%  File man/render.animation.Rd in package ndtv, part of the Statnet suite
%  of packages for network analysis, http://statnet.org .
%
%  This software is distributed under the GPL-3 license.  It is free,
%  open source, and has the attribution requirements (GPL Section 7) at
%  http://statnet.org/attribution
%
%  Copyright 2012-2013 Statnet Commons
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\name{render.animation}
\alias{render.animation}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Render animations of \code{networkDynamic} objects as movies in various formats
}
\description{
Takes a network object which describes a sequence of changes in properties of a network and graphically renders it out as a sequence of images to create an animation . Normally the coordinates determining the vertex positions at specified time points should have been calculated and stored in the network object, along with the \code{slice.par} list of parameters describing when and how the network should be divided up in time. If the coordinate data is not found, \code{\link{compute.animation}} will be called with default arguments.

Appropriate `static' networks for each time region will be generated by \code{network.collapse}.  The rendering of each frame is drawn by \link{plot.network} and most arguments are supported and are passed through to control the details of rendering. The rendered images are stored using the \code{\link{animation}} package and can be played in the plot window (\code{\link{ani.replay}}) or saved to a movie file with \code{\link{saveVideo}}.
}
\usage{
render.animation(net, 
  render.par=list(tween.frames=10, show.time=TRUE, show.stats=NULL, extraPlotCmds=NULL),
  plot.par=list(bg='white'), ani.options = list(interval=0.1), verbose=TRUE, 
  label, displaylabels=!missing(label), xlab, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{net}{
The networkDynamic object to be rendered, usually containing pre-computed vertex positions as dynamic attributes. 
}
  \item{render.par}{
Named list of parameters to specify the behavior of the animation.
  \itemize{
  \item \code{tween.frames} the number of interpolated frames to generate between each calculated network layout (default 10). 
  \item \code{show.time} If \code{TRUE}, labels the plot with onset and terminus time for each slice. 
  \item \code{show.stats} \code{NULL}, or a string containing a formula to be passed to summary.stergm to display the network statistics for the current slice on the plot
  \item \code{extraPlotCmds} NULL, or additional plot commands to draw on each frame of the animation. 

  }
}
 \item{plot.par}{
   list of `high-level' plotting control arguments to be passed to \code{\link{par}}. e.g. \code{bg} for background color, margins, fonts, etc. 
 }
 \item{ani.options}{
  list of control arguments for the animation library. For example. \code{interval} controls the delay between frames in playback, \code{ani.dev} and \code{ani.type} can be used to set non-default graphics devices for rendering (i.e \code{'jpeg'} instead of \code{'png'}). See \code{\link{ani.options}}
 }
\item{verbose}{
 If \code{TRUE}, update text will be printed to the console to report the status of the rendering process.
}
\item{label}{
  name of vertex variable or vector of values to be used as vertex labels (see \code{\link{plot.network.default}}).  If not specified, will be set to \code{\link{network.vertex.names}} of appropriate slice network
}
\item{displaylabels}{
  logical. should labels be drawn on the plot for each vertex? See \code{\link{plot.network.default}}}
\item{xlab}{label for x-axis of the plot. (see \code{\link{plot.default}}) If \code{render.par$show.stats} or \code{render.par$show.time} are set, the x-axis label will be used to display the appropriate values. }

  \item{...}{
 Other parameters to control network rendering. See \code{link{plot.network.default}. TEA attribute names can be used}
 }
}
  
  
\details{

Most of the network plotting arguments are passed directly to \code{\link{plot.network.default}}.  

If no \code{slice.par} network attribute is found to define the time range to render,  it will make one up using the smallest and largest non-Inf time values and unit-length non-overlapping time steps. 

If no dynamic coordinate information has been stored, \code{\link{compute.animation}} will be called with default values to try to do the layout before giving up. 

Additional plot commands passed in via the \code{extraPlotCmds} argument will be  passed to \code{eval()} after each frame is rendered and can be used to add extra annotations to the plot. 
}

\note{
A few of the network drawing arguments have slightly different interpretations than their \link{plot.network} equivalents:
\itemize{
\item \code{xlab} will be used to display time and network statistics if those \code{render.par} parameters are set
\item \code{xlim} and \code{ylim} will be set using max and min observed coordinate values so that all network slices will appear on the plot

\item \code{label} if not set explicitly, will default to the vertex names for the appropriate slice network. 
}
If the background color is transparent and not explicitly set, it will be reset to white to prevent unintentional behavior when exporting movies via ffmpeg. 
}

\value{
The recorded plots are stored as a list in \code{.ani.env$.images} (this will probably change in the future)
}
\references{
Skye Bender-deMoll and McFarland, Daniel A. (2006) The Art and Science of Dynamic Network Visualization. \emph{Journal of Social Structure. Volume 7, Number 2} \url{http://www.cmu.edu/joss/content/articles/volume7/deMollMcFarland/}
}
\author{
Skye Bender-deMoll, and the statnet team. 
}
\note{
This is still a draft implementation and most of the package. Most specifics (including the function name) are likely to change.
}


\seealso{
\link{compute.animation},\link{ani.record}, \link{plot.network} and the package vignette \code{vignette('ndtv')}
}
\examples{

require(ndtv)
# trivial example

triangle <- network.initialize(3) # create a toy network
add.edge(triangle,1,2)
# add an edge between vertices 1 and 2
add.edge(triangle,2,3)
# add a more edges
activate.edges(triangle,at=1) # turn on all edges at time 1 only
activate.edges(triangle,onset=2, terminus=3,
e=get.edgeIDs(triangle,v=1,alter=2))
add.edges.active(triangle,onset=4, length=2,tail=3,head=1)
render.animation(triangle)
ani.replay() 

# an example with changing attributes
wheel <- network.initialize(10)  # create a toy network
add.edges.active(wheel,tail=1:9,head=c(2:9,1),onset=1:9, terminus=11)
add.edges.active(wheel,tail=10,head=c(1:9),onset=10, terminus=12)
# set a dynamic value for edge widths
activate.edge.attribute(wheel,'width',1,onset=0,terminus=3) 
activate.edge.attribute(wheel,'width',5,onset=3,terminus=7)
activate.edge.attribute(wheel,'width',10,onset=3,terminus=Inf)
# set a value for vertex sizes
activate.vertex.attribute(wheel,'mySize',1, onset=-Inf,terminus=Inf)
activate.vertex.attribute(wheel,'mySize',3, onset=5,terminus=10,v=4:8)
# set values for vertex colors
activate.vertex.attribute(wheel,'color','gray',onset=-Inf,terminus=Inf)
activate.vertex.attribute(wheel,'color','red',onset=5,terminus=6,v=4)
activate.vertex.attribute(wheel,'color','green',onset=6,terminus=7,v=5)
activate.vertex.attribute(wheel,'color','blue',onset=7,terminus=8,v=6)
activate.vertex.attribute(wheel,'color','pink',onset=8,terminus=9,v=7)
# render it all
render.animation(wheel,edge.lwd='width',vertex.cex='mySize',vertex.col='color')
ani.replay()

# simulation based example
# disabled to save time when testing
\dontrun{
require(tergm)
# load in example data, results of a basic stergm sim
data(stergm.sim.1)

# (optional) pre-compute coordinates for set time range
# (optional) limit time range to a few steps to speek example
slice.par=list(start=0,end=3,interval=1, aggregate.dur=1,rule='any')
compute.animation(stergm.sim.1,slice.par=slice.par)


# define the number of inbetween frames and a formula for stats to display
render.par<-list(tween.frames=5,show.time=TRUE,
                show.stats="~edges+gwesp(0,fixed=TRUE)")
                
# render the movie, with labels, smaller vertecies, etc
render.animation(stergm.sim.1,render.par=render.par,
                 edge.col="darkgray",displaylabels=TRUE,
                 label.cex=.6,label.col="blue")
                 
# preview the movie in the plot window
ani.replay()     

# save the movie as mp4 compressed video (if FFMPEG installed)
saveVideo(ani.replay(),video.name="stergm.sim.1.mp4", 
                    other.opts="-b 5000k",clean=TRUE)
}

}

