% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{multiEstimate}
\alias{multiEstimate}
\title{Wavelet deconvolution signal estimate from the noisy multichannel convoluted signal}
\usage{
multiEstimate(Y, G = directBlur(nrow(as.matrix(Y)), ncol(as.matrix(Y))),
  alpha = rep(1, dim(as.matrix(Y))[2]),
  resolution = resolutionMethod(detectBlur(G)), sigma = as.numeric(c()),
  j0 = 3L, j1 = NA_integer_, eta = NA_real_,
  thresh = multiThresh(as.matrix(Y), G = G, alpha = alpha, j0 = j0, j1 = j1,
  eta = eta, deg = 3L), shrinkType = "hard", deg = 3L)
}
\arguments{
\item{Y}{An input signal either an n by m matrix containing the multichannel signal to be analysed or single vector of n elements for the single channel. In the multichannel case, each of the m columns represents a channel of n observations.}

\item{G}{The input multichannel blur matrix/vector (needs to be the same dimension/length as the signal input which is a matrix or vector for the multichannel or single channel case respectively). This argument dictates the form of blur present in each of the channels.}

\item{alpha}{A numeric vector, with m elements, specifying the level of long memory for the noise process within each channel of the form alpha = 2 - 2H, where H is the Hurst parameter. If alpha is a single element, that same element is repeated across all required channels.}

\item{resolution}{A character string describing which resolution selection method is to be applied.\itemize{
\item 'smooth': Smooth stopping rule in Fourier domain applied piecewise in each channel and maximum selected which is appropriate if blurring kernel is of regular smooth blur type or direct model (no convolution).
\item 'block': Blockwise variance selection method is used which is appropriate for box car type.}
The default choice uses the detectBlur function to identify what type of blur matrix, G, is input and then maps that identification to the resolution type via a simple switch statement in the hidden \code{resolutionMethod} function, whereby, identified 'smooth' and 'direct' blur use the smooth resolution selection while box.car uses the blockwise resolution selection method.}

\item{sigma}{A numeric vector with m elements that specifies the level of noise (standard deviation) in each channel. The default method uses the Median Absolute Deviation of wavelet coefficients in the finest resolution (see \code{\link{multiSigma}}) for details.}

\item{j0}{The coarsest resolution level for the wavelet expansion.}

\item{j1}{The finest resolution level for the wavelet expansion. If unspecified, the function will compute all thresholds up to the maximum possible resolution level at j1 = log2(n) - 1.}

\item{eta}{The smoothing parameter. The default level is \eqn{2\sqrt(\alpha^*)} where \eqn{\alpha^*} is an optimal level depending on the type of blur. (see Kulik, Sapatinas and Wishart (2014) for details and justification)}

\item{thresh}{A numeric vector of resolution level thresholds to use in the wavelet thresholded estimator of the true signal. It should have enough elements to construct the required expansion with all resolutions. That is, have \code{j1} - \code{j0} + 2 elements. If a single element is input, it is repeated to be the universal threshold across all resolutions.}

\item{shrinkType}{A character string that specifies which thresholding regime to use. 
Available choices are the 'hard', 'soft' or 'garrote'.}

\item{deg}{The degree of the auxiliary polynomial used in the Meyer wavelet.}
}
\value{
A numeric vector of the estimate of the underlying signal of interest.
}
\description{
Estimates the underlying signal of interest from a multichannel noisy deconvolution model.
}
\details{
Function requires input of a noisy multichannel signal matrix, Y, which contains the information for each channel in each of the m columns. Optional inputs are a matrix, G, the same dimension as Y, that gives the multichannel blur information.
}
\examples{
library(mwaved)
# Simulate the multichannel doppler signal.
m <- 3
n <- 2^10
x <- (1:n)/n
signal <- makeDoppler(n)
# Noise levels per channel
e <- rnorm(m * n)
# Create Gamma blur
shape <- seq(from = 0.5, to = 1, length = m)
scale <- rep(0.25, m)
G <- gammaBlur(n, shape = shape, scale = scale)
# Convolve the signal
X <- blurSignal(signal, G)
# Create error with custom signal to noise ratio
SNR <- c(10, 15, 20)
sigma <- sigmaSNR(X, SNR)
if (requireNamespace("fracdiff", quietly = TRUE)) {
  alpha <- c(0.75, 0.8, 1)
} else {
  alpha <- rep(1, m)
}
E <- multiNoise(n, sigma, alpha)
# Create noisy & blurred multichannel signal
Y <- X + E
# Estimate the underlying doppler signal
dopplerEstimate <- multiEstimate(Y, G = G, alpha = rep(1, m))
# Plot the result and compare with truth
par(mfrow=c(2, 1))
matplot(x, Y, type = 'l', main = 'Noisy multichannel signal')
plot(x, signal, type = 'l', lty = 2, main = 'True Doppler signal and estimate', col = 'red')
lines(x, dopplerEstimate)

}
