% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/center.R
\name{center}
\alias{center}
\title{Centering Predictor Variables in Single-Level and Multilevel Data}
\usage{
center(data, ..., cluster = NULL, type = c("CGM", "CWC"),
       cwc.mean = c("L2", "L3"), value = NULL, append = TRUE, name = ".c",
       as.na = NULL, check = TRUE)
}
\arguments{
\item{data}{a numeric vector for centering a predictor variable, or a
data frame for centering more than one predictor variable.}

\item{...}{an expression indicating the variable names in \code{data} e.g.,
\code{center(dat, x1, x2)} for centering the variables \code{x1}
and \code{x2} in the data frame \code{dat}. Note that the
operators \code{.}, \code{+}, \code{-}, \code{~}, \code{:},
\code{::}, and \code{!} can also be used to select variables,
see 'Details' in the \code{\link{df.subset}} function.}

\item{cluster}{a character string indicating the name of the cluster variable
in \code{data} for a two-level model, a character vector
indicating the names of the cluster variables in \code{data}
for a three-level model, or a vector or data frame representing
the nested grouping structure (i.e., group or cluster variables).
Alternatively, a character string or character vector indicating
the variable name(s) of the cluster variable(s) in \code{data}.
Note that the cluster variable at Level 3 come first in a
three-level model, i.e., \code{cluster = c("level3", "level2")}.}

\item{type}{a character string indicating the type of centering, i.e.,
\code{"CGM"} for centering at the grand mean (i.e., grand mean
centering, default when \code{cluster = NULL}) or \code{"CWC"}
for centering within cluster (i.e., group mean centering, default
when specifying the argument \code{cluster}).}

\item{cwc.mean}{a character string indicating the type of centering of a level-1
predictor variable in a three-level model, i.e., \code{L2}
(default) for centering the predictor variable at the level-2
cluster means, and  \code{L3} for centering the predictor
variable at the level-3 cluster means.}

\item{value}{a numeric value for centering on a specific user-defined value.
Note that this option is only available when specifying a
single-level predictor variable, i.e., \code{cluster = NULL}.}

\item{append}{logical: if \code{TRUE} (default), centered variable(s) are
appended to the data frame specified in the argument \code{data}.}

\item{name}{a character string or character vector indicating the names of
the centered predictor variables. By default, centered predictor
variables are named with the ending \code{".c"} resulting in
e.g. \code{"x1.c"} and \code{"x2.c"}. Variable names can also
be specified by using a character vector matching the number
of variables (e.g., \code{name = c("center.x1", "center.x2")}).}

\item{as.na}{a numeric vector indicating user-defined missing values, i.e.
these values are converted to \code{NA} before conducting the
analysis. Note that \code{as.na()} function is only applied to
\code{data} but not to \code{cluster}.}

\item{check}{logical: if \code{TRUE} (default), argument specification is checked.}
}
\value{
Returns a numeric vector or data frame with the same length or same number of
rows as \code{data} containing the centered variable(s).
}
\description{
This function centers predictor variables in single-level data, two-level
data, and three-level data at the grand mean (CGM, i.e., grand mean centering)
or within cluster (CWC, i.e., group mean centering).
}
\details{
\describe{
\item{\strong{Single-Level Data}}{\strong{Predictor variables in single-level
data} can only be centered at the grand mean (CGM) by specifying
\code{type = "CGM"}:

\deqn{x_{i} - \bar{x}_{.}}

where \eqn{x_{i}} is the predictor value of observation \eqn{i} and
\eqn{\bar{x}_{.}} is the average \eqn{x} score. Note that predictor variables
can be centered on any meaningful value specifying the argument \code{value},
e.g., a predictor variable centered at 5 by applying following formula:

\deqn{x_{i} - \bar{x}_{.} + 5}

resulting in a mean of the centered predictor variable of 5.
}
\item{\strong{Two-Level Data}}{\strong{Level-1 (L1) predictor variables} in
two-level data can be centered at the grand mean (CGM) by specifying
\code{type = "CGM"}:

\deqn{x_{ij} - \bar{x}_{..}}

where \eqn{x_{ij}} is the predictor value of observation \eqn{i} in L2 cluster
\eqn{j} and \eqn{\bar{x}_{..}} is the average \eqn{x} score.

L1 predictor variables are centered at the group mean (CWC) by specifying
\code{type = "CWC"} (Default):

\deqn{x_{ij} - \bar{x}_{.j}}

where \eqn{\bar{x_{.j}}} is the average \eqn{x} score in cluster \eqn{j}.

\strong{Level-2 (L1) predictor variables} in two-level data can only be
centered at the grand mean:

\deqn{x_{.j} - \bar{x}_{..}}

where \eqn{x_{.j}} is the predictor value of Level 2 cluster \eqn{j} and
\eqn{\bar{x}_{..}} is the average Level-2 cluster score. Note that the cluster
membership variable needs to be specified when centering a L2 predictor variable
in two-level data. Otherwise the average \eqn{x_{ij}} individual score instead
of the average \eqn{x_{.j}} cluster score is used to center the predictor
variable.
}
\item{\strong{Three-Level Data}}{\strong{Level-1 (L1) predictor variables} in
three-level data can be centered at the grand mean (CGM) by specifying
\code{type = "CGM"}:

\deqn{x_{ijk} - \bar{x}_{...}}

where \eqn{x_{ijk}} is the predictor value of observation \eqn{i} in Level-2
cluster \eqn{j} within Level-3 cluster \eqn{k} and \eqn{\bar{x}_{...}} is the average
 \eqn{x} score.

L1 predictor variables are centered within cluster (CWC) by specifying
\code{type = "CWC"} (Default). However, L1 predictor variables can be either
centered within Level-2 clusters (\code{cwc.mean = "L2"}, Default, see Brincks et
al., 2017):

\deqn{x_{ijk} - \bar{x}_{.jk}}

or within Level-3 clusters (\code{cwc.mean = "L3"}, see Enders, 2013):

\deqn{x_{ijk} - \bar{x}_{..k}}

where \eqn{\bar{x}_{.jk}} is the average \eqn{x} score in Level-2 cluster
\eqn{j} within Level-3 cluster \eqn{k} and \eqn{\bar{x}_{..k}} is the average
\eqn{x} score in Level-3 cluster \eqn{k}.

\strong{Level-2 (L2) predictor variables} in three-level data can be centered
at the grand mean (CGM) by specifying \code{type = "CGM"}:

\deqn{x_{.jk} - \bar{x}_{...}}

where \eqn{x_{.jk}} is the predictor value of Level-2 cluster \eqn{j} within
Level-3 cluster \eqn{k} and \eqn{\bar{x}_{...}} is the average Level-2 cluster
score.

L2 predictor variables are centered within cluster (CWC) by specifying
\code{type = "CWC"} (Default):

\deqn{x_{.jk} - \bar{x}_{..k}}

where \eqn{\bar{x}_{..k}} is the average \eqn{x} score in Level-3 cluster
\eqn{k}.

\strong{Level-3 (L3) predictor variables} in three-level data can only be
centered at the grand mean:

\deqn{x_{..k} - \bar{x}_{...}}

where \eqn{x_{..k}} is the predictor value of Level-3 cluster \eqn{k} and
\eqn{\bar{x}_{...}} is the average Level-3 cluster score. Note that the cluster
membership variables at Level 2 and Level 3 need to be specified when centering
a L3 predictor variable in three-level data.}
}
}
\examples{
#----------------------------------------------------------------------------
# Predictor Variables in Single-Level Data

# Example 1a: Center predictor 'disp' at the grand mean
center(mtcars, disp, append = FALSE)

# Alternative specification without using the '...' argument
center(mtcars$disp)

# Example 1b: Center predictors 'disp' and 'hp' at the grand mean and append to 'mtcars'
center(mtcars, disp, hp)

# Alternative specification without using the '...' argument
cbind(mtcars, center(mtcars[, c("disp", "hp")]))

# Example 1c: Center predictor 'disp' at the value 3
center(mtcars, disp, value = 3)

# Example 1d: Center predictors 'disp' and 'hp' and label with the suffix ".v"
center(mtcars, disp, hp, name = ".v")

#----------------------------------------------------------------------------
# Predictor Variables in Two-Level Data

# Load data set "Demo.twolevel" in the lavaan package
data("Demo.twolevel", package = "lavaan")

# Example 2a: Center L1 predictor 'y1' within cluster
center(Demo.twolevel, y1, cluster = "cluster")

# Alternative specification without using the '...' argument
center(Demo.twolevel$y1, cluster = Demo.twolevel$cluster)

# Example 2b: Center L2 predictor 'w2' at the grand mean
center(Demo.twolevel, w1, cluster = "cluster")

# Example 2c: Center L1 predictor 'y1' within cluster and L2 predictor 'w1' at the grand mean
center(Demo.twolevel, y1, w1, cluster = "cluster")

#----------------------------------------------------------------------------
# Predictor Variables in Three-Level Data

# Create arbitrary three-level data
Demo.threelevel <- data.frame(Demo.twolevel, cluster2 = Demo.twolevel$cluster,
                                             cluster3 = rep(1:10, each = 250))

# Example 3a: Center L1 predictor 'y1' within L2 cluster
center(Demo.threelevel, y1, cluster = c("cluster3", "cluster2"))

# Example 3b: Center L1 predictor 'y1' within L3 cluster
center(Demo.threelevel, y1, cluster = c("cluster3", "cluster2"), cwc.mean = "L3")

# Example 3c: Center L1 predictor 'y1' within L2 cluster and L2 predictor 'w1' within L3 cluster
center(Demo.threelevel, y1, w1, cluster = c("cluster3", "cluster2"))}
\references{
Brincks, A. M., Enders, C. K., Llabre, M. M., Bulotsky-Shearer, R. J., Prado, G.,
& Feaster, D. J. (2017). Centering predictor variables in three-level contextual
models. \emph{Multivariate Behavioral Research, 52}(2), 149–163.
https://doi.org/10.1080/00273171.2016.1256753

Chang, C.-N., & Kwok, O.-M. (2022) Partitioning Variance for a Within-Level
Predictor in Multilevel Models. \emph{Structural Equation Modeling: A
Multidisciplinary Journal}. Advance online publication.
https://doi.org/10.1080/10705511.2022.2051175

Enders, C. K. (2013). Centering predictors and contextual effects. In M. A.
Scott, J. S. Simonoff, & B. D. Marx (Eds.), \emph{The Sage handbook of
multilevel modeling} (pp. 89-109). Sage. https://dx.doi.org/10.4135/9781446247600

Enders, C. K., & Tofighi, D. (2007). Centering predictor variables in
cross-sectional multilevel models: A new look at an old issue. \emph{Psychological
Methods, 12}, 121-138. https://doi.org/10.1037/1082-989X.12.2.121

Rights, J. D., Preacher, K. J., & Cole, D. A. (2020). The danger of conflating
level-specific effects of control variables when primary interest lies in
level-2 effects. \emph{British Journal of Mathematical & Statistical Psychology,
73}, 194-211. https://doi.org/10.1111/bmsp.12194

Yaremych, H. E., Preacher, K. J., & Hedeker, D. (2021). Centering categorical
predictors in multilevel models: Best practices and interpretation.
\emph{Psychological Methods}. Advance online publication.
https://doi.org/10.1037/met0000434
}
\seealso{
\code{\link{coding}}, \code{\link{cluster.scores}}, \code{\link{rec}},
\code{\link{item.reverse}}, \code{\link{rwg.lindell}}, \code{\link{item.scores}}.
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
