\name{ivmediate} 
\alias{ivmediate} 
\title{Causal Mediation Analysis with Treatment Noncompliance} 
\description{ 
'ivmediate' is used to estimate local average causal mediation effects, local average natural direct effects and local average treatment effects for compliers using the method of Yamamoto (2013).
} 

\usage{

ivmediate(model.t, model.m, model.y, ci = TRUE, sims = 1000, boot = TRUE,
          enc = "enc.name", treat = "treat.name", mediator = "med.name",
          conf.level = .95, long = TRUE, dropobs = FALSE,
          multicore = FALSE, mc.cores = getOption("mc.cores", 2L))

} 

\arguments{ 
\item{model.t}{ a fitted model object for actual treatment. Can be of class 'lm' or 'glm'.}
\item{model.m}{ a fitted model object for mediator.  Can be of class 'lm', or 'glm'.} 
\item{model.y}{ a fitted model object for outcome.  Can be of class 'lm', or 'glm'.} 
\item{ci}{ a logical value. if 'TRUE' both point estimates and confidence intervals are calculated; if 'FALSE' only point estimates are produced. Default is 'TRUE'.}
\item{sims}{ number of Monte Carlo draws for nonparametric bootstrap or quasi-Bayesian approximation.} 
\item{boot}{ a logical value. if 'FALSE' a quasi-Bayesian approximation is used for confidence intervals; if 'TRUE' nonparametric bootstrap will be used. Default is 'TRUE'.}
\item{enc}{ a character string indicating the name of the encouragement variable used in the models. Must be a binary variable.}
\item{treat}{ a character string indicating the name of the actual treatment variable used in the models.  Must be a binary variable.}
\item{mediator}{ a character string indicating the name of the mediator variable used in the models.}
\item{conf.level}{ a numeric vector indicating the levels of the returned two-sided confidence intervals. Default is to return the 2.5 and 97.5 percentiles of the simulated quantities.} 
\item{long}{ a logical value. If 'TRUE', the output will contain the entire sets of simulation draws of the estimated effects. Default is 'TRUE'.}
\item{dropobs}{ a logical value indicating the behavior when the model frames of 'model.t', 'model.m' and 'model.y' are composed of different observations. If 'TRUE', models will be re-fitted using common data rows. If 'FALSE', error is returned. Default is 'FALSE'.}
\item{multicore}{ a logical value indicating whether to parallelize simulations into multiple cores. Default is 'FALSE'. Note that this option is currently unavailable on Windows.}
\item{mc.cores}{ number of cores to be used in the parallelization of bootstrap or Monte Carlo simulations. Default is the value in the 'mc.cores' option (see \code{\link{mclapply}}).}
}

\details{This is the workhorse function for estimating local causal mediation effects for compliers using the approach of Yamamoto (2013).

Although the method is quite general and accommodate various types of mediator and outcome variables, the current function can only handle binary variables (modeled via \code{glm} with \code{family=binomial}) and continuous variables (modeled via \code{lm}). In addition, when the mediator is continuous and its model contains any predictor other than the encouragement and actual treatment, computation of confidence intervals is extremely slow because it requires numerical integration via \code{\link{integrate}} for each observation in each simulation iteration. Users are advised to use a powerful computer (preferably equipped with multiple CPU cores and a non-Windows operating system) for such calculation, especially if the data contain many rows and/or the desired number of simulations is large.
}


\value{\code{ivmediate} returns an object of class '\code{ivmediate}', a list that contains the components listed below.  Some of these elements are not available depending on the values of the 'ci' and 'long' options.

The function \code{summary} (i.e., \code{summary.ivmediate}) can be used to obtain a table of the results.  The function \code{plot} (i.e., \code{plot.ivmediate}) can be used to produce a plot of the estimated effects along with their confidence intervals. 

\item{dc0, dc1}{point estimates for the local average causal mediation effects under the control and treatment conditions.}
\item{dc0.ci, dc1.ci}{confidence intervals for the local average causal mediation effects. The confidence levels are set at the value specified in 'conf.level'.}
\item{dc0.issue, dc1.issue}{number of observations for which the numerical integration via \code{\link{integrate}} encountered computational problems when calculating dc0 and dc1, respectively.}
\item{dc0.inf, dc1.inf}{number of observations for which the numerical integration produced non-finite values when calculating dc0 and dc1, respectively. (Such values are excluded from the calculation of average effects.)}
\item{dc0.sims, dc1.sims}{vectors of length 'sims' containing simulation draws of local average causal mediation effects.}
\item{zc0, zc1}{point estimates for the local average natural direct effects under the control and treatment conditions.}
\item{zc0.ci, zc1.ci}{confidence intervals for the local average natural direct effects.}
\item{zc0.issue, zc1.issue}{number of observations for which the numerical integration encountered computational problems when calculating zc0 and zc1, respectively.}
\item{zc0.inf, zc1.inf}{number of observations for which the numerical integration produced non-finite values for zc0 and zc1.}
\item{zc0.sims, zc1.sims}{vectors of length 'sims' containing simulation draws of local average natural direct effects.}
\item{tauc}{point estimate for the local average treatment effect.}
\item{tauc.ci}{confidence interval for the local average treatment effect.}
\item{tauc.sims}{a vector of length 'sims' containing simulation draws of the local average treatment effect.}
\item{boot}{logical, the 'boot' argument used.}
\item{enc}{a character string indicating the name of the 'enc' variable used.}
\item{treat}{a character string indicating the name of the 'treat' variable used.}
\item{mediator}{a character string indicating the name of the 'mediator' variable used.}
\item{conf.level}{the confidence levels used. }
\item{nobs}{number of observations in the model frame for 'model.t', 'model.m' and 'model.y'. May differ from the numbers in the original models input to 'ivmediate' if 'dropobs' was 'TRUE'.}
\item{sims}{number of simulation draws used.}
}

\references{
Yamamoto, T. (2013). Identification and Estimation of Causal Mediation Effects with Treatment Noncompliance. Unpublished manuscript.
} 

\author{Teppei Yamamoto, Massachusetts Institute of Technology, \email{teppei@mit.edu}.}
 
\seealso{\code{\link{plot.ivmediate}}, \code{\link{summary.ivmediate}}}

\examples{ 
# Examples with JOBS II Field Experiment

# ** For illustration purposes a small number of simulations are used **

require(parallel)
require(MASS)

data(jobs)

a <- lm(comply ~ treat + sex + age + marital + nonwhite + educ + income, 
        data = jobs)
b <- glm(job_dich ~ comply + treat + sex + age + marital + nonwhite + educ + income, 
        data = jobs, family = binomial)
c <- lm(depress2 ~ job_dich * (comply + treat) + sex + age + marital + nonwhite + educ + income, 
        data = jobs)

out <- ivmediate(a, b, c, sims = 50, boot = FALSE, 
                 enc = "treat", treat = "comply", mediator = "job_dich")
                 
summary(out)
plot(out)

} 

