% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matrixpls.weights.R
\name{weightAlgorithms}
\alias{weight.factor}
\alias{weight.fixed}
\alias{weight.optim}
\alias{weight.pls}
\alias{weight.principal}
\alias{weightAlgorithms}
\title{Indicator weigth algoritms}
\usage{
weight.pls(S, model, W.model, outerEstimators = NULL,
  innerEstimator = inner.path, ..., convCheck = convCheck.absolute,
  variant = "lohmoller", tol = 1e-05, iter = 100, validateInput = TRUE)

weight.optim(S, model, W.model, parameterEstimator = params.separate,
  optimCriterion = optim.maximizeInnerR2, method = "BFGS", ...,
  validateInput = TRUE, standardize = TRUE)

weight.fixed(S, model, W.model = NULL, ..., standardize = TRUE)

weight.factor(S, model, W.model = NULL, ..., fm = "minres",
  standardize = TRUE)

weight.principal(S, model, W.model = NULL, ..., standardize = TRUE)
}
\arguments{
\item{S}{Covariance matrix of the data.}

\item{model}{There are two options for this argument: 1. lavaan script or lavaan parameter
table, or 2. a list containing three matrices
\code{inner}, \code{reflective}, and \code{formative} defining the free regression paths
in the model.}

\item{W.model}{A matrix specifying the weight relationships and their starting values.}

\item{outerEstimators}{A function or a list of functions used for outer estimation. If
the value of this parameter is a function, the same function is applied to all
composites. If the value is a list, the composite \code{n} is estimated
with the estimator in the \code{n}th position in the list. If this argument is
\code{NULL} the \code{\link{outer.modeA}} is used for all composites that are linked to at least
one indicator in the \code{reflective} matrix.\code{\link{outer.modeB}} is used for all other
composites. See \code{\link{outerEstimators}}.}

\item{innerEstimator}{A function used for inner estimation. The default is \code{\link{inner.path}}.
See \code{\link{innerEstimators}}.}

\item{...}{All other arguments are passed through to other estimation functions.}

\item{convCheck}{A function that takes the old and new weight matrices and
returns a scalar that is compared against \code{tol} to check for convergence. The default
is \code{\link{convCheck.absolute}}. See \code{\link{convergenceCheck}}.}

\item{variant}{Choose either Lohmöller's (\code{"lohmoller"}, default) or Wold's (\code{"wold"}) 
variant of PLS. In Wold's variant the inner and outer estimation steps are repeated for each
indicator block whereas in Lohmöller's variant the weights for all composites are calculated
simultaneously.}

\item{tol}{Decimal value indicating the tolerance criterion for convergence.}

\item{iter}{An integer indicating the maximum number of iterations.}

\item{validateInput}{A boolean indicating whether the validity of the parameter values should be tested.}

\item{parameterEstimator}{A function that takes three or more arguments, the data covariance matrix \code{S},
model specification \code{model}, and weights \code{W} and returns a named vector of parameter estimates. The default is \code{\link{params.separate}}}

\item{optimCriterion}{A function that taking an object of class class 
\code{matrixpls} and returning a scalar. The default is \code{\link{optim.maximizeInnerR2}}. 
See \code{\link{optimCriterion}}}

\item{method}{The minimization algorithm to be used. See \code{\link[stats]{optim}}
for details. Default is \code{"BFGS"}.}

\item{standardize}{A boolean indicating whether \code{S} the weights should be scaled to produce
standardized composites.}

\item{fm}{factoring method for estimating the common factor model. Possible values are
\code{minres}, \code{wls}, \code{gls}, \code{pa}, and \code{ml}. The parameter is passed through to
to \code{\link[psych]{fa}}.}
}
\value{
An object of class \code{"matrixplsweights"}, which is a matrix containing the weights with the following attributes:

\item{iterations}{Number of iterations performed}

\item{converged}{A boolean indicating if the algorithm converged}

\item{history}{A data.frame containing the weights for each iteration}

\code{weight.pls} returns the following as attributes:

\item{S}{the sample covariance matrix.}
\item{E}{inner weight matrix.}
\item{iterations}{the number of iterations used by the weight algorithm.}
\item{converged}{\code{TRUE} if the weight algorithm converged and \code{FALSE} otherwise.}
\item{history}{weight optimization history as a matrix.}
}
\description{
Estimates a weight matrix using Partial Least Squares or a related algorithm.

\code{weight.factor} calculates weights by estimating a common factor analysis model with a single factor for each 
indicator block and using the resulting estimates to calculate factor score weights

\code{weight.principal} calculates weights by calculating a principal component analysis for each 
indicator block and returning the weights for the first principal component.
}
\details{
Model can be specified in the lavaan format or the native matrixpls format.
The native model format is a list of three binary matrices, \code{inner}, \code{reflective},
and \code{formative} specifying the free parameters of a model: \code{inner} (\code{l x l}) specifies the 
regressions between composites, \code{reflective} (\code{k x l}) specifies the regressions of observed
data on composites, and \code{formative} (\code{l x k}) specifies the regressions of composites on the
observed data. Here \code{k} is the number of observed variables and \code{l} is the number of composites.

If the model is specified in lavaan format, the native
format model is derived from this model by assigning all regressions between latent
variables to \code{inner}, all factor loadings to \code{reflective}, and all regressions
of latent variables on observed variables to \code{formative}. Regressions between
observed variables and all free covariances are ignored. All parameters that are
specified in the model will be treated as free parameters.

The original papers about Partial Least Squares, as well as many of the current PLS
implementations, impose restrictions on the matrices \code{inner},
\code{reflective}, and \code{formative}: \code{inner} must be a lower triangular matrix,
\code{reflective} must have exactly one non-zero value on each row and must have at least
one non-zero value on each column, and \code{formative} must only contain zeros.
Some PLS implementations allow \code{formative} to contain non-zero values, but impose a
restriction that the sum of \code{reflective} and \code{t(formative)} must satisfy
the original restrictions of \code{reflective}. The only restrictions that matrixpls
imposes on \code{inner}, \code{reflective}, and \code{formative} is that these must be
binary matrices and that the diagonal of \code{inner} must be zeros.

The argument \code{W.model} is a (\code{l x k}) matrix that indicates
how the indicators are combined to form the composites. The original papers about
Partial Least Squares as well as all current PLS implementations define this as
\code{t(reflective) | formative}, which means that the weight patter must match the
model specified in \code{reflective} and \code{formative}. Matrixpls does not
require that \code{W.model} needs to match \code{reflective} and \code{formative}, but
accepts any numeric matrix. If this argument is not specified, all elements of \code{W.model} that
correspond to non-zero elements in the \code{reflective} or \code{formative} formative
matrices receive the value 1.

\code{weight.pls} calculates indicator weights by calling the 
\code{innerEstimator} and \code{outerEstimators} iteratively until either the convergence criterion or
maximum number of iterations is reached and provides the results in a matrix.

\code{weight.optim} calculates indicator weights by optimizing the indicator
weights against the criterion function using \code{\link[stats]{optim}}. The
algoritmh works by first estimating the model with the starting weights. The
resulting \code{matrixpls} object is passed to the \code{optimCriterion}
function, which evaluates the optimization criterion for the weights. The
weights are adjusted and new estimates are calculated until the optimization
criterion converges.
}
\section{Functions}{
\itemize{
\item \code{weight.pls}: partial Least Squares and other iterative two-stage weight algorithms.

\item \code{weight.optim}: calculates a set of weights to minimize an optimization criterion.

\item \code{weight.fixed}: returns the starting weights.

\item \code{weight.factor}: blockwise factor score weights.

\item \code{weight.principal}: blockwise principal component weights.
}}
\examples{
library(plspm)

# Run the customer satisfaction examle form plspm

# load dataset satisfaction
data(satisfaction)
# inner model matrix
IMAG = c(0,0,0,0,0,0)
EXPE = c(1,0,0,0,0,0)
QUAL = c(0,1,0,0,0,0)
VAL = c(0,1,1,0,0,0)
SAT = c(1,1,1,1,0,0)
LOY = c(1,0,0,0,1,0)
inner = rbind(IMAG, EXPE, QUAL, VAL, SAT, LOY)
colnames(inner) <- rownames(inner)

# Reflective model
list(1:5, 6:10, 11:15, 16:19, 20:23, 24:27)

reflective<- matrix(
  c(1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1),
  27,6, dimnames = list(colnames(satisfaction)[1:27],colnames(inner)))

# empty formative model

formative <- matrix(0, 6, 27, dimnames = list(colnames(inner), colnames(satisfaction)[1:27]))

# Estimation using covariance matrix
model <- list(inner = inner,
              reflective = reflective,
              formative = formative)

S <- cov(satisfaction[,1:27])

matrixpls.ModeA <- matrixpls(S, model)
matrixpls.ModeB <- matrixpls(S, model, outerEstimators = outer.modeB)
matrixpls.MaxR2 <- matrixpls(S, model, weightFunction = weight.optim)

# Compare the R2s from the different estimations

R2s <- cbind(r2(matrixpls.ModeA), r2(matrixpls.ModeB), r2(matrixpls.MaxR2))
print(R2s)
apply(R2s,2,mean)

# Optimization against custom function

maximizeSumOfCorrelations <- function(matrixpls.res){
  C <- attr(matrixpls.res,"C")
  model <- attr(matrixpls.res,"model")
  - sum(C[model$inner != 0])
}

matrixpls.MaxCor <- matrixpls(S, model, weightFunction = weight.optim,
                             optimCriterion = maximizeSumOfCorrelations)

# Compare the Mode B and optimized solutions

C <- attr(matrixpls.ModeB,"C")
print(C)
print(sum(C[inner != 0]))
C <- attr(matrixpls.MaxCor,"C")
print(C)
print(sum(C[inner != 0]))
}
\references{
Rosseel, Y. (2012). lavaan: An R Package for Structural Equation Modeling. \emph{Journal of Statistical Software}, 48(2), 1–36. Retrieved from http://www.jstatsoft.org/v48/i02
}

