% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fuse_ind_computation.R
\name{fuse}
\alias{fuse}
\title{Compute FUSE (Functionally Unique, Specialized and Endangered)}
\usage{
fuse(sp_dist, sp_faxes_coord, nb_NN = 5, GE, standGE = FALSE)
}
\arguments{
\item{sp_dist}{a dist object provided by \code{\link{funct.dist}},
\code{\link[cluster]{daisy}} or \code{\link[ade4]{dist.ktab}}.}

\item{sp_faxes_coord}{a data frame with the coordinates of the species on a
multidimensional space based on a selected number of axes derived from a
Principal Coordinate Analysis (PCoA). The species are in rows and the PCOA
axes are in column.}

\item{nb_NN}{a numerical value giving the number of nearest neighbor to
consider. Default: \code{nb_NN = 5}.}

\item{GE}{a numerical vector giving the IUCN status rank (DD = NA, LC = 0,
NT = 1, VU = 2, EN = 3, CR = 4) or the IUCN extinction probability
associated with each status. See Mooers \emph{et al.} (2008) for further
information. For example, DD = NA, LC = 0, NT = 0.1, VU = 0.4, EN = 0.666,
and CR = 0.999).}

\item{standGE}{a logical value to standardize the GE values.}
}
\value{
A data frame with species in rows and the following metrics in
columns:
\itemize{
\item FUSE: functionally unique, specialized and endangered (see
Pimiento \emph{et al.} (2020);
\item FUn_std: functional uniqueness standardized between 0 and 1 (see
Mouillot \emph{et al.} (2013);
\item FSp_std: functional specialization standardized between 0 and 1
(see Mouillot \emph{et al.} (2013);
}
}
\description{
This index takes into account species functional uniqueness (also called
Functional Originality), species specialisation and species IUCN status.
}
\examples{
# Load species traits data:
sp_tr <- read.csv(system.file('extdata', 'data_traits_MMA_ursus.csv', 
  package = 'mFD'), dec = ',', sep = ';', header = TRUE, row.names = 1,
  na.strings='NA')

# Trait compilation and ordination:
dimorphism      <- ordered(sp_tr$dimorphism)
breeding_site   <- ordered(sp_tr$breeding_site)
social_behavior <- ordered(sp_tr$social_behavior)
weight_max      <- log(sp_tr$adult_weight_max)
social_group    <- log(sp_tr$social_group_mean)
 
# Trait Matrix construction:
sp_tr_end <- data.frame(
  main_diet = sp_tr$main_diet, 
  foraging_water_depth = sp_tr$foraging_water_depth,
  foraging_location = sp_tr$foraging_location, 
  fasting_strategy = sp_tr$fasting_strategy,
  female_sexual_maturity = sp_tr$female_sexual_maturity, 
  weaning = sp_tr$weaning,
  gestation = sp_tr$gestation, inter_litter = sp_tr$inter_litter,
  breeding_site = sp_tr$breeding_site, 
  social_group = sp_tr$social_group_mean,
  social_behavior = sp_tr$social_behavior, 
  weight_max = sp_tr$adult_weight_max,
  dimorphism = sp_tr$dimorphism)
  
rownames(sp_tr_end) <- rownames(sp_tr)
 
# Function weigthing vector:
v <- c(0.25, 0.25, 0.25, 0.25, 0.20, 0.20, 0.20, 0.20, 0.20, 0.5, 0.5, 0.5, 
  0.5)
   
# Gower distance calculation:
sp_tr_end$main_diet <- as.factor(sp_tr_end$main_diet)
sp_tr_end$foraging_water_depth <- as.factor(sp_tr_end$foraging_water_depth)
sp_tr_end$foraging_location <- as.factor(sp_tr_end$foraging_location)
sp_tr_end$breeding_site <- as.factor(sp_tr_end$breeding_site)
sp_tr_end$social_behavior <- as.factor(sp_tr_end$social_behavior)

sp_dist_tr <- cluster::daisy(sp_tr_end, metric = c('gower'), 
  type = list(symm = c(4)), weights = v)
  
# Principal coordinate analyses
Pcoa <- ade4::dudi.pco(ade4::quasieuclid(sp_dist_tr), scann = FALSE, 
                       nf = 40)

sp_faxes_coord <- Pcoa$li[1:40]
 
# FUSE calculation:
 FUSE_res <- mFD::fuse(
    sp_dist        = sp_dist_tr, 
    sp_faxes_coord = as.matrix(sp_faxes_coord), 
    nb_NN          = 5,  
    GE             = sp_tr$IUCN_num,
    standGE        = TRUE)
 FUSE_res
    
 FUSE_res2 <- mFD::fuse(
    sp_dist        = sp_dist_tr, 
    sp_faxes_coord = as.matrix(sp_faxes_coord), 
    nb_NN          = 5,
    GE             = sp_tr$IUCN_50,
    standGE        = TRUE)
 FUSE_res2
    
 FUSE_res3 <- mFD::fuse(
    sp_dist        = sp_dist_tr, 
    sp_faxes_coord = as.matrix(sp_faxes_coord), 
    nb_NN          = 5, 
    GE             = sp_tr$IUCN_100,
    standGE        = TRUE)
 FUSE_res3
}
\references{
Mouillot \emph{et al.} (2013) Rare species support vulnerable functions in
high-diversity ecosystems. \emph{PLoS Biology}, \strong{11}, e1001569.\cr
Pimiento \emph{et al.} (2020) Functional diversity of marine megafauna in the
Anthropocene. \emph{Science Advances}, \strong{6}, eaay7650.\cr
Violle \emph{et al.} (2007) Let the concept of trait be functional! \emph{Oikos},
\strong{116}, 882-892.
}
\author{
Fabien Leprieur and Camille Albouy
}
