% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate.lvm.R
\name{estimate.lvm}
\alias{estimate.lvm}
\title{Estimation of parameters in a Latent Variable Model (lvm)}
\usage{
\method{estimate}{lvm}(x, data = parent.frame(), estimator = NULL,
  control = list(), missing = FALSE, weights, weightsname, data2, id, fix,
  index = !quick, graph = FALSE, messages = lava.options()$messages,
  quick = FALSE, method, param, cluster, p, ...)
}
\arguments{
\item{x}{\code{lvm}-object}

\item{data}{\code{data.frame}}

\item{estimator}{String defining the estimator (see details below)}

\item{control}{control/optimization parameters (see details below)}

\item{missing}{Logical variable indiciating how to treat missing data.
Setting to FALSE leads to complete case analysis. In the other case
likelihood based inference is obtained by integrating out the missing data
under assumption the assumption that data is missing at random (MAR).}

\item{weights}{Optional weights to used by the chosen estimator.}

\item{weightsname}{Weights names (variable names of the model) in case
\code{weights} was given as a vector of column names of \code{data}}

\item{data2}{Optional additional dataset used by the chosen
estimator.}

\item{id}{Vector (or name of column in \code{data}) that identifies
correlated groups of observations in the data leading to variance estimates
based on a sandwich estimator}

\item{fix}{Logical variable indicating whether parameter restriction
automatically should be imposed (e.g. intercepts of latent variables set to
0 and at least one regression parameter of each measurement model fixed to
ensure identifiability.)}

\item{index}{For internal use only}

\item{graph}{For internal use only}

\item{messages}{Control how much information should be
printed during estimation (0: none)}

\item{quick}{If TRUE the parameter estimates are calculated but all
additional information such as standard errors are skipped}

\item{method}{Optimization method}

\item{param}{set parametrization (see \code{help(lava.options)})}

\item{cluster}{Obsolete. Alias for 'id'.}

\item{p}{Evaluate model in parameter 'p' (no optimization)}

\item{...}{Additional arguments to be passed to lower-level functions}
}
\value{
A \code{lvmfit}-object.
}
\description{
Estimate parameters. MLE, IV or user-defined estimator.
}
\details{
A list of parameters controlling the estimation and optimization procedures
is parsed via the \code{control} argument. By default Maximum Likelihood is
used assuming multivariate normal distributed measurement errors. A list
with one or more of the following elements is expected:

\describe{
\item{start:}{Starting value. The order of the parameters can be shown by
calling \code{coef} (with \code{mean=TRUE}) on the \code{lvm}-object or with
\code{plot(..., labels=TRUE)}. Note that this requires a check that it is
actual the model being estimated, as \code{estimate} might add additional
restriction to the model, e.g. through the \code{fix} and \code{exo.fix}
arguments. The \code{lvm}-object of a fitted model can be extracted with the
\code{Model}-function.}

\item{starterfun:}{Starter-function with syntax
\code{function(lvm, S, mu)}.  Three builtin functions are available:
\code{startvalues}, \code{startvalues0}, \code{startvalues1}, ...}

\item{estimator:}{ String defining which estimator to use (Defaults to
``\code{gaussian}'')}

\item{meanstructure}{Logical variable indicating
whether to fit model with meanstructure.}

\item{method:}{ String pointing to
alternative optimizer (e.g. \code{optim} to use simulated annealing).}

\item{control:}{ Parameters passed to the optimizer (default
\code{stats::nlminb}).}

\item{tol:}{ Tolerance of optimization constraints on lower limit of
variance parameters.  } }
}
\examples{
dd <- read.table(header=TRUE,
text="x1 x2 x3
 0.0 -0.5 -2.5
-0.5 -2.0  0.0
 1.0  1.5  1.0
 0.0  0.5  0.0
-2.5 -1.5 -1.0")
e <- estimate(lvm(c(x1,x2,x3)~u),dd)

## Simulation example
m <- lvm(list(y~v1+v2+v3+v4,c(v1,v2,v3,v4)~x))
covariance(m) <- v1~v2+v3+v4
dd <- sim(m,10000) ## Simulate 10000 observations from model
e <- estimate(m, dd) ## Estimate parameters
e

## Using just sufficient statistics
n <- nrow(dd)
e0 <- estimate(m,data=list(S=cov(dd)*(n-1)/n,mu=colMeans(dd),n=n))
rm(dd)

## Multiple group analysis
m <- lvm()
regression(m) <- c(y1,y2,y3)~u
regression(m) <- u~x
d1 <- sim(m,100,p=c("u,u"=1,"u~x"=1))
d2 <- sim(m,100,p=c("u,u"=2,"u~x"=-1))

mm <- baptize(m)
regression(mm,u~x) <- NA
covariance(mm,~u) <- NA
intercept(mm,~u) <- NA
ee <- estimate(list(mm,mm),list(d1,d2))

## Missing data
d0 <- makemissing(d1,cols=1:2)
e0 <- estimate(m,d0,missing=TRUE)
e0
}
\seealso{
estimate.default score, information
}
\author{
Klaus K. Holst
}
\keyword{models}
\keyword{regression}
