% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/largeVis.R
\name{largeVis}
\alias{largeVis}
\title{Apply the LargeVis algorithm for visualizing large high-dimensional datasets.}
\usage{
largeVis(x, dim = 2, K = 50, n_trees = 50, tree_threshold = max(10,
  ncol(x)), max_iter = 1, distance_method = "Euclidean",
  perplexity = max(50, K/3), save_neighbors = TRUE, threads = NULL,
  verbose = getOption("verbose", TRUE), ...)
}
\arguments{
\item{x}{A matrix, where the features are rows and the examples are columns.}

\item{dim}{The number of dimensions in the output}

\item{K}{The number of nearest-neighbors to use in computing the kNN graph}

\item{n_trees}{See \code{\link{randomProjectionTreeSearch}}.  The default is set at 50, which is the number
used in the examples in the original paper.}

\item{tree_threshold}{See \code{\link{randomProjectionTreeSearch}}.  By default, this is the number of features
in the input set.}

\item{max_iter}{See \code{\link{randomProjectionTreeSearch}}.}

\item{distance_method}{One of "Euclidean" or "Cosine."  See \code{\link{randomProjectionTreeSearch}}.}

\item{perplexity}{See \code{\link{buildWijMatrix}}.}

\item{save_neighbors}{Whether to include in the output the adjacency matrix of nearest neighbors.}

\item{threads}{The maximum number of threads to spawn. Determined automatically if \code{NULL} (the default).  It is unlikely that
this parameter should ever need to be adjusted.  It is only available to make it possible to abide by the CRAN limitation that no package
use more than two cores.}

\item{verbose}{Verbosity}

\item{...}{Additional arguments passed to \code{\link{projectKNNs}}.}
}
\value{
A `largeVis` object with the following slots:
 \describe{
   \item{'knns'}{An [N,K] 0-indexed integer matrix, which is an adjacency list of each vertex' identified nearest neighbors.
   If the algorithm failed to find \code{K} neighbors, the matrix is padded with \code{NA}'s. Note that this matrix is not identical to the output
   from \code{\link{randomProjectionTreeSearch}}: missing neighbors are \code{NA}'s rather than \code{-1}'s, and the matrix is transposed.}
   \item{'wij'}{A sparse [N,N] matrix where each cell represents \eqn{w_{ij}}.}
   \item{'call'}{The call.}
   \item{'coords'}{A [D,N] matrix of the embedding of the dataset in the low-dimensional space.}
 }
}
\description{
Apply the LargeVis algorithm for visualizing large high-dimensional datasets.
}
\examples{
\dontrun{
# iris
data(iris)
dat <- as.matrix(iris[,1:4])
dat <- scale(dat)
dupes = which(duplicated(dat))
dat <- dat[-dupes,] # duplicates can cause the algorithm to fail
dat <- t(dat)
visObject <- largeVis(dat, max_iter = 20, K = 10, sgd_batches = 10000)
plot(t(visObject$coords))

# mnist
load("./mnist.Rda")
dat <- mnist$images
dim(dat) <- c(42000, 28 * 28)
dat <- (dat / 255) - 0.5
dat <- t(dat)
visObject <- largeVis(dat, n_trees = 50, tree_th = 200, K = 50)
plot(t(visObject$coords))
}

}
\references{
Jian Tang, Jingzhou Liu, Ming Zhang, Qiaozhu Mei. \href{https://arxiv.org/abs/1602.00370}{Visualizing Large-scale and High-dimensional Data.}
}

