\name{plot.kde}
\alias{plot.kde}
\title{Plot for kernel density estimate}
\description{
  Plot for kernel density estimate for 1- to 3-dimensional data.
}

\usage{
\method{plot}{kde}(x, ...)
}

\arguments{
  \item{x}{an object of class \code{kde} (output from \code{\link{kde}})}
  \item{...}{other graphics parameters}
}
  

\value{
  Plot of 1-d and 2-d kernel density estimates are sent to graphics window. Plot
  for 3-d is generated by the \code{misc3d} and \code{rgl}
  libraries and is sent to RGL window.
}


\details{
  For \code{kde} objects, the function headers for the different dimensional data are  
  \preformatted{
  ## univariate
  plot(x, xlab, ylab="Density function", add=FALSE, drawpoints=FALSE, 
    ptcol="blue", col="black", cont.col=1, cont.lwd, jitter=FALSE, cont, 
    abs.cont, approx.cont=FALSE, ...)
  
  ## bivariate
  plot(x, display="slice", cont=c(25,50,75), abs.cont, 
    approx.cont=FALSE, xlab, ylab, zlab="Density function", add=FALSE, 
    drawpoints=FALSE, drawlabels=TRUE, theta=-30, phi=40, d=4,
    ptcol="blue", col="black", ...)

  ## trivariate
  plot(x, cont=c(25,50,75), abs.cont, approx.cont=FALSE, colors,
    add=FALSE, drawpoints=FALSE, alpha, alphavec, xlab, ylab, zlab, 
    size=3, ptcol="blue", ...)   
  }

  The arguments are
  \describe{
  \item{\code{display}}{type of display, "slice" for contour plot, "persp" for perspective plot, 
    "image" for image plot, "filled.contour" for filled contour plot (1st form),
   "filled.contour2" (2nd form) (2-d plot)}
  \item{\code{cont}}{vector of percentages for contour level curves}
  \item{\code{abs.cont} }{ vector of absolute density estimate heights for contour level curves}
  \item{\code{approx.cont} }{ flag to compute approximate contour levels. Default is FALSE.} 
  \item{\code{ptcol} }{ plotting colour for data points}
  \item{\code{col} }{ plotting colour for density estimate (1-d, 2-d plot)}
  \item{\code{colors} }{ vector of colours for each contour (3-d plot)}
  \item{\code{jitter} }{ flag to jitter rug plot (1-d plot). Default is TRUE.}
  \item{\code{xlim, ylim} }{ axes limits}
  \item{\code{xlab,ylab,zlab} }{ axes labels}
  \item{\code{add} }{ flag to add to current plot. Default is FALSE.}
  \item{\code{theta,phi,d} }{ graphics parameters for perspective plots (2-d plot)}
  \item{\code{drawpoints} }{ flag to draw data points on density estimate. Default is FALSE.}
  \item{\code{drawlabels} }{ flag to draw contour labels (2-d plot). Default is TRUE.}
  \item{\code{alpha} }{ transparency value of plotting symbol (3-d plot)}
  \item{\code{alphavec} }{ vector of transparency values for contours (3-d plot)}
  \item{\code{size} }{ size of plotting symbol (3-d plot)}
  \item{\code{prior.prob} }{ vector of prior probabilities}  
  \item{\code{partcol} }{ vector of colours for partition classes (1-d, 2-d plot)} 
  }

  The 1-d plot is a standard plot of a 1-d curve. If
  \code{drawpoints=TRUE} then a rug plot is added. If \code{cont} is specified,
  the horizontal line on the x-axis indicates the \code{cont}\% highest 
  density level set.  
  
  There are different types of plotting displays for 2-d data available,
  controlled by the \code{display} parameter.  (a) If \code{display="slice"} then a slice/contour plot
  is generated using \code{contour}.  
  (b) If \code{display} is \code{"filled.contour"} or \code{"filled.contour2"} 
  then a filled contour plot is generated.
  The default contours are at 25\%, 50\%, 75\% or
  \code{cont=c(25,50,75)} which are upper percentages of
  highest density regions. See below for alternative
  contour levels. (c) \code{display="persp"} then a perspective/wire-frame plot
  is generated.  The default z-axis limits \code{zlim} are the default
  from the usual \code{persp} command. (d) If \code{display="image"} then an image plot
  is generated. Default colours are the default from the usual
  \code{image} command.

  For 3-dimensional data, the interactive plot is a series of nested
  3-d contours. 
  The default contours are \code{cont=c(25,50,75)}. See below for
  alternative  contour levels. The
  default \code{colors} are \code{heat.colors} and the
  default opacity \code{alphavec} ranges from 0.1 to 0.5.

  To specify contours, either one of \code{cont} or \code{abs.cont}
  is required. \code{cont} specifies upper percentages which
  correspond to probability contour regions. If \code{abs.cont} is set to particular
  values, then contours at these levels are drawn.
  This second option is useful for plotting
  multiple density estimates with common contour levels. See
  \code{\link{contourLevels}} for details on computing contour levels.   
  If \code{approx=FALSE}, then the exact KDE at x is computed. Otherwise the exact 
  KDE is replaced by the KDE at the nearest 
  grid point. This can dramatically reduce computation time for large data sets.
}
 

\seealso{\code{\link{kde}}}

\examples{
## univariate example
x <- rnorm.mixt(n=100, mus=1, sigmas=1, props=1)
fhat <- kde(x=x, h=hpi(x))  
plot(fhat, cont=50, cont.col="blue", cont.lwd=2)

## univariate positive data example
z <- 2^x
fhatz <- kde(x=z, h=hpi(log(z)), positive=TRUE)
  ## better to use hpi on log data
plot(fhatz)

## bivariate example
data(unicef)
H.scv <- Hscv(x=unicef)
fhat <- kde(x=unicef, H=H.scv, compute.cont=TRUE)
plot(fhat, drawpoints=TRUE, drawlabels=FALSE, col=3, lwd=3, cex=0.5)
plot(fhat, display="persp", border=NA, shade=0.3)
plot(fhat, display="image", col=rev(heat.colors(100)))
plot(fhat, display="filled.contour2", cont=seq(10,90,by=10))

## large bivariate sample, using binned estimation 
x <- rmvnorm.mixt(10000, c(0,0), invvech(c(1, 0.8, 1)))    
H <- Hpi(x, binned=TRUE)
fhat <- kde(x, H=H, binned=TRUE, compute.cont=TRUE)      
plot(fhat, display="filled.contour")  
}


\keyword{hplot}
