% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernelshap.R
\name{kernelshap}
\alias{kernelshap}
\title{Kernel SHAP}
\usage{
kernelshap(
  X,
  pred_fun,
  bg_X,
  bg_w = NULL,
  paired_sampling = TRUE,
  m = "auto",
  tol = 0.01,
  max_iter = 250,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{X}{Matrix or data.frame containing the observations to be explained.
Should only contain model features.}

\item{pred_fun}{A function taking objects like \code{X} or \code{bg_X} as input and providing numeric
predictions. Example: If "fit" denotes a logistic regression fitted via \code{stats::glm},
and SHAP values should be on the probability scale, then this argument is
\code{function(X) predict(fit, X, type = "response")}.}

\item{bg_X}{Matrix or data.frame used as background data to calculate marginal
expectations. Its column structure must be similar to \code{X}.
This data should neither be too small nor too large (50-200 rows). A large background
data slows down the calculations, while a small data set leads to imprecise SHAP values.}

\item{bg_w}{Optional vector of case weights for each row of \code{bg_X}.}

\item{paired_sampling}{Logical flag indicating whether to use paired sampling.
The default is \code{TRUE}. This means that with every feature subset S,
also its complement is evaluated, which leads to faster convergence.}

\item{m}{Number of feature subsets S to be evaluated during one iteration.
The default, "auto", equals \code{trunc(20 * sqrt(ncol(X)))}.
For the paired sampling strategy, 2m evaluations are done per iteration.}

\item{tol}{Tolerance determining when to stop. The algorithm keeps iterating until
max(sigma_n) / diff(range(beta_n)) < tol, where sigma_n are the standard errors
and beta_n are the SHAP values of a given observation.}

\item{max_iter}{If the stopping criterion (see \code{tol}) is not reached after
\code{max_iter} iterations, then the algorithm stops.}

\item{verbose}{Set to \code{FALSE} to suppress messages, warnings, and the progress bar.}

\item{...}{Currently unused.}
}
\value{
An object of class "kernelshap" with the following components:
\itemize{
\item \code{S}: Matrix with SHAP values.
\item \code{X}: Same as parameter \code{X}.
\item \code{baseline}: The average prediction on the background data.
\item \code{SE}: Standard errors corresponding to \code{S}.
\item \code{n_iter}: Number of iterations until convergence per row.
\item \code{converged}: Logical vector indicating convergence per row.
}
}
\description{
This function implements the model-agnostic Kernel SHAP algorithm explained in detail in
Covert and Lee (2021). It is an iterative refinement of the original Kernel SHAP algorithm
of Lundberg and Lee (2017).
The algorithm is applied to each row in \code{X}. Due to its iterative nature, approximate
standard errors of the resulting SHAP values are provided, and convergence is monitored.
The data rows \code{X} to be explained and the
background data \code{bg_X} should only contain feature columns required by the
prediction function \code{pred_fun}. The latter is a function taking
a data structure like \code{X} and \code{bg_X} and providing one numeric
prediction per row.
During each iteration, \code{m} subsets are evaluated until the worst standard error of the SHAP
values is small enough relative to the range of the SHAP values. This exactly follows the logic
used by Covert and Lee (2021).
}
\examples{
fit <- stats::lm(Sepal.Length ~ ., data = iris)
pred_fun <- function(X) stats::predict(fit, X)
s <- kernelshap(iris[1:2, -1], pred_fun = pred_fun, iris[-1])
s

# Matrix input works as well, and pred_fun may contain preprocessing steps.
fit <- stats::lm(Sepal.Length ~ ., data = iris[1:4])
pred_fun <- function(X) stats::predict(fit, as.data.frame(X))
X <- data.matrix(iris[2:4])
s <- kernelshap(X[1:3, ], pred_fun = pred_fun, X)
s
}
\references{
\enumerate{
\item Ian Covert and Su-In Lee. Improving KernelSHAP: Practical Shapley Value Estimation Using Linear Regression. Proceedings of The 24th International Conference on Artificial Intelligence and Statistics, PMLR 130:3457-3465, 2021.
\item Scott M. Lundberg and Su-In Lee. A Unified Approach to Interpreting Model Predictions. Advances in Neural Information Processing Systems 30, 2017.
}
}
