% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/irboost_aft.R
\name{irboost_aft}
\alias{irboost_aft}
\title{fit a robust accelerated failure time model with iteratively reweighted boosting algorithm}
\usage{
irboost_aft(
  params,
  data,
  cfun = "ccave",
  s = 1,
  delta = 0.1,
  iter = 10,
  nrounds = 100,
  del = 1e-10,
  trace = FALSE,
  ...
)
}
\arguments{
\item{params}{the list of parameters used in \code{xgb.train} of \pkg{xgboost}. Must include \code{aft_loss_distribution}, \code{aft_loss_distribution_scale}, but there is no need to include \code{objective}. The complete list of parameters is
available in the \href{http://xgboost.readthedocs.io/en/latest/parameter.html}{online documentation}.}

\item{data}{training dataset. \code{irboost_aft} accepts only an \code{xgb.DMatrix} as the input.}

\item{cfun}{concave component of CC-family, can be \code{"hacve", "acave", "bcave", "ccave"}, 
\code{"dcave", "ecave", "gcave", "hcave"}. See Table 2 in https://arxiv.org/pdf/2010.02848.pdf}

\item{s}{tuning parameter of \code{cfun}. \code{s > 0} and can be equal to 0 for \code{cfun="tcave"}. If \code{s} is too close to 0 for                     \code{cfun="acave", "bcave", "ccave"}, the calculated weights can become 0 for all observations, thus crash the program}

\item{delta}{a small positive number provided by user only if \code{cfun="gcave"} and \code{0 < s <1}}

\item{iter}{number of iteration in the IRCO algorithm}

\item{nrounds}{boosting iterations in \code{xgb.train} within each IRCO iteration}

\item{del}{convergency criteria in the IRCO algorithm, no relation to \code{delta}}

\item{trace}{if \code{TRUE}, fitting progress is reported}

\item{...}{other arguments passing to \code{xgb.train}}
}
\value{
An object of class \code{xgb.Booster} with additional elements:
\itemize{
  \item \code{weight_update_log} a matrix of \code{nobs} row by {iter} column of observation weights in each iteration of the IRCO algorithm
  \item \code{weight_update} a vector of observation weights in the last IRCO iteration that produces the final model fit
  \item \code{loss_log} sum of loss value of the composite function \code{cfun(survival_aft_distribution)} in each IRCO iteration
}
}
\description{
Fit an accelerated failure time model with the iteratively reweighted convex optimization   (IRCO) that minimizes the robust loss functions in the CC-family (concave-convex).     The convex optimization is conducted by functional descent boosting algorithm   in the R package \pkg{xgboost}. The iteratively reweighted boosting (IRBoost) algorithm reduces the weight of the        observation that leads to a large loss; it also provides weights to help        identify outliers. For time-to-event data, an accelerated failure time model (AFT
model) provides an alternative to the commonly used proportional hazards models. Note, \code{irboost} with \code{dfun=survival:aft} is the wrapper of \code{irboost_aft}, which was developed to facilitate a different data input format used in \code{xgb.train} not in \code{xgboost} at the time.
}
\examples{
\donttest{
library("xgboost")
X <- matrix(1:5, ncol=1)

# Associate ranged labels with the data matrix.
# This example shows each kind of censored labels.
#                   uncensored  right  left  interval
y_lower = c(10,  15, -Inf, 30, 100)
y_upper = c(Inf, Inf,   20, 50, Inf)
dtrain <- xgb.DMatrix(data=X, label_lower_bound=y_lower, label_upper_bound=y_upper)
                      params = list(objective="survival:aft", aft_loss_distribution="normal",
                      aft_loss_distribution_scale=1, max_depth=3, min_child_weight= 0)
watchlist <- list(train = dtrain)
bst <- xgb.train(params, dtrain, nrounds=15, watchlist=watchlist)
predict(bst, dtrain)
bst_cc <- irboost_aft(params, dtrain, nrounds=15, watchlist=watchlist, cfun="hcave", 
                      s=1.5, trace=TRUE, verbose=0)
bst_cc$weight_update
predict(bst_cc, dtrain)
}

}
\references{
Wang, Zhu (2021), \emph{Unified Robust Boosting}, arXiv eprint, \url{https://arxiv.org/abs/2101.07718}
}
\seealso{
\code{\link{irboost}}
}
\author{
Zhu Wang\cr Maintainer: Zhu Wang \email{zhuwang@gmail.com}
}
\keyword{regression}
\keyword{survival}
