#' Create a Web Map Using The National Map Services
#'
#' This function creates a \href{http://leafletjs.com/}{Leaflet} map widget with base maps offered through
#' The National Map (\href{https://nationalmap.gov/}{TNM}).
#' Information about the content of these base maps can be found within the
#' \href{https://viewer.nationalmap.gov/help/3.0\%20TNM\%20Base\%20Maps.htm}{TNM Base Maps} document.
#'
#' @param ...
#'   Leaflet options to be passed to the \code{\link[leaflet]{leafletOptions}} function.
#' @param collapsed 'logical'.
#'   If true, the layers control will be rendered as an icon that expands when hovered over.
#'
#' @details A number of \href{https://viewer.nationalmap.gov/services/}{map services} are offered through TNM.
#'   There are no use restrictions on these services.
#'   However, map content is limited to the United States and Territories.
#'   This function integrates TNM services within an interactive web map using
#'   \href{https://rstudio.github.io/leaflet/}{Leaflet for R}.
#'
#' @return Returns a 'leaflet' Hypertext Markup Language (HTML) widget object with TNM base maps.
#'   See example for instructions on how to add additional graphic layers to the map widget.
#'
#' @author J.C. Fisher, U.S. Geological Survey, Idaho Water Science Center
#'
#' @seealso \code{\link[leaflet]{addWMSTiles}}
#'
#' @keywords hplot
#'
#' @export
#'
#' @examples
#' map <- CreateWebMap()
#' lng <- c(-112.049705, -122.171257, -77.367458, -149.803565, -80.248344)
#' lat <- c(43.517810, 37.456526, 38.947206, 61.187905, 26.080860)
#' pop <- c("ID", "CA", "VA", "AK", "FL")
#' map <- leaflet::addMarkers(map, lng, lat, popup = pop)
#' map
#'

CreateWebMap <- function(..., collapsed=TRUE) {

  # establish layers
  basemap <- c("Topo"          = "USGSTopo",
               "Imagery"       = "USGSImageryOnly",
               "Imagery Topo"  = "USGSImageryTopo",
               "Hydrography"   = "USGSHydroCached",
               "Shaded Relief" = "USGSShadedReliefOnly")

  # initialize map widget
  map <- leaflet::leaflet(options=leaflet::leafletOptions(...))

  # specify attribution
  att <- paste("<a href='https://www.usgs.gov/'>U.S. Geological Survey</a> |",
               "<a href='https://www.usgs.gov/laws/policies_notices.html'>Policies</a>")

  # add tiled basemaps
  url <- .GetURL(basemap)
  opt <- leaflet::WMSTileOptions(version="1.3.0", maxNativeZoom=15)
  for (i in seq_along(basemap)) {
    map <- leaflet::addWMSTiles(map, url[i], group=names(basemap)[i], attribution=att,
                                options=opt, layers="0")
  }

  # add control feature
  opt <- leaflet::layersControlOptions(collapsed=collapsed)
  map <- leaflet::addLayersControl(map, position="topleft",
                                   baseGroups=names(basemap), options=opt)

  # add scale bar
  map <- leaflet::addScaleBar(map, position="bottomleft")

  # add mouse coordinates and zoom level;
  # derived from mapview::addMouseCoordinates function, accessed on 2017-07-17.
  lab <- paste("' longitude: ' + (e.latlng.lng).toFixed(5) +",
               "' | latitude: ' + (e.latlng.lat).toFixed(5) +",
               "' | zoom: ' + map.getZoom() + ' '")
  js <- sprintf("function(el, x, data) {
                   var map = this;
                   function addElement () {
                     var newDiv = $(document.createElement('div'));
                     $(el).append(newDiv);
                     newDiv.addClass('lnlt');
                     newDiv.css({
                       'position': 'relative',
                       'bottomleft':  '0px',
                       'background-color': 'rgba(255, 255, 255, 0.7)',
                       'box-shadow': '0 0 2px #bbb',
                       'background-clip': 'padding-box',
                       'margin': '0',
                       'text-align': 'center',
                       'color': '#333',
                       'font': '9px/1.5 \"Helvetica Neue\", Arial, Helvetica, sans-serif',
                     });
                     return newDiv;
                   }
                   var lnlt = $(el).find('.lnlt');
                   if(!lnlt.length) {
                     lnlt = addElement();
                     map.on('mousemove', function (e) {
                       lnlt.text(%s);
                     })
                   };
                 }", lab)
  map <- htmlwidgets::onRender(map, gsub(" +", " ", js))

  # return html widget
  return(map)
}


.GetURL <- function(service, host="basemap.nationalmap.gov") {
  sprintf("https://%s/arcgis/services/%s/MapServer/WmsServer?", host, service)
}
