% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bru.inference.R
\name{lgcp}
\alias{lgcp}
\title{Log Gaussian Cox process (LGCP) inference using INLA}
\usage{
lgcp(
  components,
  data,
  domain = NULL,
  samplers = NULL,
  ips = NULL,
  formula = . ~ .,
  ...,
  options = list(),
  .envir = parent.frame()
)
}
\arguments{
\item{components}{A \code{formula}-like specification of latent components.
Also used to define a default linear additive predictor.  See
\code{\link[=bru_component]{bru_component()}} for details.}

\item{data}{Likelihood-specific data, as a \code{data.frame} or
\code{SpatialPoints[DataFrame]}
object.}

\item{domain, samplers, ips}{Arguments used for \code{family="cp"}.
\describe{
\item{\code{domain}}{Named list of domain definitions.}
\item{\code{samplers}}{Integration subdomain for 'cp' family.}
\item{\code{ips}}{Integration points for 'cp' family. Defaults
to \code{fmesher::fm_int(domain, samplers)}. If explicitly given,
overrides \code{domain} and \code{samplers}.}
}}

\item{formula}{a \code{formula} where the right hand side is a general R
expression defines the predictor used in the model.}

\item{\dots}{Further arguments passed on to \code{\link[=bru_obs]{bru_obs()}}. In particular,
optional \code{E}, a single numeric used rescale all integration weights by a
fixed factor.}

\item{options}{A \link{bru_options} options object or a list of options passed
on to \code{\link[=bru_options]{bru_options()}}}

\item{.envir}{The evaluation environment to use for special arguments (\code{E},
\code{Ntrials}, \code{weights}, and \code{scale}) if not found in \code{response_data} or
\code{data}. Defaults to the calling environment.}
}
\value{
An \code{\link[=bru]{bru()}} object
}
\description{
This function performs inference on a LGCP observed via points residing
possibly multiple dimensions. These dimensions are defined via the left hand
side of the formula provided via the model parameter. The left hand side
determines the intensity function that is assumed to drive the LGCP. This may
include effects that lead to a thinning (filtering) of the point process. By
default, the log intensity is assumed to be a linear combination of the
effects defined by the formula's RHS.

More sophisticated models, e.g.
non-linear thinning, can be achieved by using the predictor argument. The
latter requires multiple runs of INLA for improving the required
approximation of the predictor. In many applications the LGCP is only
observed through subsets of the dimensions the process is living in. For
example, spatial point realizations may only be known in sub-areas of the
modelled space. These observed subsets of the LGCP domain are called samplers
and can be provided via the respective parameter. If samplers is NULL it is
assumed that all of the LGCP's dimensions have been observed completely.
}
\examples{
\donttest{
if (bru_safe_inla() &&
  require(ggplot2, quietly = TRUE) &&
  require(fmesher, quietly = TRUE) &&
  require(sn, quietly = TRUE)) {
  # Load the Gorilla data
  data <- gorillas_sf

  # Plot the Gorilla nests, the mesh and the survey boundary
  ggplot() +
    geom_fm(data = data$mesh) +
    gg(data$boundary, fill = "blue", alpha = 0.2) +
    gg(data$nests, col = "red", alpha = 0.2)

  # Define SPDE prior
  matern <- INLA::inla.spde2.pcmatern(
    data$mesh,
    prior.sigma = c(0.1, 0.01),
    prior.range = c(0.1, 0.01)
  )

  # Define domain of the LGCP as well as the model components (spatial SPDE
  # effect and Intercept)
  cmp <- geometry ~ field(geometry, model = matern) + Intercept(1)

  # Fit the model (with int.strategy="eb" to make the example take less time)
  fit <- lgcp(cmp, data$nests,
    samplers = data$boundary,
    domain = list(geometry = data$mesh),
    options = list(control.inla = list(int.strategy = "eb"))
  )

  # Predict the spatial intensity surface
  lambda <- predict(
    fit,
    fm_pixels(data$mesh, mask = data$boundary),
    ~ exp(field + Intercept)
  )

  # Plot the intensity
  ggplot() +
    gg(lambda, geom = "tile") +
    geom_fm(data = data$mesh, alpha = 0, linewidth = 0.05) +
    gg(data$nests, col = "red", alpha = 0.2)
}
}

}
