library(hutilscpp)
library(hutils)
library(data.table)

x <- rep_len(c(TRUE, FALSE), 1e4)
expect_false(any(or3s(x %between% c(TRUE, FALSE))))
expect_false(any(or3s(x %(between)% c(FALSE, TRUE))))
expect_false(any(or3s(x %]between[% c(FALSE, TRUE))))
expect_true(all(or3s(x %between% c(FALSE, TRUE))))
expect_equal(or3s(x %between% c(TRUE, TRUE), type = "which"), which(x))
expect_equal(or3s(x %between% c(FALSE, FALSE), type = "which"), which(!x))
y1 <- rep_len(c(TRUE, FALSE, FALSE), 1e4)
y2 <- rep_len(c(FALSE, TRUE, FALSE, FALSE), 1e4)
y3 <- rep_len(c(FALSE, FALSE, FALSE, FALSE, TRUE), 1e4)
expect_equal(or3s(x, y1, y2, y3, x), x | y1 | y2 | y3 | x)


# test_that("big doubles", {
x <- rep_len(1:5 + 0.5 + .Machine$integer.max, 1e4)
expect_equal(or3s(x > 2147483649.5), x > 2147483649.5)
expect_equal(sum_or3s(x > 2147483649.5), sum(x > 2147483649.5))
expect_equal(and3s(x > 2147483649.5), x > 2147483649.5)
expect_equal(sum_and3s(x > 2147483649.5), sum(x > 2147483649.5))

x <- c(1L, 3L, 4L, -3L, 2L, -1L, -2L, 0L, -5L, 5L, -4L, 6L)
x <- rep_len(x, 1001)
expect_equal(and3s(x %in% -1:5), x %in% -1:5)
expect_equal(or3s(x == -1L,,x == 2L), x %in% c(-1L, 2L))

"%(between)%" <- hutilscpp:::`%(between)%`
"%]between[%" <- hutilscpp:::`%]between[%`

# test_that("A in len>100 ", {
A <-
  c(-945028649L, -705457251L, 807204080L, 1708708214L, -885957403L,
    1862209884L, 1740762002L, 1546012157L, -1233491410L, 1256036667L,
    1009745233L, -815497191L, 137228285L, 2012907335L, -314954938L,
    -1234120580L, 2138414482L, 2089828880L, -1494606089L, 1669541061L,
    -1635694586L, 913293496L, 657757461L)
A <- rep_len(A, 101)
B <- c(-705457251L, 1:1000)
D <- rep(TRUE, length(B))
E <- rep(TRUE, length(B))


bor3 <- function(x, y, z, ...) {
  if (missing(z)) return(x | y)
  if (missing(..1)) ((x | y) | z) else bor3(x | y, z, ...)
}
expect_equal(or3s(B %in% A, D, E),
             bor3(B %in% A, D, E))
expect_equal(or3s(D, B %in% A, D, E),
             bor3(D, B %in% A, D, E))
expect_equal(or3s(D, E, B %in% A, D, E),
             bor3(D, E, B %in% A, D, E))
expect_equal(or3s(D, E, E, B %in% A, D, E),
             bor3(D, E, E, B %in% A, D, E))
expect_equal(or3s(D, E, E, B %in% A, !D, !E),
             bor3(D, E, E, B %in% A, !D, !E))
z <- function() or3s(D, E, B %in% A)
expect_equal(z(), bor3(D, E, B %in% A))

AA <- rep_len(A, length(B))
BB <- rep_len(B, length(AA))
expect_equal(or3s(AA != AA, AA != BB), AA != BB)
expect_equal(or3s(AA != AA, AA == BB), AA == BB)
expect_equal(or3s(AA != AA, AA >= BB), AA >= BB)
expect_equal(or3s(AA != AA, AA <= BB), AA <= BB)
expect_equal(or3s(AA != AA, AA > BB), AA > BB)
expect_equal(or3s(AA != AA, AA < BB), AA < BB)
expect_equal(or3s(AA != AA, AA %in% BB), AA %in% BB)
expect_equal(or3s(AA != AA, AA %notin% BB), AA %notin% BB)
expect_equal(or3s(AA %notin% BB), AA %notin% BB)
expect_equal(or3s(AA != AA, AA %between% c(1L, 2L)), AA %between% c(1L, 2L))
expect_equal(or3s(AA != AA, AA %between% c(1L, 2)), AA %between% c(1L, 2L))
expect_equal(or3s(AA != AA, AA %between% c(1L, 2.2)), AA %between% c(1L, 2L))
expect_equal(or3s(AA != AA, AA %between% c(NA, 1)), AA %between% c(NA, 1))
expect_equal(or3s(AA != AA, AA %(between)% c(1L, 1L), type = "raw"), raw(length(AA)))
expect_equal(or3s(AA != AA, AA %]between[% c(1L, 1L)), AA %]between[% c(1L, 1L))
expect_equal(or3s(AA != AA, AA %(between)% c(NA, 1)), AA < 1)
expect_equal(or3s(AA != AA, AA %]between[% c(NA, 1)), AA >= 1)
expect_equal(or3s(AA != AA, AA %between% c(1L, NaN)), AA %between% c(1L, NaN))
expect_equal(or3s(AA != AA, AA %(between)% c(1L, NaN)), AA > 1)
expect_equal(or3s(AA != AA, AA %]between[% c(1L, NaN)), AA <= 1)
expect_equal(or3s(AA != AA, AA %between% c(.Machine$integer.max, .Machine$integer.max)), AA == .Machine$integer.max)
expect_equal(or3s(AA != AA, AA %between% c(-.Machine$integer.max, -.Machine$integer.max)), AA == -.Machine$integer.max)
expect_equal(or3s(AA != AA, AA %between% c(NA, .Machine$integer.max)), rep(TRUE, length(AA)))
expect_equal(or3s(AA != AA, AA %between% c(2L, 1L)), logical(length(AA)))
expect_equal(or3s(AA != AA, AA %between% c(2L, 1)), logical(length(AA)))
BB[2] <- BB[2] + 0.5
expect_equal(or3s(AA != AA, AA != BB), AA != BB)
expect_equal(or3s(AA != AA, AA == BB), AA == BB)
expect_equal(or3s(AA != AA, AA >= BB), AA >= BB)
expect_equal(or3s(AA != AA, AA <= BB), AA <= BB)
expect_equal(or3s(AA != AA, AA > BB), AA > BB)
expect_equal(or3s(AA != AA, AA < BB), AA < BB)
expect_equal(or3s(AA != AA, AA %in% BB), AA %in% BB)
AA <- as.double(AA)
BB <- as.integer(BB)
expect_equal(or3s(AA != AA, AA != BB), AA != BB)
expect_equal(or3s(AA != AA, AA == BB), AA == BB)
expect_equal(or3s(AA != AA, AA >= BB), AA >= BB)
expect_equal(or3s(AA != AA, AA <= BB), AA <= BB)
expect_equal(or3s(AA != AA, AA > BB), AA > BB)
expect_equal(or3s(AA != AA, AA < BB), AA < BB)
expect_equal(or3s(AA != AA, AA %in% BB), AA %in% BB)
expect_equal(or3s(AA != AA, AA != BB[1]), AA != BB[1])
expect_equal(or3s(AA != AA, AA == BB[1]), AA == BB[1])
expect_equal(or3s(AA != AA, AA >= BB[1]), AA >= BB[1])
expect_equal(or3s(AA != AA, AA <= BB[1]), AA <= BB[1])
expect_equal(or3s(AA != AA, AA > BB[1]), AA > BB[1])
expect_equal(or3s(AA != AA, AA < BB[1]), AA < BB[1])
expect_equal(or3s(AA != AA, AA %in% BB[1]), AA %in% BB[1])
expect_equal(or3s(AA != AA, AA %between% BB[1:2]), AA %between% BB[1:2])
BB[2] <- BB[2] + 0.5
expect_equal(or3s(AA != AA, AA != BB), AA != BB)
expect_equal(or3s(AA != AA, AA == BB), AA == BB)
expect_equal(or3s(AA != AA, AA >= BB), AA >= BB)
expect_equal(or3s(AA != AA, AA <= BB), AA <= BB)
expect_equal(or3s(AA != AA, AA > BB), AA > BB)
expect_equal(or3s(AA != AA, AA < BB), AA < BB)
expect_equal(or3s(AA != AA, AA %in% BB), AA %in% BB)

AA <- rep_len(c(TRUE, FALSE, TRUE), length(AA))
BB <- rep_len(c(FALSE, FALSE, TRUE, TRUE, TRUE), length(AA))
DD <- rep(FALSE, length(AA))
expect_equal(or3s(DD, AA != BB), AA != BB)
expect_equal(or3s(DD, AA == BB), AA == BB)
expect_equal(or3s(DD, AA >= BB), AA >= BB)
expect_equal(or3s(DD, AA <= BB), AA <= BB)
expect_equal(or3s(DD, AA > BB), AA > BB)
expect_equal(or3s(DD, AA < BB), AA < BB)

expect_equal(or3s(DD, AA != TRUE), AA != TRUE)
expect_equal(or3s(DD, AA == TRUE), AA == TRUE)
expect_equal(or3s(DD, AA >= TRUE), AA >= TRUE)
expect_equal(or3s(DD, AA <= TRUE), AA <= TRUE)
expect_equal(or3s(DD, AA > TRUE), AA > TRUE)
expect_equal(or3s(DD, AA < TRUE), AA < TRUE)

expect_equal(or3s(DD, AA != FALSE), AA != FALSE)
expect_equal(or3s(DD, AA == FALSE), AA == FALSE)
expect_equal(or3s(DD, AA >= FALSE), AA >= FALSE)
expect_equal(or3s(DD, AA <= FALSE), AA <= FALSE)
expect_equal(or3s(DD, AA > FALSE), AA > FALSE)
expect_equal(or3s(DD, AA < FALSE), AA < FALSE)

expect_equal(or3s(DD, AA %in% BB), AA %in% BB)
expect_equal(or3s(DD, AA %between% c(FALSE, TRUE)), !logical(length(DD)))
expect_equal(or3s(DD, AA %between% c(TRUE, FALSE)), logical(length(DD)))
rm(AA)
A <-
  c(-945028649L, -705457251L, 807204080L, 1708708214L, -885957403L,
    1862209884L, 1740762002L, 1546012157L, -1233491410L, 1256036667L,
    1009745233L, -815497191L, 137228285L, 2012907335L, -314954938L,
    -1234120580L, 2138414482L, 2089828880L, -1494606089L, 1669541061L,
    -1635694586L, 913293496L, 657757461L)
A <- rep_len(A, 1005)
expect_equal(or3s(A > 2^32), A > 2^32)
expect_equal(or3s(A < 2^32), A < 2^32)
expect_equal(or3s(A <= -2^32), A <= -2^32)
expect_equal(or3s(A <= 2^32), A <= 2^32)
expect_equal(or3s(A < 2^32), A < 2^32)
expect_equal(or3s(A < -2^32), A < -2^32)
expect_equal(or3s(A >= -2^32), A >= -2^32)
expect_equal(or3s(A > -2^32), A > -2^32)
expect_equal(or3s(A == 913293496, A != 913293496),
             bor3(A == 913293496, A != 913293496))
expect_equal(or3s(A == 913293496.5, A != 913293496),
             bor3(A == 913293496.5, A != 913293496))
expect_equal(or3s(A != 913293496.5, A != 913293496),
             bor3(A != 913293496.5, A != 913293496))
expect_equal(or3s(A >= Inf, A > 0.5), A > 0.5)
expect_equal(or3s(A >= Inf, A < 0.5), A < 0.5)
expect_equal(or3s(A >= Inf, A <= 0.5), A <= 0.5)
expect_equal(or3s(A >= Inf, A >= 0.5), A >= 0.5)


expect_equal(or3s(A == 913293496L), A == 913293496L)
expect_equal(or3s(A != 913293496L), A != 913293496L)
expect_equal(or3s(A >= 913293496L), A >= 913293496L)
expect_equal(or3s(A <= 913293496L), A <= 913293496L)
expect_equal(or3s(A > 913293496L), A > 913293496L)
expect_equal(or3s(A < 913293496L), A < 913293496L)
expect_equal(or3s(A == 913293496), A == 913293496)
expect_equal(or3s(A != 913293496), A != 913293496)
expect_equal(or3s(A >= 913293496), A >= 913293496)
expect_equal(or3s(A <= 913293496), A <= 913293496)
expect_equal(or3s(A > 913293496), A > 913293496)
expect_equal(or3s(A < 913293496), A < 913293496)

A <- as.double(A)
expect_equal(or3s(A > 2^32), A > 2^32)
expect_equal(or3s(A < 2^32), A < 2^32)
expect_equal(or3s(A == 913293496, A != 913293496),
             bor3(A == 913293496, A != 913293496))
expect_equal(or3s(A == 913293496.5, A != 913293496),
             bor3(A == 913293496.5, A != 913293496))
expect_equal(or3s(A != 913293496.5, A != 913293496),
             bor3(A != 913293496.5, A != 913293496))
expect_equal(or3s(A >= Inf, A > 0.5), A > 0.5)
expect_equal(or3s(A >= Inf, A < 0.5), A < 0.5)
expect_equal(or3s(A >= Inf, A <= 0.5), A <= 0.5)
expect_equal(or3s(A >= Inf, A >= 0.5), A >= 0.5)
expect_equal(or3s(A %between% c(1, 5)), A %between% c(1, 5))
expect_equal(or3s(A %(between)% c(1, 5)), A %(between)% c(1, 5))
expect_equal(or3s(A %(between)% c(1L, 5L)), A %(between)% c(1L, 5L))
expect_equal(or3s(A %]between[% c(1L, 5L)), A %]between[% c(1L, 5L))


expect_equal(or3s(B %in% A, D, E),
             bor3(B %in% A, D, E))
expect_equal(or3s(D, B %in% A, D, E),
             bor3(D, B %in% A, D, E))
expect_equal(or3s(D, E, B %in% A, D, E),
             bor3(D, E, B %in% A, D, E))
expect_equal(or3s(D, E, E, B %in% A, D, E),
             bor3(D, E, E, B %in% A, D, E))

# })

# test_that("%(between)% and %]between[%", {
xi <- 1:10e3 + 0L
yi <- rep_len(1:1e4, length(xi))
xd <- as.double(xi)
yd <- as.double(yi)
expect_equal(or3s(xi %(between)% c(11L, 110L),
                  yi %]between[% c(14L, 16L)),
             `|`(xi %(between)% c(11L, 110L),
                 yi %]between[% c(14L, 16L)))
expect_equal(or3s(xd %(between)% c(11.1, 110),
                  yd %]between[% c(14, 6)),
             `|`(xd %(between)% c(11.1, 110),
                 yd %]between[% c(14, 16)))



# test_that("'Make coverage explicit'", {
A <- c(FALSE, TRUE, TRUE)
expect_equal(or3s(A, A, !A), A | A | !A)
expect_equal(or3s(A, !A, !A), A | !A | !A)



# test_that("or3s works", {
# skip_if_not_installed("magrittr")
# skip_if_not_installed("hutils")
# skip_if_not_installed("data.table")
library(data.table)
library(hutils)
samp <- function(..., loud = FALSE) hutils::samp(..., loud = loud)

bor3 <- function(exprA, exprB, exprC) {
  if (missing(exprC) && missing(exprB)) {
    return(exprA)
  }
  if (missing(exprC)) {
    return(exprA | exprB)
  }
  if (missing(exprB)) {
    return(exprA | exprC)
  }
  (exprA | exprB) | exprC
}


library(magrittr)
#
gen_one_expr <- function(lhs = "", op = "", val = 1L, In_As_Between = FALSE) {
  if (op == "") {
    return(paste0("logi_", lhs))
  }
  if (op == "!") {
    return(paste0("!", paste0("logi_", lhs)))
  }
  switch(op,
         "%between%" = paste(lhs, "%between% c(-1L, 1L)"),
         "%in%" = paste(lhs, "%in%",
                        if (In_As_Between) {
                          "1:4"
                        } else {
                          "c(4L, -3L, 2L, 7L, 8L)"
                        }),
         paste(lhs, op, val))
}


DT <-
  CJ(Op1 = c("", "!", "!=", "==", ">=", "<=", ">", "<", "%in%", "%between%"),
     Op2 = c("", "!", "!=", "==", ">=", "<=", ">", "<", "%in%", "%between%"),
     Op3 = c("", "!", "!=", "==", ">=", "<=", ">", "<", "%in%", "%between%"),
     In_As_Between = c(FALSE, TRUE),
     e2 = c(FALSE, TRUE),
     e3 = c(FALSE, TRUE),
     val = c("0L", "1L", "9L")) %>%
  .[implies(In_As_Between, Op1 == "%in%" | Op2 == "%in%" | Op3 == "%in%")]

i2letters <- function(I) {
  out <- character(length(I))

  M <- matrix(0L, byrow = FALSE, ncol = ceiling(log(length(I), 26L)), nrow = length(I))

  M[, 1] <- rep_len(1:26, length(I))
  for (j in 2:ncol(M)) {
    eachj <- 26 ^ (j - 1L)
    if (eachj > length(I)) {
      break
    }
    M[, j] <- rep_len(rep(0:25, each = eachj), length(I))
  }

  M2 <- matrix("", ncol = ncol(M), nrow = nrow(M))

  for (j in 1:ncol(M2)) {
    M2[, ncol(M2) - j + 1] <- hutils::if_else(M[, j] > 0L, c("", letters)[M[, j] + 1L], "")
  }

  for (i in seq_along(out)) {
    out[i] <- paste0(M2[i, ], collapse = "")
  }
  out

}

DT[, ii := .I]
DT[, lhs1 := i2letters(ii)]
DT[lhs1 %in% c("for", "if", "else", "repeat", "in", "next", "break"), lhs1 := toupper(lhs1)]
DT[, lhs2 := samp(letters, size = .N)]
DT[, lhs3 := samp(letters, size = .N)]
#
# DT[, ii := .I]
DT[, expr_one   := gen_one_expr(lhs1, Op1, val = val, In_As_Between = In_As_Between), by = "ii"]
DT[, expr_two   := gen_one_expr(lhs2, Op2, val = val, In_As_Between = In_As_Between), by = "ii"]
DT[, expr_three := gen_one_expr(lhs3, Op3, val = val, In_As_Between = In_As_Between), by = "ii"]

DT[, the_expr := paste0("expect_equal(",
                        paste0("or3s(",
                               expr_one, ", ",
                               if_else(e2, expr_two, ""), ", ",
                               if_else(e3, expr_three, ""), ")"),
                        ",\n",
                        paste0("bor3(",
                               expr_one, ", ",
                               if_else(e2, expr_two, ""), ", ",
                               if_else(e3, expr_three, ""), ")"),
                        ")")]

generate_int <- function() rep_len(samp(-15:101, size = 99), 231)
generate_lgl <- function() rep_len(hutils::samp(c(TRUE, FALSE), size = 20, loud = FALSE), 231)

for (tmp__0 in unique(c(DT[["lhs1"]], DT[["lhs2"]], DT[["lhs3"]], letters))) {
  assign(paste0("logi_", tmp__0), value = generate_lgl())
  assign(tmp__0, value = generate_int())
}

# cat(DT[["the_expr"]], sep = "\n", file = "temp.R")



expect_equal(or3s(logi_a, , ),
             bor3(logi_a, , ))
expect_equal(or3s(logi_b, , ),
             bor3(logi_b, , ))
expect_equal(or3s(logi_c, , ),
             bor3(logi_c, , ))
expect_equal(or3s(logi_d, , logi_d),
             bor3(logi_d, , logi_d))
expect_equal(or3s(logi_e, , logi_p),
             bor3(logi_e, , logi_p))
expect_equal(or3s(logi_f, , logi_x),
             bor3(logi_f, , logi_x))
expect_equal(or3s(logi_g, logi_b, ),
             bor3(logi_g, logi_b, ))
expect_equal(or3s(logi_h, logi_g, ),
             bor3(logi_h, logi_g, ))
expect_equal(or3s(logi_i, logi_p, ),
             bor3(logi_i, logi_p, ))
expect_equal(or3s(logi_j, logi_s, logi_d),
             bor3(logi_j, logi_s, logi_d))
expect_equal(or3s(logi_k, logi_q, logi_x),
             bor3(logi_k, logi_q, logi_x))
expect_equal(or3s(logi_l, logi_a, logi_z),
             bor3(logi_l, logi_a, logi_z))
expect_equal(or3s(logi_m, , ),
             bor3(logi_m, , ))
expect_equal(or3s(logi_n, , ),
             bor3(logi_n, , ))
expect_equal(or3s(logi_o, , ),
             bor3(logi_o, , ))
expect_equal(or3s(logi_p, , !logi_g),
             bor3(logi_p, , !logi_g))
expect_equal(or3s(logi_q, , !logi_p),
             bor3(logi_q, , !logi_p))
expect_equal(or3s(logi_r, , !logi_m),
             bor3(logi_r, , !logi_m))
expect_equal(or3s(logi_s, logi_v, ),
             bor3(logi_s, logi_v, ))
expect_equal(or3s(logi_t, logi_z, ),
             bor3(logi_t, logi_z, ))
expect_equal(or3s(logi_u, logi_s, ),
             bor3(logi_u, logi_s, ))
expect_equal(or3s(logi_v, logi_y, !logi_v),
             bor3(logi_v, logi_y, !logi_v))
expect_equal(or3s(logi_w, logi_m, !logi_a),
             bor3(logi_w, logi_m, !logi_a))
expect_equal(or3s(logi_x, logi_f, !logi_w),
             bor3(logi_x, logi_f, !logi_w))
expect_equal(or3s(logi_y, , ),
             bor3(logi_y, , ))
expect_equal(or3s(logi_z, , ),
             bor3(logi_z, , ))
expect_equal(or3s(logi_aa, , ),
             bor3(logi_aa, , ))
expect_equal(or3s(logi_ab, , n != 0L),
             bor3(logi_ab, , n != 0L))
expect_equal(or3s(logi_ac, , f != 1L),
             bor3(logi_ac, , f != 1L))
expect_equal(or3s(logi_ad, , d != 9L),
             bor3(logi_ad, , d != 9L))
expect_equal(or3s(logi_ae, logi_x, ),
             bor3(logi_ae, logi_x, ))
expect_equal(or3s(logi_af, logi_x, ),
             bor3(logi_af, logi_x, ))
expect_equal(or3s(logi_ag, logi_v, ),
             bor3(logi_ag, logi_v, ))
expect_equal(or3s(logi_ah, logi_y, j != 0L),
             bor3(logi_ah, logi_y, j != 0L))
expect_equal(or3s(logi_ai, logi_i, j != 1L),
             bor3(logi_ai, logi_i, j != 1L))
expect_equal(or3s(logi_aj, logi_p, o != 9L),
             bor3(logi_aj, logi_p, o != 9L))
expect_equal(or3s(logi_ak, , ),
             bor3(logi_ak, , ))
expect_equal(or3s(logi_al, , ),
             bor3(logi_al, , ))
expect_equal(or3s(logi_am, , ),
             bor3(logi_am, , ))
expect_equal(or3s(logi_an, , j %between% c(-1L, 1L)),
             bor3(logi_an, , j %between% c(-1L, 1L)))
expect_equal(or3s(logi_ao, , d %between% c(-1L, 1L)),
             bor3(logi_ao, , d %between% c(-1L, 1L)))
expect_equal(or3s(logi_ap, , n %between% c(-1L, 1L)),
             bor3(logi_ap, , n %between% c(-1L, 1L)))
expect_equal(or3s(logi_aq, logi_t, ),
             bor3(logi_aq, logi_t, ))
expect_equal(or3s(logi_ar, logi_q, ),
             bor3(logi_ar, logi_q, ))
expect_equal(or3s(logi_as, logi_z, ),
             bor3(logi_as, logi_z, ))
expect_equal(or3s(logi_at, logi_h, l %between% c(-1L, 1L)),
             bor3(logi_at, logi_h, l %between% c(-1L, 1L)))
expect_equal(or3s(logi_au, logi_n, j %between% c(-1L, 1L)),
             bor3(logi_au, logi_n, j %between% c(-1L, 1L)))
expect_equal(or3s(logi_av, logi_e, v %between% c(-1L, 1L)),
             bor3(logi_av, logi_e, v %between% c(-1L, 1L)))
expect_equal(or3s(logi_aw, , ),
             bor3(logi_aw, , ))
expect_equal(or3s(logi_ax, , ),
             bor3(logi_ax, , ))
expect_equal(or3s(logi_ay, , ),
             bor3(logi_ay, , ))
expect_equal(or3s(logi_az, , e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_az, , e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ba, , o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ba, , o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_bb, , o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_bb, , o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_bc, logi_t, ),
             bor3(logi_bc, logi_t, ))
expect_equal(or3s(logi_bd, logi_n, ),
             bor3(logi_bd, logi_n, ))
expect_equal(or3s(logi_be, logi_s, ),
             bor3(logi_be, logi_s, ))
expect_equal(or3s(logi_bf, logi_x, s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_bf, logi_x, s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_bg, logi_e, h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_bg, logi_e, h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_bh, logi_n, a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_bh, logi_n, a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_bi, , ),
             bor3(logi_bi, , ))
expect_equal(or3s(logi_bj, , ),
             bor3(logi_bj, , ))
expect_equal(or3s(logi_bk, , ),
             bor3(logi_bk, , ))
expect_equal(or3s(logi_bl, , q %in% 1:4),
             bor3(logi_bl, , q %in% 1:4))
expect_equal(or3s(logi_bm, , u %in% 1:4),
             bor3(logi_bm, , u %in% 1:4))
expect_equal(or3s(logi_bn, , r %in% 1:4),
             bor3(logi_bn, , r %in% 1:4))
expect_equal(or3s(logi_bo, logi_q, ),
             bor3(logi_bo, logi_q, ))
expect_equal(or3s(logi_bp, logi_i, ),
             bor3(logi_bp, logi_i, ))
expect_equal(or3s(logi_bq, logi_p, ),
             bor3(logi_bq, logi_p, ))
expect_equal(or3s(logi_br, logi_l, b %in% 1:4),
             bor3(logi_br, logi_l, b %in% 1:4))
expect_equal(or3s(logi_bs, logi_b, f %in% 1:4),
             bor3(logi_bs, logi_b, f %in% 1:4))
expect_equal(or3s(logi_bt, logi_l, j %in% 1:4),
             bor3(logi_bt, logi_l, j %in% 1:4))
expect_equal(or3s(logi_bu, , ),
             bor3(logi_bu, , ))
expect_equal(or3s(logi_bv, , ),
             bor3(logi_bv, , ))
expect_equal(or3s(logi_bw, , ),
             bor3(logi_bw, , ))
expect_equal(or3s(logi_bx, , i < 0L),
             bor3(logi_bx, , i < 0L))
expect_equal(or3s(logi_by, , n < 1L),
             bor3(logi_by, , n < 1L))
expect_equal(or3s(logi_bz, , k < 9L),
             bor3(logi_bz, , k < 9L))
expect_equal(or3s(logi_ca, logi_n, ),
             bor3(logi_ca, logi_n, ))
expect_equal(or3s(logi_cb, logi_e, ),
             bor3(logi_cb, logi_e, ))
expect_equal(or3s(logi_cc, logi_x, ),
             bor3(logi_cc, logi_x, ))
expect_equal(or3s(logi_cd, logi_s, f < 0L),
             bor3(logi_cd, logi_s, f < 0L))
expect_equal(or3s(logi_ce, logi_q, h < 1L),
             bor3(logi_ce, logi_q, h < 1L))
expect_equal(or3s(logi_cf, logi_r, p < 9L),
             bor3(logi_cf, logi_r, p < 9L))
expect_equal(or3s(logi_cg, , ),
             bor3(logi_cg, , ))
expect_equal(or3s(logi_ch, , ),
             bor3(logi_ch, , ))
expect_equal(or3s(logi_ci, , ),
             bor3(logi_ci, , ))
expect_equal(or3s(logi_cj, , k <= 0L),
             bor3(logi_cj, , k <= 0L))
expect_equal(or3s(logi_ck, , a <= 1L),
             bor3(logi_ck, , a <= 1L))
expect_equal(or3s(logi_cl, , r <= 9L),
             bor3(logi_cl, , r <= 9L))
expect_equal(or3s(logi_cm, logi_s, ),
             bor3(logi_cm, logi_s, ))
expect_equal(or3s(logi_cn, logi_j, ),
             bor3(logi_cn, logi_j, ))
expect_equal(or3s(logi_co, logi_t, ),
             bor3(logi_co, logi_t, ))
expect_equal(or3s(logi_cp, logi_x, t <= 0L),
             bor3(logi_cp, logi_x, t <= 0L))
expect_equal(or3s(logi_cq, logi_c, v <= 1L),
             bor3(logi_cq, logi_c, v <= 1L))
expect_equal(or3s(logi_cr, logi_v, q <= 9L),
             bor3(logi_cr, logi_v, q <= 9L))
expect_equal(or3s(logi_cs, , ),
             bor3(logi_cs, , ))
expect_equal(or3s(logi_ct, , ),
             bor3(logi_ct, , ))
expect_equal(or3s(logi_cu, , ),
             bor3(logi_cu, , ))
expect_equal(or3s(logi_cv, , g == 0L),
             bor3(logi_cv, , g == 0L))
expect_equal(or3s(logi_cw, , a == 1L),
             bor3(logi_cw, , a == 1L))
expect_equal(or3s(logi_cx, , d == 9L),
             bor3(logi_cx, , d == 9L))
expect_equal(or3s(logi_cy, logi_i, ),
             bor3(logi_cy, logi_i, ))
expect_equal(or3s(logi_cz, logi_g, ),
             bor3(logi_cz, logi_g, ))
expect_equal(or3s(logi_da, logi_k, ),
             bor3(logi_da, logi_k, ))
expect_equal(or3s(logi_db, logi_t, b == 0L),
             bor3(logi_db, logi_t, b == 0L))
expect_equal(or3s(logi_dc, logi_o, l == 1L),
             bor3(logi_dc, logi_o, l == 1L))
expect_equal(or3s(logi_dd, logi_m, w == 9L),
             bor3(logi_dd, logi_m, w == 9L))
expect_equal(or3s(logi_de, , ),
             bor3(logi_de, , ))
expect_equal(or3s(logi_df, , ),
             bor3(logi_df, , ))
expect_equal(or3s(logi_dg, , ),
             bor3(logi_dg, , ))
expect_equal(or3s(logi_dh, , u > 0L),
             bor3(logi_dh, , u > 0L))
expect_equal(or3s(logi_di, , p > 1L),
             bor3(logi_di, , p > 1L))
expect_equal(or3s(logi_dj, , y > 9L),
             bor3(logi_dj, , y > 9L))
expect_equal(or3s(logi_dk, logi_x, ),
             bor3(logi_dk, logi_x, ))
expect_equal(or3s(logi_dl, logi_q, ),
             bor3(logi_dl, logi_q, ))
expect_equal(or3s(logi_dm, logi_k, ),
             bor3(logi_dm, logi_k, ))
expect_equal(or3s(logi_dn, logi_b, o > 0L),
             bor3(logi_dn, logi_b, o > 0L))
expect_equal(or3s(logi_do, logi_y, g > 1L),
             bor3(logi_do, logi_y, g > 1L))
expect_equal(or3s(logi_dp, logi_t, k > 9L),
             bor3(logi_dp, logi_t, k > 9L))
expect_equal(or3s(logi_dq, , ),
             bor3(logi_dq, , ))
expect_equal(or3s(logi_dr, , ),
             bor3(logi_dr, , ))
expect_equal(or3s(logi_ds, , ),
             bor3(logi_ds, , ))
expect_equal(or3s(logi_dt, , d >= 0L),
             bor3(logi_dt, , d >= 0L))
expect_equal(or3s(logi_du, , k >= 1L),
             bor3(logi_du, , k >= 1L))
expect_equal(or3s(logi_dv, , k >= 9L),
             bor3(logi_dv, , k >= 9L))
expect_equal(or3s(logi_dw, logi_g, ),
             bor3(logi_dw, logi_g, ))
expect_equal(or3s(logi_dx, logi_r, ),
             bor3(logi_dx, logi_r, ))
expect_equal(or3s(logi_dy, logi_s, ),
             bor3(logi_dy, logi_s, ))
expect_equal(or3s(logi_dz, logi_z, x >= 0L),
             bor3(logi_dz, logi_z, x >= 0L))
expect_equal(or3s(logi_ea, logi_g, t >= 1L),
             bor3(logi_ea, logi_g, t >= 1L))
expect_equal(or3s(logi_eb, logi_c, p >= 9L),
             bor3(logi_eb, logi_c, p >= 9L))
expect_equal(or3s(logi_ec, , ),
             bor3(logi_ec, , ))
expect_equal(or3s(logi_ed, , ),
             bor3(logi_ed, , ))
expect_equal(or3s(logi_ee, , ),
             bor3(logi_ee, , ))
expect_equal(or3s(logi_ef, , logi_n),
             bor3(logi_ef, , logi_n))
expect_equal(or3s(logi_eg, , logi_e),
             bor3(logi_eg, , logi_e))
expect_equal(or3s(logi_eh, , logi_k),
             bor3(logi_eh, , logi_k))
expect_equal(or3s(logi_ei, !logi_s, ),
             bor3(logi_ei, !logi_s, ))
expect_equal(or3s(logi_ej, !logi_j, ),
             bor3(logi_ej, !logi_j, ))
expect_equal(or3s(logi_ek, !logi_m, ),
             bor3(logi_ek, !logi_m, ))
expect_equal(or3s(logi_el, !logi_p, logi_k),
             bor3(logi_el, !logi_p, logi_k))
expect_equal(or3s(logi_em, !logi_e, logi_o),
             bor3(logi_em, !logi_e, logi_o))
expect_equal(or3s(logi_en, !logi_x, logi_a),
             bor3(logi_en, !logi_x, logi_a))
expect_equal(or3s(logi_eo, , ),
             bor3(logi_eo, , ))
expect_equal(or3s(logi_ep, , ),
             bor3(logi_ep, , ))
expect_equal(or3s(logi_eq, , ),
             bor3(logi_eq, , ))
expect_equal(or3s(logi_er, , !logi_n),
             bor3(logi_er, , !logi_n))
expect_equal(or3s(logi_es, , !logi_d),
             bor3(logi_es, , !logi_d))
expect_equal(or3s(logi_et, , !logi_y),
             bor3(logi_et, , !logi_y))
expect_equal(or3s(logi_eu, !logi_f, ),
             bor3(logi_eu, !logi_f, ))
expect_equal(or3s(logi_ev, !logi_l, ),
             bor3(logi_ev, !logi_l, ))
expect_equal(or3s(logi_ew, !logi_r, ),
             bor3(logi_ew, !logi_r, ))
expect_equal(or3s(logi_ex, !logi_d, !logi_m),
             bor3(logi_ex, !logi_d, !logi_m))
expect_equal(or3s(logi_ey, !logi_o, !logi_m),
             bor3(logi_ey, !logi_o, !logi_m))
expect_equal(or3s(logi_ez, !logi_f, !logi_j),
             bor3(logi_ez, !logi_f, !logi_j))
expect_equal(or3s(logi_fa, , ),
             bor3(logi_fa, , ))
expect_equal(or3s(logi_fb, , ),
             bor3(logi_fb, , ))
expect_equal(or3s(logi_fc, , ),
             bor3(logi_fc, , ))
expect_equal(or3s(logi_fd, , y != 0L),
             bor3(logi_fd, , y != 0L))
expect_equal(or3s(logi_fe, , h != 1L),
             bor3(logi_fe, , h != 1L))
expect_equal(or3s(logi_ff, , t != 9L),
             bor3(logi_ff, , t != 9L))
expect_equal(or3s(logi_fg, !logi_m, ),
             bor3(logi_fg, !logi_m, ))
expect_equal(or3s(logi_fh, !logi_z, ),
             bor3(logi_fh, !logi_z, ))
expect_equal(or3s(logi_fi, !logi_s, ),
             bor3(logi_fi, !logi_s, ))
expect_equal(or3s(logi_fj, !logi_m, t != 0L),
             bor3(logi_fj, !logi_m, t != 0L))
expect_equal(or3s(logi_fk, !logi_t, l != 1L),
             bor3(logi_fk, !logi_t, l != 1L))
expect_equal(or3s(logi_fl, !logi_l, l != 9L),
             bor3(logi_fl, !logi_l, l != 9L))
expect_equal(or3s(logi_fm, , ),
             bor3(logi_fm, , ))
expect_equal(or3s(logi_fn, , ),
             bor3(logi_fn, , ))
expect_equal(or3s(logi_fo, , ),
             bor3(logi_fo, , ))
expect_equal(or3s(logi_fp, , n %between% c(-1L, 1L)),
             bor3(logi_fp, , n %between% c(-1L, 1L)))
expect_equal(or3s(logi_fq, , x %between% c(-1L, 1L)),
             bor3(logi_fq, , x %between% c(-1L, 1L)))
expect_equal(or3s(logi_fr, , r %between% c(-1L, 1L)),
             bor3(logi_fr, , r %between% c(-1L, 1L)))
expect_equal(or3s(logi_fs, !logi_l, ),
             bor3(logi_fs, !logi_l, ))
expect_equal(or3s(logi_ft, !logi_n, ),
             bor3(logi_ft, !logi_n, ))
expect_equal(or3s(logi_fu, !logi_p, ),
             bor3(logi_fu, !logi_p, ))
expect_equal(or3s(logi_fv, !logi_f, s %between% c(-1L, 1L)),
             bor3(logi_fv, !logi_f, s %between% c(-1L, 1L)))
expect_equal(or3s(logi_fw, !logi_c, t %between% c(-1L, 1L)),
             bor3(logi_fw, !logi_c, t %between% c(-1L, 1L)))
expect_equal(or3s(logi_fx, !logi_x, a %between% c(-1L, 1L)),
             bor3(logi_fx, !logi_x, a %between% c(-1L, 1L)))
expect_equal(or3s(logi_fy, , ),
             bor3(logi_fy, , ))
expect_equal(or3s(logi_fz, , ),
             bor3(logi_fz, , ))
expect_equal(or3s(logi_ga, , ),
             bor3(logi_ga, , ))
expect_equal(or3s(logi_gb, , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_gb, , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_gc, , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_gc, , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_gd, , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_gd, , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ge, !logi_z, ),
             bor3(logi_ge, !logi_z, ))
expect_equal(or3s(logi_gf, !logi_r, ),
             bor3(logi_gf, !logi_r, ))
expect_equal(or3s(logi_gg, !logi_b, ),
             bor3(logi_gg, !logi_b, ))
expect_equal(or3s(logi_gh, !logi_v, z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_gh, !logi_v, z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_gi, !logi_r, z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_gi, !logi_r, z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_gj, !logi_y, q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_gj, !logi_y, q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_gk, , ),
             bor3(logi_gk, , ))
expect_equal(or3s(logi_gl, , ),
             bor3(logi_gl, , ))
expect_equal(or3s(logi_gm, , ),
             bor3(logi_gm, , ))
expect_equal(or3s(logi_gn, , h %in% 1:4),
             bor3(logi_gn, , h %in% 1:4))
expect_equal(or3s(logi_go, , a %in% 1:4),
             bor3(logi_go, , a %in% 1:4))
expect_equal(or3s(logi_gp, , q %in% 1:4),
             bor3(logi_gp, , q %in% 1:4))
expect_equal(or3s(logi_gq, !logi_q, ),
             bor3(logi_gq, !logi_q, ))
expect_equal(or3s(logi_gr, !logi_x, ),
             bor3(logi_gr, !logi_x, ))
expect_equal(or3s(logi_gs, !logi_b, ),
             bor3(logi_gs, !logi_b, ))
expect_equal(or3s(logi_gt, !logi_z, a %in% 1:4),
             bor3(logi_gt, !logi_z, a %in% 1:4))
expect_equal(or3s(logi_gu, !logi_k, z %in% 1:4),
             bor3(logi_gu, !logi_k, z %in% 1:4))
expect_equal(or3s(logi_gv, !logi_e, w %in% 1:4),
             bor3(logi_gv, !logi_e, w %in% 1:4))
expect_equal(or3s(logi_gw, , ),
             bor3(logi_gw, , ))
expect_equal(or3s(logi_gx, , ),
             bor3(logi_gx, , ))
expect_equal(or3s(logi_gy, , ),
             bor3(logi_gy, , ))
expect_equal(or3s(logi_gz, , i < 0L),
             bor3(logi_gz, , i < 0L))
expect_equal(or3s(logi_ha, , x < 1L),
             bor3(logi_ha, , x < 1L))
expect_equal(or3s(logi_hb, , j < 9L),
             bor3(logi_hb, , j < 9L))
expect_equal(or3s(logi_hc, !logi_r, ),
             bor3(logi_hc, !logi_r, ))
expect_equal(or3s(logi_hd, !logi_v, ),
             bor3(logi_hd, !logi_v, ))
expect_equal(or3s(logi_he, !logi_w, ),
             bor3(logi_he, !logi_w, ))
expect_equal(or3s(logi_hf, !logi_t, t < 0L),
             bor3(logi_hf, !logi_t, t < 0L))
expect_equal(or3s(logi_hg, !logi_t, c < 1L),
             bor3(logi_hg, !logi_t, c < 1L))
expect_equal(or3s(logi_hh, !logi_f, u < 9L),
             bor3(logi_hh, !logi_f, u < 9L))
expect_equal(or3s(logi_hi, , ),
             bor3(logi_hi, , ))
expect_equal(or3s(logi_hj, , ),
             bor3(logi_hj, , ))
expect_equal(or3s(logi_hk, , ),
             bor3(logi_hk, , ))
expect_equal(or3s(logi_hl, , y <= 0L),
             bor3(logi_hl, , y <= 0L))
expect_equal(or3s(logi_hm, , m <= 1L),
             bor3(logi_hm, , m <= 1L))
expect_equal(or3s(logi_hn, , e <= 9L),
             bor3(logi_hn, , e <= 9L))
expect_equal(or3s(logi_ho, !logi_r, ),
             bor3(logi_ho, !logi_r, ))
expect_equal(or3s(logi_hp, !logi_g, ),
             bor3(logi_hp, !logi_g, ))
expect_equal(or3s(logi_hq, !logi_t, ),
             bor3(logi_hq, !logi_t, ))
expect_equal(or3s(logi_hr, !logi_e, d <= 0L),
             bor3(logi_hr, !logi_e, d <= 0L))
expect_equal(or3s(logi_hs, !logi_y, d <= 1L),
             bor3(logi_hs, !logi_y, d <= 1L))
expect_equal(or3s(logi_ht, !logi_p, m <= 9L),
             bor3(logi_ht, !logi_p, m <= 9L))
expect_equal(or3s(logi_hu, , ),
             bor3(logi_hu, , ))
expect_equal(or3s(logi_hv, , ),
             bor3(logi_hv, , ))
expect_equal(or3s(logi_hw, , ),
             bor3(logi_hw, , ))
expect_equal(or3s(logi_hx, , m == 0L),
             bor3(logi_hx, , m == 0L))
expect_equal(or3s(logi_hy, , x == 1L),
             bor3(logi_hy, , x == 1L))
expect_equal(or3s(logi_hz, , n == 9L),
             bor3(logi_hz, , n == 9L))
expect_equal(or3s(logi_ia, !logi_y, ),
             bor3(logi_ia, !logi_y, ))
expect_equal(or3s(logi_ib, !logi_h, ),
             bor3(logi_ib, !logi_h, ))
expect_equal(or3s(logi_ic, !logi_c, ),
             bor3(logi_ic, !logi_c, ))
expect_equal(or3s(logi_id, !logi_t, f == 0L),
             bor3(logi_id, !logi_t, f == 0L))
expect_equal(or3s(logi_ie, !logi_e, w == 1L),
             bor3(logi_ie, !logi_e, w == 1L))
expect_equal(or3s(logi_IF, !logi_y, p == 9L),
             bor3(logi_IF, !logi_y, p == 9L))
expect_equal(or3s(logi_ig, , ),
             bor3(logi_ig, , ))
expect_equal(or3s(logi_ih, , ),
             bor3(logi_ih, , ))
expect_equal(or3s(logi_ii, , ),
             bor3(logi_ii, , ))
expect_equal(or3s(logi_ij, , z > 0L),
             bor3(logi_ij, , z > 0L))
expect_equal(or3s(logi_ik, , r > 1L),
             bor3(logi_ik, , r > 1L))
expect_equal(or3s(logi_il, , e > 9L),
             bor3(logi_il, , e > 9L))
expect_equal(or3s(logi_im, !logi_d, ),
             bor3(logi_im, !logi_d, ))
expect_equal(or3s(logi_IN, !logi_o, ),
             bor3(logi_IN, !logi_o, ))
expect_equal(or3s(logi_io, !logi_o, ),
             bor3(logi_io, !logi_o, ))
expect_equal(or3s(logi_ip, !logi_f, m > 0L),
             bor3(logi_ip, !logi_f, m > 0L))
expect_equal(or3s(logi_iq, !logi_i, y > 1L),
             bor3(logi_iq, !logi_i, y > 1L))
expect_equal(or3s(logi_ir, !logi_w, s > 9L),
             bor3(logi_ir, !logi_w, s > 9L))
expect_equal(or3s(logi_is, , ),
             bor3(logi_is, , ))
expect_equal(or3s(logi_it, , ),
             bor3(logi_it, , ))
expect_equal(or3s(logi_iu, , ),
             bor3(logi_iu, , ))
expect_equal(or3s(logi_iv, , y >= 0L),
             bor3(logi_iv, , y >= 0L))
expect_equal(or3s(logi_iw, , d >= 1L),
             bor3(logi_iw, , d >= 1L))
expect_equal(or3s(logi_ix, , w >= 9L),
             bor3(logi_ix, , w >= 9L))
expect_equal(or3s(logi_iy, !logi_i, ),
             bor3(logi_iy, !logi_i, ))
expect_equal(or3s(logi_iz, !logi_f, ),
             bor3(logi_iz, !logi_f, ))
expect_equal(or3s(logi_ja, !logi_x, ),
             bor3(logi_ja, !logi_x, ))
expect_equal(or3s(logi_jb, !logi_j, e >= 0L),
             bor3(logi_jb, !logi_j, e >= 0L))
expect_equal(or3s(logi_jc, !logi_v, j >= 1L),
             bor3(logi_jc, !logi_v, j >= 1L))
expect_equal(or3s(logi_jd, !logi_q, x >= 9L),
             bor3(logi_jd, !logi_q, x >= 9L))
expect_equal(or3s(logi_je, , ),
             bor3(logi_je, , ))
expect_equal(or3s(logi_jf, , ),
             bor3(logi_jf, , ))
expect_equal(or3s(logi_jg, , ),
             bor3(logi_jg, , ))
expect_equal(or3s(logi_jh, , logi_q),
             bor3(logi_jh, , logi_q))
expect_equal(or3s(logi_ji, , logi_x),
             bor3(logi_ji, , logi_x))
expect_equal(or3s(logi_jj, , logi_z),
             bor3(logi_jj, , logi_z))
expect_equal(or3s(logi_jk, j != 0L, ),
             bor3(logi_jk, j != 0L, ))
expect_equal(or3s(logi_jl, m != 1L, ),
             bor3(logi_jl, m != 1L, ))
expect_equal(or3s(logi_jm, o != 9L, ),
             bor3(logi_jm, o != 9L, ))
expect_equal(or3s(logi_jn, f != 0L, logi_u),
             bor3(logi_jn, f != 0L, logi_u))
expect_equal(or3s(logi_jo, f != 1L, logi_q),
             bor3(logi_jo, f != 1L, logi_q))
expect_equal(or3s(logi_jp, w != 9L, logi_x),
             bor3(logi_jp, w != 9L, logi_x))
expect_equal(or3s(logi_jq, , ),
             bor3(logi_jq, , ))
expect_equal(or3s(logi_jr, , ),
             bor3(logi_jr, , ))
expect_equal(or3s(logi_js, , ),
             bor3(logi_js, , ))
expect_equal(or3s(logi_jt, , !logi_s),
             bor3(logi_jt, , !logi_s))
expect_equal(or3s(logi_ju, , !logi_n),
             bor3(logi_ju, , !logi_n))
expect_equal(or3s(logi_jv, , !logi_z),
             bor3(logi_jv, , !logi_z))
expect_equal(or3s(logi_jw, c != 0L, ),
             bor3(logi_jw, c != 0L, ))
expect_equal(or3s(logi_jx, h != 1L, ),
             bor3(logi_jx, h != 1L, ))
expect_equal(or3s(logi_jy, c != 9L, ),
             bor3(logi_jy, c != 9L, ))
expect_equal(or3s(logi_jz, d != 0L, !logi_a),
             bor3(logi_jz, d != 0L, !logi_a))
expect_equal(or3s(logi_ka, x != 1L, !logi_i),
             bor3(logi_ka, x != 1L, !logi_i))
expect_equal(or3s(logi_kb, s != 9L, !logi_r),
             bor3(logi_kb, s != 9L, !logi_r))
expect_equal(or3s(logi_kc, , ),
             bor3(logi_kc, , ))
expect_equal(or3s(logi_kd, , ),
             bor3(logi_kd, , ))
expect_equal(or3s(logi_ke, , ),
             bor3(logi_ke, , ))
expect_equal(or3s(logi_kf, , r != 0L),
             bor3(logi_kf, , r != 0L))
expect_equal(or3s(logi_kg, , z != 1L),
             bor3(logi_kg, , z != 1L))
expect_equal(or3s(logi_kh, , h != 9L),
             bor3(logi_kh, , h != 9L))
expect_equal(or3s(logi_ki, a != 0L, ),
             bor3(logi_ki, a != 0L, ))
expect_equal(or3s(logi_kj, u != 1L, ),
             bor3(logi_kj, u != 1L, ))
expect_equal(or3s(logi_kk, j != 9L, ),
             bor3(logi_kk, j != 9L, ))
expect_equal(or3s(logi_kl, m != 0L, g != 0L),
             bor3(logi_kl, m != 0L, g != 0L))
expect_equal(or3s(logi_km, m != 1L, l != 1L),
             bor3(logi_km, m != 1L, l != 1L))
expect_equal(or3s(logi_kn, b != 9L, u != 9L),
             bor3(logi_kn, b != 9L, u != 9L))
expect_equal(or3s(logi_ko, , ),
             bor3(logi_ko, , ))
expect_equal(or3s(logi_kp, , ),
             bor3(logi_kp, , ))
expect_equal(or3s(logi_kq, , ),
             bor3(logi_kq, , ))
expect_equal(or3s(logi_kr, , i %between% c(-1L, 1L)),
             bor3(logi_kr, , i %between% c(-1L, 1L)))
expect_equal(or3s(logi_ks, , o %between% c(-1L, 1L)),
             bor3(logi_ks, , o %between% c(-1L, 1L)))
expect_equal(or3s(logi_kt, , j %between% c(-1L, 1L)),
             bor3(logi_kt, , j %between% c(-1L, 1L)))
expect_equal(or3s(logi_ku, g != 0L, ),
             bor3(logi_ku, g != 0L, ))
expect_equal(or3s(logi_kv, m != 1L, ),
             bor3(logi_kv, m != 1L, ))
expect_equal(or3s(logi_kw, b != 9L, ),
             bor3(logi_kw, b != 9L, ))
expect_equal(or3s(logi_kx, f != 0L, e %between% c(-1L, 1L)),
             bor3(logi_kx, f != 0L, e %between% c(-1L, 1L)))
expect_equal(or3s(logi_ky, w != 1L, j %between% c(-1L, 1L)),
             bor3(logi_ky, w != 1L, j %between% c(-1L, 1L)))
expect_equal(or3s(logi_kz, e != 9L, w %between% c(-1L, 1L)),
             bor3(logi_kz, e != 9L, w %between% c(-1L, 1L)))
expect_equal(or3s(logi_la, , ),
             bor3(logi_la, , ))
expect_equal(or3s(logi_lb, , ),
             bor3(logi_lb, , ))
expect_equal(or3s(logi_lc, , ),
             bor3(logi_lc, , ))
expect_equal(or3s(logi_ld, , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ld, , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_le, , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_le, , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_lf, , k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_lf, , k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_lg, m != 0L, ),
             bor3(logi_lg, m != 0L, ))
expect_equal(or3s(logi_lh, c != 1L, ),
             bor3(logi_lh, c != 1L, ))
expect_equal(or3s(logi_li, j != 9L, ),
             bor3(logi_li, j != 9L, ))
expect_equal(or3s(logi_lj, v != 0L, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_lj, v != 0L, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_lk, p != 1L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_lk, p != 1L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ll, a != 9L, e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ll, a != 9L, e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_lm, , ),
             bor3(logi_lm, , ))
expect_equal(or3s(logi_ln, , ),
             bor3(logi_ln, , ))
expect_equal(or3s(logi_lo, , ),
             bor3(logi_lo, , ))
expect_equal(or3s(logi_lp, , h %in% 1:4),
             bor3(logi_lp, , h %in% 1:4))
expect_equal(or3s(logi_lq, , n %in% 1:4),
             bor3(logi_lq, , n %in% 1:4))
expect_equal(or3s(logi_lr, , t %in% 1:4),
             bor3(logi_lr, , t %in% 1:4))
expect_equal(or3s(logi_ls, k != 0L, ),
             bor3(logi_ls, k != 0L, ))
expect_equal(or3s(logi_lt, w != 1L, ),
             bor3(logi_lt, w != 1L, ))
expect_equal(or3s(logi_lu, j != 9L, ),
             bor3(logi_lu, j != 9L, ))
expect_equal(or3s(logi_lv, s != 0L, o %in% 1:4),
             bor3(logi_lv, s != 0L, o %in% 1:4))
expect_equal(or3s(logi_lw, b != 1L, u %in% 1:4),
             bor3(logi_lw, b != 1L, u %in% 1:4))
expect_equal(or3s(logi_lx, n != 9L, x %in% 1:4),
             bor3(logi_lx, n != 9L, x %in% 1:4))
expect_equal(or3s(logi_ly, , ),
             bor3(logi_ly, , ))
expect_equal(or3s(logi_lz, , ),
             bor3(logi_lz, , ))
expect_equal(or3s(logi_ma, , ),
             bor3(logi_ma, , ))
expect_equal(or3s(logi_mb, , i < 0L),
             bor3(logi_mb, , i < 0L))
expect_equal(or3s(logi_mc, , d < 1L),
             bor3(logi_mc, , d < 1L))
expect_equal(or3s(logi_md, , r < 9L),
             bor3(logi_md, , r < 9L))
expect_equal(or3s(logi_me, e != 0L, ),
             bor3(logi_me, e != 0L, ))
expect_equal(or3s(logi_mf, d != 1L, ),
             bor3(logi_mf, d != 1L, ))
expect_equal(or3s(logi_mg, k != 9L, ),
             bor3(logi_mg, k != 9L, ))
expect_equal(or3s(logi_mh, n != 0L, c < 0L),
             bor3(logi_mh, n != 0L, c < 0L))
expect_equal(or3s(logi_mi, u != 1L, t < 1L),
             bor3(logi_mi, u != 1L, t < 1L))
expect_equal(or3s(logi_mj, x != 9L, r < 9L),
             bor3(logi_mj, x != 9L, r < 9L))
expect_equal(or3s(logi_mk, , ),
             bor3(logi_mk, , ))
expect_equal(or3s(logi_ml, , ),
             bor3(logi_ml, , ))
expect_equal(or3s(logi_mm, , ),
             bor3(logi_mm, , ))
expect_equal(or3s(logi_mn, , h <= 0L),
             bor3(logi_mn, , h <= 0L))
expect_equal(or3s(logi_mo, , i <= 1L),
             bor3(logi_mo, , i <= 1L))
expect_equal(or3s(logi_mp, , n <= 9L),
             bor3(logi_mp, , n <= 9L))
expect_equal(or3s(logi_mq, q != 0L, ),
             bor3(logi_mq, q != 0L, ))
expect_equal(or3s(logi_mr, l != 1L, ),
             bor3(logi_mr, l != 1L, ))
expect_equal(or3s(logi_ms, a != 9L, ),
             bor3(logi_ms, a != 9L, ))
expect_equal(or3s(logi_mt, o != 0L, a <= 0L),
             bor3(logi_mt, o != 0L, a <= 0L))
expect_equal(or3s(logi_mu, z != 1L, e <= 1L),
             bor3(logi_mu, z != 1L, e <= 1L))
expect_equal(or3s(logi_mv, k != 9L, l <= 9L),
             bor3(logi_mv, k != 9L, l <= 9L))
expect_equal(or3s(logi_mw, , ),
             bor3(logi_mw, , ))
expect_equal(or3s(logi_mx, , ),
             bor3(logi_mx, , ))
expect_equal(or3s(logi_my, , ),
             bor3(logi_my, , ))
expect_equal(or3s(logi_mz, , i == 0L),
             bor3(logi_mz, , i == 0L))
expect_equal(or3s(logi_na, , g == 1L),
             bor3(logi_na, , g == 1L))
expect_equal(or3s(logi_nb, , b == 9L),
             bor3(logi_nb, , b == 9L))
expect_equal(or3s(logi_nc, t != 0L, ),
             bor3(logi_nc, t != 0L, ))
expect_equal(or3s(logi_nd, l != 1L, ),
             bor3(logi_nd, l != 1L, ))
expect_equal(or3s(logi_ne, b != 9L, ),
             bor3(logi_ne, b != 9L, ))
expect_equal(or3s(logi_nf, j != 0L, s == 0L),
             bor3(logi_nf, j != 0L, s == 0L))
expect_equal(or3s(logi_ng, z != 1L, x == 1L),
             bor3(logi_ng, z != 1L, x == 1L))
expect_equal(or3s(logi_nh, f != 9L, k == 9L),
             bor3(logi_nh, f != 9L, k == 9L))
expect_equal(or3s(logi_ni, , ),
             bor3(logi_ni, , ))
expect_equal(or3s(logi_nj, , ),
             bor3(logi_nj, , ))
expect_equal(or3s(logi_nk, , ),
             bor3(logi_nk, , ))
expect_equal(or3s(logi_nl, , x > 0L),
             bor3(logi_nl, , x > 0L))
expect_equal(or3s(logi_nm, , b > 1L),
             bor3(logi_nm, , b > 1L))
expect_equal(or3s(logi_nn, , r > 9L),
             bor3(logi_nn, , r > 9L))
expect_equal(or3s(logi_no, f != 0L, ),
             bor3(logi_no, f != 0L, ))
expect_equal(or3s(logi_np, z != 1L, ),
             bor3(logi_np, z != 1L, ))
expect_equal(or3s(logi_nq, m != 9L, ),
             bor3(logi_nq, m != 9L, ))
expect_equal(or3s(logi_nr, b != 0L, o > 0L),
             bor3(logi_nr, b != 0L, o > 0L))
expect_equal(or3s(logi_ns, q != 1L, d > 1L),
             bor3(logi_ns, q != 1L, d > 1L))
expect_equal(or3s(logi_nt, k != 9L, z > 9L),
             bor3(logi_nt, k != 9L, z > 9L))
expect_equal(or3s(logi_nu, , ),
             bor3(logi_nu, , ))
expect_equal(or3s(logi_nv, , ),
             bor3(logi_nv, , ))
expect_equal(or3s(logi_nw, , ),
             bor3(logi_nw, , ))
expect_equal(or3s(logi_nx, , u >= 0L),
             bor3(logi_nx, , u >= 0L))
expect_equal(or3s(logi_ny, , j >= 1L),
             bor3(logi_ny, , j >= 1L))
expect_equal(or3s(logi_nz, , j >= 9L),
             bor3(logi_nz, , j >= 9L))
expect_equal(or3s(logi_oa, t != 0L, ),
             bor3(logi_oa, t != 0L, ))
expect_equal(or3s(logi_ob, n != 1L, ),
             bor3(logi_ob, n != 1L, ))
expect_equal(or3s(logi_oc, b != 9L, ),
             bor3(logi_oc, b != 9L, ))
expect_equal(or3s(logi_od, r != 0L, c >= 0L),
             bor3(logi_od, r != 0L, c >= 0L))
expect_equal(or3s(logi_oe, c != 1L, m >= 1L),
             bor3(logi_oe, c != 1L, m >= 1L))
expect_equal(or3s(logi_of, c != 9L, h >= 9L),
             bor3(logi_of, c != 9L, h >= 9L))
expect_equal(or3s(logi_og, , ),
             bor3(logi_og, , ))
expect_equal(or3s(logi_oh, , ),
             bor3(logi_oh, , ))
expect_equal(or3s(logi_oi, , ),
             bor3(logi_oi, , ))
expect_equal(or3s(logi_oj, , logi_q),
             bor3(logi_oj, , logi_q))
expect_equal(or3s(logi_ok, , logi_m),
             bor3(logi_ok, , logi_m))
expect_equal(or3s(logi_ol, , logi_p),
             bor3(logi_ol, , logi_p))
expect_equal(or3s(logi_om, w %between% c(-1L, 1L), ),
             bor3(logi_om, w %between% c(-1L, 1L), ))
expect_equal(or3s(logi_on, k %between% c(-1L, 1L), ),
             bor3(logi_on, k %between% c(-1L, 1L), ))
expect_equal(or3s(logi_oo, w %between% c(-1L, 1L), ),
             bor3(logi_oo, w %between% c(-1L, 1L), ))
expect_equal(or3s(logi_op, g %between% c(-1L, 1L), logi_j),
             bor3(logi_op, g %between% c(-1L, 1L), logi_j))
expect_equal(or3s(logi_oq, z %between% c(-1L, 1L), logi_z),
             bor3(logi_oq, z %between% c(-1L, 1L), logi_z))
expect_equal(or3s(logi_or, x %between% c(-1L, 1L), logi_d),
             bor3(logi_or, x %between% c(-1L, 1L), logi_d))
expect_equal(or3s(logi_os, , ),
             bor3(logi_os, , ))
expect_equal(or3s(logi_ot, , ),
             bor3(logi_ot, , ))
expect_equal(or3s(logi_ou, , ),
             bor3(logi_ou, , ))
expect_equal(or3s(logi_ov, , !logi_i),
             bor3(logi_ov, , !logi_i))
expect_equal(or3s(logi_ow, , !logi_b),
             bor3(logi_ow, , !logi_b))
expect_equal(or3s(logi_ox, , !logi_b),
             bor3(logi_ox, , !logi_b))
expect_equal(or3s(logi_oy, r %between% c(-1L, 1L), ),
             bor3(logi_oy, r %between% c(-1L, 1L), ))
expect_equal(or3s(logi_oz, f %between% c(-1L, 1L), ),
             bor3(logi_oz, f %between% c(-1L, 1L), ))
expect_equal(or3s(logi_pa, o %between% c(-1L, 1L), ),
             bor3(logi_pa, o %between% c(-1L, 1L), ))
expect_equal(or3s(logi_pb, l %between% c(-1L, 1L), !logi_s),
             bor3(logi_pb, l %between% c(-1L, 1L), !logi_s))
expect_equal(or3s(logi_pc, g %between% c(-1L, 1L), !logi_w),
             bor3(logi_pc, g %between% c(-1L, 1L), !logi_w))
expect_equal(or3s(logi_pd, s %between% c(-1L, 1L), !logi_y),
             bor3(logi_pd, s %between% c(-1L, 1L), !logi_y))
expect_equal(or3s(logi_pe, , ),
             bor3(logi_pe, , ))
expect_equal(or3s(logi_pf, , ),
             bor3(logi_pf, , ))
expect_equal(or3s(logi_pg, , ),
             bor3(logi_pg, , ))
expect_equal(or3s(logi_ph, , m != 0L),
             bor3(logi_ph, , m != 0L))
expect_equal(or3s(logi_pi, , u != 1L),
             bor3(logi_pi, , u != 1L))
expect_equal(or3s(logi_pj, , d != 9L),
             bor3(logi_pj, , d != 9L))
expect_equal(or3s(logi_pk, p %between% c(-1L, 1L), ),
             bor3(logi_pk, p %between% c(-1L, 1L), ))
expect_equal(or3s(logi_pl, n %between% c(-1L, 1L), ),
             bor3(logi_pl, n %between% c(-1L, 1L), ))
expect_equal(or3s(logi_pm, u %between% c(-1L, 1L), ),
             bor3(logi_pm, u %between% c(-1L, 1L), ))
expect_equal(or3s(logi_pn, h %between% c(-1L, 1L), v != 0L),
             bor3(logi_pn, h %between% c(-1L, 1L), v != 0L))
expect_equal(or3s(logi_po, y %between% c(-1L, 1L), y != 1L),
             bor3(logi_po, y %between% c(-1L, 1L), y != 1L))
expect_equal(or3s(logi_pp, t %between% c(-1L, 1L), e != 9L),
             bor3(logi_pp, t %between% c(-1L, 1L), e != 9L))
expect_equal(or3s(logi_pq, , ),
             bor3(logi_pq, , ))
expect_equal(or3s(logi_pr, , ),
             bor3(logi_pr, , ))
expect_equal(or3s(logi_ps, , ),
             bor3(logi_ps, , ))
expect_equal(or3s(logi_pt, , w %between% c(-1L, 1L)),
             bor3(logi_pt, , w %between% c(-1L, 1L)))
expect_equal(or3s(logi_pu, , o %between% c(-1L, 1L)),
             bor3(logi_pu, , o %between% c(-1L, 1L)))
expect_equal(or3s(logi_pv, , n %between% c(-1L, 1L)),
             bor3(logi_pv, , n %between% c(-1L, 1L)))
expect_equal(or3s(logi_pw, f %between% c(-1L, 1L), ),
             bor3(logi_pw, f %between% c(-1L, 1L), ))
expect_equal(or3s(logi_px, t %between% c(-1L, 1L), ),
             bor3(logi_px, t %between% c(-1L, 1L), ))
expect_equal(or3s(logi_py, z %between% c(-1L, 1L), ),
             bor3(logi_py, z %between% c(-1L, 1L), ))
expect_equal(or3s(logi_pz, d %between% c(-1L, 1L), p %between% c(-1L, 1L)),
             bor3(logi_pz, d %between% c(-1L, 1L), p %between% c(-1L, 1L)))
expect_equal(or3s(logi_qa, t %between% c(-1L, 1L), j %between% c(-1L, 1L)),
             bor3(logi_qa, t %between% c(-1L, 1L), j %between% c(-1L, 1L)))
expect_equal(or3s(logi_qb, i %between% c(-1L, 1L), n %between% c(-1L, 1L)),
             bor3(logi_qb, i %between% c(-1L, 1L), n %between% c(-1L, 1L)))
expect_equal(or3s(logi_qc, , ),
             bor3(logi_qc, , ))
expect_equal(or3s(logi_qd, , ),
             bor3(logi_qd, , ))
expect_equal(or3s(logi_qe, , ),
             bor3(logi_qe, , ))
expect_equal(or3s(logi_qf, , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_qf, , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_qg, , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_qg, , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_qh, , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_qh, , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_qi, u %between% c(-1L, 1L), ),
             bor3(logi_qi, u %between% c(-1L, 1L), ))
expect_equal(or3s(logi_qj, d %between% c(-1L, 1L), ),
             bor3(logi_qj, d %between% c(-1L, 1L), ))
expect_equal(or3s(logi_qk, p %between% c(-1L, 1L), ),
             bor3(logi_qk, p %between% c(-1L, 1L), ))
expect_equal(or3s(logi_ql, r %between% c(-1L, 1L), a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ql, r %between% c(-1L, 1L), a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_qm, n %between% c(-1L, 1L), m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_qm, n %between% c(-1L, 1L), m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_qn, m %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_qn, m %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_qo, , ),
             bor3(logi_qo, , ))
expect_equal(or3s(logi_qp, , ),
             bor3(logi_qp, , ))
expect_equal(or3s(logi_qq, , ),
             bor3(logi_qq, , ))
expect_equal(or3s(logi_qr, , f %in% 1:4),
             bor3(logi_qr, , f %in% 1:4))
expect_equal(or3s(logi_qs, , s %in% 1:4),
             bor3(logi_qs, , s %in% 1:4))
expect_equal(or3s(logi_qt, , d %in% 1:4),
             bor3(logi_qt, , d %in% 1:4))
expect_equal(or3s(logi_qu, c %between% c(-1L, 1L), ),
             bor3(logi_qu, c %between% c(-1L, 1L), ))
expect_equal(or3s(logi_qv, l %between% c(-1L, 1L), ),
             bor3(logi_qv, l %between% c(-1L, 1L), ))
expect_equal(or3s(logi_qw, d %between% c(-1L, 1L), ),
             bor3(logi_qw, d %between% c(-1L, 1L), ))
expect_equal(or3s(logi_qx, b %between% c(-1L, 1L), y %in% 1:4),
             bor3(logi_qx, b %between% c(-1L, 1L), y %in% 1:4))
expect_equal(or3s(logi_qy, m %between% c(-1L, 1L), f %in% 1:4),
             bor3(logi_qy, m %between% c(-1L, 1L), f %in% 1:4))
expect_equal(or3s(logi_qz, z %between% c(-1L, 1L), a %in% 1:4),
             bor3(logi_qz, z %between% c(-1L, 1L), a %in% 1:4))
expect_equal(or3s(logi_ra, , ),
             bor3(logi_ra, , ))
expect_equal(or3s(logi_rb, , ),
             bor3(logi_rb, , ))
expect_equal(or3s(logi_rc, , ),
             bor3(logi_rc, , ))
expect_equal(or3s(logi_rd, , n < 0L),
             bor3(logi_rd, , n < 0L))
expect_equal(or3s(logi_re, , u < 1L),
             bor3(logi_re, , u < 1L))
expect_equal(or3s(logi_rf, , l < 9L),
             bor3(logi_rf, , l < 9L))
expect_equal(or3s(logi_rg, z %between% c(-1L, 1L), ),
             bor3(logi_rg, z %between% c(-1L, 1L), ))
expect_equal(or3s(logi_rh, u %between% c(-1L, 1L), ),
             bor3(logi_rh, u %between% c(-1L, 1L), ))
expect_equal(or3s(logi_ri, y %between% c(-1L, 1L), ),
             bor3(logi_ri, y %between% c(-1L, 1L), ))
expect_equal(or3s(logi_rj, o %between% c(-1L, 1L), d < 0L),
             bor3(logi_rj, o %between% c(-1L, 1L), d < 0L))
expect_equal(or3s(logi_rk, a %between% c(-1L, 1L), j < 1L),
             bor3(logi_rk, a %between% c(-1L, 1L), j < 1L))
expect_equal(or3s(logi_rl, z %between% c(-1L, 1L), r < 9L),
             bor3(logi_rl, z %between% c(-1L, 1L), r < 9L))
expect_equal(or3s(logi_rm, , ),
             bor3(logi_rm, , ))
expect_equal(or3s(logi_rn, , ),
             bor3(logi_rn, , ))
expect_equal(or3s(logi_ro, , ),
             bor3(logi_ro, , ))
expect_equal(or3s(logi_rp, , s <= 0L),
             bor3(logi_rp, , s <= 0L))
expect_equal(or3s(logi_rq, , l <= 1L),
             bor3(logi_rq, , l <= 1L))
expect_equal(or3s(logi_rr, , n <= 9L),
             bor3(logi_rr, , n <= 9L))
expect_equal(or3s(logi_rs, b %between% c(-1L, 1L), ),
             bor3(logi_rs, b %between% c(-1L, 1L), ))
expect_equal(or3s(logi_rt, n %between% c(-1L, 1L), ),
             bor3(logi_rt, n %between% c(-1L, 1L), ))
expect_equal(or3s(logi_ru, g %between% c(-1L, 1L), ),
             bor3(logi_ru, g %between% c(-1L, 1L), ))
expect_equal(or3s(logi_rv, n %between% c(-1L, 1L), o <= 0L),
             bor3(logi_rv, n %between% c(-1L, 1L), o <= 0L))
expect_equal(or3s(logi_rw, i %between% c(-1L, 1L), k <= 1L),
             bor3(logi_rw, i %between% c(-1L, 1L), k <= 1L))
expect_equal(or3s(logi_rx, z %between% c(-1L, 1L), q <= 9L),
             bor3(logi_rx, z %between% c(-1L, 1L), q <= 9L))
expect_equal(or3s(logi_ry, , ),
             bor3(logi_ry, , ))
expect_equal(or3s(logi_rz, , ),
             bor3(logi_rz, , ))
expect_equal(or3s(logi_sa, , ),
             bor3(logi_sa, , ))
expect_equal(or3s(logi_sb, , m == 0L),
             bor3(logi_sb, , m == 0L))
expect_equal(or3s(logi_sc, , u == 1L),
             bor3(logi_sc, , u == 1L))
expect_equal(or3s(logi_sd, , t == 9L),
             bor3(logi_sd, , t == 9L))
expect_equal(or3s(logi_se, n %between% c(-1L, 1L), ),
             bor3(logi_se, n %between% c(-1L, 1L), ))
expect_equal(or3s(logi_sf, f %between% c(-1L, 1L), ),
             bor3(logi_sf, f %between% c(-1L, 1L), ))
expect_equal(or3s(logi_sg, n %between% c(-1L, 1L), ),
             bor3(logi_sg, n %between% c(-1L, 1L), ))
expect_equal(or3s(logi_sh, i %between% c(-1L, 1L), e == 0L),
             bor3(logi_sh, i %between% c(-1L, 1L), e == 0L))
expect_equal(or3s(logi_si, f %between% c(-1L, 1L), b == 1L),
             bor3(logi_si, f %between% c(-1L, 1L), b == 1L))
expect_equal(or3s(logi_sj, b %between% c(-1L, 1L), t == 9L),
             bor3(logi_sj, b %between% c(-1L, 1L), t == 9L))
expect_equal(or3s(logi_sk, , ),
             bor3(logi_sk, , ))
expect_equal(or3s(logi_sl, , ),
             bor3(logi_sl, , ))
expect_equal(or3s(logi_sm, , ),
             bor3(logi_sm, , ))
expect_equal(or3s(logi_sn, , g > 0L),
             bor3(logi_sn, , g > 0L))
expect_equal(or3s(logi_so, , w > 1L),
             bor3(logi_so, , w > 1L))
expect_equal(or3s(logi_sp, , n > 9L),
             bor3(logi_sp, , n > 9L))
expect_equal(or3s(logi_sq, q %between% c(-1L, 1L), ),
             bor3(logi_sq, q %between% c(-1L, 1L), ))
expect_equal(or3s(logi_sr, v %between% c(-1L, 1L), ),
             bor3(logi_sr, v %between% c(-1L, 1L), ))
expect_equal(or3s(logi_ss, j %between% c(-1L, 1L), ),
             bor3(logi_ss, j %between% c(-1L, 1L), ))
expect_equal(or3s(logi_st, a %between% c(-1L, 1L), a > 0L),
             bor3(logi_st, a %between% c(-1L, 1L), a > 0L))
expect_equal(or3s(logi_su, y %between% c(-1L, 1L), i > 1L),
             bor3(logi_su, y %between% c(-1L, 1L), i > 1L))
expect_equal(or3s(logi_sv, p %between% c(-1L, 1L), c > 9L),
             bor3(logi_sv, p %between% c(-1L, 1L), c > 9L))
expect_equal(or3s(logi_sw, , ),
             bor3(logi_sw, , ))
expect_equal(or3s(logi_sx, , ),
             bor3(logi_sx, , ))
expect_equal(or3s(logi_sy, , ),
             bor3(logi_sy, , ))
expect_equal(or3s(logi_sz, , h >= 0L),
             bor3(logi_sz, , h >= 0L))
expect_equal(or3s(logi_ta, , d >= 1L),
             bor3(logi_ta, , d >= 1L))
expect_equal(or3s(logi_tb, , m >= 9L),
             bor3(logi_tb, , m >= 9L))
expect_equal(or3s(logi_tc, f %between% c(-1L, 1L), ),
             bor3(logi_tc, f %between% c(-1L, 1L), ))
expect_equal(or3s(logi_td, z %between% c(-1L, 1L), ),
             bor3(logi_td, z %between% c(-1L, 1L), ))
expect_equal(or3s(logi_te, e %between% c(-1L, 1L), ),
             bor3(logi_te, e %between% c(-1L, 1L), ))
expect_equal(or3s(logi_tf, s %between% c(-1L, 1L), c >= 0L),
             bor3(logi_tf, s %between% c(-1L, 1L), c >= 0L))
expect_equal(or3s(logi_tg, c %between% c(-1L, 1L), q >= 1L),
             bor3(logi_tg, c %between% c(-1L, 1L), q >= 1L))
expect_equal(or3s(logi_th, r %between% c(-1L, 1L), v >= 9L),
             bor3(logi_th, r %between% c(-1L, 1L), v >= 9L))
expect_equal(or3s(logi_ti, , ),
             bor3(logi_ti, , ))
expect_equal(or3s(logi_tj, , ),
             bor3(logi_tj, , ))
expect_equal(or3s(logi_tk, , ),
             bor3(logi_tk, , ))
expect_equal(or3s(logi_tl, , logi_n),
             bor3(logi_tl, , logi_n))
expect_equal(or3s(logi_tm, , logi_c),
             bor3(logi_tm, , logi_c))
expect_equal(or3s(logi_tn, , logi_a),
             bor3(logi_tn, , logi_a))
expect_equal(or3s(logi_to, b %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_to, b %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_tp, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_tp, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_tq, i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_tq, i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_tr, j %in% c(4L, -3L, 2L, 7L, 8L), logi_p),
             bor3(logi_tr, j %in% c(4L, -3L, 2L, 7L, 8L), logi_p))
expect_equal(or3s(logi_ts, v %in% c(4L, -3L, 2L, 7L, 8L), logi_u),
             bor3(logi_ts, v %in% c(4L, -3L, 2L, 7L, 8L), logi_u))
expect_equal(or3s(logi_tt, g %in% c(4L, -3L, 2L, 7L, 8L), logi_l),
             bor3(logi_tt, g %in% c(4L, -3L, 2L, 7L, 8L), logi_l))
expect_equal(or3s(logi_tu, , ),
             bor3(logi_tu, , ))
expect_equal(or3s(logi_tv, , ),
             bor3(logi_tv, , ))
expect_equal(or3s(logi_tw, , ),
             bor3(logi_tw, , ))
expect_equal(or3s(logi_tx, , logi_c),
             bor3(logi_tx, , logi_c))
expect_equal(or3s(logi_ty, , logi_y),
             bor3(logi_ty, , logi_y))
expect_equal(or3s(logi_tz, , logi_d),
             bor3(logi_tz, , logi_d))
expect_equal(or3s(logi_ua, m %in% 1:4, ),
             bor3(logi_ua, m %in% 1:4, ))
expect_equal(or3s(logi_ub, j %in% 1:4, ),
             bor3(logi_ub, j %in% 1:4, ))
expect_equal(or3s(logi_uc, n %in% 1:4, ),
             bor3(logi_uc, n %in% 1:4, ))
expect_equal(or3s(logi_ud, h %in% 1:4, logi_h),
             bor3(logi_ud, h %in% 1:4, logi_h))
expect_equal(or3s(logi_ue, b %in% 1:4, logi_k),
             bor3(logi_ue, b %in% 1:4, logi_k))
expect_equal(or3s(logi_uf, a %in% 1:4, logi_i),
             bor3(logi_uf, a %in% 1:4, logi_i))
expect_equal(or3s(logi_ug, , ),
             bor3(logi_ug, , ))
expect_equal(or3s(logi_uh, , ),
             bor3(logi_uh, , ))
expect_equal(or3s(logi_ui, , ),
             bor3(logi_ui, , ))
expect_equal(or3s(logi_uj, , !logi_f),
             bor3(logi_uj, , !logi_f))
expect_equal(or3s(logi_uk, , !logi_o),
             bor3(logi_uk, , !logi_o))
expect_equal(or3s(logi_ul, , !logi_n),
             bor3(logi_ul, , !logi_n))
expect_equal(or3s(logi_um, n %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_um, n %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_un, j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_un, j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_uo, j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_uo, j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_up, s %in% c(4L, -3L, 2L, 7L, 8L), !logi_b),
             bor3(logi_up, s %in% c(4L, -3L, 2L, 7L, 8L), !logi_b))
expect_equal(or3s(logi_uq, u %in% c(4L, -3L, 2L, 7L, 8L), !logi_z),
             bor3(logi_uq, u %in% c(4L, -3L, 2L, 7L, 8L), !logi_z))
expect_equal(or3s(logi_ur, l %in% c(4L, -3L, 2L, 7L, 8L), !logi_w),
             bor3(logi_ur, l %in% c(4L, -3L, 2L, 7L, 8L), !logi_w))
expect_equal(or3s(logi_us, , ),
             bor3(logi_us, , ))
expect_equal(or3s(logi_ut, , ),
             bor3(logi_ut, , ))
expect_equal(or3s(logi_uu, , ),
             bor3(logi_uu, , ))
expect_equal(or3s(logi_uv, , !logi_x),
             bor3(logi_uv, , !logi_x))
expect_equal(or3s(logi_uw, , !logi_g),
             bor3(logi_uw, , !logi_g))
expect_equal(or3s(logi_ux, , !logi_z),
             bor3(logi_ux, , !logi_z))
expect_equal(or3s(logi_uy, u %in% 1:4, ),
             bor3(logi_uy, u %in% 1:4, ))
expect_equal(or3s(logi_uz, f %in% 1:4, ),
             bor3(logi_uz, f %in% 1:4, ))
expect_equal(or3s(logi_va, r %in% 1:4, ),
             bor3(logi_va, r %in% 1:4, ))
expect_equal(or3s(logi_vb, b %in% 1:4, !logi_m),
             bor3(logi_vb, b %in% 1:4, !logi_m))
expect_equal(or3s(logi_vc, y %in% 1:4, !logi_h),
             bor3(logi_vc, y %in% 1:4, !logi_h))
expect_equal(or3s(logi_vd, w %in% 1:4, !logi_a),
             bor3(logi_vd, w %in% 1:4, !logi_a))
expect_equal(or3s(logi_ve, , ),
             bor3(logi_ve, , ))
expect_equal(or3s(logi_vf, , ),
             bor3(logi_vf, , ))
expect_equal(or3s(logi_vg, , ),
             bor3(logi_vg, , ))
expect_equal(or3s(logi_vh, , x != 0L),
             bor3(logi_vh, , x != 0L))
expect_equal(or3s(logi_vi, , i != 1L),
             bor3(logi_vi, , i != 1L))
expect_equal(or3s(logi_vj, , l != 9L),
             bor3(logi_vj, , l != 9L))
expect_equal(or3s(logi_vk, f %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_vk, f %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_vl, z %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_vl, z %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_vm, j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_vm, j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_vn, w %in% c(4L, -3L, 2L, 7L, 8L), o != 0L),
             bor3(logi_vn, w %in% c(4L, -3L, 2L, 7L, 8L), o != 0L))
expect_equal(or3s(logi_vo, o %in% c(4L, -3L, 2L, 7L, 8L), k != 1L),
             bor3(logi_vo, o %in% c(4L, -3L, 2L, 7L, 8L), k != 1L))
expect_equal(or3s(logi_vp, g %in% c(4L, -3L, 2L, 7L, 8L), m != 9L),
             bor3(logi_vp, g %in% c(4L, -3L, 2L, 7L, 8L), m != 9L))
expect_equal(or3s(logi_vq, , ),
             bor3(logi_vq, , ))
expect_equal(or3s(logi_vr, , ),
             bor3(logi_vr, , ))
expect_equal(or3s(logi_vs, , ),
             bor3(logi_vs, , ))
expect_equal(or3s(logi_vt, , x != 0L),
             bor3(logi_vt, , x != 0L))
expect_equal(or3s(logi_vu, , l != 1L),
             bor3(logi_vu, , l != 1L))
expect_equal(or3s(logi_vv, , m != 9L),
             bor3(logi_vv, , m != 9L))
expect_equal(or3s(logi_vw, z %in% 1:4, ),
             bor3(logi_vw, z %in% 1:4, ))
expect_equal(or3s(logi_vx, h %in% 1:4, ),
             bor3(logi_vx, h %in% 1:4, ))
expect_equal(or3s(logi_vy, i %in% 1:4, ),
             bor3(logi_vy, i %in% 1:4, ))
expect_equal(or3s(logi_vz, p %in% 1:4, v != 0L),
             bor3(logi_vz, p %in% 1:4, v != 0L))
expect_equal(or3s(logi_wa, s %in% 1:4, o != 1L),
             bor3(logi_wa, s %in% 1:4, o != 1L))
expect_equal(or3s(logi_wb, d %in% 1:4, a != 9L),
             bor3(logi_wb, d %in% 1:4, a != 9L))
expect_equal(or3s(logi_wc, , ),
             bor3(logi_wc, , ))
expect_equal(or3s(logi_wd, , ),
             bor3(logi_wd, , ))
expect_equal(or3s(logi_we, , ),
             bor3(logi_we, , ))
expect_equal(or3s(logi_wf, , n %between% c(-1L, 1L)),
             bor3(logi_wf, , n %between% c(-1L, 1L)))
expect_equal(or3s(logi_wg, , i %between% c(-1L, 1L)),
             bor3(logi_wg, , i %between% c(-1L, 1L)))
expect_equal(or3s(logi_wh, , b %between% c(-1L, 1L)),
             bor3(logi_wh, , b %between% c(-1L, 1L)))
expect_equal(or3s(logi_wi, o %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_wi, o %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_wj, i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_wj, i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_wk, d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_wk, d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_wl, l %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L)),
             bor3(logi_wl, l %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L)))
expect_equal(or3s(logi_wm, x %in% c(4L, -3L, 2L, 7L, 8L), w %between% c(-1L, 1L)),
             bor3(logi_wm, x %in% c(4L, -3L, 2L, 7L, 8L), w %between% c(-1L, 1L)))
expect_equal(or3s(logi_wn, m %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L)),
             bor3(logi_wn, m %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L)))
expect_equal(or3s(logi_wo, , ),
             bor3(logi_wo, , ))
expect_equal(or3s(logi_wp, , ),
             bor3(logi_wp, , ))
expect_equal(or3s(logi_wq, , ),
             bor3(logi_wq, , ))
expect_equal(or3s(logi_wr, , d %between% c(-1L, 1L)),
             bor3(logi_wr, , d %between% c(-1L, 1L)))
expect_equal(or3s(logi_ws, , d %between% c(-1L, 1L)),
             bor3(logi_ws, , d %between% c(-1L, 1L)))
expect_equal(or3s(logi_wt, , g %between% c(-1L, 1L)),
             bor3(logi_wt, , g %between% c(-1L, 1L)))
expect_equal(or3s(logi_wu, z %in% 1:4, ),
             bor3(logi_wu, z %in% 1:4, ))
expect_equal(or3s(logi_wv, t %in% 1:4, ),
             bor3(logi_wv, t %in% 1:4, ))
expect_equal(or3s(logi_ww, r %in% 1:4, ),
             bor3(logi_ww, r %in% 1:4, ))
expect_equal(or3s(logi_wx, s %in% 1:4, j %between% c(-1L, 1L)),
             bor3(logi_wx, s %in% 1:4, j %between% c(-1L, 1L)))
expect_equal(or3s(logi_wy, u %in% 1:4, q %between% c(-1L, 1L)),
             bor3(logi_wy, u %in% 1:4, q %between% c(-1L, 1L)))
expect_equal(or3s(logi_wz, d %in% 1:4, j %between% c(-1L, 1L)),
             bor3(logi_wz, d %in% 1:4, j %between% c(-1L, 1L)))
expect_equal(or3s(logi_xa, , ),
             bor3(logi_xa, , ))
expect_equal(or3s(logi_xb, , ),
             bor3(logi_xb, , ))
expect_equal(or3s(logi_xc, , ),
             bor3(logi_xc, , ))
expect_equal(or3s(logi_xd, , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_xd, , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_xe, , d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_xe, , d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_xf, , u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_xf, , u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_xg, e %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_xg, e %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_xh, h %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_xh, h %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_xi, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_xi, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_xj, o %in% c(4L, -3L, 2L, 7L, 8L), h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_xj, o %in% c(4L, -3L, 2L, 7L, 8L), h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_xk, b %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_xk, b %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_xl, e %in% c(4L, -3L, 2L, 7L, 8L), s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_xl, e %in% c(4L, -3L, 2L, 7L, 8L), s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_xm, , ),
             bor3(logi_xm, , ))
expect_equal(or3s(logi_xn, , ),
             bor3(logi_xn, , ))
expect_equal(or3s(logi_xo, , ),
             bor3(logi_xo, , ))
expect_equal(or3s(logi_xp, , c %in% 1:4),
             bor3(logi_xp, , c %in% 1:4))
expect_equal(or3s(logi_xq, , x %in% 1:4),
             bor3(logi_xq, , x %in% 1:4))
expect_equal(or3s(logi_xr, , f %in% 1:4),
             bor3(logi_xr, , f %in% 1:4))
expect_equal(or3s(logi_xs, j %in% 1:4, ),
             bor3(logi_xs, j %in% 1:4, ))
expect_equal(or3s(logi_xt, v %in% 1:4, ),
             bor3(logi_xt, v %in% 1:4, ))
expect_equal(or3s(logi_xu, d %in% 1:4, ),
             bor3(logi_xu, d %in% 1:4, ))
expect_equal(or3s(logi_xv, f %in% 1:4, c %in% 1:4),
             bor3(logi_xv, f %in% 1:4, c %in% 1:4))
expect_equal(or3s(logi_xw, w %in% 1:4, j %in% 1:4),
             bor3(logi_xw, w %in% 1:4, j %in% 1:4))
expect_equal(or3s(logi_xx, o %in% 1:4, n %in% 1:4),
             bor3(logi_xx, o %in% 1:4, n %in% 1:4))
expect_equal(or3s(logi_xy, , ),
             bor3(logi_xy, , ))
expect_equal(or3s(logi_xz, , ),
             bor3(logi_xz, , ))
expect_equal(or3s(logi_ya, , ),
             bor3(logi_ya, , ))
expect_equal(or3s(logi_yb, , s < 0L),
             bor3(logi_yb, , s < 0L))
expect_equal(or3s(logi_yc, , e < 1L),
             bor3(logi_yc, , e < 1L))
expect_equal(or3s(logi_yd, , d < 9L),
             bor3(logi_yd, , d < 9L))
expect_equal(or3s(logi_ye, y %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_ye, y %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_yf, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_yf, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_yg, j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_yg, j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_yh, e %in% c(4L, -3L, 2L, 7L, 8L), h < 0L),
             bor3(logi_yh, e %in% c(4L, -3L, 2L, 7L, 8L), h < 0L))
expect_equal(or3s(logi_yi, g %in% c(4L, -3L, 2L, 7L, 8L), p < 1L),
             bor3(logi_yi, g %in% c(4L, -3L, 2L, 7L, 8L), p < 1L))
expect_equal(or3s(logi_yj, e %in% c(4L, -3L, 2L, 7L, 8L), o < 9L),
             bor3(logi_yj, e %in% c(4L, -3L, 2L, 7L, 8L), o < 9L))
expect_equal(or3s(logi_yk, , ),
             bor3(logi_yk, , ))
expect_equal(or3s(logi_yl, , ),
             bor3(logi_yl, , ))
expect_equal(or3s(logi_ym, , ),
             bor3(logi_ym, , ))
expect_equal(or3s(logi_yn, , j < 0L),
             bor3(logi_yn, , j < 0L))
expect_equal(or3s(logi_yo, , t < 1L),
             bor3(logi_yo, , t < 1L))
expect_equal(or3s(logi_yp, , m < 9L),
             bor3(logi_yp, , m < 9L))
expect_equal(or3s(logi_yq, m %in% 1:4, ),
             bor3(logi_yq, m %in% 1:4, ))
expect_equal(or3s(logi_yr, q %in% 1:4, ),
             bor3(logi_yr, q %in% 1:4, ))
expect_equal(or3s(logi_ys, d %in% 1:4, ),
             bor3(logi_ys, d %in% 1:4, ))
expect_equal(or3s(logi_yt, w %in% 1:4, a < 0L),
             bor3(logi_yt, w %in% 1:4, a < 0L))
expect_equal(or3s(logi_yu, k %in% 1:4, d < 1L),
             bor3(logi_yu, k %in% 1:4, d < 1L))
expect_equal(or3s(logi_yv, p %in% 1:4, l < 9L),
             bor3(logi_yv, p %in% 1:4, l < 9L))
expect_equal(or3s(logi_yw, , ),
             bor3(logi_yw, , ))
expect_equal(or3s(logi_yx, , ),
             bor3(logi_yx, , ))
expect_equal(or3s(logi_yy, , ),
             bor3(logi_yy, , ))
expect_equal(or3s(logi_yz, , a <= 0L),
             bor3(logi_yz, , a <= 0L))
expect_equal(or3s(logi_aa, , b <= 1L),
             bor3(logi_aa, , b <= 1L))
expect_equal(or3s(logi_ab, , i <= 9L),
             bor3(logi_ab, , i <= 9L))
expect_equal(or3s(logi_ac, u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_ac, u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_ad, p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_ad, p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_ae, q %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_ae, q %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_af, f %in% c(4L, -3L, 2L, 7L, 8L), x <= 0L),
             bor3(logi_af, f %in% c(4L, -3L, 2L, 7L, 8L), x <= 0L))
expect_equal(or3s(logi_ag, q %in% c(4L, -3L, 2L, 7L, 8L), a <= 1L),
             bor3(logi_ag, q %in% c(4L, -3L, 2L, 7L, 8L), a <= 1L))
expect_equal(or3s(logi_ah, j %in% c(4L, -3L, 2L, 7L, 8L), d <= 9L),
             bor3(logi_ah, j %in% c(4L, -3L, 2L, 7L, 8L), d <= 9L))
expect_equal(or3s(logi_ai, , ),
             bor3(logi_ai, , ))
expect_equal(or3s(logi_aj, , ),
             bor3(logi_aj, , ))
expect_equal(or3s(logi_ak, , ),
             bor3(logi_ak, , ))
expect_equal(or3s(logi_al, , a <= 0L),
             bor3(logi_al, , a <= 0L))
expect_equal(or3s(logi_am, , l <= 1L),
             bor3(logi_am, , l <= 1L))
expect_equal(or3s(logi_an, , w <= 9L),
             bor3(logi_an, , w <= 9L))
expect_equal(or3s(logi_ao, w %in% 1:4, ),
             bor3(logi_ao, w %in% 1:4, ))
expect_equal(or3s(logi_ap, f %in% 1:4, ),
             bor3(logi_ap, f %in% 1:4, ))
expect_equal(or3s(logi_aq, w %in% 1:4, ),
             bor3(logi_aq, w %in% 1:4, ))
expect_equal(or3s(logi_ar, w %in% 1:4, z <= 0L),
             bor3(logi_ar, w %in% 1:4, z <= 0L))
expect_equal(or3s(logi_as, b %in% 1:4, a <= 1L),
             bor3(logi_as, b %in% 1:4, a <= 1L))
expect_equal(or3s(logi_at, f %in% 1:4, f <= 9L),
             bor3(logi_at, f %in% 1:4, f <= 9L))
expect_equal(or3s(logi_au, , ),
             bor3(logi_au, , ))
expect_equal(or3s(logi_av, , ),
             bor3(logi_av, , ))
expect_equal(or3s(logi_aw, , ),
             bor3(logi_aw, , ))
expect_equal(or3s(logi_ax, , b == 0L),
             bor3(logi_ax, , b == 0L))
expect_equal(or3s(logi_ay, , a == 1L),
             bor3(logi_ay, , a == 1L))
expect_equal(or3s(logi_az, , g == 9L),
             bor3(logi_az, , g == 9L))
expect_equal(or3s(logi_aaa, n %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_aaa, n %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_aab, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_aab, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_aac, q %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_aac, q %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_aad, h %in% c(4L, -3L, 2L, 7L, 8L), i == 0L),
             bor3(logi_aad, h %in% c(4L, -3L, 2L, 7L, 8L), i == 0L))
expect_equal(or3s(logi_aae, b %in% c(4L, -3L, 2L, 7L, 8L), s == 1L),
             bor3(logi_aae, b %in% c(4L, -3L, 2L, 7L, 8L), s == 1L))
expect_equal(or3s(logi_aaf, e %in% c(4L, -3L, 2L, 7L, 8L), k == 9L),
             bor3(logi_aaf, e %in% c(4L, -3L, 2L, 7L, 8L), k == 9L))
expect_equal(or3s(logi_aag, , ),
             bor3(logi_aag, , ))
expect_equal(or3s(logi_aah, , ),
             bor3(logi_aah, , ))
expect_equal(or3s(logi_aai, , ),
             bor3(logi_aai, , ))
expect_equal(or3s(logi_aaj, , g == 0L),
             bor3(logi_aaj, , g == 0L))
expect_equal(or3s(logi_aak, , u == 1L),
             bor3(logi_aak, , u == 1L))
expect_equal(or3s(logi_aal, , q == 9L),
             bor3(logi_aal, , q == 9L))
expect_equal(or3s(logi_aam, w %in% 1:4, ),
             bor3(logi_aam, w %in% 1:4, ))
expect_equal(or3s(logi_aan, w %in% 1:4, ),
             bor3(logi_aan, w %in% 1:4, ))
expect_equal(or3s(logi_aao, r %in% 1:4, ),
             bor3(logi_aao, r %in% 1:4, ))
expect_equal(or3s(logi_aap, b %in% 1:4, b == 0L),
             bor3(logi_aap, b %in% 1:4, b == 0L))
expect_equal(or3s(logi_aaq, u %in% 1:4, x == 1L),
             bor3(logi_aaq, u %in% 1:4, x == 1L))
expect_equal(or3s(logi_aar, d %in% 1:4, p == 9L),
             bor3(logi_aar, d %in% 1:4, p == 9L))
expect_equal(or3s(logi_aas, , ),
             bor3(logi_aas, , ))
expect_equal(or3s(logi_aat, , ),
             bor3(logi_aat, , ))
expect_equal(or3s(logi_aau, , ),
             bor3(logi_aau, , ))
expect_equal(or3s(logi_aav, , y > 0L),
             bor3(logi_aav, , y > 0L))
expect_equal(or3s(logi_aaw, , b > 1L),
             bor3(logi_aaw, , b > 1L))
expect_equal(or3s(logi_aax, , b > 9L),
             bor3(logi_aax, , b > 9L))
expect_equal(or3s(logi_aay, n %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_aay, n %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_aaz, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_aaz, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_aba, p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_aba, p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_abb, s %in% c(4L, -3L, 2L, 7L, 8L), i > 0L),
             bor3(logi_abb, s %in% c(4L, -3L, 2L, 7L, 8L), i > 0L))
expect_equal(or3s(logi_abc, o %in% c(4L, -3L, 2L, 7L, 8L), x > 1L),
             bor3(logi_abc, o %in% c(4L, -3L, 2L, 7L, 8L), x > 1L))
expect_equal(or3s(logi_abd, y %in% c(4L, -3L, 2L, 7L, 8L), k > 9L),
             bor3(logi_abd, y %in% c(4L, -3L, 2L, 7L, 8L), k > 9L))
expect_equal(or3s(logi_abe, , ),
             bor3(logi_abe, , ))
expect_equal(or3s(logi_abf, , ),
             bor3(logi_abf, , ))
expect_equal(or3s(logi_abg, , ),
             bor3(logi_abg, , ))
expect_equal(or3s(logi_abh, , w > 0L),
             bor3(logi_abh, , w > 0L))
expect_equal(or3s(logi_abi, , h > 1L),
             bor3(logi_abi, , h > 1L))
expect_equal(or3s(logi_abj, , x > 9L),
             bor3(logi_abj, , x > 9L))
expect_equal(or3s(logi_abk, j %in% 1:4, ),
             bor3(logi_abk, j %in% 1:4, ))
expect_equal(or3s(logi_abl, f %in% 1:4, ),
             bor3(logi_abl, f %in% 1:4, ))
expect_equal(or3s(logi_abm, j %in% 1:4, ),
             bor3(logi_abm, j %in% 1:4, ))
expect_equal(or3s(logi_abn, d %in% 1:4, i > 0L),
             bor3(logi_abn, d %in% 1:4, i > 0L))
expect_equal(or3s(logi_abo, f %in% 1:4, r > 1L),
             bor3(logi_abo, f %in% 1:4, r > 1L))
expect_equal(or3s(logi_abp, i %in% 1:4, k > 9L),
             bor3(logi_abp, i %in% 1:4, k > 9L))
expect_equal(or3s(logi_abq, , ),
             bor3(logi_abq, , ))
expect_equal(or3s(logi_abr, , ),
             bor3(logi_abr, , ))
expect_equal(or3s(logi_abs, , ),
             bor3(logi_abs, , ))
expect_equal(or3s(logi_abt, , p >= 0L),
             bor3(logi_abt, , p >= 0L))
expect_equal(or3s(logi_abu, , u >= 1L),
             bor3(logi_abu, , u >= 1L))
expect_equal(or3s(logi_abv, , x >= 9L),
             bor3(logi_abv, , x >= 9L))
expect_equal(or3s(logi_abw, o %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_abw, o %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_abx, b %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_abx, b %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_aby, n %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(logi_aby, n %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(logi_abz, v %in% c(4L, -3L, 2L, 7L, 8L), a >= 0L),
             bor3(logi_abz, v %in% c(4L, -3L, 2L, 7L, 8L), a >= 0L))
expect_equal(or3s(logi_aca, r %in% c(4L, -3L, 2L, 7L, 8L), l >= 1L),
             bor3(logi_aca, r %in% c(4L, -3L, 2L, 7L, 8L), l >= 1L))
expect_equal(or3s(logi_acb, x %in% c(4L, -3L, 2L, 7L, 8L), i >= 9L),
             bor3(logi_acb, x %in% c(4L, -3L, 2L, 7L, 8L), i >= 9L))
expect_equal(or3s(logi_acc, , ),
             bor3(logi_acc, , ))
expect_equal(or3s(logi_acd, , ),
             bor3(logi_acd, , ))
expect_equal(or3s(logi_ace, , ),
             bor3(logi_ace, , ))
expect_equal(or3s(logi_acf, , a >= 0L),
             bor3(logi_acf, , a >= 0L))
expect_equal(or3s(logi_acg, , e >= 1L),
             bor3(logi_acg, , e >= 1L))
expect_equal(or3s(logi_ach, , r >= 9L),
             bor3(logi_ach, , r >= 9L))
expect_equal(or3s(logi_aci, n %in% 1:4, ),
             bor3(logi_aci, n %in% 1:4, ))
expect_equal(or3s(logi_acj, p %in% 1:4, ),
             bor3(logi_acj, p %in% 1:4, ))
expect_equal(or3s(logi_ack, m %in% 1:4, ),
             bor3(logi_ack, m %in% 1:4, ))
expect_equal(or3s(logi_acl, e %in% 1:4, c >= 0L),
             bor3(logi_acl, e %in% 1:4, c >= 0L))
expect_equal(or3s(logi_acm, d %in% 1:4, a >= 1L),
             bor3(logi_acm, d %in% 1:4, a >= 1L))
expect_equal(or3s(logi_acn, h %in% 1:4, c >= 9L),
             bor3(logi_acn, h %in% 1:4, c >= 9L))
expect_equal(or3s(logi_aco, , ),
             bor3(logi_aco, , ))
expect_equal(or3s(logi_acp, , ),
             bor3(logi_acp, , ))
expect_equal(or3s(logi_acq, , ),
             bor3(logi_acq, , ))
expect_equal(or3s(logi_acr, , logi_w),
             bor3(logi_acr, , logi_w))
expect_equal(or3s(logi_acs, , logi_f),
             bor3(logi_acs, , logi_f))
expect_equal(or3s(logi_act, , logi_y),
             bor3(logi_act, , logi_y))
expect_equal(or3s(logi_acu, h < 0L, ),
             bor3(logi_acu, h < 0L, ))
expect_equal(or3s(logi_acv, n < 1L, ),
             bor3(logi_acv, n < 1L, ))
expect_equal(or3s(logi_acw, w < 9L, ),
             bor3(logi_acw, w < 9L, ))
expect_equal(or3s(logi_acx, o < 0L, logi_b),
             bor3(logi_acx, o < 0L, logi_b))
expect_equal(or3s(logi_acy, l < 1L, logi_b),
             bor3(logi_acy, l < 1L, logi_b))
expect_equal(or3s(logi_acz, s < 9L, logi_g),
             bor3(logi_acz, s < 9L, logi_g))
expect_equal(or3s(logi_ada, , ),
             bor3(logi_ada, , ))
expect_equal(or3s(logi_adb, , ),
             bor3(logi_adb, , ))
expect_equal(or3s(logi_adc, , ),
             bor3(logi_adc, , ))
expect_equal(or3s(logi_add, , !logi_r),
             bor3(logi_add, , !logi_r))
expect_equal(or3s(logi_ade, , !logi_e),
             bor3(logi_ade, , !logi_e))
expect_equal(or3s(logi_adf, , !logi_g),
             bor3(logi_adf, , !logi_g))
expect_equal(or3s(logi_adg, d < 0L, ),
             bor3(logi_adg, d < 0L, ))
expect_equal(or3s(logi_adh, v < 1L, ),
             bor3(logi_adh, v < 1L, ))
expect_equal(or3s(logi_adi, q < 9L, ),
             bor3(logi_adi, q < 9L, ))
expect_equal(or3s(logi_adj, n < 0L, !logi_x),
             bor3(logi_adj, n < 0L, !logi_x))
expect_equal(or3s(logi_adk, z < 1L, !logi_i),
             bor3(logi_adk, z < 1L, !logi_i))
expect_equal(or3s(logi_adl, m < 9L, !logi_o),
             bor3(logi_adl, m < 9L, !logi_o))
expect_equal(or3s(logi_adm, , ),
             bor3(logi_adm, , ))
expect_equal(or3s(logi_adn, , ),
             bor3(logi_adn, , ))
expect_equal(or3s(logi_ado, , ),
             bor3(logi_ado, , ))
expect_equal(or3s(logi_adp, , e != 0L),
             bor3(logi_adp, , e != 0L))
expect_equal(or3s(logi_adq, , z != 1L),
             bor3(logi_adq, , z != 1L))
expect_equal(or3s(logi_adr, , u != 9L),
             bor3(logi_adr, , u != 9L))
expect_equal(or3s(logi_ads, b < 0L, ),
             bor3(logi_ads, b < 0L, ))
expect_equal(or3s(logi_adt, m < 1L, ),
             bor3(logi_adt, m < 1L, ))
expect_equal(or3s(logi_adu, y < 9L, ),
             bor3(logi_adu, y < 9L, ))
expect_equal(or3s(logi_adv, x < 0L, y != 0L),
             bor3(logi_adv, x < 0L, y != 0L))
expect_equal(or3s(logi_adw, h < 1L, k != 1L),
             bor3(logi_adw, h < 1L, k != 1L))
expect_equal(or3s(logi_adx, a < 9L, k != 9L),
             bor3(logi_adx, a < 9L, k != 9L))
expect_equal(or3s(logi_ady, , ),
             bor3(logi_ady, , ))
expect_equal(or3s(logi_adz, , ),
             bor3(logi_adz, , ))
expect_equal(or3s(logi_aea, , ),
             bor3(logi_aea, , ))
expect_equal(or3s(logi_aeb, , g %between% c(-1L, 1L)),
             bor3(logi_aeb, , g %between% c(-1L, 1L)))
expect_equal(or3s(logi_aec, , a %between% c(-1L, 1L)),
             bor3(logi_aec, , a %between% c(-1L, 1L)))
expect_equal(or3s(logi_aed, , u %between% c(-1L, 1L)),
             bor3(logi_aed, , u %between% c(-1L, 1L)))
expect_equal(or3s(logi_aee, z < 0L, ),
             bor3(logi_aee, z < 0L, ))
expect_equal(or3s(logi_aef, q < 1L, ),
             bor3(logi_aef, q < 1L, ))
expect_equal(or3s(logi_aeg, x < 9L, ),
             bor3(logi_aeg, x < 9L, ))
expect_equal(or3s(logi_aeh, d < 0L, x %between% c(-1L, 1L)),
             bor3(logi_aeh, d < 0L, x %between% c(-1L, 1L)))
expect_equal(or3s(logi_aei, l < 1L, b %between% c(-1L, 1L)),
             bor3(logi_aei, l < 1L, b %between% c(-1L, 1L)))
expect_equal(or3s(logi_aej, i < 9L, k %between% c(-1L, 1L)),
             bor3(logi_aej, i < 9L, k %between% c(-1L, 1L)))
expect_equal(or3s(logi_aek, , ),
             bor3(logi_aek, , ))
expect_equal(or3s(logi_ael, , ),
             bor3(logi_ael, , ))
expect_equal(or3s(logi_aem, , ),
             bor3(logi_aem, , ))
expect_equal(or3s(logi_aen, , m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aen, , m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aeo, , u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aeo, , u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aep, , b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aep, , b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aeq, e < 0L, ),
             bor3(logi_aeq, e < 0L, ))
expect_equal(or3s(logi_aer, d < 1L, ),
             bor3(logi_aer, d < 1L, ))
expect_equal(or3s(logi_aes, i < 9L, ),
             bor3(logi_aes, i < 9L, ))
expect_equal(or3s(logi_aet, z < 0L, n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aet, z < 0L, n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aeu, m < 1L, g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aeu, m < 1L, g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aev, q < 9L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aev, q < 9L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aew, , ),
             bor3(logi_aew, , ))
expect_equal(or3s(logi_aex, , ),
             bor3(logi_aex, , ))
expect_equal(or3s(logi_aey, , ),
             bor3(logi_aey, , ))
expect_equal(or3s(logi_aez, , s %in% 1:4),
             bor3(logi_aez, , s %in% 1:4))
expect_equal(or3s(logi_afa, , e %in% 1:4),
             bor3(logi_afa, , e %in% 1:4))
expect_equal(or3s(logi_afb, , p %in% 1:4),
             bor3(logi_afb, , p %in% 1:4))
expect_equal(or3s(logi_afc, a < 0L, ),
             bor3(logi_afc, a < 0L, ))
expect_equal(or3s(logi_afd, u < 1L, ),
             bor3(logi_afd, u < 1L, ))
expect_equal(or3s(logi_afe, z < 9L, ),
             bor3(logi_afe, z < 9L, ))
expect_equal(or3s(logi_aff, d < 0L, d %in% 1:4),
             bor3(logi_aff, d < 0L, d %in% 1:4))
expect_equal(or3s(logi_afg, a < 1L, b %in% 1:4),
             bor3(logi_afg, a < 1L, b %in% 1:4))
expect_equal(or3s(logi_afh, y < 9L, q %in% 1:4),
             bor3(logi_afh, y < 9L, q %in% 1:4))
expect_equal(or3s(logi_afi, , ),
             bor3(logi_afi, , ))
expect_equal(or3s(logi_afj, , ),
             bor3(logi_afj, , ))
expect_equal(or3s(logi_afk, , ),
             bor3(logi_afk, , ))
expect_equal(or3s(logi_afl, , q < 0L),
             bor3(logi_afl, , q < 0L))
expect_equal(or3s(logi_afm, , k < 1L),
             bor3(logi_afm, , k < 1L))
expect_equal(or3s(logi_afn, , f < 9L),
             bor3(logi_afn, , f < 9L))
expect_equal(or3s(logi_afo, y < 0L, ),
             bor3(logi_afo, y < 0L, ))
expect_equal(or3s(logi_afp, j < 1L, ),
             bor3(logi_afp, j < 1L, ))
expect_equal(or3s(logi_afq, l < 9L, ),
             bor3(logi_afq, l < 9L, ))
expect_equal(or3s(logi_afr, e < 0L, y < 0L),
             bor3(logi_afr, e < 0L, y < 0L))
expect_equal(or3s(logi_afs, w < 1L, x < 1L),
             bor3(logi_afs, w < 1L, x < 1L))
expect_equal(or3s(logi_aft, y < 9L, c < 9L),
             bor3(logi_aft, y < 9L, c < 9L))
expect_equal(or3s(logi_afu, , ),
             bor3(logi_afu, , ))
expect_equal(or3s(logi_afv, , ),
             bor3(logi_afv, , ))
expect_equal(or3s(logi_afw, , ),
             bor3(logi_afw, , ))
expect_equal(or3s(logi_afx, , o <= 0L),
             bor3(logi_afx, , o <= 0L))
expect_equal(or3s(logi_afy, , r <= 1L),
             bor3(logi_afy, , r <= 1L))
expect_equal(or3s(logi_afz, , p <= 9L),
             bor3(logi_afz, , p <= 9L))
expect_equal(or3s(logi_aga, g < 0L, ),
             bor3(logi_aga, g < 0L, ))
expect_equal(or3s(logi_agb, u < 1L, ),
             bor3(logi_agb, u < 1L, ))
expect_equal(or3s(logi_agc, i < 9L, ),
             bor3(logi_agc, i < 9L, ))
expect_equal(or3s(logi_agd, e < 0L, d <= 0L),
             bor3(logi_agd, e < 0L, d <= 0L))
expect_equal(or3s(logi_age, m < 1L, i <= 1L),
             bor3(logi_age, m < 1L, i <= 1L))
expect_equal(or3s(logi_agf, x < 9L, h <= 9L),
             bor3(logi_agf, x < 9L, h <= 9L))
expect_equal(or3s(logi_agg, , ),
             bor3(logi_agg, , ))
expect_equal(or3s(logi_agh, , ),
             bor3(logi_agh, , ))
expect_equal(or3s(logi_agi, , ),
             bor3(logi_agi, , ))
expect_equal(or3s(logi_agj, , z == 0L),
             bor3(logi_agj, , z == 0L))
expect_equal(or3s(logi_agk, , i == 1L),
             bor3(logi_agk, , i == 1L))
expect_equal(or3s(logi_agl, , o == 9L),
             bor3(logi_agl, , o == 9L))
expect_equal(or3s(logi_agm, h < 0L, ),
             bor3(logi_agm, h < 0L, ))
expect_equal(or3s(logi_agn, q < 1L, ),
             bor3(logi_agn, q < 1L, ))
expect_equal(or3s(logi_ago, j < 9L, ),
             bor3(logi_ago, j < 9L, ))
expect_equal(or3s(logi_agp, i < 0L, u == 0L),
             bor3(logi_agp, i < 0L, u == 0L))
expect_equal(or3s(logi_agq, x < 1L, e == 1L),
             bor3(logi_agq, x < 1L, e == 1L))
expect_equal(or3s(logi_agr, j < 9L, w == 9L),
             bor3(logi_agr, j < 9L, w == 9L))
expect_equal(or3s(logi_ags, , ),
             bor3(logi_ags, , ))
expect_equal(or3s(logi_agt, , ),
             bor3(logi_agt, , ))
expect_equal(or3s(logi_agu, , ),
             bor3(logi_agu, , ))
expect_equal(or3s(logi_agv, , w > 0L),
             bor3(logi_agv, , w > 0L))
expect_equal(or3s(logi_agw, , k > 1L),
             bor3(logi_agw, , k > 1L))
expect_equal(or3s(logi_agx, , n > 9L),
             bor3(logi_agx, , n > 9L))
expect_equal(or3s(logi_agy, j < 0L, ),
             bor3(logi_agy, j < 0L, ))
expect_equal(or3s(logi_agz, z < 1L, ),
             bor3(logi_agz, z < 1L, ))
expect_equal(or3s(logi_aha, e < 9L, ),
             bor3(logi_aha, e < 9L, ))
expect_equal(or3s(logi_ahb, k < 0L, j > 0L),
             bor3(logi_ahb, k < 0L, j > 0L))
expect_equal(or3s(logi_ahc, m < 1L, f > 1L),
             bor3(logi_ahc, m < 1L, f > 1L))
expect_equal(or3s(logi_ahd, g < 9L, l > 9L),
             bor3(logi_ahd, g < 9L, l > 9L))
expect_equal(or3s(logi_ahe, , ),
             bor3(logi_ahe, , ))
expect_equal(or3s(logi_ahf, , ),
             bor3(logi_ahf, , ))
expect_equal(or3s(logi_ahg, , ),
             bor3(logi_ahg, , ))
expect_equal(or3s(logi_ahh, , z >= 0L),
             bor3(logi_ahh, , z >= 0L))
expect_equal(or3s(logi_ahi, , y >= 1L),
             bor3(logi_ahi, , y >= 1L))
expect_equal(or3s(logi_ahj, , j >= 9L),
             bor3(logi_ahj, , j >= 9L))
expect_equal(or3s(logi_ahk, w < 0L, ),
             bor3(logi_ahk, w < 0L, ))
expect_equal(or3s(logi_ahl, e < 1L, ),
             bor3(logi_ahl, e < 1L, ))
expect_equal(or3s(logi_ahm, p < 9L, ),
             bor3(logi_ahm, p < 9L, ))
expect_equal(or3s(logi_ahn, u < 0L, z >= 0L),
             bor3(logi_ahn, u < 0L, z >= 0L))
expect_equal(or3s(logi_aho, z < 1L, y >= 1L),
             bor3(logi_aho, z < 1L, y >= 1L))
expect_equal(or3s(logi_ahp, u < 9L, t >= 9L),
             bor3(logi_ahp, u < 9L, t >= 9L))
expect_equal(or3s(logi_ahq, , ),
             bor3(logi_ahq, , ))
expect_equal(or3s(logi_ahr, , ),
             bor3(logi_ahr, , ))
expect_equal(or3s(logi_ahs, , ),
             bor3(logi_ahs, , ))
expect_equal(or3s(logi_aht, , logi_c),
             bor3(logi_aht, , logi_c))
expect_equal(or3s(logi_ahu, , logi_c),
             bor3(logi_ahu, , logi_c))
expect_equal(or3s(logi_ahv, , logi_u),
             bor3(logi_ahv, , logi_u))
expect_equal(or3s(logi_ahw, i <= 0L, ),
             bor3(logi_ahw, i <= 0L, ))
expect_equal(or3s(logi_ahx, q <= 1L, ),
             bor3(logi_ahx, q <= 1L, ))
expect_equal(or3s(logi_ahy, n <= 9L, ),
             bor3(logi_ahy, n <= 9L, ))
expect_equal(or3s(logi_ahz, p <= 0L, logi_q),
             bor3(logi_ahz, p <= 0L, logi_q))
expect_equal(or3s(logi_aia, n <= 1L, logi_d),
             bor3(logi_aia, n <= 1L, logi_d))
expect_equal(or3s(logi_aib, w <= 9L, logi_c),
             bor3(logi_aib, w <= 9L, logi_c))
expect_equal(or3s(logi_aic, , ),
             bor3(logi_aic, , ))
expect_equal(or3s(logi_aid, , ),
             bor3(logi_aid, , ))
expect_equal(or3s(logi_aie, , ),
             bor3(logi_aie, , ))
expect_equal(or3s(logi_aif, , !logi_k),
             bor3(logi_aif, , !logi_k))
expect_equal(or3s(logi_aig, , !logi_c),
             bor3(logi_aig, , !logi_c))
expect_equal(or3s(logi_aih, , !logi_l),
             bor3(logi_aih, , !logi_l))
expect_equal(or3s(logi_aii, e <= 0L, ),
             bor3(logi_aii, e <= 0L, ))
expect_equal(or3s(logi_aij, a <= 1L, ),
             bor3(logi_aij, a <= 1L, ))
expect_equal(or3s(logi_aik, d <= 9L, ),
             bor3(logi_aik, d <= 9L, ))
expect_equal(or3s(logi_ail, t <= 0L, !logi_q),
             bor3(logi_ail, t <= 0L, !logi_q))
expect_equal(or3s(logi_aim, g <= 1L, !logi_p),
             bor3(logi_aim, g <= 1L, !logi_p))
expect_equal(or3s(logi_ain, f <= 9L, !logi_w),
             bor3(logi_ain, f <= 9L, !logi_w))
expect_equal(or3s(logi_aio, , ),
             bor3(logi_aio, , ))
expect_equal(or3s(logi_aip, , ),
             bor3(logi_aip, , ))
expect_equal(or3s(logi_aiq, , ),
             bor3(logi_aiq, , ))
expect_equal(or3s(logi_air, , n != 0L),
             bor3(logi_air, , n != 0L))
expect_equal(or3s(logi_ais, , s != 1L),
             bor3(logi_ais, , s != 1L))
expect_equal(or3s(logi_ait, , p != 9L),
             bor3(logi_ait, , p != 9L))
expect_equal(or3s(logi_aiu, h <= 0L, ),
             bor3(logi_aiu, h <= 0L, ))
expect_equal(or3s(logi_aiv, d <= 1L, ),
             bor3(logi_aiv, d <= 1L, ))
expect_equal(or3s(logi_aiw, w <= 9L, ),
             bor3(logi_aiw, w <= 9L, ))
expect_equal(or3s(logi_aix, r <= 0L, h != 0L),
             bor3(logi_aix, r <= 0L, h != 0L))
expect_equal(or3s(logi_aiy, g <= 1L, u != 1L),
             bor3(logi_aiy, g <= 1L, u != 1L))
expect_equal(or3s(logi_aiz, p <= 9L, v != 9L),
             bor3(logi_aiz, p <= 9L, v != 9L))
expect_equal(or3s(logi_aja, , ),
             bor3(logi_aja, , ))
expect_equal(or3s(logi_ajb, , ),
             bor3(logi_ajb, , ))
expect_equal(or3s(logi_ajc, , ),
             bor3(logi_ajc, , ))
expect_equal(or3s(logi_ajd, , w %between% c(-1L, 1L)),
             bor3(logi_ajd, , w %between% c(-1L, 1L)))
expect_equal(or3s(logi_aje, , z %between% c(-1L, 1L)),
             bor3(logi_aje, , z %between% c(-1L, 1L)))
expect_equal(or3s(logi_ajf, , q %between% c(-1L, 1L)),
             bor3(logi_ajf, , q %between% c(-1L, 1L)))
expect_equal(or3s(logi_ajg, v <= 0L, ),
             bor3(logi_ajg, v <= 0L, ))
expect_equal(or3s(logi_ajh, i <= 1L, ),
             bor3(logi_ajh, i <= 1L, ))
expect_equal(or3s(logi_aji, n <= 9L, ),
             bor3(logi_aji, n <= 9L, ))
expect_equal(or3s(logi_ajj, q <= 0L, i %between% c(-1L, 1L)),
             bor3(logi_ajj, q <= 0L, i %between% c(-1L, 1L)))
expect_equal(or3s(logi_ajk, d <= 1L, s %between% c(-1L, 1L)),
             bor3(logi_ajk, d <= 1L, s %between% c(-1L, 1L)))
expect_equal(or3s(logi_ajl, p <= 9L, f %between% c(-1L, 1L)),
             bor3(logi_ajl, p <= 9L, f %between% c(-1L, 1L)))
expect_equal(or3s(logi_ajm, , ),
             bor3(logi_ajm, , ))
expect_equal(or3s(logi_ajn, , ),
             bor3(logi_ajn, , ))
expect_equal(or3s(logi_ajo, , ),
             bor3(logi_ajo, , ))
expect_equal(or3s(logi_ajp, , f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ajp, , f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ajq, , x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ajq, , x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ajr, , w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ajr, , w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ajs, v <= 0L, ),
             bor3(logi_ajs, v <= 0L, ))
expect_equal(or3s(logi_ajt, t <= 1L, ),
             bor3(logi_ajt, t <= 1L, ))
expect_equal(or3s(logi_aju, s <= 9L, ),
             bor3(logi_aju, s <= 9L, ))
expect_equal(or3s(logi_ajv, n <= 0L, l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ajv, n <= 0L, l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ajw, a <= 1L, o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ajw, a <= 1L, o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ajx, i <= 9L, b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ajx, i <= 9L, b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ajy, , ),
             bor3(logi_ajy, , ))
expect_equal(or3s(logi_ajz, , ),
             bor3(logi_ajz, , ))
expect_equal(or3s(logi_aka, , ),
             bor3(logi_aka, , ))
expect_equal(or3s(logi_akb, , t %in% 1:4),
             bor3(logi_akb, , t %in% 1:4))
expect_equal(or3s(logi_akc, , g %in% 1:4),
             bor3(logi_akc, , g %in% 1:4))
expect_equal(or3s(logi_akd, , d %in% 1:4),
             bor3(logi_akd, , d %in% 1:4))
expect_equal(or3s(logi_ake, b <= 0L, ),
             bor3(logi_ake, b <= 0L, ))
expect_equal(or3s(logi_akf, y <= 1L, ),
             bor3(logi_akf, y <= 1L, ))
expect_equal(or3s(logi_akg, y <= 9L, ),
             bor3(logi_akg, y <= 9L, ))
expect_equal(or3s(logi_akh, x <= 0L, r %in% 1:4),
             bor3(logi_akh, x <= 0L, r %in% 1:4))
expect_equal(or3s(logi_aki, i <= 1L, a %in% 1:4),
             bor3(logi_aki, i <= 1L, a %in% 1:4))
expect_equal(or3s(logi_akj, d <= 9L, n %in% 1:4),
             bor3(logi_akj, d <= 9L, n %in% 1:4))
expect_equal(or3s(logi_akk, , ),
             bor3(logi_akk, , ))
expect_equal(or3s(logi_akl, , ),
             bor3(logi_akl, , ))
expect_equal(or3s(logi_akm, , ),
             bor3(logi_akm, , ))
expect_equal(or3s(logi_akn, , p < 0L),
             bor3(logi_akn, , p < 0L))
expect_equal(or3s(logi_ako, , w < 1L),
             bor3(logi_ako, , w < 1L))
expect_equal(or3s(logi_akp, , t < 9L),
             bor3(logi_akp, , t < 9L))
expect_equal(or3s(logi_akq, r <= 0L, ),
             bor3(logi_akq, r <= 0L, ))
expect_equal(or3s(logi_akr, w <= 1L, ),
             bor3(logi_akr, w <= 1L, ))
expect_equal(or3s(logi_aks, w <= 9L, ),
             bor3(logi_aks, w <= 9L, ))
expect_equal(or3s(logi_akt, t <= 0L, j < 0L),
             bor3(logi_akt, t <= 0L, j < 0L))
expect_equal(or3s(logi_aku, x <= 1L, b < 1L),
             bor3(logi_aku, x <= 1L, b < 1L))
expect_equal(or3s(logi_akv, e <= 9L, v < 9L),
             bor3(logi_akv, e <= 9L, v < 9L))
expect_equal(or3s(logi_akw, , ),
             bor3(logi_akw, , ))
expect_equal(or3s(logi_akx, , ),
             bor3(logi_akx, , ))
expect_equal(or3s(logi_aky, , ),
             bor3(logi_aky, , ))
expect_equal(or3s(logi_akz, , r <= 0L),
             bor3(logi_akz, , r <= 0L))
expect_equal(or3s(logi_ala, , o <= 1L),
             bor3(logi_ala, , o <= 1L))
expect_equal(or3s(logi_alb, , d <= 9L),
             bor3(logi_alb, , d <= 9L))
expect_equal(or3s(logi_alc, h <= 0L, ),
             bor3(logi_alc, h <= 0L, ))
expect_equal(or3s(logi_ald, a <= 1L, ),
             bor3(logi_ald, a <= 1L, ))
expect_equal(or3s(logi_ale, r <= 9L, ),
             bor3(logi_ale, r <= 9L, ))
expect_equal(or3s(logi_alf, n <= 0L, b <= 0L),
             bor3(logi_alf, n <= 0L, b <= 0L))
expect_equal(or3s(logi_alg, s <= 1L, u <= 1L),
             bor3(logi_alg, s <= 1L, u <= 1L))
expect_equal(or3s(logi_alh, m <= 9L, e <= 9L),
             bor3(logi_alh, m <= 9L, e <= 9L))
expect_equal(or3s(logi_ali, , ),
             bor3(logi_ali, , ))
expect_equal(or3s(logi_alj, , ),
             bor3(logi_alj, , ))
expect_equal(or3s(logi_alk, , ),
             bor3(logi_alk, , ))
expect_equal(or3s(logi_all, , c == 0L),
             bor3(logi_all, , c == 0L))
expect_equal(or3s(logi_alm, , u == 1L),
             bor3(logi_alm, , u == 1L))
expect_equal(or3s(logi_aln, , k == 9L),
             bor3(logi_aln, , k == 9L))
expect_equal(or3s(logi_alo, r <= 0L, ),
             bor3(logi_alo, r <= 0L, ))
expect_equal(or3s(logi_alp, g <= 1L, ),
             bor3(logi_alp, g <= 1L, ))
expect_equal(or3s(logi_alq, m <= 9L, ),
             bor3(logi_alq, m <= 9L, ))
expect_equal(or3s(logi_alr, e <= 0L, x == 0L),
             bor3(logi_alr, e <= 0L, x == 0L))
expect_equal(or3s(logi_als, r <= 1L, a == 1L),
             bor3(logi_als, r <= 1L, a == 1L))
expect_equal(or3s(logi_alt, d <= 9L, h == 9L),
             bor3(logi_alt, d <= 9L, h == 9L))
expect_equal(or3s(logi_alu, , ),
             bor3(logi_alu, , ))
expect_equal(or3s(logi_alv, , ),
             bor3(logi_alv, , ))
expect_equal(or3s(logi_alw, , ),
             bor3(logi_alw, , ))
expect_equal(or3s(logi_alx, , h > 0L),
             bor3(logi_alx, , h > 0L))
expect_equal(or3s(logi_aly, , q > 1L),
             bor3(logi_aly, , q > 1L))
expect_equal(or3s(logi_alz, , c > 9L),
             bor3(logi_alz, , c > 9L))
expect_equal(or3s(logi_ama, e <= 0L, ),
             bor3(logi_ama, e <= 0L, ))
expect_equal(or3s(logi_amb, k <= 1L, ),
             bor3(logi_amb, k <= 1L, ))
expect_equal(or3s(logi_amc, d <= 9L, ),
             bor3(logi_amc, d <= 9L, ))
expect_equal(or3s(logi_amd, f <= 0L, z > 0L),
             bor3(logi_amd, f <= 0L, z > 0L))
expect_equal(or3s(logi_ame, x <= 1L, u > 1L),
             bor3(logi_ame, x <= 1L, u > 1L))
expect_equal(or3s(logi_amf, l <= 9L, b > 9L),
             bor3(logi_amf, l <= 9L, b > 9L))
expect_equal(or3s(logi_amg, , ),
             bor3(logi_amg, , ))
expect_equal(or3s(logi_amh, , ),
             bor3(logi_amh, , ))
expect_equal(or3s(logi_ami, , ),
             bor3(logi_ami, , ))
expect_equal(or3s(logi_amj, , q >= 0L),
             bor3(logi_amj, , q >= 0L))
expect_equal(or3s(logi_amk, , a >= 1L),
             bor3(logi_amk, , a >= 1L))
expect_equal(or3s(logi_aml, , p >= 9L),
             bor3(logi_aml, , p >= 9L))
expect_equal(or3s(logi_amm, k <= 0L, ),
             bor3(logi_amm, k <= 0L, ))
expect_equal(or3s(logi_amn, g <= 1L, ),
             bor3(logi_amn, g <= 1L, ))
expect_equal(or3s(logi_amo, e <= 9L, ),
             bor3(logi_amo, e <= 9L, ))
expect_equal(or3s(logi_amp, r <= 0L, e >= 0L),
             bor3(logi_amp, r <= 0L, e >= 0L))
expect_equal(or3s(logi_amq, x <= 1L, y >= 1L),
             bor3(logi_amq, x <= 1L, y >= 1L))
expect_equal(or3s(logi_amr, v <= 9L, u >= 9L),
             bor3(logi_amr, v <= 9L, u >= 9L))
expect_equal(or3s(logi_ams, , ),
             bor3(logi_ams, , ))
expect_equal(or3s(logi_amt, , ),
             bor3(logi_amt, , ))
expect_equal(or3s(logi_amu, , ),
             bor3(logi_amu, , ))
expect_equal(or3s(logi_amv, , logi_c),
             bor3(logi_amv, , logi_c))
expect_equal(or3s(logi_amw, , logi_j),
             bor3(logi_amw, , logi_j))
expect_equal(or3s(logi_amx, , logi_i),
             bor3(logi_amx, , logi_i))
expect_equal(or3s(logi_amy, u == 0L, ),
             bor3(logi_amy, u == 0L, ))
expect_equal(or3s(logi_amz, m == 1L, ),
             bor3(logi_amz, m == 1L, ))
expect_equal(or3s(logi_ana, p == 9L, ),
             bor3(logi_ana, p == 9L, ))
expect_equal(or3s(logi_anb, d == 0L, logi_t),
             bor3(logi_anb, d == 0L, logi_t))
expect_equal(or3s(logi_anc, y == 1L, logi_r),
             bor3(logi_anc, y == 1L, logi_r))
expect_equal(or3s(logi_and, p == 9L, logi_e),
             bor3(logi_and, p == 9L, logi_e))
expect_equal(or3s(logi_ane, , ),
             bor3(logi_ane, , ))
expect_equal(or3s(logi_anf, , ),
             bor3(logi_anf, , ))
expect_equal(or3s(logi_ang, , ),
             bor3(logi_ang, , ))
expect_equal(or3s(logi_anh, , !logi_s),
             bor3(logi_anh, , !logi_s))
expect_equal(or3s(logi_ani, , !logi_w),
             bor3(logi_ani, , !logi_w))
expect_equal(or3s(logi_anj, , !logi_y),
             bor3(logi_anj, , !logi_y))
expect_equal(or3s(logi_ank, e == 0L, ),
             bor3(logi_ank, e == 0L, ))
expect_equal(or3s(logi_anl, m == 1L, ),
             bor3(logi_anl, m == 1L, ))
expect_equal(or3s(logi_anm, p == 9L, ),
             bor3(logi_anm, p == 9L, ))
expect_equal(or3s(logi_ann, f == 0L, !logi_b),
             bor3(logi_ann, f == 0L, !logi_b))
expect_equal(or3s(logi_ano, p == 1L, !logi_l),
             bor3(logi_ano, p == 1L, !logi_l))
expect_equal(or3s(logi_anp, w == 9L, !logi_w),
             bor3(logi_anp, w == 9L, !logi_w))
expect_equal(or3s(logi_anq, , ),
             bor3(logi_anq, , ))
expect_equal(or3s(logi_anr, , ),
             bor3(logi_anr, , ))
expect_equal(or3s(logi_ans, , ),
             bor3(logi_ans, , ))
expect_equal(or3s(logi_ant, , f != 0L),
             bor3(logi_ant, , f != 0L))
expect_equal(or3s(logi_anu, , h != 1L),
             bor3(logi_anu, , h != 1L))
expect_equal(or3s(logi_anv, , r != 9L),
             bor3(logi_anv, , r != 9L))
expect_equal(or3s(logi_anw, v == 0L, ),
             bor3(logi_anw, v == 0L, ))
expect_equal(or3s(logi_anx, p == 1L, ),
             bor3(logi_anx, p == 1L, ))
expect_equal(or3s(logi_any, k == 9L, ),
             bor3(logi_any, k == 9L, ))
expect_equal(or3s(logi_anz, o == 0L, p != 0L),
             bor3(logi_anz, o == 0L, p != 0L))
expect_equal(or3s(logi_aoa, y == 1L, i != 1L),
             bor3(logi_aoa, y == 1L, i != 1L))
expect_equal(or3s(logi_aob, t == 9L, h != 9L),
             bor3(logi_aob, t == 9L, h != 9L))
expect_equal(or3s(logi_aoc, , ),
             bor3(logi_aoc, , ))
expect_equal(or3s(logi_aod, , ),
             bor3(logi_aod, , ))
expect_equal(or3s(logi_aoe, , ),
             bor3(logi_aoe, , ))
expect_equal(or3s(logi_aof, , x %between% c(-1L, 1L)),
             bor3(logi_aof, , x %between% c(-1L, 1L)))
expect_equal(or3s(logi_aog, , t %between% c(-1L, 1L)),
             bor3(logi_aog, , t %between% c(-1L, 1L)))
expect_equal(or3s(logi_aoh, , h %between% c(-1L, 1L)),
             bor3(logi_aoh, , h %between% c(-1L, 1L)))
expect_equal(or3s(logi_aoi, j == 0L, ),
             bor3(logi_aoi, j == 0L, ))
expect_equal(or3s(logi_aoj, h == 1L, ),
             bor3(logi_aoj, h == 1L, ))
expect_equal(or3s(logi_aok, r == 9L, ),
             bor3(logi_aok, r == 9L, ))
expect_equal(or3s(logi_aol, f == 0L, c %between% c(-1L, 1L)),
             bor3(logi_aol, f == 0L, c %between% c(-1L, 1L)))
expect_equal(or3s(logi_aom, f == 1L, c %between% c(-1L, 1L)),
             bor3(logi_aom, f == 1L, c %between% c(-1L, 1L)))
expect_equal(or3s(logi_aon, q == 9L, k %between% c(-1L, 1L)),
             bor3(logi_aon, q == 9L, k %between% c(-1L, 1L)))
expect_equal(or3s(logi_aoo, , ),
             bor3(logi_aoo, , ))
expect_equal(or3s(logi_aop, , ),
             bor3(logi_aop, , ))
expect_equal(or3s(logi_aoq, , ),
             bor3(logi_aoq, , ))
expect_equal(or3s(logi_aor, , a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aor, , a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aos, , r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aos, , r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aot, , s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aot, , s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aou, o == 0L, ),
             bor3(logi_aou, o == 0L, ))
expect_equal(or3s(logi_aov, p == 1L, ),
             bor3(logi_aov, p == 1L, ))
expect_equal(or3s(logi_aow, k == 9L, ),
             bor3(logi_aow, k == 9L, ))
expect_equal(or3s(logi_aox, z == 0L, j %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aox, z == 0L, j %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aoy, n == 1L, k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aoy, n == 1L, k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aoz, m == 9L, u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aoz, m == 9L, u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_apa, , ),
             bor3(logi_apa, , ))
expect_equal(or3s(logi_apb, , ),
             bor3(logi_apb, , ))
expect_equal(or3s(logi_apc, , ),
             bor3(logi_apc, , ))
expect_equal(or3s(logi_apd, , d %in% 1:4),
             bor3(logi_apd, , d %in% 1:4))
expect_equal(or3s(logi_ape, , w %in% 1:4),
             bor3(logi_ape, , w %in% 1:4))
expect_equal(or3s(logi_apf, , t %in% 1:4),
             bor3(logi_apf, , t %in% 1:4))
expect_equal(or3s(logi_apg, u == 0L, ),
             bor3(logi_apg, u == 0L, ))
expect_equal(or3s(logi_aph, m == 1L, ),
             bor3(logi_aph, m == 1L, ))
expect_equal(or3s(logi_api, o == 9L, ),
             bor3(logi_api, o == 9L, ))
expect_equal(or3s(logi_apj, a == 0L, z %in% 1:4),
             bor3(logi_apj, a == 0L, z %in% 1:4))
expect_equal(or3s(logi_apk, m == 1L, n %in% 1:4),
             bor3(logi_apk, m == 1L, n %in% 1:4))
expect_equal(or3s(logi_apl, d == 9L, d %in% 1:4),
             bor3(logi_apl, d == 9L, d %in% 1:4))
expect_equal(or3s(logi_apm, , ),
             bor3(logi_apm, , ))
expect_equal(or3s(logi_apn, , ),
             bor3(logi_apn, , ))
expect_equal(or3s(logi_apo, , ),
             bor3(logi_apo, , ))
expect_equal(or3s(logi_app, , i < 0L),
             bor3(logi_app, , i < 0L))
expect_equal(or3s(logi_apq, , k < 1L),
             bor3(logi_apq, , k < 1L))
expect_equal(or3s(logi_apr, , t < 9L),
             bor3(logi_apr, , t < 9L))
expect_equal(or3s(logi_aps, a == 0L, ),
             bor3(logi_aps, a == 0L, ))
expect_equal(or3s(logi_apt, h == 1L, ),
             bor3(logi_apt, h == 1L, ))
expect_equal(or3s(logi_apu, g == 9L, ),
             bor3(logi_apu, g == 9L, ))
expect_equal(or3s(logi_apv, i == 0L, t < 0L),
             bor3(logi_apv, i == 0L, t < 0L))
expect_equal(or3s(logi_apw, a == 1L, b < 1L),
             bor3(logi_apw, a == 1L, b < 1L))
expect_equal(or3s(logi_apx, y == 9L, n < 9L),
             bor3(logi_apx, y == 9L, n < 9L))
expect_equal(or3s(logi_apy, , ),
             bor3(logi_apy, , ))
expect_equal(or3s(logi_apz, , ),
             bor3(logi_apz, , ))
expect_equal(or3s(logi_aqa, , ),
             bor3(logi_aqa, , ))
expect_equal(or3s(logi_aqb, , l <= 0L),
             bor3(logi_aqb, , l <= 0L))
expect_equal(or3s(logi_aqc, , l <= 1L),
             bor3(logi_aqc, , l <= 1L))
expect_equal(or3s(logi_aqd, , r <= 9L),
             bor3(logi_aqd, , r <= 9L))
expect_equal(or3s(logi_aqe, r == 0L, ),
             bor3(logi_aqe, r == 0L, ))
expect_equal(or3s(logi_aqf, k == 1L, ),
             bor3(logi_aqf, k == 1L, ))
expect_equal(or3s(logi_aqg, s == 9L, ),
             bor3(logi_aqg, s == 9L, ))
expect_equal(or3s(logi_aqh, u == 0L, i <= 0L),
             bor3(logi_aqh, u == 0L, i <= 0L))
expect_equal(or3s(logi_aqi, c == 1L, j <= 1L),
             bor3(logi_aqi, c == 1L, j <= 1L))
expect_equal(or3s(logi_aqj, n == 9L, n <= 9L),
             bor3(logi_aqj, n == 9L, n <= 9L))
expect_equal(or3s(logi_aqk, , ),
             bor3(logi_aqk, , ))
expect_equal(or3s(logi_aql, , ),
             bor3(logi_aql, , ))
expect_equal(or3s(logi_aqm, , ),
             bor3(logi_aqm, , ))
expect_equal(or3s(logi_aqn, , z == 0L),
             bor3(logi_aqn, , z == 0L))
expect_equal(or3s(logi_aqo, , n == 1L),
             bor3(logi_aqo, , n == 1L))
expect_equal(or3s(logi_aqp, , l == 9L),
             bor3(logi_aqp, , l == 9L))
expect_equal(or3s(logi_aqq, p == 0L, ),
             bor3(logi_aqq, p == 0L, ))
expect_equal(or3s(logi_aqr, s == 1L, ),
             bor3(logi_aqr, s == 1L, ))
expect_equal(or3s(logi_aqs, b == 9L, ),
             bor3(logi_aqs, b == 9L, ))
expect_equal(or3s(logi_aqt, u == 0L, x == 0L),
             bor3(logi_aqt, u == 0L, x == 0L))
expect_equal(or3s(logi_aqu, n == 1L, j == 1L),
             bor3(logi_aqu, n == 1L, j == 1L))
expect_equal(or3s(logi_aqv, v == 9L, h == 9L),
             bor3(logi_aqv, v == 9L, h == 9L))
expect_equal(or3s(logi_aqw, , ),
             bor3(logi_aqw, , ))
expect_equal(or3s(logi_aqx, , ),
             bor3(logi_aqx, , ))
expect_equal(or3s(logi_aqy, , ),
             bor3(logi_aqy, , ))
expect_equal(or3s(logi_aqz, , w > 0L),
             bor3(logi_aqz, , w > 0L))
expect_equal(or3s(logi_ara, , p > 1L),
             bor3(logi_ara, , p > 1L))
expect_equal(or3s(logi_arb, , l > 9L),
             bor3(logi_arb, , l > 9L))
expect_equal(or3s(logi_arc, o == 0L, ),
             bor3(logi_arc, o == 0L, ))
expect_equal(or3s(logi_ard, l == 1L, ),
             bor3(logi_ard, l == 1L, ))
expect_equal(or3s(logi_are, h == 9L, ),
             bor3(logi_are, h == 9L, ))
expect_equal(or3s(logi_arf, h == 0L, r > 0L),
             bor3(logi_arf, h == 0L, r > 0L))
expect_equal(or3s(logi_arg, h == 1L, y > 1L),
             bor3(logi_arg, h == 1L, y > 1L))
expect_equal(or3s(logi_arh, g == 9L, q > 9L),
             bor3(logi_arh, g == 9L, q > 9L))
expect_equal(or3s(logi_ari, , ),
             bor3(logi_ari, , ))
expect_equal(or3s(logi_arj, , ),
             bor3(logi_arj, , ))
expect_equal(or3s(logi_ark, , ),
             bor3(logi_ark, , ))
expect_equal(or3s(logi_arl, , h >= 0L),
             bor3(logi_arl, , h >= 0L))
expect_equal(or3s(logi_arm, , a >= 1L),
             bor3(logi_arm, , a >= 1L))
expect_equal(or3s(logi_arn, , g >= 9L),
             bor3(logi_arn, , g >= 9L))
expect_equal(or3s(logi_aro, b == 0L, ),
             bor3(logi_aro, b == 0L, ))
expect_equal(or3s(logi_arp, a == 1L, ),
             bor3(logi_arp, a == 1L, ))
expect_equal(or3s(logi_arq, r == 9L, ),
             bor3(logi_arq, r == 9L, ))
expect_equal(or3s(logi_arr, s == 0L, k >= 0L),
             bor3(logi_arr, s == 0L, k >= 0L))
expect_equal(or3s(logi_ars, h == 1L, q >= 1L),
             bor3(logi_ars, h == 1L, q >= 1L))
expect_equal(or3s(logi_art, h == 9L, c >= 9L),
             bor3(logi_art, h == 9L, c >= 9L))
expect_equal(or3s(logi_aru, , ),
             bor3(logi_aru, , ))
expect_equal(or3s(logi_arv, , ),
             bor3(logi_arv, , ))
expect_equal(or3s(logi_arw, , ),
             bor3(logi_arw, , ))
expect_equal(or3s(logi_arx, , logi_x),
             bor3(logi_arx, , logi_x))
expect_equal(or3s(logi_ary, , logi_b),
             bor3(logi_ary, , logi_b))
expect_equal(or3s(logi_arz, , logi_q),
             bor3(logi_arz, , logi_q))
expect_equal(or3s(logi_asa, b > 0L, ),
             bor3(logi_asa, b > 0L, ))
expect_equal(or3s(logi_asb, i > 1L, ),
             bor3(logi_asb, i > 1L, ))
expect_equal(or3s(logi_asc, f > 9L, ),
             bor3(logi_asc, f > 9L, ))
expect_equal(or3s(logi_asd, s > 0L, logi_k),
             bor3(logi_asd, s > 0L, logi_k))
expect_equal(or3s(logi_ase, d > 1L, logi_u),
             bor3(logi_ase, d > 1L, logi_u))
expect_equal(or3s(logi_asf, v > 9L, logi_g),
             bor3(logi_asf, v > 9L, logi_g))
expect_equal(or3s(logi_asg, , ),
             bor3(logi_asg, , ))
expect_equal(or3s(logi_ash, , ),
             bor3(logi_ash, , ))
expect_equal(or3s(logi_asi, , ),
             bor3(logi_asi, , ))
expect_equal(or3s(logi_asj, , !logi_l),
             bor3(logi_asj, , !logi_l))
expect_equal(or3s(logi_ask, , !logi_z),
             bor3(logi_ask, , !logi_z))
expect_equal(or3s(logi_asl, , !logi_t),
             bor3(logi_asl, , !logi_t))
expect_equal(or3s(logi_asm, c > 0L, ),
             bor3(logi_asm, c > 0L, ))
expect_equal(or3s(logi_asn, m > 1L, ),
             bor3(logi_asn, m > 1L, ))
expect_equal(or3s(logi_aso, t > 9L, ),
             bor3(logi_aso, t > 9L, ))
expect_equal(or3s(logi_asp, f > 0L, !logi_n),
             bor3(logi_asp, f > 0L, !logi_n))
expect_equal(or3s(logi_asq, i > 1L, !logi_p),
             bor3(logi_asq, i > 1L, !logi_p))
expect_equal(or3s(logi_asr, i > 9L, !logi_s),
             bor3(logi_asr, i > 9L, !logi_s))
expect_equal(or3s(logi_ass, , ),
             bor3(logi_ass, , ))
expect_equal(or3s(logi_ast, , ),
             bor3(logi_ast, , ))
expect_equal(or3s(logi_asu, , ),
             bor3(logi_asu, , ))
expect_equal(or3s(logi_asv, , o != 0L),
             bor3(logi_asv, , o != 0L))
expect_equal(or3s(logi_asw, , k != 1L),
             bor3(logi_asw, , k != 1L))
expect_equal(or3s(logi_asx, , b != 9L),
             bor3(logi_asx, , b != 9L))
expect_equal(or3s(logi_asy, w > 0L, ),
             bor3(logi_asy, w > 0L, ))
expect_equal(or3s(logi_asz, r > 1L, ),
             bor3(logi_asz, r > 1L, ))
expect_equal(or3s(logi_ata, v > 9L, ),
             bor3(logi_ata, v > 9L, ))
expect_equal(or3s(logi_atb, b > 0L, i != 0L),
             bor3(logi_atb, b > 0L, i != 0L))
expect_equal(or3s(logi_atc, u > 1L, b != 1L),
             bor3(logi_atc, u > 1L, b != 1L))
expect_equal(or3s(logi_atd, k > 9L, m != 9L),
             bor3(logi_atd, k > 9L, m != 9L))
expect_equal(or3s(logi_ate, , ),
             bor3(logi_ate, , ))
expect_equal(or3s(logi_atf, , ),
             bor3(logi_atf, , ))
expect_equal(or3s(logi_atg, , ),
             bor3(logi_atg, , ))
expect_equal(or3s(logi_ath, , d %between% c(-1L, 1L)),
             bor3(logi_ath, , d %between% c(-1L, 1L)))
expect_equal(or3s(logi_ati, , y %between% c(-1L, 1L)),
             bor3(logi_ati, , y %between% c(-1L, 1L)))
expect_equal(or3s(logi_atj, , s %between% c(-1L, 1L)),
             bor3(logi_atj, , s %between% c(-1L, 1L)))
expect_equal(or3s(logi_atk, m > 0L, ),
             bor3(logi_atk, m > 0L, ))
expect_equal(or3s(logi_atl, d > 1L, ),
             bor3(logi_atl, d > 1L, ))
expect_equal(or3s(logi_atm, u > 9L, ),
             bor3(logi_atm, u > 9L, ))
expect_equal(or3s(logi_atn, n > 0L, g %between% c(-1L, 1L)),
             bor3(logi_atn, n > 0L, g %between% c(-1L, 1L)))
expect_equal(or3s(logi_ato, w > 1L, z %between% c(-1L, 1L)),
             bor3(logi_ato, w > 1L, z %between% c(-1L, 1L)))
expect_equal(or3s(logi_atp, n > 9L, x %between% c(-1L, 1L)),
             bor3(logi_atp, n > 9L, x %between% c(-1L, 1L)))
expect_equal(or3s(logi_atq, , ),
             bor3(logi_atq, , ))
expect_equal(or3s(logi_atr, , ),
             bor3(logi_atr, , ))
expect_equal(or3s(logi_ats, , ),
             bor3(logi_ats, , ))
expect_equal(or3s(logi_att, , d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_att, , d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_atu, , m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_atu, , m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_atv, , x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_atv, , x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_atw, r > 0L, ),
             bor3(logi_atw, r > 0L, ))
expect_equal(or3s(logi_atx, c > 1L, ),
             bor3(logi_atx, c > 1L, ))
expect_equal(or3s(logi_aty, i > 9L, ),
             bor3(logi_aty, i > 9L, ))
expect_equal(or3s(logi_atz, l > 0L, g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_atz, l > 0L, g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aua, p > 1L, f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aua, p > 1L, f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_aub, n > 9L, z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_aub, n > 9L, z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_auc, , ),
             bor3(logi_auc, , ))
expect_equal(or3s(logi_aud, , ),
             bor3(logi_aud, , ))
expect_equal(or3s(logi_aue, , ),
             bor3(logi_aue, , ))
expect_equal(or3s(logi_auf, , e %in% 1:4),
             bor3(logi_auf, , e %in% 1:4))
expect_equal(or3s(logi_aug, , f %in% 1:4),
             bor3(logi_aug, , f %in% 1:4))
expect_equal(or3s(logi_auh, , t %in% 1:4),
             bor3(logi_auh, , t %in% 1:4))
expect_equal(or3s(logi_aui, o > 0L, ),
             bor3(logi_aui, o > 0L, ))
expect_equal(or3s(logi_auj, x > 1L, ),
             bor3(logi_auj, x > 1L, ))
expect_equal(or3s(logi_auk, c > 9L, ),
             bor3(logi_auk, c > 9L, ))
expect_equal(or3s(logi_aul, x > 0L, c %in% 1:4),
             bor3(logi_aul, x > 0L, c %in% 1:4))
expect_equal(or3s(logi_aum, o > 1L, e %in% 1:4),
             bor3(logi_aum, o > 1L, e %in% 1:4))
expect_equal(or3s(logi_aun, w > 9L, b %in% 1:4),
             bor3(logi_aun, w > 9L, b %in% 1:4))
expect_equal(or3s(logi_auo, , ),
             bor3(logi_auo, , ))
expect_equal(or3s(logi_aup, , ),
             bor3(logi_aup, , ))
expect_equal(or3s(logi_auq, , ),
             bor3(logi_auq, , ))
expect_equal(or3s(logi_aur, , t < 0L),
             bor3(logi_aur, , t < 0L))
expect_equal(or3s(logi_aus, , s < 1L),
             bor3(logi_aus, , s < 1L))
expect_equal(or3s(logi_aut, , h < 9L),
             bor3(logi_aut, , h < 9L))
expect_equal(or3s(logi_auu, d > 0L, ),
             bor3(logi_auu, d > 0L, ))
expect_equal(or3s(logi_auv, q > 1L, ),
             bor3(logi_auv, q > 1L, ))
expect_equal(or3s(logi_auw, y > 9L, ),
             bor3(logi_auw, y > 9L, ))
expect_equal(or3s(logi_aux, i > 0L, s < 0L),
             bor3(logi_aux, i > 0L, s < 0L))
expect_equal(or3s(logi_auy, f > 1L, t < 1L),
             bor3(logi_auy, f > 1L, t < 1L))
expect_equal(or3s(logi_auz, n > 9L, m < 9L),
             bor3(logi_auz, n > 9L, m < 9L))
expect_equal(or3s(logi_ava, , ),
             bor3(logi_ava, , ))
expect_equal(or3s(logi_avb, , ),
             bor3(logi_avb, , ))
expect_equal(or3s(logi_avc, , ),
             bor3(logi_avc, , ))
expect_equal(or3s(logi_avd, , m <= 0L),
             bor3(logi_avd, , m <= 0L))
expect_equal(or3s(logi_ave, , m <= 1L),
             bor3(logi_ave, , m <= 1L))
expect_equal(or3s(logi_avf, , v <= 9L),
             bor3(logi_avf, , v <= 9L))
expect_equal(or3s(logi_avg, h > 0L, ),
             bor3(logi_avg, h > 0L, ))
expect_equal(or3s(logi_avh, b > 1L, ),
             bor3(logi_avh, b > 1L, ))
expect_equal(or3s(logi_avi, x > 9L, ),
             bor3(logi_avi, x > 9L, ))
expect_equal(or3s(logi_avj, z > 0L, m <= 0L),
             bor3(logi_avj, z > 0L, m <= 0L))
expect_equal(or3s(logi_avk, m > 1L, b <= 1L),
             bor3(logi_avk, m > 1L, b <= 1L))
expect_equal(or3s(logi_avl, v > 9L, d <= 9L),
             bor3(logi_avl, v > 9L, d <= 9L))
expect_equal(or3s(logi_avm, , ),
             bor3(logi_avm, , ))
expect_equal(or3s(logi_avn, , ),
             bor3(logi_avn, , ))
expect_equal(or3s(logi_avo, , ),
             bor3(logi_avo, , ))
expect_equal(or3s(logi_avp, , e == 0L),
             bor3(logi_avp, , e == 0L))
expect_equal(or3s(logi_avq, , g == 1L),
             bor3(logi_avq, , g == 1L))
expect_equal(or3s(logi_avr, , n == 9L),
             bor3(logi_avr, , n == 9L))
expect_equal(or3s(logi_avs, r > 0L, ),
             bor3(logi_avs, r > 0L, ))
expect_equal(or3s(logi_avt, x > 1L, ),
             bor3(logi_avt, x > 1L, ))
expect_equal(or3s(logi_avu, k > 9L, ),
             bor3(logi_avu, k > 9L, ))
expect_equal(or3s(logi_avv, e > 0L, p == 0L),
             bor3(logi_avv, e > 0L, p == 0L))
expect_equal(or3s(logi_avw, g > 1L, p == 1L),
             bor3(logi_avw, g > 1L, p == 1L))
expect_equal(or3s(logi_avx, p > 9L, m == 9L),
             bor3(logi_avx, p > 9L, m == 9L))
expect_equal(or3s(logi_avy, , ),
             bor3(logi_avy, , ))
expect_equal(or3s(logi_avz, , ),
             bor3(logi_avz, , ))
expect_equal(or3s(logi_awa, , ),
             bor3(logi_awa, , ))
expect_equal(or3s(logi_awb, , l > 0L),
             bor3(logi_awb, , l > 0L))
expect_equal(or3s(logi_awc, , z > 1L),
             bor3(logi_awc, , z > 1L))
expect_equal(or3s(logi_awd, , g > 9L),
             bor3(logi_awd, , g > 9L))
expect_equal(or3s(logi_awe, g > 0L, ),
             bor3(logi_awe, g > 0L, ))
expect_equal(or3s(logi_awf, v > 1L, ),
             bor3(logi_awf, v > 1L, ))
expect_equal(or3s(logi_awg, v > 9L, ),
             bor3(logi_awg, v > 9L, ))
expect_equal(or3s(logi_awh, p > 0L, q > 0L),
             bor3(logi_awh, p > 0L, q > 0L))
expect_equal(or3s(logi_awi, v > 1L, t > 1L),
             bor3(logi_awi, v > 1L, t > 1L))
expect_equal(or3s(logi_awj, y > 9L, y > 9L),
             bor3(logi_awj, y > 9L, y > 9L))
expect_equal(or3s(logi_awk, , ),
             bor3(logi_awk, , ))
expect_equal(or3s(logi_awl, , ),
             bor3(logi_awl, , ))
expect_equal(or3s(logi_awm, , ),
             bor3(logi_awm, , ))
expect_equal(or3s(logi_awn, , k >= 0L),
             bor3(logi_awn, , k >= 0L))
expect_equal(or3s(logi_awo, , r >= 1L),
             bor3(logi_awo, , r >= 1L))
expect_equal(or3s(logi_awp, , t >= 9L),
             bor3(logi_awp, , t >= 9L))
expect_equal(or3s(logi_awq, f > 0L, ),
             bor3(logi_awq, f > 0L, ))
expect_equal(or3s(logi_awr, o > 1L, ),
             bor3(logi_awr, o > 1L, ))
expect_equal(or3s(logi_aws, w > 9L, ),
             bor3(logi_aws, w > 9L, ))
expect_equal(or3s(logi_awt, i > 0L, a >= 0L),
             bor3(logi_awt, i > 0L, a >= 0L))
expect_equal(or3s(logi_awu, b > 1L, t >= 1L),
             bor3(logi_awu, b > 1L, t >= 1L))
expect_equal(or3s(logi_awv, z > 9L, s >= 9L),
             bor3(logi_awv, z > 9L, s >= 9L))
expect_equal(or3s(logi_aww, , ),
             bor3(logi_aww, , ))
expect_equal(or3s(logi_awx, , ),
             bor3(logi_awx, , ))
expect_equal(or3s(logi_awy, , ),
             bor3(logi_awy, , ))
expect_equal(or3s(logi_awz, , logi_h),
             bor3(logi_awz, , logi_h))
expect_equal(or3s(logi_axa, , logi_s),
             bor3(logi_axa, , logi_s))
expect_equal(or3s(logi_axb, , logi_j),
             bor3(logi_axb, , logi_j))
expect_equal(or3s(logi_axc, b >= 0L, ),
             bor3(logi_axc, b >= 0L, ))
expect_equal(or3s(logi_axd, p >= 1L, ),
             bor3(logi_axd, p >= 1L, ))
expect_equal(or3s(logi_axe, u >= 9L, ),
             bor3(logi_axe, u >= 9L, ))
expect_equal(or3s(logi_axf, c >= 0L, logi_l),
             bor3(logi_axf, c >= 0L, logi_l))
expect_equal(or3s(logi_axg, i >= 1L, logi_r),
             bor3(logi_axg, i >= 1L, logi_r))
expect_equal(or3s(logi_axh, p >= 9L, logi_k),
             bor3(logi_axh, p >= 9L, logi_k))
expect_equal(or3s(logi_axi, , ),
             bor3(logi_axi, , ))
expect_equal(or3s(logi_axj, , ),
             bor3(logi_axj, , ))
expect_equal(or3s(logi_axk, , ),
             bor3(logi_axk, , ))
expect_equal(or3s(logi_axl, , !logi_p),
             bor3(logi_axl, , !logi_p))
expect_equal(or3s(logi_axm, , !logi_i),
             bor3(logi_axm, , !logi_i))
expect_equal(or3s(logi_axn, , !logi_b),
             bor3(logi_axn, , !logi_b))
expect_equal(or3s(logi_axo, e >= 0L, ),
             bor3(logi_axo, e >= 0L, ))
expect_equal(or3s(logi_axp, d >= 1L, ),
             bor3(logi_axp, d >= 1L, ))
expect_equal(or3s(logi_axq, y >= 9L, ),
             bor3(logi_axq, y >= 9L, ))
expect_equal(or3s(logi_axr, b >= 0L, !logi_o),
             bor3(logi_axr, b >= 0L, !logi_o))
expect_equal(or3s(logi_axs, i >= 1L, !logi_z),
             bor3(logi_axs, i >= 1L, !logi_z))
expect_equal(or3s(logi_axt, x >= 9L, !logi_f),
             bor3(logi_axt, x >= 9L, !logi_f))
expect_equal(or3s(logi_axu, , ),
             bor3(logi_axu, , ))
expect_equal(or3s(logi_axv, , ),
             bor3(logi_axv, , ))
expect_equal(or3s(logi_axw, , ),
             bor3(logi_axw, , ))
expect_equal(or3s(logi_axx, , u != 0L),
             bor3(logi_axx, , u != 0L))
expect_equal(or3s(logi_axy, , u != 1L),
             bor3(logi_axy, , u != 1L))
expect_equal(or3s(logi_axz, , r != 9L),
             bor3(logi_axz, , r != 9L))
expect_equal(or3s(logi_aya, g >= 0L, ),
             bor3(logi_aya, g >= 0L, ))
expect_equal(or3s(logi_ayb, x >= 1L, ),
             bor3(logi_ayb, x >= 1L, ))
expect_equal(or3s(logi_ayc, b >= 9L, ),
             bor3(logi_ayc, b >= 9L, ))
expect_equal(or3s(logi_ayd, i >= 0L, x != 0L),
             bor3(logi_ayd, i >= 0L, x != 0L))
expect_equal(or3s(logi_aye, e >= 1L, v != 1L),
             bor3(logi_aye, e >= 1L, v != 1L))
expect_equal(or3s(logi_ayf, x >= 9L, r != 9L),
             bor3(logi_ayf, x >= 9L, r != 9L))
expect_equal(or3s(logi_ayg, , ),
             bor3(logi_ayg, , ))
expect_equal(or3s(logi_ayh, , ),
             bor3(logi_ayh, , ))
expect_equal(or3s(logi_ayi, , ),
             bor3(logi_ayi, , ))
expect_equal(or3s(logi_ayj, , e %between% c(-1L, 1L)),
             bor3(logi_ayj, , e %between% c(-1L, 1L)))
expect_equal(or3s(logi_ayk, , l %between% c(-1L, 1L)),
             bor3(logi_ayk, , l %between% c(-1L, 1L)))
expect_equal(or3s(logi_ayl, , b %between% c(-1L, 1L)),
             bor3(logi_ayl, , b %between% c(-1L, 1L)))
expect_equal(or3s(logi_aym, l >= 0L, ),
             bor3(logi_aym, l >= 0L, ))
expect_equal(or3s(logi_ayn, t >= 1L, ),
             bor3(logi_ayn, t >= 1L, ))
expect_equal(or3s(logi_ayo, n >= 9L, ),
             bor3(logi_ayo, n >= 9L, ))
expect_equal(or3s(logi_ayp, l >= 0L, r %between% c(-1L, 1L)),
             bor3(logi_ayp, l >= 0L, r %between% c(-1L, 1L)))
expect_equal(or3s(logi_ayq, m >= 1L, j %between% c(-1L, 1L)),
             bor3(logi_ayq, m >= 1L, j %between% c(-1L, 1L)))
expect_equal(or3s(logi_ayr, q >= 9L, q %between% c(-1L, 1L)),
             bor3(logi_ayr, q >= 9L, q %between% c(-1L, 1L)))
expect_equal(or3s(logi_ays, , ),
             bor3(logi_ays, , ))
expect_equal(or3s(logi_ayt, , ),
             bor3(logi_ayt, , ))
expect_equal(or3s(logi_ayu, , ),
             bor3(logi_ayu, , ))
expect_equal(or3s(logi_ayv, , i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ayv, , i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ayw, , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ayw, , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ayx, , e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_ayx, , e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_ayy, s >= 0L, ),
             bor3(logi_ayy, s >= 0L, ))
expect_equal(or3s(logi_ayz, e >= 1L, ),
             bor3(logi_ayz, e >= 1L, ))
expect_equal(or3s(logi_ba, r >= 9L, ),
             bor3(logi_ba, r >= 9L, ))
expect_equal(or3s(logi_bb, n >= 0L, f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_bb, n >= 0L, f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_bc, g >= 1L, h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_bc, g >= 1L, h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_bd, h >= 9L, v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(logi_bd, h >= 9L, v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(logi_be, , ),
             bor3(logi_be, , ))
expect_equal(or3s(logi_bf, , ),
             bor3(logi_bf, , ))
expect_equal(or3s(logi_bg, , ),
             bor3(logi_bg, , ))
expect_equal(or3s(logi_bh, , z %in% 1:4),
             bor3(logi_bh, , z %in% 1:4))
expect_equal(or3s(logi_bi, , z %in% 1:4),
             bor3(logi_bi, , z %in% 1:4))
expect_equal(or3s(logi_bj, , w %in% 1:4),
             bor3(logi_bj, , w %in% 1:4))
expect_equal(or3s(logi_bk, k >= 0L, ),
             bor3(logi_bk, k >= 0L, ))
expect_equal(or3s(logi_bl, i >= 1L, ),
             bor3(logi_bl, i >= 1L, ))
expect_equal(or3s(logi_bm, b >= 9L, ),
             bor3(logi_bm, b >= 9L, ))
expect_equal(or3s(logi_bn, d >= 0L, n %in% 1:4),
             bor3(logi_bn, d >= 0L, n %in% 1:4))
expect_equal(or3s(logi_bo, k >= 1L, c %in% 1:4),
             bor3(logi_bo, k >= 1L, c %in% 1:4))
expect_equal(or3s(logi_bp, o >= 9L, h %in% 1:4),
             bor3(logi_bp, o >= 9L, h %in% 1:4))
expect_equal(or3s(logi_bq, , ),
             bor3(logi_bq, , ))
expect_equal(or3s(logi_br, , ),
             bor3(logi_br, , ))
expect_equal(or3s(logi_bs, , ),
             bor3(logi_bs, , ))
expect_equal(or3s(logi_bt, , q < 0L),
             bor3(logi_bt, , q < 0L))
expect_equal(or3s(logi_bu, , e < 1L),
             bor3(logi_bu, , e < 1L))
expect_equal(or3s(logi_bv, , i < 9L),
             bor3(logi_bv, , i < 9L))
expect_equal(or3s(logi_bw, u >= 0L, ),
             bor3(logi_bw, u >= 0L, ))
expect_equal(or3s(logi_bx, d >= 1L, ),
             bor3(logi_bx, d >= 1L, ))
expect_equal(or3s(logi_by, n >= 9L, ),
             bor3(logi_by, n >= 9L, ))
expect_equal(or3s(logi_bz, k >= 0L, j < 0L),
             bor3(logi_bz, k >= 0L, j < 0L))
expect_equal(or3s(logi_baa, e >= 1L, r < 1L),
             bor3(logi_baa, e >= 1L, r < 1L))
expect_equal(or3s(logi_bab, x >= 9L, r < 9L),
             bor3(logi_bab, x >= 9L, r < 9L))
expect_equal(or3s(logi_bac, , ),
             bor3(logi_bac, , ))
expect_equal(or3s(logi_bad, , ),
             bor3(logi_bad, , ))
expect_equal(or3s(logi_bae, , ),
             bor3(logi_bae, , ))
expect_equal(or3s(logi_baf, , o <= 0L),
             bor3(logi_baf, , o <= 0L))
expect_equal(or3s(logi_bag, , g <= 1L),
             bor3(logi_bag, , g <= 1L))
expect_equal(or3s(logi_bah, , e <= 9L),
             bor3(logi_bah, , e <= 9L))
expect_equal(or3s(logi_bai, a >= 0L, ),
             bor3(logi_bai, a >= 0L, ))
expect_equal(or3s(logi_baj, m >= 1L, ),
             bor3(logi_baj, m >= 1L, ))
expect_equal(or3s(logi_bak, q >= 9L, ),
             bor3(logi_bak, q >= 9L, ))
expect_equal(or3s(logi_bal, q >= 0L, l <= 0L),
             bor3(logi_bal, q >= 0L, l <= 0L))
expect_equal(or3s(logi_bam, o >= 1L, c <= 1L),
             bor3(logi_bam, o >= 1L, c <= 1L))
expect_equal(or3s(logi_ban, a >= 9L, k <= 9L),
             bor3(logi_ban, a >= 9L, k <= 9L))
expect_equal(or3s(logi_bao, , ),
             bor3(logi_bao, , ))
expect_equal(or3s(logi_bap, , ),
             bor3(logi_bap, , ))
expect_equal(or3s(logi_baq, , ),
             bor3(logi_baq, , ))
expect_equal(or3s(logi_bar, , n == 0L),
             bor3(logi_bar, , n == 0L))
expect_equal(or3s(logi_bas, , i == 1L),
             bor3(logi_bas, , i == 1L))
expect_equal(or3s(logi_bat, , n == 9L),
             bor3(logi_bat, , n == 9L))
expect_equal(or3s(logi_bau, q >= 0L, ),
             bor3(logi_bau, q >= 0L, ))
expect_equal(or3s(logi_bav, r >= 1L, ),
             bor3(logi_bav, r >= 1L, ))
expect_equal(or3s(logi_baw, p >= 9L, ),
             bor3(logi_baw, p >= 9L, ))
expect_equal(or3s(logi_bax, r >= 0L, x == 0L),
             bor3(logi_bax, r >= 0L, x == 0L))
expect_equal(or3s(logi_bay, n >= 1L, u == 1L),
             bor3(logi_bay, n >= 1L, u == 1L))
expect_equal(or3s(logi_baz, z >= 9L, h == 9L),
             bor3(logi_baz, z >= 9L, h == 9L))
expect_equal(or3s(logi_bba, , ),
             bor3(logi_bba, , ))
expect_equal(or3s(logi_bbb, , ),
             bor3(logi_bbb, , ))
expect_equal(or3s(logi_bbc, , ),
             bor3(logi_bbc, , ))
expect_equal(or3s(logi_bbd, , i > 0L),
             bor3(logi_bbd, , i > 0L))
expect_equal(or3s(logi_bbe, , y > 1L),
             bor3(logi_bbe, , y > 1L))
expect_equal(or3s(logi_bbf, , t > 9L),
             bor3(logi_bbf, , t > 9L))
expect_equal(or3s(logi_bbg, t >= 0L, ),
             bor3(logi_bbg, t >= 0L, ))
expect_equal(or3s(logi_bbh, f >= 1L, ),
             bor3(logi_bbh, f >= 1L, ))
expect_equal(or3s(logi_bbi, k >= 9L, ),
             bor3(logi_bbi, k >= 9L, ))
expect_equal(or3s(logi_bbj, r >= 0L, m > 0L),
             bor3(logi_bbj, r >= 0L, m > 0L))
expect_equal(or3s(logi_bbk, i >= 1L, d > 1L),
             bor3(logi_bbk, i >= 1L, d > 1L))
expect_equal(or3s(logi_bbl, t >= 9L, k > 9L),
             bor3(logi_bbl, t >= 9L, k > 9L))
expect_equal(or3s(logi_bbm, , ),
             bor3(logi_bbm, , ))
expect_equal(or3s(logi_bbn, , ),
             bor3(logi_bbn, , ))
expect_equal(or3s(logi_bbo, , ),
             bor3(logi_bbo, , ))
expect_equal(or3s(logi_bbp, , l >= 0L),
             bor3(logi_bbp, , l >= 0L))
expect_equal(or3s(logi_bbq, , m >= 1L),
             bor3(logi_bbq, , m >= 1L))
expect_equal(or3s(logi_bbr, , x >= 9L),
             bor3(logi_bbr, , x >= 9L))
expect_equal(or3s(logi_bbs, v >= 0L, ),
             bor3(logi_bbs, v >= 0L, ))
expect_equal(or3s(logi_bbt, j >= 1L, ),
             bor3(logi_bbt, j >= 1L, ))
expect_equal(or3s(logi_bbu, s >= 9L, ),
             bor3(logi_bbu, s >= 9L, ))
expect_equal(or3s(logi_bbv, p >= 0L, u >= 0L),
             bor3(logi_bbv, p >= 0L, u >= 0L))
expect_equal(or3s(logi_bbw, w >= 1L, a >= 1L),
             bor3(logi_bbw, w >= 1L, a >= 1L))
expect_equal(or3s(logi_bbx, f >= 9L, o >= 9L),
             bor3(logi_bbx, f >= 9L, o >= 9L))
expect_equal(or3s(!logi_bby, , ),
             bor3(!logi_bby, , ))
expect_equal(or3s(!logi_bbz, , ),
             bor3(!logi_bbz, , ))
expect_equal(or3s(!logi_bca, , ),
             bor3(!logi_bca, , ))
expect_equal(or3s(!logi_bcb, , logi_l),
             bor3(!logi_bcb, , logi_l))
expect_equal(or3s(!logi_bcc, , logi_n),
             bor3(!logi_bcc, , logi_n))
expect_equal(or3s(!logi_bcd, , logi_b),
             bor3(!logi_bcd, , logi_b))
expect_equal(or3s(!logi_bce, logi_t, ),
             bor3(!logi_bce, logi_t, ))
expect_equal(or3s(!logi_bcf, logi_x, ),
             bor3(!logi_bcf, logi_x, ))
expect_equal(or3s(!logi_bcg, logi_h, ),
             bor3(!logi_bcg, logi_h, ))
expect_equal(or3s(!logi_bch, logi_h, logi_i),
             bor3(!logi_bch, logi_h, logi_i))
expect_equal(or3s(!logi_bci, logi_f, logi_v),
             bor3(!logi_bci, logi_f, logi_v))
expect_equal(or3s(!logi_bcj, logi_f, logi_i),
             bor3(!logi_bcj, logi_f, logi_i))
expect_equal(or3s(!logi_bck, , ),
             bor3(!logi_bck, , ))
expect_equal(or3s(!logi_bcl, , ),
             bor3(!logi_bcl, , ))
expect_equal(or3s(!logi_bcm, , ),
             bor3(!logi_bcm, , ))
expect_equal(or3s(!logi_bcn, , !logi_w),
             bor3(!logi_bcn, , !logi_w))
expect_equal(or3s(!logi_bco, , !logi_v),
             bor3(!logi_bco, , !logi_v))
expect_equal(or3s(!logi_bcp, , !logi_i),
             bor3(!logi_bcp, , !logi_i))
expect_equal(or3s(!logi_bcq, logi_b, ),
             bor3(!logi_bcq, logi_b, ))
expect_equal(or3s(!logi_bcr, logi_l, ),
             bor3(!logi_bcr, logi_l, ))
expect_equal(or3s(!logi_bcs, logi_l, ),
             bor3(!logi_bcs, logi_l, ))
expect_equal(or3s(!logi_bct, logi_i, !logi_b),
             bor3(!logi_bct, logi_i, !logi_b))
expect_equal(or3s(!logi_bcu, logi_h, !logi_f),
             bor3(!logi_bcu, logi_h, !logi_f))
expect_equal(or3s(!logi_bcv, logi_f, !logi_n),
             bor3(!logi_bcv, logi_f, !logi_n))
expect_equal(or3s(!logi_bcw, , ),
             bor3(!logi_bcw, , ))
expect_equal(or3s(!logi_bcx, , ),
             bor3(!logi_bcx, , ))
expect_equal(or3s(!logi_bcy, , ),
             bor3(!logi_bcy, , ))
expect_equal(or3s(!logi_bcz, , f != 0L),
             bor3(!logi_bcz, , f != 0L))
expect_equal(or3s(!logi_bda, , e != 1L),
             bor3(!logi_bda, , e != 1L))
expect_equal(or3s(!logi_bdb, , r != 9L),
             bor3(!logi_bdb, , r != 9L))
expect_equal(or3s(!logi_bdc, logi_s, ),
             bor3(!logi_bdc, logi_s, ))
expect_equal(or3s(!logi_bdd, logi_s, ),
             bor3(!logi_bdd, logi_s, ))
expect_equal(or3s(!logi_bde, logi_b, ),
             bor3(!logi_bde, logi_b, ))
expect_equal(or3s(!logi_bdf, logi_c, p != 0L),
             bor3(!logi_bdf, logi_c, p != 0L))
expect_equal(or3s(!logi_bdg, logi_n, b != 1L),
             bor3(!logi_bdg, logi_n, b != 1L))
expect_equal(or3s(!logi_bdh, logi_g, c != 9L),
             bor3(!logi_bdh, logi_g, c != 9L))
expect_equal(or3s(!logi_bdi, , ),
             bor3(!logi_bdi, , ))
expect_equal(or3s(!logi_bdj, , ),
             bor3(!logi_bdj, , ))
expect_equal(or3s(!logi_bdk, , ),
             bor3(!logi_bdk, , ))
expect_equal(or3s(!logi_bdl, , t %between% c(-1L, 1L)),
             bor3(!logi_bdl, , t %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bdm, , p %between% c(-1L, 1L)),
             bor3(!logi_bdm, , p %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bdn, , m %between% c(-1L, 1L)),
             bor3(!logi_bdn, , m %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bdo, logi_i, ),
             bor3(!logi_bdo, logi_i, ))
expect_equal(or3s(!logi_bdp, logi_w, ),
             bor3(!logi_bdp, logi_w, ))
expect_equal(or3s(!logi_bdq, logi_t, ),
             bor3(!logi_bdq, logi_t, ))
expect_equal(or3s(!logi_bdr, logi_e, t %between% c(-1L, 1L)),
             bor3(!logi_bdr, logi_e, t %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bds, logi_t, r %between% c(-1L, 1L)),
             bor3(!logi_bds, logi_t, r %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bdt, logi_j, c %between% c(-1L, 1L)),
             bor3(!logi_bdt, logi_j, c %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bdu, , ),
             bor3(!logi_bdu, , ))
expect_equal(or3s(!logi_bdv, , ),
             bor3(!logi_bdv, , ))
expect_equal(or3s(!logi_bdw, , ),
             bor3(!logi_bdw, , ))
expect_equal(or3s(!logi_bdx, , s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bdx, , s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bdy, , x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bdy, , x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bdz, , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bdz, , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bea, logi_b, ),
             bor3(!logi_bea, logi_b, ))
expect_equal(or3s(!logi_beb, logi_o, ),
             bor3(!logi_beb, logi_o, ))
expect_equal(or3s(!logi_bec, logi_d, ),
             bor3(!logi_bec, logi_d, ))
expect_equal(or3s(!logi_bed, logi_v, k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bed, logi_v, k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bee, logi_m, t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bee, logi_m, t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bef, logi_e, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bef, logi_e, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_beg, , ),
             bor3(!logi_beg, , ))
expect_equal(or3s(!logi_beh, , ),
             bor3(!logi_beh, , ))
expect_equal(or3s(!logi_bei, , ),
             bor3(!logi_bei, , ))
expect_equal(or3s(!logi_bej, , t %in% 1:4),
             bor3(!logi_bej, , t %in% 1:4))
expect_equal(or3s(!logi_bek, , v %in% 1:4),
             bor3(!logi_bek, , v %in% 1:4))
expect_equal(or3s(!logi_bel, , e %in% 1:4),
             bor3(!logi_bel, , e %in% 1:4))
expect_equal(or3s(!logi_bem, logi_l, ),
             bor3(!logi_bem, logi_l, ))
expect_equal(or3s(!logi_ben, logi_s, ),
             bor3(!logi_ben, logi_s, ))
expect_equal(or3s(!logi_beo, logi_b, ),
             bor3(!logi_beo, logi_b, ))
expect_equal(or3s(!logi_bep, logi_c, r %in% 1:4),
             bor3(!logi_bep, logi_c, r %in% 1:4))
expect_equal(or3s(!logi_beq, logi_j, u %in% 1:4),
             bor3(!logi_beq, logi_j, u %in% 1:4))
expect_equal(or3s(!logi_ber, logi_l, e %in% 1:4),
             bor3(!logi_ber, logi_l, e %in% 1:4))
expect_equal(or3s(!logi_bes, , ),
             bor3(!logi_bes, , ))
expect_equal(or3s(!logi_bet, , ),
             bor3(!logi_bet, , ))
expect_equal(or3s(!logi_beu, , ),
             bor3(!logi_beu, , ))
expect_equal(or3s(!logi_bev, , c < 0L),
             bor3(!logi_bev, , c < 0L))
expect_equal(or3s(!logi_bew, , u < 1L),
             bor3(!logi_bew, , u < 1L))
expect_equal(or3s(!logi_bex, , b < 9L),
             bor3(!logi_bex, , b < 9L))
expect_equal(or3s(!logi_bey, logi_g, ),
             bor3(!logi_bey, logi_g, ))
expect_equal(or3s(!logi_bez, logi_h, ),
             bor3(!logi_bez, logi_h, ))
expect_equal(or3s(!logi_bfa, logi_q, ),
             bor3(!logi_bfa, logi_q, ))
expect_equal(or3s(!logi_bfb, logi_u, k < 0L),
             bor3(!logi_bfb, logi_u, k < 0L))
expect_equal(or3s(!logi_bfc, logi_a, x < 1L),
             bor3(!logi_bfc, logi_a, x < 1L))
expect_equal(or3s(!logi_bfd, logi_h, u < 9L),
             bor3(!logi_bfd, logi_h, u < 9L))
expect_equal(or3s(!logi_bfe, , ),
             bor3(!logi_bfe, , ))
expect_equal(or3s(!logi_bff, , ),
             bor3(!logi_bff, , ))
expect_equal(or3s(!logi_bfg, , ),
             bor3(!logi_bfg, , ))
expect_equal(or3s(!logi_bfh, , y <= 0L),
             bor3(!logi_bfh, , y <= 0L))
expect_equal(or3s(!logi_bfi, , z <= 1L),
             bor3(!logi_bfi, , z <= 1L))
expect_equal(or3s(!logi_bfj, , r <= 9L),
             bor3(!logi_bfj, , r <= 9L))
expect_equal(or3s(!logi_bfk, logi_h, ),
             bor3(!logi_bfk, logi_h, ))
expect_equal(or3s(!logi_bfl, logi_p, ),
             bor3(!logi_bfl, logi_p, ))
expect_equal(or3s(!logi_bfm, logi_y, ),
             bor3(!logi_bfm, logi_y, ))
expect_equal(or3s(!logi_bfn, logi_z, e <= 0L),
             bor3(!logi_bfn, logi_z, e <= 0L))
expect_equal(or3s(!logi_bfo, logi_c, b <= 1L),
             bor3(!logi_bfo, logi_c, b <= 1L))
expect_equal(or3s(!logi_bfp, logi_l, b <= 9L),
             bor3(!logi_bfp, logi_l, b <= 9L))
expect_equal(or3s(!logi_bfq, , ),
             bor3(!logi_bfq, , ))
expect_equal(or3s(!logi_bfr, , ),
             bor3(!logi_bfr, , ))
expect_equal(or3s(!logi_bfs, , ),
             bor3(!logi_bfs, , ))
expect_equal(or3s(!logi_bft, , x == 0L),
             bor3(!logi_bft, , x == 0L))
expect_equal(or3s(!logi_bfu, , h == 1L),
             bor3(!logi_bfu, , h == 1L))
expect_equal(or3s(!logi_bfv, , t == 9L),
             bor3(!logi_bfv, , t == 9L))
expect_equal(or3s(!logi_bfw, logi_p, ),
             bor3(!logi_bfw, logi_p, ))
expect_equal(or3s(!logi_bfx, logi_v, ),
             bor3(!logi_bfx, logi_v, ))
expect_equal(or3s(!logi_bfy, logi_y, ),
             bor3(!logi_bfy, logi_y, ))
expect_equal(or3s(!logi_bfz, logi_v, o == 0L),
             bor3(!logi_bfz, logi_v, o == 0L))
expect_equal(or3s(!logi_bga, logi_t, k == 1L),
             bor3(!logi_bga, logi_t, k == 1L))
expect_equal(or3s(!logi_bgb, logi_d, s == 9L),
             bor3(!logi_bgb, logi_d, s == 9L))
expect_equal(or3s(!logi_bgc, , ),
             bor3(!logi_bgc, , ))
expect_equal(or3s(!logi_bgd, , ),
             bor3(!logi_bgd, , ))
expect_equal(or3s(!logi_bge, , ),
             bor3(!logi_bge, , ))
expect_equal(or3s(!logi_bgf, , x > 0L),
             bor3(!logi_bgf, , x > 0L))
expect_equal(or3s(!logi_bgg, , b > 1L),
             bor3(!logi_bgg, , b > 1L))
expect_equal(or3s(!logi_bgh, , h > 9L),
             bor3(!logi_bgh, , h > 9L))
expect_equal(or3s(!logi_bgi, logi_i, ),
             bor3(!logi_bgi, logi_i, ))
expect_equal(or3s(!logi_bgj, logi_f, ),
             bor3(!logi_bgj, logi_f, ))
expect_equal(or3s(!logi_bgk, logi_j, ),
             bor3(!logi_bgk, logi_j, ))
expect_equal(or3s(!logi_bgl, logi_o, n > 0L),
             bor3(!logi_bgl, logi_o, n > 0L))
expect_equal(or3s(!logi_bgm, logi_h, s > 1L),
             bor3(!logi_bgm, logi_h, s > 1L))
expect_equal(or3s(!logi_bgn, logi_w, s > 9L),
             bor3(!logi_bgn, logi_w, s > 9L))
expect_equal(or3s(!logi_bgo, , ),
             bor3(!logi_bgo, , ))
expect_equal(or3s(!logi_bgp, , ),
             bor3(!logi_bgp, , ))
expect_equal(or3s(!logi_bgq, , ),
             bor3(!logi_bgq, , ))
expect_equal(or3s(!logi_bgr, , u >= 0L),
             bor3(!logi_bgr, , u >= 0L))
expect_equal(or3s(!logi_bgs, , f >= 1L),
             bor3(!logi_bgs, , f >= 1L))
expect_equal(or3s(!logi_bgt, , a >= 9L),
             bor3(!logi_bgt, , a >= 9L))
expect_equal(or3s(!logi_bgu, logi_f, ),
             bor3(!logi_bgu, logi_f, ))
expect_equal(or3s(!logi_bgv, logi_a, ),
             bor3(!logi_bgv, logi_a, ))
expect_equal(or3s(!logi_bgw, logi_z, ),
             bor3(!logi_bgw, logi_z, ))
expect_equal(or3s(!logi_bgx, logi_p, i >= 0L),
             bor3(!logi_bgx, logi_p, i >= 0L))
expect_equal(or3s(!logi_bgy, logi_u, m >= 1L),
             bor3(!logi_bgy, logi_u, m >= 1L))
expect_equal(or3s(!logi_bgz, logi_d, k >= 9L),
             bor3(!logi_bgz, logi_d, k >= 9L))
expect_equal(or3s(!logi_bha, , ),
             bor3(!logi_bha, , ))
expect_equal(or3s(!logi_bhb, , ),
             bor3(!logi_bhb, , ))
expect_equal(or3s(!logi_bhc, , ),
             bor3(!logi_bhc, , ))
expect_equal(or3s(!logi_bhd, , logi_g),
             bor3(!logi_bhd, , logi_g))
expect_equal(or3s(!logi_bhe, , logi_n),
             bor3(!logi_bhe, , logi_n))
expect_equal(or3s(!logi_bhf, , logi_y),
             bor3(!logi_bhf, , logi_y))
expect_equal(or3s(!logi_bhg, !logi_d, ),
             bor3(!logi_bhg, !logi_d, ))
expect_equal(or3s(!logi_bhh, !logi_l, ),
             bor3(!logi_bhh, !logi_l, ))
expect_equal(or3s(!logi_bhi, !logi_l, ),
             bor3(!logi_bhi, !logi_l, ))
expect_equal(or3s(!logi_bhj, !logi_p, logi_q),
             bor3(!logi_bhj, !logi_p, logi_q))
expect_equal(or3s(!logi_bhk, !logi_i, logi_x),
             bor3(!logi_bhk, !logi_i, logi_x))
expect_equal(or3s(!logi_bhl, !logi_x, logi_i),
             bor3(!logi_bhl, !logi_x, logi_i))
expect_equal(or3s(!logi_bhm, , ),
             bor3(!logi_bhm, , ))
expect_equal(or3s(!logi_bhn, , ),
             bor3(!logi_bhn, , ))
expect_equal(or3s(!logi_bho, , ),
             bor3(!logi_bho, , ))
expect_equal(or3s(!logi_bhp, , !logi_a),
             bor3(!logi_bhp, , !logi_a))
expect_equal(or3s(!logi_bhq, , !logi_l),
             bor3(!logi_bhq, , !logi_l))
expect_equal(or3s(!logi_bhr, , !logi_x),
             bor3(!logi_bhr, , !logi_x))
expect_equal(or3s(!logi_bhs, !logi_g, ),
             bor3(!logi_bhs, !logi_g, ))
expect_equal(or3s(!logi_bht, !logi_b, ),
             bor3(!logi_bht, !logi_b, ))
expect_equal(or3s(!logi_bhu, !logi_g, ),
             bor3(!logi_bhu, !logi_g, ))
expect_equal(or3s(!logi_bhv, !logi_f, !logi_u),
             bor3(!logi_bhv, !logi_f, !logi_u))
expect_equal(or3s(!logi_bhw, !logi_q, !logi_d),
             bor3(!logi_bhw, !logi_q, !logi_d))
expect_equal(or3s(!logi_bhx, !logi_y, !logi_l),
             bor3(!logi_bhx, !logi_y, !logi_l))
expect_equal(or3s(!logi_bhy, , ),
             bor3(!logi_bhy, , ))
expect_equal(or3s(!logi_bhz, , ),
             bor3(!logi_bhz, , ))
expect_equal(or3s(!logi_bia, , ),
             bor3(!logi_bia, , ))
expect_equal(or3s(!logi_bib, , x != 0L),
             bor3(!logi_bib, , x != 0L))
expect_equal(or3s(!logi_bic, , o != 1L),
             bor3(!logi_bic, , o != 1L))
expect_equal(or3s(!logi_bid, , z != 9L),
             bor3(!logi_bid, , z != 9L))
expect_equal(or3s(!logi_bie, !logi_k, ),
             bor3(!logi_bie, !logi_k, ))
expect_equal(or3s(!logi_bif, !logi_x, ),
             bor3(!logi_bif, !logi_x, ))
expect_equal(or3s(!logi_big, !logi_c, ),
             bor3(!logi_big, !logi_c, ))
expect_equal(or3s(!logi_bih, !logi_q, c != 0L),
             bor3(!logi_bih, !logi_q, c != 0L))
expect_equal(or3s(!logi_bii, !logi_p, w != 1L),
             bor3(!logi_bii, !logi_p, w != 1L))
expect_equal(or3s(!logi_bij, !logi_p, u != 9L),
             bor3(!logi_bij, !logi_p, u != 9L))
expect_equal(or3s(!logi_bik, , ),
             bor3(!logi_bik, , ))
expect_equal(or3s(!logi_bil, , ),
             bor3(!logi_bil, , ))
expect_equal(or3s(!logi_bim, , ),
             bor3(!logi_bim, , ))
expect_equal(or3s(!logi_bin, , v %between% c(-1L, 1L)),
             bor3(!logi_bin, , v %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bio, , n %between% c(-1L, 1L)),
             bor3(!logi_bio, , n %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bip, , o %between% c(-1L, 1L)),
             bor3(!logi_bip, , o %between% c(-1L, 1L)))
expect_equal(or3s(!logi_biq, !logi_k, ),
             bor3(!logi_biq, !logi_k, ))
expect_equal(or3s(!logi_bir, !logi_a, ),
             bor3(!logi_bir, !logi_a, ))
expect_equal(or3s(!logi_bis, !logi_k, ),
             bor3(!logi_bis, !logi_k, ))
expect_equal(or3s(!logi_bit, !logi_t, q %between% c(-1L, 1L)),
             bor3(!logi_bit, !logi_t, q %between% c(-1L, 1L)))
expect_equal(or3s(!logi_biu, !logi_s, z %between% c(-1L, 1L)),
             bor3(!logi_biu, !logi_s, z %between% c(-1L, 1L)))
expect_equal(or3s(!logi_biv, !logi_s, m %between% c(-1L, 1L)),
             bor3(!logi_biv, !logi_s, m %between% c(-1L, 1L)))
expect_equal(or3s(!logi_biw, , ),
             bor3(!logi_biw, , ))
expect_equal(or3s(!logi_bix, , ),
             bor3(!logi_bix, , ))
expect_equal(or3s(!logi_biy, , ),
             bor3(!logi_biy, , ))
expect_equal(or3s(!logi_biz, , f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_biz, , f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bja, , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bja, , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bjb, , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bjb, , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bjc, !logi_d, ),
             bor3(!logi_bjc, !logi_d, ))
expect_equal(or3s(!logi_bjd, !logi_c, ),
             bor3(!logi_bjd, !logi_c, ))
expect_equal(or3s(!logi_bje, !logi_r, ),
             bor3(!logi_bje, !logi_r, ))
expect_equal(or3s(!logi_bjf, !logi_x, f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bjf, !logi_x, f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bjg, !logi_c, u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bjg, !logi_c, u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bjh, !logi_a, k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bjh, !logi_a, k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bji, , ),
             bor3(!logi_bji, , ))
expect_equal(or3s(!logi_bjj, , ),
             bor3(!logi_bjj, , ))
expect_equal(or3s(!logi_bjk, , ),
             bor3(!logi_bjk, , ))
expect_equal(or3s(!logi_bjl, , x %in% 1:4),
             bor3(!logi_bjl, , x %in% 1:4))
expect_equal(or3s(!logi_bjm, , d %in% 1:4),
             bor3(!logi_bjm, , d %in% 1:4))
expect_equal(or3s(!logi_bjn, , m %in% 1:4),
             bor3(!logi_bjn, , m %in% 1:4))
expect_equal(or3s(!logi_bjo, !logi_e, ),
             bor3(!logi_bjo, !logi_e, ))
expect_equal(or3s(!logi_bjp, !logi_r, ),
             bor3(!logi_bjp, !logi_r, ))
expect_equal(or3s(!logi_bjq, !logi_m, ),
             bor3(!logi_bjq, !logi_m, ))
expect_equal(or3s(!logi_bjr, !logi_z, z %in% 1:4),
             bor3(!logi_bjr, !logi_z, z %in% 1:4))
expect_equal(or3s(!logi_bjs, !logi_z, o %in% 1:4),
             bor3(!logi_bjs, !logi_z, o %in% 1:4))
expect_equal(or3s(!logi_bjt, !logi_e, q %in% 1:4),
             bor3(!logi_bjt, !logi_e, q %in% 1:4))
expect_equal(or3s(!logi_bju, , ),
             bor3(!logi_bju, , ))
expect_equal(or3s(!logi_bjv, , ),
             bor3(!logi_bjv, , ))
expect_equal(or3s(!logi_bjw, , ),
             bor3(!logi_bjw, , ))
expect_equal(or3s(!logi_bjx, , i < 0L),
             bor3(!logi_bjx, , i < 0L))
expect_equal(or3s(!logi_bjy, , o < 1L),
             bor3(!logi_bjy, , o < 1L))
expect_equal(or3s(!logi_bjz, , e < 9L),
             bor3(!logi_bjz, , e < 9L))
expect_equal(or3s(!logi_bka, !logi_a, ),
             bor3(!logi_bka, !logi_a, ))
expect_equal(or3s(!logi_bkb, !logi_m, ),
             bor3(!logi_bkb, !logi_m, ))
expect_equal(or3s(!logi_bkc, !logi_b, ),
             bor3(!logi_bkc, !logi_b, ))
expect_equal(or3s(!logi_bkd, !logi_y, r < 0L),
             bor3(!logi_bkd, !logi_y, r < 0L))
expect_equal(or3s(!logi_bke, !logi_l, u < 1L),
             bor3(!logi_bke, !logi_l, u < 1L))
expect_equal(or3s(!logi_bkf, !logi_y, l < 9L),
             bor3(!logi_bkf, !logi_y, l < 9L))
expect_equal(or3s(!logi_bkg, , ),
             bor3(!logi_bkg, , ))
expect_equal(or3s(!logi_bkh, , ),
             bor3(!logi_bkh, , ))
expect_equal(or3s(!logi_bki, , ),
             bor3(!logi_bki, , ))
expect_equal(or3s(!logi_bkj, , u <= 0L),
             bor3(!logi_bkj, , u <= 0L))
expect_equal(or3s(!logi_bkk, , i <= 1L),
             bor3(!logi_bkk, , i <= 1L))
expect_equal(or3s(!logi_bkl, , e <= 9L),
             bor3(!logi_bkl, , e <= 9L))
expect_equal(or3s(!logi_bkm, !logi_l, ),
             bor3(!logi_bkm, !logi_l, ))
expect_equal(or3s(!logi_bkn, !logi_v, ),
             bor3(!logi_bkn, !logi_v, ))
expect_equal(or3s(!logi_bko, !logi_s, ),
             bor3(!logi_bko, !logi_s, ))
expect_equal(or3s(!logi_bkp, !logi_n, o <= 0L),
             bor3(!logi_bkp, !logi_n, o <= 0L))
expect_equal(or3s(!logi_bkq, !logi_u, s <= 1L),
             bor3(!logi_bkq, !logi_u, s <= 1L))
expect_equal(or3s(!logi_bkr, !logi_d, m <= 9L),
             bor3(!logi_bkr, !logi_d, m <= 9L))
expect_equal(or3s(!logi_bks, , ),
             bor3(!logi_bks, , ))
expect_equal(or3s(!logi_bkt, , ),
             bor3(!logi_bkt, , ))
expect_equal(or3s(!logi_bku, , ),
             bor3(!logi_bku, , ))
expect_equal(or3s(!logi_bkv, , p == 0L),
             bor3(!logi_bkv, , p == 0L))
expect_equal(or3s(!logi_bkw, , j == 1L),
             bor3(!logi_bkw, , j == 1L))
expect_equal(or3s(!logi_bkx, , c == 9L),
             bor3(!logi_bkx, , c == 9L))
expect_equal(or3s(!logi_bky, !logi_e, ),
             bor3(!logi_bky, !logi_e, ))
expect_equal(or3s(!logi_bkz, !logi_r, ),
             bor3(!logi_bkz, !logi_r, ))
expect_equal(or3s(!logi_bla, !logi_y, ),
             bor3(!logi_bla, !logi_y, ))
expect_equal(or3s(!logi_blb, !logi_m, d == 0L),
             bor3(!logi_blb, !logi_m, d == 0L))
expect_equal(or3s(!logi_blc, !logi_e, g == 1L),
             bor3(!logi_blc, !logi_e, g == 1L))
expect_equal(or3s(!logi_bld, !logi_a, z == 9L),
             bor3(!logi_bld, !logi_a, z == 9L))
expect_equal(or3s(!logi_ble, , ),
             bor3(!logi_ble, , ))
expect_equal(or3s(!logi_blf, , ),
             bor3(!logi_blf, , ))
expect_equal(or3s(!logi_blg, , ),
             bor3(!logi_blg, , ))
expect_equal(or3s(!logi_blh, , j > 0L),
             bor3(!logi_blh, , j > 0L))
expect_equal(or3s(!logi_bli, , s > 1L),
             bor3(!logi_bli, , s > 1L))
expect_equal(or3s(!logi_blj, , t > 9L),
             bor3(!logi_blj, , t > 9L))
expect_equal(or3s(!logi_blk, !logi_b, ),
             bor3(!logi_blk, !logi_b, ))
expect_equal(or3s(!logi_bll, !logi_f, ),
             bor3(!logi_bll, !logi_f, ))
expect_equal(or3s(!logi_blm, !logi_k, ),
             bor3(!logi_blm, !logi_k, ))
expect_equal(or3s(!logi_bln, !logi_h, c > 0L),
             bor3(!logi_bln, !logi_h, c > 0L))
expect_equal(or3s(!logi_blo, !logi_j, t > 1L),
             bor3(!logi_blo, !logi_j, t > 1L))
expect_equal(or3s(!logi_blp, !logi_k, v > 9L),
             bor3(!logi_blp, !logi_k, v > 9L))
expect_equal(or3s(!logi_blq, , ),
             bor3(!logi_blq, , ))
expect_equal(or3s(!logi_blr, , ),
             bor3(!logi_blr, , ))
expect_equal(or3s(!logi_bls, , ),
             bor3(!logi_bls, , ))
expect_equal(or3s(!logi_blt, , n >= 0L),
             bor3(!logi_blt, , n >= 0L))
expect_equal(or3s(!logi_blu, , e >= 1L),
             bor3(!logi_blu, , e >= 1L))
expect_equal(or3s(!logi_blv, , j >= 9L),
             bor3(!logi_blv, , j >= 9L))
expect_equal(or3s(!logi_blw, !logi_f, ),
             bor3(!logi_blw, !logi_f, ))
expect_equal(or3s(!logi_blx, !logi_u, ),
             bor3(!logi_blx, !logi_u, ))
expect_equal(or3s(!logi_bly, !logi_e, ),
             bor3(!logi_bly, !logi_e, ))
expect_equal(or3s(!logi_blz, !logi_k, b >= 0L),
             bor3(!logi_blz, !logi_k, b >= 0L))
expect_equal(or3s(!logi_bma, !logi_w, g >= 1L),
             bor3(!logi_bma, !logi_w, g >= 1L))
expect_equal(or3s(!logi_bmb, !logi_d, f >= 9L),
             bor3(!logi_bmb, !logi_d, f >= 9L))
expect_equal(or3s(!logi_bmc, , ),
             bor3(!logi_bmc, , ))
expect_equal(or3s(!logi_bmd, , ),
             bor3(!logi_bmd, , ))
expect_equal(or3s(!logi_bme, , ),
             bor3(!logi_bme, , ))
expect_equal(or3s(!logi_bmf, , logi_u),
             bor3(!logi_bmf, , logi_u))
expect_equal(or3s(!logi_bmg, , logi_y),
             bor3(!logi_bmg, , logi_y))
expect_equal(or3s(!logi_bmh, , logi_u),
             bor3(!logi_bmh, , logi_u))
expect_equal(or3s(!logi_bmi, p != 0L, ),
             bor3(!logi_bmi, p != 0L, ))
expect_equal(or3s(!logi_bmj, p != 1L, ),
             bor3(!logi_bmj, p != 1L, ))
expect_equal(or3s(!logi_bmk, z != 9L, ),
             bor3(!logi_bmk, z != 9L, ))
expect_equal(or3s(!logi_bml, g != 0L, logi_c),
             bor3(!logi_bml, g != 0L, logi_c))
expect_equal(or3s(!logi_bmm, c != 1L, logi_e),
             bor3(!logi_bmm, c != 1L, logi_e))
expect_equal(or3s(!logi_bmn, d != 9L, logi_k),
             bor3(!logi_bmn, d != 9L, logi_k))
expect_equal(or3s(!logi_bmo, , ),
             bor3(!logi_bmo, , ))
expect_equal(or3s(!logi_bmp, , ),
             bor3(!logi_bmp, , ))
expect_equal(or3s(!logi_bmq, , ),
             bor3(!logi_bmq, , ))
expect_equal(or3s(!logi_bmr, , !logi_c),
             bor3(!logi_bmr, , !logi_c))
expect_equal(or3s(!logi_bms, , !logi_y),
             bor3(!logi_bms, , !logi_y))
expect_equal(or3s(!logi_bmt, , !logi_f),
             bor3(!logi_bmt, , !logi_f))
expect_equal(or3s(!logi_bmu, z != 0L, ),
             bor3(!logi_bmu, z != 0L, ))
expect_equal(or3s(!logi_bmv, a != 1L, ),
             bor3(!logi_bmv, a != 1L, ))
expect_equal(or3s(!logi_bmw, y != 9L, ),
             bor3(!logi_bmw, y != 9L, ))
expect_equal(or3s(!logi_bmx, r != 0L, !logi_n),
             bor3(!logi_bmx, r != 0L, !logi_n))
expect_equal(or3s(!logi_bmy, h != 1L, !logi_u),
             bor3(!logi_bmy, h != 1L, !logi_u))
expect_equal(or3s(!logi_bmz, e != 9L, !logi_c),
             bor3(!logi_bmz, e != 9L, !logi_c))
expect_equal(or3s(!logi_bna, , ),
             bor3(!logi_bna, , ))
expect_equal(or3s(!logi_bnb, , ),
             bor3(!logi_bnb, , ))
expect_equal(or3s(!logi_bnc, , ),
             bor3(!logi_bnc, , ))
expect_equal(or3s(!logi_bnd, , n != 0L),
             bor3(!logi_bnd, , n != 0L))
expect_equal(or3s(!logi_bne, , y != 1L),
             bor3(!logi_bne, , y != 1L))
expect_equal(or3s(!logi_bnf, , k != 9L),
             bor3(!logi_bnf, , k != 9L))
expect_equal(or3s(!logi_bng, d != 0L, ),
             bor3(!logi_bng, d != 0L, ))
expect_equal(or3s(!logi_bnh, t != 1L, ),
             bor3(!logi_bnh, t != 1L, ))
expect_equal(or3s(!logi_bni, m != 9L, ),
             bor3(!logi_bni, m != 9L, ))
expect_equal(or3s(!logi_bnj, m != 0L, v != 0L),
             bor3(!logi_bnj, m != 0L, v != 0L))
expect_equal(or3s(!logi_bnk, v != 1L, z != 1L),
             bor3(!logi_bnk, v != 1L, z != 1L))
expect_equal(or3s(!logi_bnl, o != 9L, n != 9L),
             bor3(!logi_bnl, o != 9L, n != 9L))
expect_equal(or3s(!logi_bnm, , ),
             bor3(!logi_bnm, , ))
expect_equal(or3s(!logi_bnn, , ),
             bor3(!logi_bnn, , ))
expect_equal(or3s(!logi_bno, , ),
             bor3(!logi_bno, , ))
expect_equal(or3s(!logi_bnp, , a %between% c(-1L, 1L)),
             bor3(!logi_bnp, , a %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bnq, , e %between% c(-1L, 1L)),
             bor3(!logi_bnq, , e %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bnr, , i %between% c(-1L, 1L)),
             bor3(!logi_bnr, , i %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bns, w != 0L, ),
             bor3(!logi_bns, w != 0L, ))
expect_equal(or3s(!logi_bnt, n != 1L, ),
             bor3(!logi_bnt, n != 1L, ))
expect_equal(or3s(!logi_bnu, s != 9L, ),
             bor3(!logi_bnu, s != 9L, ))
expect_equal(or3s(!logi_bnv, j != 0L, b %between% c(-1L, 1L)),
             bor3(!logi_bnv, j != 0L, b %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bnw, c != 1L, l %between% c(-1L, 1L)),
             bor3(!logi_bnw, c != 1L, l %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bnx, q != 9L, v %between% c(-1L, 1L)),
             bor3(!logi_bnx, q != 9L, v %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bny, , ),
             bor3(!logi_bny, , ))
expect_equal(or3s(!logi_bnz, , ),
             bor3(!logi_bnz, , ))
expect_equal(or3s(!logi_boa, , ),
             bor3(!logi_boa, , ))
expect_equal(or3s(!logi_bob, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bob, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_boc, , f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_boc, , f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bod, , q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bod, , q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_boe, t != 0L, ),
             bor3(!logi_boe, t != 0L, ))
expect_equal(or3s(!logi_bof, z != 1L, ),
             bor3(!logi_bof, z != 1L, ))
expect_equal(or3s(!logi_bog, t != 9L, ),
             bor3(!logi_bog, t != 9L, ))
expect_equal(or3s(!logi_boh, k != 0L, z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_boh, k != 0L, z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_boi, r != 1L, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_boi, r != 1L, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_boj, v != 9L, p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_boj, v != 9L, p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bok, , ),
             bor3(!logi_bok, , ))
expect_equal(or3s(!logi_bol, , ),
             bor3(!logi_bol, , ))
expect_equal(or3s(!logi_bom, , ),
             bor3(!logi_bom, , ))
expect_equal(or3s(!logi_bon, , b %in% 1:4),
             bor3(!logi_bon, , b %in% 1:4))
expect_equal(or3s(!logi_boo, , u %in% 1:4),
             bor3(!logi_boo, , u %in% 1:4))
expect_equal(or3s(!logi_bop, , v %in% 1:4),
             bor3(!logi_bop, , v %in% 1:4))
expect_equal(or3s(!logi_boq, i != 0L, ),
             bor3(!logi_boq, i != 0L, ))
expect_equal(or3s(!logi_bor, v != 1L, ),
             bor3(!logi_bor, v != 1L, ))
expect_equal(or3s(!logi_bos, t != 9L, ),
             bor3(!logi_bos, t != 9L, ))
expect_equal(or3s(!logi_bot, o != 0L, f %in% 1:4),
             bor3(!logi_bot, o != 0L, f %in% 1:4))
expect_equal(or3s(!logi_bou, l != 1L, p %in% 1:4),
             bor3(!logi_bou, l != 1L, p %in% 1:4))
expect_equal(or3s(!logi_bov, r != 9L, o %in% 1:4),
             bor3(!logi_bov, r != 9L, o %in% 1:4))
expect_equal(or3s(!logi_bow, , ),
             bor3(!logi_bow, , ))
expect_equal(or3s(!logi_box, , ),
             bor3(!logi_box, , ))
expect_equal(or3s(!logi_boy, , ),
             bor3(!logi_boy, , ))
expect_equal(or3s(!logi_boz, , y < 0L),
             bor3(!logi_boz, , y < 0L))
expect_equal(or3s(!logi_bpa, , e < 1L),
             bor3(!logi_bpa, , e < 1L))
expect_equal(or3s(!logi_bpb, , e < 9L),
             bor3(!logi_bpb, , e < 9L))
expect_equal(or3s(!logi_bpc, g != 0L, ),
             bor3(!logi_bpc, g != 0L, ))
expect_equal(or3s(!logi_bpd, q != 1L, ),
             bor3(!logi_bpd, q != 1L, ))
expect_equal(or3s(!logi_bpe, m != 9L, ),
             bor3(!logi_bpe, m != 9L, ))
expect_equal(or3s(!logi_bpf, t != 0L, i < 0L),
             bor3(!logi_bpf, t != 0L, i < 0L))
expect_equal(or3s(!logi_bpg, u != 1L, u < 1L),
             bor3(!logi_bpg, u != 1L, u < 1L))
expect_equal(or3s(!logi_bph, h != 9L, i < 9L),
             bor3(!logi_bph, h != 9L, i < 9L))
expect_equal(or3s(!logi_bpi, , ),
             bor3(!logi_bpi, , ))
expect_equal(or3s(!logi_bpj, , ),
             bor3(!logi_bpj, , ))
expect_equal(or3s(!logi_bpk, , ),
             bor3(!logi_bpk, , ))
expect_equal(or3s(!logi_bpl, , f <= 0L),
             bor3(!logi_bpl, , f <= 0L))
expect_equal(or3s(!logi_bpm, , s <= 1L),
             bor3(!logi_bpm, , s <= 1L))
expect_equal(or3s(!logi_bpn, , p <= 9L),
             bor3(!logi_bpn, , p <= 9L))
expect_equal(or3s(!logi_bpo, f != 0L, ),
             bor3(!logi_bpo, f != 0L, ))
expect_equal(or3s(!logi_bpp, f != 1L, ),
             bor3(!logi_bpp, f != 1L, ))
expect_equal(or3s(!logi_bpq, y != 9L, ),
             bor3(!logi_bpq, y != 9L, ))
expect_equal(or3s(!logi_bpr, y != 0L, c <= 0L),
             bor3(!logi_bpr, y != 0L, c <= 0L))
expect_equal(or3s(!logi_bps, e != 1L, j <= 1L),
             bor3(!logi_bps, e != 1L, j <= 1L))
expect_equal(or3s(!logi_bpt, a != 9L, r <= 9L),
             bor3(!logi_bpt, a != 9L, r <= 9L))
expect_equal(or3s(!logi_bpu, , ),
             bor3(!logi_bpu, , ))
expect_equal(or3s(!logi_bpv, , ),
             bor3(!logi_bpv, , ))
expect_equal(or3s(!logi_bpw, , ),
             bor3(!logi_bpw, , ))
expect_equal(or3s(!logi_bpx, , p == 0L),
             bor3(!logi_bpx, , p == 0L))
expect_equal(or3s(!logi_bpy, , q == 1L),
             bor3(!logi_bpy, , q == 1L))
expect_equal(or3s(!logi_bpz, , z == 9L),
             bor3(!logi_bpz, , z == 9L))
expect_equal(or3s(!logi_bqa, g != 0L, ),
             bor3(!logi_bqa, g != 0L, ))
expect_equal(or3s(!logi_bqb, b != 1L, ),
             bor3(!logi_bqb, b != 1L, ))
expect_equal(or3s(!logi_bqc, d != 9L, ),
             bor3(!logi_bqc, d != 9L, ))
expect_equal(or3s(!logi_bqd, j != 0L, j == 0L),
             bor3(!logi_bqd, j != 0L, j == 0L))
expect_equal(or3s(!logi_bqe, t != 1L, y == 1L),
             bor3(!logi_bqe, t != 1L, y == 1L))
expect_equal(or3s(!logi_bqf, z != 9L, n == 9L),
             bor3(!logi_bqf, z != 9L, n == 9L))
expect_equal(or3s(!logi_bqg, , ),
             bor3(!logi_bqg, , ))
expect_equal(or3s(!logi_bqh, , ),
             bor3(!logi_bqh, , ))
expect_equal(or3s(!logi_bqi, , ),
             bor3(!logi_bqi, , ))
expect_equal(or3s(!logi_bqj, , q > 0L),
             bor3(!logi_bqj, , q > 0L))
expect_equal(or3s(!logi_bqk, , w > 1L),
             bor3(!logi_bqk, , w > 1L))
expect_equal(or3s(!logi_bql, , a > 9L),
             bor3(!logi_bql, , a > 9L))
expect_equal(or3s(!logi_bqm, r != 0L, ),
             bor3(!logi_bqm, r != 0L, ))
expect_equal(or3s(!logi_bqn, w != 1L, ),
             bor3(!logi_bqn, w != 1L, ))
expect_equal(or3s(!logi_bqo, u != 9L, ),
             bor3(!logi_bqo, u != 9L, ))
expect_equal(or3s(!logi_bqp, x != 0L, e > 0L),
             bor3(!logi_bqp, x != 0L, e > 0L))
expect_equal(or3s(!logi_bqq, r != 1L, s > 1L),
             bor3(!logi_bqq, r != 1L, s > 1L))
expect_equal(or3s(!logi_bqr, e != 9L, r > 9L),
             bor3(!logi_bqr, e != 9L, r > 9L))
expect_equal(or3s(!logi_bqs, , ),
             bor3(!logi_bqs, , ))
expect_equal(or3s(!logi_bqt, , ),
             bor3(!logi_bqt, , ))
expect_equal(or3s(!logi_bqu, , ),
             bor3(!logi_bqu, , ))
expect_equal(or3s(!logi_bqv, , k >= 0L),
             bor3(!logi_bqv, , k >= 0L))
expect_equal(or3s(!logi_bqw, , s >= 1L),
             bor3(!logi_bqw, , s >= 1L))
expect_equal(or3s(!logi_bqx, , m >= 9L),
             bor3(!logi_bqx, , m >= 9L))
expect_equal(or3s(!logi_bqy, c != 0L, ),
             bor3(!logi_bqy, c != 0L, ))
expect_equal(or3s(!logi_bqz, v != 1L, ),
             bor3(!logi_bqz, v != 1L, ))
expect_equal(or3s(!logi_bra, x != 9L, ),
             bor3(!logi_bra, x != 9L, ))
expect_equal(or3s(!logi_brb, t != 0L, y >= 0L),
             bor3(!logi_brb, t != 0L, y >= 0L))
expect_equal(or3s(!logi_brc, q != 1L, f >= 1L),
             bor3(!logi_brc, q != 1L, f >= 1L))
expect_equal(or3s(!logi_brd, z != 9L, t >= 9L),
             bor3(!logi_brd, z != 9L, t >= 9L))
expect_equal(or3s(!logi_bre, , ),
             bor3(!logi_bre, , ))
expect_equal(or3s(!logi_brf, , ),
             bor3(!logi_brf, , ))
expect_equal(or3s(!logi_brg, , ),
             bor3(!logi_brg, , ))
expect_equal(or3s(!logi_brh, , logi_c),
             bor3(!logi_brh, , logi_c))
expect_equal(or3s(!logi_bri, , logi_g),
             bor3(!logi_bri, , logi_g))
expect_equal(or3s(!logi_brj, , logi_u),
             bor3(!logi_brj, , logi_u))
expect_equal(or3s(!logi_brk, p %between% c(-1L, 1L), ),
             bor3(!logi_brk, p %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_brl, y %between% c(-1L, 1L), ),
             bor3(!logi_brl, y %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_brm, d %between% c(-1L, 1L), ),
             bor3(!logi_brm, d %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_brn, e %between% c(-1L, 1L), logi_l),
             bor3(!logi_brn, e %between% c(-1L, 1L), logi_l))
expect_equal(or3s(!logi_bro, l %between% c(-1L, 1L), logi_n),
             bor3(!logi_bro, l %between% c(-1L, 1L), logi_n))
expect_equal(or3s(!logi_brp, m %between% c(-1L, 1L), logi_w),
             bor3(!logi_brp, m %between% c(-1L, 1L), logi_w))
expect_equal(or3s(!logi_brq, , ),
             bor3(!logi_brq, , ))
expect_equal(or3s(!logi_brr, , ),
             bor3(!logi_brr, , ))
expect_equal(or3s(!logi_brs, , ),
             bor3(!logi_brs, , ))
expect_equal(or3s(!logi_brt, , !logi_a),
             bor3(!logi_brt, , !logi_a))
expect_equal(or3s(!logi_bru, , !logi_w),
             bor3(!logi_bru, , !logi_w))
expect_equal(or3s(!logi_brv, , !logi_x),
             bor3(!logi_brv, , !logi_x))
expect_equal(or3s(!logi_brw, o %between% c(-1L, 1L), ),
             bor3(!logi_brw, o %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_brx, v %between% c(-1L, 1L), ),
             bor3(!logi_brx, v %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bry, c %between% c(-1L, 1L), ),
             bor3(!logi_bry, c %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_brz, l %between% c(-1L, 1L), !logi_e),
             bor3(!logi_brz, l %between% c(-1L, 1L), !logi_e))
expect_equal(or3s(!logi_bsa, n %between% c(-1L, 1L), !logi_i),
             bor3(!logi_bsa, n %between% c(-1L, 1L), !logi_i))
expect_equal(or3s(!logi_bsb, d %between% c(-1L, 1L), !logi_v),
             bor3(!logi_bsb, d %between% c(-1L, 1L), !logi_v))
expect_equal(or3s(!logi_bsc, , ),
             bor3(!logi_bsc, , ))
expect_equal(or3s(!logi_bsd, , ),
             bor3(!logi_bsd, , ))
expect_equal(or3s(!logi_bse, , ),
             bor3(!logi_bse, , ))
expect_equal(or3s(!logi_bsf, , u != 0L),
             bor3(!logi_bsf, , u != 0L))
expect_equal(or3s(!logi_bsg, , h != 1L),
             bor3(!logi_bsg, , h != 1L))
expect_equal(or3s(!logi_bsh, , d != 9L),
             bor3(!logi_bsh, , d != 9L))
expect_equal(or3s(!logi_bsi, h %between% c(-1L, 1L), ),
             bor3(!logi_bsi, h %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bsj, s %between% c(-1L, 1L), ),
             bor3(!logi_bsj, s %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bsk, y %between% c(-1L, 1L), ),
             bor3(!logi_bsk, y %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bsl, m %between% c(-1L, 1L), i != 0L),
             bor3(!logi_bsl, m %between% c(-1L, 1L), i != 0L))
expect_equal(or3s(!logi_bsm, h %between% c(-1L, 1L), v != 1L),
             bor3(!logi_bsm, h %between% c(-1L, 1L), v != 1L))
expect_equal(or3s(!logi_bsn, e %between% c(-1L, 1L), i != 9L),
             bor3(!logi_bsn, e %between% c(-1L, 1L), i != 9L))
expect_equal(or3s(!logi_bso, , ),
             bor3(!logi_bso, , ))
expect_equal(or3s(!logi_bsp, , ),
             bor3(!logi_bsp, , ))
expect_equal(or3s(!logi_bsq, , ),
             bor3(!logi_bsq, , ))
expect_equal(or3s(!logi_bsr, , o %between% c(-1L, 1L)),
             bor3(!logi_bsr, , o %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bss, , u %between% c(-1L, 1L)),
             bor3(!logi_bss, , u %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bst, , d %between% c(-1L, 1L)),
             bor3(!logi_bst, , d %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bsu, c %between% c(-1L, 1L), ),
             bor3(!logi_bsu, c %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bsv, q %between% c(-1L, 1L), ),
             bor3(!logi_bsv, q %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bsw, k %between% c(-1L, 1L), ),
             bor3(!logi_bsw, k %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bsx, l %between% c(-1L, 1L), e %between% c(-1L, 1L)),
             bor3(!logi_bsx, l %between% c(-1L, 1L), e %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bsy, e %between% c(-1L, 1L), i %between% c(-1L, 1L)),
             bor3(!logi_bsy, e %between% c(-1L, 1L), i %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bsz, b %between% c(-1L, 1L), b %between% c(-1L, 1L)),
             bor3(!logi_bsz, b %between% c(-1L, 1L), b %between% c(-1L, 1L)))
expect_equal(or3s(!logi_bta, , ),
             bor3(!logi_bta, , ))
expect_equal(or3s(!logi_btb, , ),
             bor3(!logi_btb, , ))
expect_equal(or3s(!logi_btc, , ),
             bor3(!logi_btc, , ))
expect_equal(or3s(!logi_btd, , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_btd, , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_bte, , u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_bte, , u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_btf, , s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_btf, , s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_btg, g %between% c(-1L, 1L), ),
             bor3(!logi_btg, g %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bth, m %between% c(-1L, 1L), ),
             bor3(!logi_bth, m %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bti, i %between% c(-1L, 1L), ),
             bor3(!logi_bti, i %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_btj, j %between% c(-1L, 1L), a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_btj, j %between% c(-1L, 1L), a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_btk, w %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_btk, w %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_btl, b %between% c(-1L, 1L), z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_btl, b %between% c(-1L, 1L), z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_btm, , ),
             bor3(!logi_btm, , ))
expect_equal(or3s(!logi_btn, , ),
             bor3(!logi_btn, , ))
expect_equal(or3s(!logi_bto, , ),
             bor3(!logi_bto, , ))
expect_equal(or3s(!logi_btp, , r %in% 1:4),
             bor3(!logi_btp, , r %in% 1:4))
expect_equal(or3s(!logi_btq, , j %in% 1:4),
             bor3(!logi_btq, , j %in% 1:4))
expect_equal(or3s(!logi_btr, , x %in% 1:4),
             bor3(!logi_btr, , x %in% 1:4))
expect_equal(or3s(!logi_bts, a %between% c(-1L, 1L), ),
             bor3(!logi_bts, a %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_btt, x %between% c(-1L, 1L), ),
             bor3(!logi_btt, x %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_btu, v %between% c(-1L, 1L), ),
             bor3(!logi_btu, v %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_btv, u %between% c(-1L, 1L), t %in% 1:4),
             bor3(!logi_btv, u %between% c(-1L, 1L), t %in% 1:4))
expect_equal(or3s(!logi_btw, d %between% c(-1L, 1L), y %in% 1:4),
             bor3(!logi_btw, d %between% c(-1L, 1L), y %in% 1:4))
expect_equal(or3s(!logi_btx, s %between% c(-1L, 1L), f %in% 1:4),
             bor3(!logi_btx, s %between% c(-1L, 1L), f %in% 1:4))
expect_equal(or3s(!logi_bty, , ),
             bor3(!logi_bty, , ))
expect_equal(or3s(!logi_btz, , ),
             bor3(!logi_btz, , ))
expect_equal(or3s(!logi_bua, , ),
             bor3(!logi_bua, , ))
expect_equal(or3s(!logi_bub, , g < 0L),
             bor3(!logi_bub, , g < 0L))
expect_equal(or3s(!logi_buc, , i < 1L),
             bor3(!logi_buc, , i < 1L))
expect_equal(or3s(!logi_bud, , o < 9L),
             bor3(!logi_bud, , o < 9L))
expect_equal(or3s(!logi_bue, h %between% c(-1L, 1L), ),
             bor3(!logi_bue, h %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_buf, x %between% c(-1L, 1L), ),
             bor3(!logi_buf, x %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bug, v %between% c(-1L, 1L), ),
             bor3(!logi_bug, v %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_buh, m %between% c(-1L, 1L), n < 0L),
             bor3(!logi_buh, m %between% c(-1L, 1L), n < 0L))
expect_equal(or3s(!logi_bui, c %between% c(-1L, 1L), d < 1L),
             bor3(!logi_bui, c %between% c(-1L, 1L), d < 1L))
expect_equal(or3s(!logi_buj, a %between% c(-1L, 1L), i < 9L),
             bor3(!logi_buj, a %between% c(-1L, 1L), i < 9L))
expect_equal(or3s(!logi_buk, , ),
             bor3(!logi_buk, , ))
expect_equal(or3s(!logi_bul, , ),
             bor3(!logi_bul, , ))
expect_equal(or3s(!logi_bum, , ),
             bor3(!logi_bum, , ))
expect_equal(or3s(!logi_bun, , w <= 0L),
             bor3(!logi_bun, , w <= 0L))
expect_equal(or3s(!logi_buo, , b <= 1L),
             bor3(!logi_buo, , b <= 1L))
expect_equal(or3s(!logi_bup, , i <= 9L),
             bor3(!logi_bup, , i <= 9L))
expect_equal(or3s(!logi_buq, n %between% c(-1L, 1L), ),
             bor3(!logi_buq, n %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bur, x %between% c(-1L, 1L), ),
             bor3(!logi_bur, x %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bus, j %between% c(-1L, 1L), ),
             bor3(!logi_bus, j %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_but, y %between% c(-1L, 1L), o <= 0L),
             bor3(!logi_but, y %between% c(-1L, 1L), o <= 0L))
expect_equal(or3s(!logi_buu, p %between% c(-1L, 1L), f <= 1L),
             bor3(!logi_buu, p %between% c(-1L, 1L), f <= 1L))
expect_equal(or3s(!logi_buv, o %between% c(-1L, 1L), d <= 9L),
             bor3(!logi_buv, o %between% c(-1L, 1L), d <= 9L))
expect_equal(or3s(!logi_buw, , ),
             bor3(!logi_buw, , ))
expect_equal(or3s(!logi_bux, , ),
             bor3(!logi_bux, , ))
expect_equal(or3s(!logi_buy, , ),
             bor3(!logi_buy, , ))
expect_equal(or3s(!logi_buz, , s == 0L),
             bor3(!logi_buz, , s == 0L))
expect_equal(or3s(!logi_bva, , m == 1L),
             bor3(!logi_bva, , m == 1L))
expect_equal(or3s(!logi_bvb, , l == 9L),
             bor3(!logi_bvb, , l == 9L))
expect_equal(or3s(!logi_bvc, x %between% c(-1L, 1L), ),
             bor3(!logi_bvc, x %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bvd, r %between% c(-1L, 1L), ),
             bor3(!logi_bvd, r %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bve, n %between% c(-1L, 1L), ),
             bor3(!logi_bve, n %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bvf, o %between% c(-1L, 1L), s == 0L),
             bor3(!logi_bvf, o %between% c(-1L, 1L), s == 0L))
expect_equal(or3s(!logi_bvg, d %between% c(-1L, 1L), y == 1L),
             bor3(!logi_bvg, d %between% c(-1L, 1L), y == 1L))
expect_equal(or3s(!logi_bvh, m %between% c(-1L, 1L), q == 9L),
             bor3(!logi_bvh, m %between% c(-1L, 1L), q == 9L))
expect_equal(or3s(!logi_bvi, , ),
             bor3(!logi_bvi, , ))
expect_equal(or3s(!logi_bvj, , ),
             bor3(!logi_bvj, , ))
expect_equal(or3s(!logi_bvk, , ),
             bor3(!logi_bvk, , ))
expect_equal(or3s(!logi_bvl, , s > 0L),
             bor3(!logi_bvl, , s > 0L))
expect_equal(or3s(!logi_bvm, , d > 1L),
             bor3(!logi_bvm, , d > 1L))
expect_equal(or3s(!logi_bvn, , g > 9L),
             bor3(!logi_bvn, , g > 9L))
expect_equal(or3s(!logi_bvo, x %between% c(-1L, 1L), ),
             bor3(!logi_bvo, x %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bvp, k %between% c(-1L, 1L), ),
             bor3(!logi_bvp, k %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bvq, x %between% c(-1L, 1L), ),
             bor3(!logi_bvq, x %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bvr, j %between% c(-1L, 1L), n > 0L),
             bor3(!logi_bvr, j %between% c(-1L, 1L), n > 0L))
expect_equal(or3s(!logi_bvs, r %between% c(-1L, 1L), s > 1L),
             bor3(!logi_bvs, r %between% c(-1L, 1L), s > 1L))
expect_equal(or3s(!logi_bvt, q %between% c(-1L, 1L), y > 9L),
             bor3(!logi_bvt, q %between% c(-1L, 1L), y > 9L))
expect_equal(or3s(!logi_bvu, , ),
             bor3(!logi_bvu, , ))
expect_equal(or3s(!logi_bvv, , ),
             bor3(!logi_bvv, , ))
expect_equal(or3s(!logi_bvw, , ),
             bor3(!logi_bvw, , ))
expect_equal(or3s(!logi_bvx, , l >= 0L),
             bor3(!logi_bvx, , l >= 0L))
expect_equal(or3s(!logi_bvy, , r >= 1L),
             bor3(!logi_bvy, , r >= 1L))
expect_equal(or3s(!logi_bvz, , q >= 9L),
             bor3(!logi_bvz, , q >= 9L))
expect_equal(or3s(!logi_bwa, n %between% c(-1L, 1L), ),
             bor3(!logi_bwa, n %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bwb, o %between% c(-1L, 1L), ),
             bor3(!logi_bwb, o %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bwc, l %between% c(-1L, 1L), ),
             bor3(!logi_bwc, l %between% c(-1L, 1L), ))
expect_equal(or3s(!logi_bwd, z %between% c(-1L, 1L), v >= 0L),
             bor3(!logi_bwd, z %between% c(-1L, 1L), v >= 0L))
expect_equal(or3s(!logi_bwe, d %between% c(-1L, 1L), p >= 1L),
             bor3(!logi_bwe, d %between% c(-1L, 1L), p >= 1L))
expect_equal(or3s(!logi_bwf, w %between% c(-1L, 1L), r >= 9L),
             bor3(!logi_bwf, w %between% c(-1L, 1L), r >= 9L))
expect_equal(or3s(!logi_bwg, , ),
             bor3(!logi_bwg, , ))
expect_equal(or3s(!logi_bwh, , ),
             bor3(!logi_bwh, , ))
expect_equal(or3s(!logi_bwi, , ),
             bor3(!logi_bwi, , ))
expect_equal(or3s(!logi_bwj, , logi_b),
             bor3(!logi_bwj, , logi_b))
expect_equal(or3s(!logi_bwk, , logi_j),
             bor3(!logi_bwk, , logi_j))
expect_equal(or3s(!logi_bwl, , logi_r),
             bor3(!logi_bwl, , logi_r))
expect_equal(or3s(!logi_bwm, c %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_bwm, c %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_bwn, d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_bwn, d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_bwo, u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_bwo, u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_bwp, o %in% c(4L, -3L, 2L, 7L, 8L), logi_d),
             bor3(!logi_bwp, o %in% c(4L, -3L, 2L, 7L, 8L), logi_d))
expect_equal(or3s(!logi_bwq, s %in% c(4L, -3L, 2L, 7L, 8L), logi_m),
             bor3(!logi_bwq, s %in% c(4L, -3L, 2L, 7L, 8L), logi_m))
expect_equal(or3s(!logi_bwr, k %in% c(4L, -3L, 2L, 7L, 8L), logi_m),
             bor3(!logi_bwr, k %in% c(4L, -3L, 2L, 7L, 8L), logi_m))
expect_equal(or3s(!logi_bws, , ),
             bor3(!logi_bws, , ))
expect_equal(or3s(!logi_bwt, , ),
             bor3(!logi_bwt, , ))
expect_equal(or3s(!logi_bwu, , ),
             bor3(!logi_bwu, , ))
expect_equal(or3s(!logi_bwv, , logi_u),
             bor3(!logi_bwv, , logi_u))
expect_equal(or3s(!logi_bww, , logi_v),
             bor3(!logi_bww, , logi_v))
expect_equal(or3s(!logi_bwx, , logi_y),
             bor3(!logi_bwx, , logi_y))
expect_equal(or3s(!logi_bwy, r %in% 1:4, ),
             bor3(!logi_bwy, r %in% 1:4, ))
expect_equal(or3s(!logi_bwz, t %in% 1:4, ),
             bor3(!logi_bwz, t %in% 1:4, ))
expect_equal(or3s(!logi_bxa, k %in% 1:4, ),
             bor3(!logi_bxa, k %in% 1:4, ))
expect_equal(or3s(!logi_bxb, q %in% 1:4, logi_w),
             bor3(!logi_bxb, q %in% 1:4, logi_w))
expect_equal(or3s(!logi_bxc, k %in% 1:4, logi_f),
             bor3(!logi_bxc, k %in% 1:4, logi_f))
expect_equal(or3s(!logi_bxd, u %in% 1:4, logi_r),
             bor3(!logi_bxd, u %in% 1:4, logi_r))
expect_equal(or3s(!logi_bxe, , ),
             bor3(!logi_bxe, , ))
expect_equal(or3s(!logi_bxf, , ),
             bor3(!logi_bxf, , ))
expect_equal(or3s(!logi_bxg, , ),
             bor3(!logi_bxg, , ))
expect_equal(or3s(!logi_bxh, , !logi_w),
             bor3(!logi_bxh, , !logi_w))
expect_equal(or3s(!logi_bxi, , !logi_c),
             bor3(!logi_bxi, , !logi_c))
expect_equal(or3s(!logi_bxj, , !logi_k),
             bor3(!logi_bxj, , !logi_k))
expect_equal(or3s(!logi_bxk, g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_bxk, g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_bxl, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_bxl, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_bxm, j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_bxm, j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_bxn, y %in% c(4L, -3L, 2L, 7L, 8L), !logi_e),
             bor3(!logi_bxn, y %in% c(4L, -3L, 2L, 7L, 8L), !logi_e))
expect_equal(or3s(!logi_bxo, k %in% c(4L, -3L, 2L, 7L, 8L), !logi_z),
             bor3(!logi_bxo, k %in% c(4L, -3L, 2L, 7L, 8L), !logi_z))
expect_equal(or3s(!logi_bxp, n %in% c(4L, -3L, 2L, 7L, 8L), !logi_n),
             bor3(!logi_bxp, n %in% c(4L, -3L, 2L, 7L, 8L), !logi_n))
expect_equal(or3s(!logi_bxq, , ),
             bor3(!logi_bxq, , ))
expect_equal(or3s(!logi_bxr, , ),
             bor3(!logi_bxr, , ))
expect_equal(or3s(!logi_bxs, , ),
             bor3(!logi_bxs, , ))
expect_equal(or3s(!logi_bxt, , !logi_g),
             bor3(!logi_bxt, , !logi_g))
expect_equal(or3s(!logi_bxu, , !logi_n),
             bor3(!logi_bxu, , !logi_n))
expect_equal(or3s(!logi_bxv, , !logi_m),
             bor3(!logi_bxv, , !logi_m))
expect_equal(or3s(!logi_bxw, y %in% 1:4, ),
             bor3(!logi_bxw, y %in% 1:4, ))
expect_equal(or3s(!logi_bxx, c %in% 1:4, ),
             bor3(!logi_bxx, c %in% 1:4, ))
expect_equal(or3s(!logi_bxy, l %in% 1:4, ),
             bor3(!logi_bxy, l %in% 1:4, ))
expect_equal(or3s(!logi_bxz, y %in% 1:4, !logi_o),
             bor3(!logi_bxz, y %in% 1:4, !logi_o))
expect_equal(or3s(!logi_bya, i %in% 1:4, !logi_h),
             bor3(!logi_bya, i %in% 1:4, !logi_h))
expect_equal(or3s(!logi_byb, k %in% 1:4, !logi_a),
             bor3(!logi_byb, k %in% 1:4, !logi_a))
expect_equal(or3s(!logi_byc, , ),
             bor3(!logi_byc, , ))
expect_equal(or3s(!logi_byd, , ),
             bor3(!logi_byd, , ))
expect_equal(or3s(!logi_bye, , ),
             bor3(!logi_bye, , ))
expect_equal(or3s(!logi_byf, , i != 0L),
             bor3(!logi_byf, , i != 0L))
expect_equal(or3s(!logi_byg, , g != 1L),
             bor3(!logi_byg, , g != 1L))
expect_equal(or3s(!logi_byh, , o != 9L),
             bor3(!logi_byh, , o != 9L))
expect_equal(or3s(!logi_byi, e %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_byi, e %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_byj, d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_byj, d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_byk, l %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_byk, l %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_byl, w %in% c(4L, -3L, 2L, 7L, 8L), i != 0L),
             bor3(!logi_byl, w %in% c(4L, -3L, 2L, 7L, 8L), i != 0L))
expect_equal(or3s(!logi_bym, d %in% c(4L, -3L, 2L, 7L, 8L), j != 1L),
             bor3(!logi_bym, d %in% c(4L, -3L, 2L, 7L, 8L), j != 1L))
expect_equal(or3s(!logi_byn, g %in% c(4L, -3L, 2L, 7L, 8L), k != 9L),
             bor3(!logi_byn, g %in% c(4L, -3L, 2L, 7L, 8L), k != 9L))
expect_equal(or3s(!logi_byo, , ),
             bor3(!logi_byo, , ))
expect_equal(or3s(!logi_byp, , ),
             bor3(!logi_byp, , ))
expect_equal(or3s(!logi_byq, , ),
             bor3(!logi_byq, , ))
expect_equal(or3s(!logi_byr, , y != 0L),
             bor3(!logi_byr, , y != 0L))
expect_equal(or3s(!logi_bys, , b != 1L),
             bor3(!logi_bys, , b != 1L))
expect_equal(or3s(!logi_byt, , h != 9L),
             bor3(!logi_byt, , h != 9L))
expect_equal(or3s(!logi_byu, t %in% 1:4, ),
             bor3(!logi_byu, t %in% 1:4, ))
expect_equal(or3s(!logi_byv, t %in% 1:4, ),
             bor3(!logi_byv, t %in% 1:4, ))
expect_equal(or3s(!logi_byw, r %in% 1:4, ),
             bor3(!logi_byw, r %in% 1:4, ))
expect_equal(or3s(!logi_byx, v %in% 1:4, f != 0L),
             bor3(!logi_byx, v %in% 1:4, f != 0L))
expect_equal(or3s(!logi_byy, r %in% 1:4, k != 1L),
             bor3(!logi_byy, r %in% 1:4, k != 1L))
expect_equal(or3s(!logi_byz, k %in% 1:4, k != 9L),
             bor3(!logi_byz, k %in% 1:4, k != 9L))
expect_equal(or3s(!logi_ca, , ),
             bor3(!logi_ca, , ))
expect_equal(or3s(!logi_cb, , ),
             bor3(!logi_cb, , ))
expect_equal(or3s(!logi_cc, , ),
             bor3(!logi_cc, , ))
expect_equal(or3s(!logi_cd, , r %between% c(-1L, 1L)),
             bor3(!logi_cd, , r %between% c(-1L, 1L)))
expect_equal(or3s(!logi_ce, , g %between% c(-1L, 1L)),
             bor3(!logi_ce, , g %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cf, , b %between% c(-1L, 1L)),
             bor3(!logi_cf, , b %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cg, p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cg, p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_ch, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_ch, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_ci, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_ci, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cj, k %in% c(4L, -3L, 2L, 7L, 8L), r %between% c(-1L, 1L)),
             bor3(!logi_cj, k %in% c(4L, -3L, 2L, 7L, 8L), r %between% c(-1L, 1L)))
expect_equal(or3s(!logi_ck, f %in% c(4L, -3L, 2L, 7L, 8L), y %between% c(-1L, 1L)),
             bor3(!logi_ck, f %in% c(4L, -3L, 2L, 7L, 8L), y %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cl, a %in% c(4L, -3L, 2L, 7L, 8L), n %between% c(-1L, 1L)),
             bor3(!logi_cl, a %in% c(4L, -3L, 2L, 7L, 8L), n %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cm, , ),
             bor3(!logi_cm, , ))
expect_equal(or3s(!logi_cn, , ),
             bor3(!logi_cn, , ))
expect_equal(or3s(!logi_co, , ),
             bor3(!logi_co, , ))
expect_equal(or3s(!logi_cp, , r %between% c(-1L, 1L)),
             bor3(!logi_cp, , r %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cq, , y %between% c(-1L, 1L)),
             bor3(!logi_cq, , y %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cr, , w %between% c(-1L, 1L)),
             bor3(!logi_cr, , w %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cs, g %in% 1:4, ),
             bor3(!logi_cs, g %in% 1:4, ))
expect_equal(or3s(!logi_ct, o %in% 1:4, ),
             bor3(!logi_ct, o %in% 1:4, ))
expect_equal(or3s(!logi_cu, x %in% 1:4, ),
             bor3(!logi_cu, x %in% 1:4, ))
expect_equal(or3s(!logi_cv, z %in% 1:4, t %between% c(-1L, 1L)),
             bor3(!logi_cv, z %in% 1:4, t %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cw, m %in% 1:4, b %between% c(-1L, 1L)),
             bor3(!logi_cw, m %in% 1:4, b %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cx, z %in% 1:4, o %between% c(-1L, 1L)),
             bor3(!logi_cx, z %in% 1:4, o %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cy, , ),
             bor3(!logi_cy, , ))
expect_equal(or3s(!logi_cz, , ),
             bor3(!logi_cz, , ))
expect_equal(or3s(!logi_caa, , ),
             bor3(!logi_caa, , ))
expect_equal(or3s(!logi_cab, , w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cab, , w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cac, , w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cac, , w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cad, , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cad, , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cae, y %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cae, y %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_caf, w %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_caf, w %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cag, t %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cag, t %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cah, j %in% c(4L, -3L, 2L, 7L, 8L), j %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cah, j %in% c(4L, -3L, 2L, 7L, 8L), j %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cai, y %in% c(4L, -3L, 2L, 7L, 8L), w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cai, y %in% c(4L, -3L, 2L, 7L, 8L), w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_caj, b %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_caj, b %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cak, , ),
             bor3(!logi_cak, , ))
expect_equal(or3s(!logi_cal, , ),
             bor3(!logi_cal, , ))
expect_equal(or3s(!logi_cam, , ),
             bor3(!logi_cam, , ))
expect_equal(or3s(!logi_can, , p %in% 1:4),
             bor3(!logi_can, , p %in% 1:4))
expect_equal(or3s(!logi_cao, , f %in% 1:4),
             bor3(!logi_cao, , f %in% 1:4))
expect_equal(or3s(!logi_cap, , x %in% 1:4),
             bor3(!logi_cap, , x %in% 1:4))
expect_equal(or3s(!logi_caq, k %in% 1:4, ),
             bor3(!logi_caq, k %in% 1:4, ))
expect_equal(or3s(!logi_car, i %in% 1:4, ),
             bor3(!logi_car, i %in% 1:4, ))
expect_equal(or3s(!logi_cas, q %in% 1:4, ),
             bor3(!logi_cas, q %in% 1:4, ))
expect_equal(or3s(!logi_cat, z %in% 1:4, z %in% 1:4),
             bor3(!logi_cat, z %in% 1:4, z %in% 1:4))
expect_equal(or3s(!logi_cau, c %in% 1:4, c %in% 1:4),
             bor3(!logi_cau, c %in% 1:4, c %in% 1:4))
expect_equal(or3s(!logi_cav, t %in% 1:4, l %in% 1:4),
             bor3(!logi_cav, t %in% 1:4, l %in% 1:4))
expect_equal(or3s(!logi_caw, , ),
             bor3(!logi_caw, , ))
expect_equal(or3s(!logi_cax, , ),
             bor3(!logi_cax, , ))
expect_equal(or3s(!logi_cay, , ),
             bor3(!logi_cay, , ))
expect_equal(or3s(!logi_caz, , s < 0L),
             bor3(!logi_caz, , s < 0L))
expect_equal(or3s(!logi_cba, , z < 1L),
             bor3(!logi_cba, , z < 1L))
expect_equal(or3s(!logi_cbb, , b < 9L),
             bor3(!logi_cbb, , b < 9L))
expect_equal(or3s(!logi_cbc, i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cbc, i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cbd, u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cbd, u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cbe, j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cbe, j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cbf, t %in% c(4L, -3L, 2L, 7L, 8L), c < 0L),
             bor3(!logi_cbf, t %in% c(4L, -3L, 2L, 7L, 8L), c < 0L))
expect_equal(or3s(!logi_cbg, h %in% c(4L, -3L, 2L, 7L, 8L), b < 1L),
             bor3(!logi_cbg, h %in% c(4L, -3L, 2L, 7L, 8L), b < 1L))
expect_equal(or3s(!logi_cbh, h %in% c(4L, -3L, 2L, 7L, 8L), u < 9L),
             bor3(!logi_cbh, h %in% c(4L, -3L, 2L, 7L, 8L), u < 9L))
expect_equal(or3s(!logi_cbi, , ),
             bor3(!logi_cbi, , ))
expect_equal(or3s(!logi_cbj, , ),
             bor3(!logi_cbj, , ))
expect_equal(or3s(!logi_cbk, , ),
             bor3(!logi_cbk, , ))
expect_equal(or3s(!logi_cbl, , a < 0L),
             bor3(!logi_cbl, , a < 0L))
expect_equal(or3s(!logi_cbm, , w < 1L),
             bor3(!logi_cbm, , w < 1L))
expect_equal(or3s(!logi_cbn, , d < 9L),
             bor3(!logi_cbn, , d < 9L))
expect_equal(or3s(!logi_cbo, q %in% 1:4, ),
             bor3(!logi_cbo, q %in% 1:4, ))
expect_equal(or3s(!logi_cbp, w %in% 1:4, ),
             bor3(!logi_cbp, w %in% 1:4, ))
expect_equal(or3s(!logi_cbq, r %in% 1:4, ),
             bor3(!logi_cbq, r %in% 1:4, ))
expect_equal(or3s(!logi_cbr, x %in% 1:4, u < 0L),
             bor3(!logi_cbr, x %in% 1:4, u < 0L))
expect_equal(or3s(!logi_cbs, l %in% 1:4, c < 1L),
             bor3(!logi_cbs, l %in% 1:4, c < 1L))
expect_equal(or3s(!logi_cbt, h %in% 1:4, q < 9L),
             bor3(!logi_cbt, h %in% 1:4, q < 9L))
expect_equal(or3s(!logi_cbu, , ),
             bor3(!logi_cbu, , ))
expect_equal(or3s(!logi_cbv, , ),
             bor3(!logi_cbv, , ))
expect_equal(or3s(!logi_cbw, , ),
             bor3(!logi_cbw, , ))
expect_equal(or3s(!logi_cbx, , d <= 0L),
             bor3(!logi_cbx, , d <= 0L))
expect_equal(or3s(!logi_cby, , m <= 1L),
             bor3(!logi_cby, , m <= 1L))
expect_equal(or3s(!logi_cbz, , h <= 9L),
             bor3(!logi_cbz, , h <= 9L))
expect_equal(or3s(!logi_cca, y %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cca, y %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_ccb, n %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_ccb, n %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_ccc, m %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_ccc, m %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_ccd, h %in% c(4L, -3L, 2L, 7L, 8L), q <= 0L),
             bor3(!logi_ccd, h %in% c(4L, -3L, 2L, 7L, 8L), q <= 0L))
expect_equal(or3s(!logi_cce, h %in% c(4L, -3L, 2L, 7L, 8L), f <= 1L),
             bor3(!logi_cce, h %in% c(4L, -3L, 2L, 7L, 8L), f <= 1L))
expect_equal(or3s(!logi_ccf, r %in% c(4L, -3L, 2L, 7L, 8L), a <= 9L),
             bor3(!logi_ccf, r %in% c(4L, -3L, 2L, 7L, 8L), a <= 9L))
expect_equal(or3s(!logi_ccg, , ),
             bor3(!logi_ccg, , ))
expect_equal(or3s(!logi_cch, , ),
             bor3(!logi_cch, , ))
expect_equal(or3s(!logi_cci, , ),
             bor3(!logi_cci, , ))
expect_equal(or3s(!logi_ccj, , e <= 0L),
             bor3(!logi_ccj, , e <= 0L))
expect_equal(or3s(!logi_cck, , w <= 1L),
             bor3(!logi_cck, , w <= 1L))
expect_equal(or3s(!logi_ccl, , a <= 9L),
             bor3(!logi_ccl, , a <= 9L))
expect_equal(or3s(!logi_ccm, w %in% 1:4, ),
             bor3(!logi_ccm, w %in% 1:4, ))
expect_equal(or3s(!logi_ccn, k %in% 1:4, ),
             bor3(!logi_ccn, k %in% 1:4, ))
expect_equal(or3s(!logi_cco, s %in% 1:4, ),
             bor3(!logi_cco, s %in% 1:4, ))
expect_equal(or3s(!logi_ccp, q %in% 1:4, t <= 0L),
             bor3(!logi_ccp, q %in% 1:4, t <= 0L))
expect_equal(or3s(!logi_ccq, u %in% 1:4, r <= 1L),
             bor3(!logi_ccq, u %in% 1:4, r <= 1L))
expect_equal(or3s(!logi_ccr, o %in% 1:4, i <= 9L),
             bor3(!logi_ccr, o %in% 1:4, i <= 9L))
expect_equal(or3s(!logi_ccs, , ),
             bor3(!logi_ccs, , ))
expect_equal(or3s(!logi_cct, , ),
             bor3(!logi_cct, , ))
expect_equal(or3s(!logi_ccu, , ),
             bor3(!logi_ccu, , ))
expect_equal(or3s(!logi_ccv, , v == 0L),
             bor3(!logi_ccv, , v == 0L))
expect_equal(or3s(!logi_ccw, , j == 1L),
             bor3(!logi_ccw, , j == 1L))
expect_equal(or3s(!logi_ccx, , b == 9L),
             bor3(!logi_ccx, , b == 9L))
expect_equal(or3s(!logi_ccy, w %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_ccy, w %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_ccz, p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_ccz, p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cda, z %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cda, z %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cdb, i %in% c(4L, -3L, 2L, 7L, 8L), h == 0L),
             bor3(!logi_cdb, i %in% c(4L, -3L, 2L, 7L, 8L), h == 0L))
expect_equal(or3s(!logi_cdc, f %in% c(4L, -3L, 2L, 7L, 8L), c == 1L),
             bor3(!logi_cdc, f %in% c(4L, -3L, 2L, 7L, 8L), c == 1L))
expect_equal(or3s(!logi_cdd, x %in% c(4L, -3L, 2L, 7L, 8L), d == 9L),
             bor3(!logi_cdd, x %in% c(4L, -3L, 2L, 7L, 8L), d == 9L))
expect_equal(or3s(!logi_cde, , ),
             bor3(!logi_cde, , ))
expect_equal(or3s(!logi_cdf, , ),
             bor3(!logi_cdf, , ))
expect_equal(or3s(!logi_cdg, , ),
             bor3(!logi_cdg, , ))
expect_equal(or3s(!logi_cdh, , i == 0L),
             bor3(!logi_cdh, , i == 0L))
expect_equal(or3s(!logi_cdi, , h == 1L),
             bor3(!logi_cdi, , h == 1L))
expect_equal(or3s(!logi_cdj, , c == 9L),
             bor3(!logi_cdj, , c == 9L))
expect_equal(or3s(!logi_cdk, l %in% 1:4, ),
             bor3(!logi_cdk, l %in% 1:4, ))
expect_equal(or3s(!logi_cdl, x %in% 1:4, ),
             bor3(!logi_cdl, x %in% 1:4, ))
expect_equal(or3s(!logi_cdm, a %in% 1:4, ),
             bor3(!logi_cdm, a %in% 1:4, ))
expect_equal(or3s(!logi_cdn, k %in% 1:4, p == 0L),
             bor3(!logi_cdn, k %in% 1:4, p == 0L))
expect_equal(or3s(!logi_cdo, u %in% 1:4, j == 1L),
             bor3(!logi_cdo, u %in% 1:4, j == 1L))
expect_equal(or3s(!logi_cdp, p %in% 1:4, t == 9L),
             bor3(!logi_cdp, p %in% 1:4, t == 9L))
expect_equal(or3s(!logi_cdq, , ),
             bor3(!logi_cdq, , ))
expect_equal(or3s(!logi_cdr, , ),
             bor3(!logi_cdr, , ))
expect_equal(or3s(!logi_cds, , ),
             bor3(!logi_cds, , ))
expect_equal(or3s(!logi_cdt, , y > 0L),
             bor3(!logi_cdt, , y > 0L))
expect_equal(or3s(!logi_cdu, , o > 1L),
             bor3(!logi_cdu, , o > 1L))
expect_equal(or3s(!logi_cdv, , r > 9L),
             bor3(!logi_cdv, , r > 9L))
expect_equal(or3s(!logi_cdw, e %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cdw, e %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cdx, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cdx, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cdy, m %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cdy, m %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cdz, v %in% c(4L, -3L, 2L, 7L, 8L), x > 0L),
             bor3(!logi_cdz, v %in% c(4L, -3L, 2L, 7L, 8L), x > 0L))
expect_equal(or3s(!logi_cea, c %in% c(4L, -3L, 2L, 7L, 8L), w > 1L),
             bor3(!logi_cea, c %in% c(4L, -3L, 2L, 7L, 8L), w > 1L))
expect_equal(or3s(!logi_ceb, w %in% c(4L, -3L, 2L, 7L, 8L), i > 9L),
             bor3(!logi_ceb, w %in% c(4L, -3L, 2L, 7L, 8L), i > 9L))
expect_equal(or3s(!logi_cec, , ),
             bor3(!logi_cec, , ))
expect_equal(or3s(!logi_ced, , ),
             bor3(!logi_ced, , ))
expect_equal(or3s(!logi_cee, , ),
             bor3(!logi_cee, , ))
expect_equal(or3s(!logi_cef, , c > 0L),
             bor3(!logi_cef, , c > 0L))
expect_equal(or3s(!logi_ceg, , k > 1L),
             bor3(!logi_ceg, , k > 1L))
expect_equal(or3s(!logi_ceh, , b > 9L),
             bor3(!logi_ceh, , b > 9L))
expect_equal(or3s(!logi_cei, b %in% 1:4, ),
             bor3(!logi_cei, b %in% 1:4, ))
expect_equal(or3s(!logi_cej, k %in% 1:4, ),
             bor3(!logi_cej, k %in% 1:4, ))
expect_equal(or3s(!logi_cek, k %in% 1:4, ),
             bor3(!logi_cek, k %in% 1:4, ))
expect_equal(or3s(!logi_cel, m %in% 1:4, i > 0L),
             bor3(!logi_cel, m %in% 1:4, i > 0L))
expect_equal(or3s(!logi_cem, b %in% 1:4, c > 1L),
             bor3(!logi_cem, b %in% 1:4, c > 1L))
expect_equal(or3s(!logi_cen, v %in% 1:4, b > 9L),
             bor3(!logi_cen, v %in% 1:4, b > 9L))
expect_equal(or3s(!logi_ceo, , ),
             bor3(!logi_ceo, , ))
expect_equal(or3s(!logi_cep, , ),
             bor3(!logi_cep, , ))
expect_equal(or3s(!logi_ceq, , ),
             bor3(!logi_ceq, , ))
expect_equal(or3s(!logi_cer, , b >= 0L),
             bor3(!logi_cer, , b >= 0L))
expect_equal(or3s(!logi_ces, , y >= 1L),
             bor3(!logi_ces, , y >= 1L))
expect_equal(or3s(!logi_cet, , s >= 9L),
             bor3(!logi_cet, , s >= 9L))
expect_equal(or3s(!logi_ceu, h %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_ceu, h %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cev, s %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cev, s %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cew, h %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(!logi_cew, h %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(!logi_cex, e %in% c(4L, -3L, 2L, 7L, 8L), r >= 0L),
             bor3(!logi_cex, e %in% c(4L, -3L, 2L, 7L, 8L), r >= 0L))
expect_equal(or3s(!logi_cey, s %in% c(4L, -3L, 2L, 7L, 8L), k >= 1L),
             bor3(!logi_cey, s %in% c(4L, -3L, 2L, 7L, 8L), k >= 1L))
expect_equal(or3s(!logi_cez, z %in% c(4L, -3L, 2L, 7L, 8L), l >= 9L),
             bor3(!logi_cez, z %in% c(4L, -3L, 2L, 7L, 8L), l >= 9L))
expect_equal(or3s(!logi_cfa, , ),
             bor3(!logi_cfa, , ))
expect_equal(or3s(!logi_cfb, , ),
             bor3(!logi_cfb, , ))
expect_equal(or3s(!logi_cfc, , ),
             bor3(!logi_cfc, , ))
expect_equal(or3s(!logi_cfd, , p >= 0L),
             bor3(!logi_cfd, , p >= 0L))
expect_equal(or3s(!logi_cfe, , y >= 1L),
             bor3(!logi_cfe, , y >= 1L))
expect_equal(or3s(!logi_cff, , u >= 9L),
             bor3(!logi_cff, , u >= 9L))
expect_equal(or3s(!logi_cfg, b %in% 1:4, ),
             bor3(!logi_cfg, b %in% 1:4, ))
expect_equal(or3s(!logi_cfh, q %in% 1:4, ),
             bor3(!logi_cfh, q %in% 1:4, ))
expect_equal(or3s(!logi_cfi, j %in% 1:4, ),
             bor3(!logi_cfi, j %in% 1:4, ))
expect_equal(or3s(!logi_cfj, b %in% 1:4, x >= 0L),
             bor3(!logi_cfj, b %in% 1:4, x >= 0L))
expect_equal(or3s(!logi_cfk, x %in% 1:4, k >= 1L),
             bor3(!logi_cfk, x %in% 1:4, k >= 1L))
expect_equal(or3s(!logi_cfl, d %in% 1:4, g >= 9L),
             bor3(!logi_cfl, d %in% 1:4, g >= 9L))
expect_equal(or3s(!logi_cfm, , ),
             bor3(!logi_cfm, , ))
expect_equal(or3s(!logi_cfn, , ),
             bor3(!logi_cfn, , ))
expect_equal(or3s(!logi_cfo, , ),
             bor3(!logi_cfo, , ))
expect_equal(or3s(!logi_cfp, , logi_q),
             bor3(!logi_cfp, , logi_q))
expect_equal(or3s(!logi_cfq, , logi_o),
             bor3(!logi_cfq, , logi_o))
expect_equal(or3s(!logi_cfr, , logi_s),
             bor3(!logi_cfr, , logi_s))
expect_equal(or3s(!logi_cfs, b < 0L, ),
             bor3(!logi_cfs, b < 0L, ))
expect_equal(or3s(!logi_cft, g < 1L, ),
             bor3(!logi_cft, g < 1L, ))
expect_equal(or3s(!logi_cfu, a < 9L, ),
             bor3(!logi_cfu, a < 9L, ))
expect_equal(or3s(!logi_cfv, o < 0L, logi_m),
             bor3(!logi_cfv, o < 0L, logi_m))
expect_equal(or3s(!logi_cfw, v < 1L, logi_w),
             bor3(!logi_cfw, v < 1L, logi_w))
expect_equal(or3s(!logi_cfx, i < 9L, logi_h),
             bor3(!logi_cfx, i < 9L, logi_h))
expect_equal(or3s(!logi_cfy, , ),
             bor3(!logi_cfy, , ))
expect_equal(or3s(!logi_cfz, , ),
             bor3(!logi_cfz, , ))
expect_equal(or3s(!logi_cga, , ),
             bor3(!logi_cga, , ))
expect_equal(or3s(!logi_cgb, , !logi_z),
             bor3(!logi_cgb, , !logi_z))
expect_equal(or3s(!logi_cgc, , !logi_l),
             bor3(!logi_cgc, , !logi_l))
expect_equal(or3s(!logi_cgd, , !logi_y),
             bor3(!logi_cgd, , !logi_y))
expect_equal(or3s(!logi_cge, w < 0L, ),
             bor3(!logi_cge, w < 0L, ))
expect_equal(or3s(!logi_cgf, z < 1L, ),
             bor3(!logi_cgf, z < 1L, ))
expect_equal(or3s(!logi_cgg, n < 9L, ),
             bor3(!logi_cgg, n < 9L, ))
expect_equal(or3s(!logi_cgh, q < 0L, !logi_e),
             bor3(!logi_cgh, q < 0L, !logi_e))
expect_equal(or3s(!logi_cgi, l < 1L, !logi_t),
             bor3(!logi_cgi, l < 1L, !logi_t))
expect_equal(or3s(!logi_cgj, t < 9L, !logi_h),
             bor3(!logi_cgj, t < 9L, !logi_h))
expect_equal(or3s(!logi_cgk, , ),
             bor3(!logi_cgk, , ))
expect_equal(or3s(!logi_cgl, , ),
             bor3(!logi_cgl, , ))
expect_equal(or3s(!logi_cgm, , ),
             bor3(!logi_cgm, , ))
expect_equal(or3s(!logi_cgn, , z != 0L),
             bor3(!logi_cgn, , z != 0L))
expect_equal(or3s(!logi_cgo, , c != 1L),
             bor3(!logi_cgo, , c != 1L))
expect_equal(or3s(!logi_cgp, , e != 9L),
             bor3(!logi_cgp, , e != 9L))
expect_equal(or3s(!logi_cgq, e < 0L, ),
             bor3(!logi_cgq, e < 0L, ))
expect_equal(or3s(!logi_cgr, t < 1L, ),
             bor3(!logi_cgr, t < 1L, ))
expect_equal(or3s(!logi_cgs, c < 9L, ),
             bor3(!logi_cgs, c < 9L, ))
expect_equal(or3s(!logi_cgt, x < 0L, k != 0L),
             bor3(!logi_cgt, x < 0L, k != 0L))
expect_equal(or3s(!logi_cgu, s < 1L, q != 1L),
             bor3(!logi_cgu, s < 1L, q != 1L))
expect_equal(or3s(!logi_cgv, w < 9L, h != 9L),
             bor3(!logi_cgv, w < 9L, h != 9L))
expect_equal(or3s(!logi_cgw, , ),
             bor3(!logi_cgw, , ))
expect_equal(or3s(!logi_cgx, , ),
             bor3(!logi_cgx, , ))
expect_equal(or3s(!logi_cgy, , ),
             bor3(!logi_cgy, , ))
expect_equal(or3s(!logi_cgz, , k %between% c(-1L, 1L)),
             bor3(!logi_cgz, , k %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cha, , m %between% c(-1L, 1L)),
             bor3(!logi_cha, , m %between% c(-1L, 1L)))
expect_equal(or3s(!logi_chb, , u %between% c(-1L, 1L)),
             bor3(!logi_chb, , u %between% c(-1L, 1L)))
expect_equal(or3s(!logi_chc, c < 0L, ),
             bor3(!logi_chc, c < 0L, ))
expect_equal(or3s(!logi_chd, u < 1L, ),
             bor3(!logi_chd, u < 1L, ))
expect_equal(or3s(!logi_che, a < 9L, ),
             bor3(!logi_che, a < 9L, ))
expect_equal(or3s(!logi_chf, r < 0L, m %between% c(-1L, 1L)),
             bor3(!logi_chf, r < 0L, m %between% c(-1L, 1L)))
expect_equal(or3s(!logi_chg, t < 1L, h %between% c(-1L, 1L)),
             bor3(!logi_chg, t < 1L, h %between% c(-1L, 1L)))
expect_equal(or3s(!logi_chh, c < 9L, k %between% c(-1L, 1L)),
             bor3(!logi_chh, c < 9L, k %between% c(-1L, 1L)))
expect_equal(or3s(!logi_chi, , ),
             bor3(!logi_chi, , ))
expect_equal(or3s(!logi_chj, , ),
             bor3(!logi_chj, , ))
expect_equal(or3s(!logi_chk, , ),
             bor3(!logi_chk, , ))
expect_equal(or3s(!logi_chl, , u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_chl, , u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_chm, , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_chm, , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_chn, , o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_chn, , o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cho, h < 0L, ),
             bor3(!logi_cho, h < 0L, ))
expect_equal(or3s(!logi_chp, n < 1L, ),
             bor3(!logi_chp, n < 1L, ))
expect_equal(or3s(!logi_chq, m < 9L, ),
             bor3(!logi_chq, m < 9L, ))
expect_equal(or3s(!logi_chr, c < 0L, z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_chr, c < 0L, z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_chs, o < 1L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_chs, o < 1L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cht, e < 9L, m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cht, e < 9L, m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_chu, , ),
             bor3(!logi_chu, , ))
expect_equal(or3s(!logi_chv, , ),
             bor3(!logi_chv, , ))
expect_equal(or3s(!logi_chw, , ),
             bor3(!logi_chw, , ))
expect_equal(or3s(!logi_chx, , y %in% 1:4),
             bor3(!logi_chx, , y %in% 1:4))
expect_equal(or3s(!logi_chy, , m %in% 1:4),
             bor3(!logi_chy, , m %in% 1:4))
expect_equal(or3s(!logi_chz, , r %in% 1:4),
             bor3(!logi_chz, , r %in% 1:4))
expect_equal(or3s(!logi_cia, a < 0L, ),
             bor3(!logi_cia, a < 0L, ))
expect_equal(or3s(!logi_cib, m < 1L, ),
             bor3(!logi_cib, m < 1L, ))
expect_equal(or3s(!logi_cic, n < 9L, ),
             bor3(!logi_cic, n < 9L, ))
expect_equal(or3s(!logi_cid, r < 0L, f %in% 1:4),
             bor3(!logi_cid, r < 0L, f %in% 1:4))
expect_equal(or3s(!logi_cie, v < 1L, z %in% 1:4),
             bor3(!logi_cie, v < 1L, z %in% 1:4))
expect_equal(or3s(!logi_cif, o < 9L, v %in% 1:4),
             bor3(!logi_cif, o < 9L, v %in% 1:4))
expect_equal(or3s(!logi_cig, , ),
             bor3(!logi_cig, , ))
expect_equal(or3s(!logi_cih, , ),
             bor3(!logi_cih, , ))
expect_equal(or3s(!logi_cii, , ),
             bor3(!logi_cii, , ))
expect_equal(or3s(!logi_cij, , m < 0L),
             bor3(!logi_cij, , m < 0L))
expect_equal(or3s(!logi_cik, , b < 1L),
             bor3(!logi_cik, , b < 1L))
expect_equal(or3s(!logi_cil, , j < 9L),
             bor3(!logi_cil, , j < 9L))
expect_equal(or3s(!logi_cim, y < 0L, ),
             bor3(!logi_cim, y < 0L, ))
expect_equal(or3s(!logi_cin, t < 1L, ),
             bor3(!logi_cin, t < 1L, ))
expect_equal(or3s(!logi_cio, w < 9L, ),
             bor3(!logi_cio, w < 9L, ))
expect_equal(or3s(!logi_cip, p < 0L, k < 0L),
             bor3(!logi_cip, p < 0L, k < 0L))
expect_equal(or3s(!logi_ciq, a < 1L, f < 1L),
             bor3(!logi_ciq, a < 1L, f < 1L))
expect_equal(or3s(!logi_cir, l < 9L, v < 9L),
             bor3(!logi_cir, l < 9L, v < 9L))
expect_equal(or3s(!logi_cis, , ),
             bor3(!logi_cis, , ))
expect_equal(or3s(!logi_cit, , ),
             bor3(!logi_cit, , ))
expect_equal(or3s(!logi_ciu, , ),
             bor3(!logi_ciu, , ))
expect_equal(or3s(!logi_civ, , e <= 0L),
             bor3(!logi_civ, , e <= 0L))
expect_equal(or3s(!logi_ciw, , k <= 1L),
             bor3(!logi_ciw, , k <= 1L))
expect_equal(or3s(!logi_cix, , q <= 9L),
             bor3(!logi_cix, , q <= 9L))
expect_equal(or3s(!logi_ciy, u < 0L, ),
             bor3(!logi_ciy, u < 0L, ))
expect_equal(or3s(!logi_ciz, i < 1L, ),
             bor3(!logi_ciz, i < 1L, ))
expect_equal(or3s(!logi_cja, b < 9L, ),
             bor3(!logi_cja, b < 9L, ))
expect_equal(or3s(!logi_cjb, k < 0L, t <= 0L),
             bor3(!logi_cjb, k < 0L, t <= 0L))
expect_equal(or3s(!logi_cjc, u < 1L, s <= 1L),
             bor3(!logi_cjc, u < 1L, s <= 1L))
expect_equal(or3s(!logi_cjd, u < 9L, n <= 9L),
             bor3(!logi_cjd, u < 9L, n <= 9L))
expect_equal(or3s(!logi_cje, , ),
             bor3(!logi_cje, , ))
expect_equal(or3s(!logi_cjf, , ),
             bor3(!logi_cjf, , ))
expect_equal(or3s(!logi_cjg, , ),
             bor3(!logi_cjg, , ))
expect_equal(or3s(!logi_cjh, , s == 0L),
             bor3(!logi_cjh, , s == 0L))
expect_equal(or3s(!logi_cji, , n == 1L),
             bor3(!logi_cji, , n == 1L))
expect_equal(or3s(!logi_cjj, , q == 9L),
             bor3(!logi_cjj, , q == 9L))
expect_equal(or3s(!logi_cjk, l < 0L, ),
             bor3(!logi_cjk, l < 0L, ))
expect_equal(or3s(!logi_cjl, f < 1L, ),
             bor3(!logi_cjl, f < 1L, ))
expect_equal(or3s(!logi_cjm, s < 9L, ),
             bor3(!logi_cjm, s < 9L, ))
expect_equal(or3s(!logi_cjn, t < 0L, z == 0L),
             bor3(!logi_cjn, t < 0L, z == 0L))
expect_equal(or3s(!logi_cjo, j < 1L, n == 1L),
             bor3(!logi_cjo, j < 1L, n == 1L))
expect_equal(or3s(!logi_cjp, n < 9L, j == 9L),
             bor3(!logi_cjp, n < 9L, j == 9L))
expect_equal(or3s(!logi_cjq, , ),
             bor3(!logi_cjq, , ))
expect_equal(or3s(!logi_cjr, , ),
             bor3(!logi_cjr, , ))
expect_equal(or3s(!logi_cjs, , ),
             bor3(!logi_cjs, , ))
expect_equal(or3s(!logi_cjt, , e > 0L),
             bor3(!logi_cjt, , e > 0L))
expect_equal(or3s(!logi_cju, , h > 1L),
             bor3(!logi_cju, , h > 1L))
expect_equal(or3s(!logi_cjv, , s > 9L),
             bor3(!logi_cjv, , s > 9L))
expect_equal(or3s(!logi_cjw, a < 0L, ),
             bor3(!logi_cjw, a < 0L, ))
expect_equal(or3s(!logi_cjx, v < 1L, ),
             bor3(!logi_cjx, v < 1L, ))
expect_equal(or3s(!logi_cjy, a < 9L, ),
             bor3(!logi_cjy, a < 9L, ))
expect_equal(or3s(!logi_cjz, h < 0L, i > 0L),
             bor3(!logi_cjz, h < 0L, i > 0L))
expect_equal(or3s(!logi_cka, x < 1L, s > 1L),
             bor3(!logi_cka, x < 1L, s > 1L))
expect_equal(or3s(!logi_ckb, q < 9L, f > 9L),
             bor3(!logi_ckb, q < 9L, f > 9L))
expect_equal(or3s(!logi_ckc, , ),
             bor3(!logi_ckc, , ))
expect_equal(or3s(!logi_ckd, , ),
             bor3(!logi_ckd, , ))
expect_equal(or3s(!logi_cke, , ),
             bor3(!logi_cke, , ))
expect_equal(or3s(!logi_ckf, , w >= 0L),
             bor3(!logi_ckf, , w >= 0L))
expect_equal(or3s(!logi_ckg, , j >= 1L),
             bor3(!logi_ckg, , j >= 1L))
expect_equal(or3s(!logi_ckh, , t >= 9L),
             bor3(!logi_ckh, , t >= 9L))
expect_equal(or3s(!logi_cki, s < 0L, ),
             bor3(!logi_cki, s < 0L, ))
expect_equal(or3s(!logi_ckj, q < 1L, ),
             bor3(!logi_ckj, q < 1L, ))
expect_equal(or3s(!logi_ckk, y < 9L, ),
             bor3(!logi_ckk, y < 9L, ))
expect_equal(or3s(!logi_ckl, d < 0L, v >= 0L),
             bor3(!logi_ckl, d < 0L, v >= 0L))
expect_equal(or3s(!logi_ckm, g < 1L, z >= 1L),
             bor3(!logi_ckm, g < 1L, z >= 1L))
expect_equal(or3s(!logi_ckn, j < 9L, l >= 9L),
             bor3(!logi_ckn, j < 9L, l >= 9L))
expect_equal(or3s(!logi_cko, , ),
             bor3(!logi_cko, , ))
expect_equal(or3s(!logi_ckp, , ),
             bor3(!logi_ckp, , ))
expect_equal(or3s(!logi_ckq, , ),
             bor3(!logi_ckq, , ))
expect_equal(or3s(!logi_ckr, , logi_m),
             bor3(!logi_ckr, , logi_m))
expect_equal(or3s(!logi_cks, , logi_u),
             bor3(!logi_cks, , logi_u))
expect_equal(or3s(!logi_ckt, , logi_l),
             bor3(!logi_ckt, , logi_l))
expect_equal(or3s(!logi_cku, j <= 0L, ),
             bor3(!logi_cku, j <= 0L, ))
expect_equal(or3s(!logi_ckv, t <= 1L, ),
             bor3(!logi_ckv, t <= 1L, ))
expect_equal(or3s(!logi_ckw, m <= 9L, ),
             bor3(!logi_ckw, m <= 9L, ))
expect_equal(or3s(!logi_ckx, j <= 0L, logi_e),
             bor3(!logi_ckx, j <= 0L, logi_e))
expect_equal(or3s(!logi_cky, m <= 1L, logi_w),
             bor3(!logi_cky, m <= 1L, logi_w))
expect_equal(or3s(!logi_ckz, p <= 9L, logi_u),
             bor3(!logi_ckz, p <= 9L, logi_u))
expect_equal(or3s(!logi_cla, , ),
             bor3(!logi_cla, , ))
expect_equal(or3s(!logi_clb, , ),
             bor3(!logi_clb, , ))
expect_equal(or3s(!logi_clc, , ),
             bor3(!logi_clc, , ))
expect_equal(or3s(!logi_cld, , !logi_e),
             bor3(!logi_cld, , !logi_e))
expect_equal(or3s(!logi_cle, , !logi_i),
             bor3(!logi_cle, , !logi_i))
expect_equal(or3s(!logi_clf, , !logi_k),
             bor3(!logi_clf, , !logi_k))
expect_equal(or3s(!logi_clg, d <= 0L, ),
             bor3(!logi_clg, d <= 0L, ))
expect_equal(or3s(!logi_clh, h <= 1L, ),
             bor3(!logi_clh, h <= 1L, ))
expect_equal(or3s(!logi_cli, o <= 9L, ),
             bor3(!logi_cli, o <= 9L, ))
expect_equal(or3s(!logi_clj, v <= 0L, !logi_i),
             bor3(!logi_clj, v <= 0L, !logi_i))
expect_equal(or3s(!logi_clk, u <= 1L, !logi_n),
             bor3(!logi_clk, u <= 1L, !logi_n))
expect_equal(or3s(!logi_cll, v <= 9L, !logi_l),
             bor3(!logi_cll, v <= 9L, !logi_l))
expect_equal(or3s(!logi_clm, , ),
             bor3(!logi_clm, , ))
expect_equal(or3s(!logi_cln, , ),
             bor3(!logi_cln, , ))
expect_equal(or3s(!logi_clo, , ),
             bor3(!logi_clo, , ))
expect_equal(or3s(!logi_clp, , a != 0L),
             bor3(!logi_clp, , a != 0L))
expect_equal(or3s(!logi_clq, , l != 1L),
             bor3(!logi_clq, , l != 1L))
expect_equal(or3s(!logi_clr, , y != 9L),
             bor3(!logi_clr, , y != 9L))
expect_equal(or3s(!logi_cls, o <= 0L, ),
             bor3(!logi_cls, o <= 0L, ))
expect_equal(or3s(!logi_clt, h <= 1L, ),
             bor3(!logi_clt, h <= 1L, ))
expect_equal(or3s(!logi_clu, v <= 9L, ),
             bor3(!logi_clu, v <= 9L, ))
expect_equal(or3s(!logi_clv, p <= 0L, a != 0L),
             bor3(!logi_clv, p <= 0L, a != 0L))
expect_equal(or3s(!logi_clw, a <= 1L, f != 1L),
             bor3(!logi_clw, a <= 1L, f != 1L))
expect_equal(or3s(!logi_clx, m <= 9L, x != 9L),
             bor3(!logi_clx, m <= 9L, x != 9L))
expect_equal(or3s(!logi_cly, , ),
             bor3(!logi_cly, , ))
expect_equal(or3s(!logi_clz, , ),
             bor3(!logi_clz, , ))
expect_equal(or3s(!logi_cma, , ),
             bor3(!logi_cma, , ))
expect_equal(or3s(!logi_cmb, , w %between% c(-1L, 1L)),
             bor3(!logi_cmb, , w %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cmc, , w %between% c(-1L, 1L)),
             bor3(!logi_cmc, , w %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cmd, , a %between% c(-1L, 1L)),
             bor3(!logi_cmd, , a %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cme, p <= 0L, ),
             bor3(!logi_cme, p <= 0L, ))
expect_equal(or3s(!logi_cmf, u <= 1L, ),
             bor3(!logi_cmf, u <= 1L, ))
expect_equal(or3s(!logi_cmg, n <= 9L, ),
             bor3(!logi_cmg, n <= 9L, ))
expect_equal(or3s(!logi_cmh, l <= 0L, g %between% c(-1L, 1L)),
             bor3(!logi_cmh, l <= 0L, g %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cmi, x <= 1L, s %between% c(-1L, 1L)),
             bor3(!logi_cmi, x <= 1L, s %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cmj, c <= 9L, o %between% c(-1L, 1L)),
             bor3(!logi_cmj, c <= 9L, o %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cmk, , ),
             bor3(!logi_cmk, , ))
expect_equal(or3s(!logi_cml, , ),
             bor3(!logi_cml, , ))
expect_equal(or3s(!logi_cmm, , ),
             bor3(!logi_cmm, , ))
expect_equal(or3s(!logi_cmn, , u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cmn, , u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cmo, , x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cmo, , x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cmp, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cmp, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cmq, f <= 0L, ),
             bor3(!logi_cmq, f <= 0L, ))
expect_equal(or3s(!logi_cmr, c <= 1L, ),
             bor3(!logi_cmr, c <= 1L, ))
expect_equal(or3s(!logi_cms, f <= 9L, ),
             bor3(!logi_cms, f <= 9L, ))
expect_equal(or3s(!logi_cmt, h <= 0L, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cmt, h <= 0L, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cmu, w <= 1L, j %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cmu, w <= 1L, j %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cmv, z <= 9L, a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cmv, z <= 9L, a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cmw, , ),
             bor3(!logi_cmw, , ))
expect_equal(or3s(!logi_cmx, , ),
             bor3(!logi_cmx, , ))
expect_equal(or3s(!logi_cmy, , ),
             bor3(!logi_cmy, , ))
expect_equal(or3s(!logi_cmz, , j %in% 1:4),
             bor3(!logi_cmz, , j %in% 1:4))
expect_equal(or3s(!logi_cna, , v %in% 1:4),
             bor3(!logi_cna, , v %in% 1:4))
expect_equal(or3s(!logi_cnb, , t %in% 1:4),
             bor3(!logi_cnb, , t %in% 1:4))
expect_equal(or3s(!logi_cnc, p <= 0L, ),
             bor3(!logi_cnc, p <= 0L, ))
expect_equal(or3s(!logi_cnd, b <= 1L, ),
             bor3(!logi_cnd, b <= 1L, ))
expect_equal(or3s(!logi_cne, a <= 9L, ),
             bor3(!logi_cne, a <= 9L, ))
expect_equal(or3s(!logi_cnf, s <= 0L, r %in% 1:4),
             bor3(!logi_cnf, s <= 0L, r %in% 1:4))
expect_equal(or3s(!logi_cng, x <= 1L, b %in% 1:4),
             bor3(!logi_cng, x <= 1L, b %in% 1:4))
expect_equal(or3s(!logi_cnh, p <= 9L, f %in% 1:4),
             bor3(!logi_cnh, p <= 9L, f %in% 1:4))
expect_equal(or3s(!logi_cni, , ),
             bor3(!logi_cni, , ))
expect_equal(or3s(!logi_cnj, , ),
             bor3(!logi_cnj, , ))
expect_equal(or3s(!logi_cnk, , ),
             bor3(!logi_cnk, , ))
expect_equal(or3s(!logi_cnl, , r < 0L),
             bor3(!logi_cnl, , r < 0L))
expect_equal(or3s(!logi_cnm, , f < 1L),
             bor3(!logi_cnm, , f < 1L))
expect_equal(or3s(!logi_cnn, , s < 9L),
             bor3(!logi_cnn, , s < 9L))
expect_equal(or3s(!logi_cno, z <= 0L, ),
             bor3(!logi_cno, z <= 0L, ))
expect_equal(or3s(!logi_cnp, k <= 1L, ),
             bor3(!logi_cnp, k <= 1L, ))
expect_equal(or3s(!logi_cnq, d <= 9L, ),
             bor3(!logi_cnq, d <= 9L, ))
expect_equal(or3s(!logi_cnr, f <= 0L, u < 0L),
             bor3(!logi_cnr, f <= 0L, u < 0L))
expect_equal(or3s(!logi_cns, w <= 1L, f < 1L),
             bor3(!logi_cns, w <= 1L, f < 1L))
expect_equal(or3s(!logi_cnt, x <= 9L, i < 9L),
             bor3(!logi_cnt, x <= 9L, i < 9L))
expect_equal(or3s(!logi_cnu, , ),
             bor3(!logi_cnu, , ))
expect_equal(or3s(!logi_cnv, , ),
             bor3(!logi_cnv, , ))
expect_equal(or3s(!logi_cnw, , ),
             bor3(!logi_cnw, , ))
expect_equal(or3s(!logi_cnx, , f <= 0L),
             bor3(!logi_cnx, , f <= 0L))
expect_equal(or3s(!logi_cny, , o <= 1L),
             bor3(!logi_cny, , o <= 1L))
expect_equal(or3s(!logi_cnz, , g <= 9L),
             bor3(!logi_cnz, , g <= 9L))
expect_equal(or3s(!logi_coa, d <= 0L, ),
             bor3(!logi_coa, d <= 0L, ))
expect_equal(or3s(!logi_cob, x <= 1L, ),
             bor3(!logi_cob, x <= 1L, ))
expect_equal(or3s(!logi_coc, j <= 9L, ),
             bor3(!logi_coc, j <= 9L, ))
expect_equal(or3s(!logi_cod, c <= 0L, o <= 0L),
             bor3(!logi_cod, c <= 0L, o <= 0L))
expect_equal(or3s(!logi_coe, w <= 1L, u <= 1L),
             bor3(!logi_coe, w <= 1L, u <= 1L))
expect_equal(or3s(!logi_cof, h <= 9L, m <= 9L),
             bor3(!logi_cof, h <= 9L, m <= 9L))
expect_equal(or3s(!logi_cog, , ),
             bor3(!logi_cog, , ))
expect_equal(or3s(!logi_coh, , ),
             bor3(!logi_coh, , ))
expect_equal(or3s(!logi_coi, , ),
             bor3(!logi_coi, , ))
expect_equal(or3s(!logi_coj, , y == 0L),
             bor3(!logi_coj, , y == 0L))
expect_equal(or3s(!logi_cok, , c == 1L),
             bor3(!logi_cok, , c == 1L))
expect_equal(or3s(!logi_col, , o == 9L),
             bor3(!logi_col, , o == 9L))
expect_equal(or3s(!logi_com, z <= 0L, ),
             bor3(!logi_com, z <= 0L, ))
expect_equal(or3s(!logi_con, t <= 1L, ),
             bor3(!logi_con, t <= 1L, ))
expect_equal(or3s(!logi_coo, u <= 9L, ),
             bor3(!logi_coo, u <= 9L, ))
expect_equal(or3s(!logi_cop, r <= 0L, l == 0L),
             bor3(!logi_cop, r <= 0L, l == 0L))
expect_equal(or3s(!logi_coq, a <= 1L, k == 1L),
             bor3(!logi_coq, a <= 1L, k == 1L))
expect_equal(or3s(!logi_cor, u <= 9L, h == 9L),
             bor3(!logi_cor, u <= 9L, h == 9L))
expect_equal(or3s(!logi_cos, , ),
             bor3(!logi_cos, , ))
expect_equal(or3s(!logi_cot, , ),
             bor3(!logi_cot, , ))
expect_equal(or3s(!logi_cou, , ),
             bor3(!logi_cou, , ))
expect_equal(or3s(!logi_cov, , m > 0L),
             bor3(!logi_cov, , m > 0L))
expect_equal(or3s(!logi_cow, , r > 1L),
             bor3(!logi_cow, , r > 1L))
expect_equal(or3s(!logi_cox, , h > 9L),
             bor3(!logi_cox, , h > 9L))
expect_equal(or3s(!logi_coy, k <= 0L, ),
             bor3(!logi_coy, k <= 0L, ))
expect_equal(or3s(!logi_coz, h <= 1L, ),
             bor3(!logi_coz, h <= 1L, ))
expect_equal(or3s(!logi_cpa, q <= 9L, ),
             bor3(!logi_cpa, q <= 9L, ))
expect_equal(or3s(!logi_cpb, p <= 0L, g > 0L),
             bor3(!logi_cpb, p <= 0L, g > 0L))
expect_equal(or3s(!logi_cpc, x <= 1L, q > 1L),
             bor3(!logi_cpc, x <= 1L, q > 1L))
expect_equal(or3s(!logi_cpd, v <= 9L, d > 9L),
             bor3(!logi_cpd, v <= 9L, d > 9L))
expect_equal(or3s(!logi_cpe, , ),
             bor3(!logi_cpe, , ))
expect_equal(or3s(!logi_cpf, , ),
             bor3(!logi_cpf, , ))
expect_equal(or3s(!logi_cpg, , ),
             bor3(!logi_cpg, , ))
expect_equal(or3s(!logi_cph, , t >= 0L),
             bor3(!logi_cph, , t >= 0L))
expect_equal(or3s(!logi_cpi, , q >= 1L),
             bor3(!logi_cpi, , q >= 1L))
expect_equal(or3s(!logi_cpj, , j >= 9L),
             bor3(!logi_cpj, , j >= 9L))
expect_equal(or3s(!logi_cpk, o <= 0L, ),
             bor3(!logi_cpk, o <= 0L, ))
expect_equal(or3s(!logi_cpl, s <= 1L, ),
             bor3(!logi_cpl, s <= 1L, ))
expect_equal(or3s(!logi_cpm, d <= 9L, ),
             bor3(!logi_cpm, d <= 9L, ))
expect_equal(or3s(!logi_cpn, t <= 0L, g >= 0L),
             bor3(!logi_cpn, t <= 0L, g >= 0L))
expect_equal(or3s(!logi_cpo, k <= 1L, x >= 1L),
             bor3(!logi_cpo, k <= 1L, x >= 1L))
expect_equal(or3s(!logi_cpp, y <= 9L, c >= 9L),
             bor3(!logi_cpp, y <= 9L, c >= 9L))
expect_equal(or3s(!logi_cpq, , ),
             bor3(!logi_cpq, , ))
expect_equal(or3s(!logi_cpr, , ),
             bor3(!logi_cpr, , ))
expect_equal(or3s(!logi_cps, , ),
             bor3(!logi_cps, , ))
expect_equal(or3s(!logi_cpt, , logi_f),
             bor3(!logi_cpt, , logi_f))
expect_equal(or3s(!logi_cpu, , logi_o),
             bor3(!logi_cpu, , logi_o))
expect_equal(or3s(!logi_cpv, , logi_g),
             bor3(!logi_cpv, , logi_g))
expect_equal(or3s(!logi_cpw, f == 0L, ),
             bor3(!logi_cpw, f == 0L, ))
expect_equal(or3s(!logi_cpx, a == 1L, ),
             bor3(!logi_cpx, a == 1L, ))
expect_equal(or3s(!logi_cpy, c == 9L, ),
             bor3(!logi_cpy, c == 9L, ))
expect_equal(or3s(!logi_cpz, g == 0L, logi_j),
             bor3(!logi_cpz, g == 0L, logi_j))
expect_equal(or3s(!logi_cqa, h == 1L, logi_j),
             bor3(!logi_cqa, h == 1L, logi_j))
expect_equal(or3s(!logi_cqb, i == 9L, logi_b),
             bor3(!logi_cqb, i == 9L, logi_b))
expect_equal(or3s(!logi_cqc, , ),
             bor3(!logi_cqc, , ))
expect_equal(or3s(!logi_cqd, , ),
             bor3(!logi_cqd, , ))
expect_equal(or3s(!logi_cqe, , ),
             bor3(!logi_cqe, , ))
expect_equal(or3s(!logi_cqf, , !logi_c),
             bor3(!logi_cqf, , !logi_c))
expect_equal(or3s(!logi_cqg, , !logi_g),
             bor3(!logi_cqg, , !logi_g))
expect_equal(or3s(!logi_cqh, , !logi_n),
             bor3(!logi_cqh, , !logi_n))
expect_equal(or3s(!logi_cqi, d == 0L, ),
             bor3(!logi_cqi, d == 0L, ))
expect_equal(or3s(!logi_cqj, n == 1L, ),
             bor3(!logi_cqj, n == 1L, ))
expect_equal(or3s(!logi_cqk, x == 9L, ),
             bor3(!logi_cqk, x == 9L, ))
expect_equal(or3s(!logi_cql, v == 0L, !logi_v),
             bor3(!logi_cql, v == 0L, !logi_v))
expect_equal(or3s(!logi_cqm, z == 1L, !logi_h),
             bor3(!logi_cqm, z == 1L, !logi_h))
expect_equal(or3s(!logi_cqn, y == 9L, !logi_h),
             bor3(!logi_cqn, y == 9L, !logi_h))
expect_equal(or3s(!logi_cqo, , ),
             bor3(!logi_cqo, , ))
expect_equal(or3s(!logi_cqp, , ),
             bor3(!logi_cqp, , ))
expect_equal(or3s(!logi_cqq, , ),
             bor3(!logi_cqq, , ))
expect_equal(or3s(!logi_cqr, , i != 0L),
             bor3(!logi_cqr, , i != 0L))
expect_equal(or3s(!logi_cqs, , m != 1L),
             bor3(!logi_cqs, , m != 1L))
expect_equal(or3s(!logi_cqt, , q != 9L),
             bor3(!logi_cqt, , q != 9L))
expect_equal(or3s(!logi_cqu, r == 0L, ),
             bor3(!logi_cqu, r == 0L, ))
expect_equal(or3s(!logi_cqv, m == 1L, ),
             bor3(!logi_cqv, m == 1L, ))
expect_equal(or3s(!logi_cqw, y == 9L, ),
             bor3(!logi_cqw, y == 9L, ))
expect_equal(or3s(!logi_cqx, w == 0L, u != 0L),
             bor3(!logi_cqx, w == 0L, u != 0L))
expect_equal(or3s(!logi_cqy, g == 1L, y != 1L),
             bor3(!logi_cqy, g == 1L, y != 1L))
expect_equal(or3s(!logi_cqz, l == 9L, w != 9L),
             bor3(!logi_cqz, l == 9L, w != 9L))
expect_equal(or3s(!logi_cra, , ),
             bor3(!logi_cra, , ))
expect_equal(or3s(!logi_crb, , ),
             bor3(!logi_crb, , ))
expect_equal(or3s(!logi_crc, , ),
             bor3(!logi_crc, , ))
expect_equal(or3s(!logi_crd, , z %between% c(-1L, 1L)),
             bor3(!logi_crd, , z %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cre, , j %between% c(-1L, 1L)),
             bor3(!logi_cre, , j %between% c(-1L, 1L)))
expect_equal(or3s(!logi_crf, , d %between% c(-1L, 1L)),
             bor3(!logi_crf, , d %between% c(-1L, 1L)))
expect_equal(or3s(!logi_crg, j == 0L, ),
             bor3(!logi_crg, j == 0L, ))
expect_equal(or3s(!logi_crh, s == 1L, ),
             bor3(!logi_crh, s == 1L, ))
expect_equal(or3s(!logi_cri, u == 9L, ),
             bor3(!logi_cri, u == 9L, ))
expect_equal(or3s(!logi_crj, m == 0L, a %between% c(-1L, 1L)),
             bor3(!logi_crj, m == 0L, a %between% c(-1L, 1L)))
expect_equal(or3s(!logi_crk, v == 1L, m %between% c(-1L, 1L)),
             bor3(!logi_crk, v == 1L, m %between% c(-1L, 1L)))
expect_equal(or3s(!logi_crl, y == 9L, b %between% c(-1L, 1L)),
             bor3(!logi_crl, y == 9L, b %between% c(-1L, 1L)))
expect_equal(or3s(!logi_crm, , ),
             bor3(!logi_crm, , ))
expect_equal(or3s(!logi_crn, , ),
             bor3(!logi_crn, , ))
expect_equal(or3s(!logi_cro, , ),
             bor3(!logi_cro, , ))
expect_equal(or3s(!logi_crp, , o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_crp, , o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_crq, , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_crq, , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_crr, , r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_crr, , r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_crs, y == 0L, ),
             bor3(!logi_crs, y == 0L, ))
expect_equal(or3s(!logi_crt, p == 1L, ),
             bor3(!logi_crt, p == 1L, ))
expect_equal(or3s(!logi_cru, c == 9L, ),
             bor3(!logi_cru, c == 9L, ))
expect_equal(or3s(!logi_crv, c == 0L, u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_crv, c == 0L, u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_crw, n == 1L, u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_crw, n == 1L, u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_crx, a == 9L, y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_crx, a == 9L, y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cry, , ),
             bor3(!logi_cry, , ))
expect_equal(or3s(!logi_crz, , ),
             bor3(!logi_crz, , ))
expect_equal(or3s(!logi_csa, , ),
             bor3(!logi_csa, , ))
expect_equal(or3s(!logi_csb, , i %in% 1:4),
             bor3(!logi_csb, , i %in% 1:4))
expect_equal(or3s(!logi_csc, , g %in% 1:4),
             bor3(!logi_csc, , g %in% 1:4))
expect_equal(or3s(!logi_csd, , x %in% 1:4),
             bor3(!logi_csd, , x %in% 1:4))
expect_equal(or3s(!logi_cse, e == 0L, ),
             bor3(!logi_cse, e == 0L, ))
expect_equal(or3s(!logi_csf, h == 1L, ),
             bor3(!logi_csf, h == 1L, ))
expect_equal(or3s(!logi_csg, p == 9L, ),
             bor3(!logi_csg, p == 9L, ))
expect_equal(or3s(!logi_csh, n == 0L, j %in% 1:4),
             bor3(!logi_csh, n == 0L, j %in% 1:4))
expect_equal(or3s(!logi_csi, c == 1L, d %in% 1:4),
             bor3(!logi_csi, c == 1L, d %in% 1:4))
expect_equal(or3s(!logi_csj, a == 9L, j %in% 1:4),
             bor3(!logi_csj, a == 9L, j %in% 1:4))
expect_equal(or3s(!logi_csk, , ),
             bor3(!logi_csk, , ))
expect_equal(or3s(!logi_csl, , ),
             bor3(!logi_csl, , ))
expect_equal(or3s(!logi_csm, , ),
             bor3(!logi_csm, , ))
expect_equal(or3s(!logi_csn, , d < 0L),
             bor3(!logi_csn, , d < 0L))
expect_equal(or3s(!logi_cso, , j < 1L),
             bor3(!logi_cso, , j < 1L))
expect_equal(or3s(!logi_csp, , m < 9L),
             bor3(!logi_csp, , m < 9L))
expect_equal(or3s(!logi_csq, a == 0L, ),
             bor3(!logi_csq, a == 0L, ))
expect_equal(or3s(!logi_csr, z == 1L, ),
             bor3(!logi_csr, z == 1L, ))
expect_equal(or3s(!logi_css, i == 9L, ),
             bor3(!logi_css, i == 9L, ))
expect_equal(or3s(!logi_cst, f == 0L, q < 0L),
             bor3(!logi_cst, f == 0L, q < 0L))
expect_equal(or3s(!logi_csu, j == 1L, z < 1L),
             bor3(!logi_csu, j == 1L, z < 1L))
expect_equal(or3s(!logi_csv, w == 9L, o < 9L),
             bor3(!logi_csv, w == 9L, o < 9L))
expect_equal(or3s(!logi_csw, , ),
             bor3(!logi_csw, , ))
expect_equal(or3s(!logi_csx, , ),
             bor3(!logi_csx, , ))
expect_equal(or3s(!logi_csy, , ),
             bor3(!logi_csy, , ))
expect_equal(or3s(!logi_csz, , u <= 0L),
             bor3(!logi_csz, , u <= 0L))
expect_equal(or3s(!logi_cta, , f <= 1L),
             bor3(!logi_cta, , f <= 1L))
expect_equal(or3s(!logi_ctb, , l <= 9L),
             bor3(!logi_ctb, , l <= 9L))
expect_equal(or3s(!logi_ctc, e == 0L, ),
             bor3(!logi_ctc, e == 0L, ))
expect_equal(or3s(!logi_ctd, u == 1L, ),
             bor3(!logi_ctd, u == 1L, ))
expect_equal(or3s(!logi_cte, p == 9L, ),
             bor3(!logi_cte, p == 9L, ))
expect_equal(or3s(!logi_ctf, a == 0L, f <= 0L),
             bor3(!logi_ctf, a == 0L, f <= 0L))
expect_equal(or3s(!logi_ctg, d == 1L, p <= 1L),
             bor3(!logi_ctg, d == 1L, p <= 1L))
expect_equal(or3s(!logi_cth, e == 9L, e <= 9L),
             bor3(!logi_cth, e == 9L, e <= 9L))
expect_equal(or3s(!logi_cti, , ),
             bor3(!logi_cti, , ))
expect_equal(or3s(!logi_ctj, , ),
             bor3(!logi_ctj, , ))
expect_equal(or3s(!logi_ctk, , ),
             bor3(!logi_ctk, , ))
expect_equal(or3s(!logi_ctl, , v == 0L),
             bor3(!logi_ctl, , v == 0L))
expect_equal(or3s(!logi_ctm, , j == 1L),
             bor3(!logi_ctm, , j == 1L))
expect_equal(or3s(!logi_ctn, , y == 9L),
             bor3(!logi_ctn, , y == 9L))
expect_equal(or3s(!logi_cto, y == 0L, ),
             bor3(!logi_cto, y == 0L, ))
expect_equal(or3s(!logi_ctp, v == 1L, ),
             bor3(!logi_ctp, v == 1L, ))
expect_equal(or3s(!logi_ctq, g == 9L, ),
             bor3(!logi_ctq, g == 9L, ))
expect_equal(or3s(!logi_ctr, q == 0L, n == 0L),
             bor3(!logi_ctr, q == 0L, n == 0L))
expect_equal(or3s(!logi_cts, d == 1L, m == 1L),
             bor3(!logi_cts, d == 1L, m == 1L))
expect_equal(or3s(!logi_ctt, e == 9L, g == 9L),
             bor3(!logi_ctt, e == 9L, g == 9L))
expect_equal(or3s(!logi_ctu, , ),
             bor3(!logi_ctu, , ))
expect_equal(or3s(!logi_ctv, , ),
             bor3(!logi_ctv, , ))
expect_equal(or3s(!logi_ctw, , ),
             bor3(!logi_ctw, , ))
expect_equal(or3s(!logi_ctx, , o > 0L),
             bor3(!logi_ctx, , o > 0L))
expect_equal(or3s(!logi_cty, , s > 1L),
             bor3(!logi_cty, , s > 1L))
expect_equal(or3s(!logi_ctz, , g > 9L),
             bor3(!logi_ctz, , g > 9L))
expect_equal(or3s(!logi_cua, t == 0L, ),
             bor3(!logi_cua, t == 0L, ))
expect_equal(or3s(!logi_cub, f == 1L, ),
             bor3(!logi_cub, f == 1L, ))
expect_equal(or3s(!logi_cuc, x == 9L, ),
             bor3(!logi_cuc, x == 9L, ))
expect_equal(or3s(!logi_cud, y == 0L, k > 0L),
             bor3(!logi_cud, y == 0L, k > 0L))
expect_equal(or3s(!logi_cue, e == 1L, y > 1L),
             bor3(!logi_cue, e == 1L, y > 1L))
expect_equal(or3s(!logi_cuf, r == 9L, u > 9L),
             bor3(!logi_cuf, r == 9L, u > 9L))
expect_equal(or3s(!logi_cug, , ),
             bor3(!logi_cug, , ))
expect_equal(or3s(!logi_cuh, , ),
             bor3(!logi_cuh, , ))
expect_equal(or3s(!logi_cui, , ),
             bor3(!logi_cui, , ))
expect_equal(or3s(!logi_cuj, , t >= 0L),
             bor3(!logi_cuj, , t >= 0L))
expect_equal(or3s(!logi_cuk, , a >= 1L),
             bor3(!logi_cuk, , a >= 1L))
expect_equal(or3s(!logi_cul, , l >= 9L),
             bor3(!logi_cul, , l >= 9L))
expect_equal(or3s(!logi_cum, g == 0L, ),
             bor3(!logi_cum, g == 0L, ))
expect_equal(or3s(!logi_cun, t == 1L, ),
             bor3(!logi_cun, t == 1L, ))
expect_equal(or3s(!logi_cuo, t == 9L, ),
             bor3(!logi_cuo, t == 9L, ))
expect_equal(or3s(!logi_cup, d == 0L, v >= 0L),
             bor3(!logi_cup, d == 0L, v >= 0L))
expect_equal(or3s(!logi_cuq, i == 1L, t >= 1L),
             bor3(!logi_cuq, i == 1L, t >= 1L))
expect_equal(or3s(!logi_cur, m == 9L, j >= 9L),
             bor3(!logi_cur, m == 9L, j >= 9L))
expect_equal(or3s(!logi_cus, , ),
             bor3(!logi_cus, , ))
expect_equal(or3s(!logi_cut, , ),
             bor3(!logi_cut, , ))
expect_equal(or3s(!logi_cuu, , ),
             bor3(!logi_cuu, , ))
expect_equal(or3s(!logi_cuv, , logi_m),
             bor3(!logi_cuv, , logi_m))
expect_equal(or3s(!logi_cuw, , logi_h),
             bor3(!logi_cuw, , logi_h))
expect_equal(or3s(!logi_cux, , logi_j),
             bor3(!logi_cux, , logi_j))
expect_equal(or3s(!logi_cuy, c > 0L, ),
             bor3(!logi_cuy, c > 0L, ))
expect_equal(or3s(!logi_cuz, y > 1L, ),
             bor3(!logi_cuz, y > 1L, ))
expect_equal(or3s(!logi_cva, x > 9L, ),
             bor3(!logi_cva, x > 9L, ))
expect_equal(or3s(!logi_cvb, q > 0L, logi_k),
             bor3(!logi_cvb, q > 0L, logi_k))
expect_equal(or3s(!logi_cvc, d > 1L, logi_x),
             bor3(!logi_cvc, d > 1L, logi_x))
expect_equal(or3s(!logi_cvd, w > 9L, logi_a),
             bor3(!logi_cvd, w > 9L, logi_a))
expect_equal(or3s(!logi_cve, , ),
             bor3(!logi_cve, , ))
expect_equal(or3s(!logi_cvf, , ),
             bor3(!logi_cvf, , ))
expect_equal(or3s(!logi_cvg, , ),
             bor3(!logi_cvg, , ))
expect_equal(or3s(!logi_cvh, , !logi_z),
             bor3(!logi_cvh, , !logi_z))
expect_equal(or3s(!logi_cvi, , !logi_t),
             bor3(!logi_cvi, , !logi_t))
expect_equal(or3s(!logi_cvj, , !logi_s),
             bor3(!logi_cvj, , !logi_s))
expect_equal(or3s(!logi_cvk, k > 0L, ),
             bor3(!logi_cvk, k > 0L, ))
expect_equal(or3s(!logi_cvl, t > 1L, ),
             bor3(!logi_cvl, t > 1L, ))
expect_equal(or3s(!logi_cvm, s > 9L, ),
             bor3(!logi_cvm, s > 9L, ))
expect_equal(or3s(!logi_cvn, x > 0L, !logi_d),
             bor3(!logi_cvn, x > 0L, !logi_d))
expect_equal(or3s(!logi_cvo, l > 1L, !logi_f),
             bor3(!logi_cvo, l > 1L, !logi_f))
expect_equal(or3s(!logi_cvp, p > 9L, !logi_x),
             bor3(!logi_cvp, p > 9L, !logi_x))
expect_equal(or3s(!logi_cvq, , ),
             bor3(!logi_cvq, , ))
expect_equal(or3s(!logi_cvr, , ),
             bor3(!logi_cvr, , ))
expect_equal(or3s(!logi_cvs, , ),
             bor3(!logi_cvs, , ))
expect_equal(or3s(!logi_cvt, , y != 0L),
             bor3(!logi_cvt, , y != 0L))
expect_equal(or3s(!logi_cvu, , s != 1L),
             bor3(!logi_cvu, , s != 1L))
expect_equal(or3s(!logi_cvv, , v != 9L),
             bor3(!logi_cvv, , v != 9L))
expect_equal(or3s(!logi_cvw, w > 0L, ),
             bor3(!logi_cvw, w > 0L, ))
expect_equal(or3s(!logi_cvx, u > 1L, ),
             bor3(!logi_cvx, u > 1L, ))
expect_equal(or3s(!logi_cvy, u > 9L, ),
             bor3(!logi_cvy, u > 9L, ))
expect_equal(or3s(!logi_cvz, a > 0L, g != 0L),
             bor3(!logi_cvz, a > 0L, g != 0L))
expect_equal(or3s(!logi_cwa, x > 1L, g != 1L),
             bor3(!logi_cwa, x > 1L, g != 1L))
expect_equal(or3s(!logi_cwb, h > 9L, e != 9L),
             bor3(!logi_cwb, h > 9L, e != 9L))
expect_equal(or3s(!logi_cwc, , ),
             bor3(!logi_cwc, , ))
expect_equal(or3s(!logi_cwd, , ),
             bor3(!logi_cwd, , ))
expect_equal(or3s(!logi_cwe, , ),
             bor3(!logi_cwe, , ))
expect_equal(or3s(!logi_cwf, , r %between% c(-1L, 1L)),
             bor3(!logi_cwf, , r %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cwg, , z %between% c(-1L, 1L)),
             bor3(!logi_cwg, , z %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cwh, , o %between% c(-1L, 1L)),
             bor3(!logi_cwh, , o %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cwi, q > 0L, ),
             bor3(!logi_cwi, q > 0L, ))
expect_equal(or3s(!logi_cwj, l > 1L, ),
             bor3(!logi_cwj, l > 1L, ))
expect_equal(or3s(!logi_cwk, x > 9L, ),
             bor3(!logi_cwk, x > 9L, ))
expect_equal(or3s(!logi_cwl, e > 0L, m %between% c(-1L, 1L)),
             bor3(!logi_cwl, e > 0L, m %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cwm, c > 1L, u %between% c(-1L, 1L)),
             bor3(!logi_cwm, c > 1L, u %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cwn, c > 9L, y %between% c(-1L, 1L)),
             bor3(!logi_cwn, c > 9L, y %between% c(-1L, 1L)))
expect_equal(or3s(!logi_cwo, , ),
             bor3(!logi_cwo, , ))
expect_equal(or3s(!logi_cwp, , ),
             bor3(!logi_cwp, , ))
expect_equal(or3s(!logi_cwq, , ),
             bor3(!logi_cwq, , ))
expect_equal(or3s(!logi_cwr, , o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cwr, , o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cws, , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cws, , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cwt, , a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cwt, , a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cwu, r > 0L, ),
             bor3(!logi_cwu, r > 0L, ))
expect_equal(or3s(!logi_cwv, r > 1L, ),
             bor3(!logi_cwv, r > 1L, ))
expect_equal(or3s(!logi_cww, k > 9L, ),
             bor3(!logi_cww, k > 9L, ))
expect_equal(or3s(!logi_cwx, b > 0L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cwx, b > 0L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cwy, a > 1L, p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cwy, a > 1L, p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cwz, q > 9L, n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_cwz, q > 9L, n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_cxa, , ),
             bor3(!logi_cxa, , ))
expect_equal(or3s(!logi_cxb, , ),
             bor3(!logi_cxb, , ))
expect_equal(or3s(!logi_cxc, , ),
             bor3(!logi_cxc, , ))
expect_equal(or3s(!logi_cxd, , r %in% 1:4),
             bor3(!logi_cxd, , r %in% 1:4))
expect_equal(or3s(!logi_cxe, , q %in% 1:4),
             bor3(!logi_cxe, , q %in% 1:4))
expect_equal(or3s(!logi_cxf, , z %in% 1:4),
             bor3(!logi_cxf, , z %in% 1:4))
expect_equal(or3s(!logi_cxg, z > 0L, ),
             bor3(!logi_cxg, z > 0L, ))
expect_equal(or3s(!logi_cxh, z > 1L, ),
             bor3(!logi_cxh, z > 1L, ))
expect_equal(or3s(!logi_cxi, g > 9L, ),
             bor3(!logi_cxi, g > 9L, ))
expect_equal(or3s(!logi_cxj, n > 0L, e %in% 1:4),
             bor3(!logi_cxj, n > 0L, e %in% 1:4))
expect_equal(or3s(!logi_cxk, y > 1L, a %in% 1:4),
             bor3(!logi_cxk, y > 1L, a %in% 1:4))
expect_equal(or3s(!logi_cxl, x > 9L, k %in% 1:4),
             bor3(!logi_cxl, x > 9L, k %in% 1:4))
expect_equal(or3s(!logi_cxm, , ),
             bor3(!logi_cxm, , ))
expect_equal(or3s(!logi_cxn, , ),
             bor3(!logi_cxn, , ))
expect_equal(or3s(!logi_cxo, , ),
             bor3(!logi_cxo, , ))
expect_equal(or3s(!logi_cxp, , k < 0L),
             bor3(!logi_cxp, , k < 0L))
expect_equal(or3s(!logi_cxq, , h < 1L),
             bor3(!logi_cxq, , h < 1L))
expect_equal(or3s(!logi_cxr, , t < 9L),
             bor3(!logi_cxr, , t < 9L))
expect_equal(or3s(!logi_cxs, r > 0L, ),
             bor3(!logi_cxs, r > 0L, ))
expect_equal(or3s(!logi_cxt, m > 1L, ),
             bor3(!logi_cxt, m > 1L, ))
expect_equal(or3s(!logi_cxu, j > 9L, ),
             bor3(!logi_cxu, j > 9L, ))
expect_equal(or3s(!logi_cxv, a > 0L, b < 0L),
             bor3(!logi_cxv, a > 0L, b < 0L))
expect_equal(or3s(!logi_cxw, z > 1L, e < 1L),
             bor3(!logi_cxw, z > 1L, e < 1L))
expect_equal(or3s(!logi_cxx, h > 9L, a < 9L),
             bor3(!logi_cxx, h > 9L, a < 9L))
expect_equal(or3s(!logi_cxy, , ),
             bor3(!logi_cxy, , ))
expect_equal(or3s(!logi_cxz, , ),
             bor3(!logi_cxz, , ))
expect_equal(or3s(!logi_cya, , ),
             bor3(!logi_cya, , ))
expect_equal(or3s(!logi_cyb, , q <= 0L),
             bor3(!logi_cyb, , q <= 0L))
expect_equal(or3s(!logi_cyc, , o <= 1L),
             bor3(!logi_cyc, , o <= 1L))
expect_equal(or3s(!logi_cyd, , f <= 9L),
             bor3(!logi_cyd, , f <= 9L))
expect_equal(or3s(!logi_cye, s > 0L, ),
             bor3(!logi_cye, s > 0L, ))
expect_equal(or3s(!logi_cyf, v > 1L, ),
             bor3(!logi_cyf, v > 1L, ))
expect_equal(or3s(!logi_cyg, r > 9L, ),
             bor3(!logi_cyg, r > 9L, ))
expect_equal(or3s(!logi_cyh, b > 0L, x <= 0L),
             bor3(!logi_cyh, b > 0L, x <= 0L))
expect_equal(or3s(!logi_cyi, w > 1L, m <= 1L),
             bor3(!logi_cyi, w > 1L, m <= 1L))
expect_equal(or3s(!logi_cyj, c > 9L, u <= 9L),
             bor3(!logi_cyj, c > 9L, u <= 9L))
expect_equal(or3s(!logi_cyk, , ),
             bor3(!logi_cyk, , ))
expect_equal(or3s(!logi_cyl, , ),
             bor3(!logi_cyl, , ))
expect_equal(or3s(!logi_cym, , ),
             bor3(!logi_cym, , ))
expect_equal(or3s(!logi_cyn, , x == 0L),
             bor3(!logi_cyn, , x == 0L))
expect_equal(or3s(!logi_cyo, , n == 1L),
             bor3(!logi_cyo, , n == 1L))
expect_equal(or3s(!logi_cyp, , p == 9L),
             bor3(!logi_cyp, , p == 9L))
expect_equal(or3s(!logi_cyq, r > 0L, ),
             bor3(!logi_cyq, r > 0L, ))
expect_equal(or3s(!logi_cyr, v > 1L, ),
             bor3(!logi_cyr, v > 1L, ))
expect_equal(or3s(!logi_cys, r > 9L, ),
             bor3(!logi_cys, r > 9L, ))
expect_equal(or3s(!logi_cyt, y > 0L, c == 0L),
             bor3(!logi_cyt, y > 0L, c == 0L))
expect_equal(or3s(!logi_cyu, t > 1L, r == 1L),
             bor3(!logi_cyu, t > 1L, r == 1L))
expect_equal(or3s(!logi_cyv, m > 9L, v == 9L),
             bor3(!logi_cyv, m > 9L, v == 9L))
expect_equal(or3s(!logi_cyw, , ),
             bor3(!logi_cyw, , ))
expect_equal(or3s(!logi_cyx, , ),
             bor3(!logi_cyx, , ))
expect_equal(or3s(!logi_cyy, , ),
             bor3(!logi_cyy, , ))
expect_equal(or3s(!logi_cyz, , w > 0L),
             bor3(!logi_cyz, , w > 0L))
expect_equal(or3s(!logi_da, , f > 1L),
             bor3(!logi_da, , f > 1L))
expect_equal(or3s(!logi_db, , f > 9L),
             bor3(!logi_db, , f > 9L))
expect_equal(or3s(!logi_dc, m > 0L, ),
             bor3(!logi_dc, m > 0L, ))
expect_equal(or3s(!logi_dd, d > 1L, ),
             bor3(!logi_dd, d > 1L, ))
expect_equal(or3s(!logi_de, o > 9L, ),
             bor3(!logi_de, o > 9L, ))
expect_equal(or3s(!logi_df, f > 0L, y > 0L),
             bor3(!logi_df, f > 0L, y > 0L))
expect_equal(or3s(!logi_dg, i > 1L, y > 1L),
             bor3(!logi_dg, i > 1L, y > 1L))
expect_equal(or3s(!logi_dh, b > 9L, n > 9L),
             bor3(!logi_dh, b > 9L, n > 9L))
expect_equal(or3s(!logi_di, , ),
             bor3(!logi_di, , ))
expect_equal(or3s(!logi_dj, , ),
             bor3(!logi_dj, , ))
expect_equal(or3s(!logi_dk, , ),
             bor3(!logi_dk, , ))
expect_equal(or3s(!logi_dl, , z >= 0L),
             bor3(!logi_dl, , z >= 0L))
expect_equal(or3s(!logi_dm, , v >= 1L),
             bor3(!logi_dm, , v >= 1L))
expect_equal(or3s(!logi_dn, , y >= 9L),
             bor3(!logi_dn, , y >= 9L))
expect_equal(or3s(!logi_do, f > 0L, ),
             bor3(!logi_do, f > 0L, ))
expect_equal(or3s(!logi_dp, g > 1L, ),
             bor3(!logi_dp, g > 1L, ))
expect_equal(or3s(!logi_dq, s > 9L, ),
             bor3(!logi_dq, s > 9L, ))
expect_equal(or3s(!logi_dr, a > 0L, u >= 0L),
             bor3(!logi_dr, a > 0L, u >= 0L))
expect_equal(or3s(!logi_ds, v > 1L, q >= 1L),
             bor3(!logi_ds, v > 1L, q >= 1L))
expect_equal(or3s(!logi_dt, w > 9L, o >= 9L),
             bor3(!logi_dt, w > 9L, o >= 9L))
expect_equal(or3s(!logi_du, , ),
             bor3(!logi_du, , ))
expect_equal(or3s(!logi_dv, , ),
             bor3(!logi_dv, , ))
expect_equal(or3s(!logi_dw, , ),
             bor3(!logi_dw, , ))
expect_equal(or3s(!logi_dx, , logi_z),
             bor3(!logi_dx, , logi_z))
expect_equal(or3s(!logi_dy, , logi_o),
             bor3(!logi_dy, , logi_o))
expect_equal(or3s(!logi_dz, , logi_r),
             bor3(!logi_dz, , logi_r))
expect_equal(or3s(!logi_daa, d >= 0L, ),
             bor3(!logi_daa, d >= 0L, ))
expect_equal(or3s(!logi_dab, k >= 1L, ),
             bor3(!logi_dab, k >= 1L, ))
expect_equal(or3s(!logi_dac, m >= 9L, ),
             bor3(!logi_dac, m >= 9L, ))
expect_equal(or3s(!logi_dad, h >= 0L, logi_z),
             bor3(!logi_dad, h >= 0L, logi_z))
expect_equal(or3s(!logi_dae, p >= 1L, logi_o),
             bor3(!logi_dae, p >= 1L, logi_o))
expect_equal(or3s(!logi_daf, z >= 9L, logi_x),
             bor3(!logi_daf, z >= 9L, logi_x))
expect_equal(or3s(!logi_dag, , ),
             bor3(!logi_dag, , ))
expect_equal(or3s(!logi_dah, , ),
             bor3(!logi_dah, , ))
expect_equal(or3s(!logi_dai, , ),
             bor3(!logi_dai, , ))
expect_equal(or3s(!logi_daj, , !logi_l),
             bor3(!logi_daj, , !logi_l))
expect_equal(or3s(!logi_dak, , !logi_p),
             bor3(!logi_dak, , !logi_p))
expect_equal(or3s(!logi_dal, , !logi_q),
             bor3(!logi_dal, , !logi_q))
expect_equal(or3s(!logi_dam, z >= 0L, ),
             bor3(!logi_dam, z >= 0L, ))
expect_equal(or3s(!logi_dan, p >= 1L, ),
             bor3(!logi_dan, p >= 1L, ))
expect_equal(or3s(!logi_dao, g >= 9L, ),
             bor3(!logi_dao, g >= 9L, ))
expect_equal(or3s(!logi_dap, a >= 0L, !logi_t),
             bor3(!logi_dap, a >= 0L, !logi_t))
expect_equal(or3s(!logi_daq, s >= 1L, !logi_f),
             bor3(!logi_daq, s >= 1L, !logi_f))
expect_equal(or3s(!logi_dar, l >= 9L, !logi_y),
             bor3(!logi_dar, l >= 9L, !logi_y))
expect_equal(or3s(!logi_das, , ),
             bor3(!logi_das, , ))
expect_equal(or3s(!logi_dat, , ),
             bor3(!logi_dat, , ))
expect_equal(or3s(!logi_dau, , ),
             bor3(!logi_dau, , ))
expect_equal(or3s(!logi_dav, , b != 0L),
             bor3(!logi_dav, , b != 0L))
expect_equal(or3s(!logi_daw, , k != 1L),
             bor3(!logi_daw, , k != 1L))
expect_equal(or3s(!logi_dax, , j != 9L),
             bor3(!logi_dax, , j != 9L))
expect_equal(or3s(!logi_day, w >= 0L, ),
             bor3(!logi_day, w >= 0L, ))
expect_equal(or3s(!logi_daz, c >= 1L, ),
             bor3(!logi_daz, c >= 1L, ))
expect_equal(or3s(!logi_dba, z >= 9L, ),
             bor3(!logi_dba, z >= 9L, ))
expect_equal(or3s(!logi_dbb, q >= 0L, g != 0L),
             bor3(!logi_dbb, q >= 0L, g != 0L))
expect_equal(or3s(!logi_dbc, w >= 1L, q != 1L),
             bor3(!logi_dbc, w >= 1L, q != 1L))
expect_equal(or3s(!logi_dbd, a >= 9L, x != 9L),
             bor3(!logi_dbd, a >= 9L, x != 9L))
expect_equal(or3s(!logi_dbe, , ),
             bor3(!logi_dbe, , ))
expect_equal(or3s(!logi_dbf, , ),
             bor3(!logi_dbf, , ))
expect_equal(or3s(!logi_dbg, , ),
             bor3(!logi_dbg, , ))
expect_equal(or3s(!logi_dbh, , f %between% c(-1L, 1L)),
             bor3(!logi_dbh, , f %between% c(-1L, 1L)))
expect_equal(or3s(!logi_dbi, , h %between% c(-1L, 1L)),
             bor3(!logi_dbi, , h %between% c(-1L, 1L)))
expect_equal(or3s(!logi_dbj, , r %between% c(-1L, 1L)),
             bor3(!logi_dbj, , r %between% c(-1L, 1L)))
expect_equal(or3s(!logi_dbk, x >= 0L, ),
             bor3(!logi_dbk, x >= 0L, ))
expect_equal(or3s(!logi_dbl, c >= 1L, ),
             bor3(!logi_dbl, c >= 1L, ))
expect_equal(or3s(!logi_dbm, n >= 9L, ),
             bor3(!logi_dbm, n >= 9L, ))
expect_equal(or3s(!logi_dbn, x >= 0L, k %between% c(-1L, 1L)),
             bor3(!logi_dbn, x >= 0L, k %between% c(-1L, 1L)))
expect_equal(or3s(!logi_dbo, y >= 1L, k %between% c(-1L, 1L)),
             bor3(!logi_dbo, y >= 1L, k %between% c(-1L, 1L)))
expect_equal(or3s(!logi_dbp, l >= 9L, u %between% c(-1L, 1L)),
             bor3(!logi_dbp, l >= 9L, u %between% c(-1L, 1L)))
expect_equal(or3s(!logi_dbq, , ),
             bor3(!logi_dbq, , ))
expect_equal(or3s(!logi_dbr, , ),
             bor3(!logi_dbr, , ))
expect_equal(or3s(!logi_dbs, , ),
             bor3(!logi_dbs, , ))
expect_equal(or3s(!logi_dbt, , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_dbt, , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_dbu, , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_dbu, , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_dbv, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_dbv, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_dbw, g >= 0L, ),
             bor3(!logi_dbw, g >= 0L, ))
expect_equal(or3s(!logi_dbx, u >= 1L, ),
             bor3(!logi_dbx, u >= 1L, ))
expect_equal(or3s(!logi_dby, r >= 9L, ),
             bor3(!logi_dby, r >= 9L, ))
expect_equal(or3s(!logi_dbz, c >= 0L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_dbz, c >= 0L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_dca, m >= 1L, x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_dca, m >= 1L, x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_dcb, q >= 9L, p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(!logi_dcb, q >= 9L, p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(!logi_dcc, , ),
             bor3(!logi_dcc, , ))
expect_equal(or3s(!logi_dcd, , ),
             bor3(!logi_dcd, , ))
expect_equal(or3s(!logi_dce, , ),
             bor3(!logi_dce, , ))
expect_equal(or3s(!logi_dcf, , k %in% 1:4),
             bor3(!logi_dcf, , k %in% 1:4))
expect_equal(or3s(!logi_dcg, , g %in% 1:4),
             bor3(!logi_dcg, , g %in% 1:4))
expect_equal(or3s(!logi_dch, , x %in% 1:4),
             bor3(!logi_dch, , x %in% 1:4))
expect_equal(or3s(!logi_dci, m >= 0L, ),
             bor3(!logi_dci, m >= 0L, ))
expect_equal(or3s(!logi_dcj, e >= 1L, ),
             bor3(!logi_dcj, e >= 1L, ))
expect_equal(or3s(!logi_dck, d >= 9L, ),
             bor3(!logi_dck, d >= 9L, ))
expect_equal(or3s(!logi_dcl, v >= 0L, k %in% 1:4),
             bor3(!logi_dcl, v >= 0L, k %in% 1:4))
expect_equal(or3s(!logi_dcm, p >= 1L, l %in% 1:4),
             bor3(!logi_dcm, p >= 1L, l %in% 1:4))
expect_equal(or3s(!logi_dcn, g >= 9L, s %in% 1:4),
             bor3(!logi_dcn, g >= 9L, s %in% 1:4))
expect_equal(or3s(!logi_dco, , ),
             bor3(!logi_dco, , ))
expect_equal(or3s(!logi_dcp, , ),
             bor3(!logi_dcp, , ))
expect_equal(or3s(!logi_dcq, , ),
             bor3(!logi_dcq, , ))
expect_equal(or3s(!logi_dcr, , e < 0L),
             bor3(!logi_dcr, , e < 0L))
expect_equal(or3s(!logi_dcs, , x < 1L),
             bor3(!logi_dcs, , x < 1L))
expect_equal(or3s(!logi_dct, , f < 9L),
             bor3(!logi_dct, , f < 9L))
expect_equal(or3s(!logi_dcu, v >= 0L, ),
             bor3(!logi_dcu, v >= 0L, ))
expect_equal(or3s(!logi_dcv, x >= 1L, ),
             bor3(!logi_dcv, x >= 1L, ))
expect_equal(or3s(!logi_dcw, n >= 9L, ),
             bor3(!logi_dcw, n >= 9L, ))
expect_equal(or3s(!logi_dcx, w >= 0L, l < 0L),
             bor3(!logi_dcx, w >= 0L, l < 0L))
expect_equal(or3s(!logi_dcy, n >= 1L, u < 1L),
             bor3(!logi_dcy, n >= 1L, u < 1L))
expect_equal(or3s(!logi_dcz, h >= 9L, q < 9L),
             bor3(!logi_dcz, h >= 9L, q < 9L))
expect_equal(or3s(!logi_dda, , ),
             bor3(!logi_dda, , ))
expect_equal(or3s(!logi_ddb, , ),
             bor3(!logi_ddb, , ))
expect_equal(or3s(!logi_ddc, , ),
             bor3(!logi_ddc, , ))
expect_equal(or3s(!logi_ddd, , g <= 0L),
             bor3(!logi_ddd, , g <= 0L))
expect_equal(or3s(!logi_dde, , o <= 1L),
             bor3(!logi_dde, , o <= 1L))
expect_equal(or3s(!logi_ddf, , c <= 9L),
             bor3(!logi_ddf, , c <= 9L))
expect_equal(or3s(!logi_ddg, i >= 0L, ),
             bor3(!logi_ddg, i >= 0L, ))
expect_equal(or3s(!logi_ddh, q >= 1L, ),
             bor3(!logi_ddh, q >= 1L, ))
expect_equal(or3s(!logi_ddi, o >= 9L, ),
             bor3(!logi_ddi, o >= 9L, ))
expect_equal(or3s(!logi_ddj, v >= 0L, h <= 0L),
             bor3(!logi_ddj, v >= 0L, h <= 0L))
expect_equal(or3s(!logi_ddk, y >= 1L, m <= 1L),
             bor3(!logi_ddk, y >= 1L, m <= 1L))
expect_equal(or3s(!logi_ddl, z >= 9L, l <= 9L),
             bor3(!logi_ddl, z >= 9L, l <= 9L))
expect_equal(or3s(!logi_ddm, , ),
             bor3(!logi_ddm, , ))
expect_equal(or3s(!logi_ddn, , ),
             bor3(!logi_ddn, , ))
expect_equal(or3s(!logi_ddo, , ),
             bor3(!logi_ddo, , ))
expect_equal(or3s(!logi_ddp, , p == 0L),
             bor3(!logi_ddp, , p == 0L))
expect_equal(or3s(!logi_ddq, , z == 1L),
             bor3(!logi_ddq, , z == 1L))
expect_equal(or3s(!logi_ddr, , x == 9L),
             bor3(!logi_ddr, , x == 9L))
expect_equal(or3s(!logi_dds, m >= 0L, ),
             bor3(!logi_dds, m >= 0L, ))
expect_equal(or3s(!logi_ddt, j >= 1L, ),
             bor3(!logi_ddt, j >= 1L, ))
expect_equal(or3s(!logi_ddu, s >= 9L, ),
             bor3(!logi_ddu, s >= 9L, ))
expect_equal(or3s(!logi_ddv, a >= 0L, s == 0L),
             bor3(!logi_ddv, a >= 0L, s == 0L))
expect_equal(or3s(!logi_ddw, t >= 1L, c == 1L),
             bor3(!logi_ddw, t >= 1L, c == 1L))
expect_equal(or3s(!logi_ddx, h >= 9L, x == 9L),
             bor3(!logi_ddx, h >= 9L, x == 9L))
expect_equal(or3s(!logi_ddy, , ),
             bor3(!logi_ddy, , ))
expect_equal(or3s(!logi_ddz, , ),
             bor3(!logi_ddz, , ))
expect_equal(or3s(!logi_dea, , ),
             bor3(!logi_dea, , ))
expect_equal(or3s(!logi_deb, , d > 0L),
             bor3(!logi_deb, , d > 0L))
expect_equal(or3s(!logi_dec, , u > 1L),
             bor3(!logi_dec, , u > 1L))
expect_equal(or3s(!logi_ded, , a > 9L),
             bor3(!logi_ded, , a > 9L))
expect_equal(or3s(!logi_dee, g >= 0L, ),
             bor3(!logi_dee, g >= 0L, ))
expect_equal(or3s(!logi_def, m >= 1L, ),
             bor3(!logi_def, m >= 1L, ))
expect_equal(or3s(!logi_deg, p >= 9L, ),
             bor3(!logi_deg, p >= 9L, ))
expect_equal(or3s(!logi_deh, e >= 0L, i > 0L),
             bor3(!logi_deh, e >= 0L, i > 0L))
expect_equal(or3s(!logi_dei, n >= 1L, i > 1L),
             bor3(!logi_dei, n >= 1L, i > 1L))
expect_equal(or3s(!logi_dej, n >= 9L, j > 9L),
             bor3(!logi_dej, n >= 9L, j > 9L))
expect_equal(or3s(!logi_dek, , ),
             bor3(!logi_dek, , ))
expect_equal(or3s(!logi_del, , ),
             bor3(!logi_del, , ))
expect_equal(or3s(!logi_dem, , ),
             bor3(!logi_dem, , ))
expect_equal(or3s(!logi_den, , l >= 0L),
             bor3(!logi_den, , l >= 0L))
expect_equal(or3s(!logi_deo, , v >= 1L),
             bor3(!logi_deo, , v >= 1L))
expect_equal(or3s(!logi_dep, , w >= 9L),
             bor3(!logi_dep, , w >= 9L))
expect_equal(or3s(!logi_deq, t >= 0L, ),
             bor3(!logi_deq, t >= 0L, ))
expect_equal(or3s(!logi_der, a >= 1L, ),
             bor3(!logi_der, a >= 1L, ))
expect_equal(or3s(!logi_des, n >= 9L, ),
             bor3(!logi_des, n >= 9L, ))
expect_equal(or3s(!logi_det, b >= 0L, y >= 0L),
             bor3(!logi_det, b >= 0L, y >= 0L))
expect_equal(or3s(!logi_deu, r >= 1L, q >= 1L),
             bor3(!logi_deu, r >= 1L, q >= 1L))
expect_equal(or3s(!logi_dev, q >= 9L, z >= 9L),
             bor3(!logi_dev, q >= 9L, z >= 9L))
expect_equal(or3s(dew != 0L, , ),
             bor3(dew != 0L, , ))
expect_equal(or3s(dex != 1L, , ),
             bor3(dex != 1L, , ))
expect_equal(or3s(dey != 9L, , ),
             bor3(dey != 9L, , ))
expect_equal(or3s(dez != 0L, , logi_j),
             bor3(dez != 0L, , logi_j))
expect_equal(or3s(dfa != 1L, , logi_v),
             bor3(dfa != 1L, , logi_v))
expect_equal(or3s(dfb != 9L, , logi_c),
             bor3(dfb != 9L, , logi_c))
expect_equal(or3s(dfc != 0L, logi_t, ),
             bor3(dfc != 0L, logi_t, ))
expect_equal(or3s(dfd != 1L, logi_w, ),
             bor3(dfd != 1L, logi_w, ))
expect_equal(or3s(dfe != 9L, logi_k, ),
             bor3(dfe != 9L, logi_k, ))
expect_equal(or3s(dff != 0L, logi_b, logi_u),
             bor3(dff != 0L, logi_b, logi_u))
expect_equal(or3s(dfg != 1L, logi_y, logi_o),
             bor3(dfg != 1L, logi_y, logi_o))
expect_equal(or3s(dfh != 9L, logi_h, logi_h),
             bor3(dfh != 9L, logi_h, logi_h))
expect_equal(or3s(dfi != 0L, , ),
             bor3(dfi != 0L, , ))
expect_equal(or3s(dfj != 1L, , ),
             bor3(dfj != 1L, , ))
expect_equal(or3s(dfk != 9L, , ),
             bor3(dfk != 9L, , ))
expect_equal(or3s(dfl != 0L, , !logi_n),
             bor3(dfl != 0L, , !logi_n))
expect_equal(or3s(dfm != 1L, , !logi_n),
             bor3(dfm != 1L, , !logi_n))
expect_equal(or3s(dfn != 9L, , !logi_d),
             bor3(dfn != 9L, , !logi_d))
expect_equal(or3s(dfo != 0L, logi_i, ),
             bor3(dfo != 0L, logi_i, ))
expect_equal(or3s(dfp != 1L, logi_o, ),
             bor3(dfp != 1L, logi_o, ))
expect_equal(or3s(dfq != 9L, logi_b, ),
             bor3(dfq != 9L, logi_b, ))
expect_equal(or3s(dfr != 0L, logi_w, !logi_t),
             bor3(dfr != 0L, logi_w, !logi_t))
expect_equal(or3s(dfs != 1L, logi_l, !logi_w),
             bor3(dfs != 1L, logi_l, !logi_w))
expect_equal(or3s(dft != 9L, logi_r, !logi_y),
             bor3(dft != 9L, logi_r, !logi_y))
expect_equal(or3s(dfu != 0L, , ),
             bor3(dfu != 0L, , ))
expect_equal(or3s(dfv != 1L, , ),
             bor3(dfv != 1L, , ))
expect_equal(or3s(dfw != 9L, , ),
             bor3(dfw != 9L, , ))
expect_equal(or3s(dfx != 0L, , i != 0L),
             bor3(dfx != 0L, , i != 0L))
expect_equal(or3s(dfy != 1L, , g != 1L),
             bor3(dfy != 1L, , g != 1L))
expect_equal(or3s(dfz != 9L, , q != 9L),
             bor3(dfz != 9L, , q != 9L))
expect_equal(or3s(dga != 0L, logi_e, ),
             bor3(dga != 0L, logi_e, ))
expect_equal(or3s(dgb != 1L, logi_c, ),
             bor3(dgb != 1L, logi_c, ))
expect_equal(or3s(dgc != 9L, logi_w, ),
             bor3(dgc != 9L, logi_w, ))
expect_equal(or3s(dgd != 0L, logi_f, m != 0L),
             bor3(dgd != 0L, logi_f, m != 0L))
expect_equal(or3s(dge != 1L, logi_p, h != 1L),
             bor3(dge != 1L, logi_p, h != 1L))
expect_equal(or3s(dgf != 9L, logi_l, c != 9L),
             bor3(dgf != 9L, logi_l, c != 9L))
expect_equal(or3s(dgg != 0L, , ),
             bor3(dgg != 0L, , ))
expect_equal(or3s(dgh != 1L, , ),
             bor3(dgh != 1L, , ))
expect_equal(or3s(dgi != 9L, , ),
             bor3(dgi != 9L, , ))
expect_equal(or3s(dgj != 0L, , b %between% c(-1L, 1L)),
             bor3(dgj != 0L, , b %between% c(-1L, 1L)))
expect_equal(or3s(dgk != 1L, , y %between% c(-1L, 1L)),
             bor3(dgk != 1L, , y %between% c(-1L, 1L)))
expect_equal(or3s(dgl != 9L, , l %between% c(-1L, 1L)),
             bor3(dgl != 9L, , l %between% c(-1L, 1L)))
expect_equal(or3s(dgm != 0L, logi_n, ),
             bor3(dgm != 0L, logi_n, ))
expect_equal(or3s(dgn != 1L, logi_g, ),
             bor3(dgn != 1L, logi_g, ))
expect_equal(or3s(dgo != 9L, logi_p, ),
             bor3(dgo != 9L, logi_p, ))
expect_equal(or3s(dgp != 0L, logi_w, p %between% c(-1L, 1L)),
             bor3(dgp != 0L, logi_w, p %between% c(-1L, 1L)))
expect_equal(or3s(dgq != 1L, logi_v, r %between% c(-1L, 1L)),
             bor3(dgq != 1L, logi_v, r %between% c(-1L, 1L)))
expect_equal(or3s(dgr != 9L, logi_z, g %between% c(-1L, 1L)),
             bor3(dgr != 9L, logi_z, g %between% c(-1L, 1L)))
expect_equal(or3s(dgs != 0L, , ),
             bor3(dgs != 0L, , ))
expect_equal(or3s(dgt != 1L, , ),
             bor3(dgt != 1L, , ))
expect_equal(or3s(dgu != 9L, , ),
             bor3(dgu != 9L, , ))
expect_equal(or3s(dgv != 0L, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dgv != 0L, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dgw != 1L, , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dgw != 1L, , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dgx != 9L, , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dgx != 9L, , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dgy != 0L, logi_y, ),
             bor3(dgy != 0L, logi_y, ))
expect_equal(or3s(dgz != 1L, logi_w, ),
             bor3(dgz != 1L, logi_w, ))
expect_equal(or3s(dha != 9L, logi_s, ),
             bor3(dha != 9L, logi_s, ))
expect_equal(or3s(dhb != 0L, logi_i, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dhb != 0L, logi_i, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dhc != 1L, logi_y, i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dhc != 1L, logi_y, i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dhd != 9L, logi_o, s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dhd != 9L, logi_o, s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dhe != 0L, , ),
             bor3(dhe != 0L, , ))
expect_equal(or3s(dhf != 1L, , ),
             bor3(dhf != 1L, , ))
expect_equal(or3s(dhg != 9L, , ),
             bor3(dhg != 9L, , ))
expect_equal(or3s(dhh != 0L, , i %in% 1:4),
             bor3(dhh != 0L, , i %in% 1:4))
expect_equal(or3s(dhi != 1L, , t %in% 1:4),
             bor3(dhi != 1L, , t %in% 1:4))
expect_equal(or3s(dhj != 9L, , g %in% 1:4),
             bor3(dhj != 9L, , g %in% 1:4))
expect_equal(or3s(dhk != 0L, logi_q, ),
             bor3(dhk != 0L, logi_q, ))
expect_equal(or3s(dhl != 1L, logi_q, ),
             bor3(dhl != 1L, logi_q, ))
expect_equal(or3s(dhm != 9L, logi_o, ),
             bor3(dhm != 9L, logi_o, ))
expect_equal(or3s(dhn != 0L, logi_d, g %in% 1:4),
             bor3(dhn != 0L, logi_d, g %in% 1:4))
expect_equal(or3s(dho != 1L, logi_a, a %in% 1:4),
             bor3(dho != 1L, logi_a, a %in% 1:4))
expect_equal(or3s(dhp != 9L, logi_j, d %in% 1:4),
             bor3(dhp != 9L, logi_j, d %in% 1:4))
expect_equal(or3s(dhq != 0L, , ),
             bor3(dhq != 0L, , ))
expect_equal(or3s(dhr != 1L, , ),
             bor3(dhr != 1L, , ))
expect_equal(or3s(dhs != 9L, , ),
             bor3(dhs != 9L, , ))
expect_equal(or3s(dht != 0L, , s < 0L),
             bor3(dht != 0L, , s < 0L))
expect_equal(or3s(dhu != 1L, , z < 1L),
             bor3(dhu != 1L, , z < 1L))
expect_equal(or3s(dhv != 9L, , t < 9L),
             bor3(dhv != 9L, , t < 9L))
expect_equal(or3s(dhw != 0L, logi_p, ),
             bor3(dhw != 0L, logi_p, ))
expect_equal(or3s(dhx != 1L, logi_d, ),
             bor3(dhx != 1L, logi_d, ))
expect_equal(or3s(dhy != 9L, logi_e, ),
             bor3(dhy != 9L, logi_e, ))
expect_equal(or3s(dhz != 0L, logi_f, j < 0L),
             bor3(dhz != 0L, logi_f, j < 0L))
expect_equal(or3s(dia != 1L, logi_a, n < 1L),
             bor3(dia != 1L, logi_a, n < 1L))
expect_equal(or3s(dib != 9L, logi_h, e < 9L),
             bor3(dib != 9L, logi_h, e < 9L))
expect_equal(or3s(dic != 0L, , ),
             bor3(dic != 0L, , ))
expect_equal(or3s(did != 1L, , ),
             bor3(did != 1L, , ))
expect_equal(or3s(die != 9L, , ),
             bor3(die != 9L, , ))
expect_equal(or3s(dif != 0L, , f <= 0L),
             bor3(dif != 0L, , f <= 0L))
expect_equal(or3s(dig != 1L, , z <= 1L),
             bor3(dig != 1L, , z <= 1L))
expect_equal(or3s(dih != 9L, , x <= 9L),
             bor3(dih != 9L, , x <= 9L))
expect_equal(or3s(dii != 0L, logi_r, ),
             bor3(dii != 0L, logi_r, ))
expect_equal(or3s(dij != 1L, logi_k, ),
             bor3(dij != 1L, logi_k, ))
expect_equal(or3s(dik != 9L, logi_b, ),
             bor3(dik != 9L, logi_b, ))
expect_equal(or3s(dil != 0L, logi_a, g <= 0L),
             bor3(dil != 0L, logi_a, g <= 0L))
expect_equal(or3s(dim != 1L, logi_l, w <= 1L),
             bor3(dim != 1L, logi_l, w <= 1L))
expect_equal(or3s(din != 9L, logi_z, x <= 9L),
             bor3(din != 9L, logi_z, x <= 9L))
expect_equal(or3s(dio != 0L, , ),
             bor3(dio != 0L, , ))
expect_equal(or3s(dip != 1L, , ),
             bor3(dip != 1L, , ))
expect_equal(or3s(diq != 9L, , ),
             bor3(diq != 9L, , ))
expect_equal(or3s(dir != 0L, , j == 0L),
             bor3(dir != 0L, , j == 0L))
expect_equal(or3s(dis != 1L, , k == 1L),
             bor3(dis != 1L, , k == 1L))
expect_equal(or3s(dit != 9L, , i == 9L),
             bor3(dit != 9L, , i == 9L))
expect_equal(or3s(diu != 0L, logi_l, ),
             bor3(diu != 0L, logi_l, ))
expect_equal(or3s(div != 1L, logi_u, ),
             bor3(div != 1L, logi_u, ))
expect_equal(or3s(diw != 9L, logi_q, ),
             bor3(diw != 9L, logi_q, ))
expect_equal(or3s(dix != 0L, logi_m, o == 0L),
             bor3(dix != 0L, logi_m, o == 0L))
expect_equal(or3s(diy != 1L, logi_u, t == 1L),
             bor3(diy != 1L, logi_u, t == 1L))
expect_equal(or3s(diz != 9L, logi_v, z == 9L),
             bor3(diz != 9L, logi_v, z == 9L))
expect_equal(or3s(dja != 0L, , ),
             bor3(dja != 0L, , ))
expect_equal(or3s(djb != 1L, , ),
             bor3(djb != 1L, , ))
expect_equal(or3s(djc != 9L, , ),
             bor3(djc != 9L, , ))
expect_equal(or3s(djd != 0L, , g > 0L),
             bor3(djd != 0L, , g > 0L))
expect_equal(or3s(dje != 1L, , y > 1L),
             bor3(dje != 1L, , y > 1L))
expect_equal(or3s(djf != 9L, , z > 9L),
             bor3(djf != 9L, , z > 9L))
expect_equal(or3s(djg != 0L, logi_c, ),
             bor3(djg != 0L, logi_c, ))
expect_equal(or3s(djh != 1L, logi_w, ),
             bor3(djh != 1L, logi_w, ))
expect_equal(or3s(dji != 9L, logi_e, ),
             bor3(dji != 9L, logi_e, ))
expect_equal(or3s(djj != 0L, logi_u, o > 0L),
             bor3(djj != 0L, logi_u, o > 0L))
expect_equal(or3s(djk != 1L, logi_z, q > 1L),
             bor3(djk != 1L, logi_z, q > 1L))
expect_equal(or3s(djl != 9L, logi_v, t > 9L),
             bor3(djl != 9L, logi_v, t > 9L))
expect_equal(or3s(djm != 0L, , ),
             bor3(djm != 0L, , ))
expect_equal(or3s(djn != 1L, , ),
             bor3(djn != 1L, , ))
expect_equal(or3s(djo != 9L, , ),
             bor3(djo != 9L, , ))
expect_equal(or3s(djp != 0L, , d >= 0L),
             bor3(djp != 0L, , d >= 0L))
expect_equal(or3s(djq != 1L, , g >= 1L),
             bor3(djq != 1L, , g >= 1L))
expect_equal(or3s(djr != 9L, , x >= 9L),
             bor3(djr != 9L, , x >= 9L))
expect_equal(or3s(djs != 0L, logi_i, ),
             bor3(djs != 0L, logi_i, ))
expect_equal(or3s(djt != 1L, logi_u, ),
             bor3(djt != 1L, logi_u, ))
expect_equal(or3s(dju != 9L, logi_h, ),
             bor3(dju != 9L, logi_h, ))
expect_equal(or3s(djv != 0L, logi_h, d >= 0L),
             bor3(djv != 0L, logi_h, d >= 0L))
expect_equal(or3s(djw != 1L, logi_f, d >= 1L),
             bor3(djw != 1L, logi_f, d >= 1L))
expect_equal(or3s(djx != 9L, logi_t, s >= 9L),
             bor3(djx != 9L, logi_t, s >= 9L))
expect_equal(or3s(djy != 0L, , ),
             bor3(djy != 0L, , ))
expect_equal(or3s(djz != 1L, , ),
             bor3(djz != 1L, , ))
expect_equal(or3s(dka != 9L, , ),
             bor3(dka != 9L, , ))
expect_equal(or3s(dkb != 0L, , logi_z),
             bor3(dkb != 0L, , logi_z))
expect_equal(or3s(dkc != 1L, , logi_e),
             bor3(dkc != 1L, , logi_e))
expect_equal(or3s(dkd != 9L, , logi_p),
             bor3(dkd != 9L, , logi_p))
expect_equal(or3s(dke != 0L, !logi_k, ),
             bor3(dke != 0L, !logi_k, ))
expect_equal(or3s(dkf != 1L, !logi_o, ),
             bor3(dkf != 1L, !logi_o, ))
expect_equal(or3s(dkg != 9L, !logi_b, ),
             bor3(dkg != 9L, !logi_b, ))
expect_equal(or3s(dkh != 0L, !logi_z, logi_s),
             bor3(dkh != 0L, !logi_z, logi_s))
expect_equal(or3s(dki != 1L, !logi_q, logi_i),
             bor3(dki != 1L, !logi_q, logi_i))
expect_equal(or3s(dkj != 9L, !logi_e, logi_b),
             bor3(dkj != 9L, !logi_e, logi_b))
expect_equal(or3s(dkk != 0L, , ),
             bor3(dkk != 0L, , ))
expect_equal(or3s(dkl != 1L, , ),
             bor3(dkl != 1L, , ))
expect_equal(or3s(dkm != 9L, , ),
             bor3(dkm != 9L, , ))
expect_equal(or3s(dkn != 0L, , !logi_c),
             bor3(dkn != 0L, , !logi_c))
expect_equal(or3s(dko != 1L, , !logi_j),
             bor3(dko != 1L, , !logi_j))
expect_equal(or3s(dkp != 9L, , !logi_j),
             bor3(dkp != 9L, , !logi_j))
expect_equal(or3s(dkq != 0L, !logi_i, ),
             bor3(dkq != 0L, !logi_i, ))
expect_equal(or3s(dkr != 1L, !logi_p, ),
             bor3(dkr != 1L, !logi_p, ))
expect_equal(or3s(dks != 9L, !logi_n, ),
             bor3(dks != 9L, !logi_n, ))
expect_equal(or3s(dkt != 0L, !logi_g, !logi_u),
             bor3(dkt != 0L, !logi_g, !logi_u))
expect_equal(or3s(dku != 1L, !logi_j, !logi_m),
             bor3(dku != 1L, !logi_j, !logi_m))
expect_equal(or3s(dkv != 9L, !logi_d, !logi_n),
             bor3(dkv != 9L, !logi_d, !logi_n))
expect_equal(or3s(dkw != 0L, , ),
             bor3(dkw != 0L, , ))
expect_equal(or3s(dkx != 1L, , ),
             bor3(dkx != 1L, , ))
expect_equal(or3s(dky != 9L, , ),
             bor3(dky != 9L, , ))
expect_equal(or3s(dkz != 0L, , z != 0L),
             bor3(dkz != 0L, , z != 0L))
expect_equal(or3s(dla != 1L, , k != 1L),
             bor3(dla != 1L, , k != 1L))
expect_equal(or3s(dlb != 9L, , t != 9L),
             bor3(dlb != 9L, , t != 9L))
expect_equal(or3s(dlc != 0L, !logi_e, ),
             bor3(dlc != 0L, !logi_e, ))
expect_equal(or3s(dld != 1L, !logi_n, ),
             bor3(dld != 1L, !logi_n, ))
expect_equal(or3s(dle != 9L, !logi_u, ),
             bor3(dle != 9L, !logi_u, ))
expect_equal(or3s(dlf != 0L, !logi_t, o != 0L),
             bor3(dlf != 0L, !logi_t, o != 0L))
expect_equal(or3s(dlg != 1L, !logi_s, i != 1L),
             bor3(dlg != 1L, !logi_s, i != 1L))
expect_equal(or3s(dlh != 9L, !logi_w, a != 9L),
             bor3(dlh != 9L, !logi_w, a != 9L))
expect_equal(or3s(dli != 0L, , ),
             bor3(dli != 0L, , ))
expect_equal(or3s(dlj != 1L, , ),
             bor3(dlj != 1L, , ))
expect_equal(or3s(dlk != 9L, , ),
             bor3(dlk != 9L, , ))
expect_equal(or3s(dll != 0L, , s %between% c(-1L, 1L)),
             bor3(dll != 0L, , s %between% c(-1L, 1L)))
expect_equal(or3s(dlm != 1L, , x %between% c(-1L, 1L)),
             bor3(dlm != 1L, , x %between% c(-1L, 1L)))
expect_equal(or3s(dln != 9L, , x %between% c(-1L, 1L)),
             bor3(dln != 9L, , x %between% c(-1L, 1L)))
expect_equal(or3s(dlo != 0L, !logi_p, ),
             bor3(dlo != 0L, !logi_p, ))
expect_equal(or3s(dlp != 1L, !logi_t, ),
             bor3(dlp != 1L, !logi_t, ))
expect_equal(or3s(dlq != 9L, !logi_u, ),
             bor3(dlq != 9L, !logi_u, ))
expect_equal(or3s(dlr != 0L, !logi_x, b %between% c(-1L, 1L)),
             bor3(dlr != 0L, !logi_x, b %between% c(-1L, 1L)))
expect_equal(or3s(dls != 1L, !logi_m, m %between% c(-1L, 1L)),
             bor3(dls != 1L, !logi_m, m %between% c(-1L, 1L)))
expect_equal(or3s(dlt != 9L, !logi_m, r %between% c(-1L, 1L)),
             bor3(dlt != 9L, !logi_m, r %between% c(-1L, 1L)))
expect_equal(or3s(dlu != 0L, , ),
             bor3(dlu != 0L, , ))
expect_equal(or3s(dlv != 1L, , ),
             bor3(dlv != 1L, , ))
expect_equal(or3s(dlw != 9L, , ),
             bor3(dlw != 9L, , ))
expect_equal(or3s(dlx != 0L, , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dlx != 0L, , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dly != 1L, , y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dly != 1L, , y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dlz != 9L, , r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dlz != 9L, , r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dma != 0L, !logi_v, ),
             bor3(dma != 0L, !logi_v, ))
expect_equal(or3s(dmb != 1L, !logi_k, ),
             bor3(dmb != 1L, !logi_k, ))
expect_equal(or3s(dmc != 9L, !logi_h, ),
             bor3(dmc != 9L, !logi_h, ))
expect_equal(or3s(dmd != 0L, !logi_w, o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dmd != 0L, !logi_w, o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dme != 1L, !logi_g, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dme != 1L, !logi_g, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dmf != 9L, !logi_q, y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dmf != 9L, !logi_q, y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dmg != 0L, , ),
             bor3(dmg != 0L, , ))
expect_equal(or3s(dmh != 1L, , ),
             bor3(dmh != 1L, , ))
expect_equal(or3s(dmi != 9L, , ),
             bor3(dmi != 9L, , ))
expect_equal(or3s(dmj != 0L, , a %in% 1:4),
             bor3(dmj != 0L, , a %in% 1:4))
expect_equal(or3s(dmk != 1L, , t %in% 1:4),
             bor3(dmk != 1L, , t %in% 1:4))
expect_equal(or3s(dml != 9L, , p %in% 1:4),
             bor3(dml != 9L, , p %in% 1:4))
expect_equal(or3s(dmm != 0L, !logi_a, ),
             bor3(dmm != 0L, !logi_a, ))
expect_equal(or3s(dmn != 1L, !logi_v, ),
             bor3(dmn != 1L, !logi_v, ))
expect_equal(or3s(dmo != 9L, !logi_m, ),
             bor3(dmo != 9L, !logi_m, ))
expect_equal(or3s(dmp != 0L, !logi_y, o %in% 1:4),
             bor3(dmp != 0L, !logi_y, o %in% 1:4))
expect_equal(or3s(dmq != 1L, !logi_x, d %in% 1:4),
             bor3(dmq != 1L, !logi_x, d %in% 1:4))
expect_equal(or3s(dmr != 9L, !logi_s, o %in% 1:4),
             bor3(dmr != 9L, !logi_s, o %in% 1:4))
expect_equal(or3s(dms != 0L, , ),
             bor3(dms != 0L, , ))
expect_equal(or3s(dmt != 1L, , ),
             bor3(dmt != 1L, , ))
expect_equal(or3s(dmu != 9L, , ),
             bor3(dmu != 9L, , ))
expect_equal(or3s(dmv != 0L, , q < 0L),
             bor3(dmv != 0L, , q < 0L))
expect_equal(or3s(dmw != 1L, , f < 1L),
             bor3(dmw != 1L, , f < 1L))
expect_equal(or3s(dmx != 9L, , i < 9L),
             bor3(dmx != 9L, , i < 9L))
expect_equal(or3s(dmy != 0L, !logi_g, ),
             bor3(dmy != 0L, !logi_g, ))
expect_equal(or3s(dmz != 1L, !logi_k, ),
             bor3(dmz != 1L, !logi_k, ))
expect_equal(or3s(dna != 9L, !logi_k, ),
             bor3(dna != 9L, !logi_k, ))
expect_equal(or3s(dnb != 0L, !logi_p, a < 0L),
             bor3(dnb != 0L, !logi_p, a < 0L))
expect_equal(or3s(dnc != 1L, !logi_z, g < 1L),
             bor3(dnc != 1L, !logi_z, g < 1L))
expect_equal(or3s(dnd != 9L, !logi_d, s < 9L),
             bor3(dnd != 9L, !logi_d, s < 9L))
expect_equal(or3s(dne != 0L, , ),
             bor3(dne != 0L, , ))
expect_equal(or3s(dnf != 1L, , ),
             bor3(dnf != 1L, , ))
expect_equal(or3s(dng != 9L, , ),
             bor3(dng != 9L, , ))
expect_equal(or3s(dnh != 0L, , f <= 0L),
             bor3(dnh != 0L, , f <= 0L))
expect_equal(or3s(dni != 1L, , w <= 1L),
             bor3(dni != 1L, , w <= 1L))
expect_equal(or3s(dnj != 9L, , w <= 9L),
             bor3(dnj != 9L, , w <= 9L))
expect_equal(or3s(dnk != 0L, !logi_c, ),
             bor3(dnk != 0L, !logi_c, ))
expect_equal(or3s(dnl != 1L, !logi_h, ),
             bor3(dnl != 1L, !logi_h, ))
expect_equal(or3s(dnm != 9L, !logi_k, ),
             bor3(dnm != 9L, !logi_k, ))
expect_equal(or3s(dnn != 0L, !logi_e, x <= 0L),
             bor3(dnn != 0L, !logi_e, x <= 0L))
expect_equal(or3s(dno != 1L, !logi_o, x <= 1L),
             bor3(dno != 1L, !logi_o, x <= 1L))
expect_equal(or3s(dnp != 9L, !logi_g, s <= 9L),
             bor3(dnp != 9L, !logi_g, s <= 9L))
expect_equal(or3s(dnq != 0L, , ),
             bor3(dnq != 0L, , ))
expect_equal(or3s(dnr != 1L, , ),
             bor3(dnr != 1L, , ))
expect_equal(or3s(dns != 9L, , ),
             bor3(dns != 9L, , ))
expect_equal(or3s(dnt != 0L, , b == 0L),
             bor3(dnt != 0L, , b == 0L))
expect_equal(or3s(dnu != 1L, , b == 1L),
             bor3(dnu != 1L, , b == 1L))
expect_equal(or3s(dnv != 9L, , n == 9L),
             bor3(dnv != 9L, , n == 9L))
expect_equal(or3s(dnw != 0L, !logi_v, ),
             bor3(dnw != 0L, !logi_v, ))
expect_equal(or3s(dnx != 1L, !logi_t, ),
             bor3(dnx != 1L, !logi_t, ))
expect_equal(or3s(dny != 9L, !logi_b, ),
             bor3(dny != 9L, !logi_b, ))
expect_equal(or3s(dnz != 0L, !logi_x, c == 0L),
             bor3(dnz != 0L, !logi_x, c == 0L))
expect_equal(or3s(doa != 1L, !logi_d, q == 1L),
             bor3(doa != 1L, !logi_d, q == 1L))
expect_equal(or3s(dob != 9L, !logi_k, l == 9L),
             bor3(dob != 9L, !logi_k, l == 9L))
expect_equal(or3s(doc != 0L, , ),
             bor3(doc != 0L, , ))
expect_equal(or3s(dod != 1L, , ),
             bor3(dod != 1L, , ))
expect_equal(or3s(doe != 9L, , ),
             bor3(doe != 9L, , ))
expect_equal(or3s(dof != 0L, , w > 0L),
             bor3(dof != 0L, , w > 0L))
expect_equal(or3s(dog != 1L, , a > 1L),
             bor3(dog != 1L, , a > 1L))
expect_equal(or3s(doh != 9L, , p > 9L),
             bor3(doh != 9L, , p > 9L))
expect_equal(or3s(doi != 0L, !logi_v, ),
             bor3(doi != 0L, !logi_v, ))
expect_equal(or3s(doj != 1L, !logi_j, ),
             bor3(doj != 1L, !logi_j, ))
expect_equal(or3s(dok != 9L, !logi_b, ),
             bor3(dok != 9L, !logi_b, ))
expect_equal(or3s(dol != 0L, !logi_k, s > 0L),
             bor3(dol != 0L, !logi_k, s > 0L))
expect_equal(or3s(dom != 1L, !logi_p, h > 1L),
             bor3(dom != 1L, !logi_p, h > 1L))
expect_equal(or3s(don != 9L, !logi_n, q > 9L),
             bor3(don != 9L, !logi_n, q > 9L))
expect_equal(or3s(doo != 0L, , ),
             bor3(doo != 0L, , ))
expect_equal(or3s(dop != 1L, , ),
             bor3(dop != 1L, , ))
expect_equal(or3s(doq != 9L, , ),
             bor3(doq != 9L, , ))
expect_equal(or3s(dor != 0L, , g >= 0L),
             bor3(dor != 0L, , g >= 0L))
expect_equal(or3s(dos != 1L, , l >= 1L),
             bor3(dos != 1L, , l >= 1L))
expect_equal(or3s(dot != 9L, , f >= 9L),
             bor3(dot != 9L, , f >= 9L))
expect_equal(or3s(dou != 0L, !logi_t, ),
             bor3(dou != 0L, !logi_t, ))
expect_equal(or3s(dov != 1L, !logi_b, ),
             bor3(dov != 1L, !logi_b, ))
expect_equal(or3s(dow != 9L, !logi_g, ),
             bor3(dow != 9L, !logi_g, ))
expect_equal(or3s(dox != 0L, !logi_i, r >= 0L),
             bor3(dox != 0L, !logi_i, r >= 0L))
expect_equal(or3s(doy != 1L, !logi_o, h >= 1L),
             bor3(doy != 1L, !logi_o, h >= 1L))
expect_equal(or3s(doz != 9L, !logi_m, t >= 9L),
             bor3(doz != 9L, !logi_m, t >= 9L))
expect_equal(or3s(dpa != 0L, , ),
             bor3(dpa != 0L, , ))
expect_equal(or3s(dpb != 1L, , ),
             bor3(dpb != 1L, , ))
expect_equal(or3s(dpc != 9L, , ),
             bor3(dpc != 9L, , ))
expect_equal(or3s(dpd != 0L, , logi_p),
             bor3(dpd != 0L, , logi_p))
expect_equal(or3s(dpe != 1L, , logi_f),
             bor3(dpe != 1L, , logi_f))
expect_equal(or3s(dpf != 9L, , logi_k),
             bor3(dpf != 9L, , logi_k))
expect_equal(or3s(dpg != 0L, i != 0L, ),
             bor3(dpg != 0L, i != 0L, ))
expect_equal(or3s(dph != 1L, d != 1L, ),
             bor3(dph != 1L, d != 1L, ))
expect_equal(or3s(dpi != 9L, q != 9L, ),
             bor3(dpi != 9L, q != 9L, ))
expect_equal(or3s(dpj != 0L, u != 0L, logi_z),
             bor3(dpj != 0L, u != 0L, logi_z))
expect_equal(or3s(dpk != 1L, k != 1L, logi_a),
             bor3(dpk != 1L, k != 1L, logi_a))
expect_equal(or3s(dpl != 9L, t != 9L, logi_c),
             bor3(dpl != 9L, t != 9L, logi_c))
expect_equal(or3s(dpm != 0L, , ),
             bor3(dpm != 0L, , ))
expect_equal(or3s(dpn != 1L, , ),
             bor3(dpn != 1L, , ))
expect_equal(or3s(dpo != 9L, , ),
             bor3(dpo != 9L, , ))
expect_equal(or3s(dpp != 0L, , !logi_r),
             bor3(dpp != 0L, , !logi_r))
expect_equal(or3s(dpq != 1L, , !logi_d),
             bor3(dpq != 1L, , !logi_d))
expect_equal(or3s(dpr != 9L, , !logi_p),
             bor3(dpr != 9L, , !logi_p))
expect_equal(or3s(dps != 0L, n != 0L, ),
             bor3(dps != 0L, n != 0L, ))
expect_equal(or3s(dpt != 1L, f != 1L, ),
             bor3(dpt != 1L, f != 1L, ))
expect_equal(or3s(dpu != 9L, r != 9L, ),
             bor3(dpu != 9L, r != 9L, ))
expect_equal(or3s(dpv != 0L, y != 0L, !logi_v),
             bor3(dpv != 0L, y != 0L, !logi_v))
expect_equal(or3s(dpw != 1L, b != 1L, !logi_w),
             bor3(dpw != 1L, b != 1L, !logi_w))
expect_equal(or3s(dpx != 9L, u != 9L, !logi_r),
             bor3(dpx != 9L, u != 9L, !logi_r))
expect_equal(or3s(dpy != 0L, , ),
             bor3(dpy != 0L, , ))
expect_equal(or3s(dpz != 1L, , ),
             bor3(dpz != 1L, , ))
expect_equal(or3s(dqa != 9L, , ),
             bor3(dqa != 9L, , ))
expect_equal(or3s(dqb != 0L, , t != 0L),
             bor3(dqb != 0L, , t != 0L))
expect_equal(or3s(dqc != 1L, , w != 1L),
             bor3(dqc != 1L, , w != 1L))
expect_equal(or3s(dqd != 9L, , g != 9L),
             bor3(dqd != 9L, , g != 9L))
expect_equal(or3s(dqe != 0L, o != 0L, ),
             bor3(dqe != 0L, o != 0L, ))
expect_equal(or3s(dqf != 1L, s != 1L, ),
             bor3(dqf != 1L, s != 1L, ))
expect_equal(or3s(dqg != 9L, i != 9L, ),
             bor3(dqg != 9L, i != 9L, ))
expect_equal(or3s(dqh != 0L, k != 0L, u != 0L),
             bor3(dqh != 0L, k != 0L, u != 0L))
expect_equal(or3s(dqi != 1L, u != 1L, m != 1L),
             bor3(dqi != 1L, u != 1L, m != 1L))
expect_equal(or3s(dqj != 9L, w != 9L, q != 9L),
             bor3(dqj != 9L, w != 9L, q != 9L))
expect_equal(or3s(dqk != 0L, , ),
             bor3(dqk != 0L, , ))
expect_equal(or3s(dql != 1L, , ),
             bor3(dql != 1L, , ))
expect_equal(or3s(dqm != 9L, , ),
             bor3(dqm != 9L, , ))
expect_equal(or3s(dqn != 0L, , c %between% c(-1L, 1L)),
             bor3(dqn != 0L, , c %between% c(-1L, 1L)))
expect_equal(or3s(dqo != 1L, , b %between% c(-1L, 1L)),
             bor3(dqo != 1L, , b %between% c(-1L, 1L)))
expect_equal(or3s(dqp != 9L, , y %between% c(-1L, 1L)),
             bor3(dqp != 9L, , y %between% c(-1L, 1L)))
expect_equal(or3s(dqq != 0L, i != 0L, ),
             bor3(dqq != 0L, i != 0L, ))
expect_equal(or3s(dqr != 1L, w != 1L, ),
             bor3(dqr != 1L, w != 1L, ))
expect_equal(or3s(dqs != 9L, e != 9L, ),
             bor3(dqs != 9L, e != 9L, ))
expect_equal(or3s(dqt != 0L, b != 0L, r %between% c(-1L, 1L)),
             bor3(dqt != 0L, b != 0L, r %between% c(-1L, 1L)))
expect_equal(or3s(dqu != 1L, s != 1L, v %between% c(-1L, 1L)),
             bor3(dqu != 1L, s != 1L, v %between% c(-1L, 1L)))
expect_equal(or3s(dqv != 9L, o != 9L, h %between% c(-1L, 1L)),
             bor3(dqv != 9L, o != 9L, h %between% c(-1L, 1L)))
expect_equal(or3s(dqw != 0L, , ),
             bor3(dqw != 0L, , ))
expect_equal(or3s(dqx != 1L, , ),
             bor3(dqx != 1L, , ))
expect_equal(or3s(dqy != 9L, , ),
             bor3(dqy != 9L, , ))
expect_equal(or3s(dqz != 0L, , k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dqz != 0L, , k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dra != 1L, , x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dra != 1L, , x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(drb != 9L, , a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(drb != 9L, , a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(drc != 0L, e != 0L, ),
             bor3(drc != 0L, e != 0L, ))
expect_equal(or3s(drd != 1L, e != 1L, ),
             bor3(drd != 1L, e != 1L, ))
expect_equal(or3s(dre != 9L, y != 9L, ),
             bor3(dre != 9L, y != 9L, ))
expect_equal(or3s(drf != 0L, l != 0L, q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(drf != 0L, l != 0L, q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(drg != 1L, a != 1L, k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(drg != 1L, a != 1L, k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(drh != 9L, w != 9L, f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(drh != 9L, w != 9L, f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dri != 0L, , ),
             bor3(dri != 0L, , ))
expect_equal(or3s(drj != 1L, , ),
             bor3(drj != 1L, , ))
expect_equal(or3s(drk != 9L, , ),
             bor3(drk != 9L, , ))
expect_equal(or3s(drl != 0L, , a %in% 1:4),
             bor3(drl != 0L, , a %in% 1:4))
expect_equal(or3s(drm != 1L, , n %in% 1:4),
             bor3(drm != 1L, , n %in% 1:4))
expect_equal(or3s(drn != 9L, , i %in% 1:4),
             bor3(drn != 9L, , i %in% 1:4))
expect_equal(or3s(dro != 0L, k != 0L, ),
             bor3(dro != 0L, k != 0L, ))
expect_equal(or3s(drp != 1L, z != 1L, ),
             bor3(drp != 1L, z != 1L, ))
expect_equal(or3s(drq != 9L, o != 9L, ),
             bor3(drq != 9L, o != 9L, ))
expect_equal(or3s(drr != 0L, f != 0L, a %in% 1:4),
             bor3(drr != 0L, f != 0L, a %in% 1:4))
expect_equal(or3s(drs != 1L, y != 1L, h %in% 1:4),
             bor3(drs != 1L, y != 1L, h %in% 1:4))
expect_equal(or3s(drt != 9L, n != 9L, o %in% 1:4),
             bor3(drt != 9L, n != 9L, o %in% 1:4))
expect_equal(or3s(dru != 0L, , ),
             bor3(dru != 0L, , ))
expect_equal(or3s(drv != 1L, , ),
             bor3(drv != 1L, , ))
expect_equal(or3s(drw != 9L, , ),
             bor3(drw != 9L, , ))
expect_equal(or3s(drx != 0L, , b < 0L),
             bor3(drx != 0L, , b < 0L))
expect_equal(or3s(dry != 1L, , t < 1L),
             bor3(dry != 1L, , t < 1L))
expect_equal(or3s(drz != 9L, , i < 9L),
             bor3(drz != 9L, , i < 9L))
expect_equal(or3s(dsa != 0L, j != 0L, ),
             bor3(dsa != 0L, j != 0L, ))
expect_equal(or3s(dsb != 1L, k != 1L, ),
             bor3(dsb != 1L, k != 1L, ))
expect_equal(or3s(dsc != 9L, t != 9L, ),
             bor3(dsc != 9L, t != 9L, ))
expect_equal(or3s(dsd != 0L, y != 0L, q < 0L),
             bor3(dsd != 0L, y != 0L, q < 0L))
expect_equal(or3s(dse != 1L, u != 1L, z < 1L),
             bor3(dse != 1L, u != 1L, z < 1L))
expect_equal(or3s(dsf != 9L, i != 9L, n < 9L),
             bor3(dsf != 9L, i != 9L, n < 9L))
expect_equal(or3s(dsg != 0L, , ),
             bor3(dsg != 0L, , ))
expect_equal(or3s(dsh != 1L, , ),
             bor3(dsh != 1L, , ))
expect_equal(or3s(dsi != 9L, , ),
             bor3(dsi != 9L, , ))
expect_equal(or3s(dsj != 0L, , g <= 0L),
             bor3(dsj != 0L, , g <= 0L))
expect_equal(or3s(dsk != 1L, , h <= 1L),
             bor3(dsk != 1L, , h <= 1L))
expect_equal(or3s(dsl != 9L, , u <= 9L),
             bor3(dsl != 9L, , u <= 9L))
expect_equal(or3s(dsm != 0L, h != 0L, ),
             bor3(dsm != 0L, h != 0L, ))
expect_equal(or3s(dsn != 1L, d != 1L, ),
             bor3(dsn != 1L, d != 1L, ))
expect_equal(or3s(dso != 9L, t != 9L, ),
             bor3(dso != 9L, t != 9L, ))
expect_equal(or3s(dsp != 0L, s != 0L, h <= 0L),
             bor3(dsp != 0L, s != 0L, h <= 0L))
expect_equal(or3s(dsq != 1L, d != 1L, i <= 1L),
             bor3(dsq != 1L, d != 1L, i <= 1L))
expect_equal(or3s(dsr != 9L, s != 9L, q <= 9L),
             bor3(dsr != 9L, s != 9L, q <= 9L))
expect_equal(or3s(dss != 0L, , ),
             bor3(dss != 0L, , ))
expect_equal(or3s(dst != 1L, , ),
             bor3(dst != 1L, , ))
expect_equal(or3s(dsu != 9L, , ),
             bor3(dsu != 9L, , ))
expect_equal(or3s(dsv != 0L, , x == 0L),
             bor3(dsv != 0L, , x == 0L))
expect_equal(or3s(dsw != 1L, , i == 1L),
             bor3(dsw != 1L, , i == 1L))
expect_equal(or3s(dsx != 9L, , n == 9L),
             bor3(dsx != 9L, , n == 9L))
expect_equal(or3s(dsy != 0L, z != 0L, ),
             bor3(dsy != 0L, z != 0L, ))
expect_equal(or3s(dsz != 1L, q != 1L, ),
             bor3(dsz != 1L, q != 1L, ))
expect_equal(or3s(dta != 9L, h != 9L, ),
             bor3(dta != 9L, h != 9L, ))
expect_equal(or3s(dtb != 0L, s != 0L, g == 0L),
             bor3(dtb != 0L, s != 0L, g == 0L))
expect_equal(or3s(dtc != 1L, f != 1L, s == 1L),
             bor3(dtc != 1L, f != 1L, s == 1L))
expect_equal(or3s(dtd != 9L, m != 9L, v == 9L),
             bor3(dtd != 9L, m != 9L, v == 9L))
expect_equal(or3s(dte != 0L, , ),
             bor3(dte != 0L, , ))
expect_equal(or3s(dtf != 1L, , ),
             bor3(dtf != 1L, , ))
expect_equal(or3s(dtg != 9L, , ),
             bor3(dtg != 9L, , ))
expect_equal(or3s(dth != 0L, , s > 0L),
             bor3(dth != 0L, , s > 0L))
expect_equal(or3s(dti != 1L, , u > 1L),
             bor3(dti != 1L, , u > 1L))
expect_equal(or3s(dtj != 9L, , k > 9L),
             bor3(dtj != 9L, , k > 9L))
expect_equal(or3s(dtk != 0L, d != 0L, ),
             bor3(dtk != 0L, d != 0L, ))
expect_equal(or3s(dtl != 1L, t != 1L, ),
             bor3(dtl != 1L, t != 1L, ))
expect_equal(or3s(dtm != 9L, j != 9L, ),
             bor3(dtm != 9L, j != 9L, ))
expect_equal(or3s(dtn != 0L, k != 0L, c > 0L),
             bor3(dtn != 0L, k != 0L, c > 0L))
expect_equal(or3s(dto != 1L, j != 1L, l > 1L),
             bor3(dto != 1L, j != 1L, l > 1L))
expect_equal(or3s(dtp != 9L, y != 9L, w > 9L),
             bor3(dtp != 9L, y != 9L, w > 9L))
expect_equal(or3s(dtq != 0L, , ),
             bor3(dtq != 0L, , ))
expect_equal(or3s(dtr != 1L, , ),
             bor3(dtr != 1L, , ))
expect_equal(or3s(dts != 9L, , ),
             bor3(dts != 9L, , ))
expect_equal(or3s(dtt != 0L, , v >= 0L),
             bor3(dtt != 0L, , v >= 0L))
expect_equal(or3s(dtu != 1L, , t >= 1L),
             bor3(dtu != 1L, , t >= 1L))
expect_equal(or3s(dtv != 9L, , i >= 9L),
             bor3(dtv != 9L, , i >= 9L))
expect_equal(or3s(dtw != 0L, n != 0L, ),
             bor3(dtw != 0L, n != 0L, ))
expect_equal(or3s(dtx != 1L, n != 1L, ),
             bor3(dtx != 1L, n != 1L, ))
expect_equal(or3s(dty != 9L, q != 9L, ),
             bor3(dty != 9L, q != 9L, ))
expect_equal(or3s(dtz != 0L, r != 0L, a >= 0L),
             bor3(dtz != 0L, r != 0L, a >= 0L))
expect_equal(or3s(dua != 1L, q != 1L, y >= 1L),
             bor3(dua != 1L, q != 1L, y >= 1L))
expect_equal(or3s(dub != 9L, p != 9L, n >= 9L),
             bor3(dub != 9L, p != 9L, n >= 9L))
expect_equal(or3s(duc != 0L, , ),
             bor3(duc != 0L, , ))
expect_equal(or3s(dud != 1L, , ),
             bor3(dud != 1L, , ))
expect_equal(or3s(due != 9L, , ),
             bor3(due != 9L, , ))
expect_equal(or3s(duf != 0L, , logi_s),
             bor3(duf != 0L, , logi_s))
expect_equal(or3s(dug != 1L, , logi_q),
             bor3(dug != 1L, , logi_q))
expect_equal(or3s(duh != 9L, , logi_c),
             bor3(duh != 9L, , logi_c))
expect_equal(or3s(dui != 0L, t %between% c(-1L, 1L), ),
             bor3(dui != 0L, t %between% c(-1L, 1L), ))
expect_equal(or3s(duj != 1L, e %between% c(-1L, 1L), ),
             bor3(duj != 1L, e %between% c(-1L, 1L), ))
expect_equal(or3s(duk != 9L, i %between% c(-1L, 1L), ),
             bor3(duk != 9L, i %between% c(-1L, 1L), ))
expect_equal(or3s(dul != 0L, u %between% c(-1L, 1L), logi_g),
             bor3(dul != 0L, u %between% c(-1L, 1L), logi_g))
expect_equal(or3s(dum != 1L, l %between% c(-1L, 1L), logi_v),
             bor3(dum != 1L, l %between% c(-1L, 1L), logi_v))
expect_equal(or3s(dun != 9L, t %between% c(-1L, 1L), logi_x),
             bor3(dun != 9L, t %between% c(-1L, 1L), logi_x))
expect_equal(or3s(duo != 0L, , ),
             bor3(duo != 0L, , ))
expect_equal(or3s(dup != 1L, , ),
             bor3(dup != 1L, , ))
expect_equal(or3s(duq != 9L, , ),
             bor3(duq != 9L, , ))
expect_equal(or3s(dur != 0L, , !logi_o),
             bor3(dur != 0L, , !logi_o))
expect_equal(or3s(dus != 1L, , !logi_r),
             bor3(dus != 1L, , !logi_r))
expect_equal(or3s(dut != 9L, , !logi_e),
             bor3(dut != 9L, , !logi_e))
expect_equal(or3s(duu != 0L, d %between% c(-1L, 1L), ),
             bor3(duu != 0L, d %between% c(-1L, 1L), ))
expect_equal(or3s(duv != 1L, e %between% c(-1L, 1L), ),
             bor3(duv != 1L, e %between% c(-1L, 1L), ))
expect_equal(or3s(duw != 9L, e %between% c(-1L, 1L), ),
             bor3(duw != 9L, e %between% c(-1L, 1L), ))
expect_equal(or3s(dux != 0L, s %between% c(-1L, 1L), !logi_c),
             bor3(dux != 0L, s %between% c(-1L, 1L), !logi_c))
expect_equal(or3s(duy != 1L, z %between% c(-1L, 1L), !logi_m),
             bor3(duy != 1L, z %between% c(-1L, 1L), !logi_m))
expect_equal(or3s(duz != 9L, z %between% c(-1L, 1L), !logi_k),
             bor3(duz != 9L, z %between% c(-1L, 1L), !logi_k))
expect_equal(or3s(dva != 0L, , ),
             bor3(dva != 0L, , ))
expect_equal(or3s(dvb != 1L, , ),
             bor3(dvb != 1L, , ))
expect_equal(or3s(dvc != 9L, , ),
             bor3(dvc != 9L, , ))
expect_equal(or3s(dvd != 0L, , b != 0L),
             bor3(dvd != 0L, , b != 0L))
expect_equal(or3s(dve != 1L, , e != 1L),
             bor3(dve != 1L, , e != 1L))
expect_equal(or3s(dvf != 9L, , v != 9L),
             bor3(dvf != 9L, , v != 9L))
expect_equal(or3s(dvg != 0L, w %between% c(-1L, 1L), ),
             bor3(dvg != 0L, w %between% c(-1L, 1L), ))
expect_equal(or3s(dvh != 1L, z %between% c(-1L, 1L), ),
             bor3(dvh != 1L, z %between% c(-1L, 1L), ))
expect_equal(or3s(dvi != 9L, i %between% c(-1L, 1L), ),
             bor3(dvi != 9L, i %between% c(-1L, 1L), ))
expect_equal(or3s(dvj != 0L, z %between% c(-1L, 1L), e != 0L),
             bor3(dvj != 0L, z %between% c(-1L, 1L), e != 0L))
expect_equal(or3s(dvk != 1L, f %between% c(-1L, 1L), b != 1L),
             bor3(dvk != 1L, f %between% c(-1L, 1L), b != 1L))
expect_equal(or3s(dvl != 9L, j %between% c(-1L, 1L), o != 9L),
             bor3(dvl != 9L, j %between% c(-1L, 1L), o != 9L))
expect_equal(or3s(dvm != 0L, , ),
             bor3(dvm != 0L, , ))
expect_equal(or3s(dvn != 1L, , ),
             bor3(dvn != 1L, , ))
expect_equal(or3s(dvo != 9L, , ),
             bor3(dvo != 9L, , ))
expect_equal(or3s(dvp != 0L, , c %between% c(-1L, 1L)),
             bor3(dvp != 0L, , c %between% c(-1L, 1L)))
expect_equal(or3s(dvq != 1L, , p %between% c(-1L, 1L)),
             bor3(dvq != 1L, , p %between% c(-1L, 1L)))
expect_equal(or3s(dvr != 9L, , x %between% c(-1L, 1L)),
             bor3(dvr != 9L, , x %between% c(-1L, 1L)))
expect_equal(or3s(dvs != 0L, t %between% c(-1L, 1L), ),
             bor3(dvs != 0L, t %between% c(-1L, 1L), ))
expect_equal(or3s(dvt != 1L, d %between% c(-1L, 1L), ),
             bor3(dvt != 1L, d %between% c(-1L, 1L), ))
expect_equal(or3s(dvu != 9L, w %between% c(-1L, 1L), ),
             bor3(dvu != 9L, w %between% c(-1L, 1L), ))
expect_equal(or3s(dvv != 0L, s %between% c(-1L, 1L), g %between% c(-1L, 1L)),
             bor3(dvv != 0L, s %between% c(-1L, 1L), g %between% c(-1L, 1L)))
expect_equal(or3s(dvw != 1L, k %between% c(-1L, 1L), q %between% c(-1L, 1L)),
             bor3(dvw != 1L, k %between% c(-1L, 1L), q %between% c(-1L, 1L)))
expect_equal(or3s(dvx != 9L, w %between% c(-1L, 1L), x %between% c(-1L, 1L)),
             bor3(dvx != 9L, w %between% c(-1L, 1L), x %between% c(-1L, 1L)))
expect_equal(or3s(dvy != 0L, , ),
             bor3(dvy != 0L, , ))
expect_equal(or3s(dvz != 1L, , ),
             bor3(dvz != 1L, , ))
expect_equal(or3s(dwa != 9L, , ),
             bor3(dwa != 9L, , ))
expect_equal(or3s(dwb != 0L, , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dwb != 0L, , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dwc != 1L, , e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dwc != 1L, , e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dwd != 9L, , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dwd != 9L, , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dwe != 0L, x %between% c(-1L, 1L), ),
             bor3(dwe != 0L, x %between% c(-1L, 1L), ))
expect_equal(or3s(dwf != 1L, d %between% c(-1L, 1L), ),
             bor3(dwf != 1L, d %between% c(-1L, 1L), ))
expect_equal(or3s(dwg != 9L, v %between% c(-1L, 1L), ),
             bor3(dwg != 9L, v %between% c(-1L, 1L), ))
expect_equal(or3s(dwh != 0L, w %between% c(-1L, 1L), q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dwh != 0L, w %between% c(-1L, 1L), q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dwi != 1L, n %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dwi != 1L, n %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dwj != 9L, p %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(dwj != 9L, p %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(dwk != 0L, , ),
             bor3(dwk != 0L, , ))
expect_equal(or3s(dwl != 1L, , ),
             bor3(dwl != 1L, , ))
expect_equal(or3s(dwm != 9L, , ),
             bor3(dwm != 9L, , ))
expect_equal(or3s(dwn != 0L, , x %in% 1:4),
             bor3(dwn != 0L, , x %in% 1:4))
expect_equal(or3s(dwo != 1L, , m %in% 1:4),
             bor3(dwo != 1L, , m %in% 1:4))
expect_equal(or3s(dwp != 9L, , b %in% 1:4),
             bor3(dwp != 9L, , b %in% 1:4))
expect_equal(or3s(dwq != 0L, o %between% c(-1L, 1L), ),
             bor3(dwq != 0L, o %between% c(-1L, 1L), ))
expect_equal(or3s(dwr != 1L, r %between% c(-1L, 1L), ),
             bor3(dwr != 1L, r %between% c(-1L, 1L), ))
expect_equal(or3s(dws != 9L, w %between% c(-1L, 1L), ),
             bor3(dws != 9L, w %between% c(-1L, 1L), ))
expect_equal(or3s(dwt != 0L, y %between% c(-1L, 1L), e %in% 1:4),
             bor3(dwt != 0L, y %between% c(-1L, 1L), e %in% 1:4))
expect_equal(or3s(dwu != 1L, x %between% c(-1L, 1L), s %in% 1:4),
             bor3(dwu != 1L, x %between% c(-1L, 1L), s %in% 1:4))
expect_equal(or3s(dwv != 9L, c %between% c(-1L, 1L), z %in% 1:4),
             bor3(dwv != 9L, c %between% c(-1L, 1L), z %in% 1:4))
expect_equal(or3s(dww != 0L, , ),
             bor3(dww != 0L, , ))
expect_equal(or3s(dwx != 1L, , ),
             bor3(dwx != 1L, , ))
expect_equal(or3s(dwy != 9L, , ),
             bor3(dwy != 9L, , ))
expect_equal(or3s(dwz != 0L, , m < 0L),
             bor3(dwz != 0L, , m < 0L))
expect_equal(or3s(dxa != 1L, , y < 1L),
             bor3(dxa != 1L, , y < 1L))
expect_equal(or3s(dxb != 9L, , o < 9L),
             bor3(dxb != 9L, , o < 9L))
expect_equal(or3s(dxc != 0L, c %between% c(-1L, 1L), ),
             bor3(dxc != 0L, c %between% c(-1L, 1L), ))
expect_equal(or3s(dxd != 1L, o %between% c(-1L, 1L), ),
             bor3(dxd != 1L, o %between% c(-1L, 1L), ))
expect_equal(or3s(dxe != 9L, b %between% c(-1L, 1L), ),
             bor3(dxe != 9L, b %between% c(-1L, 1L), ))
expect_equal(or3s(dxf != 0L, b %between% c(-1L, 1L), d < 0L),
             bor3(dxf != 0L, b %between% c(-1L, 1L), d < 0L))
expect_equal(or3s(dxg != 1L, n %between% c(-1L, 1L), g < 1L),
             bor3(dxg != 1L, n %between% c(-1L, 1L), g < 1L))
expect_equal(or3s(dxh != 9L, c %between% c(-1L, 1L), p < 9L),
             bor3(dxh != 9L, c %between% c(-1L, 1L), p < 9L))
expect_equal(or3s(dxi != 0L, , ),
             bor3(dxi != 0L, , ))
expect_equal(or3s(dxj != 1L, , ),
             bor3(dxj != 1L, , ))
expect_equal(or3s(dxk != 9L, , ),
             bor3(dxk != 9L, , ))
expect_equal(or3s(dxl != 0L, , i <= 0L),
             bor3(dxl != 0L, , i <= 0L))
expect_equal(or3s(dxm != 1L, , c <= 1L),
             bor3(dxm != 1L, , c <= 1L))
expect_equal(or3s(dxn != 9L, , z <= 9L),
             bor3(dxn != 9L, , z <= 9L))
expect_equal(or3s(dxo != 0L, j %between% c(-1L, 1L), ),
             bor3(dxo != 0L, j %between% c(-1L, 1L), ))
expect_equal(or3s(dxp != 1L, f %between% c(-1L, 1L), ),
             bor3(dxp != 1L, f %between% c(-1L, 1L), ))
expect_equal(or3s(dxq != 9L, n %between% c(-1L, 1L), ),
             bor3(dxq != 9L, n %between% c(-1L, 1L), ))
expect_equal(or3s(dxr != 0L, q %between% c(-1L, 1L), n <= 0L),
             bor3(dxr != 0L, q %between% c(-1L, 1L), n <= 0L))
expect_equal(or3s(dxs != 1L, k %between% c(-1L, 1L), u <= 1L),
             bor3(dxs != 1L, k %between% c(-1L, 1L), u <= 1L))
expect_equal(or3s(dxt != 9L, c %between% c(-1L, 1L), m <= 9L),
             bor3(dxt != 9L, c %between% c(-1L, 1L), m <= 9L))
expect_equal(or3s(dxu != 0L, , ),
             bor3(dxu != 0L, , ))
expect_equal(or3s(dxv != 1L, , ),
             bor3(dxv != 1L, , ))
expect_equal(or3s(dxw != 9L, , ),
             bor3(dxw != 9L, , ))
expect_equal(or3s(dxx != 0L, , m == 0L),
             bor3(dxx != 0L, , m == 0L))
expect_equal(or3s(dxy != 1L, , r == 1L),
             bor3(dxy != 1L, , r == 1L))
expect_equal(or3s(dxz != 9L, , y == 9L),
             bor3(dxz != 9L, , y == 9L))
expect_equal(or3s(dya != 0L, r %between% c(-1L, 1L), ),
             bor3(dya != 0L, r %between% c(-1L, 1L), ))
expect_equal(or3s(dyb != 1L, p %between% c(-1L, 1L), ),
             bor3(dyb != 1L, p %between% c(-1L, 1L), ))
expect_equal(or3s(dyc != 9L, j %between% c(-1L, 1L), ),
             bor3(dyc != 9L, j %between% c(-1L, 1L), ))
expect_equal(or3s(dyd != 0L, d %between% c(-1L, 1L), x == 0L),
             bor3(dyd != 0L, d %between% c(-1L, 1L), x == 0L))
expect_equal(or3s(dye != 1L, x %between% c(-1L, 1L), c == 1L),
             bor3(dye != 1L, x %between% c(-1L, 1L), c == 1L))
expect_equal(or3s(dyf != 9L, r %between% c(-1L, 1L), d == 9L),
             bor3(dyf != 9L, r %between% c(-1L, 1L), d == 9L))
expect_equal(or3s(dyg != 0L, , ),
             bor3(dyg != 0L, , ))
expect_equal(or3s(dyh != 1L, , ),
             bor3(dyh != 1L, , ))
expect_equal(or3s(dyi != 9L, , ),
             bor3(dyi != 9L, , ))
expect_equal(or3s(dyj != 0L, , l > 0L),
             bor3(dyj != 0L, , l > 0L))
expect_equal(or3s(dyk != 1L, , t > 1L),
             bor3(dyk != 1L, , t > 1L))
expect_equal(or3s(dyl != 9L, , q > 9L),
             bor3(dyl != 9L, , q > 9L))
expect_equal(or3s(dym != 0L, e %between% c(-1L, 1L), ),
             bor3(dym != 0L, e %between% c(-1L, 1L), ))
expect_equal(or3s(dyn != 1L, s %between% c(-1L, 1L), ),
             bor3(dyn != 1L, s %between% c(-1L, 1L), ))
expect_equal(or3s(dyo != 9L, a %between% c(-1L, 1L), ),
             bor3(dyo != 9L, a %between% c(-1L, 1L), ))
expect_equal(or3s(dyp != 0L, v %between% c(-1L, 1L), f > 0L),
             bor3(dyp != 0L, v %between% c(-1L, 1L), f > 0L))
expect_equal(or3s(dyq != 1L, h %between% c(-1L, 1L), q > 1L),
             bor3(dyq != 1L, h %between% c(-1L, 1L), q > 1L))
expect_equal(or3s(dyr != 9L, o %between% c(-1L, 1L), i > 9L),
             bor3(dyr != 9L, o %between% c(-1L, 1L), i > 9L))
expect_equal(or3s(dys != 0L, , ),
             bor3(dys != 0L, , ))
expect_equal(or3s(dyt != 1L, , ),
             bor3(dyt != 1L, , ))
expect_equal(or3s(dyu != 9L, , ),
             bor3(dyu != 9L, , ))
expect_equal(or3s(dyv != 0L, , x >= 0L),
             bor3(dyv != 0L, , x >= 0L))
expect_equal(or3s(dyw != 1L, , x >= 1L),
             bor3(dyw != 1L, , x >= 1L))
expect_equal(or3s(dyx != 9L, , u >= 9L),
             bor3(dyx != 9L, , u >= 9L))
expect_equal(or3s(dyy != 0L, x %between% c(-1L, 1L), ),
             bor3(dyy != 0L, x %between% c(-1L, 1L), ))
expect_equal(or3s(dyz != 1L, k %between% c(-1L, 1L), ),
             bor3(dyz != 1L, k %between% c(-1L, 1L), ))
expect_equal(or3s(ea != 9L, o %between% c(-1L, 1L), ),
             bor3(ea != 9L, o %between% c(-1L, 1L), ))
expect_equal(or3s(eb != 0L, l %between% c(-1L, 1L), b >= 0L),
             bor3(eb != 0L, l %between% c(-1L, 1L), b >= 0L))
expect_equal(or3s(ec != 1L, w %between% c(-1L, 1L), f >= 1L),
             bor3(ec != 1L, w %between% c(-1L, 1L), f >= 1L))
expect_equal(or3s(ed != 9L, q %between% c(-1L, 1L), v >= 9L),
             bor3(ed != 9L, q %between% c(-1L, 1L), v >= 9L))
expect_equal(or3s(ee != 0L, , ),
             bor3(ee != 0L, , ))
expect_equal(or3s(ef != 1L, , ),
             bor3(ef != 1L, , ))
expect_equal(or3s(eg != 9L, , ),
             bor3(eg != 9L, , ))
expect_equal(or3s(eh != 0L, , logi_z),
             bor3(eh != 0L, , logi_z))
expect_equal(or3s(ei != 1L, , logi_v),
             bor3(ei != 1L, , logi_v))
expect_equal(or3s(ej != 9L, , logi_m),
             bor3(ej != 9L, , logi_m))
expect_equal(or3s(ek != 0L, d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ek != 0L, d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(el != 1L, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(el != 1L, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(em != 9L, u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(em != 9L, u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(en != 0L, p %in% c(4L, -3L, 2L, 7L, 8L), logi_l),
             bor3(en != 0L, p %in% c(4L, -3L, 2L, 7L, 8L), logi_l))
expect_equal(or3s(eo != 1L, q %in% c(4L, -3L, 2L, 7L, 8L), logi_m),
             bor3(eo != 1L, q %in% c(4L, -3L, 2L, 7L, 8L), logi_m))
expect_equal(or3s(ep != 9L, m %in% c(4L, -3L, 2L, 7L, 8L), logi_f),
             bor3(ep != 9L, m %in% c(4L, -3L, 2L, 7L, 8L), logi_f))
expect_equal(or3s(eq != 0L, , ),
             bor3(eq != 0L, , ))
expect_equal(or3s(er != 1L, , ),
             bor3(er != 1L, , ))
expect_equal(or3s(es != 9L, , ),
             bor3(es != 9L, , ))
expect_equal(or3s(et != 0L, , logi_t),
             bor3(et != 0L, , logi_t))
expect_equal(or3s(eu != 1L, , logi_c),
             bor3(eu != 1L, , logi_c))
expect_equal(or3s(ev != 9L, , logi_l),
             bor3(ev != 9L, , logi_l))
expect_equal(or3s(ew != 0L, e %in% 1:4, ),
             bor3(ew != 0L, e %in% 1:4, ))
expect_equal(or3s(ex != 1L, h %in% 1:4, ),
             bor3(ex != 1L, h %in% 1:4, ))
expect_equal(or3s(ey != 9L, l %in% 1:4, ),
             bor3(ey != 9L, l %in% 1:4, ))
expect_equal(or3s(ez != 0L, m %in% 1:4, logi_f),
             bor3(ez != 0L, m %in% 1:4, logi_f))
expect_equal(or3s(eaa != 1L, g %in% 1:4, logi_q),
             bor3(eaa != 1L, g %in% 1:4, logi_q))
expect_equal(or3s(eab != 9L, l %in% 1:4, logi_u),
             bor3(eab != 9L, l %in% 1:4, logi_u))
expect_equal(or3s(eac != 0L, , ),
             bor3(eac != 0L, , ))
expect_equal(or3s(ead != 1L, , ),
             bor3(ead != 1L, , ))
expect_equal(or3s(eae != 9L, , ),
             bor3(eae != 9L, , ))
expect_equal(or3s(eaf != 0L, , !logi_y),
             bor3(eaf != 0L, , !logi_y))
expect_equal(or3s(eag != 1L, , !logi_q),
             bor3(eag != 1L, , !logi_q))
expect_equal(or3s(eah != 9L, , !logi_a),
             bor3(eah != 9L, , !logi_a))
expect_equal(or3s(eai != 0L, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eai != 0L, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(eaj != 1L, g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eaj != 1L, g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(eak != 9L, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eak != 9L, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(eal != 0L, j %in% c(4L, -3L, 2L, 7L, 8L), !logi_u),
             bor3(eal != 0L, j %in% c(4L, -3L, 2L, 7L, 8L), !logi_u))
expect_equal(or3s(eam != 1L, t %in% c(4L, -3L, 2L, 7L, 8L), !logi_m),
             bor3(eam != 1L, t %in% c(4L, -3L, 2L, 7L, 8L), !logi_m))
expect_equal(or3s(ean != 9L, t %in% c(4L, -3L, 2L, 7L, 8L), !logi_r),
             bor3(ean != 9L, t %in% c(4L, -3L, 2L, 7L, 8L), !logi_r))
expect_equal(or3s(eao != 0L, , ),
             bor3(eao != 0L, , ))
expect_equal(or3s(eap != 1L, , ),
             bor3(eap != 1L, , ))
expect_equal(or3s(eaq != 9L, , ),
             bor3(eaq != 9L, , ))
expect_equal(or3s(ear != 0L, , !logi_e),
             bor3(ear != 0L, , !logi_e))
expect_equal(or3s(eas != 1L, , !logi_a),
             bor3(eas != 1L, , !logi_a))
expect_equal(or3s(eat != 9L, , !logi_l),
             bor3(eat != 9L, , !logi_l))
expect_equal(or3s(eau != 0L, m %in% 1:4, ),
             bor3(eau != 0L, m %in% 1:4, ))
expect_equal(or3s(eav != 1L, w %in% 1:4, ),
             bor3(eav != 1L, w %in% 1:4, ))
expect_equal(or3s(eaw != 9L, k %in% 1:4, ),
             bor3(eaw != 9L, k %in% 1:4, ))
expect_equal(or3s(eax != 0L, n %in% 1:4, !logi_n),
             bor3(eax != 0L, n %in% 1:4, !logi_n))
expect_equal(or3s(eay != 1L, j %in% 1:4, !logi_w),
             bor3(eay != 1L, j %in% 1:4, !logi_w))
expect_equal(or3s(eaz != 9L, b %in% 1:4, !logi_l),
             bor3(eaz != 9L, b %in% 1:4, !logi_l))
expect_equal(or3s(eba != 0L, , ),
             bor3(eba != 0L, , ))
expect_equal(or3s(ebb != 1L, , ),
             bor3(ebb != 1L, , ))
expect_equal(or3s(ebc != 9L, , ),
             bor3(ebc != 9L, , ))
expect_equal(or3s(ebd != 0L, , b != 0L),
             bor3(ebd != 0L, , b != 0L))
expect_equal(or3s(ebe != 1L, , o != 1L),
             bor3(ebe != 1L, , o != 1L))
expect_equal(or3s(ebf != 9L, , s != 9L),
             bor3(ebf != 9L, , s != 9L))
expect_equal(or3s(ebg != 0L, s %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ebg != 0L, s %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ebh != 1L, n %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ebh != 1L, n %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ebi != 9L, g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ebi != 9L, g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ebj != 0L, x %in% c(4L, -3L, 2L, 7L, 8L), t != 0L),
             bor3(ebj != 0L, x %in% c(4L, -3L, 2L, 7L, 8L), t != 0L))
expect_equal(or3s(ebk != 1L, j %in% c(4L, -3L, 2L, 7L, 8L), a != 1L),
             bor3(ebk != 1L, j %in% c(4L, -3L, 2L, 7L, 8L), a != 1L))
expect_equal(or3s(ebl != 9L, r %in% c(4L, -3L, 2L, 7L, 8L), g != 9L),
             bor3(ebl != 9L, r %in% c(4L, -3L, 2L, 7L, 8L), g != 9L))
expect_equal(or3s(ebm != 0L, , ),
             bor3(ebm != 0L, , ))
expect_equal(or3s(ebn != 1L, , ),
             bor3(ebn != 1L, , ))
expect_equal(or3s(ebo != 9L, , ),
             bor3(ebo != 9L, , ))
expect_equal(or3s(ebp != 0L, , k != 0L),
             bor3(ebp != 0L, , k != 0L))
expect_equal(or3s(ebq != 1L, , g != 1L),
             bor3(ebq != 1L, , g != 1L))
expect_equal(or3s(ebr != 9L, , o != 9L),
             bor3(ebr != 9L, , o != 9L))
expect_equal(or3s(ebs != 0L, z %in% 1:4, ),
             bor3(ebs != 0L, z %in% 1:4, ))
expect_equal(or3s(ebt != 1L, x %in% 1:4, ),
             bor3(ebt != 1L, x %in% 1:4, ))
expect_equal(or3s(ebu != 9L, i %in% 1:4, ),
             bor3(ebu != 9L, i %in% 1:4, ))
expect_equal(or3s(ebv != 0L, g %in% 1:4, a != 0L),
             bor3(ebv != 0L, g %in% 1:4, a != 0L))
expect_equal(or3s(ebw != 1L, d %in% 1:4, o != 1L),
             bor3(ebw != 1L, d %in% 1:4, o != 1L))
expect_equal(or3s(ebx != 9L, c %in% 1:4, r != 9L),
             bor3(ebx != 9L, c %in% 1:4, r != 9L))
expect_equal(or3s(eby != 0L, , ),
             bor3(eby != 0L, , ))
expect_equal(or3s(ebz != 1L, , ),
             bor3(ebz != 1L, , ))
expect_equal(or3s(eca != 9L, , ),
             bor3(eca != 9L, , ))
expect_equal(or3s(ecb != 0L, , k %between% c(-1L, 1L)),
             bor3(ecb != 0L, , k %between% c(-1L, 1L)))
expect_equal(or3s(ecc != 1L, , r %between% c(-1L, 1L)),
             bor3(ecc != 1L, , r %between% c(-1L, 1L)))
expect_equal(or3s(ecd != 9L, , a %between% c(-1L, 1L)),
             bor3(ecd != 9L, , a %between% c(-1L, 1L)))
expect_equal(or3s(ece != 0L, f %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ece != 0L, f %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ecf != 1L, o %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ecf != 1L, o %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ecg != 9L, q %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ecg != 9L, q %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ech != 0L, d %in% c(4L, -3L, 2L, 7L, 8L), s %between% c(-1L, 1L)),
             bor3(ech != 0L, d %in% c(4L, -3L, 2L, 7L, 8L), s %between% c(-1L, 1L)))
expect_equal(or3s(eci != 1L, y %in% c(4L, -3L, 2L, 7L, 8L), r %between% c(-1L, 1L)),
             bor3(eci != 1L, y %in% c(4L, -3L, 2L, 7L, 8L), r %between% c(-1L, 1L)))
expect_equal(or3s(ecj != 9L, a %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L)),
             bor3(ecj != 9L, a %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L)))
expect_equal(or3s(eck != 0L, , ),
             bor3(eck != 0L, , ))
expect_equal(or3s(ecl != 1L, , ),
             bor3(ecl != 1L, , ))
expect_equal(or3s(ecm != 9L, , ),
             bor3(ecm != 9L, , ))
expect_equal(or3s(ecn != 0L, , p %between% c(-1L, 1L)),
             bor3(ecn != 0L, , p %between% c(-1L, 1L)))
expect_equal(or3s(eco != 1L, , p %between% c(-1L, 1L)),
             bor3(eco != 1L, , p %between% c(-1L, 1L)))
expect_equal(or3s(ecp != 9L, , k %between% c(-1L, 1L)),
             bor3(ecp != 9L, , k %between% c(-1L, 1L)))
expect_equal(or3s(ecq != 0L, k %in% 1:4, ),
             bor3(ecq != 0L, k %in% 1:4, ))
expect_equal(or3s(ecr != 1L, s %in% 1:4, ),
             bor3(ecr != 1L, s %in% 1:4, ))
expect_equal(or3s(ecs != 9L, q %in% 1:4, ),
             bor3(ecs != 9L, q %in% 1:4, ))
expect_equal(or3s(ect != 0L, g %in% 1:4, w %between% c(-1L, 1L)),
             bor3(ect != 0L, g %in% 1:4, w %between% c(-1L, 1L)))
expect_equal(or3s(ecu != 1L, k %in% 1:4, c %between% c(-1L, 1L)),
             bor3(ecu != 1L, k %in% 1:4, c %between% c(-1L, 1L)))
expect_equal(or3s(ecv != 9L, i %in% 1:4, a %between% c(-1L, 1L)),
             bor3(ecv != 9L, i %in% 1:4, a %between% c(-1L, 1L)))
expect_equal(or3s(ecw != 0L, , ),
             bor3(ecw != 0L, , ))
expect_equal(or3s(ecx != 1L, , ),
             bor3(ecx != 1L, , ))
expect_equal(or3s(ecy != 9L, , ),
             bor3(ecy != 9L, , ))
expect_equal(or3s(ecz != 0L, , a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ecz != 0L, , a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(eda != 1L, , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(eda != 1L, , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(edb != 9L, , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(edb != 9L, , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(edc != 0L, e %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(edc != 0L, e %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(edd != 1L, q %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(edd != 1L, q %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ede != 9L, b %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ede != 9L, b %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(edf != 0L, k %in% c(4L, -3L, 2L, 7L, 8L), p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(edf != 0L, k %in% c(4L, -3L, 2L, 7L, 8L), p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(edg != 1L, r %in% c(4L, -3L, 2L, 7L, 8L), l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(edg != 1L, r %in% c(4L, -3L, 2L, 7L, 8L), l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(edh != 9L, b %in% c(4L, -3L, 2L, 7L, 8L), x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(edh != 9L, b %in% c(4L, -3L, 2L, 7L, 8L), x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(edi != 0L, , ),
             bor3(edi != 0L, , ))
expect_equal(or3s(edj != 1L, , ),
             bor3(edj != 1L, , ))
expect_equal(or3s(edk != 9L, , ),
             bor3(edk != 9L, , ))
expect_equal(or3s(edl != 0L, , u %in% 1:4),
             bor3(edl != 0L, , u %in% 1:4))
expect_equal(or3s(edm != 1L, , v %in% 1:4),
             bor3(edm != 1L, , v %in% 1:4))
expect_equal(or3s(edn != 9L, , h %in% 1:4),
             bor3(edn != 9L, , h %in% 1:4))
expect_equal(or3s(edo != 0L, f %in% 1:4, ),
             bor3(edo != 0L, f %in% 1:4, ))
expect_equal(or3s(edp != 1L, h %in% 1:4, ),
             bor3(edp != 1L, h %in% 1:4, ))
expect_equal(or3s(edq != 9L, h %in% 1:4, ),
             bor3(edq != 9L, h %in% 1:4, ))
expect_equal(or3s(edr != 0L, y %in% 1:4, i %in% 1:4),
             bor3(edr != 0L, y %in% 1:4, i %in% 1:4))
expect_equal(or3s(eds != 1L, d %in% 1:4, s %in% 1:4),
             bor3(eds != 1L, d %in% 1:4, s %in% 1:4))
expect_equal(or3s(edt != 9L, q %in% 1:4, i %in% 1:4),
             bor3(edt != 9L, q %in% 1:4, i %in% 1:4))
expect_equal(or3s(edu != 0L, , ),
             bor3(edu != 0L, , ))
expect_equal(or3s(edv != 1L, , ),
             bor3(edv != 1L, , ))
expect_equal(or3s(edw != 9L, , ),
             bor3(edw != 9L, , ))
expect_equal(or3s(edx != 0L, , v < 0L),
             bor3(edx != 0L, , v < 0L))
expect_equal(or3s(edy != 1L, , w < 1L),
             bor3(edy != 1L, , w < 1L))
expect_equal(or3s(edz != 9L, , q < 9L),
             bor3(edz != 9L, , q < 9L))
expect_equal(or3s(eea != 0L, s %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eea != 0L, s %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(eeb != 1L, u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eeb != 1L, u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(eec != 9L, f %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eec != 9L, f %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(eed != 0L, c %in% c(4L, -3L, 2L, 7L, 8L), d < 0L),
             bor3(eed != 0L, c %in% c(4L, -3L, 2L, 7L, 8L), d < 0L))
expect_equal(or3s(eee != 1L, p %in% c(4L, -3L, 2L, 7L, 8L), y < 1L),
             bor3(eee != 1L, p %in% c(4L, -3L, 2L, 7L, 8L), y < 1L))
expect_equal(or3s(eef != 9L, n %in% c(4L, -3L, 2L, 7L, 8L), a < 9L),
             bor3(eef != 9L, n %in% c(4L, -3L, 2L, 7L, 8L), a < 9L))
expect_equal(or3s(eeg != 0L, , ),
             bor3(eeg != 0L, , ))
expect_equal(or3s(eeh != 1L, , ),
             bor3(eeh != 1L, , ))
expect_equal(or3s(eei != 9L, , ),
             bor3(eei != 9L, , ))
expect_equal(or3s(eej != 0L, , w < 0L),
             bor3(eej != 0L, , w < 0L))
expect_equal(or3s(eek != 1L, , c < 1L),
             bor3(eek != 1L, , c < 1L))
expect_equal(or3s(eel != 9L, , k < 9L),
             bor3(eel != 9L, , k < 9L))
expect_equal(or3s(eem != 0L, w %in% 1:4, ),
             bor3(eem != 0L, w %in% 1:4, ))
expect_equal(or3s(een != 1L, g %in% 1:4, ),
             bor3(een != 1L, g %in% 1:4, ))
expect_equal(or3s(eeo != 9L, v %in% 1:4, ),
             bor3(eeo != 9L, v %in% 1:4, ))
expect_equal(or3s(eep != 0L, c %in% 1:4, o < 0L),
             bor3(eep != 0L, c %in% 1:4, o < 0L))
expect_equal(or3s(eeq != 1L, u %in% 1:4, m < 1L),
             bor3(eeq != 1L, u %in% 1:4, m < 1L))
expect_equal(or3s(eer != 9L, l %in% 1:4, p < 9L),
             bor3(eer != 9L, l %in% 1:4, p < 9L))
expect_equal(or3s(ees != 0L, , ),
             bor3(ees != 0L, , ))
expect_equal(or3s(eet != 1L, , ),
             bor3(eet != 1L, , ))
expect_equal(or3s(eeu != 9L, , ),
             bor3(eeu != 9L, , ))
expect_equal(or3s(eev != 0L, , w <= 0L),
             bor3(eev != 0L, , w <= 0L))
expect_equal(or3s(eew != 1L, , w <= 1L),
             bor3(eew != 1L, , w <= 1L))
expect_equal(or3s(eex != 9L, , m <= 9L),
             bor3(eex != 9L, , m <= 9L))
expect_equal(or3s(eey != 0L, l %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eey != 0L, l %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(eez != 1L, i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eez != 1L, i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(efa != 9L, p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(efa != 9L, p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(efb != 0L, r %in% c(4L, -3L, 2L, 7L, 8L), m <= 0L),
             bor3(efb != 0L, r %in% c(4L, -3L, 2L, 7L, 8L), m <= 0L))
expect_equal(or3s(efc != 1L, n %in% c(4L, -3L, 2L, 7L, 8L), a <= 1L),
             bor3(efc != 1L, n %in% c(4L, -3L, 2L, 7L, 8L), a <= 1L))
expect_equal(or3s(efd != 9L, b %in% c(4L, -3L, 2L, 7L, 8L), k <= 9L),
             bor3(efd != 9L, b %in% c(4L, -3L, 2L, 7L, 8L), k <= 9L))
expect_equal(or3s(efe != 0L, , ),
             bor3(efe != 0L, , ))
expect_equal(or3s(eff != 1L, , ),
             bor3(eff != 1L, , ))
expect_equal(or3s(efg != 9L, , ),
             bor3(efg != 9L, , ))
expect_equal(or3s(efh != 0L, , h <= 0L),
             bor3(efh != 0L, , h <= 0L))
expect_equal(or3s(efi != 1L, , a <= 1L),
             bor3(efi != 1L, , a <= 1L))
expect_equal(or3s(efj != 9L, , g <= 9L),
             bor3(efj != 9L, , g <= 9L))
expect_equal(or3s(efk != 0L, q %in% 1:4, ),
             bor3(efk != 0L, q %in% 1:4, ))
expect_equal(or3s(efl != 1L, j %in% 1:4, ),
             bor3(efl != 1L, j %in% 1:4, ))
expect_equal(or3s(efm != 9L, x %in% 1:4, ),
             bor3(efm != 9L, x %in% 1:4, ))
expect_equal(or3s(efn != 0L, f %in% 1:4, o <= 0L),
             bor3(efn != 0L, f %in% 1:4, o <= 0L))
expect_equal(or3s(efo != 1L, s %in% 1:4, c <= 1L),
             bor3(efo != 1L, s %in% 1:4, c <= 1L))
expect_equal(or3s(efp != 9L, n %in% 1:4, e <= 9L),
             bor3(efp != 9L, n %in% 1:4, e <= 9L))
expect_equal(or3s(efq != 0L, , ),
             bor3(efq != 0L, , ))
expect_equal(or3s(efr != 1L, , ),
             bor3(efr != 1L, , ))
expect_equal(or3s(efs != 9L, , ),
             bor3(efs != 9L, , ))
expect_equal(or3s(eft != 0L, , k == 0L),
             bor3(eft != 0L, , k == 0L))
expect_equal(or3s(efu != 1L, , d == 1L),
             bor3(efu != 1L, , d == 1L))
expect_equal(or3s(efv != 9L, , t == 9L),
             bor3(efv != 9L, , t == 9L))
expect_equal(or3s(efw != 0L, i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(efw != 0L, i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(efx != 1L, y %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(efx != 1L, y %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(efy != 9L, y %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(efy != 9L, y %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(efz != 0L, h %in% c(4L, -3L, 2L, 7L, 8L), m == 0L),
             bor3(efz != 0L, h %in% c(4L, -3L, 2L, 7L, 8L), m == 0L))
expect_equal(or3s(ega != 1L, f %in% c(4L, -3L, 2L, 7L, 8L), d == 1L),
             bor3(ega != 1L, f %in% c(4L, -3L, 2L, 7L, 8L), d == 1L))
expect_equal(or3s(egb != 9L, a %in% c(4L, -3L, 2L, 7L, 8L), t == 9L),
             bor3(egb != 9L, a %in% c(4L, -3L, 2L, 7L, 8L), t == 9L))
expect_equal(or3s(egc != 0L, , ),
             bor3(egc != 0L, , ))
expect_equal(or3s(egd != 1L, , ),
             bor3(egd != 1L, , ))
expect_equal(or3s(ege != 9L, , ),
             bor3(ege != 9L, , ))
expect_equal(or3s(egf != 0L, , y == 0L),
             bor3(egf != 0L, , y == 0L))
expect_equal(or3s(egg != 1L, , w == 1L),
             bor3(egg != 1L, , w == 1L))
expect_equal(or3s(egh != 9L, , c == 9L),
             bor3(egh != 9L, , c == 9L))
expect_equal(or3s(egi != 0L, d %in% 1:4, ),
             bor3(egi != 0L, d %in% 1:4, ))
expect_equal(or3s(egj != 1L, b %in% 1:4, ),
             bor3(egj != 1L, b %in% 1:4, ))
expect_equal(or3s(egk != 9L, z %in% 1:4, ),
             bor3(egk != 9L, z %in% 1:4, ))
expect_equal(or3s(egl != 0L, q %in% 1:4, c == 0L),
             bor3(egl != 0L, q %in% 1:4, c == 0L))
expect_equal(or3s(egm != 1L, d %in% 1:4, v == 1L),
             bor3(egm != 1L, d %in% 1:4, v == 1L))
expect_equal(or3s(egn != 9L, k %in% 1:4, a == 9L),
             bor3(egn != 9L, k %in% 1:4, a == 9L))
expect_equal(or3s(ego != 0L, , ),
             bor3(ego != 0L, , ))
expect_equal(or3s(egp != 1L, , ),
             bor3(egp != 1L, , ))
expect_equal(or3s(egq != 9L, , ),
             bor3(egq != 9L, , ))
expect_equal(or3s(egr != 0L, , f > 0L),
             bor3(egr != 0L, , f > 0L))
expect_equal(or3s(egs != 1L, , w > 1L),
             bor3(egs != 1L, , w > 1L))
expect_equal(or3s(egt != 9L, , c > 9L),
             bor3(egt != 9L, , c > 9L))
expect_equal(or3s(egu != 0L, s %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(egu != 0L, s %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(egv != 1L, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(egv != 1L, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(egw != 9L, g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(egw != 9L, g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(egx != 0L, e %in% c(4L, -3L, 2L, 7L, 8L), v > 0L),
             bor3(egx != 0L, e %in% c(4L, -3L, 2L, 7L, 8L), v > 0L))
expect_equal(or3s(egy != 1L, n %in% c(4L, -3L, 2L, 7L, 8L), w > 1L),
             bor3(egy != 1L, n %in% c(4L, -3L, 2L, 7L, 8L), w > 1L))
expect_equal(or3s(egz != 9L, u %in% c(4L, -3L, 2L, 7L, 8L), h > 9L),
             bor3(egz != 9L, u %in% c(4L, -3L, 2L, 7L, 8L), h > 9L))
expect_equal(or3s(eha != 0L, , ),
             bor3(eha != 0L, , ))
expect_equal(or3s(ehb != 1L, , ),
             bor3(ehb != 1L, , ))
expect_equal(or3s(ehc != 9L, , ),
             bor3(ehc != 9L, , ))
expect_equal(or3s(ehd != 0L, , l > 0L),
             bor3(ehd != 0L, , l > 0L))
expect_equal(or3s(ehe != 1L, , q > 1L),
             bor3(ehe != 1L, , q > 1L))
expect_equal(or3s(ehf != 9L, , r > 9L),
             bor3(ehf != 9L, , r > 9L))
expect_equal(or3s(ehg != 0L, g %in% 1:4, ),
             bor3(ehg != 0L, g %in% 1:4, ))
expect_equal(or3s(ehh != 1L, h %in% 1:4, ),
             bor3(ehh != 1L, h %in% 1:4, ))
expect_equal(or3s(ehi != 9L, x %in% 1:4, ),
             bor3(ehi != 9L, x %in% 1:4, ))
expect_equal(or3s(ehj != 0L, z %in% 1:4, k > 0L),
             bor3(ehj != 0L, z %in% 1:4, k > 0L))
expect_equal(or3s(ehk != 1L, p %in% 1:4, p > 1L),
             bor3(ehk != 1L, p %in% 1:4, p > 1L))
expect_equal(or3s(ehl != 9L, o %in% 1:4, i > 9L),
             bor3(ehl != 9L, o %in% 1:4, i > 9L))
expect_equal(or3s(ehm != 0L, , ),
             bor3(ehm != 0L, , ))
expect_equal(or3s(ehn != 1L, , ),
             bor3(ehn != 1L, , ))
expect_equal(or3s(eho != 9L, , ),
             bor3(eho != 9L, , ))
expect_equal(or3s(ehp != 0L, , n >= 0L),
             bor3(ehp != 0L, , n >= 0L))
expect_equal(or3s(ehq != 1L, , n >= 1L),
             bor3(ehq != 1L, , n >= 1L))
expect_equal(or3s(ehr != 9L, , x >= 9L),
             bor3(ehr != 9L, , x >= 9L))
expect_equal(or3s(ehs != 0L, t %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ehs != 0L, t %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(eht != 1L, b %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(eht != 1L, b %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ehu != 9L, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ehu != 9L, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ehv != 0L, y %in% c(4L, -3L, 2L, 7L, 8L), w >= 0L),
             bor3(ehv != 0L, y %in% c(4L, -3L, 2L, 7L, 8L), w >= 0L))
expect_equal(or3s(ehw != 1L, w %in% c(4L, -3L, 2L, 7L, 8L), r >= 1L),
             bor3(ehw != 1L, w %in% c(4L, -3L, 2L, 7L, 8L), r >= 1L))
expect_equal(or3s(ehx != 9L, o %in% c(4L, -3L, 2L, 7L, 8L), o >= 9L),
             bor3(ehx != 9L, o %in% c(4L, -3L, 2L, 7L, 8L), o >= 9L))
expect_equal(or3s(ehy != 0L, , ),
             bor3(ehy != 0L, , ))
expect_equal(or3s(ehz != 1L, , ),
             bor3(ehz != 1L, , ))
expect_equal(or3s(eia != 9L, , ),
             bor3(eia != 9L, , ))
expect_equal(or3s(eib != 0L, , u >= 0L),
             bor3(eib != 0L, , u >= 0L))
expect_equal(or3s(eic != 1L, , d >= 1L),
             bor3(eic != 1L, , d >= 1L))
expect_equal(or3s(eid != 9L, , t >= 9L),
             bor3(eid != 9L, , t >= 9L))
expect_equal(or3s(eie != 0L, i %in% 1:4, ),
             bor3(eie != 0L, i %in% 1:4, ))
expect_equal(or3s(eif != 1L, r %in% 1:4, ),
             bor3(eif != 1L, r %in% 1:4, ))
expect_equal(or3s(eig != 9L, j %in% 1:4, ),
             bor3(eig != 9L, j %in% 1:4, ))
expect_equal(or3s(eih != 0L, p %in% 1:4, u >= 0L),
             bor3(eih != 0L, p %in% 1:4, u >= 0L))
expect_equal(or3s(eii != 1L, x %in% 1:4, k >= 1L),
             bor3(eii != 1L, x %in% 1:4, k >= 1L))
expect_equal(or3s(eij != 9L, y %in% 1:4, n >= 9L),
             bor3(eij != 9L, y %in% 1:4, n >= 9L))
expect_equal(or3s(eik != 0L, , ),
             bor3(eik != 0L, , ))
expect_equal(or3s(eil != 1L, , ),
             bor3(eil != 1L, , ))
expect_equal(or3s(eim != 9L, , ),
             bor3(eim != 9L, , ))
expect_equal(or3s(ein != 0L, , logi_r),
             bor3(ein != 0L, , logi_r))
expect_equal(or3s(eio != 1L, , logi_h),
             bor3(eio != 1L, , logi_h))
expect_equal(or3s(eip != 9L, , logi_u),
             bor3(eip != 9L, , logi_u))
expect_equal(or3s(eiq != 0L, z < 0L, ),
             bor3(eiq != 0L, z < 0L, ))
expect_equal(or3s(eir != 1L, e < 1L, ),
             bor3(eir != 1L, e < 1L, ))
expect_equal(or3s(eis != 9L, k < 9L, ),
             bor3(eis != 9L, k < 9L, ))
expect_equal(or3s(eit != 0L, k < 0L, logi_q),
             bor3(eit != 0L, k < 0L, logi_q))
expect_equal(or3s(eiu != 1L, n < 1L, logi_z),
             bor3(eiu != 1L, n < 1L, logi_z))
expect_equal(or3s(eiv != 9L, l < 9L, logi_m),
             bor3(eiv != 9L, l < 9L, logi_m))
expect_equal(or3s(eiw != 0L, , ),
             bor3(eiw != 0L, , ))
expect_equal(or3s(eix != 1L, , ),
             bor3(eix != 1L, , ))
expect_equal(or3s(eiy != 9L, , ),
             bor3(eiy != 9L, , ))
expect_equal(or3s(eiz != 0L, , !logi_v),
             bor3(eiz != 0L, , !logi_v))
expect_equal(or3s(eja != 1L, , !logi_e),
             bor3(eja != 1L, , !logi_e))
expect_equal(or3s(ejb != 9L, , !logi_m),
             bor3(ejb != 9L, , !logi_m))
expect_equal(or3s(ejc != 0L, v < 0L, ),
             bor3(ejc != 0L, v < 0L, ))
expect_equal(or3s(ejd != 1L, q < 1L, ),
             bor3(ejd != 1L, q < 1L, ))
expect_equal(or3s(eje != 9L, z < 9L, ),
             bor3(eje != 9L, z < 9L, ))
expect_equal(or3s(ejf != 0L, b < 0L, !logi_u),
             bor3(ejf != 0L, b < 0L, !logi_u))
expect_equal(or3s(ejg != 1L, f < 1L, !logi_t),
             bor3(ejg != 1L, f < 1L, !logi_t))
expect_equal(or3s(ejh != 9L, l < 9L, !logi_u),
             bor3(ejh != 9L, l < 9L, !logi_u))
expect_equal(or3s(eji != 0L, , ),
             bor3(eji != 0L, , ))
expect_equal(or3s(ejj != 1L, , ),
             bor3(ejj != 1L, , ))
expect_equal(or3s(ejk != 9L, , ),
             bor3(ejk != 9L, , ))
expect_equal(or3s(ejl != 0L, , n != 0L),
             bor3(ejl != 0L, , n != 0L))
expect_equal(or3s(ejm != 1L, , m != 1L),
             bor3(ejm != 1L, , m != 1L))
expect_equal(or3s(ejn != 9L, , k != 9L),
             bor3(ejn != 9L, , k != 9L))
expect_equal(or3s(ejo != 0L, l < 0L, ),
             bor3(ejo != 0L, l < 0L, ))
expect_equal(or3s(ejp != 1L, z < 1L, ),
             bor3(ejp != 1L, z < 1L, ))
expect_equal(or3s(ejq != 9L, u < 9L, ),
             bor3(ejq != 9L, u < 9L, ))
expect_equal(or3s(ejr != 0L, e < 0L, v != 0L),
             bor3(ejr != 0L, e < 0L, v != 0L))
expect_equal(or3s(ejs != 1L, u < 1L, z != 1L),
             bor3(ejs != 1L, u < 1L, z != 1L))
expect_equal(or3s(ejt != 9L, w < 9L, u != 9L),
             bor3(ejt != 9L, w < 9L, u != 9L))
expect_equal(or3s(eju != 0L, , ),
             bor3(eju != 0L, , ))
expect_equal(or3s(ejv != 1L, , ),
             bor3(ejv != 1L, , ))
expect_equal(or3s(ejw != 9L, , ),
             bor3(ejw != 9L, , ))
expect_equal(or3s(ejx != 0L, , j %between% c(-1L, 1L)),
             bor3(ejx != 0L, , j %between% c(-1L, 1L)))
expect_equal(or3s(ejy != 1L, , h %between% c(-1L, 1L)),
             bor3(ejy != 1L, , h %between% c(-1L, 1L)))
expect_equal(or3s(ejz != 9L, , k %between% c(-1L, 1L)),
             bor3(ejz != 9L, , k %between% c(-1L, 1L)))
expect_equal(or3s(eka != 0L, x < 0L, ),
             bor3(eka != 0L, x < 0L, ))
expect_equal(or3s(ekb != 1L, q < 1L, ),
             bor3(ekb != 1L, q < 1L, ))
expect_equal(or3s(ekc != 9L, p < 9L, ),
             bor3(ekc != 9L, p < 9L, ))
expect_equal(or3s(ekd != 0L, q < 0L, g %between% c(-1L, 1L)),
             bor3(ekd != 0L, q < 0L, g %between% c(-1L, 1L)))
expect_equal(or3s(eke != 1L, b < 1L, y %between% c(-1L, 1L)),
             bor3(eke != 1L, b < 1L, y %between% c(-1L, 1L)))
expect_equal(or3s(ekf != 9L, y < 9L, j %between% c(-1L, 1L)),
             bor3(ekf != 9L, y < 9L, j %between% c(-1L, 1L)))
expect_equal(or3s(ekg != 0L, , ),
             bor3(ekg != 0L, , ))
expect_equal(or3s(ekh != 1L, , ),
             bor3(ekh != 1L, , ))
expect_equal(or3s(eki != 9L, , ),
             bor3(eki != 9L, , ))
expect_equal(or3s(ekj != 0L, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ekj != 0L, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ekk != 1L, , i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ekk != 1L, , i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ekl != 9L, , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ekl != 9L, , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ekm != 0L, u < 0L, ),
             bor3(ekm != 0L, u < 0L, ))
expect_equal(or3s(ekn != 1L, z < 1L, ),
             bor3(ekn != 1L, z < 1L, ))
expect_equal(or3s(eko != 9L, y < 9L, ),
             bor3(eko != 9L, y < 9L, ))
expect_equal(or3s(ekp != 0L, a < 0L, o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ekp != 0L, a < 0L, o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ekq != 1L, g < 1L, x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ekq != 1L, g < 1L, x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ekr != 9L, g < 9L, s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ekr != 9L, g < 9L, s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(eks != 0L, , ),
             bor3(eks != 0L, , ))
expect_equal(or3s(ekt != 1L, , ),
             bor3(ekt != 1L, , ))
expect_equal(or3s(eku != 9L, , ),
             bor3(eku != 9L, , ))
expect_equal(or3s(ekv != 0L, , n %in% 1:4),
             bor3(ekv != 0L, , n %in% 1:4))
expect_equal(or3s(ekw != 1L, , a %in% 1:4),
             bor3(ekw != 1L, , a %in% 1:4))
expect_equal(or3s(ekx != 9L, , g %in% 1:4),
             bor3(ekx != 9L, , g %in% 1:4))
expect_equal(or3s(eky != 0L, d < 0L, ),
             bor3(eky != 0L, d < 0L, ))
expect_equal(or3s(ekz != 1L, t < 1L, ),
             bor3(ekz != 1L, t < 1L, ))
expect_equal(or3s(ela != 9L, y < 9L, ),
             bor3(ela != 9L, y < 9L, ))
expect_equal(or3s(elb != 0L, p < 0L, z %in% 1:4),
             bor3(elb != 0L, p < 0L, z %in% 1:4))
expect_equal(or3s(elc != 1L, p < 1L, b %in% 1:4),
             bor3(elc != 1L, p < 1L, b %in% 1:4))
expect_equal(or3s(eld != 9L, n < 9L, e %in% 1:4),
             bor3(eld != 9L, n < 9L, e %in% 1:4))
expect_equal(or3s(ele != 0L, , ),
             bor3(ele != 0L, , ))
expect_equal(or3s(elf != 1L, , ),
             bor3(elf != 1L, , ))
expect_equal(or3s(elg != 9L, , ),
             bor3(elg != 9L, , ))
expect_equal(or3s(elh != 0L, , c < 0L),
             bor3(elh != 0L, , c < 0L))
expect_equal(or3s(eli != 1L, , n < 1L),
             bor3(eli != 1L, , n < 1L))
expect_equal(or3s(elj != 9L, , h < 9L),
             bor3(elj != 9L, , h < 9L))
expect_equal(or3s(elk != 0L, p < 0L, ),
             bor3(elk != 0L, p < 0L, ))
expect_equal(or3s(ell != 1L, w < 1L, ),
             bor3(ell != 1L, w < 1L, ))
expect_equal(or3s(elm != 9L, s < 9L, ),
             bor3(elm != 9L, s < 9L, ))
expect_equal(or3s(eln != 0L, r < 0L, z < 0L),
             bor3(eln != 0L, r < 0L, z < 0L))
expect_equal(or3s(elo != 1L, q < 1L, u < 1L),
             bor3(elo != 1L, q < 1L, u < 1L))
expect_equal(or3s(elp != 9L, q < 9L, p < 9L),
             bor3(elp != 9L, q < 9L, p < 9L))
expect_equal(or3s(elq != 0L, , ),
             bor3(elq != 0L, , ))
expect_equal(or3s(elr != 1L, , ),
             bor3(elr != 1L, , ))
expect_equal(or3s(els != 9L, , ),
             bor3(els != 9L, , ))
expect_equal(or3s(elt != 0L, , z <= 0L),
             bor3(elt != 0L, , z <= 0L))
expect_equal(or3s(elu != 1L, , e <= 1L),
             bor3(elu != 1L, , e <= 1L))
expect_equal(or3s(elv != 9L, , a <= 9L),
             bor3(elv != 9L, , a <= 9L))
expect_equal(or3s(elw != 0L, a < 0L, ),
             bor3(elw != 0L, a < 0L, ))
expect_equal(or3s(elx != 1L, u < 1L, ),
             bor3(elx != 1L, u < 1L, ))
expect_equal(or3s(ely != 9L, x < 9L, ),
             bor3(ely != 9L, x < 9L, ))
expect_equal(or3s(elz != 0L, r < 0L, h <= 0L),
             bor3(elz != 0L, r < 0L, h <= 0L))
expect_equal(or3s(ema != 1L, j < 1L, n <= 1L),
             bor3(ema != 1L, j < 1L, n <= 1L))
expect_equal(or3s(emb != 9L, v < 9L, x <= 9L),
             bor3(emb != 9L, v < 9L, x <= 9L))
expect_equal(or3s(emc != 0L, , ),
             bor3(emc != 0L, , ))
expect_equal(or3s(emd != 1L, , ),
             bor3(emd != 1L, , ))
expect_equal(or3s(eme != 9L, , ),
             bor3(eme != 9L, , ))
expect_equal(or3s(emf != 0L, , f == 0L),
             bor3(emf != 0L, , f == 0L))
expect_equal(or3s(emg != 1L, , x == 1L),
             bor3(emg != 1L, , x == 1L))
expect_equal(or3s(emh != 9L, , u == 9L),
             bor3(emh != 9L, , u == 9L))
expect_equal(or3s(emi != 0L, x < 0L, ),
             bor3(emi != 0L, x < 0L, ))
expect_equal(or3s(emj != 1L, l < 1L, ),
             bor3(emj != 1L, l < 1L, ))
expect_equal(or3s(emk != 9L, r < 9L, ),
             bor3(emk != 9L, r < 9L, ))
expect_equal(or3s(eml != 0L, g < 0L, z == 0L),
             bor3(eml != 0L, g < 0L, z == 0L))
expect_equal(or3s(emm != 1L, j < 1L, n == 1L),
             bor3(emm != 1L, j < 1L, n == 1L))
expect_equal(or3s(emn != 9L, v < 9L, u == 9L),
             bor3(emn != 9L, v < 9L, u == 9L))
expect_equal(or3s(emo != 0L, , ),
             bor3(emo != 0L, , ))
expect_equal(or3s(emp != 1L, , ),
             bor3(emp != 1L, , ))
expect_equal(or3s(emq != 9L, , ),
             bor3(emq != 9L, , ))
expect_equal(or3s(emr != 0L, , t > 0L),
             bor3(emr != 0L, , t > 0L))
expect_equal(or3s(ems != 1L, , v > 1L),
             bor3(ems != 1L, , v > 1L))
expect_equal(or3s(emt != 9L, , e > 9L),
             bor3(emt != 9L, , e > 9L))
expect_equal(or3s(emu != 0L, m < 0L, ),
             bor3(emu != 0L, m < 0L, ))
expect_equal(or3s(emv != 1L, i < 1L, ),
             bor3(emv != 1L, i < 1L, ))
expect_equal(or3s(emw != 9L, w < 9L, ),
             bor3(emw != 9L, w < 9L, ))
expect_equal(or3s(emx != 0L, y < 0L, o > 0L),
             bor3(emx != 0L, y < 0L, o > 0L))
expect_equal(or3s(emy != 1L, x < 1L, d > 1L),
             bor3(emy != 1L, x < 1L, d > 1L))
expect_equal(or3s(emz != 9L, x < 9L, s > 9L),
             bor3(emz != 9L, x < 9L, s > 9L))
expect_equal(or3s(ena != 0L, , ),
             bor3(ena != 0L, , ))
expect_equal(or3s(enb != 1L, , ),
             bor3(enb != 1L, , ))
expect_equal(or3s(enc != 9L, , ),
             bor3(enc != 9L, , ))
expect_equal(or3s(end != 0L, , i >= 0L),
             bor3(end != 0L, , i >= 0L))
expect_equal(or3s(ene != 1L, , e >= 1L),
             bor3(ene != 1L, , e >= 1L))
expect_equal(or3s(enf != 9L, , z >= 9L),
             bor3(enf != 9L, , z >= 9L))
expect_equal(or3s(eng != 0L, q < 0L, ),
             bor3(eng != 0L, q < 0L, ))
expect_equal(or3s(enh != 1L, z < 1L, ),
             bor3(enh != 1L, z < 1L, ))
expect_equal(or3s(eni != 9L, s < 9L, ),
             bor3(eni != 9L, s < 9L, ))
expect_equal(or3s(enj != 0L, w < 0L, v >= 0L),
             bor3(enj != 0L, w < 0L, v >= 0L))
expect_equal(or3s(enk != 1L, j < 1L, t >= 1L),
             bor3(enk != 1L, j < 1L, t >= 1L))
expect_equal(or3s(enl != 9L, d < 9L, i >= 9L),
             bor3(enl != 9L, d < 9L, i >= 9L))
expect_equal(or3s(enm != 0L, , ),
             bor3(enm != 0L, , ))
expect_equal(or3s(enn != 1L, , ),
             bor3(enn != 1L, , ))
expect_equal(or3s(eno != 9L, , ),
             bor3(eno != 9L, , ))
expect_equal(or3s(enp != 0L, , logi_k),
             bor3(enp != 0L, , logi_k))
expect_equal(or3s(enq != 1L, , logi_i),
             bor3(enq != 1L, , logi_i))
expect_equal(or3s(enr != 9L, , logi_i),
             bor3(enr != 9L, , logi_i))
expect_equal(or3s(ens != 0L, e <= 0L, ),
             bor3(ens != 0L, e <= 0L, ))
expect_equal(or3s(ent != 1L, f <= 1L, ),
             bor3(ent != 1L, f <= 1L, ))
expect_equal(or3s(enu != 9L, q <= 9L, ),
             bor3(enu != 9L, q <= 9L, ))
expect_equal(or3s(env != 0L, m <= 0L, logi_a),
             bor3(env != 0L, m <= 0L, logi_a))
expect_equal(or3s(enw != 1L, z <= 1L, logi_c),
             bor3(enw != 1L, z <= 1L, logi_c))
expect_equal(or3s(enx != 9L, k <= 9L, logi_c),
             bor3(enx != 9L, k <= 9L, logi_c))
expect_equal(or3s(eny != 0L, , ),
             bor3(eny != 0L, , ))
expect_equal(or3s(enz != 1L, , ),
             bor3(enz != 1L, , ))
expect_equal(or3s(eoa != 9L, , ),
             bor3(eoa != 9L, , ))
expect_equal(or3s(eob != 0L, , !logi_g),
             bor3(eob != 0L, , !logi_g))
expect_equal(or3s(eoc != 1L, , !logi_e),
             bor3(eoc != 1L, , !logi_e))
expect_equal(or3s(eod != 9L, , !logi_d),
             bor3(eod != 9L, , !logi_d))
expect_equal(or3s(eoe != 0L, l <= 0L, ),
             bor3(eoe != 0L, l <= 0L, ))
expect_equal(or3s(eof != 1L, l <= 1L, ),
             bor3(eof != 1L, l <= 1L, ))
expect_equal(or3s(eog != 9L, a <= 9L, ),
             bor3(eog != 9L, a <= 9L, ))
expect_equal(or3s(eoh != 0L, w <= 0L, !logi_f),
             bor3(eoh != 0L, w <= 0L, !logi_f))
expect_equal(or3s(eoi != 1L, f <= 1L, !logi_n),
             bor3(eoi != 1L, f <= 1L, !logi_n))
expect_equal(or3s(eoj != 9L, x <= 9L, !logi_m),
             bor3(eoj != 9L, x <= 9L, !logi_m))
expect_equal(or3s(eok != 0L, , ),
             bor3(eok != 0L, , ))
expect_equal(or3s(eol != 1L, , ),
             bor3(eol != 1L, , ))
expect_equal(or3s(eom != 9L, , ),
             bor3(eom != 9L, , ))
expect_equal(or3s(eon != 0L, , u != 0L),
             bor3(eon != 0L, , u != 0L))
expect_equal(or3s(eoo != 1L, , z != 1L),
             bor3(eoo != 1L, , z != 1L))
expect_equal(or3s(eop != 9L, , a != 9L),
             bor3(eop != 9L, , a != 9L))
expect_equal(or3s(eoq != 0L, p <= 0L, ),
             bor3(eoq != 0L, p <= 0L, ))
expect_equal(or3s(eor != 1L, a <= 1L, ),
             bor3(eor != 1L, a <= 1L, ))
expect_equal(or3s(eos != 9L, q <= 9L, ),
             bor3(eos != 9L, q <= 9L, ))
expect_equal(or3s(eot != 0L, b <= 0L, m != 0L),
             bor3(eot != 0L, b <= 0L, m != 0L))
expect_equal(or3s(eou != 1L, g <= 1L, b != 1L),
             bor3(eou != 1L, g <= 1L, b != 1L))
expect_equal(or3s(eov != 9L, k <= 9L, u != 9L),
             bor3(eov != 9L, k <= 9L, u != 9L))
expect_equal(or3s(eow != 0L, , ),
             bor3(eow != 0L, , ))
expect_equal(or3s(eox != 1L, , ),
             bor3(eox != 1L, , ))
expect_equal(or3s(eoy != 9L, , ),
             bor3(eoy != 9L, , ))
expect_equal(or3s(eoz != 0L, , u %between% c(-1L, 1L)),
             bor3(eoz != 0L, , u %between% c(-1L, 1L)))
expect_equal(or3s(epa != 1L, , k %between% c(-1L, 1L)),
             bor3(epa != 1L, , k %between% c(-1L, 1L)))
expect_equal(or3s(epb != 9L, , v %between% c(-1L, 1L)),
             bor3(epb != 9L, , v %between% c(-1L, 1L)))
expect_equal(or3s(epc != 0L, e <= 0L, ),
             bor3(epc != 0L, e <= 0L, ))
expect_equal(or3s(epd != 1L, m <= 1L, ),
             bor3(epd != 1L, m <= 1L, ))
expect_equal(or3s(epe != 9L, r <= 9L, ),
             bor3(epe != 9L, r <= 9L, ))
expect_equal(or3s(epf != 0L, o <= 0L, v %between% c(-1L, 1L)),
             bor3(epf != 0L, o <= 0L, v %between% c(-1L, 1L)))
expect_equal(or3s(epg != 1L, d <= 1L, i %between% c(-1L, 1L)),
             bor3(epg != 1L, d <= 1L, i %between% c(-1L, 1L)))
expect_equal(or3s(eph != 9L, n <= 9L, w %between% c(-1L, 1L)),
             bor3(eph != 9L, n <= 9L, w %between% c(-1L, 1L)))
expect_equal(or3s(epi != 0L, , ),
             bor3(epi != 0L, , ))
expect_equal(or3s(epj != 1L, , ),
             bor3(epj != 1L, , ))
expect_equal(or3s(epk != 9L, , ),
             bor3(epk != 9L, , ))
expect_equal(or3s(epl != 0L, , r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(epl != 0L, , r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(epm != 1L, , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(epm != 1L, , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(epn != 9L, , i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(epn != 9L, , i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(epo != 0L, p <= 0L, ),
             bor3(epo != 0L, p <= 0L, ))
expect_equal(or3s(epp != 1L, j <= 1L, ),
             bor3(epp != 1L, j <= 1L, ))
expect_equal(or3s(epq != 9L, j <= 9L, ),
             bor3(epq != 9L, j <= 9L, ))
expect_equal(or3s(epr != 0L, h <= 0L, h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(epr != 0L, h <= 0L, h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(eps != 1L, s <= 1L, d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(eps != 1L, s <= 1L, d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ept != 9L, h <= 9L, y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ept != 9L, h <= 9L, y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(epu != 0L, , ),
             bor3(epu != 0L, , ))
expect_equal(or3s(epv != 1L, , ),
             bor3(epv != 1L, , ))
expect_equal(or3s(epw != 9L, , ),
             bor3(epw != 9L, , ))
expect_equal(or3s(epx != 0L, , s %in% 1:4),
             bor3(epx != 0L, , s %in% 1:4))
expect_equal(or3s(epy != 1L, , l %in% 1:4),
             bor3(epy != 1L, , l %in% 1:4))
expect_equal(or3s(epz != 9L, , l %in% 1:4),
             bor3(epz != 9L, , l %in% 1:4))
expect_equal(or3s(eqa != 0L, t <= 0L, ),
             bor3(eqa != 0L, t <= 0L, ))
expect_equal(or3s(eqb != 1L, t <= 1L, ),
             bor3(eqb != 1L, t <= 1L, ))
expect_equal(or3s(eqc != 9L, o <= 9L, ),
             bor3(eqc != 9L, o <= 9L, ))
expect_equal(or3s(eqd != 0L, z <= 0L, j %in% 1:4),
             bor3(eqd != 0L, z <= 0L, j %in% 1:4))
expect_equal(or3s(eqe != 1L, p <= 1L, w %in% 1:4),
             bor3(eqe != 1L, p <= 1L, w %in% 1:4))
expect_equal(or3s(eqf != 9L, t <= 9L, h %in% 1:4),
             bor3(eqf != 9L, t <= 9L, h %in% 1:4))
expect_equal(or3s(eqg != 0L, , ),
             bor3(eqg != 0L, , ))
expect_equal(or3s(eqh != 1L, , ),
             bor3(eqh != 1L, , ))
expect_equal(or3s(eqi != 9L, , ),
             bor3(eqi != 9L, , ))
expect_equal(or3s(eqj != 0L, , r < 0L),
             bor3(eqj != 0L, , r < 0L))
expect_equal(or3s(eqk != 1L, , s < 1L),
             bor3(eqk != 1L, , s < 1L))
expect_equal(or3s(eql != 9L, , w < 9L),
             bor3(eql != 9L, , w < 9L))
expect_equal(or3s(eqm != 0L, i <= 0L, ),
             bor3(eqm != 0L, i <= 0L, ))
expect_equal(or3s(eqn != 1L, m <= 1L, ),
             bor3(eqn != 1L, m <= 1L, ))
expect_equal(or3s(eqo != 9L, r <= 9L, ),
             bor3(eqo != 9L, r <= 9L, ))
expect_equal(or3s(eqp != 0L, d <= 0L, f < 0L),
             bor3(eqp != 0L, d <= 0L, f < 0L))
expect_equal(or3s(eqq != 1L, f <= 1L, c < 1L),
             bor3(eqq != 1L, f <= 1L, c < 1L))
expect_equal(or3s(eqr != 9L, w <= 9L, a < 9L),
             bor3(eqr != 9L, w <= 9L, a < 9L))
expect_equal(or3s(eqs != 0L, , ),
             bor3(eqs != 0L, , ))
expect_equal(or3s(eqt != 1L, , ),
             bor3(eqt != 1L, , ))
expect_equal(or3s(equ != 9L, , ),
             bor3(equ != 9L, , ))
expect_equal(or3s(eqv != 0L, , f <= 0L),
             bor3(eqv != 0L, , f <= 0L))
expect_equal(or3s(eqw != 1L, , a <= 1L),
             bor3(eqw != 1L, , a <= 1L))
expect_equal(or3s(eqx != 9L, , s <= 9L),
             bor3(eqx != 9L, , s <= 9L))
expect_equal(or3s(eqy != 0L, b <= 0L, ),
             bor3(eqy != 0L, b <= 0L, ))
expect_equal(or3s(eqz != 1L, q <= 1L, ),
             bor3(eqz != 1L, q <= 1L, ))
expect_equal(or3s(era != 9L, a <= 9L, ),
             bor3(era != 9L, a <= 9L, ))
expect_equal(or3s(erb != 0L, l <= 0L, v <= 0L),
             bor3(erb != 0L, l <= 0L, v <= 0L))
expect_equal(or3s(erc != 1L, j <= 1L, z <= 1L),
             bor3(erc != 1L, j <= 1L, z <= 1L))
expect_equal(or3s(erd != 9L, r <= 9L, r <= 9L),
             bor3(erd != 9L, r <= 9L, r <= 9L))
expect_equal(or3s(ere != 0L, , ),
             bor3(ere != 0L, , ))
expect_equal(or3s(erf != 1L, , ),
             bor3(erf != 1L, , ))
expect_equal(or3s(erg != 9L, , ),
             bor3(erg != 9L, , ))
expect_equal(or3s(erh != 0L, , u == 0L),
             bor3(erh != 0L, , u == 0L))
expect_equal(or3s(eri != 1L, , w == 1L),
             bor3(eri != 1L, , w == 1L))
expect_equal(or3s(erj != 9L, , x == 9L),
             bor3(erj != 9L, , x == 9L))
expect_equal(or3s(erk != 0L, m <= 0L, ),
             bor3(erk != 0L, m <= 0L, ))
expect_equal(or3s(erl != 1L, m <= 1L, ),
             bor3(erl != 1L, m <= 1L, ))
expect_equal(or3s(erm != 9L, l <= 9L, ),
             bor3(erm != 9L, l <= 9L, ))
expect_equal(or3s(ern != 0L, b <= 0L, t == 0L),
             bor3(ern != 0L, b <= 0L, t == 0L))
expect_equal(or3s(ero != 1L, x <= 1L, j == 1L),
             bor3(ero != 1L, x <= 1L, j == 1L))
expect_equal(or3s(erp != 9L, y <= 9L, e == 9L),
             bor3(erp != 9L, y <= 9L, e == 9L))
expect_equal(or3s(erq != 0L, , ),
             bor3(erq != 0L, , ))
expect_equal(or3s(err != 1L, , ),
             bor3(err != 1L, , ))
expect_equal(or3s(ers != 9L, , ),
             bor3(ers != 9L, , ))
expect_equal(or3s(ert != 0L, , v > 0L),
             bor3(ert != 0L, , v > 0L))
expect_equal(or3s(eru != 1L, , t > 1L),
             bor3(eru != 1L, , t > 1L))
expect_equal(or3s(erv != 9L, , y > 9L),
             bor3(erv != 9L, , y > 9L))
expect_equal(or3s(erw != 0L, l <= 0L, ),
             bor3(erw != 0L, l <= 0L, ))
expect_equal(or3s(erx != 1L, j <= 1L, ),
             bor3(erx != 1L, j <= 1L, ))
expect_equal(or3s(ery != 9L, h <= 9L, ),
             bor3(ery != 9L, h <= 9L, ))
expect_equal(or3s(erz != 0L, v <= 0L, a > 0L),
             bor3(erz != 0L, v <= 0L, a > 0L))
expect_equal(or3s(esa != 1L, o <= 1L, r > 1L),
             bor3(esa != 1L, o <= 1L, r > 1L))
expect_equal(or3s(esb != 9L, p <= 9L, e > 9L),
             bor3(esb != 9L, p <= 9L, e > 9L))
expect_equal(or3s(esc != 0L, , ),
             bor3(esc != 0L, , ))
expect_equal(or3s(esd != 1L, , ),
             bor3(esd != 1L, , ))
expect_equal(or3s(ese != 9L, , ),
             bor3(ese != 9L, , ))
expect_equal(or3s(esf != 0L, , i >= 0L),
             bor3(esf != 0L, , i >= 0L))
expect_equal(or3s(esg != 1L, , i >= 1L),
             bor3(esg != 1L, , i >= 1L))
expect_equal(or3s(esh != 9L, , p >= 9L),
             bor3(esh != 9L, , p >= 9L))
expect_equal(or3s(esi != 0L, h <= 0L, ),
             bor3(esi != 0L, h <= 0L, ))
expect_equal(or3s(esj != 1L, z <= 1L, ),
             bor3(esj != 1L, z <= 1L, ))
expect_equal(or3s(esk != 9L, v <= 9L, ),
             bor3(esk != 9L, v <= 9L, ))
expect_equal(or3s(esl != 0L, h <= 0L, d >= 0L),
             bor3(esl != 0L, h <= 0L, d >= 0L))
expect_equal(or3s(esm != 1L, l <= 1L, z >= 1L),
             bor3(esm != 1L, l <= 1L, z >= 1L))
expect_equal(or3s(esn != 9L, n <= 9L, q >= 9L),
             bor3(esn != 9L, n <= 9L, q >= 9L))
expect_equal(or3s(eso != 0L, , ),
             bor3(eso != 0L, , ))
expect_equal(or3s(esp != 1L, , ),
             bor3(esp != 1L, , ))
expect_equal(or3s(esq != 9L, , ),
             bor3(esq != 9L, , ))
expect_equal(or3s(esr != 0L, , logi_g),
             bor3(esr != 0L, , logi_g))
expect_equal(or3s(ess != 1L, , logi_i),
             bor3(ess != 1L, , logi_i))
expect_equal(or3s(est != 9L, , logi_k),
             bor3(est != 9L, , logi_k))
expect_equal(or3s(esu != 0L, m == 0L, ),
             bor3(esu != 0L, m == 0L, ))
expect_equal(or3s(esv != 1L, y == 1L, ),
             bor3(esv != 1L, y == 1L, ))
expect_equal(or3s(esw != 9L, l == 9L, ),
             bor3(esw != 9L, l == 9L, ))
expect_equal(or3s(esx != 0L, w == 0L, logi_g),
             bor3(esx != 0L, w == 0L, logi_g))
expect_equal(or3s(esy != 1L, i == 1L, logi_d),
             bor3(esy != 1L, i == 1L, logi_d))
expect_equal(or3s(esz != 9L, z == 9L, logi_o),
             bor3(esz != 9L, z == 9L, logi_o))
expect_equal(or3s(eta != 0L, , ),
             bor3(eta != 0L, , ))
expect_equal(or3s(etb != 1L, , ),
             bor3(etb != 1L, , ))
expect_equal(or3s(etc != 9L, , ),
             bor3(etc != 9L, , ))
expect_equal(or3s(etd != 0L, , !logi_v),
             bor3(etd != 0L, , !logi_v))
expect_equal(or3s(ete != 1L, , !logi_r),
             bor3(ete != 1L, , !logi_r))
expect_equal(or3s(etf != 9L, , !logi_o),
             bor3(etf != 9L, , !logi_o))
expect_equal(or3s(etg != 0L, p == 0L, ),
             bor3(etg != 0L, p == 0L, ))
expect_equal(or3s(eth != 1L, i == 1L, ),
             bor3(eth != 1L, i == 1L, ))
expect_equal(or3s(eti != 9L, q == 9L, ),
             bor3(eti != 9L, q == 9L, ))
expect_equal(or3s(etj != 0L, t == 0L, !logi_m),
             bor3(etj != 0L, t == 0L, !logi_m))
expect_equal(or3s(etk != 1L, n == 1L, !logi_z),
             bor3(etk != 1L, n == 1L, !logi_z))
expect_equal(or3s(etl != 9L, r == 9L, !logi_a),
             bor3(etl != 9L, r == 9L, !logi_a))
expect_equal(or3s(etm != 0L, , ),
             bor3(etm != 0L, , ))
expect_equal(or3s(etn != 1L, , ),
             bor3(etn != 1L, , ))
expect_equal(or3s(eto != 9L, , ),
             bor3(eto != 9L, , ))
expect_equal(or3s(etp != 0L, , d != 0L),
             bor3(etp != 0L, , d != 0L))
expect_equal(or3s(etq != 1L, , e != 1L),
             bor3(etq != 1L, , e != 1L))
expect_equal(or3s(etr != 9L, , v != 9L),
             bor3(etr != 9L, , v != 9L))
expect_equal(or3s(ets != 0L, q == 0L, ),
             bor3(ets != 0L, q == 0L, ))
expect_equal(or3s(ett != 1L, m == 1L, ),
             bor3(ett != 1L, m == 1L, ))
expect_equal(or3s(etu != 9L, t == 9L, ),
             bor3(etu != 9L, t == 9L, ))
expect_equal(or3s(etv != 0L, b == 0L, u != 0L),
             bor3(etv != 0L, b == 0L, u != 0L))
expect_equal(or3s(etw != 1L, n == 1L, g != 1L),
             bor3(etw != 1L, n == 1L, g != 1L))
expect_equal(or3s(etx != 9L, j == 9L, g != 9L),
             bor3(etx != 9L, j == 9L, g != 9L))
expect_equal(or3s(ety != 0L, , ),
             bor3(ety != 0L, , ))
expect_equal(or3s(etz != 1L, , ),
             bor3(etz != 1L, , ))
expect_equal(or3s(eua != 9L, , ),
             bor3(eua != 9L, , ))
expect_equal(or3s(eub != 0L, , n %between% c(-1L, 1L)),
             bor3(eub != 0L, , n %between% c(-1L, 1L)))
expect_equal(or3s(euc != 1L, , b %between% c(-1L, 1L)),
             bor3(euc != 1L, , b %between% c(-1L, 1L)))
expect_equal(or3s(eud != 9L, , w %between% c(-1L, 1L)),
             bor3(eud != 9L, , w %between% c(-1L, 1L)))
expect_equal(or3s(eue != 0L, o == 0L, ),
             bor3(eue != 0L, o == 0L, ))
expect_equal(or3s(euf != 1L, t == 1L, ),
             bor3(euf != 1L, t == 1L, ))
expect_equal(or3s(eug != 9L, o == 9L, ),
             bor3(eug != 9L, o == 9L, ))
expect_equal(or3s(euh != 0L, n == 0L, u %between% c(-1L, 1L)),
             bor3(euh != 0L, n == 0L, u %between% c(-1L, 1L)))
expect_equal(or3s(eui != 1L, b == 1L, b %between% c(-1L, 1L)),
             bor3(eui != 1L, b == 1L, b %between% c(-1L, 1L)))
expect_equal(or3s(euj != 9L, c == 9L, c %between% c(-1L, 1L)),
             bor3(euj != 9L, c == 9L, c %between% c(-1L, 1L)))
expect_equal(or3s(euk != 0L, , ),
             bor3(euk != 0L, , ))
expect_equal(or3s(eul != 1L, , ),
             bor3(eul != 1L, , ))
expect_equal(or3s(eum != 9L, , ),
             bor3(eum != 9L, , ))
expect_equal(or3s(eun != 0L, , i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(eun != 0L, , i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(euo != 1L, , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(euo != 1L, , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(eup != 9L, , d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(eup != 9L, , d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(euq != 0L, y == 0L, ),
             bor3(euq != 0L, y == 0L, ))
expect_equal(or3s(eur != 1L, q == 1L, ),
             bor3(eur != 1L, q == 1L, ))
expect_equal(or3s(eus != 9L, x == 9L, ),
             bor3(eus != 9L, x == 9L, ))
expect_equal(or3s(eut != 0L, k == 0L, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(eut != 0L, k == 0L, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(euu != 1L, g == 1L, o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(euu != 1L, g == 1L, o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(euv != 9L, g == 9L, m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(euv != 9L, g == 9L, m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(euw != 0L, , ),
             bor3(euw != 0L, , ))
expect_equal(or3s(eux != 1L, , ),
             bor3(eux != 1L, , ))
expect_equal(or3s(euy != 9L, , ),
             bor3(euy != 9L, , ))
expect_equal(or3s(euz != 0L, , f %in% 1:4),
             bor3(euz != 0L, , f %in% 1:4))
expect_equal(or3s(eva != 1L, , r %in% 1:4),
             bor3(eva != 1L, , r %in% 1:4))
expect_equal(or3s(evb != 9L, , l %in% 1:4),
             bor3(evb != 9L, , l %in% 1:4))
expect_equal(or3s(evc != 0L, s == 0L, ),
             bor3(evc != 0L, s == 0L, ))
expect_equal(or3s(evd != 1L, m == 1L, ),
             bor3(evd != 1L, m == 1L, ))
expect_equal(or3s(eve != 9L, a == 9L, ),
             bor3(eve != 9L, a == 9L, ))
expect_equal(or3s(evf != 0L, t == 0L, h %in% 1:4),
             bor3(evf != 0L, t == 0L, h %in% 1:4))
expect_equal(or3s(evg != 1L, p == 1L, g %in% 1:4),
             bor3(evg != 1L, p == 1L, g %in% 1:4))
expect_equal(or3s(evh != 9L, i == 9L, b %in% 1:4),
             bor3(evh != 9L, i == 9L, b %in% 1:4))
expect_equal(or3s(evi != 0L, , ),
             bor3(evi != 0L, , ))
expect_equal(or3s(evj != 1L, , ),
             bor3(evj != 1L, , ))
expect_equal(or3s(evk != 9L, , ),
             bor3(evk != 9L, , ))
expect_equal(or3s(evl != 0L, , k < 0L),
             bor3(evl != 0L, , k < 0L))
expect_equal(or3s(evm != 1L, , p < 1L),
             bor3(evm != 1L, , p < 1L))
expect_equal(or3s(evn != 9L, , s < 9L),
             bor3(evn != 9L, , s < 9L))
expect_equal(or3s(evo != 0L, b == 0L, ),
             bor3(evo != 0L, b == 0L, ))
expect_equal(or3s(evp != 1L, o == 1L, ),
             bor3(evp != 1L, o == 1L, ))
expect_equal(or3s(evq != 9L, u == 9L, ),
             bor3(evq != 9L, u == 9L, ))
expect_equal(or3s(evr != 0L, r == 0L, o < 0L),
             bor3(evr != 0L, r == 0L, o < 0L))
expect_equal(or3s(evs != 1L, q == 1L, w < 1L),
             bor3(evs != 1L, q == 1L, w < 1L))
expect_equal(or3s(evt != 9L, q == 9L, p < 9L),
             bor3(evt != 9L, q == 9L, p < 9L))
expect_equal(or3s(evu != 0L, , ),
             bor3(evu != 0L, , ))
expect_equal(or3s(evv != 1L, , ),
             bor3(evv != 1L, , ))
expect_equal(or3s(evw != 9L, , ),
             bor3(evw != 9L, , ))
expect_equal(or3s(evx != 0L, , i <= 0L),
             bor3(evx != 0L, , i <= 0L))
expect_equal(or3s(evy != 1L, , n <= 1L),
             bor3(evy != 1L, , n <= 1L))
expect_equal(or3s(evz != 9L, , v <= 9L),
             bor3(evz != 9L, , v <= 9L))
expect_equal(or3s(ewa != 0L, t == 0L, ),
             bor3(ewa != 0L, t == 0L, ))
expect_equal(or3s(ewb != 1L, p == 1L, ),
             bor3(ewb != 1L, p == 1L, ))
expect_equal(or3s(ewc != 9L, m == 9L, ),
             bor3(ewc != 9L, m == 9L, ))
expect_equal(or3s(ewd != 0L, j == 0L, w <= 0L),
             bor3(ewd != 0L, j == 0L, w <= 0L))
expect_equal(or3s(ewe != 1L, e == 1L, k <= 1L),
             bor3(ewe != 1L, e == 1L, k <= 1L))
expect_equal(or3s(ewf != 9L, u == 9L, e <= 9L),
             bor3(ewf != 9L, u == 9L, e <= 9L))
expect_equal(or3s(ewg != 0L, , ),
             bor3(ewg != 0L, , ))
expect_equal(or3s(ewh != 1L, , ),
             bor3(ewh != 1L, , ))
expect_equal(or3s(ewi != 9L, , ),
             bor3(ewi != 9L, , ))
expect_equal(or3s(ewj != 0L, , y == 0L),
             bor3(ewj != 0L, , y == 0L))
expect_equal(or3s(ewk != 1L, , a == 1L),
             bor3(ewk != 1L, , a == 1L))
expect_equal(or3s(ewl != 9L, , j == 9L),
             bor3(ewl != 9L, , j == 9L))
expect_equal(or3s(ewm != 0L, z == 0L, ),
             bor3(ewm != 0L, z == 0L, ))
expect_equal(or3s(ewn != 1L, n == 1L, ),
             bor3(ewn != 1L, n == 1L, ))
expect_equal(or3s(ewo != 9L, d == 9L, ),
             bor3(ewo != 9L, d == 9L, ))
expect_equal(or3s(ewp != 0L, x == 0L, v == 0L),
             bor3(ewp != 0L, x == 0L, v == 0L))
expect_equal(or3s(ewq != 1L, a == 1L, j == 1L),
             bor3(ewq != 1L, a == 1L, j == 1L))
expect_equal(or3s(ewr != 9L, x == 9L, x == 9L),
             bor3(ewr != 9L, x == 9L, x == 9L))
expect_equal(or3s(ews != 0L, , ),
             bor3(ews != 0L, , ))
expect_equal(or3s(ewt != 1L, , ),
             bor3(ewt != 1L, , ))
expect_equal(or3s(ewu != 9L, , ),
             bor3(ewu != 9L, , ))
expect_equal(or3s(ewv != 0L, , b > 0L),
             bor3(ewv != 0L, , b > 0L))
expect_equal(or3s(eww != 1L, , j > 1L),
             bor3(eww != 1L, , j > 1L))
expect_equal(or3s(ewx != 9L, , j > 9L),
             bor3(ewx != 9L, , j > 9L))
expect_equal(or3s(ewy != 0L, h == 0L, ),
             bor3(ewy != 0L, h == 0L, ))
expect_equal(or3s(ewz != 1L, o == 1L, ),
             bor3(ewz != 1L, o == 1L, ))
expect_equal(or3s(exa != 9L, t == 9L, ),
             bor3(exa != 9L, t == 9L, ))
expect_equal(or3s(exb != 0L, o == 0L, f > 0L),
             bor3(exb != 0L, o == 0L, f > 0L))
expect_equal(or3s(exc != 1L, d == 1L, t > 1L),
             bor3(exc != 1L, d == 1L, t > 1L))
expect_equal(or3s(exd != 9L, q == 9L, l > 9L),
             bor3(exd != 9L, q == 9L, l > 9L))
expect_equal(or3s(exe != 0L, , ),
             bor3(exe != 0L, , ))
expect_equal(or3s(exf != 1L, , ),
             bor3(exf != 1L, , ))
expect_equal(or3s(exg != 9L, , ),
             bor3(exg != 9L, , ))
expect_equal(or3s(exh != 0L, , b >= 0L),
             bor3(exh != 0L, , b >= 0L))
expect_equal(or3s(exi != 1L, , k >= 1L),
             bor3(exi != 1L, , k >= 1L))
expect_equal(or3s(exj != 9L, , e >= 9L),
             bor3(exj != 9L, , e >= 9L))
expect_equal(or3s(exk != 0L, g == 0L, ),
             bor3(exk != 0L, g == 0L, ))
expect_equal(or3s(exl != 1L, a == 1L, ),
             bor3(exl != 1L, a == 1L, ))
expect_equal(or3s(exm != 9L, h == 9L, ),
             bor3(exm != 9L, h == 9L, ))
expect_equal(or3s(exn != 0L, u == 0L, t >= 0L),
             bor3(exn != 0L, u == 0L, t >= 0L))
expect_equal(or3s(exo != 1L, d == 1L, w >= 1L),
             bor3(exo != 1L, d == 1L, w >= 1L))
expect_equal(or3s(exp != 9L, z == 9L, y >= 9L),
             bor3(exp != 9L, z == 9L, y >= 9L))
expect_equal(or3s(exq != 0L, , ),
             bor3(exq != 0L, , ))
expect_equal(or3s(exr != 1L, , ),
             bor3(exr != 1L, , ))
expect_equal(or3s(exs != 9L, , ),
             bor3(exs != 9L, , ))
expect_equal(or3s(ext != 0L, , logi_v),
             bor3(ext != 0L, , logi_v))
expect_equal(or3s(exu != 1L, , logi_a),
             bor3(exu != 1L, , logi_a))
expect_equal(or3s(exv != 9L, , logi_h),
             bor3(exv != 9L, , logi_h))
expect_equal(or3s(exw != 0L, d > 0L, ),
             bor3(exw != 0L, d > 0L, ))
expect_equal(or3s(exx != 1L, m > 1L, ),
             bor3(exx != 1L, m > 1L, ))
expect_equal(or3s(exy != 9L, c > 9L, ),
             bor3(exy != 9L, c > 9L, ))
expect_equal(or3s(exz != 0L, f > 0L, logi_w),
             bor3(exz != 0L, f > 0L, logi_w))
expect_equal(or3s(eya != 1L, d > 1L, logi_x),
             bor3(eya != 1L, d > 1L, logi_x))
expect_equal(or3s(eyb != 9L, h > 9L, logi_i),
             bor3(eyb != 9L, h > 9L, logi_i))
expect_equal(or3s(eyc != 0L, , ),
             bor3(eyc != 0L, , ))
expect_equal(or3s(eyd != 1L, , ),
             bor3(eyd != 1L, , ))
expect_equal(or3s(eye != 9L, , ),
             bor3(eye != 9L, , ))
expect_equal(or3s(eyf != 0L, , !logi_e),
             bor3(eyf != 0L, , !logi_e))
expect_equal(or3s(eyg != 1L, , !logi_a),
             bor3(eyg != 1L, , !logi_a))
expect_equal(or3s(eyh != 9L, , !logi_o),
             bor3(eyh != 9L, , !logi_o))
expect_equal(or3s(eyi != 0L, s > 0L, ),
             bor3(eyi != 0L, s > 0L, ))
expect_equal(or3s(eyj != 1L, n > 1L, ),
             bor3(eyj != 1L, n > 1L, ))
expect_equal(or3s(eyk != 9L, c > 9L, ),
             bor3(eyk != 9L, c > 9L, ))
expect_equal(or3s(eyl != 0L, m > 0L, !logi_v),
             bor3(eyl != 0L, m > 0L, !logi_v))
expect_equal(or3s(eym != 1L, t > 1L, !logi_l),
             bor3(eym != 1L, t > 1L, !logi_l))
expect_equal(or3s(eyn != 9L, m > 9L, !logi_v),
             bor3(eyn != 9L, m > 9L, !logi_v))
expect_equal(or3s(eyo != 0L, , ),
             bor3(eyo != 0L, , ))
expect_equal(or3s(eyp != 1L, , ),
             bor3(eyp != 1L, , ))
expect_equal(or3s(eyq != 9L, , ),
             bor3(eyq != 9L, , ))
expect_equal(or3s(eyr != 0L, , s != 0L),
             bor3(eyr != 0L, , s != 0L))
expect_equal(or3s(eys != 1L, , c != 1L),
             bor3(eys != 1L, , c != 1L))
expect_equal(or3s(eyt != 9L, , a != 9L),
             bor3(eyt != 9L, , a != 9L))
expect_equal(or3s(eyu != 0L, p > 0L, ),
             bor3(eyu != 0L, p > 0L, ))
expect_equal(or3s(eyv != 1L, t > 1L, ),
             bor3(eyv != 1L, t > 1L, ))
expect_equal(or3s(eyw != 9L, b > 9L, ),
             bor3(eyw != 9L, b > 9L, ))
expect_equal(or3s(eyx != 0L, o > 0L, m != 0L),
             bor3(eyx != 0L, o > 0L, m != 0L))
expect_equal(or3s(eyy != 1L, k > 1L, z != 1L),
             bor3(eyy != 1L, k > 1L, z != 1L))
expect_equal(or3s(eyz != 9L, m > 9L, w != 9L),
             bor3(eyz != 9L, m > 9L, w != 9L))
expect_equal(or3s(fa != 0L, , ),
             bor3(fa != 0L, , ))
expect_equal(or3s(fb != 1L, , ),
             bor3(fb != 1L, , ))
expect_equal(or3s(fc != 9L, , ),
             bor3(fc != 9L, , ))
expect_equal(or3s(fd != 0L, , a %between% c(-1L, 1L)),
             bor3(fd != 0L, , a %between% c(-1L, 1L)))
expect_equal(or3s(fe != 1L, , r %between% c(-1L, 1L)),
             bor3(fe != 1L, , r %between% c(-1L, 1L)))
expect_equal(or3s(ff != 9L, , b %between% c(-1L, 1L)),
             bor3(ff != 9L, , b %between% c(-1L, 1L)))
expect_equal(or3s(fg != 0L, z > 0L, ),
             bor3(fg != 0L, z > 0L, ))
expect_equal(or3s(fh != 1L, u > 1L, ),
             bor3(fh != 1L, u > 1L, ))
expect_equal(or3s(fi != 9L, u > 9L, ),
             bor3(fi != 9L, u > 9L, ))
expect_equal(or3s(fj != 0L, v > 0L, b %between% c(-1L, 1L)),
             bor3(fj != 0L, v > 0L, b %between% c(-1L, 1L)))
expect_equal(or3s(fk != 1L, w > 1L, p %between% c(-1L, 1L)),
             bor3(fk != 1L, w > 1L, p %between% c(-1L, 1L)))
expect_equal(or3s(fl != 9L, x > 9L, l %between% c(-1L, 1L)),
             bor3(fl != 9L, x > 9L, l %between% c(-1L, 1L)))
expect_equal(or3s(fm != 0L, , ),
             bor3(fm != 0L, , ))
expect_equal(or3s(fn != 1L, , ),
             bor3(fn != 1L, , ))
expect_equal(or3s(fo != 9L, , ),
             bor3(fo != 9L, , ))
expect_equal(or3s(fp != 0L, , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fp != 0L, , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fq != 1L, , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fq != 1L, , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fr != 9L, , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fr != 9L, , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fs != 0L, t > 0L, ),
             bor3(fs != 0L, t > 0L, ))
expect_equal(or3s(ft != 1L, d > 1L, ),
             bor3(ft != 1L, d > 1L, ))
expect_equal(or3s(fu != 9L, u > 9L, ),
             bor3(fu != 9L, u > 9L, ))
expect_equal(or3s(fv != 0L, w > 0L, f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fv != 0L, w > 0L, f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fw != 1L, u > 1L, m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fw != 1L, u > 1L, m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fx != 9L, o > 9L, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fx != 9L, o > 9L, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fy != 0L, , ),
             bor3(fy != 0L, , ))
expect_equal(or3s(fz != 1L, , ),
             bor3(fz != 1L, , ))
expect_equal(or3s(faa != 9L, , ),
             bor3(faa != 9L, , ))
expect_equal(or3s(fab != 0L, , w %in% 1:4),
             bor3(fab != 0L, , w %in% 1:4))
expect_equal(or3s(fac != 1L, , b %in% 1:4),
             bor3(fac != 1L, , b %in% 1:4))
expect_equal(or3s(fad != 9L, , n %in% 1:4),
             bor3(fad != 9L, , n %in% 1:4))
expect_equal(or3s(fae != 0L, c > 0L, ),
             bor3(fae != 0L, c > 0L, ))
expect_equal(or3s(faf != 1L, y > 1L, ),
             bor3(faf != 1L, y > 1L, ))
expect_equal(or3s(fag != 9L, v > 9L, ),
             bor3(fag != 9L, v > 9L, ))
expect_equal(or3s(fah != 0L, w > 0L, e %in% 1:4),
             bor3(fah != 0L, w > 0L, e %in% 1:4))
expect_equal(or3s(fai != 1L, r > 1L, h %in% 1:4),
             bor3(fai != 1L, r > 1L, h %in% 1:4))
expect_equal(or3s(faj != 9L, u > 9L, r %in% 1:4),
             bor3(faj != 9L, u > 9L, r %in% 1:4))
expect_equal(or3s(fak != 0L, , ),
             bor3(fak != 0L, , ))
expect_equal(or3s(fal != 1L, , ),
             bor3(fal != 1L, , ))
expect_equal(or3s(fam != 9L, , ),
             bor3(fam != 9L, , ))
expect_equal(or3s(fan != 0L, , e < 0L),
             bor3(fan != 0L, , e < 0L))
expect_equal(or3s(fao != 1L, , p < 1L),
             bor3(fao != 1L, , p < 1L))
expect_equal(or3s(fap != 9L, , c < 9L),
             bor3(fap != 9L, , c < 9L))
expect_equal(or3s(faq != 0L, h > 0L, ),
             bor3(faq != 0L, h > 0L, ))
expect_equal(or3s(far != 1L, c > 1L, ),
             bor3(far != 1L, c > 1L, ))
expect_equal(or3s(fas != 9L, c > 9L, ),
             bor3(fas != 9L, c > 9L, ))
expect_equal(or3s(fat != 0L, c > 0L, h < 0L),
             bor3(fat != 0L, c > 0L, h < 0L))
expect_equal(or3s(fau != 1L, u > 1L, g < 1L),
             bor3(fau != 1L, u > 1L, g < 1L))
expect_equal(or3s(fav != 9L, l > 9L, y < 9L),
             bor3(fav != 9L, l > 9L, y < 9L))
expect_equal(or3s(faw != 0L, , ),
             bor3(faw != 0L, , ))
expect_equal(or3s(fax != 1L, , ),
             bor3(fax != 1L, , ))
expect_equal(or3s(fay != 9L, , ),
             bor3(fay != 9L, , ))
expect_equal(or3s(faz != 0L, , i <= 0L),
             bor3(faz != 0L, , i <= 0L))
expect_equal(or3s(fba != 1L, , w <= 1L),
             bor3(fba != 1L, , w <= 1L))
expect_equal(or3s(fbb != 9L, , e <= 9L),
             bor3(fbb != 9L, , e <= 9L))
expect_equal(or3s(fbc != 0L, x > 0L, ),
             bor3(fbc != 0L, x > 0L, ))
expect_equal(or3s(fbd != 1L, q > 1L, ),
             bor3(fbd != 1L, q > 1L, ))
expect_equal(or3s(fbe != 9L, x > 9L, ),
             bor3(fbe != 9L, x > 9L, ))
expect_equal(or3s(fbf != 0L, x > 0L, e <= 0L),
             bor3(fbf != 0L, x > 0L, e <= 0L))
expect_equal(or3s(fbg != 1L, n > 1L, q <= 1L),
             bor3(fbg != 1L, n > 1L, q <= 1L))
expect_equal(or3s(fbh != 9L, l > 9L, i <= 9L),
             bor3(fbh != 9L, l > 9L, i <= 9L))
expect_equal(or3s(fbi != 0L, , ),
             bor3(fbi != 0L, , ))
expect_equal(or3s(fbj != 1L, , ),
             bor3(fbj != 1L, , ))
expect_equal(or3s(fbk != 9L, , ),
             bor3(fbk != 9L, , ))
expect_equal(or3s(fbl != 0L, , g == 0L),
             bor3(fbl != 0L, , g == 0L))
expect_equal(or3s(fbm != 1L, , y == 1L),
             bor3(fbm != 1L, , y == 1L))
expect_equal(or3s(fbn != 9L, , n == 9L),
             bor3(fbn != 9L, , n == 9L))
expect_equal(or3s(fbo != 0L, b > 0L, ),
             bor3(fbo != 0L, b > 0L, ))
expect_equal(or3s(fbp != 1L, k > 1L, ),
             bor3(fbp != 1L, k > 1L, ))
expect_equal(or3s(fbq != 9L, z > 9L, ),
             bor3(fbq != 9L, z > 9L, ))
expect_equal(or3s(fbr != 0L, l > 0L, y == 0L),
             bor3(fbr != 0L, l > 0L, y == 0L))
expect_equal(or3s(fbs != 1L, m > 1L, b == 1L),
             bor3(fbs != 1L, m > 1L, b == 1L))
expect_equal(or3s(fbt != 9L, s > 9L, m == 9L),
             bor3(fbt != 9L, s > 9L, m == 9L))
expect_equal(or3s(fbu != 0L, , ),
             bor3(fbu != 0L, , ))
expect_equal(or3s(fbv != 1L, , ),
             bor3(fbv != 1L, , ))
expect_equal(or3s(fbw != 9L, , ),
             bor3(fbw != 9L, , ))
expect_equal(or3s(fbx != 0L, , g > 0L),
             bor3(fbx != 0L, , g > 0L))
expect_equal(or3s(fby != 1L, , t > 1L),
             bor3(fby != 1L, , t > 1L))
expect_equal(or3s(fbz != 9L, , c > 9L),
             bor3(fbz != 9L, , c > 9L))
expect_equal(or3s(fca != 0L, x > 0L, ),
             bor3(fca != 0L, x > 0L, ))
expect_equal(or3s(fcb != 1L, i > 1L, ),
             bor3(fcb != 1L, i > 1L, ))
expect_equal(or3s(fcc != 9L, d > 9L, ),
             bor3(fcc != 9L, d > 9L, ))
expect_equal(or3s(fcd != 0L, j > 0L, y > 0L),
             bor3(fcd != 0L, j > 0L, y > 0L))
expect_equal(or3s(fce != 1L, n > 1L, q > 1L),
             bor3(fce != 1L, n > 1L, q > 1L))
expect_equal(or3s(fcf != 9L, q > 9L, x > 9L),
             bor3(fcf != 9L, q > 9L, x > 9L))
expect_equal(or3s(fcg != 0L, , ),
             bor3(fcg != 0L, , ))
expect_equal(or3s(fch != 1L, , ),
             bor3(fch != 1L, , ))
expect_equal(or3s(fci != 9L, , ),
             bor3(fci != 9L, , ))
expect_equal(or3s(fcj != 0L, , u >= 0L),
             bor3(fcj != 0L, , u >= 0L))
expect_equal(or3s(fck != 1L, , z >= 1L),
             bor3(fck != 1L, , z >= 1L))
expect_equal(or3s(fcl != 9L, , w >= 9L),
             bor3(fcl != 9L, , w >= 9L))
expect_equal(or3s(fcm != 0L, w > 0L, ),
             bor3(fcm != 0L, w > 0L, ))
expect_equal(or3s(fcn != 1L, p > 1L, ),
             bor3(fcn != 1L, p > 1L, ))
expect_equal(or3s(fco != 9L, d > 9L, ),
             bor3(fco != 9L, d > 9L, ))
expect_equal(or3s(fcp != 0L, i > 0L, h >= 0L),
             bor3(fcp != 0L, i > 0L, h >= 0L))
expect_equal(or3s(fcq != 1L, s > 1L, w >= 1L),
             bor3(fcq != 1L, s > 1L, w >= 1L))
expect_equal(or3s(fcr != 9L, a > 9L, z >= 9L),
             bor3(fcr != 9L, a > 9L, z >= 9L))
expect_equal(or3s(fcs != 0L, , ),
             bor3(fcs != 0L, , ))
expect_equal(or3s(fct != 1L, , ),
             bor3(fct != 1L, , ))
expect_equal(or3s(fcu != 9L, , ),
             bor3(fcu != 9L, , ))
expect_equal(or3s(fcv != 0L, , logi_m),
             bor3(fcv != 0L, , logi_m))
expect_equal(or3s(fcw != 1L, , logi_u),
             bor3(fcw != 1L, , logi_u))
expect_equal(or3s(fcx != 9L, , logi_f),
             bor3(fcx != 9L, , logi_f))
expect_equal(or3s(fcy != 0L, a >= 0L, ),
             bor3(fcy != 0L, a >= 0L, ))
expect_equal(or3s(fcz != 1L, t >= 1L, ),
             bor3(fcz != 1L, t >= 1L, ))
expect_equal(or3s(fda != 9L, c >= 9L, ),
             bor3(fda != 9L, c >= 9L, ))
expect_equal(or3s(fdb != 0L, a >= 0L, logi_v),
             bor3(fdb != 0L, a >= 0L, logi_v))
expect_equal(or3s(fdc != 1L, d >= 1L, logi_q),
             bor3(fdc != 1L, d >= 1L, logi_q))
expect_equal(or3s(fdd != 9L, v >= 9L, logi_y),
             bor3(fdd != 9L, v >= 9L, logi_y))
expect_equal(or3s(fde != 0L, , ),
             bor3(fde != 0L, , ))
expect_equal(or3s(fdf != 1L, , ),
             bor3(fdf != 1L, , ))
expect_equal(or3s(fdg != 9L, , ),
             bor3(fdg != 9L, , ))
expect_equal(or3s(fdh != 0L, , !logi_x),
             bor3(fdh != 0L, , !logi_x))
expect_equal(or3s(fdi != 1L, , !logi_e),
             bor3(fdi != 1L, , !logi_e))
expect_equal(or3s(fdj != 9L, , !logi_j),
             bor3(fdj != 9L, , !logi_j))
expect_equal(or3s(fdk != 0L, i >= 0L, ),
             bor3(fdk != 0L, i >= 0L, ))
expect_equal(or3s(fdl != 1L, o >= 1L, ),
             bor3(fdl != 1L, o >= 1L, ))
expect_equal(or3s(fdm != 9L, q >= 9L, ),
             bor3(fdm != 9L, q >= 9L, ))
expect_equal(or3s(fdn != 0L, c >= 0L, !logi_a),
             bor3(fdn != 0L, c >= 0L, !logi_a))
expect_equal(or3s(fdo != 1L, u >= 1L, !logi_s),
             bor3(fdo != 1L, u >= 1L, !logi_s))
expect_equal(or3s(fdp != 9L, d >= 9L, !logi_i),
             bor3(fdp != 9L, d >= 9L, !logi_i))
expect_equal(or3s(fdq != 0L, , ),
             bor3(fdq != 0L, , ))
expect_equal(or3s(fdr != 1L, , ),
             bor3(fdr != 1L, , ))
expect_equal(or3s(fds != 9L, , ),
             bor3(fds != 9L, , ))
expect_equal(or3s(fdt != 0L, , n != 0L),
             bor3(fdt != 0L, , n != 0L))
expect_equal(or3s(fdu != 1L, , v != 1L),
             bor3(fdu != 1L, , v != 1L))
expect_equal(or3s(fdv != 9L, , p != 9L),
             bor3(fdv != 9L, , p != 9L))
expect_equal(or3s(fdw != 0L, h >= 0L, ),
             bor3(fdw != 0L, h >= 0L, ))
expect_equal(or3s(fdx != 1L, x >= 1L, ),
             bor3(fdx != 1L, x >= 1L, ))
expect_equal(or3s(fdy != 9L, t >= 9L, ),
             bor3(fdy != 9L, t >= 9L, ))
expect_equal(or3s(fdz != 0L, k >= 0L, h != 0L),
             bor3(fdz != 0L, k >= 0L, h != 0L))
expect_equal(or3s(fea != 1L, b >= 1L, z != 1L),
             bor3(fea != 1L, b >= 1L, z != 1L))
expect_equal(or3s(feb != 9L, k >= 9L, t != 9L),
             bor3(feb != 9L, k >= 9L, t != 9L))
expect_equal(or3s(fec != 0L, , ),
             bor3(fec != 0L, , ))
expect_equal(or3s(fed != 1L, , ),
             bor3(fed != 1L, , ))
expect_equal(or3s(fee != 9L, , ),
             bor3(fee != 9L, , ))
expect_equal(or3s(fef != 0L, , e %between% c(-1L, 1L)),
             bor3(fef != 0L, , e %between% c(-1L, 1L)))
expect_equal(or3s(feg != 1L, , s %between% c(-1L, 1L)),
             bor3(feg != 1L, , s %between% c(-1L, 1L)))
expect_equal(or3s(feh != 9L, , a %between% c(-1L, 1L)),
             bor3(feh != 9L, , a %between% c(-1L, 1L)))
expect_equal(or3s(fei != 0L, a >= 0L, ),
             bor3(fei != 0L, a >= 0L, ))
expect_equal(or3s(fej != 1L, z >= 1L, ),
             bor3(fej != 1L, z >= 1L, ))
expect_equal(or3s(fek != 9L, u >= 9L, ),
             bor3(fek != 9L, u >= 9L, ))
expect_equal(or3s(fel != 0L, z >= 0L, i %between% c(-1L, 1L)),
             bor3(fel != 0L, z >= 0L, i %between% c(-1L, 1L)))
expect_equal(or3s(fem != 1L, m >= 1L, i %between% c(-1L, 1L)),
             bor3(fem != 1L, m >= 1L, i %between% c(-1L, 1L)))
expect_equal(or3s(fen != 9L, d >= 9L, f %between% c(-1L, 1L)),
             bor3(fen != 9L, d >= 9L, f %between% c(-1L, 1L)))
expect_equal(or3s(feo != 0L, , ),
             bor3(feo != 0L, , ))
expect_equal(or3s(fep != 1L, , ),
             bor3(fep != 1L, , ))
expect_equal(or3s(feq != 9L, , ),
             bor3(feq != 9L, , ))
expect_equal(or3s(fer != 0L, , e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fer != 0L, , e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fes != 1L, , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fes != 1L, , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fet != 9L, , x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fet != 9L, , x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(feu != 0L, q >= 0L, ),
             bor3(feu != 0L, q >= 0L, ))
expect_equal(or3s(fev != 1L, k >= 1L, ),
             bor3(fev != 1L, k >= 1L, ))
expect_equal(or3s(few != 9L, z >= 9L, ),
             bor3(few != 9L, z >= 9L, ))
expect_equal(or3s(fex != 0L, m >= 0L, l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fex != 0L, m >= 0L, l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fey != 1L, q >= 1L, a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fey != 1L, q >= 1L, a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fez != 9L, u >= 9L, l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fez != 9L, u >= 9L, l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ffa != 0L, , ),
             bor3(ffa != 0L, , ))
expect_equal(or3s(ffb != 1L, , ),
             bor3(ffb != 1L, , ))
expect_equal(or3s(ffc != 9L, , ),
             bor3(ffc != 9L, , ))
expect_equal(or3s(ffd != 0L, , n %in% 1:4),
             bor3(ffd != 0L, , n %in% 1:4))
expect_equal(or3s(ffe != 1L, , l %in% 1:4),
             bor3(ffe != 1L, , l %in% 1:4))
expect_equal(or3s(fff != 9L, , w %in% 1:4),
             bor3(fff != 9L, , w %in% 1:4))
expect_equal(or3s(ffg != 0L, m >= 0L, ),
             bor3(ffg != 0L, m >= 0L, ))
expect_equal(or3s(ffh != 1L, t >= 1L, ),
             bor3(ffh != 1L, t >= 1L, ))
expect_equal(or3s(ffi != 9L, m >= 9L, ),
             bor3(ffi != 9L, m >= 9L, ))
expect_equal(or3s(ffj != 0L, v >= 0L, x %in% 1:4),
             bor3(ffj != 0L, v >= 0L, x %in% 1:4))
expect_equal(or3s(ffk != 1L, w >= 1L, x %in% 1:4),
             bor3(ffk != 1L, w >= 1L, x %in% 1:4))
expect_equal(or3s(ffl != 9L, n >= 9L, h %in% 1:4),
             bor3(ffl != 9L, n >= 9L, h %in% 1:4))
expect_equal(or3s(ffm != 0L, , ),
             bor3(ffm != 0L, , ))
expect_equal(or3s(ffn != 1L, , ),
             bor3(ffn != 1L, , ))
expect_equal(or3s(ffo != 9L, , ),
             bor3(ffo != 9L, , ))
expect_equal(or3s(ffp != 0L, , c < 0L),
             bor3(ffp != 0L, , c < 0L))
expect_equal(or3s(ffq != 1L, , l < 1L),
             bor3(ffq != 1L, , l < 1L))
expect_equal(or3s(ffr != 9L, , m < 9L),
             bor3(ffr != 9L, , m < 9L))
expect_equal(or3s(ffs != 0L, l >= 0L, ),
             bor3(ffs != 0L, l >= 0L, ))
expect_equal(or3s(fft != 1L, v >= 1L, ),
             bor3(fft != 1L, v >= 1L, ))
expect_equal(or3s(ffu != 9L, x >= 9L, ),
             bor3(ffu != 9L, x >= 9L, ))
expect_equal(or3s(ffv != 0L, v >= 0L, h < 0L),
             bor3(ffv != 0L, v >= 0L, h < 0L))
expect_equal(or3s(ffw != 1L, h >= 1L, d < 1L),
             bor3(ffw != 1L, h >= 1L, d < 1L))
expect_equal(or3s(ffx != 9L, w >= 9L, o < 9L),
             bor3(ffx != 9L, w >= 9L, o < 9L))
expect_equal(or3s(ffy != 0L, , ),
             bor3(ffy != 0L, , ))
expect_equal(or3s(ffz != 1L, , ),
             bor3(ffz != 1L, , ))
expect_equal(or3s(fga != 9L, , ),
             bor3(fga != 9L, , ))
expect_equal(or3s(fgb != 0L, , d <= 0L),
             bor3(fgb != 0L, , d <= 0L))
expect_equal(or3s(fgc != 1L, , x <= 1L),
             bor3(fgc != 1L, , x <= 1L))
expect_equal(or3s(fgd != 9L, , l <= 9L),
             bor3(fgd != 9L, , l <= 9L))
expect_equal(or3s(fge != 0L, w >= 0L, ),
             bor3(fge != 0L, w >= 0L, ))
expect_equal(or3s(fgf != 1L, k >= 1L, ),
             bor3(fgf != 1L, k >= 1L, ))
expect_equal(or3s(fgg != 9L, v >= 9L, ),
             bor3(fgg != 9L, v >= 9L, ))
expect_equal(or3s(fgh != 0L, y >= 0L, o <= 0L),
             bor3(fgh != 0L, y >= 0L, o <= 0L))
expect_equal(or3s(fgi != 1L, t >= 1L, z <= 1L),
             bor3(fgi != 1L, t >= 1L, z <= 1L))
expect_equal(or3s(fgj != 9L, p >= 9L, y <= 9L),
             bor3(fgj != 9L, p >= 9L, y <= 9L))
expect_equal(or3s(fgk != 0L, , ),
             bor3(fgk != 0L, , ))
expect_equal(or3s(fgl != 1L, , ),
             bor3(fgl != 1L, , ))
expect_equal(or3s(fgm != 9L, , ),
             bor3(fgm != 9L, , ))
expect_equal(or3s(fgn != 0L, , w == 0L),
             bor3(fgn != 0L, , w == 0L))
expect_equal(or3s(fgo != 1L, , f == 1L),
             bor3(fgo != 1L, , f == 1L))
expect_equal(or3s(fgp != 9L, , c == 9L),
             bor3(fgp != 9L, , c == 9L))
expect_equal(or3s(fgq != 0L, n >= 0L, ),
             bor3(fgq != 0L, n >= 0L, ))
expect_equal(or3s(fgr != 1L, v >= 1L, ),
             bor3(fgr != 1L, v >= 1L, ))
expect_equal(or3s(fgs != 9L, y >= 9L, ),
             bor3(fgs != 9L, y >= 9L, ))
expect_equal(or3s(fgt != 0L, k >= 0L, n == 0L),
             bor3(fgt != 0L, k >= 0L, n == 0L))
expect_equal(or3s(fgu != 1L, u >= 1L, e == 1L),
             bor3(fgu != 1L, u >= 1L, e == 1L))
expect_equal(or3s(fgv != 9L, d >= 9L, n == 9L),
             bor3(fgv != 9L, d >= 9L, n == 9L))
expect_equal(or3s(fgw != 0L, , ),
             bor3(fgw != 0L, , ))
expect_equal(or3s(fgx != 1L, , ),
             bor3(fgx != 1L, , ))
expect_equal(or3s(fgy != 9L, , ),
             bor3(fgy != 9L, , ))
expect_equal(or3s(fgz != 0L, , v > 0L),
             bor3(fgz != 0L, , v > 0L))
expect_equal(or3s(fha != 1L, , m > 1L),
             bor3(fha != 1L, , m > 1L))
expect_equal(or3s(fhb != 9L, , b > 9L),
             bor3(fhb != 9L, , b > 9L))
expect_equal(or3s(fhc != 0L, y >= 0L, ),
             bor3(fhc != 0L, y >= 0L, ))
expect_equal(or3s(fhd != 1L, k >= 1L, ),
             bor3(fhd != 1L, k >= 1L, ))
expect_equal(or3s(fhe != 9L, t >= 9L, ),
             bor3(fhe != 9L, t >= 9L, ))
expect_equal(or3s(fhf != 0L, n >= 0L, t > 0L),
             bor3(fhf != 0L, n >= 0L, t > 0L))
expect_equal(or3s(fhg != 1L, g >= 1L, s > 1L),
             bor3(fhg != 1L, g >= 1L, s > 1L))
expect_equal(or3s(fhh != 9L, f >= 9L, i > 9L),
             bor3(fhh != 9L, f >= 9L, i > 9L))
expect_equal(or3s(fhi != 0L, , ),
             bor3(fhi != 0L, , ))
expect_equal(or3s(fhj != 1L, , ),
             bor3(fhj != 1L, , ))
expect_equal(or3s(fhk != 9L, , ),
             bor3(fhk != 9L, , ))
expect_equal(or3s(fhl != 0L, , h >= 0L),
             bor3(fhl != 0L, , h >= 0L))
expect_equal(or3s(fhm != 1L, , n >= 1L),
             bor3(fhm != 1L, , n >= 1L))
expect_equal(or3s(fhn != 9L, , n >= 9L),
             bor3(fhn != 9L, , n >= 9L))
expect_equal(or3s(fho != 0L, p >= 0L, ),
             bor3(fho != 0L, p >= 0L, ))
expect_equal(or3s(fhp != 1L, b >= 1L, ),
             bor3(fhp != 1L, b >= 1L, ))
expect_equal(or3s(fhq != 9L, b >= 9L, ),
             bor3(fhq != 9L, b >= 9L, ))
expect_equal(or3s(fhr != 0L, e >= 0L, c >= 0L),
             bor3(fhr != 0L, e >= 0L, c >= 0L))
expect_equal(or3s(fhs != 1L, t >= 1L, k >= 1L),
             bor3(fhs != 1L, t >= 1L, k >= 1L))
expect_equal(or3s(fht != 9L, t >= 9L, g >= 9L),
             bor3(fht != 9L, t >= 9L, g >= 9L))
expect_equal(or3s(fhu %between% c(-1L, 1L), , ),
             bor3(fhu %between% c(-1L, 1L), , ))
expect_equal(or3s(fhv %between% c(-1L, 1L), , ),
             bor3(fhv %between% c(-1L, 1L), , ))
expect_equal(or3s(fhw %between% c(-1L, 1L), , ),
             bor3(fhw %between% c(-1L, 1L), , ))
expect_equal(or3s(fhx %between% c(-1L, 1L), , logi_m),
             bor3(fhx %between% c(-1L, 1L), , logi_m))
expect_equal(or3s(fhy %between% c(-1L, 1L), , logi_e),
             bor3(fhy %between% c(-1L, 1L), , logi_e))
expect_equal(or3s(fhz %between% c(-1L, 1L), , logi_y),
             bor3(fhz %between% c(-1L, 1L), , logi_y))
expect_equal(or3s(fia %between% c(-1L, 1L), logi_k, ),
             bor3(fia %between% c(-1L, 1L), logi_k, ))
expect_equal(or3s(fib %between% c(-1L, 1L), logi_e, ),
             bor3(fib %between% c(-1L, 1L), logi_e, ))
expect_equal(or3s(fic %between% c(-1L, 1L), logi_m, ),
             bor3(fic %between% c(-1L, 1L), logi_m, ))
expect_equal(or3s(fid %between% c(-1L, 1L), logi_n, logi_f),
             bor3(fid %between% c(-1L, 1L), logi_n, logi_f))
expect_equal(or3s(fie %between% c(-1L, 1L), logi_e, logi_k),
             bor3(fie %between% c(-1L, 1L), logi_e, logi_k))
expect_equal(or3s(fif %between% c(-1L, 1L), logi_n, logi_m),
             bor3(fif %between% c(-1L, 1L), logi_n, logi_m))
expect_equal(or3s(fig %between% c(-1L, 1L), , ),
             bor3(fig %between% c(-1L, 1L), , ))
expect_equal(or3s(fih %between% c(-1L, 1L), , ),
             bor3(fih %between% c(-1L, 1L), , ))
expect_equal(or3s(fii %between% c(-1L, 1L), , ),
             bor3(fii %between% c(-1L, 1L), , ))
expect_equal(or3s(fij %between% c(-1L, 1L), , !logi_c),
             bor3(fij %between% c(-1L, 1L), , !logi_c))
expect_equal(or3s(fik %between% c(-1L, 1L), , !logi_i),
             bor3(fik %between% c(-1L, 1L), , !logi_i))
expect_equal(or3s(fil %between% c(-1L, 1L), , !logi_n),
             bor3(fil %between% c(-1L, 1L), , !logi_n))
expect_equal(or3s(fim %between% c(-1L, 1L), logi_n, ),
             bor3(fim %between% c(-1L, 1L), logi_n, ))
expect_equal(or3s(fin %between% c(-1L, 1L), logi_g, ),
             bor3(fin %between% c(-1L, 1L), logi_g, ))
expect_equal(or3s(fio %between% c(-1L, 1L), logi_r, ),
             bor3(fio %between% c(-1L, 1L), logi_r, ))
expect_equal(or3s(fip %between% c(-1L, 1L), logi_a, !logi_n),
             bor3(fip %between% c(-1L, 1L), logi_a, !logi_n))
expect_equal(or3s(fiq %between% c(-1L, 1L), logi_g, !logi_a),
             bor3(fiq %between% c(-1L, 1L), logi_g, !logi_a))
expect_equal(or3s(fir %between% c(-1L, 1L), logi_t, !logi_h),
             bor3(fir %between% c(-1L, 1L), logi_t, !logi_h))
expect_equal(or3s(fis %between% c(-1L, 1L), , ),
             bor3(fis %between% c(-1L, 1L), , ))
expect_equal(or3s(fit %between% c(-1L, 1L), , ),
             bor3(fit %between% c(-1L, 1L), , ))
expect_equal(or3s(fiu %between% c(-1L, 1L), , ),
             bor3(fiu %between% c(-1L, 1L), , ))
expect_equal(or3s(fiv %between% c(-1L, 1L), , w != 0L),
             bor3(fiv %between% c(-1L, 1L), , w != 0L))
expect_equal(or3s(fiw %between% c(-1L, 1L), , t != 1L),
             bor3(fiw %between% c(-1L, 1L), , t != 1L))
expect_equal(or3s(fix %between% c(-1L, 1L), , u != 9L),
             bor3(fix %between% c(-1L, 1L), , u != 9L))
expect_equal(or3s(fiy %between% c(-1L, 1L), logi_y, ),
             bor3(fiy %between% c(-1L, 1L), logi_y, ))
expect_equal(or3s(fiz %between% c(-1L, 1L), logi_q, ),
             bor3(fiz %between% c(-1L, 1L), logi_q, ))
expect_equal(or3s(fja %between% c(-1L, 1L), logi_a, ),
             bor3(fja %between% c(-1L, 1L), logi_a, ))
expect_equal(or3s(fjb %between% c(-1L, 1L), logi_l, u != 0L),
             bor3(fjb %between% c(-1L, 1L), logi_l, u != 0L))
expect_equal(or3s(fjc %between% c(-1L, 1L), logi_b, g != 1L),
             bor3(fjc %between% c(-1L, 1L), logi_b, g != 1L))
expect_equal(or3s(fjd %between% c(-1L, 1L), logi_e, v != 9L),
             bor3(fjd %between% c(-1L, 1L), logi_e, v != 9L))
expect_equal(or3s(fje %between% c(-1L, 1L), , ),
             bor3(fje %between% c(-1L, 1L), , ))
expect_equal(or3s(fjf %between% c(-1L, 1L), , ),
             bor3(fjf %between% c(-1L, 1L), , ))
expect_equal(or3s(fjg %between% c(-1L, 1L), , ),
             bor3(fjg %between% c(-1L, 1L), , ))
expect_equal(or3s(fjh %between% c(-1L, 1L), , x %between% c(-1L, 1L)),
             bor3(fjh %between% c(-1L, 1L), , x %between% c(-1L, 1L)))
expect_equal(or3s(fji %between% c(-1L, 1L), , c %between% c(-1L, 1L)),
             bor3(fji %between% c(-1L, 1L), , c %between% c(-1L, 1L)))
expect_equal(or3s(fjj %between% c(-1L, 1L), , u %between% c(-1L, 1L)),
             bor3(fjj %between% c(-1L, 1L), , u %between% c(-1L, 1L)))
expect_equal(or3s(fjk %between% c(-1L, 1L), logi_t, ),
             bor3(fjk %between% c(-1L, 1L), logi_t, ))
expect_equal(or3s(fjl %between% c(-1L, 1L), logi_e, ),
             bor3(fjl %between% c(-1L, 1L), logi_e, ))
expect_equal(or3s(fjm %between% c(-1L, 1L), logi_r, ),
             bor3(fjm %between% c(-1L, 1L), logi_r, ))
expect_equal(or3s(fjn %between% c(-1L, 1L), logi_e, h %between% c(-1L, 1L)),
             bor3(fjn %between% c(-1L, 1L), logi_e, h %between% c(-1L, 1L)))
expect_equal(or3s(fjo %between% c(-1L, 1L), logi_m, y %between% c(-1L, 1L)),
             bor3(fjo %between% c(-1L, 1L), logi_m, y %between% c(-1L, 1L)))
expect_equal(or3s(fjp %between% c(-1L, 1L), logi_g, a %between% c(-1L, 1L)),
             bor3(fjp %between% c(-1L, 1L), logi_g, a %between% c(-1L, 1L)))
expect_equal(or3s(fjq %between% c(-1L, 1L), , ),
             bor3(fjq %between% c(-1L, 1L), , ))
expect_equal(or3s(fjr %between% c(-1L, 1L), , ),
             bor3(fjr %between% c(-1L, 1L), , ))
expect_equal(or3s(fjs %between% c(-1L, 1L), , ),
             bor3(fjs %between% c(-1L, 1L), , ))
expect_equal(or3s(fjt %between% c(-1L, 1L), , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fjt %between% c(-1L, 1L), , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fju %between% c(-1L, 1L), , y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fju %between% c(-1L, 1L), , y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fjv %between% c(-1L, 1L), , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fjv %between% c(-1L, 1L), , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fjw %between% c(-1L, 1L), logi_k, ),
             bor3(fjw %between% c(-1L, 1L), logi_k, ))
expect_equal(or3s(fjx %between% c(-1L, 1L), logi_d, ),
             bor3(fjx %between% c(-1L, 1L), logi_d, ))
expect_equal(or3s(fjy %between% c(-1L, 1L), logi_x, ),
             bor3(fjy %between% c(-1L, 1L), logi_x, ))
expect_equal(or3s(fjz %between% c(-1L, 1L), logi_c, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fjz %between% c(-1L, 1L), logi_c, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fka %between% c(-1L, 1L), logi_l, v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fka %between% c(-1L, 1L), logi_l, v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fkb %between% c(-1L, 1L), logi_q, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fkb %between% c(-1L, 1L), logi_q, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fkc %between% c(-1L, 1L), , ),
             bor3(fkc %between% c(-1L, 1L), , ))
expect_equal(or3s(fkd %between% c(-1L, 1L), , ),
             bor3(fkd %between% c(-1L, 1L), , ))
expect_equal(or3s(fke %between% c(-1L, 1L), , ),
             bor3(fke %between% c(-1L, 1L), , ))
expect_equal(or3s(fkf %between% c(-1L, 1L), , h %in% 1:4),
             bor3(fkf %between% c(-1L, 1L), , h %in% 1:4))
expect_equal(or3s(fkg %between% c(-1L, 1L), , v %in% 1:4),
             bor3(fkg %between% c(-1L, 1L), , v %in% 1:4))
expect_equal(or3s(fkh %between% c(-1L, 1L), , w %in% 1:4),
             bor3(fkh %between% c(-1L, 1L), , w %in% 1:4))
expect_equal(or3s(fki %between% c(-1L, 1L), logi_e, ),
             bor3(fki %between% c(-1L, 1L), logi_e, ))
expect_equal(or3s(fkj %between% c(-1L, 1L), logi_x, ),
             bor3(fkj %between% c(-1L, 1L), logi_x, ))
expect_equal(or3s(fkk %between% c(-1L, 1L), logi_s, ),
             bor3(fkk %between% c(-1L, 1L), logi_s, ))
expect_equal(or3s(fkl %between% c(-1L, 1L), logi_w, a %in% 1:4),
             bor3(fkl %between% c(-1L, 1L), logi_w, a %in% 1:4))
expect_equal(or3s(fkm %between% c(-1L, 1L), logi_b, r %in% 1:4),
             bor3(fkm %between% c(-1L, 1L), logi_b, r %in% 1:4))
expect_equal(or3s(fkn %between% c(-1L, 1L), logi_k, a %in% 1:4),
             bor3(fkn %between% c(-1L, 1L), logi_k, a %in% 1:4))
expect_equal(or3s(fko %between% c(-1L, 1L), , ),
             bor3(fko %between% c(-1L, 1L), , ))
expect_equal(or3s(fkp %between% c(-1L, 1L), , ),
             bor3(fkp %between% c(-1L, 1L), , ))
expect_equal(or3s(fkq %between% c(-1L, 1L), , ),
             bor3(fkq %between% c(-1L, 1L), , ))
expect_equal(or3s(fkr %between% c(-1L, 1L), , e < 0L),
             bor3(fkr %between% c(-1L, 1L), , e < 0L))
expect_equal(or3s(fks %between% c(-1L, 1L), , n < 1L),
             bor3(fks %between% c(-1L, 1L), , n < 1L))
expect_equal(or3s(fkt %between% c(-1L, 1L), , a < 9L),
             bor3(fkt %between% c(-1L, 1L), , a < 9L))
expect_equal(or3s(fku %between% c(-1L, 1L), logi_e, ),
             bor3(fku %between% c(-1L, 1L), logi_e, ))
expect_equal(or3s(fkv %between% c(-1L, 1L), logi_i, ),
             bor3(fkv %between% c(-1L, 1L), logi_i, ))
expect_equal(or3s(fkw %between% c(-1L, 1L), logi_o, ),
             bor3(fkw %between% c(-1L, 1L), logi_o, ))
expect_equal(or3s(fkx %between% c(-1L, 1L), logi_e, m < 0L),
             bor3(fkx %between% c(-1L, 1L), logi_e, m < 0L))
expect_equal(or3s(fky %between% c(-1L, 1L), logi_d, d < 1L),
             bor3(fky %between% c(-1L, 1L), logi_d, d < 1L))
expect_equal(or3s(fkz %between% c(-1L, 1L), logi_u, a < 9L),
             bor3(fkz %between% c(-1L, 1L), logi_u, a < 9L))
expect_equal(or3s(fla %between% c(-1L, 1L), , ),
             bor3(fla %between% c(-1L, 1L), , ))
expect_equal(or3s(flb %between% c(-1L, 1L), , ),
             bor3(flb %between% c(-1L, 1L), , ))
expect_equal(or3s(flc %between% c(-1L, 1L), , ),
             bor3(flc %between% c(-1L, 1L), , ))
expect_equal(or3s(fld %between% c(-1L, 1L), , s <= 0L),
             bor3(fld %between% c(-1L, 1L), , s <= 0L))
expect_equal(or3s(fle %between% c(-1L, 1L), , w <= 1L),
             bor3(fle %between% c(-1L, 1L), , w <= 1L))
expect_equal(or3s(flf %between% c(-1L, 1L), , f <= 9L),
             bor3(flf %between% c(-1L, 1L), , f <= 9L))
expect_equal(or3s(flg %between% c(-1L, 1L), logi_g, ),
             bor3(flg %between% c(-1L, 1L), logi_g, ))
expect_equal(or3s(flh %between% c(-1L, 1L), logi_t, ),
             bor3(flh %between% c(-1L, 1L), logi_t, ))
expect_equal(or3s(fli %between% c(-1L, 1L), logi_e, ),
             bor3(fli %between% c(-1L, 1L), logi_e, ))
expect_equal(or3s(flj %between% c(-1L, 1L), logi_o, s <= 0L),
             bor3(flj %between% c(-1L, 1L), logi_o, s <= 0L))
expect_equal(or3s(flk %between% c(-1L, 1L), logi_n, c <= 1L),
             bor3(flk %between% c(-1L, 1L), logi_n, c <= 1L))
expect_equal(or3s(fll %between% c(-1L, 1L), logi_n, c <= 9L),
             bor3(fll %between% c(-1L, 1L), logi_n, c <= 9L))
expect_equal(or3s(flm %between% c(-1L, 1L), , ),
             bor3(flm %between% c(-1L, 1L), , ))
expect_equal(or3s(fln %between% c(-1L, 1L), , ),
             bor3(fln %between% c(-1L, 1L), , ))
expect_equal(or3s(flo %between% c(-1L, 1L), , ),
             bor3(flo %between% c(-1L, 1L), , ))
expect_equal(or3s(flp %between% c(-1L, 1L), , a == 0L),
             bor3(flp %between% c(-1L, 1L), , a == 0L))
expect_equal(or3s(flq %between% c(-1L, 1L), , p == 1L),
             bor3(flq %between% c(-1L, 1L), , p == 1L))
expect_equal(or3s(flr %between% c(-1L, 1L), , k == 9L),
             bor3(flr %between% c(-1L, 1L), , k == 9L))
expect_equal(or3s(fls %between% c(-1L, 1L), logi_n, ),
             bor3(fls %between% c(-1L, 1L), logi_n, ))
expect_equal(or3s(flt %between% c(-1L, 1L), logi_e, ),
             bor3(flt %between% c(-1L, 1L), logi_e, ))
expect_equal(or3s(flu %between% c(-1L, 1L), logi_s, ),
             bor3(flu %between% c(-1L, 1L), logi_s, ))
expect_equal(or3s(flv %between% c(-1L, 1L), logi_d, f == 0L),
             bor3(flv %between% c(-1L, 1L), logi_d, f == 0L))
expect_equal(or3s(flw %between% c(-1L, 1L), logi_a, g == 1L),
             bor3(flw %between% c(-1L, 1L), logi_a, g == 1L))
expect_equal(or3s(flx %between% c(-1L, 1L), logi_b, q == 9L),
             bor3(flx %between% c(-1L, 1L), logi_b, q == 9L))
expect_equal(or3s(fly %between% c(-1L, 1L), , ),
             bor3(fly %between% c(-1L, 1L), , ))
expect_equal(or3s(flz %between% c(-1L, 1L), , ),
             bor3(flz %between% c(-1L, 1L), , ))
expect_equal(or3s(fma %between% c(-1L, 1L), , ),
             bor3(fma %between% c(-1L, 1L), , ))
expect_equal(or3s(fmb %between% c(-1L, 1L), , p > 0L),
             bor3(fmb %between% c(-1L, 1L), , p > 0L))
expect_equal(or3s(fmc %between% c(-1L, 1L), , y > 1L),
             bor3(fmc %between% c(-1L, 1L), , y > 1L))
expect_equal(or3s(fmd %between% c(-1L, 1L), , u > 9L),
             bor3(fmd %between% c(-1L, 1L), , u > 9L))
expect_equal(or3s(fme %between% c(-1L, 1L), logi_c, ),
             bor3(fme %between% c(-1L, 1L), logi_c, ))
expect_equal(or3s(fmf %between% c(-1L, 1L), logi_e, ),
             bor3(fmf %between% c(-1L, 1L), logi_e, ))
expect_equal(or3s(fmg %between% c(-1L, 1L), logi_k, ),
             bor3(fmg %between% c(-1L, 1L), logi_k, ))
expect_equal(or3s(fmh %between% c(-1L, 1L), logi_k, n > 0L),
             bor3(fmh %between% c(-1L, 1L), logi_k, n > 0L))
expect_equal(or3s(fmi %between% c(-1L, 1L), logi_g, q > 1L),
             bor3(fmi %between% c(-1L, 1L), logi_g, q > 1L))
expect_equal(or3s(fmj %between% c(-1L, 1L), logi_t, r > 9L),
             bor3(fmj %between% c(-1L, 1L), logi_t, r > 9L))
expect_equal(or3s(fmk %between% c(-1L, 1L), , ),
             bor3(fmk %between% c(-1L, 1L), , ))
expect_equal(or3s(fml %between% c(-1L, 1L), , ),
             bor3(fml %between% c(-1L, 1L), , ))
expect_equal(or3s(fmm %between% c(-1L, 1L), , ),
             bor3(fmm %between% c(-1L, 1L), , ))
expect_equal(or3s(fmn %between% c(-1L, 1L), , b >= 0L),
             bor3(fmn %between% c(-1L, 1L), , b >= 0L))
expect_equal(or3s(fmo %between% c(-1L, 1L), , l >= 1L),
             bor3(fmo %between% c(-1L, 1L), , l >= 1L))
expect_equal(or3s(fmp %between% c(-1L, 1L), , v >= 9L),
             bor3(fmp %between% c(-1L, 1L), , v >= 9L))
expect_equal(or3s(fmq %between% c(-1L, 1L), logi_e, ),
             bor3(fmq %between% c(-1L, 1L), logi_e, ))
expect_equal(or3s(fmr %between% c(-1L, 1L), logi_h, ),
             bor3(fmr %between% c(-1L, 1L), logi_h, ))
expect_equal(or3s(fms %between% c(-1L, 1L), logi_v, ),
             bor3(fms %between% c(-1L, 1L), logi_v, ))
expect_equal(or3s(fmt %between% c(-1L, 1L), logi_u, b >= 0L),
             bor3(fmt %between% c(-1L, 1L), logi_u, b >= 0L))
expect_equal(or3s(fmu %between% c(-1L, 1L), logi_w, v >= 1L),
             bor3(fmu %between% c(-1L, 1L), logi_w, v >= 1L))
expect_equal(or3s(fmv %between% c(-1L, 1L), logi_d, l >= 9L),
             bor3(fmv %between% c(-1L, 1L), logi_d, l >= 9L))
expect_equal(or3s(fmw %between% c(-1L, 1L), , ),
             bor3(fmw %between% c(-1L, 1L), , ))
expect_equal(or3s(fmx %between% c(-1L, 1L), , ),
             bor3(fmx %between% c(-1L, 1L), , ))
expect_equal(or3s(fmy %between% c(-1L, 1L), , ),
             bor3(fmy %between% c(-1L, 1L), , ))
expect_equal(or3s(fmz %between% c(-1L, 1L), , logi_w),
             bor3(fmz %between% c(-1L, 1L), , logi_w))
expect_equal(or3s(fna %between% c(-1L, 1L), , logi_w),
             bor3(fna %between% c(-1L, 1L), , logi_w))
expect_equal(or3s(fnb %between% c(-1L, 1L), , logi_f),
             bor3(fnb %between% c(-1L, 1L), , logi_f))
expect_equal(or3s(fnc %between% c(-1L, 1L), !logi_m, ),
             bor3(fnc %between% c(-1L, 1L), !logi_m, ))
expect_equal(or3s(fnd %between% c(-1L, 1L), !logi_z, ),
             bor3(fnd %between% c(-1L, 1L), !logi_z, ))
expect_equal(or3s(fne %between% c(-1L, 1L), !logi_s, ),
             bor3(fne %between% c(-1L, 1L), !logi_s, ))
expect_equal(or3s(fnf %between% c(-1L, 1L), !logi_r, logi_q),
             bor3(fnf %between% c(-1L, 1L), !logi_r, logi_q))
expect_equal(or3s(fng %between% c(-1L, 1L), !logi_d, logi_y),
             bor3(fng %between% c(-1L, 1L), !logi_d, logi_y))
expect_equal(or3s(fnh %between% c(-1L, 1L), !logi_f, logi_b),
             bor3(fnh %between% c(-1L, 1L), !logi_f, logi_b))
expect_equal(or3s(fni %between% c(-1L, 1L), , ),
             bor3(fni %between% c(-1L, 1L), , ))
expect_equal(or3s(fnj %between% c(-1L, 1L), , ),
             bor3(fnj %between% c(-1L, 1L), , ))
expect_equal(or3s(fnk %between% c(-1L, 1L), , ),
             bor3(fnk %between% c(-1L, 1L), , ))
expect_equal(or3s(fnl %between% c(-1L, 1L), , !logi_j),
             bor3(fnl %between% c(-1L, 1L), , !logi_j))
expect_equal(or3s(fnm %between% c(-1L, 1L), , !logi_b),
             bor3(fnm %between% c(-1L, 1L), , !logi_b))
expect_equal(or3s(fnn %between% c(-1L, 1L), , !logi_f),
             bor3(fnn %between% c(-1L, 1L), , !logi_f))
expect_equal(or3s(fno %between% c(-1L, 1L), !logi_n, ),
             bor3(fno %between% c(-1L, 1L), !logi_n, ))
expect_equal(or3s(fnp %between% c(-1L, 1L), !logi_e, ),
             bor3(fnp %between% c(-1L, 1L), !logi_e, ))
expect_equal(or3s(fnq %between% c(-1L, 1L), !logi_u, ),
             bor3(fnq %between% c(-1L, 1L), !logi_u, ))
expect_equal(or3s(fnr %between% c(-1L, 1L), !logi_x, !logi_p),
             bor3(fnr %between% c(-1L, 1L), !logi_x, !logi_p))
expect_equal(or3s(fns %between% c(-1L, 1L), !logi_z, !logi_n),
             bor3(fns %between% c(-1L, 1L), !logi_z, !logi_n))
expect_equal(or3s(fnt %between% c(-1L, 1L), !logi_m, !logi_l),
             bor3(fnt %between% c(-1L, 1L), !logi_m, !logi_l))
expect_equal(or3s(fnu %between% c(-1L, 1L), , ),
             bor3(fnu %between% c(-1L, 1L), , ))
expect_equal(or3s(fnv %between% c(-1L, 1L), , ),
             bor3(fnv %between% c(-1L, 1L), , ))
expect_equal(or3s(fnw %between% c(-1L, 1L), , ),
             bor3(fnw %between% c(-1L, 1L), , ))
expect_equal(or3s(fnx %between% c(-1L, 1L), , b != 0L),
             bor3(fnx %between% c(-1L, 1L), , b != 0L))
expect_equal(or3s(fny %between% c(-1L, 1L), , y != 1L),
             bor3(fny %between% c(-1L, 1L), , y != 1L))
expect_equal(or3s(fnz %between% c(-1L, 1L), , s != 9L),
             bor3(fnz %between% c(-1L, 1L), , s != 9L))
expect_equal(or3s(foa %between% c(-1L, 1L), !logi_j, ),
             bor3(foa %between% c(-1L, 1L), !logi_j, ))
expect_equal(or3s(fob %between% c(-1L, 1L), !logi_m, ),
             bor3(fob %between% c(-1L, 1L), !logi_m, ))
expect_equal(or3s(foc %between% c(-1L, 1L), !logi_y, ),
             bor3(foc %between% c(-1L, 1L), !logi_y, ))
expect_equal(or3s(fod %between% c(-1L, 1L), !logi_k, g != 0L),
             bor3(fod %between% c(-1L, 1L), !logi_k, g != 0L))
expect_equal(or3s(foe %between% c(-1L, 1L), !logi_b, u != 1L),
             bor3(foe %between% c(-1L, 1L), !logi_b, u != 1L))
expect_equal(or3s(fof %between% c(-1L, 1L), !logi_q, s != 9L),
             bor3(fof %between% c(-1L, 1L), !logi_q, s != 9L))
expect_equal(or3s(fog %between% c(-1L, 1L), , ),
             bor3(fog %between% c(-1L, 1L), , ))
expect_equal(or3s(foh %between% c(-1L, 1L), , ),
             bor3(foh %between% c(-1L, 1L), , ))
expect_equal(or3s(foi %between% c(-1L, 1L), , ),
             bor3(foi %between% c(-1L, 1L), , ))
expect_equal(or3s(foj %between% c(-1L, 1L), , o %between% c(-1L, 1L)),
             bor3(foj %between% c(-1L, 1L), , o %between% c(-1L, 1L)))
expect_equal(or3s(fok %between% c(-1L, 1L), , v %between% c(-1L, 1L)),
             bor3(fok %between% c(-1L, 1L), , v %between% c(-1L, 1L)))
expect_equal(or3s(fol %between% c(-1L, 1L), , h %between% c(-1L, 1L)),
             bor3(fol %between% c(-1L, 1L), , h %between% c(-1L, 1L)))
expect_equal(or3s(fom %between% c(-1L, 1L), !logi_w, ),
             bor3(fom %between% c(-1L, 1L), !logi_w, ))
expect_equal(or3s(fon %between% c(-1L, 1L), !logi_k, ),
             bor3(fon %between% c(-1L, 1L), !logi_k, ))
expect_equal(or3s(foo %between% c(-1L, 1L), !logi_w, ),
             bor3(foo %between% c(-1L, 1L), !logi_w, ))
expect_equal(or3s(fop %between% c(-1L, 1L), !logi_y, p %between% c(-1L, 1L)),
             bor3(fop %between% c(-1L, 1L), !logi_y, p %between% c(-1L, 1L)))
expect_equal(or3s(foq %between% c(-1L, 1L), !logi_b, h %between% c(-1L, 1L)),
             bor3(foq %between% c(-1L, 1L), !logi_b, h %between% c(-1L, 1L)))
expect_equal(or3s(FOR %between% c(-1L, 1L), !logi_h, v %between% c(-1L, 1L)),
             bor3(FOR %between% c(-1L, 1L), !logi_h, v %between% c(-1L, 1L)))
expect_equal(or3s(fos %between% c(-1L, 1L), , ),
             bor3(fos %between% c(-1L, 1L), , ))
expect_equal(or3s(fot %between% c(-1L, 1L), , ),
             bor3(fot %between% c(-1L, 1L), , ))
expect_equal(or3s(fou %between% c(-1L, 1L), , ),
             bor3(fou %between% c(-1L, 1L), , ))
expect_equal(or3s(fov %between% c(-1L, 1L), , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fov %between% c(-1L, 1L), , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fow %between% c(-1L, 1L), , s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fow %between% c(-1L, 1L), , s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fox %between% c(-1L, 1L), , d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fox %between% c(-1L, 1L), , d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(foy %between% c(-1L, 1L), !logi_x, ),
             bor3(foy %between% c(-1L, 1L), !logi_x, ))
expect_equal(or3s(foz %between% c(-1L, 1L), !logi_b, ),
             bor3(foz %between% c(-1L, 1L), !logi_b, ))
expect_equal(or3s(fpa %between% c(-1L, 1L), !logi_o, ),
             bor3(fpa %between% c(-1L, 1L), !logi_o, ))
expect_equal(or3s(fpb %between% c(-1L, 1L), !logi_b, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fpb %between% c(-1L, 1L), !logi_b, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fpc %between% c(-1L, 1L), !logi_n, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fpc %between% c(-1L, 1L), !logi_n, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fpd %between% c(-1L, 1L), !logi_c, d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fpd %between% c(-1L, 1L), !logi_c, d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fpe %between% c(-1L, 1L), , ),
             bor3(fpe %between% c(-1L, 1L), , ))
expect_equal(or3s(fpf %between% c(-1L, 1L), , ),
             bor3(fpf %between% c(-1L, 1L), , ))
expect_equal(or3s(fpg %between% c(-1L, 1L), , ),
             bor3(fpg %between% c(-1L, 1L), , ))
expect_equal(or3s(fph %between% c(-1L, 1L), , e %in% 1:4),
             bor3(fph %between% c(-1L, 1L), , e %in% 1:4))
expect_equal(or3s(fpi %between% c(-1L, 1L), , e %in% 1:4),
             bor3(fpi %between% c(-1L, 1L), , e %in% 1:4))
expect_equal(or3s(fpj %between% c(-1L, 1L), , u %in% 1:4),
             bor3(fpj %between% c(-1L, 1L), , u %in% 1:4))
expect_equal(or3s(fpk %between% c(-1L, 1L), !logi_e, ),
             bor3(fpk %between% c(-1L, 1L), !logi_e, ))
expect_equal(or3s(fpl %between% c(-1L, 1L), !logi_x, ),
             bor3(fpl %between% c(-1L, 1L), !logi_x, ))
expect_equal(or3s(fpm %between% c(-1L, 1L), !logi_m, ),
             bor3(fpm %between% c(-1L, 1L), !logi_m, ))
expect_equal(or3s(fpn %between% c(-1L, 1L), !logi_j, l %in% 1:4),
             bor3(fpn %between% c(-1L, 1L), !logi_j, l %in% 1:4))
expect_equal(or3s(fpo %between% c(-1L, 1L), !logi_w, k %in% 1:4),
             bor3(fpo %between% c(-1L, 1L), !logi_w, k %in% 1:4))
expect_equal(or3s(fpp %between% c(-1L, 1L), !logi_y, v %in% 1:4),
             bor3(fpp %between% c(-1L, 1L), !logi_y, v %in% 1:4))
expect_equal(or3s(fpq %between% c(-1L, 1L), , ),
             bor3(fpq %between% c(-1L, 1L), , ))
expect_equal(or3s(fpr %between% c(-1L, 1L), , ),
             bor3(fpr %between% c(-1L, 1L), , ))
expect_equal(or3s(fps %between% c(-1L, 1L), , ),
             bor3(fps %between% c(-1L, 1L), , ))
expect_equal(or3s(fpt %between% c(-1L, 1L), , l < 0L),
             bor3(fpt %between% c(-1L, 1L), , l < 0L))
expect_equal(or3s(fpu %between% c(-1L, 1L), , u < 1L),
             bor3(fpu %between% c(-1L, 1L), , u < 1L))
expect_equal(or3s(fpv %between% c(-1L, 1L), , r < 9L),
             bor3(fpv %between% c(-1L, 1L), , r < 9L))
expect_equal(or3s(fpw %between% c(-1L, 1L), !logi_d, ),
             bor3(fpw %between% c(-1L, 1L), !logi_d, ))
expect_equal(or3s(fpx %between% c(-1L, 1L), !logi_j, ),
             bor3(fpx %between% c(-1L, 1L), !logi_j, ))
expect_equal(or3s(fpy %between% c(-1L, 1L), !logi_c, ),
             bor3(fpy %between% c(-1L, 1L), !logi_c, ))
expect_equal(or3s(fpz %between% c(-1L, 1L), !logi_d, n < 0L),
             bor3(fpz %between% c(-1L, 1L), !logi_d, n < 0L))
expect_equal(or3s(fqa %between% c(-1L, 1L), !logi_g, h < 1L),
             bor3(fqa %between% c(-1L, 1L), !logi_g, h < 1L))
expect_equal(or3s(fqb %between% c(-1L, 1L), !logi_f, c < 9L),
             bor3(fqb %between% c(-1L, 1L), !logi_f, c < 9L))
expect_equal(or3s(fqc %between% c(-1L, 1L), , ),
             bor3(fqc %between% c(-1L, 1L), , ))
expect_equal(or3s(fqd %between% c(-1L, 1L), , ),
             bor3(fqd %between% c(-1L, 1L), , ))
expect_equal(or3s(fqe %between% c(-1L, 1L), , ),
             bor3(fqe %between% c(-1L, 1L), , ))
expect_equal(or3s(fqf %between% c(-1L, 1L), , f <= 0L),
             bor3(fqf %between% c(-1L, 1L), , f <= 0L))
expect_equal(or3s(fqg %between% c(-1L, 1L), , h <= 1L),
             bor3(fqg %between% c(-1L, 1L), , h <= 1L))
expect_equal(or3s(fqh %between% c(-1L, 1L), , b <= 9L),
             bor3(fqh %between% c(-1L, 1L), , b <= 9L))
expect_equal(or3s(fqi %between% c(-1L, 1L), !logi_q, ),
             bor3(fqi %between% c(-1L, 1L), !logi_q, ))
expect_equal(or3s(fqj %between% c(-1L, 1L), !logi_n, ),
             bor3(fqj %between% c(-1L, 1L), !logi_n, ))
expect_equal(or3s(fqk %between% c(-1L, 1L), !logi_t, ),
             bor3(fqk %between% c(-1L, 1L), !logi_t, ))
expect_equal(or3s(fql %between% c(-1L, 1L), !logi_e, o <= 0L),
             bor3(fql %between% c(-1L, 1L), !logi_e, o <= 0L))
expect_equal(or3s(fqm %between% c(-1L, 1L), !logi_i, r <= 1L),
             bor3(fqm %between% c(-1L, 1L), !logi_i, r <= 1L))
expect_equal(or3s(fqn %between% c(-1L, 1L), !logi_w, d <= 9L),
             bor3(fqn %between% c(-1L, 1L), !logi_w, d <= 9L))
expect_equal(or3s(fqo %between% c(-1L, 1L), , ),
             bor3(fqo %between% c(-1L, 1L), , ))
expect_equal(or3s(fqp %between% c(-1L, 1L), , ),
             bor3(fqp %between% c(-1L, 1L), , ))
expect_equal(or3s(fqq %between% c(-1L, 1L), , ),
             bor3(fqq %between% c(-1L, 1L), , ))
expect_equal(or3s(fqr %between% c(-1L, 1L), , m == 0L),
             bor3(fqr %between% c(-1L, 1L), , m == 0L))
expect_equal(or3s(fqs %between% c(-1L, 1L), , d == 1L),
             bor3(fqs %between% c(-1L, 1L), , d == 1L))
expect_equal(or3s(fqt %between% c(-1L, 1L), , p == 9L),
             bor3(fqt %between% c(-1L, 1L), , p == 9L))
expect_equal(or3s(fqu %between% c(-1L, 1L), !logi_o, ),
             bor3(fqu %between% c(-1L, 1L), !logi_o, ))
expect_equal(or3s(fqv %between% c(-1L, 1L), !logi_r, ),
             bor3(fqv %between% c(-1L, 1L), !logi_r, ))
expect_equal(or3s(fqw %between% c(-1L, 1L), !logi_n, ),
             bor3(fqw %between% c(-1L, 1L), !logi_n, ))
expect_equal(or3s(fqx %between% c(-1L, 1L), !logi_l, b == 0L),
             bor3(fqx %between% c(-1L, 1L), !logi_l, b == 0L))
expect_equal(or3s(fqy %between% c(-1L, 1L), !logi_d, d == 1L),
             bor3(fqy %between% c(-1L, 1L), !logi_d, d == 1L))
expect_equal(or3s(fqz %between% c(-1L, 1L), !logi_t, h == 9L),
             bor3(fqz %between% c(-1L, 1L), !logi_t, h == 9L))
expect_equal(or3s(fra %between% c(-1L, 1L), , ),
             bor3(fra %between% c(-1L, 1L), , ))
expect_equal(or3s(frb %between% c(-1L, 1L), , ),
             bor3(frb %between% c(-1L, 1L), , ))
expect_equal(or3s(frc %between% c(-1L, 1L), , ),
             bor3(frc %between% c(-1L, 1L), , ))
expect_equal(or3s(frd %between% c(-1L, 1L), , m > 0L),
             bor3(frd %between% c(-1L, 1L), , m > 0L))
expect_equal(or3s(fre %between% c(-1L, 1L), , w > 1L),
             bor3(fre %between% c(-1L, 1L), , w > 1L))
expect_equal(or3s(frf %between% c(-1L, 1L), , x > 9L),
             bor3(frf %between% c(-1L, 1L), , x > 9L))
expect_equal(or3s(frg %between% c(-1L, 1L), !logi_d, ),
             bor3(frg %between% c(-1L, 1L), !logi_d, ))
expect_equal(or3s(frh %between% c(-1L, 1L), !logi_m, ),
             bor3(frh %between% c(-1L, 1L), !logi_m, ))
expect_equal(or3s(fri %between% c(-1L, 1L), !logi_l, ),
             bor3(fri %between% c(-1L, 1L), !logi_l, ))
expect_equal(or3s(frj %between% c(-1L, 1L), !logi_z, j > 0L),
             bor3(frj %between% c(-1L, 1L), !logi_z, j > 0L))
expect_equal(or3s(frk %between% c(-1L, 1L), !logi_c, d > 1L),
             bor3(frk %between% c(-1L, 1L), !logi_c, d > 1L))
expect_equal(or3s(frl %between% c(-1L, 1L), !logi_r, q > 9L),
             bor3(frl %between% c(-1L, 1L), !logi_r, q > 9L))
expect_equal(or3s(frm %between% c(-1L, 1L), , ),
             bor3(frm %between% c(-1L, 1L), , ))
expect_equal(or3s(frn %between% c(-1L, 1L), , ),
             bor3(frn %between% c(-1L, 1L), , ))
expect_equal(or3s(fro %between% c(-1L, 1L), , ),
             bor3(fro %between% c(-1L, 1L), , ))
expect_equal(or3s(frp %between% c(-1L, 1L), , t >= 0L),
             bor3(frp %between% c(-1L, 1L), , t >= 0L))
expect_equal(or3s(frq %between% c(-1L, 1L), , c >= 1L),
             bor3(frq %between% c(-1L, 1L), , c >= 1L))
expect_equal(or3s(frr %between% c(-1L, 1L), , m >= 9L),
             bor3(frr %between% c(-1L, 1L), , m >= 9L))
expect_equal(or3s(frs %between% c(-1L, 1L), !logi_e, ),
             bor3(frs %between% c(-1L, 1L), !logi_e, ))
expect_equal(or3s(frt %between% c(-1L, 1L), !logi_a, ),
             bor3(frt %between% c(-1L, 1L), !logi_a, ))
expect_equal(or3s(fru %between% c(-1L, 1L), !logi_l, ),
             bor3(fru %between% c(-1L, 1L), !logi_l, ))
expect_equal(or3s(frv %between% c(-1L, 1L), !logi_k, d >= 0L),
             bor3(frv %between% c(-1L, 1L), !logi_k, d >= 0L))
expect_equal(or3s(frw %between% c(-1L, 1L), !logi_t, l >= 1L),
             bor3(frw %between% c(-1L, 1L), !logi_t, l >= 1L))
expect_equal(or3s(frx %between% c(-1L, 1L), !logi_b, b >= 9L),
             bor3(frx %between% c(-1L, 1L), !logi_b, b >= 9L))
expect_equal(or3s(fry %between% c(-1L, 1L), , ),
             bor3(fry %between% c(-1L, 1L), , ))
expect_equal(or3s(frz %between% c(-1L, 1L), , ),
             bor3(frz %between% c(-1L, 1L), , ))
expect_equal(or3s(fsa %between% c(-1L, 1L), , ),
             bor3(fsa %between% c(-1L, 1L), , ))
expect_equal(or3s(fsb %between% c(-1L, 1L), , logi_h),
             bor3(fsb %between% c(-1L, 1L), , logi_h))
expect_equal(or3s(fsc %between% c(-1L, 1L), , logi_h),
             bor3(fsc %between% c(-1L, 1L), , logi_h))
expect_equal(or3s(fsd %between% c(-1L, 1L), , logi_q),
             bor3(fsd %between% c(-1L, 1L), , logi_q))
expect_equal(or3s(fse %between% c(-1L, 1L), u != 0L, ),
             bor3(fse %between% c(-1L, 1L), u != 0L, ))
expect_equal(or3s(fsf %between% c(-1L, 1L), t != 1L, ),
             bor3(fsf %between% c(-1L, 1L), t != 1L, ))
expect_equal(or3s(fsg %between% c(-1L, 1L), u != 9L, ),
             bor3(fsg %between% c(-1L, 1L), u != 9L, ))
expect_equal(or3s(fsh %between% c(-1L, 1L), f != 0L, logi_h),
             bor3(fsh %between% c(-1L, 1L), f != 0L, logi_h))
expect_equal(or3s(fsi %between% c(-1L, 1L), g != 1L, logi_d),
             bor3(fsi %between% c(-1L, 1L), g != 1L, logi_d))
expect_equal(or3s(fsj %between% c(-1L, 1L), n != 9L, logi_i),
             bor3(fsj %between% c(-1L, 1L), n != 9L, logi_i))
expect_equal(or3s(fsk %between% c(-1L, 1L), , ),
             bor3(fsk %between% c(-1L, 1L), , ))
expect_equal(or3s(fsl %between% c(-1L, 1L), , ),
             bor3(fsl %between% c(-1L, 1L), , ))
expect_equal(or3s(fsm %between% c(-1L, 1L), , ),
             bor3(fsm %between% c(-1L, 1L), , ))
expect_equal(or3s(fsn %between% c(-1L, 1L), , !logi_y),
             bor3(fsn %between% c(-1L, 1L), , !logi_y))
expect_equal(or3s(fso %between% c(-1L, 1L), , !logi_w),
             bor3(fso %between% c(-1L, 1L), , !logi_w))
expect_equal(or3s(fsp %between% c(-1L, 1L), , !logi_z),
             bor3(fsp %between% c(-1L, 1L), , !logi_z))
expect_equal(or3s(fsq %between% c(-1L, 1L), b != 0L, ),
             bor3(fsq %between% c(-1L, 1L), b != 0L, ))
expect_equal(or3s(fsr %between% c(-1L, 1L), c != 1L, ),
             bor3(fsr %between% c(-1L, 1L), c != 1L, ))
expect_equal(or3s(fss %between% c(-1L, 1L), m != 9L, ),
             bor3(fss %between% c(-1L, 1L), m != 9L, ))
expect_equal(or3s(fst %between% c(-1L, 1L), s != 0L, !logi_d),
             bor3(fst %between% c(-1L, 1L), s != 0L, !logi_d))
expect_equal(or3s(fsu %between% c(-1L, 1L), m != 1L, !logi_q),
             bor3(fsu %between% c(-1L, 1L), m != 1L, !logi_q))
expect_equal(or3s(fsv %between% c(-1L, 1L), x != 9L, !logi_n),
             bor3(fsv %between% c(-1L, 1L), x != 9L, !logi_n))
expect_equal(or3s(fsw %between% c(-1L, 1L), , ),
             bor3(fsw %between% c(-1L, 1L), , ))
expect_equal(or3s(fsx %between% c(-1L, 1L), , ),
             bor3(fsx %between% c(-1L, 1L), , ))
expect_equal(or3s(fsy %between% c(-1L, 1L), , ),
             bor3(fsy %between% c(-1L, 1L), , ))
expect_equal(or3s(fsz %between% c(-1L, 1L), , m != 0L),
             bor3(fsz %between% c(-1L, 1L), , m != 0L))
expect_equal(or3s(fta %between% c(-1L, 1L), , n != 1L),
             bor3(fta %between% c(-1L, 1L), , n != 1L))
expect_equal(or3s(ftb %between% c(-1L, 1L), , n != 9L),
             bor3(ftb %between% c(-1L, 1L), , n != 9L))
expect_equal(or3s(ftc %between% c(-1L, 1L), l != 0L, ),
             bor3(ftc %between% c(-1L, 1L), l != 0L, ))
expect_equal(or3s(ftd %between% c(-1L, 1L), b != 1L, ),
             bor3(ftd %between% c(-1L, 1L), b != 1L, ))
expect_equal(or3s(fte %between% c(-1L, 1L), p != 9L, ),
             bor3(fte %between% c(-1L, 1L), p != 9L, ))
expect_equal(or3s(ftf %between% c(-1L, 1L), h != 0L, l != 0L),
             bor3(ftf %between% c(-1L, 1L), h != 0L, l != 0L))
expect_equal(or3s(ftg %between% c(-1L, 1L), n != 1L, l != 1L),
             bor3(ftg %between% c(-1L, 1L), n != 1L, l != 1L))
expect_equal(or3s(fth %between% c(-1L, 1L), i != 9L, e != 9L),
             bor3(fth %between% c(-1L, 1L), i != 9L, e != 9L))
expect_equal(or3s(fti %between% c(-1L, 1L), , ),
             bor3(fti %between% c(-1L, 1L), , ))
expect_equal(or3s(ftj %between% c(-1L, 1L), , ),
             bor3(ftj %between% c(-1L, 1L), , ))
expect_equal(or3s(ftk %between% c(-1L, 1L), , ),
             bor3(ftk %between% c(-1L, 1L), , ))
expect_equal(or3s(ftl %between% c(-1L, 1L), , i %between% c(-1L, 1L)),
             bor3(ftl %between% c(-1L, 1L), , i %between% c(-1L, 1L)))
expect_equal(or3s(ftm %between% c(-1L, 1L), , k %between% c(-1L, 1L)),
             bor3(ftm %between% c(-1L, 1L), , k %between% c(-1L, 1L)))
expect_equal(or3s(ftn %between% c(-1L, 1L), , g %between% c(-1L, 1L)),
             bor3(ftn %between% c(-1L, 1L), , g %between% c(-1L, 1L)))
expect_equal(or3s(fto %between% c(-1L, 1L), e != 0L, ),
             bor3(fto %between% c(-1L, 1L), e != 0L, ))
expect_equal(or3s(ftp %between% c(-1L, 1L), d != 1L, ),
             bor3(ftp %between% c(-1L, 1L), d != 1L, ))
expect_equal(or3s(ftq %between% c(-1L, 1L), z != 9L, ),
             bor3(ftq %between% c(-1L, 1L), z != 9L, ))
expect_equal(or3s(ftr %between% c(-1L, 1L), n != 0L, s %between% c(-1L, 1L)),
             bor3(ftr %between% c(-1L, 1L), n != 0L, s %between% c(-1L, 1L)))
expect_equal(or3s(fts %between% c(-1L, 1L), x != 1L, i %between% c(-1L, 1L)),
             bor3(fts %between% c(-1L, 1L), x != 1L, i %between% c(-1L, 1L)))
expect_equal(or3s(ftt %between% c(-1L, 1L), o != 9L, z %between% c(-1L, 1L)),
             bor3(ftt %between% c(-1L, 1L), o != 9L, z %between% c(-1L, 1L)))
expect_equal(or3s(ftu %between% c(-1L, 1L), , ),
             bor3(ftu %between% c(-1L, 1L), , ))
expect_equal(or3s(ftv %between% c(-1L, 1L), , ),
             bor3(ftv %between% c(-1L, 1L), , ))
expect_equal(or3s(ftw %between% c(-1L, 1L), , ),
             bor3(ftw %between% c(-1L, 1L), , ))
expect_equal(or3s(ftx %between% c(-1L, 1L), , i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ftx %between% c(-1L, 1L), , i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fty %between% c(-1L, 1L), , a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fty %between% c(-1L, 1L), , a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ftz %between% c(-1L, 1L), , a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ftz %between% c(-1L, 1L), , a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fua %between% c(-1L, 1L), p != 0L, ),
             bor3(fua %between% c(-1L, 1L), p != 0L, ))
expect_equal(or3s(fub %between% c(-1L, 1L), m != 1L, ),
             bor3(fub %between% c(-1L, 1L), m != 1L, ))
expect_equal(or3s(fuc %between% c(-1L, 1L), i != 9L, ),
             bor3(fuc %between% c(-1L, 1L), i != 9L, ))
expect_equal(or3s(fud %between% c(-1L, 1L), x != 0L, x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fud %between% c(-1L, 1L), x != 0L, x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fue %between% c(-1L, 1L), a != 1L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fue %between% c(-1L, 1L), a != 1L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fuf %between% c(-1L, 1L), y != 9L, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fuf %between% c(-1L, 1L), y != 9L, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(fug %between% c(-1L, 1L), , ),
             bor3(fug %between% c(-1L, 1L), , ))
expect_equal(or3s(fuh %between% c(-1L, 1L), , ),
             bor3(fuh %between% c(-1L, 1L), , ))
expect_equal(or3s(fui %between% c(-1L, 1L), , ),
             bor3(fui %between% c(-1L, 1L), , ))
expect_equal(or3s(fuj %between% c(-1L, 1L), , a %in% 1:4),
             bor3(fuj %between% c(-1L, 1L), , a %in% 1:4))
expect_equal(or3s(fuk %between% c(-1L, 1L), , o %in% 1:4),
             bor3(fuk %between% c(-1L, 1L), , o %in% 1:4))
expect_equal(or3s(ful %between% c(-1L, 1L), , l %in% 1:4),
             bor3(ful %between% c(-1L, 1L), , l %in% 1:4))
expect_equal(or3s(fum %between% c(-1L, 1L), q != 0L, ),
             bor3(fum %between% c(-1L, 1L), q != 0L, ))
expect_equal(or3s(fun %between% c(-1L, 1L), y != 1L, ),
             bor3(fun %between% c(-1L, 1L), y != 1L, ))
expect_equal(or3s(fuo %between% c(-1L, 1L), k != 9L, ),
             bor3(fuo %between% c(-1L, 1L), k != 9L, ))
expect_equal(or3s(fup %between% c(-1L, 1L), t != 0L, w %in% 1:4),
             bor3(fup %between% c(-1L, 1L), t != 0L, w %in% 1:4))
expect_equal(or3s(fuq %between% c(-1L, 1L), g != 1L, q %in% 1:4),
             bor3(fuq %between% c(-1L, 1L), g != 1L, q %in% 1:4))
expect_equal(or3s(fur %between% c(-1L, 1L), x != 9L, c %in% 1:4),
             bor3(fur %between% c(-1L, 1L), x != 9L, c %in% 1:4))
expect_equal(or3s(fus %between% c(-1L, 1L), , ),
             bor3(fus %between% c(-1L, 1L), , ))
expect_equal(or3s(fut %between% c(-1L, 1L), , ),
             bor3(fut %between% c(-1L, 1L), , ))
expect_equal(or3s(fuu %between% c(-1L, 1L), , ),
             bor3(fuu %between% c(-1L, 1L), , ))
expect_equal(or3s(fuv %between% c(-1L, 1L), , o < 0L),
             bor3(fuv %between% c(-1L, 1L), , o < 0L))
expect_equal(or3s(fuw %between% c(-1L, 1L), , z < 1L),
             bor3(fuw %between% c(-1L, 1L), , z < 1L))
expect_equal(or3s(fux %between% c(-1L, 1L), , l < 9L),
             bor3(fux %between% c(-1L, 1L), , l < 9L))
expect_equal(or3s(fuy %between% c(-1L, 1L), k != 0L, ),
             bor3(fuy %between% c(-1L, 1L), k != 0L, ))
expect_equal(or3s(fuz %between% c(-1L, 1L), e != 1L, ),
             bor3(fuz %between% c(-1L, 1L), e != 1L, ))
expect_equal(or3s(fva %between% c(-1L, 1L), x != 9L, ),
             bor3(fva %between% c(-1L, 1L), x != 9L, ))
expect_equal(or3s(fvb %between% c(-1L, 1L), n != 0L, b < 0L),
             bor3(fvb %between% c(-1L, 1L), n != 0L, b < 0L))
expect_equal(or3s(fvc %between% c(-1L, 1L), i != 1L, h < 1L),
             bor3(fvc %between% c(-1L, 1L), i != 1L, h < 1L))
expect_equal(or3s(fvd %between% c(-1L, 1L), x != 9L, o < 9L),
             bor3(fvd %between% c(-1L, 1L), x != 9L, o < 9L))
expect_equal(or3s(fve %between% c(-1L, 1L), , ),
             bor3(fve %between% c(-1L, 1L), , ))
expect_equal(or3s(fvf %between% c(-1L, 1L), , ),
             bor3(fvf %between% c(-1L, 1L), , ))
expect_equal(or3s(fvg %between% c(-1L, 1L), , ),
             bor3(fvg %between% c(-1L, 1L), , ))
expect_equal(or3s(fvh %between% c(-1L, 1L), , o <= 0L),
             bor3(fvh %between% c(-1L, 1L), , o <= 0L))
expect_equal(or3s(fvi %between% c(-1L, 1L), , d <= 1L),
             bor3(fvi %between% c(-1L, 1L), , d <= 1L))
expect_equal(or3s(fvj %between% c(-1L, 1L), , s <= 9L),
             bor3(fvj %between% c(-1L, 1L), , s <= 9L))
expect_equal(or3s(fvk %between% c(-1L, 1L), u != 0L, ),
             bor3(fvk %between% c(-1L, 1L), u != 0L, ))
expect_equal(or3s(fvl %between% c(-1L, 1L), f != 1L, ),
             bor3(fvl %between% c(-1L, 1L), f != 1L, ))
expect_equal(or3s(fvm %between% c(-1L, 1L), o != 9L, ),
             bor3(fvm %between% c(-1L, 1L), o != 9L, ))
expect_equal(or3s(fvn %between% c(-1L, 1L), z != 0L, b <= 0L),
             bor3(fvn %between% c(-1L, 1L), z != 0L, b <= 0L))
expect_equal(or3s(fvo %between% c(-1L, 1L), g != 1L, j <= 1L),
             bor3(fvo %between% c(-1L, 1L), g != 1L, j <= 1L))
expect_equal(or3s(fvp %between% c(-1L, 1L), w != 9L, v <= 9L),
             bor3(fvp %between% c(-1L, 1L), w != 9L, v <= 9L))
expect_equal(or3s(fvq %between% c(-1L, 1L), , ),
             bor3(fvq %between% c(-1L, 1L), , ))
expect_equal(or3s(fvr %between% c(-1L, 1L), , ),
             bor3(fvr %between% c(-1L, 1L), , ))
expect_equal(or3s(fvs %between% c(-1L, 1L), , ),
             bor3(fvs %between% c(-1L, 1L), , ))
expect_equal(or3s(fvt %between% c(-1L, 1L), , y == 0L),
             bor3(fvt %between% c(-1L, 1L), , y == 0L))
expect_equal(or3s(fvu %between% c(-1L, 1L), , e == 1L),
             bor3(fvu %between% c(-1L, 1L), , e == 1L))
expect_equal(or3s(fvv %between% c(-1L, 1L), , j == 9L),
             bor3(fvv %between% c(-1L, 1L), , j == 9L))
expect_equal(or3s(fvw %between% c(-1L, 1L), j != 0L, ),
             bor3(fvw %between% c(-1L, 1L), j != 0L, ))
expect_equal(or3s(fvx %between% c(-1L, 1L), a != 1L, ),
             bor3(fvx %between% c(-1L, 1L), a != 1L, ))
expect_equal(or3s(fvy %between% c(-1L, 1L), u != 9L, ),
             bor3(fvy %between% c(-1L, 1L), u != 9L, ))
expect_equal(or3s(fvz %between% c(-1L, 1L), d != 0L, k == 0L),
             bor3(fvz %between% c(-1L, 1L), d != 0L, k == 0L))
expect_equal(or3s(fwa %between% c(-1L, 1L), m != 1L, a == 1L),
             bor3(fwa %between% c(-1L, 1L), m != 1L, a == 1L))
expect_equal(or3s(fwb %between% c(-1L, 1L), h != 9L, j == 9L),
             bor3(fwb %between% c(-1L, 1L), h != 9L, j == 9L))
expect_equal(or3s(fwc %between% c(-1L, 1L), , ),
             bor3(fwc %between% c(-1L, 1L), , ))
expect_equal(or3s(fwd %between% c(-1L, 1L), , ),
             bor3(fwd %between% c(-1L, 1L), , ))
expect_equal(or3s(fwe %between% c(-1L, 1L), , ),
             bor3(fwe %between% c(-1L, 1L), , ))
expect_equal(or3s(fwf %between% c(-1L, 1L), , c > 0L),
             bor3(fwf %between% c(-1L, 1L), , c > 0L))
expect_equal(or3s(fwg %between% c(-1L, 1L), , a > 1L),
             bor3(fwg %between% c(-1L, 1L), , a > 1L))
expect_equal(or3s(fwh %between% c(-1L, 1L), , j > 9L),
             bor3(fwh %between% c(-1L, 1L), , j > 9L))
expect_equal(or3s(fwi %between% c(-1L, 1L), t != 0L, ),
             bor3(fwi %between% c(-1L, 1L), t != 0L, ))
expect_equal(or3s(fwj %between% c(-1L, 1L), b != 1L, ),
             bor3(fwj %between% c(-1L, 1L), b != 1L, ))
expect_equal(or3s(fwk %between% c(-1L, 1L), t != 9L, ),
             bor3(fwk %between% c(-1L, 1L), t != 9L, ))
expect_equal(or3s(fwl %between% c(-1L, 1L), g != 0L, y > 0L),
             bor3(fwl %between% c(-1L, 1L), g != 0L, y > 0L))
expect_equal(or3s(fwm %between% c(-1L, 1L), f != 1L, c > 1L),
             bor3(fwm %between% c(-1L, 1L), f != 1L, c > 1L))
expect_equal(or3s(fwn %between% c(-1L, 1L), h != 9L, m > 9L),
             bor3(fwn %between% c(-1L, 1L), h != 9L, m > 9L))
expect_equal(or3s(fwo %between% c(-1L, 1L), , ),
             bor3(fwo %between% c(-1L, 1L), , ))
expect_equal(or3s(fwp %between% c(-1L, 1L), , ),
             bor3(fwp %between% c(-1L, 1L), , ))
expect_equal(or3s(fwq %between% c(-1L, 1L), , ),
             bor3(fwq %between% c(-1L, 1L), , ))
expect_equal(or3s(fwr %between% c(-1L, 1L), , m >= 0L),
             bor3(fwr %between% c(-1L, 1L), , m >= 0L))
expect_equal(or3s(fws %between% c(-1L, 1L), , u >= 1L),
             bor3(fws %between% c(-1L, 1L), , u >= 1L))
expect_equal(or3s(fwt %between% c(-1L, 1L), , r >= 9L),
             bor3(fwt %between% c(-1L, 1L), , r >= 9L))
expect_equal(or3s(fwu %between% c(-1L, 1L), o != 0L, ),
             bor3(fwu %between% c(-1L, 1L), o != 0L, ))
expect_equal(or3s(fwv %between% c(-1L, 1L), q != 1L, ),
             bor3(fwv %between% c(-1L, 1L), q != 1L, ))
expect_equal(or3s(fww %between% c(-1L, 1L), n != 9L, ),
             bor3(fww %between% c(-1L, 1L), n != 9L, ))
expect_equal(or3s(fwx %between% c(-1L, 1L), n != 0L, g >= 0L),
             bor3(fwx %between% c(-1L, 1L), n != 0L, g >= 0L))
expect_equal(or3s(fwy %between% c(-1L, 1L), i != 1L, u >= 1L),
             bor3(fwy %between% c(-1L, 1L), i != 1L, u >= 1L))
expect_equal(or3s(fwz %between% c(-1L, 1L), a != 9L, z >= 9L),
             bor3(fwz %between% c(-1L, 1L), a != 9L, z >= 9L))
expect_equal(or3s(fxa %between% c(-1L, 1L), , ),
             bor3(fxa %between% c(-1L, 1L), , ))
expect_equal(or3s(fxb %between% c(-1L, 1L), , ),
             bor3(fxb %between% c(-1L, 1L), , ))
expect_equal(or3s(fxc %between% c(-1L, 1L), , ),
             bor3(fxc %between% c(-1L, 1L), , ))
expect_equal(or3s(fxd %between% c(-1L, 1L), , logi_e),
             bor3(fxd %between% c(-1L, 1L), , logi_e))
expect_equal(or3s(fxe %between% c(-1L, 1L), , logi_h),
             bor3(fxe %between% c(-1L, 1L), , logi_h))
expect_equal(or3s(fxf %between% c(-1L, 1L), , logi_l),
             bor3(fxf %between% c(-1L, 1L), , logi_l))
expect_equal(or3s(fxg %between% c(-1L, 1L), p %between% c(-1L, 1L), ),
             bor3(fxg %between% c(-1L, 1L), p %between% c(-1L, 1L), ))
expect_equal(or3s(fxh %between% c(-1L, 1L), b %between% c(-1L, 1L), ),
             bor3(fxh %between% c(-1L, 1L), b %between% c(-1L, 1L), ))
expect_equal(or3s(fxi %between% c(-1L, 1L), s %between% c(-1L, 1L), ),
             bor3(fxi %between% c(-1L, 1L), s %between% c(-1L, 1L), ))
expect_equal(or3s(fxj %between% c(-1L, 1L), v %between% c(-1L, 1L), logi_u),
             bor3(fxj %between% c(-1L, 1L), v %between% c(-1L, 1L), logi_u))
expect_equal(or3s(fxk %between% c(-1L, 1L), s %between% c(-1L, 1L), logi_b),
             bor3(fxk %between% c(-1L, 1L), s %between% c(-1L, 1L), logi_b))
expect_equal(or3s(fxl %between% c(-1L, 1L), f %between% c(-1L, 1L), logi_q),
             bor3(fxl %between% c(-1L, 1L), f %between% c(-1L, 1L), logi_q))
expect_equal(or3s(fxm %between% c(-1L, 1L), , ),
             bor3(fxm %between% c(-1L, 1L), , ))
expect_equal(or3s(fxn %between% c(-1L, 1L), , ),
             bor3(fxn %between% c(-1L, 1L), , ))
expect_equal(or3s(fxo %between% c(-1L, 1L), , ),
             bor3(fxo %between% c(-1L, 1L), , ))
expect_equal(or3s(fxp %between% c(-1L, 1L), , !logi_l),
             bor3(fxp %between% c(-1L, 1L), , !logi_l))
expect_equal(or3s(fxq %between% c(-1L, 1L), , !logi_u),
             bor3(fxq %between% c(-1L, 1L), , !logi_u))
expect_equal(or3s(fxr %between% c(-1L, 1L), , !logi_u),
             bor3(fxr %between% c(-1L, 1L), , !logi_u))
expect_equal(or3s(fxs %between% c(-1L, 1L), c %between% c(-1L, 1L), ),
             bor3(fxs %between% c(-1L, 1L), c %between% c(-1L, 1L), ))
expect_equal(or3s(fxt %between% c(-1L, 1L), h %between% c(-1L, 1L), ),
             bor3(fxt %between% c(-1L, 1L), h %between% c(-1L, 1L), ))
expect_equal(or3s(fxu %between% c(-1L, 1L), a %between% c(-1L, 1L), ),
             bor3(fxu %between% c(-1L, 1L), a %between% c(-1L, 1L), ))
expect_equal(or3s(fxv %between% c(-1L, 1L), t %between% c(-1L, 1L), !logi_o),
             bor3(fxv %between% c(-1L, 1L), t %between% c(-1L, 1L), !logi_o))
expect_equal(or3s(fxw %between% c(-1L, 1L), f %between% c(-1L, 1L), !logi_l),
             bor3(fxw %between% c(-1L, 1L), f %between% c(-1L, 1L), !logi_l))
expect_equal(or3s(fxx %between% c(-1L, 1L), z %between% c(-1L, 1L), !logi_c),
             bor3(fxx %between% c(-1L, 1L), z %between% c(-1L, 1L), !logi_c))
expect_equal(or3s(fxy %between% c(-1L, 1L), , ),
             bor3(fxy %between% c(-1L, 1L), , ))
expect_equal(or3s(fxz %between% c(-1L, 1L), , ),
             bor3(fxz %between% c(-1L, 1L), , ))
expect_equal(or3s(fya %between% c(-1L, 1L), , ),
             bor3(fya %between% c(-1L, 1L), , ))
expect_equal(or3s(fyb %between% c(-1L, 1L), , y != 0L),
             bor3(fyb %between% c(-1L, 1L), , y != 0L))
expect_equal(or3s(fyc %between% c(-1L, 1L), , r != 1L),
             bor3(fyc %between% c(-1L, 1L), , r != 1L))
expect_equal(or3s(fyd %between% c(-1L, 1L), , y != 9L),
             bor3(fyd %between% c(-1L, 1L), , y != 9L))
expect_equal(or3s(fye %between% c(-1L, 1L), n %between% c(-1L, 1L), ),
             bor3(fye %between% c(-1L, 1L), n %between% c(-1L, 1L), ))
expect_equal(or3s(fyf %between% c(-1L, 1L), e %between% c(-1L, 1L), ),
             bor3(fyf %between% c(-1L, 1L), e %between% c(-1L, 1L), ))
expect_equal(or3s(fyg %between% c(-1L, 1L), k %between% c(-1L, 1L), ),
             bor3(fyg %between% c(-1L, 1L), k %between% c(-1L, 1L), ))
expect_equal(or3s(fyh %between% c(-1L, 1L), i %between% c(-1L, 1L), s != 0L),
             bor3(fyh %between% c(-1L, 1L), i %between% c(-1L, 1L), s != 0L))
expect_equal(or3s(fyi %between% c(-1L, 1L), q %between% c(-1L, 1L), k != 1L),
             bor3(fyi %between% c(-1L, 1L), q %between% c(-1L, 1L), k != 1L))
expect_equal(or3s(fyj %between% c(-1L, 1L), c %between% c(-1L, 1L), q != 9L),
             bor3(fyj %between% c(-1L, 1L), c %between% c(-1L, 1L), q != 9L))
expect_equal(or3s(fyk %between% c(-1L, 1L), , ),
             bor3(fyk %between% c(-1L, 1L), , ))
expect_equal(or3s(fyl %between% c(-1L, 1L), , ),
             bor3(fyl %between% c(-1L, 1L), , ))
expect_equal(or3s(fym %between% c(-1L, 1L), , ),
             bor3(fym %between% c(-1L, 1L), , ))
expect_equal(or3s(fyn %between% c(-1L, 1L), , n %between% c(-1L, 1L)),
             bor3(fyn %between% c(-1L, 1L), , n %between% c(-1L, 1L)))
expect_equal(or3s(fyo %between% c(-1L, 1L), , i %between% c(-1L, 1L)),
             bor3(fyo %between% c(-1L, 1L), , i %between% c(-1L, 1L)))
expect_equal(or3s(fyp %between% c(-1L, 1L), , t %between% c(-1L, 1L)),
             bor3(fyp %between% c(-1L, 1L), , t %between% c(-1L, 1L)))
expect_equal(or3s(fyq %between% c(-1L, 1L), x %between% c(-1L, 1L), ),
             bor3(fyq %between% c(-1L, 1L), x %between% c(-1L, 1L), ))
expect_equal(or3s(fyr %between% c(-1L, 1L), s %between% c(-1L, 1L), ),
             bor3(fyr %between% c(-1L, 1L), s %between% c(-1L, 1L), ))
expect_equal(or3s(fys %between% c(-1L, 1L), g %between% c(-1L, 1L), ),
             bor3(fys %between% c(-1L, 1L), g %between% c(-1L, 1L), ))
expect_equal(or3s(fyt %between% c(-1L, 1L), v %between% c(-1L, 1L), u %between% c(-1L, 1L)),
             bor3(fyt %between% c(-1L, 1L), v %between% c(-1L, 1L), u %between% c(-1L, 1L)))
expect_equal(or3s(fyu %between% c(-1L, 1L), d %between% c(-1L, 1L), b %between% c(-1L, 1L)),
             bor3(fyu %between% c(-1L, 1L), d %between% c(-1L, 1L), b %between% c(-1L, 1L)))
expect_equal(or3s(fyv %between% c(-1L, 1L), i %between% c(-1L, 1L), q %between% c(-1L, 1L)),
             bor3(fyv %between% c(-1L, 1L), i %between% c(-1L, 1L), q %between% c(-1L, 1L)))
expect_equal(or3s(fyw %between% c(-1L, 1L), , ),
             bor3(fyw %between% c(-1L, 1L), , ))
expect_equal(or3s(fyx %between% c(-1L, 1L), , ),
             bor3(fyx %between% c(-1L, 1L), , ))
expect_equal(or3s(fyy %between% c(-1L, 1L), , ),
             bor3(fyy %between% c(-1L, 1L), , ))
expect_equal(or3s(fyz %between% c(-1L, 1L), , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(fyz %between% c(-1L, 1L), , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ga %between% c(-1L, 1L), , u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ga %between% c(-1L, 1L), , u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gb %between% c(-1L, 1L), , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gb %between% c(-1L, 1L), , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gc %between% c(-1L, 1L), h %between% c(-1L, 1L), ),
             bor3(gc %between% c(-1L, 1L), h %between% c(-1L, 1L), ))
expect_equal(or3s(gd %between% c(-1L, 1L), r %between% c(-1L, 1L), ),
             bor3(gd %between% c(-1L, 1L), r %between% c(-1L, 1L), ))
expect_equal(or3s(ge %between% c(-1L, 1L), g %between% c(-1L, 1L), ),
             bor3(ge %between% c(-1L, 1L), g %between% c(-1L, 1L), ))
expect_equal(or3s(gf %between% c(-1L, 1L), j %between% c(-1L, 1L), j %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gf %between% c(-1L, 1L), j %between% c(-1L, 1L), j %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gg %between% c(-1L, 1L), v %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gg %between% c(-1L, 1L), v %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gh %between% c(-1L, 1L), h %between% c(-1L, 1L), c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gh %between% c(-1L, 1L), h %between% c(-1L, 1L), c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gi %between% c(-1L, 1L), , ),
             bor3(gi %between% c(-1L, 1L), , ))
expect_equal(or3s(gj %between% c(-1L, 1L), , ),
             bor3(gj %between% c(-1L, 1L), , ))
expect_equal(or3s(gk %between% c(-1L, 1L), , ),
             bor3(gk %between% c(-1L, 1L), , ))
expect_equal(or3s(gl %between% c(-1L, 1L), , u %in% 1:4),
             bor3(gl %between% c(-1L, 1L), , u %in% 1:4))
expect_equal(or3s(gm %between% c(-1L, 1L), , r %in% 1:4),
             bor3(gm %between% c(-1L, 1L), , r %in% 1:4))
expect_equal(or3s(gn %between% c(-1L, 1L), , x %in% 1:4),
             bor3(gn %between% c(-1L, 1L), , x %in% 1:4))
expect_equal(or3s(go %between% c(-1L, 1L), v %between% c(-1L, 1L), ),
             bor3(go %between% c(-1L, 1L), v %between% c(-1L, 1L), ))
expect_equal(or3s(gp %between% c(-1L, 1L), u %between% c(-1L, 1L), ),
             bor3(gp %between% c(-1L, 1L), u %between% c(-1L, 1L), ))
expect_equal(or3s(gq %between% c(-1L, 1L), m %between% c(-1L, 1L), ),
             bor3(gq %between% c(-1L, 1L), m %between% c(-1L, 1L), ))
expect_equal(or3s(gr %between% c(-1L, 1L), v %between% c(-1L, 1L), t %in% 1:4),
             bor3(gr %between% c(-1L, 1L), v %between% c(-1L, 1L), t %in% 1:4))
expect_equal(or3s(gs %between% c(-1L, 1L), t %between% c(-1L, 1L), x %in% 1:4),
             bor3(gs %between% c(-1L, 1L), t %between% c(-1L, 1L), x %in% 1:4))
expect_equal(or3s(gt %between% c(-1L, 1L), m %between% c(-1L, 1L), t %in% 1:4),
             bor3(gt %between% c(-1L, 1L), m %between% c(-1L, 1L), t %in% 1:4))
expect_equal(or3s(gu %between% c(-1L, 1L), , ),
             bor3(gu %between% c(-1L, 1L), , ))
expect_equal(or3s(gv %between% c(-1L, 1L), , ),
             bor3(gv %between% c(-1L, 1L), , ))
expect_equal(or3s(gw %between% c(-1L, 1L), , ),
             bor3(gw %between% c(-1L, 1L), , ))
expect_equal(or3s(gx %between% c(-1L, 1L), , q < 0L),
             bor3(gx %between% c(-1L, 1L), , q < 0L))
expect_equal(or3s(gy %between% c(-1L, 1L), , o < 1L),
             bor3(gy %between% c(-1L, 1L), , o < 1L))
expect_equal(or3s(gz %between% c(-1L, 1L), , u < 9L),
             bor3(gz %between% c(-1L, 1L), , u < 9L))
expect_equal(or3s(gaa %between% c(-1L, 1L), y %between% c(-1L, 1L), ),
             bor3(gaa %between% c(-1L, 1L), y %between% c(-1L, 1L), ))
expect_equal(or3s(gab %between% c(-1L, 1L), h %between% c(-1L, 1L), ),
             bor3(gab %between% c(-1L, 1L), h %between% c(-1L, 1L), ))
expect_equal(or3s(gac %between% c(-1L, 1L), t %between% c(-1L, 1L), ),
             bor3(gac %between% c(-1L, 1L), t %between% c(-1L, 1L), ))
expect_equal(or3s(gad %between% c(-1L, 1L), i %between% c(-1L, 1L), s < 0L),
             bor3(gad %between% c(-1L, 1L), i %between% c(-1L, 1L), s < 0L))
expect_equal(or3s(gae %between% c(-1L, 1L), e %between% c(-1L, 1L), e < 1L),
             bor3(gae %between% c(-1L, 1L), e %between% c(-1L, 1L), e < 1L))
expect_equal(or3s(gaf %between% c(-1L, 1L), w %between% c(-1L, 1L), s < 9L),
             bor3(gaf %between% c(-1L, 1L), w %between% c(-1L, 1L), s < 9L))
expect_equal(or3s(gag %between% c(-1L, 1L), , ),
             bor3(gag %between% c(-1L, 1L), , ))
expect_equal(or3s(gah %between% c(-1L, 1L), , ),
             bor3(gah %between% c(-1L, 1L), , ))
expect_equal(or3s(gai %between% c(-1L, 1L), , ),
             bor3(gai %between% c(-1L, 1L), , ))
expect_equal(or3s(gaj %between% c(-1L, 1L), , k <= 0L),
             bor3(gaj %between% c(-1L, 1L), , k <= 0L))
expect_equal(or3s(gak %between% c(-1L, 1L), , c <= 1L),
             bor3(gak %between% c(-1L, 1L), , c <= 1L))
expect_equal(or3s(gal %between% c(-1L, 1L), , h <= 9L),
             bor3(gal %between% c(-1L, 1L), , h <= 9L))
expect_equal(or3s(gam %between% c(-1L, 1L), n %between% c(-1L, 1L), ),
             bor3(gam %between% c(-1L, 1L), n %between% c(-1L, 1L), ))
expect_equal(or3s(gan %between% c(-1L, 1L), i %between% c(-1L, 1L), ),
             bor3(gan %between% c(-1L, 1L), i %between% c(-1L, 1L), ))
expect_equal(or3s(gao %between% c(-1L, 1L), n %between% c(-1L, 1L), ),
             bor3(gao %between% c(-1L, 1L), n %between% c(-1L, 1L), ))
expect_equal(or3s(gap %between% c(-1L, 1L), n %between% c(-1L, 1L), f <= 0L),
             bor3(gap %between% c(-1L, 1L), n %between% c(-1L, 1L), f <= 0L))
expect_equal(or3s(gaq %between% c(-1L, 1L), q %between% c(-1L, 1L), d <= 1L),
             bor3(gaq %between% c(-1L, 1L), q %between% c(-1L, 1L), d <= 1L))
expect_equal(or3s(gar %between% c(-1L, 1L), b %between% c(-1L, 1L), z <= 9L),
             bor3(gar %between% c(-1L, 1L), b %between% c(-1L, 1L), z <= 9L))
expect_equal(or3s(gas %between% c(-1L, 1L), , ),
             bor3(gas %between% c(-1L, 1L), , ))
expect_equal(or3s(gat %between% c(-1L, 1L), , ),
             bor3(gat %between% c(-1L, 1L), , ))
expect_equal(or3s(gau %between% c(-1L, 1L), , ),
             bor3(gau %between% c(-1L, 1L), , ))
expect_equal(or3s(gav %between% c(-1L, 1L), , d == 0L),
             bor3(gav %between% c(-1L, 1L), , d == 0L))
expect_equal(or3s(gaw %between% c(-1L, 1L), , f == 1L),
             bor3(gaw %between% c(-1L, 1L), , f == 1L))
expect_equal(or3s(gax %between% c(-1L, 1L), , v == 9L),
             bor3(gax %between% c(-1L, 1L), , v == 9L))
expect_equal(or3s(gay %between% c(-1L, 1L), j %between% c(-1L, 1L), ),
             bor3(gay %between% c(-1L, 1L), j %between% c(-1L, 1L), ))
expect_equal(or3s(gaz %between% c(-1L, 1L), o %between% c(-1L, 1L), ),
             bor3(gaz %between% c(-1L, 1L), o %between% c(-1L, 1L), ))
expect_equal(or3s(gba %between% c(-1L, 1L), h %between% c(-1L, 1L), ),
             bor3(gba %between% c(-1L, 1L), h %between% c(-1L, 1L), ))
expect_equal(or3s(gbb %between% c(-1L, 1L), d %between% c(-1L, 1L), o == 0L),
             bor3(gbb %between% c(-1L, 1L), d %between% c(-1L, 1L), o == 0L))
expect_equal(or3s(gbc %between% c(-1L, 1L), w %between% c(-1L, 1L), t == 1L),
             bor3(gbc %between% c(-1L, 1L), w %between% c(-1L, 1L), t == 1L))
expect_equal(or3s(gbd %between% c(-1L, 1L), v %between% c(-1L, 1L), p == 9L),
             bor3(gbd %between% c(-1L, 1L), v %between% c(-1L, 1L), p == 9L))
expect_equal(or3s(gbe %between% c(-1L, 1L), , ),
             bor3(gbe %between% c(-1L, 1L), , ))
expect_equal(or3s(gbf %between% c(-1L, 1L), , ),
             bor3(gbf %between% c(-1L, 1L), , ))
expect_equal(or3s(gbg %between% c(-1L, 1L), , ),
             bor3(gbg %between% c(-1L, 1L), , ))
expect_equal(or3s(gbh %between% c(-1L, 1L), , k > 0L),
             bor3(gbh %between% c(-1L, 1L), , k > 0L))
expect_equal(or3s(gbi %between% c(-1L, 1L), , q > 1L),
             bor3(gbi %between% c(-1L, 1L), , q > 1L))
expect_equal(or3s(gbj %between% c(-1L, 1L), , l > 9L),
             bor3(gbj %between% c(-1L, 1L), , l > 9L))
expect_equal(or3s(gbk %between% c(-1L, 1L), u %between% c(-1L, 1L), ),
             bor3(gbk %between% c(-1L, 1L), u %between% c(-1L, 1L), ))
expect_equal(or3s(gbl %between% c(-1L, 1L), t %between% c(-1L, 1L), ),
             bor3(gbl %between% c(-1L, 1L), t %between% c(-1L, 1L), ))
expect_equal(or3s(gbm %between% c(-1L, 1L), e %between% c(-1L, 1L), ),
             bor3(gbm %between% c(-1L, 1L), e %between% c(-1L, 1L), ))
expect_equal(or3s(gbn %between% c(-1L, 1L), d %between% c(-1L, 1L), j > 0L),
             bor3(gbn %between% c(-1L, 1L), d %between% c(-1L, 1L), j > 0L))
expect_equal(or3s(gbo %between% c(-1L, 1L), l %between% c(-1L, 1L), n > 1L),
             bor3(gbo %between% c(-1L, 1L), l %between% c(-1L, 1L), n > 1L))
expect_equal(or3s(gbp %between% c(-1L, 1L), r %between% c(-1L, 1L), y > 9L),
             bor3(gbp %between% c(-1L, 1L), r %between% c(-1L, 1L), y > 9L))
expect_equal(or3s(gbq %between% c(-1L, 1L), , ),
             bor3(gbq %between% c(-1L, 1L), , ))
expect_equal(or3s(gbr %between% c(-1L, 1L), , ),
             bor3(gbr %between% c(-1L, 1L), , ))
expect_equal(or3s(gbs %between% c(-1L, 1L), , ),
             bor3(gbs %between% c(-1L, 1L), , ))
expect_equal(or3s(gbt %between% c(-1L, 1L), , d >= 0L),
             bor3(gbt %between% c(-1L, 1L), , d >= 0L))
expect_equal(or3s(gbu %between% c(-1L, 1L), , l >= 1L),
             bor3(gbu %between% c(-1L, 1L), , l >= 1L))
expect_equal(or3s(gbv %between% c(-1L, 1L), , x >= 9L),
             bor3(gbv %between% c(-1L, 1L), , x >= 9L))
expect_equal(or3s(gbw %between% c(-1L, 1L), u %between% c(-1L, 1L), ),
             bor3(gbw %between% c(-1L, 1L), u %between% c(-1L, 1L), ))
expect_equal(or3s(gbx %between% c(-1L, 1L), p %between% c(-1L, 1L), ),
             bor3(gbx %between% c(-1L, 1L), p %between% c(-1L, 1L), ))
expect_equal(or3s(gby %between% c(-1L, 1L), t %between% c(-1L, 1L), ),
             bor3(gby %between% c(-1L, 1L), t %between% c(-1L, 1L), ))
expect_equal(or3s(gbz %between% c(-1L, 1L), p %between% c(-1L, 1L), g >= 0L),
             bor3(gbz %between% c(-1L, 1L), p %between% c(-1L, 1L), g >= 0L))
expect_equal(or3s(gca %between% c(-1L, 1L), l %between% c(-1L, 1L), u >= 1L),
             bor3(gca %between% c(-1L, 1L), l %between% c(-1L, 1L), u >= 1L))
expect_equal(or3s(gcb %between% c(-1L, 1L), r %between% c(-1L, 1L), a >= 9L),
             bor3(gcb %between% c(-1L, 1L), r %between% c(-1L, 1L), a >= 9L))
expect_equal(or3s(gcc %between% c(-1L, 1L), , ),
             bor3(gcc %between% c(-1L, 1L), , ))
expect_equal(or3s(gcd %between% c(-1L, 1L), , ),
             bor3(gcd %between% c(-1L, 1L), , ))
expect_equal(or3s(gce %between% c(-1L, 1L), , ),
             bor3(gce %between% c(-1L, 1L), , ))
expect_equal(or3s(gcf %between% c(-1L, 1L), , logi_z),
             bor3(gcf %between% c(-1L, 1L), , logi_z))
expect_equal(or3s(gcg %between% c(-1L, 1L), , logi_q),
             bor3(gcg %between% c(-1L, 1L), , logi_q))
expect_equal(or3s(gch %between% c(-1L, 1L), , logi_d),
             bor3(gch %between% c(-1L, 1L), , logi_d))
expect_equal(or3s(gci %between% c(-1L, 1L), j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gci %between% c(-1L, 1L), j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gcj %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gcj %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gck %between% c(-1L, 1L), m %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gck %between% c(-1L, 1L), m %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gcl %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), logi_h),
             bor3(gcl %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), logi_h))
expect_equal(or3s(gcm %between% c(-1L, 1L), z %in% c(4L, -3L, 2L, 7L, 8L), logi_i),
             bor3(gcm %between% c(-1L, 1L), z %in% c(4L, -3L, 2L, 7L, 8L), logi_i))
expect_equal(or3s(gcn %between% c(-1L, 1L), c %in% c(4L, -3L, 2L, 7L, 8L), logi_y),
             bor3(gcn %between% c(-1L, 1L), c %in% c(4L, -3L, 2L, 7L, 8L), logi_y))
expect_equal(or3s(gco %between% c(-1L, 1L), , ),
             bor3(gco %between% c(-1L, 1L), , ))
expect_equal(or3s(gcp %between% c(-1L, 1L), , ),
             bor3(gcp %between% c(-1L, 1L), , ))
expect_equal(or3s(gcq %between% c(-1L, 1L), , ),
             bor3(gcq %between% c(-1L, 1L), , ))
expect_equal(or3s(gcr %between% c(-1L, 1L), , logi_q),
             bor3(gcr %between% c(-1L, 1L), , logi_q))
expect_equal(or3s(gcs %between% c(-1L, 1L), , logi_a),
             bor3(gcs %between% c(-1L, 1L), , logi_a))
expect_equal(or3s(gct %between% c(-1L, 1L), , logi_g),
             bor3(gct %between% c(-1L, 1L), , logi_g))
expect_equal(or3s(gcu %between% c(-1L, 1L), d %in% 1:4, ),
             bor3(gcu %between% c(-1L, 1L), d %in% 1:4, ))
expect_equal(or3s(gcv %between% c(-1L, 1L), a %in% 1:4, ),
             bor3(gcv %between% c(-1L, 1L), a %in% 1:4, ))
expect_equal(or3s(gcw %between% c(-1L, 1L), a %in% 1:4, ),
             bor3(gcw %between% c(-1L, 1L), a %in% 1:4, ))
expect_equal(or3s(gcx %between% c(-1L, 1L), i %in% 1:4, logi_n),
             bor3(gcx %between% c(-1L, 1L), i %in% 1:4, logi_n))
expect_equal(or3s(gcy %between% c(-1L, 1L), y %in% 1:4, logi_u),
             bor3(gcy %between% c(-1L, 1L), y %in% 1:4, logi_u))
expect_equal(or3s(gcz %between% c(-1L, 1L), z %in% 1:4, logi_g),
             bor3(gcz %between% c(-1L, 1L), z %in% 1:4, logi_g))
expect_equal(or3s(gda %between% c(-1L, 1L), , ),
             bor3(gda %between% c(-1L, 1L), , ))
expect_equal(or3s(gdb %between% c(-1L, 1L), , ),
             bor3(gdb %between% c(-1L, 1L), , ))
expect_equal(or3s(gdc %between% c(-1L, 1L), , ),
             bor3(gdc %between% c(-1L, 1L), , ))
expect_equal(or3s(gdd %between% c(-1L, 1L), , !logi_m),
             bor3(gdd %between% c(-1L, 1L), , !logi_m))
expect_equal(or3s(gde %between% c(-1L, 1L), , !logi_n),
             bor3(gde %between% c(-1L, 1L), , !logi_n))
expect_equal(or3s(gdf %between% c(-1L, 1L), , !logi_c),
             bor3(gdf %between% c(-1L, 1L), , !logi_c))
expect_equal(or3s(gdg %between% c(-1L, 1L), i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gdg %between% c(-1L, 1L), i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gdh %between% c(-1L, 1L), w %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gdh %between% c(-1L, 1L), w %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gdi %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gdi %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gdj %between% c(-1L, 1L), q %in% c(4L, -3L, 2L, 7L, 8L), !logi_j),
             bor3(gdj %between% c(-1L, 1L), q %in% c(4L, -3L, 2L, 7L, 8L), !logi_j))
expect_equal(or3s(gdk %between% c(-1L, 1L), w %in% c(4L, -3L, 2L, 7L, 8L), !logi_c),
             bor3(gdk %between% c(-1L, 1L), w %in% c(4L, -3L, 2L, 7L, 8L), !logi_c))
expect_equal(or3s(gdl %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), !logi_e),
             bor3(gdl %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), !logi_e))
expect_equal(or3s(gdm %between% c(-1L, 1L), , ),
             bor3(gdm %between% c(-1L, 1L), , ))
expect_equal(or3s(gdn %between% c(-1L, 1L), , ),
             bor3(gdn %between% c(-1L, 1L), , ))
expect_equal(or3s(gdo %between% c(-1L, 1L), , ),
             bor3(gdo %between% c(-1L, 1L), , ))
expect_equal(or3s(gdp %between% c(-1L, 1L), , !logi_u),
             bor3(gdp %between% c(-1L, 1L), , !logi_u))
expect_equal(or3s(gdq %between% c(-1L, 1L), , !logi_v),
             bor3(gdq %between% c(-1L, 1L), , !logi_v))
expect_equal(or3s(gdr %between% c(-1L, 1L), , !logi_n),
             bor3(gdr %between% c(-1L, 1L), , !logi_n))
expect_equal(or3s(gds %between% c(-1L, 1L), s %in% 1:4, ),
             bor3(gds %between% c(-1L, 1L), s %in% 1:4, ))
expect_equal(or3s(gdt %between% c(-1L, 1L), k %in% 1:4, ),
             bor3(gdt %between% c(-1L, 1L), k %in% 1:4, ))
expect_equal(or3s(gdu %between% c(-1L, 1L), j %in% 1:4, ),
             bor3(gdu %between% c(-1L, 1L), j %in% 1:4, ))
expect_equal(or3s(gdv %between% c(-1L, 1L), e %in% 1:4, !logi_r),
             bor3(gdv %between% c(-1L, 1L), e %in% 1:4, !logi_r))
expect_equal(or3s(gdw %between% c(-1L, 1L), u %in% 1:4, !logi_o),
             bor3(gdw %between% c(-1L, 1L), u %in% 1:4, !logi_o))
expect_equal(or3s(gdx %between% c(-1L, 1L), a %in% 1:4, !logi_k),
             bor3(gdx %between% c(-1L, 1L), a %in% 1:4, !logi_k))
expect_equal(or3s(gdy %between% c(-1L, 1L), , ),
             bor3(gdy %between% c(-1L, 1L), , ))
expect_equal(or3s(gdz %between% c(-1L, 1L), , ),
             bor3(gdz %between% c(-1L, 1L), , ))
expect_equal(or3s(gea %between% c(-1L, 1L), , ),
             bor3(gea %between% c(-1L, 1L), , ))
expect_equal(or3s(geb %between% c(-1L, 1L), , k != 0L),
             bor3(geb %between% c(-1L, 1L), , k != 0L))
expect_equal(or3s(gec %between% c(-1L, 1L), , h != 1L),
             bor3(gec %between% c(-1L, 1L), , h != 1L))
expect_equal(or3s(ged %between% c(-1L, 1L), , x != 9L),
             bor3(ged %between% c(-1L, 1L), , x != 9L))
expect_equal(or3s(gee %between% c(-1L, 1L), a %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gee %between% c(-1L, 1L), a %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gef %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gef %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(geg %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(geg %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(geh %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L), l != 0L),
             bor3(geh %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L), l != 0L))
expect_equal(or3s(gei %between% c(-1L, 1L), a %in% c(4L, -3L, 2L, 7L, 8L), e != 1L),
             bor3(gei %between% c(-1L, 1L), a %in% c(4L, -3L, 2L, 7L, 8L), e != 1L))
expect_equal(or3s(gej %between% c(-1L, 1L), g %in% c(4L, -3L, 2L, 7L, 8L), i != 9L),
             bor3(gej %between% c(-1L, 1L), g %in% c(4L, -3L, 2L, 7L, 8L), i != 9L))
expect_equal(or3s(gek %between% c(-1L, 1L), , ),
             bor3(gek %between% c(-1L, 1L), , ))
expect_equal(or3s(gel %between% c(-1L, 1L), , ),
             bor3(gel %between% c(-1L, 1L), , ))
expect_equal(or3s(gem %between% c(-1L, 1L), , ),
             bor3(gem %between% c(-1L, 1L), , ))
expect_equal(or3s(gen %between% c(-1L, 1L), , a != 0L),
             bor3(gen %between% c(-1L, 1L), , a != 0L))
expect_equal(or3s(geo %between% c(-1L, 1L), , m != 1L),
             bor3(geo %between% c(-1L, 1L), , m != 1L))
expect_equal(or3s(gep %between% c(-1L, 1L), , r != 9L),
             bor3(gep %between% c(-1L, 1L), , r != 9L))
expect_equal(or3s(geq %between% c(-1L, 1L), e %in% 1:4, ),
             bor3(geq %between% c(-1L, 1L), e %in% 1:4, ))
expect_equal(or3s(ger %between% c(-1L, 1L), l %in% 1:4, ),
             bor3(ger %between% c(-1L, 1L), l %in% 1:4, ))
expect_equal(or3s(ges %between% c(-1L, 1L), q %in% 1:4, ),
             bor3(ges %between% c(-1L, 1L), q %in% 1:4, ))
expect_equal(or3s(get %between% c(-1L, 1L), q %in% 1:4, s != 0L),
             bor3(get %between% c(-1L, 1L), q %in% 1:4, s != 0L))
expect_equal(or3s(geu %between% c(-1L, 1L), k %in% 1:4, q != 1L),
             bor3(geu %between% c(-1L, 1L), k %in% 1:4, q != 1L))
expect_equal(or3s(gev %between% c(-1L, 1L), s %in% 1:4, w != 9L),
             bor3(gev %between% c(-1L, 1L), s %in% 1:4, w != 9L))
expect_equal(or3s(gew %between% c(-1L, 1L), , ),
             bor3(gew %between% c(-1L, 1L), , ))
expect_equal(or3s(gex %between% c(-1L, 1L), , ),
             bor3(gex %between% c(-1L, 1L), , ))
expect_equal(or3s(gey %between% c(-1L, 1L), , ),
             bor3(gey %between% c(-1L, 1L), , ))
expect_equal(or3s(gez %between% c(-1L, 1L), , z %between% c(-1L, 1L)),
             bor3(gez %between% c(-1L, 1L), , z %between% c(-1L, 1L)))
expect_equal(or3s(gfa %between% c(-1L, 1L), , z %between% c(-1L, 1L)),
             bor3(gfa %between% c(-1L, 1L), , z %between% c(-1L, 1L)))
expect_equal(or3s(gfb %between% c(-1L, 1L), , r %between% c(-1L, 1L)),
             bor3(gfb %between% c(-1L, 1L), , r %between% c(-1L, 1L)))
expect_equal(or3s(gfc %between% c(-1L, 1L), s %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gfc %between% c(-1L, 1L), s %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gfd %between% c(-1L, 1L), u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gfd %between% c(-1L, 1L), u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gfe %between% c(-1L, 1L), c %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gfe %between% c(-1L, 1L), c %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gff %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), x %between% c(-1L, 1L)),
             bor3(gff %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), x %between% c(-1L, 1L)))
expect_equal(or3s(gfg %between% c(-1L, 1L), z %in% c(4L, -3L, 2L, 7L, 8L), l %between% c(-1L, 1L)),
             bor3(gfg %between% c(-1L, 1L), z %in% c(4L, -3L, 2L, 7L, 8L), l %between% c(-1L, 1L)))
expect_equal(or3s(gfh %between% c(-1L, 1L), k %in% c(4L, -3L, 2L, 7L, 8L), p %between% c(-1L, 1L)),
             bor3(gfh %between% c(-1L, 1L), k %in% c(4L, -3L, 2L, 7L, 8L), p %between% c(-1L, 1L)))
expect_equal(or3s(gfi %between% c(-1L, 1L), , ),
             bor3(gfi %between% c(-1L, 1L), , ))
expect_equal(or3s(gfj %between% c(-1L, 1L), , ),
             bor3(gfj %between% c(-1L, 1L), , ))
expect_equal(or3s(gfk %between% c(-1L, 1L), , ),
             bor3(gfk %between% c(-1L, 1L), , ))
expect_equal(or3s(gfl %between% c(-1L, 1L), , n %between% c(-1L, 1L)),
             bor3(gfl %between% c(-1L, 1L), , n %between% c(-1L, 1L)))
expect_equal(or3s(gfm %between% c(-1L, 1L), , r %between% c(-1L, 1L)),
             bor3(gfm %between% c(-1L, 1L), , r %between% c(-1L, 1L)))
expect_equal(or3s(gfn %between% c(-1L, 1L), , t %between% c(-1L, 1L)),
             bor3(gfn %between% c(-1L, 1L), , t %between% c(-1L, 1L)))
expect_equal(or3s(gfo %between% c(-1L, 1L), j %in% 1:4, ),
             bor3(gfo %between% c(-1L, 1L), j %in% 1:4, ))
expect_equal(or3s(gfp %between% c(-1L, 1L), j %in% 1:4, ),
             bor3(gfp %between% c(-1L, 1L), j %in% 1:4, ))
expect_equal(or3s(gfq %between% c(-1L, 1L), m %in% 1:4, ),
             bor3(gfq %between% c(-1L, 1L), m %in% 1:4, ))
expect_equal(or3s(gfr %between% c(-1L, 1L), q %in% 1:4, x %between% c(-1L, 1L)),
             bor3(gfr %between% c(-1L, 1L), q %in% 1:4, x %between% c(-1L, 1L)))
expect_equal(or3s(gfs %between% c(-1L, 1L), m %in% 1:4, p %between% c(-1L, 1L)),
             bor3(gfs %between% c(-1L, 1L), m %in% 1:4, p %between% c(-1L, 1L)))
expect_equal(or3s(gft %between% c(-1L, 1L), x %in% 1:4, j %between% c(-1L, 1L)),
             bor3(gft %between% c(-1L, 1L), x %in% 1:4, j %between% c(-1L, 1L)))
expect_equal(or3s(gfu %between% c(-1L, 1L), , ),
             bor3(gfu %between% c(-1L, 1L), , ))
expect_equal(or3s(gfv %between% c(-1L, 1L), , ),
             bor3(gfv %between% c(-1L, 1L), , ))
expect_equal(or3s(gfw %between% c(-1L, 1L), , ),
             bor3(gfw %between% c(-1L, 1L), , ))
expect_equal(or3s(gfx %between% c(-1L, 1L), , r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gfx %between% c(-1L, 1L), , r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gfy %between% c(-1L, 1L), , x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gfy %between% c(-1L, 1L), , x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gfz %between% c(-1L, 1L), , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gfz %between% c(-1L, 1L), , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gga %between% c(-1L, 1L), r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gga %between% c(-1L, 1L), r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ggb %between% c(-1L, 1L), j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ggb %between% c(-1L, 1L), j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ggc %between% c(-1L, 1L), d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ggc %between% c(-1L, 1L), d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ggd %between% c(-1L, 1L), q %in% c(4L, -3L, 2L, 7L, 8L), j %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ggd %between% c(-1L, 1L), q %in% c(4L, -3L, 2L, 7L, 8L), j %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gge %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L), r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gge %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L), r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ggf %between% c(-1L, 1L), g %in% c(4L, -3L, 2L, 7L, 8L), v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ggf %between% c(-1L, 1L), g %in% c(4L, -3L, 2L, 7L, 8L), v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ggg %between% c(-1L, 1L), , ),
             bor3(ggg %between% c(-1L, 1L), , ))
expect_equal(or3s(ggh %between% c(-1L, 1L), , ),
             bor3(ggh %between% c(-1L, 1L), , ))
expect_equal(or3s(ggi %between% c(-1L, 1L), , ),
             bor3(ggi %between% c(-1L, 1L), , ))
expect_equal(or3s(ggj %between% c(-1L, 1L), , b %in% 1:4),
             bor3(ggj %between% c(-1L, 1L), , b %in% 1:4))
expect_equal(or3s(ggk %between% c(-1L, 1L), , s %in% 1:4),
             bor3(ggk %between% c(-1L, 1L), , s %in% 1:4))
expect_equal(or3s(ggl %between% c(-1L, 1L), , f %in% 1:4),
             bor3(ggl %between% c(-1L, 1L), , f %in% 1:4))
expect_equal(or3s(ggm %between% c(-1L, 1L), n %in% 1:4, ),
             bor3(ggm %between% c(-1L, 1L), n %in% 1:4, ))
expect_equal(or3s(ggn %between% c(-1L, 1L), u %in% 1:4, ),
             bor3(ggn %between% c(-1L, 1L), u %in% 1:4, ))
expect_equal(or3s(ggo %between% c(-1L, 1L), x %in% 1:4, ),
             bor3(ggo %between% c(-1L, 1L), x %in% 1:4, ))
expect_equal(or3s(ggp %between% c(-1L, 1L), x %in% 1:4, c %in% 1:4),
             bor3(ggp %between% c(-1L, 1L), x %in% 1:4, c %in% 1:4))
expect_equal(or3s(ggq %between% c(-1L, 1L), t %in% 1:4, n %in% 1:4),
             bor3(ggq %between% c(-1L, 1L), t %in% 1:4, n %in% 1:4))
expect_equal(or3s(ggr %between% c(-1L, 1L), c %in% 1:4, y %in% 1:4),
             bor3(ggr %between% c(-1L, 1L), c %in% 1:4, y %in% 1:4))
expect_equal(or3s(ggs %between% c(-1L, 1L), , ),
             bor3(ggs %between% c(-1L, 1L), , ))
expect_equal(or3s(ggt %between% c(-1L, 1L), , ),
             bor3(ggt %between% c(-1L, 1L), , ))
expect_equal(or3s(ggu %between% c(-1L, 1L), , ),
             bor3(ggu %between% c(-1L, 1L), , ))
expect_equal(or3s(ggv %between% c(-1L, 1L), , o < 0L),
             bor3(ggv %between% c(-1L, 1L), , o < 0L))
expect_equal(or3s(ggw %between% c(-1L, 1L), , a < 1L),
             bor3(ggw %between% c(-1L, 1L), , a < 1L))
expect_equal(or3s(ggx %between% c(-1L, 1L), , j < 9L),
             bor3(ggx %between% c(-1L, 1L), , j < 9L))
expect_equal(or3s(ggy %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ggy %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ggz %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ggz %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gha %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gha %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ghb %between% c(-1L, 1L), f %in% c(4L, -3L, 2L, 7L, 8L), g < 0L),
             bor3(ghb %between% c(-1L, 1L), f %in% c(4L, -3L, 2L, 7L, 8L), g < 0L))
expect_equal(or3s(ghc %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), d < 1L),
             bor3(ghc %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), d < 1L))
expect_equal(or3s(ghd %between% c(-1L, 1L), s %in% c(4L, -3L, 2L, 7L, 8L), d < 9L),
             bor3(ghd %between% c(-1L, 1L), s %in% c(4L, -3L, 2L, 7L, 8L), d < 9L))
expect_equal(or3s(ghe %between% c(-1L, 1L), , ),
             bor3(ghe %between% c(-1L, 1L), , ))
expect_equal(or3s(ghf %between% c(-1L, 1L), , ),
             bor3(ghf %between% c(-1L, 1L), , ))
expect_equal(or3s(ghg %between% c(-1L, 1L), , ),
             bor3(ghg %between% c(-1L, 1L), , ))
expect_equal(or3s(ghh %between% c(-1L, 1L), , a < 0L),
             bor3(ghh %between% c(-1L, 1L), , a < 0L))
expect_equal(or3s(ghi %between% c(-1L, 1L), , v < 1L),
             bor3(ghi %between% c(-1L, 1L), , v < 1L))
expect_equal(or3s(ghj %between% c(-1L, 1L), , w < 9L),
             bor3(ghj %between% c(-1L, 1L), , w < 9L))
expect_equal(or3s(ghk %between% c(-1L, 1L), s %in% 1:4, ),
             bor3(ghk %between% c(-1L, 1L), s %in% 1:4, ))
expect_equal(or3s(ghl %between% c(-1L, 1L), w %in% 1:4, ),
             bor3(ghl %between% c(-1L, 1L), w %in% 1:4, ))
expect_equal(or3s(ghm %between% c(-1L, 1L), r %in% 1:4, ),
             bor3(ghm %between% c(-1L, 1L), r %in% 1:4, ))
expect_equal(or3s(ghn %between% c(-1L, 1L), t %in% 1:4, j < 0L),
             bor3(ghn %between% c(-1L, 1L), t %in% 1:4, j < 0L))
expect_equal(or3s(gho %between% c(-1L, 1L), w %in% 1:4, o < 1L),
             bor3(gho %between% c(-1L, 1L), w %in% 1:4, o < 1L))
expect_equal(or3s(ghp %between% c(-1L, 1L), x %in% 1:4, f < 9L),
             bor3(ghp %between% c(-1L, 1L), x %in% 1:4, f < 9L))
expect_equal(or3s(ghq %between% c(-1L, 1L), , ),
             bor3(ghq %between% c(-1L, 1L), , ))
expect_equal(or3s(ghr %between% c(-1L, 1L), , ),
             bor3(ghr %between% c(-1L, 1L), , ))
expect_equal(or3s(ghs %between% c(-1L, 1L), , ),
             bor3(ghs %between% c(-1L, 1L), , ))
expect_equal(or3s(ght %between% c(-1L, 1L), , g <= 0L),
             bor3(ght %between% c(-1L, 1L), , g <= 0L))
expect_equal(or3s(ghu %between% c(-1L, 1L), , r <= 1L),
             bor3(ghu %between% c(-1L, 1L), , r <= 1L))
expect_equal(or3s(ghv %between% c(-1L, 1L), , b <= 9L),
             bor3(ghv %between% c(-1L, 1L), , b <= 9L))
expect_equal(or3s(ghw %between% c(-1L, 1L), w %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ghw %between% c(-1L, 1L), w %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ghx %between% c(-1L, 1L), m %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ghx %between% c(-1L, 1L), m %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ghy %between% c(-1L, 1L), k %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ghy %between% c(-1L, 1L), k %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ghz %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), b <= 0L),
             bor3(ghz %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), b <= 0L))
expect_equal(or3s(gia %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L), b <= 1L),
             bor3(gia %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L), b <= 1L))
expect_equal(or3s(gib %between% c(-1L, 1L), m %in% c(4L, -3L, 2L, 7L, 8L), d <= 9L),
             bor3(gib %between% c(-1L, 1L), m %in% c(4L, -3L, 2L, 7L, 8L), d <= 9L))
expect_equal(or3s(gic %between% c(-1L, 1L), , ),
             bor3(gic %between% c(-1L, 1L), , ))
expect_equal(or3s(gid %between% c(-1L, 1L), , ),
             bor3(gid %between% c(-1L, 1L), , ))
expect_equal(or3s(gie %between% c(-1L, 1L), , ),
             bor3(gie %between% c(-1L, 1L), , ))
expect_equal(or3s(gif %between% c(-1L, 1L), , w <= 0L),
             bor3(gif %between% c(-1L, 1L), , w <= 0L))
expect_equal(or3s(gig %between% c(-1L, 1L), , o <= 1L),
             bor3(gig %between% c(-1L, 1L), , o <= 1L))
expect_equal(or3s(gih %between% c(-1L, 1L), , d <= 9L),
             bor3(gih %between% c(-1L, 1L), , d <= 9L))
expect_equal(or3s(gii %between% c(-1L, 1L), l %in% 1:4, ),
             bor3(gii %between% c(-1L, 1L), l %in% 1:4, ))
expect_equal(or3s(gij %between% c(-1L, 1L), p %in% 1:4, ),
             bor3(gij %between% c(-1L, 1L), p %in% 1:4, ))
expect_equal(or3s(gik %between% c(-1L, 1L), k %in% 1:4, ),
             bor3(gik %between% c(-1L, 1L), k %in% 1:4, ))
expect_equal(or3s(gil %between% c(-1L, 1L), h %in% 1:4, s <= 0L),
             bor3(gil %between% c(-1L, 1L), h %in% 1:4, s <= 0L))
expect_equal(or3s(gim %between% c(-1L, 1L), j %in% 1:4, t <= 1L),
             bor3(gim %between% c(-1L, 1L), j %in% 1:4, t <= 1L))
expect_equal(or3s(gin %between% c(-1L, 1L), p %in% 1:4, b <= 9L),
             bor3(gin %between% c(-1L, 1L), p %in% 1:4, b <= 9L))
expect_equal(or3s(gio %between% c(-1L, 1L), , ),
             bor3(gio %between% c(-1L, 1L), , ))
expect_equal(or3s(gip %between% c(-1L, 1L), , ),
             bor3(gip %between% c(-1L, 1L), , ))
expect_equal(or3s(giq %between% c(-1L, 1L), , ),
             bor3(giq %between% c(-1L, 1L), , ))
expect_equal(or3s(gir %between% c(-1L, 1L), , v == 0L),
             bor3(gir %between% c(-1L, 1L), , v == 0L))
expect_equal(or3s(gis %between% c(-1L, 1L), , b == 1L),
             bor3(gis %between% c(-1L, 1L), , b == 1L))
expect_equal(or3s(git %between% c(-1L, 1L), , f == 9L),
             bor3(git %between% c(-1L, 1L), , f == 9L))
expect_equal(or3s(giu %between% c(-1L, 1L), f %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(giu %between% c(-1L, 1L), f %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(giv %between% c(-1L, 1L), v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(giv %between% c(-1L, 1L), v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(giw %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(giw %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gix %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), u == 0L),
             bor3(gix %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), u == 0L))
expect_equal(or3s(giy %between% c(-1L, 1L), f %in% c(4L, -3L, 2L, 7L, 8L), m == 1L),
             bor3(giy %between% c(-1L, 1L), f %in% c(4L, -3L, 2L, 7L, 8L), m == 1L))
expect_equal(or3s(giz %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L), a == 9L),
             bor3(giz %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L), a == 9L))
expect_equal(or3s(gja %between% c(-1L, 1L), , ),
             bor3(gja %between% c(-1L, 1L), , ))
expect_equal(or3s(gjb %between% c(-1L, 1L), , ),
             bor3(gjb %between% c(-1L, 1L), , ))
expect_equal(or3s(gjc %between% c(-1L, 1L), , ),
             bor3(gjc %between% c(-1L, 1L), , ))
expect_equal(or3s(gjd %between% c(-1L, 1L), , r == 0L),
             bor3(gjd %between% c(-1L, 1L), , r == 0L))
expect_equal(or3s(gje %between% c(-1L, 1L), , n == 1L),
             bor3(gje %between% c(-1L, 1L), , n == 1L))
expect_equal(or3s(gjf %between% c(-1L, 1L), , c == 9L),
             bor3(gjf %between% c(-1L, 1L), , c == 9L))
expect_equal(or3s(gjg %between% c(-1L, 1L), f %in% 1:4, ),
             bor3(gjg %between% c(-1L, 1L), f %in% 1:4, ))
expect_equal(or3s(gjh %between% c(-1L, 1L), h %in% 1:4, ),
             bor3(gjh %between% c(-1L, 1L), h %in% 1:4, ))
expect_equal(or3s(gji %between% c(-1L, 1L), n %in% 1:4, ),
             bor3(gji %between% c(-1L, 1L), n %in% 1:4, ))
expect_equal(or3s(gjj %between% c(-1L, 1L), r %in% 1:4, g == 0L),
             bor3(gjj %between% c(-1L, 1L), r %in% 1:4, g == 0L))
expect_equal(or3s(gjk %between% c(-1L, 1L), m %in% 1:4, i == 1L),
             bor3(gjk %between% c(-1L, 1L), m %in% 1:4, i == 1L))
expect_equal(or3s(gjl %between% c(-1L, 1L), t %in% 1:4, e == 9L),
             bor3(gjl %between% c(-1L, 1L), t %in% 1:4, e == 9L))
expect_equal(or3s(gjm %between% c(-1L, 1L), , ),
             bor3(gjm %between% c(-1L, 1L), , ))
expect_equal(or3s(gjn %between% c(-1L, 1L), , ),
             bor3(gjn %between% c(-1L, 1L), , ))
expect_equal(or3s(gjo %between% c(-1L, 1L), , ),
             bor3(gjo %between% c(-1L, 1L), , ))
expect_equal(or3s(gjp %between% c(-1L, 1L), , v > 0L),
             bor3(gjp %between% c(-1L, 1L), , v > 0L))
expect_equal(or3s(gjq %between% c(-1L, 1L), , l > 1L),
             bor3(gjq %between% c(-1L, 1L), , l > 1L))
expect_equal(or3s(gjr %between% c(-1L, 1L), , u > 9L),
             bor3(gjr %between% c(-1L, 1L), , u > 9L))
expect_equal(or3s(gjs %between% c(-1L, 1L), z %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gjs %between% c(-1L, 1L), z %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gjt %between% c(-1L, 1L), i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gjt %between% c(-1L, 1L), i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gju %between% c(-1L, 1L), x %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gju %between% c(-1L, 1L), x %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gjv %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), v > 0L),
             bor3(gjv %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), v > 0L))
expect_equal(or3s(gjw %between% c(-1L, 1L), e %in% c(4L, -3L, 2L, 7L, 8L), e > 1L),
             bor3(gjw %between% c(-1L, 1L), e %in% c(4L, -3L, 2L, 7L, 8L), e > 1L))
expect_equal(or3s(gjx %between% c(-1L, 1L), d %in% c(4L, -3L, 2L, 7L, 8L), n > 9L),
             bor3(gjx %between% c(-1L, 1L), d %in% c(4L, -3L, 2L, 7L, 8L), n > 9L))
expect_equal(or3s(gjy %between% c(-1L, 1L), , ),
             bor3(gjy %between% c(-1L, 1L), , ))
expect_equal(or3s(gjz %between% c(-1L, 1L), , ),
             bor3(gjz %between% c(-1L, 1L), , ))
expect_equal(or3s(gka %between% c(-1L, 1L), , ),
             bor3(gka %between% c(-1L, 1L), , ))
expect_equal(or3s(gkb %between% c(-1L, 1L), , c > 0L),
             bor3(gkb %between% c(-1L, 1L), , c > 0L))
expect_equal(or3s(gkc %between% c(-1L, 1L), , j > 1L),
             bor3(gkc %between% c(-1L, 1L), , j > 1L))
expect_equal(or3s(gkd %between% c(-1L, 1L), , s > 9L),
             bor3(gkd %between% c(-1L, 1L), , s > 9L))
expect_equal(or3s(gke %between% c(-1L, 1L), d %in% 1:4, ),
             bor3(gke %between% c(-1L, 1L), d %in% 1:4, ))
expect_equal(or3s(gkf %between% c(-1L, 1L), t %in% 1:4, ),
             bor3(gkf %between% c(-1L, 1L), t %in% 1:4, ))
expect_equal(or3s(gkg %between% c(-1L, 1L), k %in% 1:4, ),
             bor3(gkg %between% c(-1L, 1L), k %in% 1:4, ))
expect_equal(or3s(gkh %between% c(-1L, 1L), t %in% 1:4, g > 0L),
             bor3(gkh %between% c(-1L, 1L), t %in% 1:4, g > 0L))
expect_equal(or3s(gki %between% c(-1L, 1L), z %in% 1:4, n > 1L),
             bor3(gki %between% c(-1L, 1L), z %in% 1:4, n > 1L))
expect_equal(or3s(gkj %between% c(-1L, 1L), p %in% 1:4, w > 9L),
             bor3(gkj %between% c(-1L, 1L), p %in% 1:4, w > 9L))
expect_equal(or3s(gkk %between% c(-1L, 1L), , ),
             bor3(gkk %between% c(-1L, 1L), , ))
expect_equal(or3s(gkl %between% c(-1L, 1L), , ),
             bor3(gkl %between% c(-1L, 1L), , ))
expect_equal(or3s(gkm %between% c(-1L, 1L), , ),
             bor3(gkm %between% c(-1L, 1L), , ))
expect_equal(or3s(gkn %between% c(-1L, 1L), , x >= 0L),
             bor3(gkn %between% c(-1L, 1L), , x >= 0L))
expect_equal(or3s(gko %between% c(-1L, 1L), , c >= 1L),
             bor3(gko %between% c(-1L, 1L), , c >= 1L))
expect_equal(or3s(gkp %between% c(-1L, 1L), , x >= 9L),
             bor3(gkp %between% c(-1L, 1L), , x >= 9L))
expect_equal(or3s(gkq %between% c(-1L, 1L), r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gkq %between% c(-1L, 1L), r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gkr %between% c(-1L, 1L), v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gkr %between% c(-1L, 1L), v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gks %between% c(-1L, 1L), o %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(gks %between% c(-1L, 1L), o %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(gkt %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), r >= 0L),
             bor3(gkt %between% c(-1L, 1L), y %in% c(4L, -3L, 2L, 7L, 8L), r >= 0L))
expect_equal(or3s(gku %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L), x >= 1L),
             bor3(gku %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L), x >= 1L))
expect_equal(or3s(gkv %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), f >= 9L),
             bor3(gkv %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L), f >= 9L))
expect_equal(or3s(gkw %between% c(-1L, 1L), , ),
             bor3(gkw %between% c(-1L, 1L), , ))
expect_equal(or3s(gkx %between% c(-1L, 1L), , ),
             bor3(gkx %between% c(-1L, 1L), , ))
expect_equal(or3s(gky %between% c(-1L, 1L), , ),
             bor3(gky %between% c(-1L, 1L), , ))
expect_equal(or3s(gkz %between% c(-1L, 1L), , j >= 0L),
             bor3(gkz %between% c(-1L, 1L), , j >= 0L))
expect_equal(or3s(gla %between% c(-1L, 1L), , r >= 1L),
             bor3(gla %between% c(-1L, 1L), , r >= 1L))
expect_equal(or3s(glb %between% c(-1L, 1L), , h >= 9L),
             bor3(glb %between% c(-1L, 1L), , h >= 9L))
expect_equal(or3s(glc %between% c(-1L, 1L), m %in% 1:4, ),
             bor3(glc %between% c(-1L, 1L), m %in% 1:4, ))
expect_equal(or3s(gld %between% c(-1L, 1L), o %in% 1:4, ),
             bor3(gld %between% c(-1L, 1L), o %in% 1:4, ))
expect_equal(or3s(gle %between% c(-1L, 1L), x %in% 1:4, ),
             bor3(gle %between% c(-1L, 1L), x %in% 1:4, ))
expect_equal(or3s(glf %between% c(-1L, 1L), d %in% 1:4, r >= 0L),
             bor3(glf %between% c(-1L, 1L), d %in% 1:4, r >= 0L))
expect_equal(or3s(glg %between% c(-1L, 1L), z %in% 1:4, p >= 1L),
             bor3(glg %between% c(-1L, 1L), z %in% 1:4, p >= 1L))
expect_equal(or3s(glh %between% c(-1L, 1L), f %in% 1:4, a >= 9L),
             bor3(glh %between% c(-1L, 1L), f %in% 1:4, a >= 9L))
expect_equal(or3s(gli %between% c(-1L, 1L), , ),
             bor3(gli %between% c(-1L, 1L), , ))
expect_equal(or3s(glj %between% c(-1L, 1L), , ),
             bor3(glj %between% c(-1L, 1L), , ))
expect_equal(or3s(glk %between% c(-1L, 1L), , ),
             bor3(glk %between% c(-1L, 1L), , ))
expect_equal(or3s(gll %between% c(-1L, 1L), , logi_f),
             bor3(gll %between% c(-1L, 1L), , logi_f))
expect_equal(or3s(glm %between% c(-1L, 1L), , logi_s),
             bor3(glm %between% c(-1L, 1L), , logi_s))
expect_equal(or3s(gln %between% c(-1L, 1L), , logi_s),
             bor3(gln %between% c(-1L, 1L), , logi_s))
expect_equal(or3s(glo %between% c(-1L, 1L), b < 0L, ),
             bor3(glo %between% c(-1L, 1L), b < 0L, ))
expect_equal(or3s(glp %between% c(-1L, 1L), l < 1L, ),
             bor3(glp %between% c(-1L, 1L), l < 1L, ))
expect_equal(or3s(glq %between% c(-1L, 1L), e < 9L, ),
             bor3(glq %between% c(-1L, 1L), e < 9L, ))
expect_equal(or3s(glr %between% c(-1L, 1L), c < 0L, logi_d),
             bor3(glr %between% c(-1L, 1L), c < 0L, logi_d))
expect_equal(or3s(gls %between% c(-1L, 1L), n < 1L, logi_x),
             bor3(gls %between% c(-1L, 1L), n < 1L, logi_x))
expect_equal(or3s(glt %between% c(-1L, 1L), s < 9L, logi_k),
             bor3(glt %between% c(-1L, 1L), s < 9L, logi_k))
expect_equal(or3s(glu %between% c(-1L, 1L), , ),
             bor3(glu %between% c(-1L, 1L), , ))
expect_equal(or3s(glv %between% c(-1L, 1L), , ),
             bor3(glv %between% c(-1L, 1L), , ))
expect_equal(or3s(glw %between% c(-1L, 1L), , ),
             bor3(glw %between% c(-1L, 1L), , ))
expect_equal(or3s(glx %between% c(-1L, 1L), , !logi_c),
             bor3(glx %between% c(-1L, 1L), , !logi_c))
expect_equal(or3s(gly %between% c(-1L, 1L), , !logi_w),
             bor3(gly %between% c(-1L, 1L), , !logi_w))
expect_equal(or3s(glz %between% c(-1L, 1L), , !logi_a),
             bor3(glz %between% c(-1L, 1L), , !logi_a))
expect_equal(or3s(gma %between% c(-1L, 1L), o < 0L, ),
             bor3(gma %between% c(-1L, 1L), o < 0L, ))
expect_equal(or3s(gmb %between% c(-1L, 1L), u < 1L, ),
             bor3(gmb %between% c(-1L, 1L), u < 1L, ))
expect_equal(or3s(gmc %between% c(-1L, 1L), p < 9L, ),
             bor3(gmc %between% c(-1L, 1L), p < 9L, ))
expect_equal(or3s(gmd %between% c(-1L, 1L), v < 0L, !logi_u),
             bor3(gmd %between% c(-1L, 1L), v < 0L, !logi_u))
expect_equal(or3s(gme %between% c(-1L, 1L), v < 1L, !logi_e),
             bor3(gme %between% c(-1L, 1L), v < 1L, !logi_e))
expect_equal(or3s(gmf %between% c(-1L, 1L), s < 9L, !logi_r),
             bor3(gmf %between% c(-1L, 1L), s < 9L, !logi_r))
expect_equal(or3s(gmg %between% c(-1L, 1L), , ),
             bor3(gmg %between% c(-1L, 1L), , ))
expect_equal(or3s(gmh %between% c(-1L, 1L), , ),
             bor3(gmh %between% c(-1L, 1L), , ))
expect_equal(or3s(gmi %between% c(-1L, 1L), , ),
             bor3(gmi %between% c(-1L, 1L), , ))
expect_equal(or3s(gmj %between% c(-1L, 1L), , o != 0L),
             bor3(gmj %between% c(-1L, 1L), , o != 0L))
expect_equal(or3s(gmk %between% c(-1L, 1L), , b != 1L),
             bor3(gmk %between% c(-1L, 1L), , b != 1L))
expect_equal(or3s(gml %between% c(-1L, 1L), , m != 9L),
             bor3(gml %between% c(-1L, 1L), , m != 9L))
expect_equal(or3s(gmm %between% c(-1L, 1L), o < 0L, ),
             bor3(gmm %between% c(-1L, 1L), o < 0L, ))
expect_equal(or3s(gmn %between% c(-1L, 1L), j < 1L, ),
             bor3(gmn %between% c(-1L, 1L), j < 1L, ))
expect_equal(or3s(gmo %between% c(-1L, 1L), o < 9L, ),
             bor3(gmo %between% c(-1L, 1L), o < 9L, ))
expect_equal(or3s(gmp %between% c(-1L, 1L), w < 0L, x != 0L),
             bor3(gmp %between% c(-1L, 1L), w < 0L, x != 0L))
expect_equal(or3s(gmq %between% c(-1L, 1L), h < 1L, u != 1L),
             bor3(gmq %between% c(-1L, 1L), h < 1L, u != 1L))
expect_equal(or3s(gmr %between% c(-1L, 1L), c < 9L, c != 9L),
             bor3(gmr %between% c(-1L, 1L), c < 9L, c != 9L))
expect_equal(or3s(gms %between% c(-1L, 1L), , ),
             bor3(gms %between% c(-1L, 1L), , ))
expect_equal(or3s(gmt %between% c(-1L, 1L), , ),
             bor3(gmt %between% c(-1L, 1L), , ))
expect_equal(or3s(gmu %between% c(-1L, 1L), , ),
             bor3(gmu %between% c(-1L, 1L), , ))
expect_equal(or3s(gmv %between% c(-1L, 1L), , d %between% c(-1L, 1L)),
             bor3(gmv %between% c(-1L, 1L), , d %between% c(-1L, 1L)))
expect_equal(or3s(gmw %between% c(-1L, 1L), , b %between% c(-1L, 1L)),
             bor3(gmw %between% c(-1L, 1L), , b %between% c(-1L, 1L)))
expect_equal(or3s(gmx %between% c(-1L, 1L), , l %between% c(-1L, 1L)),
             bor3(gmx %between% c(-1L, 1L), , l %between% c(-1L, 1L)))
expect_equal(or3s(gmy %between% c(-1L, 1L), o < 0L, ),
             bor3(gmy %between% c(-1L, 1L), o < 0L, ))
expect_equal(or3s(gmz %between% c(-1L, 1L), w < 1L, ),
             bor3(gmz %between% c(-1L, 1L), w < 1L, ))
expect_equal(or3s(gna %between% c(-1L, 1L), u < 9L, ),
             bor3(gna %between% c(-1L, 1L), u < 9L, ))
expect_equal(or3s(gnb %between% c(-1L, 1L), x < 0L, b %between% c(-1L, 1L)),
             bor3(gnb %between% c(-1L, 1L), x < 0L, b %between% c(-1L, 1L)))
expect_equal(or3s(gnc %between% c(-1L, 1L), t < 1L, e %between% c(-1L, 1L)),
             bor3(gnc %between% c(-1L, 1L), t < 1L, e %between% c(-1L, 1L)))
expect_equal(or3s(gnd %between% c(-1L, 1L), w < 9L, n %between% c(-1L, 1L)),
             bor3(gnd %between% c(-1L, 1L), w < 9L, n %between% c(-1L, 1L)))
expect_equal(or3s(gne %between% c(-1L, 1L), , ),
             bor3(gne %between% c(-1L, 1L), , ))
expect_equal(or3s(gnf %between% c(-1L, 1L), , ),
             bor3(gnf %between% c(-1L, 1L), , ))
expect_equal(or3s(gng %between% c(-1L, 1L), , ),
             bor3(gng %between% c(-1L, 1L), , ))
expect_equal(or3s(gnh %between% c(-1L, 1L), , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gnh %between% c(-1L, 1L), , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gni %between% c(-1L, 1L), , m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gni %between% c(-1L, 1L), , m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gnj %between% c(-1L, 1L), , f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gnj %between% c(-1L, 1L), , f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gnk %between% c(-1L, 1L), x < 0L, ),
             bor3(gnk %between% c(-1L, 1L), x < 0L, ))
expect_equal(or3s(gnl %between% c(-1L, 1L), n < 1L, ),
             bor3(gnl %between% c(-1L, 1L), n < 1L, ))
expect_equal(or3s(gnm %between% c(-1L, 1L), r < 9L, ),
             bor3(gnm %between% c(-1L, 1L), r < 9L, ))
expect_equal(or3s(gnn %between% c(-1L, 1L), z < 0L, m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gnn %between% c(-1L, 1L), z < 0L, m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gno %between% c(-1L, 1L), d < 1L, n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gno %between% c(-1L, 1L), d < 1L, n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gnp %between% c(-1L, 1L), s < 9L, a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gnp %between% c(-1L, 1L), s < 9L, a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gnq %between% c(-1L, 1L), , ),
             bor3(gnq %between% c(-1L, 1L), , ))
expect_equal(or3s(gnr %between% c(-1L, 1L), , ),
             bor3(gnr %between% c(-1L, 1L), , ))
expect_equal(or3s(gns %between% c(-1L, 1L), , ),
             bor3(gns %between% c(-1L, 1L), , ))
expect_equal(or3s(gnt %between% c(-1L, 1L), , r %in% 1:4),
             bor3(gnt %between% c(-1L, 1L), , r %in% 1:4))
expect_equal(or3s(gnu %between% c(-1L, 1L), , s %in% 1:4),
             bor3(gnu %between% c(-1L, 1L), , s %in% 1:4))
expect_equal(or3s(gnv %between% c(-1L, 1L), , f %in% 1:4),
             bor3(gnv %between% c(-1L, 1L), , f %in% 1:4))
expect_equal(or3s(gnw %between% c(-1L, 1L), d < 0L, ),
             bor3(gnw %between% c(-1L, 1L), d < 0L, ))
expect_equal(or3s(gnx %between% c(-1L, 1L), n < 1L, ),
             bor3(gnx %between% c(-1L, 1L), n < 1L, ))
expect_equal(or3s(gny %between% c(-1L, 1L), c < 9L, ),
             bor3(gny %between% c(-1L, 1L), c < 9L, ))
expect_equal(or3s(gnz %between% c(-1L, 1L), u < 0L, a %in% 1:4),
             bor3(gnz %between% c(-1L, 1L), u < 0L, a %in% 1:4))
expect_equal(or3s(goa %between% c(-1L, 1L), p < 1L, q %in% 1:4),
             bor3(goa %between% c(-1L, 1L), p < 1L, q %in% 1:4))
expect_equal(or3s(gob %between% c(-1L, 1L), f < 9L, x %in% 1:4),
             bor3(gob %between% c(-1L, 1L), f < 9L, x %in% 1:4))
expect_equal(or3s(goc %between% c(-1L, 1L), , ),
             bor3(goc %between% c(-1L, 1L), , ))
expect_equal(or3s(god %between% c(-1L, 1L), , ),
             bor3(god %between% c(-1L, 1L), , ))
expect_equal(or3s(goe %between% c(-1L, 1L), , ),
             bor3(goe %between% c(-1L, 1L), , ))
expect_equal(or3s(gof %between% c(-1L, 1L), , n < 0L),
             bor3(gof %between% c(-1L, 1L), , n < 0L))
expect_equal(or3s(gog %between% c(-1L, 1L), , i < 1L),
             bor3(gog %between% c(-1L, 1L), , i < 1L))
expect_equal(or3s(goh %between% c(-1L, 1L), , o < 9L),
             bor3(goh %between% c(-1L, 1L), , o < 9L))
expect_equal(or3s(goi %between% c(-1L, 1L), u < 0L, ),
             bor3(goi %between% c(-1L, 1L), u < 0L, ))
expect_equal(or3s(goj %between% c(-1L, 1L), w < 1L, ),
             bor3(goj %between% c(-1L, 1L), w < 1L, ))
expect_equal(or3s(gok %between% c(-1L, 1L), l < 9L, ),
             bor3(gok %between% c(-1L, 1L), l < 9L, ))
expect_equal(or3s(gol %between% c(-1L, 1L), f < 0L, g < 0L),
             bor3(gol %between% c(-1L, 1L), f < 0L, g < 0L))
expect_equal(or3s(gom %between% c(-1L, 1L), f < 1L, h < 1L),
             bor3(gom %between% c(-1L, 1L), f < 1L, h < 1L))
expect_equal(or3s(gon %between% c(-1L, 1L), k < 9L, z < 9L),
             bor3(gon %between% c(-1L, 1L), k < 9L, z < 9L))
expect_equal(or3s(goo %between% c(-1L, 1L), , ),
             bor3(goo %between% c(-1L, 1L), , ))
expect_equal(or3s(gop %between% c(-1L, 1L), , ),
             bor3(gop %between% c(-1L, 1L), , ))
expect_equal(or3s(goq %between% c(-1L, 1L), , ),
             bor3(goq %between% c(-1L, 1L), , ))
expect_equal(or3s(gor %between% c(-1L, 1L), , d <= 0L),
             bor3(gor %between% c(-1L, 1L), , d <= 0L))
expect_equal(or3s(gos %between% c(-1L, 1L), , b <= 1L),
             bor3(gos %between% c(-1L, 1L), , b <= 1L))
expect_equal(or3s(got %between% c(-1L, 1L), , y <= 9L),
             bor3(got %between% c(-1L, 1L), , y <= 9L))
expect_equal(or3s(gou %between% c(-1L, 1L), h < 0L, ),
             bor3(gou %between% c(-1L, 1L), h < 0L, ))
expect_equal(or3s(gov %between% c(-1L, 1L), b < 1L, ),
             bor3(gov %between% c(-1L, 1L), b < 1L, ))
expect_equal(or3s(gow %between% c(-1L, 1L), s < 9L, ),
             bor3(gow %between% c(-1L, 1L), s < 9L, ))
expect_equal(or3s(gox %between% c(-1L, 1L), g < 0L, n <= 0L),
             bor3(gox %between% c(-1L, 1L), g < 0L, n <= 0L))
expect_equal(or3s(goy %between% c(-1L, 1L), p < 1L, k <= 1L),
             bor3(goy %between% c(-1L, 1L), p < 1L, k <= 1L))
expect_equal(or3s(goz %between% c(-1L, 1L), l < 9L, v <= 9L),
             bor3(goz %between% c(-1L, 1L), l < 9L, v <= 9L))
expect_equal(or3s(gpa %between% c(-1L, 1L), , ),
             bor3(gpa %between% c(-1L, 1L), , ))
expect_equal(or3s(gpb %between% c(-1L, 1L), , ),
             bor3(gpb %between% c(-1L, 1L), , ))
expect_equal(or3s(gpc %between% c(-1L, 1L), , ),
             bor3(gpc %between% c(-1L, 1L), , ))
expect_equal(or3s(gpd %between% c(-1L, 1L), , f == 0L),
             bor3(gpd %between% c(-1L, 1L), , f == 0L))
expect_equal(or3s(gpe %between% c(-1L, 1L), , n == 1L),
             bor3(gpe %between% c(-1L, 1L), , n == 1L))
expect_equal(or3s(gpf %between% c(-1L, 1L), , g == 9L),
             bor3(gpf %between% c(-1L, 1L), , g == 9L))
expect_equal(or3s(gpg %between% c(-1L, 1L), x < 0L, ),
             bor3(gpg %between% c(-1L, 1L), x < 0L, ))
expect_equal(or3s(gph %between% c(-1L, 1L), t < 1L, ),
             bor3(gph %between% c(-1L, 1L), t < 1L, ))
expect_equal(or3s(gpi %between% c(-1L, 1L), y < 9L, ),
             bor3(gpi %between% c(-1L, 1L), y < 9L, ))
expect_equal(or3s(gpj %between% c(-1L, 1L), y < 0L, g == 0L),
             bor3(gpj %between% c(-1L, 1L), y < 0L, g == 0L))
expect_equal(or3s(gpk %between% c(-1L, 1L), y < 1L, x == 1L),
             bor3(gpk %between% c(-1L, 1L), y < 1L, x == 1L))
expect_equal(or3s(gpl %between% c(-1L, 1L), m < 9L, z == 9L),
             bor3(gpl %between% c(-1L, 1L), m < 9L, z == 9L))
expect_equal(or3s(gpm %between% c(-1L, 1L), , ),
             bor3(gpm %between% c(-1L, 1L), , ))
expect_equal(or3s(gpn %between% c(-1L, 1L), , ),
             bor3(gpn %between% c(-1L, 1L), , ))
expect_equal(or3s(gpo %between% c(-1L, 1L), , ),
             bor3(gpo %between% c(-1L, 1L), , ))
expect_equal(or3s(gpp %between% c(-1L, 1L), , k > 0L),
             bor3(gpp %between% c(-1L, 1L), , k > 0L))
expect_equal(or3s(gpq %between% c(-1L, 1L), , n > 1L),
             bor3(gpq %between% c(-1L, 1L), , n > 1L))
expect_equal(or3s(gpr %between% c(-1L, 1L), , q > 9L),
             bor3(gpr %between% c(-1L, 1L), , q > 9L))
expect_equal(or3s(gps %between% c(-1L, 1L), s < 0L, ),
             bor3(gps %between% c(-1L, 1L), s < 0L, ))
expect_equal(or3s(gpt %between% c(-1L, 1L), d < 1L, ),
             bor3(gpt %between% c(-1L, 1L), d < 1L, ))
expect_equal(or3s(gpu %between% c(-1L, 1L), n < 9L, ),
             bor3(gpu %between% c(-1L, 1L), n < 9L, ))
expect_equal(or3s(gpv %between% c(-1L, 1L), x < 0L, j > 0L),
             bor3(gpv %between% c(-1L, 1L), x < 0L, j > 0L))
expect_equal(or3s(gpw %between% c(-1L, 1L), x < 1L, h > 1L),
             bor3(gpw %between% c(-1L, 1L), x < 1L, h > 1L))
expect_equal(or3s(gpx %between% c(-1L, 1L), z < 9L, z > 9L),
             bor3(gpx %between% c(-1L, 1L), z < 9L, z > 9L))
expect_equal(or3s(gpy %between% c(-1L, 1L), , ),
             bor3(gpy %between% c(-1L, 1L), , ))
expect_equal(or3s(gpz %between% c(-1L, 1L), , ),
             bor3(gpz %between% c(-1L, 1L), , ))
expect_equal(or3s(gqa %between% c(-1L, 1L), , ),
             bor3(gqa %between% c(-1L, 1L), , ))
expect_equal(or3s(gqb %between% c(-1L, 1L), , f >= 0L),
             bor3(gqb %between% c(-1L, 1L), , f >= 0L))
expect_equal(or3s(gqc %between% c(-1L, 1L), , o >= 1L),
             bor3(gqc %between% c(-1L, 1L), , o >= 1L))
expect_equal(or3s(gqd %between% c(-1L, 1L), , v >= 9L),
             bor3(gqd %between% c(-1L, 1L), , v >= 9L))
expect_equal(or3s(gqe %between% c(-1L, 1L), h < 0L, ),
             bor3(gqe %between% c(-1L, 1L), h < 0L, ))
expect_equal(or3s(gqf %between% c(-1L, 1L), i < 1L, ),
             bor3(gqf %between% c(-1L, 1L), i < 1L, ))
expect_equal(or3s(gqg %between% c(-1L, 1L), b < 9L, ),
             bor3(gqg %between% c(-1L, 1L), b < 9L, ))
expect_equal(or3s(gqh %between% c(-1L, 1L), x < 0L, u >= 0L),
             bor3(gqh %between% c(-1L, 1L), x < 0L, u >= 0L))
expect_equal(or3s(gqi %between% c(-1L, 1L), u < 1L, r >= 1L),
             bor3(gqi %between% c(-1L, 1L), u < 1L, r >= 1L))
expect_equal(or3s(gqj %between% c(-1L, 1L), z < 9L, i >= 9L),
             bor3(gqj %between% c(-1L, 1L), z < 9L, i >= 9L))
expect_equal(or3s(gqk %between% c(-1L, 1L), , ),
             bor3(gqk %between% c(-1L, 1L), , ))
expect_equal(or3s(gql %between% c(-1L, 1L), , ),
             bor3(gql %between% c(-1L, 1L), , ))
expect_equal(or3s(gqm %between% c(-1L, 1L), , ),
             bor3(gqm %between% c(-1L, 1L), , ))
expect_equal(or3s(gqn %between% c(-1L, 1L), , logi_k),
             bor3(gqn %between% c(-1L, 1L), , logi_k))
expect_equal(or3s(gqo %between% c(-1L, 1L), , logi_u),
             bor3(gqo %between% c(-1L, 1L), , logi_u))
expect_equal(or3s(gqp %between% c(-1L, 1L), , logi_l),
             bor3(gqp %between% c(-1L, 1L), , logi_l))
expect_equal(or3s(gqq %between% c(-1L, 1L), w <= 0L, ),
             bor3(gqq %between% c(-1L, 1L), w <= 0L, ))
expect_equal(or3s(gqr %between% c(-1L, 1L), b <= 1L, ),
             bor3(gqr %between% c(-1L, 1L), b <= 1L, ))
expect_equal(or3s(gqs %between% c(-1L, 1L), v <= 9L, ),
             bor3(gqs %between% c(-1L, 1L), v <= 9L, ))
expect_equal(or3s(gqt %between% c(-1L, 1L), m <= 0L, logi_i),
             bor3(gqt %between% c(-1L, 1L), m <= 0L, logi_i))
expect_equal(or3s(gqu %between% c(-1L, 1L), f <= 1L, logi_z),
             bor3(gqu %between% c(-1L, 1L), f <= 1L, logi_z))
expect_equal(or3s(gqv %between% c(-1L, 1L), e <= 9L, logi_a),
             bor3(gqv %between% c(-1L, 1L), e <= 9L, logi_a))
expect_equal(or3s(gqw %between% c(-1L, 1L), , ),
             bor3(gqw %between% c(-1L, 1L), , ))
expect_equal(or3s(gqx %between% c(-1L, 1L), , ),
             bor3(gqx %between% c(-1L, 1L), , ))
expect_equal(or3s(gqy %between% c(-1L, 1L), , ),
             bor3(gqy %between% c(-1L, 1L), , ))
expect_equal(or3s(gqz %between% c(-1L, 1L), , !logi_h),
             bor3(gqz %between% c(-1L, 1L), , !logi_h))
expect_equal(or3s(gra %between% c(-1L, 1L), , !logi_m),
             bor3(gra %between% c(-1L, 1L), , !logi_m))
expect_equal(or3s(grb %between% c(-1L, 1L), , !logi_k),
             bor3(grb %between% c(-1L, 1L), , !logi_k))
expect_equal(or3s(grc %between% c(-1L, 1L), o <= 0L, ),
             bor3(grc %between% c(-1L, 1L), o <= 0L, ))
expect_equal(or3s(grd %between% c(-1L, 1L), u <= 1L, ),
             bor3(grd %between% c(-1L, 1L), u <= 1L, ))
expect_equal(or3s(gre %between% c(-1L, 1L), n <= 9L, ),
             bor3(gre %between% c(-1L, 1L), n <= 9L, ))
expect_equal(or3s(grf %between% c(-1L, 1L), r <= 0L, !logi_y),
             bor3(grf %between% c(-1L, 1L), r <= 0L, !logi_y))
expect_equal(or3s(grg %between% c(-1L, 1L), s <= 1L, !logi_z),
             bor3(grg %between% c(-1L, 1L), s <= 1L, !logi_z))
expect_equal(or3s(grh %between% c(-1L, 1L), n <= 9L, !logi_i),
             bor3(grh %between% c(-1L, 1L), n <= 9L, !logi_i))
expect_equal(or3s(gri %between% c(-1L, 1L), , ),
             bor3(gri %between% c(-1L, 1L), , ))
expect_equal(or3s(grj %between% c(-1L, 1L), , ),
             bor3(grj %between% c(-1L, 1L), , ))
expect_equal(or3s(grk %between% c(-1L, 1L), , ),
             bor3(grk %between% c(-1L, 1L), , ))
expect_equal(or3s(grl %between% c(-1L, 1L), , s != 0L),
             bor3(grl %between% c(-1L, 1L), , s != 0L))
expect_equal(or3s(grm %between% c(-1L, 1L), , v != 1L),
             bor3(grm %between% c(-1L, 1L), , v != 1L))
expect_equal(or3s(grn %between% c(-1L, 1L), , i != 9L),
             bor3(grn %between% c(-1L, 1L), , i != 9L))
expect_equal(or3s(gro %between% c(-1L, 1L), m <= 0L, ),
             bor3(gro %between% c(-1L, 1L), m <= 0L, ))
expect_equal(or3s(grp %between% c(-1L, 1L), b <= 1L, ),
             bor3(grp %between% c(-1L, 1L), b <= 1L, ))
expect_equal(or3s(grq %between% c(-1L, 1L), r <= 9L, ),
             bor3(grq %between% c(-1L, 1L), r <= 9L, ))
expect_equal(or3s(grr %between% c(-1L, 1L), a <= 0L, s != 0L),
             bor3(grr %between% c(-1L, 1L), a <= 0L, s != 0L))
expect_equal(or3s(grs %between% c(-1L, 1L), j <= 1L, q != 1L),
             bor3(grs %between% c(-1L, 1L), j <= 1L, q != 1L))
expect_equal(or3s(grt %between% c(-1L, 1L), t <= 9L, p != 9L),
             bor3(grt %between% c(-1L, 1L), t <= 9L, p != 9L))
expect_equal(or3s(gru %between% c(-1L, 1L), , ),
             bor3(gru %between% c(-1L, 1L), , ))
expect_equal(or3s(grv %between% c(-1L, 1L), , ),
             bor3(grv %between% c(-1L, 1L), , ))
expect_equal(or3s(grw %between% c(-1L, 1L), , ),
             bor3(grw %between% c(-1L, 1L), , ))
expect_equal(or3s(grx %between% c(-1L, 1L), , a %between% c(-1L, 1L)),
             bor3(grx %between% c(-1L, 1L), , a %between% c(-1L, 1L)))
expect_equal(or3s(gry %between% c(-1L, 1L), , p %between% c(-1L, 1L)),
             bor3(gry %between% c(-1L, 1L), , p %between% c(-1L, 1L)))
expect_equal(or3s(grz %between% c(-1L, 1L), , q %between% c(-1L, 1L)),
             bor3(grz %between% c(-1L, 1L), , q %between% c(-1L, 1L)))
expect_equal(or3s(gsa %between% c(-1L, 1L), u <= 0L, ),
             bor3(gsa %between% c(-1L, 1L), u <= 0L, ))
expect_equal(or3s(gsb %between% c(-1L, 1L), a <= 1L, ),
             bor3(gsb %between% c(-1L, 1L), a <= 1L, ))
expect_equal(or3s(gsc %between% c(-1L, 1L), u <= 9L, ),
             bor3(gsc %between% c(-1L, 1L), u <= 9L, ))
expect_equal(or3s(gsd %between% c(-1L, 1L), r <= 0L, w %between% c(-1L, 1L)),
             bor3(gsd %between% c(-1L, 1L), r <= 0L, w %between% c(-1L, 1L)))
expect_equal(or3s(gse %between% c(-1L, 1L), i <= 1L, w %between% c(-1L, 1L)),
             bor3(gse %between% c(-1L, 1L), i <= 1L, w %between% c(-1L, 1L)))
expect_equal(or3s(gsf %between% c(-1L, 1L), g <= 9L, b %between% c(-1L, 1L)),
             bor3(gsf %between% c(-1L, 1L), g <= 9L, b %between% c(-1L, 1L)))
expect_equal(or3s(gsg %between% c(-1L, 1L), , ),
             bor3(gsg %between% c(-1L, 1L), , ))
expect_equal(or3s(gsh %between% c(-1L, 1L), , ),
             bor3(gsh %between% c(-1L, 1L), , ))
expect_equal(or3s(gsi %between% c(-1L, 1L), , ),
             bor3(gsi %between% c(-1L, 1L), , ))
expect_equal(or3s(gsj %between% c(-1L, 1L), , f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gsj %between% c(-1L, 1L), , f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gsk %between% c(-1L, 1L), , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gsk %between% c(-1L, 1L), , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gsl %between% c(-1L, 1L), , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gsl %between% c(-1L, 1L), , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gsm %between% c(-1L, 1L), m <= 0L, ),
             bor3(gsm %between% c(-1L, 1L), m <= 0L, ))
expect_equal(or3s(gsn %between% c(-1L, 1L), r <= 1L, ),
             bor3(gsn %between% c(-1L, 1L), r <= 1L, ))
expect_equal(or3s(gso %between% c(-1L, 1L), e <= 9L, ),
             bor3(gso %between% c(-1L, 1L), e <= 9L, ))
expect_equal(or3s(gsp %between% c(-1L, 1L), v <= 0L, e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gsp %between% c(-1L, 1L), v <= 0L, e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gsq %between% c(-1L, 1L), n <= 1L, d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gsq %between% c(-1L, 1L), n <= 1L, d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gsr %between% c(-1L, 1L), d <= 9L, i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gsr %between% c(-1L, 1L), d <= 9L, i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gss %between% c(-1L, 1L), , ),
             bor3(gss %between% c(-1L, 1L), , ))
expect_equal(or3s(gst %between% c(-1L, 1L), , ),
             bor3(gst %between% c(-1L, 1L), , ))
expect_equal(or3s(gsu %between% c(-1L, 1L), , ),
             bor3(gsu %between% c(-1L, 1L), , ))
expect_equal(or3s(gsv %between% c(-1L, 1L), , a %in% 1:4),
             bor3(gsv %between% c(-1L, 1L), , a %in% 1:4))
expect_equal(or3s(gsw %between% c(-1L, 1L), , f %in% 1:4),
             bor3(gsw %between% c(-1L, 1L), , f %in% 1:4))
expect_equal(or3s(gsx %between% c(-1L, 1L), , x %in% 1:4),
             bor3(gsx %between% c(-1L, 1L), , x %in% 1:4))
expect_equal(or3s(gsy %between% c(-1L, 1L), o <= 0L, ),
             bor3(gsy %between% c(-1L, 1L), o <= 0L, ))
expect_equal(or3s(gsz %between% c(-1L, 1L), k <= 1L, ),
             bor3(gsz %between% c(-1L, 1L), k <= 1L, ))
expect_equal(or3s(gta %between% c(-1L, 1L), k <= 9L, ),
             bor3(gta %between% c(-1L, 1L), k <= 9L, ))
expect_equal(or3s(gtb %between% c(-1L, 1L), i <= 0L, a %in% 1:4),
             bor3(gtb %between% c(-1L, 1L), i <= 0L, a %in% 1:4))
expect_equal(or3s(gtc %between% c(-1L, 1L), w <= 1L, g %in% 1:4),
             bor3(gtc %between% c(-1L, 1L), w <= 1L, g %in% 1:4))
expect_equal(or3s(gtd %between% c(-1L, 1L), v <= 9L, h %in% 1:4),
             bor3(gtd %between% c(-1L, 1L), v <= 9L, h %in% 1:4))
expect_equal(or3s(gte %between% c(-1L, 1L), , ),
             bor3(gte %between% c(-1L, 1L), , ))
expect_equal(or3s(gtf %between% c(-1L, 1L), , ),
             bor3(gtf %between% c(-1L, 1L), , ))
expect_equal(or3s(gtg %between% c(-1L, 1L), , ),
             bor3(gtg %between% c(-1L, 1L), , ))
expect_equal(or3s(gth %between% c(-1L, 1L), , d < 0L),
             bor3(gth %between% c(-1L, 1L), , d < 0L))
expect_equal(or3s(gti %between% c(-1L, 1L), , p < 1L),
             bor3(gti %between% c(-1L, 1L), , p < 1L))
expect_equal(or3s(gtj %between% c(-1L, 1L), , c < 9L),
             bor3(gtj %between% c(-1L, 1L), , c < 9L))
expect_equal(or3s(gtk %between% c(-1L, 1L), f <= 0L, ),
             bor3(gtk %between% c(-1L, 1L), f <= 0L, ))
expect_equal(or3s(gtl %between% c(-1L, 1L), f <= 1L, ),
             bor3(gtl %between% c(-1L, 1L), f <= 1L, ))
expect_equal(or3s(gtm %between% c(-1L, 1L), q <= 9L, ),
             bor3(gtm %between% c(-1L, 1L), q <= 9L, ))
expect_equal(or3s(gtn %between% c(-1L, 1L), l <= 0L, x < 0L),
             bor3(gtn %between% c(-1L, 1L), l <= 0L, x < 0L))
expect_equal(or3s(gto %between% c(-1L, 1L), q <= 1L, p < 1L),
             bor3(gto %between% c(-1L, 1L), q <= 1L, p < 1L))
expect_equal(or3s(gtp %between% c(-1L, 1L), j <= 9L, n < 9L),
             bor3(gtp %between% c(-1L, 1L), j <= 9L, n < 9L))
expect_equal(or3s(gtq %between% c(-1L, 1L), , ),
             bor3(gtq %between% c(-1L, 1L), , ))
expect_equal(or3s(gtr %between% c(-1L, 1L), , ),
             bor3(gtr %between% c(-1L, 1L), , ))
expect_equal(or3s(gts %between% c(-1L, 1L), , ),
             bor3(gts %between% c(-1L, 1L), , ))
expect_equal(or3s(gtt %between% c(-1L, 1L), , k <= 0L),
             bor3(gtt %between% c(-1L, 1L), , k <= 0L))
expect_equal(or3s(gtu %between% c(-1L, 1L), , u <= 1L),
             bor3(gtu %between% c(-1L, 1L), , u <= 1L))
expect_equal(or3s(gtv %between% c(-1L, 1L), , c <= 9L),
             bor3(gtv %between% c(-1L, 1L), , c <= 9L))
expect_equal(or3s(gtw %between% c(-1L, 1L), s <= 0L, ),
             bor3(gtw %between% c(-1L, 1L), s <= 0L, ))
expect_equal(or3s(gtx %between% c(-1L, 1L), e <= 1L, ),
             bor3(gtx %between% c(-1L, 1L), e <= 1L, ))
expect_equal(or3s(gty %between% c(-1L, 1L), v <= 9L, ),
             bor3(gty %between% c(-1L, 1L), v <= 9L, ))
expect_equal(or3s(gtz %between% c(-1L, 1L), d <= 0L, k <= 0L),
             bor3(gtz %between% c(-1L, 1L), d <= 0L, k <= 0L))
expect_equal(or3s(gua %between% c(-1L, 1L), c <= 1L, d <= 1L),
             bor3(gua %between% c(-1L, 1L), c <= 1L, d <= 1L))
expect_equal(or3s(gub %between% c(-1L, 1L), g <= 9L, r <= 9L),
             bor3(gub %between% c(-1L, 1L), g <= 9L, r <= 9L))
expect_equal(or3s(guc %between% c(-1L, 1L), , ),
             bor3(guc %between% c(-1L, 1L), , ))
expect_equal(or3s(gud %between% c(-1L, 1L), , ),
             bor3(gud %between% c(-1L, 1L), , ))
expect_equal(or3s(gue %between% c(-1L, 1L), , ),
             bor3(gue %between% c(-1L, 1L), , ))
expect_equal(or3s(guf %between% c(-1L, 1L), , q == 0L),
             bor3(guf %between% c(-1L, 1L), , q == 0L))
expect_equal(or3s(gug %between% c(-1L, 1L), , k == 1L),
             bor3(gug %between% c(-1L, 1L), , k == 1L))
expect_equal(or3s(guh %between% c(-1L, 1L), , x == 9L),
             bor3(guh %between% c(-1L, 1L), , x == 9L))
expect_equal(or3s(gui %between% c(-1L, 1L), b <= 0L, ),
             bor3(gui %between% c(-1L, 1L), b <= 0L, ))
expect_equal(or3s(guj %between% c(-1L, 1L), v <= 1L, ),
             bor3(guj %between% c(-1L, 1L), v <= 1L, ))
expect_equal(or3s(guk %between% c(-1L, 1L), k <= 9L, ),
             bor3(guk %between% c(-1L, 1L), k <= 9L, ))
expect_equal(or3s(gul %between% c(-1L, 1L), h <= 0L, m == 0L),
             bor3(gul %between% c(-1L, 1L), h <= 0L, m == 0L))
expect_equal(or3s(gum %between% c(-1L, 1L), h <= 1L, k == 1L),
             bor3(gum %between% c(-1L, 1L), h <= 1L, k == 1L))
expect_equal(or3s(gun %between% c(-1L, 1L), j <= 9L, v == 9L),
             bor3(gun %between% c(-1L, 1L), j <= 9L, v == 9L))
expect_equal(or3s(guo %between% c(-1L, 1L), , ),
             bor3(guo %between% c(-1L, 1L), , ))
expect_equal(or3s(gup %between% c(-1L, 1L), , ),
             bor3(gup %between% c(-1L, 1L), , ))
expect_equal(or3s(guq %between% c(-1L, 1L), , ),
             bor3(guq %between% c(-1L, 1L), , ))
expect_equal(or3s(gur %between% c(-1L, 1L), , n > 0L),
             bor3(gur %between% c(-1L, 1L), , n > 0L))
expect_equal(or3s(gus %between% c(-1L, 1L), , p > 1L),
             bor3(gus %between% c(-1L, 1L), , p > 1L))
expect_equal(or3s(gut %between% c(-1L, 1L), , j > 9L),
             bor3(gut %between% c(-1L, 1L), , j > 9L))
expect_equal(or3s(guu %between% c(-1L, 1L), u <= 0L, ),
             bor3(guu %between% c(-1L, 1L), u <= 0L, ))
expect_equal(or3s(guv %between% c(-1L, 1L), s <= 1L, ),
             bor3(guv %between% c(-1L, 1L), s <= 1L, ))
expect_equal(or3s(guw %between% c(-1L, 1L), k <= 9L, ),
             bor3(guw %between% c(-1L, 1L), k <= 9L, ))
expect_equal(or3s(gux %between% c(-1L, 1L), e <= 0L, e > 0L),
             bor3(gux %between% c(-1L, 1L), e <= 0L, e > 0L))
expect_equal(or3s(guy %between% c(-1L, 1L), y <= 1L, u > 1L),
             bor3(guy %between% c(-1L, 1L), y <= 1L, u > 1L))
expect_equal(or3s(guz %between% c(-1L, 1L), p <= 9L, m > 9L),
             bor3(guz %between% c(-1L, 1L), p <= 9L, m > 9L))
expect_equal(or3s(gva %between% c(-1L, 1L), , ),
             bor3(gva %between% c(-1L, 1L), , ))
expect_equal(or3s(gvb %between% c(-1L, 1L), , ),
             bor3(gvb %between% c(-1L, 1L), , ))
expect_equal(or3s(gvc %between% c(-1L, 1L), , ),
             bor3(gvc %between% c(-1L, 1L), , ))
expect_equal(or3s(gvd %between% c(-1L, 1L), , k >= 0L),
             bor3(gvd %between% c(-1L, 1L), , k >= 0L))
expect_equal(or3s(gve %between% c(-1L, 1L), , b >= 1L),
             bor3(gve %between% c(-1L, 1L), , b >= 1L))
expect_equal(or3s(gvf %between% c(-1L, 1L), , o >= 9L),
             bor3(gvf %between% c(-1L, 1L), , o >= 9L))
expect_equal(or3s(gvg %between% c(-1L, 1L), s <= 0L, ),
             bor3(gvg %between% c(-1L, 1L), s <= 0L, ))
expect_equal(or3s(gvh %between% c(-1L, 1L), p <= 1L, ),
             bor3(gvh %between% c(-1L, 1L), p <= 1L, ))
expect_equal(or3s(gvi %between% c(-1L, 1L), z <= 9L, ),
             bor3(gvi %between% c(-1L, 1L), z <= 9L, ))
expect_equal(or3s(gvj %between% c(-1L, 1L), t <= 0L, k >= 0L),
             bor3(gvj %between% c(-1L, 1L), t <= 0L, k >= 0L))
expect_equal(or3s(gvk %between% c(-1L, 1L), d <= 1L, b >= 1L),
             bor3(gvk %between% c(-1L, 1L), d <= 1L, b >= 1L))
expect_equal(or3s(gvl %between% c(-1L, 1L), c <= 9L, t >= 9L),
             bor3(gvl %between% c(-1L, 1L), c <= 9L, t >= 9L))
expect_equal(or3s(gvm %between% c(-1L, 1L), , ),
             bor3(gvm %between% c(-1L, 1L), , ))
expect_equal(or3s(gvn %between% c(-1L, 1L), , ),
             bor3(gvn %between% c(-1L, 1L), , ))
expect_equal(or3s(gvo %between% c(-1L, 1L), , ),
             bor3(gvo %between% c(-1L, 1L), , ))
expect_equal(or3s(gvp %between% c(-1L, 1L), , logi_l),
             bor3(gvp %between% c(-1L, 1L), , logi_l))
expect_equal(or3s(gvq %between% c(-1L, 1L), , logi_f),
             bor3(gvq %between% c(-1L, 1L), , logi_f))
expect_equal(or3s(gvr %between% c(-1L, 1L), , logi_f),
             bor3(gvr %between% c(-1L, 1L), , logi_f))
expect_equal(or3s(gvs %between% c(-1L, 1L), e == 0L, ),
             bor3(gvs %between% c(-1L, 1L), e == 0L, ))
expect_equal(or3s(gvt %between% c(-1L, 1L), e == 1L, ),
             bor3(gvt %between% c(-1L, 1L), e == 1L, ))
expect_equal(or3s(gvu %between% c(-1L, 1L), z == 9L, ),
             bor3(gvu %between% c(-1L, 1L), z == 9L, ))
expect_equal(or3s(gvv %between% c(-1L, 1L), k == 0L, logi_a),
             bor3(gvv %between% c(-1L, 1L), k == 0L, logi_a))
expect_equal(or3s(gvw %between% c(-1L, 1L), p == 1L, logi_o),
             bor3(gvw %between% c(-1L, 1L), p == 1L, logi_o))
expect_equal(or3s(gvx %between% c(-1L, 1L), m == 9L, logi_v),
             bor3(gvx %between% c(-1L, 1L), m == 9L, logi_v))
expect_equal(or3s(gvy %between% c(-1L, 1L), , ),
             bor3(gvy %between% c(-1L, 1L), , ))
expect_equal(or3s(gvz %between% c(-1L, 1L), , ),
             bor3(gvz %between% c(-1L, 1L), , ))
expect_equal(or3s(gwa %between% c(-1L, 1L), , ),
             bor3(gwa %between% c(-1L, 1L), , ))
expect_equal(or3s(gwb %between% c(-1L, 1L), , !logi_z),
             bor3(gwb %between% c(-1L, 1L), , !logi_z))
expect_equal(or3s(gwc %between% c(-1L, 1L), , !logi_b),
             bor3(gwc %between% c(-1L, 1L), , !logi_b))
expect_equal(or3s(gwd %between% c(-1L, 1L), , !logi_r),
             bor3(gwd %between% c(-1L, 1L), , !logi_r))
expect_equal(or3s(gwe %between% c(-1L, 1L), f == 0L, ),
             bor3(gwe %between% c(-1L, 1L), f == 0L, ))
expect_equal(or3s(gwf %between% c(-1L, 1L), i == 1L, ),
             bor3(gwf %between% c(-1L, 1L), i == 1L, ))
expect_equal(or3s(gwg %between% c(-1L, 1L), a == 9L, ),
             bor3(gwg %between% c(-1L, 1L), a == 9L, ))
expect_equal(or3s(gwh %between% c(-1L, 1L), t == 0L, !logi_d),
             bor3(gwh %between% c(-1L, 1L), t == 0L, !logi_d))
expect_equal(or3s(gwi %between% c(-1L, 1L), a == 1L, !logi_o),
             bor3(gwi %between% c(-1L, 1L), a == 1L, !logi_o))
expect_equal(or3s(gwj %between% c(-1L, 1L), b == 9L, !logi_x),
             bor3(gwj %between% c(-1L, 1L), b == 9L, !logi_x))
expect_equal(or3s(gwk %between% c(-1L, 1L), , ),
             bor3(gwk %between% c(-1L, 1L), , ))
expect_equal(or3s(gwl %between% c(-1L, 1L), , ),
             bor3(gwl %between% c(-1L, 1L), , ))
expect_equal(or3s(gwm %between% c(-1L, 1L), , ),
             bor3(gwm %between% c(-1L, 1L), , ))
expect_equal(or3s(gwn %between% c(-1L, 1L), , d != 0L),
             bor3(gwn %between% c(-1L, 1L), , d != 0L))
expect_equal(or3s(gwo %between% c(-1L, 1L), , f != 1L),
             bor3(gwo %between% c(-1L, 1L), , f != 1L))
expect_equal(or3s(gwp %between% c(-1L, 1L), , z != 9L),
             bor3(gwp %between% c(-1L, 1L), , z != 9L))
expect_equal(or3s(gwq %between% c(-1L, 1L), l == 0L, ),
             bor3(gwq %between% c(-1L, 1L), l == 0L, ))
expect_equal(or3s(gwr %between% c(-1L, 1L), h == 1L, ),
             bor3(gwr %between% c(-1L, 1L), h == 1L, ))
expect_equal(or3s(gws %between% c(-1L, 1L), g == 9L, ),
             bor3(gws %between% c(-1L, 1L), g == 9L, ))
expect_equal(or3s(gwt %between% c(-1L, 1L), n == 0L, v != 0L),
             bor3(gwt %between% c(-1L, 1L), n == 0L, v != 0L))
expect_equal(or3s(gwu %between% c(-1L, 1L), s == 1L, a != 1L),
             bor3(gwu %between% c(-1L, 1L), s == 1L, a != 1L))
expect_equal(or3s(gwv %between% c(-1L, 1L), w == 9L, k != 9L),
             bor3(gwv %between% c(-1L, 1L), w == 9L, k != 9L))
expect_equal(or3s(gww %between% c(-1L, 1L), , ),
             bor3(gww %between% c(-1L, 1L), , ))
expect_equal(or3s(gwx %between% c(-1L, 1L), , ),
             bor3(gwx %between% c(-1L, 1L), , ))
expect_equal(or3s(gwy %between% c(-1L, 1L), , ),
             bor3(gwy %between% c(-1L, 1L), , ))
expect_equal(or3s(gwz %between% c(-1L, 1L), , l %between% c(-1L, 1L)),
             bor3(gwz %between% c(-1L, 1L), , l %between% c(-1L, 1L)))
expect_equal(or3s(gxa %between% c(-1L, 1L), , k %between% c(-1L, 1L)),
             bor3(gxa %between% c(-1L, 1L), , k %between% c(-1L, 1L)))
expect_equal(or3s(gxb %between% c(-1L, 1L), , i %between% c(-1L, 1L)),
             bor3(gxb %between% c(-1L, 1L), , i %between% c(-1L, 1L)))
expect_equal(or3s(gxc %between% c(-1L, 1L), v == 0L, ),
             bor3(gxc %between% c(-1L, 1L), v == 0L, ))
expect_equal(or3s(gxd %between% c(-1L, 1L), j == 1L, ),
             bor3(gxd %between% c(-1L, 1L), j == 1L, ))
expect_equal(or3s(gxe %between% c(-1L, 1L), l == 9L, ),
             bor3(gxe %between% c(-1L, 1L), l == 9L, ))
expect_equal(or3s(gxf %between% c(-1L, 1L), u == 0L, y %between% c(-1L, 1L)),
             bor3(gxf %between% c(-1L, 1L), u == 0L, y %between% c(-1L, 1L)))
expect_equal(or3s(gxg %between% c(-1L, 1L), j == 1L, h %between% c(-1L, 1L)),
             bor3(gxg %between% c(-1L, 1L), j == 1L, h %between% c(-1L, 1L)))
expect_equal(or3s(gxh %between% c(-1L, 1L), o == 9L, s %between% c(-1L, 1L)),
             bor3(gxh %between% c(-1L, 1L), o == 9L, s %between% c(-1L, 1L)))
expect_equal(or3s(gxi %between% c(-1L, 1L), , ),
             bor3(gxi %between% c(-1L, 1L), , ))
expect_equal(or3s(gxj %between% c(-1L, 1L), , ),
             bor3(gxj %between% c(-1L, 1L), , ))
expect_equal(or3s(gxk %between% c(-1L, 1L), , ),
             bor3(gxk %between% c(-1L, 1L), , ))
expect_equal(or3s(gxl %between% c(-1L, 1L), , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gxl %between% c(-1L, 1L), , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gxm %between% c(-1L, 1L), , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gxm %between% c(-1L, 1L), , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gxn %between% c(-1L, 1L), , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gxn %between% c(-1L, 1L), , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gxo %between% c(-1L, 1L), a == 0L, ),
             bor3(gxo %between% c(-1L, 1L), a == 0L, ))
expect_equal(or3s(gxp %between% c(-1L, 1L), t == 1L, ),
             bor3(gxp %between% c(-1L, 1L), t == 1L, ))
expect_equal(or3s(gxq %between% c(-1L, 1L), o == 9L, ),
             bor3(gxq %between% c(-1L, 1L), o == 9L, ))
expect_equal(or3s(gxr %between% c(-1L, 1L), g == 0L, e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gxr %between% c(-1L, 1L), g == 0L, e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gxs %between% c(-1L, 1L), t == 1L, i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gxs %between% c(-1L, 1L), t == 1L, i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gxt %between% c(-1L, 1L), h == 9L, f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(gxt %between% c(-1L, 1L), h == 9L, f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(gxu %between% c(-1L, 1L), , ),
             bor3(gxu %between% c(-1L, 1L), , ))
expect_equal(or3s(gxv %between% c(-1L, 1L), , ),
             bor3(gxv %between% c(-1L, 1L), , ))
expect_equal(or3s(gxw %between% c(-1L, 1L), , ),
             bor3(gxw %between% c(-1L, 1L), , ))
expect_equal(or3s(gxx %between% c(-1L, 1L), , v %in% 1:4),
             bor3(gxx %between% c(-1L, 1L), , v %in% 1:4))
expect_equal(or3s(gxy %between% c(-1L, 1L), , i %in% 1:4),
             bor3(gxy %between% c(-1L, 1L), , i %in% 1:4))
expect_equal(or3s(gxz %between% c(-1L, 1L), , e %in% 1:4),
             bor3(gxz %between% c(-1L, 1L), , e %in% 1:4))
expect_equal(or3s(gya %between% c(-1L, 1L), u == 0L, ),
             bor3(gya %between% c(-1L, 1L), u == 0L, ))
expect_equal(or3s(gyb %between% c(-1L, 1L), e == 1L, ),
             bor3(gyb %between% c(-1L, 1L), e == 1L, ))
expect_equal(or3s(gyc %between% c(-1L, 1L), o == 9L, ),
             bor3(gyc %between% c(-1L, 1L), o == 9L, ))
expect_equal(or3s(gyd %between% c(-1L, 1L), b == 0L, r %in% 1:4),
             bor3(gyd %between% c(-1L, 1L), b == 0L, r %in% 1:4))
expect_equal(or3s(gye %between% c(-1L, 1L), v == 1L, z %in% 1:4),
             bor3(gye %between% c(-1L, 1L), v == 1L, z %in% 1:4))
expect_equal(or3s(gyf %between% c(-1L, 1L), w == 9L, a %in% 1:4),
             bor3(gyf %between% c(-1L, 1L), w == 9L, a %in% 1:4))
expect_equal(or3s(gyg %between% c(-1L, 1L), , ),
             bor3(gyg %between% c(-1L, 1L), , ))
expect_equal(or3s(gyh %between% c(-1L, 1L), , ),
             bor3(gyh %between% c(-1L, 1L), , ))
expect_equal(or3s(gyi %between% c(-1L, 1L), , ),
             bor3(gyi %between% c(-1L, 1L), , ))
expect_equal(or3s(gyj %between% c(-1L, 1L), , o < 0L),
             bor3(gyj %between% c(-1L, 1L), , o < 0L))
expect_equal(or3s(gyk %between% c(-1L, 1L), , t < 1L),
             bor3(gyk %between% c(-1L, 1L), , t < 1L))
expect_equal(or3s(gyl %between% c(-1L, 1L), , v < 9L),
             bor3(gyl %between% c(-1L, 1L), , v < 9L))
expect_equal(or3s(gym %between% c(-1L, 1L), y == 0L, ),
             bor3(gym %between% c(-1L, 1L), y == 0L, ))
expect_equal(or3s(gyn %between% c(-1L, 1L), a == 1L, ),
             bor3(gyn %between% c(-1L, 1L), a == 1L, ))
expect_equal(or3s(gyo %between% c(-1L, 1L), c == 9L, ),
             bor3(gyo %between% c(-1L, 1L), c == 9L, ))
expect_equal(or3s(gyp %between% c(-1L, 1L), d == 0L, y < 0L),
             bor3(gyp %between% c(-1L, 1L), d == 0L, y < 0L))
expect_equal(or3s(gyq %between% c(-1L, 1L), x == 1L, i < 1L),
             bor3(gyq %between% c(-1L, 1L), x == 1L, i < 1L))
expect_equal(or3s(gyr %between% c(-1L, 1L), g == 9L, t < 9L),
             bor3(gyr %between% c(-1L, 1L), g == 9L, t < 9L))
expect_equal(or3s(gys %between% c(-1L, 1L), , ),
             bor3(gys %between% c(-1L, 1L), , ))
expect_equal(or3s(gyt %between% c(-1L, 1L), , ),
             bor3(gyt %between% c(-1L, 1L), , ))
expect_equal(or3s(gyu %between% c(-1L, 1L), , ),
             bor3(gyu %between% c(-1L, 1L), , ))
expect_equal(or3s(gyv %between% c(-1L, 1L), , e <= 0L),
             bor3(gyv %between% c(-1L, 1L), , e <= 0L))
expect_equal(or3s(gyw %between% c(-1L, 1L), , r <= 1L),
             bor3(gyw %between% c(-1L, 1L), , r <= 1L))
expect_equal(or3s(gyx %between% c(-1L, 1L), , e <= 9L),
             bor3(gyx %between% c(-1L, 1L), , e <= 9L))
expect_equal(or3s(gyy %between% c(-1L, 1L), v == 0L, ),
             bor3(gyy %between% c(-1L, 1L), v == 0L, ))
expect_equal(or3s(gyz %between% c(-1L, 1L), l == 1L, ),
             bor3(gyz %between% c(-1L, 1L), l == 1L, ))
expect_equal(or3s(ha %between% c(-1L, 1L), q == 9L, ),
             bor3(ha %between% c(-1L, 1L), q == 9L, ))
expect_equal(or3s(hb %between% c(-1L, 1L), g == 0L, z <= 0L),
             bor3(hb %between% c(-1L, 1L), g == 0L, z <= 0L))
expect_equal(or3s(hc %between% c(-1L, 1L), f == 1L, d <= 1L),
             bor3(hc %between% c(-1L, 1L), f == 1L, d <= 1L))
expect_equal(or3s(hd %between% c(-1L, 1L), r == 9L, q <= 9L),
             bor3(hd %between% c(-1L, 1L), r == 9L, q <= 9L))
expect_equal(or3s(he %between% c(-1L, 1L), , ),
             bor3(he %between% c(-1L, 1L), , ))
expect_equal(or3s(hf %between% c(-1L, 1L), , ),
             bor3(hf %between% c(-1L, 1L), , ))
expect_equal(or3s(hg %between% c(-1L, 1L), , ),
             bor3(hg %between% c(-1L, 1L), , ))
expect_equal(or3s(hh %between% c(-1L, 1L), , o == 0L),
             bor3(hh %between% c(-1L, 1L), , o == 0L))
expect_equal(or3s(hi %between% c(-1L, 1L), , a == 1L),
             bor3(hi %between% c(-1L, 1L), , a == 1L))
expect_equal(or3s(hj %between% c(-1L, 1L), , g == 9L),
             bor3(hj %between% c(-1L, 1L), , g == 9L))
expect_equal(or3s(hk %between% c(-1L, 1L), h == 0L, ),
             bor3(hk %between% c(-1L, 1L), h == 0L, ))
expect_equal(or3s(hl %between% c(-1L, 1L), w == 1L, ),
             bor3(hl %between% c(-1L, 1L), w == 1L, ))
expect_equal(or3s(hm %between% c(-1L, 1L), s == 9L, ),
             bor3(hm %between% c(-1L, 1L), s == 9L, ))
expect_equal(or3s(hn %between% c(-1L, 1L), q == 0L, n == 0L),
             bor3(hn %between% c(-1L, 1L), q == 0L, n == 0L))
expect_equal(or3s(ho %between% c(-1L, 1L), m == 1L, z == 1L),
             bor3(ho %between% c(-1L, 1L), m == 1L, z == 1L))
expect_equal(or3s(hp %between% c(-1L, 1L), o == 9L, x == 9L),
             bor3(hp %between% c(-1L, 1L), o == 9L, x == 9L))
expect_equal(or3s(hq %between% c(-1L, 1L), , ),
             bor3(hq %between% c(-1L, 1L), , ))
expect_equal(or3s(hr %between% c(-1L, 1L), , ),
             bor3(hr %between% c(-1L, 1L), , ))
expect_equal(or3s(hs %between% c(-1L, 1L), , ),
             bor3(hs %between% c(-1L, 1L), , ))
expect_equal(or3s(ht %between% c(-1L, 1L), , q > 0L),
             bor3(ht %between% c(-1L, 1L), , q > 0L))
expect_equal(or3s(hu %between% c(-1L, 1L), , s > 1L),
             bor3(hu %between% c(-1L, 1L), , s > 1L))
expect_equal(or3s(hv %between% c(-1L, 1L), , o > 9L),
             bor3(hv %between% c(-1L, 1L), , o > 9L))
expect_equal(or3s(hw %between% c(-1L, 1L), i == 0L, ),
             bor3(hw %between% c(-1L, 1L), i == 0L, ))
expect_equal(or3s(hx %between% c(-1L, 1L), v == 1L, ),
             bor3(hx %between% c(-1L, 1L), v == 1L, ))
expect_equal(or3s(hy %between% c(-1L, 1L), e == 9L, ),
             bor3(hy %between% c(-1L, 1L), e == 9L, ))
expect_equal(or3s(hz %between% c(-1L, 1L), b == 0L, t > 0L),
             bor3(hz %between% c(-1L, 1L), b == 0L, t > 0L))
expect_equal(or3s(haa %between% c(-1L, 1L), r == 1L, l > 1L),
             bor3(haa %between% c(-1L, 1L), r == 1L, l > 1L))
expect_equal(or3s(hab %between% c(-1L, 1L), f == 9L, s > 9L),
             bor3(hab %between% c(-1L, 1L), f == 9L, s > 9L))
expect_equal(or3s(hac %between% c(-1L, 1L), , ),
             bor3(hac %between% c(-1L, 1L), , ))
expect_equal(or3s(had %between% c(-1L, 1L), , ),
             bor3(had %between% c(-1L, 1L), , ))
expect_equal(or3s(hae %between% c(-1L, 1L), , ),
             bor3(hae %between% c(-1L, 1L), , ))
expect_equal(or3s(haf %between% c(-1L, 1L), , e >= 0L),
             bor3(haf %between% c(-1L, 1L), , e >= 0L))
expect_equal(or3s(hag %between% c(-1L, 1L), , v >= 1L),
             bor3(hag %between% c(-1L, 1L), , v >= 1L))
expect_equal(or3s(hah %between% c(-1L, 1L), , a >= 9L),
             bor3(hah %between% c(-1L, 1L), , a >= 9L))
expect_equal(or3s(hai %between% c(-1L, 1L), r == 0L, ),
             bor3(hai %between% c(-1L, 1L), r == 0L, ))
expect_equal(or3s(haj %between% c(-1L, 1L), g == 1L, ),
             bor3(haj %between% c(-1L, 1L), g == 1L, ))
expect_equal(or3s(hak %between% c(-1L, 1L), z == 9L, ),
             bor3(hak %between% c(-1L, 1L), z == 9L, ))
expect_equal(or3s(hal %between% c(-1L, 1L), n == 0L, a >= 0L),
             bor3(hal %between% c(-1L, 1L), n == 0L, a >= 0L))
expect_equal(or3s(ham %between% c(-1L, 1L), z == 1L, l >= 1L),
             bor3(ham %between% c(-1L, 1L), z == 1L, l >= 1L))
expect_equal(or3s(han %between% c(-1L, 1L), s == 9L, g >= 9L),
             bor3(han %between% c(-1L, 1L), s == 9L, g >= 9L))
expect_equal(or3s(hao %between% c(-1L, 1L), , ),
             bor3(hao %between% c(-1L, 1L), , ))
expect_equal(or3s(hap %between% c(-1L, 1L), , ),
             bor3(hap %between% c(-1L, 1L), , ))
expect_equal(or3s(haq %between% c(-1L, 1L), , ),
             bor3(haq %between% c(-1L, 1L), , ))
expect_equal(or3s(har %between% c(-1L, 1L), , logi_j),
             bor3(har %between% c(-1L, 1L), , logi_j))
expect_equal(or3s(has %between% c(-1L, 1L), , logi_p),
             bor3(has %between% c(-1L, 1L), , logi_p))
expect_equal(or3s(hat %between% c(-1L, 1L), , logi_k),
             bor3(hat %between% c(-1L, 1L), , logi_k))
expect_equal(or3s(hau %between% c(-1L, 1L), l > 0L, ),
             bor3(hau %between% c(-1L, 1L), l > 0L, ))
expect_equal(or3s(hav %between% c(-1L, 1L), i > 1L, ),
             bor3(hav %between% c(-1L, 1L), i > 1L, ))
expect_equal(or3s(haw %between% c(-1L, 1L), m > 9L, ),
             bor3(haw %between% c(-1L, 1L), m > 9L, ))
expect_equal(or3s(hax %between% c(-1L, 1L), l > 0L, logi_d),
             bor3(hax %between% c(-1L, 1L), l > 0L, logi_d))
expect_equal(or3s(hay %between% c(-1L, 1L), f > 1L, logi_y),
             bor3(hay %between% c(-1L, 1L), f > 1L, logi_y))
expect_equal(or3s(haz %between% c(-1L, 1L), z > 9L, logi_a),
             bor3(haz %between% c(-1L, 1L), z > 9L, logi_a))
expect_equal(or3s(hba %between% c(-1L, 1L), , ),
             bor3(hba %between% c(-1L, 1L), , ))
expect_equal(or3s(hbb %between% c(-1L, 1L), , ),
             bor3(hbb %between% c(-1L, 1L), , ))
expect_equal(or3s(hbc %between% c(-1L, 1L), , ),
             bor3(hbc %between% c(-1L, 1L), , ))
expect_equal(or3s(hbd %between% c(-1L, 1L), , !logi_p),
             bor3(hbd %between% c(-1L, 1L), , !logi_p))
expect_equal(or3s(hbe %between% c(-1L, 1L), , !logi_e),
             bor3(hbe %between% c(-1L, 1L), , !logi_e))
expect_equal(or3s(hbf %between% c(-1L, 1L), , !logi_n),
             bor3(hbf %between% c(-1L, 1L), , !logi_n))
expect_equal(or3s(hbg %between% c(-1L, 1L), l > 0L, ),
             bor3(hbg %between% c(-1L, 1L), l > 0L, ))
expect_equal(or3s(hbh %between% c(-1L, 1L), e > 1L, ),
             bor3(hbh %between% c(-1L, 1L), e > 1L, ))
expect_equal(or3s(hbi %between% c(-1L, 1L), v > 9L, ),
             bor3(hbi %between% c(-1L, 1L), v > 9L, ))
expect_equal(or3s(hbj %between% c(-1L, 1L), m > 0L, !logi_l),
             bor3(hbj %between% c(-1L, 1L), m > 0L, !logi_l))
expect_equal(or3s(hbk %between% c(-1L, 1L), k > 1L, !logi_e),
             bor3(hbk %between% c(-1L, 1L), k > 1L, !logi_e))
expect_equal(or3s(hbl %between% c(-1L, 1L), c > 9L, !logi_f),
             bor3(hbl %between% c(-1L, 1L), c > 9L, !logi_f))
expect_equal(or3s(hbm %between% c(-1L, 1L), , ),
             bor3(hbm %between% c(-1L, 1L), , ))
expect_equal(or3s(hbn %between% c(-1L, 1L), , ),
             bor3(hbn %between% c(-1L, 1L), , ))
expect_equal(or3s(hbo %between% c(-1L, 1L), , ),
             bor3(hbo %between% c(-1L, 1L), , ))
expect_equal(or3s(hbp %between% c(-1L, 1L), , o != 0L),
             bor3(hbp %between% c(-1L, 1L), , o != 0L))
expect_equal(or3s(hbq %between% c(-1L, 1L), , d != 1L),
             bor3(hbq %between% c(-1L, 1L), , d != 1L))
expect_equal(or3s(hbr %between% c(-1L, 1L), , s != 9L),
             bor3(hbr %between% c(-1L, 1L), , s != 9L))
expect_equal(or3s(hbs %between% c(-1L, 1L), e > 0L, ),
             bor3(hbs %between% c(-1L, 1L), e > 0L, ))
expect_equal(or3s(hbt %between% c(-1L, 1L), d > 1L, ),
             bor3(hbt %between% c(-1L, 1L), d > 1L, ))
expect_equal(or3s(hbu %between% c(-1L, 1L), d > 9L, ),
             bor3(hbu %between% c(-1L, 1L), d > 9L, ))
expect_equal(or3s(hbv %between% c(-1L, 1L), d > 0L, m != 0L),
             bor3(hbv %between% c(-1L, 1L), d > 0L, m != 0L))
expect_equal(or3s(hbw %between% c(-1L, 1L), c > 1L, e != 1L),
             bor3(hbw %between% c(-1L, 1L), c > 1L, e != 1L))
expect_equal(or3s(hbx %between% c(-1L, 1L), s > 9L, q != 9L),
             bor3(hbx %between% c(-1L, 1L), s > 9L, q != 9L))
expect_equal(or3s(hby %between% c(-1L, 1L), , ),
             bor3(hby %between% c(-1L, 1L), , ))
expect_equal(or3s(hbz %between% c(-1L, 1L), , ),
             bor3(hbz %between% c(-1L, 1L), , ))
expect_equal(or3s(hca %between% c(-1L, 1L), , ),
             bor3(hca %between% c(-1L, 1L), , ))
expect_equal(or3s(hcb %between% c(-1L, 1L), , v %between% c(-1L, 1L)),
             bor3(hcb %between% c(-1L, 1L), , v %between% c(-1L, 1L)))
expect_equal(or3s(hcc %between% c(-1L, 1L), , f %between% c(-1L, 1L)),
             bor3(hcc %between% c(-1L, 1L), , f %between% c(-1L, 1L)))
expect_equal(or3s(hcd %between% c(-1L, 1L), , w %between% c(-1L, 1L)),
             bor3(hcd %between% c(-1L, 1L), , w %between% c(-1L, 1L)))
expect_equal(or3s(hce %between% c(-1L, 1L), p > 0L, ),
             bor3(hce %between% c(-1L, 1L), p > 0L, ))
expect_equal(or3s(hcf %between% c(-1L, 1L), r > 1L, ),
             bor3(hcf %between% c(-1L, 1L), r > 1L, ))
expect_equal(or3s(hcg %between% c(-1L, 1L), s > 9L, ),
             bor3(hcg %between% c(-1L, 1L), s > 9L, ))
expect_equal(or3s(hch %between% c(-1L, 1L), s > 0L, c %between% c(-1L, 1L)),
             bor3(hch %between% c(-1L, 1L), s > 0L, c %between% c(-1L, 1L)))
expect_equal(or3s(hci %between% c(-1L, 1L), e > 1L, r %between% c(-1L, 1L)),
             bor3(hci %between% c(-1L, 1L), e > 1L, r %between% c(-1L, 1L)))
expect_equal(or3s(hcj %between% c(-1L, 1L), b > 9L, b %between% c(-1L, 1L)),
             bor3(hcj %between% c(-1L, 1L), b > 9L, b %between% c(-1L, 1L)))
expect_equal(or3s(hck %between% c(-1L, 1L), , ),
             bor3(hck %between% c(-1L, 1L), , ))
expect_equal(or3s(hcl %between% c(-1L, 1L), , ),
             bor3(hcl %between% c(-1L, 1L), , ))
expect_equal(or3s(hcm %between% c(-1L, 1L), , ),
             bor3(hcm %between% c(-1L, 1L), , ))
expect_equal(or3s(hcn %between% c(-1L, 1L), , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hcn %between% c(-1L, 1L), , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hco %between% c(-1L, 1L), , b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hco %between% c(-1L, 1L), , b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hcp %between% c(-1L, 1L), , s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hcp %between% c(-1L, 1L), , s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hcq %between% c(-1L, 1L), b > 0L, ),
             bor3(hcq %between% c(-1L, 1L), b > 0L, ))
expect_equal(or3s(hcr %between% c(-1L, 1L), i > 1L, ),
             bor3(hcr %between% c(-1L, 1L), i > 1L, ))
expect_equal(or3s(hcs %between% c(-1L, 1L), z > 9L, ),
             bor3(hcs %between% c(-1L, 1L), z > 9L, ))
expect_equal(or3s(hct %between% c(-1L, 1L), r > 0L, i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hct %between% c(-1L, 1L), r > 0L, i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hcu %between% c(-1L, 1L), a > 1L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hcu %between% c(-1L, 1L), a > 1L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hcv %between% c(-1L, 1L), m > 9L, g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hcv %between% c(-1L, 1L), m > 9L, g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hcw %between% c(-1L, 1L), , ),
             bor3(hcw %between% c(-1L, 1L), , ))
expect_equal(or3s(hcx %between% c(-1L, 1L), , ),
             bor3(hcx %between% c(-1L, 1L), , ))
expect_equal(or3s(hcy %between% c(-1L, 1L), , ),
             bor3(hcy %between% c(-1L, 1L), , ))
expect_equal(or3s(hcz %between% c(-1L, 1L), , b %in% 1:4),
             bor3(hcz %between% c(-1L, 1L), , b %in% 1:4))
expect_equal(or3s(hda %between% c(-1L, 1L), , w %in% 1:4),
             bor3(hda %between% c(-1L, 1L), , w %in% 1:4))
expect_equal(or3s(hdb %between% c(-1L, 1L), , c %in% 1:4),
             bor3(hdb %between% c(-1L, 1L), , c %in% 1:4))
expect_equal(or3s(hdc %between% c(-1L, 1L), d > 0L, ),
             bor3(hdc %between% c(-1L, 1L), d > 0L, ))
expect_equal(or3s(hdd %between% c(-1L, 1L), k > 1L, ),
             bor3(hdd %between% c(-1L, 1L), k > 1L, ))
expect_equal(or3s(hde %between% c(-1L, 1L), v > 9L, ),
             bor3(hde %between% c(-1L, 1L), v > 9L, ))
expect_equal(or3s(hdf %between% c(-1L, 1L), q > 0L, h %in% 1:4),
             bor3(hdf %between% c(-1L, 1L), q > 0L, h %in% 1:4))
expect_equal(or3s(hdg %between% c(-1L, 1L), a > 1L, j %in% 1:4),
             bor3(hdg %between% c(-1L, 1L), a > 1L, j %in% 1:4))
expect_equal(or3s(hdh %between% c(-1L, 1L), p > 9L, c %in% 1:4),
             bor3(hdh %between% c(-1L, 1L), p > 9L, c %in% 1:4))
expect_equal(or3s(hdi %between% c(-1L, 1L), , ),
             bor3(hdi %between% c(-1L, 1L), , ))
expect_equal(or3s(hdj %between% c(-1L, 1L), , ),
             bor3(hdj %between% c(-1L, 1L), , ))
expect_equal(or3s(hdk %between% c(-1L, 1L), , ),
             bor3(hdk %between% c(-1L, 1L), , ))
expect_equal(or3s(hdl %between% c(-1L, 1L), , j < 0L),
             bor3(hdl %between% c(-1L, 1L), , j < 0L))
expect_equal(or3s(hdm %between% c(-1L, 1L), , x < 1L),
             bor3(hdm %between% c(-1L, 1L), , x < 1L))
expect_equal(or3s(hdn %between% c(-1L, 1L), , o < 9L),
             bor3(hdn %between% c(-1L, 1L), , o < 9L))
expect_equal(or3s(hdo %between% c(-1L, 1L), f > 0L, ),
             bor3(hdo %between% c(-1L, 1L), f > 0L, ))
expect_equal(or3s(hdp %between% c(-1L, 1L), v > 1L, ),
             bor3(hdp %between% c(-1L, 1L), v > 1L, ))
expect_equal(or3s(hdq %between% c(-1L, 1L), m > 9L, ),
             bor3(hdq %between% c(-1L, 1L), m > 9L, ))
expect_equal(or3s(hdr %between% c(-1L, 1L), u > 0L, n < 0L),
             bor3(hdr %between% c(-1L, 1L), u > 0L, n < 0L))
expect_equal(or3s(hds %between% c(-1L, 1L), b > 1L, c < 1L),
             bor3(hds %between% c(-1L, 1L), b > 1L, c < 1L))
expect_equal(or3s(hdt %between% c(-1L, 1L), o > 9L, u < 9L),
             bor3(hdt %between% c(-1L, 1L), o > 9L, u < 9L))
expect_equal(or3s(hdu %between% c(-1L, 1L), , ),
             bor3(hdu %between% c(-1L, 1L), , ))
expect_equal(or3s(hdv %between% c(-1L, 1L), , ),
             bor3(hdv %between% c(-1L, 1L), , ))
expect_equal(or3s(hdw %between% c(-1L, 1L), , ),
             bor3(hdw %between% c(-1L, 1L), , ))
expect_equal(or3s(hdx %between% c(-1L, 1L), , l <= 0L),
             bor3(hdx %between% c(-1L, 1L), , l <= 0L))
expect_equal(or3s(hdy %between% c(-1L, 1L), , t <= 1L),
             bor3(hdy %between% c(-1L, 1L), , t <= 1L))
expect_equal(or3s(hdz %between% c(-1L, 1L), , u <= 9L),
             bor3(hdz %between% c(-1L, 1L), , u <= 9L))
expect_equal(or3s(hea %between% c(-1L, 1L), p > 0L, ),
             bor3(hea %between% c(-1L, 1L), p > 0L, ))
expect_equal(or3s(heb %between% c(-1L, 1L), q > 1L, ),
             bor3(heb %between% c(-1L, 1L), q > 1L, ))
expect_equal(or3s(hec %between% c(-1L, 1L), k > 9L, ),
             bor3(hec %between% c(-1L, 1L), k > 9L, ))
expect_equal(or3s(hed %between% c(-1L, 1L), b > 0L, v <= 0L),
             bor3(hed %between% c(-1L, 1L), b > 0L, v <= 0L))
expect_equal(or3s(hee %between% c(-1L, 1L), r > 1L, x <= 1L),
             bor3(hee %between% c(-1L, 1L), r > 1L, x <= 1L))
expect_equal(or3s(hef %between% c(-1L, 1L), h > 9L, s <= 9L),
             bor3(hef %between% c(-1L, 1L), h > 9L, s <= 9L))
expect_equal(or3s(heg %between% c(-1L, 1L), , ),
             bor3(heg %between% c(-1L, 1L), , ))
expect_equal(or3s(heh %between% c(-1L, 1L), , ),
             bor3(heh %between% c(-1L, 1L), , ))
expect_equal(or3s(hei %between% c(-1L, 1L), , ),
             bor3(hei %between% c(-1L, 1L), , ))
expect_equal(or3s(hej %between% c(-1L, 1L), , h == 0L),
             bor3(hej %between% c(-1L, 1L), , h == 0L))
expect_equal(or3s(hek %between% c(-1L, 1L), , h == 1L),
             bor3(hek %between% c(-1L, 1L), , h == 1L))
expect_equal(or3s(hel %between% c(-1L, 1L), , j == 9L),
             bor3(hel %between% c(-1L, 1L), , j == 9L))
expect_equal(or3s(hem %between% c(-1L, 1L), t > 0L, ),
             bor3(hem %between% c(-1L, 1L), t > 0L, ))
expect_equal(or3s(hen %between% c(-1L, 1L), h > 1L, ),
             bor3(hen %between% c(-1L, 1L), h > 1L, ))
expect_equal(or3s(heo %between% c(-1L, 1L), d > 9L, ),
             bor3(heo %between% c(-1L, 1L), d > 9L, ))
expect_equal(or3s(hep %between% c(-1L, 1L), m > 0L, g == 0L),
             bor3(hep %between% c(-1L, 1L), m > 0L, g == 0L))
expect_equal(or3s(heq %between% c(-1L, 1L), r > 1L, v == 1L),
             bor3(heq %between% c(-1L, 1L), r > 1L, v == 1L))
expect_equal(or3s(her %between% c(-1L, 1L), u > 9L, g == 9L),
             bor3(her %between% c(-1L, 1L), u > 9L, g == 9L))
expect_equal(or3s(hes %between% c(-1L, 1L), , ),
             bor3(hes %between% c(-1L, 1L), , ))
expect_equal(or3s(het %between% c(-1L, 1L), , ),
             bor3(het %between% c(-1L, 1L), , ))
expect_equal(or3s(heu %between% c(-1L, 1L), , ),
             bor3(heu %between% c(-1L, 1L), , ))
expect_equal(or3s(hev %between% c(-1L, 1L), , r > 0L),
             bor3(hev %between% c(-1L, 1L), , r > 0L))
expect_equal(or3s(hew %between% c(-1L, 1L), , q > 1L),
             bor3(hew %between% c(-1L, 1L), , q > 1L))
expect_equal(or3s(hex %between% c(-1L, 1L), , f > 9L),
             bor3(hex %between% c(-1L, 1L), , f > 9L))
expect_equal(or3s(hey %between% c(-1L, 1L), o > 0L, ),
             bor3(hey %between% c(-1L, 1L), o > 0L, ))
expect_equal(or3s(hez %between% c(-1L, 1L), p > 1L, ),
             bor3(hez %between% c(-1L, 1L), p > 1L, ))
expect_equal(or3s(hfa %between% c(-1L, 1L), w > 9L, ),
             bor3(hfa %between% c(-1L, 1L), w > 9L, ))
expect_equal(or3s(hfb %between% c(-1L, 1L), p > 0L, f > 0L),
             bor3(hfb %between% c(-1L, 1L), p > 0L, f > 0L))
expect_equal(or3s(hfc %between% c(-1L, 1L), s > 1L, n > 1L),
             bor3(hfc %between% c(-1L, 1L), s > 1L, n > 1L))
expect_equal(or3s(hfd %between% c(-1L, 1L), f > 9L, o > 9L),
             bor3(hfd %between% c(-1L, 1L), f > 9L, o > 9L))
expect_equal(or3s(hfe %between% c(-1L, 1L), , ),
             bor3(hfe %between% c(-1L, 1L), , ))
expect_equal(or3s(hff %between% c(-1L, 1L), , ),
             bor3(hff %between% c(-1L, 1L), , ))
expect_equal(or3s(hfg %between% c(-1L, 1L), , ),
             bor3(hfg %between% c(-1L, 1L), , ))
expect_equal(or3s(hfh %between% c(-1L, 1L), , o >= 0L),
             bor3(hfh %between% c(-1L, 1L), , o >= 0L))
expect_equal(or3s(hfi %between% c(-1L, 1L), , a >= 1L),
             bor3(hfi %between% c(-1L, 1L), , a >= 1L))
expect_equal(or3s(hfj %between% c(-1L, 1L), , g >= 9L),
             bor3(hfj %between% c(-1L, 1L), , g >= 9L))
expect_equal(or3s(hfk %between% c(-1L, 1L), u > 0L, ),
             bor3(hfk %between% c(-1L, 1L), u > 0L, ))
expect_equal(or3s(hfl %between% c(-1L, 1L), u > 1L, ),
             bor3(hfl %between% c(-1L, 1L), u > 1L, ))
expect_equal(or3s(hfm %between% c(-1L, 1L), w > 9L, ),
             bor3(hfm %between% c(-1L, 1L), w > 9L, ))
expect_equal(or3s(hfn %between% c(-1L, 1L), c > 0L, r >= 0L),
             bor3(hfn %between% c(-1L, 1L), c > 0L, r >= 0L))
expect_equal(or3s(hfo %between% c(-1L, 1L), b > 1L, q >= 1L),
             bor3(hfo %between% c(-1L, 1L), b > 1L, q >= 1L))
expect_equal(or3s(hfp %between% c(-1L, 1L), k > 9L, u >= 9L),
             bor3(hfp %between% c(-1L, 1L), k > 9L, u >= 9L))
expect_equal(or3s(hfq %between% c(-1L, 1L), , ),
             bor3(hfq %between% c(-1L, 1L), , ))
expect_equal(or3s(hfr %between% c(-1L, 1L), , ),
             bor3(hfr %between% c(-1L, 1L), , ))
expect_equal(or3s(hfs %between% c(-1L, 1L), , ),
             bor3(hfs %between% c(-1L, 1L), , ))
expect_equal(or3s(hft %between% c(-1L, 1L), , logi_a),
             bor3(hft %between% c(-1L, 1L), , logi_a))
expect_equal(or3s(hfu %between% c(-1L, 1L), , logi_u),
             bor3(hfu %between% c(-1L, 1L), , logi_u))
expect_equal(or3s(hfv %between% c(-1L, 1L), , logi_o),
             bor3(hfv %between% c(-1L, 1L), , logi_o))
expect_equal(or3s(hfw %between% c(-1L, 1L), l >= 0L, ),
             bor3(hfw %between% c(-1L, 1L), l >= 0L, ))
expect_equal(or3s(hfx %between% c(-1L, 1L), i >= 1L, ),
             bor3(hfx %between% c(-1L, 1L), i >= 1L, ))
expect_equal(or3s(hfy %between% c(-1L, 1L), l >= 9L, ),
             bor3(hfy %between% c(-1L, 1L), l >= 9L, ))
expect_equal(or3s(hfz %between% c(-1L, 1L), w >= 0L, logi_g),
             bor3(hfz %between% c(-1L, 1L), w >= 0L, logi_g))
expect_equal(or3s(hga %between% c(-1L, 1L), k >= 1L, logi_j),
             bor3(hga %between% c(-1L, 1L), k >= 1L, logi_j))
expect_equal(or3s(hgb %between% c(-1L, 1L), p >= 9L, logi_k),
             bor3(hgb %between% c(-1L, 1L), p >= 9L, logi_k))
expect_equal(or3s(hgc %between% c(-1L, 1L), , ),
             bor3(hgc %between% c(-1L, 1L), , ))
expect_equal(or3s(hgd %between% c(-1L, 1L), , ),
             bor3(hgd %between% c(-1L, 1L), , ))
expect_equal(or3s(hge %between% c(-1L, 1L), , ),
             bor3(hge %between% c(-1L, 1L), , ))
expect_equal(or3s(hgf %between% c(-1L, 1L), , !logi_t),
             bor3(hgf %between% c(-1L, 1L), , !logi_t))
expect_equal(or3s(hgg %between% c(-1L, 1L), , !logi_s),
             bor3(hgg %between% c(-1L, 1L), , !logi_s))
expect_equal(or3s(hgh %between% c(-1L, 1L), , !logi_y),
             bor3(hgh %between% c(-1L, 1L), , !logi_y))
expect_equal(or3s(hgi %between% c(-1L, 1L), d >= 0L, ),
             bor3(hgi %between% c(-1L, 1L), d >= 0L, ))
expect_equal(or3s(hgj %between% c(-1L, 1L), h >= 1L, ),
             bor3(hgj %between% c(-1L, 1L), h >= 1L, ))
expect_equal(or3s(hgk %between% c(-1L, 1L), q >= 9L, ),
             bor3(hgk %between% c(-1L, 1L), q >= 9L, ))
expect_equal(or3s(hgl %between% c(-1L, 1L), o >= 0L, !logi_v),
             bor3(hgl %between% c(-1L, 1L), o >= 0L, !logi_v))
expect_equal(or3s(hgm %between% c(-1L, 1L), f >= 1L, !logi_h),
             bor3(hgm %between% c(-1L, 1L), f >= 1L, !logi_h))
expect_equal(or3s(hgn %between% c(-1L, 1L), h >= 9L, !logi_g),
             bor3(hgn %between% c(-1L, 1L), h >= 9L, !logi_g))
expect_equal(or3s(hgo %between% c(-1L, 1L), , ),
             bor3(hgo %between% c(-1L, 1L), , ))
expect_equal(or3s(hgp %between% c(-1L, 1L), , ),
             bor3(hgp %between% c(-1L, 1L), , ))
expect_equal(or3s(hgq %between% c(-1L, 1L), , ),
             bor3(hgq %between% c(-1L, 1L), , ))
expect_equal(or3s(hgr %between% c(-1L, 1L), , u != 0L),
             bor3(hgr %between% c(-1L, 1L), , u != 0L))
expect_equal(or3s(hgs %between% c(-1L, 1L), , h != 1L),
             bor3(hgs %between% c(-1L, 1L), , h != 1L))
expect_equal(or3s(hgt %between% c(-1L, 1L), , z != 9L),
             bor3(hgt %between% c(-1L, 1L), , z != 9L))
expect_equal(or3s(hgu %between% c(-1L, 1L), k >= 0L, ),
             bor3(hgu %between% c(-1L, 1L), k >= 0L, ))
expect_equal(or3s(hgv %between% c(-1L, 1L), b >= 1L, ),
             bor3(hgv %between% c(-1L, 1L), b >= 1L, ))
expect_equal(or3s(hgw %between% c(-1L, 1L), k >= 9L, ),
             bor3(hgw %between% c(-1L, 1L), k >= 9L, ))
expect_equal(or3s(hgx %between% c(-1L, 1L), l >= 0L, r != 0L),
             bor3(hgx %between% c(-1L, 1L), l >= 0L, r != 0L))
expect_equal(or3s(hgy %between% c(-1L, 1L), q >= 1L, y != 1L),
             bor3(hgy %between% c(-1L, 1L), q >= 1L, y != 1L))
expect_equal(or3s(hgz %between% c(-1L, 1L), s >= 9L, u != 9L),
             bor3(hgz %between% c(-1L, 1L), s >= 9L, u != 9L))
expect_equal(or3s(hha %between% c(-1L, 1L), , ),
             bor3(hha %between% c(-1L, 1L), , ))
expect_equal(or3s(hhb %between% c(-1L, 1L), , ),
             bor3(hhb %between% c(-1L, 1L), , ))
expect_equal(or3s(hhc %between% c(-1L, 1L), , ),
             bor3(hhc %between% c(-1L, 1L), , ))
expect_equal(or3s(hhd %between% c(-1L, 1L), , f %between% c(-1L, 1L)),
             bor3(hhd %between% c(-1L, 1L), , f %between% c(-1L, 1L)))
expect_equal(or3s(hhe %between% c(-1L, 1L), , y %between% c(-1L, 1L)),
             bor3(hhe %between% c(-1L, 1L), , y %between% c(-1L, 1L)))
expect_equal(or3s(hhf %between% c(-1L, 1L), , v %between% c(-1L, 1L)),
             bor3(hhf %between% c(-1L, 1L), , v %between% c(-1L, 1L)))
expect_equal(or3s(hhg %between% c(-1L, 1L), i >= 0L, ),
             bor3(hhg %between% c(-1L, 1L), i >= 0L, ))
expect_equal(or3s(hhh %between% c(-1L, 1L), s >= 1L, ),
             bor3(hhh %between% c(-1L, 1L), s >= 1L, ))
expect_equal(or3s(hhi %between% c(-1L, 1L), g >= 9L, ),
             bor3(hhi %between% c(-1L, 1L), g >= 9L, ))
expect_equal(or3s(hhj %between% c(-1L, 1L), j >= 0L, y %between% c(-1L, 1L)),
             bor3(hhj %between% c(-1L, 1L), j >= 0L, y %between% c(-1L, 1L)))
expect_equal(or3s(hhk %between% c(-1L, 1L), g >= 1L, q %between% c(-1L, 1L)),
             bor3(hhk %between% c(-1L, 1L), g >= 1L, q %between% c(-1L, 1L)))
expect_equal(or3s(hhl %between% c(-1L, 1L), u >= 9L, k %between% c(-1L, 1L)),
             bor3(hhl %between% c(-1L, 1L), u >= 9L, k %between% c(-1L, 1L)))
expect_equal(or3s(hhm %between% c(-1L, 1L), , ),
             bor3(hhm %between% c(-1L, 1L), , ))
expect_equal(or3s(hhn %between% c(-1L, 1L), , ),
             bor3(hhn %between% c(-1L, 1L), , ))
expect_equal(or3s(hho %between% c(-1L, 1L), , ),
             bor3(hho %between% c(-1L, 1L), , ))
expect_equal(or3s(hhp %between% c(-1L, 1L), , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hhp %between% c(-1L, 1L), , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hhq %between% c(-1L, 1L), , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hhq %between% c(-1L, 1L), , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hhr %between% c(-1L, 1L), , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hhr %between% c(-1L, 1L), , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hhs %between% c(-1L, 1L), h >= 0L, ),
             bor3(hhs %between% c(-1L, 1L), h >= 0L, ))
expect_equal(or3s(hht %between% c(-1L, 1L), z >= 1L, ),
             bor3(hht %between% c(-1L, 1L), z >= 1L, ))
expect_equal(or3s(hhu %between% c(-1L, 1L), j >= 9L, ),
             bor3(hhu %between% c(-1L, 1L), j >= 9L, ))
expect_equal(or3s(hhv %between% c(-1L, 1L), g >= 0L, n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hhv %between% c(-1L, 1L), g >= 0L, n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hhw %between% c(-1L, 1L), v >= 1L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hhw %between% c(-1L, 1L), v >= 1L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hhx %between% c(-1L, 1L), d >= 9L, m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hhx %between% c(-1L, 1L), d >= 9L, m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hhy %between% c(-1L, 1L), , ),
             bor3(hhy %between% c(-1L, 1L), , ))
expect_equal(or3s(hhz %between% c(-1L, 1L), , ),
             bor3(hhz %between% c(-1L, 1L), , ))
expect_equal(or3s(hia %between% c(-1L, 1L), , ),
             bor3(hia %between% c(-1L, 1L), , ))
expect_equal(or3s(hib %between% c(-1L, 1L), , b %in% 1:4),
             bor3(hib %between% c(-1L, 1L), , b %in% 1:4))
expect_equal(or3s(hic %between% c(-1L, 1L), , t %in% 1:4),
             bor3(hic %between% c(-1L, 1L), , t %in% 1:4))
expect_equal(or3s(hid %between% c(-1L, 1L), , t %in% 1:4),
             bor3(hid %between% c(-1L, 1L), , t %in% 1:4))
expect_equal(or3s(hie %between% c(-1L, 1L), r >= 0L, ),
             bor3(hie %between% c(-1L, 1L), r >= 0L, ))
expect_equal(or3s(hif %between% c(-1L, 1L), a >= 1L, ),
             bor3(hif %between% c(-1L, 1L), a >= 1L, ))
expect_equal(or3s(hig %between% c(-1L, 1L), s >= 9L, ),
             bor3(hig %between% c(-1L, 1L), s >= 9L, ))
expect_equal(or3s(hih %between% c(-1L, 1L), e >= 0L, z %in% 1:4),
             bor3(hih %between% c(-1L, 1L), e >= 0L, z %in% 1:4))
expect_equal(or3s(hii %between% c(-1L, 1L), r >= 1L, f %in% 1:4),
             bor3(hii %between% c(-1L, 1L), r >= 1L, f %in% 1:4))
expect_equal(or3s(hij %between% c(-1L, 1L), i >= 9L, m %in% 1:4),
             bor3(hij %between% c(-1L, 1L), i >= 9L, m %in% 1:4))
expect_equal(or3s(hik %between% c(-1L, 1L), , ),
             bor3(hik %between% c(-1L, 1L), , ))
expect_equal(or3s(hil %between% c(-1L, 1L), , ),
             bor3(hil %between% c(-1L, 1L), , ))
expect_equal(or3s(him %between% c(-1L, 1L), , ),
             bor3(him %between% c(-1L, 1L), , ))
expect_equal(or3s(hin %between% c(-1L, 1L), , f < 0L),
             bor3(hin %between% c(-1L, 1L), , f < 0L))
expect_equal(or3s(hio %between% c(-1L, 1L), , d < 1L),
             bor3(hio %between% c(-1L, 1L), , d < 1L))
expect_equal(or3s(hip %between% c(-1L, 1L), , h < 9L),
             bor3(hip %between% c(-1L, 1L), , h < 9L))
expect_equal(or3s(hiq %between% c(-1L, 1L), r >= 0L, ),
             bor3(hiq %between% c(-1L, 1L), r >= 0L, ))
expect_equal(or3s(hir %between% c(-1L, 1L), s >= 1L, ),
             bor3(hir %between% c(-1L, 1L), s >= 1L, ))
expect_equal(or3s(his %between% c(-1L, 1L), f >= 9L, ),
             bor3(his %between% c(-1L, 1L), f >= 9L, ))
expect_equal(or3s(hit %between% c(-1L, 1L), x >= 0L, r < 0L),
             bor3(hit %between% c(-1L, 1L), x >= 0L, r < 0L))
expect_equal(or3s(hiu %between% c(-1L, 1L), u >= 1L, s < 1L),
             bor3(hiu %between% c(-1L, 1L), u >= 1L, s < 1L))
expect_equal(or3s(hiv %between% c(-1L, 1L), o >= 9L, u < 9L),
             bor3(hiv %between% c(-1L, 1L), o >= 9L, u < 9L))
expect_equal(or3s(hiw %between% c(-1L, 1L), , ),
             bor3(hiw %between% c(-1L, 1L), , ))
expect_equal(or3s(hix %between% c(-1L, 1L), , ),
             bor3(hix %between% c(-1L, 1L), , ))
expect_equal(or3s(hiy %between% c(-1L, 1L), , ),
             bor3(hiy %between% c(-1L, 1L), , ))
expect_equal(or3s(hiz %between% c(-1L, 1L), , g <= 0L),
             bor3(hiz %between% c(-1L, 1L), , g <= 0L))
expect_equal(or3s(hja %between% c(-1L, 1L), , k <= 1L),
             bor3(hja %between% c(-1L, 1L), , k <= 1L))
expect_equal(or3s(hjb %between% c(-1L, 1L), , b <= 9L),
             bor3(hjb %between% c(-1L, 1L), , b <= 9L))
expect_equal(or3s(hjc %between% c(-1L, 1L), p >= 0L, ),
             bor3(hjc %between% c(-1L, 1L), p >= 0L, ))
expect_equal(or3s(hjd %between% c(-1L, 1L), f >= 1L, ),
             bor3(hjd %between% c(-1L, 1L), f >= 1L, ))
expect_equal(or3s(hje %between% c(-1L, 1L), a >= 9L, ),
             bor3(hje %between% c(-1L, 1L), a >= 9L, ))
expect_equal(or3s(hjf %between% c(-1L, 1L), s >= 0L, p <= 0L),
             bor3(hjf %between% c(-1L, 1L), s >= 0L, p <= 0L))
expect_equal(or3s(hjg %between% c(-1L, 1L), u >= 1L, b <= 1L),
             bor3(hjg %between% c(-1L, 1L), u >= 1L, b <= 1L))
expect_equal(or3s(hjh %between% c(-1L, 1L), z >= 9L, s <= 9L),
             bor3(hjh %between% c(-1L, 1L), z >= 9L, s <= 9L))
expect_equal(or3s(hji %between% c(-1L, 1L), , ),
             bor3(hji %between% c(-1L, 1L), , ))
expect_equal(or3s(hjj %between% c(-1L, 1L), , ),
             bor3(hjj %between% c(-1L, 1L), , ))
expect_equal(or3s(hjk %between% c(-1L, 1L), , ),
             bor3(hjk %between% c(-1L, 1L), , ))
expect_equal(or3s(hjl %between% c(-1L, 1L), , e == 0L),
             bor3(hjl %between% c(-1L, 1L), , e == 0L))
expect_equal(or3s(hjm %between% c(-1L, 1L), , d == 1L),
             bor3(hjm %between% c(-1L, 1L), , d == 1L))
expect_equal(or3s(hjn %between% c(-1L, 1L), , c == 9L),
             bor3(hjn %between% c(-1L, 1L), , c == 9L))
expect_equal(or3s(hjo %between% c(-1L, 1L), h >= 0L, ),
             bor3(hjo %between% c(-1L, 1L), h >= 0L, ))
expect_equal(or3s(hjp %between% c(-1L, 1L), k >= 1L, ),
             bor3(hjp %between% c(-1L, 1L), k >= 1L, ))
expect_equal(or3s(hjq %between% c(-1L, 1L), i >= 9L, ),
             bor3(hjq %between% c(-1L, 1L), i >= 9L, ))
expect_equal(or3s(hjr %between% c(-1L, 1L), o >= 0L, r == 0L),
             bor3(hjr %between% c(-1L, 1L), o >= 0L, r == 0L))
expect_equal(or3s(hjs %between% c(-1L, 1L), y >= 1L, u == 1L),
             bor3(hjs %between% c(-1L, 1L), y >= 1L, u == 1L))
expect_equal(or3s(hjt %between% c(-1L, 1L), z >= 9L, e == 9L),
             bor3(hjt %between% c(-1L, 1L), z >= 9L, e == 9L))
expect_equal(or3s(hju %between% c(-1L, 1L), , ),
             bor3(hju %between% c(-1L, 1L), , ))
expect_equal(or3s(hjv %between% c(-1L, 1L), , ),
             bor3(hjv %between% c(-1L, 1L), , ))
expect_equal(or3s(hjw %between% c(-1L, 1L), , ),
             bor3(hjw %between% c(-1L, 1L), , ))
expect_equal(or3s(hjx %between% c(-1L, 1L), , o > 0L),
             bor3(hjx %between% c(-1L, 1L), , o > 0L))
expect_equal(or3s(hjy %between% c(-1L, 1L), , p > 1L),
             bor3(hjy %between% c(-1L, 1L), , p > 1L))
expect_equal(or3s(hjz %between% c(-1L, 1L), , b > 9L),
             bor3(hjz %between% c(-1L, 1L), , b > 9L))
expect_equal(or3s(hka %between% c(-1L, 1L), s >= 0L, ),
             bor3(hka %between% c(-1L, 1L), s >= 0L, ))
expect_equal(or3s(hkb %between% c(-1L, 1L), g >= 1L, ),
             bor3(hkb %between% c(-1L, 1L), g >= 1L, ))
expect_equal(or3s(hkc %between% c(-1L, 1L), y >= 9L, ),
             bor3(hkc %between% c(-1L, 1L), y >= 9L, ))
expect_equal(or3s(hkd %between% c(-1L, 1L), d >= 0L, a > 0L),
             bor3(hkd %between% c(-1L, 1L), d >= 0L, a > 0L))
expect_equal(or3s(hke %between% c(-1L, 1L), h >= 1L, p > 1L),
             bor3(hke %between% c(-1L, 1L), h >= 1L, p > 1L))
expect_equal(or3s(hkf %between% c(-1L, 1L), z >= 9L, g > 9L),
             bor3(hkf %between% c(-1L, 1L), z >= 9L, g > 9L))
expect_equal(or3s(hkg %between% c(-1L, 1L), , ),
             bor3(hkg %between% c(-1L, 1L), , ))
expect_equal(or3s(hkh %between% c(-1L, 1L), , ),
             bor3(hkh %between% c(-1L, 1L), , ))
expect_equal(or3s(hki %between% c(-1L, 1L), , ),
             bor3(hki %between% c(-1L, 1L), , ))
expect_equal(or3s(hkj %between% c(-1L, 1L), , i >= 0L),
             bor3(hkj %between% c(-1L, 1L), , i >= 0L))
expect_equal(or3s(hkk %between% c(-1L, 1L), , g >= 1L),
             bor3(hkk %between% c(-1L, 1L), , g >= 1L))
expect_equal(or3s(hkl %between% c(-1L, 1L), , n >= 9L),
             bor3(hkl %between% c(-1L, 1L), , n >= 9L))
expect_equal(or3s(hkm %between% c(-1L, 1L), s >= 0L, ),
             bor3(hkm %between% c(-1L, 1L), s >= 0L, ))
expect_equal(or3s(hkn %between% c(-1L, 1L), s >= 1L, ),
             bor3(hkn %between% c(-1L, 1L), s >= 1L, ))
expect_equal(or3s(hko %between% c(-1L, 1L), o >= 9L, ),
             bor3(hko %between% c(-1L, 1L), o >= 9L, ))
expect_equal(or3s(hkp %between% c(-1L, 1L), c >= 0L, l >= 0L),
             bor3(hkp %between% c(-1L, 1L), c >= 0L, l >= 0L))
expect_equal(or3s(hkq %between% c(-1L, 1L), e >= 1L, o >= 1L),
             bor3(hkq %between% c(-1L, 1L), e >= 1L, o >= 1L))
expect_equal(or3s(hkr %between% c(-1L, 1L), c >= 9L, f >= 9L),
             bor3(hkr %between% c(-1L, 1L), c >= 9L, f >= 9L))
expect_equal(or3s(hks %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hks %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hkt %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hkt %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hku %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hku %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hkv %in% c(4L, -3L, 2L, 7L, 8L), , logi_k),
             bor3(hkv %in% c(4L, -3L, 2L, 7L, 8L), , logi_k))
expect_equal(or3s(hkw %in% c(4L, -3L, 2L, 7L, 8L), , logi_e),
             bor3(hkw %in% c(4L, -3L, 2L, 7L, 8L), , logi_e))
expect_equal(or3s(hkx %in% c(4L, -3L, 2L, 7L, 8L), , logi_g),
             bor3(hkx %in% c(4L, -3L, 2L, 7L, 8L), , logi_g))
expect_equal(or3s(hky %in% c(4L, -3L, 2L, 7L, 8L), logi_q, ),
             bor3(hky %in% c(4L, -3L, 2L, 7L, 8L), logi_q, ))
expect_equal(or3s(hkz %in% c(4L, -3L, 2L, 7L, 8L), logi_a, ),
             bor3(hkz %in% c(4L, -3L, 2L, 7L, 8L), logi_a, ))
expect_equal(or3s(hla %in% c(4L, -3L, 2L, 7L, 8L), logi_z, ),
             bor3(hla %in% c(4L, -3L, 2L, 7L, 8L), logi_z, ))
expect_equal(or3s(hlb %in% c(4L, -3L, 2L, 7L, 8L), logi_h, logi_n),
             bor3(hlb %in% c(4L, -3L, 2L, 7L, 8L), logi_h, logi_n))
expect_equal(or3s(hlc %in% c(4L, -3L, 2L, 7L, 8L), logi_c, logi_i),
             bor3(hlc %in% c(4L, -3L, 2L, 7L, 8L), logi_c, logi_i))
expect_equal(or3s(hld %in% c(4L, -3L, 2L, 7L, 8L), logi_i, logi_c),
             bor3(hld %in% c(4L, -3L, 2L, 7L, 8L), logi_i, logi_c))
expect_equal(or3s(hle %in% 1:4, , ),
             bor3(hle %in% 1:4, , ))
expect_equal(or3s(hlf %in% 1:4, , ),
             bor3(hlf %in% 1:4, , ))
expect_equal(or3s(hlg %in% 1:4, , ),
             bor3(hlg %in% 1:4, , ))
expect_equal(or3s(hlh %in% 1:4, , logi_v),
             bor3(hlh %in% 1:4, , logi_v))
expect_equal(or3s(hli %in% 1:4, , logi_g),
             bor3(hli %in% 1:4, , logi_g))
expect_equal(or3s(hlj %in% 1:4, , logi_y),
             bor3(hlj %in% 1:4, , logi_y))
expect_equal(or3s(hlk %in% 1:4, logi_u, ),
             bor3(hlk %in% 1:4, logi_u, ))
expect_equal(or3s(hll %in% 1:4, logi_r, ),
             bor3(hll %in% 1:4, logi_r, ))
expect_equal(or3s(hlm %in% 1:4, logi_x, ),
             bor3(hlm %in% 1:4, logi_x, ))
expect_equal(or3s(hln %in% 1:4, logi_u, logi_y),
             bor3(hln %in% 1:4, logi_u, logi_y))
expect_equal(or3s(hlo %in% 1:4, logi_z, logi_l),
             bor3(hlo %in% 1:4, logi_z, logi_l))
expect_equal(or3s(hlp %in% 1:4, logi_m, logi_p),
             bor3(hlp %in% 1:4, logi_m, logi_p))
expect_equal(or3s(hlq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hlq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hlr %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hlr %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hls %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hls %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hlt %in% c(4L, -3L, 2L, 7L, 8L), , !logi_u),
             bor3(hlt %in% c(4L, -3L, 2L, 7L, 8L), , !logi_u))
expect_equal(or3s(hlu %in% c(4L, -3L, 2L, 7L, 8L), , !logi_r),
             bor3(hlu %in% c(4L, -3L, 2L, 7L, 8L), , !logi_r))
expect_equal(or3s(hlv %in% c(4L, -3L, 2L, 7L, 8L), , !logi_e),
             bor3(hlv %in% c(4L, -3L, 2L, 7L, 8L), , !logi_e))
expect_equal(or3s(hlw %in% c(4L, -3L, 2L, 7L, 8L), logi_j, ),
             bor3(hlw %in% c(4L, -3L, 2L, 7L, 8L), logi_j, ))
expect_equal(or3s(hlx %in% c(4L, -3L, 2L, 7L, 8L), logi_h, ),
             bor3(hlx %in% c(4L, -3L, 2L, 7L, 8L), logi_h, ))
expect_equal(or3s(hly %in% c(4L, -3L, 2L, 7L, 8L), logi_a, ),
             bor3(hly %in% c(4L, -3L, 2L, 7L, 8L), logi_a, ))
expect_equal(or3s(hlz %in% c(4L, -3L, 2L, 7L, 8L), logi_b, !logi_d),
             bor3(hlz %in% c(4L, -3L, 2L, 7L, 8L), logi_b, !logi_d))
expect_equal(or3s(hma %in% c(4L, -3L, 2L, 7L, 8L), logi_r, !logi_m),
             bor3(hma %in% c(4L, -3L, 2L, 7L, 8L), logi_r, !logi_m))
expect_equal(or3s(hmb %in% c(4L, -3L, 2L, 7L, 8L), logi_h, !logi_q),
             bor3(hmb %in% c(4L, -3L, 2L, 7L, 8L), logi_h, !logi_q))
expect_equal(or3s(hmc %in% 1:4, , ),
             bor3(hmc %in% 1:4, , ))
expect_equal(or3s(hmd %in% 1:4, , ),
             bor3(hmd %in% 1:4, , ))
expect_equal(or3s(hme %in% 1:4, , ),
             bor3(hme %in% 1:4, , ))
expect_equal(or3s(hmf %in% 1:4, , !logi_x),
             bor3(hmf %in% 1:4, , !logi_x))
expect_equal(or3s(hmg %in% 1:4, , !logi_t),
             bor3(hmg %in% 1:4, , !logi_t))
expect_equal(or3s(hmh %in% 1:4, , !logi_x),
             bor3(hmh %in% 1:4, , !logi_x))
expect_equal(or3s(hmi %in% 1:4, logi_r, ),
             bor3(hmi %in% 1:4, logi_r, ))
expect_equal(or3s(hmj %in% 1:4, logi_m, ),
             bor3(hmj %in% 1:4, logi_m, ))
expect_equal(or3s(hmk %in% 1:4, logi_f, ),
             bor3(hmk %in% 1:4, logi_f, ))
expect_equal(or3s(hml %in% 1:4, logi_t, !logi_t),
             bor3(hml %in% 1:4, logi_t, !logi_t))
expect_equal(or3s(hmm %in% 1:4, logi_a, !logi_g),
             bor3(hmm %in% 1:4, logi_a, !logi_g))
expect_equal(or3s(hmn %in% 1:4, logi_f, !logi_r),
             bor3(hmn %in% 1:4, logi_f, !logi_r))
expect_equal(or3s(hmo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hmo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hmp %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hmp %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hmq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hmq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hmr %in% c(4L, -3L, 2L, 7L, 8L), , m != 0L),
             bor3(hmr %in% c(4L, -3L, 2L, 7L, 8L), , m != 0L))
expect_equal(or3s(hms %in% c(4L, -3L, 2L, 7L, 8L), , d != 1L),
             bor3(hms %in% c(4L, -3L, 2L, 7L, 8L), , d != 1L))
expect_equal(or3s(hmt %in% c(4L, -3L, 2L, 7L, 8L), , n != 9L),
             bor3(hmt %in% c(4L, -3L, 2L, 7L, 8L), , n != 9L))
expect_equal(or3s(hmu %in% c(4L, -3L, 2L, 7L, 8L), logi_g, ),
             bor3(hmu %in% c(4L, -3L, 2L, 7L, 8L), logi_g, ))
expect_equal(or3s(hmv %in% c(4L, -3L, 2L, 7L, 8L), logi_s, ),
             bor3(hmv %in% c(4L, -3L, 2L, 7L, 8L), logi_s, ))
expect_equal(or3s(hmw %in% c(4L, -3L, 2L, 7L, 8L), logi_e, ),
             bor3(hmw %in% c(4L, -3L, 2L, 7L, 8L), logi_e, ))
expect_equal(or3s(hmx %in% c(4L, -3L, 2L, 7L, 8L), logi_a, s != 0L),
             bor3(hmx %in% c(4L, -3L, 2L, 7L, 8L), logi_a, s != 0L))
expect_equal(or3s(hmy %in% c(4L, -3L, 2L, 7L, 8L), logi_x, j != 1L),
             bor3(hmy %in% c(4L, -3L, 2L, 7L, 8L), logi_x, j != 1L))
expect_equal(or3s(hmz %in% c(4L, -3L, 2L, 7L, 8L), logi_j, g != 9L),
             bor3(hmz %in% c(4L, -3L, 2L, 7L, 8L), logi_j, g != 9L))
expect_equal(or3s(hna %in% 1:4, , ),
             bor3(hna %in% 1:4, , ))
expect_equal(or3s(hnb %in% 1:4, , ),
             bor3(hnb %in% 1:4, , ))
expect_equal(or3s(hnc %in% 1:4, , ),
             bor3(hnc %in% 1:4, , ))
expect_equal(or3s(hnd %in% 1:4, , k != 0L),
             bor3(hnd %in% 1:4, , k != 0L))
expect_equal(or3s(hne %in% 1:4, , u != 1L),
             bor3(hne %in% 1:4, , u != 1L))
expect_equal(or3s(hnf %in% 1:4, , i != 9L),
             bor3(hnf %in% 1:4, , i != 9L))
expect_equal(or3s(hng %in% 1:4, logi_y, ),
             bor3(hng %in% 1:4, logi_y, ))
expect_equal(or3s(hnh %in% 1:4, logi_j, ),
             bor3(hnh %in% 1:4, logi_j, ))
expect_equal(or3s(hni %in% 1:4, logi_y, ),
             bor3(hni %in% 1:4, logi_y, ))
expect_equal(or3s(hnj %in% 1:4, logi_j, n != 0L),
             bor3(hnj %in% 1:4, logi_j, n != 0L))
expect_equal(or3s(hnk %in% 1:4, logi_i, o != 1L),
             bor3(hnk %in% 1:4, logi_i, o != 1L))
expect_equal(or3s(hnl %in% 1:4, logi_x, h != 9L),
             bor3(hnl %in% 1:4, logi_x, h != 9L))
expect_equal(or3s(hnm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hnm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hnn %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hnn %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hno %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hno %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hnp %in% c(4L, -3L, 2L, 7L, 8L), , l %between% c(-1L, 1L)),
             bor3(hnp %in% c(4L, -3L, 2L, 7L, 8L), , l %between% c(-1L, 1L)))
expect_equal(or3s(hnq %in% c(4L, -3L, 2L, 7L, 8L), , d %between% c(-1L, 1L)),
             bor3(hnq %in% c(4L, -3L, 2L, 7L, 8L), , d %between% c(-1L, 1L)))
expect_equal(or3s(hnr %in% c(4L, -3L, 2L, 7L, 8L), , c %between% c(-1L, 1L)),
             bor3(hnr %in% c(4L, -3L, 2L, 7L, 8L), , c %between% c(-1L, 1L)))
expect_equal(or3s(hns %in% c(4L, -3L, 2L, 7L, 8L), logi_m, ),
             bor3(hns %in% c(4L, -3L, 2L, 7L, 8L), logi_m, ))
expect_equal(or3s(hnt %in% c(4L, -3L, 2L, 7L, 8L), logi_p, ),
             bor3(hnt %in% c(4L, -3L, 2L, 7L, 8L), logi_p, ))
expect_equal(or3s(hnu %in% c(4L, -3L, 2L, 7L, 8L), logi_f, ),
             bor3(hnu %in% c(4L, -3L, 2L, 7L, 8L), logi_f, ))
expect_equal(or3s(hnv %in% c(4L, -3L, 2L, 7L, 8L), logi_k, q %between% c(-1L, 1L)),
             bor3(hnv %in% c(4L, -3L, 2L, 7L, 8L), logi_k, q %between% c(-1L, 1L)))
expect_equal(or3s(hnw %in% c(4L, -3L, 2L, 7L, 8L), logi_r, h %between% c(-1L, 1L)),
             bor3(hnw %in% c(4L, -3L, 2L, 7L, 8L), logi_r, h %between% c(-1L, 1L)))
expect_equal(or3s(hnx %in% c(4L, -3L, 2L, 7L, 8L), logi_f, d %between% c(-1L, 1L)),
             bor3(hnx %in% c(4L, -3L, 2L, 7L, 8L), logi_f, d %between% c(-1L, 1L)))
expect_equal(or3s(hny %in% 1:4, , ),
             bor3(hny %in% 1:4, , ))
expect_equal(or3s(hnz %in% 1:4, , ),
             bor3(hnz %in% 1:4, , ))
expect_equal(or3s(hoa %in% 1:4, , ),
             bor3(hoa %in% 1:4, , ))
expect_equal(or3s(hob %in% 1:4, , w %between% c(-1L, 1L)),
             bor3(hob %in% 1:4, , w %between% c(-1L, 1L)))
expect_equal(or3s(hoc %in% 1:4, , r %between% c(-1L, 1L)),
             bor3(hoc %in% 1:4, , r %between% c(-1L, 1L)))
expect_equal(or3s(hod %in% 1:4, , c %between% c(-1L, 1L)),
             bor3(hod %in% 1:4, , c %between% c(-1L, 1L)))
expect_equal(or3s(hoe %in% 1:4, logi_b, ),
             bor3(hoe %in% 1:4, logi_b, ))
expect_equal(or3s(hof %in% 1:4, logi_t, ),
             bor3(hof %in% 1:4, logi_t, ))
expect_equal(or3s(hog %in% 1:4, logi_o, ),
             bor3(hog %in% 1:4, logi_o, ))
expect_equal(or3s(hoh %in% 1:4, logi_f, a %between% c(-1L, 1L)),
             bor3(hoh %in% 1:4, logi_f, a %between% c(-1L, 1L)))
expect_equal(or3s(hoi %in% 1:4, logi_k, e %between% c(-1L, 1L)),
             bor3(hoi %in% 1:4, logi_k, e %between% c(-1L, 1L)))
expect_equal(or3s(hoj %in% 1:4, logi_k, q %between% c(-1L, 1L)),
             bor3(hoj %in% 1:4, logi_k, q %between% c(-1L, 1L)))
expect_equal(or3s(hok %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hok %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hol %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hol %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hom %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hom %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hon %in% c(4L, -3L, 2L, 7L, 8L), , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hon %in% c(4L, -3L, 2L, 7L, 8L), , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hoo %in% c(4L, -3L, 2L, 7L, 8L), , d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hoo %in% c(4L, -3L, 2L, 7L, 8L), , d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hop %in% c(4L, -3L, 2L, 7L, 8L), , w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hop %in% c(4L, -3L, 2L, 7L, 8L), , w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hoq %in% c(4L, -3L, 2L, 7L, 8L), logi_h, ),
             bor3(hoq %in% c(4L, -3L, 2L, 7L, 8L), logi_h, ))
expect_equal(or3s(hor %in% c(4L, -3L, 2L, 7L, 8L), logi_x, ),
             bor3(hor %in% c(4L, -3L, 2L, 7L, 8L), logi_x, ))
expect_equal(or3s(hos %in% c(4L, -3L, 2L, 7L, 8L), logi_s, ),
             bor3(hos %in% c(4L, -3L, 2L, 7L, 8L), logi_s, ))
expect_equal(or3s(hot %in% c(4L, -3L, 2L, 7L, 8L), logi_w, h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hot %in% c(4L, -3L, 2L, 7L, 8L), logi_w, h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hou %in% c(4L, -3L, 2L, 7L, 8L), logi_y, k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hou %in% c(4L, -3L, 2L, 7L, 8L), logi_y, k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hov %in% c(4L, -3L, 2L, 7L, 8L), logi_k, k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hov %in% c(4L, -3L, 2L, 7L, 8L), logi_k, k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(how %in% 1:4, , ),
             bor3(how %in% 1:4, , ))
expect_equal(or3s(hox %in% 1:4, , ),
             bor3(hox %in% 1:4, , ))
expect_equal(or3s(hoy %in% 1:4, , ),
             bor3(hoy %in% 1:4, , ))
expect_equal(or3s(hoz %in% 1:4, , h %in% 1:4),
             bor3(hoz %in% 1:4, , h %in% 1:4))
expect_equal(or3s(hpa %in% 1:4, , k %in% 1:4),
             bor3(hpa %in% 1:4, , k %in% 1:4))
expect_equal(or3s(hpb %in% 1:4, , i %in% 1:4),
             bor3(hpb %in% 1:4, , i %in% 1:4))
expect_equal(or3s(hpc %in% 1:4, logi_w, ),
             bor3(hpc %in% 1:4, logi_w, ))
expect_equal(or3s(hpd %in% 1:4, logi_q, ),
             bor3(hpd %in% 1:4, logi_q, ))
expect_equal(or3s(hpe %in% 1:4, logi_c, ),
             bor3(hpe %in% 1:4, logi_c, ))
expect_equal(or3s(hpf %in% 1:4, logi_d, e %in% 1:4),
             bor3(hpf %in% 1:4, logi_d, e %in% 1:4))
expect_equal(or3s(hpg %in% 1:4, logi_j, y %in% 1:4),
             bor3(hpg %in% 1:4, logi_j, y %in% 1:4))
expect_equal(or3s(hph %in% 1:4, logi_j, u %in% 1:4),
             bor3(hph %in% 1:4, logi_j, u %in% 1:4))
expect_equal(or3s(hpi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hpi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hpj %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hpj %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hpk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hpk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hpl %in% c(4L, -3L, 2L, 7L, 8L), , d < 0L),
             bor3(hpl %in% c(4L, -3L, 2L, 7L, 8L), , d < 0L))
expect_equal(or3s(hpm %in% c(4L, -3L, 2L, 7L, 8L), , n < 1L),
             bor3(hpm %in% c(4L, -3L, 2L, 7L, 8L), , n < 1L))
expect_equal(or3s(hpn %in% c(4L, -3L, 2L, 7L, 8L), , g < 9L),
             bor3(hpn %in% c(4L, -3L, 2L, 7L, 8L), , g < 9L))
expect_equal(or3s(hpo %in% c(4L, -3L, 2L, 7L, 8L), logi_t, ),
             bor3(hpo %in% c(4L, -3L, 2L, 7L, 8L), logi_t, ))
expect_equal(or3s(hpp %in% c(4L, -3L, 2L, 7L, 8L), logi_t, ),
             bor3(hpp %in% c(4L, -3L, 2L, 7L, 8L), logi_t, ))
expect_equal(or3s(hpq %in% c(4L, -3L, 2L, 7L, 8L), logi_r, ),
             bor3(hpq %in% c(4L, -3L, 2L, 7L, 8L), logi_r, ))
expect_equal(or3s(hpr %in% c(4L, -3L, 2L, 7L, 8L), logi_k, b < 0L),
             bor3(hpr %in% c(4L, -3L, 2L, 7L, 8L), logi_k, b < 0L))
expect_equal(or3s(hps %in% c(4L, -3L, 2L, 7L, 8L), logi_d, h < 1L),
             bor3(hps %in% c(4L, -3L, 2L, 7L, 8L), logi_d, h < 1L))
expect_equal(or3s(hpt %in% c(4L, -3L, 2L, 7L, 8L), logi_v, i < 9L),
             bor3(hpt %in% c(4L, -3L, 2L, 7L, 8L), logi_v, i < 9L))
expect_equal(or3s(hpu %in% 1:4, , ),
             bor3(hpu %in% 1:4, , ))
expect_equal(or3s(hpv %in% 1:4, , ),
             bor3(hpv %in% 1:4, , ))
expect_equal(or3s(hpw %in% 1:4, , ),
             bor3(hpw %in% 1:4, , ))
expect_equal(or3s(hpx %in% 1:4, , f < 0L),
             bor3(hpx %in% 1:4, , f < 0L))
expect_equal(or3s(hpy %in% 1:4, , n < 1L),
             bor3(hpy %in% 1:4, , n < 1L))
expect_equal(or3s(hpz %in% 1:4, , d < 9L),
             bor3(hpz %in% 1:4, , d < 9L))
expect_equal(or3s(hqa %in% 1:4, logi_d, ),
             bor3(hqa %in% 1:4, logi_d, ))
expect_equal(or3s(hqb %in% 1:4, logi_z, ),
             bor3(hqb %in% 1:4, logi_z, ))
expect_equal(or3s(hqc %in% 1:4, logi_q, ),
             bor3(hqc %in% 1:4, logi_q, ))
expect_equal(or3s(hqd %in% 1:4, logi_e, b < 0L),
             bor3(hqd %in% 1:4, logi_e, b < 0L))
expect_equal(or3s(hqe %in% 1:4, logi_t, y < 1L),
             bor3(hqe %in% 1:4, logi_t, y < 1L))
expect_equal(or3s(hqf %in% 1:4, logi_x, l < 9L),
             bor3(hqf %in% 1:4, logi_x, l < 9L))
expect_equal(or3s(hqg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hqg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hqh %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hqh %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hqi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hqi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hqj %in% c(4L, -3L, 2L, 7L, 8L), , s <= 0L),
             bor3(hqj %in% c(4L, -3L, 2L, 7L, 8L), , s <= 0L))
expect_equal(or3s(hqk %in% c(4L, -3L, 2L, 7L, 8L), , i <= 1L),
             bor3(hqk %in% c(4L, -3L, 2L, 7L, 8L), , i <= 1L))
expect_equal(or3s(hql %in% c(4L, -3L, 2L, 7L, 8L), , l <= 9L),
             bor3(hql %in% c(4L, -3L, 2L, 7L, 8L), , l <= 9L))
expect_equal(or3s(hqm %in% c(4L, -3L, 2L, 7L, 8L), logi_i, ),
             bor3(hqm %in% c(4L, -3L, 2L, 7L, 8L), logi_i, ))
expect_equal(or3s(hqn %in% c(4L, -3L, 2L, 7L, 8L), logi_y, ),
             bor3(hqn %in% c(4L, -3L, 2L, 7L, 8L), logi_y, ))
expect_equal(or3s(hqo %in% c(4L, -3L, 2L, 7L, 8L), logi_p, ),
             bor3(hqo %in% c(4L, -3L, 2L, 7L, 8L), logi_p, ))
expect_equal(or3s(hqp %in% c(4L, -3L, 2L, 7L, 8L), logi_q, x <= 0L),
             bor3(hqp %in% c(4L, -3L, 2L, 7L, 8L), logi_q, x <= 0L))
expect_equal(or3s(hqq %in% c(4L, -3L, 2L, 7L, 8L), logi_v, f <= 1L),
             bor3(hqq %in% c(4L, -3L, 2L, 7L, 8L), logi_v, f <= 1L))
expect_equal(or3s(hqr %in% c(4L, -3L, 2L, 7L, 8L), logi_l, l <= 9L),
             bor3(hqr %in% c(4L, -3L, 2L, 7L, 8L), logi_l, l <= 9L))
expect_equal(or3s(hqs %in% 1:4, , ),
             bor3(hqs %in% 1:4, , ))
expect_equal(or3s(hqt %in% 1:4, , ),
             bor3(hqt %in% 1:4, , ))
expect_equal(or3s(hqu %in% 1:4, , ),
             bor3(hqu %in% 1:4, , ))
expect_equal(or3s(hqv %in% 1:4, , j <= 0L),
             bor3(hqv %in% 1:4, , j <= 0L))
expect_equal(or3s(hqw %in% 1:4, , g <= 1L),
             bor3(hqw %in% 1:4, , g <= 1L))
expect_equal(or3s(hqx %in% 1:4, , x <= 9L),
             bor3(hqx %in% 1:4, , x <= 9L))
expect_equal(or3s(hqy %in% 1:4, logi_q, ),
             bor3(hqy %in% 1:4, logi_q, ))
expect_equal(or3s(hqz %in% 1:4, logi_m, ),
             bor3(hqz %in% 1:4, logi_m, ))
expect_equal(or3s(hra %in% 1:4, logi_z, ),
             bor3(hra %in% 1:4, logi_z, ))
expect_equal(or3s(hrb %in% 1:4, logi_l, k <= 0L),
             bor3(hrb %in% 1:4, logi_l, k <= 0L))
expect_equal(or3s(hrc %in% 1:4, logi_a, d <= 1L),
             bor3(hrc %in% 1:4, logi_a, d <= 1L))
expect_equal(or3s(hrd %in% 1:4, logi_s, o <= 9L),
             bor3(hrd %in% 1:4, logi_s, o <= 9L))
expect_equal(or3s(hre %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hre %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hrf %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hrf %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hrg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hrg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hrh %in% c(4L, -3L, 2L, 7L, 8L), , v == 0L),
             bor3(hrh %in% c(4L, -3L, 2L, 7L, 8L), , v == 0L))
expect_equal(or3s(hri %in% c(4L, -3L, 2L, 7L, 8L), , u == 1L),
             bor3(hri %in% c(4L, -3L, 2L, 7L, 8L), , u == 1L))
expect_equal(or3s(hrj %in% c(4L, -3L, 2L, 7L, 8L), , y == 9L),
             bor3(hrj %in% c(4L, -3L, 2L, 7L, 8L), , y == 9L))
expect_equal(or3s(hrk %in% c(4L, -3L, 2L, 7L, 8L), logi_i, ),
             bor3(hrk %in% c(4L, -3L, 2L, 7L, 8L), logi_i, ))
expect_equal(or3s(hrl %in% c(4L, -3L, 2L, 7L, 8L), logi_v, ),
             bor3(hrl %in% c(4L, -3L, 2L, 7L, 8L), logi_v, ))
expect_equal(or3s(hrm %in% c(4L, -3L, 2L, 7L, 8L), logi_e, ),
             bor3(hrm %in% c(4L, -3L, 2L, 7L, 8L), logi_e, ))
expect_equal(or3s(hrn %in% c(4L, -3L, 2L, 7L, 8L), logi_p, i == 0L),
             bor3(hrn %in% c(4L, -3L, 2L, 7L, 8L), logi_p, i == 0L))
expect_equal(or3s(hro %in% c(4L, -3L, 2L, 7L, 8L), logi_u, y == 1L),
             bor3(hro %in% c(4L, -3L, 2L, 7L, 8L), logi_u, y == 1L))
expect_equal(or3s(hrp %in% c(4L, -3L, 2L, 7L, 8L), logi_n, t == 9L),
             bor3(hrp %in% c(4L, -3L, 2L, 7L, 8L), logi_n, t == 9L))
expect_equal(or3s(hrq %in% 1:4, , ),
             bor3(hrq %in% 1:4, , ))
expect_equal(or3s(hrr %in% 1:4, , ),
             bor3(hrr %in% 1:4, , ))
expect_equal(or3s(hrs %in% 1:4, , ),
             bor3(hrs %in% 1:4, , ))
expect_equal(or3s(hrt %in% 1:4, , q == 0L),
             bor3(hrt %in% 1:4, , q == 0L))
expect_equal(or3s(hru %in% 1:4, , c == 1L),
             bor3(hru %in% 1:4, , c == 1L))
expect_equal(or3s(hrv %in% 1:4, , a == 9L),
             bor3(hrv %in% 1:4, , a == 9L))
expect_equal(or3s(hrw %in% 1:4, logi_b, ),
             bor3(hrw %in% 1:4, logi_b, ))
expect_equal(or3s(hrx %in% 1:4, logi_o, ),
             bor3(hrx %in% 1:4, logi_o, ))
expect_equal(or3s(hry %in% 1:4, logi_o, ),
             bor3(hry %in% 1:4, logi_o, ))
expect_equal(or3s(hrz %in% 1:4, logi_i, w == 0L),
             bor3(hrz %in% 1:4, logi_i, w == 0L))
expect_equal(or3s(hsa %in% 1:4, logi_r, p == 1L),
             bor3(hsa %in% 1:4, logi_r, p == 1L))
expect_equal(or3s(hsb %in% 1:4, logi_d, p == 9L),
             bor3(hsb %in% 1:4, logi_d, p == 9L))
expect_equal(or3s(hsc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hsc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hsd %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hsd %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hse %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hse %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hsf %in% c(4L, -3L, 2L, 7L, 8L), , e > 0L),
             bor3(hsf %in% c(4L, -3L, 2L, 7L, 8L), , e > 0L))
expect_equal(or3s(hsg %in% c(4L, -3L, 2L, 7L, 8L), , e > 1L),
             bor3(hsg %in% c(4L, -3L, 2L, 7L, 8L), , e > 1L))
expect_equal(or3s(hsh %in% c(4L, -3L, 2L, 7L, 8L), , w > 9L),
             bor3(hsh %in% c(4L, -3L, 2L, 7L, 8L), , w > 9L))
expect_equal(or3s(hsi %in% c(4L, -3L, 2L, 7L, 8L), logi_b, ),
             bor3(hsi %in% c(4L, -3L, 2L, 7L, 8L), logi_b, ))
expect_equal(or3s(hsj %in% c(4L, -3L, 2L, 7L, 8L), logi_r, ),
             bor3(hsj %in% c(4L, -3L, 2L, 7L, 8L), logi_r, ))
expect_equal(or3s(hsk %in% c(4L, -3L, 2L, 7L, 8L), logi_b, ),
             bor3(hsk %in% c(4L, -3L, 2L, 7L, 8L), logi_b, ))
expect_equal(or3s(hsl %in% c(4L, -3L, 2L, 7L, 8L), logi_p, j > 0L),
             bor3(hsl %in% c(4L, -3L, 2L, 7L, 8L), logi_p, j > 0L))
expect_equal(or3s(hsm %in% c(4L, -3L, 2L, 7L, 8L), logi_u, r > 1L),
             bor3(hsm %in% c(4L, -3L, 2L, 7L, 8L), logi_u, r > 1L))
expect_equal(or3s(hsn %in% c(4L, -3L, 2L, 7L, 8L), logi_e, p > 9L),
             bor3(hsn %in% c(4L, -3L, 2L, 7L, 8L), logi_e, p > 9L))
expect_equal(or3s(hso %in% 1:4, , ),
             bor3(hso %in% 1:4, , ))
expect_equal(or3s(hsp %in% 1:4, , ),
             bor3(hsp %in% 1:4, , ))
expect_equal(or3s(hsq %in% 1:4, , ),
             bor3(hsq %in% 1:4, , ))
expect_equal(or3s(hsr %in% 1:4, , d > 0L),
             bor3(hsr %in% 1:4, , d > 0L))
expect_equal(or3s(hss %in% 1:4, , s > 1L),
             bor3(hss %in% 1:4, , s > 1L))
expect_equal(or3s(hst %in% 1:4, , s > 9L),
             bor3(hst %in% 1:4, , s > 9L))
expect_equal(or3s(hsu %in% 1:4, logi_w, ),
             bor3(hsu %in% 1:4, logi_w, ))
expect_equal(or3s(hsv %in% 1:4, logi_r, ),
             bor3(hsv %in% 1:4, logi_r, ))
expect_equal(or3s(hsw %in% 1:4, logi_j, ),
             bor3(hsw %in% 1:4, logi_j, ))
expect_equal(or3s(hsx %in% 1:4, logi_z, d > 0L),
             bor3(hsx %in% 1:4, logi_z, d > 0L))
expect_equal(or3s(hsy %in% 1:4, logi_l, o > 1L),
             bor3(hsy %in% 1:4, logi_l, o > 1L))
expect_equal(or3s(hsz %in% 1:4, logi_m, k > 9L),
             bor3(hsz %in% 1:4, logi_m, k > 9L))
expect_equal(or3s(hta %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hta %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(htb %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(htb %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(htc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(htc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(htd %in% c(4L, -3L, 2L, 7L, 8L), , t >= 0L),
             bor3(htd %in% c(4L, -3L, 2L, 7L, 8L), , t >= 0L))
expect_equal(or3s(hte %in% c(4L, -3L, 2L, 7L, 8L), , p >= 1L),
             bor3(hte %in% c(4L, -3L, 2L, 7L, 8L), , p >= 1L))
expect_equal(or3s(htf %in% c(4L, -3L, 2L, 7L, 8L), , r >= 9L),
             bor3(htf %in% c(4L, -3L, 2L, 7L, 8L), , r >= 9L))
expect_equal(or3s(htg %in% c(4L, -3L, 2L, 7L, 8L), logi_m, ),
             bor3(htg %in% c(4L, -3L, 2L, 7L, 8L), logi_m, ))
expect_equal(or3s(hth %in% c(4L, -3L, 2L, 7L, 8L), logi_q, ),
             bor3(hth %in% c(4L, -3L, 2L, 7L, 8L), logi_q, ))
expect_equal(or3s(hti %in% c(4L, -3L, 2L, 7L, 8L), logi_c, ),
             bor3(hti %in% c(4L, -3L, 2L, 7L, 8L), logi_c, ))
expect_equal(or3s(htj %in% c(4L, -3L, 2L, 7L, 8L), logi_a, b >= 0L),
             bor3(htj %in% c(4L, -3L, 2L, 7L, 8L), logi_a, b >= 0L))
expect_equal(or3s(htk %in% c(4L, -3L, 2L, 7L, 8L), logi_c, i >= 1L),
             bor3(htk %in% c(4L, -3L, 2L, 7L, 8L), logi_c, i >= 1L))
expect_equal(or3s(htl %in% c(4L, -3L, 2L, 7L, 8L), logi_s, h >= 9L),
             bor3(htl %in% c(4L, -3L, 2L, 7L, 8L), logi_s, h >= 9L))
expect_equal(or3s(htm %in% 1:4, , ),
             bor3(htm %in% 1:4, , ))
expect_equal(or3s(htn %in% 1:4, , ),
             bor3(htn %in% 1:4, , ))
expect_equal(or3s(hto %in% 1:4, , ),
             bor3(hto %in% 1:4, , ))
expect_equal(or3s(htp %in% 1:4, , e >= 0L),
             bor3(htp %in% 1:4, , e >= 0L))
expect_equal(or3s(htq %in% 1:4, , s >= 1L),
             bor3(htq %in% 1:4, , s >= 1L))
expect_equal(or3s(htr %in% 1:4, , l >= 9L),
             bor3(htr %in% 1:4, , l >= 9L))
expect_equal(or3s(hts %in% 1:4, logi_n, ),
             bor3(hts %in% 1:4, logi_n, ))
expect_equal(or3s(htt %in% 1:4, logi_p, ),
             bor3(htt %in% 1:4, logi_p, ))
expect_equal(or3s(htu %in% 1:4, logi_e, ),
             bor3(htu %in% 1:4, logi_e, ))
expect_equal(or3s(htv %in% 1:4, logi_l, x >= 0L),
             bor3(htv %in% 1:4, logi_l, x >= 0L))
expect_equal(or3s(htw %in% 1:4, logi_t, d >= 1L),
             bor3(htw %in% 1:4, logi_t, d >= 1L))
expect_equal(or3s(htx %in% 1:4, logi_i, m >= 9L),
             bor3(htx %in% 1:4, logi_i, m >= 9L))
expect_equal(or3s(hty %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hty %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(htz %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(htz %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hua %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hua %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hub %in% c(4L, -3L, 2L, 7L, 8L), , logi_c),
             bor3(hub %in% c(4L, -3L, 2L, 7L, 8L), , logi_c))
expect_equal(or3s(huc %in% c(4L, -3L, 2L, 7L, 8L), , logi_j),
             bor3(huc %in% c(4L, -3L, 2L, 7L, 8L), , logi_j))
expect_equal(or3s(hud %in% c(4L, -3L, 2L, 7L, 8L), , logi_x),
             bor3(hud %in% c(4L, -3L, 2L, 7L, 8L), , logi_x))
expect_equal(or3s(hue %in% c(4L, -3L, 2L, 7L, 8L), !logi_g, ),
             bor3(hue %in% c(4L, -3L, 2L, 7L, 8L), !logi_g, ))
expect_equal(or3s(huf %in% c(4L, -3L, 2L, 7L, 8L), !logi_m, ),
             bor3(huf %in% c(4L, -3L, 2L, 7L, 8L), !logi_m, ))
expect_equal(or3s(hug %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, ),
             bor3(hug %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, ))
expect_equal(or3s(huh %in% c(4L, -3L, 2L, 7L, 8L), !logi_y, logi_h),
             bor3(huh %in% c(4L, -3L, 2L, 7L, 8L), !logi_y, logi_h))
expect_equal(or3s(hui %in% c(4L, -3L, 2L, 7L, 8L), !logi_f, logi_g),
             bor3(hui %in% c(4L, -3L, 2L, 7L, 8L), !logi_f, logi_g))
expect_equal(or3s(huj %in% c(4L, -3L, 2L, 7L, 8L), !logi_q, logi_i),
             bor3(huj %in% c(4L, -3L, 2L, 7L, 8L), !logi_q, logi_i))
expect_equal(or3s(huk %in% 1:4, , ),
             bor3(huk %in% 1:4, , ))
expect_equal(or3s(hul %in% 1:4, , ),
             bor3(hul %in% 1:4, , ))
expect_equal(or3s(hum %in% 1:4, , ),
             bor3(hum %in% 1:4, , ))
expect_equal(or3s(hun %in% 1:4, , logi_d),
             bor3(hun %in% 1:4, , logi_d))
expect_equal(or3s(huo %in% 1:4, , logi_w),
             bor3(huo %in% 1:4, , logi_w))
expect_equal(or3s(hup %in% 1:4, , logi_f),
             bor3(hup %in% 1:4, , logi_f))
expect_equal(or3s(huq %in% 1:4, !logi_h, ),
             bor3(huq %in% 1:4, !logi_h, ))
expect_equal(or3s(hur %in% 1:4, !logi_q, ),
             bor3(hur %in% 1:4, !logi_q, ))
expect_equal(or3s(hus %in% 1:4, !logi_e, ),
             bor3(hus %in% 1:4, !logi_e, ))
expect_equal(or3s(hut %in% 1:4, !logi_s, logi_g),
             bor3(hut %in% 1:4, !logi_s, logi_g))
expect_equal(or3s(huu %in% 1:4, !logi_h, logi_k),
             bor3(huu %in% 1:4, !logi_h, logi_k))
expect_equal(or3s(huv %in% 1:4, !logi_q, logi_n),
             bor3(huv %in% 1:4, !logi_q, logi_n))
expect_equal(or3s(huw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(huw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hux %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hux %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(huy %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(huy %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(huz %in% c(4L, -3L, 2L, 7L, 8L), , !logi_u),
             bor3(huz %in% c(4L, -3L, 2L, 7L, 8L), , !logi_u))
expect_equal(or3s(hva %in% c(4L, -3L, 2L, 7L, 8L), , !logi_f),
             bor3(hva %in% c(4L, -3L, 2L, 7L, 8L), , !logi_f))
expect_equal(or3s(hvb %in% c(4L, -3L, 2L, 7L, 8L), , !logi_f),
             bor3(hvb %in% c(4L, -3L, 2L, 7L, 8L), , !logi_f))
expect_equal(or3s(hvc %in% c(4L, -3L, 2L, 7L, 8L), !logi_k, ),
             bor3(hvc %in% c(4L, -3L, 2L, 7L, 8L), !logi_k, ))
expect_equal(or3s(hvd %in% c(4L, -3L, 2L, 7L, 8L), !logi_m, ),
             bor3(hvd %in% c(4L, -3L, 2L, 7L, 8L), !logi_m, ))
expect_equal(or3s(hve %in% c(4L, -3L, 2L, 7L, 8L), !logi_r, ),
             bor3(hve %in% c(4L, -3L, 2L, 7L, 8L), !logi_r, ))
expect_equal(or3s(hvf %in% c(4L, -3L, 2L, 7L, 8L), !logi_b, !logi_z),
             bor3(hvf %in% c(4L, -3L, 2L, 7L, 8L), !logi_b, !logi_z))
expect_equal(or3s(hvg %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, !logi_o),
             bor3(hvg %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, !logi_o))
expect_equal(or3s(hvh %in% c(4L, -3L, 2L, 7L, 8L), !logi_v, !logi_p),
             bor3(hvh %in% c(4L, -3L, 2L, 7L, 8L), !logi_v, !logi_p))
expect_equal(or3s(hvi %in% 1:4, , ),
             bor3(hvi %in% 1:4, , ))
expect_equal(or3s(hvj %in% 1:4, , ),
             bor3(hvj %in% 1:4, , ))
expect_equal(or3s(hvk %in% 1:4, , ),
             bor3(hvk %in% 1:4, , ))
expect_equal(or3s(hvl %in% 1:4, , !logi_g),
             bor3(hvl %in% 1:4, , !logi_g))
expect_equal(or3s(hvm %in% 1:4, , !logi_d),
             bor3(hvm %in% 1:4, , !logi_d))
expect_equal(or3s(hvn %in% 1:4, , !logi_d),
             bor3(hvn %in% 1:4, , !logi_d))
expect_equal(or3s(hvo %in% 1:4, !logi_g, ),
             bor3(hvo %in% 1:4, !logi_g, ))
expect_equal(or3s(hvp %in% 1:4, !logi_q, ),
             bor3(hvp %in% 1:4, !logi_q, ))
expect_equal(or3s(hvq %in% 1:4, !logi_i, ),
             bor3(hvq %in% 1:4, !logi_i, ))
expect_equal(or3s(hvr %in% 1:4, !logi_w, !logi_n),
             bor3(hvr %in% 1:4, !logi_w, !logi_n))
expect_equal(or3s(hvs %in% 1:4, !logi_z, !logi_f),
             bor3(hvs %in% 1:4, !logi_z, !logi_f))
expect_equal(or3s(hvt %in% 1:4, !logi_a, !logi_y),
             bor3(hvt %in% 1:4, !logi_a, !logi_y))
expect_equal(or3s(hvu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hvu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hvv %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hvv %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hvw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hvw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hvx %in% c(4L, -3L, 2L, 7L, 8L), , f != 0L),
             bor3(hvx %in% c(4L, -3L, 2L, 7L, 8L), , f != 0L))
expect_equal(or3s(hvy %in% c(4L, -3L, 2L, 7L, 8L), , t != 1L),
             bor3(hvy %in% c(4L, -3L, 2L, 7L, 8L), , t != 1L))
expect_equal(or3s(hvz %in% c(4L, -3L, 2L, 7L, 8L), , z != 9L),
             bor3(hvz %in% c(4L, -3L, 2L, 7L, 8L), , z != 9L))
expect_equal(or3s(hwa %in% c(4L, -3L, 2L, 7L, 8L), !logi_u, ),
             bor3(hwa %in% c(4L, -3L, 2L, 7L, 8L), !logi_u, ))
expect_equal(or3s(hwb %in% c(4L, -3L, 2L, 7L, 8L), !logi_j, ),
             bor3(hwb %in% c(4L, -3L, 2L, 7L, 8L), !logi_j, ))
expect_equal(or3s(hwc %in% c(4L, -3L, 2L, 7L, 8L), !logi_q, ),
             bor3(hwc %in% c(4L, -3L, 2L, 7L, 8L), !logi_q, ))
expect_equal(or3s(hwd %in% c(4L, -3L, 2L, 7L, 8L), !logi_m, u != 0L),
             bor3(hwd %in% c(4L, -3L, 2L, 7L, 8L), !logi_m, u != 0L))
expect_equal(or3s(hwe %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, c != 1L),
             bor3(hwe %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, c != 1L))
expect_equal(or3s(hwf %in% c(4L, -3L, 2L, 7L, 8L), !logi_j, h != 9L),
             bor3(hwf %in% c(4L, -3L, 2L, 7L, 8L), !logi_j, h != 9L))
expect_equal(or3s(hwg %in% 1:4, , ),
             bor3(hwg %in% 1:4, , ))
expect_equal(or3s(hwh %in% 1:4, , ),
             bor3(hwh %in% 1:4, , ))
expect_equal(or3s(hwi %in% 1:4, , ),
             bor3(hwi %in% 1:4, , ))
expect_equal(or3s(hwj %in% 1:4, , r != 0L),
             bor3(hwj %in% 1:4, , r != 0L))
expect_equal(or3s(hwk %in% 1:4, , v != 1L),
             bor3(hwk %in% 1:4, , v != 1L))
expect_equal(or3s(hwl %in% 1:4, , n != 9L),
             bor3(hwl %in% 1:4, , n != 9L))
expect_equal(or3s(hwm %in% 1:4, !logi_k, ),
             bor3(hwm %in% 1:4, !logi_k, ))
expect_equal(or3s(hwn %in% 1:4, !logi_a, ),
             bor3(hwn %in% 1:4, !logi_a, ))
expect_equal(or3s(hwo %in% 1:4, !logi_v, ),
             bor3(hwo %in% 1:4, !logi_v, ))
expect_equal(or3s(hwp %in% 1:4, !logi_u, r != 0L),
             bor3(hwp %in% 1:4, !logi_u, r != 0L))
expect_equal(or3s(hwq %in% 1:4, !logi_w, k != 1L),
             bor3(hwq %in% 1:4, !logi_w, k != 1L))
expect_equal(or3s(hwr %in% 1:4, !logi_b, a != 9L),
             bor3(hwr %in% 1:4, !logi_b, a != 9L))
expect_equal(or3s(hws %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hws %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hwt %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hwt %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hwu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hwu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hwv %in% c(4L, -3L, 2L, 7L, 8L), , u %between% c(-1L, 1L)),
             bor3(hwv %in% c(4L, -3L, 2L, 7L, 8L), , u %between% c(-1L, 1L)))
expect_equal(or3s(hww %in% c(4L, -3L, 2L, 7L, 8L), , q %between% c(-1L, 1L)),
             bor3(hww %in% c(4L, -3L, 2L, 7L, 8L), , q %between% c(-1L, 1L)))
expect_equal(or3s(hwx %in% c(4L, -3L, 2L, 7L, 8L), , z %between% c(-1L, 1L)),
             bor3(hwx %in% c(4L, -3L, 2L, 7L, 8L), , z %between% c(-1L, 1L)))
expect_equal(or3s(hwy %in% c(4L, -3L, 2L, 7L, 8L), !logi_j, ),
             bor3(hwy %in% c(4L, -3L, 2L, 7L, 8L), !logi_j, ))
expect_equal(or3s(hwz %in% c(4L, -3L, 2L, 7L, 8L), !logi_d, ),
             bor3(hwz %in% c(4L, -3L, 2L, 7L, 8L), !logi_d, ))
expect_equal(or3s(hxa %in% c(4L, -3L, 2L, 7L, 8L), !logi_n, ),
             bor3(hxa %in% c(4L, -3L, 2L, 7L, 8L), !logi_n, ))
expect_equal(or3s(hxb %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, b %between% c(-1L, 1L)),
             bor3(hxb %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, b %between% c(-1L, 1L)))
expect_equal(or3s(hxc %in% c(4L, -3L, 2L, 7L, 8L), !logi_c, f %between% c(-1L, 1L)),
             bor3(hxc %in% c(4L, -3L, 2L, 7L, 8L), !logi_c, f %between% c(-1L, 1L)))
expect_equal(or3s(hxd %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, a %between% c(-1L, 1L)),
             bor3(hxd %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, a %between% c(-1L, 1L)))
expect_equal(or3s(hxe %in% 1:4, , ),
             bor3(hxe %in% 1:4, , ))
expect_equal(or3s(hxf %in% 1:4, , ),
             bor3(hxf %in% 1:4, , ))
expect_equal(or3s(hxg %in% 1:4, , ),
             bor3(hxg %in% 1:4, , ))
expect_equal(or3s(hxh %in% 1:4, , e %between% c(-1L, 1L)),
             bor3(hxh %in% 1:4, , e %between% c(-1L, 1L)))
expect_equal(or3s(hxi %in% 1:4, , n %between% c(-1L, 1L)),
             bor3(hxi %in% 1:4, , n %between% c(-1L, 1L)))
expect_equal(or3s(hxj %in% 1:4, , y %between% c(-1L, 1L)),
             bor3(hxj %in% 1:4, , y %between% c(-1L, 1L)))
expect_equal(or3s(hxk %in% 1:4, !logi_u, ),
             bor3(hxk %in% 1:4, !logi_u, ))
expect_equal(or3s(hxl %in% 1:4, !logi_z, ),
             bor3(hxl %in% 1:4, !logi_z, ))
expect_equal(or3s(hxm %in% 1:4, !logi_q, ),
             bor3(hxm %in% 1:4, !logi_q, ))
expect_equal(or3s(hxn %in% 1:4, !logi_o, f %between% c(-1L, 1L)),
             bor3(hxn %in% 1:4, !logi_o, f %between% c(-1L, 1L)))
expect_equal(or3s(hxo %in% 1:4, !logi_a, v %between% c(-1L, 1L)),
             bor3(hxo %in% 1:4, !logi_a, v %between% c(-1L, 1L)))
expect_equal(or3s(hxp %in% 1:4, !logi_h, i %between% c(-1L, 1L)),
             bor3(hxp %in% 1:4, !logi_h, i %between% c(-1L, 1L)))
expect_equal(or3s(hxq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hxq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hxr %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hxr %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hxs %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hxs %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hxt %in% c(4L, -3L, 2L, 7L, 8L), , q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hxt %in% c(4L, -3L, 2L, 7L, 8L), , q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hxu %in% c(4L, -3L, 2L, 7L, 8L), , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hxu %in% c(4L, -3L, 2L, 7L, 8L), , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hxv %in% c(4L, -3L, 2L, 7L, 8L), , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hxv %in% c(4L, -3L, 2L, 7L, 8L), , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hxw %in% c(4L, -3L, 2L, 7L, 8L), !logi_b, ),
             bor3(hxw %in% c(4L, -3L, 2L, 7L, 8L), !logi_b, ))
expect_equal(or3s(hxx %in% c(4L, -3L, 2L, 7L, 8L), !logi_i, ),
             bor3(hxx %in% c(4L, -3L, 2L, 7L, 8L), !logi_i, ))
expect_equal(or3s(hxy %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, ),
             bor3(hxy %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, ))
expect_equal(or3s(hxz %in% c(4L, -3L, 2L, 7L, 8L), !logi_t, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hxz %in% c(4L, -3L, 2L, 7L, 8L), !logi_t, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hya %in% c(4L, -3L, 2L, 7L, 8L), !logi_r, d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hya %in% c(4L, -3L, 2L, 7L, 8L), !logi_r, d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hyb %in% c(4L, -3L, 2L, 7L, 8L), !logi_a, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(hyb %in% c(4L, -3L, 2L, 7L, 8L), !logi_a, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(hyc %in% 1:4, , ),
             bor3(hyc %in% 1:4, , ))
expect_equal(or3s(hyd %in% 1:4, , ),
             bor3(hyd %in% 1:4, , ))
expect_equal(or3s(hye %in% 1:4, , ),
             bor3(hye %in% 1:4, , ))
expect_equal(or3s(hyf %in% 1:4, , t %in% 1:4),
             bor3(hyf %in% 1:4, , t %in% 1:4))
expect_equal(or3s(hyg %in% 1:4, , q %in% 1:4),
             bor3(hyg %in% 1:4, , q %in% 1:4))
expect_equal(or3s(hyh %in% 1:4, , l %in% 1:4),
             bor3(hyh %in% 1:4, , l %in% 1:4))
expect_equal(or3s(hyi %in% 1:4, !logi_u, ),
             bor3(hyi %in% 1:4, !logi_u, ))
expect_equal(or3s(hyj %in% 1:4, !logi_v, ),
             bor3(hyj %in% 1:4, !logi_v, ))
expect_equal(or3s(hyk %in% 1:4, !logi_p, ),
             bor3(hyk %in% 1:4, !logi_p, ))
expect_equal(or3s(hyl %in% 1:4, !logi_t, r %in% 1:4),
             bor3(hyl %in% 1:4, !logi_t, r %in% 1:4))
expect_equal(or3s(hym %in% 1:4, !logi_v, d %in% 1:4),
             bor3(hym %in% 1:4, !logi_v, d %in% 1:4))
expect_equal(or3s(hyn %in% 1:4, !logi_x, y %in% 1:4),
             bor3(hyn %in% 1:4, !logi_x, y %in% 1:4))
expect_equal(or3s(hyo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hyo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hyp %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hyp %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hyq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(hyq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(hyr %in% c(4L, -3L, 2L, 7L, 8L), , u < 0L),
             bor3(hyr %in% c(4L, -3L, 2L, 7L, 8L), , u < 0L))
expect_equal(or3s(hys %in% c(4L, -3L, 2L, 7L, 8L), , p < 1L),
             bor3(hys %in% c(4L, -3L, 2L, 7L, 8L), , p < 1L))
expect_equal(or3s(hyt %in% c(4L, -3L, 2L, 7L, 8L), , s < 9L),
             bor3(hyt %in% c(4L, -3L, 2L, 7L, 8L), , s < 9L))
expect_equal(or3s(hyu %in% c(4L, -3L, 2L, 7L, 8L), !logi_q, ),
             bor3(hyu %in% c(4L, -3L, 2L, 7L, 8L), !logi_q, ))
expect_equal(or3s(hyv %in% c(4L, -3L, 2L, 7L, 8L), !logi_b, ),
             bor3(hyv %in% c(4L, -3L, 2L, 7L, 8L), !logi_b, ))
expect_equal(or3s(hyw %in% c(4L, -3L, 2L, 7L, 8L), !logi_p, ),
             bor3(hyw %in% c(4L, -3L, 2L, 7L, 8L), !logi_p, ))
expect_equal(or3s(hyx %in% c(4L, -3L, 2L, 7L, 8L), !logi_n, w < 0L),
             bor3(hyx %in% c(4L, -3L, 2L, 7L, 8L), !logi_n, w < 0L))
expect_equal(or3s(hyy %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, u < 1L),
             bor3(hyy %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, u < 1L))
expect_equal(or3s(hyz %in% c(4L, -3L, 2L, 7L, 8L), !logi_r, s < 9L),
             bor3(hyz %in% c(4L, -3L, 2L, 7L, 8L), !logi_r, s < 9L))
expect_equal(or3s(ia %in% 1:4, , ),
             bor3(ia %in% 1:4, , ))
expect_equal(or3s(ib %in% 1:4, , ),
             bor3(ib %in% 1:4, , ))
expect_equal(or3s(ic %in% 1:4, , ),
             bor3(ic %in% 1:4, , ))
expect_equal(or3s(id %in% 1:4, , d < 0L),
             bor3(id %in% 1:4, , d < 0L))
expect_equal(or3s(ie %in% 1:4, , k < 1L),
             bor3(ie %in% 1:4, , k < 1L))
expect_equal(or3s(IF %in% 1:4, , t < 9L),
             bor3(IF %in% 1:4, , t < 9L))
expect_equal(or3s(ig %in% 1:4, !logi_d, ),
             bor3(ig %in% 1:4, !logi_d, ))
expect_equal(or3s(ih %in% 1:4, !logi_n, ),
             bor3(ih %in% 1:4, !logi_n, ))
expect_equal(or3s(ii %in% 1:4, !logi_c, ),
             bor3(ii %in% 1:4, !logi_c, ))
expect_equal(or3s(ij %in% 1:4, !logi_x, h < 0L),
             bor3(ij %in% 1:4, !logi_x, h < 0L))
expect_equal(or3s(ik %in% 1:4, !logi_z, q < 1L),
             bor3(ik %in% 1:4, !logi_z, q < 1L))
expect_equal(or3s(il %in% 1:4, !logi_v, j < 9L),
             bor3(il %in% 1:4, !logi_v, j < 9L))
expect_equal(or3s(im %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(im %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(IN %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(IN %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(io %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(io %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ip %in% c(4L, -3L, 2L, 7L, 8L), , c <= 0L),
             bor3(ip %in% c(4L, -3L, 2L, 7L, 8L), , c <= 0L))
expect_equal(or3s(iq %in% c(4L, -3L, 2L, 7L, 8L), , w <= 1L),
             bor3(iq %in% c(4L, -3L, 2L, 7L, 8L), , w <= 1L))
expect_equal(or3s(ir %in% c(4L, -3L, 2L, 7L, 8L), , c <= 9L),
             bor3(ir %in% c(4L, -3L, 2L, 7L, 8L), , c <= 9L))
expect_equal(or3s(is %in% c(4L, -3L, 2L, 7L, 8L), !logi_s, ),
             bor3(is %in% c(4L, -3L, 2L, 7L, 8L), !logi_s, ))
expect_equal(or3s(it %in% c(4L, -3L, 2L, 7L, 8L), !logi_w, ),
             bor3(it %in% c(4L, -3L, 2L, 7L, 8L), !logi_w, ))
expect_equal(or3s(iu %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, ),
             bor3(iu %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, ))
expect_equal(or3s(iv %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, o <= 0L),
             bor3(iv %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, o <= 0L))
expect_equal(or3s(iw %in% c(4L, -3L, 2L, 7L, 8L), !logi_f, q <= 1L),
             bor3(iw %in% c(4L, -3L, 2L, 7L, 8L), !logi_f, q <= 1L))
expect_equal(or3s(ix %in% c(4L, -3L, 2L, 7L, 8L), !logi_q, m <= 9L),
             bor3(ix %in% c(4L, -3L, 2L, 7L, 8L), !logi_q, m <= 9L))
expect_equal(or3s(iy %in% 1:4, , ),
             bor3(iy %in% 1:4, , ))
expect_equal(or3s(iz %in% 1:4, , ),
             bor3(iz %in% 1:4, , ))
expect_equal(or3s(iaa %in% 1:4, , ),
             bor3(iaa %in% 1:4, , ))
expect_equal(or3s(iab %in% 1:4, , j <= 0L),
             bor3(iab %in% 1:4, , j <= 0L))
expect_equal(or3s(iac %in% 1:4, , h <= 1L),
             bor3(iac %in% 1:4, , h <= 1L))
expect_equal(or3s(iad %in% 1:4, , u <= 9L),
             bor3(iad %in% 1:4, , u <= 9L))
expect_equal(or3s(iae %in% 1:4, !logi_k, ),
             bor3(iae %in% 1:4, !logi_k, ))
expect_equal(or3s(iaf %in% 1:4, !logi_k, ),
             bor3(iaf %in% 1:4, !logi_k, ))
expect_equal(or3s(iag %in% 1:4, !logi_p, ),
             bor3(iag %in% 1:4, !logi_p, ))
expect_equal(or3s(iah %in% 1:4, !logi_t, g <= 0L),
             bor3(iah %in% 1:4, !logi_t, g <= 0L))
expect_equal(or3s(iai %in% 1:4, !logi_r, r <= 1L),
             bor3(iai %in% 1:4, !logi_r, r <= 1L))
expect_equal(or3s(iaj %in% 1:4, !logi_a, n <= 9L),
             bor3(iaj %in% 1:4, !logi_a, n <= 9L))
expect_equal(or3s(iak %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iak %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ial %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ial %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iam %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iam %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ian %in% c(4L, -3L, 2L, 7L, 8L), , i == 0L),
             bor3(ian %in% c(4L, -3L, 2L, 7L, 8L), , i == 0L))
expect_equal(or3s(iao %in% c(4L, -3L, 2L, 7L, 8L), , d == 1L),
             bor3(iao %in% c(4L, -3L, 2L, 7L, 8L), , d == 1L))
expect_equal(or3s(iap %in% c(4L, -3L, 2L, 7L, 8L), , r == 9L),
             bor3(iap %in% c(4L, -3L, 2L, 7L, 8L), , r == 9L))
expect_equal(or3s(iaq %in% c(4L, -3L, 2L, 7L, 8L), !logi_s, ),
             bor3(iaq %in% c(4L, -3L, 2L, 7L, 8L), !logi_s, ))
expect_equal(or3s(iar %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, ),
             bor3(iar %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, ))
expect_equal(or3s(ias %in% c(4L, -3L, 2L, 7L, 8L), !logi_w, ),
             bor3(ias %in% c(4L, -3L, 2L, 7L, 8L), !logi_w, ))
expect_equal(or3s(iat %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, z == 0L),
             bor3(iat %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, z == 0L))
expect_equal(or3s(iau %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, z == 1L),
             bor3(iau %in% c(4L, -3L, 2L, 7L, 8L), !logi_h, z == 1L))
expect_equal(or3s(iav %in% c(4L, -3L, 2L, 7L, 8L), !logi_v, t == 9L),
             bor3(iav %in% c(4L, -3L, 2L, 7L, 8L), !logi_v, t == 9L))
expect_equal(or3s(iaw %in% 1:4, , ),
             bor3(iaw %in% 1:4, , ))
expect_equal(or3s(iax %in% 1:4, , ),
             bor3(iax %in% 1:4, , ))
expect_equal(or3s(iay %in% 1:4, , ),
             bor3(iay %in% 1:4, , ))
expect_equal(or3s(iaz %in% 1:4, , d == 0L),
             bor3(iaz %in% 1:4, , d == 0L))
expect_equal(or3s(iba %in% 1:4, , n == 1L),
             bor3(iba %in% 1:4, , n == 1L))
expect_equal(or3s(ibb %in% 1:4, , n == 9L),
             bor3(ibb %in% 1:4, , n == 9L))
expect_equal(or3s(ibc %in% 1:4, !logi_r, ),
             bor3(ibc %in% 1:4, !logi_r, ))
expect_equal(or3s(ibd %in% 1:4, !logi_a, ),
             bor3(ibd %in% 1:4, !logi_a, ))
expect_equal(or3s(ibe %in% 1:4, !logi_w, ),
             bor3(ibe %in% 1:4, !logi_w, ))
expect_equal(or3s(ibf %in% 1:4, !logi_r, x == 0L),
             bor3(ibf %in% 1:4, !logi_r, x == 0L))
expect_equal(or3s(ibg %in% 1:4, !logi_y, i == 1L),
             bor3(ibg %in% 1:4, !logi_y, i == 1L))
expect_equal(or3s(ibh %in% 1:4, !logi_w, r == 9L),
             bor3(ibh %in% 1:4, !logi_w, r == 9L))
expect_equal(or3s(ibi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ibi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ibj %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ibj %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ibk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ibk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ibl %in% c(4L, -3L, 2L, 7L, 8L), , w > 0L),
             bor3(ibl %in% c(4L, -3L, 2L, 7L, 8L), , w > 0L))
expect_equal(or3s(ibm %in% c(4L, -3L, 2L, 7L, 8L), , i > 1L),
             bor3(ibm %in% c(4L, -3L, 2L, 7L, 8L), , i > 1L))
expect_equal(or3s(ibn %in% c(4L, -3L, 2L, 7L, 8L), , u > 9L),
             bor3(ibn %in% c(4L, -3L, 2L, 7L, 8L), , u > 9L))
expect_equal(or3s(ibo %in% c(4L, -3L, 2L, 7L, 8L), !logi_d, ),
             bor3(ibo %in% c(4L, -3L, 2L, 7L, 8L), !logi_d, ))
expect_equal(or3s(ibp %in% c(4L, -3L, 2L, 7L, 8L), !logi_w, ),
             bor3(ibp %in% c(4L, -3L, 2L, 7L, 8L), !logi_w, ))
expect_equal(or3s(ibq %in% c(4L, -3L, 2L, 7L, 8L), !logi_t, ),
             bor3(ibq %in% c(4L, -3L, 2L, 7L, 8L), !logi_t, ))
expect_equal(or3s(ibr %in% c(4L, -3L, 2L, 7L, 8L), !logi_w, y > 0L),
             bor3(ibr %in% c(4L, -3L, 2L, 7L, 8L), !logi_w, y > 0L))
expect_equal(or3s(ibs %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, d > 1L),
             bor3(ibs %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, d > 1L))
expect_equal(or3s(ibt %in% c(4L, -3L, 2L, 7L, 8L), !logi_r, o > 9L),
             bor3(ibt %in% c(4L, -3L, 2L, 7L, 8L), !logi_r, o > 9L))
expect_equal(or3s(ibu %in% 1:4, , ),
             bor3(ibu %in% 1:4, , ))
expect_equal(or3s(ibv %in% 1:4, , ),
             bor3(ibv %in% 1:4, , ))
expect_equal(or3s(ibw %in% 1:4, , ),
             bor3(ibw %in% 1:4, , ))
expect_equal(or3s(ibx %in% 1:4, , q > 0L),
             bor3(ibx %in% 1:4, , q > 0L))
expect_equal(or3s(iby %in% 1:4, , h > 1L),
             bor3(iby %in% 1:4, , h > 1L))
expect_equal(or3s(ibz %in% 1:4, , d > 9L),
             bor3(ibz %in% 1:4, , d > 9L))
expect_equal(or3s(ica %in% 1:4, !logi_h, ),
             bor3(ica %in% 1:4, !logi_h, ))
expect_equal(or3s(icb %in% 1:4, !logi_n, ),
             bor3(icb %in% 1:4, !logi_n, ))
expect_equal(or3s(icc %in% 1:4, !logi_r, ),
             bor3(icc %in% 1:4, !logi_r, ))
expect_equal(or3s(icd %in% 1:4, !logi_d, c > 0L),
             bor3(icd %in% 1:4, !logi_d, c > 0L))
expect_equal(or3s(ice %in% 1:4, !logi_h, y > 1L),
             bor3(ice %in% 1:4, !logi_h, y > 1L))
expect_equal(or3s(icf %in% 1:4, !logi_z, z > 9L),
             bor3(icf %in% 1:4, !logi_z, z > 9L))
expect_equal(or3s(icg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(icg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ich %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ich %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ici %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ici %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(icj %in% c(4L, -3L, 2L, 7L, 8L), , f >= 0L),
             bor3(icj %in% c(4L, -3L, 2L, 7L, 8L), , f >= 0L))
expect_equal(or3s(ick %in% c(4L, -3L, 2L, 7L, 8L), , f >= 1L),
             bor3(ick %in% c(4L, -3L, 2L, 7L, 8L), , f >= 1L))
expect_equal(or3s(icl %in% c(4L, -3L, 2L, 7L, 8L), , t >= 9L),
             bor3(icl %in% c(4L, -3L, 2L, 7L, 8L), , t >= 9L))
expect_equal(or3s(icm %in% c(4L, -3L, 2L, 7L, 8L), !logi_u, ),
             bor3(icm %in% c(4L, -3L, 2L, 7L, 8L), !logi_u, ))
expect_equal(or3s(icn %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, ),
             bor3(icn %in% c(4L, -3L, 2L, 7L, 8L), !logi_l, ))
expect_equal(or3s(ico %in% c(4L, -3L, 2L, 7L, 8L), !logi_m, ),
             bor3(ico %in% c(4L, -3L, 2L, 7L, 8L), !logi_m, ))
expect_equal(or3s(icp %in% c(4L, -3L, 2L, 7L, 8L), !logi_a, z >= 0L),
             bor3(icp %in% c(4L, -3L, 2L, 7L, 8L), !logi_a, z >= 0L))
expect_equal(or3s(icq %in% c(4L, -3L, 2L, 7L, 8L), !logi_v, p >= 1L),
             bor3(icq %in% c(4L, -3L, 2L, 7L, 8L), !logi_v, p >= 1L))
expect_equal(or3s(icr %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, t >= 9L),
             bor3(icr %in% c(4L, -3L, 2L, 7L, 8L), !logi_z, t >= 9L))
expect_equal(or3s(ics %in% 1:4, , ),
             bor3(ics %in% 1:4, , ))
expect_equal(or3s(ict %in% 1:4, , ),
             bor3(ict %in% 1:4, , ))
expect_equal(or3s(icu %in% 1:4, , ),
             bor3(icu %in% 1:4, , ))
expect_equal(or3s(icv %in% 1:4, , r >= 0L),
             bor3(icv %in% 1:4, , r >= 0L))
expect_equal(or3s(icw %in% 1:4, , z >= 1L),
             bor3(icw %in% 1:4, , z >= 1L))
expect_equal(or3s(icx %in% 1:4, , h >= 9L),
             bor3(icx %in% 1:4, , h >= 9L))
expect_equal(or3s(icy %in% 1:4, !logi_a, ),
             bor3(icy %in% 1:4, !logi_a, ))
expect_equal(or3s(icz %in% 1:4, !logi_r, ),
             bor3(icz %in% 1:4, !logi_r, ))
expect_equal(or3s(ida %in% 1:4, !logi_l, ),
             bor3(ida %in% 1:4, !logi_l, ))
expect_equal(or3s(idb %in% 1:4, !logi_n, d >= 0L),
             bor3(idb %in% 1:4, !logi_n, d >= 0L))
expect_equal(or3s(idc %in% 1:4, !logi_z, j >= 1L),
             bor3(idc %in% 1:4, !logi_z, j >= 1L))
expect_equal(or3s(idd %in% 1:4, !logi_q, z >= 9L),
             bor3(idd %in% 1:4, !logi_q, z >= 9L))
expect_equal(or3s(ide %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ide %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(idf %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(idf %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(idg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(idg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(idh %in% c(4L, -3L, 2L, 7L, 8L), , logi_n),
             bor3(idh %in% c(4L, -3L, 2L, 7L, 8L), , logi_n))
expect_equal(or3s(idi %in% c(4L, -3L, 2L, 7L, 8L), , logi_o),
             bor3(idi %in% c(4L, -3L, 2L, 7L, 8L), , logi_o))
expect_equal(or3s(idj %in% c(4L, -3L, 2L, 7L, 8L), , logi_l),
             bor3(idj %in% c(4L, -3L, 2L, 7L, 8L), , logi_l))
expect_equal(or3s(idk %in% c(4L, -3L, 2L, 7L, 8L), d != 0L, ),
             bor3(idk %in% c(4L, -3L, 2L, 7L, 8L), d != 0L, ))
expect_equal(or3s(idl %in% c(4L, -3L, 2L, 7L, 8L), g != 1L, ),
             bor3(idl %in% c(4L, -3L, 2L, 7L, 8L), g != 1L, ))
expect_equal(or3s(idm %in% c(4L, -3L, 2L, 7L, 8L), w != 9L, ),
             bor3(idm %in% c(4L, -3L, 2L, 7L, 8L), w != 9L, ))
expect_equal(or3s(idn %in% c(4L, -3L, 2L, 7L, 8L), t != 0L, logi_v),
             bor3(idn %in% c(4L, -3L, 2L, 7L, 8L), t != 0L, logi_v))
expect_equal(or3s(ido %in% c(4L, -3L, 2L, 7L, 8L), u != 1L, logi_j),
             bor3(ido %in% c(4L, -3L, 2L, 7L, 8L), u != 1L, logi_j))
expect_equal(or3s(idp %in% c(4L, -3L, 2L, 7L, 8L), d != 9L, logi_u),
             bor3(idp %in% c(4L, -3L, 2L, 7L, 8L), d != 9L, logi_u))
expect_equal(or3s(idq %in% 1:4, , ),
             bor3(idq %in% 1:4, , ))
expect_equal(or3s(idr %in% 1:4, , ),
             bor3(idr %in% 1:4, , ))
expect_equal(or3s(ids %in% 1:4, , ),
             bor3(ids %in% 1:4, , ))
expect_equal(or3s(idt %in% 1:4, , logi_o),
             bor3(idt %in% 1:4, , logi_o))
expect_equal(or3s(idu %in% 1:4, , logi_z),
             bor3(idu %in% 1:4, , logi_z))
expect_equal(or3s(idv %in% 1:4, , logi_h),
             bor3(idv %in% 1:4, , logi_h))
expect_equal(or3s(idw %in% 1:4, f != 0L, ),
             bor3(idw %in% 1:4, f != 0L, ))
expect_equal(or3s(idx %in% 1:4, z != 1L, ),
             bor3(idx %in% 1:4, z != 1L, ))
expect_equal(or3s(idy %in% 1:4, x != 9L, ),
             bor3(idy %in% 1:4, x != 9L, ))
expect_equal(or3s(idz %in% 1:4, h != 0L, logi_e),
             bor3(idz %in% 1:4, h != 0L, logi_e))
expect_equal(or3s(iea %in% 1:4, c != 1L, logi_q),
             bor3(iea %in% 1:4, c != 1L, logi_q))
expect_equal(or3s(ieb %in% 1:4, j != 9L, logi_z),
             bor3(ieb %in% 1:4, j != 9L, logi_z))
expect_equal(or3s(iec %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iec %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ied %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ied %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iee %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iee %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ief %in% c(4L, -3L, 2L, 7L, 8L), , !logi_q),
             bor3(ief %in% c(4L, -3L, 2L, 7L, 8L), , !logi_q))
expect_equal(or3s(ieg %in% c(4L, -3L, 2L, 7L, 8L), , !logi_u),
             bor3(ieg %in% c(4L, -3L, 2L, 7L, 8L), , !logi_u))
expect_equal(or3s(ieh %in% c(4L, -3L, 2L, 7L, 8L), , !logi_d),
             bor3(ieh %in% c(4L, -3L, 2L, 7L, 8L), , !logi_d))
expect_equal(or3s(iei %in% c(4L, -3L, 2L, 7L, 8L), a != 0L, ),
             bor3(iei %in% c(4L, -3L, 2L, 7L, 8L), a != 0L, ))
expect_equal(or3s(iej %in% c(4L, -3L, 2L, 7L, 8L), m != 1L, ),
             bor3(iej %in% c(4L, -3L, 2L, 7L, 8L), m != 1L, ))
expect_equal(or3s(iek %in% c(4L, -3L, 2L, 7L, 8L), b != 9L, ),
             bor3(iek %in% c(4L, -3L, 2L, 7L, 8L), b != 9L, ))
expect_equal(or3s(iel %in% c(4L, -3L, 2L, 7L, 8L), i != 0L, !logi_k),
             bor3(iel %in% c(4L, -3L, 2L, 7L, 8L), i != 0L, !logi_k))
expect_equal(or3s(iem %in% c(4L, -3L, 2L, 7L, 8L), p != 1L, !logi_g),
             bor3(iem %in% c(4L, -3L, 2L, 7L, 8L), p != 1L, !logi_g))
expect_equal(or3s(ien %in% c(4L, -3L, 2L, 7L, 8L), q != 9L, !logi_p),
             bor3(ien %in% c(4L, -3L, 2L, 7L, 8L), q != 9L, !logi_p))
expect_equal(or3s(ieo %in% 1:4, , ),
             bor3(ieo %in% 1:4, , ))
expect_equal(or3s(iep %in% 1:4, , ),
             bor3(iep %in% 1:4, , ))
expect_equal(or3s(ieq %in% 1:4, , ),
             bor3(ieq %in% 1:4, , ))
expect_equal(or3s(ier %in% 1:4, , !logi_x),
             bor3(ier %in% 1:4, , !logi_x))
expect_equal(or3s(ies %in% 1:4, , !logi_p),
             bor3(ies %in% 1:4, , !logi_p))
expect_equal(or3s(iet %in% 1:4, , !logi_l),
             bor3(iet %in% 1:4, , !logi_l))
expect_equal(or3s(ieu %in% 1:4, m != 0L, ),
             bor3(ieu %in% 1:4, m != 0L, ))
expect_equal(or3s(iev %in% 1:4, h != 1L, ),
             bor3(iev %in% 1:4, h != 1L, ))
expect_equal(or3s(iew %in% 1:4, p != 9L, ),
             bor3(iew %in% 1:4, p != 9L, ))
expect_equal(or3s(iex %in% 1:4, w != 0L, !logi_b),
             bor3(iex %in% 1:4, w != 0L, !logi_b))
expect_equal(or3s(iey %in% 1:4, e != 1L, !logi_d),
             bor3(iey %in% 1:4, e != 1L, !logi_d))
expect_equal(or3s(iez %in% 1:4, m != 9L, !logi_f),
             bor3(iez %in% 1:4, m != 9L, !logi_f))
expect_equal(or3s(ifa %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ifa %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ifb %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ifb %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ifc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ifc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ifd %in% c(4L, -3L, 2L, 7L, 8L), , t != 0L),
             bor3(ifd %in% c(4L, -3L, 2L, 7L, 8L), , t != 0L))
expect_equal(or3s(ife %in% c(4L, -3L, 2L, 7L, 8L), , v != 1L),
             bor3(ife %in% c(4L, -3L, 2L, 7L, 8L), , v != 1L))
expect_equal(or3s(iff %in% c(4L, -3L, 2L, 7L, 8L), , o != 9L),
             bor3(iff %in% c(4L, -3L, 2L, 7L, 8L), , o != 9L))
expect_equal(or3s(ifg %in% c(4L, -3L, 2L, 7L, 8L), e != 0L, ),
             bor3(ifg %in% c(4L, -3L, 2L, 7L, 8L), e != 0L, ))
expect_equal(or3s(ifh %in% c(4L, -3L, 2L, 7L, 8L), o != 1L, ),
             bor3(ifh %in% c(4L, -3L, 2L, 7L, 8L), o != 1L, ))
expect_equal(or3s(ifi %in% c(4L, -3L, 2L, 7L, 8L), i != 9L, ),
             bor3(ifi %in% c(4L, -3L, 2L, 7L, 8L), i != 9L, ))
expect_equal(or3s(ifj %in% c(4L, -3L, 2L, 7L, 8L), e != 0L, y != 0L),
             bor3(ifj %in% c(4L, -3L, 2L, 7L, 8L), e != 0L, y != 0L))
expect_equal(or3s(ifk %in% c(4L, -3L, 2L, 7L, 8L), b != 1L, m != 1L),
             bor3(ifk %in% c(4L, -3L, 2L, 7L, 8L), b != 1L, m != 1L))
expect_equal(or3s(ifl %in% c(4L, -3L, 2L, 7L, 8L), u != 9L, s != 9L),
             bor3(ifl %in% c(4L, -3L, 2L, 7L, 8L), u != 9L, s != 9L))
expect_equal(or3s(ifm %in% 1:4, , ),
             bor3(ifm %in% 1:4, , ))
expect_equal(or3s(ifn %in% 1:4, , ),
             bor3(ifn %in% 1:4, , ))
expect_equal(or3s(ifo %in% 1:4, , ),
             bor3(ifo %in% 1:4, , ))
expect_equal(or3s(ifp %in% 1:4, , r != 0L),
             bor3(ifp %in% 1:4, , r != 0L))
expect_equal(or3s(ifq %in% 1:4, , w != 1L),
             bor3(ifq %in% 1:4, , w != 1L))
expect_equal(or3s(ifr %in% 1:4, , z != 9L),
             bor3(ifr %in% 1:4, , z != 9L))
expect_equal(or3s(ifs %in% 1:4, i != 0L, ),
             bor3(ifs %in% 1:4, i != 0L, ))
expect_equal(or3s(ift %in% 1:4, h != 1L, ),
             bor3(ift %in% 1:4, h != 1L, ))
expect_equal(or3s(ifu %in% 1:4, w != 9L, ),
             bor3(ifu %in% 1:4, w != 9L, ))
expect_equal(or3s(ifv %in% 1:4, p != 0L, m != 0L),
             bor3(ifv %in% 1:4, p != 0L, m != 0L))
expect_equal(or3s(ifw %in% 1:4, w != 1L, l != 1L),
             bor3(ifw %in% 1:4, w != 1L, l != 1L))
expect_equal(or3s(ifx %in% 1:4, r != 9L, r != 9L),
             bor3(ifx %in% 1:4, r != 9L, r != 9L))
expect_equal(or3s(ify %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ify %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ifz %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ifz %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iga %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iga %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(igb %in% c(4L, -3L, 2L, 7L, 8L), , p %between% c(-1L, 1L)),
             bor3(igb %in% c(4L, -3L, 2L, 7L, 8L), , p %between% c(-1L, 1L)))
expect_equal(or3s(igc %in% c(4L, -3L, 2L, 7L, 8L), , u %between% c(-1L, 1L)),
             bor3(igc %in% c(4L, -3L, 2L, 7L, 8L), , u %between% c(-1L, 1L)))
expect_equal(or3s(igd %in% c(4L, -3L, 2L, 7L, 8L), , o %between% c(-1L, 1L)),
             bor3(igd %in% c(4L, -3L, 2L, 7L, 8L), , o %between% c(-1L, 1L)))
expect_equal(or3s(ige %in% c(4L, -3L, 2L, 7L, 8L), z != 0L, ),
             bor3(ige %in% c(4L, -3L, 2L, 7L, 8L), z != 0L, ))
expect_equal(or3s(igf %in% c(4L, -3L, 2L, 7L, 8L), o != 1L, ),
             bor3(igf %in% c(4L, -3L, 2L, 7L, 8L), o != 1L, ))
expect_equal(or3s(igg %in% c(4L, -3L, 2L, 7L, 8L), b != 9L, ),
             bor3(igg %in% c(4L, -3L, 2L, 7L, 8L), b != 9L, ))
expect_equal(or3s(igh %in% c(4L, -3L, 2L, 7L, 8L), v != 0L, t %between% c(-1L, 1L)),
             bor3(igh %in% c(4L, -3L, 2L, 7L, 8L), v != 0L, t %between% c(-1L, 1L)))
expect_equal(or3s(igi %in% c(4L, -3L, 2L, 7L, 8L), m != 1L, e %between% c(-1L, 1L)),
             bor3(igi %in% c(4L, -3L, 2L, 7L, 8L), m != 1L, e %between% c(-1L, 1L)))
expect_equal(or3s(igj %in% c(4L, -3L, 2L, 7L, 8L), n != 9L, g %between% c(-1L, 1L)),
             bor3(igj %in% c(4L, -3L, 2L, 7L, 8L), n != 9L, g %between% c(-1L, 1L)))
expect_equal(or3s(igk %in% 1:4, , ),
             bor3(igk %in% 1:4, , ))
expect_equal(or3s(igl %in% 1:4, , ),
             bor3(igl %in% 1:4, , ))
expect_equal(or3s(igm %in% 1:4, , ),
             bor3(igm %in% 1:4, , ))
expect_equal(or3s(ign %in% 1:4, , k %between% c(-1L, 1L)),
             bor3(ign %in% 1:4, , k %between% c(-1L, 1L)))
expect_equal(or3s(igo %in% 1:4, , q %between% c(-1L, 1L)),
             bor3(igo %in% 1:4, , q %between% c(-1L, 1L)))
expect_equal(or3s(igp %in% 1:4, , f %between% c(-1L, 1L)),
             bor3(igp %in% 1:4, , f %between% c(-1L, 1L)))
expect_equal(or3s(igq %in% 1:4, w != 0L, ),
             bor3(igq %in% 1:4, w != 0L, ))
expect_equal(or3s(igr %in% 1:4, w != 1L, ),
             bor3(igr %in% 1:4, w != 1L, ))
expect_equal(or3s(igs %in% 1:4, z != 9L, ),
             bor3(igs %in% 1:4, z != 9L, ))
expect_equal(or3s(igt %in% 1:4, c != 0L, e %between% c(-1L, 1L)),
             bor3(igt %in% 1:4, c != 0L, e %between% c(-1L, 1L)))
expect_equal(or3s(igu %in% 1:4, d != 1L, x %between% c(-1L, 1L)),
             bor3(igu %in% 1:4, d != 1L, x %between% c(-1L, 1L)))
expect_equal(or3s(igv %in% 1:4, o != 9L, n %between% c(-1L, 1L)),
             bor3(igv %in% 1:4, o != 9L, n %between% c(-1L, 1L)))
expect_equal(or3s(igw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(igw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(igx %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(igx %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(igy %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(igy %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(igz %in% c(4L, -3L, 2L, 7L, 8L), , w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(igz %in% c(4L, -3L, 2L, 7L, 8L), , w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(iha %in% c(4L, -3L, 2L, 7L, 8L), , r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(iha %in% c(4L, -3L, 2L, 7L, 8L), , r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ihb %in% c(4L, -3L, 2L, 7L, 8L), , r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ihb %in% c(4L, -3L, 2L, 7L, 8L), , r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ihc %in% c(4L, -3L, 2L, 7L, 8L), k != 0L, ),
             bor3(ihc %in% c(4L, -3L, 2L, 7L, 8L), k != 0L, ))
expect_equal(or3s(ihd %in% c(4L, -3L, 2L, 7L, 8L), o != 1L, ),
             bor3(ihd %in% c(4L, -3L, 2L, 7L, 8L), o != 1L, ))
expect_equal(or3s(ihe %in% c(4L, -3L, 2L, 7L, 8L), u != 9L, ),
             bor3(ihe %in% c(4L, -3L, 2L, 7L, 8L), u != 9L, ))
expect_equal(or3s(ihf %in% c(4L, -3L, 2L, 7L, 8L), r != 0L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ihf %in% c(4L, -3L, 2L, 7L, 8L), r != 0L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ihg %in% c(4L, -3L, 2L, 7L, 8L), r != 1L, x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ihg %in% c(4L, -3L, 2L, 7L, 8L), r != 1L, x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ihh %in% c(4L, -3L, 2L, 7L, 8L), y != 9L, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ihh %in% c(4L, -3L, 2L, 7L, 8L), y != 9L, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ihi %in% 1:4, , ),
             bor3(ihi %in% 1:4, , ))
expect_equal(or3s(ihj %in% 1:4, , ),
             bor3(ihj %in% 1:4, , ))
expect_equal(or3s(ihk %in% 1:4, , ),
             bor3(ihk %in% 1:4, , ))
expect_equal(or3s(ihl %in% 1:4, , g %in% 1:4),
             bor3(ihl %in% 1:4, , g %in% 1:4))
expect_equal(or3s(ihm %in% 1:4, , w %in% 1:4),
             bor3(ihm %in% 1:4, , w %in% 1:4))
expect_equal(or3s(ihn %in% 1:4, , b %in% 1:4),
             bor3(ihn %in% 1:4, , b %in% 1:4))
expect_equal(or3s(iho %in% 1:4, x != 0L, ),
             bor3(iho %in% 1:4, x != 0L, ))
expect_equal(or3s(ihp %in% 1:4, t != 1L, ),
             bor3(ihp %in% 1:4, t != 1L, ))
expect_equal(or3s(ihq %in% 1:4, i != 9L, ),
             bor3(ihq %in% 1:4, i != 9L, ))
expect_equal(or3s(ihr %in% 1:4, p != 0L, v %in% 1:4),
             bor3(ihr %in% 1:4, p != 0L, v %in% 1:4))
expect_equal(or3s(ihs %in% 1:4, p != 1L, l %in% 1:4),
             bor3(ihs %in% 1:4, p != 1L, l %in% 1:4))
expect_equal(or3s(iht %in% 1:4, h != 9L, k %in% 1:4),
             bor3(iht %in% 1:4, h != 9L, k %in% 1:4))
expect_equal(or3s(ihu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ihu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ihv %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ihv %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ihw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ihw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ihx %in% c(4L, -3L, 2L, 7L, 8L), , c < 0L),
             bor3(ihx %in% c(4L, -3L, 2L, 7L, 8L), , c < 0L))
expect_equal(or3s(ihy %in% c(4L, -3L, 2L, 7L, 8L), , d < 1L),
             bor3(ihy %in% c(4L, -3L, 2L, 7L, 8L), , d < 1L))
expect_equal(or3s(ihz %in% c(4L, -3L, 2L, 7L, 8L), , g < 9L),
             bor3(ihz %in% c(4L, -3L, 2L, 7L, 8L), , g < 9L))
expect_equal(or3s(iia %in% c(4L, -3L, 2L, 7L, 8L), r != 0L, ),
             bor3(iia %in% c(4L, -3L, 2L, 7L, 8L), r != 0L, ))
expect_equal(or3s(iib %in% c(4L, -3L, 2L, 7L, 8L), y != 1L, ),
             bor3(iib %in% c(4L, -3L, 2L, 7L, 8L), y != 1L, ))
expect_equal(or3s(iic %in% c(4L, -3L, 2L, 7L, 8L), y != 9L, ),
             bor3(iic %in% c(4L, -3L, 2L, 7L, 8L), y != 9L, ))
expect_equal(or3s(iid %in% c(4L, -3L, 2L, 7L, 8L), m != 0L, b < 0L),
             bor3(iid %in% c(4L, -3L, 2L, 7L, 8L), m != 0L, b < 0L))
expect_equal(or3s(iie %in% c(4L, -3L, 2L, 7L, 8L), r != 1L, e < 1L),
             bor3(iie %in% c(4L, -3L, 2L, 7L, 8L), r != 1L, e < 1L))
expect_equal(or3s(iif %in% c(4L, -3L, 2L, 7L, 8L), d != 9L, w < 9L),
             bor3(iif %in% c(4L, -3L, 2L, 7L, 8L), d != 9L, w < 9L))
expect_equal(or3s(iig %in% 1:4, , ),
             bor3(iig %in% 1:4, , ))
expect_equal(or3s(iih %in% 1:4, , ),
             bor3(iih %in% 1:4, , ))
expect_equal(or3s(iii %in% 1:4, , ),
             bor3(iii %in% 1:4, , ))
expect_equal(or3s(iij %in% 1:4, , m < 0L),
             bor3(iij %in% 1:4, , m < 0L))
expect_equal(or3s(iik %in% 1:4, , p < 1L),
             bor3(iik %in% 1:4, , p < 1L))
expect_equal(or3s(iil %in% 1:4, , p < 9L),
             bor3(iil %in% 1:4, , p < 9L))
expect_equal(or3s(iim %in% 1:4, t != 0L, ),
             bor3(iim %in% 1:4, t != 0L, ))
expect_equal(or3s(iin %in% 1:4, h != 1L, ),
             bor3(iin %in% 1:4, h != 1L, ))
expect_equal(or3s(iio %in% 1:4, g != 9L, ),
             bor3(iio %in% 1:4, g != 9L, ))
expect_equal(or3s(iip %in% 1:4, k != 0L, t < 0L),
             bor3(iip %in% 1:4, k != 0L, t < 0L))
expect_equal(or3s(iiq %in% 1:4, u != 1L, r < 1L),
             bor3(iiq %in% 1:4, u != 1L, r < 1L))
expect_equal(or3s(iir %in% 1:4, x != 9L, g < 9L),
             bor3(iir %in% 1:4, x != 9L, g < 9L))
expect_equal(or3s(iis %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iis %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iit %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iit %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iiu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iiu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iiv %in% c(4L, -3L, 2L, 7L, 8L), , c <= 0L),
             bor3(iiv %in% c(4L, -3L, 2L, 7L, 8L), , c <= 0L))
expect_equal(or3s(iiw %in% c(4L, -3L, 2L, 7L, 8L), , c <= 1L),
             bor3(iiw %in% c(4L, -3L, 2L, 7L, 8L), , c <= 1L))
expect_equal(or3s(iix %in% c(4L, -3L, 2L, 7L, 8L), , v <= 9L),
             bor3(iix %in% c(4L, -3L, 2L, 7L, 8L), , v <= 9L))
expect_equal(or3s(iiy %in% c(4L, -3L, 2L, 7L, 8L), y != 0L, ),
             bor3(iiy %in% c(4L, -3L, 2L, 7L, 8L), y != 0L, ))
expect_equal(or3s(iiz %in% c(4L, -3L, 2L, 7L, 8L), a != 1L, ),
             bor3(iiz %in% c(4L, -3L, 2L, 7L, 8L), a != 1L, ))
expect_equal(or3s(ija %in% c(4L, -3L, 2L, 7L, 8L), o != 9L, ),
             bor3(ija %in% c(4L, -3L, 2L, 7L, 8L), o != 9L, ))
expect_equal(or3s(ijb %in% c(4L, -3L, 2L, 7L, 8L), r != 0L, m <= 0L),
             bor3(ijb %in% c(4L, -3L, 2L, 7L, 8L), r != 0L, m <= 0L))
expect_equal(or3s(ijc %in% c(4L, -3L, 2L, 7L, 8L), h != 1L, h <= 1L),
             bor3(ijc %in% c(4L, -3L, 2L, 7L, 8L), h != 1L, h <= 1L))
expect_equal(or3s(ijd %in% c(4L, -3L, 2L, 7L, 8L), s != 9L, g <= 9L),
             bor3(ijd %in% c(4L, -3L, 2L, 7L, 8L), s != 9L, g <= 9L))
expect_equal(or3s(ije %in% 1:4, , ),
             bor3(ije %in% 1:4, , ))
expect_equal(or3s(ijf %in% 1:4, , ),
             bor3(ijf %in% 1:4, , ))
expect_equal(or3s(ijg %in% 1:4, , ),
             bor3(ijg %in% 1:4, , ))
expect_equal(or3s(ijh %in% 1:4, , r <= 0L),
             bor3(ijh %in% 1:4, , r <= 0L))
expect_equal(or3s(iji %in% 1:4, , s <= 1L),
             bor3(iji %in% 1:4, , s <= 1L))
expect_equal(or3s(ijj %in% 1:4, , n <= 9L),
             bor3(ijj %in% 1:4, , n <= 9L))
expect_equal(or3s(ijk %in% 1:4, w != 0L, ),
             bor3(ijk %in% 1:4, w != 0L, ))
expect_equal(or3s(ijl %in% 1:4, v != 1L, ),
             bor3(ijl %in% 1:4, v != 1L, ))
expect_equal(or3s(ijm %in% 1:4, y != 9L, ),
             bor3(ijm %in% 1:4, y != 9L, ))
expect_equal(or3s(ijn %in% 1:4, v != 0L, y <= 0L),
             bor3(ijn %in% 1:4, v != 0L, y <= 0L))
expect_equal(or3s(ijo %in% 1:4, k != 1L, e <= 1L),
             bor3(ijo %in% 1:4, k != 1L, e <= 1L))
expect_equal(or3s(ijp %in% 1:4, r != 9L, w <= 9L),
             bor3(ijp %in% 1:4, r != 9L, w <= 9L))
expect_equal(or3s(ijq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ijq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ijr %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ijr %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ijs %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ijs %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ijt %in% c(4L, -3L, 2L, 7L, 8L), , q == 0L),
             bor3(ijt %in% c(4L, -3L, 2L, 7L, 8L), , q == 0L))
expect_equal(or3s(iju %in% c(4L, -3L, 2L, 7L, 8L), , u == 1L),
             bor3(iju %in% c(4L, -3L, 2L, 7L, 8L), , u == 1L))
expect_equal(or3s(ijv %in% c(4L, -3L, 2L, 7L, 8L), , f == 9L),
             bor3(ijv %in% c(4L, -3L, 2L, 7L, 8L), , f == 9L))
expect_equal(or3s(ijw %in% c(4L, -3L, 2L, 7L, 8L), a != 0L, ),
             bor3(ijw %in% c(4L, -3L, 2L, 7L, 8L), a != 0L, ))
expect_equal(or3s(ijx %in% c(4L, -3L, 2L, 7L, 8L), i != 1L, ),
             bor3(ijx %in% c(4L, -3L, 2L, 7L, 8L), i != 1L, ))
expect_equal(or3s(ijy %in% c(4L, -3L, 2L, 7L, 8L), l != 9L, ),
             bor3(ijy %in% c(4L, -3L, 2L, 7L, 8L), l != 9L, ))
expect_equal(or3s(ijz %in% c(4L, -3L, 2L, 7L, 8L), r != 0L, m == 0L),
             bor3(ijz %in% c(4L, -3L, 2L, 7L, 8L), r != 0L, m == 0L))
expect_equal(or3s(ika %in% c(4L, -3L, 2L, 7L, 8L), z != 1L, p == 1L),
             bor3(ika %in% c(4L, -3L, 2L, 7L, 8L), z != 1L, p == 1L))
expect_equal(or3s(ikb %in% c(4L, -3L, 2L, 7L, 8L), e != 9L, w == 9L),
             bor3(ikb %in% c(4L, -3L, 2L, 7L, 8L), e != 9L, w == 9L))
expect_equal(or3s(ikc %in% 1:4, , ),
             bor3(ikc %in% 1:4, , ))
expect_equal(or3s(ikd %in% 1:4, , ),
             bor3(ikd %in% 1:4, , ))
expect_equal(or3s(ike %in% 1:4, , ),
             bor3(ike %in% 1:4, , ))
expect_equal(or3s(ikf %in% 1:4, , z == 0L),
             bor3(ikf %in% 1:4, , z == 0L))
expect_equal(or3s(ikg %in% 1:4, , q == 1L),
             bor3(ikg %in% 1:4, , q == 1L))
expect_equal(or3s(ikh %in% 1:4, , n == 9L),
             bor3(ikh %in% 1:4, , n == 9L))
expect_equal(or3s(iki %in% 1:4, t != 0L, ),
             bor3(iki %in% 1:4, t != 0L, ))
expect_equal(or3s(ikj %in% 1:4, a != 1L, ),
             bor3(ikj %in% 1:4, a != 1L, ))
expect_equal(or3s(ikk %in% 1:4, e != 9L, ),
             bor3(ikk %in% 1:4, e != 9L, ))
expect_equal(or3s(ikl %in% 1:4, t != 0L, n == 0L),
             bor3(ikl %in% 1:4, t != 0L, n == 0L))
expect_equal(or3s(ikm %in% 1:4, t != 1L, l == 1L),
             bor3(ikm %in% 1:4, t != 1L, l == 1L))
expect_equal(or3s(ikn %in% 1:4, n != 9L, i == 9L),
             bor3(ikn %in% 1:4, n != 9L, i == 9L))
expect_equal(or3s(iko %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iko %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ikp %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ikp %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ikq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ikq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ikr %in% c(4L, -3L, 2L, 7L, 8L), , x > 0L),
             bor3(ikr %in% c(4L, -3L, 2L, 7L, 8L), , x > 0L))
expect_equal(or3s(iks %in% c(4L, -3L, 2L, 7L, 8L), , m > 1L),
             bor3(iks %in% c(4L, -3L, 2L, 7L, 8L), , m > 1L))
expect_equal(or3s(ikt %in% c(4L, -3L, 2L, 7L, 8L), , n > 9L),
             bor3(ikt %in% c(4L, -3L, 2L, 7L, 8L), , n > 9L))
expect_equal(or3s(iku %in% c(4L, -3L, 2L, 7L, 8L), q != 0L, ),
             bor3(iku %in% c(4L, -3L, 2L, 7L, 8L), q != 0L, ))
expect_equal(or3s(ikv %in% c(4L, -3L, 2L, 7L, 8L), l != 1L, ),
             bor3(ikv %in% c(4L, -3L, 2L, 7L, 8L), l != 1L, ))
expect_equal(or3s(ikw %in% c(4L, -3L, 2L, 7L, 8L), e != 9L, ),
             bor3(ikw %in% c(4L, -3L, 2L, 7L, 8L), e != 9L, ))
expect_equal(or3s(ikx %in% c(4L, -3L, 2L, 7L, 8L), l != 0L, c > 0L),
             bor3(ikx %in% c(4L, -3L, 2L, 7L, 8L), l != 0L, c > 0L))
expect_equal(or3s(iky %in% c(4L, -3L, 2L, 7L, 8L), f != 1L, a > 1L),
             bor3(iky %in% c(4L, -3L, 2L, 7L, 8L), f != 1L, a > 1L))
expect_equal(or3s(ikz %in% c(4L, -3L, 2L, 7L, 8L), j != 9L, p > 9L),
             bor3(ikz %in% c(4L, -3L, 2L, 7L, 8L), j != 9L, p > 9L))
expect_equal(or3s(ila %in% 1:4, , ),
             bor3(ila %in% 1:4, , ))
expect_equal(or3s(ilb %in% 1:4, , ),
             bor3(ilb %in% 1:4, , ))
expect_equal(or3s(ilc %in% 1:4, , ),
             bor3(ilc %in% 1:4, , ))
expect_equal(or3s(ild %in% 1:4, , n > 0L),
             bor3(ild %in% 1:4, , n > 0L))
expect_equal(or3s(ile %in% 1:4, , s > 1L),
             bor3(ile %in% 1:4, , s > 1L))
expect_equal(or3s(ilf %in% 1:4, , y > 9L),
             bor3(ilf %in% 1:4, , y > 9L))
expect_equal(or3s(ilg %in% 1:4, l != 0L, ),
             bor3(ilg %in% 1:4, l != 0L, ))
expect_equal(or3s(ilh %in% 1:4, m != 1L, ),
             bor3(ilh %in% 1:4, m != 1L, ))
expect_equal(or3s(ili %in% 1:4, n != 9L, ),
             bor3(ili %in% 1:4, n != 9L, ))
expect_equal(or3s(ilj %in% 1:4, z != 0L, u > 0L),
             bor3(ilj %in% 1:4, z != 0L, u > 0L))
expect_equal(or3s(ilk %in% 1:4, p != 1L, z > 1L),
             bor3(ilk %in% 1:4, p != 1L, z > 1L))
expect_equal(or3s(ill %in% 1:4, t != 9L, n > 9L),
             bor3(ill %in% 1:4, t != 9L, n > 9L))
expect_equal(or3s(ilm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ilm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iln %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iln %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ilo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ilo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ilp %in% c(4L, -3L, 2L, 7L, 8L), , s >= 0L),
             bor3(ilp %in% c(4L, -3L, 2L, 7L, 8L), , s >= 0L))
expect_equal(or3s(ilq %in% c(4L, -3L, 2L, 7L, 8L), , w >= 1L),
             bor3(ilq %in% c(4L, -3L, 2L, 7L, 8L), , w >= 1L))
expect_equal(or3s(ilr %in% c(4L, -3L, 2L, 7L, 8L), , p >= 9L),
             bor3(ilr %in% c(4L, -3L, 2L, 7L, 8L), , p >= 9L))
expect_equal(or3s(ils %in% c(4L, -3L, 2L, 7L, 8L), p != 0L, ),
             bor3(ils %in% c(4L, -3L, 2L, 7L, 8L), p != 0L, ))
expect_equal(or3s(ilt %in% c(4L, -3L, 2L, 7L, 8L), s != 1L, ),
             bor3(ilt %in% c(4L, -3L, 2L, 7L, 8L), s != 1L, ))
expect_equal(or3s(ilu %in% c(4L, -3L, 2L, 7L, 8L), d != 9L, ),
             bor3(ilu %in% c(4L, -3L, 2L, 7L, 8L), d != 9L, ))
expect_equal(or3s(ilv %in% c(4L, -3L, 2L, 7L, 8L), p != 0L, b >= 0L),
             bor3(ilv %in% c(4L, -3L, 2L, 7L, 8L), p != 0L, b >= 0L))
expect_equal(or3s(ilw %in% c(4L, -3L, 2L, 7L, 8L), p != 1L, k >= 1L),
             bor3(ilw %in% c(4L, -3L, 2L, 7L, 8L), p != 1L, k >= 1L))
expect_equal(or3s(ilx %in% c(4L, -3L, 2L, 7L, 8L), r != 9L, p >= 9L),
             bor3(ilx %in% c(4L, -3L, 2L, 7L, 8L), r != 9L, p >= 9L))
expect_equal(or3s(ily %in% 1:4, , ),
             bor3(ily %in% 1:4, , ))
expect_equal(or3s(ilz %in% 1:4, , ),
             bor3(ilz %in% 1:4, , ))
expect_equal(or3s(ima %in% 1:4, , ),
             bor3(ima %in% 1:4, , ))
expect_equal(or3s(imb %in% 1:4, , h >= 0L),
             bor3(imb %in% 1:4, , h >= 0L))
expect_equal(or3s(imc %in% 1:4, , r >= 1L),
             bor3(imc %in% 1:4, , r >= 1L))
expect_equal(or3s(imd %in% 1:4, , i >= 9L),
             bor3(imd %in% 1:4, , i >= 9L))
expect_equal(or3s(ime %in% 1:4, v != 0L, ),
             bor3(ime %in% 1:4, v != 0L, ))
expect_equal(or3s(imf %in% 1:4, n != 1L, ),
             bor3(imf %in% 1:4, n != 1L, ))
expect_equal(or3s(img %in% 1:4, f != 9L, ),
             bor3(img %in% 1:4, f != 9L, ))
expect_equal(or3s(imh %in% 1:4, e != 0L, z >= 0L),
             bor3(imh %in% 1:4, e != 0L, z >= 0L))
expect_equal(or3s(imi %in% 1:4, w != 1L, u >= 1L),
             bor3(imi %in% 1:4, w != 1L, u >= 1L))
expect_equal(or3s(imj %in% 1:4, i != 9L, e >= 9L),
             bor3(imj %in% 1:4, i != 9L, e >= 9L))
expect_equal(or3s(imk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(imk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iml %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iml %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(imm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(imm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(imn %in% c(4L, -3L, 2L, 7L, 8L), , logi_z),
             bor3(imn %in% c(4L, -3L, 2L, 7L, 8L), , logi_z))
expect_equal(or3s(imo %in% c(4L, -3L, 2L, 7L, 8L), , logi_l),
             bor3(imo %in% c(4L, -3L, 2L, 7L, 8L), , logi_l))
expect_equal(or3s(imp %in% c(4L, -3L, 2L, 7L, 8L), , logi_p),
             bor3(imp %in% c(4L, -3L, 2L, 7L, 8L), , logi_p))
expect_equal(or3s(imq %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L), ),
             bor3(imq %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L), ))
expect_equal(or3s(imr %in% c(4L, -3L, 2L, 7L, 8L), g %between% c(-1L, 1L), ),
             bor3(imr %in% c(4L, -3L, 2L, 7L, 8L), g %between% c(-1L, 1L), ))
expect_equal(or3s(ims %in% c(4L, -3L, 2L, 7L, 8L), l %between% c(-1L, 1L), ),
             bor3(ims %in% c(4L, -3L, 2L, 7L, 8L), l %between% c(-1L, 1L), ))
expect_equal(or3s(imt %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L), logi_g),
             bor3(imt %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L), logi_g))
expect_equal(or3s(imu %in% c(4L, -3L, 2L, 7L, 8L), n %between% c(-1L, 1L), logi_t),
             bor3(imu %in% c(4L, -3L, 2L, 7L, 8L), n %between% c(-1L, 1L), logi_t))
expect_equal(or3s(imv %in% c(4L, -3L, 2L, 7L, 8L), w %between% c(-1L, 1L), logi_e),
             bor3(imv %in% c(4L, -3L, 2L, 7L, 8L), w %between% c(-1L, 1L), logi_e))
expect_equal(or3s(imw %in% 1:4, , ),
             bor3(imw %in% 1:4, , ))
expect_equal(or3s(imx %in% 1:4, , ),
             bor3(imx %in% 1:4, , ))
expect_equal(or3s(imy %in% 1:4, , ),
             bor3(imy %in% 1:4, , ))
expect_equal(or3s(imz %in% 1:4, , logi_y),
             bor3(imz %in% 1:4, , logi_y))
expect_equal(or3s(ina %in% 1:4, , logi_u),
             bor3(ina %in% 1:4, , logi_u))
expect_equal(or3s(inb %in% 1:4, , logi_y),
             bor3(inb %in% 1:4, , logi_y))
expect_equal(or3s(inc %in% 1:4, u %between% c(-1L, 1L), ),
             bor3(inc %in% 1:4, u %between% c(-1L, 1L), ))
expect_equal(or3s(ind %in% 1:4, n %between% c(-1L, 1L), ),
             bor3(ind %in% 1:4, n %between% c(-1L, 1L), ))
expect_equal(or3s(ine %in% 1:4, j %between% c(-1L, 1L), ),
             bor3(ine %in% 1:4, j %between% c(-1L, 1L), ))
expect_equal(or3s(inf %in% 1:4, p %between% c(-1L, 1L), logi_v),
             bor3(inf %in% 1:4, p %between% c(-1L, 1L), logi_v))
expect_equal(or3s(ing %in% 1:4, w %between% c(-1L, 1L), logi_f),
             bor3(ing %in% 1:4, w %between% c(-1L, 1L), logi_f))
expect_equal(or3s(inh %in% 1:4, q %between% c(-1L, 1L), logi_k),
             bor3(inh %in% 1:4, q %between% c(-1L, 1L), logi_k))
expect_equal(or3s(ini %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ini %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(inj %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(inj %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ink %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ink %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(inl %in% c(4L, -3L, 2L, 7L, 8L), , !logi_w),
             bor3(inl %in% c(4L, -3L, 2L, 7L, 8L), , !logi_w))
expect_equal(or3s(inm %in% c(4L, -3L, 2L, 7L, 8L), , !logi_t),
             bor3(inm %in% c(4L, -3L, 2L, 7L, 8L), , !logi_t))
expect_equal(or3s(inn %in% c(4L, -3L, 2L, 7L, 8L), , !logi_k),
             bor3(inn %in% c(4L, -3L, 2L, 7L, 8L), , !logi_k))
expect_equal(or3s(ino %in% c(4L, -3L, 2L, 7L, 8L), b %between% c(-1L, 1L), ),
             bor3(ino %in% c(4L, -3L, 2L, 7L, 8L), b %between% c(-1L, 1L), ))
expect_equal(or3s(inp %in% c(4L, -3L, 2L, 7L, 8L), c %between% c(-1L, 1L), ),
             bor3(inp %in% c(4L, -3L, 2L, 7L, 8L), c %between% c(-1L, 1L), ))
expect_equal(or3s(inq %in% c(4L, -3L, 2L, 7L, 8L), l %between% c(-1L, 1L), ),
             bor3(inq %in% c(4L, -3L, 2L, 7L, 8L), l %between% c(-1L, 1L), ))
expect_equal(or3s(inr %in% c(4L, -3L, 2L, 7L, 8L), h %between% c(-1L, 1L), !logi_x),
             bor3(inr %in% c(4L, -3L, 2L, 7L, 8L), h %between% c(-1L, 1L), !logi_x))
expect_equal(or3s(ins %in% c(4L, -3L, 2L, 7L, 8L), m %between% c(-1L, 1L), !logi_n),
             bor3(ins %in% c(4L, -3L, 2L, 7L, 8L), m %between% c(-1L, 1L), !logi_n))
expect_equal(or3s(int %in% c(4L, -3L, 2L, 7L, 8L), q %between% c(-1L, 1L), !logi_w),
             bor3(int %in% c(4L, -3L, 2L, 7L, 8L), q %between% c(-1L, 1L), !logi_w))
expect_equal(or3s(inu %in% 1:4, , ),
             bor3(inu %in% 1:4, , ))
expect_equal(or3s(inv %in% 1:4, , ),
             bor3(inv %in% 1:4, , ))
expect_equal(or3s(inw %in% 1:4, , ),
             bor3(inw %in% 1:4, , ))
expect_equal(or3s(inx %in% 1:4, , !logi_n),
             bor3(inx %in% 1:4, , !logi_n))
expect_equal(or3s(iny %in% 1:4, , !logi_b),
             bor3(iny %in% 1:4, , !logi_b))
expect_equal(or3s(inz %in% 1:4, , !logi_d),
             bor3(inz %in% 1:4, , !logi_d))
expect_equal(or3s(ioa %in% 1:4, g %between% c(-1L, 1L), ),
             bor3(ioa %in% 1:4, g %between% c(-1L, 1L), ))
expect_equal(or3s(iob %in% 1:4, u %between% c(-1L, 1L), ),
             bor3(iob %in% 1:4, u %between% c(-1L, 1L), ))
expect_equal(or3s(ioc %in% 1:4, g %between% c(-1L, 1L), ),
             bor3(ioc %in% 1:4, g %between% c(-1L, 1L), ))
expect_equal(or3s(iod %in% 1:4, n %between% c(-1L, 1L), !logi_b),
             bor3(iod %in% 1:4, n %between% c(-1L, 1L), !logi_b))
expect_equal(or3s(ioe %in% 1:4, s %between% c(-1L, 1L), !logi_s),
             bor3(ioe %in% 1:4, s %between% c(-1L, 1L), !logi_s))
expect_equal(or3s(iof %in% 1:4, o %between% c(-1L, 1L), !logi_p),
             bor3(iof %in% 1:4, o %between% c(-1L, 1L), !logi_p))
expect_equal(or3s(iog %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iog %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ioh %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ioh %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ioi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ioi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ioj %in% c(4L, -3L, 2L, 7L, 8L), , d != 0L),
             bor3(ioj %in% c(4L, -3L, 2L, 7L, 8L), , d != 0L))
expect_equal(or3s(iok %in% c(4L, -3L, 2L, 7L, 8L), , j != 1L),
             bor3(iok %in% c(4L, -3L, 2L, 7L, 8L), , j != 1L))
expect_equal(or3s(iol %in% c(4L, -3L, 2L, 7L, 8L), , i != 9L),
             bor3(iol %in% c(4L, -3L, 2L, 7L, 8L), , i != 9L))
expect_equal(or3s(iom %in% c(4L, -3L, 2L, 7L, 8L), s %between% c(-1L, 1L), ),
             bor3(iom %in% c(4L, -3L, 2L, 7L, 8L), s %between% c(-1L, 1L), ))
expect_equal(or3s(ion %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), ),
             bor3(ion %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), ))
expect_equal(or3s(ioo %in% c(4L, -3L, 2L, 7L, 8L), x %between% c(-1L, 1L), ),
             bor3(ioo %in% c(4L, -3L, 2L, 7L, 8L), x %between% c(-1L, 1L), ))
expect_equal(or3s(iop %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), h != 0L),
             bor3(iop %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), h != 0L))
expect_equal(or3s(ioq %in% c(4L, -3L, 2L, 7L, 8L), t %between% c(-1L, 1L), y != 1L),
             bor3(ioq %in% c(4L, -3L, 2L, 7L, 8L), t %between% c(-1L, 1L), y != 1L))
expect_equal(or3s(ior %in% c(4L, -3L, 2L, 7L, 8L), n %between% c(-1L, 1L), a != 9L),
             bor3(ior %in% c(4L, -3L, 2L, 7L, 8L), n %between% c(-1L, 1L), a != 9L))
expect_equal(or3s(ios %in% 1:4, , ),
             bor3(ios %in% 1:4, , ))
expect_equal(or3s(iot %in% 1:4, , ),
             bor3(iot %in% 1:4, , ))
expect_equal(or3s(iou %in% 1:4, , ),
             bor3(iou %in% 1:4, , ))
expect_equal(or3s(iov %in% 1:4, , q != 0L),
             bor3(iov %in% 1:4, , q != 0L))
expect_equal(or3s(iow %in% 1:4, , w != 1L),
             bor3(iow %in% 1:4, , w != 1L))
expect_equal(or3s(iox %in% 1:4, , l != 9L),
             bor3(iox %in% 1:4, , l != 9L))
expect_equal(or3s(ioy %in% 1:4, n %between% c(-1L, 1L), ),
             bor3(ioy %in% 1:4, n %between% c(-1L, 1L), ))
expect_equal(or3s(ioz %in% 1:4, t %between% c(-1L, 1L), ),
             bor3(ioz %in% 1:4, t %between% c(-1L, 1L), ))
expect_equal(or3s(ipa %in% 1:4, l %between% c(-1L, 1L), ),
             bor3(ipa %in% 1:4, l %between% c(-1L, 1L), ))
expect_equal(or3s(ipb %in% 1:4, j %between% c(-1L, 1L), g != 0L),
             bor3(ipb %in% 1:4, j %between% c(-1L, 1L), g != 0L))
expect_equal(or3s(ipc %in% 1:4, j %between% c(-1L, 1L), k != 1L),
             bor3(ipc %in% 1:4, j %between% c(-1L, 1L), k != 1L))
expect_equal(or3s(ipd %in% 1:4, g %between% c(-1L, 1L), b != 9L),
             bor3(ipd %in% 1:4, g %between% c(-1L, 1L), b != 9L))
expect_equal(or3s(ipe %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ipe %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ipf %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ipf %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ipg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ipg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iph %in% c(4L, -3L, 2L, 7L, 8L), , p %between% c(-1L, 1L)),
             bor3(iph %in% c(4L, -3L, 2L, 7L, 8L), , p %between% c(-1L, 1L)))
expect_equal(or3s(ipi %in% c(4L, -3L, 2L, 7L, 8L), , w %between% c(-1L, 1L)),
             bor3(ipi %in% c(4L, -3L, 2L, 7L, 8L), , w %between% c(-1L, 1L)))
expect_equal(or3s(ipj %in% c(4L, -3L, 2L, 7L, 8L), , j %between% c(-1L, 1L)),
             bor3(ipj %in% c(4L, -3L, 2L, 7L, 8L), , j %between% c(-1L, 1L)))
expect_equal(or3s(ipk %in% c(4L, -3L, 2L, 7L, 8L), f %between% c(-1L, 1L), ),
             bor3(ipk %in% c(4L, -3L, 2L, 7L, 8L), f %between% c(-1L, 1L), ))
expect_equal(or3s(ipl %in% c(4L, -3L, 2L, 7L, 8L), s %between% c(-1L, 1L), ),
             bor3(ipl %in% c(4L, -3L, 2L, 7L, 8L), s %between% c(-1L, 1L), ))
expect_equal(or3s(ipm %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), ),
             bor3(ipm %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), ))
expect_equal(or3s(ipn %in% c(4L, -3L, 2L, 7L, 8L), q %between% c(-1L, 1L), p %between% c(-1L, 1L)),
             bor3(ipn %in% c(4L, -3L, 2L, 7L, 8L), q %between% c(-1L, 1L), p %between% c(-1L, 1L)))
expect_equal(or3s(ipo %in% c(4L, -3L, 2L, 7L, 8L), w %between% c(-1L, 1L), y %between% c(-1L, 1L)),
             bor3(ipo %in% c(4L, -3L, 2L, 7L, 8L), w %between% c(-1L, 1L), y %between% c(-1L, 1L)))
expect_equal(or3s(ipp %in% c(4L, -3L, 2L, 7L, 8L), i %between% c(-1L, 1L), t %between% c(-1L, 1L)),
             bor3(ipp %in% c(4L, -3L, 2L, 7L, 8L), i %between% c(-1L, 1L), t %between% c(-1L, 1L)))
expect_equal(or3s(ipq %in% 1:4, , ),
             bor3(ipq %in% 1:4, , ))
expect_equal(or3s(ipr %in% 1:4, , ),
             bor3(ipr %in% 1:4, , ))
expect_equal(or3s(ips %in% 1:4, , ),
             bor3(ips %in% 1:4, , ))
expect_equal(or3s(ipt %in% 1:4, , f %between% c(-1L, 1L)),
             bor3(ipt %in% 1:4, , f %between% c(-1L, 1L)))
expect_equal(or3s(ipu %in% 1:4, , u %between% c(-1L, 1L)),
             bor3(ipu %in% 1:4, , u %between% c(-1L, 1L)))
expect_equal(or3s(ipv %in% 1:4, , l %between% c(-1L, 1L)),
             bor3(ipv %in% 1:4, , l %between% c(-1L, 1L)))
expect_equal(or3s(ipw %in% 1:4, k %between% c(-1L, 1L), ),
             bor3(ipw %in% 1:4, k %between% c(-1L, 1L), ))
expect_equal(or3s(ipx %in% 1:4, v %between% c(-1L, 1L), ),
             bor3(ipx %in% 1:4, v %between% c(-1L, 1L), ))
expect_equal(or3s(ipy %in% 1:4, p %between% c(-1L, 1L), ),
             bor3(ipy %in% 1:4, p %between% c(-1L, 1L), ))
expect_equal(or3s(ipz %in% 1:4, d %between% c(-1L, 1L), a %between% c(-1L, 1L)),
             bor3(ipz %in% 1:4, d %between% c(-1L, 1L), a %between% c(-1L, 1L)))
expect_equal(or3s(iqa %in% 1:4, v %between% c(-1L, 1L), b %between% c(-1L, 1L)),
             bor3(iqa %in% 1:4, v %between% c(-1L, 1L), b %between% c(-1L, 1L)))
expect_equal(or3s(iqb %in% 1:4, m %between% c(-1L, 1L), r %between% c(-1L, 1L)),
             bor3(iqb %in% 1:4, m %between% c(-1L, 1L), r %between% c(-1L, 1L)))
expect_equal(or3s(iqc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iqc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iqd %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iqd %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iqe %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iqe %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iqf %in% c(4L, -3L, 2L, 7L, 8L), , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(iqf %in% c(4L, -3L, 2L, 7L, 8L), , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(iqg %in% c(4L, -3L, 2L, 7L, 8L), , e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(iqg %in% c(4L, -3L, 2L, 7L, 8L), , e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(iqh %in% c(4L, -3L, 2L, 7L, 8L), , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(iqh %in% c(4L, -3L, 2L, 7L, 8L), , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(iqi %in% c(4L, -3L, 2L, 7L, 8L), d %between% c(-1L, 1L), ),
             bor3(iqi %in% c(4L, -3L, 2L, 7L, 8L), d %between% c(-1L, 1L), ))
expect_equal(or3s(iqj %in% c(4L, -3L, 2L, 7L, 8L), d %between% c(-1L, 1L), ),
             bor3(iqj %in% c(4L, -3L, 2L, 7L, 8L), d %between% c(-1L, 1L), ))
expect_equal(or3s(iqk %in% c(4L, -3L, 2L, 7L, 8L), v %between% c(-1L, 1L), ),
             bor3(iqk %in% c(4L, -3L, 2L, 7L, 8L), v %between% c(-1L, 1L), ))
expect_equal(or3s(iql %in% c(4L, -3L, 2L, 7L, 8L), f %between% c(-1L, 1L), u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(iql %in% c(4L, -3L, 2L, 7L, 8L), f %between% c(-1L, 1L), u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(iqm %in% c(4L, -3L, 2L, 7L, 8L), o %between% c(-1L, 1L), s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(iqm %in% c(4L, -3L, 2L, 7L, 8L), o %between% c(-1L, 1L), s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(iqn %in% c(4L, -3L, 2L, 7L, 8L), z %between% c(-1L, 1L), k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(iqn %in% c(4L, -3L, 2L, 7L, 8L), z %between% c(-1L, 1L), k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(iqo %in% 1:4, , ),
             bor3(iqo %in% 1:4, , ))
expect_equal(or3s(iqp %in% 1:4, , ),
             bor3(iqp %in% 1:4, , ))
expect_equal(or3s(iqq %in% 1:4, , ),
             bor3(iqq %in% 1:4, , ))
expect_equal(or3s(iqr %in% 1:4, , o %in% 1:4),
             bor3(iqr %in% 1:4, , o %in% 1:4))
expect_equal(or3s(iqs %in% 1:4, , o %in% 1:4),
             bor3(iqs %in% 1:4, , o %in% 1:4))
expect_equal(or3s(iqt %in% 1:4, , p %in% 1:4),
             bor3(iqt %in% 1:4, , p %in% 1:4))
expect_equal(or3s(iqu %in% 1:4, j %between% c(-1L, 1L), ),
             bor3(iqu %in% 1:4, j %between% c(-1L, 1L), ))
expect_equal(or3s(iqv %in% 1:4, s %between% c(-1L, 1L), ),
             bor3(iqv %in% 1:4, s %between% c(-1L, 1L), ))
expect_equal(or3s(iqw %in% 1:4, d %between% c(-1L, 1L), ),
             bor3(iqw %in% 1:4, d %between% c(-1L, 1L), ))
expect_equal(or3s(iqx %in% 1:4, x %between% c(-1L, 1L), x %in% 1:4),
             bor3(iqx %in% 1:4, x %between% c(-1L, 1L), x %in% 1:4))
expect_equal(or3s(iqy %in% 1:4, c %between% c(-1L, 1L), h %in% 1:4),
             bor3(iqy %in% 1:4, c %between% c(-1L, 1L), h %in% 1:4))
expect_equal(or3s(iqz %in% 1:4, m %between% c(-1L, 1L), l %in% 1:4),
             bor3(iqz %in% 1:4, m %between% c(-1L, 1L), l %in% 1:4))
expect_equal(or3s(ira %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ira %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(irb %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(irb %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(irc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(irc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ird %in% c(4L, -3L, 2L, 7L, 8L), , o < 0L),
             bor3(ird %in% c(4L, -3L, 2L, 7L, 8L), , o < 0L))
expect_equal(or3s(ire %in% c(4L, -3L, 2L, 7L, 8L), , d < 1L),
             bor3(ire %in% c(4L, -3L, 2L, 7L, 8L), , d < 1L))
expect_equal(or3s(irf %in% c(4L, -3L, 2L, 7L, 8L), , y < 9L),
             bor3(irf %in% c(4L, -3L, 2L, 7L, 8L), , y < 9L))
expect_equal(or3s(irg %in% c(4L, -3L, 2L, 7L, 8L), z %between% c(-1L, 1L), ),
             bor3(irg %in% c(4L, -3L, 2L, 7L, 8L), z %between% c(-1L, 1L), ))
expect_equal(or3s(irh %in% c(4L, -3L, 2L, 7L, 8L), v %between% c(-1L, 1L), ),
             bor3(irh %in% c(4L, -3L, 2L, 7L, 8L), v %between% c(-1L, 1L), ))
expect_equal(or3s(iri %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), ),
             bor3(iri %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), ))
expect_equal(or3s(irj %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), q < 0L),
             bor3(irj %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), q < 0L))
expect_equal(or3s(irk %in% c(4L, -3L, 2L, 7L, 8L), q %between% c(-1L, 1L), i < 1L),
             bor3(irk %in% c(4L, -3L, 2L, 7L, 8L), q %between% c(-1L, 1L), i < 1L))
expect_equal(or3s(irl %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), k < 9L),
             bor3(irl %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), k < 9L))
expect_equal(or3s(irm %in% 1:4, , ),
             bor3(irm %in% 1:4, , ))
expect_equal(or3s(irn %in% 1:4, , ),
             bor3(irn %in% 1:4, , ))
expect_equal(or3s(iro %in% 1:4, , ),
             bor3(iro %in% 1:4, , ))
expect_equal(or3s(irp %in% 1:4, , h < 0L),
             bor3(irp %in% 1:4, , h < 0L))
expect_equal(or3s(irq %in% 1:4, , p < 1L),
             bor3(irq %in% 1:4, , p < 1L))
expect_equal(or3s(irr %in% 1:4, , n < 9L),
             bor3(irr %in% 1:4, , n < 9L))
expect_equal(or3s(irs %in% 1:4, y %between% c(-1L, 1L), ),
             bor3(irs %in% 1:4, y %between% c(-1L, 1L), ))
expect_equal(or3s(irt %in% 1:4, v %between% c(-1L, 1L), ),
             bor3(irt %in% 1:4, v %between% c(-1L, 1L), ))
expect_equal(or3s(iru %in% 1:4, j %between% c(-1L, 1L), ),
             bor3(iru %in% 1:4, j %between% c(-1L, 1L), ))
expect_equal(or3s(irv %in% 1:4, s %between% c(-1L, 1L), q < 0L),
             bor3(irv %in% 1:4, s %between% c(-1L, 1L), q < 0L))
expect_equal(or3s(irw %in% 1:4, w %between% c(-1L, 1L), f < 1L),
             bor3(irw %in% 1:4, w %between% c(-1L, 1L), f < 1L))
expect_equal(or3s(irx %in% 1:4, i %between% c(-1L, 1L), t < 9L),
             bor3(irx %in% 1:4, i %between% c(-1L, 1L), t < 9L))
expect_equal(or3s(iry %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iry %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(irz %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(irz %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(isa %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(isa %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(isb %in% c(4L, -3L, 2L, 7L, 8L), , c <= 0L),
             bor3(isb %in% c(4L, -3L, 2L, 7L, 8L), , c <= 0L))
expect_equal(or3s(isc %in% c(4L, -3L, 2L, 7L, 8L), , k <= 1L),
             bor3(isc %in% c(4L, -3L, 2L, 7L, 8L), , k <= 1L))
expect_equal(or3s(isd %in% c(4L, -3L, 2L, 7L, 8L), , r <= 9L),
             bor3(isd %in% c(4L, -3L, 2L, 7L, 8L), , r <= 9L))
expect_equal(or3s(ise %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), ),
             bor3(ise %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), ))
expect_equal(or3s(isf %in% c(4L, -3L, 2L, 7L, 8L), h %between% c(-1L, 1L), ),
             bor3(isf %in% c(4L, -3L, 2L, 7L, 8L), h %between% c(-1L, 1L), ))
expect_equal(or3s(isg %in% c(4L, -3L, 2L, 7L, 8L), c %between% c(-1L, 1L), ),
             bor3(isg %in% c(4L, -3L, 2L, 7L, 8L), c %between% c(-1L, 1L), ))
expect_equal(or3s(ish %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), i <= 0L),
             bor3(ish %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), i <= 0L))
expect_equal(or3s(isi %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), w <= 1L),
             bor3(isi %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), w <= 1L))
expect_equal(or3s(isj %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), u <= 9L),
             bor3(isj %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), u <= 9L))
expect_equal(or3s(isk %in% 1:4, , ),
             bor3(isk %in% 1:4, , ))
expect_equal(or3s(isl %in% 1:4, , ),
             bor3(isl %in% 1:4, , ))
expect_equal(or3s(ism %in% 1:4, , ),
             bor3(ism %in% 1:4, , ))
expect_equal(or3s(isn %in% 1:4, , o <= 0L),
             bor3(isn %in% 1:4, , o <= 0L))
expect_equal(or3s(iso %in% 1:4, , p <= 1L),
             bor3(iso %in% 1:4, , p <= 1L))
expect_equal(or3s(isp %in% 1:4, , m <= 9L),
             bor3(isp %in% 1:4, , m <= 9L))
expect_equal(or3s(isq %in% 1:4, l %between% c(-1L, 1L), ),
             bor3(isq %in% 1:4, l %between% c(-1L, 1L), ))
expect_equal(or3s(isr %in% 1:4, j %between% c(-1L, 1L), ),
             bor3(isr %in% 1:4, j %between% c(-1L, 1L), ))
expect_equal(or3s(iss %in% 1:4, f %between% c(-1L, 1L), ),
             bor3(iss %in% 1:4, f %between% c(-1L, 1L), ))
expect_equal(or3s(ist %in% 1:4, b %between% c(-1L, 1L), j <= 0L),
             bor3(ist %in% 1:4, b %between% c(-1L, 1L), j <= 0L))
expect_equal(or3s(isu %in% 1:4, q %between% c(-1L, 1L), k <= 1L),
             bor3(isu %in% 1:4, q %between% c(-1L, 1L), k <= 1L))
expect_equal(or3s(isv %in% 1:4, e %between% c(-1L, 1L), d <= 9L),
             bor3(isv %in% 1:4, e %between% c(-1L, 1L), d <= 9L))
expect_equal(or3s(isw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(isw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(isx %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(isx %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(isy %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(isy %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(isz %in% c(4L, -3L, 2L, 7L, 8L), , y == 0L),
             bor3(isz %in% c(4L, -3L, 2L, 7L, 8L), , y == 0L))
expect_equal(or3s(ita %in% c(4L, -3L, 2L, 7L, 8L), , q == 1L),
             bor3(ita %in% c(4L, -3L, 2L, 7L, 8L), , q == 1L))
expect_equal(or3s(itb %in% c(4L, -3L, 2L, 7L, 8L), , v == 9L),
             bor3(itb %in% c(4L, -3L, 2L, 7L, 8L), , v == 9L))
expect_equal(or3s(itc %in% c(4L, -3L, 2L, 7L, 8L), p %between% c(-1L, 1L), ),
             bor3(itc %in% c(4L, -3L, 2L, 7L, 8L), p %between% c(-1L, 1L), ))
expect_equal(or3s(itd %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), ),
             bor3(itd %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), ))
expect_equal(or3s(ite %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), ),
             bor3(ite %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), ))
expect_equal(or3s(itf %in% c(4L, -3L, 2L, 7L, 8L), y %between% c(-1L, 1L), m == 0L),
             bor3(itf %in% c(4L, -3L, 2L, 7L, 8L), y %between% c(-1L, 1L), m == 0L))
expect_equal(or3s(itg %in% c(4L, -3L, 2L, 7L, 8L), t %between% c(-1L, 1L), e == 1L),
             bor3(itg %in% c(4L, -3L, 2L, 7L, 8L), t %between% c(-1L, 1L), e == 1L))
expect_equal(or3s(ith %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L), b == 9L),
             bor3(ith %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L), b == 9L))
expect_equal(or3s(iti %in% 1:4, , ),
             bor3(iti %in% 1:4, , ))
expect_equal(or3s(itj %in% 1:4, , ),
             bor3(itj %in% 1:4, , ))
expect_equal(or3s(itk %in% 1:4, , ),
             bor3(itk %in% 1:4, , ))
expect_equal(or3s(itl %in% 1:4, , k == 0L),
             bor3(itl %in% 1:4, , k == 0L))
expect_equal(or3s(itm %in% 1:4, , k == 1L),
             bor3(itm %in% 1:4, , k == 1L))
expect_equal(or3s(itn %in% 1:4, , o == 9L),
             bor3(itn %in% 1:4, , o == 9L))
expect_equal(or3s(ito %in% 1:4, r %between% c(-1L, 1L), ),
             bor3(ito %in% 1:4, r %between% c(-1L, 1L), ))
expect_equal(or3s(itp %in% 1:4, q %between% c(-1L, 1L), ),
             bor3(itp %in% 1:4, q %between% c(-1L, 1L), ))
expect_equal(or3s(itq %in% 1:4, x %between% c(-1L, 1L), ),
             bor3(itq %in% 1:4, x %between% c(-1L, 1L), ))
expect_equal(or3s(itr %in% 1:4, j %between% c(-1L, 1L), t == 0L),
             bor3(itr %in% 1:4, j %between% c(-1L, 1L), t == 0L))
expect_equal(or3s(its %in% 1:4, k %between% c(-1L, 1L), n == 1L),
             bor3(its %in% 1:4, k %between% c(-1L, 1L), n == 1L))
expect_equal(or3s(itt %in% 1:4, r %between% c(-1L, 1L), g == 9L),
             bor3(itt %in% 1:4, r %between% c(-1L, 1L), g == 9L))
expect_equal(or3s(itu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(itu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(itv %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(itv %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(itw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(itw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(itx %in% c(4L, -3L, 2L, 7L, 8L), , c > 0L),
             bor3(itx %in% c(4L, -3L, 2L, 7L, 8L), , c > 0L))
expect_equal(or3s(ity %in% c(4L, -3L, 2L, 7L, 8L), , e > 1L),
             bor3(ity %in% c(4L, -3L, 2L, 7L, 8L), , e > 1L))
expect_equal(or3s(itz %in% c(4L, -3L, 2L, 7L, 8L), , p > 9L),
             bor3(itz %in% c(4L, -3L, 2L, 7L, 8L), , p > 9L))
expect_equal(or3s(iua %in% c(4L, -3L, 2L, 7L, 8L), b %between% c(-1L, 1L), ),
             bor3(iua %in% c(4L, -3L, 2L, 7L, 8L), b %between% c(-1L, 1L), ))
expect_equal(or3s(iub %in% c(4L, -3L, 2L, 7L, 8L), x %between% c(-1L, 1L), ),
             bor3(iub %in% c(4L, -3L, 2L, 7L, 8L), x %between% c(-1L, 1L), ))
expect_equal(or3s(iuc %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), ),
             bor3(iuc %in% c(4L, -3L, 2L, 7L, 8L), k %between% c(-1L, 1L), ))
expect_equal(or3s(iud %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), b > 0L),
             bor3(iud %in% c(4L, -3L, 2L, 7L, 8L), u %between% c(-1L, 1L), b > 0L))
expect_equal(or3s(iue %in% c(4L, -3L, 2L, 7L, 8L), p %between% c(-1L, 1L), x > 1L),
             bor3(iue %in% c(4L, -3L, 2L, 7L, 8L), p %between% c(-1L, 1L), x > 1L))
expect_equal(or3s(iuf %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), e > 9L),
             bor3(iuf %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), e > 9L))
expect_equal(or3s(iug %in% 1:4, , ),
             bor3(iug %in% 1:4, , ))
expect_equal(or3s(iuh %in% 1:4, , ),
             bor3(iuh %in% 1:4, , ))
expect_equal(or3s(iui %in% 1:4, , ),
             bor3(iui %in% 1:4, , ))
expect_equal(or3s(iuj %in% 1:4, , s > 0L),
             bor3(iuj %in% 1:4, , s > 0L))
expect_equal(or3s(iuk %in% 1:4, , w > 1L),
             bor3(iuk %in% 1:4, , w > 1L))
expect_equal(or3s(iul %in% 1:4, , z > 9L),
             bor3(iul %in% 1:4, , z > 9L))
expect_equal(or3s(ium %in% 1:4, q %between% c(-1L, 1L), ),
             bor3(ium %in% 1:4, q %between% c(-1L, 1L), ))
expect_equal(or3s(iun %in% 1:4, o %between% c(-1L, 1L), ),
             bor3(iun %in% 1:4, o %between% c(-1L, 1L), ))
expect_equal(or3s(iuo %in% 1:4, t %between% c(-1L, 1L), ),
             bor3(iuo %in% 1:4, t %between% c(-1L, 1L), ))
expect_equal(or3s(iup %in% 1:4, j %between% c(-1L, 1L), a > 0L),
             bor3(iup %in% 1:4, j %between% c(-1L, 1L), a > 0L))
expect_equal(or3s(iuq %in% 1:4, d %between% c(-1L, 1L), i > 1L),
             bor3(iuq %in% 1:4, d %between% c(-1L, 1L), i > 1L))
expect_equal(or3s(iur %in% 1:4, x %between% c(-1L, 1L), i > 9L),
             bor3(iur %in% 1:4, x %between% c(-1L, 1L), i > 9L))
expect_equal(or3s(ius %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ius %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iut %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iut %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iuu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iuu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iuv %in% c(4L, -3L, 2L, 7L, 8L), , z >= 0L),
             bor3(iuv %in% c(4L, -3L, 2L, 7L, 8L), , z >= 0L))
expect_equal(or3s(iuw %in% c(4L, -3L, 2L, 7L, 8L), , j >= 1L),
             bor3(iuw %in% c(4L, -3L, 2L, 7L, 8L), , j >= 1L))
expect_equal(or3s(iux %in% c(4L, -3L, 2L, 7L, 8L), , l >= 9L),
             bor3(iux %in% c(4L, -3L, 2L, 7L, 8L), , l >= 9L))
expect_equal(or3s(iuy %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), ),
             bor3(iuy %in% c(4L, -3L, 2L, 7L, 8L), e %between% c(-1L, 1L), ))
expect_equal(or3s(iuz %in% c(4L, -3L, 2L, 7L, 8L), t %between% c(-1L, 1L), ),
             bor3(iuz %in% c(4L, -3L, 2L, 7L, 8L), t %between% c(-1L, 1L), ))
expect_equal(or3s(iva %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L), ),
             bor3(iva %in% c(4L, -3L, 2L, 7L, 8L), j %between% c(-1L, 1L), ))
expect_equal(or3s(ivb %in% c(4L, -3L, 2L, 7L, 8L), a %between% c(-1L, 1L), z >= 0L),
             bor3(ivb %in% c(4L, -3L, 2L, 7L, 8L), a %between% c(-1L, 1L), z >= 0L))
expect_equal(or3s(ivc %in% c(4L, -3L, 2L, 7L, 8L), i %between% c(-1L, 1L), c >= 1L),
             bor3(ivc %in% c(4L, -3L, 2L, 7L, 8L), i %between% c(-1L, 1L), c >= 1L))
expect_equal(or3s(ivd %in% c(4L, -3L, 2L, 7L, 8L), g %between% c(-1L, 1L), k >= 9L),
             bor3(ivd %in% c(4L, -3L, 2L, 7L, 8L), g %between% c(-1L, 1L), k >= 9L))
expect_equal(or3s(ive %in% 1:4, , ),
             bor3(ive %in% 1:4, , ))
expect_equal(or3s(ivf %in% 1:4, , ),
             bor3(ivf %in% 1:4, , ))
expect_equal(or3s(ivg %in% 1:4, , ),
             bor3(ivg %in% 1:4, , ))
expect_equal(or3s(ivh %in% 1:4, , l >= 0L),
             bor3(ivh %in% 1:4, , l >= 0L))
expect_equal(or3s(ivi %in% 1:4, , w >= 1L),
             bor3(ivi %in% 1:4, , w >= 1L))
expect_equal(or3s(ivj %in% 1:4, , e >= 9L),
             bor3(ivj %in% 1:4, , e >= 9L))
expect_equal(or3s(ivk %in% 1:4, x %between% c(-1L, 1L), ),
             bor3(ivk %in% 1:4, x %between% c(-1L, 1L), ))
expect_equal(or3s(ivl %in% 1:4, a %between% c(-1L, 1L), ),
             bor3(ivl %in% 1:4, a %between% c(-1L, 1L), ))
expect_equal(or3s(ivm %in% 1:4, a %between% c(-1L, 1L), ),
             bor3(ivm %in% 1:4, a %between% c(-1L, 1L), ))
expect_equal(or3s(ivn %in% 1:4, s %between% c(-1L, 1L), e >= 0L),
             bor3(ivn %in% 1:4, s %between% c(-1L, 1L), e >= 0L))
expect_equal(or3s(ivo %in% 1:4, r %between% c(-1L, 1L), j >= 1L),
             bor3(ivo %in% 1:4, r %between% c(-1L, 1L), j >= 1L))
expect_equal(or3s(ivp %in% 1:4, u %between% c(-1L, 1L), l >= 9L),
             bor3(ivp %in% 1:4, u %between% c(-1L, 1L), l >= 9L))
expect_equal(or3s(ivq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ivq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ivr %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ivr %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ivs %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ivs %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ivt %in% c(4L, -3L, 2L, 7L, 8L), , logi_h),
             bor3(ivt %in% c(4L, -3L, 2L, 7L, 8L), , logi_h))
expect_equal(or3s(ivu %in% c(4L, -3L, 2L, 7L, 8L), , logi_o),
             bor3(ivu %in% c(4L, -3L, 2L, 7L, 8L), , logi_o))
expect_equal(or3s(ivv %in% c(4L, -3L, 2L, 7L, 8L), , logi_q),
             bor3(ivv %in% c(4L, -3L, 2L, 7L, 8L), , logi_q))
expect_equal(or3s(ivw %in% c(4L, -3L, 2L, 7L, 8L), o %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ivw %in% c(4L, -3L, 2L, 7L, 8L), o %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ivx %in% c(4L, -3L, 2L, 7L, 8L), z %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ivx %in% c(4L, -3L, 2L, 7L, 8L), z %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ivy %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ivy %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ivz %in% c(4L, -3L, 2L, 7L, 8L), p %in% c(4L, -3L, 2L, 7L, 8L), logi_l),
             bor3(ivz %in% c(4L, -3L, 2L, 7L, 8L), p %in% c(4L, -3L, 2L, 7L, 8L), logi_l))
expect_equal(or3s(iwa %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L), logi_i),
             bor3(iwa %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L), logi_i))
expect_equal(or3s(iwb %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L), logi_c),
             bor3(iwb %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L), logi_c))
expect_equal(or3s(iwc %in% 1:4, , ),
             bor3(iwc %in% 1:4, , ))
expect_equal(or3s(iwd %in% 1:4, , ),
             bor3(iwd %in% 1:4, , ))
expect_equal(or3s(iwe %in% 1:4, , ),
             bor3(iwe %in% 1:4, , ))
expect_equal(or3s(iwf %in% 1:4, , logi_v),
             bor3(iwf %in% 1:4, , logi_v))
expect_equal(or3s(iwg %in% 1:4, , logi_k),
             bor3(iwg %in% 1:4, , logi_k))
expect_equal(or3s(iwh %in% 1:4, , logi_z),
             bor3(iwh %in% 1:4, , logi_z))
expect_equal(or3s(iwi %in% 1:4, v %in% 1:4, ),
             bor3(iwi %in% 1:4, v %in% 1:4, ))
expect_equal(or3s(iwj %in% 1:4, c %in% 1:4, ),
             bor3(iwj %in% 1:4, c %in% 1:4, ))
expect_equal(or3s(iwk %in% 1:4, h %in% 1:4, ),
             bor3(iwk %in% 1:4, h %in% 1:4, ))
expect_equal(or3s(iwl %in% 1:4, i %in% 1:4, logi_n),
             bor3(iwl %in% 1:4, i %in% 1:4, logi_n))
expect_equal(or3s(iwm %in% 1:4, n %in% 1:4, logi_s),
             bor3(iwm %in% 1:4, n %in% 1:4, logi_s))
expect_equal(or3s(iwn %in% 1:4, x %in% 1:4, logi_t),
             bor3(iwn %in% 1:4, x %in% 1:4, logi_t))
expect_equal(or3s(iwo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iwo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iwp %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iwp %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iwq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iwq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iwr %in% c(4L, -3L, 2L, 7L, 8L), , !logi_a),
             bor3(iwr %in% c(4L, -3L, 2L, 7L, 8L), , !logi_a))
expect_equal(or3s(iws %in% c(4L, -3L, 2L, 7L, 8L), , !logi_t),
             bor3(iws %in% c(4L, -3L, 2L, 7L, 8L), , !logi_t))
expect_equal(or3s(iwt %in% c(4L, -3L, 2L, 7L, 8L), , !logi_z),
             bor3(iwt %in% c(4L, -3L, 2L, 7L, 8L), , !logi_z))
expect_equal(or3s(iwu %in% c(4L, -3L, 2L, 7L, 8L), s %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(iwu %in% c(4L, -3L, 2L, 7L, 8L), s %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(iwv %in% c(4L, -3L, 2L, 7L, 8L), l %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(iwv %in% c(4L, -3L, 2L, 7L, 8L), l %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(iww %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(iww %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(iwx %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), !logi_k),
             bor3(iwx %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), !logi_k))
expect_equal(or3s(iwy %in% c(4L, -3L, 2L, 7L, 8L), b %in% c(4L, -3L, 2L, 7L, 8L), !logi_k),
             bor3(iwy %in% c(4L, -3L, 2L, 7L, 8L), b %in% c(4L, -3L, 2L, 7L, 8L), !logi_k))
expect_equal(or3s(iwz %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), !logi_x),
             bor3(iwz %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), !logi_x))
expect_equal(or3s(ixa %in% 1:4, , ),
             bor3(ixa %in% 1:4, , ))
expect_equal(or3s(ixb %in% 1:4, , ),
             bor3(ixb %in% 1:4, , ))
expect_equal(or3s(ixc %in% 1:4, , ),
             bor3(ixc %in% 1:4, , ))
expect_equal(or3s(ixd %in% 1:4, , !logi_r),
             bor3(ixd %in% 1:4, , !logi_r))
expect_equal(or3s(ixe %in% 1:4, , !logi_b),
             bor3(ixe %in% 1:4, , !logi_b))
expect_equal(or3s(ixf %in% 1:4, , !logi_d),
             bor3(ixf %in% 1:4, , !logi_d))
expect_equal(or3s(ixg %in% 1:4, f %in% 1:4, ),
             bor3(ixg %in% 1:4, f %in% 1:4, ))
expect_equal(or3s(ixh %in% 1:4, y %in% 1:4, ),
             bor3(ixh %in% 1:4, y %in% 1:4, ))
expect_equal(or3s(ixi %in% 1:4, q %in% 1:4, ),
             bor3(ixi %in% 1:4, q %in% 1:4, ))
expect_equal(or3s(ixj %in% 1:4, z %in% 1:4, !logi_v),
             bor3(ixj %in% 1:4, z %in% 1:4, !logi_v))
expect_equal(or3s(ixk %in% 1:4, h %in% 1:4, !logi_y),
             bor3(ixk %in% 1:4, h %in% 1:4, !logi_y))
expect_equal(or3s(ixl %in% 1:4, o %in% 1:4, !logi_q),
             bor3(ixl %in% 1:4, o %in% 1:4, !logi_q))
expect_equal(or3s(ixm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ixm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ixn %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ixn %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ixo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ixo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ixp %in% c(4L, -3L, 2L, 7L, 8L), , m != 0L),
             bor3(ixp %in% c(4L, -3L, 2L, 7L, 8L), , m != 0L))
expect_equal(or3s(ixq %in% c(4L, -3L, 2L, 7L, 8L), , q != 1L),
             bor3(ixq %in% c(4L, -3L, 2L, 7L, 8L), , q != 1L))
expect_equal(or3s(ixr %in% c(4L, -3L, 2L, 7L, 8L), , v != 9L),
             bor3(ixr %in% c(4L, -3L, 2L, 7L, 8L), , v != 9L))
expect_equal(or3s(ixs %in% c(4L, -3L, 2L, 7L, 8L), a %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ixs %in% c(4L, -3L, 2L, 7L, 8L), a %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ixt %in% c(4L, -3L, 2L, 7L, 8L), p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ixt %in% c(4L, -3L, 2L, 7L, 8L), p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ixu %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ixu %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ixv %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), c != 0L),
             bor3(ixv %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), c != 0L))
expect_equal(or3s(ixw %in% c(4L, -3L, 2L, 7L, 8L), l %in% c(4L, -3L, 2L, 7L, 8L), z != 1L),
             bor3(ixw %in% c(4L, -3L, 2L, 7L, 8L), l %in% c(4L, -3L, 2L, 7L, 8L), z != 1L))
expect_equal(or3s(ixx %in% c(4L, -3L, 2L, 7L, 8L), v %in% c(4L, -3L, 2L, 7L, 8L), k != 9L),
             bor3(ixx %in% c(4L, -3L, 2L, 7L, 8L), v %in% c(4L, -3L, 2L, 7L, 8L), k != 9L))
expect_equal(or3s(ixy %in% 1:4, , ),
             bor3(ixy %in% 1:4, , ))
expect_equal(or3s(ixz %in% 1:4, , ),
             bor3(ixz %in% 1:4, , ))
expect_equal(or3s(iya %in% 1:4, , ),
             bor3(iya %in% 1:4, , ))
expect_equal(or3s(iyb %in% 1:4, , r != 0L),
             bor3(iyb %in% 1:4, , r != 0L))
expect_equal(or3s(iyc %in% 1:4, , i != 1L),
             bor3(iyc %in% 1:4, , i != 1L))
expect_equal(or3s(iyd %in% 1:4, , r != 9L),
             bor3(iyd %in% 1:4, , r != 9L))
expect_equal(or3s(iye %in% 1:4, m %in% 1:4, ),
             bor3(iye %in% 1:4, m %in% 1:4, ))
expect_equal(or3s(iyf %in% 1:4, t %in% 1:4, ),
             bor3(iyf %in% 1:4, t %in% 1:4, ))
expect_equal(or3s(iyg %in% 1:4, u %in% 1:4, ),
             bor3(iyg %in% 1:4, u %in% 1:4, ))
expect_equal(or3s(iyh %in% 1:4, k %in% 1:4, t != 0L),
             bor3(iyh %in% 1:4, k %in% 1:4, t != 0L))
expect_equal(or3s(iyi %in% 1:4, a %in% 1:4, d != 1L),
             bor3(iyi %in% 1:4, a %in% 1:4, d != 1L))
expect_equal(or3s(iyj %in% 1:4, l %in% 1:4, o != 9L),
             bor3(iyj %in% 1:4, l %in% 1:4, o != 9L))
expect_equal(or3s(iyk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iyk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iyl %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iyl %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iym %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(iym %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(iyn %in% c(4L, -3L, 2L, 7L, 8L), , e %between% c(-1L, 1L)),
             bor3(iyn %in% c(4L, -3L, 2L, 7L, 8L), , e %between% c(-1L, 1L)))
expect_equal(or3s(iyo %in% c(4L, -3L, 2L, 7L, 8L), , o %between% c(-1L, 1L)),
             bor3(iyo %in% c(4L, -3L, 2L, 7L, 8L), , o %between% c(-1L, 1L)))
expect_equal(or3s(iyp %in% c(4L, -3L, 2L, 7L, 8L), , s %between% c(-1L, 1L)),
             bor3(iyp %in% c(4L, -3L, 2L, 7L, 8L), , s %between% c(-1L, 1L)))
expect_equal(or3s(iyq %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(iyq %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(iyr %in% c(4L, -3L, 2L, 7L, 8L), l %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(iyr %in% c(4L, -3L, 2L, 7L, 8L), l %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(iys %in% c(4L, -3L, 2L, 7L, 8L), x %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(iys %in% c(4L, -3L, 2L, 7L, 8L), x %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(iyt %in% c(4L, -3L, 2L, 7L, 8L), u %in% c(4L, -3L, 2L, 7L, 8L), b %between% c(-1L, 1L)),
             bor3(iyt %in% c(4L, -3L, 2L, 7L, 8L), u %in% c(4L, -3L, 2L, 7L, 8L), b %between% c(-1L, 1L)))
expect_equal(or3s(iyu %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L), n %between% c(-1L, 1L)),
             bor3(iyu %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L), n %between% c(-1L, 1L)))
expect_equal(or3s(iyv %in% c(4L, -3L, 2L, 7L, 8L), w %in% c(4L, -3L, 2L, 7L, 8L), g %between% c(-1L, 1L)),
             bor3(iyv %in% c(4L, -3L, 2L, 7L, 8L), w %in% c(4L, -3L, 2L, 7L, 8L), g %between% c(-1L, 1L)))
expect_equal(or3s(iyw %in% 1:4, , ),
             bor3(iyw %in% 1:4, , ))
expect_equal(or3s(iyx %in% 1:4, , ),
             bor3(iyx %in% 1:4, , ))
expect_equal(or3s(iyy %in% 1:4, , ),
             bor3(iyy %in% 1:4, , ))
expect_equal(or3s(iyz %in% 1:4, , j %between% c(-1L, 1L)),
             bor3(iyz %in% 1:4, , j %between% c(-1L, 1L)))
expect_equal(or3s(ja %in% 1:4, , f %between% c(-1L, 1L)),
             bor3(ja %in% 1:4, , f %between% c(-1L, 1L)))
expect_equal(or3s(jb %in% 1:4, , z %between% c(-1L, 1L)),
             bor3(jb %in% 1:4, , z %between% c(-1L, 1L)))
expect_equal(or3s(jc %in% 1:4, z %in% 1:4, ),
             bor3(jc %in% 1:4, z %in% 1:4, ))
expect_equal(or3s(jd %in% 1:4, h %in% 1:4, ),
             bor3(jd %in% 1:4, h %in% 1:4, ))
expect_equal(or3s(je %in% 1:4, e %in% 1:4, ),
             bor3(je %in% 1:4, e %in% 1:4, ))
expect_equal(or3s(jf %in% 1:4, k %in% 1:4, c %between% c(-1L, 1L)),
             bor3(jf %in% 1:4, k %in% 1:4, c %between% c(-1L, 1L)))
expect_equal(or3s(jg %in% 1:4, n %in% 1:4, l %between% c(-1L, 1L)),
             bor3(jg %in% 1:4, n %in% 1:4, l %between% c(-1L, 1L)))
expect_equal(or3s(jh %in% 1:4, h %in% 1:4, o %between% c(-1L, 1L)),
             bor3(jh %in% 1:4, h %in% 1:4, o %between% c(-1L, 1L)))
expect_equal(or3s(ji %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ji %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jj %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jj %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jl %in% c(4L, -3L, 2L, 7L, 8L), , o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jl %in% c(4L, -3L, 2L, 7L, 8L), , o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jm %in% c(4L, -3L, 2L, 7L, 8L), , y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jm %in% c(4L, -3L, 2L, 7L, 8L), , y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jn %in% c(4L, -3L, 2L, 7L, 8L), , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jn %in% c(4L, -3L, 2L, 7L, 8L), , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jo %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jo %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jp %in% c(4L, -3L, 2L, 7L, 8L), q %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jp %in% c(4L, -3L, 2L, 7L, 8L), q %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jq %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jq %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jr %in% c(4L, -3L, 2L, 7L, 8L), d %in% c(4L, -3L, 2L, 7L, 8L), r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jr %in% c(4L, -3L, 2L, 7L, 8L), d %in% c(4L, -3L, 2L, 7L, 8L), r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(js %in% c(4L, -3L, 2L, 7L, 8L), x %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(js %in% c(4L, -3L, 2L, 7L, 8L), x %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jt %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jt %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ju %in% 1:4, , ),
             bor3(ju %in% 1:4, , ))
expect_equal(or3s(jv %in% 1:4, , ),
             bor3(jv %in% 1:4, , ))
expect_equal(or3s(jw %in% 1:4, , ),
             bor3(jw %in% 1:4, , ))
expect_equal(or3s(jx %in% 1:4, , b %in% 1:4),
             bor3(jx %in% 1:4, , b %in% 1:4))
expect_equal(or3s(jy %in% 1:4, , f %in% 1:4),
             bor3(jy %in% 1:4, , f %in% 1:4))
expect_equal(or3s(jz %in% 1:4, , h %in% 1:4),
             bor3(jz %in% 1:4, , h %in% 1:4))
expect_equal(or3s(jaa %in% 1:4, a %in% 1:4, ),
             bor3(jaa %in% 1:4, a %in% 1:4, ))
expect_equal(or3s(jab %in% 1:4, e %in% 1:4, ),
             bor3(jab %in% 1:4, e %in% 1:4, ))
expect_equal(or3s(jac %in% 1:4, r %in% 1:4, ),
             bor3(jac %in% 1:4, r %in% 1:4, ))
expect_equal(or3s(jad %in% 1:4, g %in% 1:4, h %in% 1:4),
             bor3(jad %in% 1:4, g %in% 1:4, h %in% 1:4))
expect_equal(or3s(jae %in% 1:4, e %in% 1:4, k %in% 1:4),
             bor3(jae %in% 1:4, e %in% 1:4, k %in% 1:4))
expect_equal(or3s(jaf %in% 1:4, n %in% 1:4, v %in% 1:4),
             bor3(jaf %in% 1:4, n %in% 1:4, v %in% 1:4))
expect_equal(or3s(jag %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jag %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jah %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jah %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jai %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jai %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jaj %in% c(4L, -3L, 2L, 7L, 8L), , w < 0L),
             bor3(jaj %in% c(4L, -3L, 2L, 7L, 8L), , w < 0L))
expect_equal(or3s(jak %in% c(4L, -3L, 2L, 7L, 8L), , r < 1L),
             bor3(jak %in% c(4L, -3L, 2L, 7L, 8L), , r < 1L))
expect_equal(or3s(jal %in% c(4L, -3L, 2L, 7L, 8L), , p < 9L),
             bor3(jal %in% c(4L, -3L, 2L, 7L, 8L), , p < 9L))
expect_equal(or3s(jam %in% c(4L, -3L, 2L, 7L, 8L), t %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jam %in% c(4L, -3L, 2L, 7L, 8L), t %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jan %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jan %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jao %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jao %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jap %in% c(4L, -3L, 2L, 7L, 8L), b %in% c(4L, -3L, 2L, 7L, 8L), j < 0L),
             bor3(jap %in% c(4L, -3L, 2L, 7L, 8L), b %in% c(4L, -3L, 2L, 7L, 8L), j < 0L))
expect_equal(or3s(jaq %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L), u < 1L),
             bor3(jaq %in% c(4L, -3L, 2L, 7L, 8L), e %in% c(4L, -3L, 2L, 7L, 8L), u < 1L))
expect_equal(or3s(jar %in% c(4L, -3L, 2L, 7L, 8L), u %in% c(4L, -3L, 2L, 7L, 8L), z < 9L),
             bor3(jar %in% c(4L, -3L, 2L, 7L, 8L), u %in% c(4L, -3L, 2L, 7L, 8L), z < 9L))
expect_equal(or3s(jas %in% 1:4, , ),
             bor3(jas %in% 1:4, , ))
expect_equal(or3s(jat %in% 1:4, , ),
             bor3(jat %in% 1:4, , ))
expect_equal(or3s(jau %in% 1:4, , ),
             bor3(jau %in% 1:4, , ))
expect_equal(or3s(jav %in% 1:4, , h < 0L),
             bor3(jav %in% 1:4, , h < 0L))
expect_equal(or3s(jaw %in% 1:4, , l < 1L),
             bor3(jaw %in% 1:4, , l < 1L))
expect_equal(or3s(jax %in% 1:4, , n < 9L),
             bor3(jax %in% 1:4, , n < 9L))
expect_equal(or3s(jay %in% 1:4, m %in% 1:4, ),
             bor3(jay %in% 1:4, m %in% 1:4, ))
expect_equal(or3s(jaz %in% 1:4, a %in% 1:4, ),
             bor3(jaz %in% 1:4, a %in% 1:4, ))
expect_equal(or3s(jba %in% 1:4, p %in% 1:4, ),
             bor3(jba %in% 1:4, p %in% 1:4, ))
expect_equal(or3s(jbb %in% 1:4, c %in% 1:4, w < 0L),
             bor3(jbb %in% 1:4, c %in% 1:4, w < 0L))
expect_equal(or3s(jbc %in% 1:4, v %in% 1:4, l < 1L),
             bor3(jbc %in% 1:4, v %in% 1:4, l < 1L))
expect_equal(or3s(jbd %in% 1:4, i %in% 1:4, a < 9L),
             bor3(jbd %in% 1:4, i %in% 1:4, a < 9L))
expect_equal(or3s(jbe %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jbe %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jbf %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jbf %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jbg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jbg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jbh %in% c(4L, -3L, 2L, 7L, 8L), , d <= 0L),
             bor3(jbh %in% c(4L, -3L, 2L, 7L, 8L), , d <= 0L))
expect_equal(or3s(jbi %in% c(4L, -3L, 2L, 7L, 8L), , i <= 1L),
             bor3(jbi %in% c(4L, -3L, 2L, 7L, 8L), , i <= 1L))
expect_equal(or3s(jbj %in% c(4L, -3L, 2L, 7L, 8L), , n <= 9L),
             bor3(jbj %in% c(4L, -3L, 2L, 7L, 8L), , n <= 9L))
expect_equal(or3s(jbk %in% c(4L, -3L, 2L, 7L, 8L), x %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jbk %in% c(4L, -3L, 2L, 7L, 8L), x %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jbl %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jbl %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jbm %in% c(4L, -3L, 2L, 7L, 8L), q %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jbm %in% c(4L, -3L, 2L, 7L, 8L), q %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jbn %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), e <= 0L),
             bor3(jbn %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), e <= 0L))
expect_equal(or3s(jbo %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L), x <= 1L),
             bor3(jbo %in% c(4L, -3L, 2L, 7L, 8L), c %in% c(4L, -3L, 2L, 7L, 8L), x <= 1L))
expect_equal(or3s(jbp %in% c(4L, -3L, 2L, 7L, 8L), k %in% c(4L, -3L, 2L, 7L, 8L), a <= 9L),
             bor3(jbp %in% c(4L, -3L, 2L, 7L, 8L), k %in% c(4L, -3L, 2L, 7L, 8L), a <= 9L))
expect_equal(or3s(jbq %in% 1:4, , ),
             bor3(jbq %in% 1:4, , ))
expect_equal(or3s(jbr %in% 1:4, , ),
             bor3(jbr %in% 1:4, , ))
expect_equal(or3s(jbs %in% 1:4, , ),
             bor3(jbs %in% 1:4, , ))
expect_equal(or3s(jbt %in% 1:4, , k <= 0L),
             bor3(jbt %in% 1:4, , k <= 0L))
expect_equal(or3s(jbu %in% 1:4, , w <= 1L),
             bor3(jbu %in% 1:4, , w <= 1L))
expect_equal(or3s(jbv %in% 1:4, , f <= 9L),
             bor3(jbv %in% 1:4, , f <= 9L))
expect_equal(or3s(jbw %in% 1:4, x %in% 1:4, ),
             bor3(jbw %in% 1:4, x %in% 1:4, ))
expect_equal(or3s(jbx %in% 1:4, k %in% 1:4, ),
             bor3(jbx %in% 1:4, k %in% 1:4, ))
expect_equal(or3s(jby %in% 1:4, n %in% 1:4, ),
             bor3(jby %in% 1:4, n %in% 1:4, ))
expect_equal(or3s(jbz %in% 1:4, x %in% 1:4, f <= 0L),
             bor3(jbz %in% 1:4, x %in% 1:4, f <= 0L))
expect_equal(or3s(jca %in% 1:4, r %in% 1:4, h <= 1L),
             bor3(jca %in% 1:4, r %in% 1:4, h <= 1L))
expect_equal(or3s(jcb %in% 1:4, o %in% 1:4, c <= 9L),
             bor3(jcb %in% 1:4, o %in% 1:4, c <= 9L))
expect_equal(or3s(jcc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jcc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jcd %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jcd %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jce %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jce %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jcf %in% c(4L, -3L, 2L, 7L, 8L), , z == 0L),
             bor3(jcf %in% c(4L, -3L, 2L, 7L, 8L), , z == 0L))
expect_equal(or3s(jcg %in% c(4L, -3L, 2L, 7L, 8L), , o == 1L),
             bor3(jcg %in% c(4L, -3L, 2L, 7L, 8L), , o == 1L))
expect_equal(or3s(jch %in% c(4L, -3L, 2L, 7L, 8L), , f == 9L),
             bor3(jch %in% c(4L, -3L, 2L, 7L, 8L), , f == 9L))
expect_equal(or3s(jci %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jci %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jcj %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jcj %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jck %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jck %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jcl %in% c(4L, -3L, 2L, 7L, 8L), u %in% c(4L, -3L, 2L, 7L, 8L), s == 0L),
             bor3(jcl %in% c(4L, -3L, 2L, 7L, 8L), u %in% c(4L, -3L, 2L, 7L, 8L), s == 0L))
expect_equal(or3s(jcm %in% c(4L, -3L, 2L, 7L, 8L), h %in% c(4L, -3L, 2L, 7L, 8L), l == 1L),
             bor3(jcm %in% c(4L, -3L, 2L, 7L, 8L), h %in% c(4L, -3L, 2L, 7L, 8L), l == 1L))
expect_equal(or3s(jcn %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), r == 9L),
             bor3(jcn %in% c(4L, -3L, 2L, 7L, 8L), f %in% c(4L, -3L, 2L, 7L, 8L), r == 9L))
expect_equal(or3s(jco %in% 1:4, , ),
             bor3(jco %in% 1:4, , ))
expect_equal(or3s(jcp %in% 1:4, , ),
             bor3(jcp %in% 1:4, , ))
expect_equal(or3s(jcq %in% 1:4, , ),
             bor3(jcq %in% 1:4, , ))
expect_equal(or3s(jcr %in% 1:4, , n == 0L),
             bor3(jcr %in% 1:4, , n == 0L))
expect_equal(or3s(jcs %in% 1:4, , g == 1L),
             bor3(jcs %in% 1:4, , g == 1L))
expect_equal(or3s(jct %in% 1:4, , r == 9L),
             bor3(jct %in% 1:4, , r == 9L))
expect_equal(or3s(jcu %in% 1:4, m %in% 1:4, ),
             bor3(jcu %in% 1:4, m %in% 1:4, ))
expect_equal(or3s(jcv %in% 1:4, h %in% 1:4, ),
             bor3(jcv %in% 1:4, h %in% 1:4, ))
expect_equal(or3s(jcw %in% 1:4, f %in% 1:4, ),
             bor3(jcw %in% 1:4, f %in% 1:4, ))
expect_equal(or3s(jcx %in% 1:4, c %in% 1:4, l == 0L),
             bor3(jcx %in% 1:4, c %in% 1:4, l == 0L))
expect_equal(or3s(jcy %in% 1:4, s %in% 1:4, e == 1L),
             bor3(jcy %in% 1:4, s %in% 1:4, e == 1L))
expect_equal(or3s(jcz %in% 1:4, f %in% 1:4, p == 9L),
             bor3(jcz %in% 1:4, f %in% 1:4, p == 9L))
expect_equal(or3s(jda %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jda %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jdb %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jdb %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jdc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jdc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jdd %in% c(4L, -3L, 2L, 7L, 8L), , x > 0L),
             bor3(jdd %in% c(4L, -3L, 2L, 7L, 8L), , x > 0L))
expect_equal(or3s(jde %in% c(4L, -3L, 2L, 7L, 8L), , g > 1L),
             bor3(jde %in% c(4L, -3L, 2L, 7L, 8L), , g > 1L))
expect_equal(or3s(jdf %in% c(4L, -3L, 2L, 7L, 8L), , u > 9L),
             bor3(jdf %in% c(4L, -3L, 2L, 7L, 8L), , u > 9L))
expect_equal(or3s(jdg %in% c(4L, -3L, 2L, 7L, 8L), v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jdg %in% c(4L, -3L, 2L, 7L, 8L), v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jdh %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jdh %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jdi %in% c(4L, -3L, 2L, 7L, 8L), k %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jdi %in% c(4L, -3L, 2L, 7L, 8L), k %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jdj %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), x > 0L),
             bor3(jdj %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), x > 0L))
expect_equal(or3s(jdk %in% c(4L, -3L, 2L, 7L, 8L), z %in% c(4L, -3L, 2L, 7L, 8L), m > 1L),
             bor3(jdk %in% c(4L, -3L, 2L, 7L, 8L), z %in% c(4L, -3L, 2L, 7L, 8L), m > 1L))
expect_equal(or3s(jdl %in% c(4L, -3L, 2L, 7L, 8L), z %in% c(4L, -3L, 2L, 7L, 8L), o > 9L),
             bor3(jdl %in% c(4L, -3L, 2L, 7L, 8L), z %in% c(4L, -3L, 2L, 7L, 8L), o > 9L))
expect_equal(or3s(jdm %in% 1:4, , ),
             bor3(jdm %in% 1:4, , ))
expect_equal(or3s(jdn %in% 1:4, , ),
             bor3(jdn %in% 1:4, , ))
expect_equal(or3s(jdo %in% 1:4, , ),
             bor3(jdo %in% 1:4, , ))
expect_equal(or3s(jdp %in% 1:4, , n > 0L),
             bor3(jdp %in% 1:4, , n > 0L))
expect_equal(or3s(jdq %in% 1:4, , k > 1L),
             bor3(jdq %in% 1:4, , k > 1L))
expect_equal(or3s(jdr %in% 1:4, , c > 9L),
             bor3(jdr %in% 1:4, , c > 9L))
expect_equal(or3s(jds %in% 1:4, p %in% 1:4, ),
             bor3(jds %in% 1:4, p %in% 1:4, ))
expect_equal(or3s(jdt %in% 1:4, b %in% 1:4, ),
             bor3(jdt %in% 1:4, b %in% 1:4, ))
expect_equal(or3s(jdu %in% 1:4, n %in% 1:4, ),
             bor3(jdu %in% 1:4, n %in% 1:4, ))
expect_equal(or3s(jdv %in% 1:4, t %in% 1:4, d > 0L),
             bor3(jdv %in% 1:4, t %in% 1:4, d > 0L))
expect_equal(or3s(jdw %in% 1:4, h %in% 1:4, r > 1L),
             bor3(jdw %in% 1:4, h %in% 1:4, r > 1L))
expect_equal(or3s(jdx %in% 1:4, u %in% 1:4, t > 9L),
             bor3(jdx %in% 1:4, u %in% 1:4, t > 9L))
expect_equal(or3s(jdy %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jdy %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jdz %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jdz %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jea %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jea %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jeb %in% c(4L, -3L, 2L, 7L, 8L), , d >= 0L),
             bor3(jeb %in% c(4L, -3L, 2L, 7L, 8L), , d >= 0L))
expect_equal(or3s(jec %in% c(4L, -3L, 2L, 7L, 8L), , i >= 1L),
             bor3(jec %in% c(4L, -3L, 2L, 7L, 8L), , i >= 1L))
expect_equal(or3s(jed %in% c(4L, -3L, 2L, 7L, 8L), , x >= 9L),
             bor3(jed %in% c(4L, -3L, 2L, 7L, 8L), , x >= 9L))
expect_equal(or3s(jee %in% c(4L, -3L, 2L, 7L, 8L), w %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jee %in% c(4L, -3L, 2L, 7L, 8L), w %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jef %in% c(4L, -3L, 2L, 7L, 8L), d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jef %in% c(4L, -3L, 2L, 7L, 8L), d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jeg %in% c(4L, -3L, 2L, 7L, 8L), q %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(jeg %in% c(4L, -3L, 2L, 7L, 8L), q %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(jeh %in% c(4L, -3L, 2L, 7L, 8L), o %in% c(4L, -3L, 2L, 7L, 8L), p >= 0L),
             bor3(jeh %in% c(4L, -3L, 2L, 7L, 8L), o %in% c(4L, -3L, 2L, 7L, 8L), p >= 0L))
expect_equal(or3s(jei %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), b >= 1L),
             bor3(jei %in% c(4L, -3L, 2L, 7L, 8L), i %in% c(4L, -3L, 2L, 7L, 8L), b >= 1L))
expect_equal(or3s(jej %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L), x >= 9L),
             bor3(jej %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L), x >= 9L))
expect_equal(or3s(jek %in% 1:4, , ),
             bor3(jek %in% 1:4, , ))
expect_equal(or3s(jel %in% 1:4, , ),
             bor3(jel %in% 1:4, , ))
expect_equal(or3s(jem %in% 1:4, , ),
             bor3(jem %in% 1:4, , ))
expect_equal(or3s(jen %in% 1:4, , c >= 0L),
             bor3(jen %in% 1:4, , c >= 0L))
expect_equal(or3s(jeo %in% 1:4, , f >= 1L),
             bor3(jeo %in% 1:4, , f >= 1L))
expect_equal(or3s(jep %in% 1:4, , r >= 9L),
             bor3(jep %in% 1:4, , r >= 9L))
expect_equal(or3s(jeq %in% 1:4, l %in% 1:4, ),
             bor3(jeq %in% 1:4, l %in% 1:4, ))
expect_equal(or3s(jer %in% 1:4, l %in% 1:4, ),
             bor3(jer %in% 1:4, l %in% 1:4, ))
expect_equal(or3s(jes %in% 1:4, f %in% 1:4, ),
             bor3(jes %in% 1:4, f %in% 1:4, ))
expect_equal(or3s(jet %in% 1:4, x %in% 1:4, l >= 0L),
             bor3(jet %in% 1:4, x %in% 1:4, l >= 0L))
expect_equal(or3s(jeu %in% 1:4, r %in% 1:4, k >= 1L),
             bor3(jeu %in% 1:4, r %in% 1:4, k >= 1L))
expect_equal(or3s(jev %in% 1:4, i %in% 1:4, f >= 9L),
             bor3(jev %in% 1:4, i %in% 1:4, f >= 9L))
expect_equal(or3s(jew %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jew %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jex %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jex %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jey %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jey %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jez %in% c(4L, -3L, 2L, 7L, 8L), , logi_x),
             bor3(jez %in% c(4L, -3L, 2L, 7L, 8L), , logi_x))
expect_equal(or3s(jfa %in% c(4L, -3L, 2L, 7L, 8L), , logi_t),
             bor3(jfa %in% c(4L, -3L, 2L, 7L, 8L), , logi_t))
expect_equal(or3s(jfb %in% c(4L, -3L, 2L, 7L, 8L), , logi_j),
             bor3(jfb %in% c(4L, -3L, 2L, 7L, 8L), , logi_j))
expect_equal(or3s(jfc %in% c(4L, -3L, 2L, 7L, 8L), a < 0L, ),
             bor3(jfc %in% c(4L, -3L, 2L, 7L, 8L), a < 0L, ))
expect_equal(or3s(jfd %in% c(4L, -3L, 2L, 7L, 8L), g < 1L, ),
             bor3(jfd %in% c(4L, -3L, 2L, 7L, 8L), g < 1L, ))
expect_equal(or3s(jfe %in% c(4L, -3L, 2L, 7L, 8L), q < 9L, ),
             bor3(jfe %in% c(4L, -3L, 2L, 7L, 8L), q < 9L, ))
expect_equal(or3s(jff %in% c(4L, -3L, 2L, 7L, 8L), q < 0L, logi_b),
             bor3(jff %in% c(4L, -3L, 2L, 7L, 8L), q < 0L, logi_b))
expect_equal(or3s(jfg %in% c(4L, -3L, 2L, 7L, 8L), p < 1L, logi_z),
             bor3(jfg %in% c(4L, -3L, 2L, 7L, 8L), p < 1L, logi_z))
expect_equal(or3s(jfh %in% c(4L, -3L, 2L, 7L, 8L), o < 9L, logi_a),
             bor3(jfh %in% c(4L, -3L, 2L, 7L, 8L), o < 9L, logi_a))
expect_equal(or3s(jfi %in% 1:4, , ),
             bor3(jfi %in% 1:4, , ))
expect_equal(or3s(jfj %in% 1:4, , ),
             bor3(jfj %in% 1:4, , ))
expect_equal(or3s(jfk %in% 1:4, , ),
             bor3(jfk %in% 1:4, , ))
expect_equal(or3s(jfl %in% 1:4, , logi_p),
             bor3(jfl %in% 1:4, , logi_p))
expect_equal(or3s(jfm %in% 1:4, , logi_u),
             bor3(jfm %in% 1:4, , logi_u))
expect_equal(or3s(jfn %in% 1:4, , logi_n),
             bor3(jfn %in% 1:4, , logi_n))
expect_equal(or3s(jfo %in% 1:4, g < 0L, ),
             bor3(jfo %in% 1:4, g < 0L, ))
expect_equal(or3s(jfp %in% 1:4, s < 1L, ),
             bor3(jfp %in% 1:4, s < 1L, ))
expect_equal(or3s(jfq %in% 1:4, j < 9L, ),
             bor3(jfq %in% 1:4, j < 9L, ))
expect_equal(or3s(jfr %in% 1:4, a < 0L, logi_m),
             bor3(jfr %in% 1:4, a < 0L, logi_m))
expect_equal(or3s(jfs %in% 1:4, z < 1L, logi_j),
             bor3(jfs %in% 1:4, z < 1L, logi_j))
expect_equal(or3s(jft %in% 1:4, m < 9L, logi_n),
             bor3(jft %in% 1:4, m < 9L, logi_n))
expect_equal(or3s(jfu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jfu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jfv %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jfv %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jfw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jfw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jfx %in% c(4L, -3L, 2L, 7L, 8L), , !logi_z),
             bor3(jfx %in% c(4L, -3L, 2L, 7L, 8L), , !logi_z))
expect_equal(or3s(jfy %in% c(4L, -3L, 2L, 7L, 8L), , !logi_o),
             bor3(jfy %in% c(4L, -3L, 2L, 7L, 8L), , !logi_o))
expect_equal(or3s(jfz %in% c(4L, -3L, 2L, 7L, 8L), , !logi_h),
             bor3(jfz %in% c(4L, -3L, 2L, 7L, 8L), , !logi_h))
expect_equal(or3s(jga %in% c(4L, -3L, 2L, 7L, 8L), v < 0L, ),
             bor3(jga %in% c(4L, -3L, 2L, 7L, 8L), v < 0L, ))
expect_equal(or3s(jgb %in% c(4L, -3L, 2L, 7L, 8L), o < 1L, ),
             bor3(jgb %in% c(4L, -3L, 2L, 7L, 8L), o < 1L, ))
expect_equal(or3s(jgc %in% c(4L, -3L, 2L, 7L, 8L), d < 9L, ),
             bor3(jgc %in% c(4L, -3L, 2L, 7L, 8L), d < 9L, ))
expect_equal(or3s(jgd %in% c(4L, -3L, 2L, 7L, 8L), p < 0L, !logi_v),
             bor3(jgd %in% c(4L, -3L, 2L, 7L, 8L), p < 0L, !logi_v))
expect_equal(or3s(jge %in% c(4L, -3L, 2L, 7L, 8L), c < 1L, !logi_t),
             bor3(jge %in% c(4L, -3L, 2L, 7L, 8L), c < 1L, !logi_t))
expect_equal(or3s(jgf %in% c(4L, -3L, 2L, 7L, 8L), q < 9L, !logi_q),
             bor3(jgf %in% c(4L, -3L, 2L, 7L, 8L), q < 9L, !logi_q))
expect_equal(or3s(jgg %in% 1:4, , ),
             bor3(jgg %in% 1:4, , ))
expect_equal(or3s(jgh %in% 1:4, , ),
             bor3(jgh %in% 1:4, , ))
expect_equal(or3s(jgi %in% 1:4, , ),
             bor3(jgi %in% 1:4, , ))
expect_equal(or3s(jgj %in% 1:4, , !logi_y),
             bor3(jgj %in% 1:4, , !logi_y))
expect_equal(or3s(jgk %in% 1:4, , !logi_m),
             bor3(jgk %in% 1:4, , !logi_m))
expect_equal(or3s(jgl %in% 1:4, , !logi_c),
             bor3(jgl %in% 1:4, , !logi_c))
expect_equal(or3s(jgm %in% 1:4, s < 0L, ),
             bor3(jgm %in% 1:4, s < 0L, ))
expect_equal(or3s(jgn %in% 1:4, a < 1L, ),
             bor3(jgn %in% 1:4, a < 1L, ))
expect_equal(or3s(jgo %in% 1:4, k < 9L, ),
             bor3(jgo %in% 1:4, k < 9L, ))
expect_equal(or3s(jgp %in% 1:4, g < 0L, !logi_s),
             bor3(jgp %in% 1:4, g < 0L, !logi_s))
expect_equal(or3s(jgq %in% 1:4, z < 1L, !logi_t),
             bor3(jgq %in% 1:4, z < 1L, !logi_t))
expect_equal(or3s(jgr %in% 1:4, w < 9L, !logi_k),
             bor3(jgr %in% 1:4, w < 9L, !logi_k))
expect_equal(or3s(jgs %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jgs %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jgt %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jgt %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jgu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jgu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jgv %in% c(4L, -3L, 2L, 7L, 8L), , g != 0L),
             bor3(jgv %in% c(4L, -3L, 2L, 7L, 8L), , g != 0L))
expect_equal(or3s(jgw %in% c(4L, -3L, 2L, 7L, 8L), , u != 1L),
             bor3(jgw %in% c(4L, -3L, 2L, 7L, 8L), , u != 1L))
expect_equal(or3s(jgx %in% c(4L, -3L, 2L, 7L, 8L), , u != 9L),
             bor3(jgx %in% c(4L, -3L, 2L, 7L, 8L), , u != 9L))
expect_equal(or3s(jgy %in% c(4L, -3L, 2L, 7L, 8L), y < 0L, ),
             bor3(jgy %in% c(4L, -3L, 2L, 7L, 8L), y < 0L, ))
expect_equal(or3s(jgz %in% c(4L, -3L, 2L, 7L, 8L), b < 1L, ),
             bor3(jgz %in% c(4L, -3L, 2L, 7L, 8L), b < 1L, ))
expect_equal(or3s(jha %in% c(4L, -3L, 2L, 7L, 8L), f < 9L, ),
             bor3(jha %in% c(4L, -3L, 2L, 7L, 8L), f < 9L, ))
expect_equal(or3s(jhb %in% c(4L, -3L, 2L, 7L, 8L), y < 0L, o != 0L),
             bor3(jhb %in% c(4L, -3L, 2L, 7L, 8L), y < 0L, o != 0L))
expect_equal(or3s(jhc %in% c(4L, -3L, 2L, 7L, 8L), r < 1L, z != 1L),
             bor3(jhc %in% c(4L, -3L, 2L, 7L, 8L), r < 1L, z != 1L))
expect_equal(or3s(jhd %in% c(4L, -3L, 2L, 7L, 8L), d < 9L, p != 9L),
             bor3(jhd %in% c(4L, -3L, 2L, 7L, 8L), d < 9L, p != 9L))
expect_equal(or3s(jhe %in% 1:4, , ),
             bor3(jhe %in% 1:4, , ))
expect_equal(or3s(jhf %in% 1:4, , ),
             bor3(jhf %in% 1:4, , ))
expect_equal(or3s(jhg %in% 1:4, , ),
             bor3(jhg %in% 1:4, , ))
expect_equal(or3s(jhh %in% 1:4, , p != 0L),
             bor3(jhh %in% 1:4, , p != 0L))
expect_equal(or3s(jhi %in% 1:4, , o != 1L),
             bor3(jhi %in% 1:4, , o != 1L))
expect_equal(or3s(jhj %in% 1:4, , s != 9L),
             bor3(jhj %in% 1:4, , s != 9L))
expect_equal(or3s(jhk %in% 1:4, f < 0L, ),
             bor3(jhk %in% 1:4, f < 0L, ))
expect_equal(or3s(jhl %in% 1:4, h < 1L, ),
             bor3(jhl %in% 1:4, h < 1L, ))
expect_equal(or3s(jhm %in% 1:4, l < 9L, ),
             bor3(jhm %in% 1:4, l < 9L, ))
expect_equal(or3s(jhn %in% 1:4, o < 0L, i != 0L),
             bor3(jhn %in% 1:4, o < 0L, i != 0L))
expect_equal(or3s(jho %in% 1:4, p < 1L, x != 1L),
             bor3(jho %in% 1:4, p < 1L, x != 1L))
expect_equal(or3s(jhp %in% 1:4, p < 9L, e != 9L),
             bor3(jhp %in% 1:4, p < 9L, e != 9L))
expect_equal(or3s(jhq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jhq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jhr %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jhr %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jhs %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jhs %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jht %in% c(4L, -3L, 2L, 7L, 8L), , i %between% c(-1L, 1L)),
             bor3(jht %in% c(4L, -3L, 2L, 7L, 8L), , i %between% c(-1L, 1L)))
expect_equal(or3s(jhu %in% c(4L, -3L, 2L, 7L, 8L), , s %between% c(-1L, 1L)),
             bor3(jhu %in% c(4L, -3L, 2L, 7L, 8L), , s %between% c(-1L, 1L)))
expect_equal(or3s(jhv %in% c(4L, -3L, 2L, 7L, 8L), , m %between% c(-1L, 1L)),
             bor3(jhv %in% c(4L, -3L, 2L, 7L, 8L), , m %between% c(-1L, 1L)))
expect_equal(or3s(jhw %in% c(4L, -3L, 2L, 7L, 8L), f < 0L, ),
             bor3(jhw %in% c(4L, -3L, 2L, 7L, 8L), f < 0L, ))
expect_equal(or3s(jhx %in% c(4L, -3L, 2L, 7L, 8L), l < 1L, ),
             bor3(jhx %in% c(4L, -3L, 2L, 7L, 8L), l < 1L, ))
expect_equal(or3s(jhy %in% c(4L, -3L, 2L, 7L, 8L), w < 9L, ),
             bor3(jhy %in% c(4L, -3L, 2L, 7L, 8L), w < 9L, ))
expect_equal(or3s(jhz %in% c(4L, -3L, 2L, 7L, 8L), p < 0L, n %between% c(-1L, 1L)),
             bor3(jhz %in% c(4L, -3L, 2L, 7L, 8L), p < 0L, n %between% c(-1L, 1L)))
expect_equal(or3s(jia %in% c(4L, -3L, 2L, 7L, 8L), s < 1L, y %between% c(-1L, 1L)),
             bor3(jia %in% c(4L, -3L, 2L, 7L, 8L), s < 1L, y %between% c(-1L, 1L)))
expect_equal(or3s(jib %in% c(4L, -3L, 2L, 7L, 8L), k < 9L, y %between% c(-1L, 1L)),
             bor3(jib %in% c(4L, -3L, 2L, 7L, 8L), k < 9L, y %between% c(-1L, 1L)))
expect_equal(or3s(jic %in% 1:4, , ),
             bor3(jic %in% 1:4, , ))
expect_equal(or3s(jid %in% 1:4, , ),
             bor3(jid %in% 1:4, , ))
expect_equal(or3s(jie %in% 1:4, , ),
             bor3(jie %in% 1:4, , ))
expect_equal(or3s(jif %in% 1:4, , e %between% c(-1L, 1L)),
             bor3(jif %in% 1:4, , e %between% c(-1L, 1L)))
expect_equal(or3s(jig %in% 1:4, , l %between% c(-1L, 1L)),
             bor3(jig %in% 1:4, , l %between% c(-1L, 1L)))
expect_equal(or3s(jih %in% 1:4, , b %between% c(-1L, 1L)),
             bor3(jih %in% 1:4, , b %between% c(-1L, 1L)))
expect_equal(or3s(jii %in% 1:4, c < 0L, ),
             bor3(jii %in% 1:4, c < 0L, ))
expect_equal(or3s(jij %in% 1:4, y < 1L, ),
             bor3(jij %in% 1:4, y < 1L, ))
expect_equal(or3s(jik %in% 1:4, e < 9L, ),
             bor3(jik %in% 1:4, e < 9L, ))
expect_equal(or3s(jil %in% 1:4, a < 0L, x %between% c(-1L, 1L)),
             bor3(jil %in% 1:4, a < 0L, x %between% c(-1L, 1L)))
expect_equal(or3s(jim %in% 1:4, x < 1L, d %between% c(-1L, 1L)),
             bor3(jim %in% 1:4, x < 1L, d %between% c(-1L, 1L)))
expect_equal(or3s(jin %in% 1:4, x < 9L, h %between% c(-1L, 1L)),
             bor3(jin %in% 1:4, x < 9L, h %between% c(-1L, 1L)))
expect_equal(or3s(jio %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jio %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jip %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jip %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jiq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jiq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jir %in% c(4L, -3L, 2L, 7L, 8L), , e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jir %in% c(4L, -3L, 2L, 7L, 8L), , e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jis %in% c(4L, -3L, 2L, 7L, 8L), , b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jis %in% c(4L, -3L, 2L, 7L, 8L), , b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jit %in% c(4L, -3L, 2L, 7L, 8L), , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jit %in% c(4L, -3L, 2L, 7L, 8L), , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jiu %in% c(4L, -3L, 2L, 7L, 8L), k < 0L, ),
             bor3(jiu %in% c(4L, -3L, 2L, 7L, 8L), k < 0L, ))
expect_equal(or3s(jiv %in% c(4L, -3L, 2L, 7L, 8L), a < 1L, ),
             bor3(jiv %in% c(4L, -3L, 2L, 7L, 8L), a < 1L, ))
expect_equal(or3s(jiw %in% c(4L, -3L, 2L, 7L, 8L), c < 9L, ),
             bor3(jiw %in% c(4L, -3L, 2L, 7L, 8L), c < 9L, ))
expect_equal(or3s(jix %in% c(4L, -3L, 2L, 7L, 8L), d < 0L, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jix %in% c(4L, -3L, 2L, 7L, 8L), d < 0L, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jiy %in% c(4L, -3L, 2L, 7L, 8L), f < 1L, x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jiy %in% c(4L, -3L, 2L, 7L, 8L), f < 1L, x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jiz %in% c(4L, -3L, 2L, 7L, 8L), e < 9L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jiz %in% c(4L, -3L, 2L, 7L, 8L), e < 9L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jja %in% 1:4, , ),
             bor3(jja %in% 1:4, , ))
expect_equal(or3s(jjb %in% 1:4, , ),
             bor3(jjb %in% 1:4, , ))
expect_equal(or3s(jjc %in% 1:4, , ),
             bor3(jjc %in% 1:4, , ))
expect_equal(or3s(jjd %in% 1:4, , a %in% 1:4),
             bor3(jjd %in% 1:4, , a %in% 1:4))
expect_equal(or3s(jje %in% 1:4, , y %in% 1:4),
             bor3(jje %in% 1:4, , y %in% 1:4))
expect_equal(or3s(jjf %in% 1:4, , c %in% 1:4),
             bor3(jjf %in% 1:4, , c %in% 1:4))
expect_equal(or3s(jjg %in% 1:4, h < 0L, ),
             bor3(jjg %in% 1:4, h < 0L, ))
expect_equal(or3s(jjh %in% 1:4, y < 1L, ),
             bor3(jjh %in% 1:4, y < 1L, ))
expect_equal(or3s(jji %in% 1:4, e < 9L, ),
             bor3(jji %in% 1:4, e < 9L, ))
expect_equal(or3s(jjj %in% 1:4, u < 0L, e %in% 1:4),
             bor3(jjj %in% 1:4, u < 0L, e %in% 1:4))
expect_equal(or3s(jjk %in% 1:4, e < 1L, m %in% 1:4),
             bor3(jjk %in% 1:4, e < 1L, m %in% 1:4))
expect_equal(or3s(jjl %in% 1:4, b < 9L, p %in% 1:4),
             bor3(jjl %in% 1:4, b < 9L, p %in% 1:4))
expect_equal(or3s(jjm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jjm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jjn %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jjn %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jjo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jjo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jjp %in% c(4L, -3L, 2L, 7L, 8L), , q < 0L),
             bor3(jjp %in% c(4L, -3L, 2L, 7L, 8L), , q < 0L))
expect_equal(or3s(jjq %in% c(4L, -3L, 2L, 7L, 8L), , x < 1L),
             bor3(jjq %in% c(4L, -3L, 2L, 7L, 8L), , x < 1L))
expect_equal(or3s(jjr %in% c(4L, -3L, 2L, 7L, 8L), , n < 9L),
             bor3(jjr %in% c(4L, -3L, 2L, 7L, 8L), , n < 9L))
expect_equal(or3s(jjs %in% c(4L, -3L, 2L, 7L, 8L), j < 0L, ),
             bor3(jjs %in% c(4L, -3L, 2L, 7L, 8L), j < 0L, ))
expect_equal(or3s(jjt %in% c(4L, -3L, 2L, 7L, 8L), x < 1L, ),
             bor3(jjt %in% c(4L, -3L, 2L, 7L, 8L), x < 1L, ))
expect_equal(or3s(jju %in% c(4L, -3L, 2L, 7L, 8L), b < 9L, ),
             bor3(jju %in% c(4L, -3L, 2L, 7L, 8L), b < 9L, ))
expect_equal(or3s(jjv %in% c(4L, -3L, 2L, 7L, 8L), a < 0L, q < 0L),
             bor3(jjv %in% c(4L, -3L, 2L, 7L, 8L), a < 0L, q < 0L))
expect_equal(or3s(jjw %in% c(4L, -3L, 2L, 7L, 8L), f < 1L, s < 1L),
             bor3(jjw %in% c(4L, -3L, 2L, 7L, 8L), f < 1L, s < 1L))
expect_equal(or3s(jjx %in% c(4L, -3L, 2L, 7L, 8L), s < 9L, n < 9L),
             bor3(jjx %in% c(4L, -3L, 2L, 7L, 8L), s < 9L, n < 9L))
expect_equal(or3s(jjy %in% 1:4, , ),
             bor3(jjy %in% 1:4, , ))
expect_equal(or3s(jjz %in% 1:4, , ),
             bor3(jjz %in% 1:4, , ))
expect_equal(or3s(jka %in% 1:4, , ),
             bor3(jka %in% 1:4, , ))
expect_equal(or3s(jkb %in% 1:4, , y < 0L),
             bor3(jkb %in% 1:4, , y < 0L))
expect_equal(or3s(jkc %in% 1:4, , u < 1L),
             bor3(jkc %in% 1:4, , u < 1L))
expect_equal(or3s(jkd %in% 1:4, , l < 9L),
             bor3(jkd %in% 1:4, , l < 9L))
expect_equal(or3s(jke %in% 1:4, j < 0L, ),
             bor3(jke %in% 1:4, j < 0L, ))
expect_equal(or3s(jkf %in% 1:4, r < 1L, ),
             bor3(jkf %in% 1:4, r < 1L, ))
expect_equal(or3s(jkg %in% 1:4, n < 9L, ),
             bor3(jkg %in% 1:4, n < 9L, ))
expect_equal(or3s(jkh %in% 1:4, p < 0L, k < 0L),
             bor3(jkh %in% 1:4, p < 0L, k < 0L))
expect_equal(or3s(jki %in% 1:4, q < 1L, t < 1L),
             bor3(jki %in% 1:4, q < 1L, t < 1L))
expect_equal(or3s(jkj %in% 1:4, u < 9L, u < 9L),
             bor3(jkj %in% 1:4, u < 9L, u < 9L))
expect_equal(or3s(jkk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jkk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jkl %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jkl %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jkm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jkm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jkn %in% c(4L, -3L, 2L, 7L, 8L), , w <= 0L),
             bor3(jkn %in% c(4L, -3L, 2L, 7L, 8L), , w <= 0L))
expect_equal(or3s(jko %in% c(4L, -3L, 2L, 7L, 8L), , j <= 1L),
             bor3(jko %in% c(4L, -3L, 2L, 7L, 8L), , j <= 1L))
expect_equal(or3s(jkp %in% c(4L, -3L, 2L, 7L, 8L), , e <= 9L),
             bor3(jkp %in% c(4L, -3L, 2L, 7L, 8L), , e <= 9L))
expect_equal(or3s(jkq %in% c(4L, -3L, 2L, 7L, 8L), s < 0L, ),
             bor3(jkq %in% c(4L, -3L, 2L, 7L, 8L), s < 0L, ))
expect_equal(or3s(jkr %in% c(4L, -3L, 2L, 7L, 8L), k < 1L, ),
             bor3(jkr %in% c(4L, -3L, 2L, 7L, 8L), k < 1L, ))
expect_equal(or3s(jks %in% c(4L, -3L, 2L, 7L, 8L), v < 9L, ),
             bor3(jks %in% c(4L, -3L, 2L, 7L, 8L), v < 9L, ))
expect_equal(or3s(jkt %in% c(4L, -3L, 2L, 7L, 8L), x < 0L, t <= 0L),
             bor3(jkt %in% c(4L, -3L, 2L, 7L, 8L), x < 0L, t <= 0L))
expect_equal(or3s(jku %in% c(4L, -3L, 2L, 7L, 8L), y < 1L, z <= 1L),
             bor3(jku %in% c(4L, -3L, 2L, 7L, 8L), y < 1L, z <= 1L))
expect_equal(or3s(jkv %in% c(4L, -3L, 2L, 7L, 8L), r < 9L, h <= 9L),
             bor3(jkv %in% c(4L, -3L, 2L, 7L, 8L), r < 9L, h <= 9L))
expect_equal(or3s(jkw %in% 1:4, , ),
             bor3(jkw %in% 1:4, , ))
expect_equal(or3s(jkx %in% 1:4, , ),
             bor3(jkx %in% 1:4, , ))
expect_equal(or3s(jky %in% 1:4, , ),
             bor3(jky %in% 1:4, , ))
expect_equal(or3s(jkz %in% 1:4, , h <= 0L),
             bor3(jkz %in% 1:4, , h <= 0L))
expect_equal(or3s(jla %in% 1:4, , y <= 1L),
             bor3(jla %in% 1:4, , y <= 1L))
expect_equal(or3s(jlb %in% 1:4, , y <= 9L),
             bor3(jlb %in% 1:4, , y <= 9L))
expect_equal(or3s(jlc %in% 1:4, p < 0L, ),
             bor3(jlc %in% 1:4, p < 0L, ))
expect_equal(or3s(jld %in% 1:4, i < 1L, ),
             bor3(jld %in% 1:4, i < 1L, ))
expect_equal(or3s(jle %in% 1:4, v < 9L, ),
             bor3(jle %in% 1:4, v < 9L, ))
expect_equal(or3s(jlf %in% 1:4, l < 0L, s <= 0L),
             bor3(jlf %in% 1:4, l < 0L, s <= 0L))
expect_equal(or3s(jlg %in% 1:4, d < 1L, v <= 1L),
             bor3(jlg %in% 1:4, d < 1L, v <= 1L))
expect_equal(or3s(jlh %in% 1:4, n < 9L, z <= 9L),
             bor3(jlh %in% 1:4, n < 9L, z <= 9L))
expect_equal(or3s(jli %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jli %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jlj %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jlj %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jlk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jlk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jll %in% c(4L, -3L, 2L, 7L, 8L), , v == 0L),
             bor3(jll %in% c(4L, -3L, 2L, 7L, 8L), , v == 0L))
expect_equal(or3s(jlm %in% c(4L, -3L, 2L, 7L, 8L), , n == 1L),
             bor3(jlm %in% c(4L, -3L, 2L, 7L, 8L), , n == 1L))
expect_equal(or3s(jln %in% c(4L, -3L, 2L, 7L, 8L), , c == 9L),
             bor3(jln %in% c(4L, -3L, 2L, 7L, 8L), , c == 9L))
expect_equal(or3s(jlo %in% c(4L, -3L, 2L, 7L, 8L), o < 0L, ),
             bor3(jlo %in% c(4L, -3L, 2L, 7L, 8L), o < 0L, ))
expect_equal(or3s(jlp %in% c(4L, -3L, 2L, 7L, 8L), t < 1L, ),
             bor3(jlp %in% c(4L, -3L, 2L, 7L, 8L), t < 1L, ))
expect_equal(or3s(jlq %in% c(4L, -3L, 2L, 7L, 8L), w < 9L, ),
             bor3(jlq %in% c(4L, -3L, 2L, 7L, 8L), w < 9L, ))
expect_equal(or3s(jlr %in% c(4L, -3L, 2L, 7L, 8L), i < 0L, a == 0L),
             bor3(jlr %in% c(4L, -3L, 2L, 7L, 8L), i < 0L, a == 0L))
expect_equal(or3s(jls %in% c(4L, -3L, 2L, 7L, 8L), n < 1L, t == 1L),
             bor3(jls %in% c(4L, -3L, 2L, 7L, 8L), n < 1L, t == 1L))
expect_equal(or3s(jlt %in% c(4L, -3L, 2L, 7L, 8L), w < 9L, n == 9L),
             bor3(jlt %in% c(4L, -3L, 2L, 7L, 8L), w < 9L, n == 9L))
expect_equal(or3s(jlu %in% 1:4, , ),
             bor3(jlu %in% 1:4, , ))
expect_equal(or3s(jlv %in% 1:4, , ),
             bor3(jlv %in% 1:4, , ))
expect_equal(or3s(jlw %in% 1:4, , ),
             bor3(jlw %in% 1:4, , ))
expect_equal(or3s(jlx %in% 1:4, , j == 0L),
             bor3(jlx %in% 1:4, , j == 0L))
expect_equal(or3s(jly %in% 1:4, , z == 1L),
             bor3(jly %in% 1:4, , z == 1L))
expect_equal(or3s(jlz %in% 1:4, , n == 9L),
             bor3(jlz %in% 1:4, , n == 9L))
expect_equal(or3s(jma %in% 1:4, l < 0L, ),
             bor3(jma %in% 1:4, l < 0L, ))
expect_equal(or3s(jmb %in% 1:4, r < 1L, ),
             bor3(jmb %in% 1:4, r < 1L, ))
expect_equal(or3s(jmc %in% 1:4, g < 9L, ),
             bor3(jmc %in% 1:4, g < 9L, ))
expect_equal(or3s(jmd %in% 1:4, z < 0L, c == 0L),
             bor3(jmd %in% 1:4, z < 0L, c == 0L))
expect_equal(or3s(jme %in% 1:4, n < 1L, s == 1L),
             bor3(jme %in% 1:4, n < 1L, s == 1L))
expect_equal(or3s(jmf %in% 1:4, a < 9L, g == 9L),
             bor3(jmf %in% 1:4, a < 9L, g == 9L))
expect_equal(or3s(jmg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jmg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jmh %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jmh %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jmi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jmi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jmj %in% c(4L, -3L, 2L, 7L, 8L), , o > 0L),
             bor3(jmj %in% c(4L, -3L, 2L, 7L, 8L), , o > 0L))
expect_equal(or3s(jmk %in% c(4L, -3L, 2L, 7L, 8L), , z > 1L),
             bor3(jmk %in% c(4L, -3L, 2L, 7L, 8L), , z > 1L))
expect_equal(or3s(jml %in% c(4L, -3L, 2L, 7L, 8L), , t > 9L),
             bor3(jml %in% c(4L, -3L, 2L, 7L, 8L), , t > 9L))
expect_equal(or3s(jmm %in% c(4L, -3L, 2L, 7L, 8L), w < 0L, ),
             bor3(jmm %in% c(4L, -3L, 2L, 7L, 8L), w < 0L, ))
expect_equal(or3s(jmn %in% c(4L, -3L, 2L, 7L, 8L), d < 1L, ),
             bor3(jmn %in% c(4L, -3L, 2L, 7L, 8L), d < 1L, ))
expect_equal(or3s(jmo %in% c(4L, -3L, 2L, 7L, 8L), e < 9L, ),
             bor3(jmo %in% c(4L, -3L, 2L, 7L, 8L), e < 9L, ))
expect_equal(or3s(jmp %in% c(4L, -3L, 2L, 7L, 8L), w < 0L, n > 0L),
             bor3(jmp %in% c(4L, -3L, 2L, 7L, 8L), w < 0L, n > 0L))
expect_equal(or3s(jmq %in% c(4L, -3L, 2L, 7L, 8L), a < 1L, o > 1L),
             bor3(jmq %in% c(4L, -3L, 2L, 7L, 8L), a < 1L, o > 1L))
expect_equal(or3s(jmr %in% c(4L, -3L, 2L, 7L, 8L), g < 9L, l > 9L),
             bor3(jmr %in% c(4L, -3L, 2L, 7L, 8L), g < 9L, l > 9L))
expect_equal(or3s(jms %in% 1:4, , ),
             bor3(jms %in% 1:4, , ))
expect_equal(or3s(jmt %in% 1:4, , ),
             bor3(jmt %in% 1:4, , ))
expect_equal(or3s(jmu %in% 1:4, , ),
             bor3(jmu %in% 1:4, , ))
expect_equal(or3s(jmv %in% 1:4, , e > 0L),
             bor3(jmv %in% 1:4, , e > 0L))
expect_equal(or3s(jmw %in% 1:4, , n > 1L),
             bor3(jmw %in% 1:4, , n > 1L))
expect_equal(or3s(jmx %in% 1:4, , l > 9L),
             bor3(jmx %in% 1:4, , l > 9L))
expect_equal(or3s(jmy %in% 1:4, q < 0L, ),
             bor3(jmy %in% 1:4, q < 0L, ))
expect_equal(or3s(jmz %in% 1:4, a < 1L, ),
             bor3(jmz %in% 1:4, a < 1L, ))
expect_equal(or3s(jna %in% 1:4, x < 9L, ),
             bor3(jna %in% 1:4, x < 9L, ))
expect_equal(or3s(jnb %in% 1:4, j < 0L, n > 0L),
             bor3(jnb %in% 1:4, j < 0L, n > 0L))
expect_equal(or3s(jnc %in% 1:4, x < 1L, b > 1L),
             bor3(jnc %in% 1:4, x < 1L, b > 1L))
expect_equal(or3s(jnd %in% 1:4, k < 9L, n > 9L),
             bor3(jnd %in% 1:4, k < 9L, n > 9L))
expect_equal(or3s(jne %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jne %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jnf %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jnf %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jng %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jng %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jnh %in% c(4L, -3L, 2L, 7L, 8L), , w >= 0L),
             bor3(jnh %in% c(4L, -3L, 2L, 7L, 8L), , w >= 0L))
expect_equal(or3s(jni %in% c(4L, -3L, 2L, 7L, 8L), , v >= 1L),
             bor3(jni %in% c(4L, -3L, 2L, 7L, 8L), , v >= 1L))
expect_equal(or3s(jnj %in% c(4L, -3L, 2L, 7L, 8L), , j >= 9L),
             bor3(jnj %in% c(4L, -3L, 2L, 7L, 8L), , j >= 9L))
expect_equal(or3s(jnk %in% c(4L, -3L, 2L, 7L, 8L), e < 0L, ),
             bor3(jnk %in% c(4L, -3L, 2L, 7L, 8L), e < 0L, ))
expect_equal(or3s(jnl %in% c(4L, -3L, 2L, 7L, 8L), j < 1L, ),
             bor3(jnl %in% c(4L, -3L, 2L, 7L, 8L), j < 1L, ))
expect_equal(or3s(jnm %in% c(4L, -3L, 2L, 7L, 8L), i < 9L, ),
             bor3(jnm %in% c(4L, -3L, 2L, 7L, 8L), i < 9L, ))
expect_equal(or3s(jnn %in% c(4L, -3L, 2L, 7L, 8L), p < 0L, q >= 0L),
             bor3(jnn %in% c(4L, -3L, 2L, 7L, 8L), p < 0L, q >= 0L))
expect_equal(or3s(jno %in% c(4L, -3L, 2L, 7L, 8L), s < 1L, n >= 1L),
             bor3(jno %in% c(4L, -3L, 2L, 7L, 8L), s < 1L, n >= 1L))
expect_equal(or3s(jnp %in% c(4L, -3L, 2L, 7L, 8L), d < 9L, n >= 9L),
             bor3(jnp %in% c(4L, -3L, 2L, 7L, 8L), d < 9L, n >= 9L))
expect_equal(or3s(jnq %in% 1:4, , ),
             bor3(jnq %in% 1:4, , ))
expect_equal(or3s(jnr %in% 1:4, , ),
             bor3(jnr %in% 1:4, , ))
expect_equal(or3s(jns %in% 1:4, , ),
             bor3(jns %in% 1:4, , ))
expect_equal(or3s(jnt %in% 1:4, , v >= 0L),
             bor3(jnt %in% 1:4, , v >= 0L))
expect_equal(or3s(jnu %in% 1:4, , i >= 1L),
             bor3(jnu %in% 1:4, , i >= 1L))
expect_equal(or3s(jnv %in% 1:4, , s >= 9L),
             bor3(jnv %in% 1:4, , s >= 9L))
expect_equal(or3s(jnw %in% 1:4, c < 0L, ),
             bor3(jnw %in% 1:4, c < 0L, ))
expect_equal(or3s(jnx %in% 1:4, s < 1L, ),
             bor3(jnx %in% 1:4, s < 1L, ))
expect_equal(or3s(jny %in% 1:4, x < 9L, ),
             bor3(jny %in% 1:4, x < 9L, ))
expect_equal(or3s(jnz %in% 1:4, w < 0L, o >= 0L),
             bor3(jnz %in% 1:4, w < 0L, o >= 0L))
expect_equal(or3s(joa %in% 1:4, g < 1L, r >= 1L),
             bor3(joa %in% 1:4, g < 1L, r >= 1L))
expect_equal(or3s(job %in% 1:4, o < 9L, y >= 9L),
             bor3(job %in% 1:4, o < 9L, y >= 9L))
expect_equal(or3s(joc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(joc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jod %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jod %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(joe %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(joe %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jof %in% c(4L, -3L, 2L, 7L, 8L), , logi_k),
             bor3(jof %in% c(4L, -3L, 2L, 7L, 8L), , logi_k))
expect_equal(or3s(jog %in% c(4L, -3L, 2L, 7L, 8L), , logi_s),
             bor3(jog %in% c(4L, -3L, 2L, 7L, 8L), , logi_s))
expect_equal(or3s(joh %in% c(4L, -3L, 2L, 7L, 8L), , logi_z),
             bor3(joh %in% c(4L, -3L, 2L, 7L, 8L), , logi_z))
expect_equal(or3s(joi %in% c(4L, -3L, 2L, 7L, 8L), a <= 0L, ),
             bor3(joi %in% c(4L, -3L, 2L, 7L, 8L), a <= 0L, ))
expect_equal(or3s(joj %in% c(4L, -3L, 2L, 7L, 8L), s <= 1L, ),
             bor3(joj %in% c(4L, -3L, 2L, 7L, 8L), s <= 1L, ))
expect_equal(or3s(jok %in% c(4L, -3L, 2L, 7L, 8L), y <= 9L, ),
             bor3(jok %in% c(4L, -3L, 2L, 7L, 8L), y <= 9L, ))
expect_equal(or3s(jol %in% c(4L, -3L, 2L, 7L, 8L), f <= 0L, logi_p),
             bor3(jol %in% c(4L, -3L, 2L, 7L, 8L), f <= 0L, logi_p))
expect_equal(or3s(jom %in% c(4L, -3L, 2L, 7L, 8L), f <= 1L, logi_n),
             bor3(jom %in% c(4L, -3L, 2L, 7L, 8L), f <= 1L, logi_n))
expect_equal(or3s(jon %in% c(4L, -3L, 2L, 7L, 8L), x <= 9L, logi_y),
             bor3(jon %in% c(4L, -3L, 2L, 7L, 8L), x <= 9L, logi_y))
expect_equal(or3s(joo %in% 1:4, , ),
             bor3(joo %in% 1:4, , ))
expect_equal(or3s(jop %in% 1:4, , ),
             bor3(jop %in% 1:4, , ))
expect_equal(or3s(joq %in% 1:4, , ),
             bor3(joq %in% 1:4, , ))
expect_equal(or3s(jor %in% 1:4, , logi_s),
             bor3(jor %in% 1:4, , logi_s))
expect_equal(or3s(jos %in% 1:4, , logi_e),
             bor3(jos %in% 1:4, , logi_e))
expect_equal(or3s(jot %in% 1:4, , logi_p),
             bor3(jot %in% 1:4, , logi_p))
expect_equal(or3s(jou %in% 1:4, s <= 0L, ),
             bor3(jou %in% 1:4, s <= 0L, ))
expect_equal(or3s(jov %in% 1:4, l <= 1L, ),
             bor3(jov %in% 1:4, l <= 1L, ))
expect_equal(or3s(jow %in% 1:4, h <= 9L, ),
             bor3(jow %in% 1:4, h <= 9L, ))
expect_equal(or3s(jox %in% 1:4, x <= 0L, logi_s),
             bor3(jox %in% 1:4, x <= 0L, logi_s))
expect_equal(or3s(joy %in% 1:4, h <= 1L, logi_u),
             bor3(joy %in% 1:4, h <= 1L, logi_u))
expect_equal(or3s(joz %in% 1:4, n <= 9L, logi_z),
             bor3(joz %in% 1:4, n <= 9L, logi_z))
expect_equal(or3s(jpa %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jpa %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jpb %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jpb %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jpc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jpc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jpd %in% c(4L, -3L, 2L, 7L, 8L), , !logi_t),
             bor3(jpd %in% c(4L, -3L, 2L, 7L, 8L), , !logi_t))
expect_equal(or3s(jpe %in% c(4L, -3L, 2L, 7L, 8L), , !logi_a),
             bor3(jpe %in% c(4L, -3L, 2L, 7L, 8L), , !logi_a))
expect_equal(or3s(jpf %in% c(4L, -3L, 2L, 7L, 8L), , !logi_h),
             bor3(jpf %in% c(4L, -3L, 2L, 7L, 8L), , !logi_h))
expect_equal(or3s(jpg %in% c(4L, -3L, 2L, 7L, 8L), w <= 0L, ),
             bor3(jpg %in% c(4L, -3L, 2L, 7L, 8L), w <= 0L, ))
expect_equal(or3s(jph %in% c(4L, -3L, 2L, 7L, 8L), s <= 1L, ),
             bor3(jph %in% c(4L, -3L, 2L, 7L, 8L), s <= 1L, ))
expect_equal(or3s(jpi %in% c(4L, -3L, 2L, 7L, 8L), w <= 9L, ),
             bor3(jpi %in% c(4L, -3L, 2L, 7L, 8L), w <= 9L, ))
expect_equal(or3s(jpj %in% c(4L, -3L, 2L, 7L, 8L), j <= 0L, !logi_q),
             bor3(jpj %in% c(4L, -3L, 2L, 7L, 8L), j <= 0L, !logi_q))
expect_equal(or3s(jpk %in% c(4L, -3L, 2L, 7L, 8L), v <= 1L, !logi_x),
             bor3(jpk %in% c(4L, -3L, 2L, 7L, 8L), v <= 1L, !logi_x))
expect_equal(or3s(jpl %in% c(4L, -3L, 2L, 7L, 8L), a <= 9L, !logi_h),
             bor3(jpl %in% c(4L, -3L, 2L, 7L, 8L), a <= 9L, !logi_h))
expect_equal(or3s(jpm %in% 1:4, , ),
             bor3(jpm %in% 1:4, , ))
expect_equal(or3s(jpn %in% 1:4, , ),
             bor3(jpn %in% 1:4, , ))
expect_equal(or3s(jpo %in% 1:4, , ),
             bor3(jpo %in% 1:4, , ))
expect_equal(or3s(jpp %in% 1:4, , !logi_s),
             bor3(jpp %in% 1:4, , !logi_s))
expect_equal(or3s(jpq %in% 1:4, , !logi_z),
             bor3(jpq %in% 1:4, , !logi_z))
expect_equal(or3s(jpr %in% 1:4, , !logi_f),
             bor3(jpr %in% 1:4, , !logi_f))
expect_equal(or3s(jps %in% 1:4, z <= 0L, ),
             bor3(jps %in% 1:4, z <= 0L, ))
expect_equal(or3s(jpt %in% 1:4, k <= 1L, ),
             bor3(jpt %in% 1:4, k <= 1L, ))
expect_equal(or3s(jpu %in% 1:4, d <= 9L, ),
             bor3(jpu %in% 1:4, d <= 9L, ))
expect_equal(or3s(jpv %in% 1:4, i <= 0L, !logi_a),
             bor3(jpv %in% 1:4, i <= 0L, !logi_a))
expect_equal(or3s(jpw %in% 1:4, d <= 1L, !logi_e),
             bor3(jpw %in% 1:4, d <= 1L, !logi_e))
expect_equal(or3s(jpx %in% 1:4, b <= 9L, !logi_u),
             bor3(jpx %in% 1:4, b <= 9L, !logi_u))
expect_equal(or3s(jpy %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jpy %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jpz %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jpz %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jqa %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jqa %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jqb %in% c(4L, -3L, 2L, 7L, 8L), , o != 0L),
             bor3(jqb %in% c(4L, -3L, 2L, 7L, 8L), , o != 0L))
expect_equal(or3s(jqc %in% c(4L, -3L, 2L, 7L, 8L), , o != 1L),
             bor3(jqc %in% c(4L, -3L, 2L, 7L, 8L), , o != 1L))
expect_equal(or3s(jqd %in% c(4L, -3L, 2L, 7L, 8L), , v != 9L),
             bor3(jqd %in% c(4L, -3L, 2L, 7L, 8L), , v != 9L))
expect_equal(or3s(jqe %in% c(4L, -3L, 2L, 7L, 8L), q <= 0L, ),
             bor3(jqe %in% c(4L, -3L, 2L, 7L, 8L), q <= 0L, ))
expect_equal(or3s(jqf %in% c(4L, -3L, 2L, 7L, 8L), j <= 1L, ),
             bor3(jqf %in% c(4L, -3L, 2L, 7L, 8L), j <= 1L, ))
expect_equal(or3s(jqg %in% c(4L, -3L, 2L, 7L, 8L), f <= 9L, ),
             bor3(jqg %in% c(4L, -3L, 2L, 7L, 8L), f <= 9L, ))
expect_equal(or3s(jqh %in% c(4L, -3L, 2L, 7L, 8L), m <= 0L, s != 0L),
             bor3(jqh %in% c(4L, -3L, 2L, 7L, 8L), m <= 0L, s != 0L))
expect_equal(or3s(jqi %in% c(4L, -3L, 2L, 7L, 8L), f <= 1L, f != 1L),
             bor3(jqi %in% c(4L, -3L, 2L, 7L, 8L), f <= 1L, f != 1L))
expect_equal(or3s(jqj %in% c(4L, -3L, 2L, 7L, 8L), v <= 9L, u != 9L),
             bor3(jqj %in% c(4L, -3L, 2L, 7L, 8L), v <= 9L, u != 9L))
expect_equal(or3s(jqk %in% 1:4, , ),
             bor3(jqk %in% 1:4, , ))
expect_equal(or3s(jql %in% 1:4, , ),
             bor3(jql %in% 1:4, , ))
expect_equal(or3s(jqm %in% 1:4, , ),
             bor3(jqm %in% 1:4, , ))
expect_equal(or3s(jqn %in% 1:4, , n != 0L),
             bor3(jqn %in% 1:4, , n != 0L))
expect_equal(or3s(jqo %in% 1:4, , v != 1L),
             bor3(jqo %in% 1:4, , v != 1L))
expect_equal(or3s(jqp %in% 1:4, , y != 9L),
             bor3(jqp %in% 1:4, , y != 9L))
expect_equal(or3s(jqq %in% 1:4, b <= 0L, ),
             bor3(jqq %in% 1:4, b <= 0L, ))
expect_equal(or3s(jqr %in% 1:4, b <= 1L, ),
             bor3(jqr %in% 1:4, b <= 1L, ))
expect_equal(or3s(jqs %in% 1:4, d <= 9L, ),
             bor3(jqs %in% 1:4, d <= 9L, ))
expect_equal(or3s(jqt %in% 1:4, p <= 0L, u != 0L),
             bor3(jqt %in% 1:4, p <= 0L, u != 0L))
expect_equal(or3s(jqu %in% 1:4, y <= 1L, k != 1L),
             bor3(jqu %in% 1:4, y <= 1L, k != 1L))
expect_equal(or3s(jqv %in% 1:4, m <= 9L, k != 9L),
             bor3(jqv %in% 1:4, m <= 9L, k != 9L))
expect_equal(or3s(jqw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jqw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jqx %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jqx %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jqy %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jqy %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jqz %in% c(4L, -3L, 2L, 7L, 8L), , q %between% c(-1L, 1L)),
             bor3(jqz %in% c(4L, -3L, 2L, 7L, 8L), , q %between% c(-1L, 1L)))
expect_equal(or3s(jra %in% c(4L, -3L, 2L, 7L, 8L), , v %between% c(-1L, 1L)),
             bor3(jra %in% c(4L, -3L, 2L, 7L, 8L), , v %between% c(-1L, 1L)))
expect_equal(or3s(jrb %in% c(4L, -3L, 2L, 7L, 8L), , m %between% c(-1L, 1L)),
             bor3(jrb %in% c(4L, -3L, 2L, 7L, 8L), , m %between% c(-1L, 1L)))
expect_equal(or3s(jrc %in% c(4L, -3L, 2L, 7L, 8L), k <= 0L, ),
             bor3(jrc %in% c(4L, -3L, 2L, 7L, 8L), k <= 0L, ))
expect_equal(or3s(jrd %in% c(4L, -3L, 2L, 7L, 8L), v <= 1L, ),
             bor3(jrd %in% c(4L, -3L, 2L, 7L, 8L), v <= 1L, ))
expect_equal(or3s(jre %in% c(4L, -3L, 2L, 7L, 8L), v <= 9L, ),
             bor3(jre %in% c(4L, -3L, 2L, 7L, 8L), v <= 9L, ))
expect_equal(or3s(jrf %in% c(4L, -3L, 2L, 7L, 8L), e <= 0L, q %between% c(-1L, 1L)),
             bor3(jrf %in% c(4L, -3L, 2L, 7L, 8L), e <= 0L, q %between% c(-1L, 1L)))
expect_equal(or3s(jrg %in% c(4L, -3L, 2L, 7L, 8L), g <= 1L, y %between% c(-1L, 1L)),
             bor3(jrg %in% c(4L, -3L, 2L, 7L, 8L), g <= 1L, y %between% c(-1L, 1L)))
expect_equal(or3s(jrh %in% c(4L, -3L, 2L, 7L, 8L), s <= 9L, z %between% c(-1L, 1L)),
             bor3(jrh %in% c(4L, -3L, 2L, 7L, 8L), s <= 9L, z %between% c(-1L, 1L)))
expect_equal(or3s(jri %in% 1:4, , ),
             bor3(jri %in% 1:4, , ))
expect_equal(or3s(jrj %in% 1:4, , ),
             bor3(jrj %in% 1:4, , ))
expect_equal(or3s(jrk %in% 1:4, , ),
             bor3(jrk %in% 1:4, , ))
expect_equal(or3s(jrl %in% 1:4, , s %between% c(-1L, 1L)),
             bor3(jrl %in% 1:4, , s %between% c(-1L, 1L)))
expect_equal(or3s(jrm %in% 1:4, , e %between% c(-1L, 1L)),
             bor3(jrm %in% 1:4, , e %between% c(-1L, 1L)))
expect_equal(or3s(jrn %in% 1:4, , i %between% c(-1L, 1L)),
             bor3(jrn %in% 1:4, , i %between% c(-1L, 1L)))
expect_equal(or3s(jro %in% 1:4, h <= 0L, ),
             bor3(jro %in% 1:4, h <= 0L, ))
expect_equal(or3s(jrp %in% 1:4, s <= 1L, ),
             bor3(jrp %in% 1:4, s <= 1L, ))
expect_equal(or3s(jrq %in% 1:4, l <= 9L, ),
             bor3(jrq %in% 1:4, l <= 9L, ))
expect_equal(or3s(jrr %in% 1:4, h <= 0L, f %between% c(-1L, 1L)),
             bor3(jrr %in% 1:4, h <= 0L, f %between% c(-1L, 1L)))
expect_equal(or3s(jrs %in% 1:4, h <= 1L, l %between% c(-1L, 1L)),
             bor3(jrs %in% 1:4, h <= 1L, l %between% c(-1L, 1L)))
expect_equal(or3s(jrt %in% 1:4, r <= 9L, g %between% c(-1L, 1L)),
             bor3(jrt %in% 1:4, r <= 9L, g %between% c(-1L, 1L)))
expect_equal(or3s(jru %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jru %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jrv %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jrv %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jrw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jrw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jrx %in% c(4L, -3L, 2L, 7L, 8L), , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jrx %in% c(4L, -3L, 2L, 7L, 8L), , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jry %in% c(4L, -3L, 2L, 7L, 8L), , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jry %in% c(4L, -3L, 2L, 7L, 8L), , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jrz %in% c(4L, -3L, 2L, 7L, 8L), , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jrz %in% c(4L, -3L, 2L, 7L, 8L), , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jsa %in% c(4L, -3L, 2L, 7L, 8L), e <= 0L, ),
             bor3(jsa %in% c(4L, -3L, 2L, 7L, 8L), e <= 0L, ))
expect_equal(or3s(jsb %in% c(4L, -3L, 2L, 7L, 8L), y <= 1L, ),
             bor3(jsb %in% c(4L, -3L, 2L, 7L, 8L), y <= 1L, ))
expect_equal(or3s(jsc %in% c(4L, -3L, 2L, 7L, 8L), c <= 9L, ),
             bor3(jsc %in% c(4L, -3L, 2L, 7L, 8L), c <= 9L, ))
expect_equal(or3s(jsd %in% c(4L, -3L, 2L, 7L, 8L), j <= 0L, j %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jsd %in% c(4L, -3L, 2L, 7L, 8L), j <= 0L, j %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jse %in% c(4L, -3L, 2L, 7L, 8L), b <= 1L, s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jse %in% c(4L, -3L, 2L, 7L, 8L), b <= 1L, s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jsf %in% c(4L, -3L, 2L, 7L, 8L), w <= 9L, v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(jsf %in% c(4L, -3L, 2L, 7L, 8L), w <= 9L, v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(jsg %in% 1:4, , ),
             bor3(jsg %in% 1:4, , ))
expect_equal(or3s(jsh %in% 1:4, , ),
             bor3(jsh %in% 1:4, , ))
expect_equal(or3s(jsi %in% 1:4, , ),
             bor3(jsi %in% 1:4, , ))
expect_equal(or3s(jsj %in% 1:4, , b %in% 1:4),
             bor3(jsj %in% 1:4, , b %in% 1:4))
expect_equal(or3s(jsk %in% 1:4, , x %in% 1:4),
             bor3(jsk %in% 1:4, , x %in% 1:4))
expect_equal(or3s(jsl %in% 1:4, , x %in% 1:4),
             bor3(jsl %in% 1:4, , x %in% 1:4))
expect_equal(or3s(jsm %in% 1:4, p <= 0L, ),
             bor3(jsm %in% 1:4, p <= 0L, ))
expect_equal(or3s(jsn %in% 1:4, b <= 1L, ),
             bor3(jsn %in% 1:4, b <= 1L, ))
expect_equal(or3s(jso %in% 1:4, g <= 9L, ),
             bor3(jso %in% 1:4, g <= 9L, ))
expect_equal(or3s(jsp %in% 1:4, w <= 0L, k %in% 1:4),
             bor3(jsp %in% 1:4, w <= 0L, k %in% 1:4))
expect_equal(or3s(jsq %in% 1:4, q <= 1L, k %in% 1:4),
             bor3(jsq %in% 1:4, q <= 1L, k %in% 1:4))
expect_equal(or3s(jsr %in% 1:4, f <= 9L, b %in% 1:4),
             bor3(jsr %in% 1:4, f <= 9L, b %in% 1:4))
expect_equal(or3s(jss %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jss %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jst %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jst %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jsu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jsu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jsv %in% c(4L, -3L, 2L, 7L, 8L), , g < 0L),
             bor3(jsv %in% c(4L, -3L, 2L, 7L, 8L), , g < 0L))
expect_equal(or3s(jsw %in% c(4L, -3L, 2L, 7L, 8L), , v < 1L),
             bor3(jsw %in% c(4L, -3L, 2L, 7L, 8L), , v < 1L))
expect_equal(or3s(jsx %in% c(4L, -3L, 2L, 7L, 8L), , h < 9L),
             bor3(jsx %in% c(4L, -3L, 2L, 7L, 8L), , h < 9L))
expect_equal(or3s(jsy %in% c(4L, -3L, 2L, 7L, 8L), p <= 0L, ),
             bor3(jsy %in% c(4L, -3L, 2L, 7L, 8L), p <= 0L, ))
expect_equal(or3s(jsz %in% c(4L, -3L, 2L, 7L, 8L), d <= 1L, ),
             bor3(jsz %in% c(4L, -3L, 2L, 7L, 8L), d <= 1L, ))
expect_equal(or3s(jta %in% c(4L, -3L, 2L, 7L, 8L), n <= 9L, ),
             bor3(jta %in% c(4L, -3L, 2L, 7L, 8L), n <= 9L, ))
expect_equal(or3s(jtb %in% c(4L, -3L, 2L, 7L, 8L), r <= 0L, u < 0L),
             bor3(jtb %in% c(4L, -3L, 2L, 7L, 8L), r <= 0L, u < 0L))
expect_equal(or3s(jtc %in% c(4L, -3L, 2L, 7L, 8L), x <= 1L, w < 1L),
             bor3(jtc %in% c(4L, -3L, 2L, 7L, 8L), x <= 1L, w < 1L))
expect_equal(or3s(jtd %in% c(4L, -3L, 2L, 7L, 8L), c <= 9L, m < 9L),
             bor3(jtd %in% c(4L, -3L, 2L, 7L, 8L), c <= 9L, m < 9L))
expect_equal(or3s(jte %in% 1:4, , ),
             bor3(jte %in% 1:4, , ))
expect_equal(or3s(jtf %in% 1:4, , ),
             bor3(jtf %in% 1:4, , ))
expect_equal(or3s(jtg %in% 1:4, , ),
             bor3(jtg %in% 1:4, , ))
expect_equal(or3s(jth %in% 1:4, , q < 0L),
             bor3(jth %in% 1:4, , q < 0L))
expect_equal(or3s(jti %in% 1:4, , t < 1L),
             bor3(jti %in% 1:4, , t < 1L))
expect_equal(or3s(jtj %in% 1:4, , m < 9L),
             bor3(jtj %in% 1:4, , m < 9L))
expect_equal(or3s(jtk %in% 1:4, s <= 0L, ),
             bor3(jtk %in% 1:4, s <= 0L, ))
expect_equal(or3s(jtl %in% 1:4, l <= 1L, ),
             bor3(jtl %in% 1:4, l <= 1L, ))
expect_equal(or3s(jtm %in% 1:4, v <= 9L, ),
             bor3(jtm %in% 1:4, v <= 9L, ))
expect_equal(or3s(jtn %in% 1:4, i <= 0L, e < 0L),
             bor3(jtn %in% 1:4, i <= 0L, e < 0L))
expect_equal(or3s(jto %in% 1:4, j <= 1L, y < 1L),
             bor3(jto %in% 1:4, j <= 1L, y < 1L))
expect_equal(or3s(jtp %in% 1:4, o <= 9L, x < 9L),
             bor3(jtp %in% 1:4, o <= 9L, x < 9L))
expect_equal(or3s(jtq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jtq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jtr %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jtr %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jts %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jts %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jtt %in% c(4L, -3L, 2L, 7L, 8L), , t <= 0L),
             bor3(jtt %in% c(4L, -3L, 2L, 7L, 8L), , t <= 0L))
expect_equal(or3s(jtu %in% c(4L, -3L, 2L, 7L, 8L), , y <= 1L),
             bor3(jtu %in% c(4L, -3L, 2L, 7L, 8L), , y <= 1L))
expect_equal(or3s(jtv %in% c(4L, -3L, 2L, 7L, 8L), , q <= 9L),
             bor3(jtv %in% c(4L, -3L, 2L, 7L, 8L), , q <= 9L))
expect_equal(or3s(jtw %in% c(4L, -3L, 2L, 7L, 8L), z <= 0L, ),
             bor3(jtw %in% c(4L, -3L, 2L, 7L, 8L), z <= 0L, ))
expect_equal(or3s(jtx %in% c(4L, -3L, 2L, 7L, 8L), p <= 1L, ),
             bor3(jtx %in% c(4L, -3L, 2L, 7L, 8L), p <= 1L, ))
expect_equal(or3s(jty %in% c(4L, -3L, 2L, 7L, 8L), l <= 9L, ),
             bor3(jty %in% c(4L, -3L, 2L, 7L, 8L), l <= 9L, ))
expect_equal(or3s(jtz %in% c(4L, -3L, 2L, 7L, 8L), d <= 0L, y <= 0L),
             bor3(jtz %in% c(4L, -3L, 2L, 7L, 8L), d <= 0L, y <= 0L))
expect_equal(or3s(jua %in% c(4L, -3L, 2L, 7L, 8L), s <= 1L, j <= 1L),
             bor3(jua %in% c(4L, -3L, 2L, 7L, 8L), s <= 1L, j <= 1L))
expect_equal(or3s(jub %in% c(4L, -3L, 2L, 7L, 8L), j <= 9L, x <= 9L),
             bor3(jub %in% c(4L, -3L, 2L, 7L, 8L), j <= 9L, x <= 9L))
expect_equal(or3s(juc %in% 1:4, , ),
             bor3(juc %in% 1:4, , ))
expect_equal(or3s(jud %in% 1:4, , ),
             bor3(jud %in% 1:4, , ))
expect_equal(or3s(jue %in% 1:4, , ),
             bor3(jue %in% 1:4, , ))
expect_equal(or3s(juf %in% 1:4, , x <= 0L),
             bor3(juf %in% 1:4, , x <= 0L))
expect_equal(or3s(jug %in% 1:4, , n <= 1L),
             bor3(jug %in% 1:4, , n <= 1L))
expect_equal(or3s(juh %in% 1:4, , v <= 9L),
             bor3(juh %in% 1:4, , v <= 9L))
expect_equal(or3s(jui %in% 1:4, v <= 0L, ),
             bor3(jui %in% 1:4, v <= 0L, ))
expect_equal(or3s(juj %in% 1:4, o <= 1L, ),
             bor3(juj %in% 1:4, o <= 1L, ))
expect_equal(or3s(juk %in% 1:4, b <= 9L, ),
             bor3(juk %in% 1:4, b <= 9L, ))
expect_equal(or3s(jul %in% 1:4, j <= 0L, h <= 0L),
             bor3(jul %in% 1:4, j <= 0L, h <= 0L))
expect_equal(or3s(jum %in% 1:4, a <= 1L, k <= 1L),
             bor3(jum %in% 1:4, a <= 1L, k <= 1L))
expect_equal(or3s(jun %in% 1:4, x <= 9L, u <= 9L),
             bor3(jun %in% 1:4, x <= 9L, u <= 9L))
expect_equal(or3s(juo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(juo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jup %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jup %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(juq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(juq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jur %in% c(4L, -3L, 2L, 7L, 8L), , u == 0L),
             bor3(jur %in% c(4L, -3L, 2L, 7L, 8L), , u == 0L))
expect_equal(or3s(jus %in% c(4L, -3L, 2L, 7L, 8L), , k == 1L),
             bor3(jus %in% c(4L, -3L, 2L, 7L, 8L), , k == 1L))
expect_equal(or3s(jut %in% c(4L, -3L, 2L, 7L, 8L), , r == 9L),
             bor3(jut %in% c(4L, -3L, 2L, 7L, 8L), , r == 9L))
expect_equal(or3s(juu %in% c(4L, -3L, 2L, 7L, 8L), b <= 0L, ),
             bor3(juu %in% c(4L, -3L, 2L, 7L, 8L), b <= 0L, ))
expect_equal(or3s(juv %in% c(4L, -3L, 2L, 7L, 8L), q <= 1L, ),
             bor3(juv %in% c(4L, -3L, 2L, 7L, 8L), q <= 1L, ))
expect_equal(or3s(juw %in% c(4L, -3L, 2L, 7L, 8L), s <= 9L, ),
             bor3(juw %in% c(4L, -3L, 2L, 7L, 8L), s <= 9L, ))
expect_equal(or3s(jux %in% c(4L, -3L, 2L, 7L, 8L), i <= 0L, t == 0L),
             bor3(jux %in% c(4L, -3L, 2L, 7L, 8L), i <= 0L, t == 0L))
expect_equal(or3s(juy %in% c(4L, -3L, 2L, 7L, 8L), v <= 1L, p == 1L),
             bor3(juy %in% c(4L, -3L, 2L, 7L, 8L), v <= 1L, p == 1L))
expect_equal(or3s(juz %in% c(4L, -3L, 2L, 7L, 8L), c <= 9L, u == 9L),
             bor3(juz %in% c(4L, -3L, 2L, 7L, 8L), c <= 9L, u == 9L))
expect_equal(or3s(jva %in% 1:4, , ),
             bor3(jva %in% 1:4, , ))
expect_equal(or3s(jvb %in% 1:4, , ),
             bor3(jvb %in% 1:4, , ))
expect_equal(or3s(jvc %in% 1:4, , ),
             bor3(jvc %in% 1:4, , ))
expect_equal(or3s(jvd %in% 1:4, , i == 0L),
             bor3(jvd %in% 1:4, , i == 0L))
expect_equal(or3s(jve %in% 1:4, , j == 1L),
             bor3(jve %in% 1:4, , j == 1L))
expect_equal(or3s(jvf %in% 1:4, , e == 9L),
             bor3(jvf %in% 1:4, , e == 9L))
expect_equal(or3s(jvg %in% 1:4, c <= 0L, ),
             bor3(jvg %in% 1:4, c <= 0L, ))
expect_equal(or3s(jvh %in% 1:4, x <= 1L, ),
             bor3(jvh %in% 1:4, x <= 1L, ))
expect_equal(or3s(jvi %in% 1:4, d <= 9L, ),
             bor3(jvi %in% 1:4, d <= 9L, ))
expect_equal(or3s(jvj %in% 1:4, i <= 0L, v == 0L),
             bor3(jvj %in% 1:4, i <= 0L, v == 0L))
expect_equal(or3s(jvk %in% 1:4, j <= 1L, j == 1L),
             bor3(jvk %in% 1:4, j <= 1L, j == 1L))
expect_equal(or3s(jvl %in% 1:4, f <= 9L, q == 9L),
             bor3(jvl %in% 1:4, f <= 9L, q == 9L))
expect_equal(or3s(jvm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jvm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jvn %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jvn %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jvo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jvo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jvp %in% c(4L, -3L, 2L, 7L, 8L), , p > 0L),
             bor3(jvp %in% c(4L, -3L, 2L, 7L, 8L), , p > 0L))
expect_equal(or3s(jvq %in% c(4L, -3L, 2L, 7L, 8L), , u > 1L),
             bor3(jvq %in% c(4L, -3L, 2L, 7L, 8L), , u > 1L))
expect_equal(or3s(jvr %in% c(4L, -3L, 2L, 7L, 8L), , x > 9L),
             bor3(jvr %in% c(4L, -3L, 2L, 7L, 8L), , x > 9L))
expect_equal(or3s(jvs %in% c(4L, -3L, 2L, 7L, 8L), k <= 0L, ),
             bor3(jvs %in% c(4L, -3L, 2L, 7L, 8L), k <= 0L, ))
expect_equal(or3s(jvt %in% c(4L, -3L, 2L, 7L, 8L), x <= 1L, ),
             bor3(jvt %in% c(4L, -3L, 2L, 7L, 8L), x <= 1L, ))
expect_equal(or3s(jvu %in% c(4L, -3L, 2L, 7L, 8L), z <= 9L, ),
             bor3(jvu %in% c(4L, -3L, 2L, 7L, 8L), z <= 9L, ))
expect_equal(or3s(jvv %in% c(4L, -3L, 2L, 7L, 8L), s <= 0L, j > 0L),
             bor3(jvv %in% c(4L, -3L, 2L, 7L, 8L), s <= 0L, j > 0L))
expect_equal(or3s(jvw %in% c(4L, -3L, 2L, 7L, 8L), u <= 1L, i > 1L),
             bor3(jvw %in% c(4L, -3L, 2L, 7L, 8L), u <= 1L, i > 1L))
expect_equal(or3s(jvx %in% c(4L, -3L, 2L, 7L, 8L), m <= 9L, p > 9L),
             bor3(jvx %in% c(4L, -3L, 2L, 7L, 8L), m <= 9L, p > 9L))
expect_equal(or3s(jvy %in% 1:4, , ),
             bor3(jvy %in% 1:4, , ))
expect_equal(or3s(jvz %in% 1:4, , ),
             bor3(jvz %in% 1:4, , ))
expect_equal(or3s(jwa %in% 1:4, , ),
             bor3(jwa %in% 1:4, , ))
expect_equal(or3s(jwb %in% 1:4, , y > 0L),
             bor3(jwb %in% 1:4, , y > 0L))
expect_equal(or3s(jwc %in% 1:4, , d > 1L),
             bor3(jwc %in% 1:4, , d > 1L))
expect_equal(or3s(jwd %in% 1:4, , d > 9L),
             bor3(jwd %in% 1:4, , d > 9L))
expect_equal(or3s(jwe %in% 1:4, h <= 0L, ),
             bor3(jwe %in% 1:4, h <= 0L, ))
expect_equal(or3s(jwf %in% 1:4, n <= 1L, ),
             bor3(jwf %in% 1:4, n <= 1L, ))
expect_equal(or3s(jwg %in% 1:4, r <= 9L, ),
             bor3(jwg %in% 1:4, r <= 9L, ))
expect_equal(or3s(jwh %in% 1:4, l <= 0L, r > 0L),
             bor3(jwh %in% 1:4, l <= 0L, r > 0L))
expect_equal(or3s(jwi %in% 1:4, i <= 1L, c > 1L),
             bor3(jwi %in% 1:4, i <= 1L, c > 1L))
expect_equal(or3s(jwj %in% 1:4, u <= 9L, k > 9L),
             bor3(jwj %in% 1:4, u <= 9L, k > 9L))
expect_equal(or3s(jwk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jwk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jwl %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jwl %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jwm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jwm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jwn %in% c(4L, -3L, 2L, 7L, 8L), , x >= 0L),
             bor3(jwn %in% c(4L, -3L, 2L, 7L, 8L), , x >= 0L))
expect_equal(or3s(jwo %in% c(4L, -3L, 2L, 7L, 8L), , y >= 1L),
             bor3(jwo %in% c(4L, -3L, 2L, 7L, 8L), , y >= 1L))
expect_equal(or3s(jwp %in% c(4L, -3L, 2L, 7L, 8L), , l >= 9L),
             bor3(jwp %in% c(4L, -3L, 2L, 7L, 8L), , l >= 9L))
expect_equal(or3s(jwq %in% c(4L, -3L, 2L, 7L, 8L), a <= 0L, ),
             bor3(jwq %in% c(4L, -3L, 2L, 7L, 8L), a <= 0L, ))
expect_equal(or3s(jwr %in% c(4L, -3L, 2L, 7L, 8L), n <= 1L, ),
             bor3(jwr %in% c(4L, -3L, 2L, 7L, 8L), n <= 1L, ))
expect_equal(or3s(jws %in% c(4L, -3L, 2L, 7L, 8L), t <= 9L, ),
             bor3(jws %in% c(4L, -3L, 2L, 7L, 8L), t <= 9L, ))
expect_equal(or3s(jwt %in% c(4L, -3L, 2L, 7L, 8L), r <= 0L, z >= 0L),
             bor3(jwt %in% c(4L, -3L, 2L, 7L, 8L), r <= 0L, z >= 0L))
expect_equal(or3s(jwu %in% c(4L, -3L, 2L, 7L, 8L), r <= 1L, d >= 1L),
             bor3(jwu %in% c(4L, -3L, 2L, 7L, 8L), r <= 1L, d >= 1L))
expect_equal(or3s(jwv %in% c(4L, -3L, 2L, 7L, 8L), t <= 9L, a >= 9L),
             bor3(jwv %in% c(4L, -3L, 2L, 7L, 8L), t <= 9L, a >= 9L))
expect_equal(or3s(jww %in% 1:4, , ),
             bor3(jww %in% 1:4, , ))
expect_equal(or3s(jwx %in% 1:4, , ),
             bor3(jwx %in% 1:4, , ))
expect_equal(or3s(jwy %in% 1:4, , ),
             bor3(jwy %in% 1:4, , ))
expect_equal(or3s(jwz %in% 1:4, , v >= 0L),
             bor3(jwz %in% 1:4, , v >= 0L))
expect_equal(or3s(jxa %in% 1:4, , u >= 1L),
             bor3(jxa %in% 1:4, , u >= 1L))
expect_equal(or3s(jxb %in% 1:4, , f >= 9L),
             bor3(jxb %in% 1:4, , f >= 9L))
expect_equal(or3s(jxc %in% 1:4, q <= 0L, ),
             bor3(jxc %in% 1:4, q <= 0L, ))
expect_equal(or3s(jxd %in% 1:4, w <= 1L, ),
             bor3(jxd %in% 1:4, w <= 1L, ))
expect_equal(or3s(jxe %in% 1:4, l <= 9L, ),
             bor3(jxe %in% 1:4, l <= 9L, ))
expect_equal(or3s(jxf %in% 1:4, u <= 0L, v >= 0L),
             bor3(jxf %in% 1:4, u <= 0L, v >= 0L))
expect_equal(or3s(jxg %in% 1:4, k <= 1L, o >= 1L),
             bor3(jxg %in% 1:4, k <= 1L, o >= 1L))
expect_equal(or3s(jxh %in% 1:4, z <= 9L, l >= 9L),
             bor3(jxh %in% 1:4, z <= 9L, l >= 9L))
expect_equal(or3s(jxi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jxi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jxj %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jxj %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jxk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jxk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jxl %in% c(4L, -3L, 2L, 7L, 8L), , logi_b),
             bor3(jxl %in% c(4L, -3L, 2L, 7L, 8L), , logi_b))
expect_equal(or3s(jxm %in% c(4L, -3L, 2L, 7L, 8L), , logi_q),
             bor3(jxm %in% c(4L, -3L, 2L, 7L, 8L), , logi_q))
expect_equal(or3s(jxn %in% c(4L, -3L, 2L, 7L, 8L), , logi_j),
             bor3(jxn %in% c(4L, -3L, 2L, 7L, 8L), , logi_j))
expect_equal(or3s(jxo %in% c(4L, -3L, 2L, 7L, 8L), v == 0L, ),
             bor3(jxo %in% c(4L, -3L, 2L, 7L, 8L), v == 0L, ))
expect_equal(or3s(jxp %in% c(4L, -3L, 2L, 7L, 8L), a == 1L, ),
             bor3(jxp %in% c(4L, -3L, 2L, 7L, 8L), a == 1L, ))
expect_equal(or3s(jxq %in% c(4L, -3L, 2L, 7L, 8L), g == 9L, ),
             bor3(jxq %in% c(4L, -3L, 2L, 7L, 8L), g == 9L, ))
expect_equal(or3s(jxr %in% c(4L, -3L, 2L, 7L, 8L), j == 0L, logi_q),
             bor3(jxr %in% c(4L, -3L, 2L, 7L, 8L), j == 0L, logi_q))
expect_equal(or3s(jxs %in% c(4L, -3L, 2L, 7L, 8L), u == 1L, logi_b),
             bor3(jxs %in% c(4L, -3L, 2L, 7L, 8L), u == 1L, logi_b))
expect_equal(or3s(jxt %in% c(4L, -3L, 2L, 7L, 8L), z == 9L, logi_p),
             bor3(jxt %in% c(4L, -3L, 2L, 7L, 8L), z == 9L, logi_p))
expect_equal(or3s(jxu %in% 1:4, , ),
             bor3(jxu %in% 1:4, , ))
expect_equal(or3s(jxv %in% 1:4, , ),
             bor3(jxv %in% 1:4, , ))
expect_equal(or3s(jxw %in% 1:4, , ),
             bor3(jxw %in% 1:4, , ))
expect_equal(or3s(jxx %in% 1:4, , logi_a),
             bor3(jxx %in% 1:4, , logi_a))
expect_equal(or3s(jxy %in% 1:4, , logi_v),
             bor3(jxy %in% 1:4, , logi_v))
expect_equal(or3s(jxz %in% 1:4, , logi_m),
             bor3(jxz %in% 1:4, , logi_m))
expect_equal(or3s(jya %in% 1:4, q == 0L, ),
             bor3(jya %in% 1:4, q == 0L, ))
expect_equal(or3s(jyb %in% 1:4, c == 1L, ),
             bor3(jyb %in% 1:4, c == 1L, ))
expect_equal(or3s(jyc %in% 1:4, c == 9L, ),
             bor3(jyc %in% 1:4, c == 9L, ))
expect_equal(or3s(jyd %in% 1:4, b == 0L, logi_p),
             bor3(jyd %in% 1:4, b == 0L, logi_p))
expect_equal(or3s(jye %in% 1:4, q == 1L, logi_k),
             bor3(jye %in% 1:4, q == 1L, logi_k))
expect_equal(or3s(jyf %in% 1:4, e == 9L, logi_u),
             bor3(jyf %in% 1:4, e == 9L, logi_u))
expect_equal(or3s(jyg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jyg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jyh %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jyh %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jyi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(jyi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(jyj %in% c(4L, -3L, 2L, 7L, 8L), , !logi_b),
             bor3(jyj %in% c(4L, -3L, 2L, 7L, 8L), , !logi_b))
expect_equal(or3s(jyk %in% c(4L, -3L, 2L, 7L, 8L), , !logi_c),
             bor3(jyk %in% c(4L, -3L, 2L, 7L, 8L), , !logi_c))
expect_equal(or3s(jyl %in% c(4L, -3L, 2L, 7L, 8L), , !logi_a),
             bor3(jyl %in% c(4L, -3L, 2L, 7L, 8L), , !logi_a))
expect_equal(or3s(jym %in% c(4L, -3L, 2L, 7L, 8L), e == 0L, ),
             bor3(jym %in% c(4L, -3L, 2L, 7L, 8L), e == 0L, ))
expect_equal(or3s(jyn %in% c(4L, -3L, 2L, 7L, 8L), x == 1L, ),
             bor3(jyn %in% c(4L, -3L, 2L, 7L, 8L), x == 1L, ))
expect_equal(or3s(jyo %in% c(4L, -3L, 2L, 7L, 8L), n == 9L, ),
             bor3(jyo %in% c(4L, -3L, 2L, 7L, 8L), n == 9L, ))
expect_equal(or3s(jyp %in% c(4L, -3L, 2L, 7L, 8L), e == 0L, !logi_c),
             bor3(jyp %in% c(4L, -3L, 2L, 7L, 8L), e == 0L, !logi_c))
expect_equal(or3s(jyq %in% c(4L, -3L, 2L, 7L, 8L), k == 1L, !logi_x),
             bor3(jyq %in% c(4L, -3L, 2L, 7L, 8L), k == 1L, !logi_x))
expect_equal(or3s(jyr %in% c(4L, -3L, 2L, 7L, 8L), n == 9L, !logi_v),
             bor3(jyr %in% c(4L, -3L, 2L, 7L, 8L), n == 9L, !logi_v))
expect_equal(or3s(jys %in% 1:4, , ),
             bor3(jys %in% 1:4, , ))
expect_equal(or3s(jyt %in% 1:4, , ),
             bor3(jyt %in% 1:4, , ))
expect_equal(or3s(jyu %in% 1:4, , ),
             bor3(jyu %in% 1:4, , ))
expect_equal(or3s(jyv %in% 1:4, , !logi_t),
             bor3(jyv %in% 1:4, , !logi_t))
expect_equal(or3s(jyw %in% 1:4, , !logi_b),
             bor3(jyw %in% 1:4, , !logi_b))
expect_equal(or3s(jyx %in% 1:4, , !logi_p),
             bor3(jyx %in% 1:4, , !logi_p))
expect_equal(or3s(jyy %in% 1:4, m == 0L, ),
             bor3(jyy %in% 1:4, m == 0L, ))
expect_equal(or3s(jyz %in% 1:4, f == 1L, ),
             bor3(jyz %in% 1:4, f == 1L, ))
expect_equal(or3s(ka %in% 1:4, o == 9L, ),
             bor3(ka %in% 1:4, o == 9L, ))
expect_equal(or3s(kb %in% 1:4, v == 0L, !logi_m),
             bor3(kb %in% 1:4, v == 0L, !logi_m))
expect_equal(or3s(kc %in% 1:4, a == 1L, !logi_d),
             bor3(kc %in% 1:4, a == 1L, !logi_d))
expect_equal(or3s(kd %in% 1:4, b == 9L, !logi_x),
             bor3(kd %in% 1:4, b == 9L, !logi_x))
expect_equal(or3s(ke %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ke %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kf %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kf %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kh %in% c(4L, -3L, 2L, 7L, 8L), , c != 0L),
             bor3(kh %in% c(4L, -3L, 2L, 7L, 8L), , c != 0L))
expect_equal(or3s(ki %in% c(4L, -3L, 2L, 7L, 8L), , k != 1L),
             bor3(ki %in% c(4L, -3L, 2L, 7L, 8L), , k != 1L))
expect_equal(or3s(kj %in% c(4L, -3L, 2L, 7L, 8L), , b != 9L),
             bor3(kj %in% c(4L, -3L, 2L, 7L, 8L), , b != 9L))
expect_equal(or3s(kk %in% c(4L, -3L, 2L, 7L, 8L), j == 0L, ),
             bor3(kk %in% c(4L, -3L, 2L, 7L, 8L), j == 0L, ))
expect_equal(or3s(kl %in% c(4L, -3L, 2L, 7L, 8L), z == 1L, ),
             bor3(kl %in% c(4L, -3L, 2L, 7L, 8L), z == 1L, ))
expect_equal(or3s(km %in% c(4L, -3L, 2L, 7L, 8L), q == 9L, ),
             bor3(km %in% c(4L, -3L, 2L, 7L, 8L), q == 9L, ))
expect_equal(or3s(kn %in% c(4L, -3L, 2L, 7L, 8L), y == 0L, m != 0L),
             bor3(kn %in% c(4L, -3L, 2L, 7L, 8L), y == 0L, m != 0L))
expect_equal(or3s(ko %in% c(4L, -3L, 2L, 7L, 8L), m == 1L, s != 1L),
             bor3(ko %in% c(4L, -3L, 2L, 7L, 8L), m == 1L, s != 1L))
expect_equal(or3s(kp %in% c(4L, -3L, 2L, 7L, 8L), b == 9L, o != 9L),
             bor3(kp %in% c(4L, -3L, 2L, 7L, 8L), b == 9L, o != 9L))
expect_equal(or3s(kq %in% 1:4, , ),
             bor3(kq %in% 1:4, , ))
expect_equal(or3s(kr %in% 1:4, , ),
             bor3(kr %in% 1:4, , ))
expect_equal(or3s(ks %in% 1:4, , ),
             bor3(ks %in% 1:4, , ))
expect_equal(or3s(kt %in% 1:4, , g != 0L),
             bor3(kt %in% 1:4, , g != 0L))
expect_equal(or3s(ku %in% 1:4, , c != 1L),
             bor3(ku %in% 1:4, , c != 1L))
expect_equal(or3s(kv %in% 1:4, , v != 9L),
             bor3(kv %in% 1:4, , v != 9L))
expect_equal(or3s(kw %in% 1:4, b == 0L, ),
             bor3(kw %in% 1:4, b == 0L, ))
expect_equal(or3s(kx %in% 1:4, f == 1L, ),
             bor3(kx %in% 1:4, f == 1L, ))
expect_equal(or3s(ky %in% 1:4, a == 9L, ),
             bor3(ky %in% 1:4, a == 9L, ))
expect_equal(or3s(kz %in% 1:4, n == 0L, y != 0L),
             bor3(kz %in% 1:4, n == 0L, y != 0L))
expect_equal(or3s(kaa %in% 1:4, c == 1L, h != 1L),
             bor3(kaa %in% 1:4, c == 1L, h != 1L))
expect_equal(or3s(kab %in% 1:4, z == 9L, y != 9L),
             bor3(kab %in% 1:4, z == 9L, y != 9L))
expect_equal(or3s(kac %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kac %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kad %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kad %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kae %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kae %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kaf %in% c(4L, -3L, 2L, 7L, 8L), , v %between% c(-1L, 1L)),
             bor3(kaf %in% c(4L, -3L, 2L, 7L, 8L), , v %between% c(-1L, 1L)))
expect_equal(or3s(kag %in% c(4L, -3L, 2L, 7L, 8L), , d %between% c(-1L, 1L)),
             bor3(kag %in% c(4L, -3L, 2L, 7L, 8L), , d %between% c(-1L, 1L)))
expect_equal(or3s(kah %in% c(4L, -3L, 2L, 7L, 8L), , r %between% c(-1L, 1L)),
             bor3(kah %in% c(4L, -3L, 2L, 7L, 8L), , r %between% c(-1L, 1L)))
expect_equal(or3s(kai %in% c(4L, -3L, 2L, 7L, 8L), n == 0L, ),
             bor3(kai %in% c(4L, -3L, 2L, 7L, 8L), n == 0L, ))
expect_equal(or3s(kaj %in% c(4L, -3L, 2L, 7L, 8L), e == 1L, ),
             bor3(kaj %in% c(4L, -3L, 2L, 7L, 8L), e == 1L, ))
expect_equal(or3s(kak %in% c(4L, -3L, 2L, 7L, 8L), j == 9L, ),
             bor3(kak %in% c(4L, -3L, 2L, 7L, 8L), j == 9L, ))
expect_equal(or3s(kal %in% c(4L, -3L, 2L, 7L, 8L), y == 0L, z %between% c(-1L, 1L)),
             bor3(kal %in% c(4L, -3L, 2L, 7L, 8L), y == 0L, z %between% c(-1L, 1L)))
expect_equal(or3s(kam %in% c(4L, -3L, 2L, 7L, 8L), b == 1L, n %between% c(-1L, 1L)),
             bor3(kam %in% c(4L, -3L, 2L, 7L, 8L), b == 1L, n %between% c(-1L, 1L)))
expect_equal(or3s(kan %in% c(4L, -3L, 2L, 7L, 8L), s == 9L, g %between% c(-1L, 1L)),
             bor3(kan %in% c(4L, -3L, 2L, 7L, 8L), s == 9L, g %between% c(-1L, 1L)))
expect_equal(or3s(kao %in% 1:4, , ),
             bor3(kao %in% 1:4, , ))
expect_equal(or3s(kap %in% 1:4, , ),
             bor3(kap %in% 1:4, , ))
expect_equal(or3s(kaq %in% 1:4, , ),
             bor3(kaq %in% 1:4, , ))
expect_equal(or3s(kar %in% 1:4, , n %between% c(-1L, 1L)),
             bor3(kar %in% 1:4, , n %between% c(-1L, 1L)))
expect_equal(or3s(kas %in% 1:4, , s %between% c(-1L, 1L)),
             bor3(kas %in% 1:4, , s %between% c(-1L, 1L)))
expect_equal(or3s(kat %in% 1:4, , s %between% c(-1L, 1L)),
             bor3(kat %in% 1:4, , s %between% c(-1L, 1L)))
expect_equal(or3s(kau %in% 1:4, m == 0L, ),
             bor3(kau %in% 1:4, m == 0L, ))
expect_equal(or3s(kav %in% 1:4, z == 1L, ),
             bor3(kav %in% 1:4, z == 1L, ))
expect_equal(or3s(kaw %in% 1:4, w == 9L, ),
             bor3(kaw %in% 1:4, w == 9L, ))
expect_equal(or3s(kax %in% 1:4, v == 0L, o %between% c(-1L, 1L)),
             bor3(kax %in% 1:4, v == 0L, o %between% c(-1L, 1L)))
expect_equal(or3s(kay %in% 1:4, r == 1L, u %between% c(-1L, 1L)),
             bor3(kay %in% 1:4, r == 1L, u %between% c(-1L, 1L)))
expect_equal(or3s(kaz %in% 1:4, z == 9L, r %between% c(-1L, 1L)),
             bor3(kaz %in% 1:4, z == 9L, r %between% c(-1L, 1L)))
expect_equal(or3s(kba %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kba %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kbb %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kbb %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kbc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kbc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kbd %in% c(4L, -3L, 2L, 7L, 8L), , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kbd %in% c(4L, -3L, 2L, 7L, 8L), , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kbe %in% c(4L, -3L, 2L, 7L, 8L), , q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kbe %in% c(4L, -3L, 2L, 7L, 8L), , q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kbf %in% c(4L, -3L, 2L, 7L, 8L), , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kbf %in% c(4L, -3L, 2L, 7L, 8L), , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kbg %in% c(4L, -3L, 2L, 7L, 8L), o == 0L, ),
             bor3(kbg %in% c(4L, -3L, 2L, 7L, 8L), o == 0L, ))
expect_equal(or3s(kbh %in% c(4L, -3L, 2L, 7L, 8L), o == 1L, ),
             bor3(kbh %in% c(4L, -3L, 2L, 7L, 8L), o == 1L, ))
expect_equal(or3s(kbi %in% c(4L, -3L, 2L, 7L, 8L), h == 9L, ),
             bor3(kbi %in% c(4L, -3L, 2L, 7L, 8L), h == 9L, ))
expect_equal(or3s(kbj %in% c(4L, -3L, 2L, 7L, 8L), p == 0L, b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kbj %in% c(4L, -3L, 2L, 7L, 8L), p == 0L, b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kbk %in% c(4L, -3L, 2L, 7L, 8L), e == 1L, y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kbk %in% c(4L, -3L, 2L, 7L, 8L), e == 1L, y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kbl %in% c(4L, -3L, 2L, 7L, 8L), o == 9L, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kbl %in% c(4L, -3L, 2L, 7L, 8L), o == 9L, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kbm %in% 1:4, , ),
             bor3(kbm %in% 1:4, , ))
expect_equal(or3s(kbn %in% 1:4, , ),
             bor3(kbn %in% 1:4, , ))
expect_equal(or3s(kbo %in% 1:4, , ),
             bor3(kbo %in% 1:4, , ))
expect_equal(or3s(kbp %in% 1:4, , k %in% 1:4),
             bor3(kbp %in% 1:4, , k %in% 1:4))
expect_equal(or3s(kbq %in% 1:4, , q %in% 1:4),
             bor3(kbq %in% 1:4, , q %in% 1:4))
expect_equal(or3s(kbr %in% 1:4, , h %in% 1:4),
             bor3(kbr %in% 1:4, , h %in% 1:4))
expect_equal(or3s(kbs %in% 1:4, z == 0L, ),
             bor3(kbs %in% 1:4, z == 0L, ))
expect_equal(or3s(kbt %in% 1:4, p == 1L, ),
             bor3(kbt %in% 1:4, p == 1L, ))
expect_equal(or3s(kbu %in% 1:4, z == 9L, ),
             bor3(kbu %in% 1:4, z == 9L, ))
expect_equal(or3s(kbv %in% 1:4, f == 0L, q %in% 1:4),
             bor3(kbv %in% 1:4, f == 0L, q %in% 1:4))
expect_equal(or3s(kbw %in% 1:4, z == 1L, k %in% 1:4),
             bor3(kbw %in% 1:4, z == 1L, k %in% 1:4))
expect_equal(or3s(kbx %in% 1:4, p == 9L, u %in% 1:4),
             bor3(kbx %in% 1:4, p == 9L, u %in% 1:4))
expect_equal(or3s(kby %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kby %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kbz %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kbz %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kca %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kca %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kcb %in% c(4L, -3L, 2L, 7L, 8L), , w < 0L),
             bor3(kcb %in% c(4L, -3L, 2L, 7L, 8L), , w < 0L))
expect_equal(or3s(kcc %in% c(4L, -3L, 2L, 7L, 8L), , m < 1L),
             bor3(kcc %in% c(4L, -3L, 2L, 7L, 8L), , m < 1L))
expect_equal(or3s(kcd %in% c(4L, -3L, 2L, 7L, 8L), , p < 9L),
             bor3(kcd %in% c(4L, -3L, 2L, 7L, 8L), , p < 9L))
expect_equal(or3s(kce %in% c(4L, -3L, 2L, 7L, 8L), d == 0L, ),
             bor3(kce %in% c(4L, -3L, 2L, 7L, 8L), d == 0L, ))
expect_equal(or3s(kcf %in% c(4L, -3L, 2L, 7L, 8L), m == 1L, ),
             bor3(kcf %in% c(4L, -3L, 2L, 7L, 8L), m == 1L, ))
expect_equal(or3s(kcg %in% c(4L, -3L, 2L, 7L, 8L), b == 9L, ),
             bor3(kcg %in% c(4L, -3L, 2L, 7L, 8L), b == 9L, ))
expect_equal(or3s(kch %in% c(4L, -3L, 2L, 7L, 8L), h == 0L, h < 0L),
             bor3(kch %in% c(4L, -3L, 2L, 7L, 8L), h == 0L, h < 0L))
expect_equal(or3s(kci %in% c(4L, -3L, 2L, 7L, 8L), k == 1L, o < 1L),
             bor3(kci %in% c(4L, -3L, 2L, 7L, 8L), k == 1L, o < 1L))
expect_equal(or3s(kcj %in% c(4L, -3L, 2L, 7L, 8L), n == 9L, y < 9L),
             bor3(kcj %in% c(4L, -3L, 2L, 7L, 8L), n == 9L, y < 9L))
expect_equal(or3s(kck %in% 1:4, , ),
             bor3(kck %in% 1:4, , ))
expect_equal(or3s(kcl %in% 1:4, , ),
             bor3(kcl %in% 1:4, , ))
expect_equal(or3s(kcm %in% 1:4, , ),
             bor3(kcm %in% 1:4, , ))
expect_equal(or3s(kcn %in% 1:4, , n < 0L),
             bor3(kcn %in% 1:4, , n < 0L))
expect_equal(or3s(kco %in% 1:4, , f < 1L),
             bor3(kco %in% 1:4, , f < 1L))
expect_equal(or3s(kcp %in% 1:4, , b < 9L),
             bor3(kcp %in% 1:4, , b < 9L))
expect_equal(or3s(kcq %in% 1:4, r == 0L, ),
             bor3(kcq %in% 1:4, r == 0L, ))
expect_equal(or3s(kcr %in% 1:4, v == 1L, ),
             bor3(kcr %in% 1:4, v == 1L, ))
expect_equal(or3s(kcs %in% 1:4, m == 9L, ),
             bor3(kcs %in% 1:4, m == 9L, ))
expect_equal(or3s(kct %in% 1:4, u == 0L, a < 0L),
             bor3(kct %in% 1:4, u == 0L, a < 0L))
expect_equal(or3s(kcu %in% 1:4, d == 1L, s < 1L),
             bor3(kcu %in% 1:4, d == 1L, s < 1L))
expect_equal(or3s(kcv %in% 1:4, w == 9L, s < 9L),
             bor3(kcv %in% 1:4, w == 9L, s < 9L))
expect_equal(or3s(kcw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kcw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kcx %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kcx %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kcy %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kcy %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kcz %in% c(4L, -3L, 2L, 7L, 8L), , w <= 0L),
             bor3(kcz %in% c(4L, -3L, 2L, 7L, 8L), , w <= 0L))
expect_equal(or3s(kda %in% c(4L, -3L, 2L, 7L, 8L), , d <= 1L),
             bor3(kda %in% c(4L, -3L, 2L, 7L, 8L), , d <= 1L))
expect_equal(or3s(kdb %in% c(4L, -3L, 2L, 7L, 8L), , b <= 9L),
             bor3(kdb %in% c(4L, -3L, 2L, 7L, 8L), , b <= 9L))
expect_equal(or3s(kdc %in% c(4L, -3L, 2L, 7L, 8L), x == 0L, ),
             bor3(kdc %in% c(4L, -3L, 2L, 7L, 8L), x == 0L, ))
expect_equal(or3s(kdd %in% c(4L, -3L, 2L, 7L, 8L), g == 1L, ),
             bor3(kdd %in% c(4L, -3L, 2L, 7L, 8L), g == 1L, ))
expect_equal(or3s(kde %in% c(4L, -3L, 2L, 7L, 8L), h == 9L, ),
             bor3(kde %in% c(4L, -3L, 2L, 7L, 8L), h == 9L, ))
expect_equal(or3s(kdf %in% c(4L, -3L, 2L, 7L, 8L), s == 0L, r <= 0L),
             bor3(kdf %in% c(4L, -3L, 2L, 7L, 8L), s == 0L, r <= 0L))
expect_equal(or3s(kdg %in% c(4L, -3L, 2L, 7L, 8L), h == 1L, i <= 1L),
             bor3(kdg %in% c(4L, -3L, 2L, 7L, 8L), h == 1L, i <= 1L))
expect_equal(or3s(kdh %in% c(4L, -3L, 2L, 7L, 8L), z == 9L, e <= 9L),
             bor3(kdh %in% c(4L, -3L, 2L, 7L, 8L), z == 9L, e <= 9L))
expect_equal(or3s(kdi %in% 1:4, , ),
             bor3(kdi %in% 1:4, , ))
expect_equal(or3s(kdj %in% 1:4, , ),
             bor3(kdj %in% 1:4, , ))
expect_equal(or3s(kdk %in% 1:4, , ),
             bor3(kdk %in% 1:4, , ))
expect_equal(or3s(kdl %in% 1:4, , r <= 0L),
             bor3(kdl %in% 1:4, , r <= 0L))
expect_equal(or3s(kdm %in% 1:4, , l <= 1L),
             bor3(kdm %in% 1:4, , l <= 1L))
expect_equal(or3s(kdn %in% 1:4, , m <= 9L),
             bor3(kdn %in% 1:4, , m <= 9L))
expect_equal(or3s(kdo %in% 1:4, z == 0L, ),
             bor3(kdo %in% 1:4, z == 0L, ))
expect_equal(or3s(kdp %in% 1:4, s == 1L, ),
             bor3(kdp %in% 1:4, s == 1L, ))
expect_equal(or3s(kdq %in% 1:4, r == 9L, ),
             bor3(kdq %in% 1:4, r == 9L, ))
expect_equal(or3s(kdr %in% 1:4, m == 0L, s <= 0L),
             bor3(kdr %in% 1:4, m == 0L, s <= 0L))
expect_equal(or3s(kds %in% 1:4, d == 1L, w <= 1L),
             bor3(kds %in% 1:4, d == 1L, w <= 1L))
expect_equal(or3s(kdt %in% 1:4, k == 9L, q <= 9L),
             bor3(kdt %in% 1:4, k == 9L, q <= 9L))
expect_equal(or3s(kdu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kdu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kdv %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kdv %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kdw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kdw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kdx %in% c(4L, -3L, 2L, 7L, 8L), , k == 0L),
             bor3(kdx %in% c(4L, -3L, 2L, 7L, 8L), , k == 0L))
expect_equal(or3s(kdy %in% c(4L, -3L, 2L, 7L, 8L), , q == 1L),
             bor3(kdy %in% c(4L, -3L, 2L, 7L, 8L), , q == 1L))
expect_equal(or3s(kdz %in% c(4L, -3L, 2L, 7L, 8L), , o == 9L),
             bor3(kdz %in% c(4L, -3L, 2L, 7L, 8L), , o == 9L))
expect_equal(or3s(kea %in% c(4L, -3L, 2L, 7L, 8L), c == 0L, ),
             bor3(kea %in% c(4L, -3L, 2L, 7L, 8L), c == 0L, ))
expect_equal(or3s(keb %in% c(4L, -3L, 2L, 7L, 8L), g == 1L, ),
             bor3(keb %in% c(4L, -3L, 2L, 7L, 8L), g == 1L, ))
expect_equal(or3s(kec %in% c(4L, -3L, 2L, 7L, 8L), q == 9L, ),
             bor3(kec %in% c(4L, -3L, 2L, 7L, 8L), q == 9L, ))
expect_equal(or3s(ked %in% c(4L, -3L, 2L, 7L, 8L), q == 0L, x == 0L),
             bor3(ked %in% c(4L, -3L, 2L, 7L, 8L), q == 0L, x == 0L))
expect_equal(or3s(kee %in% c(4L, -3L, 2L, 7L, 8L), j == 1L, j == 1L),
             bor3(kee %in% c(4L, -3L, 2L, 7L, 8L), j == 1L, j == 1L))
expect_equal(or3s(kef %in% c(4L, -3L, 2L, 7L, 8L), f == 9L, p == 9L),
             bor3(kef %in% c(4L, -3L, 2L, 7L, 8L), f == 9L, p == 9L))
expect_equal(or3s(keg %in% 1:4, , ),
             bor3(keg %in% 1:4, , ))
expect_equal(or3s(keh %in% 1:4, , ),
             bor3(keh %in% 1:4, , ))
expect_equal(or3s(kei %in% 1:4, , ),
             bor3(kei %in% 1:4, , ))
expect_equal(or3s(kej %in% 1:4, , w == 0L),
             bor3(kej %in% 1:4, , w == 0L))
expect_equal(or3s(kek %in% 1:4, , l == 1L),
             bor3(kek %in% 1:4, , l == 1L))
expect_equal(or3s(kel %in% 1:4, , l == 9L),
             bor3(kel %in% 1:4, , l == 9L))
expect_equal(or3s(kem %in% 1:4, o == 0L, ),
             bor3(kem %in% 1:4, o == 0L, ))
expect_equal(or3s(ken %in% 1:4, o == 1L, ),
             bor3(ken %in% 1:4, o == 1L, ))
expect_equal(or3s(keo %in% 1:4, y == 9L, ),
             bor3(keo %in% 1:4, y == 9L, ))
expect_equal(or3s(kep %in% 1:4, d == 0L, o == 0L),
             bor3(kep %in% 1:4, d == 0L, o == 0L))
expect_equal(or3s(keq %in% 1:4, x == 1L, i == 1L),
             bor3(keq %in% 1:4, x == 1L, i == 1L))
expect_equal(or3s(ker %in% 1:4, m == 9L, r == 9L),
             bor3(ker %in% 1:4, m == 9L, r == 9L))
expect_equal(or3s(kes %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kes %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ket %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ket %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(keu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(keu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kev %in% c(4L, -3L, 2L, 7L, 8L), , p > 0L),
             bor3(kev %in% c(4L, -3L, 2L, 7L, 8L), , p > 0L))
expect_equal(or3s(kew %in% c(4L, -3L, 2L, 7L, 8L), , y > 1L),
             bor3(kew %in% c(4L, -3L, 2L, 7L, 8L), , y > 1L))
expect_equal(or3s(kex %in% c(4L, -3L, 2L, 7L, 8L), , e > 9L),
             bor3(kex %in% c(4L, -3L, 2L, 7L, 8L), , e > 9L))
expect_equal(or3s(key %in% c(4L, -3L, 2L, 7L, 8L), l == 0L, ),
             bor3(key %in% c(4L, -3L, 2L, 7L, 8L), l == 0L, ))
expect_equal(or3s(kez %in% c(4L, -3L, 2L, 7L, 8L), a == 1L, ),
             bor3(kez %in% c(4L, -3L, 2L, 7L, 8L), a == 1L, ))
expect_equal(or3s(kfa %in% c(4L, -3L, 2L, 7L, 8L), s == 9L, ),
             bor3(kfa %in% c(4L, -3L, 2L, 7L, 8L), s == 9L, ))
expect_equal(or3s(kfb %in% c(4L, -3L, 2L, 7L, 8L), s == 0L, n > 0L),
             bor3(kfb %in% c(4L, -3L, 2L, 7L, 8L), s == 0L, n > 0L))
expect_equal(or3s(kfc %in% c(4L, -3L, 2L, 7L, 8L), r == 1L, v > 1L),
             bor3(kfc %in% c(4L, -3L, 2L, 7L, 8L), r == 1L, v > 1L))
expect_equal(or3s(kfd %in% c(4L, -3L, 2L, 7L, 8L), p == 9L, c > 9L),
             bor3(kfd %in% c(4L, -3L, 2L, 7L, 8L), p == 9L, c > 9L))
expect_equal(or3s(kfe %in% 1:4, , ),
             bor3(kfe %in% 1:4, , ))
expect_equal(or3s(kff %in% 1:4, , ),
             bor3(kff %in% 1:4, , ))
expect_equal(or3s(kfg %in% 1:4, , ),
             bor3(kfg %in% 1:4, , ))
expect_equal(or3s(kfh %in% 1:4, , z > 0L),
             bor3(kfh %in% 1:4, , z > 0L))
expect_equal(or3s(kfi %in% 1:4, , w > 1L),
             bor3(kfi %in% 1:4, , w > 1L))
expect_equal(or3s(kfj %in% 1:4, , c > 9L),
             bor3(kfj %in% 1:4, , c > 9L))
expect_equal(or3s(kfk %in% 1:4, t == 0L, ),
             bor3(kfk %in% 1:4, t == 0L, ))
expect_equal(or3s(kfl %in% 1:4, h == 1L, ),
             bor3(kfl %in% 1:4, h == 1L, ))
expect_equal(or3s(kfm %in% 1:4, a == 9L, ),
             bor3(kfm %in% 1:4, a == 9L, ))
expect_equal(or3s(kfn %in% 1:4, o == 0L, g > 0L),
             bor3(kfn %in% 1:4, o == 0L, g > 0L))
expect_equal(or3s(kfo %in% 1:4, q == 1L, p > 1L),
             bor3(kfo %in% 1:4, q == 1L, p > 1L))
expect_equal(or3s(kfp %in% 1:4, e == 9L, d > 9L),
             bor3(kfp %in% 1:4, e == 9L, d > 9L))
expect_equal(or3s(kfq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kfq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kfr %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kfr %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kfs %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kfs %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kft %in% c(4L, -3L, 2L, 7L, 8L), , d >= 0L),
             bor3(kft %in% c(4L, -3L, 2L, 7L, 8L), , d >= 0L))
expect_equal(or3s(kfu %in% c(4L, -3L, 2L, 7L, 8L), , p >= 1L),
             bor3(kfu %in% c(4L, -3L, 2L, 7L, 8L), , p >= 1L))
expect_equal(or3s(kfv %in% c(4L, -3L, 2L, 7L, 8L), , v >= 9L),
             bor3(kfv %in% c(4L, -3L, 2L, 7L, 8L), , v >= 9L))
expect_equal(or3s(kfw %in% c(4L, -3L, 2L, 7L, 8L), w == 0L, ),
             bor3(kfw %in% c(4L, -3L, 2L, 7L, 8L), w == 0L, ))
expect_equal(or3s(kfx %in% c(4L, -3L, 2L, 7L, 8L), r == 1L, ),
             bor3(kfx %in% c(4L, -3L, 2L, 7L, 8L), r == 1L, ))
expect_equal(or3s(kfy %in% c(4L, -3L, 2L, 7L, 8L), h == 9L, ),
             bor3(kfy %in% c(4L, -3L, 2L, 7L, 8L), h == 9L, ))
expect_equal(or3s(kfz %in% c(4L, -3L, 2L, 7L, 8L), n == 0L, a >= 0L),
             bor3(kfz %in% c(4L, -3L, 2L, 7L, 8L), n == 0L, a >= 0L))
expect_equal(or3s(kga %in% c(4L, -3L, 2L, 7L, 8L), u == 1L, a >= 1L),
             bor3(kga %in% c(4L, -3L, 2L, 7L, 8L), u == 1L, a >= 1L))
expect_equal(or3s(kgb %in% c(4L, -3L, 2L, 7L, 8L), z == 9L, j >= 9L),
             bor3(kgb %in% c(4L, -3L, 2L, 7L, 8L), z == 9L, j >= 9L))
expect_equal(or3s(kgc %in% 1:4, , ),
             bor3(kgc %in% 1:4, , ))
expect_equal(or3s(kgd %in% 1:4, , ),
             bor3(kgd %in% 1:4, , ))
expect_equal(or3s(kge %in% 1:4, , ),
             bor3(kge %in% 1:4, , ))
expect_equal(or3s(kgf %in% 1:4, , a >= 0L),
             bor3(kgf %in% 1:4, , a >= 0L))
expect_equal(or3s(kgg %in% 1:4, , d >= 1L),
             bor3(kgg %in% 1:4, , d >= 1L))
expect_equal(or3s(kgh %in% 1:4, , l >= 9L),
             bor3(kgh %in% 1:4, , l >= 9L))
expect_equal(or3s(kgi %in% 1:4, a == 0L, ),
             bor3(kgi %in% 1:4, a == 0L, ))
expect_equal(or3s(kgj %in% 1:4, t == 1L, ),
             bor3(kgj %in% 1:4, t == 1L, ))
expect_equal(or3s(kgk %in% 1:4, p == 9L, ),
             bor3(kgk %in% 1:4, p == 9L, ))
expect_equal(or3s(kgl %in% 1:4, c == 0L, l >= 0L),
             bor3(kgl %in% 1:4, c == 0L, l >= 0L))
expect_equal(or3s(kgm %in% 1:4, v == 1L, r >= 1L),
             bor3(kgm %in% 1:4, v == 1L, r >= 1L))
expect_equal(or3s(kgn %in% 1:4, y == 9L, b >= 9L),
             bor3(kgn %in% 1:4, y == 9L, b >= 9L))
expect_equal(or3s(kgo %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kgo %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kgp %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kgp %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kgq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kgq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kgr %in% c(4L, -3L, 2L, 7L, 8L), , logi_e),
             bor3(kgr %in% c(4L, -3L, 2L, 7L, 8L), , logi_e))
expect_equal(or3s(kgs %in% c(4L, -3L, 2L, 7L, 8L), , logi_p),
             bor3(kgs %in% c(4L, -3L, 2L, 7L, 8L), , logi_p))
expect_equal(or3s(kgt %in% c(4L, -3L, 2L, 7L, 8L), , logi_n),
             bor3(kgt %in% c(4L, -3L, 2L, 7L, 8L), , logi_n))
expect_equal(or3s(kgu %in% c(4L, -3L, 2L, 7L, 8L), v > 0L, ),
             bor3(kgu %in% c(4L, -3L, 2L, 7L, 8L), v > 0L, ))
expect_equal(or3s(kgv %in% c(4L, -3L, 2L, 7L, 8L), u > 1L, ),
             bor3(kgv %in% c(4L, -3L, 2L, 7L, 8L), u > 1L, ))
expect_equal(or3s(kgw %in% c(4L, -3L, 2L, 7L, 8L), i > 9L, ),
             bor3(kgw %in% c(4L, -3L, 2L, 7L, 8L), i > 9L, ))
expect_equal(or3s(kgx %in% c(4L, -3L, 2L, 7L, 8L), u > 0L, logi_y),
             bor3(kgx %in% c(4L, -3L, 2L, 7L, 8L), u > 0L, logi_y))
expect_equal(or3s(kgy %in% c(4L, -3L, 2L, 7L, 8L), s > 1L, logi_w),
             bor3(kgy %in% c(4L, -3L, 2L, 7L, 8L), s > 1L, logi_w))
expect_equal(or3s(kgz %in% c(4L, -3L, 2L, 7L, 8L), w > 9L, logi_c),
             bor3(kgz %in% c(4L, -3L, 2L, 7L, 8L), w > 9L, logi_c))
expect_equal(or3s(kha %in% 1:4, , ),
             bor3(kha %in% 1:4, , ))
expect_equal(or3s(khb %in% 1:4, , ),
             bor3(khb %in% 1:4, , ))
expect_equal(or3s(khc %in% 1:4, , ),
             bor3(khc %in% 1:4, , ))
expect_equal(or3s(khd %in% 1:4, , logi_k),
             bor3(khd %in% 1:4, , logi_k))
expect_equal(or3s(khe %in% 1:4, , logi_q),
             bor3(khe %in% 1:4, , logi_q))
expect_equal(or3s(khf %in% 1:4, , logi_o),
             bor3(khf %in% 1:4, , logi_o))
expect_equal(or3s(khg %in% 1:4, v > 0L, ),
             bor3(khg %in% 1:4, v > 0L, ))
expect_equal(or3s(khh %in% 1:4, t > 1L, ),
             bor3(khh %in% 1:4, t > 1L, ))
expect_equal(or3s(khi %in% 1:4, r > 9L, ),
             bor3(khi %in% 1:4, r > 9L, ))
expect_equal(or3s(khj %in% 1:4, h > 0L, logi_w),
             bor3(khj %in% 1:4, h > 0L, logi_w))
expect_equal(or3s(khk %in% 1:4, t > 1L, logi_k),
             bor3(khk %in% 1:4, t > 1L, logi_k))
expect_equal(or3s(khl %in% 1:4, x > 9L, logi_i),
             bor3(khl %in% 1:4, x > 9L, logi_i))
expect_equal(or3s(khm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(khm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(khn %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(khn %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kho %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kho %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(khp %in% c(4L, -3L, 2L, 7L, 8L), , !logi_a),
             bor3(khp %in% c(4L, -3L, 2L, 7L, 8L), , !logi_a))
expect_equal(or3s(khq %in% c(4L, -3L, 2L, 7L, 8L), , !logi_h),
             bor3(khq %in% c(4L, -3L, 2L, 7L, 8L), , !logi_h))
expect_equal(or3s(khr %in% c(4L, -3L, 2L, 7L, 8L), , !logi_f),
             bor3(khr %in% c(4L, -3L, 2L, 7L, 8L), , !logi_f))
expect_equal(or3s(khs %in% c(4L, -3L, 2L, 7L, 8L), c > 0L, ),
             bor3(khs %in% c(4L, -3L, 2L, 7L, 8L), c > 0L, ))
expect_equal(or3s(kht %in% c(4L, -3L, 2L, 7L, 8L), m > 1L, ),
             bor3(kht %in% c(4L, -3L, 2L, 7L, 8L), m > 1L, ))
expect_equal(or3s(khu %in% c(4L, -3L, 2L, 7L, 8L), o > 9L, ),
             bor3(khu %in% c(4L, -3L, 2L, 7L, 8L), o > 9L, ))
expect_equal(or3s(khv %in% c(4L, -3L, 2L, 7L, 8L), t > 0L, !logi_b),
             bor3(khv %in% c(4L, -3L, 2L, 7L, 8L), t > 0L, !logi_b))
expect_equal(or3s(khw %in% c(4L, -3L, 2L, 7L, 8L), v > 1L, !logi_j),
             bor3(khw %in% c(4L, -3L, 2L, 7L, 8L), v > 1L, !logi_j))
expect_equal(or3s(khx %in% c(4L, -3L, 2L, 7L, 8L), w > 9L, !logi_j),
             bor3(khx %in% c(4L, -3L, 2L, 7L, 8L), w > 9L, !logi_j))
expect_equal(or3s(khy %in% 1:4, , ),
             bor3(khy %in% 1:4, , ))
expect_equal(or3s(khz %in% 1:4, , ),
             bor3(khz %in% 1:4, , ))
expect_equal(or3s(kia %in% 1:4, , ),
             bor3(kia %in% 1:4, , ))
expect_equal(or3s(kib %in% 1:4, , !logi_i),
             bor3(kib %in% 1:4, , !logi_i))
expect_equal(or3s(kic %in% 1:4, , !logi_p),
             bor3(kic %in% 1:4, , !logi_p))
expect_equal(or3s(kid %in% 1:4, , !logi_v),
             bor3(kid %in% 1:4, , !logi_v))
expect_equal(or3s(kie %in% 1:4, o > 0L, ),
             bor3(kie %in% 1:4, o > 0L, ))
expect_equal(or3s(kif %in% 1:4, c > 1L, ),
             bor3(kif %in% 1:4, c > 1L, ))
expect_equal(or3s(kig %in% 1:4, q > 9L, ),
             bor3(kig %in% 1:4, q > 9L, ))
expect_equal(or3s(kih %in% 1:4, v > 0L, !logi_t),
             bor3(kih %in% 1:4, v > 0L, !logi_t))
expect_equal(or3s(kii %in% 1:4, p > 1L, !logi_g),
             bor3(kii %in% 1:4, p > 1L, !logi_g))
expect_equal(or3s(kij %in% 1:4, r > 9L, !logi_k),
             bor3(kij %in% 1:4, r > 9L, !logi_k))
expect_equal(or3s(kik %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kik %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kil %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kil %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kim %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kim %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kin %in% c(4L, -3L, 2L, 7L, 8L), , g != 0L),
             bor3(kin %in% c(4L, -3L, 2L, 7L, 8L), , g != 0L))
expect_equal(or3s(kio %in% c(4L, -3L, 2L, 7L, 8L), , h != 1L),
             bor3(kio %in% c(4L, -3L, 2L, 7L, 8L), , h != 1L))
expect_equal(or3s(kip %in% c(4L, -3L, 2L, 7L, 8L), , q != 9L),
             bor3(kip %in% c(4L, -3L, 2L, 7L, 8L), , q != 9L))
expect_equal(or3s(kiq %in% c(4L, -3L, 2L, 7L, 8L), m > 0L, ),
             bor3(kiq %in% c(4L, -3L, 2L, 7L, 8L), m > 0L, ))
expect_equal(or3s(kir %in% c(4L, -3L, 2L, 7L, 8L), y > 1L, ),
             bor3(kir %in% c(4L, -3L, 2L, 7L, 8L), y > 1L, ))
expect_equal(or3s(kis %in% c(4L, -3L, 2L, 7L, 8L), k > 9L, ),
             bor3(kis %in% c(4L, -3L, 2L, 7L, 8L), k > 9L, ))
expect_equal(or3s(kit %in% c(4L, -3L, 2L, 7L, 8L), p > 0L, d != 0L),
             bor3(kit %in% c(4L, -3L, 2L, 7L, 8L), p > 0L, d != 0L))
expect_equal(or3s(kiu %in% c(4L, -3L, 2L, 7L, 8L), l > 1L, x != 1L),
             bor3(kiu %in% c(4L, -3L, 2L, 7L, 8L), l > 1L, x != 1L))
expect_equal(or3s(kiv %in% c(4L, -3L, 2L, 7L, 8L), s > 9L, d != 9L),
             bor3(kiv %in% c(4L, -3L, 2L, 7L, 8L), s > 9L, d != 9L))
expect_equal(or3s(kiw %in% 1:4, , ),
             bor3(kiw %in% 1:4, , ))
expect_equal(or3s(kix %in% 1:4, , ),
             bor3(kix %in% 1:4, , ))
expect_equal(or3s(kiy %in% 1:4, , ),
             bor3(kiy %in% 1:4, , ))
expect_equal(or3s(kiz %in% 1:4, , z != 0L),
             bor3(kiz %in% 1:4, , z != 0L))
expect_equal(or3s(kja %in% 1:4, , n != 1L),
             bor3(kja %in% 1:4, , n != 1L))
expect_equal(or3s(kjb %in% 1:4, , b != 9L),
             bor3(kjb %in% 1:4, , b != 9L))
expect_equal(or3s(kjc %in% 1:4, b > 0L, ),
             bor3(kjc %in% 1:4, b > 0L, ))
expect_equal(or3s(kjd %in% 1:4, j > 1L, ),
             bor3(kjd %in% 1:4, j > 1L, ))
expect_equal(or3s(kje %in% 1:4, j > 9L, ),
             bor3(kje %in% 1:4, j > 9L, ))
expect_equal(or3s(kjf %in% 1:4, y > 0L, k != 0L),
             bor3(kjf %in% 1:4, y > 0L, k != 0L))
expect_equal(or3s(kjg %in% 1:4, n > 1L, m != 1L),
             bor3(kjg %in% 1:4, n > 1L, m != 1L))
expect_equal(or3s(kjh %in% 1:4, u > 9L, q != 9L),
             bor3(kjh %in% 1:4, u > 9L, q != 9L))
expect_equal(or3s(kji %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kji %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kjj %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kjj %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kjk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kjk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kjl %in% c(4L, -3L, 2L, 7L, 8L), , y %between% c(-1L, 1L)),
             bor3(kjl %in% c(4L, -3L, 2L, 7L, 8L), , y %between% c(-1L, 1L)))
expect_equal(or3s(kjm %in% c(4L, -3L, 2L, 7L, 8L), , a %between% c(-1L, 1L)),
             bor3(kjm %in% c(4L, -3L, 2L, 7L, 8L), , a %between% c(-1L, 1L)))
expect_equal(or3s(kjn %in% c(4L, -3L, 2L, 7L, 8L), , c %between% c(-1L, 1L)),
             bor3(kjn %in% c(4L, -3L, 2L, 7L, 8L), , c %between% c(-1L, 1L)))
expect_equal(or3s(kjo %in% c(4L, -3L, 2L, 7L, 8L), v > 0L, ),
             bor3(kjo %in% c(4L, -3L, 2L, 7L, 8L), v > 0L, ))
expect_equal(or3s(kjp %in% c(4L, -3L, 2L, 7L, 8L), z > 1L, ),
             bor3(kjp %in% c(4L, -3L, 2L, 7L, 8L), z > 1L, ))
expect_equal(or3s(kjq %in% c(4L, -3L, 2L, 7L, 8L), k > 9L, ),
             bor3(kjq %in% c(4L, -3L, 2L, 7L, 8L), k > 9L, ))
expect_equal(or3s(kjr %in% c(4L, -3L, 2L, 7L, 8L), g > 0L, u %between% c(-1L, 1L)),
             bor3(kjr %in% c(4L, -3L, 2L, 7L, 8L), g > 0L, u %between% c(-1L, 1L)))
expect_equal(or3s(kjs %in% c(4L, -3L, 2L, 7L, 8L), w > 1L, s %between% c(-1L, 1L)),
             bor3(kjs %in% c(4L, -3L, 2L, 7L, 8L), w > 1L, s %between% c(-1L, 1L)))
expect_equal(or3s(kjt %in% c(4L, -3L, 2L, 7L, 8L), x > 9L, w %between% c(-1L, 1L)),
             bor3(kjt %in% c(4L, -3L, 2L, 7L, 8L), x > 9L, w %between% c(-1L, 1L)))
expect_equal(or3s(kju %in% 1:4, , ),
             bor3(kju %in% 1:4, , ))
expect_equal(or3s(kjv %in% 1:4, , ),
             bor3(kjv %in% 1:4, , ))
expect_equal(or3s(kjw %in% 1:4, , ),
             bor3(kjw %in% 1:4, , ))
expect_equal(or3s(kjx %in% 1:4, , g %between% c(-1L, 1L)),
             bor3(kjx %in% 1:4, , g %between% c(-1L, 1L)))
expect_equal(or3s(kjy %in% 1:4, , e %between% c(-1L, 1L)),
             bor3(kjy %in% 1:4, , e %between% c(-1L, 1L)))
expect_equal(or3s(kjz %in% 1:4, , l %between% c(-1L, 1L)),
             bor3(kjz %in% 1:4, , l %between% c(-1L, 1L)))
expect_equal(or3s(kka %in% 1:4, k > 0L, ),
             bor3(kka %in% 1:4, k > 0L, ))
expect_equal(or3s(kkb %in% 1:4, u > 1L, ),
             bor3(kkb %in% 1:4, u > 1L, ))
expect_equal(or3s(kkc %in% 1:4, p > 9L, ),
             bor3(kkc %in% 1:4, p > 9L, ))
expect_equal(or3s(kkd %in% 1:4, v > 0L, g %between% c(-1L, 1L)),
             bor3(kkd %in% 1:4, v > 0L, g %between% c(-1L, 1L)))
expect_equal(or3s(kke %in% 1:4, d > 1L, i %between% c(-1L, 1L)),
             bor3(kke %in% 1:4, d > 1L, i %between% c(-1L, 1L)))
expect_equal(or3s(kkf %in% 1:4, i > 9L, u %between% c(-1L, 1L)),
             bor3(kkf %in% 1:4, i > 9L, u %between% c(-1L, 1L)))
expect_equal(or3s(kkg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kkg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kkh %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kkh %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kki %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kki %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kkj %in% c(4L, -3L, 2L, 7L, 8L), , w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kkj %in% c(4L, -3L, 2L, 7L, 8L), , w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kkk %in% c(4L, -3L, 2L, 7L, 8L), , r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kkk %in% c(4L, -3L, 2L, 7L, 8L), , r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kkl %in% c(4L, -3L, 2L, 7L, 8L), , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kkl %in% c(4L, -3L, 2L, 7L, 8L), , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kkm %in% c(4L, -3L, 2L, 7L, 8L), b > 0L, ),
             bor3(kkm %in% c(4L, -3L, 2L, 7L, 8L), b > 0L, ))
expect_equal(or3s(kkn %in% c(4L, -3L, 2L, 7L, 8L), r > 1L, ),
             bor3(kkn %in% c(4L, -3L, 2L, 7L, 8L), r > 1L, ))
expect_equal(or3s(kko %in% c(4L, -3L, 2L, 7L, 8L), l > 9L, ),
             bor3(kko %in% c(4L, -3L, 2L, 7L, 8L), l > 9L, ))
expect_equal(or3s(kkp %in% c(4L, -3L, 2L, 7L, 8L), o > 0L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kkp %in% c(4L, -3L, 2L, 7L, 8L), o > 0L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kkq %in% c(4L, -3L, 2L, 7L, 8L), v > 1L, d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kkq %in% c(4L, -3L, 2L, 7L, 8L), v > 1L, d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kkr %in% c(4L, -3L, 2L, 7L, 8L), o > 9L, o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(kkr %in% c(4L, -3L, 2L, 7L, 8L), o > 9L, o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kks %in% 1:4, , ),
             bor3(kks %in% 1:4, , ))
expect_equal(or3s(kkt %in% 1:4, , ),
             bor3(kkt %in% 1:4, , ))
expect_equal(or3s(kku %in% 1:4, , ),
             bor3(kku %in% 1:4, , ))
expect_equal(or3s(kkv %in% 1:4, , l %in% 1:4),
             bor3(kkv %in% 1:4, , l %in% 1:4))
expect_equal(or3s(kkw %in% 1:4, , k %in% 1:4),
             bor3(kkw %in% 1:4, , k %in% 1:4))
expect_equal(or3s(kkx %in% 1:4, , s %in% 1:4),
             bor3(kkx %in% 1:4, , s %in% 1:4))
expect_equal(or3s(kky %in% 1:4, i > 0L, ),
             bor3(kky %in% 1:4, i > 0L, ))
expect_equal(or3s(kkz %in% 1:4, y > 1L, ),
             bor3(kkz %in% 1:4, y > 1L, ))
expect_equal(or3s(kla %in% 1:4, c > 9L, ),
             bor3(kla %in% 1:4, c > 9L, ))
expect_equal(or3s(klb %in% 1:4, v > 0L, s %in% 1:4),
             bor3(klb %in% 1:4, v > 0L, s %in% 1:4))
expect_equal(or3s(klc %in% 1:4, v > 1L, r %in% 1:4),
             bor3(klc %in% 1:4, v > 1L, r %in% 1:4))
expect_equal(or3s(kld %in% 1:4, u > 9L, b %in% 1:4),
             bor3(kld %in% 1:4, u > 9L, b %in% 1:4))
expect_equal(or3s(kle %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kle %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(klf %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(klf %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(klg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(klg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(klh %in% c(4L, -3L, 2L, 7L, 8L), , f < 0L),
             bor3(klh %in% c(4L, -3L, 2L, 7L, 8L), , f < 0L))
expect_equal(or3s(kli %in% c(4L, -3L, 2L, 7L, 8L), , x < 1L),
             bor3(kli %in% c(4L, -3L, 2L, 7L, 8L), , x < 1L))
expect_equal(or3s(klj %in% c(4L, -3L, 2L, 7L, 8L), , g < 9L),
             bor3(klj %in% c(4L, -3L, 2L, 7L, 8L), , g < 9L))
expect_equal(or3s(klk %in% c(4L, -3L, 2L, 7L, 8L), z > 0L, ),
             bor3(klk %in% c(4L, -3L, 2L, 7L, 8L), z > 0L, ))
expect_equal(or3s(kll %in% c(4L, -3L, 2L, 7L, 8L), a > 1L, ),
             bor3(kll %in% c(4L, -3L, 2L, 7L, 8L), a > 1L, ))
expect_equal(or3s(klm %in% c(4L, -3L, 2L, 7L, 8L), t > 9L, ),
             bor3(klm %in% c(4L, -3L, 2L, 7L, 8L), t > 9L, ))
expect_equal(or3s(kln %in% c(4L, -3L, 2L, 7L, 8L), d > 0L, t < 0L),
             bor3(kln %in% c(4L, -3L, 2L, 7L, 8L), d > 0L, t < 0L))
expect_equal(or3s(klo %in% c(4L, -3L, 2L, 7L, 8L), g > 1L, i < 1L),
             bor3(klo %in% c(4L, -3L, 2L, 7L, 8L), g > 1L, i < 1L))
expect_equal(or3s(klp %in% c(4L, -3L, 2L, 7L, 8L), p > 9L, s < 9L),
             bor3(klp %in% c(4L, -3L, 2L, 7L, 8L), p > 9L, s < 9L))
expect_equal(or3s(klq %in% 1:4, , ),
             bor3(klq %in% 1:4, , ))
expect_equal(or3s(klr %in% 1:4, , ),
             bor3(klr %in% 1:4, , ))
expect_equal(or3s(kls %in% 1:4, , ),
             bor3(kls %in% 1:4, , ))
expect_equal(or3s(klt %in% 1:4, , k < 0L),
             bor3(klt %in% 1:4, , k < 0L))
expect_equal(or3s(klu %in% 1:4, , c < 1L),
             bor3(klu %in% 1:4, , c < 1L))
expect_equal(or3s(klv %in% 1:4, , d < 9L),
             bor3(klv %in% 1:4, , d < 9L))
expect_equal(or3s(klw %in% 1:4, x > 0L, ),
             bor3(klw %in% 1:4, x > 0L, ))
expect_equal(or3s(klx %in% 1:4, f > 1L, ),
             bor3(klx %in% 1:4, f > 1L, ))
expect_equal(or3s(kly %in% 1:4, n > 9L, ),
             bor3(kly %in% 1:4, n > 9L, ))
expect_equal(or3s(klz %in% 1:4, u > 0L, i < 0L),
             bor3(klz %in% 1:4, u > 0L, i < 0L))
expect_equal(or3s(kma %in% 1:4, t > 1L, u < 1L),
             bor3(kma %in% 1:4, t > 1L, u < 1L))
expect_equal(or3s(kmb %in% 1:4, p > 9L, w < 9L),
             bor3(kmb %in% 1:4, p > 9L, w < 9L))
expect_equal(or3s(kmc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kmc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kmd %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kmd %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kme %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kme %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kmf %in% c(4L, -3L, 2L, 7L, 8L), , e <= 0L),
             bor3(kmf %in% c(4L, -3L, 2L, 7L, 8L), , e <= 0L))
expect_equal(or3s(kmg %in% c(4L, -3L, 2L, 7L, 8L), , i <= 1L),
             bor3(kmg %in% c(4L, -3L, 2L, 7L, 8L), , i <= 1L))
expect_equal(or3s(kmh %in% c(4L, -3L, 2L, 7L, 8L), , c <= 9L),
             bor3(kmh %in% c(4L, -3L, 2L, 7L, 8L), , c <= 9L))
expect_equal(or3s(kmi %in% c(4L, -3L, 2L, 7L, 8L), o > 0L, ),
             bor3(kmi %in% c(4L, -3L, 2L, 7L, 8L), o > 0L, ))
expect_equal(or3s(kmj %in% c(4L, -3L, 2L, 7L, 8L), u > 1L, ),
             bor3(kmj %in% c(4L, -3L, 2L, 7L, 8L), u > 1L, ))
expect_equal(or3s(kmk %in% c(4L, -3L, 2L, 7L, 8L), z > 9L, ),
             bor3(kmk %in% c(4L, -3L, 2L, 7L, 8L), z > 9L, ))
expect_equal(or3s(kml %in% c(4L, -3L, 2L, 7L, 8L), j > 0L, o <= 0L),
             bor3(kml %in% c(4L, -3L, 2L, 7L, 8L), j > 0L, o <= 0L))
expect_equal(or3s(kmm %in% c(4L, -3L, 2L, 7L, 8L), p > 1L, b <= 1L),
             bor3(kmm %in% c(4L, -3L, 2L, 7L, 8L), p > 1L, b <= 1L))
expect_equal(or3s(kmn %in% c(4L, -3L, 2L, 7L, 8L), n > 9L, v <= 9L),
             bor3(kmn %in% c(4L, -3L, 2L, 7L, 8L), n > 9L, v <= 9L))
expect_equal(or3s(kmo %in% 1:4, , ),
             bor3(kmo %in% 1:4, , ))
expect_equal(or3s(kmp %in% 1:4, , ),
             bor3(kmp %in% 1:4, , ))
expect_equal(or3s(kmq %in% 1:4, , ),
             bor3(kmq %in% 1:4, , ))
expect_equal(or3s(kmr %in% 1:4, , b <= 0L),
             bor3(kmr %in% 1:4, , b <= 0L))
expect_equal(or3s(kms %in% 1:4, , l <= 1L),
             bor3(kms %in% 1:4, , l <= 1L))
expect_equal(or3s(kmt %in% 1:4, , q <= 9L),
             bor3(kmt %in% 1:4, , q <= 9L))
expect_equal(or3s(kmu %in% 1:4, s > 0L, ),
             bor3(kmu %in% 1:4, s > 0L, ))
expect_equal(or3s(kmv %in% 1:4, k > 1L, ),
             bor3(kmv %in% 1:4, k > 1L, ))
expect_equal(or3s(kmw %in% 1:4, r > 9L, ),
             bor3(kmw %in% 1:4, r > 9L, ))
expect_equal(or3s(kmx %in% 1:4, v > 0L, y <= 0L),
             bor3(kmx %in% 1:4, v > 0L, y <= 0L))
expect_equal(or3s(kmy %in% 1:4, v > 1L, v <= 1L),
             bor3(kmy %in% 1:4, v > 1L, v <= 1L))
expect_equal(or3s(kmz %in% 1:4, k > 9L, l <= 9L),
             bor3(kmz %in% 1:4, k > 9L, l <= 9L))
expect_equal(or3s(kna %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kna %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(knb %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(knb %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(knc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(knc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(knd %in% c(4L, -3L, 2L, 7L, 8L), , i == 0L),
             bor3(knd %in% c(4L, -3L, 2L, 7L, 8L), , i == 0L))
expect_equal(or3s(kne %in% c(4L, -3L, 2L, 7L, 8L), , w == 1L),
             bor3(kne %in% c(4L, -3L, 2L, 7L, 8L), , w == 1L))
expect_equal(or3s(knf %in% c(4L, -3L, 2L, 7L, 8L), , l == 9L),
             bor3(knf %in% c(4L, -3L, 2L, 7L, 8L), , l == 9L))
expect_equal(or3s(kng %in% c(4L, -3L, 2L, 7L, 8L), g > 0L, ),
             bor3(kng %in% c(4L, -3L, 2L, 7L, 8L), g > 0L, ))
expect_equal(or3s(knh %in% c(4L, -3L, 2L, 7L, 8L), h > 1L, ),
             bor3(knh %in% c(4L, -3L, 2L, 7L, 8L), h > 1L, ))
expect_equal(or3s(kni %in% c(4L, -3L, 2L, 7L, 8L), b > 9L, ),
             bor3(kni %in% c(4L, -3L, 2L, 7L, 8L), b > 9L, ))
expect_equal(or3s(knj %in% c(4L, -3L, 2L, 7L, 8L), c > 0L, t == 0L),
             bor3(knj %in% c(4L, -3L, 2L, 7L, 8L), c > 0L, t == 0L))
expect_equal(or3s(knk %in% c(4L, -3L, 2L, 7L, 8L), h > 1L, v == 1L),
             bor3(knk %in% c(4L, -3L, 2L, 7L, 8L), h > 1L, v == 1L))
expect_equal(or3s(knl %in% c(4L, -3L, 2L, 7L, 8L), t > 9L, b == 9L),
             bor3(knl %in% c(4L, -3L, 2L, 7L, 8L), t > 9L, b == 9L))
expect_equal(or3s(knm %in% 1:4, , ),
             bor3(knm %in% 1:4, , ))
expect_equal(or3s(knn %in% 1:4, , ),
             bor3(knn %in% 1:4, , ))
expect_equal(or3s(kno %in% 1:4, , ),
             bor3(kno %in% 1:4, , ))
expect_equal(or3s(knp %in% 1:4, , w == 0L),
             bor3(knp %in% 1:4, , w == 0L))
expect_equal(or3s(knq %in% 1:4, , h == 1L),
             bor3(knq %in% 1:4, , h == 1L))
expect_equal(or3s(knr %in% 1:4, , h == 9L),
             bor3(knr %in% 1:4, , h == 9L))
expect_equal(or3s(kns %in% 1:4, u > 0L, ),
             bor3(kns %in% 1:4, u > 0L, ))
expect_equal(or3s(knt %in% 1:4, y > 1L, ),
             bor3(knt %in% 1:4, y > 1L, ))
expect_equal(or3s(knu %in% 1:4, c > 9L, ),
             bor3(knu %in% 1:4, c > 9L, ))
expect_equal(or3s(knv %in% 1:4, f > 0L, k == 0L),
             bor3(knv %in% 1:4, f > 0L, k == 0L))
expect_equal(or3s(knw %in% 1:4, f > 1L, m == 1L),
             bor3(knw %in% 1:4, f > 1L, m == 1L))
expect_equal(or3s(knx %in% 1:4, m > 9L, e == 9L),
             bor3(knx %in% 1:4, m > 9L, e == 9L))
expect_equal(or3s(kny %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kny %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(knz %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(knz %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(koa %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(koa %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kob %in% c(4L, -3L, 2L, 7L, 8L), , n > 0L),
             bor3(kob %in% c(4L, -3L, 2L, 7L, 8L), , n > 0L))
expect_equal(or3s(koc %in% c(4L, -3L, 2L, 7L, 8L), , o > 1L),
             bor3(koc %in% c(4L, -3L, 2L, 7L, 8L), , o > 1L))
expect_equal(or3s(kod %in% c(4L, -3L, 2L, 7L, 8L), , h > 9L),
             bor3(kod %in% c(4L, -3L, 2L, 7L, 8L), , h > 9L))
expect_equal(or3s(koe %in% c(4L, -3L, 2L, 7L, 8L), r > 0L, ),
             bor3(koe %in% c(4L, -3L, 2L, 7L, 8L), r > 0L, ))
expect_equal(or3s(kof %in% c(4L, -3L, 2L, 7L, 8L), d > 1L, ),
             bor3(kof %in% c(4L, -3L, 2L, 7L, 8L), d > 1L, ))
expect_equal(or3s(kog %in% c(4L, -3L, 2L, 7L, 8L), a > 9L, ),
             bor3(kog %in% c(4L, -3L, 2L, 7L, 8L), a > 9L, ))
expect_equal(or3s(koh %in% c(4L, -3L, 2L, 7L, 8L), i > 0L, u > 0L),
             bor3(koh %in% c(4L, -3L, 2L, 7L, 8L), i > 0L, u > 0L))
expect_equal(or3s(koi %in% c(4L, -3L, 2L, 7L, 8L), y > 1L, a > 1L),
             bor3(koi %in% c(4L, -3L, 2L, 7L, 8L), y > 1L, a > 1L))
expect_equal(or3s(koj %in% c(4L, -3L, 2L, 7L, 8L), g > 9L, r > 9L),
             bor3(koj %in% c(4L, -3L, 2L, 7L, 8L), g > 9L, r > 9L))
expect_equal(or3s(kok %in% 1:4, , ),
             bor3(kok %in% 1:4, , ))
expect_equal(or3s(kol %in% 1:4, , ),
             bor3(kol %in% 1:4, , ))
expect_equal(or3s(kom %in% 1:4, , ),
             bor3(kom %in% 1:4, , ))
expect_equal(or3s(kon %in% 1:4, , v > 0L),
             bor3(kon %in% 1:4, , v > 0L))
expect_equal(or3s(koo %in% 1:4, , g > 1L),
             bor3(koo %in% 1:4, , g > 1L))
expect_equal(or3s(kop %in% 1:4, , q > 9L),
             bor3(kop %in% 1:4, , q > 9L))
expect_equal(or3s(koq %in% 1:4, j > 0L, ),
             bor3(koq %in% 1:4, j > 0L, ))
expect_equal(or3s(kor %in% 1:4, m > 1L, ),
             bor3(kor %in% 1:4, m > 1L, ))
expect_equal(or3s(kos %in% 1:4, e > 9L, ),
             bor3(kos %in% 1:4, e > 9L, ))
expect_equal(or3s(kot %in% 1:4, l > 0L, r > 0L),
             bor3(kot %in% 1:4, l > 0L, r > 0L))
expect_equal(or3s(kou %in% 1:4, h > 1L, c > 1L),
             bor3(kou %in% 1:4, h > 1L, c > 1L))
expect_equal(or3s(kov %in% 1:4, e > 9L, a > 9L),
             bor3(kov %in% 1:4, e > 9L, a > 9L))
expect_equal(or3s(kow %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kow %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kox %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kox %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(koy %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(koy %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(koz %in% c(4L, -3L, 2L, 7L, 8L), , y >= 0L),
             bor3(koz %in% c(4L, -3L, 2L, 7L, 8L), , y >= 0L))
expect_equal(or3s(kpa %in% c(4L, -3L, 2L, 7L, 8L), , c >= 1L),
             bor3(kpa %in% c(4L, -3L, 2L, 7L, 8L), , c >= 1L))
expect_equal(or3s(kpb %in% c(4L, -3L, 2L, 7L, 8L), , a >= 9L),
             bor3(kpb %in% c(4L, -3L, 2L, 7L, 8L), , a >= 9L))
expect_equal(or3s(kpc %in% c(4L, -3L, 2L, 7L, 8L), g > 0L, ),
             bor3(kpc %in% c(4L, -3L, 2L, 7L, 8L), g > 0L, ))
expect_equal(or3s(kpd %in% c(4L, -3L, 2L, 7L, 8L), l > 1L, ),
             bor3(kpd %in% c(4L, -3L, 2L, 7L, 8L), l > 1L, ))
expect_equal(or3s(kpe %in% c(4L, -3L, 2L, 7L, 8L), t > 9L, ),
             bor3(kpe %in% c(4L, -3L, 2L, 7L, 8L), t > 9L, ))
expect_equal(or3s(kpf %in% c(4L, -3L, 2L, 7L, 8L), z > 0L, y >= 0L),
             bor3(kpf %in% c(4L, -3L, 2L, 7L, 8L), z > 0L, y >= 0L))
expect_equal(or3s(kpg %in% c(4L, -3L, 2L, 7L, 8L), r > 1L, b >= 1L),
             bor3(kpg %in% c(4L, -3L, 2L, 7L, 8L), r > 1L, b >= 1L))
expect_equal(or3s(kph %in% c(4L, -3L, 2L, 7L, 8L), e > 9L, p >= 9L),
             bor3(kph %in% c(4L, -3L, 2L, 7L, 8L), e > 9L, p >= 9L))
expect_equal(or3s(kpi %in% 1:4, , ),
             bor3(kpi %in% 1:4, , ))
expect_equal(or3s(kpj %in% 1:4, , ),
             bor3(kpj %in% 1:4, , ))
expect_equal(or3s(kpk %in% 1:4, , ),
             bor3(kpk %in% 1:4, , ))
expect_equal(or3s(kpl %in% 1:4, , u >= 0L),
             bor3(kpl %in% 1:4, , u >= 0L))
expect_equal(or3s(kpm %in% 1:4, , r >= 1L),
             bor3(kpm %in% 1:4, , r >= 1L))
expect_equal(or3s(kpn %in% 1:4, , y >= 9L),
             bor3(kpn %in% 1:4, , y >= 9L))
expect_equal(or3s(kpo %in% 1:4, v > 0L, ),
             bor3(kpo %in% 1:4, v > 0L, ))
expect_equal(or3s(kpp %in% 1:4, z > 1L, ),
             bor3(kpp %in% 1:4, z > 1L, ))
expect_equal(or3s(kpq %in% 1:4, h > 9L, ),
             bor3(kpq %in% 1:4, h > 9L, ))
expect_equal(or3s(kpr %in% 1:4, a > 0L, b >= 0L),
             bor3(kpr %in% 1:4, a > 0L, b >= 0L))
expect_equal(or3s(kps %in% 1:4, e > 1L, q >= 1L),
             bor3(kps %in% 1:4, e > 1L, q >= 1L))
expect_equal(or3s(kpt %in% 1:4, w > 9L, r >= 9L),
             bor3(kpt %in% 1:4, w > 9L, r >= 9L))
expect_equal(or3s(kpu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kpu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kpv %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kpv %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kpw %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kpw %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kpx %in% c(4L, -3L, 2L, 7L, 8L), , logi_v),
             bor3(kpx %in% c(4L, -3L, 2L, 7L, 8L), , logi_v))
expect_equal(or3s(kpy %in% c(4L, -3L, 2L, 7L, 8L), , logi_h),
             bor3(kpy %in% c(4L, -3L, 2L, 7L, 8L), , logi_h))
expect_equal(or3s(kpz %in% c(4L, -3L, 2L, 7L, 8L), , logi_k),
             bor3(kpz %in% c(4L, -3L, 2L, 7L, 8L), , logi_k))
expect_equal(or3s(kqa %in% c(4L, -3L, 2L, 7L, 8L), b >= 0L, ),
             bor3(kqa %in% c(4L, -3L, 2L, 7L, 8L), b >= 0L, ))
expect_equal(or3s(kqb %in% c(4L, -3L, 2L, 7L, 8L), x >= 1L, ),
             bor3(kqb %in% c(4L, -3L, 2L, 7L, 8L), x >= 1L, ))
expect_equal(or3s(kqc %in% c(4L, -3L, 2L, 7L, 8L), i >= 9L, ),
             bor3(kqc %in% c(4L, -3L, 2L, 7L, 8L), i >= 9L, ))
expect_equal(or3s(kqd %in% c(4L, -3L, 2L, 7L, 8L), e >= 0L, logi_c),
             bor3(kqd %in% c(4L, -3L, 2L, 7L, 8L), e >= 0L, logi_c))
expect_equal(or3s(kqe %in% c(4L, -3L, 2L, 7L, 8L), y >= 1L, logi_i),
             bor3(kqe %in% c(4L, -3L, 2L, 7L, 8L), y >= 1L, logi_i))
expect_equal(or3s(kqf %in% c(4L, -3L, 2L, 7L, 8L), w >= 9L, logi_y),
             bor3(kqf %in% c(4L, -3L, 2L, 7L, 8L), w >= 9L, logi_y))
expect_equal(or3s(kqg %in% 1:4, , ),
             bor3(kqg %in% 1:4, , ))
expect_equal(or3s(kqh %in% 1:4, , ),
             bor3(kqh %in% 1:4, , ))
expect_equal(or3s(kqi %in% 1:4, , ),
             bor3(kqi %in% 1:4, , ))
expect_equal(or3s(kqj %in% 1:4, , logi_b),
             bor3(kqj %in% 1:4, , logi_b))
expect_equal(or3s(kqk %in% 1:4, , logi_o),
             bor3(kqk %in% 1:4, , logi_o))
expect_equal(or3s(kql %in% 1:4, , logi_m),
             bor3(kql %in% 1:4, , logi_m))
expect_equal(or3s(kqm %in% 1:4, e >= 0L, ),
             bor3(kqm %in% 1:4, e >= 0L, ))
expect_equal(or3s(kqn %in% 1:4, s >= 1L, ),
             bor3(kqn %in% 1:4, s >= 1L, ))
expect_equal(or3s(kqo %in% 1:4, d >= 9L, ),
             bor3(kqo %in% 1:4, d >= 9L, ))
expect_equal(or3s(kqp %in% 1:4, u >= 0L, logi_n),
             bor3(kqp %in% 1:4, u >= 0L, logi_n))
expect_equal(or3s(kqq %in% 1:4, s >= 1L, logi_k),
             bor3(kqq %in% 1:4, s >= 1L, logi_k))
expect_equal(or3s(kqr %in% 1:4, h >= 9L, logi_n),
             bor3(kqr %in% 1:4, h >= 9L, logi_n))
expect_equal(or3s(kqs %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kqs %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kqt %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kqt %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kqu %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kqu %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kqv %in% c(4L, -3L, 2L, 7L, 8L), , !logi_b),
             bor3(kqv %in% c(4L, -3L, 2L, 7L, 8L), , !logi_b))
expect_equal(or3s(kqw %in% c(4L, -3L, 2L, 7L, 8L), , !logi_m),
             bor3(kqw %in% c(4L, -3L, 2L, 7L, 8L), , !logi_m))
expect_equal(or3s(kqx %in% c(4L, -3L, 2L, 7L, 8L), , !logi_x),
             bor3(kqx %in% c(4L, -3L, 2L, 7L, 8L), , !logi_x))
expect_equal(or3s(kqy %in% c(4L, -3L, 2L, 7L, 8L), r >= 0L, ),
             bor3(kqy %in% c(4L, -3L, 2L, 7L, 8L), r >= 0L, ))
expect_equal(or3s(kqz %in% c(4L, -3L, 2L, 7L, 8L), f >= 1L, ),
             bor3(kqz %in% c(4L, -3L, 2L, 7L, 8L), f >= 1L, ))
expect_equal(or3s(kra %in% c(4L, -3L, 2L, 7L, 8L), l >= 9L, ),
             bor3(kra %in% c(4L, -3L, 2L, 7L, 8L), l >= 9L, ))
expect_equal(or3s(krb %in% c(4L, -3L, 2L, 7L, 8L), q >= 0L, !logi_f),
             bor3(krb %in% c(4L, -3L, 2L, 7L, 8L), q >= 0L, !logi_f))
expect_equal(or3s(krc %in% c(4L, -3L, 2L, 7L, 8L), e >= 1L, !logi_l),
             bor3(krc %in% c(4L, -3L, 2L, 7L, 8L), e >= 1L, !logi_l))
expect_equal(or3s(krd %in% c(4L, -3L, 2L, 7L, 8L), t >= 9L, !logi_t),
             bor3(krd %in% c(4L, -3L, 2L, 7L, 8L), t >= 9L, !logi_t))
expect_equal(or3s(kre %in% 1:4, , ),
             bor3(kre %in% 1:4, , ))
expect_equal(or3s(krf %in% 1:4, , ),
             bor3(krf %in% 1:4, , ))
expect_equal(or3s(krg %in% 1:4, , ),
             bor3(krg %in% 1:4, , ))
expect_equal(or3s(krh %in% 1:4, , !logi_d),
             bor3(krh %in% 1:4, , !logi_d))
expect_equal(or3s(kri %in% 1:4, , !logi_r),
             bor3(kri %in% 1:4, , !logi_r))
expect_equal(or3s(krj %in% 1:4, , !logi_n),
             bor3(krj %in% 1:4, , !logi_n))
expect_equal(or3s(krk %in% 1:4, q >= 0L, ),
             bor3(krk %in% 1:4, q >= 0L, ))
expect_equal(or3s(krl %in% 1:4, z >= 1L, ),
             bor3(krl %in% 1:4, z >= 1L, ))
expect_equal(or3s(krm %in% 1:4, j >= 9L, ),
             bor3(krm %in% 1:4, j >= 9L, ))
expect_equal(or3s(krn %in% 1:4, b >= 0L, !logi_q),
             bor3(krn %in% 1:4, b >= 0L, !logi_q))
expect_equal(or3s(kro %in% 1:4, q >= 1L, !logi_y),
             bor3(kro %in% 1:4, q >= 1L, !logi_y))
expect_equal(or3s(krp %in% 1:4, f >= 9L, !logi_z),
             bor3(krp %in% 1:4, f >= 9L, !logi_z))
expect_equal(or3s(krq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(krq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(krr %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(krr %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(krs %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(krs %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(krt %in% c(4L, -3L, 2L, 7L, 8L), , o != 0L),
             bor3(krt %in% c(4L, -3L, 2L, 7L, 8L), , o != 0L))
expect_equal(or3s(kru %in% c(4L, -3L, 2L, 7L, 8L), , e != 1L),
             bor3(kru %in% c(4L, -3L, 2L, 7L, 8L), , e != 1L))
expect_equal(or3s(krv %in% c(4L, -3L, 2L, 7L, 8L), , z != 9L),
             bor3(krv %in% c(4L, -3L, 2L, 7L, 8L), , z != 9L))
expect_equal(or3s(krw %in% c(4L, -3L, 2L, 7L, 8L), k >= 0L, ),
             bor3(krw %in% c(4L, -3L, 2L, 7L, 8L), k >= 0L, ))
expect_equal(or3s(krx %in% c(4L, -3L, 2L, 7L, 8L), r >= 1L, ),
             bor3(krx %in% c(4L, -3L, 2L, 7L, 8L), r >= 1L, ))
expect_equal(or3s(kry %in% c(4L, -3L, 2L, 7L, 8L), l >= 9L, ),
             bor3(kry %in% c(4L, -3L, 2L, 7L, 8L), l >= 9L, ))
expect_equal(or3s(krz %in% c(4L, -3L, 2L, 7L, 8L), q >= 0L, c != 0L),
             bor3(krz %in% c(4L, -3L, 2L, 7L, 8L), q >= 0L, c != 0L))
expect_equal(or3s(ksa %in% c(4L, -3L, 2L, 7L, 8L), m >= 1L, y != 1L),
             bor3(ksa %in% c(4L, -3L, 2L, 7L, 8L), m >= 1L, y != 1L))
expect_equal(or3s(ksb %in% c(4L, -3L, 2L, 7L, 8L), p >= 9L, f != 9L),
             bor3(ksb %in% c(4L, -3L, 2L, 7L, 8L), p >= 9L, f != 9L))
expect_equal(or3s(ksc %in% 1:4, , ),
             bor3(ksc %in% 1:4, , ))
expect_equal(or3s(ksd %in% 1:4, , ),
             bor3(ksd %in% 1:4, , ))
expect_equal(or3s(kse %in% 1:4, , ),
             bor3(kse %in% 1:4, , ))
expect_equal(or3s(ksf %in% 1:4, , g != 0L),
             bor3(ksf %in% 1:4, , g != 0L))
expect_equal(or3s(ksg %in% 1:4, , q != 1L),
             bor3(ksg %in% 1:4, , q != 1L))
expect_equal(or3s(ksh %in% 1:4, , s != 9L),
             bor3(ksh %in% 1:4, , s != 9L))
expect_equal(or3s(ksi %in% 1:4, y >= 0L, ),
             bor3(ksi %in% 1:4, y >= 0L, ))
expect_equal(or3s(ksj %in% 1:4, f >= 1L, ),
             bor3(ksj %in% 1:4, f >= 1L, ))
expect_equal(or3s(ksk %in% 1:4, b >= 9L, ),
             bor3(ksk %in% 1:4, b >= 9L, ))
expect_equal(or3s(ksl %in% 1:4, v >= 0L, r != 0L),
             bor3(ksl %in% 1:4, v >= 0L, r != 0L))
expect_equal(or3s(ksm %in% 1:4, d >= 1L, n != 1L),
             bor3(ksm %in% 1:4, d >= 1L, n != 1L))
expect_equal(or3s(ksn %in% 1:4, b >= 9L, u != 9L),
             bor3(ksn %in% 1:4, b >= 9L, u != 9L))
expect_equal(or3s(kso %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kso %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ksp %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ksp %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ksq %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ksq %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ksr %in% c(4L, -3L, 2L, 7L, 8L), , b %between% c(-1L, 1L)),
             bor3(ksr %in% c(4L, -3L, 2L, 7L, 8L), , b %between% c(-1L, 1L)))
expect_equal(or3s(kss %in% c(4L, -3L, 2L, 7L, 8L), , p %between% c(-1L, 1L)),
             bor3(kss %in% c(4L, -3L, 2L, 7L, 8L), , p %between% c(-1L, 1L)))
expect_equal(or3s(kst %in% c(4L, -3L, 2L, 7L, 8L), , y %between% c(-1L, 1L)),
             bor3(kst %in% c(4L, -3L, 2L, 7L, 8L), , y %between% c(-1L, 1L)))
expect_equal(or3s(ksu %in% c(4L, -3L, 2L, 7L, 8L), v >= 0L, ),
             bor3(ksu %in% c(4L, -3L, 2L, 7L, 8L), v >= 0L, ))
expect_equal(or3s(ksv %in% c(4L, -3L, 2L, 7L, 8L), o >= 1L, ),
             bor3(ksv %in% c(4L, -3L, 2L, 7L, 8L), o >= 1L, ))
expect_equal(or3s(ksw %in% c(4L, -3L, 2L, 7L, 8L), m >= 9L, ),
             bor3(ksw %in% c(4L, -3L, 2L, 7L, 8L), m >= 9L, ))
expect_equal(or3s(ksx %in% c(4L, -3L, 2L, 7L, 8L), e >= 0L, e %between% c(-1L, 1L)),
             bor3(ksx %in% c(4L, -3L, 2L, 7L, 8L), e >= 0L, e %between% c(-1L, 1L)))
expect_equal(or3s(ksy %in% c(4L, -3L, 2L, 7L, 8L), e >= 1L, t %between% c(-1L, 1L)),
             bor3(ksy %in% c(4L, -3L, 2L, 7L, 8L), e >= 1L, t %between% c(-1L, 1L)))
expect_equal(or3s(ksz %in% c(4L, -3L, 2L, 7L, 8L), w >= 9L, a %between% c(-1L, 1L)),
             bor3(ksz %in% c(4L, -3L, 2L, 7L, 8L), w >= 9L, a %between% c(-1L, 1L)))
expect_equal(or3s(kta %in% 1:4, , ),
             bor3(kta %in% 1:4, , ))
expect_equal(or3s(ktb %in% 1:4, , ),
             bor3(ktb %in% 1:4, , ))
expect_equal(or3s(ktc %in% 1:4, , ),
             bor3(ktc %in% 1:4, , ))
expect_equal(or3s(ktd %in% 1:4, , a %between% c(-1L, 1L)),
             bor3(ktd %in% 1:4, , a %between% c(-1L, 1L)))
expect_equal(or3s(kte %in% 1:4, , u %between% c(-1L, 1L)),
             bor3(kte %in% 1:4, , u %between% c(-1L, 1L)))
expect_equal(or3s(ktf %in% 1:4, , n %between% c(-1L, 1L)),
             bor3(ktf %in% 1:4, , n %between% c(-1L, 1L)))
expect_equal(or3s(ktg %in% 1:4, f >= 0L, ),
             bor3(ktg %in% 1:4, f >= 0L, ))
expect_equal(or3s(kth %in% 1:4, l >= 1L, ),
             bor3(kth %in% 1:4, l >= 1L, ))
expect_equal(or3s(kti %in% 1:4, l >= 9L, ),
             bor3(kti %in% 1:4, l >= 9L, ))
expect_equal(or3s(ktj %in% 1:4, l >= 0L, f %between% c(-1L, 1L)),
             bor3(ktj %in% 1:4, l >= 0L, f %between% c(-1L, 1L)))
expect_equal(or3s(ktk %in% 1:4, g >= 1L, a %between% c(-1L, 1L)),
             bor3(ktk %in% 1:4, g >= 1L, a %between% c(-1L, 1L)))
expect_equal(or3s(ktl %in% 1:4, m >= 9L, r %between% c(-1L, 1L)),
             bor3(ktl %in% 1:4, m >= 9L, r %between% c(-1L, 1L)))
expect_equal(or3s(ktm %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ktm %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ktn %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(ktn %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kto %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kto %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(ktp %in% c(4L, -3L, 2L, 7L, 8L), , u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ktp %in% c(4L, -3L, 2L, 7L, 8L), , u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ktq %in% c(4L, -3L, 2L, 7L, 8L), , k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ktq %in% c(4L, -3L, 2L, 7L, 8L), , k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ktr %in% c(4L, -3L, 2L, 7L, 8L), , y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ktr %in% c(4L, -3L, 2L, 7L, 8L), , y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kts %in% c(4L, -3L, 2L, 7L, 8L), c >= 0L, ),
             bor3(kts %in% c(4L, -3L, 2L, 7L, 8L), c >= 0L, ))
expect_equal(or3s(ktt %in% c(4L, -3L, 2L, 7L, 8L), n >= 1L, ),
             bor3(ktt %in% c(4L, -3L, 2L, 7L, 8L), n >= 1L, ))
expect_equal(or3s(ktu %in% c(4L, -3L, 2L, 7L, 8L), p >= 9L, ),
             bor3(ktu %in% c(4L, -3L, 2L, 7L, 8L), p >= 9L, ))
expect_equal(or3s(ktv %in% c(4L, -3L, 2L, 7L, 8L), g >= 0L, k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ktv %in% c(4L, -3L, 2L, 7L, 8L), g >= 0L, k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ktw %in% c(4L, -3L, 2L, 7L, 8L), k >= 1L, y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ktw %in% c(4L, -3L, 2L, 7L, 8L), k >= 1L, y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ktx %in% c(4L, -3L, 2L, 7L, 8L), j >= 9L, t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ktx %in% c(4L, -3L, 2L, 7L, 8L), j >= 9L, t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(kty %in% 1:4, , ),
             bor3(kty %in% 1:4, , ))
expect_equal(or3s(ktz %in% 1:4, , ),
             bor3(ktz %in% 1:4, , ))
expect_equal(or3s(kua %in% 1:4, , ),
             bor3(kua %in% 1:4, , ))
expect_equal(or3s(kub %in% 1:4, , t %in% 1:4),
             bor3(kub %in% 1:4, , t %in% 1:4))
expect_equal(or3s(kuc %in% 1:4, , q %in% 1:4),
             bor3(kuc %in% 1:4, , q %in% 1:4))
expect_equal(or3s(kud %in% 1:4, , g %in% 1:4),
             bor3(kud %in% 1:4, , g %in% 1:4))
expect_equal(or3s(kue %in% 1:4, b >= 0L, ),
             bor3(kue %in% 1:4, b >= 0L, ))
expect_equal(or3s(kuf %in% 1:4, v >= 1L, ),
             bor3(kuf %in% 1:4, v >= 1L, ))
expect_equal(or3s(kug %in% 1:4, n >= 9L, ),
             bor3(kug %in% 1:4, n >= 9L, ))
expect_equal(or3s(kuh %in% 1:4, g >= 0L, g %in% 1:4),
             bor3(kuh %in% 1:4, g >= 0L, g %in% 1:4))
expect_equal(or3s(kui %in% 1:4, h >= 1L, y %in% 1:4),
             bor3(kui %in% 1:4, h >= 1L, y %in% 1:4))
expect_equal(or3s(kuj %in% 1:4, n >= 9L, x %in% 1:4),
             bor3(kuj %in% 1:4, n >= 9L, x %in% 1:4))
expect_equal(or3s(kuk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kuk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kul %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kul %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kum %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kum %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kun %in% c(4L, -3L, 2L, 7L, 8L), , q < 0L),
             bor3(kun %in% c(4L, -3L, 2L, 7L, 8L), , q < 0L))
expect_equal(or3s(kuo %in% c(4L, -3L, 2L, 7L, 8L), , r < 1L),
             bor3(kuo %in% c(4L, -3L, 2L, 7L, 8L), , r < 1L))
expect_equal(or3s(kup %in% c(4L, -3L, 2L, 7L, 8L), , f < 9L),
             bor3(kup %in% c(4L, -3L, 2L, 7L, 8L), , f < 9L))
expect_equal(or3s(kuq %in% c(4L, -3L, 2L, 7L, 8L), e >= 0L, ),
             bor3(kuq %in% c(4L, -3L, 2L, 7L, 8L), e >= 0L, ))
expect_equal(or3s(kur %in% c(4L, -3L, 2L, 7L, 8L), y >= 1L, ),
             bor3(kur %in% c(4L, -3L, 2L, 7L, 8L), y >= 1L, ))
expect_equal(or3s(kus %in% c(4L, -3L, 2L, 7L, 8L), x >= 9L, ),
             bor3(kus %in% c(4L, -3L, 2L, 7L, 8L), x >= 9L, ))
expect_equal(or3s(kut %in% c(4L, -3L, 2L, 7L, 8L), t >= 0L, c < 0L),
             bor3(kut %in% c(4L, -3L, 2L, 7L, 8L), t >= 0L, c < 0L))
expect_equal(or3s(kuu %in% c(4L, -3L, 2L, 7L, 8L), l >= 1L, r < 1L),
             bor3(kuu %in% c(4L, -3L, 2L, 7L, 8L), l >= 1L, r < 1L))
expect_equal(or3s(kuv %in% c(4L, -3L, 2L, 7L, 8L), q >= 9L, e < 9L),
             bor3(kuv %in% c(4L, -3L, 2L, 7L, 8L), q >= 9L, e < 9L))
expect_equal(or3s(kuw %in% 1:4, , ),
             bor3(kuw %in% 1:4, , ))
expect_equal(or3s(kux %in% 1:4, , ),
             bor3(kux %in% 1:4, , ))
expect_equal(or3s(kuy %in% 1:4, , ),
             bor3(kuy %in% 1:4, , ))
expect_equal(or3s(kuz %in% 1:4, , x < 0L),
             bor3(kuz %in% 1:4, , x < 0L))
expect_equal(or3s(kva %in% 1:4, , i < 1L),
             bor3(kva %in% 1:4, , i < 1L))
expect_equal(or3s(kvb %in% 1:4, , z < 9L),
             bor3(kvb %in% 1:4, , z < 9L))
expect_equal(or3s(kvc %in% 1:4, x >= 0L, ),
             bor3(kvc %in% 1:4, x >= 0L, ))
expect_equal(or3s(kvd %in% 1:4, p >= 1L, ),
             bor3(kvd %in% 1:4, p >= 1L, ))
expect_equal(or3s(kve %in% 1:4, h >= 9L, ),
             bor3(kve %in% 1:4, h >= 9L, ))
expect_equal(or3s(kvf %in% 1:4, a >= 0L, b < 0L),
             bor3(kvf %in% 1:4, a >= 0L, b < 0L))
expect_equal(or3s(kvg %in% 1:4, r >= 1L, c < 1L),
             bor3(kvg %in% 1:4, r >= 1L, c < 1L))
expect_equal(or3s(kvh %in% 1:4, b >= 9L, d < 9L),
             bor3(kvh %in% 1:4, b >= 9L, d < 9L))
expect_equal(or3s(kvi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kvi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kvj %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kvj %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kvk %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kvk %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kvl %in% c(4L, -3L, 2L, 7L, 8L), , l <= 0L),
             bor3(kvl %in% c(4L, -3L, 2L, 7L, 8L), , l <= 0L))
expect_equal(or3s(kvm %in% c(4L, -3L, 2L, 7L, 8L), , r <= 1L),
             bor3(kvm %in% c(4L, -3L, 2L, 7L, 8L), , r <= 1L))
expect_equal(or3s(kvn %in% c(4L, -3L, 2L, 7L, 8L), , x <= 9L),
             bor3(kvn %in% c(4L, -3L, 2L, 7L, 8L), , x <= 9L))
expect_equal(or3s(kvo %in% c(4L, -3L, 2L, 7L, 8L), e >= 0L, ),
             bor3(kvo %in% c(4L, -3L, 2L, 7L, 8L), e >= 0L, ))
expect_equal(or3s(kvp %in% c(4L, -3L, 2L, 7L, 8L), b >= 1L, ),
             bor3(kvp %in% c(4L, -3L, 2L, 7L, 8L), b >= 1L, ))
expect_equal(or3s(kvq %in% c(4L, -3L, 2L, 7L, 8L), o >= 9L, ),
             bor3(kvq %in% c(4L, -3L, 2L, 7L, 8L), o >= 9L, ))
expect_equal(or3s(kvr %in% c(4L, -3L, 2L, 7L, 8L), z >= 0L, d <= 0L),
             bor3(kvr %in% c(4L, -3L, 2L, 7L, 8L), z >= 0L, d <= 0L))
expect_equal(or3s(kvs %in% c(4L, -3L, 2L, 7L, 8L), c >= 1L, j <= 1L),
             bor3(kvs %in% c(4L, -3L, 2L, 7L, 8L), c >= 1L, j <= 1L))
expect_equal(or3s(kvt %in% c(4L, -3L, 2L, 7L, 8L), a >= 9L, u <= 9L),
             bor3(kvt %in% c(4L, -3L, 2L, 7L, 8L), a >= 9L, u <= 9L))
expect_equal(or3s(kvu %in% 1:4, , ),
             bor3(kvu %in% 1:4, , ))
expect_equal(or3s(kvv %in% 1:4, , ),
             bor3(kvv %in% 1:4, , ))
expect_equal(or3s(kvw %in% 1:4, , ),
             bor3(kvw %in% 1:4, , ))
expect_equal(or3s(kvx %in% 1:4, , m <= 0L),
             bor3(kvx %in% 1:4, , m <= 0L))
expect_equal(or3s(kvy %in% 1:4, , c <= 1L),
             bor3(kvy %in% 1:4, , c <= 1L))
expect_equal(or3s(kvz %in% 1:4, , q <= 9L),
             bor3(kvz %in% 1:4, , q <= 9L))
expect_equal(or3s(kwa %in% 1:4, x >= 0L, ),
             bor3(kwa %in% 1:4, x >= 0L, ))
expect_equal(or3s(kwb %in% 1:4, o >= 1L, ),
             bor3(kwb %in% 1:4, o >= 1L, ))
expect_equal(or3s(kwc %in% 1:4, w >= 9L, ),
             bor3(kwc %in% 1:4, w >= 9L, ))
expect_equal(or3s(kwd %in% 1:4, v >= 0L, o <= 0L),
             bor3(kwd %in% 1:4, v >= 0L, o <= 0L))
expect_equal(or3s(kwe %in% 1:4, k >= 1L, w <= 1L),
             bor3(kwe %in% 1:4, k >= 1L, w <= 1L))
expect_equal(or3s(kwf %in% 1:4, l >= 9L, y <= 9L),
             bor3(kwf %in% 1:4, l >= 9L, y <= 9L))
expect_equal(or3s(kwg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kwg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kwh %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kwh %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kwi %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kwi %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kwj %in% c(4L, -3L, 2L, 7L, 8L), , e == 0L),
             bor3(kwj %in% c(4L, -3L, 2L, 7L, 8L), , e == 0L))
expect_equal(or3s(kwk %in% c(4L, -3L, 2L, 7L, 8L), , r == 1L),
             bor3(kwk %in% c(4L, -3L, 2L, 7L, 8L), , r == 1L))
expect_equal(or3s(kwl %in% c(4L, -3L, 2L, 7L, 8L), , i == 9L),
             bor3(kwl %in% c(4L, -3L, 2L, 7L, 8L), , i == 9L))
expect_equal(or3s(kwm %in% c(4L, -3L, 2L, 7L, 8L), t >= 0L, ),
             bor3(kwm %in% c(4L, -3L, 2L, 7L, 8L), t >= 0L, ))
expect_equal(or3s(kwn %in% c(4L, -3L, 2L, 7L, 8L), s >= 1L, ),
             bor3(kwn %in% c(4L, -3L, 2L, 7L, 8L), s >= 1L, ))
expect_equal(or3s(kwo %in% c(4L, -3L, 2L, 7L, 8L), o >= 9L, ),
             bor3(kwo %in% c(4L, -3L, 2L, 7L, 8L), o >= 9L, ))
expect_equal(or3s(kwp %in% c(4L, -3L, 2L, 7L, 8L), z >= 0L, x == 0L),
             bor3(kwp %in% c(4L, -3L, 2L, 7L, 8L), z >= 0L, x == 0L))
expect_equal(or3s(kwq %in% c(4L, -3L, 2L, 7L, 8L), r >= 1L, t == 1L),
             bor3(kwq %in% c(4L, -3L, 2L, 7L, 8L), r >= 1L, t == 1L))
expect_equal(or3s(kwr %in% c(4L, -3L, 2L, 7L, 8L), j >= 9L, r == 9L),
             bor3(kwr %in% c(4L, -3L, 2L, 7L, 8L), j >= 9L, r == 9L))
expect_equal(or3s(kws %in% 1:4, , ),
             bor3(kws %in% 1:4, , ))
expect_equal(or3s(kwt %in% 1:4, , ),
             bor3(kwt %in% 1:4, , ))
expect_equal(or3s(kwu %in% 1:4, , ),
             bor3(kwu %in% 1:4, , ))
expect_equal(or3s(kwv %in% 1:4, , o == 0L),
             bor3(kwv %in% 1:4, , o == 0L))
expect_equal(or3s(kww %in% 1:4, , j == 1L),
             bor3(kww %in% 1:4, , j == 1L))
expect_equal(or3s(kwx %in% 1:4, , k == 9L),
             bor3(kwx %in% 1:4, , k == 9L))
expect_equal(or3s(kwy %in% 1:4, h >= 0L, ),
             bor3(kwy %in% 1:4, h >= 0L, ))
expect_equal(or3s(kwz %in% 1:4, f >= 1L, ),
             bor3(kwz %in% 1:4, f >= 1L, ))
expect_equal(or3s(kxa %in% 1:4, c >= 9L, ),
             bor3(kxa %in% 1:4, c >= 9L, ))
expect_equal(or3s(kxb %in% 1:4, c >= 0L, v == 0L),
             bor3(kxb %in% 1:4, c >= 0L, v == 0L))
expect_equal(or3s(kxc %in% 1:4, z >= 1L, e == 1L),
             bor3(kxc %in% 1:4, z >= 1L, e == 1L))
expect_equal(or3s(kxd %in% 1:4, v >= 9L, e == 9L),
             bor3(kxd %in% 1:4, v >= 9L, e == 9L))
expect_equal(or3s(kxe %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kxe %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kxf %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kxf %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kxg %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kxg %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kxh %in% c(4L, -3L, 2L, 7L, 8L), , h > 0L),
             bor3(kxh %in% c(4L, -3L, 2L, 7L, 8L), , h > 0L))
expect_equal(or3s(kxi %in% c(4L, -3L, 2L, 7L, 8L), , l > 1L),
             bor3(kxi %in% c(4L, -3L, 2L, 7L, 8L), , l > 1L))
expect_equal(or3s(kxj %in% c(4L, -3L, 2L, 7L, 8L), , f > 9L),
             bor3(kxj %in% c(4L, -3L, 2L, 7L, 8L), , f > 9L))
expect_equal(or3s(kxk %in% c(4L, -3L, 2L, 7L, 8L), j >= 0L, ),
             bor3(kxk %in% c(4L, -3L, 2L, 7L, 8L), j >= 0L, ))
expect_equal(or3s(kxl %in% c(4L, -3L, 2L, 7L, 8L), n >= 1L, ),
             bor3(kxl %in% c(4L, -3L, 2L, 7L, 8L), n >= 1L, ))
expect_equal(or3s(kxm %in% c(4L, -3L, 2L, 7L, 8L), h >= 9L, ),
             bor3(kxm %in% c(4L, -3L, 2L, 7L, 8L), h >= 9L, ))
expect_equal(or3s(kxn %in% c(4L, -3L, 2L, 7L, 8L), x >= 0L, e > 0L),
             bor3(kxn %in% c(4L, -3L, 2L, 7L, 8L), x >= 0L, e > 0L))
expect_equal(or3s(kxo %in% c(4L, -3L, 2L, 7L, 8L), n >= 1L, v > 1L),
             bor3(kxo %in% c(4L, -3L, 2L, 7L, 8L), n >= 1L, v > 1L))
expect_equal(or3s(kxp %in% c(4L, -3L, 2L, 7L, 8L), a >= 9L, f > 9L),
             bor3(kxp %in% c(4L, -3L, 2L, 7L, 8L), a >= 9L, f > 9L))
expect_equal(or3s(kxq %in% 1:4, , ),
             bor3(kxq %in% 1:4, , ))
expect_equal(or3s(kxr %in% 1:4, , ),
             bor3(kxr %in% 1:4, , ))
expect_equal(or3s(kxs %in% 1:4, , ),
             bor3(kxs %in% 1:4, , ))
expect_equal(or3s(kxt %in% 1:4, , p > 0L),
             bor3(kxt %in% 1:4, , p > 0L))
expect_equal(or3s(kxu %in% 1:4, , w > 1L),
             bor3(kxu %in% 1:4, , w > 1L))
expect_equal(or3s(kxv %in% 1:4, , a > 9L),
             bor3(kxv %in% 1:4, , a > 9L))
expect_equal(or3s(kxw %in% 1:4, j >= 0L, ),
             bor3(kxw %in% 1:4, j >= 0L, ))
expect_equal(or3s(kxx %in% 1:4, d >= 1L, ),
             bor3(kxx %in% 1:4, d >= 1L, ))
expect_equal(or3s(kxy %in% 1:4, t >= 9L, ),
             bor3(kxy %in% 1:4, t >= 9L, ))
expect_equal(or3s(kxz %in% 1:4, l >= 0L, z > 0L),
             bor3(kxz %in% 1:4, l >= 0L, z > 0L))
expect_equal(or3s(kya %in% 1:4, r >= 1L, m > 1L),
             bor3(kya %in% 1:4, r >= 1L, m > 1L))
expect_equal(or3s(kyb %in% 1:4, z >= 9L, a > 9L),
             bor3(kyb %in% 1:4, z >= 9L, a > 9L))
expect_equal(or3s(kyc %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kyc %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kyd %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kyd %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kye %in% c(4L, -3L, 2L, 7L, 8L), , ),
             bor3(kye %in% c(4L, -3L, 2L, 7L, 8L), , ))
expect_equal(or3s(kyf %in% c(4L, -3L, 2L, 7L, 8L), , i >= 0L),
             bor3(kyf %in% c(4L, -3L, 2L, 7L, 8L), , i >= 0L))
expect_equal(or3s(kyg %in% c(4L, -3L, 2L, 7L, 8L), , y >= 1L),
             bor3(kyg %in% c(4L, -3L, 2L, 7L, 8L), , y >= 1L))
expect_equal(or3s(kyh %in% c(4L, -3L, 2L, 7L, 8L), , k >= 9L),
             bor3(kyh %in% c(4L, -3L, 2L, 7L, 8L), , k >= 9L))
expect_equal(or3s(kyi %in% c(4L, -3L, 2L, 7L, 8L), y >= 0L, ),
             bor3(kyi %in% c(4L, -3L, 2L, 7L, 8L), y >= 0L, ))
expect_equal(or3s(kyj %in% c(4L, -3L, 2L, 7L, 8L), l >= 1L, ),
             bor3(kyj %in% c(4L, -3L, 2L, 7L, 8L), l >= 1L, ))
expect_equal(or3s(kyk %in% c(4L, -3L, 2L, 7L, 8L), y >= 9L, ),
             bor3(kyk %in% c(4L, -3L, 2L, 7L, 8L), y >= 9L, ))
expect_equal(or3s(kyl %in% c(4L, -3L, 2L, 7L, 8L), l >= 0L, p >= 0L),
             bor3(kyl %in% c(4L, -3L, 2L, 7L, 8L), l >= 0L, p >= 0L))
expect_equal(or3s(kym %in% c(4L, -3L, 2L, 7L, 8L), d >= 1L, c >= 1L),
             bor3(kym %in% c(4L, -3L, 2L, 7L, 8L), d >= 1L, c >= 1L))
expect_equal(or3s(kyn %in% c(4L, -3L, 2L, 7L, 8L), r >= 9L, p >= 9L),
             bor3(kyn %in% c(4L, -3L, 2L, 7L, 8L), r >= 9L, p >= 9L))
expect_equal(or3s(kyo %in% 1:4, , ),
             bor3(kyo %in% 1:4, , ))
expect_equal(or3s(kyp %in% 1:4, , ),
             bor3(kyp %in% 1:4, , ))
expect_equal(or3s(kyq %in% 1:4, , ),
             bor3(kyq %in% 1:4, , ))
expect_equal(or3s(kyr %in% 1:4, , x >= 0L),
             bor3(kyr %in% 1:4, , x >= 0L))
expect_equal(or3s(kys %in% 1:4, , k >= 1L),
             bor3(kys %in% 1:4, , k >= 1L))
expect_equal(or3s(kyt %in% 1:4, , k >= 9L),
             bor3(kyt %in% 1:4, , k >= 9L))
expect_equal(or3s(kyu %in% 1:4, c >= 0L, ),
             bor3(kyu %in% 1:4, c >= 0L, ))
expect_equal(or3s(kyv %in% 1:4, h >= 1L, ),
             bor3(kyv %in% 1:4, h >= 1L, ))
expect_equal(or3s(kyw %in% 1:4, g >= 9L, ),
             bor3(kyw %in% 1:4, g >= 9L, ))
expect_equal(or3s(kyx %in% 1:4, b >= 0L, t >= 0L),
             bor3(kyx %in% 1:4, b >= 0L, t >= 0L))
expect_equal(or3s(kyy %in% 1:4, t >= 1L, t >= 1L),
             bor3(kyy %in% 1:4, t >= 1L, t >= 1L))
expect_equal(or3s(kyz %in% 1:4, s >= 9L, f >= 9L),
             bor3(kyz %in% 1:4, s >= 9L, f >= 9L))
expect_equal(or3s(la < 0L, , ),
             bor3(la < 0L, , ))
expect_equal(or3s(lb < 1L, , ),
             bor3(lb < 1L, , ))
expect_equal(or3s(lc < 9L, , ),
             bor3(lc < 9L, , ))
expect_equal(or3s(ld < 0L, , logi_z),
             bor3(ld < 0L, , logi_z))
expect_equal(or3s(le < 1L, , logi_v),
             bor3(le < 1L, , logi_v))
expect_equal(or3s(lf < 9L, , logi_o),
             bor3(lf < 9L, , logi_o))
expect_equal(or3s(lg < 0L, logi_j, ),
             bor3(lg < 0L, logi_j, ))
expect_equal(or3s(lh < 1L, logi_z, ),
             bor3(lh < 1L, logi_z, ))
expect_equal(or3s(li < 9L, logi_x, ),
             bor3(li < 9L, logi_x, ))
expect_equal(or3s(lj < 0L, logi_v, logi_j),
             bor3(lj < 0L, logi_v, logi_j))
expect_equal(or3s(lk < 1L, logi_k, logi_l),
             bor3(lk < 1L, logi_k, logi_l))
expect_equal(or3s(ll < 9L, logi_m, logi_c),
             bor3(ll < 9L, logi_m, logi_c))
expect_equal(or3s(lm < 0L, , ),
             bor3(lm < 0L, , ))
expect_equal(or3s(ln < 1L, , ),
             bor3(ln < 1L, , ))
expect_equal(or3s(lo < 9L, , ),
             bor3(lo < 9L, , ))
expect_equal(or3s(lp < 0L, , !logi_j),
             bor3(lp < 0L, , !logi_j))
expect_equal(or3s(lq < 1L, , !logi_z),
             bor3(lq < 1L, , !logi_z))
expect_equal(or3s(lr < 9L, , !logi_j),
             bor3(lr < 9L, , !logi_j))
expect_equal(or3s(ls < 0L, logi_h, ),
             bor3(ls < 0L, logi_h, ))
expect_equal(or3s(lt < 1L, logi_u, ),
             bor3(lt < 1L, logi_u, ))
expect_equal(or3s(lu < 9L, logi_w, ),
             bor3(lu < 9L, logi_w, ))
expect_equal(or3s(lv < 0L, logi_v, !logi_p),
             bor3(lv < 0L, logi_v, !logi_p))
expect_equal(or3s(lw < 1L, logi_q, !logi_g),
             bor3(lw < 1L, logi_q, !logi_g))
expect_equal(or3s(lx < 9L, logi_t, !logi_u),
             bor3(lx < 9L, logi_t, !logi_u))
expect_equal(or3s(ly < 0L, , ),
             bor3(ly < 0L, , ))
expect_equal(or3s(lz < 1L, , ),
             bor3(lz < 1L, , ))
expect_equal(or3s(laa < 9L, , ),
             bor3(laa < 9L, , ))
expect_equal(or3s(lab < 0L, , h != 0L),
             bor3(lab < 0L, , h != 0L))
expect_equal(or3s(lac < 1L, , n != 1L),
             bor3(lac < 1L, , n != 1L))
expect_equal(or3s(lad < 9L, , r != 9L),
             bor3(lad < 9L, , r != 9L))
expect_equal(or3s(lae < 0L, logi_g, ),
             bor3(lae < 0L, logi_g, ))
expect_equal(or3s(laf < 1L, logi_v, ),
             bor3(laf < 1L, logi_v, ))
expect_equal(or3s(lag < 9L, logi_z, ),
             bor3(lag < 9L, logi_z, ))
expect_equal(or3s(lah < 0L, logi_e, h != 0L),
             bor3(lah < 0L, logi_e, h != 0L))
expect_equal(or3s(lai < 1L, logi_z, l != 1L),
             bor3(lai < 1L, logi_z, l != 1L))
expect_equal(or3s(laj < 9L, logi_t, y != 9L),
             bor3(laj < 9L, logi_t, y != 9L))
expect_equal(or3s(lak < 0L, , ),
             bor3(lak < 0L, , ))
expect_equal(or3s(lal < 1L, , ),
             bor3(lal < 1L, , ))
expect_equal(or3s(lam < 9L, , ),
             bor3(lam < 9L, , ))
expect_equal(or3s(lan < 0L, , o %between% c(-1L, 1L)),
             bor3(lan < 0L, , o %between% c(-1L, 1L)))
expect_equal(or3s(lao < 1L, , y %between% c(-1L, 1L)),
             bor3(lao < 1L, , y %between% c(-1L, 1L)))
expect_equal(or3s(lap < 9L, , c %between% c(-1L, 1L)),
             bor3(lap < 9L, , c %between% c(-1L, 1L)))
expect_equal(or3s(laq < 0L, logi_u, ),
             bor3(laq < 0L, logi_u, ))
expect_equal(or3s(lar < 1L, logi_u, ),
             bor3(lar < 1L, logi_u, ))
expect_equal(or3s(las < 9L, logi_r, ),
             bor3(las < 9L, logi_r, ))
expect_equal(or3s(lat < 0L, logi_x, v %between% c(-1L, 1L)),
             bor3(lat < 0L, logi_x, v %between% c(-1L, 1L)))
expect_equal(or3s(lau < 1L, logi_k, v %between% c(-1L, 1L)),
             bor3(lau < 1L, logi_k, v %between% c(-1L, 1L)))
expect_equal(or3s(lav < 9L, logi_d, c %between% c(-1L, 1L)),
             bor3(lav < 9L, logi_d, c %between% c(-1L, 1L)))
expect_equal(or3s(law < 0L, , ),
             bor3(law < 0L, , ))
expect_equal(or3s(lax < 1L, , ),
             bor3(lax < 1L, , ))
expect_equal(or3s(lay < 9L, , ),
             bor3(lay < 9L, , ))
expect_equal(or3s(laz < 0L, , y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(laz < 0L, , y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lba < 1L, , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lba < 1L, , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lbb < 9L, , u %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lbb < 9L, , u %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lbc < 0L, logi_i, ),
             bor3(lbc < 0L, logi_i, ))
expect_equal(or3s(lbd < 1L, logi_n, ),
             bor3(lbd < 1L, logi_n, ))
expect_equal(or3s(lbe < 9L, logi_a, ),
             bor3(lbe < 9L, logi_a, ))
expect_equal(or3s(lbf < 0L, logi_p, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lbf < 0L, logi_p, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lbg < 1L, logi_h, s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lbg < 1L, logi_h, s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lbh < 9L, logi_b, m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lbh < 9L, logi_b, m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lbi < 0L, , ),
             bor3(lbi < 0L, , ))
expect_equal(or3s(lbj < 1L, , ),
             bor3(lbj < 1L, , ))
expect_equal(or3s(lbk < 9L, , ),
             bor3(lbk < 9L, , ))
expect_equal(or3s(lbl < 0L, , y %in% 1:4),
             bor3(lbl < 0L, , y %in% 1:4))
expect_equal(or3s(lbm < 1L, , h %in% 1:4),
             bor3(lbm < 1L, , h %in% 1:4))
expect_equal(or3s(lbn < 9L, , y %in% 1:4),
             bor3(lbn < 9L, , y %in% 1:4))
expect_equal(or3s(lbo < 0L, logi_o, ),
             bor3(lbo < 0L, logi_o, ))
expect_equal(or3s(lbp < 1L, logi_n, ),
             bor3(lbp < 1L, logi_n, ))
expect_equal(or3s(lbq < 9L, logi_j, ),
             bor3(lbq < 9L, logi_j, ))
expect_equal(or3s(lbr < 0L, logi_m, e %in% 1:4),
             bor3(lbr < 0L, logi_m, e %in% 1:4))
expect_equal(or3s(lbs < 1L, logi_d, k %in% 1:4),
             bor3(lbs < 1L, logi_d, k %in% 1:4))
expect_equal(or3s(lbt < 9L, logi_y, i %in% 1:4),
             bor3(lbt < 9L, logi_y, i %in% 1:4))
expect_equal(or3s(lbu < 0L, , ),
             bor3(lbu < 0L, , ))
expect_equal(or3s(lbv < 1L, , ),
             bor3(lbv < 1L, , ))
expect_equal(or3s(lbw < 9L, , ),
             bor3(lbw < 9L, , ))
expect_equal(or3s(lbx < 0L, , g < 0L),
             bor3(lbx < 0L, , g < 0L))
expect_equal(or3s(lby < 1L, , t < 1L),
             bor3(lby < 1L, , t < 1L))
expect_equal(or3s(lbz < 9L, , h < 9L),
             bor3(lbz < 9L, , h < 9L))
expect_equal(or3s(lca < 0L, logi_q, ),
             bor3(lca < 0L, logi_q, ))
expect_equal(or3s(lcb < 1L, logi_z, ),
             bor3(lcb < 1L, logi_z, ))
expect_equal(or3s(lcc < 9L, logi_v, ),
             bor3(lcc < 9L, logi_v, ))
expect_equal(or3s(lcd < 0L, logi_x, m < 0L),
             bor3(lcd < 0L, logi_x, m < 0L))
expect_equal(or3s(lce < 1L, logi_t, m < 1L),
             bor3(lce < 1L, logi_t, m < 1L))
expect_equal(or3s(lcf < 9L, logi_s, n < 9L),
             bor3(lcf < 9L, logi_s, n < 9L))
expect_equal(or3s(lcg < 0L, , ),
             bor3(lcg < 0L, , ))
expect_equal(or3s(lch < 1L, , ),
             bor3(lch < 1L, , ))
expect_equal(or3s(lci < 9L, , ),
             bor3(lci < 9L, , ))
expect_equal(or3s(lcj < 0L, , d <= 0L),
             bor3(lcj < 0L, , d <= 0L))
expect_equal(or3s(lck < 1L, , h <= 1L),
             bor3(lck < 1L, , h <= 1L))
expect_equal(or3s(lcl < 9L, , d <= 9L),
             bor3(lcl < 9L, , d <= 9L))
expect_equal(or3s(lcm < 0L, logi_k, ),
             bor3(lcm < 0L, logi_k, ))
expect_equal(or3s(lcn < 1L, logi_j, ),
             bor3(lcn < 1L, logi_j, ))
expect_equal(or3s(lco < 9L, logi_k, ),
             bor3(lco < 9L, logi_k, ))
expect_equal(or3s(lcp < 0L, logi_k, i <= 0L),
             bor3(lcp < 0L, logi_k, i <= 0L))
expect_equal(or3s(lcq < 1L, logi_b, s <= 1L),
             bor3(lcq < 1L, logi_b, s <= 1L))
expect_equal(or3s(lcr < 9L, logi_i, s <= 9L),
             bor3(lcr < 9L, logi_i, s <= 9L))
expect_equal(or3s(lcs < 0L, , ),
             bor3(lcs < 0L, , ))
expect_equal(or3s(lct < 1L, , ),
             bor3(lct < 1L, , ))
expect_equal(or3s(lcu < 9L, , ),
             bor3(lcu < 9L, , ))
expect_equal(or3s(lcv < 0L, , n == 0L),
             bor3(lcv < 0L, , n == 0L))
expect_equal(or3s(lcw < 1L, , n == 1L),
             bor3(lcw < 1L, , n == 1L))
expect_equal(or3s(lcx < 9L, , d == 9L),
             bor3(lcx < 9L, , d == 9L))
expect_equal(or3s(lcy < 0L, logi_m, ),
             bor3(lcy < 0L, logi_m, ))
expect_equal(or3s(lcz < 1L, logi_x, ),
             bor3(lcz < 1L, logi_x, ))
expect_equal(or3s(lda < 9L, logi_q, ),
             bor3(lda < 9L, logi_q, ))
expect_equal(or3s(ldb < 0L, logi_m, b == 0L),
             bor3(ldb < 0L, logi_m, b == 0L))
expect_equal(or3s(ldc < 1L, logi_j, o == 1L),
             bor3(ldc < 1L, logi_j, o == 1L))
expect_equal(or3s(ldd < 9L, logi_a, h == 9L),
             bor3(ldd < 9L, logi_a, h == 9L))
expect_equal(or3s(lde < 0L, , ),
             bor3(lde < 0L, , ))
expect_equal(or3s(ldf < 1L, , ),
             bor3(ldf < 1L, , ))
expect_equal(or3s(ldg < 9L, , ),
             bor3(ldg < 9L, , ))
expect_equal(or3s(ldh < 0L, , c > 0L),
             bor3(ldh < 0L, , c > 0L))
expect_equal(or3s(ldi < 1L, , v > 1L),
             bor3(ldi < 1L, , v > 1L))
expect_equal(or3s(ldj < 9L, , u > 9L),
             bor3(ldj < 9L, , u > 9L))
expect_equal(or3s(ldk < 0L, logi_t, ),
             bor3(ldk < 0L, logi_t, ))
expect_equal(or3s(ldl < 1L, logi_a, ),
             bor3(ldl < 1L, logi_a, ))
expect_equal(or3s(ldm < 9L, logi_m, ),
             bor3(ldm < 9L, logi_m, ))
expect_equal(or3s(ldn < 0L, logi_b, p > 0L),
             bor3(ldn < 0L, logi_b, p > 0L))
expect_equal(or3s(ldo < 1L, logi_d, j > 1L),
             bor3(ldo < 1L, logi_d, j > 1L))
expect_equal(or3s(ldp < 9L, logi_e, s > 9L),
             bor3(ldp < 9L, logi_e, s > 9L))
expect_equal(or3s(ldq < 0L, , ),
             bor3(ldq < 0L, , ))
expect_equal(or3s(ldr < 1L, , ),
             bor3(ldr < 1L, , ))
expect_equal(or3s(lds < 9L, , ),
             bor3(lds < 9L, , ))
expect_equal(or3s(ldt < 0L, , s >= 0L),
             bor3(ldt < 0L, , s >= 0L))
expect_equal(or3s(ldu < 1L, , z >= 1L),
             bor3(ldu < 1L, , z >= 1L))
expect_equal(or3s(ldv < 9L, , j >= 9L),
             bor3(ldv < 9L, , j >= 9L))
expect_equal(or3s(ldw < 0L, logi_f, ),
             bor3(ldw < 0L, logi_f, ))
expect_equal(or3s(ldx < 1L, logi_h, ),
             bor3(ldx < 1L, logi_h, ))
expect_equal(or3s(ldy < 9L, logi_w, ),
             bor3(ldy < 9L, logi_w, ))
expect_equal(or3s(ldz < 0L, logi_x, c >= 0L),
             bor3(ldz < 0L, logi_x, c >= 0L))
expect_equal(or3s(lea < 1L, logi_j, d >= 1L),
             bor3(lea < 1L, logi_j, d >= 1L))
expect_equal(or3s(leb < 9L, logi_v, s >= 9L),
             bor3(leb < 9L, logi_v, s >= 9L))
expect_equal(or3s(lec < 0L, , ),
             bor3(lec < 0L, , ))
expect_equal(or3s(led < 1L, , ),
             bor3(led < 1L, , ))
expect_equal(or3s(lee < 9L, , ),
             bor3(lee < 9L, , ))
expect_equal(or3s(lef < 0L, , logi_e),
             bor3(lef < 0L, , logi_e))
expect_equal(or3s(leg < 1L, , logi_i),
             bor3(leg < 1L, , logi_i))
expect_equal(or3s(leh < 9L, , logi_f),
             bor3(leh < 9L, , logi_f))
expect_equal(or3s(lei < 0L, !logi_a, ),
             bor3(lei < 0L, !logi_a, ))
expect_equal(or3s(lej < 1L, !logi_s, ),
             bor3(lej < 1L, !logi_s, ))
expect_equal(or3s(lek < 9L, !logi_t, ),
             bor3(lek < 9L, !logi_t, ))
expect_equal(or3s(lel < 0L, !logi_r, logi_i),
             bor3(lel < 0L, !logi_r, logi_i))
expect_equal(or3s(lem < 1L, !logi_d, logi_j),
             bor3(lem < 1L, !logi_d, logi_j))
expect_equal(or3s(len < 9L, !logi_f, logi_g),
             bor3(len < 9L, !logi_f, logi_g))
expect_equal(or3s(leo < 0L, , ),
             bor3(leo < 0L, , ))
expect_equal(or3s(lep < 1L, , ),
             bor3(lep < 1L, , ))
expect_equal(or3s(leq < 9L, , ),
             bor3(leq < 9L, , ))
expect_equal(or3s(ler < 0L, , !logi_r),
             bor3(ler < 0L, , !logi_r))
expect_equal(or3s(les < 1L, , !logi_q),
             bor3(les < 1L, , !logi_q))
expect_equal(or3s(let < 9L, , !logi_d),
             bor3(let < 9L, , !logi_d))
expect_equal(or3s(leu < 0L, !logi_k, ),
             bor3(leu < 0L, !logi_k, ))
expect_equal(or3s(lev < 1L, !logi_z, ),
             bor3(lev < 1L, !logi_z, ))
expect_equal(or3s(lew < 9L, !logi_i, ),
             bor3(lew < 9L, !logi_i, ))
expect_equal(or3s(lex < 0L, !logi_l, !logi_f),
             bor3(lex < 0L, !logi_l, !logi_f))
expect_equal(or3s(ley < 1L, !logi_d, !logi_p),
             bor3(ley < 1L, !logi_d, !logi_p))
expect_equal(or3s(lez < 9L, !logi_b, !logi_u),
             bor3(lez < 9L, !logi_b, !logi_u))
expect_equal(or3s(lfa < 0L, , ),
             bor3(lfa < 0L, , ))
expect_equal(or3s(lfb < 1L, , ),
             bor3(lfb < 1L, , ))
expect_equal(or3s(lfc < 9L, , ),
             bor3(lfc < 9L, , ))
expect_equal(or3s(lfd < 0L, , n != 0L),
             bor3(lfd < 0L, , n != 0L))
expect_equal(or3s(lfe < 1L, , t != 1L),
             bor3(lfe < 1L, , t != 1L))
expect_equal(or3s(lff < 9L, , r != 9L),
             bor3(lff < 9L, , r != 9L))
expect_equal(or3s(lfg < 0L, !logi_m, ),
             bor3(lfg < 0L, !logi_m, ))
expect_equal(or3s(lfh < 1L, !logi_q, ),
             bor3(lfh < 1L, !logi_q, ))
expect_equal(or3s(lfi < 9L, !logi_m, ),
             bor3(lfi < 9L, !logi_m, ))
expect_equal(or3s(lfj < 0L, !logi_m, a != 0L),
             bor3(lfj < 0L, !logi_m, a != 0L))
expect_equal(or3s(lfk < 1L, !logi_w, k != 1L),
             bor3(lfk < 1L, !logi_w, k != 1L))
expect_equal(or3s(lfl < 9L, !logi_o, v != 9L),
             bor3(lfl < 9L, !logi_o, v != 9L))
expect_equal(or3s(lfm < 0L, , ),
             bor3(lfm < 0L, , ))
expect_equal(or3s(lfn < 1L, , ),
             bor3(lfn < 1L, , ))
expect_equal(or3s(lfo < 9L, , ),
             bor3(lfo < 9L, , ))
expect_equal(or3s(lfp < 0L, , w %between% c(-1L, 1L)),
             bor3(lfp < 0L, , w %between% c(-1L, 1L)))
expect_equal(or3s(lfq < 1L, , l %between% c(-1L, 1L)),
             bor3(lfq < 1L, , l %between% c(-1L, 1L)))
expect_equal(or3s(lfr < 9L, , q %between% c(-1L, 1L)),
             bor3(lfr < 9L, , q %between% c(-1L, 1L)))
expect_equal(or3s(lfs < 0L, !logi_d, ),
             bor3(lfs < 0L, !logi_d, ))
expect_equal(or3s(lft < 1L, !logi_w, ),
             bor3(lft < 1L, !logi_w, ))
expect_equal(or3s(lfu < 9L, !logi_p, ),
             bor3(lfu < 9L, !logi_p, ))
expect_equal(or3s(lfv < 0L, !logi_c, f %between% c(-1L, 1L)),
             bor3(lfv < 0L, !logi_c, f %between% c(-1L, 1L)))
expect_equal(or3s(lfw < 1L, !logi_i, h %between% c(-1L, 1L)),
             bor3(lfw < 1L, !logi_i, h %between% c(-1L, 1L)))
expect_equal(or3s(lfx < 9L, !logi_n, f %between% c(-1L, 1L)),
             bor3(lfx < 9L, !logi_n, f %between% c(-1L, 1L)))
expect_equal(or3s(lfy < 0L, , ),
             bor3(lfy < 0L, , ))
expect_equal(or3s(lfz < 1L, , ),
             bor3(lfz < 1L, , ))
expect_equal(or3s(lga < 9L, , ),
             bor3(lga < 9L, , ))
expect_equal(or3s(lgb < 0L, , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lgb < 0L, , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lgc < 1L, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lgc < 1L, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lgd < 9L, , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lgd < 9L, , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lge < 0L, !logi_k, ),
             bor3(lge < 0L, !logi_k, ))
expect_equal(or3s(lgf < 1L, !logi_d, ),
             bor3(lgf < 1L, !logi_d, ))
expect_equal(or3s(lgg < 9L, !logi_b, ),
             bor3(lgg < 9L, !logi_b, ))
expect_equal(or3s(lgh < 0L, !logi_w, e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lgh < 0L, !logi_w, e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lgi < 1L, !logi_p, p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lgi < 1L, !logi_p, p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lgj < 9L, !logi_j, t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lgj < 9L, !logi_j, t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lgk < 0L, , ),
             bor3(lgk < 0L, , ))
expect_equal(or3s(lgl < 1L, , ),
             bor3(lgl < 1L, , ))
expect_equal(or3s(lgm < 9L, , ),
             bor3(lgm < 9L, , ))
expect_equal(or3s(lgn < 0L, , n %in% 1:4),
             bor3(lgn < 0L, , n %in% 1:4))
expect_equal(or3s(lgo < 1L, , w %in% 1:4),
             bor3(lgo < 1L, , w %in% 1:4))
expect_equal(or3s(lgp < 9L, , k %in% 1:4),
             bor3(lgp < 9L, , k %in% 1:4))
expect_equal(or3s(lgq < 0L, !logi_n, ),
             bor3(lgq < 0L, !logi_n, ))
expect_equal(or3s(lgr < 1L, !logi_t, ),
             bor3(lgr < 1L, !logi_t, ))
expect_equal(or3s(lgs < 9L, !logi_t, ),
             bor3(lgs < 9L, !logi_t, ))
expect_equal(or3s(lgt < 0L, !logi_k, q %in% 1:4),
             bor3(lgt < 0L, !logi_k, q %in% 1:4))
expect_equal(or3s(lgu < 1L, !logi_i, c %in% 1:4),
             bor3(lgu < 1L, !logi_i, c %in% 1:4))
expect_equal(or3s(lgv < 9L, !logi_z, z %in% 1:4),
             bor3(lgv < 9L, !logi_z, z %in% 1:4))
expect_equal(or3s(lgw < 0L, , ),
             bor3(lgw < 0L, , ))
expect_equal(or3s(lgx < 1L, , ),
             bor3(lgx < 1L, , ))
expect_equal(or3s(lgy < 9L, , ),
             bor3(lgy < 9L, , ))
expect_equal(or3s(lgz < 0L, , d < 0L),
             bor3(lgz < 0L, , d < 0L))
expect_equal(or3s(lha < 1L, , j < 1L),
             bor3(lha < 1L, , j < 1L))
expect_equal(or3s(lhb < 9L, , d < 9L),
             bor3(lhb < 9L, , d < 9L))
expect_equal(or3s(lhc < 0L, !logi_h, ),
             bor3(lhc < 0L, !logi_h, ))
expect_equal(or3s(lhd < 1L, !logi_x, ),
             bor3(lhd < 1L, !logi_x, ))
expect_equal(or3s(lhe < 9L, !logi_v, ),
             bor3(lhe < 9L, !logi_v, ))
expect_equal(or3s(lhf < 0L, !logi_t, u < 0L),
             bor3(lhf < 0L, !logi_t, u < 0L))
expect_equal(or3s(lhg < 1L, !logi_h, o < 1L),
             bor3(lhg < 1L, !logi_h, o < 1L))
expect_equal(or3s(lhh < 9L, !logi_i, p < 9L),
             bor3(lhh < 9L, !logi_i, p < 9L))
expect_equal(or3s(lhi < 0L, , ),
             bor3(lhi < 0L, , ))
expect_equal(or3s(lhj < 1L, , ),
             bor3(lhj < 1L, , ))
expect_equal(or3s(lhk < 9L, , ),
             bor3(lhk < 9L, , ))
expect_equal(or3s(lhl < 0L, , y <= 0L),
             bor3(lhl < 0L, , y <= 0L))
expect_equal(or3s(lhm < 1L, , j <= 1L),
             bor3(lhm < 1L, , j <= 1L))
expect_equal(or3s(lhn < 9L, , v <= 9L),
             bor3(lhn < 9L, , v <= 9L))
expect_equal(or3s(lho < 0L, !logi_a, ),
             bor3(lho < 0L, !logi_a, ))
expect_equal(or3s(lhp < 1L, !logi_v, ),
             bor3(lhp < 1L, !logi_v, ))
expect_equal(or3s(lhq < 9L, !logi_r, ),
             bor3(lhq < 9L, !logi_r, ))
expect_equal(or3s(lhr < 0L, !logi_k, x <= 0L),
             bor3(lhr < 0L, !logi_k, x <= 0L))
expect_equal(or3s(lhs < 1L, !logi_n, x <= 1L),
             bor3(lhs < 1L, !logi_n, x <= 1L))
expect_equal(or3s(lht < 9L, !logi_x, b <= 9L),
             bor3(lht < 9L, !logi_x, b <= 9L))
expect_equal(or3s(lhu < 0L, , ),
             bor3(lhu < 0L, , ))
expect_equal(or3s(lhv < 1L, , ),
             bor3(lhv < 1L, , ))
expect_equal(or3s(lhw < 9L, , ),
             bor3(lhw < 9L, , ))
expect_equal(or3s(lhx < 0L, , c == 0L),
             bor3(lhx < 0L, , c == 0L))
expect_equal(or3s(lhy < 1L, , g == 1L),
             bor3(lhy < 1L, , g == 1L))
expect_equal(or3s(lhz < 9L, , e == 9L),
             bor3(lhz < 9L, , e == 9L))
expect_equal(or3s(lia < 0L, !logi_t, ),
             bor3(lia < 0L, !logi_t, ))
expect_equal(or3s(lib < 1L, !logi_c, ),
             bor3(lib < 1L, !logi_c, ))
expect_equal(or3s(lic < 9L, !logi_p, ),
             bor3(lic < 9L, !logi_p, ))
expect_equal(or3s(lid < 0L, !logi_i, m == 0L),
             bor3(lid < 0L, !logi_i, m == 0L))
expect_equal(or3s(lie < 1L, !logi_a, b == 1L),
             bor3(lie < 1L, !logi_a, b == 1L))
expect_equal(or3s(lif < 9L, !logi_u, o == 9L),
             bor3(lif < 9L, !logi_u, o == 9L))
expect_equal(or3s(lig < 0L, , ),
             bor3(lig < 0L, , ))
expect_equal(or3s(lih < 1L, , ),
             bor3(lih < 1L, , ))
expect_equal(or3s(lii < 9L, , ),
             bor3(lii < 9L, , ))
expect_equal(or3s(lij < 0L, , q > 0L),
             bor3(lij < 0L, , q > 0L))
expect_equal(or3s(lik < 1L, , u > 1L),
             bor3(lik < 1L, , u > 1L))
expect_equal(or3s(lil < 9L, , u > 9L),
             bor3(lil < 9L, , u > 9L))
expect_equal(or3s(lim < 0L, !logi_j, ),
             bor3(lim < 0L, !logi_j, ))
expect_equal(or3s(lin < 1L, !logi_s, ),
             bor3(lin < 1L, !logi_s, ))
expect_equal(or3s(lio < 9L, !logi_k, ),
             bor3(lio < 9L, !logi_k, ))
expect_equal(or3s(lip < 0L, !logi_w, s > 0L),
             bor3(lip < 0L, !logi_w, s > 0L))
expect_equal(or3s(liq < 1L, !logi_v, o > 1L),
             bor3(liq < 1L, !logi_v, o > 1L))
expect_equal(or3s(lir < 9L, !logi_g, a > 9L),
             bor3(lir < 9L, !logi_g, a > 9L))
expect_equal(or3s(lis < 0L, , ),
             bor3(lis < 0L, , ))
expect_equal(or3s(lit < 1L, , ),
             bor3(lit < 1L, , ))
expect_equal(or3s(liu < 9L, , ),
             bor3(liu < 9L, , ))
expect_equal(or3s(liv < 0L, , v >= 0L),
             bor3(liv < 0L, , v >= 0L))
expect_equal(or3s(liw < 1L, , a >= 1L),
             bor3(liw < 1L, , a >= 1L))
expect_equal(or3s(lix < 9L, , o >= 9L),
             bor3(lix < 9L, , o >= 9L))
expect_equal(or3s(liy < 0L, !logi_h, ),
             bor3(liy < 0L, !logi_h, ))
expect_equal(or3s(liz < 1L, !logi_x, ),
             bor3(liz < 1L, !logi_x, ))
expect_equal(or3s(lja < 9L, !logi_h, ),
             bor3(lja < 9L, !logi_h, ))
expect_equal(or3s(ljb < 0L, !logi_n, p >= 0L),
             bor3(ljb < 0L, !logi_n, p >= 0L))
expect_equal(or3s(ljc < 1L, !logi_u, i >= 1L),
             bor3(ljc < 1L, !logi_u, i >= 1L))
expect_equal(or3s(ljd < 9L, !logi_x, k >= 9L),
             bor3(ljd < 9L, !logi_x, k >= 9L))
expect_equal(or3s(lje < 0L, , ),
             bor3(lje < 0L, , ))
expect_equal(or3s(ljf < 1L, , ),
             bor3(ljf < 1L, , ))
expect_equal(or3s(ljg < 9L, , ),
             bor3(ljg < 9L, , ))
expect_equal(or3s(ljh < 0L, , logi_l),
             bor3(ljh < 0L, , logi_l))
expect_equal(or3s(lji < 1L, , logi_v),
             bor3(lji < 1L, , logi_v))
expect_equal(or3s(ljj < 9L, , logi_b),
             bor3(ljj < 9L, , logi_b))
expect_equal(or3s(ljk < 0L, c != 0L, ),
             bor3(ljk < 0L, c != 0L, ))
expect_equal(or3s(ljl < 1L, s != 1L, ),
             bor3(ljl < 1L, s != 1L, ))
expect_equal(or3s(ljm < 9L, d != 9L, ),
             bor3(ljm < 9L, d != 9L, ))
expect_equal(or3s(ljn < 0L, z != 0L, logi_u),
             bor3(ljn < 0L, z != 0L, logi_u))
expect_equal(or3s(ljo < 1L, k != 1L, logi_x),
             bor3(ljo < 1L, k != 1L, logi_x))
expect_equal(or3s(ljp < 9L, x != 9L, logi_v),
             bor3(ljp < 9L, x != 9L, logi_v))
expect_equal(or3s(ljq < 0L, , ),
             bor3(ljq < 0L, , ))
expect_equal(or3s(ljr < 1L, , ),
             bor3(ljr < 1L, , ))
expect_equal(or3s(ljs < 9L, , ),
             bor3(ljs < 9L, , ))
expect_equal(or3s(ljt < 0L, , !logi_b),
             bor3(ljt < 0L, , !logi_b))
expect_equal(or3s(lju < 1L, , !logi_p),
             bor3(lju < 1L, , !logi_p))
expect_equal(or3s(ljv < 9L, , !logi_w),
             bor3(ljv < 9L, , !logi_w))
expect_equal(or3s(ljw < 0L, q != 0L, ),
             bor3(ljw < 0L, q != 0L, ))
expect_equal(or3s(ljx < 1L, x != 1L, ),
             bor3(ljx < 1L, x != 1L, ))
expect_equal(or3s(ljy < 9L, q != 9L, ),
             bor3(ljy < 9L, q != 9L, ))
expect_equal(or3s(ljz < 0L, a != 0L, !logi_t),
             bor3(ljz < 0L, a != 0L, !logi_t))
expect_equal(or3s(lka < 1L, f != 1L, !logi_d),
             bor3(lka < 1L, f != 1L, !logi_d))
expect_equal(or3s(lkb < 9L, o != 9L, !logi_o),
             bor3(lkb < 9L, o != 9L, !logi_o))
expect_equal(or3s(lkc < 0L, , ),
             bor3(lkc < 0L, , ))
expect_equal(or3s(lkd < 1L, , ),
             bor3(lkd < 1L, , ))
expect_equal(or3s(lke < 9L, , ),
             bor3(lke < 9L, , ))
expect_equal(or3s(lkf < 0L, , x != 0L),
             bor3(lkf < 0L, , x != 0L))
expect_equal(or3s(lkg < 1L, , z != 1L),
             bor3(lkg < 1L, , z != 1L))
expect_equal(or3s(lkh < 9L, , e != 9L),
             bor3(lkh < 9L, , e != 9L))
expect_equal(or3s(lki < 0L, k != 0L, ),
             bor3(lki < 0L, k != 0L, ))
expect_equal(or3s(lkj < 1L, s != 1L, ),
             bor3(lkj < 1L, s != 1L, ))
expect_equal(or3s(lkk < 9L, j != 9L, ),
             bor3(lkk < 9L, j != 9L, ))
expect_equal(or3s(lkl < 0L, e != 0L, e != 0L),
             bor3(lkl < 0L, e != 0L, e != 0L))
expect_equal(or3s(lkm < 1L, k != 1L, l != 1L),
             bor3(lkm < 1L, k != 1L, l != 1L))
expect_equal(or3s(lkn < 9L, e != 9L, e != 9L),
             bor3(lkn < 9L, e != 9L, e != 9L))
expect_equal(or3s(lko < 0L, , ),
             bor3(lko < 0L, , ))
expect_equal(or3s(lkp < 1L, , ),
             bor3(lkp < 1L, , ))
expect_equal(or3s(lkq < 9L, , ),
             bor3(lkq < 9L, , ))
expect_equal(or3s(lkr < 0L, , z %between% c(-1L, 1L)),
             bor3(lkr < 0L, , z %between% c(-1L, 1L)))
expect_equal(or3s(lks < 1L, , j %between% c(-1L, 1L)),
             bor3(lks < 1L, , j %between% c(-1L, 1L)))
expect_equal(or3s(lkt < 9L, , s %between% c(-1L, 1L)),
             bor3(lkt < 9L, , s %between% c(-1L, 1L)))
expect_equal(or3s(lku < 0L, s != 0L, ),
             bor3(lku < 0L, s != 0L, ))
expect_equal(or3s(lkv < 1L, c != 1L, ),
             bor3(lkv < 1L, c != 1L, ))
expect_equal(or3s(lkw < 9L, m != 9L, ),
             bor3(lkw < 9L, m != 9L, ))
expect_equal(or3s(lkx < 0L, q != 0L, p %between% c(-1L, 1L)),
             bor3(lkx < 0L, q != 0L, p %between% c(-1L, 1L)))
expect_equal(or3s(lky < 1L, v != 1L, d %between% c(-1L, 1L)),
             bor3(lky < 1L, v != 1L, d %between% c(-1L, 1L)))
expect_equal(or3s(lkz < 9L, b != 9L, q %between% c(-1L, 1L)),
             bor3(lkz < 9L, b != 9L, q %between% c(-1L, 1L)))
expect_equal(or3s(lla < 0L, , ),
             bor3(lla < 0L, , ))
expect_equal(or3s(llb < 1L, , ),
             bor3(llb < 1L, , ))
expect_equal(or3s(llc < 9L, , ),
             bor3(llc < 9L, , ))
expect_equal(or3s(lld < 0L, , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lld < 0L, , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lle < 1L, , o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lle < 1L, , o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(llf < 9L, , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(llf < 9L, , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(llg < 0L, b != 0L, ),
             bor3(llg < 0L, b != 0L, ))
expect_equal(or3s(llh < 1L, k != 1L, ),
             bor3(llh < 1L, k != 1L, ))
expect_equal(or3s(lli < 9L, u != 9L, ),
             bor3(lli < 9L, u != 9L, ))
expect_equal(or3s(llj < 0L, p != 0L, s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(llj < 0L, p != 0L, s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(llk < 1L, c != 1L, e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(llk < 1L, c != 1L, e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lll < 9L, q != 9L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lll < 9L, q != 9L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(llm < 0L, , ),
             bor3(llm < 0L, , ))
expect_equal(or3s(lln < 1L, , ),
             bor3(lln < 1L, , ))
expect_equal(or3s(llo < 9L, , ),
             bor3(llo < 9L, , ))
expect_equal(or3s(llp < 0L, , n %in% 1:4),
             bor3(llp < 0L, , n %in% 1:4))
expect_equal(or3s(llq < 1L, , x %in% 1:4),
             bor3(llq < 1L, , x %in% 1:4))
expect_equal(or3s(llr < 9L, , p %in% 1:4),
             bor3(llr < 9L, , p %in% 1:4))
expect_equal(or3s(lls < 0L, b != 0L, ),
             bor3(lls < 0L, b != 0L, ))
expect_equal(or3s(llt < 1L, q != 1L, ),
             bor3(llt < 1L, q != 1L, ))
expect_equal(or3s(llu < 9L, y != 9L, ),
             bor3(llu < 9L, y != 9L, ))
expect_equal(or3s(llv < 0L, t != 0L, y %in% 1:4),
             bor3(llv < 0L, t != 0L, y %in% 1:4))
expect_equal(or3s(llw < 1L, t != 1L, a %in% 1:4),
             bor3(llw < 1L, t != 1L, a %in% 1:4))
expect_equal(or3s(llx < 9L, a != 9L, h %in% 1:4),
             bor3(llx < 9L, a != 9L, h %in% 1:4))
expect_equal(or3s(lly < 0L, , ),
             bor3(lly < 0L, , ))
expect_equal(or3s(llz < 1L, , ),
             bor3(llz < 1L, , ))
expect_equal(or3s(lma < 9L, , ),
             bor3(lma < 9L, , ))
expect_equal(or3s(lmb < 0L, , s < 0L),
             bor3(lmb < 0L, , s < 0L))
expect_equal(or3s(lmc < 1L, , d < 1L),
             bor3(lmc < 1L, , d < 1L))
expect_equal(or3s(lmd < 9L, , k < 9L),
             bor3(lmd < 9L, , k < 9L))
expect_equal(or3s(lme < 0L, g != 0L, ),
             bor3(lme < 0L, g != 0L, ))
expect_equal(or3s(lmf < 1L, i != 1L, ),
             bor3(lmf < 1L, i != 1L, ))
expect_equal(or3s(lmg < 9L, q != 9L, ),
             bor3(lmg < 9L, q != 9L, ))
expect_equal(or3s(lmh < 0L, n != 0L, h < 0L),
             bor3(lmh < 0L, n != 0L, h < 0L))
expect_equal(or3s(lmi < 1L, w != 1L, h < 1L),
             bor3(lmi < 1L, w != 1L, h < 1L))
expect_equal(or3s(lmj < 9L, n != 9L, k < 9L),
             bor3(lmj < 9L, n != 9L, k < 9L))
expect_equal(or3s(lmk < 0L, , ),
             bor3(lmk < 0L, , ))
expect_equal(or3s(lml < 1L, , ),
             bor3(lml < 1L, , ))
expect_equal(or3s(lmm < 9L, , ),
             bor3(lmm < 9L, , ))
expect_equal(or3s(lmn < 0L, , u <= 0L),
             bor3(lmn < 0L, , u <= 0L))
expect_equal(or3s(lmo < 1L, , e <= 1L),
             bor3(lmo < 1L, , e <= 1L))
expect_equal(or3s(lmp < 9L, , e <= 9L),
             bor3(lmp < 9L, , e <= 9L))
expect_equal(or3s(lmq < 0L, l != 0L, ),
             bor3(lmq < 0L, l != 0L, ))
expect_equal(or3s(lmr < 1L, x != 1L, ),
             bor3(lmr < 1L, x != 1L, ))
expect_equal(or3s(lms < 9L, r != 9L, ),
             bor3(lms < 9L, r != 9L, ))
expect_equal(or3s(lmt < 0L, i != 0L, i <= 0L),
             bor3(lmt < 0L, i != 0L, i <= 0L))
expect_equal(or3s(lmu < 1L, r != 1L, m <= 1L),
             bor3(lmu < 1L, r != 1L, m <= 1L))
expect_equal(or3s(lmv < 9L, v != 9L, h <= 9L),
             bor3(lmv < 9L, v != 9L, h <= 9L))
expect_equal(or3s(lmw < 0L, , ),
             bor3(lmw < 0L, , ))
expect_equal(or3s(lmx < 1L, , ),
             bor3(lmx < 1L, , ))
expect_equal(or3s(lmy < 9L, , ),
             bor3(lmy < 9L, , ))
expect_equal(or3s(lmz < 0L, , c == 0L),
             bor3(lmz < 0L, , c == 0L))
expect_equal(or3s(lna < 1L, , e == 1L),
             bor3(lna < 1L, , e == 1L))
expect_equal(or3s(lnb < 9L, , q == 9L),
             bor3(lnb < 9L, , q == 9L))
expect_equal(or3s(lnc < 0L, z != 0L, ),
             bor3(lnc < 0L, z != 0L, ))
expect_equal(or3s(lnd < 1L, y != 1L, ),
             bor3(lnd < 1L, y != 1L, ))
expect_equal(or3s(lne < 9L, a != 9L, ),
             bor3(lne < 9L, a != 9L, ))
expect_equal(or3s(lnf < 0L, o != 0L, s == 0L),
             bor3(lnf < 0L, o != 0L, s == 0L))
expect_equal(or3s(lng < 1L, m != 1L, l == 1L),
             bor3(lng < 1L, m != 1L, l == 1L))
expect_equal(or3s(lnh < 9L, n != 9L, u == 9L),
             bor3(lnh < 9L, n != 9L, u == 9L))
expect_equal(or3s(lni < 0L, , ),
             bor3(lni < 0L, , ))
expect_equal(or3s(lnj < 1L, , ),
             bor3(lnj < 1L, , ))
expect_equal(or3s(lnk < 9L, , ),
             bor3(lnk < 9L, , ))
expect_equal(or3s(lnl < 0L, , t > 0L),
             bor3(lnl < 0L, , t > 0L))
expect_equal(or3s(lnm < 1L, , g > 1L),
             bor3(lnm < 1L, , g > 1L))
expect_equal(or3s(lnn < 9L, , o > 9L),
             bor3(lnn < 9L, , o > 9L))
expect_equal(or3s(lno < 0L, z != 0L, ),
             bor3(lno < 0L, z != 0L, ))
expect_equal(or3s(lnp < 1L, p != 1L, ),
             bor3(lnp < 1L, p != 1L, ))
expect_equal(or3s(lnq < 9L, q != 9L, ),
             bor3(lnq < 9L, q != 9L, ))
expect_equal(or3s(lnr < 0L, s != 0L, g > 0L),
             bor3(lnr < 0L, s != 0L, g > 0L))
expect_equal(or3s(lns < 1L, p != 1L, v > 1L),
             bor3(lns < 1L, p != 1L, v > 1L))
expect_equal(or3s(lnt < 9L, l != 9L, t > 9L),
             bor3(lnt < 9L, l != 9L, t > 9L))
expect_equal(or3s(lnu < 0L, , ),
             bor3(lnu < 0L, , ))
expect_equal(or3s(lnv < 1L, , ),
             bor3(lnv < 1L, , ))
expect_equal(or3s(lnw < 9L, , ),
             bor3(lnw < 9L, , ))
expect_equal(or3s(lnx < 0L, , c >= 0L),
             bor3(lnx < 0L, , c >= 0L))
expect_equal(or3s(lny < 1L, , l >= 1L),
             bor3(lny < 1L, , l >= 1L))
expect_equal(or3s(lnz < 9L, , m >= 9L),
             bor3(lnz < 9L, , m >= 9L))
expect_equal(or3s(loa < 0L, b != 0L, ),
             bor3(loa < 0L, b != 0L, ))
expect_equal(or3s(lob < 1L, u != 1L, ),
             bor3(lob < 1L, u != 1L, ))
expect_equal(or3s(loc < 9L, x != 9L, ),
             bor3(loc < 9L, x != 9L, ))
expect_equal(or3s(lod < 0L, m != 0L, c >= 0L),
             bor3(lod < 0L, m != 0L, c >= 0L))
expect_equal(or3s(loe < 1L, o != 1L, m >= 1L),
             bor3(loe < 1L, o != 1L, m >= 1L))
expect_equal(or3s(lof < 9L, p != 9L, i >= 9L),
             bor3(lof < 9L, p != 9L, i >= 9L))
expect_equal(or3s(log < 0L, , ),
             bor3(log < 0L, , ))
expect_equal(or3s(loh < 1L, , ),
             bor3(loh < 1L, , ))
expect_equal(or3s(loi < 9L, , ),
             bor3(loi < 9L, , ))
expect_equal(or3s(loj < 0L, , logi_l),
             bor3(loj < 0L, , logi_l))
expect_equal(or3s(lok < 1L, , logi_h),
             bor3(lok < 1L, , logi_h))
expect_equal(or3s(lol < 9L, , logi_n),
             bor3(lol < 9L, , logi_n))
expect_equal(or3s(lom < 0L, o %between% c(-1L, 1L), ),
             bor3(lom < 0L, o %between% c(-1L, 1L), ))
expect_equal(or3s(lon < 1L, p %between% c(-1L, 1L), ),
             bor3(lon < 1L, p %between% c(-1L, 1L), ))
expect_equal(or3s(loo < 9L, j %between% c(-1L, 1L), ),
             bor3(loo < 9L, j %between% c(-1L, 1L), ))
expect_equal(or3s(lop < 0L, f %between% c(-1L, 1L), logi_g),
             bor3(lop < 0L, f %between% c(-1L, 1L), logi_g))
expect_equal(or3s(loq < 1L, i %between% c(-1L, 1L), logi_x),
             bor3(loq < 1L, i %between% c(-1L, 1L), logi_x))
expect_equal(or3s(lor < 9L, z %between% c(-1L, 1L), logi_u),
             bor3(lor < 9L, z %between% c(-1L, 1L), logi_u))
expect_equal(or3s(los < 0L, , ),
             bor3(los < 0L, , ))
expect_equal(or3s(lot < 1L, , ),
             bor3(lot < 1L, , ))
expect_equal(or3s(lou < 9L, , ),
             bor3(lou < 9L, , ))
expect_equal(or3s(lov < 0L, , !logi_u),
             bor3(lov < 0L, , !logi_u))
expect_equal(or3s(low < 1L, , !logi_c),
             bor3(low < 1L, , !logi_c))
expect_equal(or3s(lox < 9L, , !logi_i),
             bor3(lox < 9L, , !logi_i))
expect_equal(or3s(loy < 0L, d %between% c(-1L, 1L), ),
             bor3(loy < 0L, d %between% c(-1L, 1L), ))
expect_equal(or3s(loz < 1L, n %between% c(-1L, 1L), ),
             bor3(loz < 1L, n %between% c(-1L, 1L), ))
expect_equal(or3s(lpa < 9L, z %between% c(-1L, 1L), ),
             bor3(lpa < 9L, z %between% c(-1L, 1L), ))
expect_equal(or3s(lpb < 0L, j %between% c(-1L, 1L), !logi_f),
             bor3(lpb < 0L, j %between% c(-1L, 1L), !logi_f))
expect_equal(or3s(lpc < 1L, z %between% c(-1L, 1L), !logi_d),
             bor3(lpc < 1L, z %between% c(-1L, 1L), !logi_d))
expect_equal(or3s(lpd < 9L, t %between% c(-1L, 1L), !logi_h),
             bor3(lpd < 9L, t %between% c(-1L, 1L), !logi_h))
expect_equal(or3s(lpe < 0L, , ),
             bor3(lpe < 0L, , ))
expect_equal(or3s(lpf < 1L, , ),
             bor3(lpf < 1L, , ))
expect_equal(or3s(lpg < 9L, , ),
             bor3(lpg < 9L, , ))
expect_equal(or3s(lph < 0L, , a != 0L),
             bor3(lph < 0L, , a != 0L))
expect_equal(or3s(lpi < 1L, , m != 1L),
             bor3(lpi < 1L, , m != 1L))
expect_equal(or3s(lpj < 9L, , o != 9L),
             bor3(lpj < 9L, , o != 9L))
expect_equal(or3s(lpk < 0L, g %between% c(-1L, 1L), ),
             bor3(lpk < 0L, g %between% c(-1L, 1L), ))
expect_equal(or3s(lpl < 1L, o %between% c(-1L, 1L), ),
             bor3(lpl < 1L, o %between% c(-1L, 1L), ))
expect_equal(or3s(lpm < 9L, q %between% c(-1L, 1L), ),
             bor3(lpm < 9L, q %between% c(-1L, 1L), ))
expect_equal(or3s(lpn < 0L, u %between% c(-1L, 1L), c != 0L),
             bor3(lpn < 0L, u %between% c(-1L, 1L), c != 0L))
expect_equal(or3s(lpo < 1L, w %between% c(-1L, 1L), k != 1L),
             bor3(lpo < 1L, w %between% c(-1L, 1L), k != 1L))
expect_equal(or3s(lpp < 9L, b %between% c(-1L, 1L), h != 9L),
             bor3(lpp < 9L, b %between% c(-1L, 1L), h != 9L))
expect_equal(or3s(lpq < 0L, , ),
             bor3(lpq < 0L, , ))
expect_equal(or3s(lpr < 1L, , ),
             bor3(lpr < 1L, , ))
expect_equal(or3s(lps < 9L, , ),
             bor3(lps < 9L, , ))
expect_equal(or3s(lpt < 0L, , j %between% c(-1L, 1L)),
             bor3(lpt < 0L, , j %between% c(-1L, 1L)))
expect_equal(or3s(lpu < 1L, , e %between% c(-1L, 1L)),
             bor3(lpu < 1L, , e %between% c(-1L, 1L)))
expect_equal(or3s(lpv < 9L, , q %between% c(-1L, 1L)),
             bor3(lpv < 9L, , q %between% c(-1L, 1L)))
expect_equal(or3s(lpw < 0L, q %between% c(-1L, 1L), ),
             bor3(lpw < 0L, q %between% c(-1L, 1L), ))
expect_equal(or3s(lpx < 1L, e %between% c(-1L, 1L), ),
             bor3(lpx < 1L, e %between% c(-1L, 1L), ))
expect_equal(or3s(lpy < 9L, l %between% c(-1L, 1L), ),
             bor3(lpy < 9L, l %between% c(-1L, 1L), ))
expect_equal(or3s(lpz < 0L, l %between% c(-1L, 1L), j %between% c(-1L, 1L)),
             bor3(lpz < 0L, l %between% c(-1L, 1L), j %between% c(-1L, 1L)))
expect_equal(or3s(lqa < 1L, w %between% c(-1L, 1L), s %between% c(-1L, 1L)),
             bor3(lqa < 1L, w %between% c(-1L, 1L), s %between% c(-1L, 1L)))
expect_equal(or3s(lqb < 9L, t %between% c(-1L, 1L), z %between% c(-1L, 1L)),
             bor3(lqb < 9L, t %between% c(-1L, 1L), z %between% c(-1L, 1L)))
expect_equal(or3s(lqc < 0L, , ),
             bor3(lqc < 0L, , ))
expect_equal(or3s(lqd < 1L, , ),
             bor3(lqd < 1L, , ))
expect_equal(or3s(lqe < 9L, , ),
             bor3(lqe < 9L, , ))
expect_equal(or3s(lqf < 0L, , i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lqf < 0L, , i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lqg < 1L, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lqg < 1L, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lqh < 9L, , n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lqh < 9L, , n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lqi < 0L, q %between% c(-1L, 1L), ),
             bor3(lqi < 0L, q %between% c(-1L, 1L), ))
expect_equal(or3s(lqj < 1L, f %between% c(-1L, 1L), ),
             bor3(lqj < 1L, f %between% c(-1L, 1L), ))
expect_equal(or3s(lqk < 9L, u %between% c(-1L, 1L), ),
             bor3(lqk < 9L, u %between% c(-1L, 1L), ))
expect_equal(or3s(lql < 0L, a %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lql < 0L, a %between% c(-1L, 1L), t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lqm < 1L, v %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lqm < 1L, v %between% c(-1L, 1L), b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lqn < 9L, n %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lqn < 9L, n %between% c(-1L, 1L), p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lqo < 0L, , ),
             bor3(lqo < 0L, , ))
expect_equal(or3s(lqp < 1L, , ),
             bor3(lqp < 1L, , ))
expect_equal(or3s(lqq < 9L, , ),
             bor3(lqq < 9L, , ))
expect_equal(or3s(lqr < 0L, , h %in% 1:4),
             bor3(lqr < 0L, , h %in% 1:4))
expect_equal(or3s(lqs < 1L, , d %in% 1:4),
             bor3(lqs < 1L, , d %in% 1:4))
expect_equal(or3s(lqt < 9L, , j %in% 1:4),
             bor3(lqt < 9L, , j %in% 1:4))
expect_equal(or3s(lqu < 0L, d %between% c(-1L, 1L), ),
             bor3(lqu < 0L, d %between% c(-1L, 1L), ))
expect_equal(or3s(lqv < 1L, v %between% c(-1L, 1L), ),
             bor3(lqv < 1L, v %between% c(-1L, 1L), ))
expect_equal(or3s(lqw < 9L, b %between% c(-1L, 1L), ),
             bor3(lqw < 9L, b %between% c(-1L, 1L), ))
expect_equal(or3s(lqx < 0L, y %between% c(-1L, 1L), b %in% 1:4),
             bor3(lqx < 0L, y %between% c(-1L, 1L), b %in% 1:4))
expect_equal(or3s(lqy < 1L, j %between% c(-1L, 1L), a %in% 1:4),
             bor3(lqy < 1L, j %between% c(-1L, 1L), a %in% 1:4))
expect_equal(or3s(lqz < 9L, j %between% c(-1L, 1L), l %in% 1:4),
             bor3(lqz < 9L, j %between% c(-1L, 1L), l %in% 1:4))
expect_equal(or3s(lra < 0L, , ),
             bor3(lra < 0L, , ))
expect_equal(or3s(lrb < 1L, , ),
             bor3(lrb < 1L, , ))
expect_equal(or3s(lrc < 9L, , ),
             bor3(lrc < 9L, , ))
expect_equal(or3s(lrd < 0L, , e < 0L),
             bor3(lrd < 0L, , e < 0L))
expect_equal(or3s(lre < 1L, , y < 1L),
             bor3(lre < 1L, , y < 1L))
expect_equal(or3s(lrf < 9L, , m < 9L),
             bor3(lrf < 9L, , m < 9L))
expect_equal(or3s(lrg < 0L, c %between% c(-1L, 1L), ),
             bor3(lrg < 0L, c %between% c(-1L, 1L), ))
expect_equal(or3s(lrh < 1L, r %between% c(-1L, 1L), ),
             bor3(lrh < 1L, r %between% c(-1L, 1L), ))
expect_equal(or3s(lri < 9L, j %between% c(-1L, 1L), ),
             bor3(lri < 9L, j %between% c(-1L, 1L), ))
expect_equal(or3s(lrj < 0L, z %between% c(-1L, 1L), j < 0L),
             bor3(lrj < 0L, z %between% c(-1L, 1L), j < 0L))
expect_equal(or3s(lrk < 1L, x %between% c(-1L, 1L), h < 1L),
             bor3(lrk < 1L, x %between% c(-1L, 1L), h < 1L))
expect_equal(or3s(lrl < 9L, s %between% c(-1L, 1L), i < 9L),
             bor3(lrl < 9L, s %between% c(-1L, 1L), i < 9L))
expect_equal(or3s(lrm < 0L, , ),
             bor3(lrm < 0L, , ))
expect_equal(or3s(lrn < 1L, , ),
             bor3(lrn < 1L, , ))
expect_equal(or3s(lro < 9L, , ),
             bor3(lro < 9L, , ))
expect_equal(or3s(lrp < 0L, , r <= 0L),
             bor3(lrp < 0L, , r <= 0L))
expect_equal(or3s(lrq < 1L, , d <= 1L),
             bor3(lrq < 1L, , d <= 1L))
expect_equal(or3s(lrr < 9L, , v <= 9L),
             bor3(lrr < 9L, , v <= 9L))
expect_equal(or3s(lrs < 0L, h %between% c(-1L, 1L), ),
             bor3(lrs < 0L, h %between% c(-1L, 1L), ))
expect_equal(or3s(lrt < 1L, y %between% c(-1L, 1L), ),
             bor3(lrt < 1L, y %between% c(-1L, 1L), ))
expect_equal(or3s(lru < 9L, b %between% c(-1L, 1L), ),
             bor3(lru < 9L, b %between% c(-1L, 1L), ))
expect_equal(or3s(lrv < 0L, o %between% c(-1L, 1L), l <= 0L),
             bor3(lrv < 0L, o %between% c(-1L, 1L), l <= 0L))
expect_equal(or3s(lrw < 1L, q %between% c(-1L, 1L), o <= 1L),
             bor3(lrw < 1L, q %between% c(-1L, 1L), o <= 1L))
expect_equal(or3s(lrx < 9L, y %between% c(-1L, 1L), h <= 9L),
             bor3(lrx < 9L, y %between% c(-1L, 1L), h <= 9L))
expect_equal(or3s(lry < 0L, , ),
             bor3(lry < 0L, , ))
expect_equal(or3s(lrz < 1L, , ),
             bor3(lrz < 1L, , ))
expect_equal(or3s(lsa < 9L, , ),
             bor3(lsa < 9L, , ))
expect_equal(or3s(lsb < 0L, , u == 0L),
             bor3(lsb < 0L, , u == 0L))
expect_equal(or3s(lsc < 1L, , k == 1L),
             bor3(lsc < 1L, , k == 1L))
expect_equal(or3s(lsd < 9L, , i == 9L),
             bor3(lsd < 9L, , i == 9L))
expect_equal(or3s(lse < 0L, j %between% c(-1L, 1L), ),
             bor3(lse < 0L, j %between% c(-1L, 1L), ))
expect_equal(or3s(lsf < 1L, g %between% c(-1L, 1L), ),
             bor3(lsf < 1L, g %between% c(-1L, 1L), ))
expect_equal(or3s(lsg < 9L, w %between% c(-1L, 1L), ),
             bor3(lsg < 9L, w %between% c(-1L, 1L), ))
expect_equal(or3s(lsh < 0L, s %between% c(-1L, 1L), c == 0L),
             bor3(lsh < 0L, s %between% c(-1L, 1L), c == 0L))
expect_equal(or3s(lsi < 1L, q %between% c(-1L, 1L), q == 1L),
             bor3(lsi < 1L, q %between% c(-1L, 1L), q == 1L))
expect_equal(or3s(lsj < 9L, y %between% c(-1L, 1L), w == 9L),
             bor3(lsj < 9L, y %between% c(-1L, 1L), w == 9L))
expect_equal(or3s(lsk < 0L, , ),
             bor3(lsk < 0L, , ))
expect_equal(or3s(lsl < 1L, , ),
             bor3(lsl < 1L, , ))
expect_equal(or3s(lsm < 9L, , ),
             bor3(lsm < 9L, , ))
expect_equal(or3s(lsn < 0L, , n > 0L),
             bor3(lsn < 0L, , n > 0L))
expect_equal(or3s(lso < 1L, , c > 1L),
             bor3(lso < 1L, , c > 1L))
expect_equal(or3s(lsp < 9L, , e > 9L),
             bor3(lsp < 9L, , e > 9L))
expect_equal(or3s(lsq < 0L, r %between% c(-1L, 1L), ),
             bor3(lsq < 0L, r %between% c(-1L, 1L), ))
expect_equal(or3s(lsr < 1L, e %between% c(-1L, 1L), ),
             bor3(lsr < 1L, e %between% c(-1L, 1L), ))
expect_equal(or3s(lss < 9L, e %between% c(-1L, 1L), ),
             bor3(lss < 9L, e %between% c(-1L, 1L), ))
expect_equal(or3s(lst < 0L, r %between% c(-1L, 1L), q > 0L),
             bor3(lst < 0L, r %between% c(-1L, 1L), q > 0L))
expect_equal(or3s(lsu < 1L, p %between% c(-1L, 1L), o > 1L),
             bor3(lsu < 1L, p %between% c(-1L, 1L), o > 1L))
expect_equal(or3s(lsv < 9L, w %between% c(-1L, 1L), m > 9L),
             bor3(lsv < 9L, w %between% c(-1L, 1L), m > 9L))
expect_equal(or3s(lsw < 0L, , ),
             bor3(lsw < 0L, , ))
expect_equal(or3s(lsx < 1L, , ),
             bor3(lsx < 1L, , ))
expect_equal(or3s(lsy < 9L, , ),
             bor3(lsy < 9L, , ))
expect_equal(or3s(lsz < 0L, , k >= 0L),
             bor3(lsz < 0L, , k >= 0L))
expect_equal(or3s(lta < 1L, , f >= 1L),
             bor3(lta < 1L, , f >= 1L))
expect_equal(or3s(ltb < 9L, , g >= 9L),
             bor3(ltb < 9L, , g >= 9L))
expect_equal(or3s(ltc < 0L, j %between% c(-1L, 1L), ),
             bor3(ltc < 0L, j %between% c(-1L, 1L), ))
expect_equal(or3s(ltd < 1L, l %between% c(-1L, 1L), ),
             bor3(ltd < 1L, l %between% c(-1L, 1L), ))
expect_equal(or3s(lte < 9L, z %between% c(-1L, 1L), ),
             bor3(lte < 9L, z %between% c(-1L, 1L), ))
expect_equal(or3s(ltf < 0L, t %between% c(-1L, 1L), f >= 0L),
             bor3(ltf < 0L, t %between% c(-1L, 1L), f >= 0L))
expect_equal(or3s(ltg < 1L, o %between% c(-1L, 1L), j >= 1L),
             bor3(ltg < 1L, o %between% c(-1L, 1L), j >= 1L))
expect_equal(or3s(lth < 9L, h %between% c(-1L, 1L), o >= 9L),
             bor3(lth < 9L, h %between% c(-1L, 1L), o >= 9L))
expect_equal(or3s(lti < 0L, , ),
             bor3(lti < 0L, , ))
expect_equal(or3s(ltj < 1L, , ),
             bor3(ltj < 1L, , ))
expect_equal(or3s(ltk < 9L, , ),
             bor3(ltk < 9L, , ))
expect_equal(or3s(ltl < 0L, , logi_n),
             bor3(ltl < 0L, , logi_n))
expect_equal(or3s(ltm < 1L, , logi_g),
             bor3(ltm < 1L, , logi_g))
expect_equal(or3s(ltn < 9L, , logi_w),
             bor3(ltn < 9L, , logi_w))
expect_equal(or3s(lto < 0L, d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lto < 0L, d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ltp < 1L, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ltp < 1L, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ltq < 9L, h %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(ltq < 9L, h %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(ltr < 0L, k %in% c(4L, -3L, 2L, 7L, 8L), logi_e),
             bor3(ltr < 0L, k %in% c(4L, -3L, 2L, 7L, 8L), logi_e))
expect_equal(or3s(lts < 1L, k %in% c(4L, -3L, 2L, 7L, 8L), logi_v),
             bor3(lts < 1L, k %in% c(4L, -3L, 2L, 7L, 8L), logi_v))
expect_equal(or3s(ltt < 9L, x %in% c(4L, -3L, 2L, 7L, 8L), logi_x),
             bor3(ltt < 9L, x %in% c(4L, -3L, 2L, 7L, 8L), logi_x))
expect_equal(or3s(ltu < 0L, , ),
             bor3(ltu < 0L, , ))
expect_equal(or3s(ltv < 1L, , ),
             bor3(ltv < 1L, , ))
expect_equal(or3s(ltw < 9L, , ),
             bor3(ltw < 9L, , ))
expect_equal(or3s(ltx < 0L, , logi_r),
             bor3(ltx < 0L, , logi_r))
expect_equal(or3s(lty < 1L, , logi_i),
             bor3(lty < 1L, , logi_i))
expect_equal(or3s(ltz < 9L, , logi_d),
             bor3(ltz < 9L, , logi_d))
expect_equal(or3s(lua < 0L, n %in% 1:4, ),
             bor3(lua < 0L, n %in% 1:4, ))
expect_equal(or3s(lub < 1L, g %in% 1:4, ),
             bor3(lub < 1L, g %in% 1:4, ))
expect_equal(or3s(luc < 9L, m %in% 1:4, ),
             bor3(luc < 9L, m %in% 1:4, ))
expect_equal(or3s(lud < 0L, u %in% 1:4, logi_g),
             bor3(lud < 0L, u %in% 1:4, logi_g))
expect_equal(or3s(lue < 1L, z %in% 1:4, logi_b),
             bor3(lue < 1L, z %in% 1:4, logi_b))
expect_equal(or3s(luf < 9L, w %in% 1:4, logi_p),
             bor3(luf < 9L, w %in% 1:4, logi_p))
expect_equal(or3s(lug < 0L, , ),
             bor3(lug < 0L, , ))
expect_equal(or3s(luh < 1L, , ),
             bor3(luh < 1L, , ))
expect_equal(or3s(lui < 9L, , ),
             bor3(lui < 9L, , ))
expect_equal(or3s(luj < 0L, , !logi_c),
             bor3(luj < 0L, , !logi_c))
expect_equal(or3s(luk < 1L, , !logi_i),
             bor3(luk < 1L, , !logi_i))
expect_equal(or3s(lul < 9L, , !logi_n),
             bor3(lul < 9L, , !logi_n))
expect_equal(or3s(lum < 0L, x %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lum < 0L, x %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lun < 1L, t %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lun < 1L, t %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(luo < 9L, x %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(luo < 9L, x %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lup < 0L, w %in% c(4L, -3L, 2L, 7L, 8L), !logi_w),
             bor3(lup < 0L, w %in% c(4L, -3L, 2L, 7L, 8L), !logi_w))
expect_equal(or3s(luq < 1L, c %in% c(4L, -3L, 2L, 7L, 8L), !logi_y),
             bor3(luq < 1L, c %in% c(4L, -3L, 2L, 7L, 8L), !logi_y))
expect_equal(or3s(lur < 9L, d %in% c(4L, -3L, 2L, 7L, 8L), !logi_n),
             bor3(lur < 9L, d %in% c(4L, -3L, 2L, 7L, 8L), !logi_n))
expect_equal(or3s(lus < 0L, , ),
             bor3(lus < 0L, , ))
expect_equal(or3s(lut < 1L, , ),
             bor3(lut < 1L, , ))
expect_equal(or3s(luu < 9L, , ),
             bor3(luu < 9L, , ))
expect_equal(or3s(luv < 0L, , !logi_p),
             bor3(luv < 0L, , !logi_p))
expect_equal(or3s(luw < 1L, , !logi_l),
             bor3(luw < 1L, , !logi_l))
expect_equal(or3s(lux < 9L, , !logi_n),
             bor3(lux < 9L, , !logi_n))
expect_equal(or3s(luy < 0L, y %in% 1:4, ),
             bor3(luy < 0L, y %in% 1:4, ))
expect_equal(or3s(luz < 1L, j %in% 1:4, ),
             bor3(luz < 1L, j %in% 1:4, ))
expect_equal(or3s(lva < 9L, z %in% 1:4, ),
             bor3(lva < 9L, z %in% 1:4, ))
expect_equal(or3s(lvb < 0L, f %in% 1:4, !logi_u),
             bor3(lvb < 0L, f %in% 1:4, !logi_u))
expect_equal(or3s(lvc < 1L, p %in% 1:4, !logi_u),
             bor3(lvc < 1L, p %in% 1:4, !logi_u))
expect_equal(or3s(lvd < 9L, b %in% 1:4, !logi_e),
             bor3(lvd < 9L, b %in% 1:4, !logi_e))
expect_equal(or3s(lve < 0L, , ),
             bor3(lve < 0L, , ))
expect_equal(or3s(lvf < 1L, , ),
             bor3(lvf < 1L, , ))
expect_equal(or3s(lvg < 9L, , ),
             bor3(lvg < 9L, , ))
expect_equal(or3s(lvh < 0L, , e != 0L),
             bor3(lvh < 0L, , e != 0L))
expect_equal(or3s(lvi < 1L, , c != 1L),
             bor3(lvi < 1L, , c != 1L))
expect_equal(or3s(lvj < 9L, , d != 9L),
             bor3(lvj < 9L, , d != 9L))
expect_equal(or3s(lvk < 0L, c %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lvk < 0L, c %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lvl < 1L, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lvl < 1L, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lvm < 9L, g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lvm < 9L, g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lvn < 0L, w %in% c(4L, -3L, 2L, 7L, 8L), z != 0L),
             bor3(lvn < 0L, w %in% c(4L, -3L, 2L, 7L, 8L), z != 0L))
expect_equal(or3s(lvo < 1L, f %in% c(4L, -3L, 2L, 7L, 8L), q != 1L),
             bor3(lvo < 1L, f %in% c(4L, -3L, 2L, 7L, 8L), q != 1L))
expect_equal(or3s(lvp < 9L, y %in% c(4L, -3L, 2L, 7L, 8L), o != 9L),
             bor3(lvp < 9L, y %in% c(4L, -3L, 2L, 7L, 8L), o != 9L))
expect_equal(or3s(lvq < 0L, , ),
             bor3(lvq < 0L, , ))
expect_equal(or3s(lvr < 1L, , ),
             bor3(lvr < 1L, , ))
expect_equal(or3s(lvs < 9L, , ),
             bor3(lvs < 9L, , ))
expect_equal(or3s(lvt < 0L, , t != 0L),
             bor3(lvt < 0L, , t != 0L))
expect_equal(or3s(lvu < 1L, , w != 1L),
             bor3(lvu < 1L, , w != 1L))
expect_equal(or3s(lvv < 9L, , t != 9L),
             bor3(lvv < 9L, , t != 9L))
expect_equal(or3s(lvw < 0L, f %in% 1:4, ),
             bor3(lvw < 0L, f %in% 1:4, ))
expect_equal(or3s(lvx < 1L, k %in% 1:4, ),
             bor3(lvx < 1L, k %in% 1:4, ))
expect_equal(or3s(lvy < 9L, u %in% 1:4, ),
             bor3(lvy < 9L, u %in% 1:4, ))
expect_equal(or3s(lvz < 0L, t %in% 1:4, e != 0L),
             bor3(lvz < 0L, t %in% 1:4, e != 0L))
expect_equal(or3s(lwa < 1L, m %in% 1:4, c != 1L),
             bor3(lwa < 1L, m %in% 1:4, c != 1L))
expect_equal(or3s(lwb < 9L, d %in% 1:4, k != 9L),
             bor3(lwb < 9L, d %in% 1:4, k != 9L))
expect_equal(or3s(lwc < 0L, , ),
             bor3(lwc < 0L, , ))
expect_equal(or3s(lwd < 1L, , ),
             bor3(lwd < 1L, , ))
expect_equal(or3s(lwe < 9L, , ),
             bor3(lwe < 9L, , ))
expect_equal(or3s(lwf < 0L, , j %between% c(-1L, 1L)),
             bor3(lwf < 0L, , j %between% c(-1L, 1L)))
expect_equal(or3s(lwg < 1L, , m %between% c(-1L, 1L)),
             bor3(lwg < 1L, , m %between% c(-1L, 1L)))
expect_equal(or3s(lwh < 9L, , e %between% c(-1L, 1L)),
             bor3(lwh < 9L, , e %between% c(-1L, 1L)))
expect_equal(or3s(lwi < 0L, u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lwi < 0L, u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lwj < 1L, d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lwj < 1L, d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lwk < 9L, w %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lwk < 9L, w %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lwl < 0L, r %in% c(4L, -3L, 2L, 7L, 8L), a %between% c(-1L, 1L)),
             bor3(lwl < 0L, r %in% c(4L, -3L, 2L, 7L, 8L), a %between% c(-1L, 1L)))
expect_equal(or3s(lwm < 1L, a %in% c(4L, -3L, 2L, 7L, 8L), r %between% c(-1L, 1L)),
             bor3(lwm < 1L, a %in% c(4L, -3L, 2L, 7L, 8L), r %between% c(-1L, 1L)))
expect_equal(or3s(lwn < 9L, t %in% c(4L, -3L, 2L, 7L, 8L), v %between% c(-1L, 1L)),
             bor3(lwn < 9L, t %in% c(4L, -3L, 2L, 7L, 8L), v %between% c(-1L, 1L)))
expect_equal(or3s(lwo < 0L, , ),
             bor3(lwo < 0L, , ))
expect_equal(or3s(lwp < 1L, , ),
             bor3(lwp < 1L, , ))
expect_equal(or3s(lwq < 9L, , ),
             bor3(lwq < 9L, , ))
expect_equal(or3s(lwr < 0L, , g %between% c(-1L, 1L)),
             bor3(lwr < 0L, , g %between% c(-1L, 1L)))
expect_equal(or3s(lws < 1L, , l %between% c(-1L, 1L)),
             bor3(lws < 1L, , l %between% c(-1L, 1L)))
expect_equal(or3s(lwt < 9L, , a %between% c(-1L, 1L)),
             bor3(lwt < 9L, , a %between% c(-1L, 1L)))
expect_equal(or3s(lwu < 0L, i %in% 1:4, ),
             bor3(lwu < 0L, i %in% 1:4, ))
expect_equal(or3s(lwv < 1L, l %in% 1:4, ),
             bor3(lwv < 1L, l %in% 1:4, ))
expect_equal(or3s(lww < 9L, c %in% 1:4, ),
             bor3(lww < 9L, c %in% 1:4, ))
expect_equal(or3s(lwx < 0L, d %in% 1:4, g %between% c(-1L, 1L)),
             bor3(lwx < 0L, d %in% 1:4, g %between% c(-1L, 1L)))
expect_equal(or3s(lwy < 1L, r %in% 1:4, l %between% c(-1L, 1L)),
             bor3(lwy < 1L, r %in% 1:4, l %between% c(-1L, 1L)))
expect_equal(or3s(lwz < 9L, j %in% 1:4, s %between% c(-1L, 1L)),
             bor3(lwz < 9L, j %in% 1:4, s %between% c(-1L, 1L)))
expect_equal(or3s(lxa < 0L, , ),
             bor3(lxa < 0L, , ))
expect_equal(or3s(lxb < 1L, , ),
             bor3(lxb < 1L, , ))
expect_equal(or3s(lxc < 9L, , ),
             bor3(lxc < 9L, , ))
expect_equal(or3s(lxd < 0L, , a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lxd < 0L, , a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lxe < 1L, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lxe < 1L, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lxf < 9L, , b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lxf < 9L, , b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lxg < 0L, j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lxg < 0L, j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lxh < 1L, r %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lxh < 1L, r %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lxi < 9L, p %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lxi < 9L, p %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lxj < 0L, s %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lxj < 0L, s %in% c(4L, -3L, 2L, 7L, 8L), g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lxk < 1L, s %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lxk < 1L, s %in% c(4L, -3L, 2L, 7L, 8L), m %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lxl < 9L, r %in% c(4L, -3L, 2L, 7L, 8L), n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(lxl < 9L, r %in% c(4L, -3L, 2L, 7L, 8L), n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(lxm < 0L, , ),
             bor3(lxm < 0L, , ))
expect_equal(or3s(lxn < 1L, , ),
             bor3(lxn < 1L, , ))
expect_equal(or3s(lxo < 9L, , ),
             bor3(lxo < 9L, , ))
expect_equal(or3s(lxp < 0L, , w %in% 1:4),
             bor3(lxp < 0L, , w %in% 1:4))
expect_equal(or3s(lxq < 1L, , w %in% 1:4),
             bor3(lxq < 1L, , w %in% 1:4))
expect_equal(or3s(lxr < 9L, , a %in% 1:4),
             bor3(lxr < 9L, , a %in% 1:4))
expect_equal(or3s(lxs < 0L, p %in% 1:4, ),
             bor3(lxs < 0L, p %in% 1:4, ))
expect_equal(or3s(lxt < 1L, o %in% 1:4, ),
             bor3(lxt < 1L, o %in% 1:4, ))
expect_equal(or3s(lxu < 9L, q %in% 1:4, ),
             bor3(lxu < 9L, q %in% 1:4, ))
expect_equal(or3s(lxv < 0L, e %in% 1:4, s %in% 1:4),
             bor3(lxv < 0L, e %in% 1:4, s %in% 1:4))
expect_equal(or3s(lxw < 1L, s %in% 1:4, d %in% 1:4),
             bor3(lxw < 1L, s %in% 1:4, d %in% 1:4))
expect_equal(or3s(lxx < 9L, a %in% 1:4, m %in% 1:4),
             bor3(lxx < 9L, a %in% 1:4, m %in% 1:4))
expect_equal(or3s(lxy < 0L, , ),
             bor3(lxy < 0L, , ))
expect_equal(or3s(lxz < 1L, , ),
             bor3(lxz < 1L, , ))
expect_equal(or3s(lya < 9L, , ),
             bor3(lya < 9L, , ))
expect_equal(or3s(lyb < 0L, , h < 0L),
             bor3(lyb < 0L, , h < 0L))
expect_equal(or3s(lyc < 1L, , s < 1L),
             bor3(lyc < 1L, , s < 1L))
expect_equal(or3s(lyd < 9L, , k < 9L),
             bor3(lyd < 9L, , k < 9L))
expect_equal(or3s(lye < 0L, u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lye < 0L, u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lyf < 1L, a %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lyf < 1L, a %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lyg < 9L, z %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(lyg < 9L, z %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(lyh < 0L, w %in% c(4L, -3L, 2L, 7L, 8L), i < 0L),
             bor3(lyh < 0L, w %in% c(4L, -3L, 2L, 7L, 8L), i < 0L))
expect_equal(or3s(lyi < 1L, o %in% c(4L, -3L, 2L, 7L, 8L), r < 1L),
             bor3(lyi < 1L, o %in% c(4L, -3L, 2L, 7L, 8L), r < 1L))
expect_equal(or3s(lyj < 9L, j %in% c(4L, -3L, 2L, 7L, 8L), d < 9L),
             bor3(lyj < 9L, j %in% c(4L, -3L, 2L, 7L, 8L), d < 9L))
expect_equal(or3s(lyk < 0L, , ),
             bor3(lyk < 0L, , ))
expect_equal(or3s(lyl < 1L, , ),
             bor3(lyl < 1L, , ))
expect_equal(or3s(lym < 9L, , ),
             bor3(lym < 9L, , ))
expect_equal(or3s(lyn < 0L, , i < 0L),
             bor3(lyn < 0L, , i < 0L))
expect_equal(or3s(lyo < 1L, , x < 1L),
             bor3(lyo < 1L, , x < 1L))
expect_equal(or3s(lyp < 9L, , d < 9L),
             bor3(lyp < 9L, , d < 9L))
expect_equal(or3s(lyq < 0L, h %in% 1:4, ),
             bor3(lyq < 0L, h %in% 1:4, ))
expect_equal(or3s(lyr < 1L, y %in% 1:4, ),
             bor3(lyr < 1L, y %in% 1:4, ))
expect_equal(or3s(lys < 9L, t %in% 1:4, ),
             bor3(lys < 9L, t %in% 1:4, ))
expect_equal(or3s(lyt < 0L, y %in% 1:4, g < 0L),
             bor3(lyt < 0L, y %in% 1:4, g < 0L))
expect_equal(or3s(lyu < 1L, h %in% 1:4, e < 1L),
             bor3(lyu < 1L, h %in% 1:4, e < 1L))
expect_equal(or3s(lyv < 9L, z %in% 1:4, p < 9L),
             bor3(lyv < 9L, z %in% 1:4, p < 9L))
expect_equal(or3s(lyw < 0L, , ),
             bor3(lyw < 0L, , ))
expect_equal(or3s(lyx < 1L, , ),
             bor3(lyx < 1L, , ))
expect_equal(or3s(lyy < 9L, , ),
             bor3(lyy < 9L, , ))
expect_equal(or3s(lyz < 0L, , u <= 0L),
             bor3(lyz < 0L, , u <= 0L))
expect_equal(or3s(ma < 1L, , h <= 1L),
             bor3(ma < 1L, , h <= 1L))
expect_equal(or3s(mb < 9L, , y <= 9L),
             bor3(mb < 9L, , y <= 9L))
expect_equal(or3s(mc < 0L, v %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(mc < 0L, v %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(md < 1L, d %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(md < 1L, d %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(me < 9L, f %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(me < 9L, f %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mf < 0L, c %in% c(4L, -3L, 2L, 7L, 8L), u <= 0L),
             bor3(mf < 0L, c %in% c(4L, -3L, 2L, 7L, 8L), u <= 0L))
expect_equal(or3s(mg < 1L, r %in% c(4L, -3L, 2L, 7L, 8L), r <= 1L),
             bor3(mg < 1L, r %in% c(4L, -3L, 2L, 7L, 8L), r <= 1L))
expect_equal(or3s(mh < 9L, i %in% c(4L, -3L, 2L, 7L, 8L), m <= 9L),
             bor3(mh < 9L, i %in% c(4L, -3L, 2L, 7L, 8L), m <= 9L))
expect_equal(or3s(mi < 0L, , ),
             bor3(mi < 0L, , ))
expect_equal(or3s(mj < 1L, , ),
             bor3(mj < 1L, , ))
expect_equal(or3s(mk < 9L, , ),
             bor3(mk < 9L, , ))
expect_equal(or3s(ml < 0L, , r <= 0L),
             bor3(ml < 0L, , r <= 0L))
expect_equal(or3s(mm < 1L, , u <= 1L),
             bor3(mm < 1L, , u <= 1L))
expect_equal(or3s(mn < 9L, , q <= 9L),
             bor3(mn < 9L, , q <= 9L))
expect_equal(or3s(mo < 0L, d %in% 1:4, ),
             bor3(mo < 0L, d %in% 1:4, ))
expect_equal(or3s(mp < 1L, d %in% 1:4, ),
             bor3(mp < 1L, d %in% 1:4, ))
expect_equal(or3s(mq < 9L, p %in% 1:4, ),
             bor3(mq < 9L, p %in% 1:4, ))
expect_equal(or3s(mr < 0L, u %in% 1:4, j <= 0L),
             bor3(mr < 0L, u %in% 1:4, j <= 0L))
expect_equal(or3s(ms < 1L, k %in% 1:4, y <= 1L),
             bor3(ms < 1L, k %in% 1:4, y <= 1L))
expect_equal(or3s(mt < 9L, a %in% 1:4, n <= 9L),
             bor3(mt < 9L, a %in% 1:4, n <= 9L))
expect_equal(or3s(mu < 0L, , ),
             bor3(mu < 0L, , ))
expect_equal(or3s(mv < 1L, , ),
             bor3(mv < 1L, , ))
expect_equal(or3s(mw < 9L, , ),
             bor3(mw < 9L, , ))
expect_equal(or3s(mx < 0L, , j == 0L),
             bor3(mx < 0L, , j == 0L))
expect_equal(or3s(my < 1L, , r == 1L),
             bor3(my < 1L, , r == 1L))
expect_equal(or3s(mz < 9L, , g == 9L),
             bor3(mz < 9L, , g == 9L))
expect_equal(or3s(maa < 0L, z %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(maa < 0L, z %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mab < 1L, h %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(mab < 1L, h %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mac < 9L, h %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(mac < 9L, h %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mad < 0L, a %in% c(4L, -3L, 2L, 7L, 8L), g == 0L),
             bor3(mad < 0L, a %in% c(4L, -3L, 2L, 7L, 8L), g == 0L))
expect_equal(or3s(mae < 1L, t %in% c(4L, -3L, 2L, 7L, 8L), x == 1L),
             bor3(mae < 1L, t %in% c(4L, -3L, 2L, 7L, 8L), x == 1L))
expect_equal(or3s(maf < 9L, g %in% c(4L, -3L, 2L, 7L, 8L), i == 9L),
             bor3(maf < 9L, g %in% c(4L, -3L, 2L, 7L, 8L), i == 9L))
expect_equal(or3s(mag < 0L, , ),
             bor3(mag < 0L, , ))
expect_equal(or3s(mah < 1L, , ),
             bor3(mah < 1L, , ))
expect_equal(or3s(mai < 9L, , ),
             bor3(mai < 9L, , ))
expect_equal(or3s(maj < 0L, , t == 0L),
             bor3(maj < 0L, , t == 0L))
expect_equal(or3s(mak < 1L, , w == 1L),
             bor3(mak < 1L, , w == 1L))
expect_equal(or3s(mal < 9L, , q == 9L),
             bor3(mal < 9L, , q == 9L))
expect_equal(or3s(mam < 0L, b %in% 1:4, ),
             bor3(mam < 0L, b %in% 1:4, ))
expect_equal(or3s(man < 1L, b %in% 1:4, ),
             bor3(man < 1L, b %in% 1:4, ))
expect_equal(or3s(mao < 9L, r %in% 1:4, ),
             bor3(mao < 9L, r %in% 1:4, ))
expect_equal(or3s(map < 0L, s %in% 1:4, b == 0L),
             bor3(map < 0L, s %in% 1:4, b == 0L))
expect_equal(or3s(maq < 1L, b %in% 1:4, k == 1L),
             bor3(maq < 1L, b %in% 1:4, k == 1L))
expect_equal(or3s(mar < 9L, z %in% 1:4, a == 9L),
             bor3(mar < 9L, z %in% 1:4, a == 9L))
expect_equal(or3s(mas < 0L, , ),
             bor3(mas < 0L, , ))
expect_equal(or3s(mat < 1L, , ),
             bor3(mat < 1L, , ))
expect_equal(or3s(mau < 9L, , ),
             bor3(mau < 9L, , ))
expect_equal(or3s(mav < 0L, , y > 0L),
             bor3(mav < 0L, , y > 0L))
expect_equal(or3s(maw < 1L, , l > 1L),
             bor3(maw < 1L, , l > 1L))
expect_equal(or3s(max < 9L, , b > 9L),
             bor3(max < 9L, , b > 9L))
expect_equal(or3s(may < 0L, u %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(may < 0L, u %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(maz < 1L, a %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(maz < 1L, a %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mba < 9L, c %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(mba < 9L, c %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mbb < 0L, x %in% c(4L, -3L, 2L, 7L, 8L), u > 0L),
             bor3(mbb < 0L, x %in% c(4L, -3L, 2L, 7L, 8L), u > 0L))
expect_equal(or3s(mbc < 1L, x %in% c(4L, -3L, 2L, 7L, 8L), f > 1L),
             bor3(mbc < 1L, x %in% c(4L, -3L, 2L, 7L, 8L), f > 1L))
expect_equal(or3s(mbd < 9L, o %in% c(4L, -3L, 2L, 7L, 8L), e > 9L),
             bor3(mbd < 9L, o %in% c(4L, -3L, 2L, 7L, 8L), e > 9L))
expect_equal(or3s(mbe < 0L, , ),
             bor3(mbe < 0L, , ))
expect_equal(or3s(mbf < 1L, , ),
             bor3(mbf < 1L, , ))
expect_equal(or3s(mbg < 9L, , ),
             bor3(mbg < 9L, , ))
expect_equal(or3s(mbh < 0L, , j > 0L),
             bor3(mbh < 0L, , j > 0L))
expect_equal(or3s(mbi < 1L, , y > 1L),
             bor3(mbi < 1L, , y > 1L))
expect_equal(or3s(mbj < 9L, , o > 9L),
             bor3(mbj < 9L, , o > 9L))
expect_equal(or3s(mbk < 0L, z %in% 1:4, ),
             bor3(mbk < 0L, z %in% 1:4, ))
expect_equal(or3s(mbl < 1L, j %in% 1:4, ),
             bor3(mbl < 1L, j %in% 1:4, ))
expect_equal(or3s(mbm < 9L, x %in% 1:4, ),
             bor3(mbm < 9L, x %in% 1:4, ))
expect_equal(or3s(mbn < 0L, e %in% 1:4, c > 0L),
             bor3(mbn < 0L, e %in% 1:4, c > 0L))
expect_equal(or3s(mbo < 1L, y %in% 1:4, p > 1L),
             bor3(mbo < 1L, y %in% 1:4, p > 1L))
expect_equal(or3s(mbp < 9L, p %in% 1:4, r > 9L),
             bor3(mbp < 9L, p %in% 1:4, r > 9L))
expect_equal(or3s(mbq < 0L, , ),
             bor3(mbq < 0L, , ))
expect_equal(or3s(mbr < 1L, , ),
             bor3(mbr < 1L, , ))
expect_equal(or3s(mbs < 9L, , ),
             bor3(mbs < 9L, , ))
expect_equal(or3s(mbt < 0L, , n >= 0L),
             bor3(mbt < 0L, , n >= 0L))
expect_equal(or3s(mbu < 1L, , w >= 1L),
             bor3(mbu < 1L, , w >= 1L))
expect_equal(or3s(mbv < 9L, , q >= 9L),
             bor3(mbv < 9L, , q >= 9L))
expect_equal(or3s(mbw < 0L, l %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(mbw < 0L, l %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mbx < 1L, h %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(mbx < 1L, h %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mby < 9L, n %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(mby < 9L, n %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(mbz < 0L, n %in% c(4L, -3L, 2L, 7L, 8L), y >= 0L),
             bor3(mbz < 0L, n %in% c(4L, -3L, 2L, 7L, 8L), y >= 0L))
expect_equal(or3s(mca < 1L, x %in% c(4L, -3L, 2L, 7L, 8L), w >= 1L),
             bor3(mca < 1L, x %in% c(4L, -3L, 2L, 7L, 8L), w >= 1L))
expect_equal(or3s(mcb < 9L, g %in% c(4L, -3L, 2L, 7L, 8L), x >= 9L),
             bor3(mcb < 9L, g %in% c(4L, -3L, 2L, 7L, 8L), x >= 9L))
expect_equal(or3s(mcc < 0L, , ),
             bor3(mcc < 0L, , ))
expect_equal(or3s(mcd < 1L, , ),
             bor3(mcd < 1L, , ))
expect_equal(or3s(mce < 9L, , ),
             bor3(mce < 9L, , ))
expect_equal(or3s(mcf < 0L, , z >= 0L),
             bor3(mcf < 0L, , z >= 0L))
expect_equal(or3s(mcg < 1L, , m >= 1L),
             bor3(mcg < 1L, , m >= 1L))
expect_equal(or3s(mch < 9L, , h >= 9L),
             bor3(mch < 9L, , h >= 9L))
expect_equal(or3s(mci < 0L, d %in% 1:4, ),
             bor3(mci < 0L, d %in% 1:4, ))
expect_equal(or3s(mcj < 1L, u %in% 1:4, ),
             bor3(mcj < 1L, u %in% 1:4, ))
expect_equal(or3s(mck < 9L, j %in% 1:4, ),
             bor3(mck < 9L, j %in% 1:4, ))
expect_equal(or3s(mcl < 0L, b %in% 1:4, k >= 0L),
             bor3(mcl < 0L, b %in% 1:4, k >= 0L))
expect_equal(or3s(mcm < 1L, y %in% 1:4, u >= 1L),
             bor3(mcm < 1L, y %in% 1:4, u >= 1L))
expect_equal(or3s(mcn < 9L, t %in% 1:4, f >= 9L),
             bor3(mcn < 9L, t %in% 1:4, f >= 9L))
expect_equal(or3s(mco < 0L, , ),
             bor3(mco < 0L, , ))
expect_equal(or3s(mcp < 1L, , ),
             bor3(mcp < 1L, , ))
expect_equal(or3s(mcq < 9L, , ),
             bor3(mcq < 9L, , ))
expect_equal(or3s(mcr < 0L, , logi_d),
             bor3(mcr < 0L, , logi_d))
expect_equal(or3s(mcs < 1L, , logi_m),
             bor3(mcs < 1L, , logi_m))
expect_equal(or3s(mct < 9L, , logi_v),
             bor3(mct < 9L, , logi_v))
expect_equal(or3s(mcu < 0L, r < 0L, ),
             bor3(mcu < 0L, r < 0L, ))
expect_equal(or3s(mcv < 1L, b < 1L, ),
             bor3(mcv < 1L, b < 1L, ))
expect_equal(or3s(mcw < 9L, b < 9L, ),
             bor3(mcw < 9L, b < 9L, ))
expect_equal(or3s(mcx < 0L, r < 0L, logi_s),
             bor3(mcx < 0L, r < 0L, logi_s))
expect_equal(or3s(mcy < 1L, d < 1L, logi_t),
             bor3(mcy < 1L, d < 1L, logi_t))
expect_equal(or3s(mcz < 9L, i < 9L, logi_l),
             bor3(mcz < 9L, i < 9L, logi_l))
expect_equal(or3s(mda < 0L, , ),
             bor3(mda < 0L, , ))
expect_equal(or3s(mdb < 1L, , ),
             bor3(mdb < 1L, , ))
expect_equal(or3s(mdc < 9L, , ),
             bor3(mdc < 9L, , ))
expect_equal(or3s(mdd < 0L, , !logi_u),
             bor3(mdd < 0L, , !logi_u))
expect_equal(or3s(mde < 1L, , !logi_z),
             bor3(mde < 1L, , !logi_z))
expect_equal(or3s(mdf < 9L, , !logi_l),
             bor3(mdf < 9L, , !logi_l))
expect_equal(or3s(mdg < 0L, a < 0L, ),
             bor3(mdg < 0L, a < 0L, ))
expect_equal(or3s(mdh < 1L, j < 1L, ),
             bor3(mdh < 1L, j < 1L, ))
expect_equal(or3s(mdi < 9L, y < 9L, ),
             bor3(mdi < 9L, y < 9L, ))
expect_equal(or3s(mdj < 0L, w < 0L, !logi_d),
             bor3(mdj < 0L, w < 0L, !logi_d))
expect_equal(or3s(mdk < 1L, m < 1L, !logi_s),
             bor3(mdk < 1L, m < 1L, !logi_s))
expect_equal(or3s(mdl < 9L, w < 9L, !logi_v),
             bor3(mdl < 9L, w < 9L, !logi_v))
expect_equal(or3s(mdm < 0L, , ),
             bor3(mdm < 0L, , ))
expect_equal(or3s(mdn < 1L, , ),
             bor3(mdn < 1L, , ))
expect_equal(or3s(mdo < 9L, , ),
             bor3(mdo < 9L, , ))
expect_equal(or3s(mdp < 0L, , y != 0L),
             bor3(mdp < 0L, , y != 0L))
expect_equal(or3s(mdq < 1L, , i != 1L),
             bor3(mdq < 1L, , i != 1L))
expect_equal(or3s(mdr < 9L, , x != 9L),
             bor3(mdr < 9L, , x != 9L))
expect_equal(or3s(mds < 0L, u < 0L, ),
             bor3(mds < 0L, u < 0L, ))
expect_equal(or3s(mdt < 1L, j < 1L, ),
             bor3(mdt < 1L, j < 1L, ))
expect_equal(or3s(mdu < 9L, v < 9L, ),
             bor3(mdu < 9L, v < 9L, ))
expect_equal(or3s(mdv < 0L, z < 0L, t != 0L),
             bor3(mdv < 0L, z < 0L, t != 0L))
expect_equal(or3s(mdw < 1L, g < 1L, c != 1L),
             bor3(mdw < 1L, g < 1L, c != 1L))
expect_equal(or3s(mdx < 9L, u < 9L, f != 9L),
             bor3(mdx < 9L, u < 9L, f != 9L))
expect_equal(or3s(mdy < 0L, , ),
             bor3(mdy < 0L, , ))
expect_equal(or3s(mdz < 1L, , ),
             bor3(mdz < 1L, , ))
expect_equal(or3s(mea < 9L, , ),
             bor3(mea < 9L, , ))
expect_equal(or3s(meb < 0L, , j %between% c(-1L, 1L)),
             bor3(meb < 0L, , j %between% c(-1L, 1L)))
expect_equal(or3s(mec < 1L, , o %between% c(-1L, 1L)),
             bor3(mec < 1L, , o %between% c(-1L, 1L)))
expect_equal(or3s(med < 9L, , i %between% c(-1L, 1L)),
             bor3(med < 9L, , i %between% c(-1L, 1L)))
expect_equal(or3s(mee < 0L, f < 0L, ),
             bor3(mee < 0L, f < 0L, ))
expect_equal(or3s(mef < 1L, o < 1L, ),
             bor3(mef < 1L, o < 1L, ))
expect_equal(or3s(meg < 9L, b < 9L, ),
             bor3(meg < 9L, b < 9L, ))
expect_equal(or3s(meh < 0L, q < 0L, w %between% c(-1L, 1L)),
             bor3(meh < 0L, q < 0L, w %between% c(-1L, 1L)))
expect_equal(or3s(mei < 1L, v < 1L, k %between% c(-1L, 1L)),
             bor3(mei < 1L, v < 1L, k %between% c(-1L, 1L)))
expect_equal(or3s(mej < 9L, e < 9L, h %between% c(-1L, 1L)),
             bor3(mej < 9L, e < 9L, h %between% c(-1L, 1L)))
expect_equal(or3s(mek < 0L, , ),
             bor3(mek < 0L, , ))
expect_equal(or3s(mel < 1L, , ),
             bor3(mel < 1L, , ))
expect_equal(or3s(mem < 9L, , ),
             bor3(mem < 9L, , ))
expect_equal(or3s(men < 0L, , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(men < 0L, , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(meo < 1L, , q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(meo < 1L, , q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mep < 9L, , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mep < 9L, , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(meq < 0L, f < 0L, ),
             bor3(meq < 0L, f < 0L, ))
expect_equal(or3s(mer < 1L, p < 1L, ),
             bor3(mer < 1L, p < 1L, ))
expect_equal(or3s(mes < 9L, w < 9L, ),
             bor3(mes < 9L, w < 9L, ))
expect_equal(or3s(met < 0L, n < 0L, b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(met < 0L, n < 0L, b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(meu < 1L, u < 1L, v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(meu < 1L, u < 1L, v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mev < 9L, q < 9L, c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mev < 9L, q < 9L, c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mew < 0L, , ),
             bor3(mew < 0L, , ))
expect_equal(or3s(mex < 1L, , ),
             bor3(mex < 1L, , ))
expect_equal(or3s(mey < 9L, , ),
             bor3(mey < 9L, , ))
expect_equal(or3s(mez < 0L, , h %in% 1:4),
             bor3(mez < 0L, , h %in% 1:4))
expect_equal(or3s(mfa < 1L, , d %in% 1:4),
             bor3(mfa < 1L, , d %in% 1:4))
expect_equal(or3s(mfb < 9L, , c %in% 1:4),
             bor3(mfb < 9L, , c %in% 1:4))
expect_equal(or3s(mfc < 0L, e < 0L, ),
             bor3(mfc < 0L, e < 0L, ))
expect_equal(or3s(mfd < 1L, f < 1L, ),
             bor3(mfd < 1L, f < 1L, ))
expect_equal(or3s(mfe < 9L, b < 9L, ),
             bor3(mfe < 9L, b < 9L, ))
expect_equal(or3s(mff < 0L, y < 0L, r %in% 1:4),
             bor3(mff < 0L, y < 0L, r %in% 1:4))
expect_equal(or3s(mfg < 1L, b < 1L, p %in% 1:4),
             bor3(mfg < 1L, b < 1L, p %in% 1:4))
expect_equal(or3s(mfh < 9L, k < 9L, r %in% 1:4),
             bor3(mfh < 9L, k < 9L, r %in% 1:4))
expect_equal(or3s(mfi < 0L, , ),
             bor3(mfi < 0L, , ))
expect_equal(or3s(mfj < 1L, , ),
             bor3(mfj < 1L, , ))
expect_equal(or3s(mfk < 9L, , ),
             bor3(mfk < 9L, , ))
expect_equal(or3s(mfl < 0L, , b < 0L),
             bor3(mfl < 0L, , b < 0L))
expect_equal(or3s(mfm < 1L, , h < 1L),
             bor3(mfm < 1L, , h < 1L))
expect_equal(or3s(mfn < 9L, , k < 9L),
             bor3(mfn < 9L, , k < 9L))
expect_equal(or3s(mfo < 0L, f < 0L, ),
             bor3(mfo < 0L, f < 0L, ))
expect_equal(or3s(mfp < 1L, i < 1L, ),
             bor3(mfp < 1L, i < 1L, ))
expect_equal(or3s(mfq < 9L, b < 9L, ),
             bor3(mfq < 9L, b < 9L, ))
expect_equal(or3s(mfr < 0L, j < 0L, k < 0L),
             bor3(mfr < 0L, j < 0L, k < 0L))
expect_equal(or3s(mfs < 1L, m < 1L, y < 1L),
             bor3(mfs < 1L, m < 1L, y < 1L))
expect_equal(or3s(mft < 9L, x < 9L, i < 9L),
             bor3(mft < 9L, x < 9L, i < 9L))
expect_equal(or3s(mfu < 0L, , ),
             bor3(mfu < 0L, , ))
expect_equal(or3s(mfv < 1L, , ),
             bor3(mfv < 1L, , ))
expect_equal(or3s(mfw < 9L, , ),
             bor3(mfw < 9L, , ))
expect_equal(or3s(mfx < 0L, , x <= 0L),
             bor3(mfx < 0L, , x <= 0L))
expect_equal(or3s(mfy < 1L, , o <= 1L),
             bor3(mfy < 1L, , o <= 1L))
expect_equal(or3s(mfz < 9L, , w <= 9L),
             bor3(mfz < 9L, , w <= 9L))
expect_equal(or3s(mga < 0L, d < 0L, ),
             bor3(mga < 0L, d < 0L, ))
expect_equal(or3s(mgb < 1L, p < 1L, ),
             bor3(mgb < 1L, p < 1L, ))
expect_equal(or3s(mgc < 9L, w < 9L, ),
             bor3(mgc < 9L, w < 9L, ))
expect_equal(or3s(mgd < 0L, p < 0L, i <= 0L),
             bor3(mgd < 0L, p < 0L, i <= 0L))
expect_equal(or3s(mge < 1L, v < 1L, j <= 1L),
             bor3(mge < 1L, v < 1L, j <= 1L))
expect_equal(or3s(mgf < 9L, e < 9L, m <= 9L),
             bor3(mgf < 9L, e < 9L, m <= 9L))
expect_equal(or3s(mgg < 0L, , ),
             bor3(mgg < 0L, , ))
expect_equal(or3s(mgh < 1L, , ),
             bor3(mgh < 1L, , ))
expect_equal(or3s(mgi < 9L, , ),
             bor3(mgi < 9L, , ))
expect_equal(or3s(mgj < 0L, , x == 0L),
             bor3(mgj < 0L, , x == 0L))
expect_equal(or3s(mgk < 1L, , k == 1L),
             bor3(mgk < 1L, , k == 1L))
expect_equal(or3s(mgl < 9L, , f == 9L),
             bor3(mgl < 9L, , f == 9L))
expect_equal(or3s(mgm < 0L, c < 0L, ),
             bor3(mgm < 0L, c < 0L, ))
expect_equal(or3s(mgn < 1L, v < 1L, ),
             bor3(mgn < 1L, v < 1L, ))
expect_equal(or3s(mgo < 9L, c < 9L, ),
             bor3(mgo < 9L, c < 9L, ))
expect_equal(or3s(mgp < 0L, g < 0L, z == 0L),
             bor3(mgp < 0L, g < 0L, z == 0L))
expect_equal(or3s(mgq < 1L, h < 1L, g == 1L),
             bor3(mgq < 1L, h < 1L, g == 1L))
expect_equal(or3s(mgr < 9L, u < 9L, u == 9L),
             bor3(mgr < 9L, u < 9L, u == 9L))
expect_equal(or3s(mgs < 0L, , ),
             bor3(mgs < 0L, , ))
expect_equal(or3s(mgt < 1L, , ),
             bor3(mgt < 1L, , ))
expect_equal(or3s(mgu < 9L, , ),
             bor3(mgu < 9L, , ))
expect_equal(or3s(mgv < 0L, , z > 0L),
             bor3(mgv < 0L, , z > 0L))
expect_equal(or3s(mgw < 1L, , e > 1L),
             bor3(mgw < 1L, , e > 1L))
expect_equal(or3s(mgx < 9L, , a > 9L),
             bor3(mgx < 9L, , a > 9L))
expect_equal(or3s(mgy < 0L, e < 0L, ),
             bor3(mgy < 0L, e < 0L, ))
expect_equal(or3s(mgz < 1L, f < 1L, ),
             bor3(mgz < 1L, f < 1L, ))
expect_equal(or3s(mha < 9L, r < 9L, ),
             bor3(mha < 9L, r < 9L, ))
expect_equal(or3s(mhb < 0L, b < 0L, j > 0L),
             bor3(mhb < 0L, b < 0L, j > 0L))
expect_equal(or3s(mhc < 1L, h < 1L, o > 1L),
             bor3(mhc < 1L, h < 1L, o > 1L))
expect_equal(or3s(mhd < 9L, k < 9L, x > 9L),
             bor3(mhd < 9L, k < 9L, x > 9L))
expect_equal(or3s(mhe < 0L, , ),
             bor3(mhe < 0L, , ))
expect_equal(or3s(mhf < 1L, , ),
             bor3(mhf < 1L, , ))
expect_equal(or3s(mhg < 9L, , ),
             bor3(mhg < 9L, , ))
expect_equal(or3s(mhh < 0L, , s >= 0L),
             bor3(mhh < 0L, , s >= 0L))
expect_equal(or3s(mhi < 1L, , s >= 1L),
             bor3(mhi < 1L, , s >= 1L))
expect_equal(or3s(mhj < 9L, , z >= 9L),
             bor3(mhj < 9L, , z >= 9L))
expect_equal(or3s(mhk < 0L, o < 0L, ),
             bor3(mhk < 0L, o < 0L, ))
expect_equal(or3s(mhl < 1L, q < 1L, ),
             bor3(mhl < 1L, q < 1L, ))
expect_equal(or3s(mhm < 9L, h < 9L, ),
             bor3(mhm < 9L, h < 9L, ))
expect_equal(or3s(mhn < 0L, z < 0L, j >= 0L),
             bor3(mhn < 0L, z < 0L, j >= 0L))
expect_equal(or3s(mho < 1L, b < 1L, b >= 1L),
             bor3(mho < 1L, b < 1L, b >= 1L))
expect_equal(or3s(mhp < 9L, e < 9L, o >= 9L),
             bor3(mhp < 9L, e < 9L, o >= 9L))
expect_equal(or3s(mhq < 0L, , ),
             bor3(mhq < 0L, , ))
expect_equal(or3s(mhr < 1L, , ),
             bor3(mhr < 1L, , ))
expect_equal(or3s(mhs < 9L, , ),
             bor3(mhs < 9L, , ))
expect_equal(or3s(mht < 0L, , logi_i),
             bor3(mht < 0L, , logi_i))
expect_equal(or3s(mhu < 1L, , logi_r),
             bor3(mhu < 1L, , logi_r))
expect_equal(or3s(mhv < 9L, , logi_v),
             bor3(mhv < 9L, , logi_v))
expect_equal(or3s(mhw < 0L, f <= 0L, ),
             bor3(mhw < 0L, f <= 0L, ))
expect_equal(or3s(mhx < 1L, x <= 1L, ),
             bor3(mhx < 1L, x <= 1L, ))
expect_equal(or3s(mhy < 9L, t <= 9L, ),
             bor3(mhy < 9L, t <= 9L, ))
expect_equal(or3s(mhz < 0L, p <= 0L, logi_j),
             bor3(mhz < 0L, p <= 0L, logi_j))
expect_equal(or3s(mia < 1L, m <= 1L, logi_y),
             bor3(mia < 1L, m <= 1L, logi_y))
expect_equal(or3s(mib < 9L, f <= 9L, logi_q),
             bor3(mib < 9L, f <= 9L, logi_q))
expect_equal(or3s(mic < 0L, , ),
             bor3(mic < 0L, , ))
expect_equal(or3s(mid < 1L, , ),
             bor3(mid < 1L, , ))
expect_equal(or3s(mie < 9L, , ),
             bor3(mie < 9L, , ))
expect_equal(or3s(mif < 0L, , !logi_a),
             bor3(mif < 0L, , !logi_a))
expect_equal(or3s(mig < 1L, , !logi_d),
             bor3(mig < 1L, , !logi_d))
expect_equal(or3s(mih < 9L, , !logi_h),
             bor3(mih < 9L, , !logi_h))
expect_equal(or3s(mii < 0L, l <= 0L, ),
             bor3(mii < 0L, l <= 0L, ))
expect_equal(or3s(mij < 1L, y <= 1L, ),
             bor3(mij < 1L, y <= 1L, ))
expect_equal(or3s(mik < 9L, w <= 9L, ),
             bor3(mik < 9L, w <= 9L, ))
expect_equal(or3s(mil < 0L, a <= 0L, !logi_g),
             bor3(mil < 0L, a <= 0L, !logi_g))
expect_equal(or3s(mim < 1L, o <= 1L, !logi_k),
             bor3(mim < 1L, o <= 1L, !logi_k))
expect_equal(or3s(min < 9L, w <= 9L, !logi_w),
             bor3(min < 9L, w <= 9L, !logi_w))
expect_equal(or3s(mio < 0L, , ),
             bor3(mio < 0L, , ))
expect_equal(or3s(mip < 1L, , ),
             bor3(mip < 1L, , ))
expect_equal(or3s(miq < 9L, , ),
             bor3(miq < 9L, , ))
expect_equal(or3s(mir < 0L, , e != 0L),
             bor3(mir < 0L, , e != 0L))
expect_equal(or3s(mis < 1L, , v != 1L),
             bor3(mis < 1L, , v != 1L))
expect_equal(or3s(mit < 9L, , x != 9L),
             bor3(mit < 9L, , x != 9L))
expect_equal(or3s(miu < 0L, g <= 0L, ),
             bor3(miu < 0L, g <= 0L, ))
expect_equal(or3s(miv < 1L, j <= 1L, ),
             bor3(miv < 1L, j <= 1L, ))
expect_equal(or3s(miw < 9L, o <= 9L, ),
             bor3(miw < 9L, o <= 9L, ))
expect_equal(or3s(mix < 0L, s <= 0L, x != 0L),
             bor3(mix < 0L, s <= 0L, x != 0L))
expect_equal(or3s(miy < 1L, q <= 1L, l != 1L),
             bor3(miy < 1L, q <= 1L, l != 1L))
expect_equal(or3s(miz < 9L, i <= 9L, p != 9L),
             bor3(miz < 9L, i <= 9L, p != 9L))
expect_equal(or3s(mja < 0L, , ),
             bor3(mja < 0L, , ))
expect_equal(or3s(mjb < 1L, , ),
             bor3(mjb < 1L, , ))
expect_equal(or3s(mjc < 9L, , ),
             bor3(mjc < 9L, , ))
expect_equal(or3s(mjd < 0L, , k %between% c(-1L, 1L)),
             bor3(mjd < 0L, , k %between% c(-1L, 1L)))
expect_equal(or3s(mje < 1L, , a %between% c(-1L, 1L)),
             bor3(mje < 1L, , a %between% c(-1L, 1L)))
expect_equal(or3s(mjf < 9L, , q %between% c(-1L, 1L)),
             bor3(mjf < 9L, , q %between% c(-1L, 1L)))
expect_equal(or3s(mjg < 0L, u <= 0L, ),
             bor3(mjg < 0L, u <= 0L, ))
expect_equal(or3s(mjh < 1L, g <= 1L, ),
             bor3(mjh < 1L, g <= 1L, ))
expect_equal(or3s(mji < 9L, u <= 9L, ),
             bor3(mji < 9L, u <= 9L, ))
expect_equal(or3s(mjj < 0L, q <= 0L, f %between% c(-1L, 1L)),
             bor3(mjj < 0L, q <= 0L, f %between% c(-1L, 1L)))
expect_equal(or3s(mjk < 1L, u <= 1L, t %between% c(-1L, 1L)),
             bor3(mjk < 1L, u <= 1L, t %between% c(-1L, 1L)))
expect_equal(or3s(mjl < 9L, h <= 9L, v %between% c(-1L, 1L)),
             bor3(mjl < 9L, h <= 9L, v %between% c(-1L, 1L)))
expect_equal(or3s(mjm < 0L, , ),
             bor3(mjm < 0L, , ))
expect_equal(or3s(mjn < 1L, , ),
             bor3(mjn < 1L, , ))
expect_equal(or3s(mjo < 9L, , ),
             bor3(mjo < 9L, , ))
expect_equal(or3s(mjp < 0L, , i %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mjp < 0L, , i %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mjq < 1L, , j %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mjq < 1L, , j %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mjr < 9L, , a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mjr < 9L, , a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mjs < 0L, a <= 0L, ),
             bor3(mjs < 0L, a <= 0L, ))
expect_equal(or3s(mjt < 1L, w <= 1L, ),
             bor3(mjt < 1L, w <= 1L, ))
expect_equal(or3s(mju < 9L, v <= 9L, ),
             bor3(mju < 9L, v <= 9L, ))
expect_equal(or3s(mjv < 0L, y <= 0L, g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mjv < 0L, y <= 0L, g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mjw < 1L, o <= 1L, p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mjw < 1L, o <= 1L, p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mjx < 9L, r <= 9L, n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mjx < 9L, r <= 9L, n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mjy < 0L, , ),
             bor3(mjy < 0L, , ))
expect_equal(or3s(mjz < 1L, , ),
             bor3(mjz < 1L, , ))
expect_equal(or3s(mka < 9L, , ),
             bor3(mka < 9L, , ))
expect_equal(or3s(mkb < 0L, , e %in% 1:4),
             bor3(mkb < 0L, , e %in% 1:4))
expect_equal(or3s(mkc < 1L, , p %in% 1:4),
             bor3(mkc < 1L, , p %in% 1:4))
expect_equal(or3s(mkd < 9L, , v %in% 1:4),
             bor3(mkd < 9L, , v %in% 1:4))
expect_equal(or3s(mke < 0L, g <= 0L, ),
             bor3(mke < 0L, g <= 0L, ))
expect_equal(or3s(mkf < 1L, i <= 1L, ),
             bor3(mkf < 1L, i <= 1L, ))
expect_equal(or3s(mkg < 9L, g <= 9L, ),
             bor3(mkg < 9L, g <= 9L, ))
expect_equal(or3s(mkh < 0L, h <= 0L, m %in% 1:4),
             bor3(mkh < 0L, h <= 0L, m %in% 1:4))
expect_equal(or3s(mki < 1L, y <= 1L, i %in% 1:4),
             bor3(mki < 1L, y <= 1L, i %in% 1:4))
expect_equal(or3s(mkj < 9L, r <= 9L, q %in% 1:4),
             bor3(mkj < 9L, r <= 9L, q %in% 1:4))
expect_equal(or3s(mkk < 0L, , ),
             bor3(mkk < 0L, , ))
expect_equal(or3s(mkl < 1L, , ),
             bor3(mkl < 1L, , ))
expect_equal(or3s(mkm < 9L, , ),
             bor3(mkm < 9L, , ))
expect_equal(or3s(mkn < 0L, , n < 0L),
             bor3(mkn < 0L, , n < 0L))
expect_equal(or3s(mko < 1L, , r < 1L),
             bor3(mko < 1L, , r < 1L))
expect_equal(or3s(mkp < 9L, , q < 9L),
             bor3(mkp < 9L, , q < 9L))
expect_equal(or3s(mkq < 0L, q <= 0L, ),
             bor3(mkq < 0L, q <= 0L, ))
expect_equal(or3s(mkr < 1L, y <= 1L, ),
             bor3(mkr < 1L, y <= 1L, ))
expect_equal(or3s(mks < 9L, k <= 9L, ),
             bor3(mks < 9L, k <= 9L, ))
expect_equal(or3s(mkt < 0L, b <= 0L, c < 0L),
             bor3(mkt < 0L, b <= 0L, c < 0L))
expect_equal(or3s(mku < 1L, s <= 1L, t < 1L),
             bor3(mku < 1L, s <= 1L, t < 1L))
expect_equal(or3s(mkv < 9L, g <= 9L, v < 9L),
             bor3(mkv < 9L, g <= 9L, v < 9L))
expect_equal(or3s(mkw < 0L, , ),
             bor3(mkw < 0L, , ))
expect_equal(or3s(mkx < 1L, , ),
             bor3(mkx < 1L, , ))
expect_equal(or3s(mky < 9L, , ),
             bor3(mky < 9L, , ))
expect_equal(or3s(mkz < 0L, , j <= 0L),
             bor3(mkz < 0L, , j <= 0L))
expect_equal(or3s(mla < 1L, , w <= 1L),
             bor3(mla < 1L, , w <= 1L))
expect_equal(or3s(mlb < 9L, , l <= 9L),
             bor3(mlb < 9L, , l <= 9L))
expect_equal(or3s(mlc < 0L, w <= 0L, ),
             bor3(mlc < 0L, w <= 0L, ))
expect_equal(or3s(mld < 1L, j <= 1L, ),
             bor3(mld < 1L, j <= 1L, ))
expect_equal(or3s(mle < 9L, m <= 9L, ),
             bor3(mle < 9L, m <= 9L, ))
expect_equal(or3s(mlf < 0L, t <= 0L, w <= 0L),
             bor3(mlf < 0L, t <= 0L, w <= 0L))
expect_equal(or3s(mlg < 1L, l <= 1L, z <= 1L),
             bor3(mlg < 1L, l <= 1L, z <= 1L))
expect_equal(or3s(mlh < 9L, k <= 9L, x <= 9L),
             bor3(mlh < 9L, k <= 9L, x <= 9L))
expect_equal(or3s(mli < 0L, , ),
             bor3(mli < 0L, , ))
expect_equal(or3s(mlj < 1L, , ),
             bor3(mlj < 1L, , ))
expect_equal(or3s(mlk < 9L, , ),
             bor3(mlk < 9L, , ))
expect_equal(or3s(mll < 0L, , c == 0L),
             bor3(mll < 0L, , c == 0L))
expect_equal(or3s(mlm < 1L, , v == 1L),
             bor3(mlm < 1L, , v == 1L))
expect_equal(or3s(mln < 9L, , a == 9L),
             bor3(mln < 9L, , a == 9L))
expect_equal(or3s(mlo < 0L, a <= 0L, ),
             bor3(mlo < 0L, a <= 0L, ))
expect_equal(or3s(mlp < 1L, e <= 1L, ),
             bor3(mlp < 1L, e <= 1L, ))
expect_equal(or3s(mlq < 9L, b <= 9L, ),
             bor3(mlq < 9L, b <= 9L, ))
expect_equal(or3s(mlr < 0L, s <= 0L, k == 0L),
             bor3(mlr < 0L, s <= 0L, k == 0L))
expect_equal(or3s(mls < 1L, g <= 1L, s == 1L),
             bor3(mls < 1L, g <= 1L, s == 1L))
expect_equal(or3s(mlt < 9L, t <= 9L, t == 9L),
             bor3(mlt < 9L, t <= 9L, t == 9L))
expect_equal(or3s(mlu < 0L, , ),
             bor3(mlu < 0L, , ))
expect_equal(or3s(mlv < 1L, , ),
             bor3(mlv < 1L, , ))
expect_equal(or3s(mlw < 9L, , ),
             bor3(mlw < 9L, , ))
expect_equal(or3s(mlx < 0L, , n > 0L),
             bor3(mlx < 0L, , n > 0L))
expect_equal(or3s(mly < 1L, , d > 1L),
             bor3(mly < 1L, , d > 1L))
expect_equal(or3s(mlz < 9L, , u > 9L),
             bor3(mlz < 9L, , u > 9L))
expect_equal(or3s(mma < 0L, w <= 0L, ),
             bor3(mma < 0L, w <= 0L, ))
expect_equal(or3s(mmb < 1L, k <= 1L, ),
             bor3(mmb < 1L, k <= 1L, ))
expect_equal(or3s(mmc < 9L, v <= 9L, ),
             bor3(mmc < 9L, v <= 9L, ))
expect_equal(or3s(mmd < 0L, v <= 0L, f > 0L),
             bor3(mmd < 0L, v <= 0L, f > 0L))
expect_equal(or3s(mme < 1L, p <= 1L, b > 1L),
             bor3(mme < 1L, p <= 1L, b > 1L))
expect_equal(or3s(mmf < 9L, u <= 9L, h > 9L),
             bor3(mmf < 9L, u <= 9L, h > 9L))
expect_equal(or3s(mmg < 0L, , ),
             bor3(mmg < 0L, , ))
expect_equal(or3s(mmh < 1L, , ),
             bor3(mmh < 1L, , ))
expect_equal(or3s(mmi < 9L, , ),
             bor3(mmi < 9L, , ))
expect_equal(or3s(mmj < 0L, , f >= 0L),
             bor3(mmj < 0L, , f >= 0L))
expect_equal(or3s(mmk < 1L, , w >= 1L),
             bor3(mmk < 1L, , w >= 1L))
expect_equal(or3s(mml < 9L, , y >= 9L),
             bor3(mml < 9L, , y >= 9L))
expect_equal(or3s(mmm < 0L, b <= 0L, ),
             bor3(mmm < 0L, b <= 0L, ))
expect_equal(or3s(mmn < 1L, d <= 1L, ),
             bor3(mmn < 1L, d <= 1L, ))
expect_equal(or3s(mmo < 9L, u <= 9L, ),
             bor3(mmo < 9L, u <= 9L, ))
expect_equal(or3s(mmp < 0L, j <= 0L, f >= 0L),
             bor3(mmp < 0L, j <= 0L, f >= 0L))
expect_equal(or3s(mmq < 1L, p <= 1L, j >= 1L),
             bor3(mmq < 1L, p <= 1L, j >= 1L))
expect_equal(or3s(mmr < 9L, g <= 9L, t >= 9L),
             bor3(mmr < 9L, g <= 9L, t >= 9L))
expect_equal(or3s(mms < 0L, , ),
             bor3(mms < 0L, , ))
expect_equal(or3s(mmt < 1L, , ),
             bor3(mmt < 1L, , ))
expect_equal(or3s(mmu < 9L, , ),
             bor3(mmu < 9L, , ))
expect_equal(or3s(mmv < 0L, , logi_n),
             bor3(mmv < 0L, , logi_n))
expect_equal(or3s(mmw < 1L, , logi_b),
             bor3(mmw < 1L, , logi_b))
expect_equal(or3s(mmx < 9L, , logi_x),
             bor3(mmx < 9L, , logi_x))
expect_equal(or3s(mmy < 0L, q == 0L, ),
             bor3(mmy < 0L, q == 0L, ))
expect_equal(or3s(mmz < 1L, n == 1L, ),
             bor3(mmz < 1L, n == 1L, ))
expect_equal(or3s(mna < 9L, l == 9L, ),
             bor3(mna < 9L, l == 9L, ))
expect_equal(or3s(mnb < 0L, j == 0L, logi_n),
             bor3(mnb < 0L, j == 0L, logi_n))
expect_equal(or3s(mnc < 1L, q == 1L, logi_c),
             bor3(mnc < 1L, q == 1L, logi_c))
expect_equal(or3s(mnd < 9L, q == 9L, logi_d),
             bor3(mnd < 9L, q == 9L, logi_d))
expect_equal(or3s(mne < 0L, , ),
             bor3(mne < 0L, , ))
expect_equal(or3s(mnf < 1L, , ),
             bor3(mnf < 1L, , ))
expect_equal(or3s(mng < 9L, , ),
             bor3(mng < 9L, , ))
expect_equal(or3s(mnh < 0L, , !logi_u),
             bor3(mnh < 0L, , !logi_u))
expect_equal(or3s(mni < 1L, , !logi_j),
             bor3(mni < 1L, , !logi_j))
expect_equal(or3s(mnj < 9L, , !logi_t),
             bor3(mnj < 9L, , !logi_t))
expect_equal(or3s(mnk < 0L, z == 0L, ),
             bor3(mnk < 0L, z == 0L, ))
expect_equal(or3s(mnl < 1L, v == 1L, ),
             bor3(mnl < 1L, v == 1L, ))
expect_equal(or3s(mnm < 9L, x == 9L, ),
             bor3(mnm < 9L, x == 9L, ))
expect_equal(or3s(mnn < 0L, p == 0L, !logi_p),
             bor3(mnn < 0L, p == 0L, !logi_p))
expect_equal(or3s(mno < 1L, k == 1L, !logi_w),
             bor3(mno < 1L, k == 1L, !logi_w))
expect_equal(or3s(mnp < 9L, n == 9L, !logi_t),
             bor3(mnp < 9L, n == 9L, !logi_t))
expect_equal(or3s(mnq < 0L, , ),
             bor3(mnq < 0L, , ))
expect_equal(or3s(mnr < 1L, , ),
             bor3(mnr < 1L, , ))
expect_equal(or3s(mns < 9L, , ),
             bor3(mns < 9L, , ))
expect_equal(or3s(mnt < 0L, , t != 0L),
             bor3(mnt < 0L, , t != 0L))
expect_equal(or3s(mnu < 1L, , o != 1L),
             bor3(mnu < 1L, , o != 1L))
expect_equal(or3s(mnv < 9L, , r != 9L),
             bor3(mnv < 9L, , r != 9L))
expect_equal(or3s(mnw < 0L, t == 0L, ),
             bor3(mnw < 0L, t == 0L, ))
expect_equal(or3s(mnx < 1L, c == 1L, ),
             bor3(mnx < 1L, c == 1L, ))
expect_equal(or3s(mny < 9L, i == 9L, ),
             bor3(mny < 9L, i == 9L, ))
expect_equal(or3s(mnz < 0L, m == 0L, m != 0L),
             bor3(mnz < 0L, m == 0L, m != 0L))
expect_equal(or3s(moa < 1L, s == 1L, h != 1L),
             bor3(moa < 1L, s == 1L, h != 1L))
expect_equal(or3s(mob < 9L, o == 9L, r != 9L),
             bor3(mob < 9L, o == 9L, r != 9L))
expect_equal(or3s(moc < 0L, , ),
             bor3(moc < 0L, , ))
expect_equal(or3s(mod < 1L, , ),
             bor3(mod < 1L, , ))
expect_equal(or3s(moe < 9L, , ),
             bor3(moe < 9L, , ))
expect_equal(or3s(mof < 0L, , d %between% c(-1L, 1L)),
             bor3(mof < 0L, , d %between% c(-1L, 1L)))
expect_equal(or3s(mog < 1L, , y %between% c(-1L, 1L)),
             bor3(mog < 1L, , y %between% c(-1L, 1L)))
expect_equal(or3s(moh < 9L, , e %between% c(-1L, 1L)),
             bor3(moh < 9L, , e %between% c(-1L, 1L)))
expect_equal(or3s(moi < 0L, i == 0L, ),
             bor3(moi < 0L, i == 0L, ))
expect_equal(or3s(moj < 1L, h == 1L, ),
             bor3(moj < 1L, h == 1L, ))
expect_equal(or3s(mok < 9L, q == 9L, ),
             bor3(mok < 9L, q == 9L, ))
expect_equal(or3s(mol < 0L, f == 0L, r %between% c(-1L, 1L)),
             bor3(mol < 0L, f == 0L, r %between% c(-1L, 1L)))
expect_equal(or3s(mom < 1L, l == 1L, k %between% c(-1L, 1L)),
             bor3(mom < 1L, l == 1L, k %between% c(-1L, 1L)))
expect_equal(or3s(mon < 9L, l == 9L, x %between% c(-1L, 1L)),
             bor3(mon < 9L, l == 9L, x %between% c(-1L, 1L)))
expect_equal(or3s(moo < 0L, , ),
             bor3(moo < 0L, , ))
expect_equal(or3s(mop < 1L, , ),
             bor3(mop < 1L, , ))
expect_equal(or3s(moq < 9L, , ),
             bor3(moq < 9L, , ))
expect_equal(or3s(mor < 0L, , g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mor < 0L, , g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mos < 1L, , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mos < 1L, , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mot < 9L, , f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mot < 9L, , f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mou < 0L, b == 0L, ),
             bor3(mou < 0L, b == 0L, ))
expect_equal(or3s(mov < 1L, c == 1L, ),
             bor3(mov < 1L, c == 1L, ))
expect_equal(or3s(mow < 9L, c == 9L, ),
             bor3(mow < 9L, c == 9L, ))
expect_equal(or3s(mox < 0L, u == 0L, q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mox < 0L, u == 0L, q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(moy < 1L, o == 1L, a %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(moy < 1L, o == 1L, a %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(moz < 9L, x == 9L, y %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(moz < 9L, x == 9L, y %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mpa < 0L, , ),
             bor3(mpa < 0L, , ))
expect_equal(or3s(mpb < 1L, , ),
             bor3(mpb < 1L, , ))
expect_equal(or3s(mpc < 9L, , ),
             bor3(mpc < 9L, , ))
expect_equal(or3s(mpd < 0L, , y %in% 1:4),
             bor3(mpd < 0L, , y %in% 1:4))
expect_equal(or3s(mpe < 1L, , n %in% 1:4),
             bor3(mpe < 1L, , n %in% 1:4))
expect_equal(or3s(mpf < 9L, , f %in% 1:4),
             bor3(mpf < 9L, , f %in% 1:4))
expect_equal(or3s(mpg < 0L, o == 0L, ),
             bor3(mpg < 0L, o == 0L, ))
expect_equal(or3s(mph < 1L, d == 1L, ),
             bor3(mph < 1L, d == 1L, ))
expect_equal(or3s(mpi < 9L, x == 9L, ),
             bor3(mpi < 9L, x == 9L, ))
expect_equal(or3s(mpj < 0L, h == 0L, e %in% 1:4),
             bor3(mpj < 0L, h == 0L, e %in% 1:4))
expect_equal(or3s(mpk < 1L, d == 1L, y %in% 1:4),
             bor3(mpk < 1L, d == 1L, y %in% 1:4))
expect_equal(or3s(mpl < 9L, o == 9L, z %in% 1:4),
             bor3(mpl < 9L, o == 9L, z %in% 1:4))
expect_equal(or3s(mpm < 0L, , ),
             bor3(mpm < 0L, , ))
expect_equal(or3s(mpn < 1L, , ),
             bor3(mpn < 1L, , ))
expect_equal(or3s(mpo < 9L, , ),
             bor3(mpo < 9L, , ))
expect_equal(or3s(mpp < 0L, , i < 0L),
             bor3(mpp < 0L, , i < 0L))
expect_equal(or3s(mpq < 1L, , r < 1L),
             bor3(mpq < 1L, , r < 1L))
expect_equal(or3s(mpr < 9L, , w < 9L),
             bor3(mpr < 9L, , w < 9L))
expect_equal(or3s(mps < 0L, d == 0L, ),
             bor3(mps < 0L, d == 0L, ))
expect_equal(or3s(mpt < 1L, v == 1L, ),
             bor3(mpt < 1L, v == 1L, ))
expect_equal(or3s(mpu < 9L, q == 9L, ),
             bor3(mpu < 9L, q == 9L, ))
expect_equal(or3s(mpv < 0L, y == 0L, y < 0L),
             bor3(mpv < 0L, y == 0L, y < 0L))
expect_equal(or3s(mpw < 1L, h == 1L, m < 1L),
             bor3(mpw < 1L, h == 1L, m < 1L))
expect_equal(or3s(mpx < 9L, f == 9L, g < 9L),
             bor3(mpx < 9L, f == 9L, g < 9L))
expect_equal(or3s(mpy < 0L, , ),
             bor3(mpy < 0L, , ))
expect_equal(or3s(mpz < 1L, , ),
             bor3(mpz < 1L, , ))
expect_equal(or3s(mqa < 9L, , ),
             bor3(mqa < 9L, , ))
expect_equal(or3s(mqb < 0L, , d <= 0L),
             bor3(mqb < 0L, , d <= 0L))
expect_equal(or3s(mqc < 1L, , b <= 1L),
             bor3(mqc < 1L, , b <= 1L))
expect_equal(or3s(mqd < 9L, , w <= 9L),
             bor3(mqd < 9L, , w <= 9L))
expect_equal(or3s(mqe < 0L, d == 0L, ),
             bor3(mqe < 0L, d == 0L, ))
expect_equal(or3s(mqf < 1L, x == 1L, ),
             bor3(mqf < 1L, x == 1L, ))
expect_equal(or3s(mqg < 9L, d == 9L, ),
             bor3(mqg < 9L, d == 9L, ))
expect_equal(or3s(mqh < 0L, a == 0L, j <= 0L),
             bor3(mqh < 0L, a == 0L, j <= 0L))
expect_equal(or3s(mqi < 1L, c == 1L, n <= 1L),
             bor3(mqi < 1L, c == 1L, n <= 1L))
expect_equal(or3s(mqj < 9L, u == 9L, d <= 9L),
             bor3(mqj < 9L, u == 9L, d <= 9L))
expect_equal(or3s(mqk < 0L, , ),
             bor3(mqk < 0L, , ))
expect_equal(or3s(mql < 1L, , ),
             bor3(mql < 1L, , ))
expect_equal(or3s(mqm < 9L, , ),
             bor3(mqm < 9L, , ))
expect_equal(or3s(mqn < 0L, , x == 0L),
             bor3(mqn < 0L, , x == 0L))
expect_equal(or3s(mqo < 1L, , o == 1L),
             bor3(mqo < 1L, , o == 1L))
expect_equal(or3s(mqp < 9L, , u == 9L),
             bor3(mqp < 9L, , u == 9L))
expect_equal(or3s(mqq < 0L, r == 0L, ),
             bor3(mqq < 0L, r == 0L, ))
expect_equal(or3s(mqr < 1L, r == 1L, ),
             bor3(mqr < 1L, r == 1L, ))
expect_equal(or3s(mqs < 9L, d == 9L, ),
             bor3(mqs < 9L, d == 9L, ))
expect_equal(or3s(mqt < 0L, f == 0L, s == 0L),
             bor3(mqt < 0L, f == 0L, s == 0L))
expect_equal(or3s(mqu < 1L, e == 1L, t == 1L),
             bor3(mqu < 1L, e == 1L, t == 1L))
expect_equal(or3s(mqv < 9L, r == 9L, e == 9L),
             bor3(mqv < 9L, r == 9L, e == 9L))
expect_equal(or3s(mqw < 0L, , ),
             bor3(mqw < 0L, , ))
expect_equal(or3s(mqx < 1L, , ),
             bor3(mqx < 1L, , ))
expect_equal(or3s(mqy < 9L, , ),
             bor3(mqy < 9L, , ))
expect_equal(or3s(mqz < 0L, , y > 0L),
             bor3(mqz < 0L, , y > 0L))
expect_equal(or3s(mra < 1L, , q > 1L),
             bor3(mra < 1L, , q > 1L))
expect_equal(or3s(mrb < 9L, , j > 9L),
             bor3(mrb < 9L, , j > 9L))
expect_equal(or3s(mrc < 0L, l == 0L, ),
             bor3(mrc < 0L, l == 0L, ))
expect_equal(or3s(mrd < 1L, a == 1L, ),
             bor3(mrd < 1L, a == 1L, ))
expect_equal(or3s(mre < 9L, h == 9L, ),
             bor3(mre < 9L, h == 9L, ))
expect_equal(or3s(mrf < 0L, f == 0L, y > 0L),
             bor3(mrf < 0L, f == 0L, y > 0L))
expect_equal(or3s(mrg < 1L, u == 1L, w > 1L),
             bor3(mrg < 1L, u == 1L, w > 1L))
expect_equal(or3s(mrh < 9L, j == 9L, y > 9L),
             bor3(mrh < 9L, j == 9L, y > 9L))
expect_equal(or3s(mri < 0L, , ),
             bor3(mri < 0L, , ))
expect_equal(or3s(mrj < 1L, , ),
             bor3(mrj < 1L, , ))
expect_equal(or3s(mrk < 9L, , ),
             bor3(mrk < 9L, , ))
expect_equal(or3s(mrl < 0L, , i >= 0L),
             bor3(mrl < 0L, , i >= 0L))
expect_equal(or3s(mrm < 1L, , l >= 1L),
             bor3(mrm < 1L, , l >= 1L))
expect_equal(or3s(mrn < 9L, , y >= 9L),
             bor3(mrn < 9L, , y >= 9L))
expect_equal(or3s(mro < 0L, q == 0L, ),
             bor3(mro < 0L, q == 0L, ))
expect_equal(or3s(mrp < 1L, h == 1L, ),
             bor3(mrp < 1L, h == 1L, ))
expect_equal(or3s(mrq < 9L, l == 9L, ),
             bor3(mrq < 9L, l == 9L, ))
expect_equal(or3s(mrr < 0L, z == 0L, u >= 0L),
             bor3(mrr < 0L, z == 0L, u >= 0L))
expect_equal(or3s(mrs < 1L, m == 1L, p >= 1L),
             bor3(mrs < 1L, m == 1L, p >= 1L))
expect_equal(or3s(mrt < 9L, r == 9L, p >= 9L),
             bor3(mrt < 9L, r == 9L, p >= 9L))
expect_equal(or3s(mru < 0L, , ),
             bor3(mru < 0L, , ))
expect_equal(or3s(mrv < 1L, , ),
             bor3(mrv < 1L, , ))
expect_equal(or3s(mrw < 9L, , ),
             bor3(mrw < 9L, , ))
expect_equal(or3s(mrx < 0L, , logi_v),
             bor3(mrx < 0L, , logi_v))
expect_equal(or3s(mry < 1L, , logi_x),
             bor3(mry < 1L, , logi_x))
expect_equal(or3s(mrz < 9L, , logi_w),
             bor3(mrz < 9L, , logi_w))
expect_equal(or3s(msa < 0L, s > 0L, ),
             bor3(msa < 0L, s > 0L, ))
expect_equal(or3s(msb < 1L, j > 1L, ),
             bor3(msb < 1L, j > 1L, ))
expect_equal(or3s(msc < 9L, c > 9L, ),
             bor3(msc < 9L, c > 9L, ))
expect_equal(or3s(msd < 0L, z > 0L, logi_w),
             bor3(msd < 0L, z > 0L, logi_w))
expect_equal(or3s(mse < 1L, k > 1L, logi_r),
             bor3(mse < 1L, k > 1L, logi_r))
expect_equal(or3s(msf < 9L, i > 9L, logi_d),
             bor3(msf < 9L, i > 9L, logi_d))
expect_equal(or3s(msg < 0L, , ),
             bor3(msg < 0L, , ))
expect_equal(or3s(msh < 1L, , ),
             bor3(msh < 1L, , ))
expect_equal(or3s(msi < 9L, , ),
             bor3(msi < 9L, , ))
expect_equal(or3s(msj < 0L, , !logi_k),
             bor3(msj < 0L, , !logi_k))
expect_equal(or3s(msk < 1L, , !logi_v),
             bor3(msk < 1L, , !logi_v))
expect_equal(or3s(msl < 9L, , !logi_y),
             bor3(msl < 9L, , !logi_y))
expect_equal(or3s(msm < 0L, k > 0L, ),
             bor3(msm < 0L, k > 0L, ))
expect_equal(or3s(msn < 1L, h > 1L, ),
             bor3(msn < 1L, h > 1L, ))
expect_equal(or3s(mso < 9L, o > 9L, ),
             bor3(mso < 9L, o > 9L, ))
expect_equal(or3s(msp < 0L, l > 0L, !logi_e),
             bor3(msp < 0L, l > 0L, !logi_e))
expect_equal(or3s(msq < 1L, a > 1L, !logi_n),
             bor3(msq < 1L, a > 1L, !logi_n))
expect_equal(or3s(msr < 9L, p > 9L, !logi_t),
             bor3(msr < 9L, p > 9L, !logi_t))
expect_equal(or3s(mss < 0L, , ),
             bor3(mss < 0L, , ))
expect_equal(or3s(mst < 1L, , ),
             bor3(mst < 1L, , ))
expect_equal(or3s(msu < 9L, , ),
             bor3(msu < 9L, , ))
expect_equal(or3s(msv < 0L, , t != 0L),
             bor3(msv < 0L, , t != 0L))
expect_equal(or3s(msw < 1L, , c != 1L),
             bor3(msw < 1L, , c != 1L))
expect_equal(or3s(msx < 9L, , q != 9L),
             bor3(msx < 9L, , q != 9L))
expect_equal(or3s(msy < 0L, a > 0L, ),
             bor3(msy < 0L, a > 0L, ))
expect_equal(or3s(msz < 1L, p > 1L, ),
             bor3(msz < 1L, p > 1L, ))
expect_equal(or3s(mta < 9L, h > 9L, ),
             bor3(mta < 9L, h > 9L, ))
expect_equal(or3s(mtb < 0L, y > 0L, h != 0L),
             bor3(mtb < 0L, y > 0L, h != 0L))
expect_equal(or3s(mtc < 1L, c > 1L, s != 1L),
             bor3(mtc < 1L, c > 1L, s != 1L))
expect_equal(or3s(mtd < 9L, c > 9L, n != 9L),
             bor3(mtd < 9L, c > 9L, n != 9L))
expect_equal(or3s(mte < 0L, , ),
             bor3(mte < 0L, , ))
expect_equal(or3s(mtf < 1L, , ),
             bor3(mtf < 1L, , ))
expect_equal(or3s(mtg < 9L, , ),
             bor3(mtg < 9L, , ))
expect_equal(or3s(mth < 0L, , s %between% c(-1L, 1L)),
             bor3(mth < 0L, , s %between% c(-1L, 1L)))
expect_equal(or3s(mti < 1L, , p %between% c(-1L, 1L)),
             bor3(mti < 1L, , p %between% c(-1L, 1L)))
expect_equal(or3s(mtj < 9L, , y %between% c(-1L, 1L)),
             bor3(mtj < 9L, , y %between% c(-1L, 1L)))
expect_equal(or3s(mtk < 0L, m > 0L, ),
             bor3(mtk < 0L, m > 0L, ))
expect_equal(or3s(mtl < 1L, z > 1L, ),
             bor3(mtl < 1L, z > 1L, ))
expect_equal(or3s(mtm < 9L, g > 9L, ),
             bor3(mtm < 9L, g > 9L, ))
expect_equal(or3s(mtn < 0L, l > 0L, b %between% c(-1L, 1L)),
             bor3(mtn < 0L, l > 0L, b %between% c(-1L, 1L)))
expect_equal(or3s(mto < 1L, y > 1L, i %between% c(-1L, 1L)),
             bor3(mto < 1L, y > 1L, i %between% c(-1L, 1L)))
expect_equal(or3s(mtp < 9L, v > 9L, l %between% c(-1L, 1L)),
             bor3(mtp < 9L, v > 9L, l %between% c(-1L, 1L)))
expect_equal(or3s(mtq < 0L, , ),
             bor3(mtq < 0L, , ))
expect_equal(or3s(mtr < 1L, , ),
             bor3(mtr < 1L, , ))
expect_equal(or3s(mts < 9L, , ),
             bor3(mts < 9L, , ))
expect_equal(or3s(mtt < 0L, , w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mtt < 0L, , w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mtu < 1L, , f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mtu < 1L, , f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mtv < 9L, , b %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mtv < 9L, , b %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mtw < 0L, f > 0L, ),
             bor3(mtw < 0L, f > 0L, ))
expect_equal(or3s(mtx < 1L, v > 1L, ),
             bor3(mtx < 1L, v > 1L, ))
expect_equal(or3s(mty < 9L, v > 9L, ),
             bor3(mty < 9L, v > 9L, ))
expect_equal(or3s(mtz < 0L, o > 0L, n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mtz < 0L, o > 0L, n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mua < 1L, s > 1L, p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mua < 1L, s > 1L, p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(mub < 9L, m > 9L, o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(mub < 9L, m > 9L, o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(muc < 0L, , ),
             bor3(muc < 0L, , ))
expect_equal(or3s(mud < 1L, , ),
             bor3(mud < 1L, , ))
expect_equal(or3s(mue < 9L, , ),
             bor3(mue < 9L, , ))
expect_equal(or3s(muf < 0L, , a %in% 1:4),
             bor3(muf < 0L, , a %in% 1:4))
expect_equal(or3s(mug < 1L, , o %in% 1:4),
             bor3(mug < 1L, , o %in% 1:4))
expect_equal(or3s(muh < 9L, , g %in% 1:4),
             bor3(muh < 9L, , g %in% 1:4))
expect_equal(or3s(mui < 0L, e > 0L, ),
             bor3(mui < 0L, e > 0L, ))
expect_equal(or3s(muj < 1L, d > 1L, ),
             bor3(muj < 1L, d > 1L, ))
expect_equal(or3s(muk < 9L, h > 9L, ),
             bor3(muk < 9L, h > 9L, ))
expect_equal(or3s(mul < 0L, t > 0L, i %in% 1:4),
             bor3(mul < 0L, t > 0L, i %in% 1:4))
expect_equal(or3s(mum < 1L, h > 1L, a %in% 1:4),
             bor3(mum < 1L, h > 1L, a %in% 1:4))
expect_equal(or3s(mun < 9L, v > 9L, k %in% 1:4),
             bor3(mun < 9L, v > 9L, k %in% 1:4))
expect_equal(or3s(muo < 0L, , ),
             bor3(muo < 0L, , ))
expect_equal(or3s(mup < 1L, , ),
             bor3(mup < 1L, , ))
expect_equal(or3s(muq < 9L, , ),
             bor3(muq < 9L, , ))
expect_equal(or3s(mur < 0L, , y < 0L),
             bor3(mur < 0L, , y < 0L))
expect_equal(or3s(mus < 1L, , k < 1L),
             bor3(mus < 1L, , k < 1L))
expect_equal(or3s(mut < 9L, , h < 9L),
             bor3(mut < 9L, , h < 9L))
expect_equal(or3s(muu < 0L, z > 0L, ),
             bor3(muu < 0L, z > 0L, ))
expect_equal(or3s(muv < 1L, x > 1L, ),
             bor3(muv < 1L, x > 1L, ))
expect_equal(or3s(muw < 9L, c > 9L, ),
             bor3(muw < 9L, c > 9L, ))
expect_equal(or3s(mux < 0L, d > 0L, e < 0L),
             bor3(mux < 0L, d > 0L, e < 0L))
expect_equal(or3s(muy < 1L, g > 1L, v < 1L),
             bor3(muy < 1L, g > 1L, v < 1L))
expect_equal(or3s(muz < 9L, w > 9L, q < 9L),
             bor3(muz < 9L, w > 9L, q < 9L))
expect_equal(or3s(mva < 0L, , ),
             bor3(mva < 0L, , ))
expect_equal(or3s(mvb < 1L, , ),
             bor3(mvb < 1L, , ))
expect_equal(or3s(mvc < 9L, , ),
             bor3(mvc < 9L, , ))
expect_equal(or3s(mvd < 0L, , m <= 0L),
             bor3(mvd < 0L, , m <= 0L))
expect_equal(or3s(mve < 1L, , n <= 1L),
             bor3(mve < 1L, , n <= 1L))
expect_equal(or3s(mvf < 9L, , t <= 9L),
             bor3(mvf < 9L, , t <= 9L))
expect_equal(or3s(mvg < 0L, t > 0L, ),
             bor3(mvg < 0L, t > 0L, ))
expect_equal(or3s(mvh < 1L, b > 1L, ),
             bor3(mvh < 1L, b > 1L, ))
expect_equal(or3s(mvi < 9L, t > 9L, ),
             bor3(mvi < 9L, t > 9L, ))
expect_equal(or3s(mvj < 0L, t > 0L, e <= 0L),
             bor3(mvj < 0L, t > 0L, e <= 0L))
expect_equal(or3s(mvk < 1L, s > 1L, d <= 1L),
             bor3(mvk < 1L, s > 1L, d <= 1L))
expect_equal(or3s(mvl < 9L, c > 9L, b <= 9L),
             bor3(mvl < 9L, c > 9L, b <= 9L))
expect_equal(or3s(mvm < 0L, , ),
             bor3(mvm < 0L, , ))
expect_equal(or3s(mvn < 1L, , ),
             bor3(mvn < 1L, , ))
expect_equal(or3s(mvo < 9L, , ),
             bor3(mvo < 9L, , ))
expect_equal(or3s(mvp < 0L, , s == 0L),
             bor3(mvp < 0L, , s == 0L))
expect_equal(or3s(mvq < 1L, , o == 1L),
             bor3(mvq < 1L, , o == 1L))
expect_equal(or3s(mvr < 9L, , m == 9L),
             bor3(mvr < 9L, , m == 9L))
expect_equal(or3s(mvs < 0L, t > 0L, ),
             bor3(mvs < 0L, t > 0L, ))
expect_equal(or3s(mvt < 1L, a > 1L, ),
             bor3(mvt < 1L, a > 1L, ))
expect_equal(or3s(mvu < 9L, j > 9L, ),
             bor3(mvu < 9L, j > 9L, ))
expect_equal(or3s(mvv < 0L, p > 0L, y == 0L),
             bor3(mvv < 0L, p > 0L, y == 0L))
expect_equal(or3s(mvw < 1L, b > 1L, f == 1L),
             bor3(mvw < 1L, b > 1L, f == 1L))
expect_equal(or3s(mvx < 9L, k > 9L, r == 9L),
             bor3(mvx < 9L, k > 9L, r == 9L))
expect_equal(or3s(mvy < 0L, , ),
             bor3(mvy < 0L, , ))
expect_equal(or3s(mvz < 1L, , ),
             bor3(mvz < 1L, , ))
expect_equal(or3s(mwa < 9L, , ),
             bor3(mwa < 9L, , ))
expect_equal(or3s(mwb < 0L, , k > 0L),
             bor3(mwb < 0L, , k > 0L))
expect_equal(or3s(mwc < 1L, , v > 1L),
             bor3(mwc < 1L, , v > 1L))
expect_equal(or3s(mwd < 9L, , v > 9L),
             bor3(mwd < 9L, , v > 9L))
expect_equal(or3s(mwe < 0L, r > 0L, ),
             bor3(mwe < 0L, r > 0L, ))
expect_equal(or3s(mwf < 1L, n > 1L, ),
             bor3(mwf < 1L, n > 1L, ))
expect_equal(or3s(mwg < 9L, l > 9L, ),
             bor3(mwg < 9L, l > 9L, ))
expect_equal(or3s(mwh < 0L, f > 0L, k > 0L),
             bor3(mwh < 0L, f > 0L, k > 0L))
expect_equal(or3s(mwi < 1L, w > 1L, g > 1L),
             bor3(mwi < 1L, w > 1L, g > 1L))
expect_equal(or3s(mwj < 9L, j > 9L, r > 9L),
             bor3(mwj < 9L, j > 9L, r > 9L))
expect_equal(or3s(mwk < 0L, , ),
             bor3(mwk < 0L, , ))
expect_equal(or3s(mwl < 1L, , ),
             bor3(mwl < 1L, , ))
expect_equal(or3s(mwm < 9L, , ),
             bor3(mwm < 9L, , ))
expect_equal(or3s(mwn < 0L, , p >= 0L),
             bor3(mwn < 0L, , p >= 0L))
expect_equal(or3s(mwo < 1L, , z >= 1L),
             bor3(mwo < 1L, , z >= 1L))
expect_equal(or3s(mwp < 9L, , u >= 9L),
             bor3(mwp < 9L, , u >= 9L))
expect_equal(or3s(mwq < 0L, u > 0L, ),
             bor3(mwq < 0L, u > 0L, ))
expect_equal(or3s(mwr < 1L, u > 1L, ),
             bor3(mwr < 1L, u > 1L, ))
expect_equal(or3s(mws < 9L, e > 9L, ),
             bor3(mws < 9L, e > 9L, ))
expect_equal(or3s(mwt < 0L, b > 0L, e >= 0L),
             bor3(mwt < 0L, b > 0L, e >= 0L))
expect_equal(or3s(mwu < 1L, x > 1L, s >= 1L),
             bor3(mwu < 1L, x > 1L, s >= 1L))
expect_equal(or3s(mwv < 9L, h > 9L, r >= 9L),
             bor3(mwv < 9L, h > 9L, r >= 9L))
expect_equal(or3s(mww < 0L, , ),
             bor3(mww < 0L, , ))
expect_equal(or3s(mwx < 1L, , ),
             bor3(mwx < 1L, , ))
expect_equal(or3s(mwy < 9L, , ),
             bor3(mwy < 9L, , ))
expect_equal(or3s(mwz < 0L, , logi_m),
             bor3(mwz < 0L, , logi_m))
expect_equal(or3s(mxa < 1L, , logi_r),
             bor3(mxa < 1L, , logi_r))
expect_equal(or3s(mxb < 9L, , logi_k),
             bor3(mxb < 9L, , logi_k))
expect_equal(or3s(mxc < 0L, b >= 0L, ),
             bor3(mxc < 0L, b >= 0L, ))
expect_equal(or3s(mxd < 1L, s >= 1L, ),
             bor3(mxd < 1L, s >= 1L, ))
expect_equal(or3s(mxe < 9L, s >= 9L, ),
             bor3(mxe < 9L, s >= 9L, ))
expect_equal(or3s(mxf < 0L, l >= 0L, logi_k),
             bor3(mxf < 0L, l >= 0L, logi_k))
expect_equal(or3s(mxg < 1L, d >= 1L, logi_c),
             bor3(mxg < 1L, d >= 1L, logi_c))
expect_equal(or3s(mxh < 9L, c >= 9L, logi_v),
             bor3(mxh < 9L, c >= 9L, logi_v))
expect_equal(or3s(mxi < 0L, , ),
             bor3(mxi < 0L, , ))
expect_equal(or3s(mxj < 1L, , ),
             bor3(mxj < 1L, , ))
expect_equal(or3s(mxk < 9L, , ),
             bor3(mxk < 9L, , ))
expect_equal(or3s(mxl < 0L, , !logi_v),
             bor3(mxl < 0L, , !logi_v))
expect_equal(or3s(mxm < 1L, , !logi_g),
             bor3(mxm < 1L, , !logi_g))
expect_equal(or3s(mxn < 9L, , !logi_u),
             bor3(mxn < 9L, , !logi_u))
expect_equal(or3s(mxo < 0L, l >= 0L, ),
             bor3(mxo < 0L, l >= 0L, ))
expect_equal(or3s(mxp < 1L, l >= 1L, ),
             bor3(mxp < 1L, l >= 1L, ))
expect_equal(or3s(mxq < 9L, v >= 9L, ),
             bor3(mxq < 9L, v >= 9L, ))
expect_equal(or3s(mxr < 0L, r >= 0L, !logi_m),
             bor3(mxr < 0L, r >= 0L, !logi_m))
expect_equal(or3s(mxs < 1L, z >= 1L, !logi_n),
             bor3(mxs < 1L, z >= 1L, !logi_n))
expect_equal(or3s(mxt < 9L, p >= 9L, !logi_a),
             bor3(mxt < 9L, p >= 9L, !logi_a))
expect_equal(or3s(mxu < 0L, , ),
             bor3(mxu < 0L, , ))
expect_equal(or3s(mxv < 1L, , ),
             bor3(mxv < 1L, , ))
expect_equal(or3s(mxw < 9L, , ),
             bor3(mxw < 9L, , ))
expect_equal(or3s(mxx < 0L, , r != 0L),
             bor3(mxx < 0L, , r != 0L))
expect_equal(or3s(mxy < 1L, , i != 1L),
             bor3(mxy < 1L, , i != 1L))
expect_equal(or3s(mxz < 9L, , r != 9L),
             bor3(mxz < 9L, , r != 9L))
expect_equal(or3s(mya < 0L, q >= 0L, ),
             bor3(mya < 0L, q >= 0L, ))
expect_equal(or3s(myb < 1L, x >= 1L, ),
             bor3(myb < 1L, x >= 1L, ))
expect_equal(or3s(myc < 9L, p >= 9L, ),
             bor3(myc < 9L, p >= 9L, ))
expect_equal(or3s(myd < 0L, p >= 0L, l != 0L),
             bor3(myd < 0L, p >= 0L, l != 0L))
expect_equal(or3s(mye < 1L, j >= 1L, c != 1L),
             bor3(mye < 1L, j >= 1L, c != 1L))
expect_equal(or3s(myf < 9L, f >= 9L, h != 9L),
             bor3(myf < 9L, f >= 9L, h != 9L))
expect_equal(or3s(myg < 0L, , ),
             bor3(myg < 0L, , ))
expect_equal(or3s(myh < 1L, , ),
             bor3(myh < 1L, , ))
expect_equal(or3s(myi < 9L, , ),
             bor3(myi < 9L, , ))
expect_equal(or3s(myj < 0L, , a %between% c(-1L, 1L)),
             bor3(myj < 0L, , a %between% c(-1L, 1L)))
expect_equal(or3s(myk < 1L, , y %between% c(-1L, 1L)),
             bor3(myk < 1L, , y %between% c(-1L, 1L)))
expect_equal(or3s(myl < 9L, , r %between% c(-1L, 1L)),
             bor3(myl < 9L, , r %between% c(-1L, 1L)))
expect_equal(or3s(mym < 0L, f >= 0L, ),
             bor3(mym < 0L, f >= 0L, ))
expect_equal(or3s(myn < 1L, y >= 1L, ),
             bor3(myn < 1L, y >= 1L, ))
expect_equal(or3s(myo < 9L, i >= 9L, ),
             bor3(myo < 9L, i >= 9L, ))
expect_equal(or3s(myp < 0L, x >= 0L, m %between% c(-1L, 1L)),
             bor3(myp < 0L, x >= 0L, m %between% c(-1L, 1L)))
expect_equal(or3s(myq < 1L, h >= 1L, g %between% c(-1L, 1L)),
             bor3(myq < 1L, h >= 1L, g %between% c(-1L, 1L)))
expect_equal(or3s(myr < 9L, c >= 9L, q %between% c(-1L, 1L)),
             bor3(myr < 9L, c >= 9L, q %between% c(-1L, 1L)))
expect_equal(or3s(mys < 0L, , ),
             bor3(mys < 0L, , ))
expect_equal(or3s(myt < 1L, , ),
             bor3(myt < 1L, , ))
expect_equal(or3s(myu < 9L, , ),
             bor3(myu < 9L, , ))
expect_equal(or3s(myv < 0L, , f %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(myv < 0L, , f %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(myw < 1L, , e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(myw < 1L, , e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(myx < 9L, , z %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(myx < 9L, , z %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(myy < 0L, j >= 0L, ),
             bor3(myy < 0L, j >= 0L, ))
expect_equal(or3s(myz < 1L, i >= 1L, ),
             bor3(myz < 1L, i >= 1L, ))
expect_equal(or3s(na < 9L, u >= 9L, ),
             bor3(na < 9L, u >= 9L, ))
expect_equal(or3s(nb < 0L, g >= 0L, d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nb < 0L, g >= 0L, d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nc < 1L, c >= 1L, v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nc < 1L, c >= 1L, v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nd < 9L, l >= 9L, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nd < 9L, l >= 9L, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ne < 0L, , ),
             bor3(ne < 0L, , ))
expect_equal(or3s(nf < 1L, , ),
             bor3(nf < 1L, , ))
expect_equal(or3s(ng < 9L, , ),
             bor3(ng < 9L, , ))
expect_equal(or3s(nh < 0L, , q %in% 1:4),
             bor3(nh < 0L, , q %in% 1:4))
expect_equal(or3s(ni < 1L, , g %in% 1:4),
             bor3(ni < 1L, , g %in% 1:4))
expect_equal(or3s(nj < 9L, , l %in% 1:4),
             bor3(nj < 9L, , l %in% 1:4))
expect_equal(or3s(nk < 0L, g >= 0L, ),
             bor3(nk < 0L, g >= 0L, ))
expect_equal(or3s(nl < 1L, u >= 1L, ),
             bor3(nl < 1L, u >= 1L, ))
expect_equal(or3s(nm < 9L, z >= 9L, ),
             bor3(nm < 9L, z >= 9L, ))
expect_equal(or3s(nn < 0L, z >= 0L, g %in% 1:4),
             bor3(nn < 0L, z >= 0L, g %in% 1:4))
expect_equal(or3s(no < 1L, y >= 1L, x %in% 1:4),
             bor3(no < 1L, y >= 1L, x %in% 1:4))
expect_equal(or3s(np < 9L, l >= 9L, a %in% 1:4),
             bor3(np < 9L, l >= 9L, a %in% 1:4))
expect_equal(or3s(nq < 0L, , ),
             bor3(nq < 0L, , ))
expect_equal(or3s(nr < 1L, , ),
             bor3(nr < 1L, , ))
expect_equal(or3s(ns < 9L, , ),
             bor3(ns < 9L, , ))
expect_equal(or3s(nt < 0L, , r < 0L),
             bor3(nt < 0L, , r < 0L))
expect_equal(or3s(nu < 1L, , g < 1L),
             bor3(nu < 1L, , g < 1L))
expect_equal(or3s(nv < 9L, , d < 9L),
             bor3(nv < 9L, , d < 9L))
expect_equal(or3s(nw < 0L, c >= 0L, ),
             bor3(nw < 0L, c >= 0L, ))
expect_equal(or3s(nx < 1L, f >= 1L, ),
             bor3(nx < 1L, f >= 1L, ))
expect_equal(or3s(ny < 9L, c >= 9L, ),
             bor3(ny < 9L, c >= 9L, ))
expect_equal(or3s(nz < 0L, g >= 0L, e < 0L),
             bor3(nz < 0L, g >= 0L, e < 0L))
expect_equal(or3s(naa < 1L, m >= 1L, t < 1L),
             bor3(naa < 1L, m >= 1L, t < 1L))
expect_equal(or3s(nab < 9L, d >= 9L, l < 9L),
             bor3(nab < 9L, d >= 9L, l < 9L))
expect_equal(or3s(nac < 0L, , ),
             bor3(nac < 0L, , ))
expect_equal(or3s(nad < 1L, , ),
             bor3(nad < 1L, , ))
expect_equal(or3s(nae < 9L, , ),
             bor3(nae < 9L, , ))
expect_equal(or3s(naf < 0L, , c <= 0L),
             bor3(naf < 0L, , c <= 0L))
expect_equal(or3s(nag < 1L, , x <= 1L),
             bor3(nag < 1L, , x <= 1L))
expect_equal(or3s(nah < 9L, , h <= 9L),
             bor3(nah < 9L, , h <= 9L))
expect_equal(or3s(nai < 0L, z >= 0L, ),
             bor3(nai < 0L, z >= 0L, ))
expect_equal(or3s(naj < 1L, c >= 1L, ),
             bor3(naj < 1L, c >= 1L, ))
expect_equal(or3s(nak < 9L, m >= 9L, ),
             bor3(nak < 9L, m >= 9L, ))
expect_equal(or3s(nal < 0L, a >= 0L, k <= 0L),
             bor3(nal < 0L, a >= 0L, k <= 0L))
expect_equal(or3s(nam < 1L, m >= 1L, b <= 1L),
             bor3(nam < 1L, m >= 1L, b <= 1L))
expect_equal(or3s(nan < 9L, u >= 9L, m <= 9L),
             bor3(nan < 9L, u >= 9L, m <= 9L))
expect_equal(or3s(nao < 0L, , ),
             bor3(nao < 0L, , ))
expect_equal(or3s(nap < 1L, , ),
             bor3(nap < 1L, , ))
expect_equal(or3s(naq < 9L, , ),
             bor3(naq < 9L, , ))
expect_equal(or3s(nar < 0L, , g == 0L),
             bor3(nar < 0L, , g == 0L))
expect_equal(or3s(nas < 1L, , n == 1L),
             bor3(nas < 1L, , n == 1L))
expect_equal(or3s(nat < 9L, , c == 9L),
             bor3(nat < 9L, , c == 9L))
expect_equal(or3s(nau < 0L, x >= 0L, ),
             bor3(nau < 0L, x >= 0L, ))
expect_equal(or3s(nav < 1L, r >= 1L, ),
             bor3(nav < 1L, r >= 1L, ))
expect_equal(or3s(naw < 9L, f >= 9L, ),
             bor3(naw < 9L, f >= 9L, ))
expect_equal(or3s(nax < 0L, j >= 0L, f == 0L),
             bor3(nax < 0L, j >= 0L, f == 0L))
expect_equal(or3s(nay < 1L, z >= 1L, v == 1L),
             bor3(nay < 1L, z >= 1L, v == 1L))
expect_equal(or3s(naz < 9L, v >= 9L, y == 9L),
             bor3(naz < 9L, v >= 9L, y == 9L))
expect_equal(or3s(nba < 0L, , ),
             bor3(nba < 0L, , ))
expect_equal(or3s(nbb < 1L, , ),
             bor3(nbb < 1L, , ))
expect_equal(or3s(nbc < 9L, , ),
             bor3(nbc < 9L, , ))
expect_equal(or3s(nbd < 0L, , i > 0L),
             bor3(nbd < 0L, , i > 0L))
expect_equal(or3s(nbe < 1L, , q > 1L),
             bor3(nbe < 1L, , q > 1L))
expect_equal(or3s(nbf < 9L, , j > 9L),
             bor3(nbf < 9L, , j > 9L))
expect_equal(or3s(nbg < 0L, g >= 0L, ),
             bor3(nbg < 0L, g >= 0L, ))
expect_equal(or3s(nbh < 1L, o >= 1L, ),
             bor3(nbh < 1L, o >= 1L, ))
expect_equal(or3s(nbi < 9L, t >= 9L, ),
             bor3(nbi < 9L, t >= 9L, ))
expect_equal(or3s(nbj < 0L, j >= 0L, a > 0L),
             bor3(nbj < 0L, j >= 0L, a > 0L))
expect_equal(or3s(nbk < 1L, y >= 1L, o > 1L),
             bor3(nbk < 1L, y >= 1L, o > 1L))
expect_equal(or3s(nbl < 9L, f >= 9L, w > 9L),
             bor3(nbl < 9L, f >= 9L, w > 9L))
expect_equal(or3s(nbm < 0L, , ),
             bor3(nbm < 0L, , ))
expect_equal(or3s(nbn < 1L, , ),
             bor3(nbn < 1L, , ))
expect_equal(or3s(nbo < 9L, , ),
             bor3(nbo < 9L, , ))
expect_equal(or3s(nbp < 0L, , u >= 0L),
             bor3(nbp < 0L, , u >= 0L))
expect_equal(or3s(nbq < 1L, , k >= 1L),
             bor3(nbq < 1L, , k >= 1L))
expect_equal(or3s(nbr < 9L, , y >= 9L),
             bor3(nbr < 9L, , y >= 9L))
expect_equal(or3s(nbs < 0L, u >= 0L, ),
             bor3(nbs < 0L, u >= 0L, ))
expect_equal(or3s(nbt < 1L, y >= 1L, ),
             bor3(nbt < 1L, y >= 1L, ))
expect_equal(or3s(nbu < 9L, l >= 9L, ),
             bor3(nbu < 9L, l >= 9L, ))
expect_equal(or3s(nbv < 0L, g >= 0L, k >= 0L),
             bor3(nbv < 0L, g >= 0L, k >= 0L))
expect_equal(or3s(nbw < 1L, r >= 1L, k >= 1L),
             bor3(nbw < 1L, r >= 1L, k >= 1L))
expect_equal(or3s(nbx < 9L, c >= 9L, y >= 9L),
             bor3(nbx < 9L, c >= 9L, y >= 9L))
expect_equal(or3s(nby <= 0L, , ),
             bor3(nby <= 0L, , ))
expect_equal(or3s(nbz <= 1L, , ),
             bor3(nbz <= 1L, , ))
expect_equal(or3s(nca <= 9L, , ),
             bor3(nca <= 9L, , ))
expect_equal(or3s(ncb <= 0L, , logi_e),
             bor3(ncb <= 0L, , logi_e))
expect_equal(or3s(ncc <= 1L, , logi_r),
             bor3(ncc <= 1L, , logi_r))
expect_equal(or3s(ncd <= 9L, , logi_e),
             bor3(ncd <= 9L, , logi_e))
expect_equal(or3s(nce <= 0L, logi_i, ),
             bor3(nce <= 0L, logi_i, ))
expect_equal(or3s(ncf <= 1L, logi_u, ),
             bor3(ncf <= 1L, logi_u, ))
expect_equal(or3s(ncg <= 9L, logi_p, ),
             bor3(ncg <= 9L, logi_p, ))
expect_equal(or3s(nch <= 0L, logi_x, logi_c),
             bor3(nch <= 0L, logi_x, logi_c))
expect_equal(or3s(nci <= 1L, logi_r, logi_w),
             bor3(nci <= 1L, logi_r, logi_w))
expect_equal(or3s(ncj <= 9L, logi_n, logi_z),
             bor3(ncj <= 9L, logi_n, logi_z))
expect_equal(or3s(nck <= 0L, , ),
             bor3(nck <= 0L, , ))
expect_equal(or3s(ncl <= 1L, , ),
             bor3(ncl <= 1L, , ))
expect_equal(or3s(ncm <= 9L, , ),
             bor3(ncm <= 9L, , ))
expect_equal(or3s(ncn <= 0L, , !logi_e),
             bor3(ncn <= 0L, , !logi_e))
expect_equal(or3s(nco <= 1L, , !logi_u),
             bor3(nco <= 1L, , !logi_u))
expect_equal(or3s(ncp <= 9L, , !logi_q),
             bor3(ncp <= 9L, , !logi_q))
expect_equal(or3s(ncq <= 0L, logi_w, ),
             bor3(ncq <= 0L, logi_w, ))
expect_equal(or3s(ncr <= 1L, logi_d, ),
             bor3(ncr <= 1L, logi_d, ))
expect_equal(or3s(ncs <= 9L, logi_v, ),
             bor3(ncs <= 9L, logi_v, ))
expect_equal(or3s(nct <= 0L, logi_k, !logi_j),
             bor3(nct <= 0L, logi_k, !logi_j))
expect_equal(or3s(ncu <= 1L, logi_j, !logi_k),
             bor3(ncu <= 1L, logi_j, !logi_k))
expect_equal(or3s(ncv <= 9L, logi_z, !logi_q),
             bor3(ncv <= 9L, logi_z, !logi_q))
expect_equal(or3s(ncw <= 0L, , ),
             bor3(ncw <= 0L, , ))
expect_equal(or3s(ncx <= 1L, , ),
             bor3(ncx <= 1L, , ))
expect_equal(or3s(ncy <= 9L, , ),
             bor3(ncy <= 9L, , ))
expect_equal(or3s(ncz <= 0L, , c != 0L),
             bor3(ncz <= 0L, , c != 0L))
expect_equal(or3s(nda <= 1L, , d != 1L),
             bor3(nda <= 1L, , d != 1L))
expect_equal(or3s(ndb <= 9L, , o != 9L),
             bor3(ndb <= 9L, , o != 9L))
expect_equal(or3s(ndc <= 0L, logi_a, ),
             bor3(ndc <= 0L, logi_a, ))
expect_equal(or3s(ndd <= 1L, logi_r, ),
             bor3(ndd <= 1L, logi_r, ))
expect_equal(or3s(nde <= 9L, logi_o, ),
             bor3(nde <= 9L, logi_o, ))
expect_equal(or3s(ndf <= 0L, logi_n, f != 0L),
             bor3(ndf <= 0L, logi_n, f != 0L))
expect_equal(or3s(ndg <= 1L, logi_e, e != 1L),
             bor3(ndg <= 1L, logi_e, e != 1L))
expect_equal(or3s(ndh <= 9L, logi_z, k != 9L),
             bor3(ndh <= 9L, logi_z, k != 9L))
expect_equal(or3s(ndi <= 0L, , ),
             bor3(ndi <= 0L, , ))
expect_equal(or3s(ndj <= 1L, , ),
             bor3(ndj <= 1L, , ))
expect_equal(or3s(ndk <= 9L, , ),
             bor3(ndk <= 9L, , ))
expect_equal(or3s(ndl <= 0L, , u %between% c(-1L, 1L)),
             bor3(ndl <= 0L, , u %between% c(-1L, 1L)))
expect_equal(or3s(ndm <= 1L, , t %between% c(-1L, 1L)),
             bor3(ndm <= 1L, , t %between% c(-1L, 1L)))
expect_equal(or3s(ndn <= 9L, , l %between% c(-1L, 1L)),
             bor3(ndn <= 9L, , l %between% c(-1L, 1L)))
expect_equal(or3s(ndo <= 0L, logi_s, ),
             bor3(ndo <= 0L, logi_s, ))
expect_equal(or3s(ndp <= 1L, logi_d, ),
             bor3(ndp <= 1L, logi_d, ))
expect_equal(or3s(ndq <= 9L, logi_e, ),
             bor3(ndq <= 9L, logi_e, ))
expect_equal(or3s(ndr <= 0L, logi_h, z %between% c(-1L, 1L)),
             bor3(ndr <= 0L, logi_h, z %between% c(-1L, 1L)))
expect_equal(or3s(nds <= 1L, logi_w, w %between% c(-1L, 1L)),
             bor3(nds <= 1L, logi_w, w %between% c(-1L, 1L)))
expect_equal(or3s(ndt <= 9L, logi_n, y %between% c(-1L, 1L)),
             bor3(ndt <= 9L, logi_n, y %between% c(-1L, 1L)))
expect_equal(or3s(ndu <= 0L, , ),
             bor3(ndu <= 0L, , ))
expect_equal(or3s(ndv <= 1L, , ),
             bor3(ndv <= 1L, , ))
expect_equal(or3s(ndw <= 9L, , ),
             bor3(ndw <= 9L, , ))
expect_equal(or3s(ndx <= 0L, , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ndx <= 0L, , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ndy <= 1L, , l %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ndy <= 1L, , l %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ndz <= 9L, , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ndz <= 9L, , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nea <= 0L, logi_t, ),
             bor3(nea <= 0L, logi_t, ))
expect_equal(or3s(neb <= 1L, logi_m, ),
             bor3(neb <= 1L, logi_m, ))
expect_equal(or3s(nec <= 9L, logi_r, ),
             bor3(nec <= 9L, logi_r, ))
expect_equal(or3s(ned <= 0L, logi_f, r %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ned <= 0L, logi_f, r %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nee <= 1L, logi_l, s %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nee <= 1L, logi_l, s %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nef <= 9L, logi_l, q %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nef <= 9L, logi_l, q %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(neg <= 0L, , ),
             bor3(neg <= 0L, , ))
expect_equal(or3s(neh <= 1L, , ),
             bor3(neh <= 1L, , ))
expect_equal(or3s(nei <= 9L, , ),
             bor3(nei <= 9L, , ))
expect_equal(or3s(nej <= 0L, , r %in% 1:4),
             bor3(nej <= 0L, , r %in% 1:4))
expect_equal(or3s(nek <= 1L, , o %in% 1:4),
             bor3(nek <= 1L, , o %in% 1:4))
expect_equal(or3s(nel <= 9L, , c %in% 1:4),
             bor3(nel <= 9L, , c %in% 1:4))
expect_equal(or3s(nem <= 0L, logi_t, ),
             bor3(nem <= 0L, logi_t, ))
expect_equal(or3s(nen <= 1L, logi_a, ),
             bor3(nen <= 1L, logi_a, ))
expect_equal(or3s(neo <= 9L, logi_q, ),
             bor3(neo <= 9L, logi_q, ))
expect_equal(or3s(nep <= 0L, logi_h, q %in% 1:4),
             bor3(nep <= 0L, logi_h, q %in% 1:4))
expect_equal(or3s(neq <= 1L, logi_b, e %in% 1:4),
             bor3(neq <= 1L, logi_b, e %in% 1:4))
expect_equal(or3s(ner <= 9L, logi_w, o %in% 1:4),
             bor3(ner <= 9L, logi_w, o %in% 1:4))
expect_equal(or3s(nes <= 0L, , ),
             bor3(nes <= 0L, , ))
expect_equal(or3s(net <= 1L, , ),
             bor3(net <= 1L, , ))
expect_equal(or3s(neu <= 9L, , ),
             bor3(neu <= 9L, , ))
expect_equal(or3s(nev <= 0L, , y < 0L),
             bor3(nev <= 0L, , y < 0L))
expect_equal(or3s(new <= 1L, , m < 1L),
             bor3(new <= 1L, , m < 1L))
expect_equal(or3s(nex <= 9L, , v < 9L),
             bor3(nex <= 9L, , v < 9L))
expect_equal(or3s(ney <= 0L, logi_c, ),
             bor3(ney <= 0L, logi_c, ))
expect_equal(or3s(nez <= 1L, logi_g, ),
             bor3(nez <= 1L, logi_g, ))
expect_equal(or3s(nfa <= 9L, logi_e, ),
             bor3(nfa <= 9L, logi_e, ))
expect_equal(or3s(nfb <= 0L, logi_y, k < 0L),
             bor3(nfb <= 0L, logi_y, k < 0L))
expect_equal(or3s(nfc <= 1L, logi_o, f < 1L),
             bor3(nfc <= 1L, logi_o, f < 1L))
expect_equal(or3s(nfd <= 9L, logi_b, j < 9L),
             bor3(nfd <= 9L, logi_b, j < 9L))
expect_equal(or3s(nfe <= 0L, , ),
             bor3(nfe <= 0L, , ))
expect_equal(or3s(nff <= 1L, , ),
             bor3(nff <= 1L, , ))
expect_equal(or3s(nfg <= 9L, , ),
             bor3(nfg <= 9L, , ))
expect_equal(or3s(nfh <= 0L, , d <= 0L),
             bor3(nfh <= 0L, , d <= 0L))
expect_equal(or3s(nfi <= 1L, , h <= 1L),
             bor3(nfi <= 1L, , h <= 1L))
expect_equal(or3s(nfj <= 9L, , c <= 9L),
             bor3(nfj <= 9L, , c <= 9L))
expect_equal(or3s(nfk <= 0L, logi_u, ),
             bor3(nfk <= 0L, logi_u, ))
expect_equal(or3s(nfl <= 1L, logi_c, ),
             bor3(nfl <= 1L, logi_c, ))
expect_equal(or3s(nfm <= 9L, logi_c, ),
             bor3(nfm <= 9L, logi_c, ))
expect_equal(or3s(nfn <= 0L, logi_e, c <= 0L),
             bor3(nfn <= 0L, logi_e, c <= 0L))
expect_equal(or3s(nfo <= 1L, logi_b, r <= 1L),
             bor3(nfo <= 1L, logi_b, r <= 1L))
expect_equal(or3s(nfp <= 9L, logi_l, x <= 9L),
             bor3(nfp <= 9L, logi_l, x <= 9L))
expect_equal(or3s(nfq <= 0L, , ),
             bor3(nfq <= 0L, , ))
expect_equal(or3s(nfr <= 1L, , ),
             bor3(nfr <= 1L, , ))
expect_equal(or3s(nfs <= 9L, , ),
             bor3(nfs <= 9L, , ))
expect_equal(or3s(nft <= 0L, , a == 0L),
             bor3(nft <= 0L, , a == 0L))
expect_equal(or3s(nfu <= 1L, , h == 1L),
             bor3(nfu <= 1L, , h == 1L))
expect_equal(or3s(nfv <= 9L, , c == 9L),
             bor3(nfv <= 9L, , c == 9L))
expect_equal(or3s(nfw <= 0L, logi_x, ),
             bor3(nfw <= 0L, logi_x, ))
expect_equal(or3s(nfx <= 1L, logi_t, ),
             bor3(nfx <= 1L, logi_t, ))
expect_equal(or3s(nfy <= 9L, logi_j, ),
             bor3(nfy <= 9L, logi_j, ))
expect_equal(or3s(nfz <= 0L, logi_o, n == 0L),
             bor3(nfz <= 0L, logi_o, n == 0L))
expect_equal(or3s(nga <= 1L, logi_t, s == 1L),
             bor3(nga <= 1L, logi_t, s == 1L))
expect_equal(or3s(ngb <= 9L, logi_c, h == 9L),
             bor3(ngb <= 9L, logi_c, h == 9L))
expect_equal(or3s(ngc <= 0L, , ),
             bor3(ngc <= 0L, , ))
expect_equal(or3s(ngd <= 1L, , ),
             bor3(ngd <= 1L, , ))
expect_equal(or3s(nge <= 9L, , ),
             bor3(nge <= 9L, , ))
expect_equal(or3s(ngf <= 0L, , n > 0L),
             bor3(ngf <= 0L, , n > 0L))
expect_equal(or3s(ngg <= 1L, , p > 1L),
             bor3(ngg <= 1L, , p > 1L))
expect_equal(or3s(ngh <= 9L, , a > 9L),
             bor3(ngh <= 9L, , a > 9L))
expect_equal(or3s(ngi <= 0L, logi_w, ),
             bor3(ngi <= 0L, logi_w, ))
expect_equal(or3s(ngj <= 1L, logi_n, ),
             bor3(ngj <= 1L, logi_n, ))
expect_equal(or3s(ngk <= 9L, logi_s, ),
             bor3(ngk <= 9L, logi_s, ))
expect_equal(or3s(ngl <= 0L, logi_f, p > 0L),
             bor3(ngl <= 0L, logi_f, p > 0L))
expect_equal(or3s(ngm <= 1L, logi_q, r > 1L),
             bor3(ngm <= 1L, logi_q, r > 1L))
expect_equal(or3s(ngn <= 9L, logi_p, e > 9L),
             bor3(ngn <= 9L, logi_p, e > 9L))
expect_equal(or3s(ngo <= 0L, , ),
             bor3(ngo <= 0L, , ))
expect_equal(or3s(ngp <= 1L, , ),
             bor3(ngp <= 1L, , ))
expect_equal(or3s(ngq <= 9L, , ),
             bor3(ngq <= 9L, , ))
expect_equal(or3s(ngr <= 0L, , z >= 0L),
             bor3(ngr <= 0L, , z >= 0L))
expect_equal(or3s(ngs <= 1L, , m >= 1L),
             bor3(ngs <= 1L, , m >= 1L))
expect_equal(or3s(ngt <= 9L, , e >= 9L),
             bor3(ngt <= 9L, , e >= 9L))
expect_equal(or3s(ngu <= 0L, logi_k, ),
             bor3(ngu <= 0L, logi_k, ))
expect_equal(or3s(ngv <= 1L, logi_l, ),
             bor3(ngv <= 1L, logi_l, ))
expect_equal(or3s(ngw <= 9L, logi_b, ),
             bor3(ngw <= 9L, logi_b, ))
expect_equal(or3s(ngx <= 0L, logi_v, n >= 0L),
             bor3(ngx <= 0L, logi_v, n >= 0L))
expect_equal(or3s(ngy <= 1L, logi_c, p >= 1L),
             bor3(ngy <= 1L, logi_c, p >= 1L))
expect_equal(or3s(ngz <= 9L, logi_i, i >= 9L),
             bor3(ngz <= 9L, logi_i, i >= 9L))
expect_equal(or3s(nha <= 0L, , ),
             bor3(nha <= 0L, , ))
expect_equal(or3s(nhb <= 1L, , ),
             bor3(nhb <= 1L, , ))
expect_equal(or3s(nhc <= 9L, , ),
             bor3(nhc <= 9L, , ))
expect_equal(or3s(nhd <= 0L, , logi_h),
             bor3(nhd <= 0L, , logi_h))
expect_equal(or3s(nhe <= 1L, , logi_b),
             bor3(nhe <= 1L, , logi_b))
expect_equal(or3s(nhf <= 9L, , logi_u),
             bor3(nhf <= 9L, , logi_u))
expect_equal(or3s(nhg <= 0L, !logi_u, ),
             bor3(nhg <= 0L, !logi_u, ))
expect_equal(or3s(nhh <= 1L, !logi_j, ),
             bor3(nhh <= 1L, !logi_j, ))
expect_equal(or3s(nhi <= 9L, !logi_b, ),
             bor3(nhi <= 9L, !logi_b, ))
expect_equal(or3s(nhj <= 0L, !logi_n, logi_p),
             bor3(nhj <= 0L, !logi_n, logi_p))
expect_equal(or3s(nhk <= 1L, !logi_o, logi_i),
             bor3(nhk <= 1L, !logi_o, logi_i))
expect_equal(or3s(nhl <= 9L, !logi_j, logi_e),
             bor3(nhl <= 9L, !logi_j, logi_e))
expect_equal(or3s(nhm <= 0L, , ),
             bor3(nhm <= 0L, , ))
expect_equal(or3s(nhn <= 1L, , ),
             bor3(nhn <= 1L, , ))
expect_equal(or3s(nho <= 9L, , ),
             bor3(nho <= 9L, , ))
expect_equal(or3s(nhp <= 0L, , !logi_s),
             bor3(nhp <= 0L, , !logi_s))
expect_equal(or3s(nhq <= 1L, , !logi_w),
             bor3(nhq <= 1L, , !logi_w))
expect_equal(or3s(nhr <= 9L, , !logi_w),
             bor3(nhr <= 9L, , !logi_w))
expect_equal(or3s(nhs <= 0L, !logi_w, ),
             bor3(nhs <= 0L, !logi_w, ))
expect_equal(or3s(nht <= 1L, !logi_b, ),
             bor3(nht <= 1L, !logi_b, ))
expect_equal(or3s(nhu <= 9L, !logi_j, ),
             bor3(nhu <= 9L, !logi_j, ))
expect_equal(or3s(nhv <= 0L, !logi_j, !logi_y),
             bor3(nhv <= 0L, !logi_j, !logi_y))
expect_equal(or3s(nhw <= 1L, !logi_e, !logi_k),
             bor3(nhw <= 1L, !logi_e, !logi_k))
expect_equal(or3s(nhx <= 9L, !logi_y, !logi_w),
             bor3(nhx <= 9L, !logi_y, !logi_w))
expect_equal(or3s(nhy <= 0L, , ),
             bor3(nhy <= 0L, , ))
expect_equal(or3s(nhz <= 1L, , ),
             bor3(nhz <= 1L, , ))
expect_equal(or3s(nia <= 9L, , ),
             bor3(nia <= 9L, , ))
expect_equal(or3s(nib <= 0L, , j != 0L),
             bor3(nib <= 0L, , j != 0L))
expect_equal(or3s(nic <= 1L, , d != 1L),
             bor3(nic <= 1L, , d != 1L))
expect_equal(or3s(nid <= 9L, , e != 9L),
             bor3(nid <= 9L, , e != 9L))
expect_equal(or3s(nie <= 0L, !logi_q, ),
             bor3(nie <= 0L, !logi_q, ))
expect_equal(or3s(nif <= 1L, !logi_u, ),
             bor3(nif <= 1L, !logi_u, ))
expect_equal(or3s(nig <= 9L, !logi_y, ),
             bor3(nig <= 9L, !logi_y, ))
expect_equal(or3s(nih <= 0L, !logi_h, e != 0L),
             bor3(nih <= 0L, !logi_h, e != 0L))
expect_equal(or3s(nii <= 1L, !logi_z, i != 1L),
             bor3(nii <= 1L, !logi_z, i != 1L))
expect_equal(or3s(nij <= 9L, !logi_o, s != 9L),
             bor3(nij <= 9L, !logi_o, s != 9L))
expect_equal(or3s(nik <= 0L, , ),
             bor3(nik <= 0L, , ))
expect_equal(or3s(nil <= 1L, , ),
             bor3(nil <= 1L, , ))
expect_equal(or3s(nim <= 9L, , ),
             bor3(nim <= 9L, , ))
expect_equal(or3s(nin <= 0L, , k %between% c(-1L, 1L)),
             bor3(nin <= 0L, , k %between% c(-1L, 1L)))
expect_equal(or3s(nio <= 1L, , q %between% c(-1L, 1L)),
             bor3(nio <= 1L, , q %between% c(-1L, 1L)))
expect_equal(or3s(nip <= 9L, , c %between% c(-1L, 1L)),
             bor3(nip <= 9L, , c %between% c(-1L, 1L)))
expect_equal(or3s(niq <= 0L, !logi_a, ),
             bor3(niq <= 0L, !logi_a, ))
expect_equal(or3s(nir <= 1L, !logi_h, ),
             bor3(nir <= 1L, !logi_h, ))
expect_equal(or3s(nis <= 9L, !logi_w, ),
             bor3(nis <= 9L, !logi_w, ))
expect_equal(or3s(nit <= 0L, !logi_x, w %between% c(-1L, 1L)),
             bor3(nit <= 0L, !logi_x, w %between% c(-1L, 1L)))
expect_equal(or3s(niu <= 1L, !logi_k, b %between% c(-1L, 1L)),
             bor3(niu <= 1L, !logi_k, b %between% c(-1L, 1L)))
expect_equal(or3s(niv <= 9L, !logi_k, w %between% c(-1L, 1L)),
             bor3(niv <= 9L, !logi_k, w %between% c(-1L, 1L)))
expect_equal(or3s(niw <= 0L, , ),
             bor3(niw <= 0L, , ))
expect_equal(or3s(nix <= 1L, , ),
             bor3(nix <= 1L, , ))
expect_equal(or3s(niy <= 9L, , ),
             bor3(niy <= 9L, , ))
expect_equal(or3s(niz <= 0L, , h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(niz <= 0L, , h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nja <= 1L, , k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nja <= 1L, , k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(njb <= 9L, , w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(njb <= 9L, , w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(njc <= 0L, !logi_t, ),
             bor3(njc <= 0L, !logi_t, ))
expect_equal(or3s(njd <= 1L, !logi_j, ),
             bor3(njd <= 1L, !logi_j, ))
expect_equal(or3s(nje <= 9L, !logi_z, ),
             bor3(nje <= 9L, !logi_z, ))
expect_equal(or3s(njf <= 0L, !logi_h, w %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(njf <= 0L, !logi_h, w %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(njg <= 1L, !logi_o, h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(njg <= 1L, !logi_o, h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(njh <= 9L, !logi_w, e %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(njh <= 9L, !logi_w, e %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nji <= 0L, , ),
             bor3(nji <= 0L, , ))
expect_equal(or3s(njj <= 1L, , ),
             bor3(njj <= 1L, , ))
expect_equal(or3s(njk <= 9L, , ),
             bor3(njk <= 9L, , ))
expect_equal(or3s(njl <= 0L, , o %in% 1:4),
             bor3(njl <= 0L, , o %in% 1:4))
expect_equal(or3s(njm <= 1L, , j %in% 1:4),
             bor3(njm <= 1L, , j %in% 1:4))
expect_equal(or3s(njn <= 9L, , i %in% 1:4),
             bor3(njn <= 9L, , i %in% 1:4))
expect_equal(or3s(njo <= 0L, !logi_b, ),
             bor3(njo <= 0L, !logi_b, ))
expect_equal(or3s(njp <= 1L, !logi_u, ),
             bor3(njp <= 1L, !logi_u, ))
expect_equal(or3s(njq <= 9L, !logi_k, ),
             bor3(njq <= 9L, !logi_k, ))
expect_equal(or3s(njr <= 0L, !logi_d, y %in% 1:4),
             bor3(njr <= 0L, !logi_d, y %in% 1:4))
expect_equal(or3s(njs <= 1L, !logi_z, z %in% 1:4),
             bor3(njs <= 1L, !logi_z, z %in% 1:4))
expect_equal(or3s(njt <= 9L, !logi_c, a %in% 1:4),
             bor3(njt <= 9L, !logi_c, a %in% 1:4))
expect_equal(or3s(nju <= 0L, , ),
             bor3(nju <= 0L, , ))
expect_equal(or3s(njv <= 1L, , ),
             bor3(njv <= 1L, , ))
expect_equal(or3s(njw <= 9L, , ),
             bor3(njw <= 9L, , ))
expect_equal(or3s(njx <= 0L, , k < 0L),
             bor3(njx <= 0L, , k < 0L))
expect_equal(or3s(njy <= 1L, , b < 1L),
             bor3(njy <= 1L, , b < 1L))
expect_equal(or3s(njz <= 9L, , a < 9L),
             bor3(njz <= 9L, , a < 9L))
expect_equal(or3s(nka <= 0L, !logi_i, ),
             bor3(nka <= 0L, !logi_i, ))
expect_equal(or3s(nkb <= 1L, !logi_j, ),
             bor3(nkb <= 1L, !logi_j, ))
expect_equal(or3s(nkc <= 9L, !logi_j, ),
             bor3(nkc <= 9L, !logi_j, ))
expect_equal(or3s(nkd <= 0L, !logi_d, t < 0L),
             bor3(nkd <= 0L, !logi_d, t < 0L))
expect_equal(or3s(nke <= 1L, !logi_j, l < 1L),
             bor3(nke <= 1L, !logi_j, l < 1L))
expect_equal(or3s(nkf <= 9L, !logi_n, b < 9L),
             bor3(nkf <= 9L, !logi_n, b < 9L))
expect_equal(or3s(nkg <= 0L, , ),
             bor3(nkg <= 0L, , ))
expect_equal(or3s(nkh <= 1L, , ),
             bor3(nkh <= 1L, , ))
expect_equal(or3s(nki <= 9L, , ),
             bor3(nki <= 9L, , ))
expect_equal(or3s(nkj <= 0L, , y <= 0L),
             bor3(nkj <= 0L, , y <= 0L))
expect_equal(or3s(nkk <= 1L, , p <= 1L),
             bor3(nkk <= 1L, , p <= 1L))
expect_equal(or3s(nkl <= 9L, , r <= 9L),
             bor3(nkl <= 9L, , r <= 9L))
expect_equal(or3s(nkm <= 0L, !logi_c, ),
             bor3(nkm <= 0L, !logi_c, ))
expect_equal(or3s(nkn <= 1L, !logi_c, ),
             bor3(nkn <= 1L, !logi_c, ))
expect_equal(or3s(nko <= 9L, !logi_w, ),
             bor3(nko <= 9L, !logi_w, ))
expect_equal(or3s(nkp <= 0L, !logi_c, n <= 0L),
             bor3(nkp <= 0L, !logi_c, n <= 0L))
expect_equal(or3s(nkq <= 1L, !logi_a, m <= 1L),
             bor3(nkq <= 1L, !logi_a, m <= 1L))
expect_equal(or3s(nkr <= 9L, !logi_e, o <= 9L),
             bor3(nkr <= 9L, !logi_e, o <= 9L))
expect_equal(or3s(nks <= 0L, , ),
             bor3(nks <= 0L, , ))
expect_equal(or3s(nkt <= 1L, , ),
             bor3(nkt <= 1L, , ))
expect_equal(or3s(nku <= 9L, , ),
             bor3(nku <= 9L, , ))
expect_equal(or3s(nkv <= 0L, , h == 0L),
             bor3(nkv <= 0L, , h == 0L))
expect_equal(or3s(nkw <= 1L, , u == 1L),
             bor3(nkw <= 1L, , u == 1L))
expect_equal(or3s(nkx <= 9L, , k == 9L),
             bor3(nkx <= 9L, , k == 9L))
expect_equal(or3s(nky <= 0L, !logi_e, ),
             bor3(nky <= 0L, !logi_e, ))
expect_equal(or3s(nkz <= 1L, !logi_z, ),
             bor3(nkz <= 1L, !logi_z, ))
expect_equal(or3s(nla <= 9L, !logi_p, ),
             bor3(nla <= 9L, !logi_p, ))
expect_equal(or3s(nlb <= 0L, !logi_h, n == 0L),
             bor3(nlb <= 0L, !logi_h, n == 0L))
expect_equal(or3s(nlc <= 1L, !logi_b, j == 1L),
             bor3(nlc <= 1L, !logi_b, j == 1L))
expect_equal(or3s(nld <= 9L, !logi_t, f == 9L),
             bor3(nld <= 9L, !logi_t, f == 9L))
expect_equal(or3s(nle <= 0L, , ),
             bor3(nle <= 0L, , ))
expect_equal(or3s(nlf <= 1L, , ),
             bor3(nlf <= 1L, , ))
expect_equal(or3s(nlg <= 9L, , ),
             bor3(nlg <= 9L, , ))
expect_equal(or3s(nlh <= 0L, , y > 0L),
             bor3(nlh <= 0L, , y > 0L))
expect_equal(or3s(nli <= 1L, , v > 1L),
             bor3(nli <= 1L, , v > 1L))
expect_equal(or3s(nlj <= 9L, , d > 9L),
             bor3(nlj <= 9L, , d > 9L))
expect_equal(or3s(nlk <= 0L, !logi_q, ),
             bor3(nlk <= 0L, !logi_q, ))
expect_equal(or3s(nll <= 1L, !logi_e, ),
             bor3(nll <= 1L, !logi_e, ))
expect_equal(or3s(nlm <= 9L, !logi_z, ),
             bor3(nlm <= 9L, !logi_z, ))
expect_equal(or3s(nln <= 0L, !logi_t, i > 0L),
             bor3(nln <= 0L, !logi_t, i > 0L))
expect_equal(or3s(nlo <= 1L, !logi_w, h > 1L),
             bor3(nlo <= 1L, !logi_w, h > 1L))
expect_equal(or3s(nlp <= 9L, !logi_p, o > 9L),
             bor3(nlp <= 9L, !logi_p, o > 9L))
expect_equal(or3s(nlq <= 0L, , ),
             bor3(nlq <= 0L, , ))
expect_equal(or3s(nlr <= 1L, , ),
             bor3(nlr <= 1L, , ))
expect_equal(or3s(nls <= 9L, , ),
             bor3(nls <= 9L, , ))
expect_equal(or3s(nlt <= 0L, , h >= 0L),
             bor3(nlt <= 0L, , h >= 0L))
expect_equal(or3s(nlu <= 1L, , z >= 1L),
             bor3(nlu <= 1L, , z >= 1L))
expect_equal(or3s(nlv <= 9L, , g >= 9L),
             bor3(nlv <= 9L, , g >= 9L))
expect_equal(or3s(nlw <= 0L, !logi_a, ),
             bor3(nlw <= 0L, !logi_a, ))
expect_equal(or3s(nlx <= 1L, !logi_h, ),
             bor3(nlx <= 1L, !logi_h, ))
expect_equal(or3s(nly <= 9L, !logi_n, ),
             bor3(nly <= 9L, !logi_n, ))
expect_equal(or3s(nlz <= 0L, !logi_u, f >= 0L),
             bor3(nlz <= 0L, !logi_u, f >= 0L))
expect_equal(or3s(nma <= 1L, !logi_r, g >= 1L),
             bor3(nma <= 1L, !logi_r, g >= 1L))
expect_equal(or3s(nmb <= 9L, !logi_c, b >= 9L),
             bor3(nmb <= 9L, !logi_c, b >= 9L))
expect_equal(or3s(nmc <= 0L, , ),
             bor3(nmc <= 0L, , ))
expect_equal(or3s(nmd <= 1L, , ),
             bor3(nmd <= 1L, , ))
expect_equal(or3s(nme <= 9L, , ),
             bor3(nme <= 9L, , ))
expect_equal(or3s(nmf <= 0L, , logi_n),
             bor3(nmf <= 0L, , logi_n))
expect_equal(or3s(nmg <= 1L, , logi_m),
             bor3(nmg <= 1L, , logi_m))
expect_equal(or3s(nmh <= 9L, , logi_z),
             bor3(nmh <= 9L, , logi_z))
expect_equal(or3s(nmi <= 0L, z != 0L, ),
             bor3(nmi <= 0L, z != 0L, ))
expect_equal(or3s(nmj <= 1L, d != 1L, ),
             bor3(nmj <= 1L, d != 1L, ))
expect_equal(or3s(nmk <= 9L, r != 9L, ),
             bor3(nmk <= 9L, r != 9L, ))
expect_equal(or3s(nml <= 0L, a != 0L, logi_m),
             bor3(nml <= 0L, a != 0L, logi_m))
expect_equal(or3s(nmm <= 1L, u != 1L, logi_c),
             bor3(nmm <= 1L, u != 1L, logi_c))
expect_equal(or3s(nmn <= 9L, s != 9L, logi_s),
             bor3(nmn <= 9L, s != 9L, logi_s))
expect_equal(or3s(nmo <= 0L, , ),
             bor3(nmo <= 0L, , ))
expect_equal(or3s(nmp <= 1L, , ),
             bor3(nmp <= 1L, , ))
expect_equal(or3s(nmq <= 9L, , ),
             bor3(nmq <= 9L, , ))
expect_equal(or3s(nmr <= 0L, , !logi_d),
             bor3(nmr <= 0L, , !logi_d))
expect_equal(or3s(nms <= 1L, , !logi_t),
             bor3(nms <= 1L, , !logi_t))
expect_equal(or3s(nmt <= 9L, , !logi_e),
             bor3(nmt <= 9L, , !logi_e))
expect_equal(or3s(nmu <= 0L, l != 0L, ),
             bor3(nmu <= 0L, l != 0L, ))
expect_equal(or3s(nmv <= 1L, n != 1L, ),
             bor3(nmv <= 1L, n != 1L, ))
expect_equal(or3s(nmw <= 9L, f != 9L, ),
             bor3(nmw <= 9L, f != 9L, ))
expect_equal(or3s(nmx <= 0L, s != 0L, !logi_o),
             bor3(nmx <= 0L, s != 0L, !logi_o))
expect_equal(or3s(nmy <= 1L, e != 1L, !logi_l),
             bor3(nmy <= 1L, e != 1L, !logi_l))
expect_equal(or3s(nmz <= 9L, d != 9L, !logi_r),
             bor3(nmz <= 9L, d != 9L, !logi_r))
expect_equal(or3s(nna <= 0L, , ),
             bor3(nna <= 0L, , ))
expect_equal(or3s(nnb <= 1L, , ),
             bor3(nnb <= 1L, , ))
expect_equal(or3s(nnc <= 9L, , ),
             bor3(nnc <= 9L, , ))
expect_equal(or3s(nnd <= 0L, , r != 0L),
             bor3(nnd <= 0L, , r != 0L))
expect_equal(or3s(nne <= 1L, , l != 1L),
             bor3(nne <= 1L, , l != 1L))
expect_equal(or3s(nnf <= 9L, , u != 9L),
             bor3(nnf <= 9L, , u != 9L))
expect_equal(or3s(nng <= 0L, h != 0L, ),
             bor3(nng <= 0L, h != 0L, ))
expect_equal(or3s(nnh <= 1L, r != 1L, ),
             bor3(nnh <= 1L, r != 1L, ))
expect_equal(or3s(nni <= 9L, j != 9L, ),
             bor3(nni <= 9L, j != 9L, ))
expect_equal(or3s(nnj <= 0L, r != 0L, s != 0L),
             bor3(nnj <= 0L, r != 0L, s != 0L))
expect_equal(or3s(nnk <= 1L, r != 1L, i != 1L),
             bor3(nnk <= 1L, r != 1L, i != 1L))
expect_equal(or3s(nnl <= 9L, m != 9L, p != 9L),
             bor3(nnl <= 9L, m != 9L, p != 9L))
expect_equal(or3s(nnm <= 0L, , ),
             bor3(nnm <= 0L, , ))
expect_equal(or3s(nnn <= 1L, , ),
             bor3(nnn <= 1L, , ))
expect_equal(or3s(nno <= 9L, , ),
             bor3(nno <= 9L, , ))
expect_equal(or3s(nnp <= 0L, , z %between% c(-1L, 1L)),
             bor3(nnp <= 0L, , z %between% c(-1L, 1L)))
expect_equal(or3s(nnq <= 1L, , m %between% c(-1L, 1L)),
             bor3(nnq <= 1L, , m %between% c(-1L, 1L)))
expect_equal(or3s(nnr <= 9L, , k %between% c(-1L, 1L)),
             bor3(nnr <= 9L, , k %between% c(-1L, 1L)))
expect_equal(or3s(nns <= 0L, f != 0L, ),
             bor3(nns <= 0L, f != 0L, ))
expect_equal(or3s(nnt <= 1L, f != 1L, ),
             bor3(nnt <= 1L, f != 1L, ))
expect_equal(or3s(nnu <= 9L, e != 9L, ),
             bor3(nnu <= 9L, e != 9L, ))
expect_equal(or3s(nnv <= 0L, b != 0L, c %between% c(-1L, 1L)),
             bor3(nnv <= 0L, b != 0L, c %between% c(-1L, 1L)))
expect_equal(or3s(nnw <= 1L, q != 1L, g %between% c(-1L, 1L)),
             bor3(nnw <= 1L, q != 1L, g %between% c(-1L, 1L)))
expect_equal(or3s(nnx <= 9L, p != 9L, s %between% c(-1L, 1L)),
             bor3(nnx <= 9L, p != 9L, s %between% c(-1L, 1L)))
expect_equal(or3s(nny <= 0L, , ),
             bor3(nny <= 0L, , ))
expect_equal(or3s(nnz <= 1L, , ),
             bor3(nnz <= 1L, , ))
expect_equal(or3s(noa <= 9L, , ),
             bor3(noa <= 9L, , ))
expect_equal(or3s(nob <= 0L, , p %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nob <= 0L, , p %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(noc <= 1L, , c %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(noc <= 1L, , c %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nod <= 9L, , v %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nod <= 9L, , v %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(noe <= 0L, k != 0L, ),
             bor3(noe <= 0L, k != 0L, ))
expect_equal(or3s(nof <= 1L, l != 1L, ),
             bor3(nof <= 1L, l != 1L, ))
expect_equal(or3s(nog <= 9L, c != 9L, ),
             bor3(nog <= 9L, c != 9L, ))
expect_equal(or3s(noh <= 0L, z != 0L, x %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(noh <= 0L, z != 0L, x %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(noi <= 1L, t != 1L, g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(noi <= 1L, t != 1L, g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(noj <= 9L, e != 9L, h %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(noj <= 9L, e != 9L, h %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nok <= 0L, , ),
             bor3(nok <= 0L, , ))
expect_equal(or3s(nol <= 1L, , ),
             bor3(nol <= 1L, , ))
expect_equal(or3s(nom <= 9L, , ),
             bor3(nom <= 9L, , ))
expect_equal(or3s(non <= 0L, , s %in% 1:4),
             bor3(non <= 0L, , s %in% 1:4))
expect_equal(or3s(noo <= 1L, , j %in% 1:4),
             bor3(noo <= 1L, , j %in% 1:4))
expect_equal(or3s(nop <= 9L, , c %in% 1:4),
             bor3(nop <= 9L, , c %in% 1:4))
expect_equal(or3s(noq <= 0L, i != 0L, ),
             bor3(noq <= 0L, i != 0L, ))
expect_equal(or3s(nor <= 1L, h != 1L, ),
             bor3(nor <= 1L, h != 1L, ))
expect_equal(or3s(nos <= 9L, s != 9L, ),
             bor3(nos <= 9L, s != 9L, ))
expect_equal(or3s(not <= 0L, j != 0L, g %in% 1:4),
             bor3(not <= 0L, j != 0L, g %in% 1:4))
expect_equal(or3s(nou <= 1L, c != 1L, d %in% 1:4),
             bor3(nou <= 1L, c != 1L, d %in% 1:4))
expect_equal(or3s(nov <= 9L, h != 9L, z %in% 1:4),
             bor3(nov <= 9L, h != 9L, z %in% 1:4))
expect_equal(or3s(now <= 0L, , ),
             bor3(now <= 0L, , ))
expect_equal(or3s(nox <= 1L, , ),
             bor3(nox <= 1L, , ))
expect_equal(or3s(noy <= 9L, , ),
             bor3(noy <= 9L, , ))
expect_equal(or3s(noz <= 0L, , g < 0L),
             bor3(noz <= 0L, , g < 0L))
expect_equal(or3s(npa <= 1L, , h < 1L),
             bor3(npa <= 1L, , h < 1L))
expect_equal(or3s(npb <= 9L, , q < 9L),
             bor3(npb <= 9L, , q < 9L))
expect_equal(or3s(npc <= 0L, h != 0L, ),
             bor3(npc <= 0L, h != 0L, ))
expect_equal(or3s(npd <= 1L, m != 1L, ),
             bor3(npd <= 1L, m != 1L, ))
expect_equal(or3s(npe <= 9L, w != 9L, ),
             bor3(npe <= 9L, w != 9L, ))
expect_equal(or3s(npf <= 0L, q != 0L, v < 0L),
             bor3(npf <= 0L, q != 0L, v < 0L))
expect_equal(or3s(npg <= 1L, e != 1L, n < 1L),
             bor3(npg <= 1L, e != 1L, n < 1L))
expect_equal(or3s(nph <= 9L, u != 9L, t < 9L),
             bor3(nph <= 9L, u != 9L, t < 9L))
expect_equal(or3s(npi <= 0L, , ),
             bor3(npi <= 0L, , ))
expect_equal(or3s(npj <= 1L, , ),
             bor3(npj <= 1L, , ))
expect_equal(or3s(npk <= 9L, , ),
             bor3(npk <= 9L, , ))
expect_equal(or3s(npl <= 0L, , x <= 0L),
             bor3(npl <= 0L, , x <= 0L))
expect_equal(or3s(npm <= 1L, , l <= 1L),
             bor3(npm <= 1L, , l <= 1L))
expect_equal(or3s(npn <= 9L, , e <= 9L),
             bor3(npn <= 9L, , e <= 9L))
expect_equal(or3s(npo <= 0L, r != 0L, ),
             bor3(npo <= 0L, r != 0L, ))
expect_equal(or3s(npp <= 1L, d != 1L, ),
             bor3(npp <= 1L, d != 1L, ))
expect_equal(or3s(npq <= 9L, n != 9L, ),
             bor3(npq <= 9L, n != 9L, ))
expect_equal(or3s(npr <= 0L, s != 0L, z <= 0L),
             bor3(npr <= 0L, s != 0L, z <= 0L))
expect_equal(or3s(nps <= 1L, h != 1L, k <= 1L),
             bor3(nps <= 1L, h != 1L, k <= 1L))
expect_equal(or3s(npt <= 9L, l != 9L, b <= 9L),
             bor3(npt <= 9L, l != 9L, b <= 9L))
expect_equal(or3s(npu <= 0L, , ),
             bor3(npu <= 0L, , ))
expect_equal(or3s(npv <= 1L, , ),
             bor3(npv <= 1L, , ))
expect_equal(or3s(npw <= 9L, , ),
             bor3(npw <= 9L, , ))
expect_equal(or3s(npx <= 0L, , x == 0L),
             bor3(npx <= 0L, , x == 0L))
expect_equal(or3s(npy <= 1L, , s == 1L),
             bor3(npy <= 1L, , s == 1L))
expect_equal(or3s(npz <= 9L, , t == 9L),
             bor3(npz <= 9L, , t == 9L))
expect_equal(or3s(nqa <= 0L, s != 0L, ),
             bor3(nqa <= 0L, s != 0L, ))
expect_equal(or3s(nqb <= 1L, o != 1L, ),
             bor3(nqb <= 1L, o != 1L, ))
expect_equal(or3s(nqc <= 9L, z != 9L, ),
             bor3(nqc <= 9L, z != 9L, ))
expect_equal(or3s(nqd <= 0L, i != 0L, t == 0L),
             bor3(nqd <= 0L, i != 0L, t == 0L))
expect_equal(or3s(nqe <= 1L, a != 1L, l == 1L),
             bor3(nqe <= 1L, a != 1L, l == 1L))
expect_equal(or3s(nqf <= 9L, j != 9L, i == 9L),
             bor3(nqf <= 9L, j != 9L, i == 9L))
expect_equal(or3s(nqg <= 0L, , ),
             bor3(nqg <= 0L, , ))
expect_equal(or3s(nqh <= 1L, , ),
             bor3(nqh <= 1L, , ))
expect_equal(or3s(nqi <= 9L, , ),
             bor3(nqi <= 9L, , ))
expect_equal(or3s(nqj <= 0L, , n > 0L),
             bor3(nqj <= 0L, , n > 0L))
expect_equal(or3s(nqk <= 1L, , l > 1L),
             bor3(nqk <= 1L, , l > 1L))
expect_equal(or3s(nql <= 9L, , y > 9L),
             bor3(nql <= 9L, , y > 9L))
expect_equal(or3s(nqm <= 0L, n != 0L, ),
             bor3(nqm <= 0L, n != 0L, ))
expect_equal(or3s(nqn <= 1L, t != 1L, ),
             bor3(nqn <= 1L, t != 1L, ))
expect_equal(or3s(nqo <= 9L, o != 9L, ),
             bor3(nqo <= 9L, o != 9L, ))
expect_equal(or3s(nqp <= 0L, y != 0L, v > 0L),
             bor3(nqp <= 0L, y != 0L, v > 0L))
expect_equal(or3s(nqq <= 1L, z != 1L, n > 1L),
             bor3(nqq <= 1L, z != 1L, n > 1L))
expect_equal(or3s(nqr <= 9L, n != 9L, g > 9L),
             bor3(nqr <= 9L, n != 9L, g > 9L))
expect_equal(or3s(nqs <= 0L, , ),
             bor3(nqs <= 0L, , ))
expect_equal(or3s(nqt <= 1L, , ),
             bor3(nqt <= 1L, , ))
expect_equal(or3s(nqu <= 9L, , ),
             bor3(nqu <= 9L, , ))
expect_equal(or3s(nqv <= 0L, , f >= 0L),
             bor3(nqv <= 0L, , f >= 0L))
expect_equal(or3s(nqw <= 1L, , s >= 1L),
             bor3(nqw <= 1L, , s >= 1L))
expect_equal(or3s(nqx <= 9L, , i >= 9L),
             bor3(nqx <= 9L, , i >= 9L))
expect_equal(or3s(nqy <= 0L, p != 0L, ),
             bor3(nqy <= 0L, p != 0L, ))
expect_equal(or3s(nqz <= 1L, v != 1L, ),
             bor3(nqz <= 1L, v != 1L, ))
expect_equal(or3s(nra <= 9L, f != 9L, ),
             bor3(nra <= 9L, f != 9L, ))
expect_equal(or3s(nrb <= 0L, u != 0L, s >= 0L),
             bor3(nrb <= 0L, u != 0L, s >= 0L))
expect_equal(or3s(nrc <= 1L, y != 1L, n >= 1L),
             bor3(nrc <= 1L, y != 1L, n >= 1L))
expect_equal(or3s(nrd <= 9L, a != 9L, h >= 9L),
             bor3(nrd <= 9L, a != 9L, h >= 9L))
expect_equal(or3s(nre <= 0L, , ),
             bor3(nre <= 0L, , ))
expect_equal(or3s(nrf <= 1L, , ),
             bor3(nrf <= 1L, , ))
expect_equal(or3s(nrg <= 9L, , ),
             bor3(nrg <= 9L, , ))
expect_equal(or3s(nrh <= 0L, , logi_w),
             bor3(nrh <= 0L, , logi_w))
expect_equal(or3s(nri <= 1L, , logi_u),
             bor3(nri <= 1L, , logi_u))
expect_equal(or3s(nrj <= 9L, , logi_a),
             bor3(nrj <= 9L, , logi_a))
expect_equal(or3s(nrk <= 0L, m %between% c(-1L, 1L), ),
             bor3(nrk <= 0L, m %between% c(-1L, 1L), ))
expect_equal(or3s(nrl <= 1L, f %between% c(-1L, 1L), ),
             bor3(nrl <= 1L, f %between% c(-1L, 1L), ))
expect_equal(or3s(nrm <= 9L, i %between% c(-1L, 1L), ),
             bor3(nrm <= 9L, i %between% c(-1L, 1L), ))
expect_equal(or3s(nrn <= 0L, q %between% c(-1L, 1L), logi_v),
             bor3(nrn <= 0L, q %between% c(-1L, 1L), logi_v))
expect_equal(or3s(nro <= 1L, j %between% c(-1L, 1L), logi_w),
             bor3(nro <= 1L, j %between% c(-1L, 1L), logi_w))
expect_equal(or3s(nrp <= 9L, d %between% c(-1L, 1L), logi_b),
             bor3(nrp <= 9L, d %between% c(-1L, 1L), logi_b))
expect_equal(or3s(nrq <= 0L, , ),
             bor3(nrq <= 0L, , ))
expect_equal(or3s(nrr <= 1L, , ),
             bor3(nrr <= 1L, , ))
expect_equal(or3s(nrs <= 9L, , ),
             bor3(nrs <= 9L, , ))
expect_equal(or3s(nrt <= 0L, , !logi_d),
             bor3(nrt <= 0L, , !logi_d))
expect_equal(or3s(nru <= 1L, , !logi_g),
             bor3(nru <= 1L, , !logi_g))
expect_equal(or3s(nrv <= 9L, , !logi_a),
             bor3(nrv <= 9L, , !logi_a))
expect_equal(or3s(nrw <= 0L, o %between% c(-1L, 1L), ),
             bor3(nrw <= 0L, o %between% c(-1L, 1L), ))
expect_equal(or3s(nrx <= 1L, k %between% c(-1L, 1L), ),
             bor3(nrx <= 1L, k %between% c(-1L, 1L), ))
expect_equal(or3s(nry <= 9L, g %between% c(-1L, 1L), ),
             bor3(nry <= 9L, g %between% c(-1L, 1L), ))
expect_equal(or3s(nrz <= 0L, l %between% c(-1L, 1L), !logi_u),
             bor3(nrz <= 0L, l %between% c(-1L, 1L), !logi_u))
expect_equal(or3s(nsa <= 1L, v %between% c(-1L, 1L), !logi_i),
             bor3(nsa <= 1L, v %between% c(-1L, 1L), !logi_i))
expect_equal(or3s(nsb <= 9L, g %between% c(-1L, 1L), !logi_w),
             bor3(nsb <= 9L, g %between% c(-1L, 1L), !logi_w))
expect_equal(or3s(nsc <= 0L, , ),
             bor3(nsc <= 0L, , ))
expect_equal(or3s(nsd <= 1L, , ),
             bor3(nsd <= 1L, , ))
expect_equal(or3s(nse <= 9L, , ),
             bor3(nse <= 9L, , ))
expect_equal(or3s(nsf <= 0L, , n != 0L),
             bor3(nsf <= 0L, , n != 0L))
expect_equal(or3s(nsg <= 1L, , a != 1L),
             bor3(nsg <= 1L, , a != 1L))
expect_equal(or3s(nsh <= 9L, , v != 9L),
             bor3(nsh <= 9L, , v != 9L))
expect_equal(or3s(nsi <= 0L, i %between% c(-1L, 1L), ),
             bor3(nsi <= 0L, i %between% c(-1L, 1L), ))
expect_equal(or3s(nsj <= 1L, p %between% c(-1L, 1L), ),
             bor3(nsj <= 1L, p %between% c(-1L, 1L), ))
expect_equal(or3s(nsk <= 9L, l %between% c(-1L, 1L), ),
             bor3(nsk <= 9L, l %between% c(-1L, 1L), ))
expect_equal(or3s(nsl <= 0L, f %between% c(-1L, 1L), d != 0L),
             bor3(nsl <= 0L, f %between% c(-1L, 1L), d != 0L))
expect_equal(or3s(nsm <= 1L, f %between% c(-1L, 1L), y != 1L),
             bor3(nsm <= 1L, f %between% c(-1L, 1L), y != 1L))
expect_equal(or3s(nsn <= 9L, b %between% c(-1L, 1L), w != 9L),
             bor3(nsn <= 9L, b %between% c(-1L, 1L), w != 9L))
expect_equal(or3s(nso <= 0L, , ),
             bor3(nso <= 0L, , ))
expect_equal(or3s(nsp <= 1L, , ),
             bor3(nsp <= 1L, , ))
expect_equal(or3s(nsq <= 9L, , ),
             bor3(nsq <= 9L, , ))
expect_equal(or3s(nsr <= 0L, , g %between% c(-1L, 1L)),
             bor3(nsr <= 0L, , g %between% c(-1L, 1L)))
expect_equal(or3s(nss <= 1L, , e %between% c(-1L, 1L)),
             bor3(nss <= 1L, , e %between% c(-1L, 1L)))
expect_equal(or3s(nst <= 9L, , a %between% c(-1L, 1L)),
             bor3(nst <= 9L, , a %between% c(-1L, 1L)))
expect_equal(or3s(nsu <= 0L, b %between% c(-1L, 1L), ),
             bor3(nsu <= 0L, b %between% c(-1L, 1L), ))
expect_equal(or3s(nsv <= 1L, l %between% c(-1L, 1L), ),
             bor3(nsv <= 1L, l %between% c(-1L, 1L), ))
expect_equal(or3s(nsw <= 9L, c %between% c(-1L, 1L), ),
             bor3(nsw <= 9L, c %between% c(-1L, 1L), ))
expect_equal(or3s(nsx <= 0L, c %between% c(-1L, 1L), f %between% c(-1L, 1L)),
             bor3(nsx <= 0L, c %between% c(-1L, 1L), f %between% c(-1L, 1L)))
expect_equal(or3s(nsy <= 1L, o %between% c(-1L, 1L), y %between% c(-1L, 1L)),
             bor3(nsy <= 1L, o %between% c(-1L, 1L), y %between% c(-1L, 1L)))
expect_equal(or3s(nsz <= 9L, k %between% c(-1L, 1L), g %between% c(-1L, 1L)),
             bor3(nsz <= 9L, k %between% c(-1L, 1L), g %between% c(-1L, 1L)))
expect_equal(or3s(nta <= 0L, , ),
             bor3(nta <= 0L, , ))
expect_equal(or3s(ntb <= 1L, , ),
             bor3(ntb <= 1L, , ))
expect_equal(or3s(ntc <= 9L, , ),
             bor3(ntc <= 9L, , ))
expect_equal(or3s(ntd <= 0L, , k %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ntd <= 0L, , k %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(nte <= 1L, , o %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(nte <= 1L, , o %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ntf <= 9L, , t %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ntf <= 9L, , t %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ntg <= 0L, w %between% c(-1L, 1L), ),
             bor3(ntg <= 0L, w %between% c(-1L, 1L), ))
expect_equal(or3s(nth <= 1L, z %between% c(-1L, 1L), ),
             bor3(nth <= 1L, z %between% c(-1L, 1L), ))
expect_equal(or3s(nti <= 9L, v %between% c(-1L, 1L), ),
             bor3(nti <= 9L, v %between% c(-1L, 1L), ))
expect_equal(or3s(ntj <= 0L, i %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ntj <= 0L, i %between% c(-1L, 1L), n %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ntk <= 1L, t %between% c(-1L, 1L), g %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ntk <= 1L, t %between% c(-1L, 1L), g %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ntl <= 9L, t %between% c(-1L, 1L), d %in% c(4L, -3L, 2L, 7L, 8L)),
             bor3(ntl <= 9L, t %between% c(-1L, 1L), d %in% c(4L, -3L, 2L, 7L, 8L)))
expect_equal(or3s(ntm <= 0L, , ),
             bor3(ntm <= 0L, , ))
expect_equal(or3s(ntn <= 1L, , ),
             bor3(ntn <= 1L, , ))
expect_equal(or3s(nto <= 9L, , ),
             bor3(nto <= 9L, , ))
expect_equal(or3s(ntp <= 0L, , q %in% 1:4),
             bor3(ntp <= 0L, , q %in% 1:4))
expect_equal(or3s(ntq <= 1L, , f %in% 1:4),
             bor3(ntq <= 1L, , f %in% 1:4))
expect_equal(or3s(ntr <= 9L, , m %in% 1:4),
             bor3(ntr <= 9L, , m %in% 1:4))
expect_equal(or3s(nts <= 0L, p %between% c(-1L, 1L), ),
             bor3(nts <= 0L, p %between% c(-1L, 1L), ))
expect_equal(or3s(ntt <= 1L, p %between% c(-1L, 1L), ),
             bor3(ntt <= 1L, p %between% c(-1L, 1L), ))
expect_equal(or3s(ntu <= 9L, c %between% c(-1L, 1L), ),
             bor3(ntu <= 9L, c %between% c(-1L, 1L), ))
expect_equal(or3s(ntv <= 0L, y %between% c(-1L, 1L), z %in% 1:4),
             bor3(ntv <= 0L, y %between% c(-1L, 1L), z %in% 1:4))
expect_equal(or3s(ntw <= 1L, s %between% c(-1L, 1L), e %in% 1:4),
             bor3(ntw <= 1L, s %between% c(-1L, 1L), e %in% 1:4))
expect_equal(or3s(ntx <= 9L, w %between% c(-1L, 1L), g %in% 1:4),
             bor3(ntx <= 9L, w %between% c(-1L, 1L), g %in% 1:4))
expect_equal(or3s(nty <= 0L, , ),
             bor3(nty <= 0L, , ))
expect_equal(or3s(ntz <= 1L, , ),
             bor3(ntz <= 1L, , ))
expect_equal(or3s(nua <= 9L, , ),
             bor3(nua <= 9L, , ))
expect_equal(or3s(nub <= 0L, , w < 0L),
             bor3(nub <= 0L, , w < 0L))
expect_equal(or3s(nuc <= 1L, , s < 1L),
             bor3(nuc <= 1L, , s < 1L))
expect_equal(or3s(nud <= 9L, , i < 9L),
             bor3(nud <= 9L, , i < 9L))
expect_equal(or3s(nue <= 0L, p %between% c(-1L, 1L), ),
             bor3(nue <= 0L, p %between% c(-1L, 1L), ))
expect_equal(or3s(nuf <= 1L, w %between% c(-1L, 1L), ),
             bor3(nuf <= 1L, w %between% c(-1L, 1L), ))
expect_equal(or3s(nug <= 9L, l %between% c(-1L, 1L), ),
             bor3(nug <= 9L, l %between% c(-1L, 1L), ))
expect_equal(or3s(nuh <= 0L, r %between% c(-1L, 1L), l < 0L),
             bor3(nuh <= 0L, r %between% c(-1L, 1L), l < 0L))
expect_equal(or3s(nui <= 1L, b %between% c(-1L, 1L), o < 1L),
             bor3(nui <= 1L, b %between% c(-1L, 1L), o < 1L))
expect_equal(or3s(nuj <= 9L, c %between% c(-1L, 1L), n < 9L),
             bor3(nuj <= 9L, c %between% c(-1L, 1L), n < 9L))
expect_equal(or3s(nuk <= 0L, , ),
             bor3(nuk <= 0L, , ))
expect_equal(or3s(nul <= 1L, , ),
             bor3(nul <= 1L, , ))
expect_equal(or3s(num <= 9L, , ),
             bor3(num <= 9L, , ))
expect_equal(or3s(nun <= 0L, , m <= 0L),
             bor3(nun <= 0L, , m <= 0L))
expect_equal(or3s(nuo <= 1L, , n <= 1L),
             bor3(nuo <= 1L, , n <= 1L))
expect_equal(or3s(nup <= 9L, , a <= 9L),
             bor3(nup <= 9L, , a <= 9L))
expect_equal(or3s(nuq <= 0L, i %between% c(-1L, 1L), ),
             bor3(nuq <= 0L, i %between% c(-1L, 1L), ))
expect_equal(or3s(nur <= 1L, t %between% c(-1L, 1L), ),
             bor3(nur <= 1L, t %between% c(-1L, 1L), ))
expect_equal(or3s(nus <= 9L, j %between% c(-1L, 1L), ),
             bor3(nus <= 9L, j %between% c(-1L, 1L), ))
expect_equal(or3s(nut <= 0L, o %between% c(-1L, 1L), v <= 0L),
             bor3(nut <= 0L, o %between% c(-1L, 1L), v <= 0L))
expect_equal(or3s(nuu <= 1L, a %between% c(-1L, 1L), m <= 1L),
             bor3(nuu <= 1L, a %between% c(-1L, 1L), m <= 1L))
expect_equal(or3s(nuv <= 9L, e %between% c(-1L, 1L), f <= 9L),
             bor3(nuv <= 9L, e %between% c(-1L, 1L), f <= 9L))
expect_equal(or3s(nuw <= 0L, , ),
             bor3(nuw <= 0L, , ))
expect_equal(or3s(nux <= 1L, , ),
             bor3(nux <= 1L, , ))
expect_equal(or3s(nuy <= 9L, , ),
             bor3(nuy <= 9L, , ))
expect_equal(or3s(nuz <= 0L, , z == 0L),
             bor3(nuz <= 0L, , z == 0L))
expect_equal(or3s(nva <= 1L, , u == 1L),
             bor3(nva <= 1L, , u == 1L))
expect_equal(or3s(nvb <= 9L, , o == 9L),
             bor3(nvb <= 9L, , o == 9L))
expect_equal(or3s(nvc <= 0L, v %between% c(-1L, 1L), ),
             bor3(nvc <= 0L, v %between% c(-1L, 1L), ))
expect_equal(or3s(nvd <= 1L, z %between% c(-1L, 1L), ),
             bor3(nvd <= 1L, z %between% c(-1L, 1L), ))
expect_equal(or3s(nve <= 9L, v %between% c(-1L, 1L), ),
             bor3(nve <= 9L, v %between% c(-1L, 1L), ))
expect_equal(or3s(nvf <= 0L, b %between% c(-1L, 1L), m == 0L),
             bor3(nvf <= 0L, b %between% c(-1L, 1L), m == 0L))
expect_equal(or3s(nvg <= 1L, x %between% c(-1L, 1L), x == 1L),
             bor3(nvg <= 1L, x %between% c(-1L, 1L), x == 1L))
expect_equal(or3s(nvh <= 9L, u %between% c(-1L, 1L), r == 9L),
             bor3(nvh <= 9L, u %between% c(-1L, 1L), r == 9L))
expect_equal(or3s(nvi <= 0L, , ),
             bor3(nvi <= 0L, , ))
expect_equal(or3s(nvj <= 1L, , ),
             bor3(nvj <= 1L, , ))
expect_equal(or3s(nvk <= 9L, , ),
             bor3(nvk <= 9L, , ))
expect_equal(or3s(nvl <= 0L, , w > 0L),
             bor3(nvl <= 0L, , w > 0L))
expect_equal(or3s(nvm <= 1L, , l > 1L),
             bor3(nvm <= 1L, , l > 1L))
expect_equal(or3s(nvn <= 9L, , j > 9L),
             bor3(nvn <= 9L, , j > 9L))
expect_equal(or3s(nvo <= 0L, r %between% c(-1L, 1L), ),
             bor3(nvo <= 0L, r %between% c(-1L, 1L), ))
expect_equal(or3s(nvp <= 1L, h %between% c(-1L, 1L), ),
             bor3(nvp <= 1L, h %between% c(-1L, 1L), ))
expect_equal(or3s(nvq <= 9L, f %between% c(-1L, 1L), ),
             bor3(nvq <= 9L, f %between% c(-1L, 1L), ))
expect_equal(or3s(nvr <= 0L, q %between% c(-1L, 1L), n > 0L),
             bor3(nvr <= 0L, q %between% c(-1L, 1L), n > 0L))
expect_equal(or3s(nvs <= 1L, l %between% c(-1L, 1L), g > 1L),
             bor3(nvs <= 1L, l %between% c(-1L, 1L), g > 1L))
expect_equal(or3s(nvt <= 9L, v %between% c(-1L, 1L), w > 9L),
             bor3(nvt <= 9L, v %between% c(-1L, 1L), w > 9L))
expect_equal(or3s(nvu <= 0L, , ),
             bor3(nvu <= 0L, , ))
expect_equal(or3s(nvv <= 1L, , ),
             bor3(nvv <= 1L, , ))
expect_equal(or3s(nvw <= 9L, , ),
             bor3(nvw <= 9L, , ))
expect_equal(or3s(nvx <= 0L, , w >= 0L),
             bor3(nvx <= 0L, , w >= 0L))
expect_equal(or3s(nvy <= 1L, , u >= 1L),
             bor3(nvy <= 1L, , u >= 1L))
expect_equal(or3s(nvz <= 9L, , g >= 9L),
             bor3(nvz <= 9L, , g >= 9L))
expect_equal(or3s(nwa <= 0L, d %between% c(-1L, 1L), ),
             bor3(nwa <= 0L, d %between% c(-1L, 1L), ))
expect_equal(or3s(nwb <= 1L, b %between% c(-1L, 1L), ),
             bor3(nwb <= 1L, b %between% c(-1L, 1L), ))
expect_equal(or3s(nwc <= 9L, f %between% c(-1L, 1L), ),
             bor3(nwc <= 9L, f %between% c(-1L, 1L), ))
expect_equal(or3s(nwd <= 0L, g %between% c(-1L, 1L), i >= 0L),
             bor3(nwd <= 0L, g %between% c(-1L, 1L), i >= 0L))
expect_equal(or3s(nwe <= 1L, x %between% c(-1L, 1L), p >= 1L),
             bor3(nwe <= 1L, x %between% c(-1L, 1L), p >= 1L))
expect_equal(or3s(nwf <= 9L, u %between% c(-1L, 1L), s >= 9L),
             bor3(nwf <= 9L, u %between% c(-1L, 1L), s >= 9L))
expect_equal(or3s(nwg <= 0L, , ),
             bor3(nwg <= 0L, , ))
expect_equal(or3s(nwh <= 1L, , ),
             bor3(nwh <= 1L, , ))
expect_equal(or3s(nwi <= 9L, , ),
             bor3(nwi <= 9L, , ))
expect_equal(or3s(nwj <= 0L, , logi_n),
             bor3(nwj <= 0L, , logi_n))
expect_equal(or3s(nwk <= 1L, , logi_u),
             bor3(nwk <= 1L, , logi_u))
expect_equal(or3s(nwl <= 9L, , logi_t),
             bor3(nwl <= 9L, , logi_t))
expect_equal(or3s(nwm <= 0L, e %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(nwm <= 0L, e %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(nwn <= 1L, g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(nwn <= 1L, g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(nwo <= 9L, g %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(nwo <= 9L, g %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(nwp <= 0L, e %in% c(4L, -3L, 2L, 7L, 8L), logi_z),
             bor3(nwp <= 0L, e %in% c(4L, -3L, 2L, 7L, 8L), logi_z))
expect_equal(or3s(nwq <= 1L, t %in% c(4L, -3L, 2L, 7L, 8L), logi_k),
             bor3(nwq <= 1L, t %in% c(4L, -3L, 2L, 7L, 8L), logi_k))
expect_equal(or3s(nwr <= 9L, n %in% c(4L, -3L, 2L, 7L, 8L), logi_r),
             bor3(nwr <= 9L, n %in% c(4L, -3L, 2L, 7L, 8L), logi_r))
expect_equal(or3s(nws <= 0L, , ),
             bor3(nws <= 0L, , ))
expect_equal(or3s(nwt <= 1L, , ),
             bor3(nwt <= 1L, , ))
expect_equal(or3s(nwu <= 9L, , ),
             bor3(nwu <= 9L, , ))
expect_equal(or3s(nwv <= 0L, , logi_j),
             bor3(nwv <= 0L, , logi_j))
expect_equal(or3s(nww <= 1L, , logi_j),
             bor3(nww <= 1L, , logi_j))
expect_equal(or3s(nwx <= 9L, , logi_i),
             bor3(nwx <= 9L, , logi_i))
expect_equal(or3s(nwy <= 0L, b %in% 1:4, ),
             bor3(nwy <= 0L, b %in% 1:4, ))
expect_equal(or3s(nwz <= 1L, x %in% 1:4, ),
             bor3(nwz <= 1L, x %in% 1:4, ))
expect_equal(or3s(nxa <= 9L, l %in% 1:4, ),
             bor3(nxa <= 9L, l %in% 1:4, ))
expect_equal(or3s(nxb <= 0L, w %in% 1:4, logi_i),
             bor3(nxb <= 0L, w %in% 1:4, logi_i))
expect_equal(or3s(nxc <= 1L, x %in% 1:4, logi_b),
             bor3(nxc <= 1L, x %in% 1:4, logi_b))
expect_equal(or3s(nxd <= 9L, h %in% 1:4, logi_e),
             bor3(nxd <= 9L, h %in% 1:4, logi_e))
expect_equal(or3s(nxe <= 0L, , ),
             bor3(nxe <= 0L, , ))
expect_equal(or3s(nxf <= 1L, , ),
             bor3(nxf <= 1L, , ))
expect_equal(or3s(nxg <= 9L, , ),
             bor3(nxg <= 9L, , ))
expect_equal(or3s(nxh <= 0L, , !logi_l),
             bor3(nxh <= 0L, , !logi_l))
expect_equal(or3s(nxi <= 1L, , !logi_k),
             bor3(nxi <= 1L, , !logi_k))
expect_equal(or3s(nxj <= 9L, , !logi_y),
             bor3(nxj <= 9L, , !logi_y))
expect_equal(or3s(nxk <= 0L, i %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(nxk <= 0L, i %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(nxl <= 1L, w %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(nxl <= 1L, w %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(nxm <= 9L, j %in% c(4L, -3L, 2L, 7L, 8L), ),
             bor3(nxm <= 9L, j %in% c(4L, -3L, 2L, 7L, 8L), ))
expect_equal(or3s(nxn <= 0L, d %in% c(4L, -3L, 2L, 7L, 8L), !logi_h),
             bor3(nxn <= 0L, d %in% c(4L, -3L, 2L, 7L, 8L), !logi_h))
expect_equal(or3s(nxo <= 1L, y %in% c(4L, -3L, 2L, 7L, 8L), !logi_f),
             bor3(nxo <= 1L, y %in% c(4L, -3L, 2L, 7L, 8L), !logi_f))
expect_equal(or3s(nxp <= 9L, p %in% c(4L, -3L, 2L, 7L, 8L), !logi_n),
             bor3(nxp <= 9L, p %in% c(4L, -3L, 2L, 7L, 8L), !logi_n))
expect_equal(or3s(nxq <= 0L, , ),
             bor3(nxq <= 0L, , ))
expect_equal(or3s(nxr <= 1L, , ),
             bor3(nxr <= 1L, , ))
expect_equal(or3s(nxs <= 9L, , ),
             bor3(nxs <= 9L, , ))
expect_equal(or3s(nxt <= 0L, , !logi_x),
             bor3(nxt <= 0L, , !logi_x))
expect_equal(or3s(nxu <= 1L, , !logi_d),
             bor3(nxu <= 1L, , !logi_d))
expect_equal(or3s(nxv <= 9L, , !logi_s),
             bor3(nxv <= 9L, , !logi_s))
expect_equal(or3s(nxw <= 0L, t %in% 1:4, ),
             bor3(nxw <= 0L, t %in% 1:4, ))
expect_equal(or3s(nxx <= 1L, e %in% 1:4, ),
             bor3(nxx <= 1L, e %in% 1:4, ))
expect_equal(or3s(nxy <= 9L, s %in% 1:4,