###########################################################################
# Copyright 2009 Nobody                                                   #
#                                                                         #
# This file is part of hergm.                                             #
#                                                                         # 
#    hergm is free software: you can redistribute it and/or modify        #
#    it under the terms of the GNU General Public License as published by #
#    the Free Software Foundation, either version 3 of the License, or    #
#    (at your option) any later version.                                  #
#                                                                         # 
#    hergm is distributed in the hope that it will be useful,             #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of       #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        #
#    GNU General Public License for more details.                         #
#                                                                         #
#    You should have received a copy of the GNU General Public License    #
#    along with hergm.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                         # 
###########################################################################

hergm.postprocess <- function(sample = NULL,
                              burnin = 0, 
                              thinning = 1,
                              relabel = FALSE,
                              name = "", 
                              ...)
# input: MCMC sample generated by function hergm, number of burn-in iterations, relabeling desired, output desired, root of name of output files 
# output: postprocessed MCMC sample
{

  # Extract arguments
  n <- sample$n
  max_number <- sample$max_number
  d1 <- sample$d1
  d2 <- sample$d2
  parallel <- sample$parallel
  simulate <- sample$simulate
  sample_size <- sample$sample_size
  mcmc <- sample$sample 

  # Check arguments
  if (burnin > sample_size) 
    {
    cat("\n\n")
    error_message <- paste("number of burn-in iterations ", burnin, " exceeds number of recorded iterations ", sample_size, ".", sep = "")
    stop(error_message, call. = FALSE)
    }
  if ((d2 == 0) || (simulate == TRUE)) relabel <- FALSE
  if ((relabel == TRUE) && (max_number > 10)) cat("\nWarning: relabeling time-consuming.\n")

  # Preprocess MCMC sample: delete burn-in iterations and transform vector into matrix, where rows correspond to MCMC draws
  d <- d1 + d2
  terms <- length_mcmc(d1, d2, max_number, n)  
  mcmc_sample <- NULL
  count <- 0
  for (i in 1:parallel) 
    {
    first <- count + (burnin * terms) + 1
    last <- count + (sample_size * terms)
    mcmc_sample <- append(mcmc_sample, mcmc[first:last])
    count <- count + (sample_size * terms)
    }
  mcmc_sample_size <- parallel * (sample_size - burnin)
  mcmc_sample <- matrix(mcmc_sample, nrow = mcmc_sample_size, ncol = terms, byrow = TRUE)
  if (thinning > 1)
    {
    for (i in sample_size:1) 
      {
      if (trunc(i / thinning) != (i / thinning)) mcmc_sample <- mcmc_sample[-i,] 
      }
    mcmc_sample_size <- nrow(mcmc_sample)
    }

  # Initialize arrays
  output <- list()
  if (d1 > 0) output$ergm_theta <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d1) 
  if (d2 > 0)
    {
    output$eta_mean <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2)
    output$eta_precision <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2)
    output$hergm_theta <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2 * (max_number + 1))
    output$indicator <- matrix(data = 0, nrow = mcmc_sample_size, ncol = n)
    output$size <- matrix(data = 0, nrow = mcmc_sample_size, ncol = max_number)
    output$p_k <- matrix(data = 0, nrow = mcmc_sample_size, ncol = max_number)
    output$alpha <- matrix(data = 0, nrow = mcmc_sample_size, ncol = 1)
    output$prediction <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d)
    }

  # Process MCMC sample
  for (row in 1:mcmc_sample_size)
    {
    column <- 0
    if (d1 > 0)
      {
      for (i in 1:d1) 
        {
        column <- column + 1
        output$ergm_theta[row,i] <- mcmc_sample[row,column]
        }
      }
    if (d2 > 0)
      {
      for (i in 1:d2) 
        {
        column <- column + 1
        output$eta_mean[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:d2) 
        {
        column <- column + 1
        output$eta_precision[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:(d2 * (max_number + 1))) 
        {
        column <- column + 1
        output$hergm_theta[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:n) 
        {
        column <- column + 1
        output$indicator[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:max_number) 
        {
        column <- column + 1
        output$size[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:max_number) 
        {
        column <- column + 1
        output$p_k[row,i] <- mcmc_sample[row,column]
        }
      column <- column + 1
      output$alpha[row,1] <- mcmc_sample[row,column]
      for (i in 1:d) 
        {
        column <- column + 1
        output$prediction[row,i] <- mcmc_sample[row,column]
        }
      }
    }

  # Relabel sample
  if (relabel == TRUE)
    {
    minimizer <- hergm.min_loss(max_number, output$indicator, 0, 100) # Specify number of iterations of post-processing algorithm
    output$loss <- minimizer$loss
    output$indicator_min <- minimizer$indicator
    output$p_i_k <- minimizer$p
    index <- 0
    for (h_term in 1:d2)
      {
      index <- index + 1
      theta <- output$hergm_theta[,index]
      for (i in 2:max_number) 
        {
        index <- index + 1
        theta <- cbind(theta, output$hergm_theta[,index])
        }
      output$hergm_theta_min <- hergm.permute_mcmc(theta, max_number, minimizer$min_permutations) 
      }
    }

  cat("\n")
 
  output

}

