/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 
#define H5G_FRIEND     

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FDprivate.h" 
#include "H5FLprivate.h" 
#include "H5Gpkg.h"      
#include "H5MMprivate.h" 

static herr_t H5F__cache_superblock_get_initial_load_size(void *udata, size_t *image_len);
static herr_t H5F__cache_superblock_get_final_load_size(const void *image_ptr, size_t image_len, void *udata,
                                                        size_t *actual_len);
static htri_t H5F__cache_superblock_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5F__cache_superblock_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5F__cache_superblock_image_len(const void *thing, size_t *image_len);
static herr_t H5F__cache_superblock_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5F__cache_superblock_free_icr(void *thing);

static herr_t H5F__cache_drvrinfo_get_initial_load_size(void *udata, size_t *image_len);
static herr_t H5F__cache_drvrinfo_get_final_load_size(const void *image_ptr, size_t image_len, void *udata,
                                                      size_t *actual_len);
static void  *H5F__cache_drvrinfo_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5F__cache_drvrinfo_image_len(const void *thing, size_t *image_len);
static herr_t H5F__cache_drvrinfo_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5F__cache_drvrinfo_free_icr(void *thing);

static herr_t H5F__superblock_prefix_decode(H5F_super_t *sblock, const uint8_t **image_ref, size_t len,
                                            const H5F_superblock_cache_ud_t *udata, bool extend_eoa);
static herr_t H5F__drvrinfo_prefix_decode(H5O_drvinfo_t *drvinfo, char *drv_name, const uint8_t **image_ref,
                                          size_t len, H5F_drvrinfo_cache_ud_t *udata, bool extend_eoa);

const H5AC_class_t H5AC_SUPERBLOCK[1] = {{
    H5AC_SUPERBLOCK_ID,                          
    "Superblock",                                
    H5FD_MEM_SUPER,                              
    H5AC__CLASS_SPECULATIVE_LOAD_FLAG,           
    H5F__cache_superblock_get_initial_load_size, 
    H5F__cache_superblock_get_final_load_size,   
    H5F__cache_superblock_verify_chksum,         
    H5F__cache_superblock_deserialize,           
    H5F__cache_superblock_image_len,             
    NULL,                                        
    H5F__cache_superblock_serialize,             
    NULL,                                        
    H5F__cache_superblock_free_icr,              
    NULL,                                        
}};

const H5AC_class_t H5AC_DRVRINFO[1] = {{
    H5AC_DRVRINFO_ID,                          
    "Driver info block",                       
    H5FD_MEM_SUPER,                            
    H5AC__CLASS_SPECULATIVE_LOAD_FLAG,         
    H5F__cache_drvrinfo_get_initial_load_size, 
    H5F__cache_drvrinfo_get_final_load_size,   
    NULL,                                      
    H5F__cache_drvrinfo_deserialize,           
    H5F__cache_drvrinfo_image_len,             
    NULL,                                      
    H5F__cache_drvrinfo_serialize,             
    NULL,                                      
    H5F__cache_drvrinfo_free_icr,              
    NULL,                                      
}};

H5FL_EXTERN(H5F_super_t);

static herr_t
H5F__superblock_prefix_decode(H5F_super_t *sblock, const uint8_t **image_ref, size_t len,
                              const H5F_superblock_cache_ud_t *udata, bool extend_eoa)
{
    const uint8_t *image     = (const uint8_t *)*image_ref; 
    const uint8_t *end       = image + len - 1;             
    htri_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(sblock);
    assert(image_ref);
    assert(image);
    assert(udata);
    assert(udata->f);

    
    if (H5_IS_BUFFER_OVERFLOW(image, H5F_SIGNATURE_LEN, end))
        HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    image += H5F_SIGNATURE_LEN;

    
    if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
        HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    sblock->super_vers = *image++;
    if (sblock->super_vers > HDF5_SUPERBLOCK_VERSION_LATEST)
        HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "bad superblock version number");

    
    if (((size_t)(image - (const uint8_t *)*image_ref)) != H5F_SUPERBLOCK_FIXED_SIZE)
        HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "bad superblock (fixed) size");

    
    if (sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2) {
        if (H5_IS_BUFFER_OVERFLOW(image, 6, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        sblock->sizeof_addr = image[4];
        sblock->sizeof_size = image[5];
    }
    else {
        if (H5_IS_BUFFER_OVERFLOW(image, 2, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        sblock->sizeof_addr = image[0];
        sblock->sizeof_size = image[1];
    }

    if (sblock->sizeof_addr != 2 && sblock->sizeof_addr != 4 && sblock->sizeof_addr != 8 &&
        sblock->sizeof_addr != 16 && sblock->sizeof_addr != 32)
        HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "bad byte number in an address");
    if (sblock->sizeof_size != 2 && sblock->sizeof_size != 4 && sblock->sizeof_size != 8 &&
        sblock->sizeof_size != 16 && sblock->sizeof_size != 32)
        HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "bad byte number for object size");

    
    if (extend_eoa) {
        size_t variable_size; 

        
        variable_size =
            (size_t)H5F_SUPERBLOCK_VARLEN_SIZE(sblock->super_vers, sblock->sizeof_addr, sblock->sizeof_size);
        if (variable_size == 0)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "variable size can't be zero");

        
        if (H5F__set_eoa(udata->f, H5FD_MEM_SUPER, (haddr_t)(H5F_SUPERBLOCK_FIXED_SIZE + variable_size)) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "set end of space allocation request failed");
    }

    
    *image_ref = image;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__drvrinfo_prefix_decode(H5O_drvinfo_t *drvrinfo, char *drv_name, const uint8_t **image_ref, size_t len,
                            H5F_drvrinfo_cache_ud_t *udata, bool extend_eoa)
{
    const uint8_t *image = (const uint8_t *)*image_ref; 
    const uint8_t *end   = image + len - 1;             
    unsigned       drv_vers;                            
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(drvrinfo);
    assert(image_ref);
    assert(image);
    assert(udata);
    assert(udata->f);

    
    if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
        HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    drv_vers = *image++;
    if (drv_vers != HDF5_DRIVERINFO_VERSION_0)
        HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, FAIL, "bad driver information block version number");

    
    if (H5_IS_BUFFER_OVERFLOW(image, 3, end))
        HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    image += 3;

    
    if (H5_IS_BUFFER_OVERFLOW(image, 4, end))
        HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    UINT32DECODE(image, drvrinfo->len);

    
    if (drv_name) {
        if (H5_IS_BUFFER_OVERFLOW(image, 8, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        H5MM_memcpy(drv_name, (const char *)image, (size_t)8);
        drv_name[8] = '\0';
        image += 8; 
    }

    
    if (extend_eoa) {
        haddr_t eoa;     
        haddr_t min_eoa; 

        
        eoa = H5FD_get_eoa(udata->f->shared->lf, H5FD_MEM_SUPER);
        if (!H5_addr_defined(eoa))
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "driver get_eoa request failed");

        
        min_eoa = udata->driver_addr + H5F_DRVINFOBLOCK_HDR_SIZE + drvrinfo->len;

        
        if (H5_addr_gt(min_eoa, eoa))
            if (H5FD_set_eoa(udata->f->shared->lf, H5FD_MEM_SUPER, min_eoa) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "set end of space allocation request failed");
    }

    
    *image_ref = image;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__cache_superblock_get_initial_load_size(void H5_ATTR_UNUSED *_udata, size_t *image_len)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(image_len);

    
    *image_len = H5F_SUPERBLOCK_SPEC_READ_SIZE;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5F__cache_superblock_get_final_load_size(const void *_image, size_t image_len, void *_udata,
                                          size_t *actual_len)
{
    const uint8_t             *image = _image;                              
    H5F_superblock_cache_ud_t *udata = (H5F_superblock_cache_ud_t *)_udata; 
    H5F_super_t                sblock;                                      
    htri_t                     ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(udata);
    assert(actual_len);
    assert(*actual_len == image_len);
    assert(image_len >= H5F_SUPERBLOCK_FIXED_SIZE + 6);

    
    if (H5F__superblock_prefix_decode(&sblock, &image, image_len, udata, true) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTDECODE, FAIL, "can't decode file superblock prefix");

    
    udata->super_vers = sblock.super_vers;

    
    *actual_len = H5F_SUPERBLOCK_FIXED_SIZE + (size_t)H5F_SUPERBLOCK_VARLEN_SIZE(
                                                  sblock.super_vers, sblock.sizeof_addr, sblock.sizeof_size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5F__cache_superblock_verify_chksum(const void *_image, size_t len, void *_udata)
{
    const uint8_t             *image = _image;                              
    H5F_superblock_cache_ud_t *udata = (H5F_superblock_cache_ud_t *)_udata; 
    uint32_t                   stored_chksum;   
    uint32_t                   computed_chksum; 
    htri_t                     ret_value = true;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(udata);

    
    if (udata->super_vers >= HDF5_SUPERBLOCK_VERSION_2) {

        
        if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get checksums");

        if (stored_chksum != computed_chksum)
            ret_value = false;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5F__cache_superblock_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5F_super_t               *sblock    = NULL;                                
    H5F_superblock_cache_ud_t *udata     = (H5F_superblock_cache_ud_t *)_udata; 
    const uint8_t             *image     = _image;          
    const uint8_t             *end       = image + len - 1; 
    H5F_super_t               *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(udata);
    assert(udata->f);
    assert(len >= H5F_SUPERBLOCK_FIXED_SIZE + 6);

    
    if (NULL == (sblock = H5FL_CALLOC(H5F_super_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5F__superblock_prefix_decode(sblock, &image, len, udata, false) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTDECODE, NULL, "can't decode file superblock prefix");

    
    if (sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2) {
        uint32_t status_flags;  
        unsigned sym_leaf_k;    
        unsigned snode_btree_k; 
        unsigned chunk_btree_k; 

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        if (HDF5_FREESPACE_VERSION != *image++)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, NULL, "bad free space version number");

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        if (HDF5_OBJECTDIR_VERSION != *image++)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, NULL, "bad object directory version number");

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        image++;

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        if (HDF5_SHAREDHEADER_VERSION != *image++)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, NULL, "bad shared-header format version number");

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        image++;
        udata->f->shared->sizeof_addr = sblock->sizeof_addr; 

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        image++;
        udata->f->shared->sizeof_size = sblock->sizeof_size; 

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        image++;

        
        if (H5_IS_BUFFER_OVERFLOW(image, 2, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        UINT16DECODE(image, sym_leaf_k);
        if (sym_leaf_k == 0)
            HGOTO_ERROR(H5E_FILE, H5E_BADRANGE, NULL, "bad symbol table leaf node 1/2 rank");
        udata->sym_leaf_k = sym_leaf_k; 

        
        if (H5_IS_BUFFER_OVERFLOW(image, 2, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        UINT16DECODE(image, snode_btree_k);
        if (snode_btree_k == 0)
            HGOTO_ERROR(H5E_FILE, H5E_BADRANGE, NULL, "bad 1/2 rank for btree internal nodes");
        udata->btree_k[H5B_SNODE_ID] = snode_btree_k;

        

        
        if (H5_IS_BUFFER_OVERFLOW(image, 4, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        UINT32DECODE(image, status_flags);
        if (status_flags > 255)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, NULL, "bad superblock status flags");
        sblock->status_flags = (uint8_t)status_flags;
        if (sblock->status_flags & ~H5F_SUPER_ALL_FLAGS)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, NULL, "bad flag value for superblock");

        
        if (sblock->super_vers > HDF5_SUPERBLOCK_VERSION_DEF) {
            if (H5_IS_BUFFER_OVERFLOW(image, 2, end))
                HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
            UINT16DECODE(image, chunk_btree_k);

            
            if (sblock->super_vers == HDF5_SUPERBLOCK_VERSION_1) {
                
                if (H5_IS_BUFFER_OVERFLOW(image, 2, end))
                    HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
                image += 2;
            }
        }
        else
            chunk_btree_k = HDF5_BTREE_CHUNK_IK_DEF;
        udata->btree_k[H5B_CHUNK_ID] = chunk_btree_k;

        
        if (H5_IS_BUFFER_OVERFLOW(image, H5F_sizeof_addr(udata->f) * 4, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &sblock->base_addr );
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &sblock->ext_addr );
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &udata->stored_eof );
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &sblock->driver_addr );

        
        if (sblock->root_ent)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, NULL, "root entry should not exist yet");
        if (NULL == (sblock->root_ent = (H5G_entry_t *)H5MM_calloc(sizeof(H5G_entry_t))))
            HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, NULL,
                        "can't allocate space for root group symbol table entry");

        
        if (H5G_ent_decode(udata->f, (const uint8_t **)&image, sblock->root_ent, end) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTDECODE, NULL, "can't decode root group symbol table entry");

        
        sblock->root_addr = sblock->root_ent->header;

        
        if (udata->ignore_drvrinfo && H5_addr_defined(sblock->driver_addr)) {
            
            sblock->driver_addr     = HADDR_UNDEF;
            udata->drvrinfo_removed = true;
        }

        
    }
    else {
        uint32_t read_chksum; 

        
        image++;
        udata->f->shared->sizeof_addr = sblock->sizeof_addr; 
        
        image++;
        udata->f->shared->sizeof_size = sblock->sizeof_size; 

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");

        
        sblock->status_flags = *image++;
        if (sblock->status_flags & ~H5F_SUPER_ALL_FLAGS)
            HGOTO_ERROR(H5E_FILE, H5E_BADVALUE, NULL, "bad flag value for superblock");

        
        if (H5_IS_BUFFER_OVERFLOW(image, H5F_SIZEOF_ADDR(udata->f) * 4, end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");

        
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &sblock->base_addr );
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &sblock->ext_addr );
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &udata->stored_eof );
        H5F_addr_decode(udata->f, (const uint8_t **)&image, &sblock->root_addr );

        

        
        if (H5_IS_BUFFER_OVERFLOW(image, sizeof(uint32_t), end))
            HGOTO_ERROR(H5E_FILE, H5E_OVERFLOW, NULL, "image pointer is out of bounds");

        
        UINT32DECODE(image, read_chksum);

        
        sblock->driver_addr = HADDR_UNDEF;
    }

    
    if ((size_t)(image - (const uint8_t *)_image) > len)
        HDONE_ERROR(H5E_FILE, H5E_BADVALUE, NULL, "bad decoded superblock size");

    ret_value = sblock;

done:
    
    if (!ret_value && sblock)
        if (H5F__super_free(sblock) < 0)
            HDONE_ERROR(H5E_FILE, H5E_CANTFREE, NULL, "unable to destroy superblock data");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__cache_superblock_image_len(const void *_thing, size_t *image_len)
{
    const H5F_super_t *sblock = (const H5F_super_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(sblock);
    assert(sblock->cache_info.type == H5AC_SUPERBLOCK);
    assert(image_len);

    
    *image_len = (size_t)H5F_SUPERBLOCK_SIZE(sblock);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5F__cache_superblock_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5F_super_t *sblock = (H5F_super_t *)_thing; 
    uint8_t     *image  = _image;                
    haddr_t      rel_eof;                        
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(image);
    assert(sblock);

    
    
    assert(sblock->cache_info.flush_me_last);

    
    H5MM_memcpy(image, H5F_SIGNATURE, (size_t)H5F_SIGNATURE_LEN);
    image += H5F_SIGNATURE_LEN;
    *image++ = (uint8_t)sblock->super_vers;

    
    if (sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2) {
        *image++ = (uint8_t)HDF5_FREESPACE_VERSION; 
        *image++ = (uint8_t)HDF5_OBJECTDIR_VERSION; 
        *image++ = 0;                               

        *image++ = (uint8_t)HDF5_SHAREDHEADER_VERSION; 
        *image++ = sblock->sizeof_addr;
        *image++ = sblock->sizeof_size;
        *image++ = 0; 

        UINT16ENCODE(image, sblock->sym_leaf_k);
        UINT16ENCODE(image, sblock->btree_k[H5B_SNODE_ID]);
        UINT32ENCODE(image, (uint32_t)sblock->status_flags);

        
        if (sblock->super_vers > HDF5_SUPERBLOCK_VERSION_DEF) {
            UINT16ENCODE(image, sblock->btree_k[H5B_CHUNK_ID]);
            *image++ = 0; 
            *image++ = 0; 
        }                 

        
        H5F_addr_encode(f, &image, sblock->base_addr);

        
        H5F_addr_encode(f, &image, sblock->ext_addr);

        
        if ((rel_eof = H5FD_get_eoa(f->shared->lf, H5FD_MEM_SUPER)) == HADDR_UNDEF)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "driver get_eoa request failed");
        H5F_addr_encode(f, &image, (rel_eof + sblock->base_addr));

        
        H5F_addr_encode(f, &image, sblock->driver_addr);

        
        if (H5G_ent_encode(f, &image, sblock->root_ent) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTENCODE, FAIL, "can't encode root group symbol table entry");

        

    }                         
    else {                    
        uint32_t   chksum;    
        H5O_loc_t *root_oloc; 

        
        *image++ = sblock->sizeof_addr;
        *image++ = sblock->sizeof_size;
        *image++ = sblock->status_flags;

        
        H5F_addr_encode(f, &image, sblock->base_addr);

        
        H5F_addr_encode(f, &image, sblock->ext_addr);

        
        if ((rel_eof = H5FD_get_eoa(f->shared->lf, H5FD_MEM_SUPER)) == HADDR_UNDEF)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "driver get_eoa request failed");
        H5F_addr_encode(f, &image, (rel_eof + sblock->base_addr));

        
        if (NULL == (root_oloc = H5G_oloc(f->shared->root_grp)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to retrieve root group information");

        
        H5F_addr_encode(f, &image, root_oloc->addr);

        
        chksum = H5_checksum_metadata(_image, ((size_t)H5F_SUPERBLOCK_SIZE(sblock) - H5F_SIZEOF_CHKSUM), 0);

        
        UINT32ENCODE(image, chksum);

        
        assert((size_t)(image - (uint8_t *)_image) == (size_t)H5F_SUPERBLOCK_SIZE(sblock));
    }

    
    assert((size_t)(image - (uint8_t *)_image) == len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__cache_superblock_free_icr(void *_thing)
{
    H5F_super_t *sblock    = (H5F_super_t *)_thing; 
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(sblock);
    assert(sblock->cache_info.type == H5AC_SUPERBLOCK);

    
    if (H5F__super_free(sblock) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL, "unable to free superblock");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__cache_drvrinfo_get_initial_load_size(void H5_ATTR_UNUSED *_udata, size_t *image_len)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(image_len);

    
    *image_len = H5F_DRVINFOBLOCK_HDR_SIZE; 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5F__cache_drvrinfo_get_final_load_size(const void *_image, size_t image_len, void *_udata,
                                        size_t *actual_len)
{
    const uint8_t           *image = _image;                            
    H5F_drvrinfo_cache_ud_t *udata = (H5F_drvrinfo_cache_ud_t *)_udata; 
    H5O_drvinfo_t            drvrinfo;                                  
    herr_t                   ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(udata);
    assert(actual_len);
    assert(*actual_len == image_len);
    assert(image_len == H5F_DRVINFOBLOCK_HDR_SIZE);

    
    if (H5F__drvrinfo_prefix_decode(&drvrinfo, NULL, &image, image_len, udata, true) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTDECODE, FAIL, "can't decode file driver info prefix");

    
    *actual_len = H5F_DRVINFOBLOCK_HDR_SIZE + drvrinfo.len;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5F__cache_drvrinfo_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5O_drvinfo_t           *drvinfo = NULL;                              
    H5F_drvrinfo_cache_ud_t *udata   = (H5F_drvrinfo_cache_ud_t *)_udata; 
    const uint8_t           *image   = _image;                            
    char                     drv_name[9];                                 
    H5O_drvinfo_t           *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(len >= H5F_DRVINFOBLOCK_HDR_SIZE);
    assert(udata);
    assert(udata->f);

    
    if (NULL == (drvinfo = (H5O_drvinfo_t *)H5MM_calloc(sizeof(H5O_drvinfo_t))))
        HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, NULL, "memory allocation failed for driver info message");

    
    if (H5F__drvrinfo_prefix_decode(drvinfo, drv_name, &image, len, udata, false) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTDECODE, NULL, "can't decode file driver info prefix");

    
    assert(len == (H5F_DRVINFOBLOCK_HDR_SIZE + drvinfo->len));

    
    if (H5FD_sb_load(udata->f->shared->lf, drv_name, image) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTDECODE, NULL, "unable to decode driver information");

    
    assert((size_t)(image - (const uint8_t *)_image) <= len);

    ret_value = drvinfo;

done:
    
    if (!ret_value && drvinfo)
        H5MM_xfree(drvinfo);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__cache_drvrinfo_image_len(const void *_thing, size_t *image_len)
{
    const H5O_drvinfo_t *drvinfo = (const H5O_drvinfo_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(drvinfo);
    assert(drvinfo->cache_info.type == H5AC_DRVRINFO);
    assert(image_len);

    
    *image_len = (size_t)(H5F_DRVINFOBLOCK_HDR_SIZE + 
                          drvinfo->len);              

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5F__cache_drvrinfo_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_thing)
{
    H5O_drvinfo_t *drvinfo = (H5O_drvinfo_t *)_thing; 
    uint8_t       *image   = _image;                  
    uint8_t       *dbuf;                              
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(image);
    assert(drvinfo);
    assert(drvinfo->cache_info.type == H5AC_DRVRINFO);
    assert(len == (size_t)(H5F_DRVINFOBLOCK_HDR_SIZE + drvinfo->len));

    
    dbuf = image;

    
    *image++ = HDF5_DRIVERINFO_VERSION_0; 
    *image++ = 0;                         
    *image++ = 0;                         
    *image++ = 0;                         

    
    UINT32ENCODE(image, drvinfo->len);

    
    if (H5FD_sb_encode(f->shared->lf, (char *)image, dbuf + H5F_DRVINFOBLOCK_HDR_SIZE) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to encode driver information");

    
    image += 8 + drvinfo->len;

    
    assert((size_t)(image - (uint8_t *)_image) == len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__cache_drvrinfo_free_icr(void *_thing)
{
    H5O_drvinfo_t *drvinfo = (H5O_drvinfo_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(drvinfo);
    assert(drvinfo->cache_info.type == H5AC_DRVRINFO);

    
    H5MM_xfree(drvinfo);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
