/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5TSmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5TSpkg.h"    

#ifdef H5_HAVE_THREADS

herr_t
H5TS_barrier_init(H5TS_barrier_t *barrier, unsigned count)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == barrier || 0 == count))
        HGOTO_DONE(FAIL);

#ifdef H5_HAVE_PTHREAD_BARRIER
    
    if (H5_UNLIKELY(pthread_barrier_init(barrier, NULL, count)))
        HGOTO_DONE(FAIL);
#else
    
    barrier->count = count;
    H5TS_atomic_init_uint(&barrier->openings, count);
    H5TS_atomic_init_uint(&barrier->generation, 0);
#endif

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_barrier_destroy(H5TS_barrier_t *barrier)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == barrier))
        HGOTO_DONE(FAIL);

#ifdef H5_HAVE_PTHREAD_BARRIER
    if (H5_UNLIKELY(pthread_barrier_destroy(barrier)))
        HGOTO_DONE(FAIL);
#else
    
    H5TS_atomic_destroy_uint(&barrier->openings);
    H5TS_atomic_destroy_uint(&barrier->generation);
#endif

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#endif 
