% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_duration}
\alias{fmt_duration}
\title{Format numeric or duration values as styled time duration strings}
\usage{
fmt_duration(
  data,
  columns,
  rows = everything(),
  input_units = NULL,
  output_units = NULL,
  duration_style = c("narrow", "wide", "colon-sep", "iso"),
  trim_zero_units = TRUE,
  max_output_units = NULL,
  pattern = "{x}",
  use_seps = TRUE,
  sep_mark = ",",
  force_sign = FALSE,
  system = c("intl", "ind"),
  locale = NULL
)
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{columns}{The columns to format. Can either be a series of column names
provided in \code{\link[=c]{c()}}, a vector of column indices, or a helper function
focused on selections. The select helper functions are: \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and
\code{\link[=everything]{everything()}}.}

\item{rows}{Optional rows to format. Providing \code{\link[=everything]{everything()}} (the
default) results in all rows in \code{columns} being formatted. Alternatively,
we can supply a vector of row captions within \code{\link[=c]{c()}}, a vector of row
indices, or a helper function focused on selections. The select helper
functions are: \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}. We can also use expressions
to filter down to the rows we need (e.g.,
\verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{input_units}{If one or more selected columns contains numeric values, a
keyword must be provided for \code{input_units} for \strong{gt} to determine how
those values are to be interpreted in terms of duration. The accepted units
are: \code{"seconds"}, \code{"minutes"}, \code{"hours"}, \code{"days"}, and \code{"weeks"}.}

\item{output_units}{Controls the output time units. The default, \code{NULL},
means that \strong{gt} will automatically choose time units based on the input
duration value. To control which time units are to be considered for output
(before trimming with \code{trim_zero_units}) we can specify a vector of one or
more of the following keywords: \code{"weeks"}, \code{"days"}, \code{"hours"},
\code{"minutes"}, or \code{"seconds"}.}

\item{duration_style}{A choice of four formatting styles for the output
duration values. With \code{"narrow"} (the default style), duration values will
be formatted with single letter time-part units (e.g., 1.35 days will be
styled as \verb{"1d 8h 24m}). With \code{"wide"}, this example value will be expanded
to \code{"1 day 8 hours 24 minutes"} after formatting. The \code{"colon-sep"} style
will put days, hours, minutes, and seconds in the \code{"([D]/)[HH]:[MM]:[SS]"}
format. The \code{"iso"} style will produce a value that conforms to the ISO
8601 rules for duration values (e.g., 1.35 days will become \code{"P1DT8H24M"}).}

\item{trim_zero_units}{Provides methods to remove output time units that have
zero values. By default this is \code{TRUE} and duration values that might
otherwise be formatted as \code{"0w 1d 0h 4m 19s"} with
\code{trim_zero_units = FALSE} are instead displayed as \code{"1d 4m 19s"}. Aside
from using \code{TRUE}/\code{FALSE} we could provide a vector of keywords for more
precise control. These keywords are: (1) \code{"leading"}, to omit all leading
zero-value time units (e.g., \code{"0w 1d"} -> \code{"1d"}), (2) \code{"trailing"}, to
omit all trailing zero-value time units (e.g., \code{"3d 5h 0s"} -> \code{"3d 5h"}),
and \code{"internal"}, which removes all internal zero-value time units (e.g.,
\code{"5d 0h 33m"} -> \code{"5d 33m"}).}

\item{max_output_units}{If \code{output_units} is \code{NULL}, where the output time
units are unspecified and left to \strong{gt} to handle, a numeric value
provided for \code{max_output_units} will be taken as the maximum number of time
units to display in all output time duration values. By default, this is
\code{NULL} and all possible time units will be displayed. This option has no
effect when \code{duration_style = "colon-sep"} (only \code{output_units} can be used
to customize that type of duration output).}

\item{pattern}{A formatting pattern that allows for decoration of the
formatted value. The value itself is represented by \code{{x}} and all other
characters are taken to be string literals.}

\item{use_seps}{An option to use digit group separators. The type of digit
group separator is set by \code{sep_mark} and overridden if a locale ID is
provided to \code{locale}. This setting is \code{TRUE} by default.}

\item{sep_mark}{The mark to use as a separator between groups of digits
(e.g., using \code{sep_mark = ","} with \code{1000} would result in a formatted value
of \verb{1,000}).}

\item{force_sign}{Should the positive sign be shown for positive values
(effectively showing a sign for all values except zero)? If so, use \code{TRUE}
for this option. The default is \code{FALSE}, where only negative value will
display a minus sign.}

\item{system}{The numbering system to use. By default, this is the
international numbering system (\code{"intl"}) whereby grouping separators
(i.e., \code{sep_mark}) are separated by three digits. The alternative system,
the Indian numbering system (\code{"ind"}) uses grouping separators that
correspond to thousand, lakh, crore, and higher quantities.}

\item{locale}{An optional locale ID that can be used for formatting the value
according the locale's rules. Examples include \code{"en"} for English (United
States) and \code{"fr"} for French (France). The use of a valid locale ID will
override any values provided in \code{sep_mark} and \code{dec_mark}. We can use the
\code{\link[=info_locales]{info_locales()}} function as a useful reference for all of the locales that
are supported. Any \code{locale} value provided here will override any global
locale setting performed in \code{\link[=gt]{gt()}}'s own \code{locale} argument.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Format input values to time duration values whether those input values are
numbers or of the \code{difftime} class. We can specify which time units any
numeric input values have (as weeks, days, hours, minutes, or seconds) and
the output can be customized with a duration style (corresponding to narrow,
wide, colon-separated, and ISO forms) and a choice of output units ranging
from weeks to seconds.
}
\section{Targeting the values to be formatted}{


Targeting of values is done through \code{columns} and additionally by \code{rows} (if
nothing is provided for \code{rows} then entire columns are selected). Conditional
formatting is possible by providing a conditional expression to the \code{rows}
argument. See the \emph{Arguments} section for more information on this.
}

\section{Output units for the colon-separated duration style}{


The colon-separated duration style (enabled when
\code{duration_style = "colon-sep"}) is essentially a clock-based output format
which uses the display logic of chronograph watch functionality. It will, by
default, display duration values in the \verb{(D/)HH:MM:SS} format. Any duration
values greater than or equal to 24 hours will have the number of days
prepended with an adjoining slash mark. While this output format is
versatile, it can be changed somewhat with the \code{output_units} option. The
following combinations of output units are permitted:
\itemize{
\item \code{c("minutes", "seconds")} -> \code{MM:SS}
\item \code{c("hours", "minutes")} -> \code{HH:MM}
\item \code{c("hours", "minutes", "seconds")} -> \code{HH:MM:SS}
\item \code{c("days", "hours", "minutes")} -> \verb{(D/)HH:MM}
}

Any other specialized combinations will result in the default set being used,
which is \code{c("days", "hours", "minutes", "seconds")}
}

\section{Examples}{


Use part of the \code{sp500} table to create a \strong{gt} table. Create a
\code{difftime}-based column and format the duration values to be displayed as the
number of days since March 30, 2020.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sp500 \%>\%
  dplyr::slice_head(n = 10) \%>\%
  dplyr::mutate(
    time_point = lubridate::ymd("2020-03-30"),
    time_passed = difftime(time_point, date)
  ) \%>\%
  dplyr::select(time_passed, open, close) \%>\%
  gt(rowname_col = "month") \%>\%
  fmt_duration(
    columns = time_passed,
    output_units = "days",
    duration_style = "wide"
  ) \%>\%
  fmt_currency(columns = c(open, close))
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_duration_1.png" alt="This image of a table was generated from the first code example in the `fmt_duration()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-14
}

\seealso{
Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_date}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_time}()},
\code{\link{fmt}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()},
\code{\link{text_transform}()}
}
\concept{data formatting functions}
