% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modify_columns.R
\name{cols_merge}
\alias{cols_merge}
\title{Merge data from two or more columns to a single column}
\usage{
cols_merge(
  data,
  columns,
  hide_columns = columns[-1],
  rows = everything(),
  pattern = NULL
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\verb{<column-targeting expression>} // \strong{required}

The columns for which the merging operations should be applied. The first
column resolved will be the target column (i.e., undergo mutation) and the
other columns will serve to provide input. Can either be a series of column
names provided in \code{c()}, a vector of column indices, or a select helper
function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). A vector is recommended because in that
case we are absolutely certain about the order of columns, and, that order
information is needed for this and other arguments.}

\item{hide_columns}{\emph{Subset of \code{columns} to hide}

\verb{<column-targeting expression>|FALSE} // \emph{default:} \code{columns[-1]}

Any column names provided here will have their state changed to \code{hidden}
(via internal use of \code{\link[=cols_hide]{cols_hide()}}) if they aren't already hidden. This is
convenient if the shared purpose of these specified columns is only to
provide string input to the target column. To suppress any hiding of
columns, \code{FALSE} can be used here.}

\item{rows}{\emph{Rows to target}

\verb{<row-targeting expression>} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
participate in the merging process. The default \code{\link[=everything]{everything()}} results in
all rows in \code{columns} being formatted. Alternatively, we can supply a
vector of row IDs within \code{c()}, a vector of row indices, or a select
helper function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}},
\code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use
expressions to filter down to the rows we need
(e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{pattern}{\emph{Formatting pattern}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

A formatting pattern that specifies the arrangement of the \code{columns} values
and any string literals. The pattern uses numbers (within \code{{ }}) that
correspond to the indices of columns provided in \code{columns}. If two columns
are provided in \code{columns} and we would like to combine the cell data onto
the first column, \code{"{1} {2}"} could be used. If a pattern isn't provided
then a space-separated pattern that includes all \code{columns} will be
generated automatically. Further details are provided in the \emph{How the
\code{pattern} works} section.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
This function takes input from two or more columns and allows the contents to
be merged into a single column by using a pattern that specifies the
arrangement. We can specify which columns to merge together in the \code{columns}
argument. The string-combining pattern is to be provided in the \code{pattern}
argument. The first column in the \code{columns} series operates as the target
column (i.e., the column that will undergo mutation) whereas all following
\code{columns} will be untouched. There is the option to hide the non-target
columns (i.e., second and subsequent columns given in \code{columns}). The
formatting of values in different columns will be preserved upon merging.
}
\section{How the \code{pattern} works}{


There are two types of templating for the \code{pattern} string:
\enumerate{
\item \code{{ }} for arranging single column values in a row-wise fashion
\item \verb{<< >>} to surround spans of text that will be removed if any of the
contained \code{{ }} yields a missing value
}

Integer values are placed in \code{{ }} and those values correspond to the columns
involved in the merge, in the order they are provided in the \code{columns}
argument. So the pattern \code{"{1} ({2}-{3})"} corresponds to the target column
value listed first in \code{columns} and the second and third columns cited
(formatted as a range in parentheses). With hypothetical values, this might
result as the merged string \code{"38.2 (3-8)"}.

Because some values involved in merging may be missing, it is likely that
something like \code{"38.2 (3-NA)"} would be undesirable. For such cases, placing
sections of text in \verb{<< >>} results in the entire span being eliminated if
there were to be an \code{NA} value (arising from \code{{ }} values). We could instead
opt for a pattern like \code{"{1}<< ({2}-{3})>>"}, which results in \code{"38.2"} if
either columns \code{{2}} or \code{{3}} have an \code{NA} value. We can even use a more
complex nesting pattern like \code{"{1}<< ({2}-<<{3}>>)>>"} to retain a lower
limit in parentheses (where \code{{3}} is \code{NA}) but remove the range altogether
if \code{{2}} is \code{NA}.

One more thing to note here is that if \code{\link[=sub_missing]{sub_missing()}} is used on values in
a column, those specific values affected won't be considered truly missing by
\code{cols_merge()} (since it's been handled with substitute text). So, the
complex pattern \code{"{1}<< ({2}-<<{3}>>)>>"} might result in something like
\code{"38.2 (3-limit)"} if \code{sub_missing(..., missing_text = "limit")} were used
on the third column supplied in \code{columns}.
}

\section{Comparison with other column-merging functions}{


There are three other column-merging functions that offer specialized
behavior that is optimized for common table tasks: \code{\link[=cols_merge_range]{cols_merge_range()}},
\code{\link[=cols_merge_uncert]{cols_merge_uncert()}}, and \code{\link[=cols_merge_n_pct]{cols_merge_n_pct()}}. These functions operate
similarly, where the non-target columns can be optionally hidden from the
output table through the \code{autohide} option.
}

\section{Examples}{


Use a subset of the \code{\link{sp500}} dataset to create a \strong{gt} table. Use the
\code{cols_merge()} function to merge the \code{open} & \code{close} columns together, and,
the \code{low} & \code{high} columns (putting an em dash between both). Relabel the
columns with \code{\link[=cols_label]{cols_label()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sp500 |>
  dplyr::slice(50:55) |>
  dplyr::select(-volume, -adj_close) |>
  gt() |>
  cols_merge(
    columns = c(open, close),
    pattern = "\{1\}&mdash;\{2\}"
  ) |>
  cols_merge(
    columns = c(low, high),
    pattern = "\{1\}&mdash;\{2\}"
  ) |>
  cols_label(
    open = "open/close",
    low = "low/high"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_merge_1.png" alt="This image of a table was generated from the first code example in the `cols_merge()` help file." style="width:100\%;">
}}

Use a portion of \code{\link{gtcars}} to create a \strong{gt} table. Use the \code{cols_merge()}
function to merge the \code{trq} & \code{trq_rpm} columns together, and, the \code{mpg_c} &
\code{mpg_h} columns. Given the presence of \code{NA} values, we can use patterns that
drop parts of the output text whenever missing values are encountered.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::filter(year == 2017) |>
  dplyr::select(mfr, model, starts_with(c("trq", "mpg"))) |>
  gt() |>
  fmt_integer(columns = trq_rpm) |>
  cols_merge(
    columns = starts_with("trq"),
    pattern = "\{1\}<< (\{2\} rpm)>>"
  ) |>
  cols_merge(
    columns = starts_with("mpg"),
    pattern = "<<\{1\} city<</\{2\} hwy>>>>"
  ) |>
  cols_label(
    mfr = "Manufacturer",
    model = "Car Model",
    trq = "Torque",
    mpg_c = "MPG"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_merge_2.png" alt="This image of a table was generated from the second code example in the `cols_merge()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

5-14
}

\section{Function Introduced}{

\code{v0.2.0.5} (March 31, 2020)
}

\seealso{
Other column modification functions: 
\code{\link{cols_add}()},
\code{\link{cols_align}()},
\code{\link{cols_align_decimal}()},
\code{\link{cols_hide}()},
\code{\link{cols_label}()},
\code{\link{cols_label_with}()},
\code{\link{cols_merge_n_pct}()},
\code{\link{cols_merge_range}()},
\code{\link{cols_merge_uncert}()},
\code{\link{cols_move}()},
\code{\link{cols_move_to_end}()},
\code{\link{cols_move_to_start}()},
\code{\link{cols_nanoplot}()},
\code{\link{cols_unhide}()},
\code{\link{cols_units}()},
\code{\link{cols_width}()}
}
\concept{column modification functions}
