% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rows_add.R
\name{rows_add}
\alias{rows_add}
\title{Add one or more rows to a \strong{gt} table}
\usage{
rows_add(
  .data,
  ...,
  .list = list2(...),
  .before = NULL,
  .after = NULL,
  .n_empty = NULL
)
}
\arguments{
\item{.data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{...}{\emph{Cell data assignments}

\verb{<multiple expressions>} // (or, use \code{.list})

Expressions for the assignment of cell values to the new rows by column
name in \code{.data}. Name-value pairs, in the form of
\verb{<column> = <value vector>} will work, so long as the \verb{<column>} value
exists in the table. Two-sided formulas with column-resolving expressions
(e.g, \verb{<expr> ~ <value vector>}) can also be used, where the left-hand side
corresponds to selections of columns. Column names should be enclosed in
\code{\link[=c]{c()}} and select helpers like \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}},
\code{\link[=matches]{matches()}}, \code{\link[=one_of]{one_of()}}, and \code{\link[=everything]{everything()}} can be used in the LHS. The
length of the longest vector in \verb{<value vector>} determines how many new
rows will be added. Single values in \verb{<value vector>} will be repeated down
in cases where there are multiple rows to be added.}

\item{.list}{\emph{Alternative to \code{...}}

\verb{<list of multiple expressions>} // (or, use \code{...})

Allows for the use of a list as an input alternative to \code{...}.}

\item{.before, .after}{\emph{Row used as anchor}

\verb{<row-targeting expression>} // \emph{default:} \code{NULL} (\code{optional})

A single row-resolving expression or row index an be given to either
\code{.before} or \code{.after}. The row specifies where the new rows should be
positioned among the existing rows in the input data table. While select
helper functions such as \code{\link[=starts_with]{starts_with()}} and \code{\link[=ends_with]{ends_with()}} can be used for
row targeting, it's recommended that a single row name or index be used.
This is to ensure that exactly one row is provided to either of these
arguments (otherwise, the function will be stopped). If nothing is provided
for either argument then any new rows will be placed at the bottom of the
table.}

\item{.n_empty}{\emph{Number of empty rows to add}

\code{scalar<numeric|integer>(val>=0)} // \emph{default:} \code{NULL} (\code{optional})

An option to add empty rows in lieu of rows containing data that would
otherwise be supplied to \code{...} or \code{.list}. If the option is taken, provide
an integer value here.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
It's possible to add new rows to your table with \code{rows_add()} by supplying
the new row data through name-value pairs or two-sided formula expressions.
The new rows are added to the bottom of the table by default but can be added
internally though by using either the \code{.before} or \code{.after} arguments. If
entirely empty rows need to be added, the \code{.n_empty} option provides a means
to specify the number of blank (i.e., all \code{NA}) rows to be inserted into the
table.
}
\section{Targeting the row for insertion with \code{.before} or \code{.after}}{


The targeting of a row for insertion is done through the \code{.before} or
\code{.after} arguments (only one of these options should be be used). This can be
done in a variety of ways. If a stub is present, then we potentially have row
identifiers. This is the ideal method to use for establishing a row target.
We can use \strong{tidyselect}-style expressions to target a row. It's also
possible to use row indices (e.g., \code{c(3, 5, 6)}) though these index values
must correspond to the row numbers of the input data (the indices won't
necessarily match those of rearranged rows if row groups are present). One
more type of expression is possible, an expression that takes column values
(can involve any of the available columns in the table) and returns a logical
vector.
}

\section{Examples}{


Let's make a simple \strong{gt} table with the \code{\link{exibble}} dataset, using the
\code{row} column for labels in the stub. We'll add a single row to the bottom of
the table with \code{rows_add()}. With name-value pairs, it's possible to add
values for the new body cells that correspond to columns available in the
table. For any columns that are missed, the related body cells will receive
\code{NA} values.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  gt(rowname_col = "row") |>
  rows_add(
    row = "row_9",
    num = 9.999E7,
    char = "ilama",
    fctr = "nine",
    group = "grp_b"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_rows_add_1.png" alt="This image of a table was generated from the first code example in the `rows_add()` help file." style="width:100\%;">
}}

If you wanted to place a row somewhere in the middle of the table, we can use
either of the \code{.before} or \code{.after} arguments in \code{rows_add()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  gt(rowname_col = "row") |>
  rows_add(
    row = "row_4.5",
    num = 9.923E3,
    char = "elderberry",
    fctr = "eighty",
    group = "grp_a",
    .after = "row_4"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_rows_add_2.png" alt="This image of a table was generated from the second code example in the `rows_add()` help file." style="width:100\%;">
}}

Putting a row at the beginning requires the use of the \code{.before} argument. We
can use an index value for the row as in \code{.before = 1} for maximum easiness:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  gt(rowname_col = "row") |>
  rows_add(
    row = "row_0",
    num = 0,
    char = "apple",
    fctr = "zero",
    group = "grp_a",
    .before = 1
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_rows_add_3.png" alt="This image of a table was generated from the third code example in the `rows_add()` help file." style="width:100\%;">
}}

Again with \code{\link{exibble}}, we can create an example where we insert 'spacer'
rows. These are rows without any content and merely serve to add extra
vertical space to the table in specific locations. In this case, we'll have a
stub with row names and row groups (set up in the \code{\link[=gt]{gt()}} call). The two rows
being added will occupy the bottom row of each group. The only data defined
for the two rows involves values for the \code{row} and \code{group} columns. It's
important that the data for \code{group} uses the group names already present in
the data (\code{"grp_a"} and \code{"grp_b"}). The corresponding values for \code{row} will
be \code{"row_a_end"} and \code{"row_b_end"}, these will be used later expressions for
targeting the rows. Here's the code needed to generate spacer rows at the end
of each row group:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  gt(rowname_col = "row", groupname_col = "group") |>
  rows_add(
    row = c("row_a_end", "row_b_end"),
    group = c("grp_a", "grp_b")
  ) |>
  tab_style(
    style = cell_borders(sides = "top", style = "hidden"),
    locations = list(
      cells_body(rows = ends_with("end")),
      cells_stub(rows = ends_with("end"))
    )
  ) |>
  sub_missing(missing_text = "") |>
  text_case_when(
    grepl("end", x) ~ "",
    .locations = cells_stub()
  ) |>
  opt_vertical_padding(scale = 0.5)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_rows_add_4.png" alt="This image of a table was generated from the fourth code example in the `rows_add()` help file." style="width:100\%;">
}}

All missing values were substituted with an empty string (\code{""}), and that was
done by using the \code{\link[=sub_missing]{sub_missing()}} function. We removed the top border of the
new rows with a call to \code{\link[=tab_style]{tab_style()}}, targeting those rows where the row
labels end with \code{"end"}. Finally, we get rid of the row labels with the use
of the \code{\link[=text_case_when]{text_case_when()}} function, using a similar strategy of targeting the
name of the row label.

Another application is starting from nothing (really just the definition of
columns) and building up a table using several invocations of \code{rows_add()}.
This might be useful in interactive or programmatic applications. Here's an
example where two columns are defined with \strong{dplyr}'s \code{tibble()} function
(and no rows are present initially); with two calls of \code{rows_add()}, two
separate rows are added.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  time = lubridate::POSIXct(),
  event = character(0)
) |>
  gt() |>
  rows_add(
    time = lubridate::ymd_hms("2022-01-23 12:36:10"),
    event = "start"
  ) |>
  rows_add(
    time = lubridate::ymd_hms("2022-01-23 13:41:26"),
    event = "completed"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_rows_add_5.png" alt="This image of a table was generated from the fifth code example in the `rows_add()` help file." style="width:100\%;">
}}

It's possible to use formula syntax in \code{rows_add()} to perform column
resolution along with attaching values for new rows. If we wanted to use an
equivalent value for multiple cells in a new row, a valid input would be in
the form of \verb{<expr> ~ <value vector>}. In the following example, we create a
simple table with six columns (the rendered \strong{gt} table displays four
columns and a stub column since the \code{group} column is used for row group
labels). Let's add a single row where some of the cell values added
correspond to columns are resolved on the LHS of the formula expressions:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  group = c("Group A", "Group B", "Group B"),
  id = c("WG-025360", "WG-025361", "WG-025362"),
  a = c(1, 6, 2),
  b = c(2, 6, 2),
  quantity_x = c(83.58, 282.71, 92.20),
  quantity_y = c(36.82, 282.71, 87.34)
) |>
  gt(rowname_col = "id", groupname_col = "group") |>
  rows_add(
    starts_with("gr") ~ "Group A",
    id = "WG-025363",
    c(a, b) ~ 5,
    starts_with("quantity") ~ 72.63
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_rows_add_6.png" alt="This image of a table was generated from the sixth code example in the `rows_add()` help file." style="width:100\%;">
}}

We can see that using \code{starts_with("gr")} yields a successful match to the
\code{group} column with the tangible result being an addition of a row to the
\code{"Group A"} group (the added row is the second one in the rendered \strong{gt}
table). Through the use of \code{c(a, b)}, it was possible to add the value \code{5} to
both the \code{a} and \code{b} columns. A similar approach was taken with adding the
\code{72.63} value to the \code{quantity_x} and \code{quantity_y} columns though we used the
\code{starts_with("quantity")} expression to get \strong{gt} to resolve those two
columns.

You can start with an empty table (i.e., no columns and no rows) and add one
or more rows to it. In the completely empty table scenario, where you would
use something like \code{dplyr::tibble()} or \code{data.frame()} with \code{\link[=gt]{gt()}}, the first
\code{rows_add()} could have rows of arbitrary width. In other words, you get to
generate table columns (and rows) with a completely empty table via
\code{rows_add()}. Here's an example of that:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gt(dplyr::tibble()) |>
  rows_add(
    msrp = c(29.95, 49.95, 79.95),
    item = c("Klax", "Rez", "Ys"),
    type = c("A", "B", "X")
  ) |>
  rows_add(
    msrp = 14.95,
    item = "D",
    type = "Z"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_rows_add_7.png" alt="This image of a table was generated from the seventh code example in the `rows_add()` help file." style="width:100\%;">
}}

In the above, three columns and three rows were generated. The second usage
of \code{rows_add()} had to use of a subset of those columns (all three were used
to create a complete, new row).

We can also start with a virtually empty table: one that has columns but no
actual rows. With this type of multi-column, zero-row table, one needs to use
a subset of the columns when generating new rows through \code{rows_add()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  msrp = numeric(0),
  item = character(0),
  type = character(0)
) |>
  gt() |>
  rows_add(
    msrp = c(29.95, 49.95, 79.95, 14.95),
    item = c("Klax", "Rez", "Ys", "D"),
    type = c("A", "B", "X", "Z")
  ) |>
  cols_add(
    genre = c("puzzle", "action", "RPG", "adventure")
  ) |>
  fmt_currency() |>
  cols_move_to_end(columns = msrp)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_rows_add_8.png" alt="This image of a table was generated from the eighth code example in the `rows_add()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

6-4
}

\section{Function Introduced}{

\code{v0.10.0} (October 7, 2023)
}

\seealso{
Other row addition/modification functions: 
\code{\link{grand_summary_rows}()},
\code{\link{row_group_order}()},
\code{\link{summary_rows}()}
}
\concept{row addition/modification functions}
