
# # Load points data
# dir.src <- "U:/Canada_Tree-Ring_Data/Shared/LFC process/analysis/detrending species Apr2024/det_Lbai by spc"
#
# f.det.lst <- list.files(path = dir.src, full.names = TRUE,  pattern = "_detrend_spc.csv$")
#
# fn.lst = f.det.lst[1:2]
# # reading all data involved



#' Read and Process Tree-Ring Series Data
#'
#' Reads a list of CSV/TSV files containing tree-ring series data, filters by year range,
#' limits values to a specified range, reshapes the data into wide format, and combines
#' all series into a single data.table.
#'
#' @param fn.lst A character vector of file paths to be read. Each file should be readable by `fread()`
#'   and contain at least the specified metadata and variable columns.
#' @param year.span A numeric vector of length two specifying the start and end years (inclusive).
#'   Default is \code{c(1801, 2017)}.
#' @param var.name A character string specifying the name of the variable to extract and reshape.
#'   Default is \code{"ratio.Bai.bv.o_p"}.
#' @param val.lim A numeric vector of length two specifying the lower and upper bounds for the variable values.
#'   Values outside this range will be clipped. Default is \code{c(-100, 100)}.
#' @param cols.meta A character vector specifying the required metadata columns in the input files.
#'   These columns are preserved in the output. Default is \code{c("uid_tree", "uid_site", "longitude", "latitude", "species", "year")}.
#'
#' @return A list with of class \code{"cfs_mapping"}, the first is a \code{data.table} , reshaped to wide format with one row per tree
#'   and year values in columns prefixed with \code{"X"}. Includes all metadata columns (except "year") and the
#'   selected variable across years. the second is a string for var.name
#'
#' @keywords internal
#' @noRd
read_series <- function(fn.lst, year.span = c(1801, 2017),
                        var.name = "ratio.Bai.bv.o_p", val.lim = c(-100, 100),
                        cols.meta = c("uid_tree", "uid_site", "longitude", "latitude", "species", "year")) {

  dt.list <- lapply(fn.lst, function(fname) {
    dt <- tryCatch(fread(fname), error = function(e) NULL)

    if (is.null(dt) || !all(c(cols.meta, var.name) %in% names(dt))) return(NULL)

    dt <- dt[, c(cols.meta, var.name), with = FALSE]

    # Filter year
    dt <- dt[year >= year.span[1] & year <= year.span[2]]

    # Limit values
    dt[, (var.name) := pmin(pmax(get(var.name), val.lim[1]), val.lim[2])]

    # Reshape to wide
    fml <- stats::as.formula(paste(paste(cols.meta[cols.meta != "year"], collapse = " + "), "~ year"))
    dt.wide <- dcast(dt, formula = fml, value.var = var.name)

    return(as.data.table(dt.wide))
  })

  # Remove NULLs from list (in case some files were skipped)
  dt.list <- dt.list[lengths(dt.list) > 0]

  # Now bind all together
  dt.w <- rbindlist(dt.list, use.names = TRUE, fill = TRUE)

  # Rename and reorder columns
  year_cols <- setdiff(names(dt.w), cols.meta)
  setcolorder(dt.w, c(cols.meta[cols.meta != "year"], sort(year_cols)))
  setnames(dt.w, old = sort(year_cols), new = paste0("X", sort(year_cols)))

  result <- list(dt.w = dt.w, var.name = var.name)
  class(result) <- c("cfs_mapping", class(result))
  return(result)
}


#' Interpolate and Map Tree-Ring Data
#'
#' This function performs inverse distance weighting (IDW) interpolation
#' of tree-ring data across a spatial grid, either for all species combined
#' or by individual species. It generates yearly interpolated raster maps
#' over a user-defined extent or the extent of the input data.
#'
#' @param data input in wide format.
#' @param year.span Numeric vector of length 2 giving the range of years to include.
#' @param extent.lim Optional numeric vector defining the spatial extent
#'   (\code{c(xmin, xmax, ymin, ymax)}). If \code{NULL}, the extent is
#'   determined from the input data.
#' @param grid.step Numeric value specifying the grid spacing in degrees.
#' @param by.spc Logical; if \code{TRUE}, maps are generated by species;
#'   if \code{FALSE}, all species are combined.
#'
#' @return An object of class \code{cfs_map}, a list of interpolated raster layers
#'   by species and year.
#'
#' @export
#' @examples
#'
#' \donttest{
#' # loading processed data
#' dt.samples_trt <- readRDS(system.file("extdata", "dt.samples_trt.rds", package = "growthTrendR"))
#' cols.meta = c("uid_tree", "uid_site", "longitude", "latitude", "species")

#' dt.mapping <- dt.samples_trt$tr_all_wide[
#'   , c(..cols.meta, as.character(1991:1995)), with = FALSE]
#' results_mapping <- CFS_mapping(dt.mapping, year.span = c(1991:1993))
#' }
#'


CFS_mapping <- function(data, year.span = c(1801, 2017),
                        extent.lim = NULL, grid.step = 0.1,
                        by.spc = FALSE) {

  #----------------------------
  # Input checks
  #----------------------------


  check_optional_deps()

  if (inherits(data, "cfs_mapping")) data <- data$dt.w

  # Add species column if needed
  if (by.spc) {
    data[, species.inuse := species]
  } else {
    data$species.inuse <- "all.spp"
  }

  #----------------------------
  # Identify year and non-year columns
  #----------------------------

  # 1. find year columns (either bare years or prefixed with "X")
  year_idx <- grep("^\\d+$", names(data))
  if (length(year_idx) == 0) {
    # try stripping leading "X"
    year_idx <- grep("^\\d+$", str_sub(names(data), 2))
  }

  # 2. select only years within year.span
  yr_vals <- intersect(as.integer(str_remove(names(data)[year_idx], "^X")), year.span[1]:year.span[2])
  year_idx_select <- year_idx[names(data)[year_idx] %in% paste0("X", yr_vals) | names(data)[year_idx] %in% as.character(yr_vals)]

  # 3. non-year columns
  non_year_idx <- setdiff(seq_along(data), year_idx)

  # 4. subset
  # data.yr <- data[, sort(c(non_year_idx, year_idx)), with = FALSE]
  data.yr <- data[, c(non_year_idx, year_idx_select), with = FALSE]
  year_idx.new <- (length(non_year_idx) + 1):ncol(data.yr)
  if (length(grep("^\\d+$", names(data))) > 0) names(data.yr)[year_idx.new] <- paste0("X", names(data.yr)[year_idx.new])

  #----------------------------
  # Create grid (Canada extent or user-defined)
  #----------------------------
  canada_ext <- if (is.null(extent.lim)) {
    raster::extent(
      min(data$longitude), max(data$longitude),
      min(data$latitude), max(data$latitude)
    )
  } else {
    raster::extent(extent.lim)
  }

  grid <- expand.grid(
    longitude = seq(from = canada_ext@xmin, to = canada_ext@xmax, by = grid.step),
    latitude  = seq(from = canada_ext@ymin, to = canada_ext@ymax, by = grid.step)
  )
  grid$grid_cell <- seq_len(nrow(grid))
  sp::coordinates(grid) <- ~longitude + latitude
  sp::gridded(grid) <- TRUE

  #----------------------------
  # Run over all species using map
  #----------------------------
  spc.lst <- sort(unique(data$species.inuse))

  results_list.spc <- purrr::map(spc.lst, function(spc.i) {
    dt.spc.i <- data.yr[species.inuse == spc.i]

    # remove columns with only NA
    setDF(dt.spc.i)
    dt.spc.i <- dt.spc.i[, colSums(!is.na(dt.spc.i)) > 0, drop = FALSE]
    setDT(dt.spc.i)
    cols.select <- names(data.yr)[year_idx.new]

    # Run over years using map
      results_list.ispc <- purrr::map(cols.select, function(col) {
      yr <- as.numeric(str_sub(col, 2))

      current_data <- dt.spc.i[!is.na(dt.spc.i[[col]]),
                               c("longitude", "latitude", col), with = FALSE]

      if (nrow(current_data) == 0) return(NULL)  # skip empty years

      sp::coordinates(current_data) <- ~longitude + latitude

      idw_model <- gstat::idw(
        as.formula(paste(col, "~1")),
        locations = current_data,
        newdata   = grid,
        idp = 2,
        nmax = 100
      )

      # df.idw <- cbind( year = yr, as.data.table(idw_model)[, -"var1.var"])
      # df.idw <- data.table(
      #   year = yr,
      #   longitude = idw_model@coords[, 1],
      #   latitude = idw_model@coords[, 2],
      #   value = idw_model$var1.pred
      # )

      df.idw <- data.table(
        year = yr,
        longitude = idw_model@coords[, 1],
        latitude = idw_model@coords[, 2],
        value = idw_model$var1.pred
      )
      df.idw
      }
      )

    # names(results_list.ispc) <- yr_vals
    # results_list.ispc

      dfs <- data.table::rbindlist(results_list.ispc, fill = TRUE)

      # Assuming your long data has: longitude, latitude, year, value
      dfs_wide <- data.table::dcast(
        dfs,
        longitude + latitude ~ year,  # id vars ~ variable to spread
        value.var = "value"            # column containing the values
      )

  })

  names(results_list.spc) <- spc.lst
  class(results_list.spc) <- c("cfs_map", class(results_list.spc))
  return(results_list.spc)
}







#' Download and Read Boreal Mask Shapefile
#'
#' Downloads the North American Boreal forest shapefile from Figshare and
#' reads it as an sf object. This is an internal function.
#'
#' @param url Character string. URL to download the shapefile zip from.
#'   Defaults to the Figshare repository URL.
#' @return An sf object containing the boreal mask polygon(s)
#' @noRd
#' @keywords internal
# get_boreal_mask <- function(url) {
#
#   # Download zip file
#   temp_zip <- tempfile(fileext = ".zip")
#   temp_dir <- tempfile()
#
#   res <- httr::GET(
#     url,
#     httr::write_disk(temp_zip, overwrite = TRUE),
#     httr::config(followlocation = TRUE),
#     httr::add_headers("User-Agent" = "Mozilla/5.0"),
#     httr::timeout(300)
#   )
#
#   if (res$status_code != 200) {
#     stop("Download failed with status code: ", res$status_code)
#   }
#
#   # Extract and read shapefile
#   utils::unzip(temp_zip, exdir = temp_dir)
#   shp_file <- list.files(temp_dir, pattern = "\\.shp$",
#                          recursive = TRUE, full.names = TRUE)[1]
#
#   if (is.na(shp_file)) {
#     stop("No shapefile found in downloaded archive")
#   }
#
#   boreal_mask <- sf::st_read(shp_file, quiet = TRUE)
#
#   # Cleanup
#   unlink(temp_zip)
#   unlink(temp_dir, recursive = TRUE)
#
#   return(boreal_mask)
# }

get_boreal_mask <- function(url) {

  temp_zip <- tempfile(fileext = ".zip")
  temp_dir <- tempfile()

  on.exit({
    if (file.exists(temp_zip)) unlink(temp_zip)
    if (dir.exists(temp_dir)) unlink(temp_dir, recursive = TRUE)
  }, add = TRUE)

  res <- tryCatch(
    httr::GET(
      url,
      httr::write_disk(temp_zip, overwrite = TRUE),
      httr::config(followlocation = TRUE),
      httr::add_headers("User-Agent" = "Mozilla/5.0"),
      httr::timeout(300)
    ),
    error = function(e) {
      message("Failed to download boreal mask: ", e$message)
      return(NULL)
    }
  )

  if (is.null(res)) return(NULL)

  if (httr::status_code(res) != 200) {
    message("Download failed (HTTP ", httr::status_code(res), ")")
    return(NULL)
  }

  # unzip safely
  unzip_ok <- tryCatch(
    {
      utils::unzip(temp_zip, exdir = temp_dir)
      TRUE
    },
    error = function(e) {
      message("Failed to unzip boreal mask: ", e$message)
      FALSE
    }
  )

  if (!unzip_ok) return(NULL)

  shp_file <- list.files(
    temp_dir,
    pattern = "\\.shp$",
    recursive = TRUE,
    full.names = TRUE
  )[1]

  if (is.na(shp_file)) {
    message("No shapefile found in downloaded archive")
    return(NULL)
  }

  boreal_mask <- tryCatch(
    sf::st_read(shp_file, quiet = TRUE),
    error = function(e) {
      message("Failed to read shapefile: ", e$message)
      NULL
    }
  )
    # Cleanup
    unlink(temp_zip)
    unlink(temp_dir, recursive = TRUE)
  boreal_mask
}


#' Plot spatially interpolated tree-ring growth maps
#'
#' Visualizes spatial interpolation results produced by
#' \code{\link{CFS_mapping}} and optionally exports raster files,
#' static maps, and animations.
#'
#' @param mapping_results
#' A list returned by \code{\link{CFS_mapping}} containing
#' spatially interpolated rasters by species and year.
#' @param crs.src Coordinate Reference System of the input data,
#'   specified as a string in the format 'EPSG:<ID>' (for example, 'EPSG:4326').
#'   The function will stop with an error if `crs.src` is not provided in this format.
#' @param parms.out
#' Character vector indicating output formats to generate.
#' Supported values are \code{"csv"}, \code{"tif"},
#' \code{"png"}, and \code{"gif"}.
#'
#' @param dir.shp Character or NULL. Path to the folder containing shapefiles for cropping data to the Canadian boreal regions. Only used for specific research purposes; if NULL (default), no cropping is applied and all data are included.

#' @param dir.out
#' Output directory used to save generated files.
#' Required when \code{parms.out} is not empty.
#'
#' @param animation_fps
#' Frames per second used when creating GIF animations.
#'
#' @param ...
#' Additional arguments passed to
#' \code{plot_tree_ring_map}, such as \code{png.text}.
#'
#' @details
#' This function assumes that spatial interpolation has already
#' been performed using \code{\link{CFS_mapping}}. The input
#' object is iterated by species and year to generate maps and
#' optional exports.
#'
#' When \code{"gif"} is requested in \code{parms.out}, yearly
#' PNG images are combined into animated GIFs.
#'
#' @return A magick-image object representing an animated GIF composed of the generated frames.

#'
#' @seealso
#' \code{\link{CFS_mapping}}
#'
#' @examples
#' \donttest{
#' # Load processed demo data
#' dt.samples_trt <- readRDS(
#'   system.file("extdata", "dt.samples_trt.rds",
#'               package = "growthTrendR")
#' )
#' # prepare data for IDW model
#' cols.meta = c("uid_tree", "uid_site", "longitude", "latitude", "species")
#' dt.mapping <- dt.samples_trt$tr_all_wide[
#'   , c(..cols.meta, as.character(1991:1995)), with = FALSE]
#'
#' # Run spatial interpolation
#' mapping_results <- CFS_mapping(
#'   dt.mapping,
#'   year.span = c(1991, 1993)
#' )
#'
#' # generate png plots
#' img_ani <- plot_mapping(
#' mapping_results = mapping_results,
#' parms.out = NULL,
#' dir.shp = NULL,
#' dir.out = NULL,
#' png.text = list(
#'   text_top  = "Ring width measurement - ",
#'   text_bott = "Source: demo-samples",
#'   text_side = "ring width (mm)"
#' )
#' )
#' }
#'
#' @export

plot_mapping <- function(mapping_results, crs.src = "EPSG:4326",
                         parms.out = c("csv", "tif", "png", "gif"),
                         dir.shp = NULL,
                         dir.out = NULL,
                         animation_fps = 1.0,
                         ...) {

  check_optional_deps()
  if (!is.character(crs.src) ||
      length(crs.src) != 1L ||
      !grepl("^EPSG:\\d+$", crs.src)) {
    stop(
      "crs.src must be a character string in the form 'EPSG:<ID>' (e.g. 'EPSG:4326')",
      call. = FALSE
    )
  }
  # --- Input validation ---
  if (!is.list(mapping_results)) stop("mapping_results must be a list from CFS_mapping()")
  allowed_out <- c("csv","tif","png","gif")
  if (length(setdiff(parms.out, allowed_out)) > 0)
    stop("parms.out only supports: ", paste(allowed_out, collapse = ", "))
  if (!is.null(parms.out) && is.null(dir.out))
    stop("Please specify an output directory (dir.out)")

  if (!is.null(dir.out)) {
    if (!dir.exists(dir.out))
      dir.create(dir.out, recursive = TRUE)
  }
  if (!is.null(dir.shp)){


    files_needed <- c("Canada120s.tif", "NABoreal_expanded.shp", "province.shp")
    paths <- file.path(dir.shp, files_needed)

    missing <- files_needed[!file.exists(paths)]

    if (length(missing) > 0) {
      stop(
        "Missing required file(s): ",
        paste(missing, collapse = ", "),
        call. = FALSE
      )
    }


    # --- Load reference layers once ---
    elevation_file <- file.path(dir.shp, "Canada120s.tif")
    elevation <- terra::rast(elevation_file)

    # using the figshare that martin provided to avoid big extdata
    # the reference is here https://doi.org/10.6084/m9.figshare.30005473.v1 (Martin 2025-10-01) then redirect to
    # https://figshare.com/articles/dataset/Spatially_detailed_tree-ring_analysis_throughout_Canada/30005473
    # the true url is by right clicking download button to copy link address for the file
    # url <- "https://figshare.com/ndownloader/files/57487996"

    # boreal_mask <- get_boreal_mask("https://figshare.com/ndownloader/files/57487996")
    # canada_mask <- sf::st_read(system.file("extdata/Mapping/province.shp", package = "growthTrendR"))
    # Transform boreal and Canada masks to raster CRS
    boreal_mask <- sf::st_read(file.path(dir.shp, "NABoreal_expanded.shp"))
    canada_mask <- sf::st_read(file.path(dir.shp, "province.shp"))

    boreal_mask_proj <- terra::vect(sf::st_transform(boreal_mask, crs.src))
    canada_mask_proj <- terra::vect(sf::st_transform(canada_mask, crs.src))
  }else elevation <- NULL
  # --- Loop over species & years ---
  spc_lst <- names(mapping_results)
  out_list <- lapply(spc_lst, function(spc.i) {
    spc_data <- mapping_results[[spc.i]]

    rasters <- terra::rast(
      spc_data,
      type = "xyz",
      crs = crs.src
    )
    df.value.lim <- terra::global(
      rasters,
      fun = range,
      na.rm = TRUE
    )

    value.lim <- c(floor(min(df.value.lim[,1]) * 10) / 10, ceiling(max(df.value.lim[,2]) * 10) / 10)


    # --- Mask rasters using terra ---
    if (!is.null(dir.shp)){
      r_boreal <- terra::mask(rasters, boreal_mask_proj)
      rasters_masked <- terra::mask(r_boreal, canada_mask_proj)
    }else rasters_masked<- rasters
    # Prepare directories once per species
    if (!is.null(dir.out) & !is.null(parms.out)){
      dir_csv <- if ("csv" %in% parms.out) file.path(dir.out, "csv") else NULL
      dir_tif <- if ("tif" %in% parms.out) file.path(dir.out, "tif") else NULL
      dir_png <- if ("png" %in% parms.out) file.path(dir.out, "png") else NULL
      dir_gif <- if ("gif" %in% parms.out) file.path(dir.out, "gif") else NULL
      for (d in c(dir_csv, dir_tif, dir_png, dir_gif)) if (!is.null(d) && !dir.exists(d)) dir.create(d, recursive = TRUE)
    }else{
      dir_csv <- dir_tif <- dir_png <- dir_gif <- NULL
    }


    gif_frames <- list()
    for (yr in names(rasters)) {
      r_masked <- rasters_masked[[yr]]


      # Apply masks sequentially (replaced by mask for whole rasters)
      # r_boreal <- terra::mask(r, boreal_mask_proj)
      # r_masked <- terra::mask(r_boreal, canada_mask_proj)


      # Save TIF
      if (!is.null(dir_tif)) {
        terra::writeRaster(
          r_masked,
          filename = file.path(dir_tif, paste0(spc.i, "_tree_rings_", yr, ".tif")),
          overwrite = TRUE
        )
      }



      # 1️⃣ Generate frame in memory
      img <- magick::image_graph(width = 2500, height = 1300, res = 300)

      plot_tree_ring_map(
        r_masked,
        elevation_raster = NULL,
        yr,
        out.png = NULL,  # always NULL for in-memory plotting
        value.lim = value.lim,
        ...
      )

      dev.off()  # closes the in-memory device and returns the image
      gif_frames[[as.character(yr)]] <- img
      # Save PNG

      if (!is.null(dir_png)) {
        png_file <- file.path(dir_png, paste0(spc.i, "_tree_rings_", yr, ".png"))
        magick::image_write(img, path = png_file, format = "png")

      }
    }

    # Save CSV
    if (!is.null(dir_csv)) {

      write.csv(spc_data, file = file.path(dir_csv, paste0(spc.i, "_tree_rings.csv")), row.names = FALSE, na = "" )
    }
    # Save GIF

    # combine frames and animate
    gif_anim <- magick::image_animate(magick::image_join(gif_frames), fps = animation_fps)

    # write GIF to disk
    if (!is.null(dir_gif)) {

      # build file path
      gif_file <- file.path(dir_gif, paste0(spc.i, "_tree_rings_animation.gif"))
      magick::image_write(image = gif_anim, path = gif_file, format = "gif")
    }


    return(gif_anim)
  })
  #
  names(out_list) <- spc_lst
  return(out_list)
}



# this is a simple version without the option of dir.shp, equivalent to plot_mapping(... dir.shp = NULL)
#' @keywords internal
#' @noRd
#'

plot_mapping2 <- function(mapping_results, crs.src = "EPSG:4326",
                          parms.out = c("csv", "tif", "png", "gif"),
                          dir.out = NULL,
                          animation_fps = 1.0,
                          ...) {

  check_optional_deps()
  if (!is.character(crs.src) ||
      length(crs.src) != 1L ||
      !grepl("^EPSG:\\d+$", crs.src)) {
    stop(
      "crs.src must be a character string in the form 'EPSG:<ID>' (e.g. 'EPSG:4326')",
      call. = FALSE
    )
  }

  # oldpar <- par(no.readonly = TRUE) # code line i
  # on.exit({
  #   if (grDevices::dev.cur() > 1) par(oldpar)
  # }, add = TRUE)

  # --- Input validation ---
  if (!is.list(mapping_results)) stop("mapping_results must be a list from CFS_mapping()")
  allowed_out <- c("csv","tif","png","gif")
  if (length(setdiff(parms.out, allowed_out)) > 0)
    stop("parms.out only supports: ", paste(allowed_out, collapse = ", "))
  if (!is.null(parms.out) && is.null(dir.out))
    stop("Please specify an output directory (dir.out)")

  if (!is.null(dir.out)) {
    if (!dir.exists(dir.out))
      dir.create(dir.out, recursive = TRUE)
  }

  # --- Loop over species & years ---
  spc_lst <- names(mapping_results)
  out_list <- lapply(spc_lst, function(spc.i) {
    spc_data <- mapping_results[[spc.i]]

    rasters <- terra::rast(
      spc_data,
      type = "xyz",
      crs = crs.src
    )
    df.value.lim <- terra::global(
      rasters,
      fun = range,
      na.rm = TRUE
    )

    value.lim <- c(floor(min(df.value.lim[,1]) * 10) / 10, ceiling(max(df.value.lim[,2]) * 10) / 10)


    # --- Mask rasters using terra REMOVED---
    #{     r_boreal <- terra::mask(rasters, boreal_mask_proj)
    # rasters_masked <- terra::mask(r_boreal, canada_mask_proj)}
    # Prepare directories once per species
    if (!is.null(dir.out) & !is.null(parms.out)){
      dir_csv <- if ("csv" %in% parms.out) file.path(dir.out, "csv") else NULL
      dir_tif <- if ("tif" %in% parms.out) file.path(dir.out, "tif") else NULL
      dir_png <- if ("png" %in% parms.out) file.path(dir.out, "png") else NULL
      dir_gif <- if ("gif" %in% parms.out) file.path(dir.out, "gif") else NULL
      for (d in c(dir_csv, dir_tif, dir_png, dir_gif)) if (!is.null(d) && !dir.exists(d)) dir.create(d, recursive = TRUE)
    }else{
      dir_csv <- dir_tif <- dir_png <- dir_gif <- NULL
    }
    png_list <- list()

    gif_frames <- list()
    for (yr in names(rasters)) {
      r_masked <- rasters[[yr]]

      # Save TIF
      if (!is.null(dir_tif)) {
        terra::writeRaster(
          r_masked,
          filename = file.path(dir_tif, paste0(spc.i, "_tree_rings_", yr, ".tif")),
          overwrite = TRUE
        )
      }


      # 1️⃣ Generate frame in memory
      img <- magick::image_graph(width = 2500, height = 1300, res = 300)

      plot_tree_ring_map(
        r_masked,
        elevation_raster = NULL,
        yr,
        out.png = NULL,  # always NULL for in-memory plotting
        value.lim = value.lim,
        ...
      )

      dev.off()  # closes the in-memory device and returns the image
      gif_frames[[as.character(yr)]] <- img
      # Save PNG

      if (!is.null(dir_png)) {
        png_file <- file.path(dir_png, paste0(spc.i, "_tree_rings_", yr, ".png"))
        magick::image_write(img, path = png_file, format = "png")

      }
    }


    # Save CSV
    if (!is.null(dir_csv)) {

      write.csv(spc_data, file = file.path(dir_csv, paste0(spc.i, "_tree_rings.csv")), row.names = FALSE, na = "" )
    }

    # Save gif

    # combine frames and animate
    gif_anim <- magick::image_animate(magick::image_join(gif_frames), fps = animation_fps)

    # write GIF to disk
    if (!is.null(dir_gif)) {

      # build file path
      gif_file <- file.path(dir_gif, paste0(spc.i, "_tree_rings_animation.gif"))

      magick::image_write(image = gif_anim, path = gif_file, format = "gif")
    }


    return(gif_anim)
  })
  #
  names(out_list) <- spc_lst
  return(out_list)
}




#' Plot tree-ring spatial map for a given year
#'
#' Generates a spatial map of interpolated tree-ring growth values for a single
#' year, optionally overlaying elevation data. The function is primarily used
#' internally by \code{plot_mapping()} to produce PNG maps and animations.
#'
#' @param tree_ring_raster A \code{SpatRaster} or \code{RasterLayer} containing
#'   interpolated tree-ring values for one year.
#' @param elevation_raster A \code{SpatRaster} or \code{RasterLayer} representing
#'   elevation data used as a grayscale background.
#' @param year Numeric year associated with the tree-ring raster.
#' @param out.png Optional character string specifying the output PNG file path.
#'   If \code{NULL}, a temporary file is created.
#' @param png.text Optional named list with text labels for the plot. Supported
#'   elements are \code{text_top}, \code{text_bott}, and \code{text_side}.
#' @param value.lim Numeric vector of length 2 giving the minimum and maximum
#'   values for the color scale.
#' @param tick_positions Numeric vector giving tick locations for the legend.
#' @param tick_labels Character or numeric vector of labels corresponding to
#'   \code{tick_positions}.
#' @param nbreaks number of breaks for the legend
#' @param ... Additional graphical parameters passed from
#'   \code{plot_mapping()}.
#'
#' @details
#' The color scale limits are typically rounded using an internal helper
#' (e.g. \code{round_limits()}) to ensure clean legend breaks. Elevation data
#' are reprojected and cropped to match the tree-ring raster before plotting.
#'
#' @return
#' Invisibly returns the file path to the generated PNG if \code{out.png} is
#' \code{NULL}; otherwise returns \code{NULL}.
#'
#' @keywords internal
#' @noRd

plot_tree_ring_map <- function(
    tree_ring_raster,
    elevation_raster = NULL,
    year,
    out.png = NULL,
    png.text = NULL,
    value.lim = NULL,
    tick_positions = NULL,
    tick_labels = NULL,
    nbreaks = 10
) {

  # oldpar <- par(no.readonly = TRUE) # code line i
  # on.exit({
  #   if (grDevices::dev.cur() > 1) par(oldpar)
  # }, add = TRUE)
  # oldpar <- par(no.readonly = TRUE)
  # par(mfrow = c(1, 1))  # example change
  #
  # on.exit({
  #   if (length(grDevices::dev.list()) > 0) {
  #     try(par(oldpar), silent = TRUE)
  #   }
  # }, add = TRUE)


  ## ----------------------------
  ## Defaults
  ## ----------------------------
  default_text <- list(
    text_top  = "my title",
    text_bott = "author/source",
    text_side = "value description"
  )

  png.text <- if (is.list(png.text)) {
    utils::modifyList(default_text, png.text)
  } else {
    default_text
  }

  ## ----------------------------
  ## Ensure SpatRaster
  ## ----------------------------
  # tree_ring_raster <- terra::rast(tree_ring_raster)
  # elevation_raster <- terra::rast(elevation_raster)

  if (!terra::hasValues(tree_ring_raster)) {
    warning("Tree-ring raster has no values for year ", year)
    return(invisible(NULL))
  }

  ## ----------------------------
  ## CRS handling
  ## ----------------------------

  if(!is.null(elevation_raster)){
  if (!terra::same.crs(elevation_raster, tree_ring_raster)) {
    elevation_raster <- terra::project(elevation_raster, tree_ring_raster)
  }

  elevation_raster <- terra::crop(elevation_raster, tree_ring_raster)
  }

  ## ----------------------------
  ## Determine value limits
  ## ----------------------------
  if (is.null(value.lim)) {
    value.lim <- terra::minmax(tree_ring_raster)
    value.lim <- as.numeric(value.lim)
    value.lim[1] <- floor(value.lim[1] * 10) / 10
    value.lim[2] <- ceiling(value.lim[2] * 10) / 10
  }

  ## Clamp raster if limits provided
  r_plot <- terra::clamp(
    tree_ring_raster,
    lower = value.lim[1],
    upper = value.lim[2],
    values = TRUE
  )
  terra::hasValues(r_plot)
  ## ----------------------------
  ## Legend breaks & ticks
  ## ----------------------------
  breaks <- seq(value.lim[1], value.lim[2], length.out = nbreaks + 1)

  if (is.null(tick_positions)) {
    tick_positions <- pretty(value.lim, n = 5)
  }

  if (is.null(tick_labels)) {
    tick_labels <- tick_positions
  }

  cols <- colorRampPalette(c("darkred", "red", "yellow", "green", "blue"))(nbreaks)

  ## ----------------------------
  ## Output file
  ## ----------------------------
  output_file <- if (is.null(out.png)) {
    tempfile(fileext = ".png")
  } else {
    out.png
  }

  ## ----------------------------
  ## Plot
  ## ----------------------------
  # png(output_file, width = 2500, height = 1300, res = 300)
  #
  # on.exit(dev.off(), add = TRUE)

  if (!is.null(out.png)) {
    png(output_file, width = 2500, height = 1300, res = 300)
    on.exit(dev.off(), add = TRUE)
  }

  oldmar <- par("mar")
  on.exit(par(mar = oldmar), add = TRUE)

  par(mar = c(4, 4, 1, 3))

  layout(matrix(c(1, 2), ncol = 2), widths = c(4, 1))
  # par(mar = c(4, 4, 1, 3))

  ## Elevation background

  if(!is.null(elevation_raster)){
  if (terra::hasValues(elevation_raster)) {
    elevation_raster <- terra::ifel(elevation_raster < 1, NA, elevation_raster)
    terra::plot(
      elevation_raster,
      col = gray.colors(100, start = 0.8, end = 0.3),
      legend = FALSE,
      main = paste(png.text$text_top, year),
      axes = TRUE
    )
  } else {
    ## Create empty spatial plot using tree-ring raster
    terra::plot(
      r_plot,
      col = NA,          # no values drawn
      legend = FALSE,
      main = paste(png.text$text_top, year),
      axes = TRUE
    )
  }
    }else{
      ## Create empty spatial plot using tree-ring raster
      terra::plot(
        r_plot,
        col = NA,          # no values drawn
        legend = FALSE,
        main = paste(png.text$text_top, year),
        axes = TRUE
      )
    }

  ## Tree-ring overlay
  terra::plot(
    r_plot,
    col = cols,
    breaks = breaks,
    add = TRUE,
    alpha = 0.6,
    legend = FALSE
  )

  grid()

  mtext(png.text$text_bott, side = 1, line = 3, adj = 0, cex = 0.8)

  ## ----------------------------
  ## Custom legend
  ## ----------------------------
  par(mar = c(5, 0, 3, 3))

  plot(c(0, 1), range(breaks), type = "n", axes = FALSE, xlab = "", ylab = "")

  for (i in seq_len(nbreaks)) {
    rect(0.1, breaks[i], 0.4, breaks[i + 1],
         col = cols[i], border = NA)
  }

  axis(4, at = tick_positions, labels = tick_labels,
       las = 1, cex.axis = 0.8, pos = 0.5)

  mtext(png.text$text_side, side = 4, line = 1.5, cex = 1.2)

  # dev.off()
  # while (grDevices::dev.cur() > 1) grDevices::dev.off()

  if (is.null(out.png)) return(output_file)
}


# Save GIF
#'
#' @keywords internal
#' @noRd

plot_gif <- function(png_list, dir_gif = NULL, animation_fps = 1.0)  {
  # gif_imgs <- magick::image_read(unlist(lapply(png_list, function(x) x$filename)))
  gif_imgs <-magick::image_read(unlist(png_list))
  gif_anim <- magick::image_animate(gif_imgs, fps = animation_fps)
  if (!is.null(dir_gif)) magick::image_write(gif_anim, file.path(dir_gif, paste0(names(png_list), "_tree_rings_animation.gif")
                                                                )) else return(gif_anim)
}



















