#' Install & load CRAN, GitHub, and GitLab packages as current on given date 
#' 
#' Groundhog maintains a separate local package library where it stores 
#' version-controlled packages, with multiple versions of the same package saved 
#' side-by-side. The `date` argument in the `groundhog.library()` function 
#' determines the version of the package that is loaded (the most recently 
#' available version on that date). #' If that version of the package is not 
#' already installed in the local `groundhog` library, #' it is automatically 
#' installed. `groundhog.library()` thus substitutes both `library()` and 
#' `install.packages()`. No changes to how R manages packages are made 
#' (e.g., no change to  #' .libPaths(), to .Rprofile, or to R Studio global 
#' settings). Therefore, to discontinue relying on `groundhog`, all you do is go 
#' back to #' executing the `install.packages()` and `library()` #' functions, 
#' instead of the `groundhog.library()` function.
#'  
#'@param pkg character string or vector with name of target package(s). 
#'Single package names need not be in quotes.
#'@param date character string  (yyyy-mm-dd), or date value, with the date which 
#'determines the version of the package, and all dependencies, to be loaded 
#'(and installed if needed). #'The most recent #'date accepted is 2 days prior 
#'to when the code is executed.
#'@param quiet.install logical, defaults to `TRUE`. When set to `FALSE`, displays
#'  output generated by `install.packages()` when installing from source
#'@param include.suggests logical, defaults to `FALSE`. When set to `TRUE`, 
#'   loads dependencies classified in the DESCRIPTION file as `suggested`.
#'@param ignore.deps an optional character vector containing dependencies which 
#'   may be already loaded in the R session and even if the loaded version does 
#'   not match the version implied by the entered date, groundhog.library() will 
#'   proceed and ignore #'   this conflict. If one version of a package is loaded, 
#'   and a different is needed for #'   groundhog, the default behavior is to 
#'   stop the request and ask the user to restart #'   the R session to unload 
#'   all packages. This will bypass that requirement.
#'@param force.source logical (defaults to `FALSE`). When set to `TRUE`, will 
#'   not attempt installing binary from CRAN or MRAN and instead download source 
#'   file and install it.
#'@param force.install logical (defaults to `FALSE`). When set to `TRUE`, will 
#    delete existing package files in groundhog folder, and install anew.
#'@param tolerate.R.version optional character string containing an R version 
#'   which `groundhog.library()` will not throw an error for using, even if the 
#'   date entered corresponds to a more recent major R release.  
#'@return a character vector containing all active packages for the session,
#'   with their version number, under the format `pkg_vrs`.
#'@examples
#'\dontrun{
#' groundhog.library("magrittr", "2022-04-01")
#'
#' pkgs <- c('pwr','metafor')
#' groundhog.library(pkgs, "2022-04-01")
#' 
#'# When running an existing  script that relied on `library()` to load packages,
#'# you can wrap the library calls in double-quotes, loading the packages with 
#'# groundhog:
#'
#'   groundhog.library(
#'        "
#'         library('pwr')
#'         library('metafor')
#'         library('tidyr')
#'         library('rio')
#'         library('this.path')
#'        "
#'        ,'2022-04-01')
#'
#' #Allow using R 3.6.3 despite entering a date that corresponds to R >=4.0.0
#'   groundhog.library('rio', '2022-04-11', tolerate.R.version='3.6.3')
#' 
#' }
#'
#' @importFrom utils capture.output
#'
#' @details For more information about groundhog check out [groundhogr.com](http://groundhogr.com)
#' @export
#'
  groundhog.library <- function(pkg, date,  quiet.install = TRUE,  include.suggests = FALSE,  
                            ignore.deps=c(), force.source = FALSE,
                            force.install = FALSE, tolerate.R.version = "" )
    {
     
  #0) Save default libpaths to return to it after exiting
        if (!exists("orig_lib_paths",envir=.pkgenv)) {
              .pkgenv[["orig_lib_paths"]] <- .libPaths()
           }
    
        #This is executed first so that upon exit() and setting it back to this, it exists for sure.

    
  #0.5) Verify a mirror has been set    
     set.default.mirror() #Function 36
    
  #0.6 Verify a personal library has been assigned
     verify.personal.library.exists()
     
  #1) Validation     
    #1.0 if ignore deps is not empty, check that it is already loaded
       if (length(ignore.deps)>0) {
         if (!all(ignore.deps %in% .packages())) {
           message1("\nGroundhog says: Error.\nAll packages included in the ignore.deps() option must be loaded prior to running\n",
                   "groundhog.library(), but the following is/are not: ",
                    paste0(dQuote(ignore.deps [!ignore.deps %in% .packages()]), collapse=" ,"))
           exit()
          } #End if some are not loaded
          } #End if ignore.deps() are requested
    
    #1.0.5) Are pkg and date set?
      if (missing(pkg) || missing(date)) {
        message1("groundhog says: you must include both a package name and a date in 'groundhog.library()' ")
        exit()
      }
     
     #1.1) Is date valid?
        date.catch <- try(typeof(date),silent=TRUE)
        if (as.character(class(date.catch))=="try-error") {
          message("Groundhog says: The object '" , as.character(substitute(date)) ,"', does not exist.")
          exit()
        }
        validate.date(date) #Function defined in utils.R
   
    #1.2) Reload databases if necessary and change them back on existr  so that any changes to cran.toc are undone (previous groundhog.library() call loading remotes)
        if (is.null(.pkgenv[['cran.toc']])) load.cran.toc()
       
    #1.4) Verify options are T/F (utils.R - function 27)
          validate.TF(include.suggests)
          validate.TF(force.source)
          validate.TF(force.install)

              
    #2) On Exit refresh libpath and cran.toc
          
            on.exit({
                    #Read cran toc again to undo any changes with remote
                       .pkgenv[['cran.toc']] <- readRDS(file.path(get.groundhog.folder(),"cran.toc.rds"))
                    
                    #Return libpath, if it has been set.
					            if  (exists("orig_lib_paths",envir=.pkgenv)) .libPaths(.pkgenv[["orig_lib_paths"]])
                    })
            
    #2.5 Drop pre-existing libpaths        
        .libPaths('')
    
    #3 If the day merits, update the database
        update_cran.toc_if.needed(date) 

    #4 validate R
        validate_R(date , tolerate.R.version)

    #5 Set of ignorable conflicts
        ignore.deps <- c(ignore.deps_default() , ignore.deps) #Add any ignore.deps explicitly stated to the default set in utils
        
        
    #6 put package name in quotes if it is not an object and was not put in quotes
        pkg.catch <- try(typeof(pkg),silent=TRUE)
        if (as.character(class(pkg.catch))=="try-error") {
          pkg <- as.character(substitute(pkg))
          } 
          
        
         
    #6.5 Sandwich possible library() commands  
        pkg <- sandwich.library(pkg)  #utils.R function 34
   
        
    #7 Add groundhog.day to hogdays to alert of possible different days used in a snowball.conflict
        if (!is.null(.pkgenv[['hogdays']])) {
            .pkgenv[['hogdays']] <- unique(c(date, .pkgenv[['hogdays']]))
          
          } else {
          
            .pkgenv[['hogdays']]<- date
        
          }
        
        
  #8 Loop running groundhog
        for (pkgk in pkg) {
              #Identify as remote based on '/'package (remotes need a /)'
                cran <- TRUE
                if (basename(pkgk)!=pkgk) cran <- FALSE
                
              #Process based on remote
                  if (cran==TRUE)   groundhog.library.single(pkgk, date, quiet.install ,  include.suggests ,  ignore.deps, force.source , force.install)
                  if (cran==FALSE)  groundhog.library.single.remote(pkgk, date, quiet.install ,  include.suggests ,  ignore.deps, force.source , force.install)
                }
  

 
  } #End of groundhog.library
        