% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/alm.R
\name{alm}
\alias{alm}
\title{Advanced Linear Model}
\usage{
alm(formula, data, subset, na.action, distribution = c("dnorm", "dlogis",
  "dlaplace", "dalaplace", "ds", "dt", "dfnorm", "dlnorm", "dllaplace", "dls",
  "dbcnorm", "dinvgauss", "dpois", "dnbinom", "dbeta", "plogis", "pnorm"),
  occurrence = c("none", "plogis", "pnorm"), ar = 0, i = 0,
  parameters = NULL, vcovProduce = FALSE, fast = FALSE, ...)
}
\arguments{
\item{formula}{an object of class "formula" (or one that can be coerced to
that class): a symbolic description of the model to be fitted.}

\item{data}{a data frame or a matrix, containing the variables in the model.}

\item{subset}{an optional vector specifying a subset of observations to be
used in the fitting process.}

\item{na.action}{a function which indicates what should happen when the
data contain NAs. The default is set by the na.action setting of
\link[base]{options}, and is \link[stats]{na.fail} if that is unset. The
factory-fresh default is \link[stats]{na.omit}. Another possible value
is NULL, no action. Value \link[stats]{na.exclude} can be useful.}

\item{distribution}{what density function to use in the process. The full
name of the distribution should be provided here. Values with "d" in the
beginning of the name refer to the density function, while "p" stands for
"probability" (cumulative distribution function). The names align with the
names of distribution functions in R. For example, see \link[stats]{dnorm}.}

\item{occurrence}{what distribution to use for occurrence variable. Can be
\code{"none"}, then nothing happens; \code{"plogis"} - then the logistic
regression using \code{alm()} is estimated for the occurrence part;
\code{"pnorm"} - then probit is constructed via \code{alm()} for the
occurrence part. In both of the latter cases, the formula used is the same
as the formula for the sizes. Finally, an "alm" model can be provided and
its estimates will be used in the model construction.

If this is not \code{"none"}, then the model is estimated
in two steps: 1. Occurrence part of the model; 2. Sizes part of the model
(excluding zeroes from the data).}

\item{ar}{the order of AR to include in the model. Only non-seasonal
orders are accepted.}

\item{i}{the order of I to include in the model. Only non-seasonal
orders are accepted.}

\item{parameters}{vector of parameters of the linear model. When \code{NULL}, it
is estimated.}

\item{vcovProduce}{whether to produce variance-covariance matrix of
coefficients or not. This is done via hessian calculation, so might be
computationally costly.}

\item{fast}{if \code{TRUE}, then the function won't check whether
the data has variability and whether the regressors are correlated. Might
cause trouble, especially in cases of multicollinearity.}

\item{...}{additional parameters to pass to distribution functions. This
includes: \code{alpha} value for Asymmetric Laplace distribution,
\code{size} for the Negative Binomial or \code{df} for the Chi-Squared and
Student's t. You can also pass parameters to the optimiser: 1.
\code{maxeval} - maximum number of evaluations to carry out (default is
100); 2. \code{xtol_rel} - the precision of the optimiser (the default is
1E-6); 3. \code{algorithm} - the algorithm to use in optimisation
(\code{"NLOPT_LN_SBPLX"} by default). 4. \code{print_level} - the level of
output for the optimiser (0 by default). You can read more about these
parameters in the documentation of \link[nloptr]{nloptr} function.}
}
\value{
Function returns \code{model} - the final model of the class
"alm", which contains:
\itemize{
\item coefficients - estimated parameters of the model,
\item vcov - covariance matrix of parameters of the model (based on Fisher
Information). Returned only when \code{vcovProduce=TRUE},
\item fitted - fitted values,
\item residuals - residuals of the model,
\item mu - the estimated location parameter of the distribution,
\item scale - the estimated scale parameter of the distribution,
\item distribution - distribution used in the estimation,
\item logLik - log-likelihood of the model,
\item df.residual - number of degrees of freedom of the residuals of the model,
\item df - number of degrees of freedom of the model,
\item call - how the model was called,
\item rank - rank of the model,
\item data - data used for the model construction,
\item occurrence - the occurrence model used in the estimation,
\item other - the list of all the other parameters either passed to the
function or estimated in the process, but not included in the standard output
(e.g. \code{alpha} for Asymmetric Laplace).
}
}
\description{
Function estimates model based on the selected distribution
}
\details{
This is a function, similar to \link[stats]{lm}, but for the cases of several
non-normal distributions. These include:
\enumerate{
\item \link[stats]{dnorm} - Normal distribution,
\item \link[stats]{dlogis} - Logistic Distribution,
\item \link[greybox]{dlaplace} - Laplace distribution,
\item dllaplace - Log Laplace distribution,
\item \link[greybox]{dalaplace} - Asymmetric Laplace distribution,
\item \link[stats]{dt} - T-distribution,
\item \link[greybox]{ds} - S-distribution,
\item dls - Log S-distribution,
\item \link[greybox]{dfnorm} - Folded normal distribution,
\item \link[stats]{dlnorm} - Log normal distribution,
\item \link[greybox]{dbcnorm} - Box-Cox normal distribution,
\item \link[statmod]{dinvgauss} - Inverse Gaussian distribution,
\item \link[stats]{dbeta} - Beta distribution,
\item \link[stats]{dpois} - Poisson Distribution,
\item \link[stats]{dnbinom} - Negative Binomial Distribution,
\item \link[stats]{plogis} - Cumulative Logistic Distribution,
\item \link[stats]{pnorm} - Cumulative Normal distribution.
}

This function can be considered as an analogue of \link[stats]{glm}, but with the
focus on time series. This is why, for example, the function has \code{ar} and
\code{i} parameters and produces time series analysis plots with \code{plot(alm(...))}.

This function is slower than \code{lm}, because it relies on likelihood estimation
of parameters, hessian calculation and matrix multiplication. So think twice when
using \code{distribution="dnorm"} here.

The estimation is done using likelihood of respective distributions.

See more details and examples in the vignette "ALM":
\code{vignette("alm","greybox")}
}
\examples{

### An example with mtcars data and factors
mtcars2 <- within(mtcars, {
   vs <- factor(vs, labels = c("V", "S"))
   am <- factor(am, labels = c("automatic", "manual"))
   cyl  <- factor(cyl)
   gear <- factor(gear)
   carb <- factor(carb)
})
# The standard model with Log Normal distribution
ourModel <- alm(mpg~., mtcars2[1:30,], distribution="dlnorm")
summary(ourModel)
plot(ourModel)

# Produce predictions with the one sided interval (upper bound)
predict(ourModel, mtcars2[-c(1:30),], interval="p", side="u")


### Artificial data for the other examples
xreg <- cbind(rlaplace(100,10,3),rnorm(100,50,5))
xreg <- cbind(100+0.5*xreg[,1]-0.75*xreg[,2]+rlaplace(100,0,3),xreg,rnorm(100,300,10))
colnames(xreg) <- c("y","x1","x2","Noise")
inSample <- xreg[1:80,]
outSample <- xreg[-c(1:80),]

# An example with Laplace distribution
ourModel <- alm(y~x1+x2, inSample, distribution="dlaplace")
summary(ourModel)
plot(predict(ourModel,outSample))

# And another one with Asymmetric Laplace distribution (quantile regression)
# with optimised alpha
ourModel <- alm(y~x1+x2, inSample, distribution="dalaplace")
summary(ourModel)
plot(predict(ourModel,outSample))

# An example with AR(1) order
ourModel <- alm(y~x1+x2, inSample, distribution="dnorm", ar=1)
summary(ourModel)
plot(predict(ourModel,outSample))

### Examples with the count data
xreg[,1] <- round(exp(xreg[,1]-70),0)
inSample <- xreg[1:80,]
outSample <- xreg[-c(1:80),]

# Negative Binomial distribution
ourModel <- alm(y~x1+x2, inSample, distribution="dnbinom")
summary(ourModel)
predict(ourModel,outSample,interval="p",side="u")

# Poisson distribution
ourModel <- alm(y~x1+x2, inSample, distribution="dpois")
summary(ourModel)
predict(ourModel,outSample,interval="p",side="u")


### Examples with binary response variable
xreg[,1] <- round(xreg[,1] / (1 + xreg[,1]),0)
inSample <- xreg[1:80,]
outSample <- xreg[-c(1:80),]

# Logistic distribution (logit regression)
ourModel <- alm(y~x1+x2, inSample, distribution="plogis")
summary(ourModel)
plot(predict(ourModel,outSample,interval="c"))

# Normal distribution (probit regression)
ourModel <- alm(y~x1+x2, inSample, distribution="pnorm")
summary(ourModel)
plot(predict(ourModel,outSample,interval="p"))

}
\seealso{
\code{\link[greybox]{stepwise}, \link[greybox]{lmCombine},
\link[greybox]{xregTransformer}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{models}
\keyword{nonlinear}
\keyword{regression}
\keyword{ts}
