#' Fill your description
#'
#' @param pkg_name The name of the package
#' @param pkg_title The title of the package
#' @param pkg_description Description of the package
#' @param author_first_name First Name of the author
#' @param author_last_name Last Name of the author
#' @param author_email Email of the author
#' @param author_orcid ORCID of the author
#' @param repo_url URL (if needed)
#' @param pkg_version The version of the package. Default is 0.0.0.9000
#' @param pkg Path to look for the DESCRIPTION. Default is `get_golem_wd()`.
#'
#'
#' @export
#'
#' @return The {desc} object, invisibly.
fill_desc <- function(
  pkg_name,
  pkg_title,
  pkg_description,
  author_first_name,
  author_last_name,
  author_email,
  author_orcid = NULL,
  repo_url = NULL,
  pkg_version = "0.0.0.9000",
  pkg = get_golem_wd()
) {
  path <- fs_path_abs(pkg)

  desc <- desc_description(
    file = fs_path(path, "DESCRIPTION")
  )

  if (!is.null(author_orcid) & !is.character(author_orcid)) {
    stop("ORCID ID must be provided as a character object")
  }


  if (is.null(author_orcid)) {
    desc$set(
      "Authors@R",
      sprintf(
        "person('%s', '%s', email = '%s', role = c('cre', 'aut'))",
        author_first_name,
        author_last_name,
        author_email
      )
    )
  } else {
    desc$set(
      "Authors@R",
      sprintf(
        "person('%s', '%s', email = '%s', role = c('cre', 'aut'), comment = c(ORCID = '%s'))",
        author_first_name,
        author_last_name,
        author_email,
        author_orcid
      )
    )
  }
  desc$del(
    keys = "Maintainer"
  )
  desc$set_version(
    version = pkg_version
  )
  set_golem_version(
    version = pkg_version,
    pkg = path
  )
  desc$set(
    Package = as.character(pkg_name)
  )
  change_app_config_name(
    name = pkg_name,
    path = pkg
  )
  set_golem_name(pkg_name)

  desc$set(
    Title = pkg_title
  )
  desc$set(
    Description = pkg_description
  )
  if_not_null(
    repo_url,
    desc$set(
      "URL",
      repo_url
    )
  )
  if_not_null(
    repo_url,
    desc$set(
      "BugReports",
      sprintf(
        "%s/issues",
        repo_url
      )
    )
  )

  desc$write(
    file = "DESCRIPTION"
  )

  cli_cat_bullet(
    "DESCRIPTION file modified",
    bullet = "tick",
    bullet_col = "green"
  )
  return(
    invisible(
      desc
    )
  )
}
