% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glmmTMB.R
\name{glmmTMB}
\alias{glmmTMB}
\title{Fit models with TMB}
\usage{
glmmTMB(formula, data = NULL, family = gaussian(), ziformula = ~0,
  dispformula = ~1, weights = NULL, offset = NULL, se = TRUE,
  verbose = FALSE, doFit = TRUE, control = glmmTMBControl())
}
\arguments{
\item{formula}{combined fixed and random effects formula, following lme4
syntax}

\item{data}{data frame}

\item{family}{family (variance/link function) information; see \code{\link{family}} for
generic family details or \code{\link{family_glmmTMB}} for details of \code{glmmTMB} specific families.  
As in \code{\link{glm}}, \code{family} can be specified as (1) a character string
referencing an existing family-construction function (e.g. \sQuote{"binomial"}); (2) a symbol referencing
such a function (\sQuote{binomial}); or (3) the output of such a function (\sQuote{binomial()}).
In addition, for families such as \code{betabinomial} that are special to \code{glmmTMB}, family
can be specified as (4) a list comprising the name of the distribution and the link function
(\sQuote{list(family="binomial", link="logit")}). However, the first 3 options are preferable.}

\item{ziformula}{a \emph{one-sided} (i.e., no response variable) formula for
    zero-inflation combining fixed and random effects:
the default \code{~0} specifies no zero-inflation.
Specifying \code{~.} will set the right-hand side of the zero-inflation
formula identical to the right-hand side of the main (conditional effects)
formula; terms can also be added or subtracted. \strong{Offset terms
will automatically be dropped from the conditional effects formula.}
The zero-inflation model uses a logit link.}

\item{dispformula}{a \emph{one-sided} formula for dispersion containing only fixed effects: the
default \code{~1} specifies the standard dispersion given any family.
The argument is ignored for families that do not have a dispersion parameter.
For an explanation of the dispersion parameter for each family, see (\code{\link{sigma}}).
The dispersion model uses a log link. 
In Gaussian mixed models, \code{dispformula=~0} fixes the parameter to be 0, forcing variance into the random effects.}

\item{weights}{weights, as in \code{glm}. Not automatically scaled to have sum 1.}

\item{offset}{offset}

\item{se}{whether to return standard errors}

\item{verbose}{logical indicating if some progress indication should be printed to the console.}

\item{doFit}{whether to fit the full model, or (if FALSE) return the preprocessed data and parameter objects,
without fitting the model}

\item{control}{control parameters; see \code{\link{glmmTMBControl}}.}
}
\description{
Fit models with TMB
}
\details{
\itemize{
\item binomial models with more than one trial (i.e., not binary/Bernoulli)
can either be specified in the form \code{prob ~ ..., weights = N} or in
the more typical two-column matrix (\code{cbind(successes,failures)~...}) form.
\item in all cases \code{glmmTMB} returns maximum likelihood estimates - random effects variance-covariance matrices are not REML (so use \code{REML=FALSE} when comparing with \code{lme4::lmer}), and residual standard deviations (\code{\link{sigma}}) are not bias-corrected. Because the \code{\link{df.residual}} method for \code{glmmTMB} currently counts the dispersion parameter, one would need to multiply by \code{sqrt(nobs(fit)/(1+df.residual(fit)))} when comparing with \code{lm} ...
\item by default, vector-valued random effects are fitted with
unstructured (general positive definite) variance-covariance matrices.
Structured variance-covariance matrices can be specified in
the form \code{struc(terms|group)}, where \code{struc} is one
of
\itemize{
\item \code{diag} (diagonal, heterogeneous variance)
\item \code{ar1} (autoregressive order-1, homogeneous variance)
\item \code{cs} (compound symmetric, heterogeneous variance)
\item \code{ou} (* Ornstein-Uhlenbeck, homogeneous variance)
\item \code{exp} (* exponential autocorrelation)
\item \code{gau} (* Gaussian autocorrelation)
\item \code{mat} (* Matérn process correlation)
\item \code{toep} (* Toeplitz)
}
(note structures marked with * are experimental/untested)
}
}
\examples{
(m1 <- glmmTMB(count~ mined + (1|site), 
  zi=~mined, 
  family=poisson, data=Salamanders))
summary(m1)
\donttest{
## Zero-inflated negative binomial model
(m2 <- glmmTMB(count~spp + mined + (1|site), 
  zi=~spp + mined, 
  family=nbinom2, Salamanders))

## Hurdle Poisson model
(m3 <- glmmTMB(count~spp + mined + (1|site), 
  zi=~spp + mined, 
  family=truncated_poisson, Salamanders))

## Binomial model
data(cbpp, package="lme4")
(tmbm1 <- glmmTMB(cbind(incidence, size-incidence) ~ period + (1 | herd),
               data=cbpp, family=binomial))

## Dispersion model
sim1=function(nfac=40, nt=100, facsd=.1, tsd=.15, mu=0, residsd=1)
{
  dat=expand.grid(fac=factor(letters[1:nfac]), t= 1:nt)
  n=nrow(dat)
  dat$REfac=rnorm(nfac, sd= facsd)[dat$fac]
  dat$REt=rnorm(nt, sd= tsd)[dat$t]
  dat$x=rnorm(n, mean=mu, sd=residsd) + dat$REfac + dat$REt
  return(dat)
}
set.seed(101)
d1 = sim1(mu=100, residsd =10)
d2 = sim1(mu=200, residsd =5)
d1$sd="ten"
d2$sd="five"
dat = rbind(d1, d2)
m0 = glmmTMB(x~sd+(1|t), dispformula=~sd, dat)
fixef(m0)$disp
c(log(5^2), log(10^2)-log(5^2)) #expected dispersion model coefficients
}
}
