#' @title Grouped pie charts with statistical tests
#' @name grouped_ggpiestats
#' @aliases grouped_ggpiestats
#' @description Helper function for `ggstatsplot::ggpiestats` to apply this
#'   function across multiple levels of a given factor and combining the
#'   resulting plots using `ggstatsplot::combine_plots`.
#' @author Indrajeet Patil, Chuck Powell
#'
#' @param grouping.var Grouping variable.
#' @param title.prefix Character specifying the prefix text for the fixed plot
#'   title (name of each factor level) (Default: `"Group"`).
#' @inheritParams ggpiestats
#' @inheritDotParams combine_plots
#'
#' @importFrom dplyr select bind_rows summarize mutate mutate_at mutate_if
#' @importFrom dplyr group_by n arrange
#' @importFrom rlang !! enquo quo_name ensym
#' @importFrom glue glue
#' @importFrom purrr map set_names
#' @importFrom tidyr nest
#'
#' @seealso \code{\link{ggpiestats}}
#'
#' @inherit ggpiestats return references
#' @inherit ggpiestats return details
#' @inherit ggpiestats return return
#'
#' @examples
#' 
#' # grouped one-sample proportion tests
#' ggstatsplot::grouped_ggpiestats(
#'   data = mtcars,
#'   grouping.var = am,
#'   main = cyl
#' )
#' 
#' # without condition and with count data
#' library(jmv)
#' 
#' ggstatsplot::grouped_ggpiestats(
#'   data = as.data.frame(HairEyeColor),
#'   main = Hair,
#'   counts = Freq,
#'   grouping.var = Sex
#' )
#' 
#' # the following will take slightly more amount of time
#' \dontrun{
#' # for reproducibility
#' set.seed(123)
#' 
#' # let's create a smaller dataframe
#' diamonds_short <- ggplot2::diamonds %>%
#'   dplyr::filter(.data = ., cut %in% c("Fair", "Very Good", "Ideal")) %>%
#'   dplyr::sample_frac(tbl = ., size = 0.10)
#' 
#' # plot
#' ggstatsplot::grouped_ggpiestats(
#'   data = diamonds_short,
#'   main = color,
#'   condition = clarity,
#'   grouping.var = cut,
#'   bf.message = TRUE,
#'   sampling.plan = "poisson",
#'   title.prefix = "Quality",
#'   slice.label = "both",
#'   messages = FALSE,
#'   perc.k = 1,
#'   nrow = 3
#' )
#' }
#' @export

# defining the function
grouped_ggpiestats <- function(data,
                               main,
                               condition = NULL,
                               counts = NULL,
                               grouping.var,
                               title.prefix = "Group",
                               ratio = NULL,
                               paired = FALSE,
                               factor.levels = NULL,
                               stat.title = NULL,
                               sample.size.label = TRUE,
                               bf.message = FALSE,
                               sampling.plan = "indepMulti",
                               fixed.margin = "rows",
                               prior.concentration = 1,
                               caption = NULL,
                               conf.level = 0.95,
                               nboot = 25,
                               simulate.p.value = FALSE,
                               B = 2000,
                               legend.title = NULL,
                               facet.wrap.name = NULL,
                               k = 2,
                               perc.k = 0,
                               slice.label = "percentage",
                               facet.proptest = TRUE,
                               ggtheme = ggplot2::theme_bw(),
                               ggstatsplot.layer = TRUE,
                               package = "RColorBrewer",
                               palette = "Dark2",
                               direction = 1,
                               messages = TRUE,
                               ...) {

  # ======================== check user input =============================

  # create a list of function call to check
  param_list <- base::as.list(base::match.call())

  # check that there is a grouping.var
  if (!"grouping.var" %in% names(param_list)) {
    base::stop("You must specify a grouping variable")
  }

  # check that conditioning and grouping.var are different
  if ("condition" %in% names(param_list)) {
    if (as.character(param_list$condition) == as.character(param_list$grouping.var)) {
      base::message(cat(
        crayon::red("\nError: "),
        crayon::blue(
          "Identical variable (",
          crayon::yellow(param_list$condition),
          ") was used for both grouping and conditioning, which is not allowed.\n"
        ),
        sep = ""
      ))
      base::return(base::invisible(param_list$condition))
    }
  }


  # ensure the grouping variable works quoted or unquoted
  grouping.var <- rlang::ensym(grouping.var)

  # ======================== preparing dataframe =============================

  # if condition variable *is* provided
  if (!missing(condition)) {
    # if the data is not tabled
    if (missing(counts)) {
      df <-
        dplyr::select(
          .data = data,
          !!rlang::enquo(grouping.var),
          !!rlang::enquo(main),
          !!rlang::enquo(condition)
        ) %>%
        dplyr::mutate(
          .data = .,
          title.text = !!rlang::enquo(grouping.var)
        )
    } else if (!missing(counts)) {
      # if data is tabled
      df <-
        dplyr::select(
          .data = data,
          !!rlang::enquo(grouping.var),
          !!rlang::enquo(main),
          !!rlang::enquo(condition),
          !!rlang::enquo(counts)
        ) %>%
        dplyr::mutate(
          .data = .,
          title.text = !!rlang::enquo(grouping.var)
        )
    }
  } else if (missing(condition)) {
    # if condition variable is *not* provided
    if (base::missing(counts)) {
      df <-
        dplyr::select(
          .data = data,
          !!rlang::enquo(grouping.var),
          !!rlang::enquo(main)
        ) %>%
        dplyr::mutate(
          .data = .,
          title.text = !!rlang::enquo(grouping.var)
        )
    } else if (!missing(counts)) {
      # if data is tabled
      df <-
        dplyr::select(
          .data = data,
          !!rlang::enquo(grouping.var),
          !!rlang::enquo(main),
          !!rlang::enquo(counts)
        ) %>%
        dplyr::mutate(
          .data = .,
          title.text = !!rlang::enquo(grouping.var)
        )
    }
  }

  # make a list of dataframes by grouping variable
  df %<>%
    dplyr::mutate_if(
      .tbl = .,
      .predicate = purrr::is_bare_character,
      .funs = ~ as.factor(.)
    ) %>%
    dplyr::mutate_if(
      .tbl = .,
      .predicate = is.factor,
      .funs = ~ base::droplevels(.)
    ) %>%
    dplyr::filter(.data = ., !is.na(!!rlang::enquo(grouping.var))) %>%
    base::split(.[[rlang::quo_text(grouping.var)]])

  # ============== build pmap list based on conditions =====================

  if (!missing(condition) && missing(counts)) {
    flexiblelist <- list(
      data = df,
      main = rlang::quo_text(ensym(main)),
      condition = rlang::quo_text(ensym(condition)),
      title = glue::glue("{title.prefix}: {names(df)}")
    )
  }

  if (missing(condition) && missing(counts)) {
    flexiblelist <- list(
      data = df,
      main = rlang::quo_text(ensym(main)),
      title = glue::glue("{title.prefix}: {names(df)}")
    )
  }

  if (!missing(condition) && !missing(counts)) {
    flexiblelist <- list(
      data = df,
      main = rlang::quo_text(ensym(main)),
      condition = rlang::quo_text(ensym(condition)),
      counts = rlang::quo_text(ensym(counts)),
      title = glue::glue("{title.prefix}: {names(df)}")
    )
  }

  if (missing(condition) && !missing(counts)) {
    flexiblelist <- list(
      data = df,
      main = rlang::quo_text(ensym(main)),
      counts = rlang::quo_text(ensym(counts)),
      title = glue::glue("{title.prefix}: {names(df)}")
    )
  }

  # ==================== creating a list of plots =======================

  # creating a list of plots using `pmap`
  plotlist_purrr <-
    purrr::pmap(
      .l = flexiblelist,
      .f = ggstatsplot::ggpiestats,
      # put common parameters here
      ratio = ratio,
      paired = paired,
      factor.levels = factor.levels,
      stat.title = stat.title,
      sample.size.label = sample.size.label,
      bf.message = bf.message,
      sampling.plan = sampling.plan,
      fixed.margin = fixed.margin,
      prior.concentration = prior.concentration,
      caption = caption,
      conf.level = conf.level,
      nboot = nboot,
      simulate.p.value = simulate.p.value,
      B = B,
      legend.title = legend.title,
      facet.wrap.name = facet.wrap.name,
      k = k,
      perc.k = perc.k,
      slice.label = slice.label,
      facet.proptest = facet.proptest,
      ggtheme = ggtheme,
      ggstatsplot.layer = ggstatsplot.layer,
      package = package,
      palette = palette,
      direction = direction,
      messages = messages
    )


  # combining the list of plots into a single plot
  combined_plot <-
    ggstatsplot::combine_plots(
      plotlist = plotlist_purrr,
      ...
    )

  # show the note about grouped_ variant producing object which is not of
  # class ggplot
  if (isTRUE(messages)) {
    grouped_message()
  }

  # return the combined plot
  return(combined_plot)
}
