% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/position-nudge-line.R
\name{position_nudge_line}
\alias{position_nudge_line}
\title{Nudge labels away from a line}
\usage{
position_nudge_line(
  x = NA_real_,
  y = NA_real_,
  xy_relative = c(0.03, 0.03),
  abline = NULL,
  method = NULL,
  formula = y ~ x,
  direction = c("automatic", "none", "split"),
  line_nudge = 1,
  kept.origin = c("original", "none")
)
}
\arguments{
\item{x, y}{Amount of vertical and horizontal distance to move. A numeric
vector of length 1, or of the same length as rows there are in \code{data},
with nudge values in data rows order.}

\item{xy_relative}{Nudge relative to \emph{x} and \emph{y} data expanse, ignored unless
\code{x} and \code{y} are both \code{NA}s.}

\item{abline}{a vector of length two giving the intercept and slope.}

\item{method}{One of \code{"spline"}, \code{"lm"} or \code{"auto"}.}

\item{formula}{A model formula for \code{\link{lm}} when \code{method =
"lm"}. Ignored otherwise.}

\item{direction}{One of \code{"automatic"}, \code{"none"}, or \code{"split"}.}

\item{line_nudge}{A positive multiplier >= 1, increasing nudging away from
the curve or line compared to nudging from points.}

\item{kept.origin}{One of \code{"original"} or \code{"none"}.}
}
\value{
A \code{"Position"} object.
}
\description{
\code{position_nudge_line()} is generally useful for adjusting the starting
position of labels or text to be repelled while preserving the original
position as the start of the segments. The difference compared to
\code{\link{position_nudge_center}} is that the nudging is away from from a
line or curve fitted to the data points or supplied as coefficients. While
\code{position_nudge_center()} is most useful for "round-shaped", vertically-
or horizontally elongated clouds of points, \code{position_nudge_line()} is
most suitable when observations follow a linear or curvilinear relationship
between \emph{x} and \emph{y} values. In contrast to
\code{\link[ggplot2]{position_nudge}}, \code{position_nudge_line()} returns
in `data` both the original coordinates and the nudged coordinates.
}
\details{
The default amount of nudging is 3% of the spread of the data along
  \emph{x} and \emph{y} axes, which in most cases is good. In most cases it is best to
  apply nudging along a direction perpendicular to the line or curve, if this
  is the aim, passing an argument to only one of \code{x}, \code{y} or
  \code{xy_relative} will be enough. When \code{direction = "split"} nudging
  is away from an implicit line or curve on either side with positive
  nudging. The line or curve can be smooth spline or linear regression fitted
  on-the-fly to the data points, or a straight line defined by its
  coefficients passed to \code{abline}. The fitting is well defined only if
  the observations fall roughly on a curve or straight line that is monotonic
  in \code{y}. By means of \code{line_nudge} one can increment nudging away
  from the line or curve compared to away from the points, which is useful
  for example to keep labels outside of a confidence band. Direction defaults
  to \code{"split"} when \code{line_nudge} > 1, and otherwise to
  \code{"none"}.
}
\note{
For \code{method = "lm"} only model formulas corresponding to
  polynomials with no missing terms are supported. If using\code{\link{poly}}
  in the model formula, \code{raw = TRUE} is required.

  In practice, \code{x} and \code{y} should have the same sign for nudging to
  work correctly.

  This position is most useful when labeling points conforming a cloud along
  an arbitrary curve or line.
}
\examples{

set.seed(16532)
df <- data.frame(
  x = -10:10,
  y = (-10:10)^2,
  yy = (-10:10)^2 + rnorm(21, 0, 4),
  yyy = (-10:10) + rnorm(21, 0, 4),
  l = letters[1:21]
)

# Setting the nudging distance

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line())

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text_s(position = position_nudge_line())

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(xy_relative = -0.03))

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(x = 0.6, y = 3.2))

ggplot(df, aes(x, y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line(x = -0.6, y = -4))

# Other curves, using defaults

ggplot(df, aes(x, -y, label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line())

ggplot(subset(df, x >= 0), aes(y, sqrt(y), label = l)) +
  geom_line(linetype = "dotted") +
  geom_point() +
  geom_text(position = position_nudge_line())

# Points scattered near a curve or line, we use 'direction = "split"'

ggplot(df, aes(x)) +
  geom_line(aes(y = y), linetype = "dotted") +
  geom_point(aes(y = yy)) +
  geom_text(aes(y = yy, label = l),
            position = position_nudge_line(direction = "split"))

ggplot(subset(df, x >= 0), aes(y, yy)) +
  stat_smooth(method = "lm", formula = y ~ x) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(direction = "split"))

# increasing the nudging for labels near the line

ggplot(subset(df, x >= 0), aes(y, yy)) +
  stat_smooth(method = "lm", formula = y ~ x) +
  geom_point() +
  geom_text(aes(label = l),
              position = position_nudge_line(line_nudge = 2,
                                             direction = "split"))

# fitting a linear model instead of the default spline

ggplot(subset(df, x >= 0), aes(y, yy)) +
  stat_smooth(method = "lm", formula = y ~ x) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(method = "lm",
                                           direction = "split"))

ggplot(subset(df, x >= 0), aes(x, x^2)) +
  stat_smooth(method = "lm", formula = y ~ poly(x, 2, raw = TRUE)) +
  geom_point() +
  geom_text(aes(label = l),
            position = position_nudge_line(method = "lm",
                                           formula = y ~ poly(x, 2, raw = TRUE)))

}
\seealso{
\code{\link[ggplot2]{position_nudge}},
 \code{\link[ggrepel]{position_nudge_repel}}.

Other position adjustments: 
\code{\link{position_dodgenudge}()},
\code{\link{position_jitternudge}()},
\code{\link{position_nudge_center}()},
\code{\link{position_nudge_to}()},
\code{\link{position_stacknudge}()}
}
\concept{position adjustments}
