% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_googlemap.R
\name{get_googlemap}
\alias{get_googlemap}
\title{Get a Google Map.}
\usage{
get_googlemap(
  center = c(lon = -95.3632715, lat = 29.7632836),
  zoom = 10,
  size = c(640, 640),
  scale = 2,
  format = c("png8", "gif", "jpg", "jpg-baseline", "png32"),
  maptype = GOOGLE_VALID_MAP_TYPES,
  language = "en-EN",
  messaging = FALSE,
  urlonly = FALSE,
  filename = NULL,
  color = c("color", "bw"),
  force = FALSE,
  where = tempdir(),
  archiving = FALSE,
  ext = "com",
  inject = "",
  region,
  markers,
  path,
  visible,
  style,
  ...
)
}
\arguments{
\item{center}{the center of the map; either a longitude/latitude numeric
vector or a string containing a location, in which case \code{\link[=geocode]{geocode()}} is
called with \code{source = "google"}. (default: \code{c(lon = -95.3632715, lat = 29.7632836)}, a reference to Houston, Texas)}

\item{zoom}{map zoom; an integer from 3 (continent) to 21 (building), default
value 10 (city)}

\item{size}{rectangular dimensions of map in pixels - horizontal x vertical -
with a max of c(640, 640). this parameter is affected in a multiplicative
way by scale.}

\item{scale}{multiplicative factor for the number of pixels returned possible
values are 1, 2, or 4 (e.g. size = c(640,640) and scale = 2 returns an
image with 1280x1280 pixels).  4 is reserved for google business users
only.  scale also affects the size of labels as well.}

\item{format}{character string providing image format - png, jpeg, and gif
formats available in various flavors}

\item{maptype}{character string providing google map theme. options available
are \code{"terrain"}, \code{"satellite"}, \code{"roadmap"}, and \code{"hybrid"}}

\item{language}{character string providing language of map labels (for themes
with them) in the format \code{"en-EN"}.  not all languages are supported; for
those which aren't the default language is used}

\item{messaging}{turn messaging on/off}

\item{urlonly}{return url only}

\item{filename}{destination file for download (file extension added according
to format). Default \code{NULL} means a random \code{\link[=tempfile]{tempfile()}}.}

\item{color}{color or black-and-white}

\item{force}{if the map is on file, should a new map be looked up?}

\item{where}{where should the file drawer be located (without terminating
"/")}

\item{archiving}{use archived maps.  note: by changing to TRUE you agree to
the one of the approved uses listed in the Google Maps API Terms of Service
: \url{https://cloud.google.com/maps-platform/terms}.}

\item{ext}{domain extension (e.g. \code{"com"}, \code{"co.nz"})}

\item{inject}{character string to add to the url}

\item{region}{borders to display as a region code specified as a
two-character ccTLD ("top-level domain") value, see
\url{https://en.wikipedia.org/wiki/List_of_Internet_top-level_domains#Country_code_top-level_domains}}

\item{markers}{data.frame with first column longitude, second column
latitude, for which google markers should be embedded in the map image, or
character string to be passed directly to api}

\item{path}{data.frame (or list of data.frames) with first column longitude,
second column latitude, for which a single path should be embedded in the
map image, or character string to be passed directly to api}

\item{visible}{a location as a longitude/latitude numeric vector (or data
frame with first column longitude, second latitude) or vector of character
string addresses which should be visible in map extent}

\item{style}{character string to be supplied directly to the api for the
style argument or a named vector (see examples). this is a powerful complex
specification, see
\url{https://developers.google.com/maps/documentation/maps-static/}}

\item{...}{...}
}
\value{
a ggmap object (a classed raster object with a bounding box
attribute)
}
\description{
\code{\link[=get_googlemap]{get_googlemap()}} queries the Google Maps Static API version 2 to download
a static map.  Note that in most cases by using this function you are
agreeing to the Google Maps API Terms of Service at
\url{https://cloud.google.com/maps-platform/terms}. Note that as of mid-2018,
registering with Google Cloud to obtain an API key is required to use any of
Google's services, including \code{\link[=get_googlemap]{get_googlemap()}}. Usage and billing may apply,
see the links under See Also further down in this documentation for more
details.
}
\examples{

\dontrun{ requires Google API key, see ?register_google

## basic usage
########################################

(map <- get_googlemap(c(-97.14667, 31.5493)))
ggmap(map)

# plotting based on a colloquial name
# this requires a geocode() call, and needs that API
get_googlemap("waco, texas") \%>\% ggmap()

# different maptypes are available
get_googlemap("waco, texas", maptype = "satellite") \%>\% ggmap()
get_googlemap("waco, texas", maptype = "hybrid") \%>\% ggmap()

# you can get the url as follows
# see ?register_google if you want the key printed
get_googlemap(urlonly = TRUE)


## other usage
########################################

# markers and paths are easy to access
d <- function(x=-95.36, y=29.76, n,r,a){
  round(data.frame(
    lon = jitter(rep(x,n), amount = a),
    lat = jitter(rep(y,n), amount = a)
  ), digits = r)
}
(df <- d(n = 50, r = 3, a = .3))
map <- get_googlemap(markers = df, path = df, scale = 2)
ggmap(map)
ggmap(map, extent = "device") +
  geom_point(aes(x = lon, y = lat), data = df, size = 3, colour = "black") +
  geom_path(aes(x = lon, y = lat), data = df)

gc <- geocode("waco, texas", source = "google")
center <- as.numeric(gc)
ggmap(get_googlemap(center = center, color = "bw", scale = 2), extent = "device")

# the scale argument can be seen in the following
# (make your graphics device as large as possible)
ggmap(get_googlemap(center, scale = 1), extent = "panel") # pixelated
ggmap(get_googlemap(center, scale = 2), extent = "panel") # fine

# archiving; note that you must meet google's terms for this condition
map <- get_googlemap(archiving = TRUE)
map <- get_googlemap()
map <- get_googlemap()
ggmap(map)


# style
map <- get_googlemap(
  maptype = "roadmap",
  style = c(feature = "all", element = "labels", visibility = "off"),
  color = "bw"
)
ggmap(map)




}

}
\seealso{
\url{https://developers.google.com/maps/documentation/maps-static/overview/},
\url{https://developers.google.com/maps/documentation/maps-static/start/},
\url{https://developers.google.com/maps/documentation/maps-static/get-api-key/},
\url{https://developers.google.com/maps/documentation/maps-static/usage-and-billing/},
\code{\link[=ggmap]{ggmap()}}, \code{\link[=register_google]{register_google()}}
}
\author{
David Kahle \email{david@kahle.io}
}
