`geoR2RF` <-
  function (cov.model, cov.pars, nugget = 0, kappa, aniso.pars) 
{
  cov.model <- match.arg(cov.model,
                         choices = c("exponential", "matern", "gaussian", "spherical",
                           "circular", "cubic", "wave", "power", "powered.exponential",
                           "cauchy", "gencauchy", "gneiting", "pure.nugget",
                           "gneiting.matern"))
  if(missing(aniso.pars)) aniso.pars <- NULL
  if(missing(kappa)) kappa <- NULL
  if (length(cov.pars) != 2) 
    stop("cov.pars must be an vector of size 2 with values for the parameters sigmasq and phi")
  RFmodel <- switch(cov.model, matern = "whittlematern", exponential = "exponential", 
                    gaussian = "gauss", spherical = "spherical", circular = "circular", 
                    cubic = "cubic", wave = "wave", power = "not compatible", 
                    powered.exponential = "stable", cauchy = "cauchy",
                    gencauchy = "gencauchy", gneiting = "gneiting",
                    gneiting.matern = "not compatible", pure.nugget = "nugget")
  if (RFmodel == "not compatible") {
    warning("geoR cov.model not compatible with RandomFields model")
    return(RFmodel)
  }
  if (any(RFmodel == "gencauchy")) 
    kappa <- rev(kappa)
  if (!any(RFmodel == c("gencauchy", "whittlematern", "stable")))
    kappa <- NULL
  if(is.null(aniso.pars))
    return(list(list(model=RFmodel, var=cov.pars[1], kappa=kappa, scale=cov.pars[2]),
                "+",
                list(model="nugget", var=nugget)))
  else{
    mat <- solve(matrix(c(cos(aniso.pars[1]), sin(aniso.pars[1]),
                          -sin(aniso.pars[1]), cos(aniso.pars[1])), nc=2)) %*%
                            diag(c(aniso.pars[2], 1)/cov.pars[2])
    return(list(list(model=RFmodel, var=cov.pars[1], kappa=kappa, aniso=mat), 
                "+",
                list(model="nugget", var=nugget, aniso=diag(1,2))))
  }
}

`grf` <-
  function (n, grid = "irreg", nx, ny, xlims = c(0, 1), ylims = c(0, 
                                                          1), nsim = 1, cov.model = "matern", cov.pars = stop("missing covariance parameters sigmasq and phi"), 
            kappa = 0.5, nugget = 0, lambda = 1, aniso.pars = NULL, mean = 0, 
            method, RF = TRUE, messages) 
{
  call.fc <- match.call()
  if (missing(messages)) 
    messages.screen <- as.logical(ifelse(is.null(getOption("geoR.messages")), 
                                         TRUE, getOption("geoR.messages")))
  else messages.screen <- messages
  cov.model <- match.arg(cov.model, choices = c("matern", "exponential", 
                                      "gaussian", "spherical", "circular", "cubic", "wave", 
                                      "power", "powered.exponential", "stable", "cauchy", "gencauchy", 
                                      "gneiting", "gneiting.matern", "pure.nugget"))
  if (cov.model == "stable") 
    cov.model <- "powered.exponential"
  if (cov.model == "matern" && kappa == 0.5) 
    cov.model <- "exponential"
  tausq <- nugget
  if (is.vector(cov.pars)) {
    sigmasq <- cov.pars[1]
    phi <- cov.pars[2]
    nst <- 1
  }
  else {
    sigmasq <- cov.pars[, 1]
    phi <- cov.pars[, 2]
    nst <- nrow(cov.pars)
  }
  sill.total <- tausq + sum(sigmasq)
  messa <- .grf.aux1(nst, nugget, sigmasq, phi, kappa, cov.model)
  if (!exists(".Random.seed", envir = .GlobalEnv, inherits = FALSE)) {
    warning(".Random.seed not initialised. Creating it with by calling runif(1)")
    runif(1)
  }
  rseed <- get(".Random.seed", envir = .GlobalEnv, inherits = FALSE)
  results <- list()
  if ((!missing(nx) && nx == 1) | (!missing(ny) && ny == 1) | 
      diff(xlims) == 0 | diff(ylims) == 0) {
    sim1d <- TRUE
    if (messages.screen) 
      cat("simulations in 1D\n")
  }
  else sim1d <- FALSE
  if (mode(grid) == "character") 
    grid <- match.arg(grid, choices = c("irreg", "reg"))
  if (is.matrix(grid) | is.data.frame(grid)) {
    results$coords <- as.matrix(grid)
    if (messages.screen) 
      cat("grf: simulation on locations provided by the user\n")
  }
  else {
    if (missing(nx)) {
      if (sim1d) 
        if (diff(xlims) == 0) 
          nx <- 1
        else nx <- n
      else if (mode(grid) == "character" && grid == "reg") 
        nx <- round(sqrt(n))
      else nx <- n
    }
    if (missing(ny)) {
      if (sim1d) 
        if (diff(ylims) == 0) 
          ny <- 1
        else ny <- n
      else if (mode(grid) == "character" && grid == "reg") 
        ny <- round(sqrt(n))
      else ny <- n
    }
    if (mode(grid) == "character" && grid == "irreg") {
      results$coords <- cbind(x = runif(nx, xlims[1], xlims[2]), 
                              y = runif(ny, ylims[1], ylims[2]))
      if (messages.screen) 
        cat(paste("grf: simulation(s) on randomly chosen locations with ", 
                  n, " points\n"))
    }
    else {
      xpts <- seq(xlims[1], xlims[2], l = nx)
      ypts <- seq(ylims[1], ylims[2], l = ny)
      results$coords <- as.matrix(expand.grid(x = xpts, 
                                              y = ypts))
      if (length(xpts) == 1) 
        xspacing <- 0
      else xspacing <- xpts[2] - xpts[1]
      if (length(ypts) == 1) 
        yspacing <- 0
      else yspacing <- ypts[2] - ypts[1]
      if (abs(xspacing - yspacing) < 1e-12) 
        equal.spacing <- TRUE
      else equal.spacing <- FALSE
      if (messages.screen) 
        cat(paste("grf: generating grid ", nx, " * ", 
                  ny, " with ", (nx * ny), " points\n"))
    }
  }
  n <- nrow(results$coords)
  if (length(unique(round(results$coords[, 1], dig = 12))) == 1 |
      length(unique(round(results$coords[, 2], dig = 12))) == 1) 
    sim1d <- TRUE
  else sim1d <- FALSE
  if (!RF && !is.null(aniso.pars)) {
    if (length(aniso.pars) != 2 | mode(aniso.pars) != "numeric") 
      stop("anisotropy parameters must be provided as a numeric vector with two elements: the rotation angle (in radians) and the anisotropy ratio (a number greater than 1)")
    if (messages.screen) 
      cat("grf: transforming to the isotropic space \n")
    results$coords <- coords.aniso(coords = results$coords, 
                                   aniso.pars = aniso.pars)
  }
  if (missing(method)) {
    method <- "cholesky"
    if (n > 500 && RF && require(RandomFields)) 
      method <- "RF"
  }
  method <- match.arg(method, choices = c("cholesky", "svd", 
                                "eigen", "RF", "circular.embedding"))
  if (method == "circular.embedding") {
    if (require(RandomFields)) {
      method <- "RF"
      if (messages.screen) 
        warning("method \"circular.embedding\" now uses algorithm from the package RandomFields")
    }
    else error("Option for method \"circular.embedding\" requires the instalation of the package RandomFields")
  }
  if (messages.screen) {
    cat(messa$nst)
    cat(messa$nugget)
    cat(messa$cov.structures)
    if (method == "RF") 
      cat("grf: simulation using the function GaussRF from package RandomFields \n")
    else cat(paste("grf: decomposition algorithm used is: ", 
                   method, "\n"))
  }
  if (all(phi) == 0) 
    results$data <- matrix(rnorm((n * nsim), mean = 0, sd = sqrt(sill.total)), 
                           nrow = n, ncol = nsim)
  else {
    if (method == "RF") {
      require(RandomFields)
      setRF <<- geoR2RF(cov.model = cov.model, cov.pars = cov.pars, 
                       nugget = nugget, kappa = kappa, aniso.pars=aniso.pars)
      if (!exists("xpts") || is.null(xpts)){
        results$data <- GaussRF(x = results$coords[, 1],y = results$coords[, 2],
                                model = setRF, grid = FALSE, n = nsim)
      }
      else{
        results$data <- drop(matrix(GaussRF(x = xpts, y = ypts, model = setRF,
                                            grid = TRUE, n = nsim), ncol = nsim))
      }
    }
    else
      results$data <- drop(crossprod(varcov.spatial(coords = results$coords, 
                                                    cov.model = cov.model, kappa = kappa,
                                                    nugget = nugget, 
                                                    cov.pars = cov.pars,
                                                    only.decomposition = TRUE,
                                                    func.in = method)$sqrt.varcov, 
                                     matrix(rnorm((n * nsim)), nrow = n, ncol = nsim)))
  }
  if (length(mean) != 1 & length(mean) != length(results$data)) 
    stop("the mean must be a scalar or a vector of the same size as the data")
  results$data <- results$data + mean
  if (lambda != 1) {
    if (lambda != 0) 
      results$data <- (results$data * lambda + 1)^(1/lambda)
    else results$data <- exp(results$data)
    messa$transformation <- paste("grf: Data transformed (Box-Cox), for lambda =", 
                                  lambda)
    if (messages.screen) 
      cat(messa$transformation)
    cat("\n")
  }
  if (!RF && !is.null(aniso.pars)) {
    if (messages.screen) 
      cat("grf: back-transforming to the anisotropic space \n")
    results$coords <- coords.aniso(coords = results$coords, 
                                   aniso.pars = aniso.pars, reverse = TRUE)
  }
  else {
    aniso.pars <- "no anisotropy parameters provided/used"
  }
  if (messages.screen) 
    cat(paste("grf: End of simulation procedure. Number of realizations:", 
              nsim, "\n"))
  results <- c(results, list(cov.model = cov.model, nugget = nugget, 
                             cov.pars = cov.pars, kappa = kappa, lambda = lambda, 
                             aniso.pars = aniso.pars, method = method, .Random.seed = rseed, 
                             messages = messa, call = call.fc))
  if (mode(grid) == "character" && grid == "reg") {
    if (equal.spacing) 
      attr(results, "spacing") <- xspacing
    else {
      attr(results, "xspacing") <- xspacing
      attr(results, "yspacing") <- yspacing
    }
  }
  attr(results, "sp.dim") <- ifelse(sim1d, "1d", "2d")
  oldClass(results) <- c("grf", "geodata", "variomodel")
  return(results)
}


".grf.aux1" <-
  function (nst, nugget, sigmasq, phi, kappa, cov.model) 
{
  cov.nst <- paste("grf: process with ", nst, " covariance structure(s)\n")
  cov.nugget <- paste("grf: nugget effect is: tausq=", nugget,"\n")
  cov.message <- NULL
  for (i in 1:nst) {
    if (phi[i] == 0) 
      cov.message[i] <- paste("grf: covariance model", i, "is a pure nugget effect\n")
    else {
      if(any(cov.model == c("matern","powered.exponential", 
          "cauchy", "gencauchy", "gneiting.matern"))) 
        cov.message[i] <- paste("grf: covariance model ", 
                                i, " is: ", cov.model, "(sigmasq=", sigmasq[i], 
                                ", phi=", phi[i], ", kappa = ", kappa, ")\n", sep = "")
      else cov.message[i] <- paste("grf: covariance model ", 
                                   i, " is: ", cov.model, "(sigmasq=", sigmasq[i], 
                                   ", phi=", phi[i], ")\n", sep = "")
    }
  }
  return(list(nst = cov.nst, nugget = cov.nugget, cov.structures = cov.message))
}

"lines.variomodel.grf" <-
  function (x, max.dist = max(dist(x$coords)), length = 100, 
            lwd = 2, ...) 
{
#  if(! "package:stats" %in% search()) require(mva)
  if(any(cov.model == c("matern","powered.exponential", 
           "cauchy", "gencauchy", "gneiting.matern"))) 
    kappa <- x$kappa
  else kappa <- NULL
  distance <- seq(0, max.dist, length = length)
  if (is.vector(x$cov.pars)) 
    sill.total <- x$nugget + x$cov.pars[1]
  else sill.total <- x$nugget + sum(x$cov.pars[, 1])
  gamma <- sill.total - cov.spatial(distance, cov.model = x$cov.model, 
                                  kappa = kappa, cov.pars = x$cov.pars)
  lines(distance, gamma, lwd = lwd, ...)
  return(invisible())
}

"image.grf" <-
  function (x, sim.number = 1, x.leg, y.leg, ...) 
{
  ##
  ## this seems to cause problems overlapping maps
  ##op <- par(no.readonly=TRUE)
  ##on.exit(par(op))
  ##
  x1vals <- unique(round(x$coords[,1], dig=12))
  x2vals <- unique(round(x$coords[,2], dig=12))
  nx <- length(x1vals)
  ny <- length(x2vals)
  ldots <- match.call(expand.dots = FALSE)$...
  ##
  ## Plotting simulations in 1-D
  ##
  if(attr(x, 'sp.dim') == "1d" | nx == 1 | ny == 1){
    do.call("plot.1d", c(list(x = x,
                              x1vals = x1vals),
                         .ldots.set(ldots, type="plot.1d",
                                   data="simulation")))
  }
  else{
    ##
    ## Plotting simulations in 2-D
    ##
    ## Checking for retangular grid
    ##
    x$data <- as.matrix(x$data)
    n <- nrow(x$data)
    if (nx * ny != n) 
      stop("cannot produce image plot probably due to irregular grid of locations")
    ##
    ## Preparing image plot elements
    ##
    do.call("image", c(list(x=x1vals, y=x2vals,
                            z=matrix(x$data[, sim.number], nc=ny)),
                       .ldots.set(ldots, type="image",
                                 data="simulation")))
    ##
    ## Adding the legend (if the case)
    ##
    if(!missing(x.leg) && !missing(y.leg)){
      if(is.null(ldots$col)) ldots$col <- heat.colors(12)
      do.call("legend.krige", c(list(x.leg=x.leg,
                                     y.leg=y.leg,
                                     values = x$data[, sim.number]),
                                     ldots))
    }
  }
  return(invisible())
}

"persp.grf" <- 
  function(x, sim.number = 1, ...)
{
  x1vals <- unique(round(x$coords[,1], dig=12))
  x2vals <- unique(round(x$coords[,2], dig=12))
  nx <- length(x1vals)
  ny <- length(x2vals)
  ldots <- match.call(expand.dots = FALSE)$...
  if(attr(x, 'sp.dim') == "1d" | nx == 1 | ny == 1){
    do.call("plot.1d", c(list(x = x,
                              x1vals = x1vals),
                       .ldots.set(ldots, type="plot.1d",
                                 data="simulation")))
  }
  else{
    x$data <- as.matrix(x$data)
    n <- nrow(x$data)
    if(nx * ny != n)
      stop("cannot produce perspective plot, probably irregular grid")
    do.call("persp", c(list(x=x1vals, y=x2vals,
                            z=matrix(x$data[, sim.number], ncol = ny)),
                       .ldots.set(ldots, type="persp",
                                 data="simulation")))
  }
  return(invisible())
}

"contour.grf" <- 
  function(x, sim.number = 1, filled = FALSE, ...)
{
  x1vals <- unique(round(x$coords[,1], dig=12))
  x2vals <- unique(round(x$coords[,2], dig=12))
  nx <- length(x1vals)
  ny <- length(x2vals)
  ldots <- match.call(expand.dots = FALSE)$...
  if(attr(x, 'sp.dim') == "1d" | nx == 1 | ny == 1){
    do.call("plot.1d", c(list(x = x,
                              x1vals = x1vals),
                       .ldots.set(ldots, type="plot.1d",
                                 data="simulation")))
  }
  else{
    x$data <- as.matrix(x$data)
    n <- nrow(x$data)
    if(nx * ny != n)
      stop("cannot produce the countour plot, probably irregular grid")
    if(filled)
      ldots.contour <- .ldots.set(ldots, type="filled.contour",
                                 data="prediction")
    else
      ldots.contour <- .ldots.set(ldots, type="contour",
                                 data="prediction")
    if(filled){
      if(is.null(ldots.contour$plot.axes)){
        ldots.contour$plot.axes <- quote({
          axis(1)
          axis(2)
          if(!is.null(coords.data)) points(coords.data, pch=20)
          if(!is.null(borders)) polygon(borders, lwd=2)
        })
      }
      do.call("filled.contour", c(list(x=x1vals, y=x2vals,
                                       z=matrix(x$data[, sim.number], ncol = ny)),
                                  ldots.contour))
    }
    else{
      do.call("contour", c(list(x=x1vals, y=x2vals,
                                z=matrix(x$data[, sim.number], ncol = ny)),
                           ldots.contour))
    }
  }
  return(invisible())
}

"plot.grf" <-
  function (x, model.line = TRUE, plot.locations = FALSE, ...) 
{
  nsim <- ncol(x$data)
  if (plot.locations){
    points.geodata(x, pt.divide="equal", xlab = "Coord X", ylab = "Coord Y")
    if(is.null(list(...)$ask)){
      ask.now <- par()$ask
      par(ask = TRUE)
      on.exit(par(ask=ask.now)) 
    }
  }
  if (is.vector(x$cov.pars)) 
    sill.total <- x$nugget + x$cov.pars[1]
  else sill.total <- x$nugget + sum(x$cov.pars[, 1])
  if (x$lambda != 1){
    if (x$lambda == 0) data <- log(x$data)
    else data <- ((x$data^x$lambda)-1)/x$lambda
  }
  else
    data <- x$data          
  sim.bin <- variog(x, data=data)
  ldots <- list(...)
  if(is.null(ldots$ylim))
    plot(sim.bin, ylim=c(0, max(c(sill.total, sim.bin$v))),...)
  else
    plot(sim.bin, ...)    
  if (model.line){
    model <- list(nugget = x$nugget, cov.pars = x$cov.pars, 
                  kappa = x$kappa, max.dist = max(sim.bin$u),
                  cov.model = x$cov.model)
    if(is.null(ldots$lty))
      lines.variomodel(model, lty=2)
    else
      lines.variomodel(model, ...)
  }
  return(invisible())
}

"print.grf" <-
  function(x, ...)
{
  print.default(x, ...)
}

"lines.grf" <- function(x, ...){
  if(attr(x, "sp.dim") != "1d")
    stop("can only be used for simulations in  1-D")
  if(is.matrix(x$data))
    matplot(x$coords[,1], x$data, add=T, ...)
  else
    lines(x$coords[,1], x$data, ...)
  return(invisible())
}
