% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/domain.R
\name{make_domain}
\alias{make_domain}
\title{Creates a list of elements that defines the domain for a multivariate distribution.}
\usage{
make_domain(type, p, lefts = NULL, rights = NULL, ineqs = NULL, rule = NULL)
}
\arguments{
\item{type}{A string, the domain type. Currently support \code{"R"}, \code{"R+"}, \code{"uniform"}, \code{"polynomial"}, \code{"simplex"}. See details.}

\item{p}{An integer, the dimension of the domain.}

\item{lefts}{Optional, required if \code{type == "uniform"} and must have the same length as \code{rights}. A non-empty vector of numbers (may contain \code{-Inf}), the left endpoints of a domain defined as a union of intervals. It is required that \code{lefts[i] <= rights[i] <= lefts[j]} for any \code{i < j}.}

\item{rights}{Optional, required if \code{type == "uniform"} and must have the same length as \code{lefts}. A non-empty vector of numbers (may contain \code{Inf}), the right endpoints of a domain defined as a union of intervals. It is required that \code{lefts[i] <= rights[i] <= lefts[j]} for any \code{i < j}.}

\item{ineqs}{Optional, required if \code{type == "polynomial"}. A list of lists, each sublist representing an inequality that defines the domain. Each sublist must contain fields \code{abs} (logical) and \code{nonnegative} (logical), and in addition either a single \code{expression} (string), or all of the following: \code{uniform} (logical), \code{larger} (logical), \code{power_numers} (1 or \code{p} integers), \code{power_denoms} (1 or \code{p} integers), \code{const} (a number), \code{coeffs} (1 or \code{p} numbers).}

\item{rule}{Optional, required if \code{type == "polynomial" && length(ineqs) > 1}. A string containing inequality numbers, spaces, parentheses, '&' and '|' only. Used to indicate the logic operation on how to combine the domains defined by each inequality, i.e. "(1 & 2 && 3) || 4 | 5". Chained operations not separated by parentheses are only allowed for the same type of operation ('&'/'|'), i.e. "1 & 2 | 3" is not allowed; it should be either "(1 & 2) | 3" or "1 & (2 | 3)".}
}
\value{
A list containing the elements that define the domain.
For all types of domains, the following are returned.
  \item{type}{A string, same as the input.}
  \item{p}{An integer, same as the input.}
  \item{p_deemed}{An integer, equal to \code{p-1} if \code{type == "simplex"} or \code{p} otherwise.}
  \item{checked}{A logical, \code{TRUE}. Used in other functions to test whether a list is returned by this function.}
In addition,
\itemize{
   \item{For \code{type == "simplex"}, returns in addition
     \describe{
        \item{\code{simplex_tol}}{Tolerance used for simplex domains. Defaults to \code{1e-10}.}
     }}
   \item{For \code{type == "uniform"}, returns in addition
     \describe{
       \item{\code{lefts}}{A non-empty vector of numbers, same as the input.}
       \item{\code{rights}}{A non-empty vector of numbers, same as the input.}
       \item{\code{left_inf}}{A logical, indicates whether \code{lefts[1]} is \code{-Inf}.}
       \item{\code{right_inf}}{A logical, indicates whether \code{rights[length(rights)]} is \code{Inf}.}
     }}
   \item{For \code{type == "polynomial"}, returns in addition
     \describe{
       \item{\code{rule}}{A string, same as the input if provided and valid; if not provided and \code{length(ineqs) == 1}, set to \code{"1"} by default.}
      \item{\code{postfix_rule}}{A string, \code{rule} in postfix notation (reverse Polish notation) containing numbers, \code{" "}, \code{"&"} and \code{"|"} only.}
      \item{\code{ineqs}}{A list of lists, each sublist representing one inequality containing the following fields:
        \describe{
        \item{\code{uniform}}{A logical, indicates whether the inequality should be uniformly applied to all components (e.g. \code{"x>1"} -> \code{"x1>1 && ... && xp>1"}).}
        \item{\code{larger}}{A logical, indicates whether the polynomial should be larger or smaller than the constant (e.g. \code{TRUE} for \code{x1 + ... + xp > C}, and \code{FALSE} for \code{x1 + ... + xp < C}).}
        \item{\code{const}}{A number, the constant the polynomial should be compared to  (e.g. \code{2.3} for \code{x1 + ... + xp > 2.3}).}
        \item{\code{abs}}{A logical, indicates whether one should evaluate the polynomial in \code{abs(x)} instead of \code{x}.}
        \item{\code{nonnegative}}{A logical, indicates whether the domain of this inequality should be restricted to the non-negative orthant.}
        \item{\code{power_numers}}{A single integer or a vector of \code{p} integers. \code{x[i]} will be raised to the power of \code{power_numers[i] / power_denoms[i]} (or without subscript if a singer integer). Note that \code{x^(0/0)} is interpreted as \code{log(x)}, and \code{x^(n/0)} as \code{exp(n*x)} for \code{n} non-zero. For a negative \code{x}, \code{x^(a/b)} is defined as \code{(-1)^a*|x|^(a/b)} if \code{b} is odd, or \code{NaN} otherwise.}
        \item{\code{power_denoms}}{A single integer or a vector of \code{p} integers.}
        \item{\code{coeffs}}{\code{NULL} if \code{uniform == TRUE}. A vector of \code{p} doubles, where \code{coeffs[i]} is the coefficient on \code{x[i]} in the inequality}
     }}}}}
}
\description{
Creates a list of elements that define the domain for a multivariate distribution.
}
\details{
The following types of domains are supported:
\describe{
  \item{\code{"R"}}{The entire \code{p}-dimensional real space. Equivalent to \code{"uniform"} type with \code{lefts=-Inf} and \code{rights=Inf}.}
  \item{\code{"R+"}}{The non-negative orthant of the \code{p}-dimensional real space. Equivalent to \code{"uniform"} type with \code{lefts=0} and \code{rights=Inf}.}
  \item{\code{"uniform"}}{A union of finitely many disjoint intervals as a uniform domain for all components. The left endpoints should be specified through \code{lefts} and the right endpoints through \code{rights}. The intervals must be disjoint and strictly increasing, i.e. \code{lefts[i] <= rights[i] <= lefts[j]} for any \code{i < j}. E.g. \code{lefts=c(0, 10)} and \code{rights=c(5, Inf)} represents the domain ([0,5]v[10,+Inf])^p.}
  \item{\code{"simplex"}}{The standard \code{p-1}-simplex with all components positive and sum to 1, i.e. \code{sum(x) == 1} and \code{x > 0}.}
  \item{\code{"polynomial"}}{A finite intersection/union of domains defined by comparing a constant to a polynomial with at most one term in each component and no interaction terms (e.g. \code{x1^3+x1^2>1} or \code{x1*x2>1} not supported). The following is supported: \code{{x1^2 + 2*x2^(3/2) > 1} && ({3.14*x1 - 0.7*x3^3 < 1} || {-exp(3*x2) + 3.7*log(x3) + 2.4*x4^(-3/2)})}.}
     To specify a polynomial-type domain, one should define the \code{ineqs}, and in case of more than one inequality, the logical \code{rule} to combine the domains defined by each inequality.
     \item{\code{rule}}{A logical rule in infix notation, e.g. \code{"(1 && 2 && 3) || (4 && 5) || 6"}, where the numbers represent the inequality numbers starting from 1. \code{"&&"} and \code{"&"} are not differentiated, and similarly for \code{"||"} and \code{"|"}. Chained operations are only allowed for the same operation (\code{"&"} or \code{"|"}), so instead of \code{"1 && 2 || 3"} one should write either \code{"(1 && 2) || 3"} or \code{"1 && (2 || 3)"} to avoid ambiguity.}
     \item{\code{ineqs}}{A list of lists, each sublist represents one inequality, and must contain the following fields:
        \describe{
          \item{\code{abs}}{A logical, indicates whether one should evaluate the polynomial in \code{abs(x)} instead of \code{x} (e.g. \code{"sum(x) > 1"} with \code{abs == TRUE} is interpreted as \code{sum(abs(x)) > 1}).}
          \item{\code{nonnegative}}{A logical, indicates whether the domain of this inequality should be restricted to the non-negative orthant.}
        }

        In addition, one must in addition specify either a single string \code{"expression"} (highly recommended, detailed below), or all of the following fields (discouraged usage):
        \describe{
          \item{\code{uniform}}{A logical, indicates whether the inequality should be uniformly applied to all components (e.g. \code{"x>1"} -> \code{"x1>1 && ... && xp>1"}).}
          \item{\code{larger}}{A logical, indicates whether the polynomial should be larger or smaller than the constant (e.g. \code{TRUE} for \code{x1 + ... + xp > C}, and \code{FALSE} for \code{x1 + ... + xp < C}).}
          \item{\code{const}}{A number, the constant the polynomial should be compared to  (e.g. \code{2.3} for \code{x1 + ... + xp > 2.3}).}
          \item{\code{power_numers}}{A single integer or a vector of \code{p} integers. \code{x[i]} will be raised to the power of \code{power_numers[i] / power_denoms[i]} (or without subscript if a singer integer). Note that \code{x^(0/0)} is interpreted as \code{log(x)}, and \code{x^(n/0)} as \code{exp(n*x)} for \code{n} non-zero. For a negative \code{x}, \code{x^(a/b)} is defined as \code{(-1)^a*|x|^(a/b)} if \code{b} is odd, or \code{NaN} otherwise. (Example: \code{c(2,3,5,0,-2)} for \code{x1^2+2*x2^(3/2)+3*x3^(5/3)+4*log(x4)+5*exp(-2*x)>1}).}
          \item{\code{power_denoms}}{A single integer or a vector of \code{p} integers.  (Example: \code{c(1,2,3,0,0)} for \code{x1^2+2*x2^(3/2)+3*x3^(5/3)+4*log(x4)+5*exp(-2*x)>1}).}
          \item{\code{coeffs}}{Required if \code{uniform == FALSE}. A vector of \code{p} doubles, where \code{coeffs[i]} is the coefficient on \code{x[i]} in the inequality.}
       }
       The user is recommended to use a single \code{expression} for ease and to avoid potential errors. The user may safely skip the explanations and directly look at the examples to get a better understanding.\cr

       The expression should have the form \code{"POLYNOMIAL SIGN CONST"}, where \code{"SIGN"} is one of \code{"<"}, \code{"<="}, \code{">"}, \code{">="}, and \code{"CONST"} is a single number (scientific notation allowed).\cr

       \code{"POLYNOMIAL"} must be (1) a single term (see below) in \code{"x"} with no coefficient (e.g. \code{"x^(2/3)"}, \code{"exp(3x)"}), or (2) such a term surrounded by \code{"sum()"} (e.g. \code{"sum(x^(2/3))"}, \code{"sum(exp(3x))"}), or (3) a sum of such terms in \code{"x1"} through \code{"xp"} (one term max for each component) with or without coefficients, separated by the plus or the minus sign (e.g. \cr\code{"2.3x1^(2/3)-3.4exp(x2)+x3^(-3/5)"}).\cr

       For (1) and (2), the term must be in one of the following forms: \code{"x"}, \code{"x^2"}, \code{"x^(-2)"}, \code{"x^(2/3)"}, \code{"x^(-2/3)"}, \code{"log(x)"}, \code{"exp(x)"}, \code{"exp(2x)"}, \code{"exp(2*x)"}, \code{"exp(-3x)"}, where the \code{2} and \code{3} can be changed to any other non-zero integers.\cr
       For (3), each term should be as above but in \code{"x1"}, ..., \code{"xp"} instead of \code{"x"}, following an optional double number and optionally a \code{"*"} sign.\cr

       Examples: For \code{p=10}, \cr
          (1) \code{"x^2 > 2"} defines the domain \code{abs(x1) > sqrt(2) && ... && abs(x10) > sqrt(2)}.\cr
          (2) \code{"sum(x^2) > 2"} defines the domain \code{x1^2 + ... + x10^2 > 2}.\cr
          (3) \code{"2.3x3^(2/3)-3.4x4+x5^(-3/5)+3.7*x6^(-4)-1.9*log(x7)+1.3e5*exp(-3x8)}\cr
          \code{-2*exp(x9)+0.5exp(2*x10) <= 2"} defines a domain using a polynomial in \code{x3} through \code{x10}, and \code{x1} and \code{x2} are thus allowed to vary freely.\cr

       Note that \code{">"} and \code{">="} are not differentiated, and so are \code{"<"} and \code{"<="}.
   }}
}
\examples{
p <- 30
# The 30-dimensional real space R^30
domain <- make_domain("R", p=p)

# The non-negative orthant of the 30-dimensional real space, R+^30
domain <- make_domain("R+", p=p)

# x such that sum(x^2) > 10 && sum(x^(1/3)) > 10 with x allowed to be negative
domain <- make_domain("polynomial", p=p, rule="1 && 2",
       ineqs=list(list("expression"="sum(x^2)>10", abs=FALSE, nonnegative=FALSE),
                      list("expression"="sum(x^(1/3))>10", abs=FALSE, nonnegative=FALSE)))
# Alternatively
domain2 <- make_domain("polynomial", p=p, rule="1 && 2",
       ineqs=list(list(uniform=FALSE, power_numers=2, power_denoms=1, const=10, coeffs=1,
                 larger=1, abs=FALSE, nonnegative=FALSE),
                 list(uniform=FALSE, power_numers=1, power_denoms=3, const=10, coeffs=1,
                 larger=1, abs=FALSE, nonnegative=FALSE)))


# ([0, 1] v [2,3]) ^ p
domain <- make_domain("uniform", p=p, lefts=c(0,2), rights=c(1,3))

# x such that {x1 > 1 && log(1.3) < x2 < 1 && x3 > log(1.3) && ... && xp > log(1.3)}
domain <- make_domain("polynomial", p=p, rule="1 && 2 && 3",
       ineqs=list(list("expression"="x1>1", abs=FALSE, nonnegative=TRUE),
                      list("expression"="x2<1", abs=FALSE, nonnegative=TRUE),
                      list("expression"="exp(x)>1.3", abs=FALSE, nonnegative=FALSE)))
# Alternatively
domain2 <- make_domain("polynomial", p=p, rule="1 && 2",
       ineqs=list(list(uniform=FALSE, power_numers=1, power_denoms=1, const=1,
                 coeffs=c(1,rep(0,p-1)), larger=1, abs=FALSE, nonnegative=TRUE),
                 list(uniform=FALSE, power_numers=1, power_denoms=1, const=1,
                 coeffs=c(0,1,rep(0,p-2)), larger=0, abs=FALSE, nonnegative=TRUE),
                 list(uniform=TRUE, power_numers=1, power_denoms=0, const=1.3,
                 larger=1, abs=FALSE, nonnegative=FALSE)))


# x in R_+^p such that {sum(log(x))<2 || (x1^(2/3)-1.3x2^(-3)<1 && exp(x1)+2.3*x2>2)}
domain <- make_domain("polynomial", p=p, rule="1 || (2 && 3)",
       ineqs=list(list("expression"="sum(log(x))<2", abs=FALSE, nonnegative=TRUE),
                      list("expression"="x1^(2/3)-1.3x2^(-3)<1", abs=FALSE, nonnegative=TRUE),
                      list("expression"="exp(x1)+2.3*x2^2>2", abs=FALSE, nonnegative=TRUE)))
# Alternatively
domain2 <- make_domain("polynomial", p=p, rule="1 && 2",
       ineqs=list(list(uniform=FALSE, power_numers=0, power_denoms=0, const=2,
                 coeffs=1, larger=0, abs=FALSE, nonnegative=TRUE),
                 list(uniform=FALSE, power_numers=c(2,-3,rep(1,p-2)), power_denoms=c(3,rep(1,p-1)),
                 const=1, coeffs=c(1.0,-1.3,rep(0,p-2)), larger=0, abs=FALSE, nonnegative=TRUE),
                 list(uniform=FALSE, power_numers=c(1,2,rep(1,p-2)), power_denoms=c(0,rep(1,p-1)),
                 const=2, coeffs=c(1,2.3,rep(0,p-2)), larger=1, abs=FALSE, nonnegative=TRUE)))


# x in R_+^p such that {x in R_+^p: sum_j j * xj <= 1}
domain <- make_domain("polynomial", p=p,
       ineqs=list(list("expression"=paste(paste(sapply(1:p,
                           function(j){paste(j, "x", j, sep="")}), collapse="+"), "<1"),
                     abs=FALSE, nonnegative=TRUE)))
# Alternatively
domain2 <- make_domain("polynomial", p=p,
       ineqs=list(list(uniform=FALSE, power_numers=1, power_denoms=1, const=1,
                 coeffs=1:p, larger=0, abs=FALSE, nonnegative=TRUE)))


# The (p-1)-simplex
domain <- make_domain("simplex", p=p)

# The l-1 ball {sum(|x|) < 1}
domain <- make_domain("polynomial", p=p,
       ineqs=list(list("expression"="sum(x)<1", abs=TRUE, nonnegative=FALSE)))
}
