% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fapply.R
\name{fapply}
\alias{fapply}
\title{Apply formatting to a vector}
\usage{
fapply(x, format = NULL, width = NULL, justify = NULL)
}
\arguments{
\item{x}{A vector, factor, or list to apply the format to.}

\item{format}{A format to be applied.}

\item{width}{The desired character width of the formatted vector.  Default
value is NULL, meaning the vector will be variable width.}

\item{justify}{Whether to justify the return vector.  Valid values are 
'left', 'right', 'center', 'centre', or 'none'.}
}
\value{
A vector of formatted values.
}
\description{
The \code{fapply} function applies formatting to a vector.
}
\details{
The \code{fapply} function accepts several types of formats:  formatting
strings, named vectors,
vectorized functions, or user-defined formats.  It also 
accepts a formatting list, composed of any of the previous types.  
The function will first determine the type of format, and then apply 
the format in the appropriate way.  Results are returned as a vector.

The function also has parameters for width and justification.

Parameters may also be passed as attributes on the vector.  See 
the \code{\link{fattr}} function for additional information on setting
formatting attributes.
}
\section{Types of Formats}{

The \code{fapply} function will process any of the following types of
formats:
\itemize{
  \item{\strong{Formatting string:} A single string will be interpreted as 
a formatting string.  See the \link{FormattingStrings} documentation for 
further details.}
  \item{\strong{Named vector:} A named vector can serve as a lookup list or 
  decode 
for a vector.  You can use a named vector to perform simple 
lookups on character vectors.}
  \item{\strong{Format object:} A format object may be created using the 
\code{\link{value}} function.  The format object is included in the 
\strong{fmtr} package, and is specially designed for data categorization.}  
  \item{\strong{Vectorized formatting function:} A vectorized function 
  provides
the most flexibility and power over your formatting.  You can use 
an existing formatting function from any package, or create 
your own vectorized formatting function using \code{\link[base]{Vectorize}}.}
}

\code{fapply} will also accept a formatting list, which can contain any 
number of formats from the above list.  To create a formatting list, 
see the \code{\link{flist}} function.
}

\examples{
## Example 1: Formatting string ##
v1 <- c(1.235, 8.363, 5.954, 2.465)

# Apply string format.
fapply(v1, "\%.1f")
# [1] "1.2" "8.4" "6.0" "2.5"

## Example 2: Named vector ##
# Set up vector
v2 <- c("A", "B", "C", "B")

# Set up named vector for formatting
fmt2 <- c(A = "Label A", B = "Label B", C = "Label C")

# Apply format to vector
fapply(v2, fmt2)
# [1] "Label A" "Label B" "Label C" "Label B"

## Example 3: User-defined format ##
# Define format
fmt3 <- value(condition(x == "A", "Label A"),
              condition(x == "B", "Label B"), 
              condition(TRUE, "Other"))
              
# Apply format to vector
fapply(v2, fmt3)
# [1] "Label A" "Label B" "Other"   "Label B"

## Example 4: Formatting function ##
# Set up vectorized function
fmt4 <- Vectorize(function(x) {

  if (x \%in\% c("A", "B"))
    ret <- paste("Label", x)
  else
    ret <- "Other" 
    
  return(ret)
})

# Apply format to vector
fapply(v2, fmt4)
# [1] "Label A" "Label B" "Other"   "Label B"

## Example 5: Formatting List - Row Type ##
# Set up data
# Notice each row has a different data type
v3 <- list(2841.258, "H", Sys.Date(),
           "L", Sys.Date() + 60, 1382.8865)
v4 <- c("int", "char", "date", "char", "date", "int")

# Create formatting list
lst <- flist(type = "row", lookup = v4,
       int = function(x) format(x, digits = 2, nsmall = 1, 
                                  big.mark=","),
       char = value(condition(x == "H", "High"),
                     condition(x == "L", "Low"),
                     condition(TRUE, "NA")),
       date = "\%d\%b\%Y")

# Apply formatting list to vector
fapply(v3, lst)
# [1] "2,841.3"   "High"      "06Jan2024" "Low"       "06Mar2024" "1,382.9"  

## Example 6: Formatting List - Column Type ##
# Set up data
v5 <- c(Sys.Date(), Sys.Date() + 30, Sys.Date() + 60)
v5
# [1] "2024-01-06" "2024-02-05" "2024-03-06"

# Create formatting list
lst <- flist("\%B", "This month is: \%s", type = "column")

# Apply formatting list to vector
fapply(v5, lst)
# [1] "This month is: January"  "This month is: February" "This month is: March" 

# Example 7: Conditional Formatting
# Data vector
v6 <- c(8.38371, 1.46938, 3.28783, NA, 0.98632)

# User-defined format
fmt5 <- value(condition(is.na(x), "Missing"),
              condition(x < 1, "Low"), 
              condition(x > 5, "High"),
              condition(TRUE, "\%.2f"))

# Apply format to data vector
fapply(v6, fmt5)
# [1] "High"    "1.47"    "3.29"    "Missing" "Low" 
}
\seealso{
\code{\link{fcat}} to create a format catalog,
\code{\link{value}} to define a format, 
\code{\link{fattr}} to easily set the formatting attributes of a vector, 
and \code{\link{flist}} to define a formatting list.  Also see 
\code{\link{fdata}} to apply formats to an entire data frame, and 
\link{FormattingStrings} for how to define a formatting string.
}
