% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_mvst.R
\name{fit_mvst}
\alias{fit_mvst}
\title{Estimate parameters of a multivariate (generalized hyperbolic) skewed t distribution to fit data}
\usage{
fit_mvst(
  X,
  nu = NULL,
  gamma = NULL,
  initial = NULL,
  max_iter = 500,
  ptol = 0.001,
  ftol = Inf,
  PXEM = TRUE,
  return_iterates = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{X}{Data matrix containing the multivariate time series (each column is one time series).}

\item{nu}{Degrees of freedom of the skewed \eqn{t} distribution (otherwise it will be iteratively estimated).}

\item{gamma}{Skewness vector of the skewed \eqn{t} distribution (otherwise it will be iteratively estimated).}

\item{initial}{List of initial values of the parameters for the iterative estimation method.
Possible elements include:
\itemize{\item{\code{nu}: default is \code{4},}
         \item{\code{mu}: default is the data sample mean,}
         \item{\code{gamma}: default is the sample skewness vector,}
         \item{\code{scatter}: default follows from the scaled sample covariance matrix,}
         }}

\item{max_iter}{Integer indicating the maximum number of iterations for the iterative estimation
method (default is \code{500}).}

\item{ptol}{Positive number indicating the relative tolerance for the change of the variables
to determine convergence of the iterative method (default is \code{1e-3}).}

\item{ftol}{Positive number indicating the relative tolerance for the change of the log-likelihood
value to determine convergence of the iterative method (default is \code{Inf}, so it is
not active). Note that using this argument might have a computational cost as a convergence
criterion due to the computation of the log-likelihood (especially when \code{X} is high-dimensional).}

\item{PXEM}{Logical value indicating whether to use the parameter expansion (PX) EM method to accelerating the convergence.}

\item{return_iterates}{Logical value indicating whether to record the values of the parameters (and possibly the
log-likelihood if \code{ftol < Inf}) at each iteration (default is \code{FALSE}).}

\item{verbose}{Logical value indicating whether to allow the function to print messages (default is \code{FALSE}).}
}
\value{
A list containing (possibly) the following elements:
        \item{\code{mu}}{Location vector estimate (not the mean).}
        \item{\code{gamma}}{Skewness vector estimate.}
        \item{\code{scatter}}{Scatter matrix estimate.}
        \item{\code{nu}}{Degrees of freedom estimate.}
        \item{\code{mean}}{Mean vector estimate: \preformatted{  mean = mu + nu/(nu-2) * gamma}}
        \item{\code{cov}}{Covariance matrix estimate: \preformatted{  cov = nu/(nu-2) * scatter + 2*nu^2 / (nu-2)^2 / (nu-4) * gamma*gamma'}}
        \item{\code{converged}}{Boolean denoting whether the algorithm has converged (\code{TRUE}) or the maximum number
                                of iterations \code{max_iter} has been reached (\code{FALSE}).}
        \item{\code{num_iterations}}{Number of iterations executed.}
        \item{\code{cpu_time}}{Elapsed overall CPU time.}
        \item{\code{log_likelihood_vs_iterations}}{Value of log-likelihood over the iterations (if \code{ftol < Inf}).}
        \item{\code{iterates_record}}{Iterates of the parameters (\code{mu}, \code{scatter}, \code{nu},
                                      and possibly \code{log_likelihood} (if \code{ftol < Inf})) along the iterations
                                      (if \code{return_iterates = TRUE}).}
        \item{\code{cpu_time_at_iter}}{Elapsed CPU time at each iteration (if \code{return_iterates = TRUE}).}
}
\description{
Estimate parameters of a multivariate (generalized hyperbolic) skewed Student's t distribution to fit data,
namely, the location vector, the scatter matrix, the skewness vector, and the degrees of freedom.
The estimation is based on the maximum likelihood estimation (MLE) and the algorithm is
obtained from the expectation-maximization (EM) method.
}
\details{
This function estimates the parameters of a (generalized hyperbolic) multivariate Student's t distribution (\code{mu},
         \code{scatter}, \code{gamma} and \code{nu}) to fit the data via the expectation-maximization (EM) algorithm.
}
\examples{
library(mvtnorm)       # to generate heavy-tailed data
library(fitHeavyTail)

# parameter setting
N <- 5
T <- 200
nu <- 6
mu <- rnorm(N)
scatter <- diag(N)
gamma <- rnorm(N)   # skewness vector

# generate GH Skew t data
taus <- rgamma(n = T, shape = nu/2, rate = nu/2)
X <- matrix(data = mu, nrow = T, ncol = N, byrow = TRUE) +
     matrix(data = gamma, nrow = T, ncol = N, byrow = TRUE) / taus +
     rmvnorm(n = T, mean = rep(0, N), sigma = scatter) / sqrt(taus)

# fit skew t model
fit_mvst(X)

# setting lower limit for nu (e.g., to guarantee existence of co-skewness and co-kurtosis matrices)
options(nu_min = 8.01)
fit_mvst(X)

}
\references{
Aas Kjersti and Ingrid Hobæk Haff. "The generalized hyperbolic skew Student’s t-distribution,"
Journal of financial econometrics, pp. 275-309, 2006.
}
\seealso{
\code{\link{fit_mvt}}
}
\author{
Rui Zhou, Xiwen Wang, and Daniel P. Palomar
}
