#' computeDuration
#
#' @description computeDuration computes the duration between when an ESM questionnaire was started and when it was finished.
#
#' @param esDf a data.frame. A single ESM dataset. It must contain the 2 columns that hold the date-time object for when an ESM questionnaire was started and finished, respectively.
#
#' @param RELEVANTVN_ES a list. This list is generated by function \code{\link{setES}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @return \code{esDf} with the additional column DUR (short for duration).
#
#' @importFrom lubridate as.interval ymd_hms as.period parse_date_time
#
#' @examples
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#' # Prerequisites in order to execute computeDuration. Start ----------
#' # RELEVANTINFO_ES is delivered with the package
#' # Use example list delivered with the package
#' RELEVANTVN_ES <- RELEVANTVN_ESext
#' intoleranceDf <- data.frame(prompt = c(2, 3, 4, 1, 1),
#' expect = c(1, 1, 1, 2, 3))
#' # expectedDf is a raw ESM dataset, delivered with the package.
#' intolLs <- intolerable(expectedDf, intoleranceDf, RELEVANTINFO_ES)
#' randSelLs <- randomMultSelection(intolLs[["cleanedDf"]])
#' # Prerequisites in order to execute computeDuration. End ------------
#' # -------------------------------------------------------
#' # Run function 26 of 28; see esmprep functions' hierarchy.
#' # -------------------------------------------------------
#' # randSelLs[["esRandSelIn"]] is the result of function 'randomMultSelection'.
#' durDf <- computeDuration(randSelLs[["esRandSelIn"]], RELEVANTVN_ES)
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#
#' @seealso Exemplary code (fully executable) in the documentation of \code{\link{esmprep}} (function 26 of 28).
#
#' @export
#
computeDuration <- function(esDf, RELEVANTVN_ES=NULL) {
	
	# Possible errors when passing arguments to the function -----------------------------
    if(!is.data.frame(esDf)) {
        stop("Argument 'esDf' must be of type data.frame.")
    }
	
	# Error handling function for all set-up lists generated by setES and setREF.
    # Both lists RELEVANTVN_ES and RELEVANTVN_REF get extended either by function
    # genDateTime or by function splitDateTime!
    SETUPLISTCheck(RELEVANTINFO_ES=NULL,
    			   RELEVANTVN_ES=RELEVANTVN_ES,
    			   RELEVANTVN_REF=NULL)
	
	
    if(any(is.na(match(c(RELEVANTVN_ES[["ES_START_DATETIME"]],
                         RELEVANTVN_ES[["ES_END_DATETIME"]]),
                       names(esDf))))) {
        stop(paste0("In order to compute the time lag the variables ",
                    RELEVANTVN_ES[["ES_START_DATETIME"]], " and ",
                    RELEVANTVN_ES[["ES_END_DATETIME"]],
                    " must be part of the data.frame that is passed to this function."))
    }

    spanDur <- lubridate::as.interval(lubridate::ymd_hms(esDf[,RELEVANTVN_ES[["ES_START_DATETIME"]]]), lubridate::ymd_hms(esDf[,RELEVANTVN_ES[["ES_END_DATETIME"]]]))
    periodDur <- lubridate::as.period(spanDur)

    durFormat <- suppressWarnings(lubridate::parse_date_time(paste0(periodDur@hour, ":", periodDur@minute, ":", periodDur@.Data), "%H%M%S"))
    timeDur <- format(durFormat, "%H:%M:%S")

    esDf[,"DUR"] <- timeDur

    return(esDf)
}
