% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run.R
\name{run_idf}
\alias{run_idf}
\alias{run_multi}
\title{Run simulations of EnergyPlus models.}
\usage{
run_idf(
  model,
  weather,
  output_dir,
  design_day = FALSE,
  annual = FALSE,
  expand_obj = TRUE,
  wait = TRUE,
  echo = TRUE,
  eplus = NULL
)

run_multi(
  model,
  weather,
  output_dir,
  design_day = FALSE,
  annual = FALSE,
  expand_obj = TRUE,
  wait = TRUE,
  echo = TRUE,
  eplus = NULL
)
}
\arguments{
\item{model}{A path (for \code{run_idf()}) or a vector of paths (for
\code{run_multi()}) of EnergyPlus IDF or IMF files.}

\item{weather}{A path (for \code{run_idf()}) or a vector of paths (for
\code{run_multi()}) of EnergyPlus EPW weather files.
If set to \code{NULL}, design-day-only simulation will be triggered,
regardless of the \code{design-day} value.
For \code{run_multi()}, \code{weather} can also be a single EPW file path. In
this case, that weather will be used for all simulations; otherwise,
\code{model} and \code{weather} should have the same length. You can set to
design-day-only simulation to some specific simulations by setting the
corresponding weather to \code{NA}.}

\item{output_dir}{Output directory path (for \code{rum_idf()}) or paths (for
\code{run_mult()}). If NULL, the directory of input model is used. For
\code{run_multi()}, \code{output_dir}, if not \code{NULL}, should have the same
length as \code{model}. Any duplicated combination of \code{model} and
\code{output_dir} is prohibited.}

\item{design_day}{Force design-day-only simulation. For \code{rum_multi()},
\code{design_day} can also be a logical vector which has the same length as
\code{model}. Note that \code{design_day} and \code{annual} cannot be all \code{TRUE} at
the same time. Default: \code{FALSE}.}

\item{annual}{Force annual simulation. For \code{rum_multi()},
\code{annual} can also be a logical vector which has the same length as
\code{model}. Note that \code{design_day} and \code{annual} cannot be all \code{TRUE} at
the same time. Default: \code{FALSE}.}

\item{expand_obj}{Whether to run ExpandObject preprocessor before simulation.
Default: TRUE.}

\item{wait}{If \code{TRUE}, R will hang on and wait all EnergyPlus simulations
finish. If \code{FALSE}, all EnergyPlus simulations are run in the
background, and a \link[processx:process]{process} object is returned.}

\item{echo}{Only applicable when \code{wait} is \code{TRUE}. Whether to show standard
output and error from EnergyPlus for \code{run_idf()} and simulation status
for \code{run_multi()}. Default: \code{TRUE}.}

\item{eplus}{An acceptable input (for \code{run_idf()}) or inputs (for
\code{run_multi()}) of \code{\link[=use_eplus]{use_eplus()}} and \code{\link[=eplus_config]{eplus_config()}}. If \code{NULL}, which
is the default, the version of EnergyPlus to use is determined by the
version of input model. For \code{run_multi()}, \code{eplus}, if not \code{NULL},
should have the same length as \code{model}.}
}
\value{
\itemize{
\item For \code{run_idf()}, if \code{wait} is \code{TRUE}, a named list of 11 elements:
}\tabular{rlll}{
   No. \tab Column \tab Type \tab Description \cr
   1 \tab \code{idf} \tab \code{character(1)} \tab Full path of input IDF file \cr
   2 \tab \code{epw} \tab \code{character(1)} or \code{NULL} \tab Full path of input EPW file \cr
   3 \tab \code{version} \tab \code{character(1)} \tab Version of called EnergyPlus \cr
   4 \tab \code{exit_status} \tab \code{integer(1)} or \code{NULL} \tab Exit status of EnergyPlus. \code{NULL} if terminated or \code{wait} is \code{FALSE} \cr
   5 \tab \code{start_time} \tab \code{POSIXct(1)} \tab Start of time of simulation \cr
   6 \tab \code{end_time} \tab \code{POSIXct(1)} or \code{NULL} \tab End of time of simulation. \code{NULL} if \code{wait} is \code{FALSE} \cr
   7 \tab \code{output_dir} \tab \code{character(1)} \tab Full path of simulation output directory \cr
   8 \tab \code{energyplus} \tab \code{character(1)} \tab Full path of called EnergyPlus executable \cr
   9 \tab \code{stdout} \tab \code{character(1)} or \code{NULL} \tab Standard output of EnergyPlus during simulation \cr
   10 \tab \code{stderr} \tab \code{character(1)} or \code{NULL} \tab Standard error of EnergyPlus during simulation \cr
   11 \tab \code{process} \tab \link[callr:r_bg]{r_process} \tab A process object which called EnergyPlus and ran the simulation \cr
}


If \code{wait} is \code{FALSE}, the \link[callr:r_bg]{R process} is directly returned.
You can get the results by calling \code{result <- proc$get_result()} (\code{proc} is
the returned process). Please note that in this case, \code{result$process} will
alwasy be \code{NULL}. But you can easily assign it by running \code{result$process <- proc}
\itemize{
\item For \code{rum_multi()}, if \code{wait} is TRUE, a
\link[data.table:data.table]{data.table} of 12 columns:\tabular{rlll}{
   No. \tab Column \tab Type \tab Description \cr
   1 \tab \code{index} \tab \code{integer} \tab Index of simulation \cr
   2 \tab \code{status} \tab \code{character} \tab Simulation status \cr
   3 \tab \code{idf} \tab \code{character} \tab Full path of input IDF file \cr
   4 \tab \code{epw} \tab \code{character} \tab Full path of input EPW file. \code{NA} for design-day-only simulation \cr
   5 \tab \code{version} \tab \code{character} \tab Version of EnergyPlus \cr
   6 \tab \code{exit_status} \tab \code{integer} \tab Exit status of EnergyPlus. \code{NA} if terminated \cr
   7 \tab \code{start_time} \tab \code{POSIXct} \tab Start of time of simulation \cr
   8 \tab \code{end_time} \tab \code{POSIXct} \tab End of time of simulation. \cr
   9 \tab \code{output_dir} \tab \code{character} \tab Full path of simulation output directory \cr
   10 \tab \code{energyplus} \tab \code{character} \tab Full path of called EnergyPlus executable \cr
   11 \tab \code{stdout} \tab \code{list} \tab Standard output of EnergyPlus during simulation \cr
   12 \tab \code{stderr} \tab \code{list} \tab Standard error of EnergyPlus during simulation \cr
}


For column \code{status}, there are 4 possible values:
\itemize{
\item \code{"completed"}: the simulation job is completed successfully
\item \code{"failed"}: the simulation job ended with error
\item \code{"terminated"}: the simulation job started but was terminated
\item \code{"cancelled"}: the simulation job was cancelled, i.e. did not start at all
}
\item For \code{run_multi()}, if \code{wait} is \code{FALSE}, a \link[callr:r_bg]{r_process}
object of background R process which handles all simulation jobs is
returned. You can check if the jobs are completed using \verb{$is_alive()} and
get the final data.table using \verb{$get_result()} method.
}
}
\description{
Run simulations of EnergyPlus models.
}
\details{
\code{run_idf()} is a wrapper of EnergyPlus itself, plus various pre-processors
and post-processors which enables to run EnergyPlus model with different
options.

\code{run_multi()} provides the functionality of running multiple models in
parallel.

It is suggested to run simulations using \link{EplusJob} class and \link{EplusGroupJob}
class, which provide much more detailed controls on the simulation and also
methods to extract simulation outputs.
}
\note{
If your input model has external file dependencies, e.g. FMU, schedule files,
etc. \code{run_idf()} and \code{run_multi()} will not work if the output directory is
different that where the input mode lives. If this is the case, parse the
model using \code{\link[=read_idf]{read_idf()}} and use \link[=Idf]{Idf$run()} or \code{\link[=eplus_job]{eplus_job()}} instead.
They are able to automatically change the paths of external files to absolute
paths or copy them into the output directory, based on your choice.
}
\examples{
\dontrun{
idf_path <- system.file("extdata/1ZoneUncontrolled.idf", package = "eplusr")

if (is_avail_eplus(8.8)) {
    # run a single model
    epw_path <- file.path(
        eplus_config(8.8)$dir,
        "WeatherData",
        "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"
    )

    run_idf(idf_path, epw_path, output_dir = tempdir())

    # run multiple model in parallel
    idf_paths <- file.path(eplus_config(8.8)$dir, "ExampleFiles",
        c("1ZoneUncontrolled.idf", "1ZoneUncontrolledFourAlgorithms.idf")
    )
    epw_paths <- rep(epw_path, times = 2L)
    output_dirs <- file.path(tempdir(), tools::file_path_sans_ext(basename(idf_paths)))
    run_multi(idf_paths, epw_paths, output_dir = output_dirs)
}
}
}
\references{
\href{https://github.com/NREL/EnergyPlus/blob/develop/doc/running-energyplus-from-command-line.md}{Running EnergyPlus from Command Line (EnergyPlus GitHub Repository)}
}
\seealso{
\link{EplusJob} class and \link{ParametricJob} class which provide a more friendly
interface to run EnergyPlus simulations and collect outputs.
}
\author{
Hongyuan Jia
}
