% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/P_TRT.R
\name{P_TRT}
\alias{P_TRT}
\title{Estimate the transitional range of temperatures based on a set of parameters}
\usage{
P_TRT(x = NULL, resultmcmc = NULL, chain = 1, equation = NULL,
  l = 0.05, replicate.CI = NULL, temperatures = NULL,
  durations = NULL, SD.temperatures = NULL, SD.durations = NULL,
  probs = c(0.025, 0.5, 0.975), warn = TRUE)
}
\arguments{
\item{x}{Set of parameters or a result of tsd()}

\item{resultmcmc}{A result of tsd_MHmcmc()}

\item{chain}{What chain to be used is resultmcmc is provided}

\item{equation}{What equation should be used. Must be provided if x is not a result of tsd()}

\item{l}{Sex ratio limits to define TRT are l and 1-l. If NULL, TRT is not estimated.}

\item{replicate.CI}{If a result of tsd() is provided, use replicate.CI replicates from the hessian matrix to estimate CI}

\item{temperatures}{If provided returns the sex ratio and its quantiles for each temperature}

\item{durations}{If provided returns the sex ratio and its quantiles for each duration}

\item{SD.temperatures}{SD of temperatures}

\item{SD.durations}{SD of durations}

\item{probs}{Probabilities used to estimate quantiles}

\item{warn}{Do the warnings must be shown ? TRUE or FALSE}
}
\value{
A list with a P_TRT object containing a matrix with lower and higher bounds for TRT, TRT and P and a P_TRT_quantiles object with quantiles for each and a sexratio_quantiles object
}
\description{
Estimate the parameters that best describe the thermal reaction norm for sex ratio when temperature-dependent sex determination occurs.\cr
It can be used also to evaluate the relationship between incubation duration and sex ratio.\cr
The parameter l was defined in Girondot (1999). The TRT is defined from the difference between the two boundary temperatures giving sex ratios of l and 1 - l.\cr
For logistic equation, exact value is used and precision iterations are used for other equations.\cr
In Girondot (1999), l was 0.05 and then the TRT was defined as being the range of temperatures producing from 5% to 95% of each sex.\cr
If l is null, TRT is not estimated and only sex ratio is estimated.\cr
}
\details{
P_TRT estimates the transitional range of temperatures based on a set of parameters
}
\examples{
\dontrun{
library("embryogrowth")
CC_AtlanticSW <- subset(DatabaseTSD, RMU=="Atlantic, SW" & 
                          Species=="Caretta caretta" & Sexed!=0)
tsdL <- with (CC_AtlanticSW, tsd(males=Males, females=Females, 
                                 temperatures=Incubation.temperature-Correction.factor, 
                                 equation="logistic"))
P_TRT(tsdL)
P_TRT(tsdL, replicate.CI=1000)
P_TRT(tsdL, replicate.CI=1000, temperatures=20:35)
P_TRT_out <- P_TRT(tsdL, replicate.CI=1000, temperatures=c(T1=20, T2=35))
attributes(P_TRT_out$sexratio_quantiles)$temperatures
P_TRT(tsdL$par, equation="logistic")
pMCMC <- tsd_MHmcmc_p(tsdL, accept=TRUE)
# Take care, it can be very long
result_mcmc_tsd <- tsd_MHmcmc(result=tsdL, 
		parametersMCMC=pMCMC, n.iter=10000, n.chains = 1,  
		n.adapt = 0, thin=1, trace=FALSE, adaptive=TRUE)
P_TRT(result_mcmc_tsd, equation="logistic")
}
}
\references{
Girondot, M. 1999. Statistical description of temperature-dependent sex determination using maximum likelihood. Evolutionary Ecology Research, 1, 479-486.

Godfrey, M.H., Delmas, V., Girondot, M., 2003. Assessment of patterns of temperature-dependent sex determination using maximum likelihood model selection. Ecoscience 10, 265-272.

Hulin, V., Delmas, V., Girondot, M., Godfrey, M.H., Guillon, J.-M., 2009. Temperature-dependent sex determination and global change: are some species at greater risk? Oecologia 160, 493-506.
}
\seealso{
Other Functions for temperature-dependent sex determination: \code{\link{DatabaseTSD}},
  \code{\link{TSP.list}}, \code{\link{plot.tsd}},
  \code{\link{predict.tsd}}, \code{\link{stages}},
  \code{\link{tsd_MHmcmc_p}}, \code{\link{tsd_MHmcmc}},
  \code{\link{tsd}}
}
\author{
Marc Girondot
}
\concept{Functions for temperature-dependent sex determination}
