#' Calculate statistics about nests
#' @title Calculte statistics about nests
#' @author Marc Girondot
#' @return A list with informations about thermosensitive period length for the series of nests analyzed
#' @param x A result file generated by searchR
#' @param parameters A set of parameters if result is not provided.
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param SE Standard error for each parameter if result is not provided.
#' @param temperatures Timeseries of temperatures. Will replace the one in result.
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings
#' @param stopattest True or False, does the plot stops when proxi of size reaches the mean test size.
#' @param M0 Measure of hatchling size proxi at laying date
#' @param series The name or number of the series to be displayed. Only one series can be displayed at a time.
#' @param TSP.borders The limits of TSP
#' @param embryo.stages The embryo stages. At least TSP.borders stages must be provided if show.TSP is TRUE
#' @param ref.stage If TRUE, the reference size for stage is the observed one in modelling. If FALSE, it is the field observed size.
#' @param replicate.CI Number of randomizations to estimate CI
#' @param progress If FALSE, the progress bar is not shown (useful for use with Sweave)
#' @description Calculate statistics about nests\cr
#' The embryo.stages is a list with stage numbers and relative size as compared to final size at the beginning of the stage.\cr
#' For example embryo.stages=list(number=21:30, size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33) indicates that the stages 21 begins at the relative size of 8.4/39.33 series can be indicated as the name of the series, its number or TRUE or FALSE. "all" indicates that all series must be printed.
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p)
#' info.nests(resultNest_4p, series=1,  
#' 	SE=c(DHA=1.396525, DHH=4.101217, T12H=0.04330405, Rho25=1.00479))
#' # to get info for all nests at the same time, use
#' infoall <- info.nests(resultNest_4p, series="all")
#' # it is easier to use after dataframe conversion
#' infoall.df <- as.data.frame(infoall$summary)
#' }
#' @export


info.nests <-
function(x, parameters=NULL, fixed.parameters=NULL, SE=NULL, temperatures=NULL, derivate=NULL, 
	test=NULL, stopattest=FALSE, M0=NULL, series="all", TSP.borders=c(21, 26), progress=TRUE,
			embryo.stages="Caretta caretta", 
				replicate.CI=100, ref.stage=TRUE) {
  
#  parameters=NULL; fixed.parameters=NULL; SE=NULL; temperatures=NULL; derivate=NULL
#  test=NULL; stopattest=FALSE; M0=NULL; series="all"; TSP.borders=c(21, 26); progress=TRUE
#  embryo.stages="Caretta caretta"
#  replicate.CI=100; ref.stage=TRUE
#  x <- resultNest_4p; parameters=fitnewp; fixed.parameters = pfixed; temperatures=formated.Georges; stopattest = TRUE; progress=TRUE; replicate.CI = 0
				
if (class(x)!="NestsResult") {
	print("A object NestsResult obtained after searchR() must be given.")

} else {

if (embryo.stages[1]=="Caretta caretta") 
	embryo.stages <- list(number=21:30, 
			size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33)



out <- plot(x, parameters=parameters, fixed.parameters=fixed.parameters, SE=SE, temperatures=temperatures, derivate=derivate, 
	test=test, stopattest=stopattest, M0=M0, series=series, TSP.borders=TSP.borders, progress=progress,
			embryo.stages=embryo.stages, replicate.CI=replicate.CI, ref.stage=ref.stage, show.plot=FALSE)

return(invisible(out))
}

}
