#' @title
#' Implementation of the NSGA-II EMOA algorithm by Deb.
#'
#' @description
#' The NSGA-II merges the current population and the generated offspring and
#' reduces it by means of the following procedure: It first applies the non
#' dominated sorting algorithm to obtain the nondominated fronts. Starting with
#' the first front, it fills the new population until the i-th front does not fit.
#' It then applies the secondary crowding distance criterion to select the missing
#' individuals from the i-th front.
#'
#' @note
#' This is a pure R implementation of the NSGA-II algorithm. It hides the regular
#' \pkg{ecr} interface and offers a more R like interface while still being quite
#' adaptable.
#'
#' @references
#' Deb, K., Pratap, A., and Agarwal, S. A Fast and Elitist Multiobjective Genetic
#' Algorithm: NSGA-II. IEEE Transactions on Evolutionary Computation, 6 (8) (2002),
#' 182-197.
#'
#' @keywords optimize
#'
#' @template arg_optimization_task
#' @param n.population [\code{integer(1)}]\cr
#'   Population size. Default is \code{100}.
#' @param n.offspring [\code{integer(1)}]\cr
#'   Offspring size, i.e., number of individuals generated by variation operators
#'   in each iteration. Default is \code{n.population}.
#' @template arg_parent_selector
#' @template arg_mutator
#' @template arg_recombinator
#' @template arg_max_iter
#' @template arg_max_evals
#' @template arg_max_time
#' @param ... [any]\cr
#'   Further arguments passed to \code{\link{setupECRControl}}.
#' @return [\code{ecr_nsga2_result, ecr_multi_objective_result}]
#' @export
nsga2 = function(
  task,
  n.population = 100L, n.offspring = n.population,
  parent.selector = setupSimpleSelector(),
  mutator = setupGaussMutator(),
  recombinator = setupCrossoverRecombinator(),
  max.iter = 100L,
  max.evals = NULL,
  max.time = NULL,
  ...) {

  if (isSmoofFunction(task)) {
    task = makeOptimizationTask(task)
  }
  assertClass(task, "ecr_optimization_task")

  # NSGA-II control object
  ctrl = setupECRControl(
    n.population = n.population,
    n.offspring = n.offspring,
    representation = "float",
    stopping.conditions = list(
      setupMaximumEvaluationsTerminator(max.evals),
      setupMaximumTimeTerminator(max.time),
      setupMaximumIterationsTerminator(max.iter)
    ),
    ...
  )
  ctrl = setupEvolutionaryOperators(
    ctrl,
    parent.selector = parent.selector,
    recombinator = recombinator,
    mutator = mutator,
    survival.selector = setupNondomSelector()
  )

  res = doTheEvolution(task, ctrl)
  res = BBmisc::addClasses(res, "ecr_nsga2_result")
  return(res)
}
