\name{plotres}

\alias{plotres}

\title{
Plot Simulation Results
}

\description{
Produces a simple standardized plot of the results generated by \code{\link{calcres}} 
or, more generally, 
plots of all or selected columns of a matrix as a function of the row names interpreted as numbers.
}

\usage{
plotres(res,colnames=list(),
        file=NA,width=7,height=7,
        ncol=NA,nrow=NA,
        lwd=2,
        col=c("black","red","blue","pink","orange","violet","cyan","brown","purple"),
        lty=c("solid",paste(2*4:1,2,sep=""),paste(paste(2*4:2,2,sep=""),22,sep="")),
        main=NA,xlab=NA,ylab=NA)
}

\arguments{
  \item{res}{
Numerical matrix, list of numerical matrices or list of lists of numerical matrices of data to be plotted.
If \code{res} is a single numerical matrix, all columns or the columns selected under \code{colnames} are plotted 
versus the row names that are converted to numerics.
If \code{res} is a list of numerical matrices, the same plots are produced as for a single numerical matrix, 
but in all plots multiple lines are produced from the different list entries.
In the legend, the column names are then appended by the names of the lists.
If \code{res} is a list of lists of numerical matrices, a set of plots according to the description above are produced.
Note that the format of \code{res} is compatible with the output of the member functions \code{\link{calcres}} and
\code{\link{calcsens}} of the class \code{\link{system-class}}.
}
  \item{colnames}{
Selection of column names to be plotted. 
Each list element defines a plot and may contain a vector of column names in which case multiple lines are plotted
into the same plot.
An empty list indicates that all columns should be plotted.
}
  \item{file}{
Optional file name to which the plot should be redirected. The file will be written in pdf format.
}
  \item{width}{
Optional width of the graphics region of the pdf in inches.
}
  \item{height}{
Optional height of the graphics region of the pdf in inches.
}
  \item{ncol}{
Optional number of columns of plots to be plotted on the same page.
}
  \item{nrow}{
Optional number of rows of plots to be plotted on the same page.
}
  \item{lwd}{
Optional line width(s); values are recycled if more lines are plotted then line widths provided.
}
  \item{lty}{
Optional line type(s); values are recycled if more lines are plotted then line types provided.
}
  \item{col}{
Optional line color(s); values are recycled if more lines are plotted then line colors provided.
}
  \item{main}{
Optional title(s) of plots; values are recycled if more plots are produced than titles provided.
}
  \item{xlab}{
Optional label(s) of the x axes; values are recycled if more plots are produced than labels provided.
}
  \item{ylab}{
Optional label(s) of the y axes; values are recycled if more plots are produced than labels provided.
}
}

\author{
Peter Reichert <peter.reichert@eawag.ch>
}

\seealso{
\code{\link{process-class}},
\code{\link{reactor-class}}
\code{\link{link-class}}
\code{\link{system-class}}
\code{\link{calcres}}.
}

\examples{
# Definition of parameters:
# =========================

param    <- list(k.gro.ALG   = 1,        # 1/d
                 k.gro.ZOO   = 0.8,      # m3/gDM/d
                 k.death.ALG = 0.4,      # 1/d
                 k.death.ZOO = 0.08,     # 1/d
                 K.HPO4      = 0.002,    # gP/m3
                 Y.ZOO       = 0.2,      # gDM/gDM
                 alpha.P.ALG = 0.002,    # gP/gDM
                 A           = 8.5e+006, # m2
                 h.epi       = 4,        # m
                 Q.in        = 4,        # m3/s
                 C.ALG.ini   = 0.05,     # gDM/m3
                 C.ZOO.ini   = 0.1,      # gDM/m3
                 C.HPO4.ini  = 0.02,     # gP/m3
                 C.HPO4.in   = 0.04)     # gP/m3             

# Definition of transformation processes:
# =======================================

# Growth of algae:
# ----------------

gro.ALG   <- new(Class  = "process",
                 name   = "Growth of algae",
                 rate   = expression(k.gro.ALG
                                     *C.HPO4/(K.HPO4+C.HPO4)
                                     *C.ALG),
                 stoich = list(C.ALG  = expression(1),              # gDM/gDM
                               C.HPO4 = expression(-alpha.P.ALG)))  # gP/gDM

# Death of algae:
# ---------------

death.ALG <- new(Class = "process",
                 name   = "Death of algae",
                 rate   = expression(k.death.ALG*C.ALG),
                 stoich = list(C.ALG  = expression(-1)))            # gDM/gDM

# Growth of zooplankton:
# ----------------------

gro.ZOO   <- new(Class  = "process",
                 name   = "Growth of zooplankton",
                 rate   = expression(k.gro.ZOO
                                     *C.ALG
                                     *C.ZOO),
                 stoich = list(C.ZOO  = expression(1),              # gDM/gDM
                               C.ALG  = expression(-1/Y.ZOO)))      # gP/gDM

# Death of zooplankton:
# ---------------------

death.ZOO <- new(Class  = "process",
                 name   = "Death of zooplankton",
                 rate   = expression(k.death.ZOO*C.ZOO),
                 stoich = list(C.ZOO  = expression(-1)))            # gDM/gDM

# Definition of reactor:
# ======================

# Epilimnion:
# -----------

epilimnion <- 
   new(Class            = "reactor",
       name             = "Epilimnion",
       volume.ini       = expression(A*h.epi),
       conc.pervol.ini  = list(C.HPO4 = expression(C.HPO4.ini),     # gP/m3
                               C.ALG  = expression(C.ALG.ini),      # gDM/m3
                               C.ZOO  = expression(C.ZOO.ini)),     # gDM/m3
       inflow           = expression(Q.in*86400),                   # m3/d
       inflow.conc      = list(C.HPO4 = expression(C.HPO4.in),
                               C.ALG  = 0,
                               C.ZOO  = 0),
       outflow          = expression(Q.in*86400),
       processes        = list(gro.ALG,death.ALG,gro.ZOO,death.ZOO))

# Definition of system:
# =====================

# Lake system:
# ------------

system <- new(Class    = "system",
              name     = "Lake",
              reactors = list(epilimnion),
              param    = param,
              t.out    = seq(0,365,by=1))

# Perform simulation:
# ===================

res <- calcres(system)

# Plot results:
# =============
                 
plotres(res)              # plot to screen

# plotres(res,file="ecosim_example_plot1.pdf")  # plot to pdf file

plotres(res, colnames=c("C.ALG", "C.ZOO"))  # plot selected variables

plotres(res, colnames=list("C.HPO4",c("C.ALG", "C.ZOO")))

plotres(res[1:100,], colnames=list("C.HPO4",c("C.ALG", "C.ZOO"))) # plot selected time steps

# plotres(res      = res,    # plot to pdf file
#         colnames = list("C.HPO4",c("C.ALG","C.ZOO")),
#         file     = "ecosim_example_plot2.pdf",
#         width    = 8,
#         height   = 4)

# Perform sensitivity analysis:
# =============================
 
res.sens <- calcsens(system,param.sens=c("k.gro.ALG","k.gro.ZOO"))

# Plot results of sensitivity analysis:
# =====================================
 
plotres(res.sens)              # plot to screen

# plotres(res.sens,file="ecosim_example_plot3.pdf")  # plot to pdf file
}
