% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dscurve.R
\name{dscurve}
\alias{dscurve}
\title{Curve objects}
\usage{
dscurve(fun, yfun = "", col = "black", image = "", lwd = 3, n = NULL,
  iters = 0, crop = TRUE, tstart = 0, tend = 1, discretize = FALSE,
  xlim = NULL, ...)
}
\arguments{
\item{fun}{A function. If \code{yfun} is provided, this is the x-equation of the parametric
equations. If not, the function's graph is used.
See sections describing graphs and parameteric equations for more info.}

\item{yfun}{The y-equation of the parameteric equations.
See sections describing parametric equations for more info.}

\item{col}{The color of the original curve, as a string.}

\item{image}{A single color as a string, or a vector of colors as a string.
See details for more information.}

\item{lwd}{Line width expressed as a double. Only used if \code{discretize} is not set.}

\item{n}{The number of points that will be calculated.
Defaults to the dsrange's \code{renderCount}.
\code{n} is used to interact with \code{discretize}.}

\item{iters}{Determines the number of iterations of the function when making a color gradient.
Use \code{col = color1, image = color2, iters = n} to create a gradient of colors between
color1 and color2. See details for more information.}

\item{crop}{If \code{crop==TRUE}, the original curve and all iterations are cropped to the range.}

\item{tstart}{Only used for parametric curves. The minimum input
for both functions. Default 0.}

\item{tend}{Only used for parametric curves. The maximum input
for the functions. Default 1.}

\item{discretize}{Set \code{discretize=TRUE} to display the calculated points, instead of
connecting them as a curve: the curve is displayed with \code{points}
instead of \code{lines}.}

\item{xlim}{Only used for the graph of a function. Determines the range of x values for which the function
is plotted. Defaults to the x limits of the model's dsrange.}

\item{...}{Further graphical parameters passed to \code{lines} or \code{points}.}
}
\description{
This function takes a description of a curve and creates an object displaying the curve, and optionally
it's behavior throughout iterations of the system.
The curve is defined either by the graph of a single function or a pair of parametric
equations. By default, rendered with the \code{lines} function.
}
\section{The graph of a function}{


If the parameter \code{fun} is a function and
the parameter \code{yfun} is not provided, then \code{dscurve} contains
the curve of points (x,fun(x)). The inputs to \code{fun} are \code{n} points between the maximum
\code{\link{dsrange}}'s x limits, but can be overwritten with the \code{xlim} parameter.
}

\section{Parametric equations}{


If the parameter \code{fun} and \code{yfun} are both provided,
\code{dscurve} contains the parametric curve described by the functions. The function is
calculated at \code{n}
points ranging from \code{tmin} to \code{tmax}.
}

\section{Images of curves}{


The \code{dscurve} object begins with an initial curve. Images of the curve may be displayed in three ways.
If the \code{image} parameter is a single color and \code{iters} is not set, then \code{dscurve} will calculate and display
the image of the curve under the model's function in that color.

If the \code{image}  parameter is a vector of k colors, then \code{dscurve} calculates and
displays k successive images of the curve using those colors.
The string "NA" may be used to avoid displaying an iteration.

If the \code{image} parameter is a single color and \code{iters} is defined, then \code{iters}
successive images are displayed, using a gradient between \code{col} and \code{image}.

In most cases, rather than specifying \code{col} and \code{image} separately, they may be
combined into a single vector.
}
\examples{
library(dsmodels)

fun <- function(X,Y) {
  list(
    X/exp(Y),
    Y/exp(X)
  )
}

model <- dsmodel(fun, title = "Points on a One-Dimensional Curve")
range <- dsrange(-2:2,-2:2, discretize = 0.5)

# Add a one-dimensional curve and its image in blue.
curve1 <- dscurve(function(x) x^2,
                col = "orange",
                image = "blue",
                discretize = TRUE,
                xlim = c(-2,2))
model +
 range +
	curve1

# Create a parametric curve with image iterations red then green.
curve2 <- dscurve(function(x) x^2,
                function(x) x,
                image = c("red", "green"),
                tstart = -2, tend = 2)
dsmodel(fun, "A Parametric Curve and Iterations of that Curve") +
 dsrange(-2:2, -2:2, discretize = 0.5) +
 curve2

}
\seealso{
\code{\link{dspoint}}
}

