#' @title Function \code{parallelism_choices}
#' @description List the types of supported parallel computing.
#' @export
#' @seealso \code{\link{make}}, \code{\link{shell_file}}
#' @return Character vector listing the types of parallel
#' computing supported.
#' @details Run \code{make(..., parallelism = x, jobs = n)} for any of
#' the following values of \code{x} to distribute targets over parallel
#' units of execution.
#' \describe{
#'  \item{'parLapply'}{launches multiple processes in a single R session
#'  using \code{parallel::\link{parLapply}()}.
#'  This is single-node, (potentially) multicore computing.
#'  It requires more overhead than the \code{'mclapply'} option,
#'  but it works on Windows. If \code{jobs} is \code{1} in
#'  \code{\link{make}()}, then no 'cluster' is created and
#'  no parallelism is used.}
#'  \item{'mclapply'}{uses multiple processes in a single R session.
#'  This is single-node, (potentially) multicore computing.
#'  Does not work on Windows for \code{jobs > 1}
#'  because \code{\link{mclapply}()} is based on forking.}
#'  \item{'Makefile'}{uses multiple R sessions
#'  by creating and running a Makefile.
#'  For distributed computing on a cluster or supercomputer,
#'  try \code{\link{make}(..., parallelism = 'Makefile',
#'  prepend = 'SHELL=./shell.sh')}.
#'  You need an auxiliary \code{shell.sh} file for this,
#'  and \code{\link{shell_file}()}
#'  writes an example.
#'
#'  Here, Makefile-level parallelism is only used for
#'  targets in your workflow plan
#'  data frame, not imports. To process imported objects and files,
#'  drake selects the best parallel
#'  backend for your system and uses
#'  the number of jobs you give to the \code{jobs}
#'  argument to \code{\link{make}()}.
#'  To use at most 2 jobs for imports and at most 4 jobs
#'  for targets, run
#'  \code{make(..., parallelism = 'Makefile', jobs = 2, args = '--jobs=4')}
#'
#'  Caution: the Makefile generated by
#'  \code{\link{make}(..., parallelism = 'Makefile')}
#'  is NOT standalone. DO NOT run it outside of
#' \code{\link{make}()} or \code{\link{make}()}.
#'  Also, Windows users will need to download and intall Rtools.
#' }}
#' @examples
#' parallelism_choices()
parallelism_choices <- function() {
  c("parLapply", "mclapply", "Makefile")
}

#' @title Function \code{default_parallelism}
#' @description Default parallelism for \code{\link{make}()}:
#' \code{'parLapply'} for Windows machines and \code{'mclapply'}
#' for other platforms.
#' @export
#' @seealso \code{\link{make}}, \code{\link{shell_file}}
#' @return default parallelism option for the current platform
#' @examples
#' default_parallelism()
default_parallelism <- function() {
  ifelse(Sys.info()["sysname"] == "Windows", "parLapply", "mclapply") %>%
    unname
}

#' @title Function \code{max_useful_jobs}
#' @description Get the maximum number of useful jobs in the next call
#' to \code{make(..., jobs = YOUR_CHOICE)}.
#' @details Any additional jobs more than \code{max_useful_jobs(...)}
#' will be superfluous, and could even slow you down for
#' \code{make(..., parallelism = 'parLapply')}. Set
#' Set the \code{imports} argument to change your assumptions about
#' how fast objects/files are imported.
#' IMPORTANT: you must be in the root directory of your project.
#' @export
#' @return a list of three data frames: one for nodes, one for edges,
#' and one for the legend/key nodes.
#' @seealso \code{\link{plot_graph}}, \code{\link{build_graph}},
#' \code{\link{shell_file}}
#' @param plan workflow plan data frame, same as for function
#' \code{\link{make}()}.
#' @param targets names of targets to bulid, same as for function
#' \code{\link{make}()}.
#' @param envir environment to import from, same as for function
#' \code{\link{make}()}.
#' @param verbose logical, whether to output messages to the console.
#' @param jobs The \code{outdated()} function is called internally,
#' and it needs to import objects and examine your
#' input files to see what has been updated. This could take some time,
#' and parallel computing may be needed
#' to speed up the process. The \code{jobs} argument is number of parallel jobs
#' to use for faster computation.
#' @param parallelism Choice of parallel backend to speed up the computation.
#' Execution order in \code{\link{make}()} is slightly different
#' when \code{parallelism} equals \code{'Makefile'}
#' because in that case, all the imports are
#' imported before any target is built.
#' Thus, \code{max_useful_jobs()} may give a
#' different answer for Makefile parallelism.
#' See \code{?parallelism_choices} for details.
#' @param packages same as for \code{\link{make}}
#' @param prework same as for \code{\link{make}}
#' @param config internal configuration list of \code{\link{make}(...)},
#' produced also with \code{\link{config}()}.
#' Computing this
#' in advance could save time if you plan multiple calls to
#' \code{dataframes_graph()}.
#' @param imports Set the \code{imports} argument to change your
#' assumptions about how fast objects/files are imported.
#' Possible values:
#' \itemize{
#'  \item{'all'}{: Factor all imported files/objects into
#'    calculating the max useful number of jobs.
#'    Note: this is not appropriate for
#'    \code{make(.., parallelism = 'Makefile')} because imports
#'    are processed sequentially for the Makefile option.}
#'  \item{'files'}{: Factor all imported files into the calculation,
#'    but ignore all the other imports.}
#'  \item{'none'}{: Ignore all the imports and just focus on the max number
#'    of useful jobs for parallelizing targets.}
#' }
#' @examples
#' \dontrun{
#' load_basic_example()
#' plot_graph(my_plan) # Look at the graph to make sense of the output.
#' max_useful_jobs(my_plan) # 8
#' max_useful_jobs(my_plan, imports = 'files') # 8
#' max_useful_jobs(my_plan, imports = 'all') # 10
#' max_useful_jobs(my_plan, imports = 'none') # 8
#' make(my_plan)
#' plot_graph(my_plan)
#' # Ignore the targets already built.
#' max_useful_jobs(my_plan) # 1
#' max_useful_jobs(my_plan, imports = 'files') # 1
#' max_useful_jobs(my_plan, imports = 'all') # 10
#' max_useful_jobs(my_plan, imports = 'none') # 0
#' # Change a function so some targets are now out of date.
#' reg2 = function(d){
#'   d$x3 = d$x^3
#'   lm(y ~ x3, data = d)
#' }
#' plot_graph(my_plan)
#' max_useful_jobs(my_plan) # 4
#' max_useful_jobs(my_plan, imports = 'files') # 4
#' max_useful_jobs(my_plan, imports = 'all') # 10
#' max_useful_jobs(my_plan, imports = 'none') # 4
#' }
max_useful_jobs <- function(plan, targets = drake::possible_targets(plan),
  envir = parent.frame(), verbose = TRUE, jobs = 1,
  parallelism = drake::default_parallelism(),
  packages = (.packages()), prework = character(0), config = NULL,
  imports = c("files", "all", "none")) {
  force(envir)
  nodes <- dataframes_graph(plan = plan, targets = targets,
    envir = envir, verbose = verbose, jobs = jobs, parallelism = parallelism,
    packages = packages, prework = prework, config = config,
    split_columns = FALSE)$nodes
  imports <- match.arg(imports)
  just_targets <- intersect(nodes$id, plan$target)
  just_files <- Filter(x = nodes$id, f = is_file)
  targets_and_files <- union(just_targets, just_files)
  if (imports == "none")
    nodes <- nodes[just_targets, ] else if (imports == "files")
    nodes <- nodes[targets_and_files, ]
  nodes <- nodes[nodes$status != "up to date", ]
  if (!nrow(nodes))
    return(0)
  dlply(nodes, "level", nrow) %>%
    unlist %>%
    max
}

#' @title Function shell_file
#' @description Write an example \code{shell.sh} file required by
#' \code{make(..., parallelism = 'Makefile', prepend = 'SHELL=./shell.sh')}
#' and do a `chmod +x` to enable execution.
#' Use this option to run your project in parallel on a computing cluster
#' or supercomputer.
#' @seealso \code{\link{make}}, \code{\link{max_useful_jobs}},
#' \code{\link{parallelism_choices}}
#' @export
#' @param path file path of the shell file
shell_file <- function(path = file.path(getwd(), "shell.sh")) {
  writeLines(
    c(
      "#!/bin/bash",
      "shift",
      "echo \"module load R; $*\" | qsub -sync y -cwd -j y"),
    path)
  system2("chmod", args = c("+x", path))
}
