#' Dissimilarity Model Frame
#'
#' @description
#' This function generates a model frame for statistical or machine learning analysis from these objects:
#'
#' \itemize{
#'   \item: Dissimilarity data frame generated by [momentum()], [momentum_ls()], or [momentum_dtw()]. The output model frame will have as many rows as this data frame.
#'   \item: Data frame with static descriptors of the time series. These descriptors are converted to distances between pairs of time series via [distance_matrix()].
#'   \item: List defining composite predictors. This feature allows grouping together predictors that have a common meaning. For example, `composite_predictors = list(temperature = c("temperature_mean", "temperature_min", "temperature_max")` generates a new predictor named "temperature", which results from computing the multivariate distances between the vectors of temperature variables of each pair of time series. Predictors in one of such groups will be scaled before distance computation if their maximum standard deviations differ by a factor of 10 or more.
#' }
#'
#'
#' The resulting data frame contains the following columns:
#' \itemize{
#'   \item `x` and `y`: names of the pair of time series represented in the row.
#'   \item response columns.
#'   \item predictors columns: representing the distance between the values of the given static predictor between `x` and `y`.
#'   \item (optional) `geographic_distance`: If `predictors_df` is an `sf` data frame, then geographic distances are computed via [sf::st_distance()].
#' }
#'
#' This function supports a parallelization setup via [future::plan()].
#'
#' @param response_df (required, data frame) output of [momentum()], [momentum_ls()], or [momentum_dtw()]. Default: NULL
#' @param predictors_df (required, data frame or sf data frame) data frame with numeric predictors for the the model frame. Must have a column with the time series names in `response_df$x` and `response_df$y`. If `sf` data frame, the column "geographic_distance" with distances between pairs of time series is added to the model frame. Default: NULL
#' @param composite_predictors (optional, list) list defining composite predictors. For example, `composite_predictors = list(a = c("b", "c"))` uses the columns `"b"` and `"c"` from `predictors_df` to generate the predictor `a` as the multivariate distance between `"b"` and `"c"` for each pair of time series in `response_df`. Default: NULL
#' @param scale (optional, logical) if TRUE, all predictors are scaled and centered with [scale()]. Default: TRUE
#' @param distance (optional, string) Method to compute the distance between predictor values for all pairs of time series in `response_df`. Default: "euclidean".
#'
#' @return data frame: with the attribute "predictors".
#' @export
#' @autoglobal
#' @examples
#'
#' #Fagus sylvatica dynamics in Europe
#' tsl <- tsl_initialize(
#'   x = fagus_dynamics,
#'   name_column = "name",
#'   time_column = "time"
#' )
#'
#' #dissimilarity analysis
#' df <- momentum_ls(tsl = tsl)
#'
#' #generate model frame
#' model_frame <- momentum_model_frame(
#'   response_df = df,
#'   predictors_df = fagus_coordinates,
#'   scale = TRUE
#' )
#'
#' head(model_frame)
#'
#' #names of response and predictors
#' #and an additive formula
#' #are stored as attributes
#' attributes(model_frame)$predictors
#'
#' @family momentum_support
momentum_model_frame <- distantia_model_frame
