% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NLR.R
\name{NLR}
\alias{NLR}
\title{DIF statistics based on non-linear regression model.}
\usage{
NLR(Data, group, model, constraints = NULL, type = "all", method = "nls",
    match = "zscore", anchor = 1:ncol(Data), start, p.adjust.method = "none", test = "LR",
    alpha = 0.05, initboot = TRUE, nrBo = 20)
}
\arguments{
\item{Data}{data.frame or matrix: dataset which rows represent scored examinee answers (\code{"1"}
correct, \code{"0"} incorrect) and columns correspond to the items.}

\item{group}{numeric: binary vector of group membership. \code{"0"} for reference group, \code{"1"} for
focal group.}

\item{model}{character: generalized logistic regression model to be fitted. See \strong{Details}.}

\item{constraints}{character: which parameters should be the same for both groups. Possible values
are any combinations of parameters \code{"a"}, \code{"b"}, \code{"c"}, and \code{"d"}. Default value
is \code{NULL}. See \strong{Details}.}

\item{type}{character: type of DIF to be tested. Possible values are \code{"all"} for detecting
difference in any parameter (default), \code{"udif"} for uniform DIF only (i.e., difference in
difficulty parameter \code{"b"}), \code{"nudif"} for non-uniform DIF only (i.e., difference in
discrimination parameter \code{"a"}), \code{"both"} for uniform and non-uniform DIF (i.e.,
difference in parameters \code{"a"} and \code{"b"}), or combination of parameters \code{"a"},
\code{"b"}, \code{"c"}, and \code{"d"}. Can be specified as a single value (for all items) or as
an item-specific vector.}

\item{method}{character: method used to estimate parameters. The options are \code{"nls"} for
non-linear least squares (default) and \code{"likelihood"} for maximum likelihood method.}

\item{match}{character or numeric: matching criterion to be used as estimate of trait. Can be
either \code{"zscore"} (default, standardized total score), \code{"score"} (total test score),
or numeric vector of the same length as number of observations in \code{Data}.}

\item{anchor}{character or numeric: specification of DIF free items. A vector of item identifiers
(integers specifying the column  number) specifying which items are currently considered as anchor
(DIF free) items. Argument is ignored if \code{match} is not \code{"zscore"} or \code{"score"}.}

\item{start}{numeric: initial values for estimation of parameters. If not specified, starting
values are calculated with \code{\link[difNLR]{startNLR}} function. Otherwise, list with as many
elements as number of items. Each element is a named numeric vector of length 8 representing initial
values for parameter estimation. Specifically, parameters \code{"a"}, \code{"b"}, \code{"c"}, and
\code{"d"} are initial values for discrimination, difficulty, guessing, and inattention for reference
group. Parameters \code{"aDif"}, \code{"bDif"}, \code{"cDif"}, and \code{"dDif"} are then differences
in these parameters between reference and focal group.}

\item{p.adjust.method}{character: method for multiple comparison correction. Possible values are
\code{"holm"}, \code{"hochberg"}, \code{"hommel"}, \code{"bonferroni"}, \code{"BH"}, \code{"BY"},
\code{"fdr"}, and \code{"none"} (default). For more details see \code{\link[stats]{p.adjust}}.}

\item{test}{character: test to be performed for DIF detection. Can be either \code{"LR"} for
likelihood ratio test of a submodel (default), or \code{"F"} for F-test of a submodel.}

\item{alpha}{numeric: significance level (default is 0.05).}

\item{initboot}{logical: in case of convergence issues, should be starting values re-calculated based on
bootstraped samples? (default is \code{TRUE}; newly calculated initial values are applied only to
items/models with convergence issues).}

\item{nrBo}{numeric: the maximal number of iterations for calculation of starting values using
bootstraped samples (default is 20).}
}
\value{
A list with the following arguments:
\describe{
  \item{\code{Sval}}{the values of \code{test} statistics.}
  \item{\code{pval}}{the p-values by \code{test}.}
  \item{\code{adjusted.pval}}{adjusted p-values by \code{p.adjust.method}.}
  \item{\code{df}}{the degress of freedom of \code{test}.}
  \item{\code{test}}{used test.}
  \item{\code{par.m0}}{the matrix of estimated item parameters for m0 model.}
  \item{\code{se.m0}}{the matrix of standard errors of item parameters for m0 model.}
  \item{\code{cov.m0}}{list of covariance matrices of item parameters for m0 model.}
  \item{\code{par.m1}}{the matrix of estimated item parameters for m1 model.}
  \item{\code{se.m1}}{the matrix of standard errors of item parameters for m1 model.}
  \item{\code{cov.m1}}{list of covariance matrices of item parameters for m1 model.}
  \item{\code{conv.fail}}{numeric: number of convergence issues.}
  \item{\code{conv.fail.which}}{the indicators of the items which did not converge.}
  \item{\code{ll.m0}}{log-likelihood of m0 model.}
  \item{\code{ll.m1}}{log-likelihood of m1 model.}
  \item{\code{startBo0}}{the binary matrix. Columns represents iterations of initial values
  re-calculations, rows represents items. The value of 0 means no convergence issue in m0 model,
  1 means convergence issue in m0 model.}
  \item{\code{startBo1}}{the binary matrix. Columns represents iterations of initial values
  re-calculations, rows represents items. The value of 0 means no convergence issue in m1 model,
  1 means convergence issue in m1 model.}
}
}
\description{
Calculates either DIF likelihood ratio statistics or F statistics for dichotomous
data based on non-linear regression model (generalized logistic regression model).
}
\details{
Calculation of the test statistics using DIF detection procedure based on non-linear regression
(extension of logistic regression procedure; Swaminathan and Rogers, 1990; Drabinova and Martinkova, 2017).

The unconstrained form of 4PL generalized logistic regression model for probability of correct
answer (i.e., \eqn{y = 1}) is
\deqn{P(y = 1) = (c + cDif*g) + (d + dDif*g - c - cDif*g)/(1 + exp(-(a + aDif*g)*(x - b - bDif*g))), }
where \eqn{x} is by default standardized total score (also called Z-score) and \eqn{g} is a group membership.
Parameters \eqn{a}, \eqn{b}, \eqn{c}, and \eqn{d} are discrimination, difficulty, guessing, and inattention.
Terms \eqn{aDif}, \eqn{bDif}, \eqn{cDif}, and \eqn{dDif} then represent differences between two groups
(reference and focal) in relevant parameters.

This 4PL model can be further constrained by \code{model} and \code{constraints} arguments.
The arguments \code{model} and \code{constraints} can be also combined. Both arguments can
be specified as a single value (for all items) or as an item-specific vector (where each
element correspond to one item).

The \code{model} argument offers several predefined models. The options are as follows:
\code{Rasch} for 1PL model with discrimination parameter fixed on value 1 for both groups,
\code{1PL} for 1PL model with discrimination parameter fixed for both groups,
\code{2PL} for logistic regression model,
\code{3PLcg} for 3PL model with fixed guessing for both groups,
\code{3PLdg} for 3PL model with fixed inattention for both groups,
\code{3PLc} (alternatively also \code{3PL}) for 3PL regression model with guessing parameter,
\code{3PLd} for 3PL model with inattention parameter,
\code{4PLcgdg} for 4PL model with fixed guessing and inattention parameter for both groups,
\code{4PLcgd} (alternatively also \code{4PLd}) for 4PL model with fixed guessing for both groups,
\code{4PLcdg} (alternatively also \code{4PLc}) for 4PL model with fixed inattention for both groups,
or \code{4PL} for 4PL model.

The \code{model} can be specified in more detail with \code{constraints} argument which specifies what
parameters should be fixed for both groups. For example, choice \code{"ad"} means that discrimination
(parameter \code{"a"}) and inattention (parameter \code{"d"}) are fixed for both groups and other parameters
(\code{"b"} and \code{"c"}) are not. The \code{NA} value for \code{constraints} means no constraints.

In case that model considers difference in guessing or inattention parameter, the different parameterization is
used and parameters with standard errors are re-calculated by delta method.
}
\examples{
\dontrun{
# loading data based on GMAT
data(GMAT)

Data <- GMAT[, 1:20]
group <- GMAT[, "group"]

# Testing both DIF effects using LR test (default)
# and model with fixed guessing for both groups
NLR(Data, group, model = "3PLcg")

# Using F test
NLR(Data, group, model = "3PLcg", test = "F")

# Testing both DIF effects with Benjamini-Hochberg correction
NLR(Data, group, model = "3PLcg", p.adjust.method = "BH")

# 4PL model with the same guessing and inattention
# to test uniform DIF
NLR(Data, group, model = "4PLcgdg", type = "udif")

# 2PL model to test non-uniform DIF
NLR(Data, group, model = "2PL", type = "nudif")

# 4PL model with fixed a and c parameter
# to test difference in b
NLR(Data, group, model = "4PL", constraints = "ac", type = "b")

# using maximum likelihood estimation method
NLR(Data, group, model = "3PLcg", method = "likelihood")
}

}
\references{
Drabinova, A. & Martinkova P. (2017). Detection of Differential Item Functioning with NonLinear Regression:
Non-IRT Approach Accounting for Guessing. Journal of Educational Measurement, 54(4), 498-517,
\url{https://doi.org/10.1111/jedm.12158}.

Swaminathan, H. & Rogers, H. J. (1990). Detecting Differential Item Functioning Using Logistic Regression Procedures.
Journal of Educational Measurement, 27(4), 361-370,
\url{https://doi.org/10.1111/j.1745-3984.1990.tb00754.x}
}
\seealso{
\code{\link[stats]{p.adjust}}
}
\author{
Adela Hladka (nee Drabinova) \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
Faculty of Mathematics and Physics, Charles University \cr
\email{hladka@cs.cas.cz} \cr

Patricia Martinkova \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
\email{martinkova@cs.cas.cz} \cr

Karel Zvara \cr
Faculty of Mathematics and Physics, Charles University \cr
}
\keyword{DIF}
