% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dalmatian.R, R/dalmatian_doc.R
\docType{package}
\name{dalmatian}
\alias{dalmatian}
\title{Run DGLM in \code{JAGS} via \code{rjags} or in \code{nimble}}
\usage{
dalmatian(
  df,
  family = "gaussian",
  mean.model,
  dispersion.model,
  joint.model = NULL,
  jags.model.args,
  coda.samples.args,
  response = NULL,
  ntrials = NULL,
  rounding = FALSE,
  lower = NULL,
  upper = NULL,
  parameters = NULL,
  svd = FALSE,
  residuals = FALSE,
  gencode = NULL,
  run.model = TRUE,
  engine = "JAGS",
  n.cores = 1L,
  drop.levels = TRUE,
  drop.missing = TRUE,
  include.checks = TRUE,
  overwrite = FALSE,
  debug = FALSE,
  saveJAGSinput = NULL
)
}
\arguments{
\item{df}{Data frame containing the response and predictor values for each individual. (data.frame)}

\item{family}{Name of family of response distribution. Currently supported families include normal (\code{gaussian}) and negative binomial (\code{nbinom}). (character)}

\item{mean.model}{Model list specifying the structure of the mean. (list)}

\item{dispersion.model}{Model list specifying the structure of the dispersion. (list)}

\item{joint.model}{Model list specifying structure with parameter shared between linear predictors of the mean and variance. (list)}

\item{jags.model.args}{List containing named arguments of \code{jags.model}. (list)}

\item{coda.samples.args}{List containing named arguments of \code{coda.samples}. (list)}

\item{response}{Name of variable in the data frame representing the response. (character)}

\item{ntrials}{Name of variable in the data frame representing the number of independent trials for each observation of the beta binomial model.}

\item{rounding}{Specifies that response has been rounded if TRUE. (logical)}

\item{lower}{Name of variable in the data frame representing the lower bound on the response if rounded. (character)}

\item{upper}{Name of variable in the data frame representing the upper bound on the response if rounded. (character)}

\item{parameters}{Names of parameters to monitor. If NULL then default values are selected. (character)}

\item{svd}{Compute Singular Variable Decomposition of model matrices to improve convergence. (logical)}

\item{residuals}{If TRUE then compute residuals in output. (logical)}

\item{gencode}{If TRUE then generate code potentially overwriting existing model file. By default generate code if the file does not exist and prompt user if it does. (logical)}

\item{run.model}{If TRUE then run sampler. Otherwise, stop once code and data have been created. (logical)}

\item{engine}{Specifies the sampling software. Packages currently supported include JAGS (the default) and nimble. (character)}

\item{n.cores}{Number of cores to use. If equal to 1 then chains will not be run in parallel. If greater than 1 then chains will be run in parallel using the designated number of cores.}

\item{drop.levels}{If TRUE then drop unused levels from all factors in \code{df}. (logical)}

\item{drop.missing}{If TRUE then remove records with missing response variable. (logical)}

\item{include.checks}{If TRUE (default) then include extra Bernoulli variables in the model to ensure that the mean and dispersion parameters remain within their support. (logical)}

\item{overwrite}{If TRUE then overwrite existing JAGS files (non-interactive sessions only). (logical)}

\item{debug}{If TRUE then enter debug model. (logical)}

\item{saveJAGSinput}{Directory to which jags.model input is saved prior to calling \code{jags.model()}. This is useful for debugging. No files saved if NULL. (character)}
}
\value{
An object of class \code{dalmatian} containing copies of the original data frame, the mean model, the
dispersion model the arguments of \code{jags.model} and \code{coda.samples}. and the output of the MCMC sampler.
}
\description{
The primary function which automates the running of \code{JAGS} and \code{nimble}.

See vignettes included in the package for full documentation. The list
of available vignettes can be generated with
\code{vignette(package="dalmatian")}.
}
\details{
The primary function in the package, dalmatian automates the generation of code, data, and initial values. These are then passed as arguments to function from the \code{rjags} package which automates the generation of samples from the posterior.
}
\examples{

\dontrun{
## Load pied flycatcher data
data(pied_flycatchers_1)

## Create variables bounding the true load
pfdata$lower=ifelse(pfdata$load==0,log(.001),log(pfdata$load-.049))
pfdata$upper=log(pfdata$load+.05)
## Mean model
mymean=list(fixed=list(name="alpha",
                      formula=~ log(IVI) + broodsize + sex,
                      priors=list(c("dnorm",0,.001))))

## Dispersion model
myvar=list(fixed=list(name="psi",
                      link="log",
                      formula=~broodsize + sex,
                      priors=list(c("dnorm",0,.001))))

## Set working directory
## By default uses a system temp directory. You probably want to change this.
workingDir <- tempdir()

## Define list of arguments for jags.model()
jm.args <- list(file=file.path(workingDir,"pied_flycatcher_1_jags.R"),n.adapt=1000)

## Define list of arguments for coda.samples()
cs.args <- list(n.iter=5000)

## Run the model using dalmatian
pfresults <- dalmatian(df=pfdata,
                         mean.model=mymean,
                         dispersion.model=myvar,
                         jags.model.args=jm.args,
                         coda.samples.args=cs.args,
                         rounding=TRUE,
                         lower="lower",
                         upper="upper",
                         debug=FALSE)
}                          
}
\references{
Bonner, S., Kim, H., Westneat, D., Mutzel, A.,
Wright, J., and Schofield, M.. (2021). \code{dalmatian}: A Package
for Fitting Double Hierarchical Linear Models in \code{R} via \code{JAGS} and
\code{nimble}. \emph{Journal of Statistical Software}, 100, 10, 1--25.
\doi{10.18637/jss.v100.i10}.
}
\author{
Simon Bonner
}
