% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_trajectory.R
\name{traj_sim}
\alias{traj_sim}
\title{Generate a Trajectory with Correlated Step Lengths and Turn Angles}
\usage{
traj_sim(
  n,
  copula,
  marginal_circ,
  marginal_lin,
  ignore_first = TRUE,
  pos_2 = NULL
)
}
\arguments{
\item{n}{\link[base]{integer}, number of trajectory steps to generate.}

\item{copula}{'\code{\linkS4class{cyl_copula}}' object.}

\item{marginal_circ}{named \link[base]{list} (for parametric estimates) or
a '\code{\link[circular]{density.circular}}' object (for kernel density estimates).
The output of function \code{\link{fit_angle}()} can be used here directly for
both cases.}

\item{marginal_lin}{named \link[base]{list} (for parametric estimates) or
a '\code{\link[stats]{density}}' object (for kernel density estimates).
The output of function \code{\link{fit_steplength}()} can be used here directly for
both cases.}

\item{ignore_first}{\link[base]{logical} value. If \code{ignore_first = TRUE} (default),
a trajectory of length \code{n+2} is generated and the first two steps of that
 trajectory are removed.}

\item{pos_2}{(optional) \link[base]{numeric} \link[base]{vector} of length 2
 containing the coordinates of the second point in the trajectory.
 The first point is always at (0,0). If
no value is specified, the second point is obtained by going in a random direction
from the first point for a distance drawn from the marginal step length distribution.}
}
\value{
A \link[base]{data.frame} containing the trajectory. It has 6 columns
containing the x and y coordinates, the turn angles, the step lengths, and
the values sampled from the copula.
}
\description{
The function draws values from a circular-linear bivariate distribution of
turn angles and step lengths specified by the marginal distributions and a
circular-linear copula. From the start
point (0,0) and the second (potentially user specified)
point, a trajectory is then built with these turn angles and step lengths.
}
\details{
Samples are drawn from the circular-linear copula and then transformed
using the quantile functions of the marginal circular and the marginal linear
distribution. To generate draws from any bivariate joint distribution (not
necessarily a circular-linear one) without also producing a trajectory,
the function \code{\link{rjoint}()} can be used.

If entered "by hand", the named lists describing the parametric distributions
(\code{marginal_circ} and \code{marginal_lin}) must contain 2 entries:
\enumerate{
   \item{\code{name}:
a \link[base]{character} string denoting the name of the distribution.
For the circular distribution, it can be \code{"vonmises"}, \code{"vonmisesmix"}, or
  \code{"wrappedcauchy"}. For the linear distribution, it must be a
  string denoting the name of a linear distribution in the environment, i.e. the name of its
   distribution function without the "p",
  e.g. "norm" for normal distribution}
   \item{\code{coef}: For the circular distribution \code{coef} is a (named) \link[base]{list} of
parameters of the circular
marginal distribution as taken by the functions
\code{\link[circular]{qvonmises}()}, \code{\link{qvonmisesmix}()},
or \code{\link{qwrappedcauchy}()}. For the linear distribution, \code{coef} is
a named list containing the parameters of the distribution given in \code{"name"}.}
}
}
\examples{
require(circular)
set.seed(123)

traj <- traj_sim(n = 5,
copula = cyl_quadsec(0.1),
marginal_circ = list(name="vonmises",coef=list(0, 1)),
marginal_lin = list(name="weibull",coef=list(shape=3))
)

traj

angles <- rvonmisesmix(100,
  mu = c(0, pi),
  kappa = c(2, 3),
  prop = c(0.4, 0.6)
)
angles <- full2half_circ(angles)
bw <- opt_circ_bw(theta = angles, method = "nrd", kappa.est = "trigmoments")
marg_ang <- fit_angle(theta = angles, parametric = FALSE, bandwidth = bw)

steplengths <- rlnorm(100, 0, 0.3)
marg_stepl <- fit_steplength(x = steplengths, parametric = "lnorm")

traj_sim(n = 5,
copula = cyl_quadsec(0.1),
marginal_circ = marg_ang,
marginal_lin = marg_stepl,
ignore_first = FALSE,
pos_2 = c(5,5)
)

}
\seealso{
\code{\link{traj_get}()},
\code{\link{fit_steplength}()}, \code{\link{fit_angle}()},
\code{\link{plot_track}()}, \code{\link{plot_cop_scat}()},
\code{\link{plot_joint_scat}()}, \code{\link{plot_joint_circ}()}.
}
