% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/knn.R
\name{cuml_knn}
\alias{cuml_knn}
\alias{cuml_knn.default}
\alias{cuml_knn.data.frame}
\alias{cuml_knn.matrix}
\alias{cuml_knn.formula}
\alias{cuml_knn.recipe}
\title{Build a KNN model.}
\usage{
cuml_knn(x, ...)

\method{cuml_knn}{default}(x, ...)

\method{cuml_knn}{data.frame}(
  x,
  y,
  algo = c("brute", "ivfflat", "ivfpq", "ivfsq"),
  metric = c("euclidean", "l2", "l1", "cityblock", "taxicab", "manhattan",
    "braycurtis", "canberra", "minkowski", "chebyshev", "jensenshannon", "cosine",
    "correlation"),
  p = 2,
  neighbors = 5L,
  ...
)

\method{cuml_knn}{matrix}(
  x,
  y,
  algo = c("brute", "ivfflat", "ivfpq", "ivfsq"),
  metric = c("euclidean", "l2", "l1", "cityblock", "taxicab", "manhattan",
    "braycurtis", "canberra", "minkowski", "chebyshev", "jensenshannon", "cosine",
    "correlation"),
  p = 2,
  neighbors = 5L,
  ...
)

\method{cuml_knn}{formula}(
  formula,
  data,
  algo = c("brute", "ivfflat", "ivfpq", "ivfsq"),
  metric = c("euclidean", "l2", "l1", "cityblock", "taxicab", "manhattan",
    "braycurtis", "canberra", "minkowski", "chebyshev", "jensenshannon", "cosine",
    "correlation"),
  p = 2,
  neighbors = 5L,
  ...
)

\method{cuml_knn}{recipe}(
  x,
  data,
  algo = c("brute", "ivfflat", "ivfpq", "ivfsq"),
  metric = c("euclidean", "l2", "l1", "cityblock", "taxicab", "manhattan",
    "braycurtis", "canberra", "minkowski", "chebyshev", "jensenshannon", "cosine",
    "correlation"),
  p = 2,
  neighbors = 5L,
  ...
)
}
\arguments{
\item{x}{Depending on the context:

  * A __data frame__ of predictors.
  * A __matrix__ of predictors.
  * A __recipe__ specifying a set of preprocessing steps
  * created from [recipes::recipe()].
  * A __formula__ specifying the predictors and the outcome.}

\item{...}{Optional arguments; currently unused.}

\item{y}{A numeric vector (for regression) or factor (for classification) of
desired responses.}

\item{algo}{The query algorithm to use. Must be one of
  {"brute", "ivfflat", "ivfpq", "ivfsq"} or a KNN algorithm specification
  constructed using the \code{cuml_knn_algo_*} family of functions.
  If the algorithm is specified by one of the \code{cuml_knn_algo_*}
  functions, then values of all required parameters of the algorithm will
  need to be specified explicitly.
  If the algorithm is specified by a character vector, then parameters for
  the algorithm are generated automatically.

  Descriptions of supported algorithms:
    - "brute": for brute-force, slow but produces exact results.
    - "ivfflat": for inverted file, divide the dataset in partitions
                 and perform search on relevant partitions only.
    - "ivfpq": for inverted file and product quantization (vectors
               are divided into sub-vectors, and each sub-vector is encoded
               using intermediary k-means clusterings to provide partial
               information).
    - "ivfsq": for inverted file and scalar quantization (vectors components
               are quantized into reduced binary representation allowing
               faster distances calculations).

  Default: "brute".}

\item{metric}{Distance metric to use. Must be one of {"euclidean", "l2",
"l1", "cityblock", "taxicab", "manhattan", "braycurtis", "canberra",
"minkowski", "lp", "chebyshev", "linf", "jensenshannon", "cosine",
"correlation"}.
Default: "euclidean".}

\item{p}{Parameter for the Minkowski metric. If p = 1, then the metric is
equivalent to manhattan distance (l1). If p = 2, the metric is equivalent
to euclidean distance (l2).}

\item{neighbors}{Number of nearest neighbors to query. Default: 5L.}

\item{formula}{A formula specifying the outcome terms on the left-hand side,
and the predictor terms on the right-hand side.}

\item{data}{When a __recipe__ or __formula__ is used, \code{data} is
specified as a  __data frame__ containing the predictors and (if
applicable) the outcome.}
}
\value{
A KNN model that can be used with the 'predict' S3 generic to make
  predictions on new data points.
  The model object contains the following:
    - "knn_index": a GPU pointer to the KNN index.
    - "algo": enum value of the algorithm being used for the KNN query.
    - "metric": enum value of the distance metric used in KNN computations.
    - "p": parameter for the Minkowski metric.
    - "n_samples": number of input data points.
    - "n_dims": dimension of each input data point.
}
\description{
Build a k-nearest-model for classification or regression tasks.
}
\examples{

library(cuml)
library(MASS)
library(magrittr)
library(purrr)

set.seed(0L)

centers <- list(c(3, 3), c(-3, -3), c(-3, 3))

gen_pts <- function(cluster_sz) {
  pts <- centers \%>\%
    map(~ mvrnorm(cluster_sz, mu = .x, Sigma = matrix(c(1, 0, 0, 1), nrow = 2)))

  rlang::exec(rbind, !!!pts) \%>\% as.matrix()
}

gen_labels <- function(cluster_sz) {
  seq_along(centers) \%>\%
    sapply(function(x) rep(x, cluster_sz)) \%>\%
    factor()
}

sample_cluster_sz <- 1000
sample_pts <- cbind(
  gen_pts(sample_cluster_sz) \%>\% as.data.frame(),
  label = gen_labels(sample_cluster_sz)
)

model <- cuml_knn(label ~ ., sample_pts, algo = "ivfflat", metric = "euclidean")

test_cluster_sz <- 10
test_pts <- gen_pts(test_cluster_sz) \%>\% as.data.frame()

predictions <- predict(model, test_pts)
print(predictions, n = 30)
}
