#' Prior plotting
#'
#' Plots priors for free model parameters in a ctStanModel.
#' 
#' @param x ctStanModel object as generated by \code{\link{ctModel}} with type='stanct' or 'standt'.
#' @param rows vector of integers denoting which rows of ctstanmodel$pars to plot priors for. 
#' Character string 'all' plots all rows with parameters to be estimated.
#' @param wait If true, user is prompted to continue before plotting next graph.
#' @param samples Numeric. Higher values increase fidelity (smoothness / accuracy) of density plots, at cost of speed.
#' @param hypersd Either 'marginalise' to sample from the specified (in the ctstanmodel) 
#' prior distribution for the population standard deviation, or a numeric value to use for the population standard deviation
#' for all subject level prior plots - the plots in dotted blue or red.
#' @param ... not used.
#' @details Plotted in black is the prior for the population mean. In red and blue are the subject level priors that result
#' given that the population mean is estimated at 1 std deviation above the mean of the prior, or 1 std deviation below. 
#' The distributions around these two points are then obtained by marginalising over the prior for the population std deviation - 
#' so the red and blue distributions do not represent any specific subject level prior, but rather characterise the general amount
#' and shape of possible subject level priors at the selected points of the population mean prior.
#' @method plot ctStanModel
#' @export

plot.ctStanModel<-function(x,rows='all',wait=FALSE,samples=1e6, hypersd='marginalise',...){
  if(class(x)!='ctStanModel') stop('not a ctStanModel object!')
  m<-x$pars
  n<-5000
   highmean=1
  lowmean=-1
  if(rows=='all') rows<-1:nrow(m)
  for(rowi in rows){
    if(is.na(m$value[rowi])){
    
    #hypersd
      if(hypersd[1]=='marginalise'){
    hypersdpriorbase<-  stats::rnorm(samples*2)
    hypersdprior<-hypersdpriorbase[hypersdpriorbase>0] * m$sdscale[rowi]
    samples<-length(hypersdprior) #adjust number of samples because of random n > 0
      } else if(is.na(as.numeric(hypersd))) stop('hypersd argument is ill specified!') else {
        hypersdprior <- rep(hypersd,samples)
      }
    
#mean
      param=stats::rnorm(samples)
      xmean=eval(parse(text=paste0(m$transform[rowi])))
      meanxlims<-stats::quantile(xmean,probs=c(.1,.9))
      
      #high
      param=stats::rnorm(samples,highmean,hypersdprior)
      xhigh=eval(parse(text=paste0(m$transform[rowi])))
      highxlims <- stats::quantile(xhigh,probs=c(.1,.9))
        
      #low
      param=stats::rnorm(samples,lowmean,hypersdprior)
      xlow=eval(parse(text=paste0(m$transform[rowi])))
      lowxlims <- stats::quantile(xlow,probs=c(.1,.9))
      
      #combined
      xlims=c(min(meanxlims[1],lowxlims[1],highxlims[1]),max(meanxlims[2],lowxlims[2],highxlims[2]))
      xdistance= ( (highxlims[1]-lowxlims[1]) + (highxlims[2]-lowxlims[2]) )/2
      
       xmean=xmean[xmean>(xlims[1]-xdistance) & xmean < (xlims[2]+xdistance)]
       xhigh=xhigh[xhigh>(xlims[1]-xdistance) & xhigh < (xlims[2]+xdistance)]
       xlow=xlow[xlow>(xlims[1]-xdistance) & xlow < (xlims[2]+xdistance)]
       
       bw=(xlims[2]-xlims[1])/300
       
      densxmean=stats::density(xmean,bw=bw,n=n)
      densxlow=stats::density(xlow,bw=bw,n=n)
      densxhigh=stats::density(xhigh,bw=bw,n=n)

    ymax= max(c(densxmean$y),c(densxlow$y),c(densxhigh$y))
    
    graphics::plot(densxmean,main=m$param[rowi],lwd=2,xlim=c(xlims[1],xlims[2]),ylim=c(0,ymax))
    graphics::points(densxhigh,lwd=2,type='l',col='red',lty=3)
    graphics::points(densxlow,lwd=2,type='l',col='blue',lty=3)

    graphics::legend('topright',c('pop mean prior', '-1sd mean','+1sd mean'),text.col=c('black','blue','red'),bty='n')
      if(wait==TRUE & rowi != utils::tail(rows,1)){
        message("Press [enter] to display next plot")
        readline()
      }
    }
  }
}
