\name{ctmm}
\alias{ctmm}
\alias{ctmm.loglike}
\alias{ctmm.fit}
\alias{ctmm.select}
\encoding{UTF-8}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Specify, fit, and select continuous-time movement models}
\description{These functions allow one to propose hypothethical movement models (with initial estimates), fit those models to the data, and select among those models via an information criteria.
The fitting functions wrap around \code{optim} and \code{ctmm.loglike} to fit continuous-time movement models to 2D animal tracking data as described in Fleming et al (2014) and Fleming et al (2015), and Fleming et al (2017).
}
\usage{
ctmm(tau=NULL,omega=FALSE,isotropic=FALSE,range=TRUE,circle=FALSE,error=FALSE,
     axes=c("x","y"),...)

ctmm.loglike(data,CTMM,REML=FALSE,profile=TRUE,zero=0,verbose=FALSE)

ctmm.fit(data,CTMM=ctmm(),method="ML",COV=TRUE,control=list(),trace=FALSE)

ctmm.select(data,CTMM,verbose=FALSE,level=1,IC="AICc",MSPE="position",trace=FALSE,cores=1,
            ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{tau}{Array of autocorrelation timescales (explained below).}
  \item{omega}{Frequency (\eqn{2\pi/period}) of oscillatory range crossings.}
  \item{isotropic}{A Boolean denoting whether or not the animal's covariance is circular or elliptical.}
  \item{range}{A Boolean denoting whether or not the movement model has a finite range.}
  \item{circle}{The period it takes the animal to stochastically circle its mean location.}
  \item{error}{A Boolean denoting whether or not to use annotated telemetry error estimates or an estimate of the error's standard deviation if the data are not annotated with error estimates or when \eqn{HDOP=1}.}
  \item{axes}{Spatial dimensions of the movement model.}
  \item{data}{ Timeseries data represented as a \code{telemetry} object. }
  \item{CTMM}{ A \code{ctmm} movement-model object containing the initial parameter guesses conforming to the basic structure of the model hypothesis. \code{ctmm.select} can accept a list of such objects.}
  \item{REML}{Use residual maximum likelihood if \code{TRUE}. Not recommended.}
  \item{profile}{Analytically solve for as many covariance parameters as possible.}
  \item{zero}{Calculates \eqn{log(likelihood) - zero}, instead of just \eqn{log(likelihood)}, in a way that maintains numerical precision if the constant \code{zero} is close to the log likelihood. Used internally by \code{ctmm.fit}.}
  \item{verbose}{Return additional information. See "Value" below.}
  \item{method}{Fitting method to use: \code{"ML"}, \code{"HREML"}, \code{"pREML"}, \code{"pHREML"}, or \code{"REML"}. See "Description" below.}
  \item{COV}{Estimate the autocorrelation parameter covariance matrix.}
  \item{control}{An optional argument list with standardized numerical arguments.}
  \item{trace}{Report progress updates. Can be among \code{0:2} with increasing detail.}
  \item{level}{Attempt to simplify a model if a feature's non-existence falls within this level of confidence interval.}
  \item{IC}{Information criteria used for selection. Can be \code{"AICc"}, \code{"AIC"}, \code{"BIC"} or none (\code{NA}). AICc is approximate.}
  \item{MSPE}{Reject non-stationary features that increase the mean square predictive error of \code{"position"}, \code{"velocity"}, or not (\code{NA}).}
  \item{cores}{Maximum number of models to fit in parallel. cores=0 will use all cores, while cores<0 will reserve abs(cores).}
  \item{...}{Further arguments passed to \code{ctmm.fit}.}
}

\details{
Model fitting and selection first requires a prototype model with guesstimated parameters (i.e., Brownian motion with a particular diffusion rate).
The initial \code{ctmm} parameter guess can be generated by the output of \code{ctmm.guess}, \code{variogram.fit} or manually specified with the function \code{ctmm(...)}, where the argument \code{tau} is explained below and additonal model options described in \code{vignette("ctmm")}.

By default, \code{tau} (\eqn{\tau}) is an ordered array of autocorrelation timescales.
If \code{length(tau)==0}, then an IID bi-variate Gaussian model is fit to the data.
If \code{length(tau)==1}, then an Ornstein-Uhlenbeck (OU) model (Brownian motion restricted to a finite home range) is fit the data, where \code{tau} is the position autocorrelation timescale. \code{tau=Inf} then yields Brownian motion (BM).
If \code{length(tau)==2}, then the OUF model (continuous-velocity motion restricted to a finite home range) is fit to the data, where \code{tau[1]} is again the position autocorrelation timescale and \code{tau[2]} is the velocity autocorrelation timescale. \code{tau[1]=Inf} then yields integrated Ornstein-Uhlenbeck (IOU) motion, which is a spatially unrestricted continuous-velocity process.

Two new models were introduced in ctmm version 0.5.2 for the case of \code{tau[1]==tau[2]}, which can happen with short tracks of data. When \code{tau[1]==tau[2]} and \code{omega==0}, the model is categorized as OUf---a special case of OUF---and the two \code{tau} parameters are treated as identical. On the other hand, when \code{tau[1]==tau[2]} and \code{omega>0}, an oscillatory model is implemented, which we refer to as OU\eqn{\Omega}.

The potential fitting methods---maximum likelihood (\code{ML}), residual maximum likelihood (\code{REML}), perturbative REML (\code{pREML}), hybrid REML (\code{HREML}), and perturbative hybrid REML (\code{pHREML})---are described in Fleming et al (2019). In general, \code{pHREML} is the best method, though when parameter estimates lie near boundaries it can fail, in which case \code{ctmm.fit} will fall back to \code{HREML}, as reported by the \code{method} slot of the resulting fit object.

The \code{control} list can take the folowing arguments, with defaults shown:
\describe{
\item{\code{method="Nelder-Mead"}}{\code{\link{optim}} method for multiple parameters.}
\item{\code{precision=1/2}}{Fraction of machine numerical precision to target in the maximized likelihood value. MLEs will necessarily have half this precision. On most computers, \code{precision=1} is approximately 16 decimal digits of precision for the likelihood and 8 for the MLEs.}
\item{\code{maxit=.Machine$integer.max}}{Maximum number of iterations allowed for optimization.}
}

Model selection in \code{ctmm.select} proceeds by first fitting the initial model guess, and then attempting to simplify the autocorrelation model and complexify the deterministic (mean) model until the information criteria fails to improve. The intent of working in these directions is to avoid fitting trends to autocorrelation.
Note that simpler models in a nested hierarchy will only be attempted if they appear credible, which can be adjusted with the \code{level} argument. \code{level=1} will, therefore, always attempt a simpler model.
}

\value{
The function \code{ctmm} returns a prototype \code{ctmm} movement-model object.
By default, \code{ctmm.loglike} returns the log-likelihood of the model \code{CTMM}.
\code{ctmm.fit} (and \code{ctmm.loglike} with \code{verbose=TRUE}) returns the maximum likelihood \code{ctmm} movement-model object with all of the components of \code{CTMM} plus the components listed below.
\code{ctmm.select} returns the best model by default or the sorted list of attempted models if \code{verbose=TRUE}.
\describe{
\item{\code{AICc}}{The approximate corrected Akaike information criterion for multivariate distributions with variable numbers of unknown mean and (structured) covariance parameters (Burnham & Anderson, Eq. 7.91). This formula is only exact for IID data.}
\item{\code{loglike}}{The log-likelihood.}
\item{\code{sigma}}{The maximum likelihood variance/covariance estimate (possibly debiased). For the endlessly diffusing BM and IOU processes, this is instead the diffusion rate estimate.}
\item{\code{mu}}{The maximum likelihood stationary mean vector estimate.}
\item{\code{COV.mu}}{The covariance matrix of the \code{mu} estimate, assuming that the covariance estimate is correct.}
\item{\code{DOF.mu}}{The effective number of degrees of freedom in the estimate of \code{mu}, assuming that the autocorrelation model is correct. This can be much smaller than \code{length(data$t)} if the data are autocorrelated.}
\item{\code{COV}}{Covariance of the autocovariance parameter estimate vector \code{c(sigma,tau,circle)}, as derived (asymptotically) from the \code{hessian} of the log-likelihood function, and where \code{sigma} is parameterized in terms of its standard \code{area}, \code{eccentricity}, and \code{angle} of orientation. Typically, \code{sigma}'s \code{area} parameter is extremely correlated to \code{tau[1]}, and sequential components of \code{tau} are slightly correlated.}
}
}

\references{
K. P. Burnham, D. R. Anderson.
Model Selection and Multimodel Inference: A Practical Information-Theoretic Approach.
Springer, 2nd edition (2003).

C. H. Fleming, J. M. Calabrese, T. Mueller, K.A. Olson, P. Leimgruber, W. F. Fagan.
From fine-scale foraging to home ranges: A semi-variance approach to identifying movement modes across spatiotemporal scales.
\href{http://www.jstor.org/discover/10.1086/675504}{The American Naturalist, 183:5, E154-E167 (2014)}.

C. H. Fleming, Y. Subasi, J. M. Calabrese.
A maximum-entropy description of animal movement.
\href{http://journals.aps.org/pre/abstract/10.1103/PhysRevE.91.032107}{Physical Review E, 91, 032107 (2015)}.

C. H. Fleming, D. Sheldon, E. Gurarie, W. F. Fagan, S. LaPoint, J. M. Calabrese.
Kálmán filters for continuous-time movement models.
\href{http://www.sciencedirect.com/science/article/pii/S1574954117301115}{Ecological Informatics, 40, 8-21 (2017)}.
}


\author{ C. H. Fleming and G. Péron. }

\note{
The default \code{\link{optim}} method here is \code{"Nelder-Mead"} with the largest allowable number of maximum iterations. Annecdotally, the faster \code{"BFGS"} and \code{"L-BFGS-B"} methods often perform poorly on these types of problems.
However, Nelder Mead is not ideal and you may want to attempt a second fit, using the first fit as its initial guess.

The AICs/BICs of endlessly diffusing models like BM and IOU cannot be easily compared to the AICs/BICs of range resident models like bivariate Gaussian, OU, and OUF, as their joint likelihood functions are infinitely different. Endlessly diffusing models have to be conditioned off of an initial state, which we derive in \code{ctmm} by taking the large range limit of a range-restricted process. I.e., BM is the limit OU(\code{Inf}) and IOU(\code{tau}) is the limit OUF(\code{Inf},\code{tau}). Using comparable likelihood functions gives up statistical efficiency and the objective prior. Moreover, comparing conditional likelihoods---with the objective prior taken from the joint likelihood---does not appear to select the true model with a likelihood ratio test. Therefore, there does not appear to be a simple approach for selecting between range resident and endlessly diffusing movement models.

Prior to v0.3.6, the univariate AICc formula was (mis)used, with the full parameter count treated as degrees of freedom in the mean. As of v.0.3.6, the mean and autocovariance parameters are treated separately in the approximate multivariate AICc formula (Burnham & Anderson, Eq. 7.91). Still, this improved formula is only exact for IID data.

Prior to v0.3.2, \code{ctmm.select} would consider every possible model.
This is no longer feasible with current versions of \code{ctmm}, as the number of possible models has grown too large.
}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{ctmm.boot}}, \code{\link{ctmm.guess}}, \code{\link{optim}}, \code{\link{summary.ctmm}}, \code{\link{variogram.fit}}. }

\examples{\donttest{
# Load package and data
library(ctmm)
data(buffalo)
Cilla <- buffalo$Cilla

GUESS <- ctmm.guess(Cilla,interactive=FALSE)
FIT <- ctmm.fit(Cilla,GUESS)

# some human-readable information
summary(FIT)
}}

% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
%\keyword{ ~kwd1 }
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line
