% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxmeg_plink.R
\name{coxmeg_plink}
\alias{coxmeg_plink}
\title{Perform GWAS using a Cox mixed-effects model with plink files as input}
\usage{
coxmeg_plink(pheno, corr, bed = NULL, tmp_dir = NULL,
  cov_file = NULL, tau = NULL, maf = 0.05, min_tau = 1e-04,
  max_tau = 5, eps = 1e-06, order = 1, detap = TRUE,
  opt = "bobyqa", eigen = TRUE, score = FALSE, dense = FALSE,
  threshold = 0, solver = 1, spd = TRUE, mc = 100,
  verbose = TRUE, invchol = TRUE)
}
\arguments{
\item{pheno}{A string value indicating the file name or the full path of a pheno file. The files must be in the working directory if the full path is not given. The file is in plink pheno format, containing the following four columns, family ID, individual ID, time and status. The status is a binary variable (1 for events/0 for censored).}

\item{corr}{A relatedness matrix. Can be a matrix or a 'dgCMatrix' class in the Matrix package. Must be symmetric positive definite or symmetric positive semidefinite.}

\item{bed}{A optional string value indicating the file name or the full path of a plink bed file (without .bed). The files must be in the working directory if the full path is not given. If not provided, only the variance component will be returned.}

\item{tmp_dir}{A optional directory to store temporary .gds files. The directory needs to be specified when \code{bed} is provided.}

\item{cov_file}{An optional string value indicating the file name or the full path of a covariate file. The files must be in the working directory if the full path is not given. Same as the cov file in plink, the first two columns are family ID and individual ID. The covariates are included in the null model for estimating the variance component. The covariates can be quantitative or binary values. Categorical variables need to be converted to dummy variables.}

\item{tau}{An optional positive value for the variance component. If tau is given, the function will skip estimating the variance component, and use the given tau to analyze the SNPs.}

\item{maf}{An optional positive value. All SNPs with MAF<maf in the bed file will not be analyzed. Default is 0.05.}

\item{min_tau}{An optional positive value indicating the lower bound in the optimization algorithm for the variance component tau. Default is 1e-4.}

\item{max_tau}{An optional positive value indicating the upper bound in the optimization algorithm for the variance component tau. Default is 5.}

\item{eps}{An optional positive value indicating the tolerance in the optimization algorithm. Default is 1e-6.}

\item{order}{An optional integer value starting from 0. Only effective when \code{dense=FALSE}. It specifies the order of approximation used in the inexact newton method. Default is 1.}

\item{detap}{An optional logical value indicating whether to use approximation for log-determinant. Default is TRUE.}

\item{opt}{An optional string value for the Optimization algorithm for tau. Can have the following values: 'bobyqa', 'Brent' or 'NM'. Default is 'bobyqa'.}

\item{eigen}{An optional logical value. Only valid when dense=FALSE. It indicates whether to use RcppEigen:LDLT to solve linear systems. Default is TRUE.}

\item{score}{An optional logical value indicating whether to perform a score test. Default is FALSE.}

\item{dense}{An optional logical value indicating whether the relatedness matrix is dense. Default is FALSE.}

\item{threshold}{An optional non-negative value. If threshold>0, coxmeg_m will reestimate HRs for those SNPs with a p-value<threshold by first estimating a variant-specific variance component. Default is 0.}

\item{solver}{An optional binary value taking either 1 or 2. Default is 1. See details.}

\item{spd}{An optional logical value indicating whether the relatedness matrix is symmetric positive definite. Default is TRUE. See details.}

\item{mc}{An optional integer value specifying the number of Monte Carlo samples used for approximating the log-determinant. Only valid when dense=TRUE and detap=TRUE. Default is 100.}

\item{verbose}{An optional logical value indicating whether to print additional messages. Default is TRUE.}

\item{invchol}{An optional logical value. Only effective when \code{dense=FALSE}. If TRUE, sparse Cholesky decomposition is used to compute the inverse of the relatedness matrix. Otherwise, sparse LU is used.}
}
\value{
beta: The estimated coefficient for each predictor in X.

HR: The estimated HR for each predictor in X.

sd_beta: The estimated standard error of beta.

p: The p-value of each SNP.

tau: The estimated variance component.

rank: The rank of the relatedness matrix.

nsam: Actual sample size.
}
\description{
\code{coxmeg_plink} first estimates the variance component under a null model with only cov if tau is not given, and then analyzing each SNP in the plink files.
}
\section{About \code{corr}}{

The subjects in \code{corr} must be in the same order as in the plink fam file.
}

\section{About missing values}{

\code{pheno} -9 for missing values, \code{cov_file} NA for missing values.
}

\section{About temporary files}{

The function will create a temporary gds file with approximately the same size as the bed file. The temporary file will be removed when the analysis is done.
}

\section{About \code{spd}}{

When \code{spd=TRUE}, the relatedness matrix is treated as SPD. If the matrix is SPSD or not sure, set \code{spd=FALSE}.
}

\section{About \code{solver}}{

When \code{solver=1} (\code{solver=2}), Cholesky decompositon (PCG) is used to solve the linear system. However, when \code{dense=FALSE} and \code{eigen=FALSE}, the solve function in the Matrix package is used regardless of \code{solver}. When \code{dense=TRUE}, it is recommended to set \code{solver=2} to have better computational performance.
}

\section{About \code{invchol}}{

Cholesky decomposition using \code{invchol=TRUE} is generally (but not always) much faster to invert a relatedness matrix (e.g., a block-diagonal matrix). But for some types of sparse matrices (e.g., a banded AR(1) matrix with rho=0.9), it sometimes can be very slow. In such cases, \code{invchol=FALSE} is can be used.
}

\examples{
library(Matrix)
library(MASS)
library(coxmeg)

## build a block-diagonal relatedness matrix
n_f <- 600
mat_list <- list()
size <- rep(5,n_f)
offd <- 0.5
for(i in 1:n_f)
{
  mat_list[[i]] <- matrix(offd,size[i],size[i])
  diag(mat_list[[i]]) <- 1
}
sigma <- as.matrix(bdiag(mat_list))

## Estimate variance component under a null model
pheno = system.file("extdata", "ex_pheno.txt", package = "coxmeg")
cov = system.file("extdata", "ex_cov.txt", package = "coxmeg")
bed = system.file("extdata", "example_null.bed", package = "coxmeg")
bed = substr(bed,1,nchar(bed)-4)
re = coxmeg_plink(pheno,sigma,bed=bed,tmp_dir=tempdir(),cov_file=cov,detap=TRUE,dense=FALSE)
re
}
\keyword{Cox}
\keyword{mixed-effects}
\keyword{model}
