#' @title Balance Statistics for `MatchIt` Objects
#' 
#' @description
#' Generates balance statistics for `matchit` objects from \pkg{MatchIt}.
#' 
#' @inheritParams bal.tab
#' @param x a `matchit` object; the output of a call to \pkgfun{MatchIt}{matchit}.
#' @param distance an optional formula or data frame containing distance values (e.g., propensity scores) or a character vector containing their names. If a formula or variable names are specified, `bal.tab()` will look in the argument to `data`, if specified. The distance measure (e.g., propensity score) generated by `matchit()` is automatically included and named "distance".
#' @param s.d.denom `character`; how the denominator for standardized mean differences should be calculated, if requested. See [col_w_smd()] for allowable options. Abbreviations allowed. If not specified, `bal.tab()` will figure out which one is best based on the estimand of the `matchit` object: if ATT, `"treated"`; if ATC, `"control"`, otherwise `"pooled"`.
#' @param s.weights Optional; either a vector containing sampling weights for each unit or a string containing the name of the sampling weight variable in `data`. These function like regular weights except that both the adjusted and unadjusted samples will be weighted according to these weights if weights are used. If `s.weights` was specified in the call to `matchit()`, they will automatically be included and do not need be specified again (though there is no harm if they are).
#' @param method a character vector containing the method of adjustment. Ignored unless subclassification was used in the original call to `matchit()`. If `"weighting"`, the subclassification weights will be used and subclasses will be ignored. If `"subclassification"`, balance will be assessed using the subclasses (see [`bal.tab.subclass()`][class-bal.tab.subclass] for details). Abbreviations allowed.
#' 
#' @returns
#' If subclassification is used and `method` is set to `"subclassification"`, an object of class `"bal.tab.subclass"` containing balance summaries within and across subclasses. See [`bal.tab.subclass()`][class-bal.tab.subclass] for details.
#' 
#' If matching is used and clusters are not specified, an object of class `"bal.tab"` containing balance summaries for the `matchit` object. See [bal.tab()] for details.
#' 
#' If clusters are specified, an object of class `"bal.tab.cluster"` containing balance summaries within each cluster and a summary of balance across clusters. See [`bal.tab.cluster()`][class-bal.tab.cluster] for details.
#' 
#' @details
#' `bal.tab.matchit()` generates a list of balance summaries for the `matchit` object given, and functions similarly to \pkgfun{MatchIt}{summary.matchit}. `bal.tab()` behaves differently depending on whether subclasses are used in conditioning or not. If they are used, `bal.tab()` creates balance statistics for each subclass and for the sample in aggregate; see [`bal.tab.subclass()`][class-bal.tab.subclass] for more information. 
#' 
#' @seealso
#' * [bal.tab()] for details of calculations.
#' 
#' @examplesIf requireNamespace("MatchIt", quietly = TRUE)
#' library(MatchIt); data("lalonde", package = "cobalt")
#' 
#' ## Nearest Neighbor matching
#' m.out1 <- matchit(treat ~ age + educ + race + 
#'                       married + nodegree + re74 + re75, 
#'                   data = lalonde, method = "nearest")
#' 
#' bal.tab(m.out1, un = TRUE, m.threshold = .1, 
#'         v.threshold = 2)
#' 
#' ## Subclassification
#' m.out2 <- matchit(treat ~ age + educ + race + 
#'                       married + nodegree + re74 + re75, 
#'                   data = lalonde, method = "subclass")
#' 
#' bal.tab(m.out2, disp.subclass = TRUE)

#' @exportS3Method bal.tab matchit
bal.tab.matchit <-    function(x,
                               stats, int = FALSE, poly = 1, distance = NULL, addl = NULL, data = NULL, continuous, binary, s.d.denom, thresholds = NULL, weights = NULL, cluster = NULL, imp = NULL, pairwise = TRUE, s.weights = NULL, abs = FALSE, subset = NULL, quick = TRUE,
                               method, ...) {
    
    tryCatch(args <- c(as.list(environment()), list(...))[-1], error = function(e) .err(conditionMessage(e)))
    
    #Adjustments to arguments
    
    args[vapply(args, rlang::is_missing, logical(1L))] <- NULL
    args[vapply(args, is_null, logical(1L)) & names(args) %nin% names(match.call())[-1]] <- NULL
    
    #Initializing variables
    X <- do.call("x2base.matchit", c(list(x), args), quote = TRUE)
    
    args[names(args) %in% names(X)] <- NULL
    
    X <- assign.X.class(X)
    
    out <- do.call("base.bal.tab", c(list(X), args),
                   quote = TRUE)
    return(out)
}
