#'Test for auto-correlation in climate.
#'
#'Tests the correlation between the climate in a specified climate window and 
#'other fitted climate windows.
#'@param reference Reference climate data to be compared. Generated by functions
#'  \code{\link{singlewin}} or \code{\link{climatewin}}.
#'@param xvar The climate variable of interest. Please specify the parent 
#'  environment and variable name (e.g. Climate$Temp).
#'@param cdate The climate date variable (dd/mm/yyyy). Please specify the parent
#'  environment and variable name (e.g. Climate$Date).
#'@param bdate The biological date variable (dd/mm/yyyy). Please specify the 
#'  parent environment and variable name (e.g. Biol$Date).
#'@param baseline The baseline model used to fit climate windows. These will be
#'  correlated with the reference climate window.
#'@param furthest The furthest number of time intervals (set by cinterval) back 
#'  from the cutoff date or biological record to include in the climate window 
#'  search.
#'@param closest The closest number of time intervals (set by cinterval) back 
#'  from the cutoff date or biological record to include in the climate window 
#'  search.
#'@param stat The aggregate statistic used to analyse the climate data. Can 
#'  currently use basic R statistics (e.g. mean, min), as well as slope. 
#'  Additional aggregate statistics can be created using the format function(x) 
#'  (...). See parameter FUN in \code{\link{apply}} for more detail.
#'@param func The function used to fit the climate variable. Can be linear 
#'  ("lin"), quadratic ("quad"), cubic ("cub"), inverse ("inv") or log ("log").
#'  Not required when a variable is provided for parameter 'centre'.
#'@param type fixed or variable, whether you wish the climate window to be variable
#'  (e.g. the number of days before each biological record is measured) or fixed
#'  (e.g. number of days before a set point in time).
#'@param cutoff.day,cutoff.month If type is fixed, the day and month of the 
#'  year from which the fixed window analysis will start.
#'@param cmissing TRUE or FALSE, determines what should be done if there are 
#'  missing climate data. If FALSE, the function will not run if missing climate
#'  data is encountered. If TRUE, any records affected by missing climate data 
#'  will be removed from climate window analysis.
#'@param cinterval The resolution at which climate window analysis will be 
#'  conducted. May be days ("day"), weeks ("week"), or months ("month"). Note the units 
#'  of parameters 'furthest' and 'closest' will differ with the choice of 
#'  cinterval.
#'@param upper Cut-off value used to determine growing degree days or positive 
#'  climate thresholds (depending on parameter thresh). Note that when values
#'  of lower and upper are both provided, autowin will instead calculate an 
#'  optimal climate zone.
#'@param lower Cut-off value used to determine chill days or negative 
#'  climate thresholds (determined by parameter thresh). Note that when values
#'  of lower and upper are both provided, autowin will instead calculate an 
#'  optimal climate zone.
#'@param thresh TRUE or FALSE. Determines whether to use values of upper and
#'  lower to calculate binary climate data (thresh = TRUE), or to use for
#'  growing degree days (thresh = FALSE).
#'@param centre Variable used for mean centring (e.g. Year, Site, Individual).
#'  Please specify the parent environment and variable name (e.g. Biol$Year). 
#'@return Will return a data frame showing the correlation between the climate 
#'  in each fitted window and the chosen reference window.
#'@author Liam D. Bailey and Martijn van de Pol
#' @examples
#' \dontrun{
#' 
#' # Test for auto-correlation using 'Mass' and 'MassClimate' data frames
#' 
#' data(Mass)
#' data(MassClimate)
#' 
#' # Fit a single climate window using the datasets Mass and MassClimate.
#' 
#' single <- singlewin(xvar = list(Temp = MassClimate$Temp), 
#'                     cdate = MassClimate$Date, bdate = Mass$Date,
#'                     baseline = lm(Mass ~ 1, data = Mass), 
#'                     furthest = 72, closest = 15, 
#'                     stat = "mean", func = "lin", type = "fixed", 
#'                     cutoff.day = 20, cutoff.month = 5, 
#'                     cmissing = FALSE, cinterval = "day")            
#' 
#' # Test the autocorrelation between the climate in this single window and other climate windows.
#' 
#' auto <- autowin(reference = single$BestModelData$climate,
#'                 xvar  = list(Temp = MassClimate$Temp), cdate = MassClimate$Date, bdate = Mass$Date,
#'                 baseline = lm(Mass ~ 1, data = Mass), furthest = 365, closest = 0, 
#'                 stat = "mean", func = "lin", 
#'                 type = "fixed", cutoff.day = 20, cutoff.month = 5,
#'                 cmissing = FALSE, cinterval = "day")
#'                 
#' # View the output
#' head(auto)
#' 
#' # Plot the output
#' plotcor(auto, type = "A")                                   
#'}
#'        
#'@export

autowin <- function(reference, xvar, cdate, bdate, baseline, furthest, 
                    closest,  stat, func, type, cutoff.day, cutoff.month, 
                    cmissing = FALSE, cinterval = "day", upper = NA,
                    lower = NA, thresh = FALSE, centre = NULL){
  
  xvar = xvar[[1]]

  print("Initialising, please wait...")
  
  if (stat == "slope" & func == "log" || stat == "slope" & func == "inv"){
    stop("stat = slope cannot be used with func = log or inv as negative values may be present")
  }
  
  if (cinterval == "day"){
    if ((min(as.Date(bdate, format = "%d/%m/%Y")) - furthest) < min(as.Date(cdate, format = "%d/%m/%Y"))){
      stop("You do not have enough climate data to search that far back. Please adjust the value of furthest or add additional climate data.")
     }
  }
  
  if (cinterval == "week"){
    if ((min(as.Date(bdate, format = "%d/%m/%Y")) - lubridate::weeks(furthest)) < min(as.Date(cdate, format = "%d/%m/%Y"))){
      stop("You do not have enough climate data to search that far back. Please adjust the value of furthest or add additional climate data.")
    }
  }
  
  if (cinterval == "month"){
    if ((min(as.Date(bdate, format = "%d/%m/%Y")) - months(furthest)) < min(as.Date(cdate, format = "%d/%m/%Y"))){
      stop("You do not have enough climate data to search that far back. Please adjust the value of furthest or add additional climate data.")
    }
  }
  
  duration   <- (furthest - closest) + 1
  maxmodno   <- (duration * (duration + 1)) / 2 
  cont       <- convertdate(bdate = bdate, cdate = cdate, xvar = xvar, 
                             cinterval = cinterval, type = type, 
                             cutoff.day = cutoff.day, cutoff.month = cutoff.month)
  modno      <- 1
  modlist    <- list()
  cmatrix    <- matrix(ncol = (duration), nrow = length(bdate))
  climate1   <- matrix(ncol = 1, nrow = length(bdate), 1)

  if (is.na(upper) == FALSE && is.na(lower) == TRUE){
    if (thresh == TRUE){
      cont$xvar <- ifelse (cont$xvar > upper, 1, 0)
    } else {
      cont$xvar <- ifelse (cont$xvar > upper, cont$xvar, 0)
    }
  }
  
  
  if (is.na(lower) == FALSE && is.na(upper) == TRUE){
    if (thresh == TRUE){
      cont$xvar <- ifelse (cont$xvar < lower, 1, 0)
    } else {
      cont$xvar <- ifelse (cont$xvar < lower, cont$xvar, 0)
    }
  }
  
  if (is.na(lower) == FALSE && is.na(upper) == FALSE){
    if (thresh == TRUE){
      cont$xvar <- ifelse (cont$xvar > lower & cont$xvar < upper, 1, 0)
    } else {
      cont$xvar <- ifelse (cont$xvar > lower & cont$xvar < upper, cont$xvar - lower, 0)
    } 
  }
  
  # Create a matrix with the climate data from closest to furthest days
  # back from each biological record
  for (i in 1:length(bdate)){
    for (j in closest:furthest){
      k <- j - closest + 1
      cmatrix[i, k] <- cont$xvar[which(cont$cintno == cont$bintno[i] - j)]    
    }
  }
  
  if (cmissing == FALSE && length(which(is.na(cmatrix))) > 0){
    if (cinterval == "day"){
      .GlobalEnv$missing <- as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1)
    }
    if (cinterval == "month"){
      .GlobalEnv$missing <- c(paste("Month:", month(as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1)),
                                    "Year:", year(as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1))))
    }
    if (cinterval == "week"){
      .GlobalEnv$missing <- c(paste("Week:", month(as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1)),
                                    "Year:", year(as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1))))
    }
    stop(c("Climate data should not contain NA values: ", length(.GlobalEnv$missing),
           " NA value(s) found. Please add missing climate data or set cmissing=TRUE.
           See object missing for all missing climate data"))
  }
  
  if (cmissing == FALSE && length(which(is.na(cmatrix))) > 0){
    if (cinterval == "day"){
      .GlobalEnv$missing <- as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1)
    }
    if (cinterval == "month"){
      .GlobalEnv$missing <- c(paste("Month:", month(as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1)),
                                    "Year:", year(as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1))))
    }
    if (cinterval == "week"){
      .GlobalEnv$missing <- c(paste("Week:", month(as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1)),
                                    "Year:", year(as.Date(cont$cintno[is.na(cont$xvar)], origin = min(as.Date(cdate, format = "%d/%m/%Y")) - 1))))
    }
    stop(c("Climate data should not contain NA values: ", length(.GlobalEnv$missing),
           " NA value(s) found. Please add missing climate data or set cmissing=TRUE.
           See object missing for all missing climate data"))
  }
  
  if (cmissing == TRUE && length(which(is.na(cmatrix))) > 0){
    reference <- reference[complete.cases(cmatrix)]
    cmatrix   <- cmatrix[complete.cases(cmatrix), ]
  } 

  modeldat           <- model.frame(baseline)
  modeldat$yvar      <- modeldat[, 1]
  modeldat$climate   <- seq(1, nrow(modeldat), 1)
  
  if (is.null(weights(baseline)) == FALSE){
    modeldat$modweights <- weights(baseline)
    baseline            <- update(baseline, .~., weights = modeldat$modweights, data = modeldat)
  }
  
  if (func == "lin"){
    modeloutput <- update(baseline, .~. + climate, data = modeldat)
  } else if (func == "quad") {
    modeloutput <- update(baseline, .~. + climate + I(climate ^ 2), data = modeldat)
  } else if (func == "cub") {
    modeloutput <- update(baseline, .~. + climate + I(climate ^ 2) + I(climate ^ 3), data = modeldat)
  } else if (func == "log") {
    modeloutput <- update(baseline, .~. + log(climate), data = modeldat)
  } else if (func == "inv") {
    modeloutput <- update (baseline, .~. + I(climate ^ -1), data = modeldat)
  } else {
    print("Define func")
  }
  
  pb <- txtProgressBar(min = 0, max = maxmodno, style = 3, char = "|")
  
  for (m in closest:furthest){
    for (n in 1:duration){
      if ( (m - n) >= (closest - 1)){  # do not use windows that overshoot the closest possible day in window   
        if (stat != "slope" || n > 1){
          windowopen  <- m - closest + 1
          windowclose <- windowopen-n + 1
          if (stat == "slope"){ 
            time       <- seq(1, n, 1)
            climate1 <- apply(cmatrix[, windowclose:windowopen], 1, FUN = function(x) coef(lm(x ~ time))[2])
          } else { 
            if (n == 1){
              climate1 <- cmatrix[, windowclose:windowopen]
            } else {
              climate1 <- apply(cmatrix[, windowclose:windowopen], 1, FUN = stat)
            }
          }
          modeloutput <- cor(climate1, reference)
          
          modlist$cor[[modno]]         <- modeloutput
          modlist$WindowOpen[[modno]]  <- m
          modlist$WindowClose[[modno]] <- m - n + 1
          modno                        <- modno + 1
        }
      }
    }  
    #Fill progress bar
    setTxtProgressBar(pb, modno - 1)
  }
  
  modlist$Furthest    <- furthest
  modlist$Closest     <- closest
  modlist$Statistics  <- stat
  modlist$Functions   <- type
  
  if (type == TRUE){
    modlist$cutoff.day   <- cutoff.day
    modlist$cutoff.month <- cutoff.month 
  }
  
  local <- as.data.frame(modlist)
  return(local)
}